unit SuFrame;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls;

type
  { TSuFrame }

  TSuFrame = class(TFrame)
  private
    FVirtualClassName: ShortString;
  protected
    procedure SetName(const Value: TComponentName); override;
  public
    Designer: TIDesigner;
    procedure ValidateRename(AComponent: TComponent; const CurName,
       NewName: string); override;
  published
    property VirtualClassName: ShortString read FVirtualClassName write FVirtualClassName;
    property Caption;
  end;

  {TFrameItem = class
  public
    FocusedControl: TWinControl;
    Frame: TFrame;
    ImageIndex: Integer;
  end;

  { TCustomFrameBar }

  TFrameBarBorder = (fbLeft, fbTop, fbRight, fbBottom);
  TFrameBarBorders = set of TFrameBarBorder;

  TCustomFrameBar = class(TCustomControl)
  private
    FrameList: TList;
    FHightLightFont: TFont;
    FFrameAnderMouse: TFrame;
    //FLastActiveFrame: TFrame;
    FActiveFrame: TFrame;
    FCloseBtn: Boolean;
    FSysButton: Boolean;
    FOnCloseBtnDown: TNotifyEvent;
    FOnSysBtnClick: TNotifyEvent;
    FItemDefaultWidth: Integer;
    FMidleLineColor: TColor;
    FImageList: TImageList;
    FWrapText: Boolean;
    FHomeButton: Boolean;
    FHomeButtonState: TButtonState;
    FSysButtonState: TButtonState;
    FOnHomeButtonClick: TNotifyEvent;
    FHomeIconIndex: Integer;
    FSysIconIndex: Integer;
    OrderList: TList;
    FHomeActionLink: TControlActionLink;
    FFrameBarBorders: TFrameBarBorders;
    FNative: Boolean;
    FFramesParent: TWinControl;
    function GetFrameCount: Integer;
    function GetHomeAction: TBasicAction;
    function GetItems(Index: Integer): TFrame;
    function GetLastActiveFrame: TFrame;
    procedure SetFrameBarBorders(AValue: TFrameBarBorders);
    procedure SetFramesParent(AValue: TWinControl);
    procedure SetHighLightFont(AValue: TFont);
    procedure SetFrameAnderMouse(AValue: TFrame);
    procedure SetHomeAction(AValue: TBasicAction);
    procedure SetHomeButton(AValue: Boolean);
    procedure SetHomeButtonState(AValue: TButtonState);
    procedure SetHomeIconIndex(AValue: Integer);
    procedure SetImageList(AValue: TImageList);
    procedure SetLastActiveFrame(AValue: TFrame);
    procedure DoHomeActionChange(Sender: TObject);
    procedure SetNative(AValue: Boolean);
    procedure SetSysButton(AValue: Boolean);
    procedure SetSysButtonState(AValue: TButtonState);
    procedure SetSysIconIndex(AValue: Integer);
  protected
    CloseBtnWidth: Integer;
    SpaceWidth: Integer;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure Paint; override;
    procedure UpdateFrameAnderMouse(P: TPoint);
    procedure AddInList(AFrame: TFrame; IconIndex: Integer);
    procedure InsertInList(AFrame: TFrame; Index: Integer; IconIndex: Integer);
    procedure MouseLeave; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer
       ); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer
       ); override;
    procedure DoCloseBtnDown(AFrame: TFrame);
    function GetCloseBtnRect(ParentRect: TRect): TRect;
    function GetFrameItem(Index: Integer): TFrameItem;
    function GetFrameItemOfFrame(AFrame: TFrame): TFrameItem;
    function GetButtonRect(Index: Integer): TRect;
    function GetButtonsRect: TRect;
    function GetSysButtonRect: TRect;
    procedure DoHomeButtonClick;
    procedure DoSysButtonClick;
    procedure HomeActionChange(Sender: TObject);

    property HomeButtonState: TButtonState read FHomeButtonState write SetHomeButtonState;
    property SysButtonState: TButtonState read FSysButtonState write SetSysButtonState;
    property FrameAnderMouse: TFrame read FFrameAnderMouse write SetFrameAnderMouse;
    property LastActiveFrame: TFrame read GetLastActiveFrame write SetLastActiveFrame;
    property HomeActionLink: TControlActionLink read FHomeActionLink write FHomeActionLink;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    property Items[Index: Integer]: TFrame read GetItems;
    procedure AddFrame(AFrame: TFrame; IconIndex: Integer = -1);
    procedure InsertFrame(AFrame, AfterFrame: TFrame; IconIndex: Integer = -1);
    procedure SaveFocusedForFrame(AFrame: TFrame; AWinControl: TWinControl);
    procedure ActivateFrame(AFrame: TFrame);
    procedure RemoveFrame(AFrame: TFrame);
    property FrameCount: Integer read GetFrameCount;
    property ActiveFrame: TFrame read FActiveFrame;
    property HighLightFont: TFont read FHightLightFont write SetHighLightFont;
    property CloseBtn: Boolean read FCloseBtn write FCloseBtn;
    property SysButton: Boolean read FSysButton write SetSysButton;
    property OnCloseBtnDown: TNotifyEvent read FOnCloseBtnDown write FOnCloseBtnDown;
    property ItemDefaultWidth: Integer read FItemDefaultWidth write FItemDefaultWidth;
    property MildeLineColor: TColor read FMidleLineColor write FMidleLineColor;
    property ImageList: TImageList read FImageList write SetImageList;
    property WrapText: Boolean read FWrapText write FWrapText;
    property HomeButton: Boolean read FHomeButton write SetHomeButton;
    property OnHomeButtonClick: TNotifyEvent read FOnHomeButtonClick write FOnHomeButtonClick;
    property OnSysButtonClick: TNotifyEvent read FOnSysBtnClick write FOnSysBtnClick;
    property HomeIconIndex: Integer read FHomeIconIndex write SetHomeIconIndex;
    property SysIconIndex: Integer read FSysIconIndex write SetSysIconIndex;
    property HomeActoin: TBasicAction read GetHomeAction write SetHomeAction;
    property FrameBarBorders: TFrameBarBorders read FFrameBarBorders write SetFrameBarBorders default [];
    property Native: Boolean read FNative write SetNative;
    property FramesParent: TWinControl read FFramesParent write SetFramesParent;
  end;

  {TFrameBar}
  TFrameBar = class(TCustomFrameBar)
  published
    property Align;
    property Color;
    property Font;
    property ParentColor;
    property ParentFont;
    property HighLightFont;
    property CloseBtn;
    property OnCloseBtnDown;
    property ItemDefaultWidth;
    property MildeLineColor;
    property ImageList;
    property WrapText;
    property HomeButton;
    property OnHomeButtonClick;
    property TabStop;
    property TabOrder;
    property HomeIconIndex;
    property HomeActoin;
    property FrameBarBorders;
    property Native;
    property SysButton;
    property SysIconIndex;
    property OnSysButtonClick;
    property FramesParent;
  end; }

implementation

{$R *.lfm}

{ TCustomFrameBar }

{procedure TCustomFrameBar.DoHomeActionChange(Sender: TObject);
begin
  if Sender = HomeActoin then HomeActionChange(Sender);
end;

procedure TCustomFrameBar.SetNative(AValue: Boolean);
begin
  if FNative=AValue then Exit;
  FNative:=AValue;
  Invalidate;
end;

procedure TCustomFrameBar.SetSysButton(AValue: Boolean);
begin
  if FSysButton <> AValue then
  begin
    FSysButton:= AValue;
    Invalidate;
  end;
end;

procedure TCustomFrameBar.SetSysButtonState(AValue: TButtonState);
var
  R: TRect;
begin
  if FSysButtonState <> AValue then
  begin
    FSysButtonState := AValue;
    R := GetSysButtonRect;
    InvalidateRect(Handle, @R, False);
  end;
end;

procedure TCustomFrameBar.SetSysIconIndex(AValue: Integer);
begin
  if FSysIconIndex <> AValue then
  begin
    FSysIconIndex:=AValue;
    Invalidate;
  end;
end;

function TCustomFrameBar.GetFrameCount: Integer;
begin
  Result := FrameList.Count;
end;

function TCustomFrameBar.GetHomeAction: TBasicAction;
begin
  if HomeActionLink <> nil then
    Result := HomeActionLink.Action
  else
    Result := nil;
end;

function TCustomFrameBar.GetItems(Index: Integer): TFrame;
begin
  Result := nil;
  if (Index >= 0) and (Index < FrameList.Count) then
  begin
    Result := TFrameItem(FrameList.Items[Index]).Frame;
  end;
end;

function TCustomFrameBar.GetLastActiveFrame: TFrame;
begin
  Result := nil;
  if OrderList.Count > 0 then Result := TFrame(OrderList.Items[0]);
end;

procedure TCustomFrameBar.SetFrameBarBorders(AValue: TFrameBarBorders);
begin
  if FFrameBarBorders=AValue then Exit;
  FFrameBarBorders:=AValue;
  Invalidate;
end;

procedure TCustomFrameBar.SetFramesParent(AValue: TWinControl);
begin
  if FFramesParent=AValue then Exit;
  FFramesParent:=AValue;
  FFramesParent.FreeNotification(Self);
end;

procedure TCustomFrameBar.SetHighLightFont(AValue: TFont);
begin
  if FHightLightFont=AValue then Exit;
  FHightLightFont:=AValue;
  Invalidate;
end;

procedure TCustomFrameBar.SetFrameAnderMouse(AValue: TFrame);
begin
  if FFrameAnderMouse <> AValue then
  begin
    FFrameAnderMouse := AValue;
    Invalidate;
  end;
end;

procedure TCustomFrameBar.SetHomeAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    HomeActionLink.Free;
    HomeActionLink := nil;
  end
  else
  begin
    if HomeActionLink = nil then
      HomeActionLink := GetActionLinkClass.Create(Self);
    HomeActionLink.Action := AValue;
    HomeActionLink.OnChange:=@DoHomeActionChange;
    HomeActionChange(AValue);
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomFrameBar.SetHomeButton(AValue: Boolean);
begin
  if FHomeButton <> AValue then
  begin
    FHomeButton:= AValue;
    Invalidate;
  end;
end;

procedure TCustomFrameBar.SetHomeButtonState(AValue: TButtonState);
var
  R: TRect;
begin
  if FHomeButtonState <> AValue then
  begin
    FHomeButtonState := AValue;
    R := GetButtonRect(FrameCount);
    InvalidateRect(Handle, @R, False);
  end;
end;

procedure TCustomFrameBar.SetHomeIconIndex(AValue: Integer);
begin
  if FHomeIconIndex <> AValue then
  begin
    FHomeIconIndex:=AValue;
    Invalidate;
  end;
end;

procedure TCustomFrameBar.SetImageList(AValue: TImageList);
begin
  if FImageList=AValue then Exit;
  FImageList:=AValue;
  Invalidate;
end;

procedure TCustomFrameBar.SetLastActiveFrame(AValue: TFrame);
var
  Find: Boolean;
  I: Integer;
  FI: TFrameItem;
begin
  if AValue = nil then Exit;

  if Screen.ActiveControl <> nil then
  begin
    if Screen.ActiveControl.Owner = AValue then
    begin
      FI := GetFrameItemOfFrame(AValue);
      FI.FocusedControl := Screen.ActiveControl;
    end;
  end;

  Find := False;
  for I := 0 to OrderList.Count - 1 do
  begin
    if TFrame(OrderList.Items[I]) = AValue then
    begin
      Find := True;
      OrderList.Move(I, 0);
      Break;
    end;
  end;

  if not Find then OrderList.Insert(0, AValue);
end;

procedure TCustomFrameBar.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (Operation = opRemove) then
  begin
    if AComponent is TFrame then RemoveFrame(TFrame(AComponent))
    else
    if AComponent = FFramesParent then
      FFramesParent := nil
    else
    if AComponent is TBasicAction then
    begin
      if AComponent = HomeActoin then HomeActoin := nil;
    end;
  end;
  inherited Notification(AComponent, Operation);
end;

procedure TCustomFrameBar.Paint;

type
  TEdgeStyle = (esNone, esRaised, esLowered);

const
  InnerStyles: array[TEdgeStyle] of Integer = (0, BDR_RAISEDINNER, BDR_SUNKENINNER);
  OuterStyles: array[TEdgeStyle] of Integer = (0, BDR_RAISEDOUTER, BDR_SUNKENOUTER);

var
  AR, TextRect, ImgRect: TRect;
  I, W, X, Y, TxtOffset: Integer;
  S: string;
  details: TThemedElementDetails;
  txtStyle: TTextStyle;
  AnderMouse: Boolean;
  ImgIndex: Integer;
  DrawFrame: TFrame;
  FEdgeBorderType : Cardinal;
begin
  AR := ClientRect;
  Canvas.Brush.Color := Color;
  if FNative then
  begin
    Details := ThemeServices.GetElementDetails(thHeaderItemNormal);
    ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
  end
  else
    Canvas.FillRect(AR);

  FEdgeBorderType := 0;
  if (fbTop in FFrameBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_TOP);
  if (fbBottom in FFrameBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_BOTTOM);
  if (fbLeft in FFrameBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_LEFT);
  if (fbRight in FFrameBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_RIGHT);
  DrawEdge(Canvas.Handle, AR,
           InnerStyles[esRaised] or OuterStyles[esLowered], FEdgeBorderType);

  AnderMouse := False;
  for I := 0 to FrameCount - 1 do
  begin
    DrawFrame := Items[I];
    AR := GetButtonRect(I);

    if I > 0 then
    begin
      Canvas.Pen.Color := MildeLineColor;
      X := AR.Left - SpaceWidth div 2 - 1;
      Canvas.MoveTo(X, AR.Top + 2);
      Canvas.LineTo(X, AR.Bottom - 2);
    end;

    AnderMouse:= False;
    if DrawFrame = FrameAnderMouse then AnderMouse:= True;

    if DrawFrame.Visible then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonHot);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
      Canvas.Font.Assign(HighLightFont);
    end
    else
    begin
      //details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      if AnderMouse then
      begin
        Canvas.Font.Assign(Font);
        details := ThemeServices.GetElementDetails(tbPushButtonNormal);
        ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
      end
      else
      begin
        //Canvas.Brush.Color := Color;
        //Canvas.FillRect(AR);
        Canvas.Font.Assign(Font);
      end;
    end;

    TxtOffset := 2;
    if (ImageList <> nil) then
    begin
      ImgIndex := GetFrameItem(I).ImageIndex;

      if ImgIndex >= 0 then
      begin
        TxtOffset := 4 + ImageList.Width + TxtOffset;
        X := AR.Left + 4;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, ImgIndex, DrawFrame.Enabled);
      end;
    end;

    S := DrawFrame.Caption;
    TextRect := AR;
    InflateRect(TextRect, -TxtOffset, -1);
    if TxtOffset <= 2 then TextRect.Right := TextRect.Right - 10;
    txtStyle.SingleLine := not WrapText;
    txtStyle.RightToLeft := False;
    txtStyle.Wordbreak := True;
    txtStyle.Alignment := taCenter;
    txtStyle.Layout := tlCenter;
    txtStyle.EndEllipsis := False;
    txtStyle.SystemFont := False;
    txtStyle.Opaque := False;

    Canvas.TextRect(TextRect, TextRect.Left, TextRect.Top, S, txtStyle);

    if AnderMouse and CloseBtn then
    begin
      AR := GetCloseBtnRect(AR);
      Canvas.Brush.Color:= clRed;
      Canvas.Pen.Color:= clGray;
      Canvas.RoundRect(AR, 2, 2);
    end;

  end;

  if HomeButton then
  begin
    AR := GetButtonRect(FrameCount);
    {if HomeButtonState = bsUp then
    begin
      Canvas.FillRect(AR);
    end
    else}
    if HomeButtonState = bsHot then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end
    else
    if HomeButtonState = bsDown then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonPressed);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end;
    if FrameCount > 0 then
    begin
      Canvas.Pen.Color := MildeLineColor;
      X := AR.Left - SpaceWidth div 2 - 1;
      Canvas.MoveTo(X, AR.Top + 2);
      Canvas.LineTo(X, AR.Bottom - 2);
    end;
    if (ImageList <> nil) then
    begin
      if (HomeIconIndex >= 0) and (HomeIconIndex < ImageList.Count) then
      begin
        X := AR.Left + (AR.Right - AR.Left - ImageList.Width) div 2;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, HomeIconIndex);
      end;
    end;
  end;

  if SysButton then
  begin
    AR := GetSysButtonRect;

    if SysButtonState = bsHot then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end
    else
    if SysButtonState = bsDown then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonPressed);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end;

    if (ImageList <> nil) then
    begin
      if (SysIconIndex >= 0) and (SysIconIndex < ImageList.Count) then
      begin
        X := AR.Left + (AR.Right - AR.Left - ImageList.Width) div 2;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, SysIconIndex);
      end;
    end;
  end;

end;

procedure TCustomFrameBar.UpdateFrameAnderMouse(P: TPoint);
var
  I: Integer;
  AR: TRect;
  AFrame: TFrame;
begin
  AFrame := nil;
  for I := 0 to FrameCount - 1 do
  begin
    AR := GetButtonRect(I);
    if PtInRect(AR, P) then
    begin
      if Items[I].Enabled then
        AFrame := Items[I];
      Break;
    end;
  end;
  FrameAnderMouse := AFrame;
end;

procedure TCustomFrameBar.AddInList(AFrame: TFrame; IconIndex: Integer);
var
  FI: TFrameItem;
begin
  FI := TFrameItem.Create;
  FI.Frame := AFrame;
  FI.ImageIndex := IconIndex;

  FrameList.Add(FI);
  AFrame.FreeNotification(Self);
  //if AFrame is TSuFrame then TSuFrame(AFrame).FrameBar := Self;
  ActivateFrame(AFrame);
end;

procedure TCustomFrameBar.InsertInList(AFrame: TFrame; Index: Integer;
  IconIndex: Integer);
var
  FI: TFrameItem;
begin
  FI := TFrameItem.Create;
  FI.Frame := AFrame;
  FI.ImageIndex := IconIndex;

  FrameList.Insert(Index, FI);
  AFrame.FreeNotification(Self);
  //if AFrame is TSuFrame then TSuFrame(AFrame).FrameBar := Self;
  ActivateFrame(AFrame);
end;

procedure TCustomFrameBar.MouseLeave;
begin
  inherited MouseLeave;
  FrameAnderMouse := nil;
  HomeButtonState := bsUp;
  SysButtonState := bsUp;
end;

procedure TCustomFrameBar.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  P: TPoint;
  AR: TRect;
  CheckHome: Boolean;
  I, W: Integer;
begin
  if Button = mbLeft then
  begin
    P.x := X;
    P.y := Y;
    W := 200;
    CheckHome := True;
    for I := 0 to FrameCount - 1 do
    begin
      AR := GetButtonRect(I);
      if PtInRect(AR, P) then
      begin
        if Items[I].Enabled then
        begin
          InflateRect(AR, -2, -1);
          if CloseBtn then
          begin
            AR := GetCloseBtnRect(AR);
            if not PtInRect(AR, P) then
              ActivateFrame(Items[I]);
          end
          else ActivateFrame(Items[I]);
        end;
        CheckHome := False;
        Break;
      end;
    end;
    if CheckHome and HomeButton then
    begin
      AR := GetButtonRect(FrameCount);
      if PtInRect(AR, P) then
      begin
        HomeButtonState := bsDown;
        CheckHome:= False;
      end;
    end;

    if CheckHome and SysButton then
    begin
      AR := GetSysButtonRect;
      if PtInRect(AR, P) then
      begin
        SysButtonState := bsDown;
      end;
    end;
  end;
end;

procedure TCustomFrameBar.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
  AR: TRect;
  B: Boolean;
begin
  P.x := X;
  P.y := Y;
  UpdateFrameAnderMouse(P);

  B := True;
  if HomeButton then
  begin
    AR := GetButtonRect(FrameCount);
    if PtInRect(AR, P) then
    begin
      B := False;
      if HomeButtonState <> bsDown then
        HomeButtonState := bsHot
    end
    else
    begin
      if HomeButtonState <> bsDown then
        HomeButtonState:= bsUp;
    end;
  end;

  if SysButton and B then
  begin
    AR := GetSysButtonRect;
    if PtInRect(AR, P) then
    begin
      if SysButtonState <> bsDown then
        SysButtonState := bsHot
    end
    else
    begin
      if SysButtonState <> bsDown then
        SysButtonState := bsUp;
    end;
  end;
end;

procedure TCustomFrameBar.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
var
  AR: TRect;
  P: TPoint;
  I: Integer;
begin
  //inherited MouseUp(Button, Shift, X, Y);
  if Button = mbLeft then
  begin
    P.x := X;
    P.y := Y;
    for I := 0 to FrameCount - 1 do
    begin
      AR := GetButtonRect(I);
      if PtInRect(AR, P) then
      begin
        if Items[I].Enabled then
        begin
          InflateRect(AR, -2, -1);
          if CloseBtn then
          begin
            AR := GetCloseBtnRect(AR);
            if PtInRect(AR, P) then
            begin
              DoCloseBtnDown(Items[I]);
              Application.ProcessMessages;
              UpdateFrameAnderMouse(P);
            end;
          end;
        end;
        Break;
      end;
    end;

    if HomeButton and (HomeButtonState = bsDown) then
    begin
      AR := GetButtonRect(FrameCount);
      if PtInRect(AR, P) then
      begin
        HomeButtonState := bsHot;
        DoHomeButtonClick;
        UpdateFrameAnderMouse(P);
      end
      else
        HomeButtonState := bsUp;
    end;

    if SysButton and (SysButtonState = bsDown) then
    begin
      AR := GetSysButtonRect;
      if PtInRect(AR, P) then
      begin
        SysButtonState := bsHot;
        DoSysButtonClick;
      end
      else
        SysButtonState := bsUp;
    end;
  end;
end;

procedure TCustomFrameBar.DoCloseBtnDown(AFrame: TFrame);
begin
  if Assigned(FOnCloseBtnDown) then FOnCloseBtnDown(AFrame);
end;

function TCustomFrameBar.GetCloseBtnRect(ParentRect: TRect): TRect;
begin
  Result.Left:= ParentRect.Right - CloseBtnWidth - 4;
  Result.Right:= Result.Left + CloseBtnWidth;
  Result.Top:= ParentRect.Top + (ParentRect.Bottom - ParentRect.Top - CloseBtnWidth) div 2;
  Result.Bottom:= Result.Top + CloseBtnWidth;
end;

function TCustomFrameBar.GetFrameItem(Index: Integer): TFrameItem;
begin
  Result := nil;
  if (Index >= 0) and (Index < FrameList.Count) then
  begin
    Result := TFrameItem(FrameList.Items[Index]);
  end;
end;

function TCustomFrameBar.GetFrameItemOfFrame(AFrame: TFrame): TFrameItem;
var
  I: Integer;
  FI: TFrameItem;
begin
  Result := nil;
  for I := 0 to FrameCount - 1 do
  begin
    FI := GetFrameItem(I);
    if FI.Frame = AFrame then
    begin
      Result := FI;
      Break;
    end;
  end;
end;

function TCustomFrameBar.GetButtonRect(Index: Integer): TRect;
var
  R, AR: TRect;
  W, ClW, I, ABottom, ATop: Integer;
begin
  R := GetButtonsRect;
  ABottom := R.Bottom;
  ATop := R.Top;

  W := ItemDefaultWidth;
  ClW:= R.Right - R.Left - Height;

  if FrameCount > 0 then
    W := Min(W, ClW div FrameCount);

  R.Right := R.Left + W;
  R.Top := ATop + 1;
  R.Bottom := ABottom - 1;
  for I := 0 to FrameCount - 1 do
  begin
    AR := R;
    AR.Right := AR.Right - SpaceWidth;
    if I = Index then
    begin
      Result := AR;
      Break;
    end;
    R.Left:= R.Right;
    R.Right := R.Left + W;
  end;

  if Index = FrameCount then
  begin
    //HomeButton
    AR := R;
    AR.Right := AR.Left + (AR.Bottom - AR.Top);
  end;
  Result := AR;
end;

function TCustomFrameBar.GetButtonsRect: TRect;
begin
  Result := ClientRect;
  Dec(Result.Bottom);
  Dec(Result.Right);
  Inc(Result.Left);
  if fbTop in FFrameBarBorders then Inc(Result.Top);
  if fbBottom in FFrameBarBorders then Dec(Result.Bottom);
  if CloseBtn then Dec(Result.Right, Height);
end;

function TCustomFrameBar.GetSysButtonRect: TRect;
begin
  Result := ClientRect;
  Dec(Result.Bottom);
  Dec(Result.Right);
  Inc(Result.Left);
  if fbTop in FFrameBarBorders then Inc(Result.Top);
  if fbBottom in FFrameBarBorders then Dec(Result.Bottom);

  Result.Left := Result.Right - (Result.Bottom - Result.Top);
end;

procedure TCustomFrameBar.DoHomeButtonClick;
begin
  if csDesigning in ComponentState then Exit;

  if (HomeActionLink <> nil) and (HomeActionLink.Action <> nil) then
    HomeActionLink.Execute(Self)
  else
  if Assigned(FOnHomeButtonClick) then FOnHomeButtonClick(Self);
end;

procedure TCustomFrameBar.DoSysButtonClick;
begin
  if csDesigning in ComponentState then Exit;

  if Assigned(FOnSysBtnClick) then FOnSysBtnClick(ActiveFrame);
end;

procedure TCustomFrameBar.HomeActionChange(Sender: TObject);
begin
  if Sender <> nil then
  begin
    if Sender is TAction then
    begin
      HomeIconIndex:= TAction(Sender).ImageIndex;
    end;
  end;
end;

constructor TCustomFrameBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:= 400;
  Height := 20;
  FrameList := TList.Create;
  OrderList := TList.Create;
  FHightLightFont := TFont.Create;
  FHightLightFont.Assign(Font);
  FrameAnderMouse := nil;
  FActiveFrame := nil;
  FCloseBtn := True;
  FSysButton := True;
  CloseBtnWidth := 14;
  SpaceWidth := 5;
  FItemDefaultWidth := 200;
  FMidleLineColor := clGray;
  FWrapText := False;
  FHomeButton := True;
  FHomeButtonState:= bsUp;
  FSysButtonState := bsUp;
  FHomeIconIndex := -1;
  FSysIconIndex := -1;
  TabStop := False;
  FFrameBarBorders := [];
  FNative:= False;
  FFramesParent := nil;
end;

destructor TCustomFrameBar.Destroy;
begin
  Clear;
  FrameList.Free;
  OrderList.Free;
  FreeThenNil(FHomeActionLink);
  inherited Destroy;
end;

procedure TCustomFrameBar.Clear;
var
  I: Integer;
begin
  for I := 0 to FrameList.Count - 1 do GetFrameItem(I).Free;
  FrameList.Clear;
  OrderList.Clear;
  Invalidate;
end;

procedure TCustomFrameBar.AddFrame(AFrame: TFrame; IconIndex: Integer);
var
  I: Integer;
  B: Boolean;
begin
  I := 0;
  B := True;
  while I < FrameList.Count do
  begin
    if Items[I] = AFrame then
    begin
      B := False;
      Break;
    end;
    Inc(I);
  end;
  if B then
  begin
    AddInList(AFrame, IconIndex);
  end;
end;

procedure TCustomFrameBar.InsertFrame(AFrame, AfterFrame: TFrame;
  IconIndex: Integer);
var
  I, FindIndex: Integer;
begin
  I := 0;
  FindIndex := -1;
  while I < FrameCount do
  begin
    if Items[I] = AfterFrame then
    begin
      FindIndex := I + 1;
      Break;
    end;
    Inc(I);
  end;
  if FindIndex >= 0 then InsertInList(AFrame, FindIndex, IconIndex)
  else AddFrame(AFrame, IconIndex);
end;

procedure TCustomFrameBar.SaveFocusedForFrame(AFrame: TFrame;
  AWinControl: TWinControl);
var
  FI: TFrameItem;
begin
  if (AWinControl.Owner = AFrame) or (AWinControl.Parent = AFrame) then
  begin
    FI := GetFrameItemOfFrame(AFrame);
    if FI <> nil then
      FI.FocusedControl := AWinControl;
  end;
end;

procedure TCustomFrameBar.ActivateFrame(AFrame: TFrame);
var
  I: Integer;
  F: TFrame;
  FI: TFrameItem;
  AControl: TWinControl;
begin
  if (ActiveFrame = AFrame) or not AFrame.Enabled then Exit;

  LastActiveFrame := ActiveFrame;
  I := 0;
  while I < FrameCount do
  begin
    F := Items[I];

    if F <> AFrame then
    begin
      if F.Visible then
      begin
        F.Hide;
        Break;
      end;
    end;
    Inc(I);
  end;
  FActiveFrame := AFrame;
  if FFramesParent = nil then
    AFrame.Parent := Self.Parent
  else
    AFrame.Parent := FFramesParent;

  AFrame.Align := alClient;
  AFrame.Show;
  FI := GetFrameItemOfFrame(FActiveFrame);
  if (FI <> nil) and (FI.FocusedControl <> nil) then
  begin
    AControl := FI.FocusedControl;
    AControl.SetFocus;
  end;
  Invalidate;
end;

procedure TCustomFrameBar.RemoveFrame(AFrame: TFrame);
var
  I: Integer;
  FI: TFrameItem;
  LAFrame: TFrame;
begin
  for I := 0 to OrderList.Count - 1 do
  begin
    if TFrame(OrderList.Items[I]) = AFrame then
    begin
      OrderList.Delete(I);
      Break;
    end;
  end;

  for I := 0 to FrameCount - 1 do
  begin
      if Items[I] = AFrame then
      begin
        if FrameAnderMouse = Items[I] then FrameAnderMouse := nil;
        if FActiveFrame = Items[I] then
        begin
          FActiveFrame := nil;
          LAFrame := LastActiveFrame;
          if LAFrame <> nil then ActivateFrame(LAFrame);
        end;
        FI := GetFrameItem(I);
        FI.Free;
        FrameList.Delete(I);
        Invalidate;
        Break;
      end;
  end;

end;}

{TSuFrame}

procedure TSuFrame.SetName(const Value: TComponentName);
begin
  inherited SetName(Value);
  if Owner = nil then
  begin
    FVirtualClassName := 'T' + Name;
  end;
end;

procedure TSuFrame.ValidateRename(AComponent: TComponent; const CurName,
       NewName: string);
begin
  if (Designer <> nil) and (CurName <> '') and (NewName <> '') then
  begin
    Designer.ValidateRename(AComponent, CurName, NewName);
  end;
end;

end.

