unit SQLEditor;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, SynEdit, synhighlightersqlex,
  Forms, Controls, Graphics, Dialogs, ExtCtrls, ComCtrls, StdCtrls, PropEdits,
  LCLType, SynCompletion;

type
  TMetadataType = (mtTable, mtView, mtStoredProc);
  TMetadataTypes = set of TMetadataType;

  { TSQLPropertyEditor }

  TSQLPropertyEditor = class(TStringsPropertyEditor)
  protected
    function AddTable(TableName: string): TTreeNode;
    function AddView(TableName: string): TTreeNode;
    function AddProcedure(ProcName: string): TTreeNode;
    procedure AddField(Node: TTreeNode; FieldName: string; PrimaryKey: Boolean = False);
    procedure AddProcParam(Node: TTreeNode; ParamName: string; InParam: Boolean);
    procedure LoadMetadata(MetaTypes: TMetadataTypes); virtual;
    procedure ClearNameList;
    procedure GetFieldFromSelected(SList: TStrings; WithPKey: Boolean = True);
    procedure GetPKeyFromSelected(SList: TStrings);
    procedure GetInParamFromSelected(SList: TStrings);
    procedure GetOutParamFromSelected(SList: TStrings);
    procedure Gen_SelectAll; virtual;
    procedure Gen_Select; virtual;
    procedure Gen_Insert; virtual;
    procedure Gen_Update; virtual;
    procedure Gen_Delete; virtual;
    procedure Gen_Execute; virtual;
    function DefaultMetaTypes: TMetadataTypes; virtual;
  public
    UseReturningField: Boolean;
    SqlDialect: TSQLDialect;
    HideRadioSqlType: Boolean;
    DefaultIndex: Integer;


    constructor Create(Hook: TPropertyEditorHook; APropCount: Integer);
       override;
    procedure Edit; override;
  end;

  TSQLCompletionItemType = (scitSQLKeyWord, scitTable, scitView,
    scitField, scitStoreProc, scitVariable, scitRetParam, scitInputParam, scitSQLType, scitUnknown);

  { TSQLCompletionItem }

  TSQLCompletionItem = class
  private
  protected
  public
    ItemType: TSQLCompletionItemType;
    ItemParam: string;
    destructor Destroy; override;
  end;

  { TSQLEditForm }

  TSQLEditForm = class(TForm)
    Button1: TButton;
    Button2: TButton;
    Button3: TButton;
    ImageList1: TImageList;
    Panel1: TPanel;
    Panel2: TPanel;
    Panel3: TPanel;
    NameList: TTreeView;
    RadioGroup1: TRadioGroup;
    Splitter1: TSplitter;
    SynEdit1: TSynEdit;
    procedure Button1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NameListKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState
      );
    procedure RadioGroup1Click(Sender: TObject);
  private
    FSQLPropEdit: TSQLPropertyEditor;
    SQLHigtLighter: TSynSQLSynEx;
    CompletionList: TStringList;
    Completion: TSynCompletion;
    procedure OnCompletion(var Value: string;
                                   SourceValue: string;
                                   var SourceStart, SourceEnd: TPoint;
                                   KeyChar: TUTF8Char;
                                   Shift: TShiftState);
    procedure ExecuteCompletion(Sender: TObject);
    procedure ClearCompletion;
    procedure CompletionSearchPosition(var Posit: integer);
    function PaintCompletionItem(const AKey: string; ACanvas: TCanvas;
      X, Y: integer; Selected: boolean; Index: integer): boolean;
    procedure OnPrevChar(Sender: TObject);
    procedure OnNextChar(Sender: TObject);
    procedure ComplUTF8KeyPress(Sender: TObject; var UTF8Key: TUTF8Char);
    procedure AddCompletionItem(AType: TSQLCompletionItemType; AName, AParam: string);
    procedure SetSQLPropEdit(AValue: TSQLPropertyEditor);
    function GetNodeOfName(AName: string): TTreeNode;
  public
    property SQLPropEdit: TSQLPropertyEditor read FSQLPropEdit write SetSQLPropEdit;
  end;

var
  SQLEditForm: TSQLEditForm;

implementation

uses typinfo, Editor, SQLParser, SynEditTypes;

{ TSQLCompletionItem }

destructor TSQLCompletionItem.Destroy;
begin
  ItemParam := '';
  inherited Destroy;
end;

{ TSQLEditForm }

procedure TSQLEditForm.Button1Click(Sender: TObject);
begin
  if NameList.Selected = nil then Exit;
  if NameList.Selected.Level > 0 then
  begin
    SynEdit1.InsertTextAtCaret(NameList.Selected.Text);
  end
  else
  case RadioGroup1.ItemIndex of
    0: SQLPropEdit.Gen_SelectAll;
    1: SQLPropEdit.Gen_Select;
    2: SQLPropEdit.Gen_Insert;
    3: SQLPropEdit.Gen_Update;
    4: SQLPropEdit.Gen_Delete;
    5: SQLPropEdit.Gen_Execute;
  end;
  SynEdit1.SetFocus;
end;

procedure TSQLEditForm.FormCreate(Sender: TObject);
begin
  SynEdit1.Font.Name := EditorDefaultFontName;
  SynEdit1.Font.Size := EditorDefaultFontSize;
  SynEdit1.Font.Quality:= fqDefault;
  SynEdit1.TabWidth := 2;
  SynEdit1.Options := SynEdit1.Options - [eoSmartTabs];
  SynEdit1.Options := SynEdit1.Options + [eoTabIndent];

  SQLHigtLighter := TSynSQLSynEx.Create(Self);
  SQLHigtLighter.KeywordAttribute.Assign(EditorForm.SynDPasSyn1.KeywordAttribute);
  SQLHigtLighter.CommentAttri.Assign(EditorForm.SynDPasSyn1.CommentAttri);
  SQLHigtLighter.StringAttri.Assign(EditorForm.SynDPasSyn1.StringAttri);
  SQLHigtLighter.SymbolAttri.Assign(EditorForm.SynDPasSyn1.SymbolAttri);
  SynEdit1.Highlighter := SQLHigtLighter;
  CompletionList := TStringList.Create;
  Completion := TSynCompletion.Create(Self);
  Completion.Editor := SynEdit1;
  Completion.LongLineHintTime := 0;
  Completion.LongLineHintType := sclpExtendRightOnly;
  Completion.SelectedColor := 10841427;
  Completion.TheForm.Font.Assign(SynEdit1.Font);
  Completion.Width := 500;
  Completion.OnExecute := @ExecuteCompletion;
  Completion.OnSearchPosition:=@CompletionSearchPosition;
  Completion.OnPaintItem := @PaintCompletionItem;
  Completion.OnCodeCompletion := @OnCompletion;
  Completion.OnKeyNextChar := @OnNextChar;
  Completion.OnKeyPrevChar := @OnPrevChar;
  Completion.OnUTF8KeyPress:= @ComplUTF8KeyPress;
end;

procedure TSQLEditForm.FormDestroy(Sender: TObject);
begin
  CompletionList.Free;
end;

procedure TSQLEditForm.NameListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_RETURN then Button1.Click;
end;

procedure TSQLEditForm.RadioGroup1Click(Sender: TObject);
begin
  NameList.Items.Clear;
  if RadioGroup1.ItemIndex in [0, 1] then
    SQLPropEdit.LoadMetadata([mtTable, mtView, mtStoredProc])
  else
  if RadioGroup1.ItemIndex in [2, 3, 4] then
    SQLPropEdit.LoadMetadata([mtTable])
  else
  if RadioGroup1.ItemIndex = 5 then
    SQLPropEdit.LoadMetadata([mtStoredProc]);
  NameList.ScrollBars := ssNone;
  NameList.ScrollBars := ssVertical;
end;

procedure TSQLEditForm.OnCompletion(var Value: string; SourceValue: string;
  var SourceStart, SourceEnd: TPoint; KeyChar: TUTF8Char; Shift: TShiftState);
var
  CI: TSQLCompletionItem;
  S: string;
  P, P2: TPoint;
begin
  CI := TSQLCompletionItem(Completion.ItemList.Objects[Completion.Position]);
  if CI.ItemType in [scitStoreProc] then
  begin
    if CI.ItemParam <> '' then
    begin
      P := SourceStart;
      Dec(P.x);
      P2 := SourceStart;
      S := SynEdit1.TextBetweenPoints[P, P2];
      if S <> '@' then
      begin
        P2 := SourceEnd;
        Inc(P2.x);
        P := SourceEnd;
        S := SynEdit1.TextBetweenPoints[P, P2];
        if S <> '(' then
        begin
          S := '(' + CI.ItemParam + ')';
          Value := Value + S;
        end;
      end;
    end;
  end
  else
  if (KeyChar = '.') or (KeyChar = ';') or (KeyChar = '(') or (KeyChar = ')') then
  begin
    Value := Value + KeyChar;
  end;
end;

procedure TSQLEditForm.ExecuteCompletion(Sender: TObject);
var
  RelationList: TStringList;
  VarList: TStringList;
  ReturnsList: TStringList;
  InputList: TStringList;

  procedure LoadRelationField(ARelation: string);
  var
    Node: TTreeNode;
  begin
    Node := GetNodeOfName(ARelation);
    if Node <> nil then
    begin
      Node := Node.GetFirstChild;
      while Node <> nil do
      begin
        AddCompletionItem(scitField, Node.Text, '');
        Node := Node.GetNextSibling;
      end;
    end;
  end;

  procedure LoadAllRelationField;
  var
    I: Integer;
  begin
    if RelationList.Count > 0 then
    begin
      for I := 0 to RelationList.Count - 1 do
        LoadRelationField(RelationList.Strings[I]);
    end;
    if VarList.Count > 0 then
    begin
      for I := 0 to VarList.Count - 1 do
        AddCompletionItem(scitVariable, VarList.Strings[I], '');
    end;
    if ReturnsList.Count > 0 then
    begin
      for I := 0 to ReturnsList.Count - 1 do
        AddCompletionItem(scitRetParam, ReturnsList.Strings[I], '');
    end;
    if InputList.Count > 0 then
    begin
      for I := 0 to InputList.Count - 1 do
        AddCompletionItem(scitInputParam, InputList.Strings[I], '');
    end;
  end;

  procedure LoadDefaultCompletion;
  var
    CompType: TSQLCompletionItemType;
    I: Integer;
    Node, ParamNode: TTreeNode;
    S, Param: string;
  begin
    CompType := scitSQLKeyWord;
    for I := 0 to SQLHigtLighter.KeyList.Count - 1 do
      AddCompletionItem(CompType, SQLHigtLighter.KeyList.Strings[I], '');
    Node := NameList.Items.GetFirstNode;
    while Node <> nil do
    begin
      Param := '';
      if Node.ImageIndex = 0 then CompType := scitTable
      else
      if Node.ImageIndex = 1 then CompType := scitView
      else
      if Node.ImageIndex = 3 then
      begin
        CompType := scitStoreProc;
        ParamNode := Node.GetFirstChild;
        while ParamNode <> nil do
        begin
          if ParamNode.ImageIndex = 4 then
          begin
            if Param <> '' then Param := Param + ', ';
            Param := Param + ':' + ParamNode.Text;
          end;
          ParamNode := ParamNode.GetNextSibling;
        end;
      end
      else CompType :=  scitUnknown;

      S := Node.Text;
      if Param <> '' then S := S + '(' + Param + ')';

      AddCompletionItem(CompType, S, '');
      //AddCompletionItem(CompType, UpperCase(S), '');
      Node := Node.GetNextSibling;
    end;
    if VarList.Count > 0 then
    begin
      for I := 0 to VarList.Count - 1 do
        AddCompletionItem(scitVariable, VarList.Strings[I], '');
    end;
    if ReturnsList.Count > 0 then
    begin
      for I := 0 to ReturnsList.Count - 1 do
        AddCompletionItem(scitRetParam, ReturnsList.Strings[I], '');
    end;
    if InputList.Count > 0 then
    begin
      for I := 0 to InputList.Count - 1 do
        AddCompletionItem(scitInputParam, InputList.Strings[I], '');
    end;
  end;



var
  CurStr, AltCurStr, RelName, AliasName, RelationName: string;
  Parser: TSQLParser;
  Tokens: TSQLTokens;
  R, CaretX, StartX, EndX: Integer;
begin
  ClearCompletion;
  Parser := TSQLParser.Create;
  Tokens := TSQLTokens.Create;
  RelationList := TStringList.Create;
  VarList := TStringList.Create;
  ReturnsList := TStringList.Create;
  InputList := TStringList.Create;
  try
    Parser.SetParseStr(SynEdit1.Text);
    Parser.GetTokensList(Tokens);
    Tokens.First;

    while Tokens.Token^.TokenID <> IDS_EOF do
    begin
      R := Tokens.Token^.Row + 1;
      if R = SynEdit1.CaretY then
      begin
        StartX:= Tokens.Token^.Col + 1;
        EndX := Tokens.Token^.EndCol;
        CaretX := SynEdit1.CaretX - 1;
        if (StartX <= CaretX) and (EndX >= CaretX) then
        begin
          if Tokens.Token^.TokenID = IDS_Colon then
          begin
            AltCurStr := Copy(Tokens.Token^.TokenName, 1, CaretX - StartX + 1);
            Break;
          end
          else
          if Tokens.Token^.TokenID = IDS_Identifier then
          begin
            AltCurStr := Copy(Tokens.Token^.TokenName, 1, CaretX - StartX + 1);
            Tokens.PrevWithSkipComment;
          end;
          if Tokens.Token^.TokenID = IDS_Period then
          begin
            Tokens.PrevWithSkipComment;
            if Tokens.Token^.TokenID = IDS_Identifier then
            begin
              CurStr := Tokens.Token^.TokenName;
              Break;
            end;
          end
          else
            Break;
        end;
      end;

      Tokens.NextWithSkipComment;
    end;

    //if CurStr <> '' then
    //begin
      Tokens.First;
      RelationName := '';
      while Tokens.Token^.TokenID <> IDS_EOF do
      begin
        if (Tokens.Token^.TokenID = IDS_UPDATE)
          or (Tokens.Token^.TokenID = IDS_FROM)
          or (Tokens.Token^.TokenID = IDS_LEFT)
          or (Tokens.Token^.TokenID = IDS_RIGHT)
          or (Tokens.Token^.TokenID = IDS_FULL) then
        begin
          if (Tokens.Token^.TokenID = IDS_LEFT)
            or (Tokens.Token^.TokenID = IDS_RIGHT)
            or (Tokens.Token^.TokenID = IDS_FULL) then
          begin
            Tokens.NextWithSkipComment;
            if (Tokens.Token^.TokenID = IDS_OUTER)
              or (Tokens.Token^.TokenID = IDS_INNER) then Tokens.NextWithSkipComment;
            if Tokens.Token^.TokenID = IDS_JOIN then Tokens.NextWithSkipComment;
          end
          else
            Tokens.NextWithSkipComment;

          while Tokens.Token^.TokenID = IDS_Identifier do
          begin
            RelName := Tokens.Token^.TokenName;
            RelationList.Append(RelName);
            if (CurStr > '') and SameText(RelName, CurStr) then
            begin
              RelationName := RelName;
              Break;
            end;
            Tokens.NextWithSkipComment;
            if Tokens.Token^.TokenID = IDS_Identifier then
            begin
              AliasName := Tokens.Token^.TokenName;
              if (CurStr <> '') and SameText(AliasName, CurStr) then
              begin
                RelationName := RelName;
                Break;
              end;
              Tokens.NextWithSkipComment;
            end;
            if Tokens.Token^.TokenID = IDS_Comma then Tokens.NextWithSkipComment;
          end;
          Tokens.PrevWithSkipComment;
        end
        else
        if Tokens.Token^.TokenID = IDS_DECLARE then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = IDS_VARIABLE then
          begin
            Tokens.NextWithSkipComment;
            if Tokens.Token^.TokenID = IDS_Identifier then
            begin
              VarList.Add(Tokens.Token^.TokenName);
              Tokens.NextWithSkipComment;
            end;
          end;
        end
        else
        if Tokens.Token^.TokenID = IDS_EXECUTE then
        begin
          if Tokens.PreviewNext^.TokenID = IDS_BLOCK then
          begin
            Tokens.Next;
            Tokens.Next;
            if Tokens.Token^.TokenID = IDS_OpenRound then
            begin
              Tokens.NextWithSkipComment;
              while Tokens.Token^.TokenID <> IDS_EOF do
              begin
                if Tokens.Token^.TokenID = IDS_CloseRound then Break;
                if Tokens.Token^.TokenID = IDS_Identifier then
                begin
                  InputList.Add(Tokens.Token^.TokenName);
                  Tokens.NextWithSkipComment;
                  Tokens.NextWithSkipComment;
                  if (Tokens.Token^.TokenID = IDS_PRECISION)
                      or (Tokens.Token^.TokenID = IDS_VARYING) then
                          Tokens.NextWithSkipComment;
                  if Tokens.Token^.TokenID = IDS_OpenRound then
                  begin
                    Tokens.NextWithSkipComment;
                    while Tokens.Token^.TokenID <> IDS_EOF do
                    begin
                      if Tokens.Token^.TokenID = IDS_CloseRound then
                      begin
                        Tokens.NextWithSkipComment;
                        Break;
                      end;
                      Tokens.NextWithSkipComment;
                    end;
                  end;
                  if Tokens.Token^.TokenID = IDS_Assignment then
                  begin
                    Tokens.NextWithSkipComment;
                    if Tokens.Token^.TokenID = IDS_QuestionMark then
                      Tokens.NextWithSkipComment
                    else
                    if Tokens.Token^.TokenID = IDS_SemiColon then
                    begin
                      Tokens.NextWithSkipComment;
                      if Tokens.Token^.TokenID = IDS_Identifier then
                        Tokens.NextWithSkipComment;
                    end
                  end
                  else
                    Break;
                end
                else Break;
                if Tokens.Token^.TokenID = IDS_Comma then
                  Tokens.NextWithSkipComment
                else Break;
              end;
            end;
          end;
        end
        else
        if Tokens.Token^.TokenID = IDS_RETURNS then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = IDS_OpenRound then
          begin
            Tokens.NextWithSkipComment;
            while Tokens.Token^.TokenID <> IDS_EOF do
            begin
              if Tokens.Token^.TokenID = IDS_CloseRound then Break;
              if Tokens.Token^.TokenID = IDS_Identifier then
              begin
                ReturnsList.Add(Tokens.Token^.TokenName);
                Tokens.NextWithSkipComment;
                Tokens.NextWithSkipComment;
                if (Tokens.Token^.TokenID = IDS_PRECISION)
                    or (Tokens.Token^.TokenID = IDS_VARYING) then
                        Tokens.NextWithSkipComment;
                if Tokens.Token^.TokenID = IDS_OpenRound then
                begin
                  Tokens.NextWithSkipComment;
                  while Tokens.Token^.TokenID <> IDS_EOF do
                  begin
                    if Tokens.Token^.TokenID = IDS_CloseRound then
                    begin
                      Tokens.NextWithSkipComment;
                      Break;
                    end;
                    Tokens.NextWithSkipComment;
                  end;
                end;
              end
              else Break;
              if Tokens.Token^.TokenID = IDS_Comma then
                Tokens.NextWithSkipComment
              else Break;
            end;
          end;
        end;
        Tokens.NextWithSkipComment;
        if RelationName <> '' then Break;
      end;
    //end;

    if RelationName <> '' then
      LoadRelationField(RelationName)
    else
    if AltCurStr = ':' then
      LoadAllRelationField
    else
      LoadDefaultCompletion;

  finally
    Parser.Free;
    Tokens.Free;
    RelationList.Free;
    VarList.Free;
    ReturnsList.Free;
    InputList.Free;
  end;

end;

procedure TSQLEditForm.ClearCompletion;
var
  I: Integer;
begin
  Completion.ItemList.Clear;
  for I := 0 to CompletionList.Count - 1 do
  begin
    CompletionList.Objects[I].Free;
    CompletionList.Objects[I] := nil;
  end;
  CompletionList.Clear;
end;

procedure TSQLEditForm.CompletionSearchPosition(var Posit: integer);
var
  I: Integer;

  procedure Add(AIndex: Integer);
  var
    S, CS: string;
    CI: TSQLCompletionItem;
    //N: Integer;
  begin
    S := CompletionList.Strings[AIndex];
    CS := Completion.CurrentString;
    if (CS <> '') then
    begin
      if CS[1] = ':' then Delete(CS, 1, 1);
    end;
    if (CS = '') or
      (pos(lowercase(CS), lowercase(s)) = 1) then
    begin
      CI := TSQLCompletionItem(CompletionList.Objects[AIndex]);
      Completion.ItemList.AddObject(S, CI);
    end;
  end;

begin
  Completion.ItemList.Clear;
  for I := 0 to CompletionList.Count - 1 do
  begin
    Add(I);
  end;
  if Completion.ItemList.Count > 0 then
    Posit := 0
  else
    Posit := -1;
end;

function TSQLEditForm.PaintCompletionItem(const AKey: string; ACanvas: TCanvas;
  X, Y: integer; Selected: boolean; Index: integer): boolean;
var
  S: string;
  CI: TSQLCompletionItem;
  L: Integer;
begin
  CI := TSQLCompletionItem(Completion.ItemList.Objects[Index]);
  S := '';

  if Selected then
    ACanvas.Font.Color := clWhite
  else
    ACanvas.Font.Color := clNavy;

  case CI.ItemType of
    scitSQLKeyWord:
    begin
      S := 'SQL    ';
      if Selected then
        ACanvas.Font.Color := clYellow
      else
        ACanvas.Font.Color := clBlue;
    end;
    scitTable:
    begin
      S := 'Table  ';
      if Selected then
        ACanvas.Font.Color := clMaroon
      else
        ACanvas.Font.Color := clPurple;
    end;
    scitView:
    begin
      S := 'View   ';
      if Selected then
        ACanvas.Font.Color := clFuchsia
      else
        ACanvas.Font.Color := clPurple;
    end;
    scitField:
    begin
      S := 'Field  ';
      if Selected then
        ACanvas.Font.Color := clOlive
      else
        ACanvas.Font.Color := clMaroon;
    end;
    scitStoreProc:
    begin
      S := 'Proc   ';
      if Selected then
        ACanvas.Font.Color := clBlue
      else
        ACanvas.Font.Color := clNavy;
    end;
    scitVariable:
    begin
      S := 'Var    ';
      if Selected then
        ACanvas.Font.Color := clBlue
      else
        ACanvas.Font.Color := clNavy;
    end;
    scitRetParam:
    begin
      S := 'Return ';
      if Selected then
        ACanvas.Font.Color := clBlue
      else
        ACanvas.Font.Color := clNavy;
    end;
    scitInputParam:
    begin
      S := 'Input  ';
      if Selected then
        ACanvas.Font.Color := clBlue
      else
        ACanvas.Font.Color := clNavy;
    end;
    scitSQLType:
    begin
      S := 'Type   ';
      if Selected then
        ACanvas.Font.Color := clLime
      else
        ACanvas.Font.Color := clGreen;
    end;
    scitUnknown:    S := '?????  ';
  end;
  ACanvas.Font.Bold := False;
  ACanvas.TextOut(X, Y, S);
  L := ACanvas.TextWidth(S);
  Inc(X, L);

  S := AKey;
  if Selected then
    ACanvas.Font.Color := clWhite
  else
    ACanvas.Font.Color := clBlack;
  ACanvas.Font.Bold := True;
  ACanvas.TextOut(X, Y, S);
  L := ACanvas.TextWidth(S);
  Inc(X, L);

  if CI.ItemParam <> '' then
  begin
    ACanvas.Font.Bold := False;
    S := '(' + CI.ItemParam + ')';
    ACanvas.TextOut(X, Y, S);
    L := ACanvas.TextWidth(S);
    Inc(X, L);
  end;
  Result := True;
end;

procedure TSQLEditForm.OnPrevChar(Sender: TObject);
var
  XY: TPoint;
  StartX: integer;
  EndX: integer;
  Line: string;
  I: Integer;
begin
  XY := SynEdit1.LogicalCaretXY;
  if XY.Y > SynEdit1.Lines.Count then exit;
  SynEdit1.GetWordBoundsAtRowCol(XY, StartX, EndX);
  Line := SynEdit1.GetWordAtRowCol(XY);
  Dec(XY.X, 1);
  if (EndX < XY.X) then exit;
  if XY.X = StartX then Line := ''
  else
  begin
    I := XY.X - StartX;
    Line:= Copy(Line, 1, I);
  end;
  SynEdit1.LogicalCaretXY := XY;
  Completion.CurrentString := Line;
end;

procedure TSQLEditForm.OnNextChar(Sender: TObject);
var
  XY: TPoint;
  StartX: integer;
  EndX: integer;
  I: Integer;
  Line: string;
begin
  XY := SynEdit1.LogicalCaretXY;
  if XY.Y > SynEdit1.Lines.Count then exit;
  SynEdit1.GetWordBoundsAtRowCol(XY, StartX, EndX);
  Line := SynEdit1.GetWordAtRowCol(XY);
  Inc(XY.X, 1);
  if (EndX < XY.X) then exit;
  if XY.X = StartX then Line := ''
  else
  begin
    I := XY.X - StartX;
    Line:= Copy(Line, 1, I);
  end;
  SynEdit1.LogicalCaretXY := XY;
  Completion.CurrentString := Line;
end;

procedure TSQLEditForm.ComplUTF8KeyPress(Sender: TObject; var UTF8Key: TUTF8Char
  );
var
  p: TPoint;
begin
  if UTF8Key = '.' then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit1.InsertTextAtCaret(UTF8Key);
    UTF8Key:= '';
    Application.ProcessMessages;
    p := SynEdit1.ClientToScreen(Point(SynEdit1.CaretXPix, SynEdit1.CaretYPix + SynEdit1.LineHeight + 1));
    Completion.Execute('', P.x, P.y);
  end
  else
  if (UTF8Key = '=') or (UTF8Key = ':') or (UTF8Key = '+')
    or (UTF8Key = '-') or (UTF8Key = '*') or (UTF8Key = '/') then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit1.InsertTextAtCaret(' ' + UTF8Key + ' ');
    UTF8Key:= '';
  end
  else
  if (UTF8Key = '(') or (UTF8Key = ')') or (UTF8Key = '[') or (UTF8Key = ']') then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit1.InsertTextAtCaret(UTF8Key);
    UTF8Key:= '';
  end;
end;

procedure TSQLEditForm.AddCompletionItem(AType: TSQLCompletionItemType; AName,
  AParam: string);
var
  CI: TSQLCompletionItem;
  S: string;
begin
  S := Trim(Completion.CurrentString);
  if (S = '') or (S = ':') or
    (pos(lowercase(S), lowercase(AName)) = 1) then
  begin
    CI := TSQLCompletionItem.Create;
    CI.ItemType := AType;
    CI.ItemParam := AParam;
    CompletionList.AddObject(AName, CI);
    Completion.ItemList.AddObject(AName, CI);
  end;
end;

procedure TSQLEditForm.SetSQLPropEdit(AValue: TSQLPropertyEditor);
begin
  if FSQLPropEdit = AValue then Exit;
  FSQLPropEdit := AValue;
  SQLHigtLighter.SQLDialect := FSQLPropEdit.SqlDialect;
end;

function TSQLEditForm.GetNodeOfName(AName: string): TTreeNode;
var
  ANode: TTreeNode;
begin
  Result := nil;
  ANode := NameList.Items.GetFirstNode;
  while ANode <> nil do
  begin
    if SameText(ANode.Text, AName) then
    begin
      Result := ANode;
      Break;
    end;
    ANode := ANode.GetNextSibling;
  end;
end;

{$R *.lfm}

{ TSQLPropertyEditor }

function TSQLPropertyEditor.AddTable(TableName: string): TTreeNode;
begin
  Result := SQLEditForm.NameList.Items.Add(nil, TableName);
  Result.ImageIndex := 0;
  Result.SelectedIndex := 0;
end;

function TSQLPropertyEditor.AddView(TableName: string): TTreeNode;
begin
  Result := SQLEditForm.NameList.Items.Add(nil, TableName);
  Result.ImageIndex := 1;
  Result.SelectedIndex := 1;
end;

function TSQLPropertyEditor.AddProcedure(ProcName: string): TTreeNode;
begin
  Result := SQLEditForm.NameList.Items.Add(nil, ProcName);
  Result.ImageIndex := 3;
  Result.SelectedIndex := 3;
end;

procedure TSQLPropertyEditor.AddField(Node: TTreeNode; FieldName: string;
  PrimaryKey: Boolean);
var
  FieldNode: TTreeNode;
begin
  FieldNode := SQLEditForm.NameList.Items.AddChild(Node, FieldName);
  if PrimaryKey then
  begin
    FieldNode.ImageIndex := 6;
    FieldNode.SelectedIndex := 6;
  end
  else
  begin
    FieldNode.ImageIndex := 2;
    FieldNode.SelectedIndex := 2;
  end;
end;

procedure TSQLPropertyEditor.AddProcParam(Node: TTreeNode; ParamName: string;
  InParam: Boolean);
var
  ParamNode: TTreeNode;
begin
  ParamNode := SQLEditForm.NameList.Items.AddChild(Node, ParamName);
  if InParam then
  begin
    ParamNode.ImageIndex := 4;
    ParamNode.SelectedIndex := 4;
  end
  else
  begin
    ParamNode.ImageIndex := 5;
    ParamNode.SelectedIndex := 5;
  end;
end;

procedure TSQLPropertyEditor.LoadMetadata(MetaTypes: TMetadataTypes);
begin

end;

procedure TSQLPropertyEditor.ClearNameList;
begin
  SQLEditForm.NameList.Items.Clear;
end;

procedure TSQLPropertyEditor.GetFieldFromSelected(SList: TStrings;
  WithPKey: Boolean);
var
  Node: TTreeNode;
  B: Boolean;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  Node := SQLEditForm.NameList.Selected.GetFirstChild;
  while Node <> nil do
  begin
    B := True;
    if not WithPKey then
    begin
      if Node.ImageIndex = 6 then B := False;
    end;
    if B then SList.Add(Node.Text);
    Node := Node.GetNextSibling;
  end;
end;

procedure TSQLPropertyEditor.GetPKeyFromSelected(SList: TStrings);
var
  Node: TTreeNode;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  Node := SQLEditForm.NameList.Selected.GetFirstChild;
  while Node <> nil do
  begin
    if Node.ImageIndex = 6 then
       SList.Add(Node.Text);
    Node := Node.GetNextSibling;
  end;
end;

procedure TSQLPropertyEditor.GetInParamFromSelected(SList: TStrings);
var
  Node: TTreeNode;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  Node := SQLEditForm.NameList.Selected.GetFirstChild;
  while Node <> nil do
  begin
    if Node.ImageIndex = 4 then
      SList.Add(Node.Text)
    else Break;
    Node := Node.GetNextSibling;
  end;
end;

procedure TSQLPropertyEditor.GetOutParamFromSelected(SList: TStrings);
var
  Node: TTreeNode;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  Node := SQLEditForm.NameList.Selected.GetFirstChild;
  while Node <> nil do
  begin
    if Node.ImageIndex = 5 then
      SList.Add(Node.Text);
    Node := Node.GetNextSibling;
  end;
end;

procedure TSQLPropertyEditor.Gen_SelectAll;
var
  AMetaType: TMetadataType;
  SList: TStringList;
  I, Cnt: Integer;
  S: string;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  SQLEditForm.SynEdit1.Text := '';
  AMetaType := mtTable;
  if SQLEditForm.NameList.Selected.ImageIndex = 3 then
    AMetaType:= mtStoredProc;

  SQLEditForm.SynEdit1.Lines.Add('SELECT *');
  SQLEditForm.SynEdit1.Lines.Add('FROM ' + SQLEditForm.NameList.Selected.Text);
  if AMetaType = mtStoredProc then
  begin
    SList := TStringList.Create;
    try
      GetInParamFromSelected(SList);
      if SList.Count > 0 then
      begin
        SQLEditForm.SynEdit1.Lines.Add('  (');
        Cnt := SList.Count - 1;
        for I := 0 to Cnt do
        begin
          S := '  :' + SList.Strings[I];
          if I < Cnt then S := S + ',';
          SQLEditForm.SynEdit1.Lines.Add(S);
        end;
        SQLEditForm.SynEdit1.Lines.Add('  )');
      end;
    finally
      SList.Free;
    end;
  end
  else
    if SQLEditForm.Panel3.Caption = 'RefreshSQL' then
    begin
      SList := TStringList.Create;
      try
        GetPKeyFromSelected(SList);
        if SList.Count > 0 then
        begin
          SQLEditForm.SynEdit1.Lines.Add('WHERE');
          SQLEditForm.SynEdit1.Lines.Add('  ' + SList.Strings[0] + ' = :' + SList.Strings[0]);
        end;

      finally
        SList.Free;
      end;
    end;
end;

procedure TSQLPropertyEditor.Gen_Select;
var
  AMetaType: TMetadataType;
  SList: TStringList;
  I, Cnt: Integer;
  S, AText: string;
  P: TPoint;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  AMetaType := mtTable;
  if SQLEditForm.NameList.Selected.ImageIndex = 3 then
    AMetaType:= mtStoredProc;

  SList := TStringList.Create;
  try
    SQLEditForm.SynEdit1.ClearAll;
    AText:= 'SELECT';
    if AMetaType = mtTable then
      GetFieldFromSelected(SList)
    else
      GetOutParamFromSelected(SList);

    if SList.Count > 0 then
    begin
      Cnt := SList.Count - 1;
      for I := 0 to Cnt do
      begin
        S := '  ' + SList.Strings[I];
        if I < Cnt then S := S + ',';
        AText := AText + #13#10 + S;
      end;
    end;
    AText:= AText + #13#10 + 'FROM ' + SQLEditForm.NameList.Selected.Text;
    if AMetaType = mtStoredProc then
    begin
      SList.Clear;
      GetInParamFromSelected(SList);
      if SList.Count > 0 then
      begin
        AText := AText + #13#10 + '  (';
        Cnt := SList.Count - 1;
        for I := 0 to Cnt do
        begin
          S := '  :' + SList.Strings[I];
          if I < Cnt then S := S + ',';
          AText := AText + #13#10 + S;
        end;
        AText:= AText + #13#10 + '  )';
      end;
    end
    else
    if SQLEditForm.Panel3.Caption = 'RefreshSQL' then
    begin
      SList.Clear;
      GetPKeyFromSelected(SList);
      if SList.Count > 0 then
        AText:= AText + #13#10 + 'WHERE ' + #13#10 + '  ' + SList.Strings[0] + ' = :' + SList.Strings[0];
    end;
    P := Point(1, 1);
    SQLEditForm.SynEdit1.TextBetweenPoints[P, P] :=  AText;
  finally
    SList.Free;
  end;

end;

procedure TSQLPropertyEditor.Gen_Insert;
var
  SList: TStringList;
  I, Cnt: Integer;
  S, Sep: string;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  SQLEditForm.SynEdit1.Lines.Text := '';
  SList := TStringList.Create;
  try
    GetFieldFromSelected(SList, not UseReturningField);
    Cnt := SList.Count - 1;
    SQLEditForm.SynEdit1.Lines.Add('INSERT INTO ' + SQLEditForm.NameList.Selected.Text);
    SQLEditForm.SynEdit1.Lines.Add('  (');
    for I := 0 to Cnt do
    begin
      S := '  ' + SList.Strings[I];
      if I < Cnt then S := S + ',';
      SQLEditForm.SynEdit1.Lines.Add(S);
    end;
    SQLEditForm.SynEdit1.Lines.Add('  )');
    SQLEditForm.SynEdit1.Lines.Add('VALUES');
    SQLEditForm.SynEdit1.Lines.Add('  (');
    for I := 0 to Cnt do
    begin
      S := '  :' + SList.Strings[I];
      if I < Cnt then S := S + ',';
      SQLEditForm.SynEdit1.Lines.Add(S);
    end;
    SQLEditForm.SynEdit1.Lines.Add('  )');
    if UseReturningField then
    begin
      SList.Clear;
      GetPKeyFromSelected(SList);
      if SList.Count > 0 then
      begin
        S := 'RETURNING';
        Sep := ' ';
        Cnt := SList.Count - 1;
        for I := 0 to Cnt do
        begin
          S := S + Sep + SList.Strings[I];
          Sep := ', ';
        end;
        SQLEditForm.SynEdit1.Lines.Add(S);
      end;
    end;
  finally
    SList.Free;
  end;
end;

procedure TSQLPropertyEditor.Gen_Update;
var
  SList: TStringList;
  I, Cnt: Integer;
  S: string;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  SQLEditForm.SynEdit1.Text := '';
  SList := TStringList.Create;
  try
    GetFieldFromSelected(SList, False);
    Cnt := SList.Count - 1;
    SQLEditForm.SynEdit1.Lines.Add('UPDATE ' + SQLEditForm.NameList.Selected.Text  + ' SET');
    for I := 0 to Cnt do
    begin
      S := '  ' + SList.Strings[I] + ' = :' + SList.Strings[I];
      if I < Cnt then S := S + ',';
      SQLEditForm.SynEdit1.Lines.Add(S);
    end;
    SList.Clear;
    GetPKeyFromSelected(SList);
    if SList.Count > 0 then
    begin
      SQLEditForm.SynEdit1.Lines.Add('WHERE');
      Cnt := SList.Count - 1;
      for I := 0 to Cnt do
      begin
        S := '  ' + SList.Strings[I] + ' = :' + SList.Strings[I];
        if I < Cnt then S := S + ',';
        SQLEditForm.SynEdit1.Lines.Add(S);
      end;
    end;
  finally
    SList.Free;
  end;
end;

procedure TSQLPropertyEditor.Gen_Delete;
var
  SList: TStringList;
  I, Cnt: Integer;
  S: string;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  SQLEditForm.SynEdit1.Text:= '';
  SQLEditForm.SynEdit1.Lines.Append('DELETE FROM ' + SQLEditForm.NameList.Selected.Text);
  SList := TStringList.Create;
  try
     GetPKeyFromSelected(SList);
     if SList.Count > 0 then
     begin
       SQLEditForm.SynEdit1.Lines.Add('WHERE');
       Cnt := SList.Count - 1;
       for I := 0 to Cnt do
       begin
         S := '  ' + SList.Strings[I] + ' = :' + SList.Strings[I];
         if I < Cnt then S := S + ',';
         SQLEditForm.SynEdit1.Lines.Add(S);
       end;
     end;
  finally
    SList.Free;
  end;
end;

procedure TSQLPropertyEditor.Gen_Execute;
var
  SList: TStringList;
  I, Cnt: Integer;
  S: string;
begin
  if SQLEditForm.NameList.Selected = nil then Exit;
  SQLEditForm.SynEdit1.Text := '';
  SList := TStringList.Create;
  try
    GetInParamFromSelected(SList);
    SQLEditForm.SynEdit1.Lines.Append('EXECUTE PROCEDURE ' + SQLEditForm.NameList.Selected.Text);
    if SList.Count > 0 then
    begin
      SQLEditForm.SynEdit1.Lines.Append('  (');
      Cnt := SList.Count - 1;
      for I := 0 to Cnt do
      begin
        S := '  :' + SList.Strings[I];
        if I < Cnt then S := S + ',';
        SQLEditForm.SynEdit1.Lines.Append(S);
      end;
      SQLEditForm.SynEdit1.Lines.Append('  )');
    end;
  finally
    SList.Free;
  end;
end;

function TSQLPropertyEditor.DefaultMetaTypes: TMetadataTypes;
begin
  Result := [mtTable, mtView, mtStoredProc];
end;

constructor TSQLPropertyEditor.Create(Hook: TPropertyEditorHook;
  APropCount: Integer);
begin
  inherited Create(Hook, APropCount);
  DefaultIndex := 0;
  HideRadioSqlType := False;
  SqlDialect := sqlStandard;
  UseReturningField := False;
end;

procedure TSQLPropertyEditor.Edit;
begin
  SQLEditForm := TSQLEditForm.Create(Application);
  SQLEditForm.SynEdit1.Lines.Assign(TStrings(GetObjectValue));
  SQLEditForm.Panel3.Caption := GetPropInfo^.Name;
  SQLEditForm.SQLPropEdit := Self;
  if HideRadioSqlType then SQLEditForm.RadioGroup1.Visible := False;
  SQLEditForm.RadioGroup1.ItemIndex := DefaultIndex;
  LoadMetadata(DefaultMetaTypes);
  if SQLEditForm.ShowModal = mrOK then
  begin
    SetPtrValue(SQLEditForm.SynEdit1.Lines);
    GlobalDesignHook.RefreshPropertyValues;
  end;
end;

end.

