unit ProjectManager;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, ComCtrls,
  XMLRead, XMLWrite, DOM, Editor, LCLType, Menus, projoption, FormDesig;

type
  PNodeObject = ^TNodeObject;
  TNodeObject = record
    Obj: string;
    ObjClassName: string;
    Code: string;
    ExtModule: Boolean;
    UseUnit: Boolean;
    ADesignHook: TDesignerHook;
  end;

   { TProjectManagerForm }

  TProjectManagerForm = class(TForm)
    ImageList1: TImageList;
    MenuItem1: TMenuItem;
    NSort: TMenuItem;
    NDelMenuItem: TMenuItem;
    OpenDialog1: TOpenDialog;
    PopupMenu1: TPopupMenu;
    SaveDialog1: TSaveDialog;
    SaveDialog2: TSaveDialog;
    TreeObj: TTreeView;
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NDelMenuItemClick(Sender: TObject);
    procedure NSortClick(Sender: TObject);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure TreeObjDblClick(Sender: TObject);
    procedure TreeObjDeletion(Sender: TObject; Node: TTreeNode);
    procedure TreeObjDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure TreeObjDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure TreeObjKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState
      );
    procedure TreeObjMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
  private
    FProjectName: string;
    FAutoObjectList: TList;
    FReadOnly: Boolean;
    FExternalTreeNode: TTreeNode;
  private
    procedure SetProjectName(AValue: string);
    procedure SetReadOnly(AValue: Boolean);
    function SortTreeObj(Node1, Node2: TTreeNode): integer;
  public
    ProjectFile: string;
    ProjectOption: TProjectOptions;
    ProjectModified: Boolean;
    HistoryList: TStringList;
    ExternalModulePath: string;
    procedure ClearUseUnit;
    function DefaultLanguage: string;
    function OpenModule(Node: TTreeNode; ShowObject: Boolean = True): TPageModule;
    function OpenModuleByName(AName: string; ShowObject: Boolean = True): TPageModule;
    procedure NewProject(ALanguage: string);
    procedure RenameProject(NewName: string);
    procedure CloseProject;
    procedure DeleteSelected;
    function ProjectExist: Boolean;
    procedure GetFrameList(AList: TStrings);
    function GetFrameLFM(AClassName: string): string;
    function GetModuleNode(ModuleName: string; IsFolder: Boolean = false): TTreeNode;
    function GetModuleCode(ModuleName: string): string;
    function SaveModule(Module: TPageModule): Boolean;
    function SaveModuleAs(Module: TPageModule): Boolean;
    function WriteExternal(Module: TPageModule; Node: TTreeNode): Boolean;
    function FindExtPathForModule: string;
    function SaveModuleInFile(Module: TPageModule): Boolean;
    function SaveModuleAsExternal(Module: TPageModule; SaveAs: Boolean): Boolean;
    function UnitExist(AUnitName: string): Boolean;
    function ObjectNameExist(AName: string): Boolean;
    procedure CloseAll;
    procedure ProjectToHistoryList;
    procedure SaveProject(SaveAs: Boolean = False);
    procedure SaveAllModules;
    procedure LoadProject(Template: Boolean);
    procedure LoadFromUrl(Template: Boolean);
    procedure OpenConfigFile(FlName: string);
    procedure GetExternalModule(SL: TStrings);
    function GetExternalNode: TTreeNode;
    function IsExternalOpen: Boolean;
    procedure UpdateExternalPath(Sender: TObject);
    function LoadExternalFromPath(ExternalNode: TTreeNode; ExtPath: string): boolean;

    procedure Modified;
    procedure ProjectOptionsFormShow;
    procedure GetModuleList(SList, ExcludeList: TStrings);
    property ProjectName: string read FProjectName write SetProjectName;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly;
    property ExternalTreeNode: TTreeNode read GetExternalNode;
  end;

var
  ProjManager: TProjectManagerForm;

const
  MainFormCaption = 'Дизель-Паскаль';

implementation

uses Main, MDialog, ProjOptForm, LResources, dpi_parser, AppConst, httpsend,
  ftpsend, synautil, LazUTF8, LangTransCode, LazFileUtils, expandenvvar, ssl_openssl3;

{$R *.lfm}

{TProjectManagerForm}

procedure TProjectManagerForm.FormClose(Sender: TObject;
  var CloseAction: TCloseAction);
begin
  CloseAction := caHide;
end;

procedure TProjectManagerForm.FormCreate(Sender: TObject);
begin
  ProjectOption := nil;
  ProjectModified := False;
  FAutoObjectList := TList.Create;
  HistoryList := TStringList.Create;
  NDelMenuItem.Caption := 'Удалить';
  ReadOnly := False;
  ExternalModulePath := '';
end;

procedure TProjectManagerForm.FormDestroy(Sender: TObject);
begin
  TreeObj.Items.Clear;
  FAutoObjectList.Free;
  HistoryList.Free;
end;

procedure TProjectManagerForm.NDelMenuItemClick(Sender: TObject);
begin
  if not ReadOnly then DeleteSelected
  else ShowMessage('Project is ReadOnly');
end;

procedure TProjectManagerForm.NSortClick(Sender: TObject);
begin
  if TreeObj.Selected <> nil then
    if TreeObj.Selected.Parent <> nil then
      TreeObj.Selected.Parent.CustomSort(@SortTreeObj);
end;

procedure TProjectManagerForm.PopupMenu1Popup(Sender: TObject);
begin
  if (TreeObj.Selected <> nil) and (TreeObj.Selected.Level > 0) then
  begin
    NDelMenuItem.Visible := True;
    NSort.Visible := True;
  end
  else
  begin
    NDelMenuItem.Visible := False;
    NSort.Visible := False;
  end;

end;

procedure TProjectManagerForm.TreeObjDblClick(Sender: TObject);
begin
  if TreeObj.Selected <> nil then
  begin
    if TreeObj.Selected.HasChildren then TreeObj.Selected.Expand(False);
    OpenModule(TreeObj.Selected);
  end;
end;

procedure TProjectManagerForm.TreeObjDeletion(Sender: TObject; Node: TTreeNode);
var
  PNodeObj: PNodeObject;
begin
  if Node = FExternalTreeNode then FExternalTreeNode := nil;
  if Node.Data <> nil then
  begin
    PNodeObj:= PNodeObject(Node.Data);
    PNodeObj^.Code := '';
    PNodeObj^.Obj := '';
    PNodeObj^.ObjClassName := '';
    Dispose(PNodeObj);
    Node.Data := nil;
  end;
end;

procedure TProjectManagerForm.TreeObjDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  Node: TTreeNode;
  Mode: TNodeAttachMode;
begin
  Node := TreeObj.GetNodeAt(X, Y);
  if Node <> nil then
  begin
    Mode := naInsert;
    if (Node.ImageIndex = 0) or (Node.Level = 0) then
      Mode := naAddChild;
      TreeObj.Selected.MoveTo(Node, Mode);
  end;
end;

procedure TProjectManagerForm.TreeObjDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  if Source = TreeObj then Accept:= True
  else Accept:= False;
end;

procedure TProjectManagerForm.TreeObjKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = VK_DELETE) and (Shift = [ssCtrl]) then
    DeleteSelected;
end;

procedure TProjectManagerForm.TreeObjMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  Node: TTreeNode;
begin
  if Button = mbLeft then
  begin
    if TreeObj.Selected <> nil then
    begin
      Node := TreeObj.GetNodeAt(X, Y);
      if Node = TreeObj.Selected then
        TreeObj.BeginDrag(False, 10);
    end;
  end;
end;

procedure TProjectManagerForm.SetProjectName(AValue: string);
begin
  if FProjectName = AValue then Exit;

  FProjectName := AValue;
  if ProjectOption <> nil then ProjectOption.AppName := FProjectName;
  if FProjectName <> '' then
    MainIDEDesForm.Caption := MainFormCaption + ' - ' + FProjectName
  else
    MainIDEDesForm.Caption := MainFormCaption;
end;

procedure TProjectManagerForm.SetReadOnly(AValue: Boolean);
var
  I: Integer;
begin
  if FReadOnly=AValue then Exit;
  FReadOnly:=AValue;

    if FReadOnly then
    begin
      if MainIDEDesForm.Inspector <> nil then
        MainIDEDesForm.Inspector.Enabled := False
    end
    else
    begin
      if MainIDEDesForm.Inspector <> nil then
        MainIDEDesForm.Inspector.Enabled := True;

      if EditorForm <> nil then
      begin
        for I := 0 to EditorForm.PageControl1.PageCount - 1 do
        begin
          if not TPageModule( EditorForm.PageControl1.Pages[I]).ExtModule then
            TPageModule( EditorForm.PageControl1.Pages[I]).ReadOnly := False;
        end;
      end;
    end;
end;

function TProjectManagerForm.SortTreeObj(Node1, Node2: TTreeNode): integer;
begin
    if ((Node1.ImageIndex = 0) and (Node2.ImageIndex = 0)) or
       ((Node1.ImageIndex > 0) and (Node2.ImageIndex > 0)) then
      Result := AnsiCompareStr(Node1.Text, Node2.Text)
    else
    if (Node1.ImageIndex = 0) and (Node2.ImageIndex > 0) then Result := 1
    else
    if (Node2.ImageIndex = 0) and (Node1.ImageIndex > 0) then Result := -1;
end;

procedure TProjectManagerForm.ClearUseUnit;
var
  Node: TTreeNode;

  procedure ClearNodeUseUnit(ANode: TTreeNode);
  var
    ChNode: TTreeNode;
  begin
    while ANode <> nil do
    begin
      if ANode.Data <> nil then
        PNodeObject(ANode.Data)^.UseUnit := False;

      ChNode := ANode.GetFirstChild;
      if ChNode <> nil then ClearNodeUseUnit(ChNode);

      ANode := ANode.GetNextSibling;
    end;
  end;

begin
  Node := TreeObj.Items.GetFirstNode;
  if Node.Data <> nil then ClearNodeUseUnit(Node);
end;

procedure TProjectManagerForm.NewProject(ALanguage: string);
var
  PNodeObj: PNodeObject;
  Node, ProjNode: TTreeNode;
  ACode, MainFormModuleName: string;
begin
  if (EditorForm <> nil) or (ProjectOption <> nil) then
  begin
    ShowMessage('Закройте текущий проект');
    Exit;
  end;
  MainIDEDesForm.BreakPointList.Clear;
  ReadOnly := False;
  ProjectName := 'NewProject';
  MainFormModuleName := 'Main';

  ProjectOption := TProjectOptions.Create(Self);
  ProjectOption.AppName := ProjectName;
  ProjectOption.Title := ProjectName;
  ProjectOption.Picture.Assign(nil);
  ProjectOption.AppIcon.Assign(nil);
  ProjectOption.AppAutor := '';
  ProjectOption.Language := ALanguage;
  ProjectOption.MinVMajor := 0;
  ProjectOption.MinVMinor := 0;
  ProjectOption.MinVRevision := 0;
  ProjectOption.OnUpdateExternalPath := @UpdateExternalPath;
  //ClearTreeItems;

    ACode := GetKeyOfLangID(ID_program) + ' NewProject;'#13#10#13#10 + GetKeyOfLangID(ID_uses) + ' Main;'#13#10#13#10 + GetKeyOfLangID(ID_begin) + #13#10'  MainForm ';
    if ALanguage = ALterLanguage then
      ACode := ACode + '='
    else
      ACode := ACode + ':=';
    ACode := ACode + ' TMainForm.Create(Application);'#13#10 + GetKeyOfLangID(ID_end) + '.';



  New(PNodeObj);
  PNodeObj^.Obj := '';
  PNodeObj^.ADesignHook := nil;
  PNodeObj^.Code:= ACode;
  PNodeObj^.ExtModule:= False;
  ProjNode := TreeObj.Items.AddChildObject(nil, ProjectName, PNodeObj);
  ProjNode.ImageIndex := 2;
  ProjNode.SelectedIndex := 2;
  ProjNode.StateIndex := 2;

  New(PNodeObj);
  PNodeObj^.ADesignHook := nil;
  PNodeObj^.ExtModule := False;
  Node := TreeObj.Items.AddChildObject(ProjNode, MainFormModuleName, PNodeObj); //SysNode AddObject(nil, 'Main', PNodeObj);
  Node.ImageIndex := 1;
  Node.SelectedIndex := 1;
  Node.StateIndex := 1;
  MainIDEDesForm.NewForm(True);
  ProjNode.Expand(False);
  //Modified;
end;

procedure TProjectManagerForm.RenameProject(NewName: string);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  SL: TStringList;
  PNodeObj: PNodeObject;
  Node: TTreeNode;

begin
  Node := TreeObj.Items.GetFirstNode;
  if Node.Text = NewName then Exit;
  Node.Text := NewName;
  PNodeObj := PNodeObject(Node.Data);
  Parser := TSuMemoParser.Create();
  Tokens := TMemoTokens.Create;
  SL := TStringList.Create;

  try
    SL.Text := PNodeObj^.Code;
    Parser.SetParseStr(SL.Text, ProjectOption.Language);
    Parser.GetTokensList(Tokens);
    while Tokens.Token^.TokenID <> ID_EOF do
    begin
      if Tokens.Token^.TokenID = ID_program then
      begin
        SL.Strings[Tokens.Token^.Row] := 'program ' + NewName + ';';
        PNodeObj^.Code := SL.Text;
        ProjectModified := True;
        Break;
      end;
      Tokens.Next;
    end;

  finally
    SL.Free;
    Tokens.Free;
    Parser.Free;
  end;
end;

function TProjectManagerForm.DefaultLanguage: string;
begin
  Result := BaseLanguage;
  if ProjectExist then Result := ProjectOption.Language;
end;

function TProjectManagerForm.OpenModule(Node: TTreeNode; ShowObject: Boolean = True): TPageModule;
var
  Form: TCustomForm;
  Root: TComponent;
  I: Integer;
  Find: Boolean;
  PNodeObj: PNodeObject;
  AReadOnly: Boolean;
begin
  Result := nil;
  if Node <> nil then
  begin
    if Node.Data = nil then
    begin
      if Node.HasChildren then Node.Expand(True);
      Exit;
    end;
      if EditorForm = nil then
      begin
        CreateEditor;
      end;
      I := 0;
      Find := False;
      while I < EditorForm.ModuleCount do
      begin
        if UTF8CompareText(EditorForm.Modules[I].Caption, Node.Text) = 0 then
        begin
          EditorForm.PageControl1.ActivePage := EditorForm.Modules[I];
          Result := EditorForm.Modules[I];
          Find := True;
          EditorForm.Show;
          {if (Result.DesignForm <> nil) and ShowObject then
          begin
            Application.ProcessMessages;
            MainIDEDesForm.DManager.ShowForm(Result.DesignForm);
          end;}
          Break;
        end;
        Inc(I);
      end;
      if not Find then
      begin
        PNodeObj := Node.Data;
        Form := nil;
        Root := nil;
        AReadOnly := ReadOnly;
        EditorForm.Show;
        if PNodeObj^.ADesignHook <> nil then
        begin
          PNodeObj^.ADesignHook.DesignOpen;
          Form := PNodeObj^.ADesignHook.Form;
          Root := PNodeObj^.ADesignHook.Root;
         end
        else
        begin
          Form := nil; Root := nil;
          MainIDEDesForm.DManager.TextToDesignObj(PNodeObj^.Obj, Root, Form);
          if Form <> nil then
            MainIDEDesForm.LastActiveForm := Form;
        end;
        Result := EditorForm.NewModule(Root, Form, Node.Text, PNodeObj^.Code, DefaultLanguage);
        Result.ExtModule := PNodeObj^.ExtModule;
        if Result.ExtModule then
          AReadOnly := True;
        Result.ReadOnly := AReadOnly;
        EditorForm.PageControl1.ActivePage := Result;

        if (Form <> nil) and ShowObject then
        begin
          //Application.ProcessMessages;
          //MainIDEDesForm.DManager.ShowForm(Form);

          //Application.ProcessMessages;
          MainIDEDesForm.DManager.PaintCurentSelection;
          TDesignerHook(Form.Designer).ResetModified;
        end;
      end;
  end;
end;

function TProjectManagerForm.OpenModuleByName(AName: string; ShowObject: Boolean
  ): TPageModule;
var
  Node: TTreeNode;
begin
  Result := nil;
  Node := GetModuleNode(AName);
  if Node <> nil then
    Result := OpenModule(Node, ShowObject);
end;

procedure TProjectManagerForm.CloseProject;
begin
  ProjectOption.Free;
  ProjectOption := nil;
  ProjectName := '';
  ProjectFile := '';
  ExternalModulePath := '';
  FExternalTreeNode := nil;
  TreeObj.Items.Clear;
  MainIDEDesForm.DManager.Clear;
  MainIDEDesForm.BreakPointList.Clear;

end;

procedure TProjectManagerForm.DeleteSelected;
begin
  if (TreeObj.Selected <> nil) and (TreeObj.Selected.Level > 0) then
  begin
      if TreeObj.Selected.HasChildren then
      begin
        ShowMessage('Объект имеет вложения и не может быть удален');
        Exit;
      end
      else
      begin
        if (TreeObj.Selected.Data <> nil) and (EditorForm <> nil) then
        begin
          if EditorForm.GetModuleByName(TreeObj.Selected.Text) <> nil then
          begin
            ShowMessage('Объект открыт в редакторе и не может быть удален');
            Exit;
          end;
        end;

        if Application.MessageBox(PChar('Удалить выбранный узел?'), PChar(Caption), MB_YESNO or MB_ICONQUESTION) = IDYES then
        begin
          TreeObj.Items.Delete(TreeObj.Selected);
          ProjectModified:= True;
        end;

      end;
  end;
end;

function TProjectManagerForm.ProjectExist: Boolean;
begin
  Result := ProjectOption <> nil;
end;

procedure TProjectManagerForm.GetFrameList(AList: TStrings);
var
  Node: TTreeNode;
  PNodeObj: PNodeObject;
  LFMType, LFMComponentName, LFMClassName: String;
begin
  Node := TreeObj.Items.GetFirstNode;
  while Node <> nil do
  begin
    if Node.Data <> nil then
    begin
      PNodeObj := PNodeObject(Node.Data);
      if PNodeObj^.Obj <> '' then
      begin
        ReadLFMHeader(PNodeObj^.Obj, LFMType, LFMComponentName, LFMClassName);
        if (UTF8CompareText(LFMClassName, 'TSuFrame') = 0) or (UTF8CompareText(LFMClassName, 'TFrame') = 0) then
          AList.AddObject(LFMComponentName + ': '+ PNodeObj^.ObjClassName, Node);
      end;
    end;
    Node := Node.GetNext;
  end;
end;

function TProjectManagerForm.GetFrameLFM(AClassName: string): string;
var
  Node: TTreeNode;
  PNodeObj: PNodeObject;
begin
  Result := '';
  Node := TreeObj.Items.GetFirstNode;
  while Node <> nil do
  begin
    if Node.Data <> nil then
    begin
      PNodeObj := PNodeObject(Node.Data);
      if UTF8CompareText(PNodeObj^.ObjClassName, AClassName) = 0 then
      begin
        Result := PNodeObj^.Obj;
        Break;
      end;
    end;
    Node := Node.GetNext;
  end;
end;

procedure TProjectManagerForm.Modified;
begin
  ProjectModified := True;
end;

procedure TProjectManagerForm.ProjectOptionsFormShow;
var
  ProjectOptionForm: TProjectOptionForm;
begin
  if not ProjectExist then Exit;
  ProjectOptionForm := TProjectOptionForm.Create(Self);
  try
    ProjectOptionForm.Edit1.Text := AnsiToUtf8(ProjectOption.AppName);
    ProjectOptionForm.Edit2.Text := AnsiToUtf8(ProjectOption.Title);
    ProjectOptionForm.Edit3.Text := AnsiToUtf8(ProjectOption.AppAutor);
    ProjectOptionForm.Image1.Picture.Assign(ProjectOption.AppIcon);
    ProjectOptionForm.Image2.Picture.Assign(ProjectOption.Picture);
    ProjectOptionForm.MinVMajorEdit.Text :=  IntToStr(ProjectOption.MinVMajor);
    ProjectOptionForm.MinVMinorEdit.Text :=  IntToStr(ProjectOption.MinVMinor);
    ProjectOptionForm.MinVRevisionEdit.Text :=  IntToStr(ProjectOption.MinVRevision);
    ProjectOptionForm.ExtModuleEdit.Text := ProjectOption.ExternalPath;
    ProjectOptionForm.ExternalListBox.Items.Assign(ProjectOption.AutoCreateExternal);
    ProjectOptionForm.StartParamEdit.Text := ProjectOption.StartParametrs;
    if ProjectOptionForm.ShowModal = mrOK then
    begin
      ProjectOption.AppName := Utf8ToAnsi(ProjectOptionForm.Edit1.Text);
      ProjectOption.Title := Utf8ToAnsi(ProjectOptionForm.Edit2.Text);
      ProjectOption.AppAutor := Utf8ToAnsi(ProjectOptionForm.Edit3.Text);
      ProjectOption.AppIcon.Assign(ProjectOptionForm.Image1.Picture.Icon);
      ProjectOption.Picture.Assign(ProjectOptionForm.Image2.Picture);
      if ProjectOptionForm.MinVMajorEdit.Text = '' then
        ProjectOption.MinVMajor := 0
      else
        ProjectOption.MinVMajor := StrToInt(ProjectOptionForm.MinVMajorEdit.Text);
      if ProjectOptionForm.MinVMinorEdit.Text = '' then
        ProjectOption.MinVMinor := 0
      else
        ProjectOption.MinVMinor := StrToInt(ProjectOptionForm.MinVMinorEdit.Text);
      if ProjectOptionForm.MinVRevisionEdit.Text = '' then
        ProjectOption.MinVRevision := 0
      else
        ProjectOption.MinVRevision := StrToInt(ProjectOptionForm.MinVRevisionEdit.Text);

      ProjectOption.ExternalPath := ProjectOptionForm.ExtModuleEdit.Text;
      ProjectOption.AutoCreateExternal.Assign(ProjectOptionForm.ExternalListBox.Items);
      ProjectOption.StartParametrs := ProjectOptionForm.StartParamEdit.Text;
      Modified;
    end;
  finally
    ProjectOptionForm.Free;
  end;
end;

procedure TProjectManagerForm.GetModuleList(SList, ExcludeList: TStrings);
var
  Node: TTreeNode;
  AName: string;
begin
  Node := TreeObj.Items.GetFirstNode;
  Node := Node.GetFirstChild;
  while Node <> nil do
  begin
    if Node.Data <> nil then
    begin
      AName:= LowerCase(Node.Text);
      if ExcludeList.IndexOf(AName) < 0 then
        SList.Add(Node.Text);
    end;
    Node := Node.GetNext;
  end;
end;

function TProjectManagerForm.GetModuleNode(ModuleName: string; IsFolder: Boolean
  ): TTreeNode;
var
  MainNode: TTreeNode;

  function GetModNode(ANode: TTreeNode): TTreeNode;
  var
    INode: TTreeNode;
    ANodeName: string;
    AFindName: string;
  begin
    Result := nil;
    AFindName:= UTF8LowerCase(ModuleName);
    while ANode <> nil do
    begin
      ANodeName:= UTF8LowerCase(ANode.Text);
      if ((ANode.Data <> nil) or (IsFolder = True)) and (ANodeName = AFindName) then
      begin
        Result := ANode;
        Exit;
      end;
      if ANode.HasChildren then
      begin
        INode := ANode.getFirstChild;
        Result := GetModNode(INode);
        if Result <> nil then Break;
      end;
      ANode := ANode.getNextSibling;
    end;
  end;
begin
  Result := nil;
  MainNode := TreeObj.Items.GetFirstNode;
  if MainNode <> nil then Result := GetModNode(MainNode);
end;

function TProjectManagerForm.GetModuleCode(ModuleName: string): string;
var
  Node: TTreeNode;
  PNodeObj: PNodeObject;
begin
  Node := GetModuleNode(ModuleName);
  if (Node <> nil) and (Node.Data <> nil) then
  begin
    PNodeObj := PNodeObject(Node.Data);
    Result := PNodeObj^.Code;
  end;
end;

function TProjectManagerForm.SaveModule(Module: TPageModule): Boolean;
var
  Node: TTreeNode;
  Dialog: TModuleDialog;
  AName: string;
  PNodeObj: PNodeObject;
begin
  Result := False;
  if ReadOnly then Exit;

  if not ProjectExist then
  begin
    Result := SaveModuleInFile(Module);
    Exit;
  end;

  Node := GetModuleNode(Module.Caption);
  if Node <> nil then
  begin
    PNodeObj := Node.Data;
  end
  else
  begin
    Dialog := TModuleDialog.Create(Self);
    Dialog.DialogStyle := mdsSave;
    Dialog.FileName := Module.Caption;
    if Dialog.Execute then
    begin
      AName := Dialog.FileName;
      Node := GetModuleNode(AName);
      if Node <> nil then
      begin
        PNodeObj := Node.Data;
      end
      else
      begin
        New(PNodeObj);
        PNodeObj^.ExtModule := False;
        PNodeObj^.ADesignHook := nil;
        Node := TreeObj.Items.AddChildObject(Dialog.NodeFolder, AName, PNodeObj);
        if Module.DesignObject <> nil then
        begin
          Node.StateIndex := 1;
          Node.SelectedIndex := 1;
          Node.ImageIndex := 1;
        end
        else
        begin
          Node.StateIndex := 2;
          Node.SelectedIndex := 2;
          Node.ImageIndex := 2;
        end;
        Module.Caption := AName;
        Node.Parent.CustomSort(@SortTreeObj);
      end;
    end
    else
      Exit;
  end;
  if PNodeObj <> nil then
  begin
    if Module.DesignObject <> nil then
    begin
      PNodeObj^.Obj := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
      PNodeObj^.ObjClassName := 'T' + Module.DesignObject.Name;
      if Module.DesignForm <> nil then
        PNodeObj^.ADesignHook := TDesignerHook(Module.DesignForm.Designer)
      else
        PNodeObj^.ADesignHook := nil;
    end
    else
    begin
      PNodeObj^.Obj := '';
      PNodeObj^.ObjClassName := '';
      PNodeObj^.ADesignHook := nil;
    end;
    if AName <> '' then
      Module.SynEdit.SetUnitName(AName);
    PNOdeObj^.Code := Module.SynEdit.Text;

    if (Node <> Nil) and ((Node.Parent <> nil) and (Node.Parent.Text = 'External')) then
    begin
       if WriteExternal(Module, Node) then Module.Modified := False;
    end
    else
      Module.Modified := False;


    //FormsDesigner.SetFormModified(Module.FForm, False);
    Modified;
    Result := True;
  end;
end;

function TProjectManagerForm.SaveModuleAs(Module: TPageModule): Boolean;
var
  Node, TestNode: TTreeNode;
  Dialog: TModuleDialog;
  AName, aOldName: string;
  PNodeObj: PNodeObject;
begin
  Result := False;
  if not ProjectExist then
  begin
    Result := SaveModuleInFile(Module);
    Exit;
  end;

  Node := GetModuleNode(Module.Caption);
  if Node = nil then
    PNodeObj := nil
  else
    PNodeObj := PNodeObject(Node.Data);

  Dialog := TModuleDialog.Create(Self);
  Dialog.DialogStyle := mdsSave;
  Dialog.FileName := Module.Caption;
  aOldName:= Module.Caption;
  if Dialog.Execute then
  begin
    AName := Dialog.FileName;
    TestNode := GetModuleNode(AName);
    if (TestNode <> nil) and (TestNode <> Node) then
    begin
      Exception.Create('Модуль с именем "' + AName + '" уже существует');
    end
    else
    begin
      if Node = nil then
      begin
        New(PNodeObj);
        PNodeObj^.ExtModule := False;
        PNodeObj^.ADesignHook := nil;
        PNodeObj^.ExtModule := False;
        Node := TreeObj.Items.AddChildObject(Dialog.NodeFolder, AName, PNodeObj);
      end;
      Node.Text := AName;
      if Node.Parent <> Dialog.NodeFolder then
        Node.MoveTo(Dialog.NodeFolder, naAddChild);
      if Module.DesignObject <> nil then
      begin
        Node.StateIndex := 1;
        Node.SelectedIndex := 1;
        Node.ImageIndex := 1;
      end
      else
      begin
        Node.StateIndex := 2;
        Node.SelectedIndex := 2;
        Node.ImageIndex := 2;
      end;
      Module.Caption := AName;
      Node.Parent.CustomSort(@SortTreeObj);
    end;
  end
  else
      Exit;

  if PNodeObj <> nil then
  begin
    if Module.DesignObject <> nil then
    begin
      PNodeObj^.Obj := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
      PNodeObj^.ObjClassName := 'T' + Module.DesignObject.Name;
      if Module.DesignForm <> nil then
        PNodeObj^.ADesignHook := TDesignerHook(Module.DesignForm.Designer)
      else
        PNodeObj^.ADesignHook := nil;
    end
    else
    begin
      PNodeObj^.Obj := '';
      PNodeObj^.ObjClassName := '';
      PNodeObj^.ADesignHook := nil;
    end;
    if AName <> '' then
      Module.SynEdit.SetUnitName(AName);
    PNOdeObj^.Code := Module.SynEdit.Text;
    PNodeObj^.ExtModule := False;
    if Node.Parent.Text = 'External' then
    begin
      if WriteExternal(Module, Node) then
         Module.Modified := False;
    end
    else
       Module.Modified := False;
  end;

  //FormsDesigner.SetFormModified(Module.FForm, False);
  Modified;
  Result := True;
end;

function TProjectManagerForm.WriteExternal(Module: TPageModule; Node: TTreeNode
  ): Boolean;
var
  SL: TStringList;
  ExtDoc: TXMLDocument;
  ExtRootNode: TDOMElement;
  AFileName, DefExt, MdFileName, ExternalPath: string;
  PNodeObj: PNodeObject;
begin
  Result := False;
  DefExt := '.pas';
  if ProjManager.ProjectExist then
  begin
    if SameText(ProjManager.ProjectOption.Language, ALterLanguage) then
    begin
      DefExt := '.dp';
    end;
  end;

  PNodeObj:= Node.Data;
  PNodeObj^.ExtModule := True;


  ExternalPath:= ExternalModulePath;
  if ExternalPath = '' then ExternalPath:= FindExtPathForModule;
  AFileName := ExternalPath + DirectorySeparator
  + Module.Caption + DefExt;

  Module.SynEdit.Lines.SaveToFile(AFileName);
  if Module.DesignObject <> nil then
  begin
    MdFileName := ExtractFileNameWithoutExt(AFileName) + '.md';
    SL := TStringList.Create;
    ExtDoc := TXMLDocument.Create;
    try
      SL.Text := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
      ExtRootNode := ExtDoc.CreateElement('Object');
      ExtRootNode['ImageIndex'] := '1';
      ExtRootNode['ClassName'] := 'T' + Module.DesignObject.Name;
      ExtRootNode.TextContent := SL.Text;
      ExtDoc.AppendChild(ExtRootNode);
      WriteXMLFile(ExtDoc, MdFileName);
      Module.Modified := False;
      Result := True;
    finally
      SL.Free;
      ExtDoc.Free;
    end;
  end;
end;

function TProjectManagerForm.FindExtPathForModule: string;
var
  SL: TStringList;
  I: Integer;
  ExtPath, ExtFileName: string;
  FTP: TFTPSend;
  Prot, User, Pass, Host, Port, Path, Para: string;
begin
  Result := '';
  SL := TStringList.Create;
  try
    SL.Delimiter := ';';
    SL.DelimitedText := ProjectOption.ExternalPath;
    for I := 0 to SL.Count - 1 do
    begin
      ExtPath := Trim(ProjManager.ProjectOption.ExpandPath(SL.Strings[I]));;
      if (ExtPath <> '') then
      begin
        if Pos('ftp://', ExtPath) = 1 then
        begin
          FTP := TFTPSend.Create;
          try
            ParseURL(ExtPath, Prot, User, Pass, Host, Port, Path, Para);
            FTP.TargetHost := Host;
            FTP.TargetPort := Port;
            FTP.UserName := User;
            FTP.Password := Pass;
            if FTP.Login then
            begin
              if FTP.ChangeWorkingDir(Path) then
              begin
                Result := ExtPath;
                Break;
              end;
            end;
          finally
            FTP.Free;;
          end;
        end
        else
        if (Pos('http://', ExtPath) = 1) or (Pos('https://', ExtPath) = 1) then
        begin
          //Not supported
          Result := '';
        end
        else
        if DirPathExists(ExtPath) then
        begin
          Result := ExcludeTrailingPathDelimiter(ExtPath);
          Break;
        end;
      end;
    end;
  finally
    SL.Free;
  end;
end;

function TProjectManagerForm.SaveModuleInFile(Module: TPageModule): Boolean;
var
  SL: TStringList;
  S: string;
  L, I: Integer;
begin
  SaveDialog2.DefaultExt := '.pas';
  SaveDialog2.Filter := 'Pascal|*.pas';
  if ProjManager.ProjectExist then
  begin
    if SameText(ProjManager.ProjectOption.Language, ALterLanguage) then
    begin
      SaveDialog2.DefaultExt := '.dp';
      SaveDialog2.Filter := 'Diesel Pascal|*.dp';
    end;
  end;
  SaveDialog2.FileName := Module.Caption;
  if SaveDialog2.Execute then
  begin
    Module.SynEdit.Lines.SaveToFile(SaveDialog2.FileName);
    if Module.DesignObject <> nil then
    begin
      SL := TStringList.Create;
      try
         S := SaveDialog2.FileName;
         L := Length(S);
         I := LastDelimiter('.', S);
         if I > 0 then
           Delete(S, I, L - I + 1);

         S := S + '.lfm';
         SL.Text := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
         SL.SaveToFile(S);
      finally
        SL.Free;
      end;
    end;
  end;
end;

function TProjectManagerForm.SaveModuleAsExternal(Module: TPageModule;
  SaveAs: Boolean): Boolean;
var
  ExtDoc: TXMLDocument;
  ExtRootNode: TDOMElement;
  MdFileName, AFileName, DefExt, AName: string;
  ExtPath, SavePath: string;
  SL: TStringList;
  PNodeObj: PNodeObject;
  Node, ExtNode, ParentNode: TTreeNode;
begin
  AFileName := '';
  SaveAs:= False;
  ExtPath := ExternalModulePath;
  if ExtPath = '' then ExtPath:= ProjectOption.ExternalPath;
  if ExtPath = '' then SaveAs:= True
  else
  if (Pos('ftp://', ExtPath) = 1) or
    (Pos('http://', ExtPath) = 1) or
    (Pos('https://', ExtPath) = 1) then
  begin
    SaveAs:= True
  end
  else
  if not SaveAs then
  begin
    DefExt := '.pas';
    if ProjManager.ProjectExist then
    begin
      if SameText(ProjManager.ProjectOption.Language, ALterLanguage) then
      begin
        DefExt := '.dp';
      end;
    end;

    if ExtPath = '' then ExtPath := FindExtPathForModule;

    AFileName:= ExtPath + DirectorySeparator + Module.Caption + DefExt;

    if not FileExists(AFileName) then  SaveAs := True;
  end;

  if SaveAs then
  begin
    SaveDialog2.InitialDir := ExtPath;

    SaveDialog2.DefaultExt := '.pas';
    SaveDialog2.Filter := 'Pascal|*.pas';
    if ProjManager.ProjectExist then
    begin
      if SameText(ProjManager.ProjectOption.Language, ALterLanguage) then
      begin
        SaveDialog2.DefaultExt := '.dp';
        SaveDialog2.Filter := 'Diesel Pascal|*.dp';
      end;
    end;

    SaveDialog2.FileName := Module.Caption + SaveDialog2.DefaultExt;
    if SaveDialog2.Execute then
    begin
      AFileName:= SaveDialog2.FileName;
    end;
  end;

  if AFileName <> '' then
  begin
    SavePath := ExtractFilePath(AFileName);

    Module.SynEdit.Lines.SaveToFile(AFileName);

    if Module.DesignObject <> nil then
    begin
      MdFileName := ExtractFileNameWithoutExt(AFileName) + '.md';
      SL := TStringList.Create;
      ExtDoc := TXMLDocument.Create;
      try
         SL.Text := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
         ExtRootNode := ExtDoc.CreateElement('Object');
         ExtRootNode['ImageIndex'] := '1';
         ExtRootNode['ClassName'] := 'T' + Module.DesignObject.Name;
         ExtRootNode.TextContent := SL.Text;
         ExtDoc.AppendChild(ExtRootNode);
         WriteXMLFile(ExtDoc, MdFileName);
      finally
        SL.Free;
        ExtDoc.Free;
      end;
    end;

    Node := GetModuleNode(Module.Caption);
    if Node = nil then
    begin
      SavePath:= ExcludeTrailingPathDelimiter(SavePath);
      if (ExternalModulePath <> '') and (SavePath = ExternalModulePath) then
      begin

        ExtNode := GetModuleNode('external', True);
        if ExtNode = nil then
        begin
          ParentNode := TreeObj.Items.GetFirstNode;;
          ExtNode := TreeObj.Items.AddChild(ParentNode, 'External');
          ExtNode.ImageIndex := 0;
          ExtNode.StateIndex := 0;
          ExtNode.SelectedIndex := 0;
        end;

        New(PNodeObj);
        PNodeObj^.ExtModule := True;
        PNodeObj^.ADesignHook := nil;
        AName:= ExtractFileNameWithoutExt(AFileName);
        AName:= ExtractFileName(AName);
        Node := TreeObj.Items.AddChildObject(ExtNode, AName, PNodeObj);
        if Module.DesignObject <> nil then
        begin
            Node.StateIndex := 1;
            Node.SelectedIndex := 1;
            Node.ImageIndex := 1;
        end
        else
        begin
          Node.StateIndex := 2;
          Node.SelectedIndex := 2;
          Node.ImageIndex := 2;
        end;
        if AName <> '' then
          Module.SynEdit.SetUnitName(AName);
        Module.Caption := AName;
        Node.Parent.CustomSort(@SortTreeObj);
      end;
    end;

    if Node <> nil then
    begin
      PNodeObj := Node.Data;
      if Module.DesignObject <> nil then
      begin
        PNodeObj^.Obj := MainIDEDesForm.DManager.ObjToText(Module.DesignObject);
        PNodeObj^.ObjClassName := 'T' + Module.DesignObject.Name;
        if Module.DesignForm <> nil then
          PNodeObj^.ADesignHook := TDesignerHook(Module.DesignForm.Designer)
        else
          PNodeObj^.ADesignHook := nil;
      end;
      PNOdeObj^.Code := Module.SynEdit.Text;
    end;

    Module.Modified := False;
  end;
end;

function TProjectManagerForm.UnitExist(AUnitName: string): Boolean;
var
  Node: TTreeNode;
  ANodeName, AName: string;
begin
  Result := False;
  Node := TreeObj.Items.GetFirstNode;
  AName:= UTF8LowerCase(AUnitName);
  while Node <> nil do
  begin
    ANodeName:= UTF8LowerCase(Node.Text);
    if ANodeName = AName then
    begin
      Result := True;
      Break;
    end;
    Node := Node.GetNext;
  end;
end;

function TProjectManagerForm.ObjectNameExist(AName: string): Boolean;
var
  Node: TTreeNode;
  PNodeObj: PNodeObject;
  LFMType, LFMComponentName, LFMClassName: string;
  LCName, FindName: string;
begin
  Result := False;
  Node := TreeObj.Items.GetFirstNode;
  FindName:= UTF8LowerCase(AName);
  while Node <> nil do
  begin
    if Node.Data <> nil then
    begin
      PNodeObj := PNodeObject(Node.Data);
      if PNodeObj^.Obj <> '' then
      begin
        ReadLFMHeader(PNodeObj^.Obj, LFMType, LFMComponentName, LFMClassName);
        LCName:= UTF8LowerCase(LFMComponentName);
        if LCName = FindName then
        begin
          Result := True;
          Break;
        end;
      end;
    end;
    Node := Node.GetNext;
  end;
end;

procedure TProjectManagerForm.CloseAll;
var
  I: Integer;
  CancelClose: Boolean;
 begin
  if EditorForm <> nil then EditorForm.CloseAll;
  Application.ProcessMessages;
  CancelClose := False;
  if EditorForm = nil then
  begin
    if ProjectExist then
    begin
      if ProjectModified then
      begin
        I := Application.MessageBox('Сохранить изменения в проекте?', PChar(ProjectName), MB_YESNOCANCEL or MB_ICONQUESTION);
        case I of
          IDYES:
          begin
            SaveProject;
            if ProjectModified then CancelClose := True;
          end;
          IDCANCEL: CancelClose := True;
        end;
      end;
      ProjectToHistoryList;
      if not CancelClose then
      begin
        CloseProject;
      end;

    end;
  end;
end;

procedure TProjectManagerForm.ProjectToHistoryList;
var
  I: Integer;
  S: string;
begin
  if ProjectFile <> '' then
  begin
    I := HistoryList.IndexOf(ProjectFile);

    if I >= 0 then
    begin
      HistoryList.Move(I, 0);
    end
    else
      HistoryList.Insert(0, ProjectFile);

    if HistoryList.Count > 10 then
    begin
          if HistoryList.Objects[HistoryList.Count - 1] <> nil then
            HistoryList.Objects[HistoryList.Count - 1].Free;
          HistoryList.Delete(HistoryList.Count - 1);
    end;

    S := MainIDEDesForm.SettingDir + 'history.txt';
    if not DirectoryExists(MainIDEDesForm.SettingDir) then
       CreateDir(MainIDEDesForm.SettingDir);
    HistoryList.SaveToFile(S);
    MainIDEDesForm.UpdateHistoryMenu;
  end;
end;

procedure TProjectManagerForm.SaveProject(SaveAs: Boolean);
var
  Doc: TXMLDocument;
  RootNode, Node: TDOMElement;
  S: string;
  MainNode: TTreeNode;
  SL: TStringList;

  procedure WriteTreeNode(XMLNode: TDOMElement; ObjectNode: TTreeNode);
  var
    SubNode, ObjXMLNode, CodeXMLNode, ChNode: TDOMElement;
    PNodeObj: PNodeObject;
  begin
    while ObjectNode <> nil do
    begin
      if not SameText(ObjectNode.Text, 'External') then
      begin
        SubNode := Doc.CreateElement(ObjectNode.Text);
        SubNode['ImageIndex'] := IntToStr(ObjectNode.ImageIndex);
        XMLNode.AppendChild(SubNode);
        if ObjectNode.Data <> nil then
        begin
          PNodeObj := PNodeObject(ObjectNode.Data);
          ObjXMLNode := Doc.CreateElement('Object');
          ObjXMLNode['ClassName'] := PNodeObj^.ObjClassName;
          ObjXMLNode.TextContent := PNodeObj^.Obj;
          SubNode.AppendChild(ObjXMLNode);
          CodeXMLNode := Doc.CreateElement('Code');
          CodeXMLNode.TextContent := PNodeObj^.Code;
          SubNode.AppendChild(CodeXMLNode);
        end;
        if ObjectNode.Count > 0 then
        begin
          ChNode := Doc.CreateElement('Child');
          SubNode.AppendChild(ChNode);
          WriteTreeNode(ChNode, ObjectNode.GetFirstChild);
        end;
      end;
      ObjectNode := ObjectNode.GetNextSibling;
    end;
  end;

begin
  if ProjectExist then
  begin
    if ReadOnly then SaveAs := True;
    if (ProjectFile = '') or SaveAs then
    begin
      SaveDialog1.FileName := ProjectName;
      if SaveDialog1.Execute then
        ProjectFile := SaveDialog1.FileName
      else Exit
    end;
    if ProjectFile <> '' then
    begin
      SaveAllModules;
      Doc := TXMLDocument.Create;
      SL := TStringList.Create;
      try
        RootNode := Doc.CreateElement('CrossEngineApp');
        RootNode['Version'] := '1';
        Doc.AppendChild(RootNode);
        S := MainIDEDesForm.DManager.ObjToText(ProjectOption);
        Node := Doc.CreateElement('Options');
        Node.TextContent := S;
        RootNode.AppendChild(Node);
        Node := Doc.CreateElement('Metadata');
        RootNode.AppendChild(Node);
        MainNode := TreeObj.Items.GetFirstNode;
        if MainNode <> nil then
          WriteTreeNode(Node, MainNode);
        WriteXMLFile(Doc, ProjectFile);
        ProjectModified := False;
        ReadOnly := False;
      finally
        SL.Free;
        Doc.Free;
      end;
      ProjectToHistoryList;
    end;
  end
  else ShowMessage('Нет открытого проекта');
end;

procedure TProjectManagerForm.SaveAllModules;
var
  I: Integer;
begin
  if EditorForm = nil then Exit;

  I := EditorForm.ModuleCount - 1;
  while I >= 0 do
  begin
    if EditorForm.Modules[I].Modified then
    begin
      if EditorForm.Modules[I].ExtModule then
        SaveModuleAsExternal(EditorForm.Modules[I], False)
      else
        SaveModule(EditorForm.Modules[I]);
    end;
    Dec(I);
  end;
end;

procedure TProjectManagerForm.LoadProject(Template: Boolean);
begin
  if ProjectExist then CloseAll;
  if not ProjectExist then
  begin
    if OpenDialog1.Execute then
    begin
      OpenConfigFile(OpenDialog1.FileName);
    end;
  end;
end;

procedure TProjectManagerForm.LoadFromUrl(Template: Boolean);
var
  L: Integer;
  S: string;
begin
  if ProjectExist then CloseAll;
  if ProjectExist then Exit;

  S := InputBox('Open Url', 'Введите URL http или ftp', 'http://');
  if S <> '' then
  begin
    if Pos('http://', S) = 1 then
    begin
      L := Length(S);
      if L > 7 then ProjManager.OpenConfigFile(S);
    end
    else
    if Pos('ftp://', S) = 1 then
    begin
      L := Length(S);
      if L > 6 then ProjManager.OpenConfigFile(S);
    end;
  end;
end;

procedure TProjectManagerForm.OpenConfigFile(FlName: string);
type
  TUseFileType = (ufHttp, ufFtp, ufFile);
var
  S, Code, Obj, ObjClassName: string;
  Doc: TXMLDocument;
  RootNode: TDOMElement;
  Node: TDOMNode;
  C: TComponent;
  PNodeObj: PNodeObject;
  MainNode: TTreeNode;
  SL, FL: TStringList;
  Stream: TStream;
  FTP: TFTPSend;
  Prot, User, Pass, Host, Port, Path, Para: string;
  ufType: TUseFileType;
  fs: TSearchRec;
const
  ErrorConfigFormat = 'Загрузка конфигурации не удалась. Не известный формат файла';

  procedure LoadExternal(ParentNode: TTreeNode);
  var
    PathList: TStringList;
    PI: Integer;
    ExtPath: string;
  begin
    PathList := TStringList.Create;
    try
      PathList.Delimiter := ';';
      PathList.DelimitedText := ProjectOption.ExternalPath;
      for PI := 0 to PathList.Count - 1 do
      begin
        ExtPath:= Trim(ProjectOption.ExpandPath(PathList.Strings[PI]));
        if ExtPath <> '' then
        begin
          if LoadExternalFromPath(ParentNode, ExtPath) then Break;
        end;
      end;
    finally
      PathList.Free;
    end;
  end;

  procedure ReadMetadata(XMLNode: TDomNode; ObjNode: TTreeNode);
  var
    ANode: TTreeNode;
    XNode: TDOMElement;
    CodeXMLNode, ObjXMLNode, ChXMLNode: TDOMNode;
    IndStr: string;
    N: Integer;
    ADesObject: TComponent;
    AForm: TCustomForm;
    LFMType, LFMComponentName, LFMClassName: string;
  begin
    while XMLNode <> nil do
    begin
      ANode := TreeObj.Items.AddChild(ObjNode, XMLNode.NodeName);
      if SameText(ANode.Text, 'Main') or (UTF8CompareText(ANode.Text, 'главный') = 0) then MainNode := ANode;
      XNode := XMLNode as TDOMElement;
      if XNode.hasAttribute('ImageIndex') then
      begin
        IndStr := XNode.AttribStrings['ImageIndex'];
        N := StrToInt(IndStr);
        ANode.ImageIndex := N;
        ANode.StateIndex := N;
        ANode.SelectedIndex := N;
      end;
      Code := '';
      Obj := '';
      ObjClassName := '';
      CodeXMLNode := XNode.FindNode('Code');
      if CodeXMLNode <> nil then Code := CodeXMLNode.TextContent;
      ObjXMLNode := XNode.FindNode('Object');
      if ObjXMLNode <> nil then
      begin
        Obj := ObjXMLNode.TextContent;
        ObjClassName := TDOMElement(ObjXMLNode).AttribStrings['ClassName'];
      end;
      if (Obj <> '') or (Code <> '') then
      begin
        New(PNodeObj);
        PNodeObj^.ExtModule:= False;
        PNodeObj^.ADesignHook := nil;
        PNodeObj^.Obj := Obj;
        PNodeObj^.ObjClassName := ObjClassName;
        PNodeObj^.Code := Code;
        ANode.Data := PNodeObj;
        if Obj <> '' then
        begin
          ReadLFMHeader(Obj, LFMType, LFMComponentName, LFMClassName);
          if LFMClassName = 'TDataModule' then
          begin
            ADesObject := nil;
            AForm := nil;
            MainIDEDesForm.DManager.TextToDesignObj(OBJ, ADesObject, AForm, False);
            PNodeObj^.ADesignHook := TDesignerHook(AForm.Designer);
            MainIDEDesForm.DManager.CurrentDesignForm := nil;
            AForm.Visible := True;
            PNodeObj^.ADesignHook.DesignClose;
            Application.ProcessMessages;
            AForm.Hide;
          end;
        end;
      end;
      ChXMLNode := XNode.FindNode('Child');
      if ChXMLNode <> nil then
      begin
        ChXMLNode := ChXMLNode.FirstChild;
        if ChXMLNode <> nil then ReadMetadata(ChXMLNode, ANode);
      end;
      XMLNode := XMLNode.NextSibling;
    end;
  end;

begin
  SetEnvironmentVariable('DIESEL_SCRIPT_NAME', FlName);
  SetEnvironmentVariable('DIESEL_SCRIPT_DIR', ExtractFilePath(FlName));
  ufType:= ufFile;
  Stream := nil;
  try
    if (Pos('http://', FlName) = 1) or (Pos('https://', FlName) = 1) then
    begin
      SL := TStringList.Create;
      try
        ufType := ufHttp;
        HttpGetText(FlName, SL);
        Stream := TStringStream.Create(SL.Text);
        ReadOnly := True;
      finally
        SL.Free;
      end;
    end
    else
    if Pos('ftp://', FlName) = 1 then
    begin
      ufType:= ufFtp;
      FTP := TFTPSend.Create;
      try
        ParseURL(FlName, Prot, User, Pass, Host, Port, Path, Para);
        FTP.TargetHost := Host;
        FTP.TargetPort := Port;
        FTP.UserName := User;
        FTP.Password := Pass;
        if FTP.Login then
        begin
          if FTP.RetrieveFile(Path, False) then
          begin
            Stream := TMemoryStream.Create;
            FTP.DataStream.Seek(0, soFromBeginning);
            Stream.CopyFrom(FTP.DataStream, FTP.DataStream.Size);
            Stream.Seek(0, soFromBeginning);
            ReadOnly := True;
          end;
          FTP.Logout;
        end;
      finally
        FTP.Free;
      end;
    end
    else
    begin
      ReadOnly := False;
      if not FileExists(FlName) then Exception.Create('Файл ' + FlName + ' не найден');
      Stream := TFileStream.Create(FlName, fmOpenRead);
    end;

    MainIDEDesForm.BreakPointList.Clear;
    Doc := nil;
      if Stream <> nil then
      begin
	try
	  ReadXMLFile(Doc, Stream);
	  RootNode := Doc.DocumentElement;
	  if RootNode.TagName = 'CrossEngineApp' then
	  begin
	    Node := RootNode.FindNode('Options');
	    if Node <> nil then
	    begin
	      S := Node.TextContent;
	      ProjectOption := TProjectOptions.Create(Self);

	      C := ProjectOption;
	      MainIDEDesForm.DManager.TextToObject(S, C);
	    end
	    else Exception.Create(ErrorConfigFormat);
	    Node := RootNode.FindNode('Metadata');
	    if Node <> nil then
	    begin
	      Node := Node.FirstChild;
	      MainNode := nil;
	      ReadMetadata(Node, nil);
	      TreeObj.Items.GetFirstNode.Expand(False);
              if ProjectOption.ExternalPath <> '' then
              begin
                ProjectOption.OnUpdateExternalPath := @UpdateExternalPath;
                UpdateExternalPath(ProjectOption);
              end;

              if MainNode <> nil then
	      begin
	        OpenModule(MainNode, True);
	        if MainNode.HasChildren then MainNode.Expand(False);
	      end;
	      ProjectFile := FlName;
	      ProjectModified := False;
	    end
	    else Exception.Create(ErrorConfigFormat);
	  end
	  else
	    Exception.Create(ErrorConfigFormat);
	finally
	  if Doc <> nil then;
	    Doc.Free;
	end;

     end;
  finally
    if Stream <> nil then Stream.Free;
  end;
end;

procedure TProjectManagerForm.GetExternalModule(SL: TStrings);
var
  Node: TTreeNode;
  PNodeObj: PNodeObject;
  LFMType, LFMComponentName, LFMClassName: string;
begin
  Node := TreeObj.Items.GetFirstNode;
  while Node <> nil do
  begin
    if SameText(Node.Text, 'External') then
    begin
      Node := Node.GetFirstChild;
      while Node <> nil do
      begin
        if Node.Data <> nil then
        begin
          PNodeObj := PNodeObject(Node.Data);
          if PNodeObj^.Obj <> '' then
          begin
            ReadLFMHeader(PNodeObj^.Obj, LFMType, LFMComponentName, LFMClassName);
            if LFMClassName = 'TDataModule' then SL.Append(Node.Text);
          end;
        end;
        Node := Node.GetNextSibling;
      end;
      Break;
    end;
    Node := Node.GetNext;
  end;
end;

function TProjectManagerForm.GetExternalNode: TTreeNode;
begin
  if FExternalTreeNode = nil then
  begin
    FExternalTreeNode := GetModuleNode('external', True);
  end;
  Result := FExternalTreeNode;
end;

function TProjectManagerForm.IsExternalOpen: Boolean;
var
  ExtNode, Node: TTreeNode;
  AModule: TPageModule;
begin
  Result := False;
  ExtNode := ExternalTreeNode;
  Node := ExtNode.GetFirstChild;
  while Node <> nil do
  begin
    AModule := EditorForm.GetModuleByName(Node.Text);
    if AModule <> nil then
    begin
      Result := True;
      Break;
    end;
    Node := Node.GetNextSibling;
  end;
end;

procedure TProjectManagerForm.UpdateExternalPath(Sender: TObject);
var
  ExtNode, ParentNode: TTreeNode;
  OldPath: string;
  procedure ClearExtNodeCildren;
  begin
    if IsExternalOpen then
        ShowMessage('Путь к внешним модулям изменился, но нет возможности их перезагрузить, т.к. есть открытые модули. Закройте их и перезагрузите внешние модули вручную или переоткройте проект')
     else
       ExtNode.DeleteChildren;
  end;

begin
  if not ProjectExist then Exit;
  OldPath := ExternalModulePath;
  ExternalModulePath := FindExtPathForModule;
  if ExternalModulePath <> '' then
  begin
    if ExternalModulePath <> '' then
    begin

      ExtNode := ExternalTreeNode;

      if (OldPath <> '') and (OldPath <> ExternalModulePath) then
      begin
        ClearExtNodeCildren;
      end;

      if ExtNode = nil then
      begin
        ParentNode := TreeObj.Items.GetFirstNode;;
        ExtNode := TreeObj.Items.AddChild(ParentNode, 'External');
        ExtNode.ImageIndex := 0;
        ExtNode.StateIndex := 0;
        ExtNode.SelectedIndex := 0;
      end;

      if OldPath <> ExternalModulePath then
        LoadExternalFromPath(ExtNode, ExternalModulePath);
    end;
  end
  else
  begin
    ExtNode := ExternalTreeNode;
    if ExtNode <> nil then
    begin
      ClearExtNodeCildren;
      if not ExtNode.HasChildren then
      begin
        TreeObj.Items.Delete(ExtNode);
      end;
    end;
  end;
end;

function TProjectManagerForm.LoadExternalFromPath(ExternalNode: TTreeNode;
  ExtPath: string): boolean;
type
  TUseFileType = (ufFtp, ufFile);
var
    ExtFileName, CodeExt, FileExt, AFileName, MdFileName, AFName: string;
    ExtDoc: TXMLDocument;
    ExtNode: TDOMElement;
    ExtStream: TMemoryStream;
    ANode: TTreeNode;
    I, N: Integer;
    AutoCreate: Boolean;
    LFMType, LFMComponentName, LFMClassName: string;
    ADesObject: TComponent;
    AForm: TCustomForm;
    URL: string;
    FTP: TFTPSend;
    Prot, User, Pass, Host, Port, Path, Para: string;
    ufType: TUseFileType;
    SL, FL: TStringList;
    PNodeObj: PNodeObject;
    fs: TSearchRec;
begin
  Result := False;
  if ExtPath <> '' then
  begin
    if Pos('ftp://', ExtPath) = 1 then
      ufType:= ufFtp
    else
      ufType:= ufFile;
  end;

  if ufType = ufFtp then
  begin
    ExtDoc := nil;
    FTP := TFTPSend.Create;
    try
      ParseURL(ExtPath, Prot, User, Pass, Host, Port, Path, Para);
      FTP.TargetHost := Host;
      FTP.TargetPort := Port;
      FTP.UserName := User;
      FTP.Password := Pass;
      if FTP.Login then
      begin
        CodeExt := '.pas';
        if ProjectOption.Language = ALterLanguage then CodeExt := '.dp';

        if FTP.ChangeWorkingDir(Path) then
        begin
          if FTP.List(Path, False) then
          begin
            Result := True;
            //ExternalModulePath := ExtPath;
            //ExternalNode := GetModuleNode('External', True);
            //if ExternalNode = nil then
            //begin
            //  TreeObj.Items.AddChild(ParentNode, 'External');
            //  ExternalNode.ImageIndex := 0;
            //  ExternalNode.StateIndex := 0;
            //  ExternalNode.SelectedIndex := 0;
            //end;
            for I := 0 to FTP.FtpList.Count - 1 do
            begin
              ExtFileName := FTP.FtpList.Items[I].FileName;
              FileExt:= ExtractFileExt(ExtFileName);
              if SameText(FileExt, CodeExt) then
              begin
                if FTP.RetrieveFile(ExtFileName, False) then
                begin
                  ExtStream := TMemoryStream.Create;
                  SL := TStringList.Create;
                  try
                    FTP.DataStream.Seek(0, soFromBeginning);
                    SL.LoadFromStream(FTP.DataStream, True);
                    AFileName := ExtractFileNameWithoutExt(ExtFileName);
                    AutoCreate := False;
                    if ProjectOption.AutoCreateExternal.IndexOf(AFileName) >= 0 then AutoCreate := True;
                    MdFileName := AFileName + '.md';
                    ANode := TreeObj.Items.AddChild(ExternalNode, AFileName);
                    ANode.ImageIndex := 2;
                    ANode.SelectedIndex := 2;
                    ANode.StateIndex := 2;
                    New(PNodeObj);
                    PNodeObj^.ExtModule := True;
                    PNodeObj^.ADesignHook := nil;
                    PNodeObj^.Code := SL.Text;
                    ANode.Data := PNodeObj;

                    for N := 0 to FTP.FtpList.Count - 1 do
                    begin

                      AFName:= ExtractFileName(FTP.FtpList.Items[N].FileName);
                      if SameText(MdFileName, AFName) then
                      begin
                        if FTP.RetrieveFile(FTP.FtpList.Items[N].FileName, False) then
                        begin
                          FTP.DataStream.Seek(0, soFromBeginning);
                          ReadXMLFile(ExtDoc, FTP.DataStream);
                          ExtNode := ExtDoc.FirstChild as TDOMElement;
                          if ExtNode.TagName = 'Object' then
                          begin
                            PNodeObj^.ObjClassName := ExtNode.AttribStrings['ClassName'];
                            PNodeObj^.Obj := ExtNode.TextContent;
                            ANode.ImageIndex := 1;
                            ANode.SelectedIndex := 1;
                            ANode.StateIndex := 1;
                          end
                          else
                            raise Exception.Create(AFName + ' не содержит метаданных');
                        end;
                        Break;
                      end;
                    end;

                    if AutoCreate and (PNodeObj^.Obj <> '') then
                    begin
                      ReadLFMHeader(PNodeObj^.Obj, LFMType, LFMComponentName, LFMClassName);
                      if LFMClassName = 'TDataModule' then
                      begin
                        ADesObject := nil;
                        AForm := nil;
                        MainIDEDesForm.DManager.TextToDesignObj(PNodeObj^.Obj, ADesObject, AForm, False);
                        PNodeObj^.ADesignHook := TDesignerHook(AForm.Designer);
                        MainIDEDesForm.DManager.CurrentDesignForm := nil;
                        AForm.Visible := True;
                        PNodeObj^.ADesignHook.DesignClose;
                        Application.ProcessMessages;
                        AForm.Hide;
                      end;
                    end;
                  finally
                    ExtStream.Free;
                    SL.Free;
                    if Assigned(ExtDoc) then ExtDoc.Free;
                  end;
                end;
              end;
            end;
          end;
        end;
        FTP.Logout;
      end;
    finally
      FTP.Free;
    end;
  end
  else
  if ufType = ufFile then
  begin
    if DirectoryExists(ExtPath) then
    begin
      Result := True;
      //ExternalModulePath := ExtPath;
      //ExternalNode := GetModuleNode('External', True);
      //if ExternalNode = nil then
      //begin
      //  ExternalNode := TreeObj.Items.AddChild(ParentNode, 'External');
      //  ExternalNode.ImageIndex := 0;
      //  ExternalNode.StateIndex := 0;
      //  ExternalNode.SelectedIndex := 0;
      //end;
      SL := TStringList.Create;
      FL := TStringList.Create;
      try
        ExtPath := ExtPath + DirectorySeparator;
        CodeExt := '.pas';
        if ProjectOption.Language = ALterLanguage then CodeExt := '.dp';
        if FindFirst(ExtPath  + '*.*', faAnyFile, fs) = 0 then
        begin
          Repeat
            if (fs.Attr and faDirectory) <> faDirectory then
            begin
              ExtFileName := fs.Name;
              FL.Append(ExtFileName);
            end;
          until FindNext(fs) <> 0;
          FindClose(fs);
        end;
        for I := 0 to FL.Count - 1 do
        begin

          ExtFileName := FL.Strings[I];
          FileExt := ExtractFileExt(ExtFileName);
          if SameText(FileExt, CodeExt) then
          begin
            SL.LoadFromFile(ExtPath + ExtFileName);
            AFileName:= ExtractFileNameWithoutExt(ExtFileName);
            AutoCreate := False;
            if ProjectOption.AutoCreateExternal.IndexOf(AFileName) >= 0 then AutoCreate := True;
            ANode := TreeObj.Items.AddChild(ExternalNode, AFileName);
            ANode.ImageIndex := 2;
            ANode.SelectedIndex := 2;
            ANode.StateIndex := 2;
            New(PNodeObj);
            PNodeObj^.ExtModule := True;
            PNodeObj^.ADesignHook := nil;
            PNodeObj^.Code := SL.Text;
            ANode.Data := PNodeObj;
            MdFileName := ExtPath + AFileName + '.md';
            if FileExists(MdFileName) then
            begin
              ReadXMLFile(ExtDoc, MdFileName);
              ExtNode := ExtDoc.FirstChild as TDOMElement;
              if ExtNode.TagName = 'Object' then
              begin
                PNodeObj^.ObjClassName := ExtNode.AttribStrings['ClassName'];
                PNodeObj^.Obj := ExtNode.TextContent;
                ANode.ImageIndex := 1;
                ANode.SelectedIndex := 1;
                ANode.StateIndex := 1;
              end
              else
                raise Exception.Create(AFName + ' не содержит метаданных');
            end;
            if AutoCreate and (PNodeObj^.Obj <> '') then
            begin
              ReadLFMHeader(PNodeObj^.Obj, LFMType, LFMComponentName, LFMClassName);
              if LFMClassName = 'TDataModule' then
              begin
                ADesObject := nil;
                AForm := nil;
                MainIDEDesForm.DManager.TextToDesignObj(PNodeObj^.Obj, ADesObject, AForm, False);
                PNodeObj^.ADesignHook := TDesignerHook(AForm.Designer);
                MainIDEDesForm.DManager.CurrentDesignForm := nil;
                AForm.Visible := True;
                PNodeObj^.ADesignHook.DesignClose;
                Application.ProcessMessages;
                AForm.Hide;
              end;
            end;
          end;
        end;
        ExternalNode.CustomSort(@SortTreeObj);
      finally
        SL.Free;
        FL.Free;
      end;
    end;
  end;
end;

end.

