////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//        Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru               //
//                                                                            //
//                            Under LGPL license                              //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
unit EasyIpc; // Easy single thread duplex text InterProcess Communication pipe.

{$IFDEF FPC}{$mode objfpc}{$ENDIF}{$H+}{$modeswitch advancedrecords}

{$WARN 6058 off : Call to subroutine "$1" marked as inline is not inlined}
{$WARN 4079 off : Converting the operands to "$1" before doing the add could prevent overflow errors.}

interface

uses
 {$IFDEF UNIX} baseunix, unix, termio, sockets, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, lclproc, lclintf, strutils,
 expandenvvar;

{$IFNDEF FPC}
const                                                                            // Constants for old Delphi
 LineEnding         = #13#10; sLineBreak = LineEnding;                           // LineEnding, sLineBreak
 DirectorySeparator = '\';    PathDelim  = DirectorySeparator;                   // DirectorySeparator
 PathSeparator      = ';';    PathSep    = PathSeparator;                        // PathSeparator
type
 PtrInt = LongInt; PtrUint = LongWord; IntPtr = PtrInt; UintPtr = PtrUint;       // Pointer-size integers
{$ENDIF}

{$IFDEF UNIX}
const // Linux-specific socket ioctls
 SIOCINQ  = FIONREAD; // input  queue size
 SIOCOUTQ =	TIOCOUTQ; // output queue size (not sent + not acked)
{$ENDIF ~UNIX}

{$IFDEF UNIX}
const
 INVALID_HANDLE_VALUE = THandle(-1);
{$ENDIF ~UNIX}

const                // Succeed error codes.
 NO_ERROR      = 0;  // No error found.
 ERROR_SUCCESS = 0;  // Success status.

const
 KiloByte    = 1024;
 MegaByte    = KiloByte * KiloByte;
 OS_PIPE_BUF = KiloByte*4;
 EOL         = LineEnding;

/////////////////////////////////////////////////////////////////////////////////
// EasyIPC is easy, single thread, duplex (bidirect) Inter Process Communication.
// It's peer to peer clent-server text communication channel. Use HEX encoding to
// transfer binary data. Use polling (for example, with timer) to Send/Recv data.
// EasyIPC makes all IO operations in non-blocking manner so thread never blocks.
// At least it is true if TimeOut=0.
//
// EasyIpc_Init(PipeName,Options) - create IPC object and return his handle or 0.
//  PipeName - Host\Name for IPC client, Name for IPC server. "." mean Localhost.
//             Name should be system-unique identifier string. Example: .\DemoIpc
//  Options  - LineEnding delimited text as OptionName=OptionValue, where options
//             is {TimeOut,RxBuffSize,TxBuffSize,LogsHistory}.
//
// EasyIpc_Free(hIpc) - Free (delete) EasyIpc object hIpc (EasyIpc_Init created).
//
// EasyIpc_Poll(hIpc) - Polling I/O procedure to be called periodically by timer.
//  EasyIpc_Poll(hIpc) call is equivalent to EasyIpc_Recv(hIpc,0), but faster.
//
// EasyIpc_Send(hIpc,TextLines)  - Send TextLines to transmitter FIFO and pipe.
//
// EasyIpc_Recv(hIpc,Count)  - Receive text (upto Count byte) from receiver FIFO.
//
// EasyIpc_Ctrl(hIpc,Request)  - Query or Control IPC object via Request command.
//  Request are "Name=Value" to set or "Name" to get the value of parameter Name.
//  Name can be {Connected,IsServer,IsClient,FileName,PipeName,HostName,BaseName,
//  Handle,TimeOut,RxBuffSize,TxBuffSize,RxLost,TxLost,RxFifoLimit,TxFifoLimit,
//  RxTotal,TxTotal,RxLength,TxLength,LogsHistory}
//
/////////////////////////////////////////////////////////////////////////////////
function EasyIpc_Init(const PipeName,Options:LongString):PtrUint;
function EasyIpc_Free(hIpc:PtrUint):Boolean;
function EasyIpc_Poll(hIpc:PtrUint):Boolean;
function EasyIpc_Send(hIpc:PtrUint; const TextLines:LongString):Boolean;
function EasyIpc_Recv(hIpc:PtrUint; Count:Integer):LongString;
function EasyIpc_Ctrl(hIpc:PtrUint; const Request:LongString):LongString;

/////////////////////////////////////////////////////////////////////////////////
// EasyIpc is simplified TEasyPipe's wrapper - specially to embed to interpreters
// like DieselPascal, DaqPascal, DaqScript etc.  Most codes located in TEasyPipe.
/////////////////////////////////////////////////////////////////////////////////

const
 DefPipeTimeOut     = 0;                                                         // Default pipe timeout
 MinPipeBuffSize    = 1024*4;                                                    // Minimal pipe buffer size
 DefPipeBuffSize    = 1024*64;                                                   // Default pipe buffer size
 MinPipeFifoLimit   = 1024*128;                                                  // Minimal pipe FIFO limit
 DefPipeFifoLimit   = 1024*1024*4;                                               // Default pipe FIFO limit
 MaxPipeFifoLimit   = 1024*1024*64;                                              // Maximal pipe FIFO limit
 DefPipeLogsHistory = 0;                                                         // Default history for list of logs

type
 TFifoRec = record
  Buff:LongString;
  Size,Limit:SizeInt;
  Total,Lost:Int64;
  procedure Clear;
  function  Count:SizeInt;
  function  Space:SizeInt;
  procedure SizeLimit(aSize,aLimit:SizeInt);
  function  GetText(Len:SizeInt=MaxInt):LongString;
  function  PutText(const aData:LongString):Boolean;
 end;
 TEasyPipe = class(TObject)                                                      // Named pipe wrapper class
 private
  myHostName    : LongString;                                                    // Peer host name (empty for server)
  myBaseName    : LongString;                                                    // Pipe base name (logic identifier)
  myFileName    : LongString;                                                    // Pipe file name (system dependent)
  myRxFifo      : TFifoRec;                                                      // Receiver    FIFO
  myTxFifo      : TFifoRec;                                                      // Transmitter FIFO
  myConnected : Boolean;                                                        // Connected?
  {$IFDEF WINDOWS}
  myFile      : THandle;                                                        // Pipe file handle
  myRxPending : Boolean;                                                        // Receiver pending IO
  myTxPending : Boolean;                                                        // Transmitter pending IO
  myRxOverlap : TOverlapped;                                                    // Receiver overlap
  myTxOverlap : TOverlapped;                                                    // Transmitter overlap
  mySecDesc   : TSecurityDescriptor;                                            // Security descriptor
  mySecAttr   : TSecurityAttributes;                                            // Security attributes
  {$ENDIF ~WINDOWS}
  {$IFDEF UNIX}
  mySock      : THandle;                                                        // Socket handle
  myConn      : THandle;                                                        // Connection (server)
  myRxBuff    : LongString;                                                     // Receiver buffer
  myTxBuff    : LongString;                                                     // Transmitter buffer
  {$ENDIF ~UNIX}
  myTimeOut      : Integer;                                                     // Timeout to check connection
  myRxLost       : Int64;                                                       // Last value of RxFifo.Lost
  myTxLost       : Int64;                                                       // Last value of TxFifo.Lost
  myLastTick     : QWord;                                                       // Every second tick
  myLastCheck    : QWord;                                                       // Time when last check done
  myLogsHistory  : Integer;                                                     // Logs history limit
  myLogsText     : TStringList;                                                 // Logs history text
  myListenPeriod : Integer;                                                     // Period to listen/accept incoming connections
  myRxBuffer     : LongString;                                                  // Receiver    buffer
  myTxBuffer     : LongString;                                                  // Transmitter buffer
  myPollingDelay : Integer;
 private
  function    GetFileName:LongString;                                           // As \\host\pipe\name
  function    GetPipeName:LongString;                                           // As host\name
  function    GetHostName:LongString;                                           // As host
  function    GetBaseName:LongString;                                           // As name
  function    InitFileName:LongString;
  function    GetHandle(i:Integer=0):THandle;
  function    GetTimeOut:Integer;
  function    GetIsServer:Boolean;
  function    GetIsClient:Boolean;
  function    GetConnected:Boolean;
  function    GetRxBuffSize:Integer;
  function    GetTxBuffSize:Integer;
  function    GetRxFifoLimit:Integer;
  procedure   SetRxFifoLimit(aLimit:Integer);
  function    GetTxFifoLimit:Integer;
  procedure   SetTxFifoLimit(aLimit:Integer);
  function    GetLogsText:TStringList;
  function    GetLogsHistory:Integer;
  procedure   SetLogsHistory(aHistory:Integer);
  function    GetRxLost:Int64;
  function    GetTxLost:Int64;
  function    GetRxTotal:Int64;
  function    GetTxTotal:Int64;
  function    GetRxLength:Integer;
  function    GetTxLength:Integer;
  function    GetLogsCount:Integer;
  function    GetLogsTextMove:LongString;
  function    GetLogsTextCopy:LongString;
  function    GetListenPeriod:Integer;
  procedure   SetListenPeriod(aPeriod:Integer);
  procedure   Report(const What:LongString; Code:Integer); // Add report to logs
  procedure   Close(aErrorCode:Integer=ERROR_SUCCESS; aRxLost:Integer=0; aTxLost:Integer=0);
 public
  constructor Create(const aHost : LongString;
                           const aName : LongString;
                           aDelay      : Integer;
                           aRxFifoSize : Integer;
                           aTxFifoSize : Integer;
                           aTimeOut    : Integer);
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
  destructor  Destroy; override;
 public
  procedure   Poll;                                                              // To be called by timer
  property    FileName     : LongString  read GetFileName;                       // Pipe UNC name as \\host\pipe\name
  property    PipeName     : LongString  read GetPipeName;                       // As host\name (client) or name (server)
  property    HostName     : LongString  read GetHostName;                       // Get host name
  property    BaseName     : LongString  read GetBaseName;                       // Get base name
  property    TimeOut      : Integer     read GetTimeOut;                        // TimeOut, ms
  property    IsServer     : Boolean     read GetIsServer;                       // Is Server mode?
  property    IsClient     : Boolean     read GetIsClient;                       // Is Client mode?
  property    Connected    : Boolean     read GetConnected;                      // Connected?
  property    RxBuffSize   : Integer     read GetRxBuffSize;                     // Rx buffer size
  property    TxBuffSize   : Integer     read GetTxBuffSize;                     // Tx buffer size
  property    RxLost       : Int64       read GetRxLost;                         // Rx lost bytes
  property    TxLost       : Int64       read GetTxLost;                         // Tx lost bytes
  property    RxTotal      : Int64       read GetRxTotal;                        // Rx received bytes
  property    TxTotal      : Int64       read GetTxTotal;                        // Tx written  bytes
  property    RxLength     : Integer     read GetRxLength;                       // Rx FIFO data length
  property    TxLength     : Integer     read GetTxLength;                       // Tx FIFO data length
  property    RxFifoLimit  : Integer     read GetRxFifoLimit write SetRxFifoLimit; // Rx Fifo size limit
  property    TxFifoLimit  : Integer     read GetTxFifoLimit write SetTxFifoLimit; // Tx Fifo size limit
  property    LogsText     : TStringList read  GetLogsText;                      // Log events text
  property    LogsHistory  : Integer     read GetLogsHistory write SetLogsHistory; // History for list of logs
  property    LogsCount    : Integer     read GetLogsCount;                      // Count of logs in list.
  property    LogsTextMove : LongString  read GetLogsTextMove;                   // Get logs as text and clear
  property    LogsTextCopy : LongString  read GetLogsTextCopy;                   // Get logs as text and stay
  property    ListenPeriod : Integer     read  GetListenPeriod write SetListenPeriod; // Period to listen/accept, ms
  function    GetRxFifo(Count:Integer=MaxInt):LongString;                        // Receiver    FIFO readout
  function    GetTxFifo(Count:Integer=MaxInt):LongString;                        // Transmitter FIFO readout
  procedure   PutRxFifo(const Data:LongString);                                  // Receiver    FIFO writing
  procedure   PutTxFifo(const Data:LongString);                                  // Transmitter FIFO writing
  function    GetProperties:LongString;                                          // Properties
  function    Ctrl(const arg:LongString):LongString;                             // Control
  procedure   Clear(const What:LongString);                                      // Clear('RxFifo,TxFifo,RxLost,TxLost,Logs')
  procedure   BugReport(E:Exception; O:TObject; M:LongString);
 public                                                                         // General I/O:
  function    Send(const aData:LongString):LongInt;                             // Send a data to pipe
  function    Recv(aMaxLen:LongInt=MaxInt):LongString;                          // Recveive a data  from pipe
 private
  class var TheDefDelay:Integer;
  class function  GetDefDelay:Integer; static;
  class procedure SetDefDelay(aDelay:Integer); static;
  class var TheDefTimeout:Integer;
  class function  GetDefTimeout:Integer; static;
  class procedure SetDefTimeout(aTimeout:Integer); static;
  class var TheDefListenPeriod:Integer;
  class function  GetDefListenPeriod:Integer; static;
  class procedure SetDefListenPeriod(aPeriod:Integer); static;
  class var TheDefFifoSize:Integer;
  class function  GetDefFifoSize:Integer; static;
  class procedure SetDefFifoSize(aFifoSize:Integer); static;
  class var TheDefHistory:Integer;
  class function  GetDefHistory:Integer; static;
  class procedure SetDefHistory(aHistory:Integer); static;
 public
  class function  ValidateDelay(aDelay:Integer):Integer; static;
  class function  ValidateTimeout(aTimeOut:Integer):Integer; static;
  class function  ValidateListenPeriod(aPeriod:Integer):Integer; static;
  class function  ValidateFifoSize(aFifoSize:Integer):Integer; static;
  class function  ValidateHistory(aHistory:Integer):Integer; static;
 public // Default parameters for all pipes
  class property DefDelay    : Integer read GetDefDelay    write SetDefDelay;   // Polling period, ms
  class property DefTimeout  : Integer read GetDefTimeout  write SetDefTimeOut; // Timeout for IO, ms
  class property DefListenPeriod : Integer read GetDefListenPeriod write SetDefListenPeriod;
  class property DefFifoSize : Integer read GetDefFifoSize write SetDefFifoSize;// Fifo size, bytes
  class property DefHistory  : Integer read GetDefHistory  write SetDefHistory; // Logs histiry, lines
 protected // Helper functions for internal use
  class procedure Note(const Msg:LongString); static; inline;
  class function  ValidHandle(fd:THandle):Boolean; static; inline;
  class procedure InvalidateHandle(var fd:THandle); static; inline;
 end;

function  NewEasyPipe(const aPipeName    : LongString;                           // Host\Name(client) or Name(server)
                            aRxBuffSize  : Integer         = DefPipeBuffSize;    // Rx Buffer size
                            aTxBuffSize  : Integer         = DefPipeBuffSize;    // Tx Buffer size
                            aTimeOut     : Integer         = DefPipeTimeOut;     // Timeout for connections
                            aLogsHistory : Integer         = DefPipeLogsHistory  // History for list of logs
                                       ) : TEasyPipe; overload;

function  NewEasyPipe(const aHost : LongString;
                      const aName : LongString;
                      aDelay      : Integer;
                      aRxFifoSize : LongInt;
                      aTxFifoSize : LongInt;
                      aTimeOut    : Integer
                                ) : TEasyPipe;  overload;

const FullEasyPipeList:TThreadList=nil; // Full list of all pipes

function IsUnix    : Boolean; inline;
function IsLinux   : Boolean; inline;
function IsWindows : Boolean; inline;

implementation

 /////////////////////////////////////////////////////
 // Private Dictionary for fast string identification.
 /////////////////////////////////////////////////////
type
 TStringIdentifier = (
 sid_Unknown,
 ////////////////////// Properties ReadOnly
 sid_Connected,
 sid_IsServer,
 sid_IsClient,
 sid_FileName,
 sid_PipeName,
 sid_HostName,
 sid_BaseName,
 sid_Handle,
 sid_TimeOut,
 sid_RxLost,
 sid_TxLost,
 sid_RxTotal,
 sid_TxTotal,
 sid_RxLength,
 sid_TxLength,
 sid_RxFifoLimit,
 sid_TxFifoLimit,
 sid_LogsCount,
 ////////////////////// Properties Writable
 sid_LogsHistory,
 sid_RxBuffSize,
 sid_TxBuffSize,
 sid_RxPipeSize,
 sid_TxPipeSize,
 sid_Polling,
 sid_Priority,
 sid_PollingDelay,
 sid_PollingPeriod,
 sid_PollingPriority,
 sid_Listen,
 sid_ListenDelay,
 sid_ListenPeriod,
 ////////////////////// Special IO
 sid_LogsTextMove,
 sid_LogsTextCopy,
 sid_Clear,
 sid_RxRecv,
 sid_TxSend,
 ////////////////////// Properties End
 sid_Asterisk,
 sid_Question,
 sid_Unused
 );

const
 Dictionary:TStringList=nil;

procedure FreeDictionary;
begin
 FreeAndNil(Dictionary);
end;

procedure InitDictionary;
 procedure AddSid(const key:LongString; sid:TStringIdentifier);
 begin
  Dictionary.AddObject(key,TObject(PtrInt(Ord(sid))));
 end;
begin
 if (Dictionary<>nil) then Exit;
 Dictionary:=TStringList.Create;
 Dictionary.Sorted:=true;
 /////////////////////////////////////////////
 // Dictionary for fast strings identification
 /////////////////////////////////////////////
 AddSid( 'Connected'           , sid_Connected);
 AddSid( 'IsServer'            , sid_IsServer);
 AddSid( 'IsClient'            , sid_IsClient);
 AddSid( 'FileName'            , sid_FileName);
 AddSid( 'PipeName'            , sid_PipeName);
 AddSid( 'HostName'            , sid_HostName);
 AddSid( 'BaseName'            , sid_BaseName);
 AddSid( 'Handle'              , sid_Handle);
 AddSid( 'TimeOut'             , sid_TimeOut);
 AddSid( 'RxBuffSize'          , sid_RxBuffSize);
 AddSid( 'TxBuffSize'          , sid_TxBuffSize);
 AddSid( 'RxLost'              , sid_RxLost);
 AddSid( 'TxLost'              , sid_TxLost);
 AddSid( 'RxTotal'             , sid_RxTotal);
 AddSid( 'TxTotal'             , sid_TxTotal);
 AddSid( 'RxLength'            , sid_RxLength);
 AddSid( 'TxLength'            , sid_TxLength);
 AddSid( 'RxFifoLimit'         , sid_RxFifoLimit);
 AddSid( 'TxFifoLimit'         , sid_TxFifoLimit);
 AddSid( 'LogsCount'           , sid_LogsCount);
 AddSid( 'LogsHistory'         , sid_LogsHistory);
 AddSid( 'RxBuffSize'          , sid_RxBuffSize);
 AddSid( 'TxBuffSize'          , sid_TxBuffSize);
 AddSid( 'RxPipeSize'          , sid_RxPipeSize);
 AddSid( 'TxPipeSize'          , sid_TxPipeSize);
 AddSid( 'Polling'             , sid_Polling);
 AddSid( 'Priority'            , sid_Priority);
 AddSid( 'PollingDelay'        , sid_PollingDelay);
 AddSid( 'PollingPeriod'       , sid_PollingPeriod);
 AddSid( 'PollingPriority'     , sid_PollingPriority);
 AddSid( 'Listen'              , sid_Listen);
 AddSid( 'ListenDelay'         , sid_ListenDelay);
 AddSid( 'ListenPeriod'        , sid_ListenPeriod);
 AddSid( 'LogsTextMove'        , sid_LogsTextMove);
 AddSid( 'LogsTextCopy'        , sid_LogsTextCopy);
 AddSid( 'Clear'               , sid_Clear);
 AddSid( 'RxRecv'              , sid_RxRecv);
 AddSid( 'TxSend'              , sid_TxSend);
 AddSid( '*'                   , sid_Asterisk);
 AddSid( '?'                   , sid_Question);
end;

function Identify(const key:LongString):TStringIdentifier;
var i,sid:Integer;
begin
 if (Dictionary=nil) then InitDictionary;
 i:=Dictionary.IndexOf(key);
 if (i<0) then sid:=Ord(sid_Unknown) else sid:=PtrInt(Dictionary.Objects[i]);
 if (sid>=Ord(Low(TStringIdentifier))) and (sid<=Ord(High(TStringIdentifier)))
 then Result:=TStringIdentifier(sid)
 else Result:=sid_Unknown;
end;

const
 sid_Ctrl_Hidden   = [sid_LogsTextMove..sid_TxSend];
 sid_Ctrl_Readable = [Succ(sid_Unknown)..Pred(sid_Asterisk)]-sid_Ctrl_Hidden;
 sid_Ctrl_Writable = [sid_LogsHistory..Pred(sid_Asterisk)]-sid_Ctrl_Hidden;

///////////////////////////
// General purpose routines
///////////////////////////

function IsUnix    : Boolean; begin Result := {$IFDEF UNIX}    true {$ELSE} false {$ENDIF}; end;
function IsLinux   : Boolean; begin Result := {$IFDEF LINUX}   true {$ELSE} false {$ENDIF}; end;
function IsWindows : Boolean; begin Result := {$IFDEF WINDOWS} true {$ELSE} false {$ENDIF}; end;

function PointerToPtrInt(P:Pointer):PtrInt;
var I : PtrInt absolute P;
begin
 Result:=I;
end;

function PointerToPtrUInt(P:Pointer):PtrUInt;
var I : PtrUInt absolute P;
begin
 Result:=I;
end;

function PtrIntToPointer(I:PtrInt):Pointer;
var P : Pointer absolute I;
begin
 Result:=P;
end;

function PtrUIntToPointer(I:PtrUInt):Pointer;
var P : Pointer absolute I;
begin
 Result:=P;
end;

procedure SafeMove(const Source; out Dest; Count: SizeInt);
begin
 if (Count>0) and (@Source<>nil) and (@Dest<>nil) then Move(Source,Dest,Count);
end;

procedure SafeFillChar(out X; Count:SizeInt; Value:Byte); overload;
begin
 if (Count>0) and (@X<>nil) then FillChar(X,Count,Value);
end;

procedure SafeFillChar(out X; Count:SizeInt; Value:Char); overload;
begin
 if (Count>0) and (@X<>nil) then FillChar(X,Count,Value);
end;

function StringBuffer(Leng:SizeInt; Filler:Char=#0):LongString; overload;
begin
 Result:='';
 if (Leng<=0) then Exit;
 SetLength(Result,Leng);
 FillChar(Pointer(Result)^,Length(Result),Filler);
end;

function StringBuffer(Buff:Pointer; Leng:SizeInt):LongString; overload;
begin
 Result:='';
 if (Leng>0) then
 if Assigned(Buff)
 then SetString(Result,Buff,Leng)
 else Result:=StringBuffer(Leng);
end;

function StringBuffer(const Source:LongString):LongString; overload;
begin
 if (Source<>'')
 then Result:=StringBuffer(PChar(Source),Length(Source))
 else Result:='';
end;


function  IsEmptyStr(const S:LongString):Boolean;
var i:SizeInt;
begin
 if (S<>'') then for i:=1 to Length(S) do if (S[i]>' ') then Exit(false);
 Result:=true;
end;

function  IsNonEmptyStr(const S:LongString):Boolean;
var i:SizeInt;
begin
 if (S<>'') then for i:=1 to Length(S) do if (S[i]>' ') then Exit(true);
 Result:=false;
end;

function WordIndex(const Name,Str:LongString; const Delims:TSysCharSet; CaseSensitive:Boolean=false):Integer;
var i:Integer; Match:Boolean;
begin
 Result:=0;
 for i:=1 to WordCount(Str,Delims) do begin
  if CaseSensitive
  then Match:=SameStr(ExtractWord(i,Str,Delims),Name)
  else Match:=SameText(ExtractWord(i,Str,Delims),Name);
  if Match then begin
   Result:=i;
   Break;
  end;
 end;
end;

procedure AddSub(var S:LongString; const sub:LongString);
begin
 if (S<>'') and (Trim(sub)<>'') then S:=IncludeTrailingPathDelimiter(S)+Trim(sub);
end;

function xdg_base_directory(const id,def,sub:LongString):LongString;
var i:Integer; pattern,item:LongString;
begin
 Result:=''; pattern:='';
 if (id<>'') then pattern:=Trim(GetEnvironmentVariable(id));
 if (pattern='') and (def<>'') then pattern:=Trim(ExpandEnvironmentVariables(def));
 for i:=1 to WordCount(pattern,[PathSep]) do begin
  item:=Trim(ExtractWord(i,pattern,[PathSep]));
  AddSub(item,sub);
  if (Result='')
  then Result:=item
  else Result:=Result+PathSep+item;
 end;
end;

function XDG_RUNTIME_DIR(const sub:LongString):LongString;
begin
 {$IFDEF UNIX}
 Result:=xdg_base_directory('XDG_RUNTIME_DIR','/run/user/'+IntToStr(FpGetuid),sub);
 {$ELSE}
 Result:='';
 {$ENDIF}
end;

function HasFlags(Mode,Flags:LongInt):Boolean;
begin
 Result:=((Mode and Flags)<>0);
end;

procedure LiftFlags(var Mode:LongInt; Flags:LongInt; Up:Boolean=true);
begin
 if Up then Mode:=Mode or Flags else Mode:=Mode and not Flags;
end;

function FileSetNonBlockFlag(fd:THandle; State:Boolean=true):Boolean;
{$IFDEF UNIX}var Flags:LongInt;{$ENDIF}
begin
 {$IFDEF UNIX}
 Flags:=fpfcntl(fd,F_GETFL);
 LiftFlags(Flags,O_NONBLOCK,State);
 Result:=(fpfcntl(fd,F_SETFL,Flags)=NO_ERROR);
 {$ELSE}
 Result:=false;
 {$ENDIF}
end;

function FileSetCloseOnExec(fd:THandle; State:Boolean=true):Boolean;
{$IFDEF UNIX}var Flags:LongInt; const FD_CLOEXEC=1;{$ENDIF}
begin
 {$IFDEF UNIX}
 Flags:=fpfcntl(fd,F_GETFD);
 if (Flags=-1) then Exit(false);
 LiftFlags(Flags,FD_CLOEXEC,State);
 Result:=(fpfcntl(fd,F_SETFD,Flags)=NO_ERROR);
 {$ELSE}
 Result:=false;
 {$ENDIF}
end;

function ePendingFileOperation(ErrorCode:Integer):Boolean;
begin
 {$if defined (WINDOWS)}
 Result:=(ErrorCode=ERROR_IO_INCOMPLETE) // Overlapped I/O event is not in a signaled state.
      or (ErrorCode=ERROR_IO_PENDING);   // Overlapped I/O operation is in progress.
 {$elseif defined(UNIX)}
 Result:=(ErrorCode=ESysEAGAIN)       // Try again
      or (ErrorCode=ESysEWOULDBLOCK)  // Try again
      or (ErrorCode=ESysEINPROGRESS); // Operation now in progress
 {$else}
 Result:=false;
 {$endif}
end;

function ExtractNameValuePair(const arg:LongString; out Name,Value:LongString;
                              const Sign:Char='='; Mode:Integer=3):Integer;
var p:Integer;
begin
 p:=Pos(Sign,arg);
 if (p>0) then begin
  Name:=Copy(arg,1,p-1);
  Value:=Copy(arg,p+1,Length(arg)-p);
 end else begin
  Name:=arg;
  Value:='';
 end;
 if HasFlags(Mode,1) and (Name<>'') then Name:=Trim(Name);
 if HasFlags(Mode,2) and (Value<>'') then Value:=Trim(Value);
 Result:=p;
end;

const
 ScanSpaces = [#0..' ',',',';','='];

procedure BugReport(E:Exception; O:TObject; M:LongString);
begin
end;

 //////////////////////////
 // TFifoRec implementation
 //////////////////////////

procedure TFifoRec.Clear;
begin
 Buff:=''; Total:=0; Lost:=0;
end;

function TFifoRec.Count:SizeInt;
begin
 Result:=Length(Buff);
end;

function TFifoRec.Space:SizeInt;
begin
 Result:=Max(0,Limit-Count);
end;

procedure TFifoRec.SizeLimit(aSize,aLimit:SizeInt);
begin
 Size:=EnsureRange(aSize,MinPipeBuffSize,MaxPipeFifoLimit);
 Limit:=EnsureRange(aLimit,MinPipeFifoLimit,MaxPipeFifoLimit)
end;

function TFifoRec.GetText(Len:SizeInt=MaxInt):LongString;
begin
 Result:='';
 if (Len<=0) then Exit;
 Len:=EnsureRange(Len,0,Count);
 if (Len>0) then begin
  Result:=Copy(Buff,1,Len);
  Delete(Buff,1,Len);
 end;
end;

function TFifoRec.PutText(const aData:LongString):Boolean;
begin
 Result:=False;
 if (aData='') then Exit;
 if (Length(aData)+Length(Buff)>Limit) then begin
  Inc(Lost,Length(aData));
  Exit;
 end;
 Inc(Total,Length(aData));
 Buff:=Buff+aData;
 Result:=True;
end;

 ///////////////////////////
 // TEasyPipe implementation
 ///////////////////////////

function TEasyPipe.GetHostName:LongString;
begin
 if Assigned(Self)
 then Result:=myHostName
 else Result:='';
end;

function TEasyPipe.GetBaseName:LongString;
begin
 if Assigned(Self)
 then Result:=myBaseName
 else Result:='';
end;

function TEasyPipe.GetPipeName:LongString;
begin
 Result:='';
 if Assigned(Self) then begin
  if (myHostName='')
  then Result:=myBaseName
  else Result:=IncludeTrailingBackslash(myHostName)+myBaseName;
 end;
end;

function TEasyPipe.GetFileName:LongString;
begin
 if Assigned(Self)
 then Result:=myFileName
 else Result:='';
end;

function TEasyPipe.InitFileName:LongString;
var dir:LongString;
begin
 Result:='';
 if Assigned(Self) then begin
  Result:=myBaseName;
  if IsWindows then begin
   if (myHostName='')
   then Result:=Format('\\%s\pipe\%s',['.',myBaseName])
   else Result:=Format('\\%s\pipe\%s',[myHostName,myBaseName]);
  end;
  if IsUnix then begin
   dir:=XDG_RUNTIME_DIR('namedpipes');
   if not DirectoryExists(dir) then MkDir(Dir);
   if not DirectoryExists(dir) then dir:=GetTempDir;
   Result:=IncludeTrailingPathDelimiter(dir)+myBaseName+'.socket';
  end;
 end;
end;

function TEasyPipe.GetIsServer:Boolean;
begin
 if Assigned(Self)
 then Result:=(myHostName='')
 else Result:=false;
end;

function TEasyPipe.GetIsClient:Boolean;
begin
 if Assigned(Self)
 then Result:=(myHostName<>'')
 else Result:=false;
end;

function TEasyPipe.GetConnected:Boolean;
begin
 if Assigned(Self)
 then Result:=myConnected
 else Result:=false;
end;

function TEasyPipe.GetTimeout:Integer;
begin
 if Assigned(Self)
 then Result:=myTimeout
 else Result:=0;
end;

function TEasyPipe.GetHandle(i:Integer=0):THandle;
begin
 Result:=0;
 if Assigned(Self) then begin
  {$IFDEF WINDOWS}
  Result:=myFile;
  {$ENDIF ~WINDOWS}
  {$IFDEF UNIX}
  case i of
   0: Result:=mySock;
   1: Result:=myConn;
  end;
  {$ENDIF ~UNIX}
 end;
end;

function TEasyPipe.GetLogsText:TStringList;
begin
 if Assigned(Self)
 then Result:=myLogsText
 else Result:=nil;
end;

function TEasyPipe.GetLogsHistory:Integer;
begin
 if Assigned(Self)
 then Result:=myLogsHistory
 else Result:=0;
end;

function TEasyPipe.GetLogsCount:Integer;
begin
 if Assigned(Self) and Assigned(myLogsText)
 then Result:=myLogsText.Count
 else Result:=0;
end;

function TEasyPipe.GetLogsTextMove:LongString;
begin
 if Assigned(Self) and Assigned(myLogsText)
 then begin Result:=myLogsText.Text; myLogsText.Clear; end
 else Result:='';
end;

function TEasyPipe.GetLogsTextCopy:LongString;
begin
 if Assigned(Self) and Assigned(myLogsText)
 then Result:=myLogsText.Text
 else Result:='';
end;

function TEasyPipe.GetListenPeriod:Integer;
begin
 if Assigned(Self)
 then Result:=myListenPeriod
 else Result:=0;
end;

procedure TEasyPipe.SetListenPeriod(aPeriod:Integer);
begin
 if Assigned(Self)
 then myListenPeriod:=IfThen(aPeriod>0,aPeriod,DefListenPeriod);
end;

procedure TEasyPipe.BugReport(E:Exception; O:TObject; M:LongString);
var What,Who:LongString; Code:Integer;
begin
 EasyIpc.BugReport(E,O,M);
 if (Self=nil) or (E=nil) then Exit;
 Who:='Unknown'; if Assigned(O) then Who:=O.ClassName;
 What:=E.ClassName+': from '+Who
 +' note '+AnsiQuotedStr(M,'"')
 +' hint '+AnsiQuotedStr(E.Message,'"');
 Code:=E.HelpContext;
 Report(What,Code);
end;

const
 StdDateTimeFormatMs = 'yyyy.mm.dd-hh:nn:ss.zzz'; // Date Time Format with ms

procedure TEasyPipe.Report(const What:LongString; Code:Integer);
var Line:LongString; When:Double;
begin
 if Assigned(Self) then
 if Assigned(LogsText) then
 try
  When:=Now;
  if (LogsHistory>0) then begin
   Line:=FormatDateTime(StdDateTimeFormatMs,When)+' => '
        +PipeName+': '+What+', '+SysErrorMessage(Code);
   LogsText.Add(Line);
  end;
  while (LogsText.Count>LogsHistory) do LogsText.Delete(0);
 except
  on E:Exception do BugReport(E,Self,'Report');
 end;
end;

function TEasyPipe.Send(const aData:LongString):LongInt;
begin
 if Assigned(Self) and (aData<>'')
 then Result:=Ord(myTxFifo.PutText(aData))*Length(aData)
 else Result:=0;
end;

function TEasyPipe.Recv(aMaxLen:LongInt=MaxInt):LongString;
begin
 if Assigned(Self) and (aMaxLen>0)
 then Result:=myRxFifo.GetText(aMaxLen)
 else Result:='';
end;

procedure TEasyPipe.Close(aErrorCode:Integer; aRxLost:Integer=0; aTxLost:Integer=0);
begin
 if Assigned(Self) then
 try
  {$IFDEF WINDOWS}
  if ValidHandle(myFile) then begin                                            // If file opened
   CloseHandle(myFile);                                                        // Close file
   InvalidateHandle(myFile);                                                   // Mark it closed
   myConnected:=false;                                                         // No connection
   myRxPending:=false;                                                         // No pending read
   myTxPending:=false;                                                         // No pending write
   SafeFillChar(myRxOverlap,sizeof(myRxOverlap),0);                            // Clear read overlap
   SafeFillChar(myTxOverlap,sizeof(myTxOverlap),0);                            // Clear write overlap
  end;
  {$ENDIF ~WINDOWS}
  {$IFDEF UNIX}
  if ValidHandle(myConn) then begin                                            // Server connection opened?
   FpShutdown(myConn,SHUT_RDWR);                                               // Shutdown connection
   FpClose(myConn);                                                            // Close connection
   InvalidateHandle(myConn);                                                   // Mark it closed
  end;
  if ValidHandle(mySock) then begin                                            // If socket opened
   FpShutdown(mySock,SHUT_RDWR);                                               // Shutdown connection
   FpClose(mySock);                                                            // Close socket
   if IsServer then FpUnlink(FileName);                                        // Unlink socket file
   InvalidateHandle(mySock);                                                   // Mark it closed
  end;
  myConnected:=false;                                                          // No connection
  Inc(aRxLost,Length(myRxBuff));                                               // This Rx data lost
  Inc(aTxLost,Length(myTxBuff));                                               // This Tx data lost
  myRxBuff:='';                                                                // Clear Rx Buffer
  myTxBuff:='';                                                                // Clear Tx Buffer
  {$ENDIF ~UNIX}
  if (aErrorCode<>ERROR_SUCCESS) or (aRxLost+aTxLost<>0) then begin            // If error occured
   Inc(myRxFifo.Lost,Length(myRxFifo.Buff)+aRxLost); myRxFifo.Buff:='';        // Receiver data lost
   Inc(myTxFifo.Lost,Length(myTxFifo.Buff)+aTxLost); myTxFifo.Buff:='';        // Transmitter data lost
  end;
  Report('Close',aErrorCode);
 except
  on E:Exception do BugReport(E,Self,'Close');
 end;
end;

constructor TEasyPipe.Create(const aHost : LongString;
                         const aName : LongString;
                         aDelay      : Integer;
                         aRxFifoSize : Integer;
                         aTxFifoSize : Integer;
                         aTimeOut    : Integer);
begin
 inherited Create;
 myHostName:=Trim(LowerCase(aHost));
 myBaseName:=Trim(LowerCase(aName));
 myFileName:=InitFileName;
 {$IFDEF WINDOWS}
 InvalidateHandle(myFile);
 myRxPending:=false;
 myTxPending:=false;
 SafeFillChar(myRxOverlap,sizeof(myRxOverlap),0);
 SafeFillChar(myTxOverlap,sizeof(myTxOverlap),0);
 if InitializeSecurityDescriptor(@mySecDesc,SECURITY_DESCRIPTOR_REVISION) and
    SetSecurityDescriptorDacl(@mySecDesc,True,nil,True)
 then Report('InitializeSecurity',ERROR_SUCCESS)
 else Report('InitializeSecurity',GetLastError);
 mySecAttr.nLength:=sizeof(mySecAttr);
 mySecAttr.lpSecurityDescriptor:=@mySecDesc;
 mySecAttr.bInheritHandle:=False;
 {$ENDIF ~WINDOWS}
 {$IFDEF UNIX}
 InvalidateHandle(mySock);
 InvalidateHandle(myConn);
 myRxBuff:='';
 myTxBuff:='';
 {$ENDIF ~UNIX}
 aDelay:=ValidateDelay(aDelay);
 aTimeOut:=ValidateTimeout(aTimeOut);
 aRxFifoSize:=ValidateFifoSize(aRxFifoSize);
 aTxFifoSize:=ValidateFifoSize(aTxFifoSize);
 myRxBuffer:=StringBuffer(aRxFifoSize);
 myTxBuffer:=StringBuffer(aTxFifoSize);
 myRxFifo.Clear; myRxFifo.SizeLimit(aRxFifoSize,DefPipeFifoLimit);
 myTxFifo.Clear; myTxFifo.SizeLimit(aTxFifoSize,DefPipeFifoLimit);
 myConnected:=false;
 myTimeOut:=aTimeOut;
 myRxLost:=0;
 myTxLost:=0;
 myLastTick:=0;
 myLastCheck:=0;
 myLogsHistory:=DefHistory;
 myLogsText:=TStringList.Create;
 myListenPeriod:=DefListenPeriod;
 myPollingDelay:=Max(0,aDelay);
end;

procedure TEasyPipe.AfterConstruction;
begin
 inherited AfterConstruction;
 FullEasyPipeList.Add(Self);
end;

procedure TEasyPipe.BeforeDestruction;
begin
 FullEasyPipeList.Remove(Self);
 inherited BeforeDestruction;
end;

destructor TEasyPipe.Destroy;
begin
 try
  Close(NO_ERROR);
  myHostName:='';
  myBaseName:='';
  myFileName:='';
  myRxBuffer:='';
  myTxBuffer:='';
  myRxFifo.Buff:='';
  myTxFifo.Buff:='';
  FreeAndNil(myLogsText);
 except
  on E:Exception do BugReport(E,Self,'Destroy');
 end;
 inherited Destroy;
end;

function TEasyPipe.GetProperties:LongString;
const PropList='HostName,BaseName,PipeName,FileName,IsServer,IsClient,'
              +'Polling,Priority,TimeOut,ListenPeriod,RxPipeSize,TxPipeSize,'
              +'RxTotal,TxTotal,RxLost,TxLost';
var Lines:TStringList; i:Integer; id:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Lines:=TStringList.Create;
  try
   for i:=1 to WordCount(PropList,ScanSpaces) do begin
    id:=ExtractWord(i,PropList,ScanSpaces);
    Lines.Add(id+' = '+Ctrl(id));
   end;
   Result:=Lines.Text;
  finally
   Lines.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'GetProperties');
 end;
end;
{$IFDEF WINDOWS}
procedure TEasyPipe.Poll;
var RxCount,TxCount:DWORD; TxBuff:LongString;
begin
 if Assigned(Self) then
 try
  //
  // If file handle is not created, try to create it
  //
  if not ValidHandle(myFile) then begin                                        // If file not opened
   if IsServer then begin                                                      // Server:
    myFile:=CreateNamedPipe(PChar(FileName),                                   // pointer to file name
                            PIPE_ACCESS_DUPLEX or FILE_FLAG_OVERLAPPED,        // pipe open mode
                            PIPE_TYPE_BYTE or PIPE_READMODE_BYTE,              // pipe-specific modes
                            1,                                                 // maximum number of instances
                            myTxFifo.Size,                                     // output buffer size, in bytes
                            myRxFifo.Size,                                     // input buffer size, in bytes
                            TimeOut,                                           // time-out time, msec
                            @mySecAttr);                                       // pointer to security attributes structure
    if ValidHandle(myFile)                                                     // If file created
    then Report('CreateNamedPipe',ERROR_SUCCESS)                               // Then report success
    else Report('CreateNamedPipe',GetLastError);                               // Else report error
   end else begin                                                              // Client:
    myFile:=CreateFile(PChar(FileName),                                        // pointer to name of the file
                       GENERIC_READ or GENERIC_WRITE,                          // access mode
                       FILE_SHARE_READ or FILE_SHARE_WRITE,                    // share mode
                       @mySecAttr,                                             // security
                       OPEN_EXISTING,                                          // how to create
                       FILE_FLAG_OVERLAPPED,                                   // use overlapped I/O
                       0);                                                     // template file - not used
    if ValidHandle(myFile)                                                     // If file created
    then Report('CreateFile',ERROR_SUCCESS)                                    // Report success
    else Report('CreateFile',GetLastError);                                    // Else report error
   end;
   if ValidHandle(myFile) then begin                                           // If file was created
    myConnected:=false;                                                        // Mark no connection
    myRxPending:=false;                                                        // Mark no pending read
    myTxPending:=false;                                                        // Mark no pending write
    SafeFillChar(myRxOverlap,sizeof(myRxOverlap),0);                           // Clear read overlap
    SafeFillChar(myTxOverlap,sizeof(myTxOverlap),0);                           // Clear write overlap
   end else Sleep(TimeOut);                                                    // If file fails,wait some time
  end;
  //
  // File created, may connect/read/write
  //
  if ValidHandle(myFile) then begin
   //
   // If not still connected, try to connect now...
   //
   RxCount:=0;
   if not myConnected then begin
    while ReadFile(myFile,PChar(myRxBuffer)^,Length(myRxBuffer),RxCount,@myRxOverlap)  // Read data to buffer
    do myRxFifo.PutText(Copy(myRxBuffer,1,RxCount));                           // Transfer to FIFO
    case GetLastError of                                                       // Analize error:
     ERROR_PIPE_LISTENING:;                                                    // Server waiting for client
     ERROR_IO_PENDING:                                                         // Connection established
      begin
       myConnected:=true;                                                      // Mark we are connected
       myRxPending:=true;                                                      // Mark pending read
       Report('Connected',ERROR_SUCCESS);                                      // Mark connection
      end;
     ERROR_HANDLE_EOF: Close(GetLastError,RxCount);                            // End of file reached
     ERROR_BROKEN_PIPE: Close(GetLastError,RxCount);                           // Connection lost
     else Close(GetLastError,RxCount);                                         // Unknown error occured
    end;
   end;
   //
   // If connection esteblished, try to read from pipe...
   //
   if myConnected then begin                                                   // When connected
    if myRxPending then begin                                                  // When pending read:
     if GetOverlappedResult(myFile,myRxOverlap,RxCount,FALSE) then begin       // If pending read complete
      myRxFifo.PutText(Copy(myRxBuffer,1,RxCount));                            // Put data to FIFO
      myRxPending:=false;                                                      // Clear pending flag
     end else                                                                  // Else
     case GetLastError of                                                      // Analize errors:
      ERROR_IO_INCOMPLETE: ;                                                   // Wait for read completion
      ERROR_HANDLE_EOF: Close(GetLastError,RxCount);                           // End of file reached
      ERROR_BROKEN_PIPE: Close(GetLastError,RxCount);                          // Connection lost
      else Close(GetLastError,RxCount);                                        // Unknown error occured
     end;
    end else begin                                                             // When no pending read:
     RxCount:=Min(Length(myRxBuffer),myRxFifo.Space);                          // Check FIFO space
     if RxCount>0 then begin                                                   // If have FIFO space
      if ReadFile(myFile,PChar(myRxBuffer)^,RxCount,RxCount,@myRxOverlap)      // If read immediatly
      then myRxFifo.PutText(Copy(myRxBuffer,1,RxCount))                        // Then put data to FIFO
      else                                                                     // If not read
      case GetLastError of                                                     // Analize error:
       ERROR_IO_PENDING: myRxPending:=true;                                    // Mark pending read
       ERROR_HANDLE_EOF: Close(GetLastError,RxCount);                          // End of file reached
       ERROR_BROKEN_PIPE: Close(GetLastError,RxCount);                         // Connection lost
       else Close(GetLastError,RxCount);                                       // Unknown error occured
      end;
     end;
    end;
   end;
   //
   // If connection esteblished, try to write to pipe...
   //
   TxCount:=0;
   if myConnected then begin                                                   // When connected
    if myTxPending then begin                                                  // When pending write:
     if GetOverlappedResult(myFile,myTxOverlap,TxCount,FALSE)                  // If write complete
     then myTxPending:=false                                                   // Clear pending flag
     else                                                                      // If not write complete
     case GetLastError of                                                      // Analize error:
      ERROR_IO_INCOMPLETE: ;                                                   // Wait for write completion
      ERROR_BROKEN_PIPE: Close(GetLastError,0,TxCount);                        // Connection lost
      else Close(GetLastError,0,TxCount);                                      // Unknown error occured
     end;
    end else begin                                                             // When no pending write:
     TxBuff:=myTxFifo.GetText(Length(myTxBuffer));                             // Copy FIFO data
     TxCount:=Length(TxBuff);                                                  // Data length to write
     if TxCount>0 then begin                                                   // If has data to write
      SafeMove(PChar(TxBuff)^,PChar(myTxBuffer)^,TxCount);                     // Copy FIFO data to buffer
      if not WriteFile(myFile,PChar(myTxBuffer)^,TxCount,TxCount,@myTxOverlap) then // If not written immediatly
      case GetLastError of                                                     // Analize error:
       ERROR_IO_PENDING: myTxPending:=true;                                    // Mark pending write
       ERROR_BROKEN_PIPE: Close(GetLastError,0,TxCount);                       // Connection lost
       else Close(GetLastError,0,TxCount);                                     // Unknown error occured
      end;
     end;
    end;
   end;
  end;
  //
  // Check if FIFO data lost, report if so
  //
  if (GetTickCount64>myLastCheck+TimeOut) then begin                           // Every TimeOut milliseconds
   if myRxFifo.Lost<>myRxLost then begin                                       // Check if RxFifo.Lost changed
    if myRxFifo.Lost>myRxLost then Report('RxLost',-1);                        // If increased, report Rx data lost
    myRxLost:=myRxFifo.Lost;                                                   // Remember for future
   end;
   if myTxFifo.Lost<>myTxLost then begin                                       // Check if TxFifo.Lost changed
    if myTxFifo.Lost>myTxLost then Report('TxLost',-1);                        // If increased, report Tx data lost
    myTxLost:=myTxFifo.Lost;                                                   // Remember for future
   end;
   myLastCheck:=GetTickCount64;                                                // Remember moment of last check
  end;
 except
  on E:Exception do BugReport(E,Self,'Poll');
 end;
end;
{$ENDIF ~WINDOWS}

{$IFDEF UNIX}
procedure TEasyPipe.Poll;
var address:sockaddr_un; ListenTick:Boolean; FakeConn:THandle;
 // Get number of bytes available to read
 function GetRxAvail(fd:THandle):LongInt;
 begin
  Result:=0;
  if (FpIoctl(fd,FIONREAD,@Result)<0) then Result:=-1;
 end;
 // Increment FIFO data lost counter
 procedure FifoDataLost(var Fifo:TFifoRec; Num:Integer); inline;
 begin
  Fifo.Lost:=Fifo.Lost+Num;
 end;
 // Receive data from (nonblock) socket and write to FIFO
 procedure DoReceive(RxSock:THandle; var RxFifo:TFifoRec; var RxBuff:LongString);
 var Num,Len:Integer; Buff:LongString;
 begin
  if ValidHandle(RxSock) then begin
   RxBuff:='';
   Num:=GetRxAvail(RxSock);
   while (Num>0) do begin
    Buff:=StringBuffer(Num);
    Len:=FpRecv(RxSock,PChar(Buff),Length(Buff),MSG_DONTWAIT);
    if (Len<0) then begin
     if not ePendingFileOperation(GetLastOsError) then begin
      Report('FpRecv',GetLastOsError);
      FifoDataLost(RxFifo,Num);
      Close(GetLastOsError);
     end;
     Break;
    end;
    if (Len<Length(Buff)) then SetLength(Buff,Len);
    if (Buff='') then Break else RxBuff:=RxBuff+Buff;
    Num:=GetRxAvail(RxSock);
   end;
   if (Num<0) then begin
    if not ePendingFileOperation(GetLastOsError) then begin
     Report('FpIoctl',GetLastOsError);
     Close(GetLastOsError);
     Exit;
    end;
   end;
   if (RxBuff<>'') then myRxFifo.PutText(RxBuff);
   RxBuff:='';
  end;
 end;
 procedure DoTransmit(TxSock:THandle; var TxFifo:TFifoRec; var TxBuff:LongString);
 var Num,Len:Integer; Deadline:QWord; Partial,HasData,HasTime,DropTail:Boolean;
 begin
  if ValidHandle(TxSock) then begin
   if (TxBuff='') then TxBuff:=myTxFifo.GetText;
   if (TxBuff<>'') then begin
    Deadline:=GetTickCount64+Timeout;
    while (TxBuff<>'') do begin
     Num:=Length(TxBuff);
     Num:=Min(Num,OS_PIPE_BUF);
     Partial:=(Num<Length(TxBuff));
     Len:=FpSend(TxSock,PChar(TxBuff),Num,MSG_NOSIGNAL or MSG_DONTWAIT);
     case Sign(Len) of
      0:  begin
           Report('FpSend',GetLastOsError);
           Close(GetLastOsError);
           Break;
          end;
      +1: Delete(TxBuff,1,Len);
      -1: if ePendingFileOperation(GetLastOsError)
          then Partial:=true
          else begin
           Report('FpSend',GetLastOsError);
           Close(GetLastOsError);
           Break;
          end;
     end;
     HasData:=(TxBuff<>'');
     if not HasData then Break;
     HasTime:=(GetTickCount64<Deadline);
     if HasData and Partial and HasTime
     then Sleep(1)
     else Break;
    end;
    DropTail:=true;
    // If we still has unwritten data, it will be lost.
    if (TxBuff<>'') and DropTail then begin
     FifoDataLost(TxFifo,Length(TxBuff));
     TxBuff:='';
    end;
   end;
  end;
 end;
 // Check connection status by sending zero data.
 procedure CheckConnectionStatus(Sock:THandle);
 const Tmp:Integer=0;
 begin
  if ValidHandle(Sock) then begin
   if (FpSend(Sock,@Tmp,0,MSG_NOSIGNAL or MSG_DONTWAIT)<0) then begin
    if not ePendingFileOperation(GetLastOsError) then begin
     Report('FpSend',GetLastOsError);
     Close(GetLastOsError);
    end;
   end;
  end;
 end;
 // Fill unix socket address struct
 procedure UnixSockAddress(out address:sockaddr_un; aName:LongString);
 begin
  address.sun_family:=AF_UNIX;
  StrLCopy(address.sun_path,PChar(aName),SizeOf(address.sun_path)-1);
 end;
begin
 if Assigned(Self) then
 try
  // Every ListenPeriod tick …
  ListenTick:=(GetTickCount64>=myLastTick+myListenPeriod);
  if ListenTick then myLastTick:=GetTickCount64;
  //
  // Try create socket if not created yet.
  //
  if ListenTick then
  if not ValidHandle(mySock) then begin
   // Create Unix domain socket.
   UnixSockAddress(address,FileName);
   if IsServer then begin
    // Create socket.
    mySock:=FpSocket(AF_UNIX,SOCK_STREAM,0);
    if ValidHandle(mySock)
    then Report('FpSocket',ERROR_SUCCESS) else begin
     Report('FpSocket',GetLastOsError);
     Close(GetLastOsError);
     Exit;
    end;
    {$IFDEF UNIX}
    // Set close-on-exec flag.
    if FileSetCloseOnExec(mySock,true)
    then Report('FileSetCloseOnExec',ERROR_SUCCESS)
    else Report('FileSetCloseOnExec',GetLastOsError);
    {$ENDIF ~UNIX}
    // Make socket nonblock.
    if FileSetNonBlockFlag(mySock,true)
    then Report('FileSetNonBlockFlag',ERROR_SUCCESS)
    else Report('FileSetNonBlockFlag',GetLastOsError);
    // Bind the socket to the address.
    if(FpBind(mySock,@address,SizeOf(address))=0)
    then Report('FpBind',ERROR_SUCCESS) else begin
     Report('FpBind',GetLastOsError);
     Close(GetLastOsError);
     Exit;
    end;
    // Listen up to N=1 connections.
    if(FpListen(mySock,1)=0)
    then Report('FpListen',ERROR_SUCCESS) else begin
     Report('FpListen',GetLastOsError);
     Close(GetLastOsError);
     Exit;
    end;
   end else begin
     // Create socket.
     mySock:=FpSocket(AF_UNIX,SOCK_STREAM,0);
     if ValidHandle(mySock)
     then Report('FpSocket',ERROR_SUCCESS) else begin
      Report('FpSocket',GetLastOsError);
      Close(GetLastOsError);
      Exit;
     end;
     {$IFDEF UNIX}
     // Set close-on-exec flag.
     if FileSetCloseOnExec(mySock,true)
     then Report('FileSetCloseOnExec',ERROR_SUCCESS)
     else Report('FileSetCloseOnExec',GetLastOsError);
     {$ENDIF ~UNIX}
     // Make socket nonblock.
     if FileSetNonBlockFlag(mySock,true)
     then Report('FileSetNonBlockFlag',ERROR_SUCCESS)
     else Report('FileSetNonBlockFlag',GetLastOsError);
   end;
  end;
  //
  // If socket opened, make I/O.
  //
  if ValidHandle(mySock) then begin
   if IsServer then begin
    if ValidHandle(myConn) then begin
     if ListenTick then begin
      // Fake Accept for excessive incoming connections.
      FakeConn:=FpAccept(mySock,nil,nil);
      if ValidHandle(FakeConn) then begin
       Report('FakeFpAccept',ERROR_SUCCESS);
       FpShutdown(FakeConn,SHUT_RDWR);
       FpClose(FakeConn);
      end else begin
       if ePendingFileOperation(GetLastOsError)
       then Note('Wait connection') else begin
        Report('FakeFpAccept',GetLastOsError);
       end;
      end;
     end;
    end else begin
     // Accept incoming connections.
     myConn:=FpAccept(mySock,nil,nil);
     if ValidHandle(myConn) then begin
      Report('FpAccept',ERROR_SUCCESS);
      Note('Server:Connected');
      myConnected:=true;
      // Make it nonblock.
      if FileSetNonBlockFlag(myConn,true)
      then Report('FileSetNonBlockFlag',ERROR_SUCCESS)
      else Report('FileSetNonBlockFlag',GetLastOsError);
     end else begin
      if ePendingFileOperation(GetLastOsError)
      then Note('Wait connection') else begin
       Report('FpAccept',GetLastOsError);
       Close(GetLastOsError);
      end;
     end;
    end;
   end else begin
    if Connected then begin
     // Client make I/O.
     DoReceive(mySock,myRxFifo,myRxBuff);
     DoTransmit(mySock,myTxFifo,myTxBuff);
     if ListenTick then CheckConnectionStatus(mySock);
    end else begin
     // Connect Unix domain socket.
     UnixSockAddress(address,FileName);
     // Connect to the server.
     if (FpConnect(mySock,@address,SizeOf(sockaddr_un))=0) then begin
      Report('FpConnect',ERROR_SUCCESS);
      myConnected:=true;
     end else begin
      if ePendingFileOperation(GetLastOsError)
      then Note('Wait connection') else begin
       Report('FpAccept',GetLastOsError);
       Close(GetLastOsError);
      end;
     end;
    end;
   end;
  end;
  //
  // If server connected…
  //
  if IsServer then
  if Connected then
  if ValidHandle(myConn) then begin
   // Apply server Rx/Tx operations.
   DoReceive(myConn,myRxFifo,myRxBuff);
   DoTransmit(myConn,myTxFifo,myTxBuff);
   if ListenTick then CheckConnectionStatus(myConn);
  end;
 except
  on E:Exception do BugReport(E,Self,'Poll');
 end;
end;
{$ENDIF ~UNIX}

function TEasyPipe.Ctrl(const arg:LongString):LongString;
var p,i,si,iv:Integer; sn,sv,par:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Result:='?'; sn:=''; sv:='';
  p:=ExtractNameValuePair(arg,sn,sv,'=',1);
  case Identify(sn) of
   sid_Connected:   Result:=IntToStr(Ord(Connected));
   sid_IsServer:    Result:=IntToStr(Ord(IsServer));
   sid_IsClient:    Result:=IntToStr(Ord(IsClient));
   sid_FileName:    Result:=FileName;
   sid_PipeName:    Result:=PipeName;
   sid_HostName:    Result:=HostName;
   sid_BaseName:    Result:=BaseName;
   sid_Handle:      Result:=IntToStr(GetHandle(StrToIntDef(Trim(sv),0)));
   sid_TimeOut:     Result:=IntToStr(TimeOut);
   sid_RxLost:      Result:=IntToStr(myRxFifo.Lost);
   sid_TxLost:      Result:=IntToStr(myTxFifo.Lost);
   sid_RxTotal:     Result:=IntToStr(myRxFifo.Total);
   sid_TxTotal:     Result:=IntToStr(myTxFifo.Total);
   sid_RxLength:    Result:=IntToStr(myRxFifo.Count);
   sid_TxLength:    Result:=IntToStr(myTxFifo.Count);
   sid_RxFifoLimit: Result:=IntToStr(myRxFifo.Limit);
   sid_TxFifoLimit: Result:=IntToStr(myTxFifo.Limit);
   sid_LogsHistory: begin
    if IsNonEmptyStr(sv) then myLogsHistory:=StrToIntDef(Trim(sv),LogsHistory);
    Result:=IntToStr(LogsHistory);
   end;
   sid_RxBuffSize,
   sid_RxPipeSize:  begin
    if (p>0) and TryStrToInt(Trim(sv),iv) then myRxFifo.Size:=ValidateFifoSize(iv);
    Result:=IntToStr(myRxFifo.Size);
   end;
   sid_TxBuffSize,
   sid_TxPipeSize:  begin
    if (p>0) and TryStrToInt(Trim(sv),iv) then myTxFifo.Size:=ValidateFifoSize(iv);
    Result:=IntToStr(myTxFifo.Size);
   end;
   sid_Polling,
   sid_PollingDelay,
   sid_PollingPeriod: begin
    if (p>0) and TryStrToInt(Trim(sv),iv) then myPollingDelay:=ValidateDelay(iv);
    Result:=IntToStr(myPollingDelay);
   end;
   sid_Listen,
   sid_ListenDelay,
   sid_ListenPeriod: begin
    if (p>0) and TryStrToInt(Trim(sv),iv) then ListenPeriod:=iv;
    Result:=IntToStr(ListenPeriod);
   end;
   sid_Priority,
   sid_PollingPriority: begin
    Result:='tpNormal';
   end;
   sid_LogsCount:    Result:=IntToStr(LogsCount);
   sid_LogsTextMove: Result:=LogsTextMove;
   sid_LogsTextCopy: Result:=LogsTextCopy;
   sid_Clear:        begin
    if IsNonEmptyStr(sv) then Clear(Trim(sv));
    Result:='';
   end;
   sid_TxSend: begin
    if (p>0) then iv:=Send(sv) else iv:=0;
    Result:=IntToStr(iv);
   end;
   sid_RxRecv: begin
    iv:=MaxInt;
    if (p>0) then iv:=StrToIntDef(Trim(sv),iv);
    Result:=Recv(iv);
   end;
   sid_Asterisk,
   sid_Question: begin
    Result:='';
    if Assigned(Dictionary) then
    for i:=0 to Dictionary.Count-1 do begin
     si:=PtrInt(Dictionary.Objects[i]);
     if (si<=Ord(sid_Unknown)) then continue;
     if (si>=Ord(sid_Asterisk)) then continue;
     if not (TStringIdentifier(si) in sid_Ctrl_Readable) then continue;
     if (p>0) and not (TStringIdentifier(si) in sid_Ctrl_Writable) then continue;
     par:=LowerCase(Dictionary.Strings[i]);
     if (sn='*') then Result:=Result+par+EOL;
     if (sn='?') then Result:=Result+par+'='+ctrl(par)+EOL;
    end;
   end;
  end;
  sn:=''; sv:='';
 except
  on E:Exception do BugReport(E,Self,'Ctrl');
 end;
end;

procedure TEasyPipe.Clear(const What:LongString);
begin
 if Assigned(Self) then begin
  if (WordIndex('RxLost',What,ScanSpaces)>0) then myRxFifo.Lost:=0;
  if (WordIndex('TxLost',What,ScanSpaces)>0) then myTxFifo.Lost:=0;
  if (WordIndex('RxFifo',What,ScanSpaces)>0) then myRxFifo.Clear;
  if (WordIndex('TxFifo',What,ScanSpaces)>0) then myTxFifo.Clear;
  if (WordIndex('RxTotal',What,ScanSpaces)>0) then myRxFifo.Total:=0;
  if (WordIndex('TxTotal',What,ScanSpaces)>0) then myTxFifo.Total:=0;
  if (WordIndex('Logs',What,ScanSpaces)>0) then LogsTextMove;
 end;
end;

function TEasyPipe.GetRxBuffSize:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=Length(myRxBuffer);
end;

function TEasyPipe.GetTxBuffSize:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=Length(myTxBuffer);
end;

function TEasyPipe.GetRxFifoLimit:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=myRxFifo.Limit;
end;

procedure TEasyPipe.SetRxFifoLimit(aLimit:Integer);
begin
 if (Self=nil) then Exit;
 myRxFifo.Limit:=EnsureRange(aLimit,MinPipeFifoLimit,MaxPipeFifoLimit);
end;

function TEasyPipe.GetTxFifoLimit:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=myTxFifo.Limit;
end;

procedure TEasyPipe.SetTxFifoLimit(aLimit:Integer);
begin
 if (Self=nil) then Exit;
 myTxFifo.Limit:=EnsureRange(aLimit,MinPipeFifoLimit,MaxPipeFifoLimit);;
end;

function TEasyPipe.GetRxLost:Int64;
begin
 if (Self=nil) then Result:=0 else Result:=myRxLost;
end;

function TEasyPipe.GetTxLost:Int64;
begin
 if (Self=nil) then Result:=0 else Result:=myTxLost;
end;

function TEasyPipe.GetRxTotal:Int64;
begin
 if (Self=nil) then Result:=0 else Result:=myRxFifo.Total;
end;

function TEasyPipe.GetTxTotal:Int64;
begin
 if (Self=nil) then Result:=0 else Result:=myTxFifo.Total;
end;

function TEasyPipe.GetRxLength:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=Length(myRxFifo.Buff);
end;

function TEasyPipe.GetTxLength:Integer;
begin
 if (Self=nil) then Result:=0 else Result:=Length(myTxFifo.Buff);
end;

function TEasyPipe.GetRxFifo(Count:Integer):LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=myRxFifo.GetText(Count);
end;

function TEasyPipe.GetTxFifo(Count:Integer):LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=myTxFifo.GetText(Count);
end;

procedure TEasyPipe.PutRxFifo(const Data:LongString);
begin
 if (Self=nil) then Exit; if (Data='') then Exit;
 if (Length(myRxFifo.Buff)+Length(Data)>myRxFifo.Limit) then begin
  Report('RxFifoOverflowDataLost',Length(Data));
  Inc(myRxLost,Length(Data));
  Exit;
 end;
 myRxFifo.PutText(Data);
end;

procedure TEasyPipe.PutTxFifo(const Data:LongString);
begin
 if (Self=nil) then Exit; if (Data='') then Exit;
 if (Length(myTxFifo.Buff)+Length(Data)>myTxFifo.Limit) then begin
  Report('TxFifoOverflowDataLost',Length(Data));
  Inc(myTxLost,Length(Data));
  Exit;
 end;
 myTxFifo.PutText(Data);
end;

procedure TEasyPipe.SetLogsHistory(aHistory:Integer);
begin
 if (Self=nil) then Exit;
 myLogsHistory:=Max(0,aHistory);
end;

class procedure TEasyPipe.Note(const Msg:LongString);
begin
end;

class function TEasyPipe.ValidHandle(fd:THandle):Boolean;
begin
 Result:=(fd<>INVALID_HANDLE_VALUE);
end;

class procedure TEasyPipe.InvalidateHandle(var fd:THandle);
begin
 fd:=INVALID_HANDLE_VALUE;
end;

class function TEasyPipe.ValidateDelay(aDelay:Integer):Integer;
begin
 if (aDelay<=0) then aDelay:=DefDelay;
 aDelay:=EnsureRange(aDelay,1,100);
 Result:=aDelay;
end;

class function TEasyPipe.GetDefDelay:Integer;
begin
 Result:=TheDefDelay;
end;

class procedure TEasyPipe.SetDefDelay(aDelay:Integer);
begin
 TheDefDelay:=ValidateDelay(aDelay);
end;

class function TEasyPipe.ValidateTimeout(aTimeOut:Integer):Integer;
begin
 if (aTimeOut<=0) then aTimeout:=DefTimeout;
 aTimeout:=EnsureRange(aTimeOut,(1000 div 5),(1000*5));
 Result:=aTimeOut;
end;

class function TEasyPipe.GetDefTimeout:Integer;
begin
 Result:=TheDefTimeout;
end;

class procedure TEasyPipe.SetDefTimeout(aTimeout:Integer);
begin
 TheDefTimeout:=ValidateTimeout(aTimeOut);
end;

class function TEasyPipe.ValidateListenPeriod(aPeriod:Integer):Integer;
begin
 if (aPeriod<=0) then aPeriod:=DefListenPeriod;
 aPeriod:=EnsureRange(aPeriod,1,5000);
 Result:=aPeriod;
end;

class function TEasyPipe.GetDefListenPeriod:Integer;
begin
 Result:=TheDefListenPeriod;
end;

class procedure TEasyPipe.SetDefListenPeriod(aPeriod:Integer);
begin
 TheDefListenPeriod:=ValidateListenPeriod(aPeriod);
end;

class function TEasyPipe.ValidateFifoSize(aFifoSize:Integer):Integer;
begin
 if (aFifoSize<=0) then aFifoSize:=DefFifoSize;
 aFifoSize:=EnsureRange(aFifoSize,KiloByte*4,MegaByte*4);
 Result:=aFifoSize;
end;

class function TEasyPipe.GetDefFifoSize:Integer;
begin
 Result:=TheDefFifoSize;
end;

class procedure TEasyPipe.SetDefFifoSize(aFifoSize:Integer);
begin
 TheDefFifoSize:=ValidateFifoSize(aFifoSize);
end;

class function TEasyPipe.ValidateHistory(aHistory:Integer):Integer;
begin
 if (aHistory<=0) then aHistory:=DefHistory;
 aHistory:=EnsureRange(aHistory,0,1000*100);
 Result:=aHistory;
end;

class function TEasyPipe.GetDefHistory:Integer;
begin
 Result:=TheDefHistory;
end;

class procedure TEasyPipe.SetDefHistory(aHistory:Integer);
begin
 TheDefHistory:=ValidateHistory(aHistory);
end;

function  NewEasyPipe(const aHost : LongString;
                      const aName : LongString;
                      aDelay      : Integer;
                      aRxFifoSize : LongInt;
                      aTxFifoSize : LongInt;
                      aTimeOut    : Integer
                                ) : TEasyPipe;
begin
 Result:=nil;
 try
  Result:=TEasyPipe.Create(aHost,aName,
          TEasyPipe.ValidateDelay(aDelay),
          TEasyPipe.ValidateFifoSize(aRxFifoSize),
          TEasyPipe.ValidateFifoSize(aTxFifoSize),
          TEasyPipe.ValidateTimeout(aTimeOut));
 except
  on E:Exception do BugReport(E,nil,'NewEasyPipe');
 end;
end;

function  NewEasyPipe(const aPipeName    : LongString;
                            aRxBuffSize  : Integer = DefPipeBuffSize;
                            aTxBuffSize  : Integer = DefPipeBuffSize;
                            aTimeOut     : Integer = DefPipeTimeout;
                            aLogsHistory : Integer = DefPipeLogsHistory
                                       ) : TEasyPipe;
var aHost,aPipe:LongString;
begin
 aHost:=Trim(ExtractFileDir(aPipeName));
 aPipe:=Trim(ExtractFileName(aPipeName));
 Result:=NewEasyPipe(aHost,aPipe,TEasyPipe.DefDelay,
         aRxBuffSize,aTxBuffSize,aTimeOut);
 if Assigned(Result)
 then Result.LogsHistory:=TEasyPipe.ValidateHistory(aLogsHistory);
end;

procedure Kill(var TheObject:TEasyPipe); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 /////////////////////////
 // EasyIPC implementation
 /////////////////////////

function EasyPipeToRef(P:TEasyPipe):PtrUint; inline;
begin
 Result:=PointerToPtrUInt(P);
end;

function RefToEasyPipe(R:PtrUint):TEasyPipe; inline;
begin
 Result:=TEasyPipe(PtrUIntToPointer(R));
end;

function EasyIpc_Init(const PipeName,Options:LongString):PtrUint;
var List:TStringList; RxBuffSize,TxBuffSize,TimeOut,LogsHistory:Integer;
begin
 TimeOut:=DefPipeTimeOut;
 RxBuffSize:=DefPipeBuffSize;
 TxBuffSize:=DefPipeBuffSize;
 LogsHistory:=DefPipeLogsHistory;
 if (Options<>'') then begin
  List:=TStringList.Create;
  try
   List.Text:=AdjustLineBreaks(Options);
   TimeOut:=StrToIntDef(Trim(List.Values['TimeOut']),DefPipeTimeOut);
   RxBuffSize:=StrToIntDef(Trim(List.Values['RxBuffSize']),DefPipeBuffSize);
   TxBuffSize:=StrToIntDef(Trim(List.Values['TxBuffSize']),DefPipeBuffSize);
   LogsHistory:=StrToIntDef(Trim(List.Values['LogsHistory']),DefPipeLogsHistory);
  finally
   List.Free;
  end;
 end;
 Result:=EasyPipeToRef(NewEasyPipe(PipeName,RxBuffSize,TxBuffSize,TimeOut,LogsHistory));
end;

function IsValidIpc(hIpc:PtrUint):Boolean;
var List:TList;
begin
 Result:=false;
 if (hIpc=0) then Exit;
 if (FullEasyPipeList<>nil) then begin
  List:=FullEasyPipeList.LockList;
  try
   if (List.IndexOf(RefToEasyPipe(hIpc))<0) then Exit;
  finally
   FullEasyPipeList.UnlockList;
  end;
 end;
 Result:=true;
end;

function EasyIpc_Free(hIpc:PtrUint):Boolean;
begin
 Result:=false;
 if not IsValidIpc(hIpc) then Exit;
 RefToEasyPipe(hIpc).Free;
 Result:=true;
end;

function EasyIpc_Poll(hIpc:PtrUint):Boolean;
begin
 Result:=false;
 if not IsValidIpc(hIpc) then Exit;
 RefToEasyPipe(hIpc).Poll;
 Result:=true;
end;

function EasyIpc_Send(hIpc:PtrUint; const TextLines:LongString):Boolean;
begin
 Result:=false;
 if not IsValidIpc(hIpc) then Exit;
 if (TextLines<>'') then RefToEasyPipe(hIpc).PutTxFifo(AdjustLineBreaks(TextLines));
 RefToEasyPipe(hIpc).Poll;
end;

function EasyIpc_Recv(hIpc:PtrUint; Count:Integer):LongString;
begin
 Result:='';
 if not IsValidIpc(hIpc) then Exit;
 RefToEasyPipe(hIpc).Poll;
 if (Count>0) then Result:=AdjustLineBreaks(RefToEasyPipe(hIpc).GetRxFifo(Count));
end;

function EasyIpc_Ctrl(hIpc:PtrUint; const Request:LongString):LongString;
var n,v:LongString; p:Integer;
 procedure AddItem(var S:LongString; const Name:LongString);
 begin
  S:=S+(Name+'='+EasyIpc_Ctrl(hIpc,Name)+LineEnding);
 end;
begin
 Result:='';
 p:=Pos('=',Request);
 if (p>0) then begin
  n:=Trim(Copy(Request,1,p-1));
  v:=Trim(Copy(Request,p+1,MaxInt));
 end else begin
  n:=Trim(Request);
  v:='';
 end;
 if (n='') then Exit;
 if not IsValidIpc(hIpc) then Exit;
 case Identify(n) of
  sid_Connected: begin
   Result:=IntToStr(Ord(RefToEasyPipe(hIpc).Connected));
   Exit;
  end;
  sid_IsServer: begin
   Result:=IntToStr(Ord(RefToEasyPipe(hIpc).IsServer));
   Exit;
  end;
  sid_IsClient: begin
   Result:=IntToStr(Ord(RefToEasyPipe(hIpc).IsClient));
   Exit;
  end;
  sid_FileName: begin
   Result:=RefToEasyPipe(hIpc).FileName;
   Exit;
  end;
  sid_PipeName: begin
   Result:=RefToEasyPipe(hIpc).PipeName;
   Exit;
  end;
  sid_HostName: begin
   Result:=RefToEasyPipe(hIpc).HostName;
   Exit;
  end;
  sid_BaseName: begin
   Result:=RefToEasyPipe(hIpc).BaseName;
   Exit;
  end;
  sid_Handle: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).GetHandle);
   Exit;
  end;
  sid_TimeOut: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TimeOut);
   Exit;
  end;
  sid_RxBuffSize: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).RxBuffSize);
   Exit;
  end;
  sid_TxBuffSize: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TxBuffSize);
   Exit;
  end;
  sid_RxLost: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).RxLost);
   Exit;
  end;
  sid_TxLost: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TxLost);
   Exit;
  end;
  sid_RxTotal: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).RxTotal);
   Exit;
  end;
  sid_TxTotal: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TxTotal);
   Exit;
  end;
  sid_RxLength: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).RxLength);
   Exit;
  end;
  sid_TxLength: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TxLength);
   Exit;
  end;
  sid_RxFifoLimit: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).RxFifoLimit);
   Exit;
  end;
  sid_TxFifoLimit: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).TxFifoLimit);
   Exit;
  end;
  sid_LogsHistory:  begin
   if (v<>'') then RefToEasyPipe(hIpc).LogsHistory:=StrToIntDef(v,RefToEasyPipe(hIpc).LogsHistory);
   Result:=IntToStr(RefToEasyPipe(hIpc).LogsHistory);
   Exit;
  end;
  sid_LogsCount: begin
   Result:=IntToStr(RefToEasyPipe(hIpc).LogsCount);
   Exit;
  end;
  sid_LogsTextMove: begin
   Result:=RefToEasyPipe(hIpc).LogsTextMove;
   Exit;
  end;
  sid_LogsTextCopy: begin
   Result:=RefToEasyPipe(hIpc).LogsTextCopy;
   Exit;
  end;
  sid_Clear: begin
   if (v<>'') then RefToEasyPipe(hIpc).Clear(v);
   Exit;
  end;
  sid_Asterisk: begin
   AddItem(Result,'Connected');
   AddItem(Result,'IsServer');
   AddItem(Result,'IsClient');
   AddItem(Result,'FileName');
   AddItem(Result,'PipeName');
   AddItem(Result,'HostName');
   AddItem(Result,'BaseName');
   AddItem(Result,'RxLost');
   AddItem(Result,'TxLost');
   AddItem(Result,'RxTotal');
   AddItem(Result,'TxTotal');
   AddItem(Result,'RxLength');
   AddItem(Result,'TxLength');
   AddItem(Result,'Handle');
   AddItem(Result,'TimeOut');
   AddItem(Result,'LogsCount');
   AddItem(Result,'LogsHistory');
   AddItem(Result,'RxBuffSize');
   AddItem(Result,'TxBuffSize');
   AddItem(Result,'RxFifoLimit');
   AddItem(Result,'TxFifoLimit');
   Exit;
  end;
 end;
end;

initialization

 InitDictionary;
 TEasyPipe.DefDelay:=4;
 TEasyPipe.DefHistory:=16;
 TEasyPipe.DefTimeout:=1000;
 TEasyPipe.DefListenPeriod:=500;
 TEasyPipe.DefFifoSize:=KiloByte*16;
 FullEasyPipeList:=TThreadList.Create;

finalization

 FreeAndNil(FullEasyPipeList);
 FreeDictionary;
 
end.

