{------------------------------------------------------------------------------}
{                                                                              }
{                              Yuriy Kopnin                                    }
{                                  LGPL                                        }
{                                                                              }
{------------------------------------------------------------------------------}

unit dplex;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, dpUtils;

const
  dsEOF     = Char(0);
  dsSymbol  = Char(1);
  dsString  = Char(2);
  dsInteger = Char(3);
  dsHexInt  = Char(4);
  dsFloat   = Char(5);
  dsText    = Char(6);
  dsChar    = Char(7);
  dsUnknSym = Char(8);

type

  TIdentToken = (
    ID_EOF,
    ID_Identifier,
    ID_SemiColon,
    ID_Comma,
    ID_Period,
    ID_Colon,
    ID_OpenRound,
    ID_CloseRound,
    ID_OpenBlock,
    ID_CloseBlock,
    ID_Assignment,
    ID_Inc,
    ID_Dec,
    ID_IncAssign,
    ID_DecAssign,
    ID_Equal,
    ID_NotEqual,
    ID_Greater,
    ID_GreaterEqual,
    ID_Less,
    ID_LessEqual,
    ID_Plus,
    ID_Minus,
    ID_Divide,
    ID_Multiply,
    ID_Integer,
    ID_Float,
    ID_String,
    ID_Char,
    ID_HexInt,
    ID_AddressOf,
    ID_Dereference,
    ID_Dots,
    ID_and,
    ID_array,
    ID_begin,
    ID_case,
    ID_const,
    ID_div,
    ID_do,
    ID_downto,
    ID_else,
    ID_elseif,
    ID_end,
    ID_for,
    ID_function,
    ID_if,
    ID_in,
    ID_mod,
    ID_not,
    ID_of,
    ID_or,
    ID_procedure,
    ID_method,
    ID_program,
    ID_repeat,
    ID_record,
    ID_set,
    ID_shl,
    ID_shr,
    ID_then,
    ID_to,
    ID_type,
    ID_until,
    ID_Break,
    ID_Continue,
    ID_Exit,
    ID_uses,
    ID_var,
    ID_while,
    ID_with,
    ID_xor,
    ID_class,
    ID_constructor,
    ID_destructor,
    ID_inherited,
    ID_private,
    ID_public,
    ID_published,
    ID_protected,
    ID_property,
    ID_virtual,
    ID_override,
    ID_overload,
    ID_As,
    ID_Is,
    ID_Unit,
    ID_Try,
    ID_Except,
    ID_On,
    ID_Finally,
    ID_External,
    ID_Forward,
    ID_Export,
    ID_Label,
    ID_Goto,
    ID_Chr,
    //ID_Ord,
    ID_Interface,
    ID_Implementation,
    ID_initialization,
    ID_finalization,
    ID_out,
    ID_nil,
    ID_False,
    ID_True,
    ID_Unknown
    );


  PTokenRec = ^TTokenRec;
  TTokenRec = record
    TokenName: string;
    TokenID: TIdentToken;
    Col: Integer;
    Row: Integer;
    HashValue: Cardinal;
  end;

  TUnitSection = (usInterface, usImplementation, usInitialization);

  TTokensList = class
  private
    FList: TList;
    FPos: Integer;
    function Get(Index: Integer): TTokenRec;
    function GetToken: TTokenRec;
    function GetPreviewNext: TTokenRec;
  protected
  public
    UnitSection: TUnitSection;
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Count: Integer;
    procedure Next;
    procedure Prev;
    function Add(TokenName: string; TokenID: TIdentToken; Col, Row: Integer; HV: Cardinal): Integer;
    property Pos: Integer read FPos write FPos;
    property Items[Index: Integer]: TTokenRec read Get;
    property Token: TTokenRec read GetToken;
    property PreviewNext: TTokenRec read GetPreviewNext;
  end;

  { TDpParser }

  TDpParser = class
  private
    FString: string;
    FStartPos: PChar;
    FEndPos: PChar;
    FRow: Longint;
    FStartCol: PChar;
    FTokenPos: Integer;
    FTokenEnd: Integer;
    FTokens: TTokensList;
    FUseTranscription: Boolean;
    FSuLanguage: TSuLanguage;
    function GetCol: Integer;
  protected
    function TokenString: String;
    function TokenText: String;
    function TokenFloat: Extended;
    function TokenInt: Integer;
  public
    //function PreviewToken: Char;
    constructor Create(UseTranscript: Boolean; ALanguage: TSuLanguage); virtual;
    destructor Destroy; override;
    function IsIdentChar(p: PChar): boolean;
    function NextToken: Char; virtual;
    procedure SetParseStr(S: String);
    function GetTokensList: TTokensList;
    property Row: LongInt read FRow;
    property Col: LongInt read GetCol;
    property TokenPos: Integer read FTokenPos;
    property TokenEnd: Integer read FTokenEnd;
    property AsString: String read TokenString;
    property AsFloat: Extended read TokenFloat;
    property AsInteger: Integer read TokenInt;
    property AsText: String read TokenText;
    property Tokens: TTokensList read FTokens;
  end;

var
  KeyAnd, KeyArray, KeyBegin, KeyEnd, KeyAs, KeyCase, KeyChr,
  KeyConst, KeyClass, KeyConstr, KeyDestr, KeyDiv, KeyDo, KeyDownto,
  KeyElse, KeyElseif, KeyExcept, KeyExit, KeyExport, KeyExternal,
  KeyFinaliz, KeyFinally, KeyFor, KeyForward, KeyFunction, KeyProcedure,
  KeyMethod, KeyIf, KeyImplement, KeyIn, KeyInherited, KeyInitializ,
  KeyInterface, KeyIs, KeyLabel, KeyMod, KeyNil, KeyNot, KeyOf, KeyOr,
  KeyOrd, KeyOut, KeyOverride, KeyOverload, KeyPrivate, KeyProgram,
  KeyProperty, KeyProtected, KeyPublic, KeyPublished, KeyRecord,
  KeyRepeat, KeySet, KeyShl, KeyShr, KeyThen, KeyTo, KeyTry, KeyType,
  KeyUnit, KeyUntil, KeyUses, KeyVar, KeyVirtual, KeyWhile, KeyWith,
  KeyXor, KeyBreak, KeyContinue, KeyTrue, KeyFalse, KeyString,
  KeyOn: string;

  KeyResult, KeySelf, KeyCreate: string;


implementation

uses character, LazUTF8;

var
  HashAnd, HashArray, HashBegin, HashEnd, HashAs, HashCase, HashChr,
  HashConst, HashClass, HashConstr, HashDestr, HashDiv, HashDo, HashDownto,
  HashElse, HashElseif, HashExcept, HashExit, HashExport, HashExternal,
  HashFinaliz, HashFinally, HashFor, HashForward, HashFunction, HashProcedure,
  HashMethod, HashIf, HashImplement, HashIn, HashInherited, HashInitializ,
  HashInterface, HashIs, HashLabel, HashMod, HashNil, HashNot, HashOf, HashOr,
  HashOrd, HashOut, HashOverride, HashOverload, HashPrivate, HashProgram,
  HashProperty, HashProtected, HashPublic, HashPublished, HashRecord,
  HashRepeat, HashSet, HashShl, HashShr, HashThen, HashTo, HashTry, HashType,
  HashUnit, HashUntil, HashUses, HashVar, HashVirtual, HashWhile, HashWith,
  HashXor, HashBreak, HashContinue, HashTrue, HashFalse, HashString,
  HashOn: Cardinal;

var
  LHashAnd, LHashArray, LHashBegin, LHashEnd, LHashAs, LHashCase, LHashChr,
  LHashConst, LHashClass, LHashConstr, LHashDestr, LHashDiv, LHashDo, LHashDownto,
  LHashElse, LHashElseif, LHashExcept, LHashExit, LHashExport, LHashExternal,
  LHashFinaliz, LHashFinally, LHashFor, LHashForward, LHashFunction, LHashProcedure,
  LHashMethod, LHashIf, LHashImplement, LHashIn, LHashInherited, LHashInitializ,
  LHashInterface, LHashIs, LHashLabel, LHashMod, LHashNil, LHashNot, LHashOf, LHashOr,
  LHashOrd, LHashOut, LHashOverride, LHashOverload, LHashPrivate, LHashProgram,
  LHashProperty, LHashProtected, LHashPublic, LHashPublished, LHashRecord,
  LHashRepeat, LHashSet, LHashShl, LHashShr, LHashThen, LHashTo, LHashTry, LHashType,
  LHashUnit, LHashUntil, LHashUses, LHashVar, LHashVirtual, LHashWhile, LHashWith,
  LHashXor, LHashBreak, LHashContinue, LHashTrue, LHashFalse, LHashString,
  LHashOn: Cardinal;

  LHashResult, LHashSelf, LHashCreate: Cardinal;



constructor TDpParser.Create(UseTranscript: Boolean; ALanguage: TSuLanguage);
begin
  inherited Create;
  FSuLanguage := ALanguage;
  FUseTranscription := UseTranscript;
  FTokens := TTokensList.Create;
end;

destructor TDpParser.Destroy;
begin
  //FTokens.Free;
  inherited Destroy;
end;

function TDpParser.IsIdentChar(p: PChar): boolean;
var
  u: UnicodeString;
  i: Integer;
  L: SizeUInt;
begin
  Result := False;
  Result := p^ in ['a'..'z','A'..'Z','0'..'9','_'];
  if Result then exit;

  if p^ <= #127 then exit;
  i := UTF8CharacterLength(p);
  SetLength(u, i);
  // wide chars of UTF-16 <= bytes of UTF-8 string
  if ConvertUTF8ToUTF16(PWideChar(u), i + 1, p, i, [toInvalidCharToSymbol], L) = trNoError then
  begin
    SetLength(u, L - 1);
    if L > 1 then
      Result := TCharacter.IsLetterOrDigit(u, 1);
  end;
end;

procedure TDpParser.SetParseStr(S: String);
var
  I: Integer;
begin
  I := Length(S);
  FString := '';
  FRow := 0;
  FTokenPos := 0;
  FTokenEnd := 0;

  if I > 0 then
  begin
    FString := S;
    I := Length(FString);
    Inc(I);
    SetLength(FString, I);
    FString[I] := #0;
    FStartPos := @FString[1];
    FEndPos := FStartPos;
    FStartCol := FStartPos;
  end
  else
  begin
    FStartPos := nil;
    FEndPos := nil;
    FStartCol := nil;
  end;
end;



function TDpParser.NextToken: Char;
var
  P: PChar;
  L: Integer;

  procedure SkipBlanks;
  begin
    case P^ of
      ' ', #8:
      begin
        while (P^ = ' ') or (P^ = #8) do
        begin
          Inc(P);
        end;
      end;

      '/':
      begin
        Inc(P, 2);
        while not (P^ in [#0, #13, #10]) do Inc(P);
      end;

      '{':
      begin
        Inc(P);
        while P^ <> dsEOF do
        begin
          case P^ of
            #10:
            begin
              FStartCol := P;
              Inc(FRow);
            end;
            '}':
            begin
              Inc(P);
              Break;
            end;
          end;
          Inc(P);
        end;
      end;

      '(':
      begin
        Inc(P, 2);
        while P^ <> dsEOF do
        begin
          case P^ of
            #10:
            begin
              Inc(FRow);
              FStartCol := P;
            end;
            '*':
            begin
              if (P + 1)^ = ')' then
              begin
                Inc(P, 2);
                Break;
              end;
            end;
          end;

          Inc(P);
        end;
      end;

      #13:
      begin
        //Inc(FRow);
        Inc(P);
      end;

      #10:
      begin
        Inc(FRow);
        Inc(P);
        FStartCol := P;
      end;
    end;

  end;

begin
  if FStartPos = nil then
  begin
    Result := dsEOF;
    Exit;
  end;
  FStartPos := FEndPos;
  P := FStartPos;

  if P^ = #0 then
  begin
    Result := dsEOF;
    Exit;
  end;

  while (P^ = ' ') or (P^ = '{') or (P^ = #8)
    or (P^ = #13) or (P^ = #10)
    or (((P^ = '/') and ((P + 1)^ = '/')))
    or ((P^ = '(') and ((P + 1)^ = '*')) do
  begin
    SkipBlanks;
  end;

  FStartPos := P;

  if P^ = #0 then
  begin
      Result := dsEOF;
  end
  else
  if P^ in ['0'..'9'] then
  begin
      Result := toInteger;
      Inc(P);
      while (P^ in ['0'..'9', '.', 'E', 'e']) do
      begin
        if (P^ = '.') then
        begin
          if ((P + 1)^ in ['0'..'9']) then
            Result := dsFloat
          else Break;
        end
        else
        if (P^ = 'e') or (P^ = 'E') then
        begin
          Result := dsFloat;
          if ((P + 1)^ = '+') or ((P + 1)^ = '-') then Inc(P);
        end;
        Inc(P);
      end;
  end
  else
  if IsIdentChar(P) then
  begin
    L := UTF8CharacterLength(P);
    Inc(P, L);
    while IsIdentChar(P) or (P^ = '$') do
    begin
      L := UTF8CharacterLength(P);
      Inc(P, L);
    end;
    Result := dsString;
  end
  else
  if P^ = '#' then
  begin
    Inc(P);
    FStartPos := P;
    while (P^ in ['0'..'9']) do Inc(P);
    if FStartPos = P then
      Result := dsUnknSym
    else
      Result := dsChar;
  end
  else
  if P^ = '$' then
  begin
      Inc(P);
      while (P^ in ['0'..'9', 'A'..'F', 'a'..'f']) do Inc(P);
      Result := dsHexInt;
  end
  else
  if P^ =  '''' then
  begin
      Inc(P);
      while (P^ <> #0) and (P^ <> #10) and (P^ <> #13) do
      begin
        if P^ = '''' then
        begin
          if (P + 1)^ = '''' then
          begin
            Inc(P, 2);
            Continue;
          end
          else
          begin
            Inc(P);
            Break;
          end;
        end;
        Inc(P);
      end;
      Result := dsText;
  end
  else
  begin
      if FSuLanguage = slPPlus then
      begin
        if ((P^ = '=') or (P^ = '<') or (P^ = '>')) and ((P + 1)^ = '=') then Inc(P)
        else
        if (P^ = '<') and ((P + 1)^ = '>') then Inc(P)
        else
        if (P^ = '<') and ((P + 1)^ = '<') then Inc(P)
        else
        if (P^ = '>') and ((P + 1)^ = '>') then Inc(P)
        else
        if (P^ = '.') and ((P + 1)^ = '.') then Inc(P)
        else
        if (P^ = '+') and ((P + 1)^ = '+') then Inc(P)
        else
        if (P^ = '+') and ((P + 1)^ = '=') then Inc(P)
        else
        if (P^ = '-') and ((P + 1)^ = '-') then Inc(P)
        else
        if (P^ = '-') and ((P + 1)^ = '=') then Inc(P);
      end
      else
      begin
        if (P^ = ':') and ((P + 1)^ = '=')  then Inc(P)
        else
        if (P^ = '<') and ((P + 1)^ = '=')  then Inc(P)
        else
        if (P^ = '>') and ((P + 1)^ = '=')  then Inc(P)
        else
        if (P^ = '=') and ((P^ = '<') or (P^ = '>')) then Inc(P)
        else
        if (P^ = '<') and ((P + 1)^ = '>') then Inc(P)
        else
        if (P^ = '<') and ((P + 1)^ = '<') then Inc(P)
        else
        if (P^ = '>') and ((P + 1)^ = '>') then Inc(P)
        else
        if (P^ = '.') and ((P + 1)^ = '.') then Inc(P)
        else
        if (P^ = '+') and ((P + 1)^ = '+') then Inc(P)
        else
        if (P^ = '+') and ((P + 1)^ = '=') then Inc(P)
        else
        if (P^ = '-') and ((P + 1)^ = '-') then Inc(P)
        else
        if (P^ = '-') and ((P + 1)^ = '=') then Inc(P);
      end;
      L := UTF8CharacterLength(p);
      Inc(P, L);
      Result := dsSymbol;
  end;
  FEndPos := P;
end;

function TDpParser.TokenString: String;
var
  I: Integer;
  R: string;
begin
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(R, FStartPos, I);
    Result := R;
    //FStartPos := FEndPos;
  end;
end;

function TDpParser.TokenText: String;
var
  I: Integer;
  R: string;
begin
  R := '';
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(R, FStartPos + 1, I - 2);
    I := Length(R);
    while I > 0 do
    begin
      if (I > 1) and (R[I] = '''') then
      begin
        if R[I - 1] = '''' then
        begin
          Delete(R, I, 1);
          Dec(I);
        end;
      end;
      Dec(I);
    end;
  end;
  Result := R;
end;

function TDpParser.TokenFloat: Extended;
begin
  Result := StrToFloat(AsString);
end;

function TDpParser.TokenInt: Integer;
begin
  Result := StrToInt(AsString);
end;

function TDpParser.GetCol: Integer;
var
  S: string;
  I: Integer;
begin
  I := FStartPos - FStartCol;
  SetString(S, FStartCol, I);
  Result := UTF8Length(FStartCol, I);
end;

function HexToInt(S: string): integer;
var
  Tempt: string;
  I: integer;
begin
  Tempt := '';
  if S = '' then
  begin
    HexToInt := 0;
  end
  else
  begin
    for i := 1 to Length(s) do
    begin
      tempt := tempt + string(IntToHex(Ord(s[i]), 2));
    end;
    HexToInt := StrToInt('$' + tempt);
  end;
end;

function TDpParser.GetTokensList: TTokensList;
var
  C: Char;
  S, HS: string;
  I: Integer;
  H: Cardinal;
begin
  Result := FTokens;
  if FString = '' then Exit;
  C := NextToken;
  while True do
  begin
    case C of
      dsSymbol:
      begin
        S := AsString;
        if FSuLanguage = slPascal then
        begin
          if S = '++' then Tokens.Add(S, ID_Inc, Col, Row, 0)
          else
          if S = '+=' then Tokens.Add(S, ID_IncAssign, Col, Row, 0)
          else
          if S = '--' then Tokens.Add(S, ID_Dec, Col, Row, 0)
          else
          if S = '-=' then Tokens.Add(S, ID_DecAssign, Col, Row, 0)
          else
          if S = '=' then Tokens.Add(S, ID_Equal, Col, Row, 0)
          else
          if S = '>=' then Tokens.Add(S, ID_GreaterEqual, Col, Row, 0)
          else
          if S = '<<' then Tokens.Add(S, ID_shl, Col, Row, 0)
          else
          if S = '>>' then Tokens.Add(S, ID_shr, Col, Row, 0)
          else
          if S = '>' then Tokens.Add(S, ID_Greater, Col, Row, 0)
          else
          if S = '<' then Tokens.Add(S, ID_Less, Col, Row, 0)
          else
          if S = '<=' then Tokens.Add(S, ID_LessEqual, Col, Row, 0)
          else
          if S = '<>' then Tokens.Add(S, ID_NotEqual, Col, Row, 0)
          else
          if S = '(' then Tokens.Add(S, ID_OpenRound, Col, Row, 0)
          else
          if S = ')' then Tokens.Add(S, ID_CloseRound, Col, Row, 0)
          else
          if S = '[' then Tokens.Add(S, ID_OpenBlock, Col, Row, 0)
          else
          if S = ']' then Tokens.Add(S, ID_CloseBlock, Col, Row, 0)
          else
          if S = ',' then Tokens.Add(S, ID_Comma, Col, Row, 0)
          else
          if S = '.' then Tokens.Add(S, ID_Period, Col, Row, 0)
          else
          if S = '..' then Tokens.Add(S, ID_Dots, Col, Row, 0)
          else
          if S = '@' then Tokens.Add(S, ID_AddressOf, Col, Row, 0)
          else
          if S = '^' then Tokens.Add(S, ID_Dereference, Col, Row, 0)
          else
          if S = ';' then Tokens.Add(S, ID_SemiColon, Col, Row, 0)
          else
          if S = ':' then Tokens.Add(S, ID_Colon, Col, Row, 0)
          else
          if S = ':=' then Tokens.Add(S, ID_Assignment, Col, Row, 0)
          else
          if S = '+' then Tokens.Add(S, ID_Plus, Col, Row, 0)
          else
          if S = '-' then Tokens.Add(S, ID_Minus, Col, Row, 0)
          else
          if S = '*' then Tokens.Add(S, ID_Multiply, Col, Row, 0)
          else
          if S = '/' then Tokens.Add(S, ID_Divide, Col, Row, 0)
          else
            Tokens.Add(S, ID_Unknown, Col, Row, 0);
        end
        else
        begin
          if S = '++' then Tokens.Add(S, ID_Inc, Col, Row, 0)
          else
          if S = '+=' then Tokens.Add(S, ID_IncAssign, Col, Row, 0)
          else
          if S = '--' then Tokens.Add(S, ID_Dec, Col, Row, 0)
          else
          if S = '-=' then Tokens.Add(S, ID_DecAssign, Col, Row, 0)
          else
          if S = '==' then Tokens.Add(S, ID_Equal, Col, Row, 0)
          else
          if S = '>=' then Tokens.Add(S, ID_GreaterEqual, Col, Row, 0)
          else
          if S = '<<' then Tokens.Add(S, ID_shl, Col, Row, 0)
          else
          if S = '>>' then Tokens.Add(S, ID_shr, Col, Row, 0)
          else
          if S = '>' then Tokens.Add(S, ID_Greater, Col, Row, 0)
          else
          if S = '<' then Tokens.Add(S, ID_Less, Col, Row, 0)
          else
          if S = '<=' then Tokens.Add(S, ID_LessEqual, Col, Row, 0)
          else
          if S = '<>' then Tokens.Add(S, ID_NotEqual, Col, Row, 0)
          else
          if S = '(' then Tokens.Add(S, ID_OpenRound, Col, Row, 0)
          else
          if S = ')' then Tokens.Add(S, ID_CloseRound, Col, Row, 0)
          else
          if S = '[' then Tokens.Add(S, ID_OpenBlock, Col, Row, 0)
          else
          if S = ']' then Tokens.Add(S, ID_CloseBlock, Col, Row, 0)
          else
          if S = ',' then Tokens.Add(S, ID_Comma, Col, Row, 0)
          else
          if S = '.' then Tokens.Add(S, ID_Period, Col, Row, 0)
          else
          if S = '..' then Tokens.Add(S, ID_Dots, Col, Row, 0)
          else
          if S = '@' then Tokens.Add(S, ID_AddressOf, Col, Row, 0)
          else
          if S = '^' then Tokens.Add(S, ID_Dereference, Col, Row, 0)
          else
          if S = ';' then Tokens.Add(S, ID_SemiColon, Col, Row, 0)
          else
          if S = ':' then Tokens.Add(S, ID_Colon, Col, Row, 0)
          else
          if S = '=' then Tokens.Add(S, ID_Assignment, Col, Row, 0)
          else
          if S = '+' then Tokens.Add(S, ID_Plus, Col, Row, 0)
          else
          if S = '-' then Tokens.Add(S, ID_Minus, Col, Row, 0)
          else
          if S = '*' then Tokens.Add(S, ID_Multiply, Col, Row, 0)
          else
          if S = '/' then Tokens.Add(S, ID_Divide, Col, Row, 0)
          else
            Tokens.Add(S, ID_Unknown, Col, Row, 0);
        end;

      end;
      dsChar:
      begin
        S := AsString;
        S := Char(Chr(StrToInt(S)));
        Tokens.Add(S, ID_Char, Col, Row, 0)
      end;
      dsString:
      begin
        S := AsString;
        HS := UTF8LowerCase(S);
        H := Hash(HS);
        I := -1;
        if FUseTranscription then
        begin
          if (H = LHashAnd) and (HS = KeyAnd) then
            I := Tokens.Add(S, ID_and, Col, Row, H)
          else
          if (H = LHashArray) and (HS = KeyArray) then
            I := Tokens.Add(S, ID_array, Col, Row, H)
          else
          if (H = LHashAs) and (HS = KeyAs) then I := Tokens.Add(S, ID_As, Col, Row, H)
          else
          if (H = LHashBegin) and (HS = KeyBegin) then I := Tokens.Add(S, ID_begin, Col, Row, H)
          else
          if (H = LHashCase) and (HS = KeyCase) then I := Tokens.Add(S, ID_case, Col, Row, H)
          else
          if (H = LHashClass) and (HS = KeyClass) then I := Tokens.Add(S, ID_class, Col, Row, H)
          else
          if (H = LHashRecord) and (HS = KeyRecord) then I := Tokens.Add(S, ID_record, Col, Row, H)
          else
          if (H = LHashConst) and (HS = KeyConst) then I := Tokens.Add(S, ID_const, Col, Row, H)
          else
          if (H = LHashConstr) and (HS = KeyConstr) then I := Tokens.Add(S, ID_constructor, Col, Row, H)
          else
          if (H = LHashDestr) and (HS = KeyDestr) then I := Tokens.Add(S, ID_destructor, Col, Row, H)
          else
          if (H = LHashDiv) and (HS = KeyDiv) then I := Tokens.Add(S, ID_div, Col, Row, H)
          else
          if (H = LHashDo) and (HS = KeyDo) then I := Tokens.Add(S, ID_do, Col, Row, H)
          else
          if (H = LHashDownto) and (HS = KeyDownto) then I := Tokens.Add(S, ID_downto, Col, Row, H)
          else
          if (H = LHashElse) and (HS = KeyElse) then I := Tokens.Add(S, ID_else, Col, Row, H)
          else
          if (H = LHashElseif) and (HS = KeyElseif) then I := Tokens.Add(S, ID_elseif, Col, Row, H)
          else
          if (H = LHashEnd) and (HS = KeyEnd) then I := Tokens.Add(S, ID_end, Col, Row, H)
          else
          if (H = LHashExcept) and (HS = KeyExcept) then I := Tokens.Add(S, ID_Except, Col, Row, H)
          else
          if (H = LHashOn) and (HS = KeyOn) then I := Tokens.Add(S, ID_On, Col, Row, H)
          else
          if (H = LHashExit) and (HS = KeyExit) then I := Tokens.Add(S, ID_exit, Col, Row, H)
          else
          if (H = LHashFinally) and (HS = KeyFinally) then I := Tokens.Add(S, ID_Finally, Col, Row, H)
          else
          if (H = LHashFor) and (HS = KeyFor) then I := Tokens.Add(S, ID_for, Col, Row, H)
          else
          if (H = LHashMethod) and (HS = KeyMethod) then I := Tokens.Add(S, ID_method, Col, Row, H)
          else
          if (H = LHashProcedure) and (HS = KeyProcedure) then I := Tokens.Add(S, ID_procedure, Col, Row, H)
          else
          if (H = LHashFunction) and (HS = KeyFunction) then I := Tokens.Add(S, ID_function, Col, Row, H)
          else
          if (H = LHashIf) and (HS = KeyIf) then I := Tokens.Add(S, ID_if, Col, Row, H)
          else
          if (H = LHashImplement) and (HS = KeyImplement) then I := Tokens.Add(S, ID_Implementation, Col, Row, H)
          else
          if (H = LHashIn) and (HS = KeyIn) then I := Tokens.Add(S, ID_in, Col, Row, H)
          else
          if (H = LHashInherited) and (HS = KeyInherited) then I := Tokens.Add(S, ID_inherited, Col, Row, H)
          else
          if (H = LHashInitializ) and (HS = KeyInitializ) then I := Tokens.Add(S, ID_initialization, Col, Row, H)
          else
          if (H = LHashInterface) and (HS = KeyInterface) then I := Tokens.Add(S, ID_Interface, Col, Row, H)
          else
          if (H = LHashIs) and (HS = KeyIs) then I := Tokens.Add(S, ID_Is, Col, Row, H)
          else
          if (H = LHashMod) and (HS = KeyMod) then I := Tokens.Add(S, ID_mod, Col, Row, H)
          else
          if (H = LHashNil) and (HS = KeyNil) then I := Tokens.Add(S, ID_nil, Col, Row, H)
          else
          if (H = LHashNot) and (HS = KeyNot) then I := Tokens.Add(S, ID_not, Col, Row, H)
          else
          if (H = LHashOf) and (HS = KeyOf) then I := Tokens.Add(S, ID_of, Col, Row, H)
          else
          if (H = LHashOr) and (HS = KeyOr) then I := Tokens.Add(S, ID_or, Col, Row, H)
          else
          if (H = LHashPrivate) and (HS = KeyPrivate) then I := Tokens.Add(S, ID_private, Col, Row, H)
          else
          if (H = LHashProgram) and (HS = KeyProgram) then I := Tokens.Add(S, ID_program, Col, Row, H)
          else
          if (H = LHashProtected) and (HS = KeyProtected) then I := Tokens.Add(S, ID_protected, Col, Row, H)
          else
          if (H = LHashPublic) and (HS = KeyPublic) then I := Tokens.Add(S, ID_public, Col, Row, H)
          else
          if (H = LHashRepeat) and (HS = KeyRepeat) then I := Tokens.Add(S, ID_repeat, Col, Row, H)
          else
          if (H = LHashSet) and (HS = KeySet) then I := Tokens.Add(S, ID_set, Col, Row, H)
          else
          if (H = LHashThen) and (HS = KeyThen) then I := Tokens.Add(S, ID_then, Col, Row, H)
          else
          if (H = LHashTo) and (HS = KeyTo) then I := Tokens.Add(S, ID_to, Col, Row, H)
          else
          if (H = LHashTry) and (HS = KeyTry) then I := Tokens.Add(S, ID_Try, Col, Row, H)
          else
          if (H = LHashType) and (HS = KeyType) then I := Tokens.Add(S, ID_type, Col, Row, H)
          else
          if (H = LHashUnit) and (HS = KeyUnit) then
            I := Tokens.Add(S, ID_Unit, Col, Row, H)
          else
          if (H = LHashUntil) and (HS = KeyUntil) then I := Tokens.Add(S, ID_until, Col, Row, H)
          else
          if (H = LHashUses) and (HS = KeyUses) then I := Tokens.Add(S, ID_uses, Col, Row, H)
          else
          if (H = LHashVar) and (HS = KeyVar) then I := Tokens.Add(S, ID_var, Col, Row, H)
          else
          if (H = LHashWhile) and (HS = KeyWhile) then I := Tokens.Add(S, ID_while, Col, Row, H)
          else
          if (H = LHashBreak) and (HS = KeyBreak) then I := Tokens.Add(S, ID_Break, Col, Row, H)
          else
          if (H = LHashContinue) and (HS = KeyContinue) then I := Tokens.Add(S, ID_Continue, Col, Row, H)
          else
          if (H = LHashForward) and (HS = KeyForward) then I := Tokens.Add(S, ID_Forward, Col, Row, H)
          else
          if (H = LHashTrue) and (HS = KeyTrue) then I := Tokens.Add(S, ID_True, Col, Row, H)
          else
          if (H = LHashFalse) and (HS = KeyFalse) then I := Tokens.Add(S, ID_False, Col, Row, H)
          else
          if (H = LHashWith) and (HS = KeyWith) then I := Tokens.Add(S, ID_with, Col, Row, H)
        end;

        if I = -1 then
        begin
          if (H = HashAnd) and (CompareText(S, 'and') = 0) then Tokens.Add(S, ID_and, Col, Row, H)
          else
          if (H = HashArray) and (CompareText(S, 'array') = 0) then Tokens.Add(S, ID_array, Col, Row, H)
          else
          if (H = HashAs) and (CompareText(S, 'as') = 0) then Tokens.Add(S, ID_As, Col, Row, H)
          else
          if (H = HashBegin) and (CompareText(S, 'begin') = 0) then Tokens.Add(S, ID_begin, Col, Row, H)
          else
          if (H = HashCase) and (CompareText(S, 'case') = 0) then Tokens.Add(S, ID_case, Col, Row, H)
          else
          //if (H = HashChr) and (CompareText(S, 'chr') = 0) then Tokens.Add(S, ID_Chr, Col, Row, H)
          //else
          if (H = HashClass) and (CompareText(S, 'class') = 0) then Tokens.Add(S, ID_class, Col, Row, H)
          else
          if (H = HashConst) and (CompareText(S, 'const') = 0) then Tokens.Add(S, ID_const, Col, Row, H)
          else
          if (H = HashConstr) and (CompareText(S, 'constructor') = 0) then Tokens.Add(S, ID_constructor, Col, Row, H)
          else
          if (H = HashDestr) and (CompareText(S, 'destructor') = 0) then Tokens.Add(S, ID_destructor, Col, Row, H)
          else
          if (H = HashDiv) and (CompareText(S, 'div') = 0) then Tokens.Add(S, ID_div, Col, Row, H)
          else
          if (H = HashDo) and (CompareText(S, 'do') = 0) then Tokens.Add(S, ID_do, Col, Row, H)
          else
          if (H = HashDownto) and (CompareText(S, 'downto') = 0) then Tokens.Add(S, ID_downto, Col, Row, H)
          else
          if (H = HashElse) and (CompareText(S, 'else') = 0) then Tokens.Add(S, ID_else, Col, Row, H)
          else
          if (H = HashElseif) and (CompareText(S, 'elseif') = 0) then Tokens.Add(S, ID_elseif, Col, Row, H)
          else
          if (H = HashEnd) and (CompareText(S, 'end') = 0) then Tokens.Add(S, ID_end, Col, Row, H)
          else
          if (H = HashExcept) and (CompareText(S, 'except') = 0) then Tokens.Add(S, ID_Except, Col, Row, H)
          else
          if (H = HashExit) and (CompareText(S, 'exit') = 0) then Tokens.Add(S, ID_exit, Col, Row, H)
          else
          if (H = HashExport) and (CompareText(S, 'export') = 0) then Tokens.Add(S, ID_Export, Col, Row, H)
          else
          if (H = HashExternal) and (CompareText(S, 'external') = 0) then Tokens.Add(S, ID_External, Col, Row, H)
          else
          if (H = HashFinaliz) and (CompareText(S, 'finalization') = 0) then Tokens.Add(S, ID_finalization, Col, Row, H)
          else
          if (H = HashFinally) and (CompareText(S, 'finally') = 0) then Tokens.Add(S, ID_Finally, Col, Row, H)
          else
          if (H = HashFor) and (CompareText(S, 'for') = 0) then Tokens.Add(S, ID_for, Col, Row, H)
          else
          if (H = HashForward) and (CompareText(S, 'forward') = 0) then Tokens.Add(S, ID_Forward, Col, Row, H)
          else
          if (H = HashFunction) and (CompareText(S, 'function') = 0) then Tokens.Add(S, ID_function, Col, Row, H)
          else
          if (H = HashMethod) and (CompareText(S, 'method') = 0) then Tokens.Add(S, ID_method, Col, Row, H)
          else
          if (H = HashIf) and (CompareText(S, 'if') = 0) then Tokens.Add(S, ID_if, Col, Row, H)
          else
          if (H = HashImplement) and (CompareText(S, 'implementation') = 0) then Tokens.Add(S, ID_Implementation, Col, Row, H)
          else
          if (H = HashIn) and (CompareText(S, 'IN') = 0) then Tokens.Add(S, ID_in, Col, Row, H)
          else
          if (H = HashInherited) and (CompareText(S, 'inherited') = 0) then Tokens.Add(S, ID_inherited, Col, Row, H)
          else
          if (H = HashInitializ) and (CompareText(S, 'initialization') = 0) then Tokens.Add(S, ID_initialization, Col, Row, H)
          else
          if (H = HashInterface) and (CompareText(S, 'interface') = 0) then Tokens.Add(S, ID_Interface, Col, Row, H)
          else
          if (H = HashIs) and (CompareText(S, 'is') = 0) then Tokens.Add(S, ID_Is, Col, Row, H)
          else
          if (H = HashMod) and (CompareText(S, 'mod') = 0) then Tokens.Add(S, ID_mod, Col, Row, H)
          else
          if (H = HashNil) and (CompareText(S, 'nil') = 0) then Tokens.Add(S, ID_nil, Col, Row, H)
          else
          if (H = HashNot) and (CompareText(S, 'not') = 0) then Tokens.Add(S, ID_not, Col, Row, H)
          else
          if (H = HashOf) and (CompareText(S, 'of') = 0) then Tokens.Add(S, ID_of, Col, Row, H)
          else
          if (H = HashOn) and (CompareText(S, 'on') = 0) then Tokens.Add(S, ID_on, Col, Row, H)
          else
          if (H = HashOr) and (CompareText(S, 'or') = 0) then Tokens.Add(S, ID_or, Col, Row, H)
          else
          {if (H = HashOrd) and (CompareText(S, 'ord') = 0) then Tokens.Add(S, ID_Ord, Col, Row, H)
          else}
          if (H = HashOut) and (CompareText(S, 'out') = 0) then Tokens.Add(S, ID_out, Col, Row, H)
          else
          if (H = HashOverride) and (CompareText(S, 'override') = 0) then Tokens.Add(S, ID_override, Col, Row, H)
          else
          if (H = HashOverload) and (CompareText(S, 'overload') = 0) then Tokens.Add(S, ID_overload, Col, Row, H)
          else
          if (H = HashPrivate) and (CompareText(S, 'private') = 0) then Tokens.Add(S, ID_private, Col, Row, H)
          else
          if (H = HashProcedure) and (CompareText(S, 'procedure') = 0) then Tokens.Add(S, ID_procedure, Col, Row, H)
          else
          if (H = HashProgram) and (CompareText(S, 'program') = 0) then Tokens.Add(S, ID_program, Col, Row, H)
          else
          if (H = HashProperty) and (CompareText(S, 'property') = 0) then Tokens.Add(S, ID_property, Col, Row, H)
          else
          if (H = HashProtected) and (CompareText(S, 'protected') = 0) then Tokens.Add(S, ID_protected, Col, Row, H)
          else
          if (H = HashPublic) and (CompareText(S, 'public') = 0) then Tokens.Add(S, ID_public, Col, Row, H)
          else
          if (H = HashPublished) and (CompareText(S, 'published') = 0) then Tokens.Add(S, ID_published, Col, Row, H)
          else
          if (H = HashRecord) and (CompareText(S, 'record') = 0) then Tokens.Add(S, ID_record, Col, Row, H)
          else
          if (H = HashRepeat) and (CompareText(S, 'repeat') = 0) then Tokens.Add(S, ID_repeat, Col, Row, H)
          else
          if (H = HashSet) and (CompareText(S, 'set') = 0) then Tokens.Add(S, ID_set, Col, Row, H)
          else
          if (H = HashShl) and (CompareText(S, 'shl') = 0) then Tokens.Add(S, ID_shl, Col, Row, H)
          else
          if (H = HashShr) and (CompareText(S, 'shr') = 0) then Tokens.Add(S, ID_shr, Col, Row, H)
          else
          if (H = HashThen) and (CompareText(S, 'then') = 0) then Tokens.Add(S, ID_then, Col, Row, H)
          else
          if (H = HashTo) and (CompareText(S, 'to') = 0) then Tokens.Add(S, ID_to, Col, Row, H)
          else
          if (H = HashTry) and (CompareText(S, 'try') = 0) then Tokens.Add(S, ID_Try, Col, Row, H)
          else
          if (H = HashType) and (CompareText(S, 'type') = 0) then Tokens.Add(S, ID_type, Col, Row, H)
          else
          if (H = HashUnit) and (CompareText(S, 'unit') = 0) then Tokens.Add(S, ID_Unit, Col, Row, H)
          else
          if (H = HashUntil) and (CompareText(S, 'until') = 0) then Tokens.Add(S, ID_until, Col, Row, H)
          else
          if (H = HashUses) and (CompareText(S, 'uses') = 0) then Tokens.Add(S, ID_uses, Col, Row, H)
          else
          if (H = HashVar) and (CompareText(S, 'var') = 0) then Tokens.Add(S, ID_var, Col, Row, H)
          else
          if (H = HashVirtual) and (CompareText(S, 'virtual') = 0) then Tokens.Add(S, ID_virtual, Col, Row, H)
          else
          if (H = HashWhile) and (CompareText(S, 'while') = 0) then Tokens.Add(S, ID_while, Col, Row, H)
          else
          if (H = HashWith) and (CompareText(S, 'with') = 0) then Tokens.Add(S, ID_with, Col, Row, H)
          else
          if (H = HashXor) and (CompareText(S, 'xor') = 0) then Tokens.Add(S, ID_xor, Col, Row, H)
          else
          if (H = HashBreak) and (CompareText(S, 'break') = 0) then Tokens.Add(S, ID_Break, Col, Row, H)
          else
          if (H = HashContinue) and (CompareText(S, 'continue') = 0) then Tokens.Add(S, ID_Continue, Col, Row, H)
          else
          if (H = HashTrue) and (CompareText(S, 'true') = 0) then Tokens.Add(S, ID_True, Col, Row, H)
          else
          if (H = HashFalse) and (CompareText(S, 'false') = 0) then Tokens.Add(S, ID_False, Col, Row, H)
          else
          begin
              Tokens.Add(S, ID_Identifier, Col, Row, H);
          end;
        end;
      end;
      dsInteger:
      begin
        S := AsString;
        Tokens.Add(S, ID_Integer, Col, Row, 0);
      end;
      dsHexInt:
      begin
        S := AsString;
        Tokens.Add(S, ID_Integer, Col, Row, 0);
      end;
      dsFloat:
      begin
        S := AsString;
        Tokens.Add(S, ID_Float, Col, Row, 0);
      end;
      dsText:
      begin
        S := AsText;
        if Length(S) = 1 then
          Tokens.Add(S, ID_Char, Col, Row, 0)
        else
          Tokens.Add(S, ID_String, Col, Row, 0);
      end;
      dsEOF:
      begin
        S := '';
        Tokens.Add(S, ID_EOF, Col, Row, 0);
        Break;
      end;
      dsUnknSym:
      begin
        S := AsString;
        Tokens.Add(S, ID_Unknown, Col, Row, 0);
      end;
    end;
    C := NextToken;
  end;
end;

{TTokensList}

constructor TTokensList.Create;
begin
  FList := TList.Create;
  FPos := 0;
  UnitSection := usInterface;
end;

destructor TTokensList.Destroy;
begin
  Clear;
  FList.Free;
  inherited Destroy;
end;

function TTokensList.Count: Integer;
begin
  Result := FList.Count;
end;

procedure TTokensList.Clear;
var
  PR: PTokenRec;
  I: Integer;
begin
  I := 0;
  while I < FList.Count do
  begin
    PR := PTokenRec(FList.Items[I]);
    Dispose(PR);
    FList.Items[I] := nil;
    Inc(I);
  end;
  FList.Clear;
end;

function TTokensList.Add(TokenName: string; TokenID: TIdentToken;
  Col, Row: Integer; HV: Cardinal): Integer;
var
  PR: PTokenRec;
begin
  New(PR);
  PR^.TokenName := TokenName;
  PR^.TokenID := TokenID;
  PR^.Col := Col;
  PR^.Row := Row;
  PR^.HashValue := HV;
  Result := FList.Add(PR);
end;

function TTokensList.Get(Index: Integer): TTokenRec;
begin

  if (Index >= 0) and (Index < FList.Count) then
    Result := PTokenRec(FList.Items[Index])^
  else raise Exception.Create('Index out of bounds. ' + IntToStr(Index));
end;

function TTokensList.GetToken: TTokenRec;
begin
  Result := Get(FPos);
end;

function TTokensList.GetPreviewNext: TTokenRec;
begin
  Result := Get(FPos + 1);
end;

procedure TTokensList.Next;
begin
  if FPos < FList.Count - 1 then Inc(FPos);
end;

procedure TTokensList.Prev;
begin
  if FPos > 0 then Dec(FPos);
end;

initialization

  HashAnd := Hash('and');
  HashBegin := Hash('begin');
  HashEnd := Hash('end');
  HashArray := Hash('array');
  HashAs := Hash('as');
  HashCase := Hash('case');
  HashChr := Hash('chr');
  HashConst := Hash('const');
  HashClass := Hash('class');
  HashConstr := Hash('constructor');
  HashDestr := Hash('destructor');
  HashDiv := Hash('div');
  HashDo := Hash('do');
  HashDownto := Hash('downto');
  HashElse := Hash('else');
  HashElseif := Hash('elseif');
  HashExcept := Hash('except');
  HashExit := Hash('exit');
  HashExport := Hash('export');
  HashExternal := Hash('external');
  HashFinaliz := Hash('finalization');
  HashFinally := Hash('finally');
  HashFor := Hash('for');
  HashForward := Hash('forward');
  HashFunction := Hash('function');
  HashProcedure := Hash('procedure');
  HashMethod := Hash('method');
  HashIf := Hash('if');
  HashImplement := Hash('implementation');
  HashIn := Hash('in');
  HashInherited := Hash('inherited');
  HashInitializ := Hash('initialization');
  HashInterface := Hash('interface');
  HashIs := Hash('is');
  HashLabel := Hash('label');
  HashMod := Hash('mod');
  HashNil := Hash('nil');
  HashNot := Hash('not');
  HashOf := Hash('of');
  HashOr := Hash('or');
  HashOrd := Hash('ord');
  HashOut := Hash('out');
  HashOverride := Hash('override');
  HashOverload := Hash('overload');
  HashPrivate := Hash('private');
  HashProtected := Hash('protected');
  HashProgram := Hash('program');
  HashProperty := Hash('property');
  HashPublic := Hash('public');
  HashPublished := Hash('published');
  HashRecord := Hash('record');
  HashRepeat := Hash('repeat');
  HashSet := Hash('set');
  HashShl := Hash('shl');
  HashShr := Hash('shr');
  HashThen := Hash('then');
  HashTo := Hash('to');
  HashTry := Hash('try');
  HashType := Hash('type');
  HashUnit := Hash('unit');
  HashUntil := Hash('until');
  HashUses := Hash('uses');
  HashVar := Hash('var');
  HashVirtual := Hash('virtual');
  HashWhile := Hash('while');
  HashWith := Hash('with');
  HashXor := Hash('xor');
  HashBreak := Hash('break');
  HashContinue := Hash('continue');
  HashTrue := Hash('true');
  HashFalse := Hash('false');
  HashString := Hash('string');
  HashOn := Hash('on');

  KeyAnd := 'и';
  KeyArray := 'массив';
  KeyBegin := 'начало';
  KeyEnd := 'конец';
  KeyAs := 'как';
  KeyCase := 'выбор';
  KeyChr := 'символ';
  KeyConst := 'конст';
  KeyClass := 'класс';
  KeyConstr := 'конструктор';
  KeyDestr := 'деструктор';
  KeyDiv := 'делить';
  KeyDo := 'выполнять';
  KeyDownto := 'вниздо';
  KeyElse := 'иначе';
  KeyElseif := 'иначеесли';
  KeyExcept := 'исключ';
  KeyExit := 'выход';
  KeyExport := 'экспорт';
  KeyExternal := 'внешний';
  KeyFinaliz := 'завершение';
  KeyFinally := 'завершить';
  KeyFor := 'для';
  KeyForward := 'опережающее';
  KeyFunction := 'функция';
  KeyProcedure := 'процедура';
  KeyMethod := 'метод';
  KeyIf := 'если';
  KeyImplement := 'реализация';
  KeyIn := 'естьв';
  KeyInherited := 'родительский';
  KeyInitializ := 'инициализация';
  KeyInterface := 'интерфейс';
  KeyIs := 'есть';
  KeyLabel := 'метка';
  KeyMod := 'мод';
  KeyNil := 'пусто';
  KeyNot := 'не';
  KeyOf := 'из';
  KeyOr := 'или';
  //KeyOrd := 'перечисл_знач';
  KeyOut := 'возвр';
  KeyOverride := 'переопределен';
  KeyOverload := 'перегружен';
  KeyPrivate := 'закрытые';
  KeyProtected := 'защищенные';
  KeyPublic := 'открытые';
  KeyPublished := 'опубликованные';
  KeyProgram := 'программа';
  KeyProperty := 'свойство';
  KeyRecord := 'запись';
  KeyRepeat := 'повторять';
  KeySet := 'множество';
  KeyShl := 'сдвигвлево';
  KeyShr := 'сдвигвправо';
  KeyThen := 'тогда';
  KeyTo := 'до';
  KeyTry := 'проба';
  KeyType := 'тип';
  KeyUnit := 'модуль';
  KeyUntil := 'повтордо';
  KeyUses := 'подключить';
  KeyVar := 'перем';
  KeyVirtual := 'виртуальный';
  KeyWhile := 'пока';
  KeyWith := 'вместе_с';
  KeyXor := 'и_или';
  KeyBreak := 'прервать';
  KeyContinue := 'продолжить';
  KeyTrue := 'правда';
  KeyFalse := 'ложь';
  KeyString := 'строка';
  KeyOn := 'на';

  LHashAnd := Hash(KeyAnd);
  LHashArray := Hash(KeyArray);
  LHashBegin := Hash(KeyBegin);
  LHashEnd := Hash(KeyEnd);
  LHashAs := Hash(KeyAs);
  LHashCase := Hash(KeyCase);
  LHashChr := Hash(KeyChr);
  LHashConst := Hash(KeyConst);
  LHashClass := Hash(KeyClass);
  LHashConstr := Hash(KeyConstr);
  LHashDestr := Hash(KeyDestr);
  LHashDiv := Hash(KeyDiv);
  LHashDo := Hash(KeyDo);
  LHashDownto := Hash(KeyDownto);
  LHashElse := Hash(KeyElse);
  LHashElseif := Hash(KeyElseif);
  LHashExcept := Hash(KeyExcept);
  LHashExit := Hash(KeyExit);
  LHashExport := Hash(KeyExport);
  LHashExternal := Hash(KeyExternal);
  LHashFinaliz := Hash(KeyFinaliz);
  LHashFinally := Hash(KeyFinally);
  LHashFor := Hash(KeyFor);
  LHashForward := Hash(KeyForward);
  LHashFunction := Hash(KeyFunction);
  LHashProcedure := Hash(KeyProcedure);
  LHashMethod := Hash(KeyMethod);
  LHashIf := Hash(KeyIf);
  LHashImplement := Hash(KeyImplement);
  LHashIn := Hash(KeyIn);
  LHashInherited := Hash(KeyInherited);
  LHashInitializ := Hash(KeyInitializ);
  LHashInterface := Hash(KeyInterface);
  LHashIs := Hash(KeyIs);
  LHashLabel := Hash(KeyLabel);
  LHashMod := Hash(KeyMod);
  LHashNil := Hash(KeyNil);
  LHashNot := Hash(KeyNot);
  LHashOf := Hash(KeyOf);
  LHashOr := Hash(KeyOr);
  //LHashOrd := Hash(KeyOrd);
  LHashOut := Hash(KeyOut);
  LHashOverride := Hash(KeyOverride);
  LHashOverload := Hash(KeyOverload);
  LHashPrivate := Hash(KeyPrivate);
  LHashProtected := Hash(KeyProtected);
  LHashPublic := Hash(KeyPublic);
  LHashPublished := Hash(KeyPublished);
  LHashProgram := Hash(KeyProgram);
  LHashProperty := Hash(KeyProperty);
  LHashRecord := Hash(KeyRecord);
  LHashRepeat := Hash(KeyRepeat);
  LHashSet := Hash(KeySet);
  LHashShl := Hash(KeyShl);
  LHashShr := Hash(KeyShr);
  LHashThen := Hash(KeyThen);
  LHashTo := Hash(KeyTo);
  LHashTry := Hash(KeyTry);
  LHashType := Hash(KeyType);
  LHashUnit := Hash(KeyUnit);
  LHashUntil := Hash(KeyUntil);
  LHashUses := Hash(KeyUses);
  LHashVar := Hash(KeyVar);
  LHashVirtual := Hash(KeyVirtual);
  LHashWhile := Hash(KeyWhile);
  LHashWith := Hash(KeyWith);
  LHashXor := Hash(KeyXor);
  LHashBreak := Hash(KeyBreak);
  LHashContinue := Hash(KeyContinue);
  LHashTrue := Hash(KeyTrue);
  LHashFalse := Hash(KeyFalse);
  LHashString := Hash(KeyString);
  LHashOn := Hash(KeyOn);

end.

