{------------------------------------------------------------------------------}
{                                                                              }
{                              Yuriy Kopnin                                    }
{                            Package VisuaTech                                 }
{                                 LGPL                                         }
{                                                                              }
{------------------------------------------------------------------------------}

unit BarCodeHook;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs,
  ExtCtrls, LCLType;

type

  { TBarCodeHook }

  TBarCodeHook = class(TComponent)
  private
    FDelayTime: Integer;
    FBarCode: string;
    FOnBarCode: TNotifyEvent;
    FActive: Boolean;
    FTimer: TIdleTimer;
    FCompletingKey: Char;
    FAllowKey: Boolean;
    FOldFormKeyUp: TKeyEvent;
    FOldFormKeyDown: TKeyEvent;
    FOldFormKeyPreview: Boolean;
    FClearKey: Boolean;
    procedure SetActive(AValue: Boolean);
  protected
    procedure ApplyBarCode;
    procedure OnTimer(Sender: TObject);
    procedure BeforeKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure BeforeKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    function GetCharOfScanCode(AKey: Word): Char;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure ResetBarCode;
    property BarCode: string read FBarCode write FBarCode;
  published
    property DelayTime: Integer read FDelayTime write FDelayTime;
    property OnBarcode: TNotifyEvent read FOnBarCode write FOnBarCode;
    property ClearKey: Boolean read FClearKey write FClearKey default False;
    property Active: Boolean read FActive write SetActive default False;
    property CompletingKey: Char read FCompletingKey write FCompletingKey default #13;
  end;

implementation

uses LCLIntf, LMessages;

{ TBarCodeHook }

procedure TBarCodeHook.SetActive(AValue: Boolean);
begin
  if not (Owner is TCustomForm) then Exit;
  if FActive=AValue then Exit;

  FActive:=AValue;

  if FActive then
  begin
    if not (csDesigning in ComponentState) then
    begin
      FOldFormKeyPreview := TCustomForm(Owner).KeyPreview;
      FOldFormKeyDown := TCustomForm(Owner).OnKeyDown;
      FOldFormKeyUp := TCustomForm(Owner).OnKeyUp;

      TCustomForm(Owner).KeyPreview := True;
      TCustomForm(Owner).OnKeyDown := @BeforeKeyDown;
      TCustomForm(Owner).OnKeyUp := @BeforeKeyUp;
    end;
  end
  else
  begin
    TCustomForm(Owner).KeyPreview := FOldFormKeyPreview;
    TCustomForm(Owner).OnKeyUp := FOldFormKeyUp;
    TCustomForm(Owner).OnKeyDown := FOldFormKeyDown;
  end;
end;

procedure TBarCodeHook.ApplyBarCode;
begin
  FTimer.Enabled := False;
  try
    if (BarCode <> '') and Assigned(FOnBarCode) then FOnBarCode(Self);
  finally
    ResetBarCode;
  end;
end;

procedure TBarCodeHook.OnTimer(Sender: TObject);
begin
  FTimer.Enabled := False;
  ResetBarCode;
end;

constructor TBarCodeHook.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBarCode := '';
  FDelayTime := 45;
  FActive:= False;
  FCompletingKey := #13;
  FClearKey := False;
  FTimer := TIdleTimer.Create(Self);
  FTimer.Interval := FDelayTime;
  FTimer.AutoEnabled := False;
  FTimer.Enabled:= False;
  FTimer.OnTimer := @OnTimer;
end;

destructor TBarCodeHook.Destroy;
begin
  FTimer.Free;
  inherited Destroy;
end;

procedure TBarCodeHook.BeforeKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  EnableTimer: Boolean;
  CharKey: Char;
begin
  FTimer.Enabled := False;
  EnableTimer := False;
  if Key = 16 then
  begin
    if ClearKey then Key := 0;
    FTimer.Enabled := True;
  end
  else
  begin
    CharKey := GetCharOfScanCode(Key);
    if ClearKey then Key := 0;
    if (CharKey <> #0) then
    begin
      if (CharKey = FCompletingKey) and (FBarCode <> '') then ApplyBarCode
      else
      begin
        FBarCode := FBarCode + CharKey;
        EnableTimer := True;
        if EnableTimer then FTimer.Enabled := True;
      end
    end
    else
    begin
      ResetBarCode;
    end;
  end;
end;

procedure TBarCodeHook.BeforeKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if ClearKey then
  begin
    if (Key = 16) or (Key = 13) or ((Key >= 48) and (Key <= 90)) then Key := 0;
  end;
end;

function TBarCodeHook.GetCharOfScanCode(AKey: Word): Char;
var
  B: Byte;
begin
  Result := #0;
  if ((AKey >= 48) and (AKey <= 57)) or ((AKey >= 65) and (AKey <= 90)) then
  begin
    case AKey of
      48: Result := '0';
      49: Result := '1';
      50: Result := '2';
      51: Result := '3';
      52: Result := '4';
      53: Result := '5';
      54: Result := '6';
      55: Result := '7';
      56: Result := '8';
      57: Result := '9';


      65: Result := 'A';
      66: Result := 'B';
      67: Result := 'C';
      68: Result := 'D';
      69: Result := 'E';
      70: Result := 'F';
      71: Result := 'G';
      72: Result := 'H';
      73: Result := 'I';
      74: Result := 'J';
      75: Result := 'K';
      76: Result := 'L';
      77: Result := 'M';
      78: Result := 'N';
      79: Result := 'O';
      80: Result := 'P';
      81: Result := 'Q';
      82: Result := 'R';
      83: Result := 'S';
      84: Result := 'T';
      85: Result := 'U';
      86: Result := 'V';
      87: Result := 'W';
      88: Result := 'X';
      89: Result := 'Y';
      90: Result := 'Z';
    end;
  end
  else
  begin
    if AKey <= 255 then
    begin
      B := AKey;
      if chr(B) = FCompletingKey then
        Result := FCompletingKey;
    end;
  end;
end;

procedure TBarCodeHook.ResetBarCode;
begin
  FBarCode := '';
  FTimer.Enabled := False;
end;

end.
