////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Color name routines: convert color names to colors codes and backward.     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20180131 - Creation, first release                                         //
// 20220213 - cl_xxx colors,FwXXX colors,KnownColorsListGroup,ColorToNumber   //
// 20230515 - Modified for FPC (A.K.)                                         //
// 20240511 - TRGBRec,ColorToWebNum,ColorToRgbNum,FindNearestKnownColor       //
// 20250809 - ColorToNumber                                                   //
////////////////////////////////////////////////////////////////////////////////

unit _crw_colors; // Named colors service routines.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, graphics, math, lclintf,
 _crw_alloc, _crw_str, _crw_hl, _crw_base64;

type // Color to RGB: Red,Green,Blue,Type.
 TRGBRec = packed record R,G,B,T:Byte; end;

 // Convert color to decimal number like 123456.
function ColorToNumber(ColorCode:TColor):LongString;

 // Convert color to Hex number $GGBBRR like $123456.
function ColorToHexNum(ColorCode:TColor; prec:Integer=6):LongString;

 // Convert code to Web number #RRGGBB like #00ff00.
function ColorToWebNum(ColorCode:TColor):LongString;

 // Convert code to RGB numbers R,G,B like 0,255,0.
function ColorToRgbNum(ColorCode:TColor):LongString;

 // Convert ColorCode to known color name or $GGBBRR code of RGB
function ColorToString(ColorCode:TColor; Prefix:LongString=''):LongString;

 // Convert ColorName (known color name or $GGBBRR code) to color code
 // Return DefaultColor if convertion failed
function StringToColor(ColorName:LongString; DefaultColor:TColor=clNone):TColor;

const KnownColorsList       : TStringList = nil; // List of known colors (names and codes)
const KnownColorsListGroup  : TList       = nil; // List of known colors (color group idn)
const KnownColorsListByName : THashList   = nil; // List of known colors indexed by name
const KnownColorsListByCode : THashList   = nil; // List of known colors indexed by code

procedure InitKnownColorsList;
procedure FreeKnownColorsList;

 // List of known colors
function KnownColorsCount:Integer;
function KnownColorsName(Index:Integer):LongString;
function KnownColorsCode(Index:Integer):TColor;
function KnownColorsGroup(Index:Integer):LongString;     // One of (JCOP,SYS,X11,WEB,SYN,VGA)
function KnownColorsListAsText(Mode:Integer):LongString; // 0/1/2/3=Names/Pas/Crc/Htm  format

 // Find nearest known named RGB color relative to given aColor.
function FindNearestKnownColor(aColor:TColor):TColor; overload;
function FindNearestKnownColor(aC:LongString):TColor; overload;

 // Specially for DaqPascal ParamStr('ColorInfo …').
function ParamStrColorInfo(arg:LongString):LongString;

const // Params for HTML generation
 KnownColorsListCookies : LongString =
  'lang=ru;charset=utf-8;'+
  'favicon=../daqsite/painter/colors.ico;'+
  'crw-daq-ico=../icon/crwdaq.ico;'+
  'id-jcop=fwcolors.pdf;'+
  'id-sys=../daqsite/painter/colors.htm;'+
  'id-x11=../daqsite/painter/colors.htm;'+
  'id-web=../daqsite/painter/colors.htm;'+
  'id-vga=../daqsite/painter/colors.htm;'+
  'id-syn=../daqsite/painter/colors.htm;';

/////////////////////////////////////////////
// Windows functional colors converted to RGB
/////////////////////////////////////////////
function cl_ScrollBar               : TColor; 
function cl_Background              : TColor; 
function cl_ActiveCaption           : TColor; 
function cl_InactiveCaption         : TColor; 
function cl_Menu                    : TColor; 
function cl_Window                  : TColor; 
function cl_WindowFrame             : TColor; 
function cl_MenuText                : TColor; 
function cl_WindowText              : TColor; 
function cl_CaptionText             : TColor; 
function cl_ActiveBorder            : TColor; 
function cl_InactiveBorder          : TColor; 
function cl_AppWorkSpace            : TColor; 
function cl_Highlight               : TColor; 
function cl_HighlightText           : TColor; 
function cl_BtnFace                 : TColor; 
function cl_BtnShadow               : TColor; 
function cl_GrayText                : TColor; 
function cl_BtnText                 : TColor; 
function cl_InactiveCaptionText     : TColor; 
function cl_BtnHighlight            : TColor; 
function cl_3DDkShadow              : TColor; 
function cl_3DLight                 : TColor; 
function cl_InfoText                : TColor; 
function cl_InfoBk                  : TColor; 
function cl_HotLight                : TColor; 
function cl_GradientActiveCaption   : TColor; 
function cl_GradientInactiveCaption : TColor; 
function cl_MenuHilight             : TColor; 
function cl_MenuBar                 : TColor; 
//////////////////////////////////////////////// Synonyms
function cl_Desktop                 : TColor; // BACKGROUND
function cl_3DFace                  : TColor; // BTNFACE
function cl_3DText                  : TColor; // BTNTEXT
function cl_3DShadow                : TColor; // BTNSHADOW
function cl_3DHighlight             : TColor; // BTNHIGHLIGHT
function cl_3DHilight               : TColor; // BTNHIGHLIGHT
function cl_BtnHilight              : TColor; // BTNHIGHLIGHT
function cl_Button                  : TColor; // BTNFACE
function cl_ButtonText              : TColor; // BTNTEXT
function cl_ButtonShadow            : TColor; // BTNSHADOW
function cl_ToolTip                 : TColor; // INFOBK
function cl_ToolTipText             : TColor; // INFOTEXT

procedure UpdateSysColors;

implementation

procedure InitKnownColorsList;
var i,g:Integer; n,h:LongString; c:TColor;
 procedure AddColor(Name:LongString; Color:TColor);
 begin
  if Assigned(KnownColorsList) then begin
   KnownColorsList.AddObject(Name,PtrIntToPointer(Color));
  end;
  if Assigned(KnownColorsListGroup) then begin
   KnownColorsListGroup.Add(PtrIntToPointer(g));
  end;
 end;
 procedure ColorGroup(id,crc,note:LongString);
 begin
  g:=Dump2i(id);
 end;
begin
 if not Assigned(KnownColorsList) then begin
  KnownColorsList:=TStringList.Create;
  KnownColorsList.Sorted:=false;
  KnownColorsListGroup:=TList.Create;
  UpdateSysColors;
  g:=Dump2i('??');
  /////////////////////////////////////////////////
  ColorGroup('JCOP','ColJFw.crc','JCOP Framework');
  /////////////////////////////////////////////////
  AddColor('FwBackground',                        cl_3DFace);
  AddColor('FwBackgroundInputText',               cl_Window);
  AddColor('FwForeground',                        cl_3DText);
  AddColor('FwForegroundInputText',               cl_WindowText);
  AddColor('DpDoesNotExist',                      $CC3399);
  AddColor('FwAlarmColoursDoNotUse',              $000000);
  AddColor('FwAlarmErrorAck',                     $3399FF);
  AddColor('FwAlarmErrorUnack',                   $3399FF);
  AddColor('FwAlarmErrorWentUnack',               $999999);
  AddColor('FwAlarmFatalAck',                     $0000FF);
  AddColor('FwAlarmFatalUnack',                   $0000FF);
  AddColor('FwAlarmFatalWentUnack',               $999999);
  AddColor('FwAlarmMasked',                       $CCCCCC);
  AddColor('FwAlarmTitleBG',                      $999999);
  AddColor('FwAlarmWarnAck',                      $33FFFF);
  AddColor('FwAlarmWarnUnack',                    $33FFFF);
  AddColor('FwAlarmWarnWentUnack',                $999999);
  AddColor('FwBackgroundUserAttention',           $8080FF);
  AddColor('FwCorporateColor',                    $FFC800);
  AddColor('FwDead',                              $CCCCCC);
  AddColor('FwDeprecatedColoursDoNotUse',         $000000);
  AddColor('FwEquipmentDisabled',                 $CCCCCC);
  AddColor('FwInstallationAltRowColorDark',       $F5F5F5);
  AddColor('FwInstallationAltRowColorLight',      $FFFFFF);
  AddColor('FwInstallationCorporateColor',        $FFC800);
  AddColor('FwInstallationTableSelection',        $FFF0C8);
  AddColor('FwModeIncluded',                      $99CC00);
  AddColor('FwModeOther',                         $FF9900);
  AddColor('FwModeTreeIncomplete',                $66FFFF);
  AddColor('FwPanelColoursDoNotUse',              $000000);
  AddColor('FwProvidedComponentsColoursDoNotUse', $000000);
  AddColor('FwStateAttention1',                   $66FFFF);
  AddColor('FwStateAttention2',                   $0099FF);
  AddColor('FwStateAttention3',                   $0000FF);
  AddColor('FwStateOKNotPhysics',                 $FF9900);
  AddColor('FwStateOKPhysics',                    $99CC00);
  AddColor('FwTableSelection',                    $FFF0C8);
  AddColor('FwTrendingCurve1',                    $0000FF);
  AddColor('FwTrendingCurve2',                    $FF0000);
  AddColor('FwTrendingCurve3',                    $009900);
  AddColor('FwTrendingCurve4',                    $FFFF00);
  AddColor('FwTrendingCurve5',                    $003366);
  AddColor('FwTrendingCurve6',                    $00CCFF);
  AddColor('FwTrendingCurve7',                    $FF00FF);
  AddColor('FwTrendingCurve8',                    $00FF00);
  AddColor('FwTrendingDataNoAccess',              $000000);
  AddColor('FwTrendingFaceplateBackground',       $C8D0D4);
  AddColor('FwTrendingFaceplateForeground',       $000000);
  AddColor('FwTrendingFaceplateTrendBackground',  $C8D0DA);
  AddColor('FwTrendingTrendBackground',           $FFFFFF);
  AddColor('FwTrendingTrendForeground',           $000000);
  AddColor('FwTrendingViewBackground',            $666666);
  AddColor('FwTrendingViewForeground',            $000000);
  /////////////////////////////////////////////////
  ColorGroup('SYS','ColWin.crc','System Colors');
  /////////////////////////////////////////////////
  // Special functional color "none"
  AddColor('_None',                    clNone);
  // Windows functional colors - GetSysColor()
  AddColor('_ScrollBar',               cl_ScrollBar);
  AddColor('_Background',              cl_Background);
  AddColor('_ActiveCaption',           cl_ActiveCaption);
  AddColor('_InactiveCaption',         cl_InactiveCaption);
  AddColor('_Menu',                    cl_Menu);
  AddColor('_Window',                  cl_Window);
  AddColor('_WindowFrame',             cl_WindowFrame);
  AddColor('_MenuText',                cl_MenuText);
  AddColor('_WindowText',              cl_WindowText);
  AddColor('_CaptionText',             cl_CaptionText);
  AddColor('_ActiveBorder',            cl_ActiveBorder);
  AddColor('_InactiveBorder',          cl_InactiveBorder);
  AddColor('_AppWorkSpace',            cl_AppWorkSpace);
  AddColor('_Highlight',               cl_Highlight);
  AddColor('_HighlightText',           cl_HighlightText);
  AddColor('_BtnFace',                 cl_BtnFace);
  AddColor('_BtnShadow',               cl_BtnShadow);
  AddColor('_GrayText',                cl_GrayText);
  AddColor('_BtnText',                 cl_BtnText);
  AddColor('_InactiveCaptionText',     cl_InactiveCaptionText);
  AddColor('_BtnHighlight',            cl_BtnHighlight);
  AddColor('_3DDkShadow',              cl_3DDkShadow);
  AddColor('_3DLight',                 cl_3DLight);
  AddColor('_InfoText',                cl_InfoText);
  AddColor('_InfoBk',                  cl_InfoBk);
  AddColor('_HotLight',                cl_HotLight);
  AddColor('_GradientActiveCaption',   cl_GradientActiveCaption);
  AddColor('_GradientInactiveCaption', cl_GradientInactiveCaption);
  AddColor('_MenuHilight',             cl_MenuHilight);
  AddColor('_MenuBar',                 cl_MenuBar);
  // Synonyms of Windows functional colors (JCOP)
  AddColor('_Button',                  cl_Button);
  AddColor('_ButtonText',              cl_ButtonText);
  AddColor('_ButtonShadow',            cl_ButtonShadow);
  AddColor('_ToolTip',                 cl_ToolTip);
  AddColor('_ToolTipText',             cl_ToolTipText);
  // Synonyms of Windows functional colors (WINAPI)
  AddColor('_Desktop',                 cl_Desktop);
  AddColor('_3DFace',                  cl_3DFace);
  AddColor('_3DText',                  cl_3DText);
  AddColor('_3DShadow',                cl_3DShadow);
  AddColor('_3DHighlight',             cl_3DHighlight);
  AddColor('_3DHilight',               cl_3DHilight);
  AddColor('_BtnHilight',              cl_BtnHilight);
  /////////////////////////////////////////////////
  ColorGroup('X11','ColX11.crc','X Window System');
  /////////////////////////////////////////////////
  AddColor('snow',$FAFAFF);
  AddColor('ghostwhite',$FFF8F8);
  AddColor('whitesmoke',$F5F5F5);
  AddColor('gainsboro',$DCDCDC);
  AddColor('floralwhite',$F0FAFF);
  AddColor('oldlace',$E6F5FD);
  AddColor('linen',$E6F0FA);
  AddColor('antiquewhite',$D7EBFA);
  AddColor('papayawhip',$D5EFFF);
  AddColor('blanchedalmond',$CDEBFF);
  AddColor('bisque',$C4E4FF);
  AddColor('peachpuff',$B9DAFF);
  AddColor('navajowhite',$ADDEFF);
  AddColor('moccasin',$B5E4FF);
  AddColor('cornsilk',$DCF8FF);
  AddColor('ivory',$F0FFFF);
  AddColor('lemonchiffon',$CDFAFF);
  AddColor('seashell',$EEF5FF);
  AddColor('honeydew',$F0FFF0);
  AddColor('mintcream',$FAFFF5);
  AddColor('azure',$FFFFF0);
  AddColor('aliceblue',$FFF8F0);
  AddColor('lavender',$FAE6E6);
  AddColor('lavenderblush',$F5F0FF);
  AddColor('mistyrose',$E1E4FF);
  AddColor('white',$FFFFFF);
  AddColor('black',$000000);
  AddColor('darkslategray',$4F4F2F);
  AddColor('darkslategrey',$4F4F2F);
  AddColor('dimgray',$696969);
  AddColor('dimgrey',$696969);
  AddColor('slategray',$908070);
  AddColor('slategrey',$908070);
  AddColor('lightslategray',$998877);
  AddColor('lightslategrey',$998877);
  AddColor('gray',$BEBEBE);
  AddColor('grey',$BEBEBE);
  AddColor('lightgrey',$D3D3D3);
  AddColor('lightgray',$D3D3D3);
  AddColor('midnightblue',$701919);
  AddColor('navy',$800000);
  AddColor('navyblue',$800000);
  AddColor('cornflowerblue',$ED9564);
  AddColor('darkslateblue',$8B3D48);
  AddColor('slateblue',$CD5A6A);
  AddColor('mediumslateblue',$EE687B);
  AddColor('lightslateblue',$FF7084);
  AddColor('mediumblue',$CD0000);
  AddColor('royalblue',$E16941);
  AddColor('blue',$FF0000);
  AddColor('dodgerblue',$FF901E);
  AddColor('deepskyblue',$FFBF00);
  AddColor('skyblue',$EBCE87);
  AddColor('lightskyblue',$FACE87);
  AddColor('steelblue',$B48246);
  AddColor('lightsteelblue',$DEC4B0);
  AddColor('lightblue',$E6D8AD);
  AddColor('powderblue',$E6E0B0);
  AddColor('paleturquoise',$EEEEAF);
  AddColor('darkturquoise',$D1CE00);
  AddColor('mediumturquoise',$CCD148);
  AddColor('turquoise',$D0E040);
  AddColor('cyan',$FFFF00);
  AddColor('lightcyan',$FFFFE0);
  AddColor('cadetblue',$A09E5F);
  AddColor('mediumaquamarine',$AACD66);
  AddColor('aquamarine',$D4FF7F);
  AddColor('darkgreen',$006400);
  AddColor('darkolivegreen',$2F6B55);
  AddColor('darkseagreen',$8FBC8F);
  AddColor('seagreen',$578B2E);
  AddColor('mediumseagreen',$71B33C);
  AddColor('lightseagreen',$AAB220);
  AddColor('palegreen',$98FB98);
  AddColor('springgreen',$7FFF00);
  AddColor('lawngreen',$00FC7C);
  AddColor('green',$00FF00);
  AddColor('chartreuse',$00FF7F);
  AddColor('mediumspringgreen',$9AFA00);
  AddColor('greenyellow',$2FFFAD);
  AddColor('limegreen',$32CD32);
  AddColor('yellowgreen',$32CD9A);
  AddColor('forestgreen',$228B22);
  AddColor('olivedrab',$238E6B);
  AddColor('darkkhaki',$6BB7BD);
  AddColor('khaki',$8CE6F0);
  AddColor('palegoldenrod',$AAE8EE);
  AddColor('lightgoldenrodyellow',$D2FAFA);
  AddColor('lightyellow',$E0FFFF);
  AddColor('yellow',$00FFFF);
  AddColor('gold',$00D7FF);
  AddColor('lightgoldenrod',$82DDEE);
  AddColor('goldenrod',$20A5DA);
  AddColor('darkgoldenrod',$0B86B8);
  AddColor('rosybrown',$8F8FBC);
  AddColor('indianred',$5C5CCD);
  AddColor('saddlebrown',$13458B);
  AddColor('sienna',$2D52A0);
  AddColor('peru',$3F85CD);
  AddColor('burlywood',$87B8DE);
  AddColor('beige',$DCF5F5);
  AddColor('wheat',$B3DEF5);
  AddColor('sandybrown',$60A4F4);
  AddColor('tan',$8CB4D2);
  AddColor('chocolate',$1E69D2);
  AddColor('firebrick',$2222B2);
  AddColor('brown',$2A2AA5);
  AddColor('darksalmon',$7A96E9);
  AddColor('salmon',$7280FA);
  AddColor('lightsalmon',$7AA0FF);
  AddColor('orange',$00A5FF);
  AddColor('darkorange',$008CFF);
  AddColor('coral',$507FFF);
  AddColor('lightcoral',$8080F0);
  AddColor('tomato',$4763FF);
  AddColor('orangered',$0045FF);
  AddColor('red',$0000FF);
  AddColor('hotpink',$B469FF);
  AddColor('deeppink',$9314FF);
  AddColor('pink',$CBC0FF);
  AddColor('lightpink',$C1B6FF);
  AddColor('palevioletred',$9370DB);
  AddColor('maroon',$6030B0);
  AddColor('mediumvioletred',$8515C7);
  AddColor('violetred',$9020D0);
  AddColor('magenta',$FF00FF);
  AddColor('violet',$EE82EE);
  AddColor('plum',$DDA0DD);
  AddColor('orchid',$D670DA);
  AddColor('mediumorchid',$D355BA);
  AddColor('darkorchid',$CC3299);
  AddColor('darkviolet',$D30094);
  AddColor('blueviolet',$E22B8A);
  AddColor('purple',$F020A0);
  AddColor('mediumpurple',$DB7093);
  AddColor('thistle',$D8BFD8);
  AddColor('snow1',$FAFAFF);
  AddColor('snow2',$E9E9EE);
  AddColor('snow3',$C9C9CD);
  AddColor('snow4',$89898B);
  AddColor('seashell1',$EEF5FF);
  AddColor('seashell2',$DEE5EE);
  AddColor('seashell3',$BFC5CD);
  AddColor('seashell4',$82868B);
  AddColor('AntiqueWhite1',$DBEFFF);
  AddColor('AntiqueWhite2',$CCDFEE);
  AddColor('AntiqueWhite3',$B0C0CD);
  AddColor('AntiqueWhite4',$78838B);
  AddColor('bisque1',$C4E4FF);
  AddColor('bisque2',$B7D5EE);
  AddColor('bisque3',$9EB7CD);
  AddColor('bisque4',$6B7D8B);
  AddColor('PeachPuff1',$B9DAFF);
  AddColor('PeachPuff2',$ADCBEE);
  AddColor('PeachPuff3',$95AFCD);
  AddColor('PeachPuff4',$65778B);
  AddColor('NavajoWhite1',$ADDEFF);
  AddColor('NavajoWhite2',$A1CFEE);
  AddColor('NavajoWhite3',$8BB3CD);
  AddColor('NavajoWhite4',$5E798B);
  AddColor('LemonChiffon1',$CDFAFF);
  AddColor('LemonChiffon2',$BFE9EE);
  AddColor('LemonChiffon3',$A5C9CD);
  AddColor('LemonChiffon4',$70898B);
  AddColor('cornsilk1',$DCF8FF);
  AddColor('cornsilk2',$CDE8EE);
  AddColor('cornsilk3',$B1C8CD);
  AddColor('cornsilk4',$78888B);
  AddColor('ivory1',$F0FFFF);
  AddColor('ivory2',$E0EEEE);
  AddColor('ivory3',$C1CDCD);
  AddColor('ivory4',$838B8B);
  AddColor('honeydew1',$F0FFF0);
  AddColor('honeydew2',$E0EEE0);
  AddColor('honeydew3',$C1CDC1);
  AddColor('honeydew4',$838B83);
  AddColor('LavenderBlush1',$F5F0FF);
  AddColor('LavenderBlush2',$E5E0EE);
  AddColor('LavenderBlush3',$C5C1CD);
  AddColor('LavenderBlush4',$86838B);
  AddColor('MistyRose1',$E1E4FF);
  AddColor('MistyRose2',$D2D5EE);
  AddColor('MistyRose3',$B5B7CD);
  AddColor('MistyRose4',$7B7D8B);
  AddColor('azure1',$FFFFF0);
  AddColor('azure2',$EEEEE0);
  AddColor('azure3',$CDCDC1);
  AddColor('azure4',$8B8B83);
  AddColor('SlateBlue1',$FF6F83);
  AddColor('SlateBlue2',$EE677A);
  AddColor('SlateBlue3',$CD5969);
  AddColor('SlateBlue4',$8B3C47);
  AddColor('RoyalBlue1',$FF7648);
  AddColor('RoyalBlue2',$EE6E43);
  AddColor('RoyalBlue3',$CD5F3A);
  AddColor('RoyalBlue4',$8B4027);
  AddColor('blue1',$FF0000);
  AddColor('blue2',$EE0000);
  AddColor('blue3',$CD0000);
  AddColor('blue4',$8B0000);
  AddColor('DodgerBlue1',$FF901E);
  AddColor('DodgerBlue2',$EE861C);
  AddColor('DodgerBlue3',$CD7418);
  AddColor('DodgerBlue4',$8B4E10);
  AddColor('SteelBlue1',$FFB863);
  AddColor('SteelBlue2',$EEAC5C);
  AddColor('SteelBlue3',$CD944F);
  AddColor('SteelBlue4',$8B6436);
  AddColor('DeepSkyBlue1',$FFBF00);
  AddColor('DeepSkyBlue2',$EEB200);
  AddColor('DeepSkyBlue3',$CD9A00);
  AddColor('DeepSkyBlue4',$8B6800);
  AddColor('SkyBlue1',$FFCE87);
  AddColor('SkyBlue2',$EEC07E);
  AddColor('SkyBlue3',$CDA66C);
  AddColor('SkyBlue4',$8B704A);
  AddColor('LightSkyBlue1',$FFE2B0);
  AddColor('LightSkyBlue2',$EED3A4);
  AddColor('LightSkyBlue3',$CDB68D);
  AddColor('LightSkyBlue4',$8B7B60);
  AddColor('SlateGray1',$FFE2C6);
  AddColor('SlateGray2',$EED3B9);
  AddColor('SlateGray3',$CDB69F);
  AddColor('SlateGray4',$8B7B6C);
  AddColor('LightSteelBlue1',$FFE1CA);
  AddColor('LightSteelBlue2',$EED2BC);
  AddColor('LightSteelBlue3',$CDB5A2);
  AddColor('LightSteelBlue4',$8B7B6E);
  AddColor('LightBlue1',$FFEFBF);
  AddColor('LightBlue2',$EEDFB2);
  AddColor('LightBlue3',$CDC09A);
  AddColor('LightBlue4',$8B8368);
  AddColor('LightCyan1',$FFFFE0);
  AddColor('LightCyan2',$EEEED1);
  AddColor('LightCyan3',$CDCDB4);
  AddColor('LightCyan4',$8B8B7A);
  AddColor('PaleTurquoise1',$FFFFBB);
  AddColor('PaleTurquoise2',$EEEEAE);
  AddColor('PaleTurquoise3',$CDCD96);
  AddColor('PaleTurquoise4',$8B8B66);
  AddColor('CadetBlue1',$FFF598);
  AddColor('CadetBlue2',$EEE58E);
  AddColor('CadetBlue3',$CDC57A);
  AddColor('CadetBlue4',$8B8653);
  AddColor('turquoise1',$FFF500);
  AddColor('turquoise2',$EEE500);
  AddColor('turquoise3',$CDC500);
  AddColor('turquoise4',$8B8600);
  AddColor('cyan1',$FFFF00);
  AddColor('cyan2',$EEEE00);
  AddColor('cyan3',$CDCD00);
  AddColor('cyan4',$8B8B00);
  AddColor('DarkSlateGray1',$FFFF97);
  AddColor('DarkSlateGray2',$EEEE8D);
  AddColor('DarkSlateGray3',$CDCD79);
  AddColor('DarkSlateGray4',$8B8B52);
  AddColor('aquamarine1',$D4FF7F);
  AddColor('aquamarine2',$C6EE76);
  AddColor('aquamarine3',$AACD66);
  AddColor('aquamarine4',$748B45);
  AddColor('DarkSeaGreen1',$C1FFC1);
  AddColor('DarkSeaGreen2',$B4EEB4);
  AddColor('DarkSeaGreen3',$9BCD9B);
  AddColor('DarkSeaGreen4',$698B69);
  AddColor('SeaGreen1',$9FFF54);
  AddColor('SeaGreen2',$94EE4E);
  AddColor('SeaGreen3',$80CD43);
  AddColor('SeaGreen4',$578B2E);
  AddColor('PaleGreen1',$9AFF9A);
  AddColor('PaleGreen2',$90EE90);
  AddColor('PaleGreen3',$7CCD7C);
  AddColor('PaleGreen4',$548B54);
  AddColor('SpringGreen1',$7FFF00);
  AddColor('SpringGreen2',$76EE00);
  AddColor('SpringGreen3',$66CD00);
  AddColor('SpringGreen4',$458B00);
  AddColor('green1',$00FF00);
  AddColor('green2',$00EE00);
  AddColor('green3',$00CD00);
  AddColor('green4',$008B00);
  AddColor('chartreuse1',$00FF7F);
  AddColor('chartreuse2',$00EE76);
  AddColor('chartreuse3',$00CD66);
  AddColor('chartreuse4',$008B45);
  AddColor('OliveDrab1',$3EFFC0);
  AddColor('OliveDrab2',$3AEEB3);
  AddColor('OliveDrab3',$32CD9A);
  AddColor('OliveDrab4',$228B69);
  AddColor('DarkOliveGreen1',$70FFCA);
  AddColor('DarkOliveGreen2',$68EEBC);
  AddColor('DarkOliveGreen3',$5ACDA2);
  AddColor('DarkOliveGreen4',$3D8B6E);
  AddColor('khaki1',$8FF6FF);
  AddColor('khaki2',$85E6EE);
  AddColor('khaki3',$73C6CD);
  AddColor('khaki4',$4E868B);
  AddColor('LightGoldenrod1',$8BECFF);
  AddColor('LightGoldenrod2',$82DCEE);
  AddColor('LightGoldenrod3',$70BECD);
  AddColor('LightGoldenrod4',$4C818B);
  AddColor('LightYellow1',$E0FFFF);
  AddColor('LightYellow2',$D1EEEE);
  AddColor('LightYellow3',$B4CDCD);
  AddColor('LightYellow4',$7A8B8B);
  AddColor('yellow1',$00FFFF);
  AddColor('yellow2',$00EEEE);
  AddColor('yellow3',$00CDCD);
  AddColor('yellow4',$008B8B);
  AddColor('gold1',$00D7FF);
  AddColor('gold2',$00C9EE);
  AddColor('gold3',$00ADCD);
  AddColor('gold4',$00758B);
  AddColor('goldenrod1',$25C1FF);
  AddColor('goldenrod2',$22B4EE);
  AddColor('goldenrod3',$1D9BCD);
  AddColor('goldenrod4',$14698B);
  AddColor('DarkGoldenrod1',$0FB9FF);
  AddColor('DarkGoldenrod2',$0EADEE);
  AddColor('DarkGoldenrod3',$0C95CD);
  AddColor('DarkGoldenrod4',$08658B);
  AddColor('RosyBrown1',$C1C1FF);
  AddColor('RosyBrown2',$B4B4EE);
  AddColor('RosyBrown3',$9B9BCD);
  AddColor('RosyBrown4',$69698B);
  AddColor('IndianRed1',$6A6AFF);
  AddColor('IndianRed2',$6363EE);
  AddColor('IndianRed3',$5555CD);
  AddColor('IndianRed4',$3A3A8B);
  AddColor('sienna1',$4782FF);
  AddColor('sienna2',$4279EE);
  AddColor('sienna3',$3968CD);
  AddColor('sienna4',$26478B);
  AddColor('burlywood1',$9BD3FF);
  AddColor('burlywood2',$91C5EE);
  AddColor('burlywood3',$7DAACD);
  AddColor('burlywood4',$55738B);
  AddColor('wheat1',$BAE7FF);
  AddColor('wheat2',$AED8EE);
  AddColor('wheat3',$96BACD);
  AddColor('wheat4',$667E8B);
  AddColor('tan1',$4FA5FF);
  AddColor('tan2',$499AEE);
  AddColor('tan3',$3F85CD);
  AddColor('tan4',$2B5A8B);
  AddColor('chocolate1',$247FFF);
  AddColor('chocolate2',$2176EE);
  AddColor('chocolate3',$1D66CD);
  AddColor('chocolate4',$13458B);
  AddColor('firebrick1',$3030FF);
  AddColor('firebrick2',$2C2CEE);
  AddColor('firebrick3',$2626CD);
  AddColor('firebrick4',$1A1A8B);
  AddColor('brown1',$4040FF);
  AddColor('brown2',$3B3BEE);
  AddColor('brown3',$3333CD);
  AddColor('brown4',$23238B);
  AddColor('salmon1',$698CFF);
  AddColor('salmon2',$6282EE);
  AddColor('salmon3',$5470CD);
  AddColor('salmon4',$394C8B);
  AddColor('LightSalmon1',$7AA0FF);
  AddColor('LightSalmon2',$7295EE);
  AddColor('LightSalmon3',$6281CD);
  AddColor('LightSalmon4',$42578B);
  AddColor('orange1',$00A5FF);
  AddColor('orange2',$009AEE);
  AddColor('orange3',$0085CD);
  AddColor('orange4',$005A8B);
  AddColor('DarkOrange1',$007FFF);
  AddColor('DarkOrange2',$0076EE);
  AddColor('DarkOrange3',$0066CD);
  AddColor('DarkOrange4',$00458B);
  AddColor('coral1',$5672FF);
  AddColor('coral2',$506AEE);
  AddColor('coral3',$455BCD);
  AddColor('coral4',$2F3E8B);
  AddColor('tomato1',$4763FF);
  AddColor('tomato2',$425CEE);
  AddColor('tomato3',$394FCD);
  AddColor('tomato4',$26368B);
  AddColor('OrangeRed1',$0045FF);
  AddColor('OrangeRed2',$0040EE);
  AddColor('OrangeRed3',$0037CD);
  AddColor('OrangeRed4',$00258B);
  AddColor('red1',$0000FF);
  AddColor('red2',$0000EE);
  AddColor('red3',$0000CD);
  AddColor('red4',$00008B);
  AddColor('DeepPink1',$9314FF);
  AddColor('DeepPink2',$8912EE);
  AddColor('DeepPink3',$7610CD);
  AddColor('DeepPink4',$500A8B);
  AddColor('HotPink1',$B46EFF);
  AddColor('HotPink2',$A76AEE);
  AddColor('HotPink3',$9060CD);
  AddColor('HotPink4',$623A8B);
  AddColor('pink1',$C5B5FF);
  AddColor('pink2',$B8A9EE);
  AddColor('pink3',$9E91CD);
  AddColor('pink4',$6C638B);
  AddColor('LightPink1',$B9AEFF);
  AddColor('LightPink2',$ADA2EE);
  AddColor('LightPink3',$958CCD);
  AddColor('LightPink4',$655F8B);
  AddColor('PaleVioletRed1',$AB82FF);
  AddColor('PaleVioletRed2',$9F79EE);
  AddColor('PaleVioletRed3',$8968CD);
  AddColor('PaleVioletRed4',$5D478B);
  AddColor('maroon1',$B334FF);
  AddColor('maroon2',$A730EE);
  AddColor('maroon3',$9029CD);
  AddColor('maroon4',$621C8B);
  AddColor('VioletRed1',$963EFF);
  AddColor('VioletRed2',$8C3AEE);
  AddColor('VioletRed3',$7832CD);
  AddColor('VioletRed4',$52228B);
  AddColor('magenta1',$FF00FF);
  AddColor('magenta2',$EE00EE);
  AddColor('magenta3',$CD00CD);
  AddColor('magenta4',$8B008B);
  AddColor('orchid1',$FA83FF);
  AddColor('orchid2',$E97AEE);
  AddColor('orchid3',$C969CD);
  AddColor('orchid4',$89478B);
  AddColor('plum1',$FFBBFF);
  AddColor('plum2',$EEAEEE);
  AddColor('plum3',$CD96CD);
  AddColor('plum4',$8B668B);
  AddColor('MediumOrchid1',$FF66E0);
  AddColor('MediumOrchid2',$EE5FD1);
  AddColor('MediumOrchid3',$CD52B4);
  AddColor('MediumOrchid4',$8B377A);
  AddColor('DarkOrchid1',$FF3EBF);
  AddColor('DarkOrchid2',$EE3AB2);
  AddColor('DarkOrchid3',$CD329A);
  AddColor('DarkOrchid4',$8B2268);
  AddColor('purple1',$FF309B);
  AddColor('purple2',$EE2C91);
  AddColor('purple3',$CD267D);
  AddColor('purple4',$8B1A55);
  AddColor('MediumPurple1',$FF82AB);
  AddColor('MediumPurple2',$EE799F);
  AddColor('MediumPurple3',$CD6889);
  AddColor('MediumPurple4',$8B475D);
  AddColor('thistle1',$FFE1FF);
  AddColor('thistle2',$EED2EE);
  AddColor('thistle3',$CDB5CD);
  AddColor('thistle4',$8B7B8B);
  AddColor('gray0',$000000);
  AddColor('grey0',$000000);
  AddColor('gray1',$030303);
  AddColor('grey1',$030303);
  AddColor('gray2',$050505);
  AddColor('grey2',$050505);
  AddColor('gray3',$080808);
  AddColor('grey3',$080808);
  AddColor('gray4',$0A0A0A);
  AddColor('grey4',$0A0A0A);
  AddColor('gray5',$0D0D0D);
  AddColor('grey5',$0D0D0D);
  AddColor('gray6',$0F0F0F);
  AddColor('grey6',$0F0F0F);
  AddColor('gray7',$121212);
  AddColor('grey7',$121212);
  AddColor('gray8',$141414);
  AddColor('grey8',$141414);
  AddColor('gray9',$171717);
  AddColor('grey9',$171717);
  AddColor('gray10',$1A1A1A);
  AddColor('grey10',$1A1A1A);
  AddColor('gray11',$1C1C1C);
  AddColor('grey11',$1C1C1C);
  AddColor('gray12',$1F1F1F);
  AddColor('grey12',$1F1F1F);
  AddColor('gray13',$212121);
  AddColor('grey13',$212121);
  AddColor('gray14',$242424);
  AddColor('grey14',$242424);
  AddColor('gray15',$262626);
  AddColor('grey15',$262626);
  AddColor('gray16',$292929);
  AddColor('grey16',$292929);
  AddColor('gray17',$2B2B2B);
  AddColor('grey17',$2B2B2B);
  AddColor('gray18',$2E2E2E);
  AddColor('grey18',$2E2E2E);
  AddColor('gray19',$303030);
  AddColor('grey19',$303030);
  AddColor('gray20',$333333);
  AddColor('grey20',$333333);
  AddColor('gray21',$363636);
  AddColor('grey21',$363636);
  AddColor('gray22',$383838);
  AddColor('grey22',$383838);
  AddColor('gray23',$3B3B3B);
  AddColor('grey23',$3B3B3B);
  AddColor('gray24',$3D3D3D);
  AddColor('grey24',$3D3D3D);
  AddColor('gray25',$404040);
  AddColor('grey25',$404040);
  AddColor('gray26',$424242);
  AddColor('grey26',$424242);
  AddColor('gray27',$454545);
  AddColor('grey27',$454545);
  AddColor('gray28',$474747);
  AddColor('grey28',$474747);
  AddColor('gray29',$4A4A4A);
  AddColor('grey29',$4A4A4A);
  AddColor('gray30',$4D4D4D);
  AddColor('grey30',$4D4D4D);
  AddColor('gray31',$4F4F4F);
  AddColor('grey31',$4F4F4F);
  AddColor('gray32',$525252);
  AddColor('grey32',$525252);
  AddColor('gray33',$545454);
  AddColor('grey33',$545454);
  AddColor('gray34',$575757);
  AddColor('grey34',$575757);
  AddColor('gray35',$595959);
  AddColor('grey35',$595959);
  AddColor('gray36',$5C5C5C);
  AddColor('grey36',$5C5C5C);
  AddColor('gray37',$5E5E5E);
  AddColor('grey37',$5E5E5E);
  AddColor('gray38',$616161);
  AddColor('grey38',$616161);
  AddColor('gray39',$636363);
  AddColor('grey39',$636363);
  AddColor('gray40',$666666);
  AddColor('grey40',$666666);
  AddColor('gray41',$696969);
  AddColor('grey41',$696969);
  AddColor('gray42',$6B6B6B);
  AddColor('grey42',$6B6B6B);
  AddColor('gray43',$6E6E6E);
  AddColor('grey43',$6E6E6E);
  AddColor('gray44',$707070);
  AddColor('grey44',$707070);
  AddColor('gray45',$737373);
  AddColor('grey45',$737373);
  AddColor('gray46',$757575);
  AddColor('grey46',$757575);
  AddColor('gray47',$787878);
  AddColor('grey47',$787878);
  AddColor('gray48',$7A7A7A);
  AddColor('grey48',$7A7A7A);
  AddColor('gray49',$7D7D7D);
  AddColor('grey49',$7D7D7D);
  AddColor('gray50',$7F7F7F);
  AddColor('grey50',$7F7F7F);
  AddColor('gray51',$828282);
  AddColor('grey51',$828282);
  AddColor('gray52',$858585);
  AddColor('grey52',$858585);
  AddColor('gray53',$878787);
  AddColor('grey53',$878787);
  AddColor('gray54',$8A8A8A);
  AddColor('grey54',$8A8A8A);
  AddColor('gray55',$8C8C8C);
  AddColor('grey55',$8C8C8C);
  AddColor('gray56',$8F8F8F);
  AddColor('grey56',$8F8F8F);
  AddColor('gray57',$919191);
  AddColor('grey57',$919191);
  AddColor('gray58',$949494);
  AddColor('grey58',$949494);
  AddColor('gray59',$969696);
  AddColor('grey59',$969696);
  AddColor('gray60',$999999);
  AddColor('grey60',$999999);
  AddColor('gray61',$9C9C9C);
  AddColor('grey61',$9C9C9C);
  AddColor('gray62',$9E9E9E);
  AddColor('grey62',$9E9E9E);
  AddColor('gray63',$A1A1A1);
  AddColor('grey63',$A1A1A1);
  AddColor('gray64',$A3A3A3);
  AddColor('grey64',$A3A3A3);
  AddColor('gray65',$A6A6A6);
  AddColor('grey65',$A6A6A6);
  AddColor('gray66',$A8A8A8);
  AddColor('grey66',$A8A8A8);
  AddColor('gray67',$ABABAB);
  AddColor('grey67',$ABABAB);
  AddColor('gray68',$ADADAD);
  AddColor('grey68',$ADADAD);
  AddColor('gray69',$B0B0B0);
  AddColor('grey69',$B0B0B0);
  AddColor('gray70',$B3B3B3);
  AddColor('grey70',$B3B3B3);
  AddColor('gray71',$B5B5B5);
  AddColor('grey71',$B5B5B5);
  AddColor('gray72',$B8B8B8);
  AddColor('grey72',$B8B8B8);
  AddColor('gray73',$BABABA);
  AddColor('grey73',$BABABA);
  AddColor('gray74',$BDBDBD);
  AddColor('grey74',$BDBDBD);
  AddColor('gray75',$BFBFBF);
  AddColor('grey75',$BFBFBF);
  AddColor('gray76',$C2C2C2);
  AddColor('grey76',$C2C2C2);
  AddColor('gray77',$C4C4C4);
  AddColor('grey77',$C4C4C4);
  AddColor('gray78',$C7C7C7);
  AddColor('grey78',$C7C7C7);
  AddColor('gray79',$C9C9C9);
  AddColor('grey79',$C9C9C9);
  AddColor('gray80',$CCCCCC);
  AddColor('grey80',$CCCCCC);
  AddColor('gray81',$CFCFCF);
  AddColor('grey81',$CFCFCF);
  AddColor('gray82',$D1D1D1);
  AddColor('grey82',$D1D1D1);
  AddColor('gray83',$D4D4D4);
  AddColor('grey83',$D4D4D4);
  AddColor('gray84',$D6D6D6);
  AddColor('grey84',$D6D6D6);
  AddColor('gray85',$D9D9D9);
  AddColor('grey85',$D9D9D9);
  AddColor('gray86',$DBDBDB);
  AddColor('grey86',$DBDBDB);
  AddColor('gray87',$DEDEDE);
  AddColor('grey87',$DEDEDE);
  AddColor('gray88',$E0E0E0);
  AddColor('grey88',$E0E0E0);
  AddColor('gray89',$E3E3E3);
  AddColor('grey89',$E3E3E3);
  AddColor('gray90',$E5E5E5);
  AddColor('grey90',$E5E5E5);
  AddColor('gray91',$E8E8E8);
  AddColor('grey91',$E8E8E8);
  AddColor('gray92',$EBEBEB);
  AddColor('grey92',$EBEBEB);
  AddColor('gray93',$EDEDED);
  AddColor('grey93',$EDEDED);
  AddColor('gray94',$F0F0F0);
  AddColor('grey94',$F0F0F0);
  AddColor('gray95',$F2F2F2);
  AddColor('grey95',$F2F2F2);
  AddColor('gray96',$F5F5F5);
  AddColor('grey96',$F5F5F5);
  AddColor('gray97',$F7F7F7);
  AddColor('grey97',$F7F7F7);
  AddColor('gray98',$FAFAFA);
  AddColor('grey98',$FAFAFA);
  AddColor('gray99',$FCFCFC);
  AddColor('grey99',$FCFCFC);
  AddColor('gray100',$FFFFFF);
  AddColor('grey100',$FFFFFF);
  AddColor('darkgrey',$A9A9A9);
  AddColor('darkgray',$A9A9A9);
  AddColor('darkblue',$8B0000);
  AddColor('darkcyan',$8B8B00);
  AddColor('darkmagenta',$8B008B);
  AddColor('darkred',$00008B);
  AddColor('lightgreen',$90EE90);
  /////////////////////////////////////////////////
  ColorGroup('WEB','ColWeb.crc','HTML W3C Stantard');
  /////////////////////////////////////////////////
  AddColor('IndianRed',$5C5CCD);
  AddColor('LightCoral',$8080F0);
  AddColor('Salmon',$7280FA);
  AddColor('DarkSalmon',$7A96E9);
  AddColor('LightSalmon',$7AA0FF);
  AddColor('Crimson',$3C14DC);
  AddColor('Red',$0000FF);
  AddColor('FireBrick',$2222B2);
  AddColor('DarkRed',$00008B);
  AddColor('Pink',$CBC0FF);
  AddColor('LightPink',$C1B6FF);
  AddColor('HotPink',$B469FF);
  AddColor('DeepPink',$9314FF);
  AddColor('MediumVioletRed',$8515C7);
  AddColor('PaleVioletRed',$9370DB);
  AddColor('LightSalmon',$7AA0FF);
  AddColor('Coral',$507FFF);
  AddColor('Tomato',$4763FF);
  AddColor('OrangeRed',$0045FF);
  AddColor('DarkOrange',$008CFF);
  AddColor('Orange',$00A5FF);
  AddColor('Gold',$00D7FF);
  AddColor('Yellow',$00FFFF);
  AddColor('LightYellow',$E0FFFF);
  AddColor('LemonChiffon',$CDFAFF);
  AddColor('LightGoldenrodYellow',$D2FAFA);
  AddColor('PapayaWhip',$D5EFFF);
  AddColor('Moccasin',$B5E4FF);
  AddColor('PeachPuff',$B9DAFF);
  AddColor('PaleGoldenrod',$AAE8EE);
  AddColor('Khaki',$8CE6F0);
  AddColor('DarkKhaki',$6BB7BD);
  AddColor('Lavender',$FAE6E6);
  AddColor('Thistle',$D8BFD8);
  AddColor('Plum',$DDA0DD);
  AddColor('Violet',$EE82EE);
  AddColor('Orchid',$D670DA);
  AddColor('Fuchsia',$FF00FF);
  AddColor('Magenta',$FF00FF);
  AddColor('MediumOrchid',$D355BA);
  AddColor('MediumPurple',$DB7093);
  AddColor('Amethyst',$CC6699);
  AddColor('BlueViolet',$E22B8A);
  AddColor('DarkViolet',$D30094);
  AddColor('DarkOrchid',$CC3299);
  AddColor('DarkMagenta',$8B008B);
  AddColor('Purple',$800080);
  AddColor('Indigo',$82004B);
  AddColor('SlateBlue',$CD5A6A);
  AddColor('DarkSlateBlue',$8B3D48);
  AddColor('MediumSlateBlue',$EE687B);
  AddColor('RebeccaPurple',$993366);
  AddColor('GreenYellow',$2FFFAD);
  AddColor('Chartreuse',$00FF7F);
  AddColor('LawnGreen',$00FC7C);
  AddColor('Lime',$00FF00);
  AddColor('LimeGreen',$32CD32);
  AddColor('PaleGreen',$98FB98);
  AddColor('LightGreen',$90EE90);
  AddColor('MediumSpringGreen',$9AFA00);
  AddColor('SpringGreen',$7FFF00);
  AddColor('MediumSeaGreen',$71B33C);
  AddColor('SeaGreen',$578B2E);
  AddColor('ForestGreen',$228B22);
  AddColor('Green',$008000);
  AddColor('DarkGreen',$006400);
  AddColor('YellowGreen',$32CD9A);
  AddColor('OliveDrab',$238E6B);
  AddColor('Olive',$008080);
  AddColor('DarkOliveGreen',$2F6B55);
  AddColor('MediumAquamarine',$AACD66);
  AddColor('DarkSeaGreen',$8FBC8F);
  AddColor('LightSeaGreen',$AAB220);
  AddColor('DarkCyan',$8B8B00);
  AddColor('Teal',$808000);
  AddColor('Aqua',$FFFF00);
  AddColor('Cyan',$FFFF00);
  AddColor('LightCyan',$FFFFE0);
  AddColor('PaleTurquoise',$EEEEAF);
  AddColor('Aquamarine',$D4FF7F);
  AddColor('Turquoise',$D0E040);
  AddColor('MediumTurquoise',$CCD148);
  AddColor('DarkTurquoise',$D1CE00);
  AddColor('CadetBlue',$A09E5F);
  AddColor('SteelBlue',$B48246);
  AddColor('LightSteelBlue',$DEC4B0);
  AddColor('PowderBlue',$E6E0B0);
  AddColor('LightBlue',$E6D8AD);
  AddColor('SkyBlue',$EBCE87);
  AddColor('LightSkyBlue',$FACE87);
  AddColor('DeepSkyBlue',$FFBF00);
  AddColor('DodgerBlue',$FF901E);
  AddColor('CornflowerBlue',$ED9564);
  AddColor('MediumSlateBlue',$EE687B);
  AddColor('RoyalBlue',$E16941);
  AddColor('Blue',$FF0000);
  AddColor('MediumBlue',$CD0000);
  AddColor('DarkBlue',$8B0000);
  AddColor('Navy',$800000);
  AddColor('MidnightBlue',$701919);
  AddColor('Cornsilk',$DCF8FF);
  AddColor('BlanchedAlmond',$CDEBFF);
  AddColor('Bisque',$C4E4FF);
  AddColor('NavajoWhite',$ADDEFF);
  AddColor('Wheat',$B3DEF5);
  AddColor('BurlyWood',$87B8DE);
  AddColor('Tan',$8CB4D2);
  AddColor('RosyBrown',$8F8FBC);
  AddColor('SandyBrown',$60A4F4);
  AddColor('Goldenrod',$20A5DA);
  AddColor('DarkGoldenrod',$0B86B8);
  AddColor('Peru',$3F85CD);
  AddColor('Chocolate',$1E69D2);
  AddColor('SaddleBrown',$13458B);
  AddColor('Sienna',$2D52A0);
  AddColor('Brown',$2A2AA5);
  AddColor('Maroon',$000080);
  AddColor('White',$FFFFFF);
  AddColor('Snow',$FAFAFF);
  AddColor('Honeydew',$F0FFF0);
  AddColor('MintCream',$FAFFF5);
  AddColor('Azure',$FFFFF0);
  AddColor('AliceBlue',$FFF8F0);
  AddColor('GhostWhite',$FFF8F8);
  AddColor('WhiteSmoke',$F5F5F5);
  AddColor('Seashell',$EEF5FF);
  AddColor('Beige',$DCF5F5);
  AddColor('OldLace',$E6F5FD);
  AddColor('FloralWhite',$F0FAFF);
  AddColor('Ivory',$F0FFFF);
  AddColor('AntiqueWhite',$D7EBFA);
  AddColor('Linen',$E6F0FA);
  AddColor('LavenderBlush',$F5F0FF);
  AddColor('MistyRose',$E1E4FF);
  AddColor('Gainsboro',$DCDCDC);
  AddColor('LightGray',$D3D3D3);
  AddColor('LightGrey',$D3D3D3);
  AddColor('Silver',$C0C0C0);
  AddColor('DarkGray',$A9A9A9);
  AddColor('DarkGrey',$A9A9A9);
  AddColor('Gray',$808080);
  AddColor('Grey',$808080);
  AddColor('DimGray',$696969);
  AddColor('DimGrey',$696969);
  AddColor('LightSlateGray',$998877);
  AddColor('LightSlateGrey',$998877);
  AddColor('SlateGray',$908070);
  AddColor('SlateGrey',$908070);
  AddColor('DarkSlateGray',$4F4F2F);
  AddColor('DarkSlateGrey',$4F4F2F);
  /////////////////////////////////////////////////
  ColorGroup('SYN','ColWin.crc','Windows Synonyms');
  /////////////////////////////////////////////////
  AddColor('LtGray',$C0C0C0);
  AddColor('DkGray',$808080);
  AddColor('LightGray',$C0C0C0);
  AddColor('DarkGray',$808080);
  /////////////////////////////////////////////////
  ColorGroup('VGA','ColWin.crc','Windows VGA Colors');
  /////////////////////////////////////////////////
  AddColor('Black',$000000);
  AddColor('Maroon',$000080);
  AddColor('Green',$008000);
  AddColor('Olive',$008080);
  AddColor('Navy',$800000);
  AddColor('Purple',$800080);
  AddColor('Teal',$808000);
  AddColor('Gray',$808080);
  AddColor('Silver',$C0C0C0);
  AddColor('Red',$0000FF);
  AddColor('Lime',$00FF00);
  AddColor('Yellow',$00FFFF);
  AddColor('Blue',$FF0000);
  AddColor('Fuchsia',$FF00FF);
  AddColor('Aqua',$FFFF00);
  AddColor('White',$FFFFFF);
 end;
 if not Assigned(KnownColorsListByName) then begin
  KnownColorsListByName:=NewHashList(false,HashList_DefaultHasher);
  KnownColorsListByName.Master:=@KnownColorsListByName;
  for i:=KnownColorsList.Count-1 downto 0 do begin
   n:=KnownColorsList[i]; c:=TColor(KnownColorsList.Objects[i]); h:='cl'+n;
   if KnownColorsListByName.IndexOf(n)<0 then KnownColorsListByName.KeyedLinks[n]:=c;
   if KnownColorsListByName.IndexOf(h)<0 then KnownColorsListByName.KeyedLinks[h]:=c;
   if (StrFetch(n,1)='_') then begin // Add BtnFace for _BtnFace
    n:=Copy(n,2,Length(n)-1); c:=TColor(KnownColorsList.Objects[i]); h:='cl'+n;
    if KnownColorsListByName.IndexOf(n)<0 then KnownColorsListByName.KeyedLinks[n]:=c;
    if KnownColorsListByName.IndexOf(h)<0 then KnownColorsListByName.KeyedLinks[h]:=c;
   end;
  end;
 end;
 if not Assigned(KnownColorsListByCode) then begin
  KnownColorsListByCode:=NewHashList(false,HashList_DefaultHasher);
  KnownColorsListByCode.Master:=@KnownColorsListByCode;
  for i:=KnownColorsList.Count-1 downto 0 do begin
   n:=KnownColorsList[i]; c:=TColor(KnownColorsList.Objects[i]); h:=ColorToHexNum(c);
   if KnownColorsListByCode.IndexOf(h)<0 then KnownColorsListByCode.KeyedParams[h]:=n;
  end;
 end;
end;

procedure FreeKnownColorsList;
begin
 FreeAndNil(KnownColorsList);
 FreeAndNil(KnownColorsListGroup);
 FreeAndNil(KnownColorsListByName);
 FreeAndNil(KnownColorsListByCode);
end;

function ColorToNumber(ColorCode:TColor):LongString;
begin
 Result:=IntToStr(ColorCode);
end;

function ColorToHexNum(ColorCode:TColor; prec:Integer=6):LongString;
begin
 if (prec<1) then prec:=1;
 if (prec>8) then prec:=8;
 Result:=Format('$%.*x',[prec,ColorCode]);
end;

function ColorToWebNum(ColorCode:TColor):LongString;
begin
 with TRGBRec(ColorCode) do Result:=Format('#%2.2x%2.2x%2.2x',[R,G,B]);
end;

function ColorToRgbNum(ColorCode:TColor):LongString;
begin
 with TRGBRec(ColorCode) do Result:=Format('%u,%u,%u',[R,G,B]);
end;

function ColorToString(ColorCode:TColor; Prefix:LongString=''):LongString;
begin
 Result:=ColorToHexNum(ColorCode);
 if Assigned(KnownColorsListByCode) then
 if KnownColorsListByCode.IndexOf(Result)>=0 then
 Result:=Prefix+KnownColorsListByCode.KeyedParams[Result];
end;

function StringToColor(ColorName:LongString; DefaultColor:TColor=clNone):TColor;
begin
 Result:=DefaultColor;
 ColorName:=Trim(ColorName);
 if (ColorName<>'') then begin
  if Assigned(KnownColorsListByName) then
  if KnownColorsListByName.IndexOf(ColorName)>=0 then begin
   Result:=KnownColorsListByName.KeyedLinks[ColorName];
   Exit;
  end;
  Result:=StrToIntDef(ColorName,DefaultColor);
 end;
end;

function KnownColorsCount:Integer;
begin
 Result:=0;
 if (KnownColorsList=nil) then Exit;
 Result:=KnownColorsList.Count;
end;

function KnownColorsName(Index:Integer):LongString;
begin
 Result:='';
 if (KnownColorsList=nil) then Exit;
 if (Index>=0) and (Index<KnownColorsList.Count)
 then Result:=KnownColorsList.Strings[Index];
end;

function KnownColorsCode(Index:Integer):TColor;
begin
 Result:=clNone;
 if (KnownColorsList=nil) then Exit;
 if (Index>=0) and (Index<KnownColorsList.Count)
 then Result:=TColor(KnownColorsList.Objects[Index]);
end;

function KnownColorsGroup(Index:Integer):LongString;
begin
 Result:='';
 if (KnownColorsListGroup=nil) then Exit;
 if (Index>=0) and (Index<KnownColorsListGroup.Count)
 then Result:=SysUtils.Trim(Dump(PointerToPtrInt(KnownColorsListGroup.Items[Index])));
end;

function Cookie(id,def:LongString):LongString;
begin
 Result:=CookieScan(KnownColorsListCookies,id,Ord(';'));
 if (Result='') then Result:=def;
end;

function KnownColorsListAsText(Mode:Integer):LongString;
type RGB = TRGBRec;
var List:TStringList; i,ml:Integer; line,name,group:LongString; code:TColor;
begin
 Result:='';
 if (KnownColorsCount=0) then Exit;
 try
  List:=TStringList.Create;
  try
   if (Mode in [3]) then begin
    List.Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0//EN">');
    List.Add('<html lang="'+Cookie('lang','ru')+'">');
    List.Add(' <head>');
    List.Add('  <link rel="shortcut icon" href="'+Cookie('favicon','../daqsite/painter/colors.ico')+'" type="image/x-icon"/>');
    List.Add('  <META http-equiv="Content-Type" content="text/html; charset='+Cookie('charset','windows-1251')+'">');
    List.Add('  <TITLE>CRW-DAQ Color Table</TITLE>');
    List.Add('  <style>');
    List.Add('  body  { text-align:justify; color:black; background-color:#E0E0E0; margin:20px 20px; padding:0px 0px; font-size:12pt; font-weight:normal; font-family:PT Mono,monospace; }');
    List.Add('  table { text-align:justify; vertical-align:top; font-size:12pt; font-weight:normal; font-family:PT Mono,monospace; }');
    List.Add('  th    { text-align:left; vertical-align:center; margin:0px; padding:8px; font-size:14pt; font-weight:bold; font-family:PT Mono,monospace; }');
    List.Add('  td    { text-align:justify; vertical-align:center; margin:0px; padding:8px; font-size:12pt; font-weight:normal; font-family:PT Mono, monospace; }');
    List.Add('  pre   { text-align:left; vertical-align:center; color:black; background-color: #FFF0F0; margin:10px; padding:10px; border:1px solid red; font-size:10pt; font-weight:normal; font-family:PT Mono,monospace; }');
    List.Add('  .hidden{display:none;} .small {font-size:60%;} .smaller {font-size:75%;} .big {font-size:140%;} .bigger {font-size:125%;}  .huge {font-size:200%;}  .huge3 {font-size:300%;}  .huge4 {font-size:400%;}');
    List.Add('  .bold  {font-weight:bold;} .red {color:red;} .blue {color:blue;}  .green {color:green;}');
    List.Add('  </style>');
    List.Add(' </head>');
    List.Add('');
    List.Add(' <body>');
    List.Add('');
    List.Add('  <hr><h1><img src="'+Cookie('crw-daq-ico','../icon/crwdaq.ico')+'"> CRW-DAQ Color Table</h1><hr>');
    List.Add('');
    List.Add('  <p>');
    List.Add('   This file contains <b>Color Table</b> actually available in <b>CRW-DAQ</b>. See <a href="#notes" class="bold">Notes</a>.');
    List.Add('  </p>');
    List.Add('');
    List.Add('  <table border="1">');
    List.Add('   <tr>');
    List.Add('    <td class="bigger bold">');
    List.Add('     #N');
    List.Add('    </td>');
    List.Add('    <td class="bigger bold">');
    List.Add('     Color Name');
    List.Add('    </td>');
    List.Add('    <td class="bigger bold">');
    List.Add('     Color View');
    List.Add('    </td>');
    List.Add('    <td class="bigger bold">');
    List.Add('     #Html');
    List.Add('    </td>');
    List.Add('    <td class="bigger bold">');
    List.Add('     Color Code');
    List.Add('    </td>');
    List.Add('    <td class="bigger bold">');
    List.Add('     Notes');
    List.Add('    </td>');
    List.Add('   </tr>');
    for i:=0 to KnownColorsCount-1 do begin
     name:=KnownColorsName(i);
     code:=KnownColorsCode(i);
     group:=KnownColorsGroup(i);
     if (group<>'') then group:=Format('<a href="#%s">%s</a>',[AnsiLowerCase(group),group]);
     List.Add('   <tr>');
     List.Add('    <td class="bold">');
     List.Add('     '+IntToStr(i+1));
     List.Add('    </td>');
     List.Add('    <td class="bold">');
     List.Add('     '+name);
     List.Add('    </td>');
     line:=Format('<div style="width:200px; height:50px; border:1px solid; margin:auto; background: #%2.2x%2.2x%2.2x;"></div>',
                  [RGB(code).R,RGB(code).G,RGB(code).B]);
     List.Add('    <td class="bold">');
     List.Add('     '+line);
     List.Add('    </td>');
     List.Add('    <td class="bold">');
     List.Add('     '+Format('#%2.2x%2.2x%2.2x',[RGB(code).R,RGB(code).G,RGB(code).B]));
     List.Add('    </td>');
     List.Add('    <td class="bold">');
     List.Add('     '+Format('<pre><b>$%8.8X</b> = RGB( %3d, %3d, %3d )</pre>',[code,RGB(code).R,RGB(code).G,RGB(code).B]));
     List.Add('    </td>');
     List.Add('    <td class="bold">');
     List.Add('     '+group);
     List.Add('    </td>');
     List.Add('   </tr>');
    end;
    List.Add('  </table>');
    List.Add('');
    List.Add('  <b id="notes" class="bigger bold">Notes:</b><br>');
    List.Add('  </ol>');
    List.Add('   <li>Named colors available in <b>DaqPascal</b>, <b>CRC</b> files and <b>Painter</b>.');
    List.Add('   <br>In <b>DaqPascal</b> colors available by call <b>ParamStr(''ColorCode Blue'')</b>.');
    List.Add('   <br>In <b>CRC</b> files colors available by name like <b>Blue</b>.');
    List.Add('   <br>In <b>Painter</b> colors available like <b>clBlue</b>.');
    List.Add('   </li>');
    List.Add('   <li id="jcop"> <a href="'+Cookie('id-jcop','fwcolors.pdf')+'" class="bold" target="_blank">JCOP</a> - named colors defined in <a href="http://cern.ch/jcop-framework" target="_blank">Joint Controls Project</a> at CERN.</li>');
    List.Add('   <li id="sys"> <a href="'+Cookie('id-sys','../daqsite/painter/colors.htm')+'" class="bold" target="_blank">SYS</a> - system (functional) colors defined in WINAPI or Delphi.</li>');
    List.Add('   <li id="x11"> <a href="'+Cookie('id-x11','../daqsite/painter/colors.htm')+'" class="bold" target="_blank">X11</a> - named colors defined in <a href="https://www.x.org/" target="_blank">X Window System</a>.</li>');
    List.Add('   <li id="web"> <a href="'+Cookie('id-web','../daqsite/painter/colors.htm')+'" class="bold" target="_blank">WEB</a> - named colors defined in <a href="https://www.w3.org/" target="_blank">HTML</a>.</li>');
    List.Add('   <li id="vga"> <a href="'+Cookie('id-vga','../daqsite/painter/colors.htm')+'" class="bold" target="_blank">VGA</a> - 16 main named colors defined in WINAPI or Delphi.</li>');
    List.Add('   <li id="syn"> <a href="'+Cookie('id-syn','../daqsite/painter/colors.htm')+'" class="bold" target="_blank">SYN</a> - synonym colors defined in WINAPI or Delphi.</li>');
    List.Add('  </ol>');
    List.Add('');
    if (Mode=3) then begin
     List.Add('  <hr><h1>CRW-DAQ Color Table (as pas code):</h1><hr>');
     List.Add('');
     List.Add('  <pre class="normal">');
     List.Add(SysUtils.TrimRight(KnownColorsListAsText(1)));
     List.Add('  </pre>');
     List.Add('');
     List.Add('  <hr><h1>CRW-DAQ Color Table (as crc code):</h1><hr>');
     List.Add('');
     List.Add('  <pre class="normal">');
     List.Add(SysUtils.TrimRight(KnownColorsListAsText(2)));
     List.Add('  </pre>');
    end;
    List.Add('');
    List.Add(' </body>');
    List.Add('</html>');
   end else
   if (Mode in [0,1,2]) then begin
    ml:=0;
    for i:=0 to KnownColorsCount-1 do begin
     name:=KnownColorsName(i);
     ml:=max(ml,Length(name));
    end;
    for i:=0 to KnownColorsCount-1 do begin
     name:=KnownColorsName(i); code:=KnownColorsCode(i); line:=name;
     if (mode=1) then line:=Format('%s%-*s = $%8.8X; // RGB( %3d, %3d, %3d )',[' ', ml,name,code,rgb(code).R,rgb(code).G,rgb(code).B]);
     if (mode=2) then line:=Format('%s%-*s = $%8.8X; // RGB( %3d, %3d, %3d )',['cl',ml,name,code,rgb(code).R,rgb(code).G,rgb(code).B]);
     List.Add(line);
    end;
   end;
   Result:=List.Text;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,nil,'KnownColorsListAsText');
 end;
end;

 ////////////////////
 // Functional colors
 ////////////////////
var
 SysColor:array[0..30] of TColor;

procedure UpdateSysColors;
var i:Integer;
begin
 for i:=Low(SysColor) to High(SysColor) do
 if (GetSysColorBrush(i)<>0)
 then SysColor[i]:=GetSysColor(i)
 else SysColor[i]:=clNone;
end;

function cl_ScrollBar               : TColor; begin Result:=SysColor[ 0]; end;
function cl_Background              : TColor; begin Result:=SysColor[ 1]; end;
function cl_ActiveCaption           : TColor; begin Result:=SysColor[ 2]; end;
function cl_InactiveCaption         : TColor; begin Result:=SysColor[ 3]; end;
function cl_Menu                    : TColor; begin Result:=SysColor[ 4]; end;
function cl_Window                  : TColor; begin Result:=SysColor[ 5]; end;
function cl_WindowFrame             : TColor; begin Result:=SysColor[ 6]; end;
function cl_MenuText                : TColor; begin Result:=SysColor[ 7]; end;
function cl_WindowText              : TColor; begin Result:=SysColor[ 8]; end;
function cl_CaptionText             : TColor; begin Result:=SysColor[ 9]; end;
function cl_ActiveBorder            : TColor; begin Result:=SysColor[10]; end;
function cl_InactiveBorder          : TColor; begin Result:=SysColor[11]; end;
function cl_AppWorkSpace            : TColor; begin Result:=SysColor[12]; end;
function cl_Highlight               : TColor; begin Result:=SysColor[13]; end;
function cl_HighlightText           : TColor; begin Result:=SysColor[14]; end;
function cl_BtnFace                 : TColor; begin Result:=SysColor[15]; end;
function cl_BtnShadow               : TColor; begin Result:=SysColor[16]; end;
function cl_GrayText                : TColor; begin Result:=SysColor[17]; end;
function cl_BtnText                 : TColor; begin Result:=SysColor[18]; end;
function cl_InactiveCaptionText     : TColor; begin Result:=SysColor[19]; end;
function cl_BtnHighlight            : TColor; begin Result:=SysColor[20]; end;
function cl_3DDkShadow              : TColor; begin Result:=SysColor[21]; end;
function cl_3DLight                 : TColor; begin Result:=SysColor[22]; end;
function cl_InfoText                : TColor; begin Result:=SysColor[23]; end;
function cl_InfoBk                  : TColor; begin Result:=SysColor[24]; end;
function cl_HotLight                : TColor; begin Result:=SysColor[26]; end;
function cl_GradientActiveCaption   : TColor; begin Result:=SysColor[27]; end;
function cl_GradientInactiveCaption : TColor; begin Result:=SysColor[28]; end;
function cl_MenuHilight             : TColor; begin Result:=SysColor[29]; end;
function cl_MenuBar                 : TColor; begin Result:=SysColor[30]; end;
// Synonyms
function cl_Desktop                 : TColor; begin Result:=SysColor[ 1]; end; // BACKGROUND
function cl_3DFace                  : TColor; begin Result:=SysColor[15]; end; // BTNFACE
function cl_3DText                  : TColor; begin Result:=SysColor[18]; end; // BTNTEXT
function cl_3DShadow                : TColor; begin Result:=SysColor[16]; end; // BTNSHADOW
function cl_3DHighlight             : TColor; begin Result:=SysColor[20]; end; // BTNHIGHLIGHT
function cl_3DHilight               : TColor; begin Result:=SysColor[20]; end; // BTNHIGHLIGHT
function cl_BtnHilight              : TColor; begin Result:=SysColor[20]; end; // BTNHIGHLIGHT
function cl_Button                  : TColor; begin Result:=SysColor[15]; end; // BTNFACE
function cl_ButtonText              : TColor; begin Result:=SysColor[18]; end; // BTNTEXT
function cl_ButtonShadow            : TColor; begin Result:=SysColor[16]; end; // BTNSHADOW
function cl_ToolTip                 : TColor; begin Result:=SysColor[24]; end; // INFOBK
function cl_ToolTipText             : TColor; begin Result:=SysColor[23]; end; // INFOTEXT

function ByteDiff(b1,b2:Byte):Byte; inline;
begin
 if (b1<b2) then Result:=b2-b1 else Result:=b1-b2;
end;

function ColorDiff(c1,c2:TColor):Integer;
type RGB=TRGBRec; var dR,dG,dB:Integer;
begin
 Result:=High(Byte);
 c1:=ColorToRGB(c1);
 c2:=ColorToRGB(c2);
 dR:=ByteDiff(RGB(c1).R,RGB(c2).R);
 dG:=ByteDiff(RGB(c1).G,RGB(c2).G);
 dB:=ByteDiff(RGB(c1).B,RGB(c2).B);
 Result:=dR*dR+dG*dG+dB*dB;
end;

function FindNearestKnownColor(aColor:TColor):TColor;
var i:Integer; ci:TColor; dm,di:Integer;
begin
 Result:=clNone; dm:=MaxInt;
 for i:=0 to KnownColorsCount-1 do begin
  ci:=KnownColorsCode(i);
  di:=ColorDiff(aColor,ci);
  if (dm<=di) then continue;
  Result:=ci; dm:=di;
  if (dm=0) then Break;
 end;
end;

function FindNearestKnownColor(aC:LongString):TColor;
var aColor:TColor;
begin
 aColor:=StringToColor(aC,clNone);
 if (aColor=clNone) then Exit(aColor);
 Result:=FindNearestKnownColor(aColor);
end;

function ParamStrColorInfo(arg:LongString):LongString;
var cn,cs:TColor; w1:LongString;
begin
 Result:='';
 w1:=ExtractWord(1,arg,ScanSpaces);
 if IsEmptyStr(w1) then Exit('');
 cn:=FindNearestKnownColor(w1);
 cs:=StringToColor(w1);
 if (cs=clNone)
 then Result:=Result+' _None'
 else Result:=Result+' '+ColorToString(cs);
 Result:=Result+' '+ColorToString(cn);
 Result:=Result+' '+ColorToHexNum(cs);
 Result:=Result+' '+ColorToHexNum(cn);
 Result:=Result+' '+ColorToWebNum(cs);
 Result:=Result+' '+ColorToWebNum(cn);
 Result:=Result+' '+ColorToRgbNum(cs);
 Result:=Result+' '+ColorToRgbNum(cn);
 Result:=Result+' '+IntToStr(cs);
 Result:=Result+' '+IntToStr(cn);
 Result:=Trim(Result);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_colors;
begin
 InitKnownColorsList;
end;

procedure Free_crw_colors;
begin
 FreeKnownColorsList;
end;

initialization

 Init_crw_colors;

finalization

 Free_crw_colors;

end.

//////////////
// END OF FILE
//////////////

