////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr RFA BFP Window.                                                //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231208 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrrfabfpwindow; // Form Spectr RFA BFP Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_CurveWindow, Form_SpectrWindow,
 Form_SpectrDaqWindow, Form_DelphiProjectEditor,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_daqdev,
 _crw_ef, _crw_sort, _crw_ee, _crw_zm,
 _crw_curves, _crw_riff, _crw_rfadata,
 _crw_calib, _crw_lsqpoly, _crw_daqsys,
 {_crw_dcc32,} _crw_pascalprojects,
 _crw_spline, _crw_crwapi, _crw_crwapiserver,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  ERfaBfpError = class(ESoftException);
  TFormSpectrRfaBfpWindow = class(TFormSpectrDaqWindow)
    ActionSpectrAnalysisRfaBfpJobWizard: TAction;
    ActionSpectrAnalysisRfaBfpJobRun: TAction;
    MenuSpectrAnalysis: TMenuItem;
    MenuSpectrAnalysisRfaBfpJobWizard: TMenuItem;
    MenuSpectrAnalysisRfaBfpJobRun: TMenuItem;
    ToolButtonSpectrAnalysisRfaBfpJobWizardSeparator: TToolButton;
    ToolButtonSpectrAnalysisRfaBfpJobWizard: TToolButton;
    ToolButtonSpectrAnalysisRfaBfpJobRun: TToolButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActionSpectrAnalysisRfaBfpJobWizardExecute(Sender: TObject);
    procedure ActionSpectrAnalysisRfaBfpJobRunExecute(Sender: TObject);
  private
    { Private declarations }
    myJobWizard      : TMasterForm;
    mySelElement     : Integer;
    mySelElements    : TByteSet;
    myConfigFile     : LongString;
    myWindowSection  : LongString;
    myDllFilePath    : LongString;
    function  GetSelElement:Integer;
    procedure SetSelElement(AtomID:Integer);
    function  GetSelElements:TByteSet;
    procedure SetSelElements(const AtomIDs:TByteSet);
    function  GetSelXRayLines:TRfaFieldsSet;
    function  EnToChan(En:Double):Double;
    function  GetExcitationFile:LongString;
    function  GetTargetFileList:LongString;
    function  GetDetectorParams:LongString;
    function  GetElementsList:LongString;
    function  GetXRayLinesList:LongString;
    function  GetRoiList:LongString;
    procedure SetStatusMessage(const S:LongString);
    function  GetConfigFile:LongString;
    function  GetWindowSection:LongString;
    function  GetDllFilePath:LongString;
  public
    { Public declarations }
    procedure Config(CfgFile,Section:LongString); override;
    procedure DrawExp; override;
    procedure DrawFul; override;
    procedure DrawExpRoiList;
    procedure DrawFulRoiList;
    procedure DrawExpSelElement;
    procedure DrawFulSelElement;
    property  SelElement     : Integer       read GetSelElement     write SetSelElement;
    property  SelElements    : TByteSet      read GetSelElements    write SetSelElements;
    property  SelXRayLines   : TRfaFieldsSet read GetSelXRayLines;
    property  ExcitationFile : LongString    read GetExcitationFile;
    property  TargetFileList : LongString    read GetTargetFileList;
    property  DetectorParams : LongString    read GetDetectorParams;
    property  ElementsList   : LongString    read GetElementsList;
    property  XRayLinesList  : LongString    read GetXRayLinesList;
    property  RoiList        : LongString    read GetRoiList;
    property  StatusMessage  : LongString                           write SetStatusMessage;
    property  ConfigFile     : LongString    read GetConfigFile;
    property  WindowSection  : LongString    read GetWindowSection;
    property  DllFilePath    : LongString    read GetDllFilePath;
    function  CreateJob      : TFormCurveWindow;
    function  ExecuteJob(Job:TFormCurveWindow):Boolean;
    function  RunJob         : Boolean;
    procedure OpenJobWizard;
    procedure EditDllSource;
  end;

const
 regSpecRfaBfpWin = 'SpecRfaBfpWin'; // RFA Based on Fundamental Parameters
 GroundLineWidth  = 2;

function  NewSpecRfaBfpWin(const aCaption    : LongString;
                                 aSpectr     : TCurve;
                                 aOwnsSpectr : Boolean):TFormSpectrWindow;
procedure Kill(var TheObject:TFormSpectrRfaBfpWindow); overload;

type
 TRoiItemRec = packed record
  Left  : Integer;
  Right : Integer;
  Power : Integer;
  Fixed : Integer;
  Coeff : packed array[0..9] of Double;
 end;
 
function RoiItemRec(L,R,P,F:Integer; const C:array of Double):TRoiItemRec;
function RoiItemRecToStr(R:TRoiItemRec):LongString;
function StrToRoiItemRec(const S:LongString):TRoiItemRec;
function RoiItemRecGround(const Z:TRoiItemRec; c:Double):Double;

implementation

{$R *.lfm}

uses
 Form_SpectrAnalysisRfaBfpJobWizard;

 {
 *******************************************************************************
 General purpose utilites
 *******************************************************************************
 }
function  NewSpecRfaBfpWin(const aCaption    : LongString;
                                 aSpectr     : TCurve;
                                 aOwnsSpectr : Boolean ) : TFormSpectrWindow;
begin
 Application.CreateForm(TFormSpectrRfaBfpWindow, Result);
 if Result.Ok then
 with Result do
 try
  LockDraw;
  if aCaption=''
  then Caption:=RusEng('Спектрометрическое окно','Spectrometry window')
  else Caption:=aCaption;
  AssignSpectr(aSpectr,aOwnsSpectr);
 finally
  UnlockDraw;
 end;
end;

procedure Kill(var TheObject:TFormSpectrRfaBfpWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 //
 // Процедуры для работы с записями ROI, хранящимися в списке виде строки
 //
procedure RoiItemRecClearTail(var R:TRoiItemRec);
var
 i : Integer;
begin
 for i:=Max(Low(R.Coeff),R.Power+1) to High(R.Coeff) do R.Coeff[i]:=0;
end;

function RoiItemRec(L,R,P,F:Integer; const C:array of Double):TRoiItemRec;
var
 i : Integer;
begin
 with Result do begin
  Left:=L; Right:=R; Power:=P; Fixed:=F;
  for i:=Low(Coeff) to High(Coeff) do
  if (i>=Low(C)) and (i<=High(C)) then Coeff[i]:=C[i] else Coeff[i]:=0;
 end;
end;

function RoiItemRecToStr(R:TRoiItemRec):LongString;
begin
 RoiItemRecClearTail(R);
 with R do Result:=Format('%5d %5d %1d %4d'+
                          ' %17.14g %17.14g %17.14g %17.14g %17.14g'+
                          ' %17.14g %17.14g %17.14g %17.14g %17.14g',
                          [Left,Right,Power,Fixed,
                          Coeff[0],Coeff[1],Coeff[2],Coeff[3],Coeff[4],
                          Coeff[5],Coeff[6],Coeff[7],Coeff[8],Coeff[9]]);
end;

function StrToRoiItemRec(const S:LongString):TRoiItemRec;
var Buff:TParsingBuffer;
begin
 Result:=Default(TRoiItemRec);
 if (ScanVarRecord(svAsIs,StrCopyBuff(Buff,S),'%i;%i;%i;%i;%f;%f;%f;%f;%f;%f;%f;%f;%f;%f',Result)=nil)
 or (Result.Power<0) or (Result.Power>9) or (Result.Left<0) or (Result.Right<Result.Left)
 then Raise EConvertError.CreateFmt(RusEng('Ошибка преобразования РОИ "%s".','Format error ROI "%s".'),[S]);
 RoiItemRecClearTail(Result);
end;

function RoiItemRecGround(const Z:TRoiItemRec; c:Double):Double;
begin
 Result:=EvalPoly(Z.Coeff[0],Z.Power+1,(c-Z.Left)/(Z.Right-Z.Left));
end;

 {
 *******************************************************************************
 TFormSpectrRfaBfpWindow implementation
 *******************************************************************************
 }
type
 TEnToChanRec = packed record
  Energy : Double;
  Spectr : TFormSpectrWindow;
 end;

function EnToChanGoal(x:Double; Custom:Pointer):Double;
begin
 with TEnToChanRec(Custom^) do Result:=Spectr.EnCalibr(x)-Energy;
end;

function TFormSpectrRfaBfpWindow.EnToChan(En:Double):Double;
var
 EnToChanRec : TEnToChanRec;
begin
 Result:=_Nan;
 if Ok then begin
  EnToChanRec.Energy:=En;
  EnToChanRec.Spectr:=Self;
  if En>=EnCalibr(0) then
  if En<=EnCalibr(SpectrSize-1) then
  if EnToChanGoal(0,@EnToChanRec)*EnToChanGoal(SpectrSize-1,@EnToChanRec)<=0
  then Result:=FindZero(EnToChanGoal,0,SpectrSize-1,1E-6,@EnToChanRec);
 end;
end;

function GaussPeak(Chan,Amplitude,Center,Fwhm:Double):Double;
const
 alfa = 2.77258872223978123766892848583271; {=4*ln(2)}
var
 alfatt : Extended;
begin
 alfatt:=alfa*sqr((Chan-Center)/Fwhm);
 if alfatt>200 then Result:=0 else Result:=Amplitude*exp(-alfatt);
end;

procedure TFormSpectrRfaBfpWindow.Config(CfgFile,Section:LongString);
begin
 CfgFile:=UnifyFileAlias(CfgFile);
 inherited;
 myConfigFile:=CfgFile;
 myWindowSection:=Section;
 if ReadIniFilePath(CfgFile,Section,'DLL_FILE_PATH',ExtractFilePath(CfgFile),myDllFilePath)
 then myDllFilePath:=DefaultExtension(myDllFilePath,'.dll')
 else myDllFilePath:='';
end;

procedure TFormSpectrRfaBfpWindow.DrawExp;
begin
 inherited;
 DrawExpRoiList;
 DrawExpSelElement;
end;

procedure TFormSpectrRfaBfpWindow.DrawFul;
begin
 inherited;
 DrawFulRoiList;
 DrawFulSelElement;
end;

procedure TFormSpectrRfaBfpWindow.DrawExpRoiList;
var
 c : Double;
 g : Double;
 i : Integer;
 j : Integer;
 n : Integer;
 L : Integer;
 R : Integer;
 Z : TRoiItemRec;
 p1,p2 : _crw_plut.TPoint2I;
begin
 if Ok then
 if myJobWizard.Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard do
 if ListBoxRoiList.Items.Count>0 then
 try
  p2:=Point2I(0,0);
  for i:=0 to ListBoxRoiList.Items.Count-1 do begin
   Z:=StrToRoiItemRec(ListBoxRoiList.Items[i]);
   L:=Max(Z.Left,BegX);
   R:=Min(Z.Right,EndX);
   if R>L then begin
    n:=ExpandView.ConvertX(R-BegX)-ExpandView.ConvertX(L-BegX);
    for j:=0 to n do begin
     c:=L+(R-L)*j/n;
     g:=RoiItemRecGround(Z,c);
     p1.x:=ExpandView.ConvertX(c-BegX);
     p1.y:=ExpandView.ConvertY(g,isLog);
     if j>0 then DrawLine(PaintBoxExpand.Canvas,p1,p2,clRed,psSolid,pmCopy,GroundLineWidth);
     if (j=0) or (j=n) then begin
      p2.x:=p1.x;
      p2.y:=ExpandView.ConvertY(0,isLog);
      DrawLine(PaintBoxExpand.Canvas,p1,p2,clRed,psSolid,pmCopy,GroundLineWidth);
     end;
     p2:=p1;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormSpectrRfaBfpWindow.DrawFulRoiList;
var
 c : Double;
 g : Double;
 i : Integer;
 j : Integer;
 n : Integer;
 L : Integer;
 R : Integer;
 Z : TRoiItemRec;
 p1,p2 : _crw_plut.TPoint2I;
begin
 if Ok then
 if myJobWizard.Ok then
 if myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard then
 with myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard do
 if ListBoxRoiList.Items.Count>0 then
 try
  p2:=Point2I(0,0);
  for i:=0 to ListBoxRoiList.Items.Count-1 do begin
   Z:=StrToRoiItemRec(ListBoxRoiList.Items[i]);
   L:=Max(Z.Left,0);
   R:=Min(Z.Right,SpectrSize-1);
   if R>L then begin
    n:=FullView.ConvertX(R)-FullView.ConvertX(L);
    for j:=0 to n do begin
     c:=L+(R-L)*j/n;
     g:=RoiItemRecGround(Z,c);
     p1.x:=FullView.ConvertX(c);
     p1.y:=FullView.ConvertY(g,true);
     if j>0 then DrawLine(PaintBoxFull.Canvas,p1,p2,clRed,psSolid,pmCopy,GroundLineWidth);
     if (j=0) or (j=n) then begin
      p2.x:=p1.x;
      p2.y:=FullView.ConvertY(0,true);
      DrawLine(PaintBoxFull.Canvas,p1,p2,clRed,psSolid,pmCopy,GroundLineWidth);
     end;
     p2:=p1;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormSpectrRfaBfpWindow.DrawExpSelElement;
var
 p1,p2    : _crw_plut.TPoint2I;
 EPos     : Double;
 Ampl     : Double;
 Chan     : Double;
 Fwhm     : Double;
 Norm     : Double;
 Spec     : TCurve;
 iChan    : Integer;
 iLast    : Integer;
 AtomId   : Integer;
 LineId   : TRfaFields;
 SelLines : TRfaFieldsSet;
begin
 if Ok then
 if SelElement>0 then
 if EnCalibrExist then
 if HwCalibrExist then
 if myJobWizard.Ok then
 if myJobWizard.Visible then
 if myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard then
 with myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard do
 if PageControl.ActivePage = TabSheetElements then
 try
  AtomId:=SelElement;
  if Rfa.Accessible[AtomId] then begin
   Spec:=NewCurve(SpectrSize);
   try
    for iChan:=0 to Spec.Count-1 do Spec[iChan]:=Point2D(EnCalibr(iChan),0);
    SelLines:=SelXRayLines;
    for LineId:=rf_FIRST_FIELD to rf_LAST_FIELD do
    if LineId in SelLines then begin
     EPos:=Rfa.XRayLine[AtomId,LineId].Energy;
     Ampl:=Rfa.XRayLine[AtomId,LineId].Height;
     if (EPos>0) and (Ampl>0) then begin
      Chan:=EnToChan(EPos);
      if not isNaN(Chan) then begin
       Fwhm:=HwCalibr(Chan);
       for iChan:=0 to Spec.Count-1 do
       Spec[iChan]:=Point2D(Spec[iChan].X,Spec[iChan].Y+GaussPeak(iChan,Ampl,Chan,Fwhm));
      end;
     end;
    end;
    iLast:=MaxInt;
    Norm:=RectSizeY(Spec.Limits);
    if Norm>0 then
    for iChan:=BegX to EndX do begin
     Ampl:=Spec[iChan].Y/Norm*0.9;
     if isLog then Ampl:=Ampl*High(Vert) else Ampl:=Ampl*Vert;
     p1.x:=ExpandView.ConvertX(iChan-BegX);
     p1.y:=ExpandView.ConvertY(0,IsLog);
     p2.x:=p1.x;
     p2.y:=ExpandView.ConvertY(Ampl,IsLog);
     if not PointIsEqual(p1,p2) and (p1.x<>iLast) then begin
      iLast:=p1.x;
      DrawLine(PaintBoxExpand.Canvas,p1,p2,clWhite,psSolid,pmXor,1);
     end;
    end;
   finally
    Kill(Spec);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormSpectrRfaBfpWindow.DrawFulSelElement;
var
 p1,p2    : _crw_plut.TPoint2I;
 EPos     : Double;
 Ampl     : Double;
 Chan     : Double;
 Fwhm     : Double;
 Norm     : Double;
 Spec     : TCurve;
 iChan    : Integer;
 iLast    : Integer;
 AtomId   : Integer;
 LineId   : TRfaFields;
 SelLines : TRfaFieldsSet;
begin
 if Ok then
 if SelElement>0 then
 if EnCalibrExist then
 if HwCalibrExist then
 if myJobWizard.Ok then
 if myJobWizard.Visible then
 if myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard then
 with myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard do
 if PageControl.ActivePage = TabSheetElements then
 try
  AtomId:=SelElement;
  if Rfa.Accessible[AtomId] then begin
   Spec:=NewCurve(SpectrSize);
   try
    for iChan:=0 to Spec.Count-1 do Spec[iChan]:=Point2D(EnCalibr(iChan),0);
    SelLines:=SelXRayLines;
    for LineId:=rf_FIRST_FIELD to rf_LAST_FIELD do
    if LineId in SelLines then begin
     EPos:=Rfa.XRayLine[AtomId,LineId].Energy;
     Ampl:=Rfa.XRayLine[AtomId,LineId].Height;
     if (EPos>0) and (Ampl>0) then begin
      Chan:=EnToChan(EPos);
      if not isNaN(Chan) then begin
       Fwhm:=HwCalibr(Chan);
       for iChan:=0 to Spec.Count-1 do
       Spec[iChan]:=Point2D(Spec[iChan].X,Spec[iChan].Y+GaussPeak(iChan,Ampl,Chan,Fwhm));
      end;
     end;
    end;
    iLast:=MaxInt;
    Norm:=RectSizeY(Spec.Limits);
    if Norm>0 then
    for iChan:=0 to SpectrSize-1 do begin
     Ampl:=Spec[iChan].Y/Norm*1e-3*High(Vert);
     p1.x:=FullView.ConvertX(iChan);
     p1.y:=FullView.ConvertY(0,true);
     p2.x:=p1.x;
     p2.y:=FullView.ConvertY(Ampl,true);
     if not PointIsEqual(p1,p2) and (p1.x<>iLast) then begin
      iLast:=p1.x;
      DrawLine(PaintBoxFull.Canvas,p1,p2,clWhite,psSolid,pmXor,1);
     end;
    end;
   finally
    Kill(Spec);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function  TFormSpectrRfaBfpWindow.GetSelElement:Integer;
begin
 if Ok then Result:=mySelElement else Result:=0;
end;

procedure TFormSpectrRfaBfpWindow.SetSelElement(AtomID:Integer);
begin
 if Ok then
 if Rfa.Feasible[AtomId] then
 if AtomID<>SelElement then begin
  mySelElement:=AtomID;
  DrawView;
 end;
end;

function  TFormSpectrRfaBfpWindow.GetSelElements:TByteSet;
begin
 if Ok then Result:=mySelElements else Result:=[];
end;

procedure TFormSpectrRfaBfpWindow.SetSelElements(const AtomIDs:TByteSet);
begin
 if Ok then mySelElements:=AtomIDs*[RfaMinAtomId..RfaMaxAtomId];
end;

function  TFormSpectrRfaBfpWindow.GetSelXRayLines:TRfaFieldsSet;
var
 i      : Integer;
 LineId : TRfaFields;
begin
 Result:=[];
 if Ok then
 if myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard then
 with myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard do
 with CheckListBoxXRayLines do
 for i:=0 to Items.Count-1 do begin
  LineId:=Rfa.FindLine(Items[i]);
  if (LineId in rf_All_Series) and Checked[i] then include(Result,LineId);
 end;
 Result:=Result*rf_All_Series;
end;

function  TFormSpectrRfaBfpWindow.GetExcitationFile:LongString;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 Result:=EditExcitation.Text;
end;

function  TFormSpectrRfaBfpWindow.GetTargetFileList:LongString;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 Result:=ListBoxTargetList.Items.Text;
end;

function  TFormSpectrRfaBfpWindow.GetDetectorParams:LongString;
var P:TText; f:Double; r:Integer; n,v:LongString;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 try
  P:=NewText;
  try
   for r:=1 to StringGridDetector.RowCount-1 do begin
    n:=Trim(StringGridDetector.Cells[0,r]);
    v:=Trim(StringGridDetector.Cells[1,r]);
    if (Length(n)>0) then begin
     if (WordIndex('DETECTOR.',UnifyAlias(n),ScanSpaces)=1) then begin
      if (WordIndex('ELEMENT',UnifyAlias(n),ScanSpaces)>0) then begin
       if not SameText(UnifyAlias(v),'AIR') then
       if not SameText(UnifyAlias(v),'NONE') then
       if (Rfa.FindAtom(v)<RfaMinAtomId) then
       Raise ERfaBfpError.CreateFmt(RusEng('Недопустимое поле "%s = %s".',
                                           'Invalid field "%s = %s".'),[n,v]);
      end else begin
       if not Str2Real(v,f) then
       Raise ERfaBfpError.CreateFmt(RusEng('Недопустимое поле "%s = %s".',
                                           'Invalid field "%s = %s".'),[n,v]);
      end;
     end;
     P.Addln(Format('%s = %s',[n,v]));
    end;
   end;
   Result:=P.Text;
  finally
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetDetectorParams');
 end;
end;

function  TFormSpectrRfaBfpWindow.GetRoiList:LongString;
var p:TText; i,j:Integer;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 try
  p:=NewText;
  try
   p.Addln(Format('ROI.Count = %d',[ListBoxRoiList.Items.Count]));
   for i:=0 to ListBoxRoiList.Items.Count-1 do
   with StrToRoiItemRec(ListBoxRoiList.Items[i]) do begin
    p.Addln(Format('ROI[%d].Left = %d',[i,Left]));
    p.Addln(Format('ROI[%d].Right = %d',[i,Right]));
    p.Addln(Format('ROI[%d].Power = %d',[i,Power]));
    p.Addln(Format('ROI[%d].Fixed = %d',[i,Fixed]));
    for j:=0 to Power do p.Addln(Format('ROI[%d].Coeff[%d] = %g',[i,j,Coeff[j]]));
   end;
   Result:=p.Text;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetRoiList');
 end;
end;

function  TFormSpectrRfaBfpWindow.GetElementsList:LongString;
var p:TText; AtomId:Integer;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 try
  p:=NewText;
  try
   p.Addln(Format('SelectedElement = %s',[Rfa.AtomSymbol[SelElement]]));
   for AtomId:=RfaMinAtomId to RfaMaxAtomId do
   if Rfa.Accessible[AtomId] and (Length(Rfa.AtomSymbol[AtomId])>0)
   then p.Addln(Format('SelectedElements[%s] = %d',[Rfa.AtomSymbol[AtomId],ord(AtomId in SelElements)]));
   Result:=p.Text;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetElementsList');
 end;
end;

function  TFormSpectrRfaBfpWindow.GetXRayLinesList:LongString;
var p:TText; LineId:TRfaFields; SelLines:TRfaFieldsSet;
begin
 Result:='';
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 try
  p:=NewText;
  try
   SelLines:=SelXRayLines;
   for LineId:=rf_FIRST_FIELD to rf_LAST_FIELD do
   if (LineId in rf_All_Series) and (Length(Rfa.LineName[LineId])>0)
   then p.Addln(Format('SelectedXRayLines[%s] = %d',[Rfa.LineName[LineId],ord(LineId in SelLines)]));
   Result:=p.Text;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetXRayLinesList');
 end;
end;

procedure TFormSpectrRfaBfpWindow.SetStatusMessage(const S:LongString);
begin
 if Ok then
 if (myJobWizard is TFormSpectrAnalysisRfaBfpJobWizard) then
 with (myJobWizard as TFormSpectrAnalysisRfaBfpJobWizard) do
 StatusBar.SimpleText:=S;
end;

function  TFormSpectrRfaBfpWindow.GetConfigFile:LongString;
begin
 if Assigned(Self) then Result:=myConfigFile else Result:='';
end;

function  TFormSpectrRfaBfpWindow.GetWindowSection:LongString;
begin
 if Assigned(Self) then Result:=myWindowSection else Result:='';
end;

function  TFormSpectrRfaBfpWindow.GetDllFilePath:LongString;
begin
 if Assigned(Self) then Result:=myDllFilePath else Result:='';
end;

function  TFormSpectrRfaBfpWindow.CreateJob:TFormCurveWindow;
var i:Integer; Target:TText; TmpSpd:LongString;
 function CloneSpectr(const aName,aSourceFile:LongString):TCurve;
 var i:Integer; Cal:TPolynomCalibration;
 begin
  Result:=NewCurve(SpectrSize,aName,clBlack,$1F);
  for i:=0 to SpectrSize-1 do Result[i]:=Point2D(EnCalibr(i),SpectrValue[i]);
  Result.AddLn(Format('SourceFile = %s',[aSourceFile]));
  Cal:=Spectrometr.Calibration[EnCalNum];
  if Cal.Ok  then begin
   Result.AddLn(Format('EnergyCalibration.Power = %d',[Cal.Power]));
   Result.AddLn(Format('EnergyCalibration.Center = %g',[Cal.Center]));
   Result.AddLn(Format('EnergyCalibration.Scale = %g',[Cal.Scale]));
   for i:=0 to Cal.Power do
   Result.AddLn(Format('EnergyCalibration.Coeff[%d] = %g',[i,Cal.Coeff[i]]));
  end;
  Cal:=Spectrometr.Calibration[HwCalNum];
  if Cal.Ok  then begin
   Result.AddLn(Format('HalfWidthCalibration.Power = %d',[Cal.Power]));
   Result.AddLn(Format('HalfWidthCalibration.Center = %g',[Cal.Center]));
   Result.AddLn(Format('HalfWidthCalibration.Scale = %g',[Cal.Scale]));
   for i:=0 to Cal.Power do
   Result.AddLn(Format('HalfWidthCalibration.Coeff[%d] = %g',[i,Cal.Coeff[i]]));
  end;
 end;
begin
 Result:=nil;
 if Ok then
 try
  if not myJobWizard.Ok then OpenJobWizard;
  Target:=NewText;
  Target.Text:=TargetFileList;
  TmpSpd:=ForcePath(TempDir,CreateTempFile('___.spd'));
  SaveSpd(TmpSpd);
  try
   // Проверить список элементов, наличие файлов для обработки
   if not FileExists(ExcitationFile)
   then Raise ERfaBfpError.CreateFmt(RusEng('Не найден спектр возбуждения "%s".',
                                            'Can''t find excitation spectrum "%s".'),[ExcitationFile]);
   if Target.Count=0
   then Raise ERfaBfpError.Create(RusEng('Нет спектров для обработки.','No spectrums to process.'));
   for i:=0 to Target.Count-1 do
   if not FileExists(Target[i])
   then Raise ERfaBfpError.CreateFmt(RusEng('Не найден спектр "%s".','Can''t find spectrum "%s".'),[Target[i]]);
   if SelElements=[]
   then Raise ERfaBfpError.Create(RusEng('Не выбраны химические элементы.','Chemical elements is not selected.'));
   if Length(RoiList)=0
   then Raise ERfaBfpError.Create(RusEng('Не задан список РОИ.','ROI list is not selected.'));
   // Создать окно и записать в комментарий общие данные
   Result:=NewCurveWindow('X-Ray Fluorescence Analysis Job',^C^M^L'  Rate,[Counts]',^R'Energy,[keV]  '^M^C);
   Result.Width:=400;
   Result.Height:=300;
   Result.Comment.Text:=DetectorParams;
   if Result.Comment.Count=0
   then Raise ERfaBfpError.Create(RusEng('Ошибка задания параметров детектора.','Detector set parameters error.'));
   Result.Comment.Text:=Result.Comment.Text+ElementsList;
   Result.Comment.Text:=Result.Comment.Text+XRayLinesList;
   Result.Comment.Text:=Result.Comment.Text+RoiList;
   // Загрузить из файлов и переписать в кривые спектры
   if not LoadSpd(ExcitationFile)
   then Raise ERfaBfpError.CreateFmt(RusEng('Ошибка ввода "%s".','Can'' load "%s".'),[ExcitationFile]);
   Result.AddCurve(CloneSpectr('ExcitationSpectrum',ExcitationFile));
   for i:=0 to Target.Count-1 do begin
    if not LoadSpd(Target[i])
    then Raise ERfaBfpError.CreateFmt(RusEng('Ошибка ввода "%s".','Can'' load "%s".'),[Target[i]]);
    Result.AddCurve(CloneSpectr(Format('TargetSpectrum[%d]',[i]),Target[i]));
   end;
   StatusMessage:=RusEng('Задание создано.','Job created.');
  finally
   Kill(Target);
   LoadSpd(TmpSpd);
   FileErase(TmpSpd);
   FreeConfigCache(0);
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'CreateJob');
   StatusMessage:=E.Message;
   SystemConsole.Activate;
   Kill(Result);
  end;
 end;
end;

function  TFormSpectrRfaBfpWindow.ExecuteJob(Job:TFormCurveWindow):Boolean;
var ErrCod:Int64; CrwApi:TCrwApiServer;
var aName,aSource,aBinary:LongString;
const Timeout=30000;
begin
 Result:=false;
 if Ok then
 if Job.Ok then
 try
  aBinary:=DllFilePath;
  aName:=LowerCase(ExtractDllBaseName(aBinary));
  aSource:=AddPathDelim(ExtractFileDir(aBinary))+aName+FavoritePascalProjectType;
  if not FileExists(DllFilePath)
  then SafeCompilePascalProject(aSource,Timeout);
  if not FileExists(DllFilePath)
  then Raise EReadError.CreateFmt(RusEng('Не найден файл "%s".',
                                         'File not found "%s".'),[DllFilePath]);
  CrwApi:=NewCrwApiServer(ForDataAnalysis,aName,aSource,aBinary);
  try
   ErrCod:=CrwApi.ExecuteDataAnalysis(DllFilePath,Job,Job.Comment.Text);
   if (ErrCod<>0)
   then Raise ERfaBfpError.CreateFmt(RusEng('Ошибка %d при выполнении задания.',
                                            'Execution error, return code %d.'),[ErrCod]);
  finally
   CrwApi.Free;
  end;
  StatusMessage:=RusEng('Задание выполнено.','Job complete.');
  Result:=true;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExecuteJob');
   StatusMessage:=E.Message;
   SystemConsole.Activate;
  end;
 end;
end;

function  TFormSpectrRfaBfpWindow.RunJob:Boolean;
begin
 Result:=ExecuteJob(CreateJob);
end;

procedure TFormSpectrRfaBfpWindow.OpenJobWizard;
begin
 if Ok then begin
  if not myJobWizard.Ok then begin
   myJobWizard:=NewFormSpectrAnalysisRfaBfpJobWizard(Self);
   myJobWizard.Master:=@myJobWizard;
   myJobWizard.Top:=Screen.DesktopTop+Screen.DesktopHeight-myJobWizard.Height;
  end;
  if myJobWizard.Ok then begin
   myJobWizard.Show;
   myJobWizard.WindowState:=wsNormal;
   myJobWizard.BringToFront;
  end;
 end;
end;

procedure TFormSpectrRfaBfpWindow.EditDllSource;
var Dpr:LongString;
begin
 if Ok then
 try
  Dpr:=ForceExtension(DllFilePath,'.dpr');
  if not FileExists(Dpr)
  then Raise ERfaBfpError.CreateFmt(RusEng('Не найден файл "%s".','Not found file "%s".'),[Dpr]);
  FindDelphiProjectEditor(Dpr,true,true);
  StatusMessage:='Ok.';
 except
  on E:Exception do begin
   BugReport(E,Self,'EditDllSource');
   StatusMessage:=E.Message;
   SystemConsole.Activate;
  end;
 end;
end;

procedure TFormSpectrRfaBfpWindow.FormCreate(Sender: TObject);
begin
 inherited;
 myJobWizard:=nil;
 mySelElement:=0;
 mySelElements:=[];
 myConfigFile:='';
 myWindowSection:='';
 myDllFilePath:='';
 UpdateMenu(MenuSpectrAnalysis,
            RusEng('Анализ','Analysis'),
            RusEng('Меню функций анализа спектров.','Menu for spectral analysis.'),
            0);
 UpdateMenu(MenuSpectrAnalysisRfaBfpJobWizard,
            RusEng('Задание на анализ','Analysis job'),
            RusEng('Редактирование параметров задания на анализ.','Edit job for spectral analysis.'),
            0);
 UpdateMenu(MenuSpectrAnalysisRfaBfpJobRun,
            RusEng('Выполнить анализ','Run analysis'),
            RusEng('Запуск задания на анализ.','Run job for spectral analysis.'),
            0);
end;

procedure TFormSpectrRfaBfpWindow.FormDestroy(Sender: TObject);
begin
 myConfigFile:='';
 myWindowSection:='';
 myDllFilePath:='';
 Kill(myJobWizard);
 inherited;
end;

procedure TFormSpectrRfaBfpWindow.ActionSpectrAnalysisRfaBfpJobWizardExecute(Sender: TObject);
begin
 inherited;
 OpenJobWizard;
end;

procedure TFormSpectrRfaBfpWindow.ActionSpectrAnalysisRfaBfpJobRunExecute(Sender: TObject);
begin
 inherited;
 RunJob;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrrfabfpwindow;
begin
 RegisterSpectrWindowConstructor(NewSpecRfaBfpWin, regSpecRfaBfpWin);
end;

procedure Free_form_spectrrfabfpwindow;
begin
end;

initialization

 Init_form_spectrrfabfpwindow;

finalization

 Free_form_spectrrfabfpwindow;

end.

//////////////
// END OF FILE
//////////////

