////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Manual Fit Control.                                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231205 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrmanfitcontrol; // Form Spectr Manual Fit Control

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Spin,
 lcltype, lclintf, Grids,
 Form_SpectrManFitWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_sort, _crw_ef, _crw_ee, _crw_zm,
 _crw_curves, _crw_riff, _crw_calib,
 _crw_daqsys, _crw_daqdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSpectrManFitControl = class(TMasterForm)
    PageControl: TPageControl;
    TabSheetPeakList: TTabSheet;
    TabSheetControl: TTabSheet;
    TabSheetPreset: TTabSheet;
    TabSheetDetails: TTabSheet;
    TabSheetErrorMatrix: TTabSheet;
    TabSheetGround: TTabSheet;
    ButtonPeakSearch: TButton;
    ButtonInterval: TButton;
    ButtonFit: TButton;
    ButtonClear: TButton;
    LabelAlgorithm: TLabel;
    ComboBoxAlgorithm: TComboBox;
    LabelMaxCalls: TLabel;
    SpinEditMaxCalls: TSpinEdit;
    LabelTolX: TLabel;
    EditTolX: TEdit;
    LabelTolF: TLabel;
    EditTolF: TEdit;
    LabelTolG: TLabel;
    EditTolG: TEdit;
    ButtonOk: TButton;
    ButtonCancel: TButton;
    GroupBoxPeakList: TGroupBox;
    ButtonPeakListOk: TButton;
    ButtonPeakListCancel: TButton;
    StringGridPeaks: TStringGrid;
    GroupBoxGround: TGroupBox;
    ButtonGroundOk: TButton;
    ButtonGroundCancel: TButton;
    StringGridGround: TStringGrid;
    ButtonPeakListInsert: TButton;
    ButtonPeakListDelete: TButton;
    MemoDetails: TMemo;
    StringGridCovar: TStringGrid;
    PanelCovarKind: TPanel;
    ComboBoxCovKind: TComboBox;
    LabelCovKind: TLabel;
    SaveDialog: TSaveDialog;
    ButtonSavePik: TButton;
    ButtonInsertPeak: TButton;
    ButtonDeletePeak: TButton;
    procedure FormCreate(Sender: TObject);
    procedure ButtonPeakSearchClick(Sender: TObject);
    procedure ButtonIntervalClick(Sender: TObject);
    procedure ButtonFitClick(Sender: TObject);
    procedure ButtonClearClick(Sender: TObject);
    procedure ButtonOkClick(Sender: TObject);
    procedure ButtonCancelClick(Sender: TObject);
    procedure ButtonPeakListOkClick(Sender: TObject);
    procedure ButtonPeakListCancelClick(Sender: TObject);
    procedure ButtonGroundOkClick(Sender: TObject);
    procedure ButtonGroundCancelClick(Sender: TObject);
    procedure StringGridPeaksClick(Sender: TObject);
    procedure ButtonPeakListInsertClick(Sender: TObject);
    procedure ButtonPeakListDeleteClick(Sender: TObject);
    procedure ComboBoxCovKindChange(Sender: TObject);
    procedure ButtonSavePikClick(Sender: TObject);
    procedure ButtonInsertPeakClick(Sender: TObject);
    procedure ButtonDeletePeakClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    Win : TFormSpectrManFitWindow;
    procedure UpdateControls(Flags:Integer);
  end;

function NewFormSpectrManFitControl(Win:TFormSpectrManFitWindow):TFormSpectrManFitControl;

implementation

{$R *.lfm}

function NewFormSpectrManFitControl(Win:TFormSpectrManFitWindow):TFormSpectrManFitControl;
begin
 Result:=nil;
 if (Win is TFormSpectrManFitWindow) then begin
  Application.CreateForm(TFormSpectrManFitControl, Result);
  Result.Win:=Win;
  Result.Caption:=Format(RusEng('Фитирование пиков - [%s]','Peak fitting - [%s]'),[Win.Caption]);
 end;
end;

procedure TFormSpectrManFitControl.UpdateControls(Flags:Integer);
var i,j,n:Integer; R:TGridRect; P:TText; ci,cj,cij,vi,vj,vij:Double;
const w1=30; w2=140;
 function  FormatValue(v:Double):LongString;
 begin
  Result:=Format('%g',[v]);
 end;
 function  FormatFixed(v:Boolean):LongString;
 begin
  if v then Result:='+' else Result:='-';
 end;
 procedure UpdateValue(const s:LongString; var v:Double);
 var
  f : Double;
 begin
  if (s<>Format('%g',[v])) then
  if Str2Real(s,f) then v:=f else begin
   Daq.OpenConsole;
   Daq.ConsoleEcho(Format('Invalid value "%s"!',[s]));
  end;
 end;
 procedure UpdateFixed(const s:LongString; var v:Boolean);
 begin
  if (s='-') then v:=false else
  if (s='+') then v:=true  else begin
   Daq.OpenConsole;
   Daq.ConsoleEcho(Format('Invalid value "%s"!',[s]));
  end;
 end;
begin
 if Ok then
 if Win.Ok then
 try
  if HasFlags(Flags,1) then begin
   { Update peak list }
   StringGridPeaks.ColCount:=10;
   StringGridPeaks.RowCount:=1+Win.ManFit.NumPeaks;
   if (StringGridPeaks.ColCount>1) then StringGridPeaks.FixedCols:=1;
   if (StringGridPeaks.RowCount>1) then StringGridPeaks.FixedRows:=1;
   StringGridPeaks.Cells[0,0]:='#N';        StringGridPeaks.ColWidths[0]:=w1;
   StringGridPeaks.Cells[1,0]:='Channel';   StringGridPeaks.ColWidths[1]:=w2;
   StringGridPeaks.Cells[2,0]:='Fix';       StringGridPeaks.ColWidths[2]:=w1;
   StringGridPeaks.Cells[3,0]:='Amplitude'; StringGridPeaks.ColWidths[3]:=w2;
   StringGridPeaks.Cells[4,0]:='Fix';       StringGridPeaks.ColWidths[4]:=w1;
   StringGridPeaks.Cells[5,0]:='FWHM';      StringGridPeaks.ColWidths[5]:=w2;
   StringGridPeaks.Cells[6,0]:='Fix';       StringGridPeaks.ColWidths[6]:=w1;
   StringGridPeaks.Cells[7,0]:='Area';      StringGridPeaks.ColWidths[7]:=w2;
   StringGridPeaks.Cells[8,0]:='Ground';    StringGridPeaks.ColWidths[8]:=w2;
   StringGridPeaks.Cells[9,0]:='Energy';    StringGridPeaks.ColWidths[9]:=w2;
   for i:=0 to Win.ManFit.NumPeaks-1 do begin
    StringGridPeaks.Cells[0,1+i]:=Format('%d',[i]);
    StringGridPeaks.Cells[1,1+i]:=FormatValue(Win.ManFit.CurrParam.Peak[i].Chan);
    StringGridPeaks.Cells[2,1+i]:=FormatFixed(Win.ManFit.CurrFixed.Peak[i].Chan);
    StringGridPeaks.Cells[3,1+i]:=FormatValue(Win.ManFit.CurrParam.Peak[i].Ampl);
    StringGridPeaks.Cells[4,1+i]:=FormatFixed(Win.ManFit.CurrFixed.Peak[i].Ampl);
    StringGridPeaks.Cells[5,1+i]:=FormatValue(Win.ManFit.CurrParam.Peak[i].FWHM);
    StringGridPeaks.Cells[6,1+i]:=FormatFixed(Win.ManFit.CurrFixed.Peak[i].FWHM);
    StringGridPeaks.Cells[7,1+i]:=FormatValue(GaussPeakArea(Win.ManFit.CurrParam.Peak[i].Ampl,
                                                            Win.ManFit.CurrParam.Peak[i].FWHM));
    StringGridPeaks.Cells[8,1+i]:=FormatValue(Win.EvalGround(Win.ManFit.CurrParam.Peak[i].Chan,
                                                             Win.ManFit.CurrParam,Win.ManFit.TempGrad));
    StringGridPeaks.Cells[9,1+i]:=FormatValue(Win.EnCalibr(Win.ManFit.CurrParam.Peak[i].Chan));
   end;
   { Update ground polynom }
   StringGridGround.ColCount:=3;
   StringGridGround.RowCount:=1+MaxManPoly;
   StringGridGround.FixedCols:=1;
   StringGridGround.FixedRows:=1;
   StringGridGround.Cells[0,0]:='Polynom';     StringGridGround.ColWidths[0]:=w2;
   StringGridGround.Cells[1,0]:='Coefficient'; StringGridGround.ColWidths[1]:=w2;
   StringGridGround.Cells[2,0]:='Fix';         StringGridGround.ColWidths[2]:=w1;
   for i:=0 to MaxManPoly-1 do begin
    StringGridGround.Cells[0,1+i]:=Format('Coeff[%d]',[i]);
    StringGridGround.Cells[1,1+i]:=FormatValue(Win.ManFit.CurrParam.Poly[i]);
    StringGridGround.Cells[2,1+i]:=FormatFixed(Win.ManFit.CurrFixed.Poly[i]);
   end;
   { Update preset }
   ComboBoxAlgorithm.ItemIndex:=Win.ManFit.Preset.Method;
   SpinEditMaxCalls.Value:=abs(Win.ManFit.Preset.MaxCalls);
   EditTolX.Text:=FormatValue(Win.ManFit.Preset.TolX);
   EditTolF.Text:=FormatValue(Win.ManFit.Preset.TolF);
   EditTolG.Text:=FormatValue(Win.ManFit.Preset.TolG);
   { Update details }
   P:=NewText;
   Win.ManFitDetailsList(P);
   MemoDetails.Text:=P.Text;
   Kill(P);
   { Update covariation }
   if Win.ManFit.Covar.Ok then begin
    {конвертируем амплитуду в площадь так как ковариация вычислена для площади}
    Win.ManFit.WorkParam:=Win.ManFit.CurrParam;
    for i:=0 to Win.ManFit.NumPeaks-1 do
    with Win.ManFit.WorkParam.Peak[i] do Ampl:=GaussPeakArea(Ampl,FWHM);
    {}
    n:=ManPeakDim*Win.ManFit.NumPeaks+MaxManPoly;
    StringGridCovar.ColCount:=1+n;
    StringGridCovar.RowCount:=1+n;
    if (StringGridCovar.ColCount>1) then StringGridCovar.FixedCols:=1;
    if (StringGridCovar.RowCount>1) then StringGridCovar.FixedRows:=1;
    for i:=0 to n-1 do begin
     if (i<MaxManPoly) then begin
      StringGridCovar.Cells[0,1+i]:=Format('Coeff[%d]',[i]);
      StringGridCovar.Cells[1+i,0]:=StringGridCovar.Cells[0,1+i];
     end else begin
      j:=i-MaxManPoly;
      case j mod ManPeakDim of
       0:StringGridCovar.Cells[0,1+i]:=Format('Chan[%d]',[j div ManPeakDim]);
       1:StringGridCovar.Cells[0,1+i]:=Format('Area[%d]',[j div ManPeakDim]);
       2:StringGridCovar.Cells[0,1+i]:=Format('FWHM[%d]',[j div ManPeakDim]);
      end;
      StringGridCovar.Cells[1+i,0]:=StringGridCovar.Cells[0,1+i];
     end;
     for j:=0 to n-1 do begin
      cij:=Win.ManFit.Covar[i,j];
      ci:=sqrt(Win.ManFit.Covar[i,i]);
      cj:=sqrt(Win.ManFit.Covar[j,j]);
      vi:=abs(Win.ManFit.WorkParam.P[i]);
      vj:=abs(Win.ManFit.WorkParam.P[j]);
      case ComboBoxCovKind.ItemIndex of
       0 :  begin
             vij:=cij;
             StringGridCovar.Cells[1+i,1+j]:=Format('%8.5g',[vij])
            end;
       1 :  begin
             vij:=sqrt(abs(cij)/vi/vj)*100;
             StringGridCovar.Cells[1+i,1+j]:=Format('%8.5g%s',[vij,'%'])
            end;
       2 :  begin
             vij:=cij/ci/cj;
             StringGridCovar.Cells[1+i,1+j]:=Format('%8.3f%s',[vij,'%'])
            end;
       3 :  begin
             vij:=cij/ci/cj*100;
             StringGridCovar.Cells[1+i,1+j]:=Format('%8.3f%s',[vij,'%'])
            end;
       else begin
             vij:=_Nan;
             StringGridCovar.Cells[1+i,1+j]:=Format('%g',[vij])
            end;
      end;
     end;
    end;
   end else begin
    StringGridCovar.ColCount:=1;
    StringGridCovar.RowCount:=1;
   end;
  end;
  if HasFlags(Flags,2) then begin
   R.Left:=1;
   R.Top:=Win.ManFitFindNearest(Win.Marker)+1;
   R.Right:=R.Left;
   R.Bottom:=R.Top;
   if (R.Top<StringGridPeaks.RowCount) then
   if (R.Left<StringGridPeaks.ColCount) then begin
    StringGridPeaks.Col:=R.Left;
    StringGridPeaks.Row:=R.Top;
    StringGridPeaks.Selection:=R;
    StringGridPeaks.Update;
   end;
  end;
  if HasFlags(Flags,4) then begin
   for i:=0 to Win.ManFit.NumPeaks-1 do begin
    UpdateValue(StringGridPeaks.Cells[1,1+i],Win.ManFit.CurrParam.Peak[i].Chan);
    UpdateFixed(StringGridPeaks.Cells[2,1+i],Win.ManFit.CurrFixed.Peak[i].Chan);
    UpdateValue(StringGridPeaks.Cells[3,1+i],Win.ManFit.CurrParam.Peak[i].Ampl);
    UpdateFixed(StringGridPeaks.Cells[4,1+i],Win.ManFit.CurrFixed.Peak[i].Ampl);
    UpdateValue(StringGridPeaks.Cells[5,1+i],Win.ManFit.CurrParam.Peak[i].FWHM);
    UpdateFixed(StringGridPeaks.Cells[6,1+i],Win.ManFit.CurrFixed.Peak[i].FWHM);
   end;
   for i:=0 to MaxManPoly-1 do begin
    UpdateValue(StringGridGround.Cells[1,1+i],Win.ManFit.CurrParam.Poly[i]);
    UpdateFixed(StringGridGround.Cells[2,1+i],Win.ManFit.CurrFixed.Poly[i]);
   end;
   Win.ManFit.Preset.Method:=ComboBoxAlgorithm.ItemIndex;
   Win.ManFit.Preset.MaxCalls:=abs(SpinEditMaxCalls.Value);
   Str2Real(EditTolX.Text,Win.ManFit.Preset.TolX);
   Str2Real(EditTolF.Text,Win.ManFit.Preset.TolF);
   Str2Real(EditTolG.Text,Win.ManFit.Preset.TolG);
  end;
  if HasFlags(Flags,8) then begin
  end;
  if HasFlags(Flags,$10) then begin
   PageControl.PageIndex:=0;
  end;
  if HasFlags(Flags,$20) then begin
   PageControl.PageIndex:=2;
  end;
  if HasFlags(Flags,$40) then begin
   PageControl.PageIndex:=3;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrManFitControl.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 LocateFormToCenterOfScreen(Self);
 Top:=0;
 TabSheetPeakList.Caption:=RusEng('Список пиков','Peak list');
 TabSheetGround.Caption:=RusEng('Подложка','Ground');
 TabSheetControl.Caption:=RusEng('Команды управления','Controls');
 TabSheetPreset.Caption:=RusEng('Параметры фитирования','Preset');
 TabSheetDetails.Caption:=RusEng('Детали минимизации','Minimization details');
 TabSheetErrorMatrix.Caption:=RusEng('Матрица ошибок','Covariation matrix');
 LabelAlgorithm.Caption:=RusEng('Алгоритм минимизации','Minimization algorithm');
 ComboBoxAlgorithm.Items.Text:='Nelder-Mead             : simplex search'+EOL+
                               'Davidon-Fletcher-Powell : quasi-Newton'+EOL+
                               'Broyden                 : quasi-Newton'+EOL+
                               'Pearson-2               : quasi-Newton'+EOL+
                               'Pearson-3               : quasi-Newton'+EOL+
                               'Zoutendijk              : quasi-Newton'+EOL+
                               'Steward                 : D-F-P,digital grad.'+EOL+
                               'Fletcher-Reeves         : conjugate gradients'+EOL+
                               'Davidon-2               : quasi-Newton';
 ButtonOk.Caption:=mrCaption(mrOk);
 ButtonCancel.Caption:=mrCaption(mrCancel);
 LabelMaxCalls.Caption:=RusEng('Предельное значение числа вызовов целевой функции',
                               'Limit for maximal number of objective function calls');
 LabelTolX.Caption:=RusEng('Допуск по X','X-Tolerance');
 LabelTolF.Caption:=RusEng('Допуск по F','F-Tolerance');
 LabelTolG.Caption:=RusEng('Допуск по G','G-Tolerance');
 GroupBoxPeakList.Caption:=RusEng('Команды','Cammands');
 ButtonPeakListOk.Caption:=mrCaption(mrOk);
 ButtonPeakListCancel.Caption:=mrCaption(mrCancel);
 ButtonPeakListInsert.Caption:=RusEng('Вставить','Insert');
 ButtonPeakListDelete.Caption:=RusEng('Удалить','Delete');
 GroupBoxGround.Caption:=RusEng('Команды','Cammands');
 ButtonGroundOk.Caption:=mrCaption(mrOk);
 ButtonGroundCancel.Caption:=mrCaption(mrCancel);
 ButtonInterval.Caption:=RusEng('Задание РОИ','Set ROI');
 ButtonPeakSearch.Caption:=RusEng('Поиск пиков','Find peaks');
 ButtonFit.Caption:=RusEng('Подгонка пиков','Start fit');
 ButtonClear.Caption:=RusEng('Очистка пиков','Clear peaks');
 ButtonSavePik.Caption:=RusEng('Сохранить *.pik','Save *.pik');
 ButtonInsertPeak.Caption:=RusEng('Вставить пик','Insert peak');
 ButtonDeletePeak.Caption:=RusEng('Удалить пик','Delete peak');
 LabelCovKind.Caption:=RusEng('Вид матрицы ошибок','Kind of error matrix');
 ComboBoxCovKind.ItemIndex:=0;
end;

procedure TFormSpectrManFitControl.ButtonIntervalClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.ManFitChangeInterval;
end;

procedure TFormSpectrManFitControl.ButtonPeakSearchClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.ManFitCallPeakin;
end;

procedure TFormSpectrManFitControl.ButtonFitClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.ManFitExecuteThread;
end;

procedure TFormSpectrManFitControl.ButtonClearClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.ManFitClear;
end;

procedure TFormSpectrManFitControl.ButtonOkClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(4);
  UpdateControls(1);
 end;
end;

procedure TFormSpectrManFitControl.ButtonCancelClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(1);
 end;
end;

procedure TFormSpectrManFitControl.ButtonPeakListOkClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(4);
  UpdateControls(1);
  Win.DrawView;
 end;
end;

procedure TFormSpectrManFitControl.ButtonPeakListCancelClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(1);
 end;
end;

procedure TFormSpectrManFitControl.ButtonGroundOkClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(4);
  UpdateControls(1);
  Win.DrawView;
 end;
end;

procedure TFormSpectrManFitControl.ButtonGroundCancelClick(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(1);
 end;
end;

procedure TFormSpectrManFitControl.StringGridPeaksClick(Sender: TObject);
var
 n : Integer;
begin
 if Ok and Win.Ok then
 try
  n:=StringGridPeaks.Row-1;
  if (n>=0) and (n<Win.ManFit.NumPeaks)
  then Win.Marker:=Round(Win.ManFit.CurrParam.Peak[n].Chan);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrManFitControl.ButtonPeakListInsertClick(Sender: TObject);
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  try
   Win.ManFitPeakInsert(Win.Marker);
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrManFitControl.ButtonPeakListDeleteClick(Sender: TObject);
var n:Integer;
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  try
   n:=StringGridPeaks.Row-1;
   if InRange(n,0,Win.ManFit.NumPeaks-1) then Win.ManFitPeakDelete(n);
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrManFitControl.ComboBoxCovKindChange(Sender: TObject);
begin
 if Ok and Win.Ok then begin
  UpdateControls(1);
 end;
end;

procedure TFormSpectrManFitControl.ButtonSavePikClick(Sender: TObject);
begin
 if Ok then
 if Win.Ok then begin
  SaveDialog.Title:=RusEng('Сохранить файл списка пиков','Save peak list file');
  SaveDialog.Filter:=RusEng('Файл списка пиков (*.pik)|*.pik',
                            'Peak list files   (*.pik)|*.pik');
  if GuardOpenDialog(SaveDialog).Execute then begin
   Win.ManFitSavePik(SaveDialog.FileName,FileExists(SaveDialog.FileName) and
                    (YesNo(RusEng('Файл существует!'+EOL+'Добавить в конец файла?',
                                  'File already exists!'+EOL+'Append to end of file?'))=mrYes));
  end;
 end;
end;

procedure TFormSpectrManFitControl.ButtonInsertPeakClick(Sender: TObject);
begin
 ButtonPeakListInsertClick(Sender);
end;

procedure TFormSpectrManFitControl.ButtonDeletePeakClick(Sender: TObject);
begin
 ButtonPeakListDeleteClick(Sender);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrmanfitcontrol;
begin
end;

procedure Free_form_spectrmanfitcontrol;
begin
end;

initialization

 Init_form_spectrmanfitcontrol;

finalization

 Free_form_spectrmanfitcontrol;

end.

//////////////
// END OF FILE
//////////////

