////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Analysis RFA BPF Job Wizard.                                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231208 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectranalysisrfabfpjobwizard; // Form Spectr Analysis RFA BPF Job Wizard

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Grids,
 lcltype, lclintf, Spin, CheckLst,
 Form_CrwDaqSysChild,
 Form_SpectrWindow, Form_SpectrDaqWindow,
 Form_SpectrRfaBfpWindow, 
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_ee, _crw_zm, _crw_sort,
 _crw_curves, _crw_riff, _crw_calib, _crw_lsqpoly,
 _crw_daqsys, _crw_daqdev, _crw_rfadata,
 _crw_appforms, _crw_apptools, _crw_apputils;


type

  { TFormSpectrAnalysisRfaBfpJobWizard }

  TFormSpectrAnalysisRfaBfpJobWizard = class(TMasterForm)
    PanelControls: TPanel;
    StatusBar: TStatusBar;
    PageControl: TPageControl;
    TabSheetFiles: TTabSheet;
    TabSheetElements: TTabSheet;
    PanelElements: TPanel;
    CheckBoxSelEnable: TCheckBox;
    PaintBoxElements: TPaintBox;
    PanelFiles: TPanel;
    OpenDialogSPD: TOpenDialog;
    GroupBoxExcitation: TGroupBox;
    BitBtnLoadExcitation: TBitBtn;
    EditExcitation: TEdit;
    GroupBoxTargetList: TGroupBox;
    ListBoxTargetList: TListBox;
    BitBtnLoadTargetList: TBitBtn;
    BitBtnClearTargetList: TBitBtn;
    LabelSelElement: TLabel;
    TabSheetDetector: TTabSheet;
    PanelDetector: TPanel;
    StringGridDetector: TStringGrid;
    TabSheetCommands: TTabSheet;
    PanelCommands: TPanel;
    BitBtnCreateJob: TBitBtn;
    BitBtnRunJob: TBitBtn;
    LabelCreateJob: TLabel;
    LabelRunJob: TLabel;
    TabSheetRoiList: TTabSheet;
    PanelRoiList: TPanel;
    GroupBoxRoiList: TGroupBox;
    ListBoxRoiList: TListBox;
    BitBtnEditDllSource: TBitBtn;
    LabelEditDllSource: TLabel;
    GroupBoxRoiCommands: TGroupBox;
    BitBtnNewRoi: TBitBtn;
    BitBtnRoiDelete: TBitBtn;
    GroupBoxGround: TGroupBox;
    SpinEditPower: TSpinEdit;
    SpinEditCoeffNum: TSpinEdit;
    LabelPower: TLabel;
    LabelCoeff: TLabel;
    EditCoeff: TEdit;
    LabelEquals: TLabel;
    CheckBoxFixed: TCheckBox;
    BitBtnRoiLeft: TBitBtn;
    BitBtnRoiRight: TBitBtn;
    LabelPoint: TLabel;
    LabelExp: TLabel;
    UpDown1: TUpDown;
    UpDown2: TUpDown;
    UpDown3: TUpDown;
    UpDown4: TUpDown;
    MemoHelp: TMemo;
    TabSheetXRayLines: TTabSheet;
    PanelXRayLines: TPanel;
    CheckListBoxXRayLines: TCheckListBox;
    BitBtnSelPrevElement: TBitBtn;
    BitBtnSelNextElement: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure PaintBoxElementsMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxElementsMouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure PaintBoxElementsMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure BitBtnLoadExcitationClick(Sender: TObject);
    procedure BitBtnLoadTargetListClick(Sender: TObject);
    procedure BitBtnClearTargetListClick(Sender: TObject);
    procedure BitBtnCreateJobClick(Sender: TObject);
    procedure BitBtnRunJobClick(Sender: TObject);
    procedure BitBtnEditDllSourceClick(Sender: TObject);
    procedure BitBtnNewRoiClick(Sender: TObject);
    procedure BitBtnRoiDeleteClick(Sender: TObject);
    procedure ListBoxRoiListClick(Sender: TObject);
    procedure BitBtnRoiLeftClick(Sender: TObject);
    procedure BitBtnRoiRightClick(Sender: TObject);
    procedure SpinButtonChangingEx(Sender: TObject; var AllowChange: Boolean;
      NewValue: SmallInt; Direction: TUpDownDirection);
    procedure SpinEditPowerChange(Sender: TObject);
    procedure SpinEditCoeffNumChange(Sender: TObject);
    procedure CheckBoxFixedClick(Sender: TObject);
    procedure EditCoeffKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure PageControlChange(Sender: TObject);
    procedure BitBtnSelPrevElementClick(Sender: TObject);
    procedure BitBtnSelNextElementClick(Sender: TObject);
  private
    { Private declarations }
    SpecWin         : TFormSpectrRfaBfpWindow;
    ButtonOfElement : packed array[RfaMinAtomId..RfaMaxAtomID] of TSpeedButton;
    ElemSelector    : TXorSelector;
    procedure ButtonOfElementClick(Sender: TObject);
    procedure UpdateControls(Flags:Integer);
  public
    { Public declarations }
  end;

function NewFormSpectrAnalysisRfaBfpJobWizard(aSpecWin:TFormSpectrRfaBfpWindow):TFormSpectrAnalysisRfaBfpJobWizard;

implementation

{$R *.lfm}

const
 idUpdateElements = 1 shl 0;
 idChangeElements = 1 shl 1;
 idUpdateGround   = 1 shl 2;
 idReadIniFile    = 1 shl 3;

function NewFormSpectrAnalysisRfaBfpJobWizard(aSpecWin:TFormSpectrRfaBfpWindow):TFormSpectrAnalysisRfaBfpJobWizard;
begin
 Application.CreateForm(TFormSpectrAnalysisRfaBfpJobWizard, Result);
 Result.SpecWin:=aSpecWin;
 Result.UpdateControls(idReadIniFile);
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.ButtonOfElementClick(Sender: TObject);
var AtomID:Byte;
begin
 if Ok then
 if SpecWin.Ok then
 if (Sender is TSpeedButton) then
 try
  AtomID:=Rfa.FindAtom((Sender as TSpeedButton).Caption);
  if Rfa.Accessible[AtomID] then begin
   SpecWin.SelElement:=AtomID;
   if CheckBoxSelEnable.Checked then UpdateControls(idChangeElements);
   UpdateControls(idUpdateElements);
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ButtonOfElementClick');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;


procedure TFormSpectrAnalysisRfaBfpJobWizard.UpdateControls(Flags:Integer);
var P:TText; i,Row,AtomID:Integer; SelElems:TByteSet; RoiItem:TRoiItemRec;
var Section,StrParam:LongString;
begin
 if Ok then
 if SpecWin.Ok then
 try
  if HasFlags(Flags,idUpdateElements) then begin
   AtomID:=SpecWin.SelElement;
   if Rfa.Accessible[AtomID] then begin
    LabelSelElement.Caption:=Format('%-3s %-2s %s %s'+EOL+
                                    'Period %s Group %-2s'+EOL+
                                    'Weight %7.3f Dens. %-7.5g',
                          [Rfa.AsString[AtomId,rf_ATOM_ID], Rfa.AsString[AtomId,rf_ATOM_SYMBOL],
                           Rfa.AsString[AtomId,rf_ATOM_NAME_ENG], Rfa.AsString[AtomId,rf_ATOM_NAME_RUS],
                           Rfa.AsString[AtomId,rf_ATOM_PERIOD], Rfa.AsString[AtomId,rf_ATOM_GROUP],
                           Rfa[AtomId,rf_ATOM_WEIGHT], Rfa[AtomId,rf_ATOM_DENSITY]]);
   end else begin
    LabelSelElement.Caption:='';
   end;
   for AtomID:=RfaMinAtomId to RfaMaxAtomID do ButtonOfElement[AtomID].Down:=(AtomID in SpecWin.SelElements);
  end;
  if HasFlags(Flags,idChangeElements) then begin
   SelElems:=[];
   for AtomID:=Low(ButtonOfElement) to High(ButtonOfElement) do
   if ButtonOfElement[AtomID].Down
   then include(SelElems,AtomID)
   else exclude(SelElems,AtomID);
   SpecWin.SelElements:=SelElems;
  end;
  if HasFlags(Flags,idUpdateGround) then begin
   with ListBoxRoiList do
   if (ItemIndex>=0) and (ItemIndex<Items.Count) then begin
    GroupBoxGround.Show;
    RoiItem:=StrToRoiItemRec(Items[ItemIndex]);
    SpinEditPower.Value:=RoiItem.Power;
    SpinEditCoeffNum.MaxValue:=RoiItem.Power;
    SpinEditCoeffNum.Value:=Min(RoiItem.Power,SpinEditCoeffNum.Value);
    EditCoeff.Text:=Format('%g',[RoiItem.Coeff[SpinEditCoeffNum.Value]]);
    CheckBoxFixed.Checked:=RoiItem.Fixed and GetBitMask(SpinEditCoeffNum.Value)<>0;
   end else begin
    GroupBoxGround.Hide;
   end;
  end;
  if HasFlags(Flags,idReadIniFile) then begin
   Section:='';   StrParam:='';
   if ReadIniFileAlpha(SpecWin.ConfigFile,SpecWin.WindowSection,'DetectorSection%a',Section) then begin
    Section:=UnifySection(Section);
    for Row:=1 to 15 do
    if ReadIniFileString(SpecWin.ConfigFile,Section,Trim(StringGridDetector.Cells[0,Row])+'%s',StrParam)
    then StringGridDetector.Cells[1,Row]:=Trim(StrParam);
   end;
   if ReadIniFileAlpha(SpecWin.ConfigFile,SpecWin.WindowSection,'ArgumentSection%a',Section) then begin
    Section:=UnifySection(Section);
    P:=ExtractListSection(SpecWin.ConfigFile,Section,efAsIs+efDelCom);
    try
     for i:=0 to P.Count-1 do if (Pos('=',P[i])>0) then
     for Row:=16 to StringGridDetector.RowCount-1 do
     if StringGridDetector.Cells[0,Row]='' then begin
      StringGridDetector.Cells[0,Row]:=Trim(Copy(P[i],1,Pos('=',P[i])-1));
      StringGridDetector.Cells[1,Row]:=Trim(Copy(P[i],Pos('=',P[i])+1,255));
      if StringGridDetector.Cells[0,Row]='' then StringGridDetector.Cells[1,Row]:='';
      Break;
     end;
    finally
     Kill(P);
    end;
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'UpdateControls');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.FormCreate(Sender: TObject);
 procedure InitElementButtons;
 const
  X0=0; Y0=0; DW=4; DH=4;
 var
  i,x,y,W,H:Integer; Btn:TSpeedButton;
 begin
  W:=((PanelElements.Width - X0 - DW) div 18) - DW;
  H:=((PanelElements.Height - Y0 - DW) div 9) - DH;
  for i:=Low(ButtonOfElement) to High(ButtonOfElement) do
  if Rfa.Accessible[i] then begin
   x:=X0+DW+round(Rfa[i,rf_ELEMPOS_X]*(W+DW));
   y:=Y0+DW+round(Rfa[i,rf_ELEMPOS_Y]*(H+DH));
   Btn:=TSpeedButton.Create(Self);
   Btn.Caption:=Rfa.AtomSymbol[i];
   Btn.AllowAllUp:=True;
   Btn.GroupIndex:=i;
   Btn.Down:=false;
   Btn.Left:=x;
   Btn.Top:=y;
   Btn.Width:=W;
   Btn.Height:=H;
   Btn.Parent:=PanelElements;
   Btn.OnClick:=ButtonOfElementClick;
   Btn.Font.Color:=Rfa.AtomColor[i];
   ButtonOfElement[i]:=Btn;
   Btn.Update;
  end;
 end;
 procedure InitXRayLines;
 var
  i      : Integer;
  LineId : TRfaFields;
 begin
  with CheckListBoxXRayLines do begin
   Items.Clear;
   for LineId:=rf_FIRST_FIELD to rf_LAST_FIELD do
   if (LineId in rf_All_Series) and (Rfa.LineName[LineId]<>'') then Items.Add(Rfa.LineName[LineId]);
   for i:=0 to Items.Count-1 do Checked[i]:=Rfa.FindLine(Items[i]) in rf_All_Series_Shorten;
  end;
 end;
begin
 inherited;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 LocateFormToCenterOfScreen(Self);
 Caption:=RusEng('Параметры задания на анализ','Analysis preset');
 TabSheetFiles.Caption:=RusEng('Файлы','Files');
 TabSheetElements.Caption:=RusEng('Элементы','Elements');
 TabSheetXRayLines.Caption:=RusEng('Линии','Lines');
 TabSheetDetector.Caption:=RusEng('Детектор','Detector');
 TabSheetRoiList.Caption:=RusEng('РОИ','ROI');
 TabSheetCommands.Caption:=RusEng('Команды','Commands');
 CheckBoxSelEnable.Caption:=RusEng('Выбор','Select');
 LabelSelElement.Caption:='';
 GroupBoxExcitation.Caption:=RusEng('Спектр возбуждения','Excitation spectra');
 BitBtnLoadExcitation.Hint:=RusEng('Ввод','Load');
 GroupBoxTargetList.Caption:=RusEng('Спектры для обработки','Target spectra list');
 BitBtnLoadTargetList.Hint:=RusEng('Ввод','Load');
 BitBtnClearTargetList.Hint:=RusEng('Сброс','Clear');
 with StringGridDetector do begin
  ColCount:=2;
  RowCount:=50;
  FixedCols:=1;
  FixedRows:=1;
  ColWidths[0]:=210;
  ColWidths[1]:=800;
  Cells[0,0]:=RusEng('Имя','Name');           Cells[1,0]:=RusEng('Значение','Value');
  Cells[0,1]:='Detector.Allow';               Cells[1,1]:='1';
  Cells[0,2]:='Detector.Element';             Cells[1,2]:='Si';
  Cells[0,3]:='Detector.WorkThickness';       Cells[1,3]:='0.2';
  Cells[0,4]:='Detector.DeadThickness';       Cells[1,4]:='0.00001';
  Cells[0,5]:='Detector.AirDensity';          Cells[1,5]:='0.0012';
  Cells[0,6]:='Detector.Layer[1].Element';    Cells[1,6]:='Air';
  Cells[0,7]:='Detector.Layer[1].Thickness';  Cells[1,7]:='3';
  Cells[0,8]:='Detector.Layer[2].Element';    Cells[1,8]:='Be';
  Cells[0,9]:='Detector.Layer[2].Thickness';  Cells[1,9]:='0.0008';
  Cells[0,10]:='Detector.Layer[3].Element';   Cells[1,10]:='Au';
  Cells[0,11]:='Detector.Layer[3].Thickness'; Cells[1,11]:='0.000002';
  Cells[0,12]:='Detector.Layer[4].Element';   Cells[1,12]:='None';
  Cells[0,13]:='Detector.Layer[4].Thickness'; Cells[1,13]:='0';
  Cells[0,14]:='Detector.Angles.Alpha';       Cells[1,14]:='45';
  Cells[0,15]:='Detector.Angles.Beta';        Cells[1,15]:='45';
 end;
 LabelCreateJob.Caption:=RusEng('Создать задание для РФА анализа ХРИ.',
                                'Create X-Ray Fluorescence Analysis Job.');
 LabelRunJob.Caption:=RusEng('Создать и Выполнить задание для РФА анализа ХРИ.',
                             'Create and Run X-Ray Fluorescence Analysis Job.');
 LabelEditDllSource.Caption:=RusEng('Редактировать алгоритм (для "профи").',
                                    'Edit Dll source file (for programmers).');
 GroupBoxRoiList.Caption:=Format('%5s %5s %1s %4s %17s %17s',['Left','Right','P','Fix','Coeff[0]','Coeff[1]']);
 GroupBoxRoiCommands.Caption:=RusEng('Команды','Command');
 BitBtnNewRoi.Hint:=RusEng('Создать РОИ','Create ROI');
 BitBtnRoiDelete.Hint:=RusEng('Удалить РОИ','Delete ROI');
 BitBtnRoiLeft.Hint:=RusEng('Предыдущее РОИ','Previouse ROI');
 BitBtnRoiRight.Hint:=RusEng('Следующее РОИ','Next ROI');
 GroupBoxGround.Caption:=RusEng('Полином фона','Ground polynom');
 MemoHelp.Lines.Clear;
 MemoHelp.Lines.Add(RusEng('Примечание: перед созданием задания надо:',
                           'Note: before create job, you need:'));
 MemoHelp.Lines.Add(RusEng('1)Задать имя файла спектра возбуждения.',
                           '1)Set file path for excitation spectrum.'));
 MemoHelp.Lines.Add(RusEng('2)Задать имена файлов для обработки.',
                           '2)Set file path for target spectrums.'));
 MemoHelp.Lines.Add(RusEng('3)Выбрать элементы таблицы Менделеева.',
                           '3)Choose elements from Mendeleev table.'));
 MemoHelp.Lines.Add(RusEng('4)Задать список областей интереса РОИ.',
                           '4)Setup list of ROI, region of interest.'));
 MemoHelp.Lines.Add(RusEng('5)Задать параметры детектора.',
                           '5)Setup detector parameters.'));
 ElemSelector:=NewXorSelector;
 ElemSelector.Mode:=sm_Rect;
 InitElementButtons;
 InitXRayLines;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.FormDestroy(Sender: TObject);
begin
 Kill(ElemSelector);
 inherited;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.FormShow(Sender: TObject);
begin
 if Ok then begin
  UpdateControls(idUpdateElements);
  UpdateControls(idUpdateGround);
  PageControl.ActivePage:=TabSheetCommands;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.FormHide(Sender: TObject);
begin
 if Ok then PageControl.ActivePage:=TabSheetCommands;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.PaintBoxElementsMouseDown(
  Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
begin
 if Ok then
 if Button = mbLeft then
 if (Sender is TPaintBox) then
 with (Sender as TPaintBox) do ElemSelector.StartXY(Canvas,X,Y);
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.PaintBoxElementsMouseMove(
  Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
 if Ok then
 if (Sender is TPaintBox) then
 with (Sender as TPaintBox) do ElemSelector.ReplaceXY(Canvas,X,Y);
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.PaintBoxElementsMouseUp(
  Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
var
 i : Integer;
 R : TRect2I;
 B : TRect2I;
 S : TByteSet;
begin
 if Ok then
 if (Button=mbLeft) then
 if (Sender is TPaintBox) then with (Sender as TPaintBox) do
 if ElemSelector.Stop(Canvas) then
 if CheckBoxSelEnable.Checked then begin
  R:=RectValidate(ElemSelector.Selection);
  for i:=Low(ButtonOfElement) to High(ButtonOfElement) do begin
   with ButtonOfElement[i] do B:=Rect2I(Left,Top,Left+Width,Top+Height);
   if RectContainsPoint(R,RectCenter(B)) then begin
    S:=SpecWin.SelElements;
    if i in S then exclude(S,i) else include(S,i);
    SpecWin.SelElements:=S;
   end;
  end;
  UpdateControls(idUpdateElements);
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnLoadExcitationClick(Sender: TObject);
begin
 if Ok then begin
  OpenDialogSPD.Title:=RusEng('Спектр возбуждения','Excitation spectra');
  OpenDialogSPD.Filter:=RusEng('Спектры (*.spd)|*.spd|Все файлы (*.*)|*.*',
                               'Spectra (*.spd)|*.spd|All files (*.*)|*.*');
  OpenDialogSPD.Options:=OpenDialogSPD.Options-[ofAllowMultiSelect];
  if GuardOpenDialog(OpenDialogSPD).Execute then
  if FileExists(OpenDialogSPD.FileName)
  then EditExcitation.Text:=FExpand(OpenDialogSPD.FileName);
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnLoadTargetListClick(Sender: TObject);
var
 i : Integer;
begin
 if Ok then begin
  OpenDialogSPD.Title:=RusEng('Список спектров для обработки','Target spectra list');
  OpenDialogSPD.Filter:=RusEng('Спектры (*.spd)|*.spd|Все файлы (*.*)|*.*',
                               'Spectra (*.spd)|*.spd|All files (*.*)|*.*');
  OpenDialogSPD.Options:=OpenDialogSPD.Options+[ofAllowMultiSelect];
  if GuardOpenDialog(OpenDialogSPD).Execute then begin
   for i:=0 to OpenDialogSPD.Files.Count-1 do
   if FileExists(OpenDialogSPD.Files[i])
   then ListBoxTargetList.Items.Add(FExpand(OpenDialogSPD.Files[i]));
  end;
 end; 
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnClearTargetListClick(Sender: TObject);
begin
 if Ok then ListBoxTargetList.Items.Clear;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnCreateJobClick(Sender: TObject);
begin
 if Ok and SpecWin.Ok then SpecWin.CreateJob;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnRunJobClick(Sender: TObject);
begin
 if Ok and SpecWin.Ok then SpecWin.RunJob;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnEditDllSourceClick(Sender: TObject);
begin
 if Ok and SpecWin.Ok then SpecWin.EditDllSource;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnNewRoiClick(Sender: TObject);
var
 i : Integer;
 R : TRoiItemRec;
 B : TRoiItemRec;
 S : LongString;
begin
 if Ok then
 if SpecWin.Ok then
 try
  with SpecWin do
  R:=RoiItemRec(MarkerL,MarkerR,1,0,[SpectrValue[MarkerL],SpectrValue[MarkerR]-SpectrValue[MarkerL]]);
  if (R.Right<=R.Left+5)
  then Raise ERfaBfpError.CreateFmt(RusEng('Интервал (%d,%d) недопустим.',
                                           'Range (%d,%d) is invalid.'),[R.Left,R.Right]);
  with ListBoxRoiList do
  for i:=0 to Items.Count-1 do begin
   B:=StrToRoiItemRec(Items[i]);
   if ((R.Left>=B.Left) and (R.Left<=B.Right)) or ((R.Right>=B.Left) and (R.Right<=B.Right))
   then Raise ERfaBfpError.Create(RusEng('Границы РОИ пересекаются с уже существующими!',
                                         'ROI bounds intersect existing one.'))
  end;
  S:=RoiItemRecToStr(R);
  ListBoxRoiList.Items.Add(S);
  with ListBoxRoiList do
  for i:=0 to Items.Count-1 do if SameText(Items[i],S) then begin
   ItemIndex:=i;
   Break;
  end;
  UpdateControls(idUpdateGround);
  SpecWin.ROIMark;
  SpecWin.StatusMessage:='Ok.';
 except
  on E:Exception do begin
   BugReport(E,Self,'BitBtnNewRoiClick');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnRoiDeleteClick(Sender: TObject);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  Items.Delete(ItemIndex);
  ItemIndex:=Min(0,Items.Count-1);
  if (ItemIndex>=0) and (ItemIndex<Items.Count) then begin
   R:=StrToRoiItemRec(Items[ItemIndex]);
   SpecWin.SetMarkerLR(R.Left,R.Right);
   SpecWin.Marker:=(R.Left+R.Right) shr 1;
   SpecWin.ROIMark;
  end else begin
   SpecWin.DrawView;
  end;
  UpdateControls(idUpdateGround);
  SpecWin.StatusMessage:='Ok.';
 except
  on E:Exception do begin
   BugReport(E,Self,'BitBtnRoiDeleteClick');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.ListBoxRoiListClick(Sender: TObject);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  R:=StrToRoiItemRec(Items[ItemIndex]);
  SpecWin.SetMarkerLR(R.Left,R.Right);
  SpecWin.Marker:=(R.Left+R.Right) shr 1;
  SpecWin.ROIMark;
  UpdateControls(idUpdateGround);
  SpecWin.StatusMessage:='Ok.';
 except
  on E:Exception do begin
   BugReport(E,Self,'ListBoxRoiListClick');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnRoiLeftClick(Sender: TObject);
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>0) then begin
  ItemIndex:=ItemIndex-1;
  ListBoxRoiListClick(Sender);
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnRoiRightClick(Sender: TObject);
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex<Items.Count-1) then begin
  ItemIndex:=ItemIndex+1;
  ListBoxRoiListClick(Sender);
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.SpinEditPowerChange(Sender: TObject);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  R:=StrToRoiItemRec(Items[ItemIndex]);
  if R.Power<>SpinEditPower.Value then begin
   R.Power:=Max(0,Min(9,SpinEditPower.Value));
   Items[ItemIndex]:=RoiItemRecToStr(R);
   ListBoxRoiListClick(Sender);
   SpecWin.StatusMessage:='Ok.';
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'SpinEditPowerChange');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.SpinEditCoeffNumChange(Sender: TObject);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  R:=StrToRoiItemRec(Items[ItemIndex]);
  if SpinEditCoeffNum.Value in [0..9] then begin
   EditCoeff.Text:=Format('%g',[R.Coeff[SpinEditCoeffNum.Value]]);
   CheckWithoutClick(CheckBoxFixed,R.Fixed and GetBitMask(SpinEditCoeffNum.Value)<>0);
   SpecWin.StatusMessage:='Ok.';
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'SpinEditCoeffNumChange');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.CheckBoxFixedClick(Sender: TObject);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  R:=StrToRoiItemRec(Items[ItemIndex]);
  if SpinEditCoeffNum.Value in [0..9] then
  if (R.Fixed and GetBitMask(SpinEditCoeffNum.Value)<>0)<>CheckBoxFixed.Checked then begin
   if CheckBoxFixed.Checked
   then R.Fixed:=R.Fixed or      GetBitMask(SpinEditCoeffNum.Value)
   else R.Fixed:=R.Fixed and not GetBitMask(SpinEditCoeffNum.Value);
   Items[ItemIndex]:=RoiItemRecToStr(R);
   ListBoxRoiListClick(Sender);
   SpecWin.StatusMessage:='Ok.';
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'CheckBoxFixedClick');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;


procedure TFormSpectrAnalysisRfaBfpJobWizard.EditCoeffKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
var
 R : TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 if Key=VK_RETURN then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  R:=StrToRoiItemRec(Items[ItemIndex]);
  if SpinEditCoeffNum.Value in [0..9] then
  if Str2Real(EditCoeff.Text,R.Coeff[SpinEditCoeffNum.Value]) then begin
   Items[ItemIndex]:=RoiItemRecToStr(R);
   ListBoxRoiListClick(Sender);
   SpecWin.StatusMessage:='Ok.';
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'EditCoeffKeyDown');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.SpinButtonChangingEx(
  Sender: TObject; var AllowChange: Boolean; NewValue: SmallInt;
  Direction: TUpDownDirection);
var d:Double; p,e,dir:Integer; s:LongString; R:TRoiItemRec;
begin
 if Ok then
 if SpecWin.Ok then
 with ListBoxRoiList do
 if (ItemIndex>=0) and (ItemIndex<Items.Count) then
 try
  case Direction of
   updUp   : dir:=+1;
   updDown : dir:=-1;
   else Exit;
  end;
  R:=StrToRoiItemRec(Items[ItemIndex]);
  if (SpinEditCoeffNum.Value in [0..9]) then begin
   d:=R.Coeff[SpinEditCoeffNum.Value];
   s:=Format('%.5e',[d]);
   p:=pos('E',s);
   if (p>0) and Str2Int(Copy(s,p+1,10),e) then begin
    if Sender=UpDown1 then d:=d+dir*IntPower(10,e);
    if Sender=UpDown2 then d:=d+dir*IntPower(10,e-1);
    if Sender=UpDown3 then d:=d+dir*IntPower(10,e-2);
    if Sender=UpDown4 then d:=d*Power(10,dir);
   end;
   if (d<>R.Coeff[SpinEditCoeffNum.Value]) then begin
    R.Coeff[SpinEditCoeffNum.Value]:=d;
    Items[ItemIndex]:=RoiItemRecToStr(R);
    ListBoxRoiListClick(Sender);
    SpecWin.StatusMessage:='Ok.';
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'SpinButtonChangingEx');
   SpecWin.StatusMessage:=E.Message;
  end;
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.PageControlChange(Sender: TObject);
begin
 if Ok then
 if SpecWin.Ok then SpecWin.DrawView;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnSelPrevElementClick(Sender: TObject);
var
 AtomId : Integer;
begin
 if Ok then
 if SpecWin.Ok then begin
  AtomId:=SpecWin.SelElement-1;
  if AtomId in [Low(ButtonOfElement)..High(ButtonOfElement)]
  then ButtonOfElementClick(ButtonOfElement[AtomId]);
 end;
end;

procedure TFormSpectrAnalysisRfaBfpJobWizard.BitBtnSelNextElementClick(Sender: TObject);
var
 AtomId : Integer;
begin
 if Ok then
 if SpecWin.Ok then begin
  AtomId:=SpecWin.SelElement+1;
  if AtomId in [Low(ButtonOfElement)..High(ButtonOfElement)]
  then ButtonOfElementClick(ButtonOfElement[AtomId]);
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectranalysisrfabfpjobwizard;
begin
end;

procedure Free_form_spectranalysisrfabfpjobwizard;
begin
end;

initialization

 Init_form_spectranalysisrfabfpjobwizard;

finalization

 Free_form_spectranalysisrfabfpjobwizard;

end.

//////////////
// END OF FILE
//////////////

