////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Dialog Device.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231207 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqdialogdevice; // Form DAQ Dialog Device

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf, CheckLst,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow,
 Form_SurfWindow, Form_CircuitWindow,
 Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator,
 Form_ListBoxSelection, Form_UartTerminal,
 Form_CalibDialog, Form_DaqEditTagDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_curves, _crw_riff, _crw_pio,
 _crw_calib, _crw_couple, _crw_daqtags, _crw_daqevnt,
 _crw_daqsys, _crw_daqdev, _crw_softdev, _crw_softdevdialog,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormDaqDialogDevice = class(TDaqDevicePropertyDialog)
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure BitBtnOkClick(Sender: TObject);
    procedure BitBtnCancelClick(Sender: TObject);
  private
    { Private declarations }
    myLink    : packed record
     Device   : TDialogDevice;
     Count    : Integer;
     Items    : packed array[0..127] of packed record
      Control : TControl;
      Tag     : Integer
     end;
    end;
    procedure   LinkTag(aControl:TControl; aTag:Integer);
    procedure   InitLinks(aDevice:TDialogDevice);
  public
    { Public declarations }
    function    Transfer(Dlg2Tag:Boolean):Boolean;
    procedure   LabelClick(Sender: TObject);
  end;

function NewFormDaqDialogDevice(aDevice:TDialogDevice):TFormDaqDialogDevice;

implementation

{$R *.lfm}

function NewFormDaqDialogDevice(aDevice:TDialogDevice):TFormDaqDialogDevice;
begin
 Application.CreateForm(TFormDaqDialogDevice, Result);
 if Result.Ok then begin
  Result.InitLinks(aDevice);
  LocateFormToCenterOfScreen(Result);
  Result.Show;
 end;
end;

procedure TFormDaqDialogDevice.LinkTag(aControl:TControl; aTag:Integer);
begin
 if Ok then
 if (aControl is TControl) then
 if (TypeTag(aTag) in [1,2,3]) then
 if (myLink.Count<High(myLink.Items)) then begin
  with myLink.Items[myLink.Count] do begin
   Control:=aControl;
   Tag:=aTag;
  end;
  inc(myLink.Count);
 end;
end;

function TFormDaqDialogDevice.Transfer(Dlg2Tag:Boolean):Boolean;
var i,j,aTag:Integer; d:LongInt; f:Double;
var aControl:TControl; s:LongString;
label Error;
begin
 Result:=false;
 if Ok then
 try
  for i:=0 to myLink.Count-1 do begin
   aTag:=myLink.Items[i].Tag;
   aControl:=myLink.Items[i].Control;
   {
   INPUTLINE
   }
   if (aControl is TComboBox) then
   if (TComboBox(aControl).Style=csDropDown) then begin
    if Dlg2Tag then begin
     RecordComboBoxHistory(TComboBox(aControl),50);
     s:=TComboBox(aControl).Text;
     case typetag(aTag) of
      1 :  if not Str2Long(s,d) then goto Error else
           if not isettag(aTag,d) then goto Error;
      2 :  if not Str2Real(s,f) then goto Error else
           if not rsettag(aTag,f) then goto Error;
      3 :  if not ssettag(aTag,s) then goto Error;
      else goto Error;
     end;
    end else begin
     case typetag(aTag) of
      1 :  s:=Format('%d',[igettag(aTag)]);
      2 :  s:=Format('%g',[rgettag(aTag)]);
      3 :  s:=Format('%s',[sgettag(aTag)]);
      else goto Error;
     end;
     TComboBox(aControl).Text:=s;
     aControl.Update;
     RecordComboBoxHistory(TComboBox(aControl),50);
    end;
   end;
   {
   CHECKBOXES
   }
   if (aControl is TCheckListBox) then begin
    if Dlg2Tag then begin
     d:=0;
     for j:=0 to TCheckListBox(aControl).Items.Count-1 do
     if TCheckListBox(aControl).Checked[j] then d:=d or GetBitMask(j);
     case typetag(aTag) of
      1 :  if not isettag(aTag,d) then goto Error;
      else goto Error;
     end;
    end else begin
     case typetag(aTag) of
      1 :  d:=igettag(aTag);
      else goto Error;
     end;
     for j:=0 to TCheckListBox(aControl).Items.Count-1 do
     TCheckListBox(aControl).Checked[j]:=HasFlags(d,GetBitMask(j));
     aControl.Update;
    end;
   end;
   {
   COMBOBOX
   }
   if (aControl is TComboBox) then
   if (TComboBox(aControl).Style=csDropDownList) then begin
    if Dlg2Tag then begin
     d:=TComboBox(aControl).ItemIndex;
     case typetag(aTag) of
      1 :  if not isettag(aTag,d) then goto Error;
      else goto Error;
     end;
    end else begin
     case typetag(aTag) of
      1 :  d:=igettag(aTag);
      else goto Error;
     end;
     if (d>=0) then
     if (d<=TComboBox(aControl).Items.Count-1) then begin
      TComboBox(aControl).ItemIndex:=d;
      TComboBox(aControl).Text:=TComboBox(aControl).Items[d];
     end;
     aControl.Update;
    end;
   end;
   {
   LISTBOX
   }
   if (aControl is TListBox) then begin
    if Dlg2Tag then begin
     d:=TListBox(aControl).ItemIndex;
     case typetag(aTag) of
      1 :  if not isettag(aTag,d) then goto Error;
      else goto Error;
     end;
    end else begin
     case typetag(aTag) of
      1 :  d:=igettag(aTag);
      else goto Error;
     end;
     if (d>=0) then
     if (d<=TListBox(aControl).Items.Count-1) then begin
      TListBox(aControl).ItemIndex:=d;
     end;
     aControl.Update;
    end;
   end;
  end;
  Result:=true;
  exit;
Error:
  Daq.ConsoleEcho(Format(RusEng('Ошибка обмена, устройство "%s", тег "%s"',
                                'Transfer error, device "%s", tag "%s"'),
                                [myLink.Device.Name, NameTag(aTag)]));
 except
  on E:Exception do BugReport(E,Self,'Transfer');
 end;
end;

procedure TFormDaqDialogDevice.InitLinks(aDevice:TDialogDevice);
const dy1=25; dy2=5; mx1=5; my1=5; mx2=25; my2=25;
label NextLine,Error;
var
 Line          : LongString;
 LineNum       : Integer;
 TempBuf       : TParsingBuffer;
 ItemCount     : Integer;
 Data          : packed record
  ID           : PureString;
  R            : TRect2I;
  PAR          : PureString;
 end;
 aTag          : Integer;
 aBevel        : TBevel;
 aLabel        : TLabel;
 aDialog       : TMasterForm;
 aListBox      : TListBox;
 aFixLabel     : TLabel;
 aComboBox     : TComboBox;
 aCheckListBox : TCheckListBox;
 aBoundsRect   : TRect2I;
 aOrigins      : TPoint2I;
 aMargins      : TPoint2I;
 aDeltaY       : Integer;
 procedure Locate(aControl:TControl; const aRect:TRect2I);
 begin
  aControl.Left:=aRect.A.X;
  aControl.Top:=aRect.A.Y+aDeltaY;
  aControl.Width:=RectSizeX(aRect);
  aControl.Height:=RectSizeY(aRect);
  if (aControl<>Self) then begin
   aBoundsRect:=RectUnion(aBoundsRect,Rect2I(aControl.BoundsRect));
   aOrigins.x:=Min(aOrigins.x,aControl.Left);
   aOrigins.y:=Min(aOrigins.y,aControl.Top);
  end;
 end;
 procedure LinkLabel(aControl:TWinControl);
 begin
  if (aControl is TWinControl) then
  if (aFixLabel is TLabel) then begin
   aFixLabel.FocusControl:=aControl;
   aFixLabel.OnClick:=LabelClick;
  end;
  aFixLabel:=nil;
 end;
begin
 aDialog:=nil;
 if Ok and aDevice.Ok and (aDevice.DialogText.Count>0) then begin
  aFixLabel:=nil;
  myLink.Device:=aDevice;
  myLink.Count:=0;
  LineNum:=0; SafeFillChar(Data,SizeOf(Data),0);
  aBoundsRect:=Rect2I(0,0,0,0);
  aOrigins:=Point2I(Width,Height);
  aDeltaY:=BitBtnOk.BoundsRect.Bottom-dy1;
  while (LineNum<myLink.Device.DialogText.Count) do begin
   Line:=myLink.Device.DialogText[LineNum];
   if IsEmptyStr(Line) then goto NextLine;
   if (ScanVarRecord(svAsIs,StrCopyBuff(TempBuf,Line),'%a;%i;%i;%i;%i;%s',Data)=nil) then goto Error;
   if RectIsEmpty(Data.R) then goto Error;
   Data.ID:=UpcaseStr(Data.ID);
   Data.PAR:=TrimChars(Data.PAR,JustBlanks,JustBlanks);
   {
   DIALOG <left> <top> <rigth> <bottom> <dialog name>
   }
   if not aDialog.Ok then
   if SameText(Data.ID,'DIALOG') then begin
    aDialog:=Self;
    Caption:=Data.PAR;
    Locate(Self,Data.R);
    LinkLabel(nil);
    goto NextLine;
   end;
   {
   FRAMEBAR <left> <top> <rigth> <bottom> <caption>
   }
   if aDialog.Ok then
   if SameText(Data.ID,'FRAMEBAR') then begin
    aBevel:=TBevel.Create(Self);
    aBevel.Parent:=Self;
    aBevel.Shape:=bsFrame;
    Locate(aBevel,Rect2I(Data.R.A.X, Data.R.A.Y+6, Data.R.B.X, Data.R.B.Y));
    aLabel:=TLabel.Create(Self);
    aLabel.Parent:=Self;
    aLabel.AutoSize:=true;
    aLabel.Color:=Self.Color;
    aLabel.Transparent:=false;
    aLabel.Caption:=' '+Data.PAR+' ';
    Locate(aLabel,Rect2I(Data.R.A.X+4, Data.R.A.Y, Data.R.A.X+4+aLabel.Width, Data.R.A.Y+aLabel.Height));
    LinkLabel(nil);
    goto NextLine;
   end;
   {
   STATICTEXT <left> <top> <right> <bottom> <text>
   }
   if aDialog.Ok then
   if SameText(Data.ID,'STATICTEXT') then begin
    aLabel:=TLabel.Create(Self);
    aLabel.Parent:=Self;
    aLabel.AutoSize:=true;
    aLabel.Caption:=Data.PAR;
    Locate(aLabel,Data.R);
    LinkLabel(nil);
    goto NextLine;
   end;
   {
   LABEL <left> <top> <right> <bottom> <caption>
   }
   if aDialog.Ok then
   if SameText(Data.ID,'LABEL') then begin
    aLabel:=TLabel.Create(Self);
    aLabel.Parent:=Self;
    aLabel.AutoSize:=true;
    aLabel.Caption:=Data.PAR;
    Locate(aLabel,Data.R);
    aFixLabel:=aLabel;
    goto NextLine;
   end;
   {
   INPUTLINE <left> <top> <right> <bottom> <tag name>
   }
   if aDialog.Ok then
   if SameText(Data.ID,'INPUTLINE') then begin
    aComboBox:=TComboBox.Create(Self);
    aComboBox.Parent:=Self;
    aComboBox.Style:=csDropDown;
    Locate(aComboBox,Data.R);
    aTag:=findtag(ExtractWord(1,Data.PAR,ScanSpaces));
    if not (typetag(aTag) in [1,2,3]) then goto Error;
    LinkTag(aComboBox,aTag);
    LinkLabel(aComboBox);
    goto NextLine;
   end;
   {
   CHECKBOXES <left> <top> <right> <bottom> <tag> <count>
    <item 1>
    ...
    <item count>
   }
   if aDialog.Ok then
   if SameText(Data.ID,'CHECKBOXES') then begin
    aCheckListBox:=TCheckListBox.Create(Self);
    aCheckListBox.Parent:=Self;
    Locate(aCheckListBox,Data.R);
    aTag:=findtag(ExtractWord(1,Data.PAR,ScanSpaces));
    if (typetag(aTag)<>1) then goto Error;
    if not Str2Int(ExtractWord(2,Data.PAR,ScanSpaces),ItemCount) then goto Error;
    if (ItemCount<1) or (ItemCount>32) then goto Error;
    while (ItemCount>0) do begin
     inc(LineNum);
     dec(ItemCount);
     Line:=Pad(TrimChars(myLink.Device.DialogText[LineNum],JustBlanks,JustBlanks),3);
     aCheckListBox.Items.Add(Line);
    end;
    LinkTag(aCheckListBox,aTag);
    LinkLabel(aCheckListBox);
    goto NextLine;
   end;
   {
   COMBOBOX left top right bottom tag count
    item 1
    ...
    item count
   }
   if aDialog.Ok then
   if SameText(Data.ID,'COMBOBOX') then begin
    aComboBox:=TComboBox.Create(Self);
    aComboBox.Parent:=Self;
    aComboBox.Style:=csDropDownList;
    Locate(aComboBox,Data.R);
    aTag:=findtag(ExtractWord(1,Data.PAR,ScanSpaces));
    if (typetag(aTag)<>1) then goto Error;
    if not Str2Int(ExtractWord(2,Data.PAR,ScanSpaces),ItemCount) then goto Error;
    if ItemCount<1 then goto Error;
    while (ItemCount>0) do begin
     inc(LineNum);
     dec(ItemCount);
     Line:=Pad(TrimChars(myLink.Device.DialogText[LineNum],JustBlanks,JustBlanks),3);
     aComboBox.Items.Add(Line);
    end;
    LinkTag(aComboBox,aTag);
    LinkLabel(aComboBox);
    goto NextLine;
   end;
   {
   LISTBOX left top right bottom tag count
    item 1
    ...
    item count
   }
   if aDialog.Ok then
   if SameText(Data.ID,'LISTBOX') or SameText(Data.ID,'RADIOBUTTONS') then begin
    aListBox:=TListBox.Create(Self);
    aListBox.Parent:=Self;
    Locate(aListBox,Data.R);
    aTag:=findtag(ExtractWord(1,Data.PAR,ScanSpaces));
    if (typetag(aTag)<>1) then goto Error;
    if not Str2Int(ExtractWord(2,Data.PAR,ScanSpaces),ItemCount) then goto Error;
    if (ItemCount<1) then goto Error;
    while (ItemCount>0) do begin
     inc(LineNum);
     dec(ItemCount);
     Line:=Pad(TrimChars(myLink.Device.DialogText[LineNum],JustBlanks,JustBlanks),3);
     aListBox.Items.Add(Line);
    end;
    LinkTag(aListBox,aTag);
    LinkLabel(aListBox);
    goto NextLine;
   end;
   {}
  Error:
   myLink.Device.FixError(ecDialog);
   if myLink.Device.DebugMode
   then DebugOut(stdfDebug,'Device: '+myLink.Device.Name+' -> Error in line "'+Line+'"');
  NextLine:
   inc(LineNum);
  end;
  {}
  aMargins.x:=EnsureRange(aOrigins.x,mx1,mx2);
  aMargins.y:=EnsureRange(aOrigins.y-BitBtnOk.BoundsRect.Bottom+dy2,my1,my2);
  aMargins.y:=Min(aMargins.x,aMargins.y);
  Width:=aBoundsRect.b.x+aMargins.x; Height:=aBoundsRect.b.y+aMargins.y;
  {}
  Transfer(false);
 end;
 if Ok then
 if (aDialog=nil) then begin
  myLink.Device.FixError(ecDialog);
  if myLink.Device.DebugMode
  then DebugOut(stdfDebug,'Device: '+myLink.Device.Name+' -> Invalid dialog section');
 end;
end;

procedure TFormDaqDialogDevice.LabelClick(Sender: TObject);
begin
 if Ok then begin
  if (Sender is TLabel) then
  if (TLabel(Sender).FocusControl<>nil) then
  ActiveControl:=TLabel(Sender).FocusControl;
 end;
end;

procedure TFormDaqDialogDevice.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 LocateFormToCenterOfScreen(Self);
 myLink.Device:=nil;
 myLink.Count:=0;
end;

procedure TFormDaqDialogDevice.BitBtnOkClick(Sender: TObject);
begin
 if Ok then begin
  if Transfer(true) then mrVoice(mrOk) else mrVoice(mrError);
  isettag(myLink.Device.NotifyTag,1);
  if myLink.Device.CloseOnEnter then Close;
 end;
end;

procedure TFormDaqDialogDevice.BitBtnCancelClick(Sender: TObject);
begin
 if Ok then begin
  if Transfer(false) then mrVoice(mrCancel) else mrVoice(mrError);
  isettag(myLink.Device.NotifyTag,2);
  if myLink.Device.CloseOnEnter then Close;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqdialogdevice;
begin
end;

procedure Free_form_daqdialogdevice;
begin
end;

initialization

 Init_form_daqdialogdevice;

finalization

 Free_form_daqdialogdevice;

end.

//////////////
// END OF FILE
//////////////

