////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// ***********************************************************************    //
// *** Библиотека чтения файлов данных *.dat                           ***    //
// *** Файлы имеют имя PPYYMMDD.DAT                                    ***    //
// *** где PP - префикс имени файла                                    ***    //
// ***     YY - год (по модулю 100)                                    ***    //
// ***     MM - месяц                                                  ***    //
// ***     DD - день                                                   ***    //
// ***********************************************************************    //
// *** Старый формат:                                                  ***    //
// *** --------------                                                  ***    //
// *** В начале файла первой строкой идет заголовок                    ***    //
// ***  Time_mS C1 C2 C3                                               ***    //
// *** где Time_mS     - имя времени (в глобальных миллисекундах)      ***    //
// ***     C1,C2,C3... - имена сохраненных кривых                      ***    //
// *** далее в файле идут строки с данными, разделенными пробелами     ***    //
// *** в первом слове идет глобальное время в миллисекундах            ***    //
// *** в следующих словах строки - данные из кривых (по порядку).      ***    //
// *** Пример файла:                                                   ***    //
// ***  Time_mS AK_TDIOD AK_T1 AK_T2 AK_T3 AK_T4 AK_T5                 ***    //
// ***  63110275202137 290.56 23.5 23.5 21 24.5 24.5                   ***    //
// ***  63110275207190 290.56 23.5 23.5 21 24.5 24.5                   ***    //
// ***  63110275212243 290.56 23.5 23.5 21 24.5 24.5                   ***    //
// ***********************************************************************    //
// *** Новый формат:                                                   ***    //
// *** -------------                                                   ***    //
// *** Секционная структура. Секция выделяется скобками [ИмяСекции]    ***    //
// *** В начале файла идет секция [CRW-DAQ DATA FILE], куда записан    ***    //
// *** список кривых CurveList = C1,C2,C3...                           ***    //
// *** Далее, чередуясь, идут секции кривых [ИмяКривой], куда записаны ***    //
// *** данные. Первой строкой секции идет @Format=XXX, где XXX-формат. ***    //
// *** Допустимые фрорматы XY:ASCII,XY:HEX,XY:BASE64.                  ***    //
// *** Далее идут данные. Для ASCII это таблица десятичных чисел X,Y.  ***    //
// *** Для HEX это шестнадцатеричный дамп X,Y, для BASE64 это MIME.    ***    //
// *** В качестве X записано глобальное время в миллисекундах.         ***    //
// *** Пример файла:                                                   ***    //
// ***  [CRW-DAQ DATA FILE]                                            ***    //
// ***  CurveList=QMS_CH0,QMS_CH1                                      ***    //
// ***  [QMS_CH0]                                                      ***    //
// ***  @Format=XY:BASE64                                              ***    //
// ***  AHidcXTEzEKt6la7k0CFP4CRnXF0xMxC3hSxwAdFhT+Aqp1xdMTMQqWYS2     ***    //
// ***  [QMS_CH1]                                                      ***    //
// ***  @Format=XY:BASE64                                              ***    //
// ***  AHidcXTEzEKt6la7k0CVP4CRnXF0xMxC3hSxwAdFlT+Aqp1xdMTMQqWYS2     ***    //
// ***********************************************************************    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20201123 - Created by A.K.                                                 //
// 20230501 - Modified for FPC (A.K.)                                         //
// 20250129 - Use TAtomicCounter                                              //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqreaddat; // CRW DAQ Read *.dat

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons,
 lcltype, lclintf,
 Form_CurveWindow,
 Form_DaqReadDat,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_daqsys, _crw_curves, _crw_base64,
 _crw_crypt, _crw_daqevnt,
 _crw_appforms, _crw_apptools, _crw_apputils;

procedure ReadDatFiles(const FileList:LongString; const Params:LongString='');

type
 EReadDatFiles = class(ESoftException);

implementation

const
 ReadDatEntry:TAtomicCounter=nil;

procedure InitDatCounters;
begin
 LockedInit(ReadDatEntry);
end;

procedure FreeDatCounters;
begin
 LockedFree(ReadDatEntry);
end;

procedure ReadDatFiles(const FileList:LongString; const Params:LongString='');
const
 cStyle    = $1F;
 cStep     = 1024*32;
 ColTab    : array[0..9] of Integer = (Black, Blue, LightBlue, LightCyan, LightGreen,
                                    LightMagenta, LightRed, DarkGray, Yellow, White);
 AbsTolX   : Double  = 0;
 RelTolX   : Double  = 1E-10;
 AbsTolY   : Double  = 0;
 RelTolY   : Double  = 0;
 PackMode  : Integer = 7;
var
 f         : Text;
 t,y       : Double;
 i,j,k,IOR : Integer;
 Win       : TFormCurveWindow;
 s,s1      : LongString;
 FName     : LongString;
 Prefix    : LongString;
 Header    : LongString;
 FBuff     : LongString;
 year      : Integer;
 month     : Integer;
 day       : Integer;
 NumC      : Integer;
 CNum      : packed array[0..63] of Integer;
 Files     : TText;
 CList     : TText;
 SortFlags : Integer;
 TimeBase  : Double;
 TimeUnits : Double;
 FileKind  : (fkUnknown,fkOld,fkModern);
 DataForm  : (dfUnknown,dfAscii,dfHex,dfBase64);
 CurrCurve : TCurve;
 bugTime   : Integer;
 bugData   : Integer;
 bugForm   : Integer;
 LastProc  : Double;
 StartTime : Double;
 NPoints   : Integer;
 Packer    : TDaqCompressor;
 procedure SortAndPackText(aText:TText);
 var i:Integer; List:TStringList; s:LongString;
 begin
  if (aText.Count>0) then begin
   List:=TStringList.Create;
   try
    for i:=0 to aText.Count-1 do begin
     s:=UpcaseStr(Trim(aText[i]));
     if (List.IndexOf(s)<0) then List.Add(s);
    end;
    List.Sort;
    aText.Text:=List.Text;
   finally
    List.Free;
   end;
  end;
 end;
 procedure ProcessMessages;
 begin
  if (msecnow-LastProc>100) then begin
   SafeApplicationProcessMessages;
   LastProc:=msecnow;
  end;
 end;
 procedure AddPoint(c:TCurve; t,y:Double);
 var x:Double; pLast,pPrev:TPoint2D;
 begin
  if c.Ok then
  if IsNanOrInf(t) then Inc(bugTime) else
  if IsNanOrInf(y) then Inc(bugData) else begin
   with c do if (Count=Capacity) then Capacity:=Capacity*2;
   if IsNanOrInf(TimeBase) then TimeBase:=t;
   x:=(t-TimeBase)/TimeUnits;
   if HasFlags(PackMode,4) then y:=Packer.Compress(y);
   if (c.Count<2) or not HasFlags(PackMode,4) then c.AddPoint(x,y) else begin
    pLast:=c[c.Count-1];
    if (y<>pLast.y) then c.AddPoint(x,y) else begin
     pPrev:=c[c.Count-2];
     if (pPrev.y<>y) then c.AddPoint(x,y) else begin
      if (x<=pLast.x) then c.AddPoint(x,y)else begin
       pLast.x:=x;
       c[c.Count-1]:=pLast;
      end;
     end;
    end;
   end;
   Inc(NPoints);
  end;
 end;
 procedure FatalBug(const Msg:LongString);
 begin
  Kill(Win);
  RAISE EReadDatFiles.Create(Msg);
 end;
begin
 if Length(FileList)>0 then
 try
  Win:=nil;
  CList:=nil;
  Files:=nil;
  Packer:=nil;
  AssignNull(f);
  IOR:=IOResult;
  FBuff:=''; FName:='';
  try
   if (LockedInc(ReadDatEntry)>1) then FatalBug('Recursion is not allowed!');
   FBuff:=StringBuffer(MaxTextBufSize);
   NumC:=0;
   Prefix:='';
   bugTime:=0;
   bugData:=0;
   bugForm:=0;
   NPoints:=0;
   TimeBase:=0;
   TimeUnits:=1e3;
   CList:=NewText;
   Files:=NewText;
   Files.Text:=FileList;
   SortAndPackText(Files);
   SafeFillChar(CNum,SizeOf(CNum),0);
   Packer:=NewDaqCompressor(AbsTolY,RelTolY);
   if Files.Count=0 then FatalBug(RusEng('Список файлов пуст!','Empty file list!'));
   s:=ExtractFileNameExt(Files[0])+'...'+ExtractFileNameExt(Files[Files.Count-1]);
   Win:=NewCurveWindow(RusEng('Данные из ','Data from ')+s,'','',wsMinimized);
   LastProc:=msecnow;
   StartTime:=msecnow;
   for i:=0 to Files.Count-1 do begin
    {
    Check file name
    }
    FName:=UnifyFileAlias(Files[i],ua_FileDefLow);
    if SystemConsole.Form.Ok then begin
     Echo(StdDateTimePrompt+RusEng('Чтение файла ','Read file ')+FName);
     ExecuteSystemConsoleMonitoring(true);
     SystemConsole.Activate;
    end;
    if not FileExists(FName)
    then FatalBug(RusEng('Не найден файл ','Not found file ')+FName);
    {
    Extract date from file name - 6 last chars
    }
    if i=0 then begin
     Prefix:=ExtractFileName(FName);
     Prefix:=Copy(Prefix,1,Length(Prefix)-6);
    end;
    if not IsSameText(Prefix,Copy(ExtractFileName(FName),1,Length(Prefix)))
    then FatalBug(RusEng('Недопустимый префикс в ','Invalid prefix in ')+FName);
    if not Str2Int(Copy(ExtractFileName(FName),Length(Prefix)+1,2),year)
    then FatalBug(RusEng('Недопустимый год в ','Invalid year in ')+FName);
    if not Str2Int(Copy(ExtractFileName(FName),Length(Prefix)+3,2),month)
    then FatalBug(RusEng('Недопустимый месяц в ','Invalid month in ')+FName);
    if not Str2Int(Copy(ExtractFileName(FName),Length(Prefix)+5,2),day)
    then FatalBug(RusEng('Недопустимый день в ','Invalid day in ')+FName);
    year:=year+2000;
    t:=DateTimeToMSec(year,month,day,0,0,0,0);
    if IsNanOrInf(t) then FatalBug(RusEng('Недопустимая дата в ','Invalid date in ')+FName);
    if i=0 then TimeBase:=t;
    {
    Read file
    }
    System.Assign(f,FName);
    System.Reset(f);
    if (Length(FBuff)>0) then SetTextBuf(f,PChar(FBuff)^,Length(FBuff));
    System.Readln(f,s);
    if (IOResult<>0) then FatalBug(RusEng('Не могу открыть ','Could not open ')+FName);
    {
    Check file header
    }
    FileKind:=fkUnknown;
    if IsSameText(Trim(s),'[CRW-DAQ DATA FILE]') then FileKind:=fkModern else
    if IsSameText(ExtractWord(1,s,ScanSpaces),'TIME_MS') then FileKind:=fkOld;
    if (FileKind=fkUnknown)
    then FatalBug(RusEng('Неизвестный заголовок в ','Invalid header in ')+FName);
    {
    If first file, create curve list
    }
    if NumC=0 then begin
     Header:=s;
     CList.Count:=0;
     case FileKind of
      fkOld:
       for j:=2 to WordCount(Header,ScanSpaces) do
        CList.Addln(ExtractWord(j,Header,ScanSpaces));
      fkModern:
       while not eof(f) do begin
        System.Readln(f,s1);
        if (IoResult<>0) then FatalBug(RusEng('Ошибка чтения ','Error read ')+FName);
        s1:=Trim(s1);
        if (StrFetch(s1,1)='[') then begin
         SmartFileClose(f);
         Reset(f);
         if (IoResult<>0) then FatalBug(RusEng('Ошибка чтения ','Error read ')+FName);
         Break;
        end;
        if IsSameText(ExtractWord(1,s1,ScanSpaces),'CurveList') then
        for j:=2 to WordCount(s1,ScanSpaces) do CList.Addln(ExtractWord(j,s1,ScanSpaces));
       end;
     end;
     SmartFileClose(f);
     if FormDaqReadDatExecute(TimeUnits,TimeBase,PackMode,
                              AbsTolX,RelTolX,AbsTolY,RelTolY,CList,Params)=mrOk
     then begin
      Packer.AbsTol:=AbsTolY;
      Packer.RelTol:=RelTolY;
      case FileKind of
       fkOld:
        for j:=0 to CList.Count-1 do begin
         k:=WordIndex(CList[j],Header,ScanSpaces);
         if (k>0) and (NumC<=High(CNum)) then begin
          Win.AddCurve(NewCurve(0,CList[j],CgaToRGBColor(ColTab[NumC mod (High(ColTab)+1)]),
                                cStyle,0,cStep));
          CNum[NumC]:=k;
          inc(NumC);
         end;
        end;
       fkModern:
        for j:=0 to CList.Count-1 do begin
         Win.AddCurve(NewCurve(0,CList[j],CgaToRGBColor(ColTab[NumC mod (High(ColTab)+1)]),
                               cStyle,0,cStep));
         inc(NumC);
        end;
      end;
     end else FatalBug(RusEng('Пользователь прервал!','User break!'));
     System.Reset(f);
     if (FileKind=fkOld) then System.Readln(f,s);
     if (IOResult<>0) then FatalBug(RusEng('Не могу открыть ','Could not open ')+FName);
     StartTime:=msecnow;
    end;
    {
    Check if curve list is Ok
    }
    if NumC=0
    then FatalBug(RusEng('Нет списка кривых в ','No curve list in ')+FName);
    if not IsSameText(Trim(s),Trim(Header))
    then FatalBug(RusEng('Плохой заголовок в ','Invalid header in ')+FName);
    {
    Now ready to read data
    }
    k:=0;
    CurrCurve:=nil;
    DataForm:=dfUnknown;
    while not System.Eof(f) do begin
     System.Readln(f,s);
     if IOResult<>0 then FatalBug(RusEng('Ошибка чтения ','Error read ')+FName);
     s:=Trim(s);
     if Length(s)>0 then
     case FileKind of
      fkOld:
       if not Str2Real(ExtractWord(1,s,ScanSpaces),t) then Inc(bugTime) else begin
        for j:=0 to NumC-1 do begin
         if Str2Real(ExtractWord(CNum[j],s,ScanSpaces),y)
         then AddPoint(Win.Curves[j],t,y)
         else Inc(bugData);
        end;
       end;
      fkModern:
       begin
        if (StrFetch(s,1)='[') then begin
         CurrCurve:=nil;
         if (StrFetch(s,Length(s))=']') then begin
          s:=Copy(s,2,Length(s)-2);
          for j:=0 to Win.Curves.Count-1 do
          if IsSameText(s,Win.Curves[j].Name) then begin
           CurrCurve:=Win.Curves[j];
           Break;
          end;
         end;
        end else
        if (StrFetch(s,1)='@') then begin
         if Assigned(CurrCurve) then
         if IsSameText(ExtractWord(1,s,ScanSpaces),'@Format') then begin
          s1:=ExtractWord(2,s,ScanSpaces);
          if IsSameText(s1,'XY:ASCII')  then DataForm:=dfAscii  else
          if IsSameText(s1,'XY:HEX')    then DataForm:=dfHex    else
          if IsSameText(s1,'XY:BASE64') then DataForm:=dfBase64 else
          DataForm:=dfUnknown;
         end;
        end else
        if Assigned(CurrCurve) then begin
         case DataForm of
          dfAscii:
           begin
            if not Str2Real(ExtractWord(1,s,ScanSpaces),t) then Inc(bugTime) else
            if not Str2Real(ExtractWord(2,s,ScanSpaces),y) then Inc(bugData) else
            AddPoint(CurrCurve,t,y);
           end;
          dfHex:
           begin
            s:=Hex_Decode(s);
            for j:=0 to (Length(s) div SizeOf(TPoint2D))-1 do begin
             t:=PPoint2DArray(@s[1])[j].x;
             y:=PPoint2DArray(@s[1])[j].y;
             AddPoint(CurrCurve,t,y);
            end;
           end;
          dfBase64:
           begin
            s:=Mime_Decode(s);
            for j:=0 to (Length(s) div SizeOf(TPoint2D))-1 do begin
             t:=PPoint2DArray(@s[1])[j].x;
             y:=PPoint2DArray(@s[1])[j].y;
             AddPoint(CurrCurve,t,y);
            end;
           end;
          else Inc(bugForm);
         end;
        end;
       end;
     end;
     if (k and $3FF = 0) then ProcessMessages;
     Inc(k);
    end;
    SmartFileClose(f);
    SendToMainConsole('@silent @integrity load.dat '+FName+EOL);
    if (bugTime+bugData+bugForm>0)
    then Echo(Format(RusEng('Найдено %d ошибок времени, %d ошибок данных, %d ошибок формата.',
                            'Found %d time errors, %d data errors, %d format errors.'),
                            [bugTime,bugData,bugForm]));
    bugTime:=0;
    bugData:=0;
    bugForm:=0;
    ProcessMessages;
   end;
   {
   Sort curves if one not sorted
   }
   SortFlags:=0;
   if HasFlags(PackMode,1) then SortFlags:=SortFlags+cfNotSortedX;
   if HasFlags(PackMode,2) then SortFlags:=SortFlags+cfDuplicatesX;
   if (SortFlags<>0) then
   for i:=0 to Win.Curves.Count-1 do
   if HasFlags(Win.Curves[i].Flags(AbsTolX,RelTolX),SortFlags) then begin
    Echo(RusEng('Сортировка кривой ','Sort curve ')+Win.Curves[i].Name);
    ProcessMessages;
    CurrCurve:=NewCurveSorted(Win.Curves[i],sfPack,AbsTolX,RelTolX);
    try Win.Curves[i].Assign(CurrCurve); finally Kill(CurrCurve); end;
   end;
   {
   Report
   }
   Echo(Format(RusEng('Прочли кривых:%d, точек:%d, время обработки:%1.0f ms',
                      'Loaded curves:%d, points:%d, processing time:%1.0f ms'),
                      [Win.Curves.Count,NPoints,msecnow-StartTime]));
   k:=0;
   for i:=0 to Win.Curves.Count-1 do Inc(k,Win.Curves[i].Count);
   if k<NPoints then
   Echo(Format(RusEng('Данные сжаты на %5.1f %%',
                      'Data was packed by %5.1f %%'),
                      [100.0*(NPoints-k)/NPoints]));
   {
   Set caption, title, legend etc.
   }
   if (Win.Curves.Count>0) then begin
    Win.WindowState:=wsNormal;
    Win.LockDraw;
    try
     Win.DefCurveNum:=0;
     Win.AutoRange;
     s:='';
     for i:=0 to Win.Curves.Count-1 do begin
      if (Length(s)>60) and (i<Win.Curves.Count-1) then begin
       s:=s+'...'+Win.Curves[Win.Curves.Count-1].Name;
       Break;
      end;
      if (Length(s)>0) then s:=s+',';
      s:=s+Win.Curves[i].Name;
     end;
     Win.Title:=^C+RusEng('Кривые ','Curves ')+s+ASCII_CR+^L' {??}';
     s:= '?';
     if TimeUnits=1 then s:=RusEng('МиллиСекунд','MilliSeconds');;
     if TimeUnits=1000 then s:=RusEng('Секунд','Seconds');
     if TimeUnits=60*1000 then s:=RusEng('Минут','Minutes');
     if TimeUnits=60*60*1000 then s:=RusEng('Часов','Hours');
     if TimeUnits=24*60*60*1000 then s:=RusEng('Суток','Days');
     Win.Legend:=^R'{'+s+'}  '+ASCII_CR+^C+RusEng('Время отсчитывается от ','Time since ')+StdDateTimeStr(TimeBase);
     Win.TimeBase:=TimeBase;
     Win.TimeUnits:=TimeUnits;
    finally
     Win.UnlockDraw;
    end;
   end;
  finally
   LockedDec(ReadDatEntry);
   Kill(Files);
   Kill(CList);
   Kill(Packer);
   SmartFileClose(f);
   SetInOutRes(IOR);
   if (Win.Curves.Count=0) then Kill(Win);
   FBuff:=''; FName:='';
  end;
 except
  on E:Exception do BugReport(E,nil,'ReadDatFiles');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqreaddat;
begin
 InitDatCounters;
end;

procedure Free_crw_daqreaddat;
begin
 FreeDatCounters;
end;

initialization

 Init_crw_daqreaddat;

finalization

 Free_crw_daqreaddat;

end.

//////////////
// END OF FILE
//////////////

