##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
# opcuasrv.py - opcua server to publish data via OPCUA.
##########################################################

'''
@help
 @Help             -  print this help
 @Exit n           -  exit with code n
 @PingEcho s       -  print echo (for testing)
 @PollCount        -  print polling loop counter
 @PollPeriod n     -  set polling loop period n ms
 @TermOnError n    -  terminate on error flag (0/1)
 @Memory           -  get process memory rss,vms,gc
 @OpcServName n    -  get/set server name (n), like OpcuaSrv
 @OpcEndpoint u    -  get/set server endpoint URL (u), like opc.tcp://0.0.0.0:4840/daqgroup/crwdaq/server
 @OpcServAddr u    -  get/set server endpoint URL (u), like opc.tcp://0.0.0.0:4840/daqgroup/crwdaq/server
 @OpcNamespace n   -  get/set server namespace (n), like http//freeopcua/defaults/modeler
 @OpcNspArray      -  get list of namespaces available on server
 @OpcNspIndex      -  get index of server namespace in list of namespaces
 @OpcXmlModel x    -  get/set server XML model (x), like /mnt/data/home/alex/projects/sandbox/opcua/opcua_test/opcuasrv.xml
 @OpcWritable n    -  make client's writable node (n), like ns=2;i=13 or ns2i13
 @OpcDataBook n    -  subscribe data change of node (n), like ns=2;i=13 or ns2i13
 @UseShortKey f    -  flag (0/1) to use short key (ns2i13 instead of ns=2;i=13)
 @OpcSubInter p    -  subscription interval (period) ms, default is 100
 @OpcStartSrv      -  start OPCUA server to publish nodes data
 @OpcDataVary n v  -  notification on node (n) changed to value (v)
 @OpcGotEvent e    -  notification on got event (e)
 @OpcDataSend n v  -  write node (n) value (v)
 @OpcDataRead n    -  read node (n) value
'''

# pylint: disable=invalid-name,superfluous-parens,consider-using-f-string
# pylint: disable=multiple-imports,unused-import,wildcard-import,unused-wildcard-import
# pylint: disable=missing-module-docstring,missing-function-docstring,missing-class-docstring
# pylint: disable=unused-argument,broad-exception-caught,broad-exception-raised,duplicate-code
# pylint: disable=too-many-public-methods,too-many-instance-attributes,too-many-arguments,line-too-long

import sys, os, time, math, asyncio, logging, argparse, re, random
from asyncua.common.subscription import Subscription
from asyncua.common.ua_utils import string_to_val
from asyncua.common.ua_utils import val_to_string
from asyncua import Server, Node, ua, uamethod
from datetime import datetime
from pycrwkit import *

###################################
# python script self identification
###################################
scriptfile = __file__                     # python script file name
scriptname = ExtractFileName(scriptfile)  # python script base name with extension
scriptbase = ExtractBaseName(scriptname)  # python script base name without extension
scriptFILE = RealPath(scriptfile)         # absolute dereferenced script file name
scriptHOME = ExtractDirName(scriptFILE)   # absolute dereferenced script directory

##################
# Common Constants
##################
EOL = os.linesep            # End Of Line separator
defSubInter = 100           # Default subscription interval (period) ms

###################################################
# Logging: NOTSET,DEBUG,INFO.WARNING,ERROR,CRITICAL
###################################################
logging.basicConfig(level=logging.WARNING)
_logger = logging.getLogger(scriptbase)

###################################################
# Default server settings
###################################################
defServerName      = "OpcuaSrv"
defServerNamespace = "http//freeopcua/defaults/modeler"
defServerXmlModel  = os.path.join(scriptHOME,scriptbase+".xml")
defServerEndpoint  = "opc.tcp://0.0.0.0:4840/daqgroup/crwdaq/server"
defServerSecurity  = [ ua.SecurityPolicyType.NoSecurity,
                       ua.SecurityPolicyType.Basic256Sha256_SignAndEncrypt,
                       ua.SecurityPolicyType.Basic256Sha256_Sign, ]
###################################################

###########################
# get OPC UA node reference
# with namespace index (ns)
# with local node index (i)
###########################
def opcuaNodeRef(ns,i):
    return "ns={0};i={1}".format(ns,i)

def varFloat(val):
    return ua.Variant(float(val), ua.VariantType.Float)
def varDouble(val):
    return ua.Variant(float(val), ua.VariantType.Double)
def varInt32(val):
    return ua.Variant(int(val), ua.VariantType.Int32)
def varInt64(val):
    return ua.Variant(int(val), ua.VariantType.Int64)

##########################################################################
# OPC UA data nodes can be identified by nodeId, a string like ns=2;i=1234
# or shortened string key like ns2i1234. useShortKey flag uses as selector
# to use full (ns=2;i=1234) or short (ns2i1234) key to identify data nodes
##########################################################################
useShortKey = 0                             # flag (0/1) to translate keys
transKeyDic = {ord(i): None for i in '=;'}  # dictionary to translate keys
##########################################################################

##########################
# get shortened nodeId key
# ns=2;i=1234 --> ns2i1234
##########################
def getShortNodeId(node):
    return str(node).translate(transKeyDic)

##############################
# get full or short nodeId key
##############################
def getKeyByNodeId(node):
    if (useShortKey > 0):
        return str(node).translate(transKeyDic)
    return str(node)

######################################
# check given string (s) is looks like
# environment variable $x, ${x} or %x%
######################################
def LooksLikeEnvVar(s):
    if isinstance(s,str):
        if ('$' in s):
            return True
        if ('%' in s) and SameText(os.name,'nt'):
            return True
    return False

##########################################################################
# Subscription Handler. To receive events from server for a subscription  
# data_change and event methods are called directly from receiving thread.
# Do not do expensive, slow or network operation there. Create another    
# thread if you need to do such a thing.                                  
##########################################################################
class SubscriptionHandler(object):
    #########################################
    # Notification on server node data change
    #########################################
    def datachange_notification(self, node, val, data):
        if isinstance(val,bool):
            val = int(val)
        print("@OpcDataVary",getKeyByNodeId(node),val)
    ##############################
    # Notification on server event
    ##############################
    def event_notification(self, event):
        print("@OpcGotEvent",event)

##########################################################
# Override DaqApplication class for Opcua Srv Application:
##########################################################
class OpcuaSrvApplication(DaqApplication):
    ###############################
    # Override Greetings on startup
    # Initialize application fields
    ###############################
    def Greetengs(self):
        super().Greetengs() # call inherited
        print("Welcome to %s program." % self.Title)
        self.subHandler = SubscriptionHandler()         # data change handler
        self.ServerStarted = False                      # flag of serving
        self.ServerReadyIO = False                      # flag ready for IO
        self.subDatChNodes = []                         # subscried Node objects
        self.uaServer = None                            # OPCUA server instance
        self.bookDatCh = []                             # list of subscriptions
        self.srvrNodes = {}                             # to get Node object by nodeId
        self.dispNodes = {}                             # to get displayName by nodeId
        self.pathNodes = {}                             # to get Node path   by nodeId
        self.typeNodes = {}                             # to get Node type   by nodeId
        self.rwioNodes = []                             # list of pending read/write
        self.wrblNodes = []                             # list of writable nodes
        self.delayedIO = []                             # list of delayed IO
        self.serverName = defServerName                 # server name
        self.serverXmlModel = defServerXmlModel         # server model file.xml
        self.serverEndpoint = defServerEndpoint         # server connection endpoint
        self.serverNamespace = defServerNamespace       # server namespace
        self.serverNspArray = None                      # server list of namespaces
        self.serverNspIndex = -1                        # server index of namespace
        self.subInter = defSubInter                     # subscription interval ms
        self.ParseParams()

    ###############################
    # Parse command line parameters
    ###############################
    def ParseParams(self):
        self.parser = argparse.ArgumentParser(prog=self.Title,
                      description='OPC UA Server test application.',
                      epilog='OPCUA server to publish data.')
        self.parser.add_argument('-u', '--url')
        self.parser.add_argument('-x', '--xml')
        self.params = self.parser.parse_args()
        if isinstance(self.params.url,str):
            self.serverEndpoint = self.params.url.strip()
            print("Option --url",self.serverEndpoint)
        if isinstance(self.params.xml,str):
            self.serverXmlModel = self.validateXmlPath(self.params.xml)
            print("Option --xml",self.serverXmlModel)

    ##################
    # Stopping actions
    ##################
    def Goodbye(self):
        if self.isatty[1]:
            print("Program terminated. Press Enter to exit.")
        return super().Goodbye()

    ########################
    # validate xml file name
    ########################
    def validateXmlPath(self,xml):
        xml = xml.strip()
        if LooksLikeEnvVar(xml):
            xml = os.path.expandvars(xml)
        if SameText(ExtractFileExt(xml),""):
            xml = xml + ".xml"
        if not os.path.isabs(xml):
            xml = os.path.abspath(xml)
        xml = os.path.normcase(xml)
        xml = os.path.normpath(xml)
        xml = os.path.realpath(xml)
        return xml.strip()

    ###########################
    # get server node by nodeId
    # uses dictionary for speed
    # nodeId expected in format
    # like: ns=2;i=12 or ns2i12
    ###########################
    def get_server_node(self,nodeId):
        node = self.srvrNodes.get(nodeId)
        if isinstance(node,Node):
            return node
        if not self.isValidNodeId(nodeId):
            nodeId = self.validateNodeId(nodeId)
            if not self.isValidNodeId(nodeId):
                return None
        if self.uaServer:
            node = self.uaServer.get_node(nodeId)
            if isinstance(node,Node):
                self.srvrNodes[nodeId] = node
                short = getShortNodeId(nodeId)
                self.srvrNodes[short] = node
        node = self.srvrNodes.get(nodeId)
        if isinstance(node,Node):
            return node
        return None

    ###################################
    # check string (s) is valid node id
    # match pattern: ^ns=(\d+);(i=\d+)$
    ###################################
    def isValidNodeId(self,s):
        m = re.match(r"^ns=(\d+);(i=\d+|s=\S+|g=\S+)$",s,re.IGNORECASE)
        return not (m is None)

    #####################################
    # check string (s) is valid short key
    # which match pattern ^ns(\d+)(i\d+)$
    #####################################
    def isValidShortKey(self,s):
        m = re.match(r"^ns(\d+)(i\d+|s\S+|g\S+)$",s,re.IGNORECASE)
        return not (m is None)

    ####################################
    # check string (s) is valid node key
    ####################################
    def isValidNodeKey(self,s):
        return self.isValidShortKey(s) or self.isValidNodeId(s)

    #######################################
    # validate node id: ns2i13 -> ns=2;i=13
    #######################################
    def validateNodeId(self,s):
        m = re.match(r"^ns(\d+)(i\d+|s\S+|g\S+)$",s,re.IGNORECASE)
        if m:
            return "ns={0};{1}={2}".format(m.group(1),m.group(2)[0],m.group(2)[1:])
        return s

    ##############################################################
    # convert string (s) to value of specified ua variant type (t)
    ##############################################################
    def str_to_val_type(self,s,t):
        if isinstance(s,str) and isinstance(t,int):
            return string_to_val(s.strip(),t)
        return s

    ##############################################################
    # convert value (v) of specified ua variant type (t) to string
    ##############################################################
    def val_to_str_type(self,v,t):
        if isinstance(t,int) and not (v is None):
            return val_to_string(v,t)
        return v

    #######################################
    # Check server can assign arg parameter
    #######################################
    def canAssignServerParam(self,cmd,arg):
        if isinstance(arg,str) and (len(arg.strip())>0):
            if self.ServerStarted:
                print("@Warning","denied after start:",cmd,arg)
                return False
            return True
        return False

    ##################################
    # Handle command @OpcServName name
    ##################################
    def DoOpcServName(self,cmd,arg):
        if not SameText(cmd,"@OpcServName"):
            return False
        if self.canAssignServerParam(cmd,arg):
            self.serverName = arg.strip()
        print("@OpcServName",self.serverName)
        return True

    #################################
    # Handle command @OpcEndpoint url
    #################################
    def DoOpcEndpoint(self,cmd,arg):
        if not SameText(cmd,"@OpcEndpoint"):
            return False
        if self.canAssignServerParam(cmd,arg):
            self.serverEndpoint = arg.strip()
        print("@OpcEndpoint",self.serverEndpoint)
        return True

    #################################
    # Handle command @OpcServAddr url
    #################################
    def DoOpcServAddr(self,cmd,arg):
        if not SameText(cmd,"@OpcServAddr"):
            return False
        if isinstance(arg,str) and (len(arg)>0):
            self.serverEndpoint = arg.strip()
        print("@OpcServAddr",self.serverEndpoint)
        return True

    ################################
    # Handle command @OpcNamespace n
    ################################
    def DoOpcNamespace(self,cmd,arg):
        if not SameText(cmd,"@OpcNamespace"):
            return False
        if self.canAssignServerParam(cmd,arg):
            self.serverNamespace = arg.strip()
        print("@OpcNamespace",self.serverNamespace)
        return True

    #################################
    # Handle command @OpcXmlModel xml
    #################################
    def DoOpcXmlModel(self,cmd,arg):
        if not SameText(cmd,"@OpcXmlModel"):
            return False
        if self.canAssignServerParam(cmd,arg):
            self.serverXmlModel = self.validateXmlPath(arg)
        print("@OpcXmlModel",self.serverXmlModel)
        return True

    #############################
    # Handle command @OpcNspArray
    #############################
    def DoOpcNspArray(self,cmd,arg):
        if not SameText(cmd,"@OpcNspArray"):
            return False
        if not self.ServerReadyIO:
            self.delayedIO.append(cmd+" "+arg)
            return True
        if self.serverNspArray is None:
            print("Server is not started.")
            return False
        print("@OpcNspArray",", ".join(self.serverNspArray))
        return True

    #############################
    # Handle command @OpcNspIndex
    #############################
    def DoOpcNspIndex(self,cmd,arg):
        if not SameText(cmd,"@OpcNspIndex"):
            return False
        if not self.ServerReadyIO:
            self.delayedIO.append(cmd+" "+arg)
            return True
        print("@OpcNspIndex",self.serverNspIndex)
        return True

    ####################################
    # Handle command @OpcWritable nodeId
    ####################################
    def DoOpcWritable(self,cmd,arg):
        if not SameText(cmd,"@OpcWritable"):
            return False
        for item in arg.split():
            nodeId = self.validateNodeId(item)
            if self.isValidNodeId(nodeId):
                if (nodeId in self.wrblNodes):
                    print("Warning: already writable",nodeId)
                    return False
                self.wrblNodes.append(nodeId)
                print("@OpcWritable",nodeId)
                return True
            else:
                print("Error: invalid nodeId",nodeId)
                return False
        return True

    ################################
    # Handle command @OpcDataBook id
    ################################
    def DoOpcDataBook(self,cmd,arg):
        if not SameText(cmd,"@OpcDataBook"):
            return False
        if self.ServerStarted:
            print("Error: could not book data after @OpcStartSrv")
            return False
        for item in arg.split():
            nodeId = self.validateNodeId(item)
            if self.isValidNodeId(nodeId):
                if (nodeId in self.bookDatCh):
                    print("Error: already booked",nodeId)
                    return False
                self.bookDatCh.append(nodeId)
                print("@OpcDataBook",nodeId)
                return True
            else:
                print("Error: invalid nodeId",nodeId)
                return False
        return True

    ####################################
    # Handle command @OpcDataRead nodeId
    ####################################
    def DoOpcDataRead(self,cmd,arg):
        if not SameText(cmd,"@OpcDataRead"):
            return False
        if not self.ServerReadyIO:
            self.delayedIO.append(cmd+" "+arg)
            return True
        nodeId = arg.strip()
        if not self.isValidNodeId(nodeId):
            nodeId = self.validateNodeId(nodeId)
        node = self.get_server_node(nodeId)
        if isinstance(node,Node):
            tup = (nodeId,node)
            self.rwioNodes.append(tup)
        return True

    ##########################################
    # Handle command @OpcDataSend nodeId Value
    ##########################################
    def DoOpcDataSend(self,cmd,arg):
        if not SameText(cmd,"@OpcDataSend"):
            return False
        if not self.ServerReadyIO:
            self.delayedIO.append(cmd+" "+arg)
            return True
        spl = arg.split(None,1)
        while (len(spl) < 2):
            spl.append("")
        nodeId = spl[0]
        value = spl[1].strip()
        if not self.isValidNodeId(nodeId):
            nodeId = self.validateNodeId(nodeId)
        node = self.get_server_node(nodeId)
        if isinstance(node,Node):
            tup = (nodeId,node,value)
            self.rwioNodes.append(tup)
        return True

    ###############################
    # Handle command @UseShortKey f
    ###############################
    def DoUseShortKey(self,cmd,arg):
        if not SameText(cmd,"@UseShortKey"):
            return False
        global useShortKey
        useShortKey = StrToIntDef(arg,useShortKey)
        print("@UseShortKey",useShortKey)
        return True

    ################################
    # Handle command @OpcSubInter p
    ################################
    def DoOpcSubInter(self,cmd,arg):
        if not SameText(cmd,"@OpcSubInter"):
            return False
        i = StrToInt(arg)
        if isinstance(i,int) and (i>0) and not self.ServerStarted:
            self.subInter = i
        print("@OpcSubInter",self.subInter)
        return True

    ##########################################
    # Handle command @OpcDataVary nodeId Value
    ##########################################
    def DoOpcDataVary(self,cmd,arg):
        if not SameText(cmd,"@OpcDataVary"):
            return False
        return True

    ###################################
    # Handle command @OpcGotEvent event
    ###################################
    def DoOpcGotEvent(self,cmd,arg):
        if not SameText(cmd,"@OpcGotEvent"):
            return False
        return True

    #############################
    # Handle command @OpcStartSrv
    #############################
    def DoOpcStartSrv(self,cmd,arg):
        if not SameText(cmd,"@OpcStartSrv"):
            return False
        self.ServerStarted = True
        print("@OpcStartSrv",int(self.ServerStarted))
        return True

    #############################
    # Handle command @OpcStartSrv
    #############################
    def ProcessDelayedIO(self):
        if (len(self.delayedIO) > 0):
            for line in self.delayedIO:
                self.StdInProcessing(line)
            self.delayedIO.clear()

async def StartServerApp(Application):
    # get server parameters
    endpoint= Application.serverEndpoint
    name = Application.serverName
    xml = Application.serverXmlModel
    # create server instance
    server = Server()
    Application.uaServer = server
    await server.init()
    server.set_server_name(name)
    server.set_endpoint(endpoint)
    # set all possible endpoint policies for clients to connect through
    server.set_security_policy(defServerSecurity)
    if not os.path.exists(xml):
        print("Error: not found",xml)
        Application.SysExit(1)
    await server.import_xml(xml)
    ###################################
    # detect namespace index for future
    ###################################
    Application.serverNspArray = await server.get_namespace_array()
    Application.serverNspIndex = await server.get_namespace_index(Application.serverNamespace)
    print("@OpcNspArray",", ".join(Application.serverNspArray))
    print("@OpcNspIndex",Application.serverNspIndex)


##############################
# async polling of application
##############################
async def AsyncPollApp(Application):
    Application.PollingLoopHandler()
    if Application.ServerStarted and Application.ServerReadyIO:
        queue = Application.rwioNodes
        while (len(queue) > 0):
            tup = queue.pop(0)
            try:
                if (len(tup) == 2):
                    nodeId = tup[0]
                    node = tup[1]
                    if isinstance(nodeId,str) and isinstance(node,Node):
                        val = await node.read_value()
                        key = getKeyByNodeId(nodeId)
                        if isinstance(val,bool):
                            val = int(val)
                        print("@OpcDataRead",key,val)
                if (len(tup) == 3):
                    nodeId = tup[0]
                    node = tup[1]
                    data = tup[2]
                    if isinstance(nodeId,str) and isinstance(node,Node):
                        typ = Application.typeNodes.get(nodeId)
                        if typ is None:
                            typ = await node.read_data_type_as_variant_type()
                            if typ:
                                Application.typeNodes[nodeId] = typ
                                shortId = getShortNodeId(nodeId)
                                Application.typeNodes[shortId] = typ
                        val = Application.str_to_val_type(data,typ)
                        await node.write_value(val,typ)
                        key = getKeyByNodeId(nodeId)
                        val = Application.val_to_str_type(val,typ)
                        print("@OpcDataSend",key,val)
            except Exception as E:
                if (Application.TermOnError > 0):
                    raise E
                Application.OnError(E)
    #Application.SleepAfterPolling()
    pass

########################
# report on subscription
########################
def reportSubscription(sub):
    if isinstance(sub,Subscription):
        if isinstance(sub._monitored_items,dict):
            n = len(sub._monitored_items)
            print("INFO: Subscription done for",n,"node(s).")

##########################################
# Create the DaqApplication object to run:
##########################################
Application = OpcuaSrvApplication(scriptbase,4)

######################
# Opcua Server Program
######################
async def execute_opcua_server():
    try:
        #####################
        # Initialize instance
        #####################
        Application.Welcome()
        ###########################################
        # Then add new application command handlers
        ###########################################
        Application.AddCommandHandler(Application.DoOpcServName,"@OpcServName n - get/set server name (n), like "+defServerName)
        Application.AddCommandHandler(Application.DoOpcEndpoint,"@OpcEndpoint u - get/set server endpoint URL (u), like "+defServerEndpoint)
        Application.AddCommandHandler(Application.DoOpcServAddr,"@OpcServAddr u - get/set server endpoint URL (u), like "+defServerEndpoint)
        Application.AddCommandHandler(Application.DoOpcNamespace,"@OpcNamespace n - get/set server namespace (n), like "+defServerNamespace)
        Application.AddCommandHandler(Application.DoOpcNspArray,"@OpcNspArray - get list of namespaces available on server")
        Application.AddCommandHandler(Application.DoOpcNspIndex,"@OpcNspIndex - get index of server namespace in list of namespaces")
        Application.AddCommandHandler(Application.DoOpcXmlModel,"@OpcXmlModel x - get/set server XML model (x), like "+defServerXmlModel)
        Application.AddCommandHandler(Application.DoOpcWritable,"@OpcWritable n - make client's writable node (n), like ns=2;i=13 or ns2i13")
        Application.AddCommandHandler(Application.DoOpcDataBook,"@OpcDataBook n - subscribe data change of node (n), like ns=2;i=13 or ns2i13")
        Application.AddCommandHandler(Application.DoUseShortKey,"@UseShortKey f - flag (0/1) to use short key (ns2i13 instead of ns=2;i=13)")
        Application.AddCommandHandler(Application.DoOpcSubInter,"@OpcSubInter p - subscription interval (period) ms, default is "+str(defSubInter))
        Application.AddCommandHandler(Application.DoOpcStartSrv,"@OpcStartSrv - start OPCUA server to publish nodes data")
        Application.AddCommandHandler(Application.DoOpcDataVary,"@OpcDataVary n v - notification on node (n) changed to value (v)")
        Application.AddCommandHandler(Application.DoOpcGotEvent,"@OpcGotEvent e - notification on got event (e)")
        Application.AddCommandHandler(Application.DoOpcDataSend,"@OpcDataSend n v - write node (n) value (v)")
        Application.AddCommandHandler(Application.DoOpcDataRead,"@OpcDataRead n - read node (n) value")
        ##############################
        # Application configuring loop
        ##############################
        while not Application.ServerStarted and not Application.Terminated:
            delay = 0.001*Application.PollPeriod
            await AsyncPollApp(Application)
            await asyncio.sleep(delay)
        if Application.Terminated:
            return
        ########################################
        # connect to server and start monitoring
        ########################################
        await StartServerApp(Application)
        async with Application.uaServer:
            for item in Application.bookDatCh:
                node = Application.get_server_node(item)
                Application.subDatChNodes.append(node)
                disp = await node.read_browse_name()
                path = await node.get_path(20,True)
                Application.dispNodes[item] = disp
                Application.pathNodes[item] = path
                short = getShortNodeId(item)
                Application.dispNodes[short] = disp
                Application.pathNodes[short] = path
                print("@OpcNodeId",short,item,disp.Name,"/".join(path))
            # print("OpcDataBook",Application.bookDatCh)
            # print("OpcServerNodes",Application.srvrNodes)
            print("Try to subscribe",len(Application.subDatChNodes),"items")
            # subscribing to data change notification of specified data nodes
            p = Application.subInter # pub/sub interval (period) milliseconds
            Application.subMan = await Application.uaServer.create_subscription(p, Application.subHandler)
            Application.subHan = await Application.subMan.subscribe_data_change(Application.subDatChNodes)
            reportSubscription(Application.subMan)
            await asyncio.sleep(0.5)
            #################################
            # subscribe to events from server
            #################################
            # TODO: await Application.subMan.subscribe_events()
            #################################
            # set writable nodes
            #################################
            for item in Application.wrblNodes:
                node = Application.get_server_node(item)
                short = getShortNodeId(item)
                await node.set_writable(True)
                print("@OpcWritable",short)
            await asyncio.sleep(0.5)
            ##########################
            # Now server ready for I/O
            ##########################
            Application.ServerReadyIO = True
            Application.ProcessDelayedIO()
            ###############################
            # Application main polling loop
            ###############################
            while not Application.Terminated:
                delay = 0.001*Application.PollPeriod
                await AsyncPollApp(Application)
                await asyncio.sleep(delay)
    except Exception as E:
        Application.OnError(E)
        Application.SysExit(1)
    finally:
        Application.Goodbye()

######
# MAIN
######

if __name__ == "__main__":
    asyncio.run(execute_opcua_server())

#############
# END OF FILE
#############
