 {
 ***********************************************************************
 MODBUS Server (Slave). At the moment IP,RTU,ASCII servers implemented.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @View node
|  Print information about logical device node.
|  If node is not specified, view table of ports & nodes.
|  node (1..64)  is logical device № (see @Node command).
| @Port port prot decl
|  Open logical MODBUS port to connect device nodes.
|  port (1..16) is logical № (i.e. port identifier).
|  prot (IP,RTU,ASCII) is MODBUS transport protocol.
|  decl is port detail declaration, looks like that:
|   tcp port 502 server 16 
|   com port 1 baudrate 9600 parity even databits 8 stopbits 1
|   See help on function pipe_init() for details.
| @PortPollRates Rx Tx
| @PortByteRates Rx Tx
| @PortPollCounters Rx Tx
| @PortByteCounters Rx Tx
| @PortErrorRates Rx Tx Ex
| @PortErrorCounters Rx Tx Ex
|  Set polls/bypes/errors counter/rate tag names of type Integer or Real
|  to count errors and rate (per second) on this port:
|  Rx - Receiver polls/bytes/errors (errors is bad request, no answer).
|  Tx - Transmitter polls/bytes/errors (errors is fail to send answer).
|  Ex - MODBUS Exceptions              (errors is bad address or data).
| @ZeroPortCounters n - Zero all stat. counters on port n (0 mean all).
| @Node node port addr
|  Declare MODBUS logical device node.
|  node (1..64)  is logical device node №.
|  port (1..16)  is logical port № where device connected.
|  addr (1..247) is device network address (slave identifier).
| @CoilStatusBase node base
| @InputStatusBase node base
| @InputRegisterBase node base
| @HoldingRegisterBase node base
|  Setup base adress for status & register inputs/outputs.
|  node (1..64) is logical device node №.
|  base is status/register input/output address base.
|  By default @CoilStatusBase=1, @InputStatusBase=10001,
|  @InputRegisterBase=30001, @HoldingRegisterBase=40001.
| @CoilStatus node addr typ src ref dst out
| @InputStatus node addr typ src ref
| @InputRegister node addr typ src ref
| @HoldingRegister node addr typ src ref dst out
|  Set address table, i.e. input/output data source/target.
|  node - device node index, 1..64
|  base - base address for this node
|  addr - address, starting from base
|  typ  - int16, int32, float, double
|  src  - AnalogInput,DigitalInput,Tag 
|  ref  - AI,DI number or tag name
|  dst  - AnalogOutput,DigitalOutput,Tag
|  out  - AO,DO number or tag name
| @ModScan32  - Run ModScan32  program to test MODBUS.
| @ModBusPoll - Run ModBusPoll program to test MODBUS.
| @ModbusHelp - Open manual - help file on &ModbusSrv.
| Implemented MODBUS protocols: TCP/IP, RTU and ASCII.
| Implemented MODBUS functions: 1,2,3,4,5,6,15,16.
|********************************************************
[]
 }
{
[Compiler.Options]            ; Size of table for ... 
Compiler.dtabmax = 1024*1024  ; -Data segment ( number of virtual machine data items )
Compiler.stabmax = 1024*8     ; -Strings      ( number of available string var+const )
Compiler.dtabmin = 1024*4     ; Min stack  space at start ( error if free space less )
Compiler.stabmin = 1024*1     ; Min string space at start ( error if free space less )
[]
}
program ModbusSrv;               { MODBUS Server (Slave)            }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_NetLibrary}            { NetLibrary constants             }
 MaxPortNum = 16;                { Max.number of ports,i.e. TCP/COM }
 MaxNodeNum = 64;                { Max.number of nodes,i.e. devices }
 MaxConnNum = 15;                { Max. RTU/ASCII connection number }
 MaxNumCoil = 1000;              { Max. number of Coil Status       }
 MaxNumInSt = 1000;              { Max. number of Input Status      }
 MaxNumIReg = 1000;              { Max. number of Input Registers   }  
 MaxNumHReg = 1000;              { Max. number of Holding Registers }
 MaxBuffer  = 8192;              { Maximum size of buffer to read   }
 mbs_SwapFL = 7;                 { MB swap float   1+2=bytes+words  }
 mbs_SwapIN = 7;                 { MB swap integer 1+2=bytes+words  }
 mbt_LIST   = 'INT16,INT32,FLOAT,DOUBLE';    { Data types           }
 mbt_LIST2  = ',BIT0,BIT1,BIT2,BIT3,BIT4,BIT5,BIT6,BIT7,BIT8,BIT9';
 mbt_LIST3  = ',BIT10,BIT11,BIT12,BIT13,BIT14,BIT15,BIT16,BIT17';
 mbt_LIST4  = ',BIT18,BIT19,BIT20,BIT21,BIT22,BIT23,BIT24,BIT25';
 mbt_LIST5  = ',BIT26,BIT27,BIT28,BIT29,BIT30,BIT31';
 mbt_Int16  = 1;                 { 16-bit integer                   }
 mbt_Int32  = 2;                 { 32-bit long integer              }
 mbt_Float  = 3;                 { 32-bit float                     }
 mbt_Double = 4;                 { 64-bit double float              }
 mbt_Bit0   = 5;                 { Status bit 0                     }
 mbs_LIST   = 'ANALOGINPUT,DIGITALINPUT,ANALOGOUTPUT,DIGITALOUTPUT,TAG';
 mbs_AI     = 1;                 { AnalogInput                      }
 mbs_DI     = 2;                 { DigitalInput                     }
 mbs_AO     = 3;                 { AnalogOutput                     }
 mbs_DO     = 4;                 { DigitalOutput                    }
 mbs_TAG    = 5;                 { Tag                              }
 RatePeriod = 1000;              { Statistics poll period           }

type
 TTagRef     = record tag:Integer; val:Real; end; { Tag ref. & val. }
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_NetLibrary}            { NetLibrary variables             }
 modbus     : record             { MODBUS related data              }
  ErrorCode : record             { MODBUS common error codes        }
   Rx       : Integer;           { Receiver errors                  }
   Tx       : Integer;           { Transmitter errors               }
   Ex       : Integer;           { MODBUS Exceptions                }
  end;
  Port      : array[1..MaxPortNum] of record { TCP/COM ports        }
   Decl     : String;            { Port declaration                 }
   Pipe     : Integer;           { TCP/COM port pipe                }
   Prot     : Integer;           { Protocol 1/2/3=ASCII/RTU/IP      }
   Req      : String;            { Temporary for answer             }
   Ans      : String;            { Temporary for request            }
   Buf      : array[0..MaxConnNum] of String;  { Temporary buffer   }
   Last     : record             { Last poll data                   }
    Poll    : Real;              { Last poll time, msec             }
    Rate    : Real;              { Last rate time, msec             }
   end;                          {                                  }
   Stat     : record             { Traffic Statistics               }
    Rate,                        { Rate                             }
    Count   : record             { Count                            }
     Bugs,                       { Errors                           }
     Bytes,                      { Bytes                            }
     Polls  : record             { Polls                            }
      Rx    : TTagRef;           { Received                         }
      Tx    : TTagRef;           { Transmitted                      }
      Ex    : TTagRef;           { MODBUS Exceptions, for bugs only }
     end;                        {                                  }
    end;                         {                                  }
   end;                          {                                  }
  end;                           {                                  }
  Node      : array[1..MaxNodeNum] of record { Nodes, i.e. devices  }
   Port     : Integer;           { Port index where node connected  }
   Addr     : Integer;           { Device address (unit identifier) }
   Coil     : record             { Coil Status table                }
    Base    : Integer;           { Base coil status          1      }
    Typ     : array[0..MaxNumCoil] of Integer; { Data type, mbt_*   }
    Src     : array[0..MaxNumCoil] of Integer; { Data source, mbs_* }
    Ref     : array[0..MaxNumCoil] of Integer; { Data reference     }
    Dst     : array[0..MaxNumCoil] of Integer; { Data destination   }
    Out     : array[0..MaxNumCoil] of Integer; { Data output        }
   end;                          {                                  }
   InSt     : record             { Input Status table               }
    Base    : Integer;           { Base input status     10001      }
    Typ     : array[0..MaxNumInSt] of Integer; { Data type, mbt_*   }
    Src     : array[0..MaxNumInSt] of Integer; { Data source, mbs_* }
    Ref     : array[0..MaxNumInSt] of Integer; { Data reference     }
   end;                          {                                  }
   IReg     : record             { Input Register table             }
    Base    : Integer;           { Base input register   30001      }
    Typ     : array[0..MaxNumIReg] of Integer; { Data type, mbt_*   }
    Src     : array[0..MaxNumIReg] of Integer; { Data source, mbs_* }
    Ref     : array[0..MaxNumIReg] of Integer; { Data reference     }
   end;                          {                                  }
   HReg     : record             { Holding Register table           }
    Base    : Integer;           { Base holding register 40001      }
    Typ     : array[0..MaxNumHReg] of Integer; { Data type, mbt_*   }
    Src     : array[0..MaxNumHReg] of Integer; { Data source, mbs_* }
    Ref     : array[0..MaxNumHReg] of Integer; { Data reference     }
    Dst     : array[0..MaxNumHReg] of Integer; { Data destination   }
    Out     : array[0..MaxNumHReg] of Integer; { Data output        }
   end;                          {                                  }
  end;                           {                                  }
 end;                            {                                  }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_NetLibrary}            { NetLibrary functions             }

 //
 // Get data dump as int16/int32/float/double, with byte reordering.
 //
 function modbus_dump_typ(typ:Integer; data:Real):String;
 begin
  if typ=mbt_int16 then modbus_dump_typ:=modbus_dump_int16(Round(data),mbs_SwapIN) else
  if typ=mbt_int32 then modbus_dump_typ:=modbus_dump_int32(Round(data),mbs_SwapIN) else
  if typ=mbt_float then modbus_dump_typ:=modbus_dump_float(data,mbs_SwapFL) else
  if typ=mbt_double then modbus_dump_typ:=modbus_dump_double(data,mbs_SwapFL) else
  modbus_dump_typ:='';
 end;
 //
 // Get data size of type int16/int32/float/double, in bytes.
 //
 function modbus_typ_size(typ:Integer):Integer;
 begin
  if typ=mbt_int16 then modbus_typ_size:=2 else
  if typ=mbt_int32 then modbus_typ_size:=4 else
  if typ=mbt_float then modbus_typ_size:=4 else
  if typ=mbt_double then modbus_typ_size:=8 else
  modbus_typ_size:=0;
 end;
 //
 // Take data from dump as int16/int32/float/double, with byte reordering.
 //
 function modbus_take_typ(typ:Integer; data:String):Real;
 begin
  if typ=mbt_int16 then modbus_take_typ:=modbus_take_int16(data,mbs_SwapIN) else
  if typ=mbt_int32 then modbus_take_typ:=modbus_take_int32(data,mbs_SwapIN) else
  if typ=mbt_float then modbus_take_typ:=modbus_take_float(data,mbs_SwapFL)  else
  if typ=mbt_double then modbus_take_typ:=modbus_take_double(data,mbs_SwapFL) else
  modbus_take_typ:=0;
 end;
 //
 // Check port, node, address numbers is good to use.
 //
 function modbus_port_ok(port:Integer):Boolean;
 begin
  modbus_port_ok:=((port>=1) and (port<=MaxPortNum));
 end;
 function modbus_node_ok(node:Integer):Boolean;
 begin
  modbus_node_ok:=((node>=1) and (node<=MaxNodeNum));
 end;
 function modbus_coil_ok(addr:Integer):Boolean;
 begin
  modbus_coil_ok:=((addr>=0) and (addr<MaxNumCoil));
 end;
 function modbus_inst_ok(addr:Integer):Boolean;
 begin
  modbus_inst_ok:=((addr>=0) and (addr<MaxNumInSt));
 end;
 function modbus_ireg_ok(addr:Integer):Boolean;
 begin
  modbus_ireg_ok:=((addr>=0) and (addr<MaxNumIReg));
 end;
 function modbus_hreg_ok(addr:Integer):Boolean;
 begin
  modbus_hreg_ok:=((addr>=0) and (addr<MaxNumHReg));
 end;
 //
 // Return string like TCP[1], RTU[1] or ASC[1].
 //
 function modbus_port_alias(port:Integer):String;
 var prot:Integer;
 begin
  if modbus_port_ok(port) then prot:=modbus.Port[port].Prot else prot:=0;
  modbus_port_alias:=modbus_prot_alias(prot)+'['+Str(port)+']';
 end;
 //
 // Clear modbus variables
 //
 procedure modbus_clear;
 var port,node,addr,conn:Integer;
 begin
  modbus.ErrorCode.Rx:=RegisterErr(DevName+' Rx');
  modbus.ErrorCode.Tx:=RegisterErr(DevName+' Tx');
  modbus.ErrorCode.Ex:=RegisterErr(DevName+' Ex');
  for port:=1 to MaxPortNum do begin
   modbus.Port[port].Decl:='';
   modbus.Port[port].Pipe:=0;
   modbus.Port[port].Prot:=0;
   modbus.Port[port].Req:='';
   modbus.Port[port].Ans:='';
   for conn:=0 to MaxConnNum do modbus.Port[port].Buf[conn]:='';
   modbus.Port[port].Last.Poll:=0;
   modbus.Port[port].Last.Rate:=0;
   modbus.Port[port].Stat.Rate.Bugs.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Count.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Ex.val:=0;
  end;
  for node:=1 to MaxNodeNum do begin
   modbus.Node[node].Port:=0;
   modbus.Node[node].Addr:=0;
   modbus.Node[node].Coil.Base:=1;
   modbus.Node[node].InSt.Base:=10001;
   modbus.Node[node].IReg.Base:=30001;
   modbus.Node[node].HReg.Base:=40001;
   for addr:=0 to MaxNumCoil do begin
    modbus.Node[node].Coil.Typ[addr]:=0;
    modbus.Node[node].Coil.Src[addr]:=0;
    modbus.Node[node].Coil.Ref[addr]:=0;
    modbus.Node[node].Coil.Dst[addr]:=0;
    modbus.Node[node].Coil.Out[addr]:=0;
   end;
   for addr:=0 to MaxNumInSt do begin
    modbus.Node[node].InSt.Typ[addr]:=0;
    modbus.Node[node].InSt.Src[addr]:=0;
    modbus.Node[node].InSt.Ref[addr]:=0;
   end;
   for addr:=0 to MaxNumIReg do begin
    modbus.Node[node].IReg.Typ[addr]:=0;
    modbus.Node[node].IReg.Src[addr]:=0;
    modbus.Node[node].IReg.Ref[addr]:=0;
   end;
   for addr:=0 to MaxNumHReg do begin
    modbus.Node[node].HReg.Typ[addr]:=0;
    modbus.Node[node].HReg.Src[addr]:=0;
    modbus.Node[node].HReg.Ref[addr]:=0;
    modbus.Node[node].HReg.Dst[addr]:=0;
    modbus.Node[node].HReg.Out[addr]:=0;
   end;
  end;
 end;
 //
 // Close specified port (0=all ports).
 //
 procedure modbus_close(port:Integer);
 var conn:Integer;
 begin
  if port=0 then begin
   for port:=1 to MaxPortNum do modbus_close(port);
  end else
  if modbus_port_ok(port) then begin
   if modbus.Port[port].Pipe<>0 then
   if pipe_free(modbus.Port[port].Pipe)
   then Success('MODBUS closed OK: '+Str(port)
               +' '+modbus_prot_name(modbus.Port[port].Prot)
               +' '+modbus.Port[port].Decl)
   else Trouble('MODBUS close FAIL: '+Str(port)
               +' '+modbus_prot_name(modbus.Port[port].Prot)
               +' '+modbus.Port[port].Decl);
   modbus.Port[port].Decl:='';
   modbus.Port[port].Pipe:=0;
   modbus.Port[port].Prot:=0;
   modbus.Port[port].Req:='';
   modbus.Port[port].Ans:='';
   for conn:=0 to MaxConnNum do modbus.Port[port].Buf[conn]:='';
   modbus.Port[port].Last.Poll:=0;
   modbus.Port[port].Last.Rate:=0;
   modbus.Port[port].Stat.Rate.Bugs.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Bytes.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Rx.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Tx.tag:=0;
   modbus.Port[port].Stat.Rate.Polls.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Bugs.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Bytes.Ex.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Rx.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Tx.tag:=0;
   modbus.Port[port].Stat.Count.Polls.Ex.tag:=0;
   modbus.Port[port].Stat.Rate.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Count.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Ex.val:=0;
  end;
 end;
 //
 // Zero statistics for specified port (0=all ports).
 //
 procedure modbus_zero_stat(port:Integer);
 begin
  if port=0 then begin
   for port:=1 to MaxPortNum do modbus_zero_stat(port);
  end else
  if modbus_port_ok(port) then begin
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bugs.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bugs.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bugs.Ex.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bytes.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bytes.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Bytes.Ex.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Polls.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Polls.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Rate.Polls.Ex.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bugs.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bugs.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bugs.Ex.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bytes.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Bytes.Ex.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Polls.Rx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Polls.Tx.tag,0));
   bNul(modbus_set_tag(modbus.Port[port].Stat.Count.Polls.Ex.tag,0));
   modbus.Port[port].Stat.Rate.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Rate.Polls.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bugs.Ex.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Rx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Tx.val:=0;
   modbus.Port[port].Stat.Count.Bytes.Ex.val:=0;
   modbus.Port[port].Stat.Count.Polls.Rx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Tx.val:=0;
   modbus.Port[port].Stat.Count.Polls.Ex.val:=0;
  end;
 end;
 //
 // Open MODBUS port
 //  arg = Port Prot Decl
 //  arg = 1    IP   tcp port 502 server 16
 //  arg = 2    RTU  com port 1 baudrate 9600 parity even databits 8 stopbits 1
 //
 function modbus_open(arg:String):Integer;
 var Port,Prot,Pipe:Integer; Decl:String;
 begin
  Port:=Val(ExtractWord(1,arg));
  Prot:=modbus_prot_byname(UpCaseStr(ExtractWord(2,arg)));
  Decl:=Trim(SkipWords(2,arg));
  if modbus_port_ok(Port) and modbus_prot_ok(Prot) and not IsEmptyStr(Decl) then begin
   Pipe:=pipe_init(Decl);
   if Pipe<>0 then begin
    modbus_close(Port);
    modbus.Port[Port].Decl:=Decl;
    modbus.Port[Port].Prot:=Prot;
    modbus.Port[Port].Pipe:=Pipe;
   end else Port:=0;
  end else Port:=0;
  modbus_open:=Port;
  Decl:='';
 end;
 //
 // Initialize modbus variables
 //
 procedure modbus_init;
 begin
  modbus_clear;
 end;
 //
 // Finalize modbus variables
 //
 procedure modbus_free;
 begin
  modbus_close(0);
  modbus_clear;
 end;
 //
 // Reset modbus to initial state
 //
 procedure modbus_reset;
 begin
  modbus_free;
  modbus_init;
 end;
 //
 // Get single coil status with address (node,addr) or -1 on error.
 //
 function GetCoilStatus(node,addr:Integer):Integer;
 var typ,src,ref,data:Integer;
 begin
  data:=-1;
  if modbus_node_ok(node) then
  if modbus_coil_ok(addr) then begin
   typ:=modbus.Node[node].Coil.Typ[addr];
   src:=modbus.Node[node].Coil.Src[addr];
   ref:=modbus.Node[node].Coil.Ref[addr];
   if (typ>=mbt_Bit0) and (src>0) and (ref>=0) then begin
    if src=mbs_TAG then begin
     if TypeTag(ref)=1 then data:=Ord(IsBit(iGetTag(ref),typ-mbt_Bit0)) else
     if TypeTag(ref)=2 then data:=Ord(IsBit(rGetTag(ref),typ-mbt_Bit0));
    end else
    if src=mbs_AI then data:=Ord(IsBit(GetAi_Yn(ref),typ-mbt_Bit0)) else
    if src=mbs_DI then data:=Ord(IsBit(GetDi_Yn(ref),typ-mbt_Bit0));
   end;
  end; 
  GetCoilStatus:=data;
 end;
 //
 // Application specific routine to get coil statuses.
 // @CoilStatus node addr typ src ref
 // node - device node, 1..MaxNodeNum
 // addr - address starting from 1
 // typ  - bit0/bit1/bit2/../bit31
 // src  - AnalogInput,DigitalInput,Tag
 // ref  - AI/DI number or tag name
 //
 function GetCoilStatuses(node,saddr,count:Integer; var ans:String):Integer;
 var errn,cntr,stat,bits:Integer; 
 begin
  ans:=''; errn:=modbus_er_OK;                          // Clear buffers & error number
  if node<1 then errn:=modbus_er_ILLADR;                // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;       // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;               // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;               // Illegal data value (number of reg)
  if count>modbus_MaxCSGet then errn:=modbus_er_ILLVAL; // Illegal data value (count too big)
  cntr:=0; bits:=0;                                     // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin   // For each address in table
   stat:=GetCoilStatus(node,saddr+cntr);                // Get coil status data
   if stat<0 then errn:=modbus_er_ILLADR;               // Invalid address?
   bits:=iOr(bits,iShift(stat,cntr mod 8));             // Accumulate bits
   if((cntr mod 8)=7) or (cntr=count-1) then begin      // If it is 8-th bit
    ans:=ans+Chr(bits);                                 // Accumulate answer
    bits:=0;                                            // Clear bits buffer
   end;                                                 // End if
   cntr:=cntr+1;                                        // Summ data counter
  end;                                                  // End while
  if (errn=modbus_er_OK) and (cntr<>count)              // Check data size
  then errn:=modbus_er_ILLVAL;                          // Unexpected data size
  GetCoilStatuses:=errn;                                // Return error number or 0 if OK
 end;
 //
 // Get single input status with address (node,addr) or -1 on error.
 //
 function GetInputStatus(node,addr:Integer):Integer;
 var typ,src,ref,data:Integer;
 begin
  data:=-1;
  if modbus_node_ok(node) then
  if modbus_inst_ok(addr) then begin
   typ:=modbus.Node[node].InSt.Typ[addr];
   src:=modbus.Node[node].InSt.Src[addr];
   ref:=modbus.Node[node].InSt.Ref[addr];
   if (typ>=mbt_Bit0) and (src>0) and (ref>=0) then begin
    if src=mbs_TAG then begin
     if TypeTag(ref)=1 then data:=Ord(IsBit(iGetTag(ref),typ-mbt_Bit0)) else
     if TypeTag(ref)=2 then data:=Ord(IsBit(rGetTag(ref),typ-mbt_Bit0));
    end else
    if src=mbs_AI then data:=Ord(IsBit(GetAi_Yn(ref),typ-mbt_Bit0)) else
    if src=mbs_DI then data:=Ord(IsBit(GetDi_Yn(ref),typ-mbt_Bit0));
   end;
  end; 
  GetInputStatus:=data;
 end;
 //
 // Application specific routine to get input statuses.
 // @InputStatus node addr typ src ref
 // node - device node, 1..MaxNodeNum
 // addr - address starting from 10001
 // typ  - bit0/bit1/bit2/../bit31
 // src  - AnalogInput,DigitalInput,Tag
 // ref  - AI/DI number or tag name
 //
 function GetInputStatuses(node,saddr,count:Integer; var ans:String):Integer;
 var errn,cntr,stat,bits:Integer; 
 begin
  ans:=''; errn:=modbus_er_OK;                          // Clear buffers & error number
  if node<1 then errn:=modbus_er_ILLADR;                // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;       // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;               // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;               // Illegal data value (number of reg)
  if count>modbus_MaxISGet then errn:=modbus_er_ILLVAL; // Illegal data value (count too big)
  cntr:=0; bits:=0;                                     // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin   // For each address in table
   stat:=GetInputStatus(node,saddr+cntr);               // Get input status data
   if stat<0 then errn:=modbus_er_ILLADR;               // Invalid address?
   bits:=iOr(bits,iShift(stat,cntr mod 8));             // Accumulate bits
   if((cntr mod 8)=7) or (cntr=count-1) then begin      // If it is 8-th bit
    ans:=ans+Chr(bits);                                 // Accumulate answer
    bits:=0;                                            // Clear bits buffer
   end;                                                 // End if
   cntr:=cntr+1;                                        // Inc data counter
  end;                                                  // End while
  if (errn=modbus_er_OK) and (cntr<>count)              // Check data size
  then errn:=modbus_er_ILLVAL;                          // Unexpected data size
  GetInputStatuses:=errn;                               // Return error number or 0 if OK
 end;
 //
 // Get data dump of input register with address (node,addr).
 //
 function DumpInputRegister(node,addr:Integer):String;
 var typ,src,ref:Integer; data:String;
 begin
  data:='';
  if modbus_node_ok(node) then
  if modbus_ireg_ok(addr) then begin
   typ:=modbus.Node[node].IReg.Typ[addr];
   src:=modbus.Node[node].IReg.Src[addr];
   ref:=modbus.Node[node].IReg.Ref[addr];
   if (typ>0) and (src>0) and (ref>=0) then begin
    if src=mbs_TAG then begin
     if TypeTag(ref)=1 then data:=modbus_dump_typ(typ,iGetTag(ref)) else
     if TypeTag(ref)=2 then data:=modbus_dump_typ(typ,rGetTag(ref));
    end else
    if src=mbs_AI then data:=modbus_dump_typ(typ,GetAi_Yn(ref)) else
    if src=mbs_DI then data:=modbus_dump_typ(typ,GetDi_Yn(ref));
   end;
  end; 
  DumpInputRegister:=data;
  data:='';
 end;
 //
 // Application specific routine to get input registers.
 // @InputRegister node addr typ src ref
 // node - device node, 1..MaxNodeNum
 // addr - address starting from 30001
 // typ  - int16/int32/float/double
 // src  - AnalogInput,DigitalInput,Tag
 // ref  - AI/DI number or tag name
 //
 function GetInputRegisters(node,saddr,count:Integer; var ans:String):Integer;
 var errn,cntr,leng:Integer; data:String;
 begin
  ans:=''; data:=''; errn:=modbus_er_OK;                // Clear buffers & error number
  if node<1 then errn:=modbus_er_ILLADR;                // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;       // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;               // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;               // Illegal data value (number of reg)
  if count>modbus_MaxIRGet then errn:=modbus_er_ILLVAL; // Illegal data value (count too big)
  cntr:=0;                                              // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin   // For each address in table
   data:=DumpInputRegister(node,saddr+cntr);            // Get input register data
   leng:=Length(data);                                  // Find data length
   if leng=0 then errn:=modbus_er_ILLADR;               // Invalid address?
   if Odd(leng) then errn:=modbus_er_ILLVAL;            // Should be even
   ans:=ans+data;                                       // Accumulate answer
   cntr:=cntr+(leng div 2);                             // Inc data counter
  end;                                                  // End loop
  if (errn=modbus_er_OK) and (cntr<>count)              // Check data size
  then errn:=modbus_er_ILLVAL;                          // Unexpected data size
  GetInputRegisters:=errn;                              // Return error number or 0 if OK
  data:='';                                             // Clear data buffer
 end; 
 //
 // Get data dump of holding register with address (node,addr).
 //
 function DumpHoldingRegister(node,addr:Integer):String;
 var typ,src,ref:Integer; data:String;
 begin
  data:='';
  if modbus_node_ok(node) then
  if modbus_hreg_ok(addr) then begin
   typ:=modbus.Node[node].HReg.Typ[addr];
   src:=modbus.Node[node].HReg.Src[addr];
   ref:=modbus.Node[node].HReg.Ref[addr];
   if (typ>0) and (src>0) and (ref>=0) then begin
    if src=mbs_TAG then begin
     if TypeTag(ref)=1 then data:=modbus_dump_typ(typ,iGetTag(ref)) else
     if TypeTag(ref)=2 then data:=modbus_dump_typ(typ,rGetTag(ref));
    end else
    if src=mbs_AI then data:=modbus_dump_typ(typ,GetAi_Yn(ref)) else
    if src=mbs_DI then data:=modbus_dump_typ(typ,GetDi_Yn(ref));
   end;
  end; 
  DumpHoldingRegister:=data;
  data:='';
 end;
 //
 // Application specific routine to get holding registers.
 // @HoldingRegister node addr typ src ref dst out
 // node - device node, 1..MaxNodeNum
 // addr - address starting from 40001
 // typ  - int16/int32/float/double
 // src  - AnalogInput,DigitalInput,Tag
 // ref  - AI/DI number or tag name
 // dst  - AnalogOutput,DigitalOutput,Tag
 // out  - AO/DO number or tag name
 //
 function GetHoldingRegisters(node,saddr,count:Integer; var ans:String):Integer;
 var errn,cntr,leng:Integer; data:String;
 begin
  ans:=''; data:=''; errn:=modbus_er_OK;                // Clear buffers & error number
  if node<1 then errn:=modbus_er_ILLADR;                // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;       // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;               // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;               // Illegal data value (number of reg)
  if count>modbus_MaxHRGet then errn:=modbus_er_ILLVAL; // Illegal data value (count too big)
  cntr:=0;                                              // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin   // For each address in table
   data:=DumpHoldingRegister(node,saddr+cntr);          // Get holding register data
   leng:=Length(data);                                  // Find data length
   if leng=0 then errn:=modbus_er_ILLADR;               // Invalid address?
   if Odd(leng) then errn:=modbus_er_ILLVAL;            // Should be even
   ans:=ans+data;                                       // Accumulate answer
   cntr:=cntr+(leng div 2);                             // Inc data counter
  end;                                                  // End loop
  if (errn=modbus_er_OK) and (cntr<>count)              // Check data size
  then errn:=modbus_er_ILLVAL;                          // Unexpected data size
  GetHoldingRegisters:=errn;                            // Return error number or 0 if OK
  data:='';                                             // Clear data buffer
 end;
 //
 // Write data dump to holding register with address (node,addr).
 // Return number of bytes written or 0.
 //
 function WriteHoldingRegister(node,addr:Integer; data:String):Integer;
 var typ,src,ref,dst,out,siz,len,cod:Integer;
 begin
  cod:=0;
  if modbus_node_ok(node) then
  if modbus_hreg_ok(addr) then begin
   typ:=modbus.Node[node].HReg.Typ[addr];
   src:=modbus.Node[node].HReg.Src[addr];
   ref:=modbus.Node[node].HReg.Ref[addr];
   dst:=modbus.Node[node].HReg.Dst[addr];
   out:=modbus.Node[node].HReg.Out[addr];
   siz:=modbus_typ_size(typ);
   len:=Length(data);
   if (typ>0) and (siz>0) and (len>=siz) and (dst>0) and (out>=0) then begin
    if dst=mbs_TAG then begin
     if TypeTag(out)=1 then begin bNul(iSetTag(out,Round(modbus_take_typ(typ,data)))); cod:=siz; end else
     if TypeTag(out)=2 then begin bNul(rSetTag(out,modbus_take_typ(typ,data))); cod:=siz; end else
    end else
    if dst=mbs_AO then begin UpdateAo(out,time,modbus_take_typ(typ,data)); cod:=siz; end else
    if dst=mbs_DO then begin UpdateDo(out,time,modbus_take_typ(typ,data)); cod:=siz; end;
   end;
  end; 
  WriteHoldingRegister:=cod;
 end;
 //
 // Application specific routine to put holding registers.
 //
 function PutHoldingRegisters(node,saddr,count,nbyte:Integer; data:String):Integer;
 var errn,cntr,leng:Integer;
 begin
  errn:=modbus_er_OK;                                    // Clear error number
  if node<1 then errn:=modbus_er_ILLADR;                 // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;        // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;                // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;                // Illegal data value (number of reg)
  if count>modbus_MaxHRPut then errn:=modbus_er_ILLVAL;  // Illegal data value (count too big)
  if nbyte<>2*count then errn:=modbus_er_ILLVAL;         // Illegal byte value (num. of bytes)
  if nbyte<>Length(data) then errn:=modbus_er_ILLVAL;    // Illegal byte value (num. of bytes)
  cntr:=0;                                               // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin    // For each address in table
   leng:=WriteHoldingRegister(node,saddr+cntr,Copy(data,1+cntr*2,8)); // Write holding register data
   if leng=0 then errn:=modbus_er_ILLADR;                // Invalid address?
   if Odd(leng) then errn:=modbus_er_ILLVAL;             // Should be even
   cntr:=cntr+(leng div 2);                              // Inc data counter
  end;                                                   // End loop
  if (errn=modbus_er_OK) and (cntr<>count)               // Check data size
  then errn:=modbus_er_ILLVAL;                           // Unexpected data size
  PutHoldingRegisters:=errn;                             // Return error number or 0 if OK
  data:='';                                              // Clear data buffer
 end;
 //
 // Write data (0/1) to coil status with address (node,addr). Return 0/1=Error/Ok.
 //
 function WriteCoilStatus(node,addr:Integer; data:Integer):Integer;
 var typ,src,ref,dst,out,cod,bit:Integer;
 begin
  cod:=0;
  data:=iAnd(data,1);
  if modbus_node_ok(node) then
  if modbus_coil_ok(addr) then begin
   typ:=modbus.Node[node].Coil.Typ[addr];
   src:=modbus.Node[node].Coil.Src[addr];
   ref:=modbus.Node[node].Coil.Ref[addr];
   dst:=modbus.Node[node].Coil.Dst[addr];
   out:=modbus.Node[node].Coil.Out[addr];
   if (typ>=mbt_Bit0) and (typ<mbt_Bit0+32) and (dst>0) and (out>=0) then begin
    bit:=typ-mbt_Bit0;
    if dst=mbs_TAG then begin
     if TypeTag(out)=1 then begin bNul(iSetTag(out,iSetBit(Round(iGetTag(out)),bit,data))); cod:=1; end else
     if TypeTag(out)=2 then begin bNul(rSetTag(out,iSetBit(Round(rGetTag(out)),bit,data))); cod:=1; end else
    end else
    if dst=mbs_AO then begin UpdateAo(out,time,data); cod:=1; end else
    if dst=mbs_DO then begin UpdateDo(out,time,data); cod:=1; end;
   end;
  end; 
  WriteCoilStatus:=cod;
 end;
 //
 // Application specific routine to put coil statuses.
 //
 function PutCoilStatuses(node,saddr,count,nbyte:Integer; data:String):Integer;
 var errn,cntr,leng,coil:Integer;
 begin
  errn:=modbus_er_OK;                                     // Clear error number
  if node<1 then errn:=modbus_er_ILLADR;                  // Illegal address    (device node)
  if node>MaxNodeNum then errn:=modbus_er_ILLADR;         // Illegal address    (device node)
  if saddr<0 then errn:=modbus_er_ILLADR;                 // Illegal address    (start address)
  if count<1 then errn:=modbus_er_ILLVAL;                 // Illegal data value (number of reg)
  if count>modbus_MaxCSPut then errn:=modbus_er_ILLVAL;   // Illegal data value (count too big)
  if nbyte<>((count+7)div 8) then errn:=modbus_er_ILLVAL; // Illegal byte value (num. of bytes)
  if nbyte<>Length(data) then errn:=modbus_er_ILLVAL;     // Illegal byte value (num. of bytes)
  cntr:=0;                                                // Clear counter
  while (cntr<count) and (errn=modbus_er_OK) do begin     // For each address in table
   coil:=Dump2i(Copy(data,1+(cntr div 8),1));             // Extract data byte of coil
   coil:=iAnd(iShift(coil,-(cntr mod 8)),1);              // Extract data bit of coil
   leng:=WriteCoilStatus(node,saddr+cntr,coil);           // Write coil status
   if leng=0 then errn:=modbus_er_ILLADR;                 // Invalid address?
   cntr:=cntr+leng;                                       // Inc data counter
  end;                                                    // End loop
  if (errn=modbus_er_OK) and (cntr<>count)                // Check data size
  then errn:=modbus_er_ILLVAL;                            // Unexpected data size
  PutCoilStatuses:=errn;                                  // Return error number or 0 if OK
  data:='';                                               // Clear data buffer
 end;
 //
 // Answer on MODBUS IP request
 //
 function modbus_ip_ans(port:Integer; req:String):String;
 var ans,dat:String; hdr:array[1..5] of Char;
     trid,prid,pdulen,unid,func,node,saddr,count,nbyte,datav,errn:Integer;
 begin
  ans:=''; dat:='';
  if modbus_port_ok(port) then                            // Check port number
  if (modbus.Port[port].Prot = modbus_pr_IP) then begin   // Check protocol IP
   pdulen:=modbus_decode_ip(req,trid,prid,unid,func,dat); // Decode message
   if pdulen>0 then begin                                 // MBAP Header is OK
    if DebugFlagEnabled(dfDetails) then begin             // Print details:
     Details('MBAP Tran '+Str(trid));                     // MBAP Transaction Id
     Details('MBAP Prot '+Str(prid));                     // MBAP Protocol Id
     Details('MBAP Leng '+Str(pdulen+1));                 // MBAP Data length
     Details('MBAP Unit '+Str(unid));                     // MBAP Unit Id
     Details('PDU  Func '+Str(func));                     // PDU Function Id
     Details('PDU  Data '+Hex_Encode(dat));               // PDU Data
    end;                                                  //
    if modbus_addr_ok(unid) then                          // If valid unit ID address
    for node:=1 to MaxNodeNum do                          // Scan all connected nodes
    if Length(ans)=0 then                                 // Skip dublicates
    if port = modbus.Node[node].Port then                 // Node connected to this port?
    if unid = modbus.Node[node].Addr then begin           // Answer only if good address found
     errn:=modbus_er_OK;                                  // Clear error number first
     if func = modbus_fn_ReadCS then begin                // Read coil statuses:
      if Length(dat)>=4 then begin                        // Expected 4+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
       end;                                               // 
       errn:=GetCoilStatuses(node,saddr,count,ans);       // Get coil statuses
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                        // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_ReadIS then begin                // Read input statuses:
      if Length(dat)>=4 then begin                        // Expected 4+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
       end;                                               // 
       errn:=GetInputStatuses(node,saddr,count,ans);      // Get input statuses
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                        // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_ReadHR then begin                // Read holding registers:
      if Length(dat)>=4 then begin                        // Expected 4+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
       end;                                               // 
       errn:=GetHoldingRegisters(node,saddr,count,ans);   // Get holding registers
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                        // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_ReadIR then begin                // Read input registers:
      if Length(dat)>=4 then begin                        // Expected 4+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
       end;                                               // 
       errn:=GetInputRegisters(node,saddr,count,ans);     // Get input registers
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                        // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_WritSC then begin                // Write single coil status:
      if Length(dat)>=4 then begin                        // Expected 4 byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);                // Data value
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Data '+Str(datav));                 // Data
       end;                                               // 
       if(datav<>0) and (datav<>modbus_0xFF00)            // Check data value
       then errn:=modbus_er_ILLVAL                        // Must be 0 or $FF00
       else errn:=PutCoilStatuses(node,saddr,1,1,Dump(Chr(Ord(datav<>0)))); // Put coil status
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Copy(dat,1,4);                               // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_WritSR then begin                // Write single holding register:
      if Length(dat)>=4 then begin                        // Expected 4 byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);                // Data value
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Data '+Str(datav));                 // Data
       end;                                               // 
       errn:=PutHoldingRegisters(node,saddr,1,2,Copy(dat,3));// Put holding register
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Copy(dat,1,4);                               // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_WritMC then begin                // Write multiple coil statuses:
      if Length(dat)>=5 then begin                        // Expected 5+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       nbyte:=Ord(hdr[5]);                                // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
        Details('PDU  Byte '+Str(nbyte));                 // Counter
       end;                                               // 
       errn:=PutCoilStatuses(node,saddr,count,nbyte,Copy(dat,6));// Put coil statuses
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Copy(dat,1,4);                               // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     if func = modbus_fn_WritMR then begin                // Write multiple holding registers:
      if Length(dat)>=5 then begin                        // Expected 5+ byte data
       hdr:=dat;                                          // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                // Number of registers
       nbyte:=Ord(hdr[5]);                                // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin          // Print details:
        Details('PDU  Addr '+Str(saddr));                 // Address
        Details('PDU  Numb '+Str(count));                 // Counter
        Details('PDU  Byte '+Str(nbyte));                 // Counter
       end;                                               // 
       errn:=PutHoldingRegisters(node,saddr,count,nbyte,Copy(dat,6));// Put holding registers
       if errn=modbus_er_OK then begin                    // Ok, data ready to encode
        ans:=Copy(dat,1,4);                               // Prepare answer
        ans:=modbus_encode_ip(trid,prid,unid,func,ans);   // Encode answer
       end;                                               // Done
      end else errn:=modbus_er_ILLVAL;                    // Illegal data value (length)
     end else
     errn:=modbus_er_ILLFUN;                              // Illegal function
     if errn<>modbus_er_OK then begin                     // Exception answer: (func or $80) + ErrorCode
      ans:=modbus_encode_ip(trid,prid,unid,iOr(func,modbus_er_EXCEPT),Dump(Chr(errn)));
      modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Ex.tag,modbus.ErrorCode.Ex,modbus_port_alias(port)
                 +' ! EXCEPTION node '+Str(node)+' addr '+Str(unid)
                 +' func $'+HexB(iOr(func,modbus_er_EXCEPT))+' code $'+HexB(errn));
     end;
    end;
   end;
  end;
  modbus_ip_ans:=ans;
  ans:=''; dat:='';
 end;
 //
 // Answer on MODBUS RTU request
 //
 function modbus_rtu_ans(port:Integer; req:String):String;
 var ans,dat:String; hdr:array[1..5] of Char;
     pdulen,unid,func,node,saddr,count,nbyte,datav,errn:Integer;
 begin
  ans:=''; dat:='';
  if modbus_port_ok(port) then                           // Check port number
  if (modbus.Port[port].Prot = modbus_pr_RTU) then begin // Check protocol RTU
   pdulen:=modbus_decode_rtu(req,unid,func,dat);         // Decode message
   if pdulen>0 then begin                                // Looks good?
    if DebugFlagEnabled(dfDetails) then begin            // Print details:
     Details('ADU Unit '+Str(unid));                     // ADU Unit Id
     Details('PDU Leng '+Str(pdulen));                   // PDU length
     Details('PDU Func '+Str(func));                     // PDU Function Id
     Details('PDU Data '+Hex_Encode(dat));               // PDU Data
    end;                                                 //
    if modbus_addr_ok(unid) then                         // If valid unit ID address
    for node:=1 to MaxNodeNum do                         // Scan all connected nodes
    if Length(ans)=0 then                                // Skip dublicates
    if port = modbus.Node[node].Port then                // Node connected to this port?
    if unid = modbus.Node[node].Addr then begin          // Answer only if good address found
     errn:=modbus_er_OK;                                 // Clear error number first
     if func = modbus_fn_ReadCS then begin               // Read coil statuses:
      if Length(dat)>=4 then begin                       // Expected 4+ byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
       end;                                              // 
       errn:=GetCoilStatuses(node,saddr,count,ans);      // Get coil statuses
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                       // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else
     if func = modbus_fn_ReadIS then begin               // Read input statuses:
      if Length(dat)>=4 then begin                       // Expected 4+ byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
       end;                                              // 
       errn:=GetInputStatuses(node,saddr,count,ans);     // Get input statuses
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                       // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else                                            //
     if func = modbus_fn_ReadHR then begin               // Read holding registers:
      if Length(dat)>=4 then begin                       // Expected 4+ byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
       end;                                              // 
       errn:=GetHoldingRegisters(node,saddr,count,ans);  // Get holding registers
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                       // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else                                            //
     if func = modbus_fn_ReadIR then begin               // Read input registers:
      if Length(dat)>=4 then begin                       // Expected 4+ byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
       end;                                              // 
       errn:=GetInputRegisters(node,saddr,count,ans);    // Get input registers
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                       // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else                                            //
     if func = modbus_fn_WritSC then begin               // Write single coil status:
      if Length(dat)>=4 then begin                       // Expected 4 byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);               // Data value
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Data '+Str(datav));                // Data
       end;                                              // 
       if(datav<>0) and (datav<>modbus_0xFF00) then errn:=modbus_er_ILLVAL else // Must be 0/$FF00
       errn:=PutCoilStatuses(node,saddr,1,1,Dump(Chr(Ord(datav<>0)))); // Put coil status
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Copy(dat,1,4);                              // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else
     if func = modbus_fn_WritSR then begin               // Write single holding register:
      if Length(dat)>=4 then begin                       // Expected 4 byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);               // Data value
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Data '+Str(datav));                // Data
       end;                                              // 
       errn:=PutHoldingRegisters(node,saddr,1,2,Copy(dat,3));// Put holding register
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Copy(dat,1,4);                              // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else
     if func = modbus_fn_WritMC then begin               // Write multiple coil statuses:
      if Length(dat)>=5 then begin                       // Expected 5+ byte data
       hdr:=dat;                                         // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       nbyte:=Ord(hdr[5]);                               // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
        Details('PDU  Byte '+Str(nbyte));                // Counter
       end;                                              // 
       errn:=PutCoilStatuses(node,saddr,count,nbyte,Copy(dat,6));// Put coil statuses
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Copy(dat,1,4);                              // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else
     if func = modbus_fn_WritMR then begin               // Write multiple holding registers:
      if Length(dat)>=5 then begin                       // Expected 5+ byte data
       hdr:=dat;                                         // Copy data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);               // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);               // Number of registers
       nbyte:=Ord(hdr[5]);                               // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin         // Print details:
        Details('PDU  Addr '+Str(saddr));                // Address
        Details('PDU  Numb '+Str(count));                // Counter
        Details('PDU  Byte '+Str(nbyte));                // Counter
       end;                                              // 
       errn:=PutHoldingRegisters(node,saddr,count,nbyte,Copy(dat,6));// Put holding registers
       if errn=modbus_er_OK then begin                   // Ok, data ready to encode
        ans:=Copy(dat,1,4);                              // Prepare answer
        ans:=modbus_encode_rtu(unid,func,ans);           // Encode answer
       end;                                              // Done
      end else errn:=modbus_er_ILLVAL;                   // Illegal data value (length)
     end else
     errn:=modbus_er_ILLFUN;                             // Illegal function
     if errn<>modbus_er_OK then begin                    // Exception answer: (func or $80) + ErrorCode
      ans:=modbus_encode_rtu(unid,iOr(func,modbus_er_EXCEPT),Dump(Chr(errn)));
      modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Ex.tag,modbus.ErrorCode.Ex,modbus_port_alias(port)
                 +' ! EXCEPTION node '+Str(node)+' addr '+Str(unid)
                 +' func $'+HexB(iOr(func,modbus_er_EXCEPT))+' code $'+HexB(errn));
     end;
    end;
   end;
  end;
  modbus_rtu_ans:=ans;
  ans:=''; dat:='';
 end;
 //
 // Answer on MODBUS ASCII request
 //
 function modbus_ascii_ans(port:Integer; req:String):String;
 var ans,dat:String; hdr:array[1..5] of Char;
     pdulen,unid,func,node,saddr,count,nbyte,datav,errn:Integer;
 begin
  ans:=''; dat:='';
  if modbus_port_ok(port) then                             // Check port number
  if (modbus.Port[port].Prot = modbus_pr_ASCII) then begin // Check protocol ASCII
   pdulen:=modbus_decode_ascii(req,unid,func,dat);         // Decode message
   if pdulen>0 then begin                                  // Looks good?
    if DebugFlagEnabled(dfDetails) then begin              // Print details:
     Details('ADU Unit '+Str(unid));                       // ADU Unit Id
     Details('PDU Leng '+Str(pdulen));                     // PDU length
     Details('PDU Func '+Str(func));                       // PDU Function Id
     Details('PDU Data '+Hex_Encode(dat));                 // PDU Data
    end;                                                   //
    if modbus_addr_ok(unid) then                           // If valid unit ID address
    for node:=1 to MaxNodeNum do                           // Scan all connected nodes
    if Length(ans)=0 then                                  // Skip dublicates
    if port = modbus.Node[node].Port then                  // Node connected to this port?
    if unid = modbus.Node[node].Addr then begin            // Answer only if good address found
     errn:=modbus_er_OK;                                   // Clear error number first
     if func = modbus_fn_ReadCS then begin                 // Read coil statuses:
      if Length(dat)>=4 then begin                         // Expected 4+ byte data
       hdr:=dat;                                           // Copy ADU header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
       end;                                                // 
       errn:=GetCoilStatuses(node,saddr,count,ans);        // Get coil statuses
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                         // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_ReadIS then begin                 // Read input statuses:
      if Length(dat)>=4 then begin                         // Expected 4+ byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
       end;                                                // 
       errn:=GetInputStatuses(node,saddr,count,ans);       // Get input statuses
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                         // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_ReadHR then begin                 // Read holding registers:
      if Length(dat)>=4 then begin                         // Expected 4+ byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
       end;                                                // 
       errn:=GetHoldingRegisters(node,saddr,count,ans);    // Get holding registers
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                         // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_ReadIR then begin                 // Read input registers:
      if Length(dat)>=4 then begin                         // Expected 4+ byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
       end;                                                // 
       errn:=GetInputRegisters(node,saddr,count,ans);      // Get input registers
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Chr(Length(ans))+ans;                         // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_WritSC then begin                 // Write single coil status:
      if Length(dat)>=4 then begin                         // Expected 4 byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Data value
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Data '+Str(datav));                  // Data
       end;                                                // 
       if(datav<>0) and (datav<>modbus_0xFF00)             // Check data value
       then errn:=modbus_er_ILLVAL                         // Must be 0 or $FF00
       else errn:=PutCoilStatuses(node,saddr,1,1,Dump(Chr(Ord(datav<>0)))); // Put coil status
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Copy(dat,1,4);                                // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_WritSR then begin                 // Write single holding register:
      if Length(dat)>=4 then begin                         // Expected 4 byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       datav:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Data value
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Data '+Str(datav));                  // Data
       end;                                                // 
       errn:=PutHoldingRegisters(node,saddr,1,2,Copy(dat,3));// Put holding register
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Copy(dat,1,4);                                // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else                                              //
     if func = modbus_fn_WritMC then begin                 // Write multiple coil statuses:
      if Length(dat)>=5 then begin                         // Expected 5+ byte data
       hdr:=dat;                                           // Copy ADU header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       nbyte:=Ord(hdr[5]);                                 // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
        Details('PDU  Byte '+Str(nbyte));                  // Counter
       end;                                                // 
       errn:=PutCoilStatuses(node,saddr,count,nbyte,Copy(dat,6));// Put coil statuses
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Copy(dat,1,4);                                // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else
     if func = modbus_fn_WritMR then begin                 // Write multiple holding registers:
      if Length(dat)>=5 then begin                         // Expected 5+ byte data
       hdr:=dat;                                           // Decode data header
       saddr:=Ord(hdr[2])+256*Ord(hdr[1]);                 // Start registers address
       count:=Ord(hdr[4])+256*Ord(hdr[3]);                 // Number of registers
       nbyte:=Ord(hdr[5]);                                 // Number of bytes
       if DebugFlagEnabled(dfDetails) then begin           // Print details:
        Details('PDU  Addr '+Str(saddr));                  // Address
        Details('PDU  Numb '+Str(count));                  // Counter
        Details('PDU  Byte '+Str(nbyte));                  // Counter
       end;                                                // 
       errn:=PutHoldingRegisters(node,saddr,count,nbyte,Copy(dat,6));// Put holding registers
       if errn=modbus_er_OK then begin                     // Ok, data ready to encode
        ans:=Copy(dat,1,4);                                // Prepare answer
        ans:=modbus_encode_ascii(unid,func,ans);           // Encode answer
       end;                                                // Done
      end else errn:=modbus_er_ILLVAL;                     // Illegal data value (length)
     end else
     errn:=modbus_er_ILLFUN;                               // Illegal function
     if errn<>modbus_er_OK then begin                      // Exception answer: (func or $80) + ErrorCode
      ans:=modbus_encode_ascii(unid,iOr(func,modbus_er_EXCEPT),Dump(Chr(errn)));
      modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Ex.tag,modbus.ErrorCode.Ex,modbus_port_alias(port)
                 +' ! EXCEPTION node '+Str(node)+' addr '+Str(unid)
                 +' func $'+HexB(iOr(func,modbus_er_EXCEPT))+' code $'+HexB(errn));
     end;
    end;
   end;
  end;
  modbus_ascii_ans:=ans;
  ans:=''; dat:='';
 end;
 //
 // Poll modbus pipe
 //
 procedure modbus_poll;
 var port,prot,pipe,conn,sid:Integer; ms,dt:Real;
  procedure CalcRate(rtag:Integer; ctag:Integer; var cval:Real; dt:Real);
  var c:Real;
  begin
   if dt>0 then begin
    c:=modbus_get_tag(ctag,0);
    bNul(modbus_set_tag(rtag,1000*(c-cval)/dt));
    cval:=c;
   end;
  end;
 begin
  //
  // Poll modbus ports
  //
  for port:=1 to MaxPortNum do begin
   pipe:=modbus.Port[port].Pipe;
   if pipe<>0 then begin
    prot:=modbus.Port[port].Prot;
    //
    // MODBUS TCP/IP server. Any number of clients supported.
    //
    if prot=modbus_pr_IP then begin
     modbus.Port[port].Req:='';
     modbus.Port[port].Ans:='';
     if pipe_connected(pipe)>0 then
     for conn:=0 to pipe_count(pipe)-1 do begin
      sid:=pipe_stream(pipe,conn);
      if pipe_connected(sid)>0 then
      if pipe_rxcount(sid)>0 then begin
       modbus.Port[port].Req:=pipe_recv(sid,MaxBuffer);
       if Length(modbus.Port[port].Req)>0 then begin
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Rx.tag,1));
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,Length(modbus.Port[port].Req)));
        if DebugFlagEnabled(dfViewImp) then ViewImp(modbus_port_alias(port)+' < '+Hex_Encode(modbus.Port[port].Req));
        modbus.Port[port].Ans:=modbus_ip_ans(port,modbus.Port[port].Req);
        if Length(modbus.Port[port].Ans)>0 then begin
         if DebugFlagEnabled(dfViewExp) then ViewExp(modbus_port_alias(port)+' > '+Hex_Encode(modbus.Port[port].Ans));
         if pipe_send(sid,modbus.Port[port].Ans)=Length(modbus.Port[port].Ans) then begin
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Tx.tag,1));
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Tx.tag,Length(modbus.Port[port].Ans)));
          if DebugFlagEnabled(dfDetails)
          then Details('Sent '+Str(Length(modbus.Port[port].Ans))+' bytes to '+pipe_ctrl(sid,'PeerIp'));
         end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Tx.tag,modbus.ErrorCode.Tx,
                                  modbus_port_alias(port)+' ! Could not send answer !');
        end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Rx.tag,modbus.ErrorCode.Rx,
                                 modbus_port_alias(port)+' ! Bad IP request, no answer !');
        modbus.Port[port].Req:='';
        modbus.Port[port].Ans:='';
       end;
      end;
     end;
    end;
    //
    // MODBUS RTU server. Up to 16 clients supported.
    //
    if prot=modbus_pr_RTU then begin
     ms:=mSecNow;
     modbus.Port[port].Req:='';
     modbus.Port[port].Ans:='';
     if pipe_connected(pipe)>0 then
     if ms>modbus.Port[port].Last.Poll then
     for conn:=0 to iMin(MaxConnNum,pipe_count(pipe)-1) do begin
      sid:=pipe_stream(pipe,conn);
      if pipe_connected(sid)>0 then
      if pipe_rxcount(sid)>0 then begin
        modbus.Port[port].Buf[conn]:=modbus.Port[port].Buf[conn]+pipe_recv(sid,MaxBuffer);
        if Length(modbus.Port[port].Buf[conn])>modbus_MaxRtuLen then begin
         modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Rx.tag,modbus.ErrorCode.Rx,
                         modbus_port_alias(port)+' ! Bad request, overflow !');
         bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,Length(modbus.Port[port].Buf[conn])));
         modbus.Port[port].Buf[conn]:='';
        end;
      end else begin
       modbus.Port[port].Req:=modbus.Port[port].Buf[conn];
       if Length(modbus.Port[port].Req)>0 then begin
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Rx.tag,1));
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,Length(modbus.Port[port].Req)));
        if DebugFlagEnabled(dfViewImp) then ViewImp(modbus_port_alias(port)+' < '+Hex_Encode(modbus.Port[port].Req));
        modbus.Port[port].Ans:=modbus_rtu_ans(port,modbus.Port[port].Req);
        if Length(modbus.Port[port].Ans)>0 then begin
         if DebugFlagEnabled(dfViewExp) then ViewExp(modbus_port_alias(port)+' > '+Hex_Encode(modbus.Port[port].Ans));
         if pipe_send(sid,modbus.Port[port].Ans)=Length(modbus.Port[port].Ans) then begin
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Tx.tag,1));
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Tx.tag,Length(modbus.Port[port].Ans)));
          if DebugFlagEnabled(dfDetails)
          then Details('Sent '+Str(Length(modbus.Port[port].Ans))+' bytes to port '+Str(port));
         end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Tx.tag,modbus.ErrorCode.Tx,
                                  modbus_port_alias(port)+' ! Could not send answer !');
        end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Rx.tag,modbus.ErrorCode.Rx,
                                 modbus_port_alias(port)+' ! Bad RTU request, no answer !');
        modbus.Port[port].Req:='';
        modbus.Port[port].Ans:='';
       end;
       modbus.Port[port].Buf[conn]:='';
      end;
     end;
     modbus.Port[port].Last.Poll:=ms;
    end;    
    //
    // MODBUS ASCII server. Up to 16 clients supported.
    //
    if prot=modbus_pr_ASCII then begin
     ms:=mSecNow;
     modbus.Port[port].Req:='';
     modbus.Port[port].Ans:='';
     if pipe_connected(pipe)>0 then
     if ms>modbus.Port[port].Last.Poll then
     for conn:=0 to iMin(MaxConnNum,pipe_count(pipe)-1) do begin
      sid:=pipe_stream(pipe,conn);
      if pipe_connected(sid)>0 then
      if pipe_readln(sid,modbus.Port[port].Req,modbus.Port[port].Buf[conn]) then begin
       if Length(modbus.Port[port].Req)>0 then begin
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Rx.tag,1));
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,Length(modbus.Port[port].Req)));
        if DebugFlagEnabled(dfViewImp) then ViewImp(modbus_port_alias(port)+' < '+modbus.Port[port].Req);
        modbus.Port[port].Ans:=modbus_ascii_ans(port,modbus.Port[port].Req);
        if Length(modbus.Port[port].Ans)>0 then begin
         if DebugFlagEnabled(dfViewExp) then ViewExp(modbus_port_alias(port)+' > '+Trim(modbus.Port[port].Ans));
         if pipe_send(sid,modbus.Port[port].Ans)=Length(modbus.Port[port].Ans) then begin
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Polls.Tx.tag,1));
          bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Tx.tag,Length(modbus.Port[port].Ans)));
          if DebugFlagEnabled(dfDetails)
          then Details('Sent '+Str(Length(modbus.Port[port].Ans))+' bytes to port '+Str(port));
         end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Tx.tag,modbus.ErrorCode.Tx,
                                  modbus_port_alias(port)+' ! Could not send answer !');
        end else modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Rx.tag,modbus.ErrorCode.Rx,
                                 modbus_port_alias(port)+' ! Bad ASC request, no answer !');
        modbus.Port[port].Req:='';
        modbus.Port[port].Ans:='';
       end;
      end else begin
       if Length(modbus.Port[port].Buf[conn])>modbus_MaxAscLen then begin
        modbus_fixerror(modbus.Port[port].Stat.Count.Bugs.Rx.tag,modbus.ErrorCode.Rx,
                        modbus_port_alias(port)+' ! Bad request, overflow !');
        bNul(modbus_inc_tag(modbus.Port[port].Stat.Count.Bytes.Rx.tag,Length(modbus.Port[port].Buf[conn])));
        modbus.Port[port].Buf[conn]:='';
       end;
      end;
     end;
     modbus.Port[port].Last.Poll:=ms;
    end;    
   end;
  end;
  //
  // Rate modbus ports
  //
  if SysTimer_Pulse(RatePeriod)>0 then begin
   ms:=mSecNow;
   for port:=1 to MaxPortNum do begin
    dt:=ms-modbus.Port[port].Last.Rate;
    CalcRate(modbus.Port[port].Stat.Rate.Polls.Rx.tag,
             modbus.Port[port].Stat.Count.Polls.Rx.tag,
             modbus.Port[port].Stat.Count.Polls.Rx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Bytes.Rx.tag,
             modbus.Port[port].Stat.Count.Bytes.Rx.tag,
             modbus.Port[port].Stat.Count.Bytes.Rx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Bugs.Rx.tag,
             modbus.Port[port].Stat.Count.Bugs.Rx.tag,
             modbus.Port[port].Stat.Count.Bugs.Rx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Polls.Tx.tag,
             modbus.Port[port].Stat.Count.Polls.Tx.tag,
             modbus.Port[port].Stat.Count.Polls.Tx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Bytes.Tx.tag,
             modbus.Port[port].Stat.Count.Bytes.Tx.tag,
             modbus.Port[port].Stat.Count.Bytes.Tx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Bugs.Tx.tag,
             modbus.Port[port].Stat.Count.Bugs.Tx.tag,
             modbus.Port[port].Stat.Count.Bugs.Tx.val,dt);
    CalcRate(modbus.Port[port].Stat.Rate.Bugs.Ex.tag,
             modbus.Port[port].Stat.Count.Bugs.Ex.tag,
             modbus.Port[port].Stat.Count.Bugs.Ex.val,dt);
    modbus.Port[port].Last.Rate:=ms;
   end;
  end;
 end;
 //
 // Test modbus routines.
 //
 procedure modbus_testing;
 var i,n,crc:Integer; s:String; r:Real;
 begin
  n:=1;
  s:=Hex_Decode('01030000000A');
  crc:=modbus_swap_int16(modbus_calc_crc(s));
  Success(HexW(crc)+' - should be $C5CD');
  //
  s:=Hex_Decode('01030000000A');
  crc:=modbus_swap_int16(modbus_calc_crc(s));
  Success(HexW(crc)+' - should be $C5CD');
  //
  s:=Hex_Decode('051000010001021388');
  for i:=1 to n do
  crc:=modbus_swap_int16(modbus_calc_crc(s));
  Success(HexW(crc)+' - should be $9817');
  s:=Hex_Decode('051000010001');
  for i:=1 to n do
  crc:=modbus_swap_int16(modbus_calc_crc(s));
  Success(HexW(crc)+' - should be $518D');
  s:=Hex_Decode('011000020001020016');
  for i:=1 to n do
  crc:=modbus_calc_lrc(s);
  Success(Hexb(crc)+'   - should be $D4');
  //
  n:=1;
  for i:=1 to n do
  s:=CharReverseStr('');
  Success(s+'   - should be ');
  for i:=1 to n do
  s:=CharReverseStr(dump('0'));
  Success(s+'   - should be 0');
  for i:=1 to n do
  s:=CharReverseStr('01');
  Success(s+'   - should be 10');
  for i:=1 to n do
  s:=CharReverseStr('012');
  Success(s+'   - should be 210');
  for i:=1 to n do
  s:=CharReverseStr('0123');
  Success(s+'   - should be 3210');
  for i:=1 to n do
  s:=CharReverseStr('01234');
  Success(s+'   - should be 43210');
  for i:=1 to n do
  s:=CharReverseStr('abcdefghijklmnopqrstuvwxyz');
  Success(s+'   - should be zyxwvutsrqponmlkjihgfedcba');
  //
  r:=3.14;
  s:=Hex_Encode(dumpf(r)); // C3F54840
  Success(Str(r)+' '+s+' '+Str(dump2f(Hex_Decode(s))));
  //
  r:=12345;
  Success(Str(r)+' int16 '+Str(modbus_take_typ(mbt_int16,modbus_dump_typ(mbt_int16,Round(r)))));
  r:=1234567890;
  Success(Str(r)+' int32 '+Str(modbus_take_typ(mbt_int32,modbus_dump_typ(mbt_int32,Round(r)))));
  r:=123.456;
  Success(Str(r)+' float '+Str(modbus_take_typ(mbt_float,modbus_dump_typ(mbt_float,r))));
  r:=12345.6789;
  Success(Str(r)+' double '+Str(modbus_take_typ(mbt_double,modbus_dump_typ(mbt_double,r))));
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  modbus_clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  InitNetLibrary;
  modbus_init;
  RunStartupScript;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  FreeNetLibrary;
  modbus_free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  modbus_poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; node,port,addr,typ,src,ref,dst,out:Integer;
  procedure ExtractAddrTypSrcRef(arg:String; Base,MaxNum:Integer);
  begin
   addr:=iValDef(ExtractWord(1,arg),-1);
   if (addr>=0) then addr:=addr-Base;
   if (addr<0) or (addr>=MaxNum) then addr:=-1;
   typ:=WordIndex(UpCaseStr(ExtractWord(2,arg)),mbt_LIST+mbt_LIST2+mbt_LIST3+mbt_LIST4+mbt_LIST5);
   src:=WordIndex(UpCaseStr(ExtractWord(3,arg)),mbs_LIST);
   if src=mbs_AI then begin ref:=iValDef(ExtractWord(4,arg),-1); if RefAi(ref)=0 then ref:=-1; end else
   if src=mbs_DI then begin ref:=iValDef(ExtractWord(4,arg),-1); if RefDi(ref)=0 then ref:=-1; end else
   if src=mbs_TAG then begin ref:=FindTag(ExtractWord(4,arg)); if ref=0 then ref:=-1; end else ref:=-1; 
   dst:=WordIndex(UpCaseStr(ExtractWord(5,arg)),mbs_LIST);
   if dst=mbs_AO then begin out:=iValDef(ExtractWord(6,arg),-1); if RefAo(out)=0 then out:=-1; end else
   if dst=mbs_DO then begin out:=iValDef(ExtractWord(6,arg),-1); if RefDo(out)=0 then out:=-1; end else
   if dst=mbs_TAG then begin out:=FindTag(ExtractWord(6,arg)); if out=0 then out:=-1; end else out:=-1;
   //Success('addr '+Str(addr)+' typ '+Str(typ)+' src '+Str(src)+' ref '+Str(ref)+' dst '+Str(dst)+' out '+Str(out));
  end;
  procedure PrintTableOk(cmd:String; node,Base:Integer);
  var msg:String;
  begin
   msg:=cmd+' '+Str(node)+' '+Str(addr+Base);
   msg:=msg+' '+ExtractWord(typ,mbt_LIST+mbt_LIST2+mbt_LIST3+mbt_LIST4+mbt_LIST5);
   msg:=msg+' '+ExtractWord(src,mbs_LIST);
   if src=mbs_TAG then msg:=msg+' '+NameTag(ref) else msg:=msg+' '+Str(ref);
   if dst>0 then msg:=msg+' '+ExtractWord(dst,mbs_LIST);
   if dst>0 then if dst=mbs_TAG then msg:=msg+' '+NameTag(out) else msg:=msg+' '+Str(out);
   Success(msg);
   msg:='';
  end;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   @Reset
   }
   if IsSameText(cmd,'@Reset') then begin
    modbus_reset;
    Success(cmd);
    Data:='';
   end else
   {
   @Port 1 ip  tcp port 502 server 16
   @Port 2 rtu com port 4 baudrate 9600 parity even databits 8 stopbits 1
   @Port PortNum[1..16] Protocol[IP,RTU,ASCII] Description[see pipe_init()]
   }
   if IsSameText(cmd,'@Port') then begin
    port:=modbus_open(Trim(arg));
    if modbus_port_ok(port)
    then Success(cmd+' '+Str(port)+' '+modbus_prot_name(modbus.Port[port].Prot)
                +' '+Trim(modbus.Port[port].Decl))
    else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortErrorCounters 1 Modbus.BugsCount.Rx Modbus.BugsCount.Tx Modbus.BugsCount.Ex
   }
   if IsSameText(cmd,'@PortErrorCounters') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Count.Bugs.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Count.Bugs.Tx.tag:=FindTag(ExtractWord(3,arg));
     modbus.Port[port].Stat.Count.Bugs.Ex.tag:=FindTag(ExtractWord(4,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Tx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Ex.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortErrorRates 1 Modbus.ErrorRates.Rx Modbus.ErrorRates.Tx Modbus.ErrorRates.Ex
   }
   if IsSameText(cmd,'@PortErrorRates') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Rate.Bugs.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Rate.Bugs.Tx.tag:=FindTag(ExtractWord(3,arg));
     modbus.Port[port].Stat.Rate.Bugs.Ex.tag:=FindTag(ExtractWord(4,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Tx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Ex.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortByteCounters 1 Modbus.Bytes.Rx Modbus.Bytes.Tx
   }
   if IsSameText(cmd,'@PortByteCounters') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Count.Bytes.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Count.Bytes.Tx.tag:=FindTag(ExtractWord(3,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bytes.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bytes.Tx.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortByteRates 1 Modbus.ByteRates.Rx Modbus.ByteRates.Tx
   }
   if IsSameText(cmd,'@PortByteRates') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Rate.Bytes.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Rate.Bytes.Tx.tag:=FindTag(ExtractWord(3,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bytes.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bytes.Tx.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortPollCounters 1 Modbus.Polls.Rx Modbus.Polls.Tx
   }
   if IsSameText(cmd,'@PortPollCounters') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Count.Polls.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Count.Polls.Tx.tag:=FindTag(ExtractWord(3,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Polls.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Polls.Tx.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @PortPollRates 1 Modbus.PollRates.Rx Modbus.PollRates.Tx
   }
   if IsSameText(cmd,'@PortPollRates') then begin
    port:=Val(ExtractWord(1,arg));
    if modbus_port_ok(port) then begin
     modbus.Port[port].Stat.Rate.Polls.Rx.tag:=FindTag(ExtractWord(2,arg));
     modbus.Port[port].Stat.Rate.Polls.Tx.tag:=FindTag(ExtractWord(3,arg));
     Success(cmd+' '+Str(port)+' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Polls.Rx.tag),'"')
                              +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Polls.Tx.tag),'"'))
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @Node 1 2 3
   @Node n p u
   n - node number, 1..32
   p - port where node connected, 1..16
   u - device unit ID address, 1..247
   }
   if IsSameText(cmd,'@Node') then begin
    node:=Val(ExtractWord(1,arg));
    port:=Val(ExtractWord(2,arg));
    addr:=Val(ExtractWord(3,arg));
    if modbus_node_ok(node) and modbus_port_ok(port) and modbus_addr_ok(addr) then begin
     modbus.Node[node].Port:=port;
     modbus.Node[node].Addr:=addr;
     Success(cmd+' '+Str(node)+' '+Str(modbus.Node[node].Port)+' '+Str(modbus.Node[node].Addr));
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @CoilStatusBase 1 1
   }
   if IsSameText(cmd,'@CoilStatusBase') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     modbus.Node[node].Coil.Base:=iValDef(ExtractWord(2,arg),modbus.Node[node].Coil.Base);
     Success(cmd+' '+Str(node)+' '+Str(modbus.Node[node].Coil.Base));
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @InputStatusBase 1 10001
   }
   if IsSameText(cmd,'@InputStatusBase') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     modbus.Node[node].InSt.Base:=iValDef(ExtractWord(2,arg),modbus.Node[node].InSt.Base);
     Success(cmd+' '+Str(node)+' '+Str(modbus.Node[node].InSt.Base));
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @InputRegisterBase 1 30001
   }
   if IsSameText(cmd,'@InputRegisterBase') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     modbus.Node[node].IReg.Base:=iValDef(ExtractWord(2,arg),modbus.Node[node].IReg.Base);
     Success(cmd+' '+Str(node)+' '+Str(modbus.Node[node].IReg.Base));
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @HoldingRegisterBase 1 40001
   }
   if IsSameText(cmd,'@HoldingRegisterBase') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     modbus.Node[node].HReg.Base:=iValDef(ExtractWord(2,arg),modbus.Node[node].HReg.Base);
     Success(cmd+' '+Str(node)+' '+Str(modbus.Node[node].HReg.Base));
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @CoilStatus 1 bit0 DigitalInput 0
   }
   if IsSameText(cmd,'@CoilStatus') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     ExtractAddrTypSrcRef(SkipWords(1,arg),modbus.Node[node].Coil.Base,MaxNumCoil);
     if (addr>=0) and (typ>0) and (src>0) and (ref>=0) then begin
       PrintTableOk(cmd,node,modbus.Node[node].Coil.Base);
       modbus.Node[node].Coil.Typ[addr]:=typ;
       modbus.Node[node].Coil.Src[addr]:=src;
       modbus.Node[node].Coil.Ref[addr]:=ref;
       modbus.Node[node].Coil.Dst[addr]:=dst;
       modbus.Node[node].Coil.Out[addr]:=out;
     end else Trouble(cmd+' '+arg+' - invalid table');
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @InputStatus 10001 bit0 DigitalInput 0
   }
   if IsSameText(cmd,'@InputStatus') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     ExtractAddrTypSrcRef(SkipWords(1,arg),modbus.Node[node].InSt.Base,MaxNumInSt);
     if (addr>=0) and (typ>0) and (src>0) and (ref>=0) then begin
       PrintTableOk(cmd,node,modbus.Node[node].InSt.Base);
       modbus.Node[node].InSt.Typ[addr]:=typ;
       modbus.Node[node].InSt.Src[addr]:=src;
       modbus.Node[node].InSt.Ref[addr]:=ref;
     end else Trouble(cmd+' '+arg+' - invalid table');
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @InputRegister 30001 float DigitalInput 0
   }
   if IsSameText(cmd,'@InputRegister') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     ExtractAddrTypSrcRef(SkipWords(1,arg),modbus.Node[node].IReg.Base,MaxNumIReg);
     if (addr>=0) and (typ>0) and (src>0) and (ref>=0) then begin
       PrintTableOk(cmd,node,modbus.Node[node].IReg.Base);
       modbus.Node[node].IReg.Typ[addr]:=typ;
       modbus.Node[node].IReg.Src[addr]:=src;
       modbus.Node[node].IReg.Ref[addr]:=ref;
     end else Trouble(cmd+' '+arg+' - invalid table');
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @HoldingRegister 40001 float AnalogInput 0
   }
   if IsSameText(cmd,'@HoldingRegister') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     ExtractAddrTypSrcRef(SkipWords(1,arg),modbus.Node[node].HReg.Base,MaxNumHReg);
     if (addr>=0) and (typ>0) and (src>0) and (ref>=0) then begin
      PrintTableOk(cmd,node,modbus.node[Node].HReg.Base);
      modbus.Node[node].HReg.Typ[addr]:=typ;
      modbus.Node[node].HReg.Src[addr]:=src;
      modbus.Node[node].HReg.Ref[addr]:=ref;
      modbus.Node[node].HReg.Dst[addr]:=dst;
      modbus.Node[node].HReg.Out[addr]:=out;
     end else Trouble(cmd+' '+arg+' - invalid table');
    end else Trouble('Invalid command arguments: '+Trim(Data));
    Data:='';
   end else
   {
   @View
   @View 1
   }
   if IsSameText(cmd,'@View') then begin
    node:=Val(ExtractWord(1,arg));
    if modbus_node_ok(node) then begin
     Success('@Node '+Str(node)+' '+Str(modbus.Node[node].Port)+' '+Str(modbus.Node[node].Addr));
     Success('@CoilStatusBase '+Str(node)+' '+Str(modbus.Node[node].Coil.Base));
     Success('@InputStatusBase '+Str(node)+' '+Str(modbus.Node[node].InSt.Base));
     Success('@InputRegisterBase '+Str(node)+' '+Str(modbus.Node[node].IReg.Base));
     Success('@HoldingRegisterBase '+Str(node)+' '+Str(modbus.Node[node].HReg.Base));
     for addr:=0 to MaxNumCoil do begin
      typ:=modbus.Node[node].Coil.Typ[addr];
      src:=modbus.Node[node].Coil.Src[addr];
      ref:=modbus.Node[node].Coil.Ref[addr];
      dst:=modbus.Node[node].Coil.Dst[addr];
      out:=modbus.Node[node].Coil.Out[addr];
      if (typ>0) and (src>0) and (ref>=0)
      then PrintTableOk('@CoilStatus',node,modbus.Node[node].Coil.Base);
     end;
     for addr:=0 to MaxNumInSt do begin
      typ:=modbus.Node[node].InSt.Typ[addr];
      src:=modbus.Node[node].InSt.Src[addr];
      ref:=modbus.Node[node].InSt.Ref[addr];
      dst:=0;
      out:=0;
      if (typ>0) and (src>0) and (ref>=0)
      then PrintTableOk('@InputStatus',node,modbus.Node[node].InSt.Base);
     end;
     for addr:=0 to MaxNumIReg do begin
      typ:=modbus.Node[node].IReg.Typ[addr];
      src:=modbus.Node[node].IReg.Src[addr];
      ref:=modbus.Node[node].IReg.Ref[addr];
      dst:=0;
      out:=0;
      if (typ>0) and (src>0) and (ref>=0)
      then PrintTableOk('@InputRegister',node,modbus.Node[node].IReg.Base);
     end;
     for addr:=0 to MaxNumHReg do begin
      typ:=modbus.Node[node].HReg.Typ[addr];
      src:=modbus.Node[node].HReg.Src[addr];
      ref:=modbus.Node[node].HReg.Ref[addr];
      dst:=modbus.Node[node].HReg.Dst[addr];
      out:=modbus.Node[node].HReg.Out[addr];
      if (typ>0) and (src>0) and (ref>=0)
      then PrintTableOk('@HoldingRegister',node,modbus.Node[node].HReg.Base);
     end;
    end else begin
     for port:=1 to MaxPortNum do
     if modbus_prot_ok(modbus.Port[port].Prot) then
     if not IsEmptyStr(modbus.Port[port].Decl) then begin
      Success('@Port '+Str(port)+' '+modbus_prot_name(modbus.Port[port].Prot)
             +' '+Trim(modbus.Port[port].Decl)
             +' : '+Str(pipe_connected(modbus.Port[port].Pipe))+' connection(s)');
      Success('@PortErrorCounters '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Tx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bugs.Ex.tag),'"'));
      Success('@PortErrorRates '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Tx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bugs.Ex.tag),'"'));
      Success('@PortByteCounters '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bytes.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Bytes.Tx.tag),'"'));
      Success('@PortByteRates '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bytes.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Bytes.Tx.tag),'"'));
      Success('@PortPollCounters '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Polls.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Count.Polls.Tx.tag),'"'));
      Success('@PortPollRates '+Str(port)
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Polls.Rx.tag),'"')
             +' '+StrAddQuotes(NameTag(modbus.Port[port].Stat.Rate.Polls.Tx.tag),'"'));
     end;
     for node:=1 to MaxNodeNum do
     if modbus_port_ok(modbus.Node[node].Port) then
     if modbus_addr_ok(modbus.Node[node].Addr) then
     Success('@Node '+Str(node)+' '+Str(modbus.Node[node].Port)+' '+Str(modbus.Node[node].Addr));
    end;
    Data:='';
   end else
   {
   @ZeroPortCounters 1
   }
   if IsSameText(cmd,'@ZeroPortCounters') then begin
    modbus_zero_stat(Val(Trim(arg)));
    Data:='';
   end else
   {
   @ModBusPoll demo
   }
   if IsSameText(cmd,'@ModBusPoll') then begin
    rNul(Eval('@system @async @run -hide '+AdaptExeFileName('modbuspoll.bat')+' '+Trim(arg)));
    Data:='';
   end else
   {
   @ModScan32 demo
   }
   if IsSameText(cmd,'@ModScan32') then begin
    rNul(Eval('@system @async @run -hide '+AdaptExeFileName('modscan32.bat')+' '+Trim(arg)));
    Data:='';
   end else
   {
   @ModBusHelp
   }
   if IsSameText(cmd,'@ModBusHelp') then begin
    rNul(Eval('@system @async @run WebBrowser '+ForceExtension(ProgramSourcePas,'.htm')));
    rNul(Eval('@system @async @run -hide '+AdaptExeFileName('modbushelp.bat')+' '+Trim(arg)));
    Data:='';
   end else
   {
   @ModBusTesting
   }
   if IsSameText(cmd,'@ModBusTesting') then begin
    modbus_testing;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
