#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to write event message to SysLog journal.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

readonly syslog="$CRW_DAQ_VAR_TMP_DIR/sys.log";

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [[ -z "$CRW_DAQ_VAR_TMP_DIR" ]]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};

function datetime_prompt(){
 date "+%Y.%m.%d-%H:%M:%S.%3N => ";
};

function match_event_pattern(){
 local msg="$*";
 local pat='^(\d+|DEBUG|INFO|WARN|ERROR|FATAL|OFF)(/\w+)?:\s+([[:graph:]]+) - ';
 if [[ ${msg^^} =~ $pat ]]; then return 0; else return 1; fi;
};

function syslog_note(){
 local lines=""; local eol=$'\n';
 local prompt="$(datetime_prompt)";
 if [[ -d "$CRW_DAQ_VAR_TMP_DIR" ]]; then
  while [[ -n "$1" ]]; do
   local line="$1"; shift;
   if match_event_pattern "$line"; then
    lines+="$prompt$line$eol";
   else
    fatal 1 "Error: arguments is not match expected pattern.";
   fi;
  done;
 else
  fatal 1 "Error: invalid log directory.";
 fi;
 if [[ -n "$lines" ]]; then
  echo -n "$lines" | tee -a "$syslog";
 fi;
};

function syslog_stdin(){
 local line="";
 local prompt="$(datetime_prompt)";
 if [[ -d "$CRW_DAQ_VAR_TMP_DIR" ]]; then
  while IFS="" read -r line || [[ -n "$line" ]]; do
   if match_event_pattern "$line"; then
    echo "${prompt}$line";
   else
    fatal 1 "Error: arguments is not match expected pattern.";
   fi;
  done;
 else
  fatal 1 "Error: invalid log directory.";
 fi;
};

function main(){
 check_environ;
 if [[ -n "$1" ]]; then
  syslog_note "$@";
  return;
 fi;
 if [[ -t 0 ]]; then
  fatal 1 "Error: no input/arguments.";
 else
  syslog_stdin | tee -a "$syslog";
 fi;
};

main "$@";

#############
# END OF FILE
#############
