 {
 ************************************************************************
 VMCD Modbus Driver.
 ************************************************************************
 Next text uses by @Help command. Do not remove it.
 ************************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Modbus.XXX    - response come from &ModbusProxy server:
| @Modbus.Reply    ref cid tim port uid fid $$ans
| @Modbus.Refuse   ref cid tim port uid fid   msg
| @Modbus.Timeout  ref cid tim port uid fid $$req
|  ref  - sender device reference, expected &ModbusProxy.
|  cid  - command id, any user defined custom data.
|  tim  - request-response time, ms.
|  port - logical port on &ModbusProxy, expected 1..16.
|  uid  - unit id, i.e.MODBUS address, expected 1..247.
|  fid  - function id, expected 1,2,3,4,5,6,15,16.
|  ans  - answer  (response), in HEX_ENCODE format.
|  req  - request (original), in HEX_ENCODE format.
|  msg  - plain text message to explain reason of Refuse.
| @PARAM.XX           - write parameter, i.e. send to MODBUS
| @PARAM.WRITEADDRESS - Запись нового адреса в EEPROM
| @PARAM.ADDRESS n    - Установка адреса n устройства
| @PARAM.BAUDRATE n   - Установка скорости передачи данных и контроля четности (см. таблицу в инструкции)
| @ENABLE.HIGHVOLTAGE - Включение высокого напряжения
| @Edit PARAM.XX      - edit parameter and write it
| @AssignTag t v      - Assign tag t to value v
| @DimTagUpdate       - Update tag from DIM
|************************************************************************
[]

 ************************************************************************
  [DeviceList]
  &VMCD.DRIVER = device software program
  [&VMCD.DRIVER]
  Comment = Modbus RTU Driver for VMCD controller.
  InquiryPeriod  = 0
  DevicePolling  = 10, tpNormal
  ProgramSource  = ..\DaqPas\VMCD_drv.pas
  DigitalFifo    = 1024
  AnalogFifo     = 1024
  DebugFlags     = 3
  OpenConsole    = 2
  Simulator      = 0
  ModbusPort     = 1
  ModbusUnitId   = 10
  ModbusTimeout  = 250
  ModbusDeadline = 60000
  ModbusPolling  = 100
  DelayOnStart   = 1000
  tagPrefix      = VMCD.D1
  ...etc...
 ************************************************************************
 }
program VMCD;
const
 {-------------------------------}{ Declare uses program constants:     }
 {$I _con_StdLibrary}             { Include all Standard constants,     }
 {-------------------------------}{ And add User defined constants:     }
 {$I _con_NetLibrary}             { NetLibrary constants                }
 dfStatist        = 32;           { DebugFlags - polling statistics     }
 MaxCmdNum        = 8;            { Maximal number of commands:         }
 cm_ReadPressure  = 1;            {  (reg. 1003)                        }
 cm_ReadHighVolt  = 2;            {  (reg. 999)                         }
 cm_ReadUARTPara  = 3;            {  (reg. 999)                         }
 cm_WriteAddress  = 4;            {  (reg. 1003)                        }
 cm_WriteHighVolt = 5;            {  (reg. 1004)                        }
 cm_SetAddress    = 6;            {  (reg. 999)                         }
 cm_WriteBaudrate = 7;            {  (reg. 1000)                        }
 ao_PRESSURE      = 0;            {  (reg. 1010)                        }
 ao_HV            = 1;            {  (reg. 999)                         }
 ao_ADDRESS       = 2;            {  (reg. 999)                         }
 ao_BAUDRATE      = 3;            {  (reg. 1000)                        }
 do_STATE_INDIC   = 0;            { Status indicator                    }
 do_POLLRATERX    = 1;            {  Poll rate Rx                       }
 do_POLLRATETX    = 2;            {  Poll rate Tx                       }
 do_POLLRATEEX    = 3;            {  Poll rate Ex                       }
 do_ERRORCOUNT    = 4;            {  Error counter                      }
 do_POLLSUMMRX    = 5;            {  Poll summ Rx                       }
 do_POLLSUMMTX    = 6;            {  Poll summ Tx                       }
 do_POLLSUMMEX    = 7;            {  Poll summ Ex                       }
 SwapModeFloat    = 1;            { Byte swap mode for float values     }
 SwapModeInt16    = 2;            { Byte swap mode for integers         }
 SwapModeInt32    = 3;            { Byte swap mode for integers         }
 st_NORMAL        = 0;            { Status: Green                       }
 st_TIMEOUT       = 1;            {  Yellow timeout                     }
 st_REFUSE        = 2;            {  Red refuse                         }
 st_SIMULAT       = 3;            {  Simulator mode                     }
 st_DISABLE       = 4;            {  Polling disabled                   }

type
 {-------------------------------}{ Declare uses program types:         }
 {$I _typ_StdLibrary}             { Include all Standard types,         }
 {-------------------------------}{ And add User defined types:         }

var
 {-------------------------------}{ Declare uses program variables:     }
 {$I _var_StdLibrary}             { Include all Standard variables,     }
 {-------------------------------}{ And add User defined variables:     }
 {$I _var_NetLibrary}             { NetLibrary variables                }
 VMCD           : record          { VMCD Driver data record             }
  Simulator     : Boolean;        { Simulator or Driver mode            }
  Modbus        : record          { Modbus data record                  }
   Port         : Integer;        { Logical Port on &ModbusProxy        }
   UnitId       : Integer;        { Modbus unit id                      }
   Timeout      : Integer;        { Modbus timeout, ms                  }
   Polling      : Integer;        { Modbus polling period, ms           }
   Deadline     : Integer;        { Modbus deadline time, ms            }
   DelayOnStart : Integer;        { Command cycle delay on start        }
   Cmd          : record          { Command cycle data record           }
    Num         : Integer;        { Current running command number      }
    Enabled     : array [1..MaxCmdNum] of Boolean; { Enable polling     }
    FuncId      : array [1..MaxCmdNum] of Integer; { Modbus Function Id }
    SAddr       : array [1..MaxCmdNum] of Integer; { Start Address      }
    Quant       : array [1..MaxCmdNum] of Integer; { Quantity of data   }
    OpData      : array [1..MaxCmdNum] of Real;    { Operation data     }
    OpBuff      : array [1..MaxCmdNum] of Real;    { Operation buffer   }
   end;                           {                                     }
   Poll         : record          { Polling transaction data record     }
    ref         : Integer;        { Last sent device reference          }
    cid         : Integer;        { Last sent command id                }
    tim         : Real;           { Last polling time, ms               }
    port        : Integer;        { Last polling port                   }
    uid         : Integer;        { Last polling unit id                }
    fid         : Integer;        { Last sent function id               }
    saddr       : Integer;        { Last sent start address             }
    quant       : Integer;        { Last sent quantity of registers     }
    dat         : String;         { Last sent PDU=(fid+dat) data        }
    Summ,                         { Poll summ counters                  }
    Rate        : record          { Poll rate on last second            }
     Rx         : Real;           { Rx poll summ/rate - receiver        }
     Tx         : Real;           { Tx poll summ/rate - transmitter     }
     Ex         : Real;           { Ex poll summ/rate - errors          }
    end;                          {                                     }
   end;                           {                                     }
  end;                            {                                     }
  POLL          : record          { Poll parameters:                    }
   ENABLE       : TTagRef;        { Poll enable flag                    }
  end;                            {                                     }
  PARAM         : record          { Device parameters:                  }
   PRESSURE     : TTagRef;        {                                     }
   STATE        : TTagRef;        { Status indicator                    }
   HV           : TTagRef;        { High Voltage                        }
   ADDRESS      : TTagRef;        { Device Address                      }
   BAUDRATE     : TTagRef;        { Device Baudrate                     }
  end;                            {                                     }
  SIM           : record          { Simulator data:                     }
   PRESSURE     : Real;           {                                     }
   STATE        : Integer;        { Status indicator                    }
   HV           : Integer;        { High Voltage                        }
   ADDRESS      : Integer;        { Device Address                      }
   BAUDRATE     : Integer;        { Device Baudrate                     }
  end;                            {                                     }
  POLLRATE      : record          {                                     }
   RX,TX,EX     : TTagRef;        {                                     }
  end;                            {                                     }
  POLLSUMM      : record          {                                     }
   RX,TX,EX     : TTagRef;        {                                     }
  end;                            {                                     }
  ERROR         : record          {                                     }
   COUNT        : TTagRef;        {                                     }
  end;                            {                                     }
 end;                             {                                     }
 cmd_mbReply      : Integer;      { Modbus commands:                    }
 cmd_mbPoll       : Integer;      {                                     }
 cmd_mbRefuse     : Integer;      {                                     }
 cmd_mbTimeout    : Integer;      {                                     }
 cmd_mbZeroPortCnt: Integer;      {                                     }
 cmd_mbClrSumm    : Integer;      {                                     }
 cmd_WriteAddress : Integer;      {                                     }
 cmd_EnableHighV  : Integer;      {                                     }
 cmd_parHighV     : Integer;      {                                     }
 cmd_parAddress   : Integer;      {                                     }
 cmd_parBaudrate  : Integer;      {                                     }
 cmd_Edit         : Integer;      { Common commands:                    }
 cmd_menuBaudrate : Integer;      {  Open baudrate menu                 }
 cmd_DimTagUpdate : Integer;      {  @DimTagUpdate                      }
 cmd_AssignTag    : Integer;      {  @AssignTag                         }
 {-------------------------------}{ Declare procedures & functions:     }
 {$I _fun_StdLibrary}             { Include all Standard functions,     }
 {-------------------------------}{ And add User defined functions:     }
 {$I _fun_NetLibrary}             { NetLibrary functions                }
 {
 Prefix for DIM @remote commands.
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 //
 // Xor bit on click (local version)
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Xor bit on click (remote version).
 }
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(tag)+' '+Str(iXor(iGetTag(tag),XorMask)));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // Command cycle routines.
 //
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsUsableCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsUsableCmdNum:=(VMCD.Modbus.Cmd.FuncId[Num]<>0)
  else IsUsableCmdNum:=False;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsEnabledCmdNum:=(VMCD.Modbus.Cmd.FuncId[Num]<>0) and VMCD.Modbus.Cmd.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer;
 begin
  i:=0;
  while (i<MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   if IsEnabledCmdNum(Num)
   then i:=MaxCmdNum
   else i:=i+1;
  end;
  NextEnabledCmdNum:=Num;
 end;
 procedure EnableCmdNum(Num:Integer; Enabled:Boolean);
 begin
  if IsUsableCmdNum(Num) then VMCD.Modbus.Cmd.Enabled[Num]:=Enabled;
 end;
 procedure HoldCmdOpData(Num:Integer; OpData:Real);
 begin
  if IsUsableCmdNum(Num) then begin
   VMCD.Modbus.Cmd.Enabled[Num]:=not IsNaN(OpData);
   VMCD.Modbus.Cmd.OpBuff[Num]:=OpData;
  end;
 end;
 procedure ApplyCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then
  if not IsNaN(VMCD.Modbus.Cmd.OpBuff[Num]) then begin
   VMCD.Modbus.Cmd.OpData[Num]:=VMCD.Modbus.Cmd.OpBuff[Num];
   VMCD.Modbus.Cmd.OpBuff[Num]:=_NaN;
  end;
 end;
 procedure ReleaseCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then HoldCmdOpData(Num,VMCD.Modbus.Cmd.OpBuff[Num]);
 end;
 function GetCmdOpData(Num:Integer):Real;
 begin
  if IsUsableCmdNum(Num)
  then GetCmdOpData:=VMCD.Modbus.Cmd.OpData[Num]
  else GetCmdOpData:=_NaN;
 end;
 procedure InitCmdItem(Num:Integer; Enabled:Boolean; FuncId,SAddr,Quant:Integer; OpData,OpBuff:Real);
 begin
  if IsValidCmdNum(Num) then begin
   VMCD.Modbus.Cmd.Enabled[Num]:=Enabled;
   VMCD.Modbus.Cmd.FuncId[Num]:=FuncId;
   VMCD.Modbus.Cmd.SAddr[Num]:=SAddr;
   VMCD.Modbus.Cmd.Quant[Num]:=Quant;
   VMCD.Modbus.Cmd.OpData[Num]:=OpData;
   VMCD.Modbus.Cmd.OpBuff[Num]:=OpBuff;
  end;
 end;
 //
 // Command table cleanup and initialization.
 //
 procedure ClearCmdTable;
 var Num:Integer;
 begin
  VMCD.Modbus.Cmd.Num:=0;
  for Num:=1 to MaxCmdNum do InitCmdItem(Num,False,0,0,0,0,_NaN);
 end;
 procedure InitCmdTable;
 begin
  ClearCmdTable;
  // Table of CommandId         Enabled FunctionId        SAddr Quant OpData OpBuff
  InitCmdItem(cm_ReadPressure,  True,   modbus_fn_ReadIR, 1003, 2,    0,     _NaN);
  InitCmdItem(cm_ReadHighVolt,  True,   modbus_fn_ReadCS, 999,  1,    0,     _NaN);
  InitCmdItem(cm_ReadUARTPara,  True,   modbus_fn_ReadHR, 999,  2,    0,     _NaN);
  InitCmdItem(cm_WriteAddress,  False,  modbus_fn_WritMC, 1002, 1,    0,     _NaN);
  InitCmdItem(cm_WriteHighVolt, False,  modbus_fn_WritMC, 999,  1,    0,     _NaN);
  InitCmdItem(cm_SetAddress,    False,  modbus_fn_WritMR, 999,  1,    0,     _NaN);
  InitCmdItem(cm_WriteBaudrate, False,  modbus_fn_WritMR, 1000, 1,    0,     _NaN);
 end;
 //
 // Assign modbus last sent polling request record.
 //
 procedure AssignModbusPoll(ref,cid:Integer; tim:Real; port,uid,fid,saddr,quant:Integer; dat:String);
 begin
  VMCD.Modbus.Poll.ref:=ref;     VMCD.Modbus.Poll.cid:=cid;     VMCD.Modbus.Poll.tim:=tim;
  VMCD.Modbus.Poll.port:=port;   VMCD.Modbus.Poll.uid:=uid;     VMCD.Modbus.Poll.fid:=fid;
  VMCD.Modbus.Poll.saddr:=saddr; VMCD.Modbus.Poll.quant:=quant; VMCD.Modbus.Poll.dat:=dat;
 end;
 //
 // Clear modbus polling request to be ready for next polling.
 //
 procedure ClearModbusPoll;
 begin
  VMCD.Modbus.Poll.cid:=0;
  VMCD.Modbus.Poll.dat:='';
 end;
 //
 // Clear modbus poll summ counters.
 //
 procedure ClearModbusSumm;
 begin
  VMCD.Modbus.Poll.Summ.Rx:=0;
  VMCD.Modbus.Poll.Summ.Tx:=0;
  VMCD.Modbus.Poll.Summ.Ex:=0;
 end;
 //
 // Clear modbus poll rate counters.
 //
 procedure ClearModbusRate;
 begin
  VMCD.Modbus.Poll.Rate.Rx:=0;
  VMCD.Modbus.Poll.Rate.Tx:=0;
  VMCD.Modbus.Poll.Rate.Ex:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Real);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 //
 // Format error message with with error counter increment.
 // Usage is like: Trouble(GotBug(...)); Problem(GotBug(...));
 //
 function GotBug(msg:String):String;
 begin
  IncSummRate(VMCD.Modbus.Poll.Summ.Ex,VMCD.Modbus.Poll.Rate.Ex);
  GotBug:=msg;
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Меню "Инструменты"');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+DevName);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand(DimRemote+'@ClearModbusSumm');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+ModbusProxy);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand(DimRemote+'@ZeroPortCounters 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Перезапустить локальный драйвер '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   if DIM_IsClientMode then begin
    n:=n+EditAddInputLn('Перезапустить удаленный драйвер '+DevName);
    n:=n+EditAddConfirm(EditGetLastInputLn);
    n:=n+EditAddCommand(DimRemote+'@SysEval @Daq Compile '+DevName);
   end;
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VMCD: Запись нового адреса в EEPROM');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.WRITEADDRESS');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VMCD: Установка адреса устройства');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit ADDRESS');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VMCD: Установка скорости передачи данных и контроля четности');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@MenuBaudrateStarter');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuCommandsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 //
 // Menu Baudrate Starter to start editing.
 //
 procedure MenuBaudrateStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Установка скорости передачи');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('9600 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('19200 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('38400 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 2');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('57600 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 3');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_BAUDRATE'),Str(iGetTag(VMCD.PARAM.BAUDRATE.tag)));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 //
 // Menu Tools Handler to handle editing.
 //
 procedure MenuBaudrateHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_BAUDRATE'));
 end;
 //
 // Find the current Proxy device to send @Modbus.Poll message.
 //
 function devTheProxy:Integer;
 begin
  if VMCD.Simulator                 // In simulation mode
  then devTheProxy:=devMySelf       // send messages to himself
  else devTheProxy:=devModbusProxy; // otherwise use &ModbusProxy
 end;
 //
 // Check if communication port is opened or not.
 // As long as we use Proxy device, check Port number & device reference.
 //
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(VMCD.Modbus.Port>0) and (devTheProxy<>0);
 end;
 //
 // Calculate Modbus request PDU by command id.
 // cid   - (in)  command id number.
 // fid   - (out) Modbus function id.
 // saddr - (out) zero-based start address of registers to read/write.
 // quant - (out) data quantity, i.e. number of r/w registers or single register r/w value.
 // dat   - (out) Modbus PDU data unit, PDU=(fid+dat).
 //
 function VMCD_CalcPDU(cid:Integer; var fid,saddr,quant:Integer; var dat:String):Boolean;
 begin
  fid:=0; saddr:=0; quant:=0; dat:='';
  if IsEnabledCmdNum(cid) then begin
   fid:=VMCD.Modbus.Cmd.FuncId[cid];
   saddr:=VMCD.Modbus.Cmd.SAddr[cid];
   quant:=VMCD.Modbus.Cmd.Quant[cid];
   if (cid=cm_ReadPressure)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,'') else
   if (cid=cm_ReadHighVolt)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,'') else
   if (cid=cm_ReadUARTPara)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,'') else
   if (cid=cm_WriteAddress)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,Str(GetCmdOpData(cid))) else
   if (cid=cm_WriteHighVolt) then dat:=modbus_encode_pdu('R',fid,saddr,quant,Str(GetCmdOpData(cid))) else
   if (cid=cm_SetAddress)    then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   if (cid=cm_WriteBaudrate) then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   Trouble(GotBug('Invalid command id '+Str(cid)));
  end;
  VMCD_CalcPDU:=modbus_func_ok(fid) and (Length(dat)>0);
 end;
 //
 // Main Modbus command handler to process Modbus device reply.
 // cid   - (in)  command id number.
 // fid   - (in)  Modbus function id.
 // saddr - (in)  zero-based start address of registers to read/write.
 // quant - (in)  data quantity, i.e. number of r/w registers or single register r/w value.
 // raw   - (in)  raw coils/registers data array.
 //
 procedure VMCD_OnCommand(cid,fid,saddr,quant:Integer; raw:String);
 var addr,offs,indic:Integer; r:Real;
  procedure Cleanup;
  begin
   addr:=0; offs:=0; indic:=0; r:=0;
  end;
 begin
  Cleanup;
  if IsValidCmdNum(cid) then begin
   if cid=cm_ReadPressure then begin
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=1003) and (quant-offs+1>=1) then begin // Давление
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(VMCD.PARAM.PRESSURE.tag,r));
     end;
    end;
   end else if cid=cm_ReadHighVolt then begin
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=999) and (quant-offs+1>=1) then begin // Высокое напряжение
      r:=modbus_ext_coil(raw,offs);
      bNul(iSetTag(VMCD.PARAM.HV.tag,round(r)));
      { EnableCmdNum(cm_ReadHighVolt,False); }
     end;
    end;
   end else if cid=cm_ReadUARTPara then begin
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=999) and (quant-offs+1>=1) then begin // Device address
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(VMCD.PARAM.ADDRESS.tag,round(r)));
     end else
     if (addr=1000) and (quant-offs+1>=1) then begin // Device baudrate
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(VMCD.PARAM.BAUDRATE.tag,round(r)));
      EnableCmdNum(cm_ReadUARTPara,False);
     end;
    end;
   end else
   if (cid=cm_WriteAddress)  then ReleaseCmdOpData(cid) else
   if (cid=cm_WriteHighVolt) then ReleaseCmdOpData(cid) else
   if (cid=cm_SetAddress)    then ReleaseCmdOpData(cid) else
   if (cid=cm_WriteBaudrate) then ReleaseCmdOpData(cid) else
   Trouble(GotBug('Unexpected command '+Str(cid)));
   if VMCD.Simulator then indic:=st_SIMULAT else indic:=st_NORMAL;
   bNul(iSetTag(VMCD.PARAM.STATE.tag,indic));
   UpdateDo(do_STATE_INDIC,time,indic);
   UpdateAo(ao_PRESSURE,time,rGetTag(VMCD.PARAM.PRESSURE.tag));
   UpdateAo(ao_HV,time,iGetTag(VMCD.PARAM.HV.tag));
   UpdateAo(ao_ADDRESS,time,iGetTag(VMCD.PARAM.ADDRESS.tag));
   UpdateAo(ao_BAUDRATE,time,iGetTag(VMCD.PARAM.BAUDRATE.tag));
  end;
  Cleanup;
 end;
 //
 // Data handler on @Modbus.Reply event. Process reply comes from Modbus device.
 //
 procedure VMCD_OnReply(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
  procedure Cleanup;
  begin
   raw:=''; saddr:=0; quant:=0;
  end;
 begin
  Cleanup;
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>VMCD.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>VMCD.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>VMCD.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>VMCD.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>VMCD.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=VMCD.Modbus.Poll.saddr; quant:=VMCD.Modbus.Poll.quant;
   if modbus_decode_pdu('A',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>VMCD.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>VMCD.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else
    if modbus_is_except(fid) then begin
     Problem(GotBug('MODBUS EXCEPTION '+Str(modbus_ext_byte(raw,0))+' ON COMMAND '+Str(cid)));
    end else begin
     IncSummRate(VMCD.Modbus.Poll.Summ.Rx,VMCD.Modbus.Poll.Rate.Rx);
     VMCD_OnCommand(cid,fid,saddr,quant,raw);
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  Cleanup;
 end;
 //
 // Data handler on @Modbus.Poll event.
 // This procedure calls in simulation mode only.
 //
 procedure VMCD_OnSimPoll(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
  procedure Cleanup;
  begin
   raw:=''; saddr:=0; quant:=0;
  end;
 begin
  Cleanup;
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>VMCD.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>VMCD.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>VMCD.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>VMCD.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>VMCD.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=VMCD.Modbus.Poll.saddr; quant:=VMCD.Modbus.Poll.quant;
   if modbus_decode_pdu('R',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>VMCD.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>VMCD.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else begin
     if DebugFlagEnabled(dfDetails) then Details('Simulate polling '+Str(cid));
     if (cid=cm_ReadPressure) then begin
      raw:=modbus_dump_float(VMCD.SIM.PRESSURE,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_ReadHighVolt) then begin
      raw:=modbus_dump_float(VMCD.SIM.HV,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_ReadUARTPara) then begin
      raw:=modbus_dump_int16(VMCD.SIM.ADDRESS,SwapModeInt32)
          +modbus_dump_int16(VMCD.SIM.BAUDRATE,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_WriteAddress) then begin
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat))
     end else if (cid=cm_WriteHighVolt) then begin
      VMCD.SIM.HV:=modbus_ext_coil(raw,0);
      { EnableCmdNum(cm_ReadHighVolt,true); }
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_SetAddress) then begin
      VMCD.SIM.ADDRESS:=modbus_ext_int16(raw,0,SwapModeFloat);
      EnableCmdNum(cm_ReadUARTPara,true);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_WriteBaudrate) then begin
      VMCD.SIM.BAUDRATE:=modbus_ext_int16(raw,0,SwapModeFloat);
      EnableCmdNum(cm_ReadUARTPara,true);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,
              modbus_as_except(fid,true),Dump(Chr(modbus_er_ILLFUN))));
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  Cleanup;
 end;

 //
 // Pressure Simulation
 //
 procedure VMCD_Simulation;
 var
 begin
  VMCD.SIM.PRESSURE:=VMCD.SIM.PRESSURE-0.0001;
  if VMCD.SIM.PRESSURE<0 then VMCD.SIM.PRESSURE:=1.5;
 end;

 //
 // Send to console command (cmd data) to set new tag value if one in range (min,max).
 //
 procedure SendTagCmd(tag:Integer; cmd,data:String; min,max:Real);
 var v:Real;
 begin
  if IsRefTag(tag) then
  if not IsEmptyStr(cmd) then
  if not IsEmptyStr(data) then
  if TypeTag(tag)<=2 then begin
   v:=rVal(data);
   if (v<min) then begin v:=min; data:=Str(min); end;
   if (v>max) then begin v:=max; data:=Str(max); end;
   if IsNaN(v) then Trouble(GotBug('Invalid tag edit')) else
   if TypeTag(tag)=1 then DevPostCmdLocal(cmd+' '+Str(Round(v))) else
   if TypeTag(tag)=2 then DevPostCmdLocal(cmd+' '+Trim(data));
  end else if TypeTag(tag)=3 then begin
   DevPostCmdLocal(cmd+' '+Trim(data));
  end;
 end;

 {
 GUI Handler to process user input...
 }
 procedure GUIHandler;
 var ClickCurve:Integer; s:String;
  procedure Cleanup;
  begin
   DIM_GuiClickBuff:=''; s:=''; ClickCurve:=0;
  end;
 begin
  Cleanup;
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Copy GUI click to DIM buffer for remote execution.
   }
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Handle local clicks
     }
     if ClickIsLocal then begin
      //
      // Handle Tag clicks...
      //
      ClickTagXorRemote(VMCD.POLL.ENABLE.tag,1);
      if ClickTag=VMCD.PARAM.HV.tag then DevSendCmdLocal(DimRemote+'@PARAM.HIGHVOLTAGE '+Str(iXor(iGetTag(VMCD.PARAM.HV.tag),1)));
      //
      // Handle sensor clicks...
      //
      if Pos('.TOOLS',ClickSensor)<>0 then begin
       bNul(Voice(snd_Click));
       MenuToolsStarter;
      end;
      if IsSameText(ClickSensor,'HELP') then begin
       DevPostCmdLocal('@BrowseHelp '+DaqFileRef(ReadIni('[DAQ] HelpFile'),'.md'));
       bNul(Voice(snd_Click));
      end;
      {
      Select Plot & Tab windows by curve...
      }
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      {
      Console commands: @url_encoded_sensor ...
      }
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     {
     Handle remote clicks comes from DIM via @DimGuiClick message.
     @DimGuiClick default handler decode and write events to FIFO,
     so we can find it as clicks and can handle it in usual way.
     }
     if ClickIsRemote then begin
      {
      Show time difference.
      }
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      {
      Handle remote console commands...
      }
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   //
   // If tag edit complete, send command to apply changes
   //
   if CheckEditTag(VMCD.PARAM.HV.tag,      s) then SendTagCmd(VMCD.PARAM.HV.tag,      DimRemote+'@PARAM.HIGHVOLTAGE',s,0, 1);
   if CheckEditTag(VMCD.PARAM.ADDRESS.tag, s) then SendTagCmd(VMCD.PARAM.ADDRESS.tag, DimRemote+'@PARAM.ADDRESS',    s,10,247);
   if CheckEditTag(VMCD.PARAM.BAUDRATE.tag,s) then SendTagCmd(VMCD.PARAM.BAUDRATE.tag,DimRemote+'@PARAM.BAUDRATE',   s,0, 19);
   //
   // Menus handler
   //
   MenuCommandsHandler;
   MenuBaudrateHandler;
   {
   Warning, Information dialog completion.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end;
  if EditStateError then begin
   Problem('Dialog error detected!');
   EditReset;
  end;
  Cleanup;
 end;
 //
 // VMCD Driver Command Cycle polling.
 //
 procedure VMCD_CMD_POLL;
  //
  // Send @Modbus.Poll ... command to &ModbusProxy and save sent request data in Modbus.Poll record.
  //
  procedure PollModbusProxy(cid:Integer);
  var dev,ref,tot,port,uid,fid,saddr,quant:Integer; dat:String;
   procedure Cleanup;
   begin
    dev:=0; ref:=0; tot:=0; port:=0; uid:=0; fid:=0; saddr:=0; quant:=0; dat:='';
   end;
  begin
   Cleanup;
   ApplyCmdOpData(cid);
   if VMCD_CalcPDU(cid,fid,saddr,quant,dat) then begin
    dev:=devTheProxy; ref:=devMySelf; tot:=VMCD.Modbus.Timeout; port:=VMCD.Modbus.Port; uid:=VMCD.Modbus.UnitId;
    if DevSend(dev,modbus_proxy_poll('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat)+CRLF)>0 then begin
     if DebugFlagEnabled(dfViewExp) then
     ViewExp('COM: '+modbus_proxy_nice('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat,32));
     AssignModbusPoll(dev,cid,mSecNow,port,uid,fid,saddr,quant,dat);
     IncSummRate(VMCD.Modbus.Poll.Summ.Tx,VMCD.Modbus.Poll.Rate.Tx);
    end else begin
     Trouble(GotBug('Fail to send command '+Str(cid)));
     ClearModbusPoll;
    end;
   end else begin
    Trouble(GotBug('Fail to calc command '+Str(cid)));
    ClearModbusPoll;
   end;
   Cleanup;
  end;
 begin
  if IsPortOpened and not DIM_IsClientMode then begin
   if iGetTag(VMCD.POLL.ENABLE.tag)<>0 then begin
    if IsValidCmdNum(VMCD.Modbus.Poll.cid) then begin
     //
     // Request in progress, waiting @Modbus.Reply/@Modbus.Refuse/@Modbus.Timeout.
     // Handle Deadline if was no responce for too long time: repeat polling again.
     //
     if (mSecNow>VMCD.Modbus.Poll.tim+VMCD.Modbus.Deadline) then begin
      Trouble(GotBug('Deadline detected, repeat polling again...'));
      PollModbusProxy(VMCD.Modbus.Poll.cid);
     end;
    end else begin
     //
     // If request is cleared, send new @Modbus.Poll request by timer.
     //
     if (mSecNow>=VMCD.Modbus.Poll.tim+VMCD.Modbus.Polling) then begin
      VMCD.Modbus.Cmd.Num:=NextEnabledCmdNum(VMCD.Modbus.Cmd.Num);
      if IsEnabledCmdNum(VMCD.Modbus.Cmd.Num)
      then PollModbusProxy(VMCD.Modbus.Cmd.Num);
     end;
    end;
   end else if iGetTag(VMCD.POLL.ENABLE.tag)=0 then begin
    UpdateDo(do_STATE_INDIC,time,st_DISABLE);
    bNul(iSetTag(VMCD.PARAM.STATE.tag,st_DISABLE));
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    bNul(rSetTag(VMCD.POLLRATE.RX.tag,VMCD.Modbus.Poll.Rate.Rx));
    bNul(rSetTag(VMCD.POLLRATE.TX.tag,VMCD.Modbus.Poll.Rate.Tx));
    bNul(rSetTag(VMCD.POLLRATE.EX.tag,VMCD.Modbus.Poll.Rate.Ex));
    bNul(rSetTag(VMCD.POLLSUMM.RX.tag,VMCD.Modbus.Poll.Summ.Rx));
    bNul(rSetTag(VMCD.POLLSUMM.TX.tag,VMCD.Modbus.Poll.Summ.Tx));
    bNul(rSetTag(VMCD.POLLSUMM.EX.tag,VMCD.Modbus.Poll.Summ.Ex));
    bNul(rSetTag(VMCD.ERROR.COUNT.tag,GetErrCount(-1)));
    UpdateDo(do_POLLRATERX,time,VMCD.Modbus.Poll.Rate.Rx);
    UpdateDo(do_POLLRATETX,time,VMCD.Modbus.Poll.Rate.Tx);
    UpdateDo(do_POLLRATEEX,time,VMCD.Modbus.Poll.Rate.Ex);
    UpdateDo(do_POLLSUMMRX,time,VMCD.Modbus.Poll.Summ.Rx);
    UpdateDo(do_POLLSUMMTX,time,VMCD.Modbus.Poll.Summ.Tx);
    UpdateDo(do_POLLSUMMEX,time,VMCD.Modbus.Poll.Summ.Ex);
    UpdateDo(do_ERRORCOUNT,time,GetErrCount(-1));
    if DebugFlagEnabled(dfStatist) then
    Success('PollRate: Rx='+Str(VMCD.Modbus.Poll.Rate.Rx)
                   +'  Tx='+Str(VMCD.Modbus.Poll.Rate.Tx)
                   +'  Ex='+Str(VMCD.Modbus.Poll.Rate.Ex));
    ClearModbusRate;
   end;
  end;
 end;
 //
 // Проверяет, находится ли давление в диапазоне ниже 1Па
 //
 function VMCD_PressureLess1Pa:Boolean;
 begin
  if rGetTag(VMCD.PARAM.PRESSURE.tag)<1
  then VMCD_PressureLess1Pa:=true
  else VMCD_PressureLess1Pa:=false;
 end;
 {
 Handle message @AssignTag arg
 }
 procedure VMCD_OnAssignTag(arg:String);
 var tag:Integer; w1,w2:String;
  procedure Cleanup;
  begin
   w1:=''; w2:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   w1:=ExtractWord(1,arg);
   tag:=FindTag(w1);
   if (tag<>0) then begin
    w2:=ExtractWord(2,arg);
    if tag=VMCD.POLL.ENABLE.tag then UpdateTag(tag,w2,0,1);
   end;
  end;
  Cleanup;
 end;
 {
 Handle message @DimUpdateTag arg
 }
 procedure VMCD_OnDimUpdateTag(arg:String);
 var tag,typ:Integer; x,y:Real;
 begin
  if (arg<>'') then begin
   if DIM_IsClientMode and not DIM_IsServerMode then begin
    tag:=FindTag(ExtractWord(1,arg));
    if (tag<>0) then begin
     typ:=TypeTag(tag);
     if (typ=1) then y:=iGetTag(tag) else
     if (typ=2) then y:=rGetTag(tag) else y:=_Nan;
     x:=time;
     if not IsNan(y) then begin
      if tag=VMCD.PARAM.PRESSURE.tag then UpdateAo(ao_PRESSURE,x,y);
      if tag=VMCD.PARAM.HV.tag       then UpdateAo(ao_HV,x,y);
      if tag=VMCD.PARAM.STATE.tag    then UpdateDo(do_STATE_INDIC,x,y);
      if tag=VMCD.ERROR.COUNT.tag    then UpdateDo(do_ERRORCOUNT,x,y);
      if tag=VMCD.POLLRATE.RX.tag    then UpdateDo(do_POLLRATERX,x,y);
      if tag=VMCD.POLLRATE.TX.tag    then UpdateDo(do_POLLRATETX,x,y);
      if tag=VMCD.POLLRATE.EX.tag    then UpdateDo(do_POLLRATEEX,x,y);
      if tag=VMCD.POLLSUMM.RX.tag    then UpdateDo(do_POLLSUMMRX,x,y);
      if tag=VMCD.POLLSUMM.TX.tag    then UpdateDo(do_POLLSUMMTX,x,y);
      if tag=VMCD.POLLSUMM.EX.tag    then UpdateDo(do_POLLSUMMEX,x,y);
     end;
    end;
   end;
  end;
 end;
 //
 // Tags filling
 //
 procedure VMCD_FillTags(InitVal:Real);
 begin
  VMCD.POLL.ENABLE.val:=InitVal;
  VMCD.PARAM.PRESSURE.val:=InitVal;
  VMCD.PARAM.STATE.val:=InitVal;
  VMCD.PARAM.HV.val:=InitVal;
  VMCD.PARAM.ADDRESS.val:=InitVal;
  VMCD.ERROR.COUNT.val:=InitVal;
  VMCD.POLLRATE.RX.val:=InitVal;
  VMCD.POLLRATE.TX.val:=InitVal;
  VMCD.POLLRATE.EX.val:=InitVal;
  VMCD.POLLSUMM.RX.val:=InitVal;
  VMCD.POLLSUMM.TX.val:=InitVal;
  VMCD.POLLSUMM.EX.val:=InitVal;
 end;
 {
 Update DIM services
 }
 procedure DimUpdateState;
 begin
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then VMCD_FillTags(-MaxReal);
   if ShouldRefresh(VMCD.POLL.ENABLE.val,    GetStampOfTag(VMCD.POLL.ENABLE.tag,0))>0    then DIM_UpdateTag(VMCD.POLL.ENABLE.tag,'');
   if SysTimer_Pulse(500)>0 then
   if ShouldRefresh(VMCD.PARAM.PRESSURE.val, GetStampOfTag(VMCD.PARAM.PRESSURE.tag,0))>0 then DIM_UpdateTag(VMCD.PARAM.PRESSURE.tag,'');
   if ShouldRefresh(VMCD.PARAM.STATE.val,    GetStampOfTag(VMCD.PARAM.STATE.tag,0))>0    then DIM_UpdateTag(VMCD.PARAM.STATE.tag,'');
   if ShouldRefresh(VMCD.PARAM.HV.val,       GetStampOfTag(VMCD.PARAM.HV.tag,0))>0       then DIM_UpdateTag(VMCD.PARAM.HV.tag,'');
   if ShouldRefresh(VMCD.PARAM.ADDRESS.val,  GetStampOfTag(VMCD.PARAM.ADDRESS.tag,0))>0  then DIM_UpdateTag(VMCD.PARAM.ADDRESS.tag,'');
   if ShouldRefresh(VMCD.PARAM.BAUDRATE.val, GetStampOfTag(VMCD.PARAM.BAUDRATE.tag,0))>0 then DIM_UpdateTag(VMCD.PARAM.BAUDRATE.tag,'');
   if ShouldRefresh(VMCD.ERROR.COUNT.val,    GetStampOfTag(VMCD.ERROR.COUNT.tag,0))>0    then DIM_UpdateTag(VMCD.ERROR.COUNT.tag,'');
   if ShouldRefresh(VMCD.POLLRATE.RX.val,    GetStampOfTag(VMCD.POLLRATE.RX.tag,0))+
      ShouldRefresh(VMCD.POLLRATE.TX.val,    GetStampOfTag(VMCD.POLLRATE.TX.tag,0))+
      ShouldRefresh(VMCD.POLLRATE.EX.val,    GetStampOfTag(VMCD.POLLRATE.EX.tag,0))>0 then DIM_UpdateTag(VMCD.POLLRATE.RX.tag,'');
   if ShouldRefresh(VMCD.POLLSUMM.RX.val,    GetStampOfTag(VMCD.POLLSUMM.RX.tag,0))+
      ShouldRefresh(VMCD.POLLSUMM.TX.val,    GetStampOfTag(VMCD.POLLSUMM.TX.tag,0))+
      ShouldRefresh(VMCD.POLLSUMM.EX.val,    GetStampOfTag(VMCD.POLLSUMM.EX.tag,0))>0 then DIM_UpdateTag(VMCD.POLLSUMM.RX.tag,'');
  end;
 end;
 //
 // VMCD tags initialization.
 //
 procedure VMCD_InitTags(Prefix:String);
 begin
  DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
  InitTag(VMCD.POLL.ENABLE.tag,    Prefix+'.ENABLE',      1);
  InitTag(VMCD.PARAM.PRESSURE.tag, Prefix+'.PRESSURE',    2);
  InitTag(VMCD.PARAM.STATE.tag,    Prefix+'.STATE',       1);
  InitTag(VMCD.PARAM.HV.tag,       Prefix+'.HV',          1);
  InitTag(VMCD.PARAM.ADDRESS.tag,  Prefix+'.ADDRESS',     1);
  InitTag(VMCD.PARAM.BAUDRATE.tag, Prefix+'.BAUDRATE',    1);
  InitTag(VMCD.POLLRATE.RX.tag,    Prefix+'.POLLRATE.RX', 2);
  InitTag(VMCD.POLLRATE.TX.tag,    Prefix+'.POLLRATE.TX', 2);
  InitTag(VMCD.POLLRATE.EX.tag,    Prefix+'.POLLRATE.EX', 2);
  InitTag(VMCD.ERROR.COUNT.tag,    Prefix+'.ERROR.COUNT', 2);
  InitTag(VMCD.POLLSUMM.RX.tag,    Prefix+'.POLLSUMM.RX', 2);
  InitTag(VMCD.POLLSUMM.TX.tag,    Prefix+'.POLLSUMM.TX', 2);
  InitTag(VMCD.POLLSUMM.EX.tag,    Prefix+'.POLLSUMM.EX', 2);
 end;
 //
 // VMCD Driver cleanup.
 //
 procedure VMCD_Clear;
 begin
  VMCD.Simulator:=False;
  VMCD.Modbus.Port:=0;
  VMCD.Modbus.UnitId:=0;
  VMCD.Modbus.Timeout:=0;
  VMCD.Modbus.Polling:=0;
  VMCD.Modbus.Deadline:=0;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
  VMCD_FillTags(-MaxReal);
 end;
 //
 // VMCD Driver initialization...
 //
 procedure VMCD_Init;
 begin
  //
  // Read ini file variables
  //
  VMCD_InitTags(ReadIni('tagPrefix'));
  VMCD.Simulator:=iValDef(ReadIni('Simulator'),0)<>0; Success('Simulator='+Str(Ord(VMCD.Simulator)));
  VMCD.Modbus.Port:=iValDef(ReadIni('ModbusPort'),1); Success('ModbusPort='+Str(VMCD.Modbus.Port));
  VMCD.Modbus.UnitId:=iValDef(ReadIni('ModbusUnitId'),1); Success('ModbusUnitId='+Str(VMCD.Modbus.UnitId));
  VMCD.Modbus.Polling:=iValDef(ReadIni('ModbusPolling'),1000); Success('ModbusPolling='+Str(VMCD.Modbus.Polling));
  VMCD.Modbus.Timeout:=iValDef(ReadIni('ModbusTimeout'),250); Success('ModbusTimeout='+Str(VMCD.Modbus.Timeout));
  VMCD.Modbus.Deadline:=iValDef(ReadIni('ModbusDeadline'),60000); Success('ModbusDeadline='+Str(VMCD.Modbus.Deadline));
  VMCD.Modbus.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000); Success('DelayOnStart='+Str(VMCD.Modbus.DelayOnStart));
  if not DIM_IsClientMode then if not VMCD.Simulator then
  if IsRefDevice(RefFind('Device '+ModbusProxy)) then Success(ModbusProxy+' device found') else Trouble(ModbusProxy+' device not found!');
  if DIM_IsServerMode then Success('Run as Server') else if DIM_IsClientMode then Success('Run as Client');
  //
  // Initialize values
  //
  VMCD.SIM.ADDRESS:=VMCD.Modbus.UnitId;
  VMCD.SIM.BAUDRATE:=1;
  //
  // Initialize Cmd command table & clear Poll record.
  //
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  InitCmdTable;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  VMCD_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  //
  // Standard initializations.
  //
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  InitNetLibrary;
  VMCD_Init;
  if not DIM_IsClientMode then
  if Val(ReadIni('CustomIniAutoLoad'))=1 then DevPostCmdLocal('@LoadIni');
  //
  // StdIn Command registration.
  //
  cmd_mbReply      :=RegisterStdInCmd('@Modbus.Reply','');
  cmd_mbPoll       :=RegisterStdInCmd('@Modbus.Poll','');
  cmd_mbRefuse     :=RegisterStdInCmd('@Modbus.Refuse','');
  cmd_mbTimeout    :=RegisterStdInCmd('@Modbus.Timeout','');
  cmd_mbZeroPortCnt:=RegisterStdInCmd('@ZeroPortCounters','');
  cmd_mbClrSumm    :=RegisterStdInCmd('@ClearModbusSumm','');
  cmd_WriteAddress :=RegisterStdInCmd('@PARAM.WRITEADDRESS','');
  cmd_EnableHighV  :=RegisterStdInCmd('@ENABLE.HIGHVOLTAGE','');
  cmd_parHighV     :=RegisterStdInCmd('@PARAM.HIGHVOLTAGE','');
  cmd_parAddress   :=RegisterStdInCmd('@PARAM.ADDRESS','');
  cmd_parBaudrate  :=RegisterStdInCmd('@PARAM.BAUDRATE','');
  cmd_Edit         :=RegisterStdInCmd('@Edit','');
  cmd_menuBaudrate :=RegisterStdInCmd('@MenuBaudrateStarter','');
  cmd_DimTagUpdate :=RegisterStdInCmd('@DimTagUpdate','');
  cmd_AssignTag    :=RegisterStdInCmd('@AssignTag',   '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if not DIM_IsClientMode then
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  FreeNetLibrary;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  GUIHandler;
  if mSecNow-FixmSecNow>VMCD.Modbus.DelayOnStart then
   if IsPortOpened then begin
    VMCD_CMD_POLL;
    if not(VMCD_PressureLess1Pa) and (iGetTag(VMCD.PARAM.HV.tag)=1) then begin
     DevPostCmdLocal('@PARAM.HIGHVOLTAGE 0');
     Problem('Выключение HV. Достигнуто предельное для режима давление.');
     ShowTooltip('text "'+DevName+': Высокое напряжение выключено! Достигнуто предельное для режима давление." preset stdWarning');
    end;
   end;
  if VMCD.Simulator then VMCD_Simulation;
  DimUpdateState;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,dat:String; cmdid,ref,cid,tim,port,uid,fid:Integer; v:Real;
  procedure Cleanup;
  begin
   cmd:=''; arg:=''; dat:='';
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Modbus.Reply ref cid tim port uid fid $$dat
   }
   if (cmdid = cmd_mbReply) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then VMCD_OnReply(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Poll ref cid tim port uid fid $$dat
   This message uses in simulation mode only
   }
   if (cmdid = cmd_mbPoll) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then VMCD_OnSimPoll(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    Data:='';
   end else
   {
   @Modbus.Refuse ...
   }
   if (cmdid = cmd_mbRefuse) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,0)))
    else Trouble(GotBug(cmd+' '+arg));
    UpdateDo(do_STATE_INDIC,time,st_REFUSE);
    bNul(iSetTag(VMCD.PARAM.STATE.tag,st_REFUSE));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Timeout ...
   }
   if (cmdid = cmd_mbTimeout) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Problem(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,64)))
    else Trouble(GotBug(cmd+' '+arg));
    UpdateDo(do_STATE_INDIC,time,st_TIMEOUT);
    bNul(iSetTag(VMCD.PARAM.STATE.tag,st_TIMEOUT));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @ZeroPortCounters 0
   }
   if (cmdid=cmd_mbZeroPortCnt) then begin
    if not DIM_IsClientMode then DevSendCmd(devModbusProxy,cmd+' '+arg);
    Data:='';
   end else
   {
   @ClearModbusSumm
   }
   if (cmdid = cmd_mbClrSumm) then begin
    if not DIM_IsClientMode then ClearModbusSumm;
    Data:='';
   end else
   {
   @PARAM.WRITEADDRESS 1
   }
   if (cmdid = cmd_WriteAddress) then begin
    if not DIM_IsClientMode then HoldCmdOpData(cm_WriteAddress,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @PARAM.HIGHVOLTAGE 1
   }
   if (cmdid = cmd_parHighV) then begin
    if not DIM_IsClientMode then begin
     v:=rVal(ExtractWord(1,arg));
     if v=0 then begin
      HoldCmdOpData(cm_WriteHighVolt,v);
      { EnableCmdNum(cm_ReadHighVolt,True); }
     end else
     if VMCD_PressureLess1Pa then begin
      HoldCmdOpData(cm_WriteHighVolt,v);
      { EnableCmdNum(cm_ReadHighVolt,True); }
     end else begin
      Problem('Не удалось включить HV. Достигнуто предельное давление.');
      ShowTooltip('text "'+DevName+': Не удалось включить HV. Достигнуто предельное давление." delay 15000 preset stdWarning');
     end;
    end;
    Data:='';
   end else
   {
   @PARAM.ADDRESS 1
   }
   if (cmdid = cmd_parAddress) then begin
    if not DIM_IsClientMode then begin
     v:=rVal(ExtractWord(1,arg));
     HoldCmdOpData(cm_SetAddress,v);
     if not IsNan(v) then begin
      bNul(iSetTag(VMCD.PARAM.ADDRESS.tag,Round(v)));
      DevPostCmdLocal('@SaveIni');
     end;
    end;
    Data:='';
   end else
   {
   @ENABLE.HIGHVOLTAGE
   }
   if (cmdid = cmd_EnableHighV) then begin
    if not DIM_IsClientMode then DevPostCmdLocal('@PARAM.HIGHVOLTAGE 1');
    Data:='';
   end else
   {
   @PARAM.BAUDRATE 1
   }
   if (cmdid = cmd_parBaudrate) then begin
    if not DIM_IsClientMode then begin
     v:=rVal(ExtractWord(1,arg));
     HoldCmdOpData(cm_WriteBaudrate,v);
     if not IsNan(v) then begin
      bNul(iSetTag(VMCD.PARAM.BAUDRATE.tag,Round(v)));
      DevPostCmdLocal('@SaveIni');
     end;
    end;
    Data:='';
   end else
   {
   @Edit ADDRESS 10
   @Edit BAUDRATE 2800
   }
   if (cmdid = cmd_Edit) then begin
    if IsSameText(ExtractWord(1,arg),'ADDRESS')  then StartEditTag(VMCD.PARAM.ADDRESS.tag, 'Введите адрес устройства (10-247)');
    if IsSameText(ExtractWord(1,arg),'BAUDRATE') then StartEditTag(VMCD.PARAM.BAUDRATE.tag,'Установка скорости передачи данных и контроля четности');
    Data:='';
   end else
   {
   @MenuBaudrateStarter
   }
   if (cmdid = cmd_menuBaudrate) then begin
    MenuBaudrateStarter;
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    VMCD_OnDimUpdateTag(arg);
    Data:='';
   end else
   {
   @AssignTag
   }
   if (cmdid=cmd_AssignTag) then begin
    VMCD_OnAssignTag(arg);
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
