 {
 ***********************************************************************
 Control driver for HART devices.
 Works as simulator if AnalogOutputs=0.
 Configuration example:
 ----------------------
 [SerialPort-COM1]  
 Port     = COM1
 BaudRate = 1200
 Parity   = ODD
 DataBits = 8
 StopBits = 1 
 DcbFlags = $3020
 []
 [DeviceList]
 &HART_DRV = device software program
 [&HART_DRV]
 Comment       = HART DRIVER
 InquiryPeriod = 1
 DevicePolling = 100, tpNormal
 ProgramSource = .\HART_DRV.PAS
 PortSection   = [SerialPort-COM1]
 HartTimeOut   = 1000
 DebugFlags    = 15
 OpenConsole   = 1
 DebugMode     = 3
 AnalogOutputs = 2
 Link AnalogOutput 0 with curve Hart.T1 history 1000
 Link AnalogOutput 1 with curve Hart.P1 history 1000
 Cmd#0 = FFFFFFFFFF82ABEF0247C501004700
 Cmd#1 = FFFFFFFFFF826BECE0F7CE0100DD00
 Ans#0 = 00FFFFFFFF86ABEF0247C5010700002041B49EB0BF
 Ans#1 = 00FFFFFFFFFF862BECE0F7CE01070040073CA2BCA45F
 []
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Help          - This help.
| @DebugFlags=n  - Set DebugFlags,1/2/4/8=!/:/>/< view
|********************************************************
[]
 }
program HART_DRV;                { HART protocol driver & simulator }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 dfViewDump        = 16;         { DebugFlags - HartDump            }
 nmax              = 127;        { Max. number of commands          }
 {---------------------------------HART Start byte codes -----------}
 sbMasterS         = 2;          { $02 Short frame master to slave  }    
 sbMasterL         = 130;        { $82 Long  frame master to slave  }    
 sbSlaveS          = 6;          { $06 Short frame slave  to master }    
 sbSlaveL          = 134;        { $86 Long  frame slave  to master }    
 sbPackedS         = 1;          { $01 Short frame packed to master }    
 sbPackedL         = 127;        { $81 Long  frame packed to master }
 {------------------------------HART commands--------------------}
 hcReadVar         = 1;          { Read variable 0:units 1-4:data F }   
 
type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }
 THartRec          = record      { HART protocol record             }
  Preambula,                     { 4..20 bytes FFFF..FF start header}
  Start,                         { 1 byte of start:01,02,06,81,82,86}
  Address,                       { 1 or 5 bytes depending of Start  }
  Command,                       { 1 or 2 bytes command,2 if 1st=254}
  NumBytes,                      { 1 byte - num.of status+data bytes}
  Status,                        { 0 or 2 bytes for master or slave }
  Data,                          { 0..25 bytes data                 }
  CheckSum         : record      { Xor for bytes starting from Start}
   Pos             : Integer;    { Position of item in data string  }
   Len             : Integer;    { Length   of item in data string  }
   Val             : Integer;    { Value    of item (if available)  }
  end;
 end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 Hart              : record      { All HART related data            }
  CmdList          : array[0..nmax] of String;    {Commands for HART}
  AnsList          : array[0..nmax] of String;    {HART reply sample}
  TimeOut          : Real;       { Timeout for wait answer, ms      }
  CmdRec           : THartRec;   { Record to decode HART command    }
  AnsRec           : THartRec;   { Record to decode HART answer     }
  Time             : Real;       { Time when sent last message      }
  Chan             : Integer;    { Current channel                  }
  Cmd              : String;     { Command for HART request         }
  Ans              : String;     { Answer on HART request           }
 end;

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Convert string to reverse order, like ABC -> CBA.
 }
 function ReverseString(data:String):String;
 var i,L:Integer; s:String;
 begin
  s:='';
  L:=Length(data);
  for i:=1 to L do s:=s+data[L-i+1];
  ReverseString:=s;
  s:='';
 end;  
 {
 Calculate Xor checksum.
 }
 function XorCheckSum(data:String):Integer;
 var i,k:Integer;
 begin
  k:=0; for i:=1 to Length(data) do k:=iXor(k,Ord(data[i]));
  XorCheckSum:=k;
 end;
 {
 Send (binary) data to COM port.
 }
 procedure SendPort(out:String);
 begin
  if Length(out)>0 then
  if ComSpace>Length(out) then begin
   if ComWrite(out)
   then ViewExp(Hex_Encode(out))
   else Trouble('ERROR SEND '+Hex_Encode(out));
  end else begin
   Trouble('COM PORT FIFO OVERFLOW');
   bNul(ComClear);
  end;
 end;
 {
 Decode HART protocol message packed in data.
 Result=0 if Ok, Result=0..5 in case of errors.
 HartRec returns position, length and value (if avail) for each item.
 }
 function HartDecode(var HartRec:THartRec; data:String):Integer;
 var i,p,ErrCode:Integer;
 begin
  {---Search scan for preambula and start byte---}
  p:=Pos(Chr(255),data);
  if p>0 then while (data[p]=Chr(255)) and (p<Length(data)) do p:=p+1;
  {---Preambula found?---}
  if (p>0) and (p<Length(data)) then begin
   {---Set preambula---}
   HartRec.Preambula.Pos:=1;
   HartRec.Preambula.Len:=p-1;
   HartRec.Preambula.Val:=255;
   {---Set StartByte---}
   HartRec.Start.Pos:=HartRec.Preambula.Pos+HartRec.Preambula.Len;
   HartRec.Start.Len:=1;
   HartRec.Start.Val:=Dump2i(Copy(data,HartRec.Start.Pos,1));
   {---Set Address depending on StartByte---}
   HartRec.Address.Pos:=HartRec.Start.Pos+HartRec.Start.Len;
   if (HartRec.Start.Val=sbMasterS) then HartRec.Address.Len:=1 else
   if (HartRec.Start.Val=sbMasterL) then HartRec.Address.Len:=5 else
   if (HartRec.Start.Val=sbSlaveS)  then HartRec.Address.Len:=1 else
   if (HartRec.Start.Val=sbSlaveL)  then HartRec.Address.Len:=5 else
   if (HartRec.Start.Val=sbPackedS) then HartRec.Address.Len:=1 else
   if (HartRec.Start.Val=sbPackedL) then HartRec.Address.Len:=5 else
   HartRec.Address.Len:=0;
   HartRec.Address.Val:=Dump2i(Copy(data,HartRec.Address.Pos,1));
   {---StartByte was correct?---}
   if (HartRec.Address.Len>0) and (Length(data)>HartRec.Address.Pos+HartRec.Address.Len+2) then begin
    {---Set command:2 bytes if 1st byte=254---}
    HartRec.Command.Pos:=HartRec.Address.Pos+HartRec.Address.Len;
    HartRec.Command.Val:=Dump2i(Copy(data,HartRec.Command.Pos,1));
    HartRec.Command.Len:=1+Ord(HartRec.Command.Val=254);
    {---Set number of status+data bytes---}
    HartRec.NumBytes.Pos:=HartRec.Command.Pos+HartRec.Command.Len;
    HartRec.NumBytes.Len:=1;
    HartRec.NumBytes.Val:=Dump2i(Copy(data,HartRec.NumBytes.Pos,1));
    {---NumBytes correct?---}
    if (HartRec.NumBytes.Val>=0) and (Length(data)>=HartRec.NumBytes.Pos+HartRec.NumBytes.Val+1) then begin
     {---Set status depending on Start byte---}
     HartRec.Status.Pos:=HartRec.NumBytes.Pos+HartRec.NumBytes.Len;
     if (HartRec.Start.Val=sbMasterS) then HartRec.Status.Len:=0 else
     if (HartRec.Start.Val=sbMasterL) then HartRec.Status.Len:=0 else
     if (HartRec.Start.Val=sbSlaveS)  then HartRec.Status.Len:=2 else
     if (HartRec.Start.Val=sbSlaveL)  then HartRec.Status.Len:=2 else
     if (HartRec.Start.Val=sbPackedS) then HartRec.Status.Len:=2 else
     if (HartRec.Start.Val=sbPackedL) then HartRec.Status.Len:=2 else
     HartRec.Status.Len:=0;
     HartRec.Status.Val:=Dump2i(ReverseString(Copy(data,HartRec.Status.Pos,HartRec.Status.Len)));
     {---Set data bytes depending on NumBytes---}
     HartRec.Data.Pos:=HartRec.Status.Pos+HartRec.Status.Len;
     HartRec.Data.Len:=HartRec.NumBytes.Val-HartRec.Status.Len;
     HartRec.Data.Val:=Dump2i(Copy(data,HartRec.Data.Pos,1));
     {---Set checksum---}
     HartRec.CheckSum.Pos:=HartRec.Data.Pos+HartRec.Data.Len;
     HartRec.CheckSum.Len:=1;
     HartRec.CheckSum.Val:=XorCheckSum(Copy(data,HartRec.Start.Pos,HartRec.CheckSum.Pos-HartRec.Start.Pos));
     {---Check data integrity---}
     if (HartRec.Data.Len>=0) and (Length(data)>=HartRec.CheckSum.Pos) then begin
      {---Compare calculated and readout checksum---}
      if (HartRec.CheckSum.Val = Dump2i(Copy(data,HartRec.CheckSum.Pos,1)))
      then ErrCode:=0                         // Ok
      else ErrCode:=5;                        // Invalid checksum
     end else ErrCode:=4;                     // Invalid data or length
    end else ErrCode:=3;                      // Invalid numbytes or length
   end else ErrCode:=2;                       // Invalid start or address or length
  end else ErrCode:=1;                        // Preambula not found
  if ErrCode<>0 then HartRec.CheckSum.Len:=0; // If error, mark HartRec as invalid
  HartDecode:=ErrCode;
 end;
 {
 Dump HART record.
 HartRec should be correct: call HartDecode(HartRec,data) before.
 }
 procedure HartDump(var HartRec:THartRec; data:String);
  procedure DumpItem(msg:String; p,l,v:Integer);
  begin
   ViewImp(' '+msg+' '+StrFix(p,2,0)+' '+StrFix(l,2,0)+' '+StrFix(v,5,0)+' '+Hex_Encode(Copy(data,p,l)));
  end;
 begin
  {---Check HartRec integrity and print dump---}
  if HartRec.CheckSum.Len=1 then begin
   DumpItem('Preambula', HartRec.Preambula.Pos, HartRec.Preambula.Len, HartRec.Preambula.Val);
   DumpItem('StartByte', HartRec.Start.Pos,     HartRec.Start.Len,     HartRec.Start.Val);
   DumpItem('Address  ', HartRec.Address.Pos,   HartRec.Address.Len,   HartRec.Address.Val);
   DumpItem('Command  ', HartRec.Command.Pos,   HartRec.Command.Len,   HartRec.Command.Val);
   DumpItem('NumBytes ', HartRec.NumBytes.Pos,  HartRec.NumBytes.Len,  HartRec.NumBytes.Val);
   DumpItem('Status   ', HartRec.Status.Pos,    HartRec.Status.Len,    HartRec.Status.Val);
   DumpItem('Data     ', HartRec.Data.Pos,      HartRec.Data.Len,      HartRec.Data.Val);
   DumpItem('CheckSum ', HartRec.CheckSum.Pos,  HartRec.CheckSum.Len,  HartRec.CheckSum.Val);
  end;
 end;
 {
 Extract HART address.
 HartRec should be correct: call HartDecode(HartRec,data) before.
 }
 function HartAddress(var HartRec:THartRec; data:String):String;
 begin
  if (HartRec.CheckSum.Len=1) and (HartRec.Address.Len=1) then begin
   if Length(data)>=HartRec.Address.Pos
   then HartAddress:=Chr(iAnd(Dump2i(Copy(data,HartRec.Address.Pos,1)),15))
   else HartAddress:='';
  end else
  if (HartRec.CheckSum.Len=1) and (HartRec.Address.Len=5) then begin
   if Length(data)>=HartRec.Address.Pos+4
   then HartAddress:=Chr(iAnd(Dump2i(Copy(data,HartRec.Address.Pos,1)),63))+Copy(data,HartRec.Address.Pos+1,4)
   else HartAddress:='';
  end else HartAddress:='';
 end;
 {
 Handle HART request for simulation.
 }
 procedure HandleRequest(Request:String);
 var i,j,k:Integer; r:Real;
 begin
  if Length(Request)>0 then
  if HartDecode(Hart.CmdRec,Request)=0 then begin
   ViewImp(Hex_Encode(Request));
   for i:=0 to nmax do
   if Length(Hart.CmdList[i])>0 then
   if Length(Hart.AnsList[i])>0 then
   if Request=Hart.CmdList[i] then begin
    Hart.Ans:=Hart.AnsList[i];
    if HartDecode(Hart.AnsRec,Hart.Ans)=0 then
    if (Hart.AnsRec.Command.Val=hcReadVar) and (Hart.AnsRec.Data.Len=5) then begin
     r:=Dump2f(ReverseString(Copy(Hart.AnsList[i],Hart.AnsRec.Data.Pos+1,4)));
     r:=r+sin(1*time*timeunits/1000)+0.05*Random(-1,1);
     Hart.Ans:=Copy(Hart.AnsList[i],1,Hart.AnsRec.Data.Pos);
     Hart.Ans:=Hart.Ans+ReverseString(DumpF(r));
     Hart.Ans:=Hart.Ans+Chr(XorCheckSum(Copy(Hart.Ans,Hart.AnsRec.Start.Pos)));
    end;
    ViewExp(Hex_Encode(Hart.Ans));
    bNul(ComWrite(Hart.Ans));
   end;
  end else Trouble('Invalid Request '+Hex_Encode(Request));
 end;
 {
 Driver should have AnalogOutputs, simulator have not.
 }
 function IsDriver:Boolean;
 begin
  IsDriver:=NumAos>0;
 end;
 {
 HART Clear all strings.
 }
 procedure HART_Clear;
 var i:Integer;
 begin
  Hart.Cmd:='';
  Hart.Ans:='';
  for i:=0 to nmax do Hart.CmdList[i]:='';
  for i:=0 to nmax do Hart.AnsList[i]:='';
 end;
 {
 HART Finalization.
 }
 procedure HART_Free;
 begin
  bNul(ComClose);
 end;
 {
 HART Initialization.
 }
 procedure HART_Init;
 var i:Integer;
 begin
  Hart.Chan:=0;
  Hart.Time:=0;
  Hart.TimeOut:=Val(ReadIni('HartTimeOut'));
  if Hart.TimeOut=0 then Hart.TimeOut:=1000;
  if IsDriver
  then Success('Start driver...')
  else Success('Start simulator...');
  {
  Read commands and answers
  }
  for i:=0 to nmax do
  if (i<NumAos) or not IsDriver then begin
   if IsDriver
   then Hart.CmdList[i]:=Hex_Decode(ReadIni('Cmd#'+Str(i)))
   else Hart.CmdList[i]:=Hex_Decode(ReadIni(ReadIni('DrvSection')+' Cmd#'+Str(i)));
   if (HartDecode(Hart.CmdRec,Hart.CmdList[i])<>0) and IsDriver
   then Trouble('Cmd#'+Str(i)+' = '+Hex_Encode(Hart.CmdList[i]))
   else Success('Cmd#'+Str(i)+' = '+Hex_Encode(Hart.CmdList[i]));
   if IsDriver
   then Hart.AnsList[i]:=Hex_Decode(ReadIni('Ans#'+Str(i)))
   else Hart.AnsList[i]:=Hex_Decode(ReadIni(ReadIni('DrvSection')+' Ans#'+Str(i)));
   if (HartDecode(Hart.AnsRec,Hart.AnsList[i])<>0) and IsDriver
   then Trouble('Ans#'+Str(i)+' = '+Hex_Encode(Hart.AnsList[i]))
   else Success('Ans#'+Str(i)+' = '+Hex_Encode(Hart.AnsList[i]));
   if HartDecode(Hart.CmdRec,Hart.CmdList[i])=0 then
   if HartDecode(Hart.AnsRec,Hart.AnsList[i])=0 then
   if HartAddress(Hart.CmdRec,Hart.CmdList[i])=HartAddress(Hart.AnsRec,Hart.AnsList[i])
   then Success('Adr#'+Str(i)+' = '+Hex_Encode(HartAddress(Hart.CmdRec,Hart.CmdList[i]))
                             +' = '+Hex_Encode(HartAddress(Hart.AnsRec,Hart.AnsList[i])))
   else Trouble('Adr#'+Str(i)+' = '+Hex_Encode(HartAddress(Hart.CmdRec,Hart.CmdList[i]))
                             +' ! '+Hex_Encode(HartAddress(Hart.AnsRec,Hart.AnsList[i])));
  end;
  {
  Open COM port
  }
  if not ComOpen(ReadIni('PortSection'))
  then Trouble('Could not open COM, stopped.')
  else bNul(ComClear);
 end; 
 {
 HART Polling procedure.
 }
 procedure HART_Poll;
 var r:Real;
 begin
  {
  HART protocol - simulator...
  }
  if not IsDriver then HandleRequest(ComRead(255));
  {
  HART protocol - driver...
  }
  if IsDriver then begin
   if Hart.Time=0 then begin
    Hart.Chan:=(Hart.Chan+1) mod NumAos;
    bNul(ComClear);
    Hart.Ans:='';
    Hart.Cmd:=Hart.CmdList[Hart.Chan];
    if Length(Hart.Cmd)>0 then begin
     SendPort(Hart.Cmd);
     Hart.Time:=msecnow;
    end; 
   end else
   if msecnow>Hart.Time+Hart.TimeOut then begin
    Trouble('TIMEOUT '+Hex_Encode(Hart.Ans));
    Hart.Cmd:='';
    Hart.Ans:='';
    Hart.Time:=0;
   end else begin
    Hart.Ans:=Hart.Ans+ComRead(255);
    if HartDecode(Hart.AnsRec,Hart.Ans)=0 then begin
     if iAnd(DebugFlags,dfViewImp)<>0 then ViewImp(Hex_Encode(Hart.Ans));
     if iAnd(DebugFlags,dfViewDump)<>0 then HartDump(Hart.AnsRec,Hart.Ans);
     if HartDecode(Hart.CmdRec,Hart.Cmd)=0 then
     if HartAddress(Hart.AnsRec,Hart.Ans)=HartAddress(Hart.CmdRec,Hart.Cmd) then
     if (Hart.AnsRec.Command.Val=hcReadVar) and (Hart.AnsRec.Data.Len=5) then begin
      r:=Dump2f(ReverseString(Copy(Hart.Ans,Hart.AnsRec.Data.Pos+1,4)));
      if RefAo(Hart.Chan)<>0 then bNul(PutAo(Hart.Chan,time,r));
     end;
     Hart.Cmd:='';
     Hart.Ans:='';
     Hart.Time:=0;
    end;
   end;
  end;
 end;
 {
 GUI Handler to process user input...
 }
 procedure GUIHandler;
 var s:String; ClickCurve:Integer;
  procedure Cleanup;
  begin
   s:=''; ClickCurve:=0;
  end;
 begin
  Cleanup;
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Handle sensor clicks...
     }
     if IsSameText(ClickSensor,'HELP') then begin
      Cron('@Browse '+DaqFileRef(ReadIni('[DAQ] HelpFile'),'.htm'));
      bNul(Voice(snd_Click));
     end;
     {
     Select Plot & Tab windows by curve...
     }
     ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
     if IsRefCurve(ClickCurve) then begin
      iNul(WinSelectByCurve(ClickCurve,ClickCurve));
      bNul(Voice(snd_Wheel));
     end;
     {
     Console commands: @url_encoded_sensor ...
     }
     if LooksLikeCommand(ClickSensor) then begin
      DevSendCmdLocal(url_decode(ClickSensor));
      bNul(Voice(snd_Click));
     end;
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   {
   Warning,Information.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  Cleanup;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  HART_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  HART_Init;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  HART_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  HART_Poll;
  GUIHandler;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid:Integer;
  procedure Cleanup;
  begin
   cmd:=''; arg:='';
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
