////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для построения экспоненты МНК f(t)=a+b*exp(-c*(t-d))
по точкам кривой в области между правой и левой вертикальной границей
"Области интересов" ROI. "Область интересов" должна быть задана в окне
- источнике перед вызовом утилиты при помощи двух маркеров ROI. Левая
граница ROI принимается за параметр d.
Эта утилита может также служить прототипом для других утилит
фитирования кривых методом наименьших квадратов.
[]
[Manual.Eng]
This plugin uses to find exponent f(t)=a+b*exp(-c*(t-d))
by curve points between left and right "Region Of Interest",
ROI, markers. Set "Region Of Interest" before call plugin.
Uses ROI markers in source data window.
You may use this plugin as template for other plugins
to fit data curves by Least Squares method.
[]
[Arguments.Rus]
Имя окна = Результат:"Экспонента кривой ROI X"
Метод[0..8] = 1
[]
[Arguments.Eng]
Caption = Result:"Expopent of curve ROI X"
Method[0..8] = 1
[]
}

library _curve_exponent_roi_x;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

const
 nmax		= 3;		// Размерность параметров.
 mmax		= 10000;	// Максимальное число входных точек.
 NumPoints	= 200;	// Число точек на графике экспоненты.

type
 TMinData = packed record	// Рабочий массив минимизации
  CrwApi    : TCrwApi;		// Ссылка на интерфейс
  m         : Integer;		// Число точек входных данных
  n         : Integer;		// Размерность массива параметров
  CallCount : Integer;		// Счетчик вызовов функции
  MaxCount  : Integer;		// Ограничение на число вызовов функции
  Method    : integer;		// Метод минимизации
  tolx      : Double;		// Критерий точности решения по аргументу
  tolf      : Double;		// Критерий точности решения по значению функции
  tolg      : Double;		// Критерий точности решения по градиенту
  Inf       : Double;		// Оценка нижней грани функции, для суммы квадратов ноль
  Size      : Double;		// Начальный размер симплекса, обычно 0.1
  Step      : Double;		// Начальный шаг численного дифференцирования
  EvalTol   : Double;   	// Оценка точности вычисления функции
  fscale    : Double;		// Масштабный фактор функции
  f         : Double; 		// Значение функции, то есть сумма квадратов
  x,g       : packed array[0..nmax-1] of Double;	// Параметры и градиент
  xLo,xHi   : packed array[0..nmax-1] of Double;	// Диапазон параметров
  t,y,w     : packed array[0..mmax-1] of Double;	// Входные данные
  Fix       : packed array[0..nmax-1] of Boolean;	// Фиксация параметров
  a         : packed array[0..(nmax+10)*(nmax+1)-1] of Double; // Рабочий массив
  t0        : Double;		// Постоянный сдвиг переменной, начало ROI
  Sigma     : Double;		// Стандартное отклонение
  SigmaRel  : Double;		// Стандартное отклонение в %
 end;

 //
 // Функция обновления экрана
 //
procedure Refreshment(CrwApi:TCrwApi; Delta:Integer);
const LastTicks : Cardinal = 0;
begin
 if LastTicks=0 then LastTicks:=GetTickCount;
 if abs(GetTickCount-LastTicks) > Delta then begin
  with CrwApi,GuiApi do begin
   ApplicationProcessMessages;
   UpdateSystemConsole;
  end;
  LastTicks:=GetTickCount;
 end;
end;

 //
 // Функция вычисляет f(t)=x0+x1*exp(-x2*(t-t0)) и градиент
 //
function Fit(t:Double; n:Integer; var x:array of Double; var f:Double;
                    var g:array of Double; Custom:Pointer):Integer;
var MinData:^TMinData;
 function exp(x:double):double;
 begin
  Result:=system.exp(max(-300,Min(300,x)));
 end;
begin
 MinData:=Custom;
 with MinData.CrwApi,SysApi do
 if n<>3 then Result:=mecBadDim else begin
  Result:=mecOk;
  f:=x[0];
  g[0]:=1;
  g[1]:=exp(-x[2]*(t-MinData.t0));
  g[2]:=-(t-MinData.t0)*x[1]*g[1];
  f:=f+x[1]*g[1];
  if isNan(f) or isInf(f) then Result:=mecBadVal;
 end;
end;

 //
 // Функция отчитывается о ходе итераций
 //
procedure Report(m:Integer; const t,y,w:array of Double; n:Integer;
                    const x:array of Double; const f:Double; const g:array of Double;
                    Custom:Pointer; Count:Integer; const Method,Comment:LongString);
const LastTicks : Cardinal = 0;
var MinData:^TMinData;
begin
 MinData:=Custom;
 if LastTicks=0 then LastTicks:=GetTickCount;
 if abs(GetTickCount-LastTicks) > 100 then begin
  writeln(Format('F = %.7g, X = %.5g, %.5g, %.5g',[f,x[0],x[1],x[2]]));
  Refreshment(MinData.CrwApi,100);
  LastTicks:=GetTickCount;
 end;
end;


//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNanInf;
var
 p : TPoint2D;
 Roi,R : TRect2D;
 i, c1, c2, err : Integer;
 MinData : ^TMinData;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('Неверное значение Target!',
                                   'Invalid Target!'));
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  Roi:=WindowRoi[swin];
  if isNAN(Roi.A.X+Roi.A.Y)
  then Raise EDanApi.Create(RusEng('Левый маркер РОИ не определен!',
                                   'Left ROI marker is not defined!'));
  if isNAN(Roi.B.X+Roi.B.Y)
  then Raise EDanApi.Create(RusEng('Правый маркер РОИ не определен!',
                                   'Right ROI marker is not defined!'));
  Roi:=Rect2DValidate(Roi);
  Roi.A.Y:=_MinusInf;
  Roi.B.Y:=_PlusInf;
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  if SelectedCurve[swin]=0 
  then Raise EDanApi.Create(RusEng('Не выбрана кривая для фитирования!',
                                   'Curve is not selected for fitting!'));
  c1:=SelectedCurve[swin];
  if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
  then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для фитирования!',
                                          'Curve %d is not convenient for fitting!'),[c1]));
  //
  // Выделяем массив для минимизации. Обязательно инициализируем ссылку CrwApi.
  //
  MinData:=Allocate(Sizeof(MinData^));
  try
   MinData.CrwApi:=CrwApi;
   //
   // Накапливаем нужные точки в массивах MinData, находим границы R
   //
   R:=Rect2D(Point2D(_PlusInf,_PlusInf),Point2D(_MinusInf,_MinusInf));
   for i:=0 to CurveLength[c1]-1 do begin
    if Rect2dContainsPoint(Roi,CurvePoint[c1,i]) then begin
     if MinData.m=mmax 
     then Raise EDanApi.Create(RusEng('Слишком много точек!','Too many points!'));
     with CurvePoint[c1,i] do begin
      MinData.t[MinData.m]:=x;
      MinData.y[MinData.m]:=y;
      MinData.w[MinData.m]:=1;
      inc(MinData.m);
      R:=Rect2DUnion(R,Rect2D(Point2D(x,y),Point2D(x,y)));
     end;
    end;
    Refreshment(CrwApi,100);
   end;
   if MinData.m<3 then Raise EDanApi.Create(RusEng('Мало точек!','No points!'));
   //
   // Инициализируем и вызываем минимизацию ...
   // При вычислении масштабов,допусков и начального приближения
   // используются эмпирические допущения и вычисленные пределы R.
   //
   with MinData^ do begin
    n:=3;
    t0:=Roi.A.X;
    MaxCount:=5000;
    if not GetArgumentAsInteger(RusEng('Метод[0..8]','Method[0..8]'),Method)
    then Method:=mDavidonFletcherPowell;
    tolx:=1e-7;
    tolf:=1e-7;
    tolg:=1e-7;
    inf:=0;
    size:=0.1;
    step:=1e-3;
    evaltol:=1e-14*abs(Rect2DSize(R).Y);
    fscale:=sqr(Rect2DSize(R).Y);
    x[0]:=Rect2DCenter(R).Y;
    x[1]:=Rect2DSize(R).Y/2;
    x[2]:=Ln(Rect2DSize(R).Y)/Rect2DSize(R).X/2;
    xLo[0]:=0; xHi[0]:=Rect2DSize(R).Y/2;
    xLo[1]:=0; xHi[1]:=Rect2DSize(R).Y/2;
    xLo[2]:=0; xHi[2]:=Ln(Rect2DSize(R).Y)/Rect2DSize(R).X/2;
    for i:=0 to n-1 do Fix[i]:=false;
    err:=LSQFit(m,t,y,w,Fit,Report,n,x,f,g,MinData,
                CallCount,MaxCount,Method,tolx,tolf,tolg,
                Inf,Size,Step,EvalTol,a,Fix,fscale,xLo,xHi);
    if err<>0
    then Raise EDanApi.Create(Format(RusEng('Ошибка %d в LsqFit!',
                                            'Error %d in LsqFit!'),[err]));
    Sigma:=sqrt(f/m);
    SigmaRel:=Sigma/Rect2DSize(R).Y*100;
    writeln('f(t)=a+b*exp(-c*(t-d))');
    writeln(Format('a = %g',[x[0]]));
    writeln(Format('b = %g',[x[1]]));
    writeln(Format('c = %g',[x[2]]));
    writeln(Format('a = %g',[t0]));
    writeln(Format('Call count      = %d',[CallCount]));
    writeln(Format('Summ of squares = %g',[f]));
    writeln(Format('Standard deviation = %g (%.5g %s)',[Sigma,SigmaRel,'%']));
    WindowTitle[twin]:=Format('^CSigma=%.5g (%.3g %s)^N^L f(t)=a+b*exp(-c*(t-d))',
                              [Sigma,SigmaRel,'%']);
    WindowLegend[twin]:=Format('^Rt  ^N^Ca=%.7g, b=%.7g, c=%.7g, d=%.7g',
                              [x[0],x[1],x[2],MinData.t0])
   end;
   R:=Roi;
   Rect2DGrow(R,Rect2DSize(R).X/2,Rect2DSize(R).Y/2);
   c2:=CreateCurve(twin);
   CurveAssign(c2,c1);
   c2:=CreateCurve(twin,'Fit:'+CurveName[c1],clWhite,$10);
   CurveLength[c2]:=NumPoints;
   for i:=0 to NumPoints-1 do begin
    p.x:=R.A.X+(R.B.X-R.A.X)*i/(NumPoints-1);
    err:=Fit(p.x,MinData.n,MinData.x,MinData.f,MinData.g,MinData);
    if err<>0
    then Raise EDanApi.Create(Format(RusEng('Ошибка %d в fit!',
                                            'Error %d in fit!'),[err]));
    p.y:=MinData.f;
    CurvePoint[c2,i]:=p;
    Refreshment(CrwApi,100);
   end;
   SelectedCurve[twin]:=0;
  finally
   Deallocate(Pointer(MinData));
  end;
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
