package flashx.textLayout.compose
{
   import flashx.textLayout.container.ContainerController;
   import flashx.textLayout.formats.BlockProgression;
   import flashx.textLayout.tlf_internal;
   import flashx.textLayout.formats.VerticalAlign;
   
   use namespace tlf_internal;
   
   public final class VerticalJustifier
   {
       
      public function VerticalJustifier()
      {
         super();
      }
      
      [ArrayElementType("text.IVerticalJustificationLine")]
      public static function applyVerticalAlignmentToColumn(cont:ContainerController, verticalAlignAttr:String, lines:Array, startIndex:int, numLines:int) : void
      {
         var helper:IVerticalAdjustmentHelper = null;
         var i:int = 0;
         if(cont.rootElement.computedFormat.blockProgression == BlockProgression.RL)
         {
            helper = new RL_VJHelper(ContainerController(cont));
         }
         else
         {
            helper = new TB_VJHelper(ContainerController(cont));
         }
         switch(verticalAlignAttr)
         {
            case VerticalAlign.MIDDLE:
               helper.computeMiddleAdjustment(lines[startIndex + numLines - 1]);
               for(i = startIndex; i < startIndex + numLines; i++)
               {
                  helper.applyMiddleAdjustment(lines[i]);
               }
               break;
            case VerticalAlign.BOTTOM:
               helper.computeBottomAdjustment(lines[startIndex + numLines - 1]);
               for(i = startIndex; i < startIndex + numLines; i++)
               {
                  helper.applyBottomAdjustment(lines[i]);
               }
               break;
            case VerticalAlign.JUSTIFY:
               helper.computeJustifyAdjustment(lines,startIndex,numLines);
               helper.applyJustifyAdjustment(lines,startIndex,numLines);
         }
      }
   }
}

import flashx.textLayout.compose.IVerticalJustificationLine;

interface IVerticalAdjustmentHelper
{
    
   function computeMiddleAdjustment(param1:IVerticalJustificationLine) : void;
   
   function applyMiddleAdjustment(param1:IVerticalJustificationLine) : void;
   
   function computeBottomAdjustment(param1:IVerticalJustificationLine) : void;
   
   function applyBottomAdjustment(param1:IVerticalJustificationLine) : void;
   
   function computeJustifyAdjustment(param1:Array, param2:int, param3:int) : void;
   
   function applyJustifyAdjustment(param1:Array, param2:int, param3:int) : void;
}

import flashx.textLayout.container.ContainerController;
import flashx.textLayout.compose.IVerticalJustificationLine;
import flashx.textLayout.compose.TextFlowLine;
import flashx.textLayout.tlf_internal;

use namespace tlf_internal;

class TB_VJHelper implements IVerticalAdjustmentHelper
{
    
   private var _textFrame:ContainerController;
   
   private var adj:Number;
   
   function TB_VJHelper(tf:ContainerController)
   {
      super();
      this._textFrame = tf;
   }
   
   private function getBottomOfLine(line:IVerticalJustificationLine) : Number
   {
      return this.getBaseline(line) + line.descent;
   }
   
   private function getBaseline(line:IVerticalJustificationLine) : Number
   {
      if(line is TextFlowLine)
      {
         return line.y + line.ascent;
      }
      return line.y;
   }
   
   private function setBaseline(line:IVerticalJustificationLine, pos:Number) : void
   {
      if(line is TextFlowLine)
      {
         line.y = pos - line.ascent;
      }
      else
      {
         line.y = pos;
      }
   }
   
   public function computeMiddleAdjustment(lastLine:IVerticalJustificationLine) : void
   {
      var frameBottom:Number = this._textFrame.compositionHeight - Number(this._textFrame.effectivePaddingBottom);
      this.adj = (frameBottom - this.getBottomOfLine(lastLine)) / 2;
      if(this.adj < 0)
      {
         this.adj = 0;
      }
   }
   
   public function applyMiddleAdjustment(line:IVerticalJustificationLine) : void
   {
      line.y = line.y + this.adj;
   }
   
   public function computeBottomAdjustment(lastLine:IVerticalJustificationLine) : void
   {
      var frameBottom:Number = this._textFrame.compositionHeight - Number(this._textFrame.effectivePaddingBottom);
      this.adj = frameBottom - this.getBottomOfLine(lastLine);
      if(this.adj < 0)
      {
         this.adj = 0;
      }
   }
   
   public function applyBottomAdjustment(line:IVerticalJustificationLine) : void
   {
      line.y = line.y + this.adj;
   }
   
   [ArrayElementType("text.compose.IVerticalJustificationLine")]
   public function computeJustifyAdjustment(lineArray:Array, firstLineIndex:int, numLines:int) : void
   {
      this.adj = 0;
      if(numLines == 1)
      {
         return;
      }
      var firstLine:IVerticalJustificationLine = lineArray[firstLineIndex];
      var firstBaseLine:Number = this.getBaseline(firstLine);
      var lastLine:IVerticalJustificationLine = lineArray[firstLineIndex + numLines - 1];
      var frameBottom:Number = this._textFrame.compositionHeight - Number(this._textFrame.effectivePaddingBottom);
      var allowance:Number = frameBottom - this.getBottomOfLine(lastLine);
      if(allowance < 0)
      {
         return;
      }
      var lastBaseLine:Number = this.getBaseline(lastLine);
      this.adj = allowance / (lastBaseLine - firstBaseLine);
   }
   
   [ArrayElementType("text.compose.IVerticalJustificationLine")]
   public function applyJustifyAdjustment(lineArray:Array, firstLineIndex:int, numLines:int) : void
   {
      var line:IVerticalJustificationLine = null;
      var currBaseLine:Number = NaN;
      var currBaseLineUnjustified:Number = NaN;
      if(Boolean(numLines == 1) || Boolean(this.adj == 0))
      {
         return;
      }
      var firstLine:IVerticalJustificationLine = lineArray[firstLineIndex];
      var prevBaseLine:Number = this.getBaseline(firstLine);
      var prevBaseLineUnjustified:Number = prevBaseLine;
      for(var i:int = 1; i < numLines; i++)
      {
         line = lineArray[i + firstLineIndex];
         currBaseLineUnjustified = this.getBaseline(line);
         currBaseLine = prevBaseLine + (currBaseLineUnjustified - prevBaseLineUnjustified) * (1 + this.adj);
         this.setBaseline(line,currBaseLine);
         prevBaseLineUnjustified = currBaseLineUnjustified;
         prevBaseLine = currBaseLine;
      }
   }
}

import flashx.textLayout.container.ContainerController;
import flashx.textLayout.compose.IVerticalJustificationLine;
import flashx.textLayout.tlf_internal;

use namespace tlf_internal;

class RL_VJHelper implements IVerticalAdjustmentHelper
{
    
   private var _textFrame:ContainerController;
   
   private var adj:Number = 0;
   
   function RL_VJHelper(tf:ContainerController)
   {
      super();
      this._textFrame = tf;
   }
   
   public function computeMiddleAdjustment(lastTextLine:IVerticalJustificationLine) : void
   {
      var frameWidth:Number = this._textFrame.compositionWidth - Number(this._textFrame.effectivePaddingLeft);
      this.adj = (frameWidth + lastTextLine.x - lastTextLine.descent) / 2;
      if(this.adj < 0)
      {
         this.adj = 0;
      }
   }
   
   public function applyMiddleAdjustment(line:IVerticalJustificationLine) : void
   {
      line.x = line.x - this.adj;
   }
   
   public function computeBottomAdjustment(lastTextLine:IVerticalJustificationLine) : void
   {
      var frameWidth:Number = this._textFrame.compositionWidth - Number(this._textFrame.effectivePaddingLeft);
      this.adj = frameWidth + lastTextLine.x - lastTextLine.descent;
      if(this.adj < 0)
      {
         this.adj = 0;
      }
   }
   
   public function applyBottomAdjustment(line:IVerticalJustificationLine) : void
   {
      line.x = line.x - this.adj;
   }
   
   [ArrayElementType("text.compose.IVerticalJustificationLine")]
   public function computeJustifyAdjustment(lineArray:Array, firstLineIndex:int, numLines:int) : void
   {
      this.adj = 0;
      if(numLines == 1)
      {
         return;
      }
      var firstLine:IVerticalJustificationLine = lineArray[firstLineIndex];
      var firstBaseLine:Number = firstLine.x;
      var lastLine:IVerticalJustificationLine = lineArray[firstLineIndex + numLines - 1];
      var frameLeft:Number = Number(this._textFrame.effectivePaddingLeft) - this._textFrame.compositionWidth;
      var allowance:Number = lastLine.x - lastLine.descent - frameLeft;
      if(allowance < 0)
      {
         return;
      }
      var lastBaseLine:Number = lastLine.x;
      this.adj = allowance / (firstBaseLine - lastBaseLine);
   }
   
   [ArrayElementType("text.IVerticalJustificationLine")]
   public function applyJustifyAdjustment(lineArray:Array, firstLineIndex:int, numLines:int) : void
   {
      var line:IVerticalJustificationLine = null;
      var currBaseLine:Number = NaN;
      var currBaseLineUnjustified:Number = NaN;
      if(Boolean(numLines == 1) || Boolean(this.adj == 0))
      {
         return;
      }
      var firstLine:IVerticalJustificationLine = lineArray[firstLineIndex];
      var prevBaseLine:Number = firstLine.x;
      var prevBaseLineUnjustified:Number = prevBaseLine;
      for(var i:int = 1; i < numLines; i++)
      {
         line = lineArray[i + firstLineIndex];
         currBaseLineUnjustified = line.x;
         currBaseLine = prevBaseLine - (prevBaseLineUnjustified - currBaseLineUnjustified) * (1 + this.adj);
         line.x = currBaseLine;
         prevBaseLineUnjustified = currBaseLineUnjustified;
         prevBaseLine = currBaseLine;
      }
   }
}
