Attribute VB_Name = "P10D10Model"
'*****************************************************************************
' Compiler:           Visual BASIC 6.0
' Operating system:   WIN NT 4.0 SP 6.0
' Author:             Gerald Burgemeister
' Date:               30.08.2002
'
' This module contains the functions to communicate with a D10 or P10 device
'
'
'*****************************************************************************

Option Explicit
Option Base 0

Const MAX_TIMEOUT = 10
Const DELAY_TIME = 100
Const TEMP2POLL_DELAY = 500

''''''''''''''''''''''''''''''''''''''''''''''''
' Reads the transmitter's ZeroScale (span start)
'
' <- zeroScale: the value
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function getZeroScale(ByRef zeroScale As String)
  Dim result As String
  Dim ret As Boolean
  Dim val As Single
  Dim expo As Integer
  Dim timeout As Integer
  
  ret = False
  timeout = 0
    
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
    
      ' send the string
      MainWindow.COM.Output = "MA" & Chr(0) & createChecksum("MA" & Chr(0)) & vbCr
      
      ' give the device a little time
      delay (DELAY_TIME)
 
      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
        
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 6 Then
          ' calculate the zero scale as described in the interface protocol
          ' chapter: 3.1.1
          val = (Asc(Mid(result, 2, 1)) * 256) + (Asc(Mid(result, 3, 1)) And &H7F)
      
          If (Asc(Mid(result, 3, 1)) And &H80) = &H80 Then
            val = val * (-1)
          End If
      
          expo = Asc(Mid(result, 4, 1)) And &H3F
      
          If (Asc(Mid(result, 4, 1)) And &H40) = &H40 Then
            expo = expo * (-1)
          End If
      
          val = val * (10 ^ expo)
      
          zeroScale = CStr(val)
          ret = True
       
        End If
      End If
      
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
  
  getZeroScale = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' Reads the transmitter's FullScale (span end)
'
' <- fullScale: the value
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function getFullScale(ByRef fullScale As String)
  Dim result As String
  Dim ret As Boolean
  Dim val As Single
  Dim expo As Integer
  Dim timeout As Integer
  
  ret = False
  timeout = 0
  
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
      
      ' send the string
      MainWindow.COM.Output = "ME" & Chr(0) & createChecksum("ME" & Chr(0)) & vbCr
      
      ' give the device a little time
      delay (DELAY_TIME)
  
      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
        
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 6 Then
          ' calculate the full scale as described in the interface protocol
          ' chapter: 3.1.2
          val = ((Asc(Mid(result, 2, 1)) And &H7F) * 256) + Asc(Mid(result, 3, 1))
      
          If (Asc(Mid(result, 2, 1)) And &H80) = &H80 Then
            val = val * (-1)
          End If
      
          expo = Asc(Mid(result, 4, 1)) And &H3F
      
          If (Asc(Mid(result, 4, 1)) And &H40) = &H40 Then
            expo = expo * (-1)
          End If
      
          val = val * (10 ^ expo)
      
          fullScale = CStr(val)
          ret = True
          
        End If
      End If
      
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
 
  getFullScale = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' Reads the transmitter's head number
'
' <- headnr: the value
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function getHeadNumber(ByRef headnr As String)
  Dim result As String
  Dim ret As Boolean
  Dim timeout As Integer
  
  ret = False
  timeout = 0
 
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
    
      ' send the string
      MainWindow.COM.Output = "KN" & Chr(0) & createChecksum("KN" & Chr(0)) & vbCr
      
      ' give the device a little time
      delay (DELAY_TIME)
 
      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
        
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 7 Then
          ' see interface protocol chapter: 3.1.6
          headnr = Mid(result, 2, 1) + Mid(result, 3, 1) + _
                   Mid(result, 4, 1) + Mid(result, 5, 1)
      
          ret = True
        End If
      End If
      
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
 
  getHeadNumber = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' Reads the transmitter's pressure unit by
' utilizing the P-Faktor of a pressure request
'
' <- unit: the value
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function getPressureUnit(ByRef unit As String)
  Dim result As String
  Dim ret As Boolean
  Dim expo As Integer
  Dim timeout As Integer
  
  ret = False
  timeout = 0
  
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
      
      ' send the string
      MainWindow.COM.Output = "PZ" & Chr(0) & createChecksum("PZ" & Chr(0)) & vbCr
      
      ' give the device a little time
      delay (DELAY_TIME)

      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
        
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 6 Then
          ' the unit is coded in the P-Faktor,
          ' thus this is the only byte we are intereseted in
          expo = Asc(Mid(result, 4, 1))
          expo = expo And &H7
          
          Select Case expo
            Case 0
              unit = "bar"
            Case 1
              unit = "psi"
            Case 2
              unit = "mmHg"
            Case 3
              unit = "inHg"
            Case 4
              unit = "mWs"
            Case 5
              unit = "inWs"
          End Select
  
          ret = True
        End If
      End If
      
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
 
  getPressureUnit = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' Reads the currently meassured pressure
'
' <- press: the value
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function getPressure(ByRef press As String)
  Dim result As String
  Dim ret As Boolean
  Dim val As Single
  Dim expo As Integer
  Dim timeout As Integer
  
  ret = False
  timeout = 0
  
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
    
      ' send the string
      MainWindow.COM.Output = "PZ" & Chr(0) & createChecksum("PZ" & Chr(0)) & vbCr
      
      ' give the device a little time
      delay (DELAY_TIME)

      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
      
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 6 Then
          ' calculate the pressure value as described in the interface protocol
          ' chapter: 3.1.3
          val = ((Asc(Mid(result, 2, 1)) And &H7F) * 256) + Asc(Mid(result, 3, 1))
    
          If (Asc(Mid(result, 2, 1)) And &H80) = &H80 Then
            val = val * (-1)
          End If
    
          expo = Asc(Mid(result, 4, 1)) And &H38
          expo = expo / 8
    
          If (Asc(Mid(result, 4, 1)) And &H40) = &H40 Then
            expo = expo * (-1)
          End If
    
          val = val * (10 ^ expo)
    
          press = CStr(val)
          ret = True
        End If
      End If
    
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
 
  getPressure = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' Switches a D10 device temporarily into pollmode
'
' Return: true, if the function was succesful,
'              otherwise false
Public Function switchD10TempToPollmode()
  Dim result As String
  Dim ret As Boolean
  Dim timeout As Integer
  
  ret = False
  timeout = 0
  
  Do
    If MainWindow.COM.PortOpen Then
      ' clear input buffer
      MainWindow.COM.InBufferCount = 0
      
      ' send the string
      MainWindow.COM.Output = "SO" & Chr(&HFF) & createChecksum("SO" & Chr(&HFF)) & vbCr
      
      ' give the device a little time
      delay (TEMP2POLL_DELAY)
            
      If MainWindow.COM.InBufferCount Then
        ' read the answer
        result = MainWindow.COM.Input
        
        ' and check it. For a more reliable check the checksum of
        ' the received telegram should be considered
        If Len(result) = 5 Then
          ' see interface protocol chapter: 2
          If ((Mid(result, 1, 1) = "s") And _
              (Mid(result, 2, 1) = "o")) Then
      
            ret = True
          End If
        End If
      End If
      
      timeout = timeout + 1
    Else
      Exit Function
    End If
  Loop Until (ret Or (timeout > MAX_TIMEOUT))
 
  If timeout > MAX_TIMEOUT Then
    ret = False
  End If
 
  switchD10TempToPollmode = ret
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' calculates the checksum. See chapter: 6 cs
'
' -> str: String of which a checksum should
'         be calculated
'
' Return: checksum
Public Function createChecksum(ByVal str As String)
  Dim i As Integer
  Dim cs As Integer
  
  cs = 0
  
  For i = 1 To Len(str)
    cs = cs + Asc(Mid(str, i, 1))
  Next
  
  ' we only need the low byte
  cs = CByte(cs And &HFF)
  
  'build the 2th complement
  cs = cs Xor &HFF
  cs = cs + 1
  
  createChecksum = Chr(cs)
End Function

''''''''''''''''''''''''''''''''''''''''''''''''
' causes a delay. Messages send to the program
' are handled during the delay
'
' -> timeInms: delaytime in ms
Public Sub delay(ByVal timeInms As Single)
  Dim Start
 
  Start = Timer * 1000
  Do While (Timer * 1000) < Start + timeInms
    DoEvents
  Loop
End Sub
