unit uhistory;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, StdCtrls, lazUTF8, Clipbrd;

type

  PStep = ^TStep;

  TStep = record
    BytePos  : SizeInt;
    SelStart : SizeInt;
    InsStr   : string;
    DelStr   : string;
  end;

  { THistory }

  THistory = class
  private
    FMemo        : TMemo;
    FSteps       : TList;         // History Records
    FIndex       : Integer;       // Based 0
    FOldOnChange : TNotifyEvent;
    FInEdit      : Boolean;
    FPrevContent : string;

    function    GetStep(AIndex: Integer): PStep;
    function    CurStep: PStep;

    procedure   AddStep(ABytePos, ASelStart: SizeInt; AInsStr, ADelStr: string);
    procedure   DelStep(Index: Integer);

    procedure   MemoOnChange(Sender: TObject);

    function    StrDiff(const ACurContent: string;
                        out ABytePos, ASelStart: SizeInt;
                        out AInsStr, ADelStr: string): Boolean;
  public
    constructor Create(AMemo: TMemo);
    destructor  Destroy; override;

    function    CanUndo: Boolean;
    function    CanRedo: Boolean;
    procedure   Undo;
    procedure   Redo;
    procedure   Paste;
  end;

implementation

{ THistory }

function THistory.GetStep(AIndex: Integer): PStep;
begin
  Result := PStep(FSteps[AIndex]);
end;

function THistory.CurStep: PStep;
begin
  Result := GetStep(FIndex);
end;

procedure THistory.AddStep(ABytePos, ASelStart: SizeInt; AInsStr, ADelStr: string);
begin
  DelStep(FIndex + 1);

  FSteps.Add(new(PStep));
  Inc(FIndex);
  with CurStep^ do begin
    BytePos  := ABytePos;
    SelStart := ASelStart;
    InsStr   := AInsStr;
    DelStr   := ADelStr;
  end;
end;

procedure THistory.DelStep(Index: Integer);
var
  i: Integer;
begin
  for i := FSteps.Count - 1 downto Index do begin
    GetStep(i)^.InsStr := '';
    GetStep(i)^.DelStr := '';
    dispose(GetStep(i));
    FSteps.Delete(i);
  end;
  FIndex := Index - 1;
end;

procedure THistory.MemoOnChange(Sender: TObject);
var
  CurContent, InsStr, DelStr: string;
  BytePos, SelStart: SizeInt;
  // TickCount: Int64;
begin
  // TickCount := GetTickCount64;
  CurContent := FMemo.Text; // Only get FMemo.Text one times.
  if FInEdit and StrDiff(CurContent, BytePos, SelStart, InsStr, DelStr) then
    AddStep(BytePos, SelStart, InsStr, DelStr);
  FPrevContent := CurContent;
  // writeln(GetTickCount64 - TickCount);

  if Assigned(FOldOnChange) then
    FOldOnChange(Sender);
end;

function THistory.StrDiff(const ACurContent: string;
  out ABytePos, ASelStart: SizeInt;
  out AInsStr, ADelStr: string): Boolean;
var
  CurStart, CurPos, CurEnd, PrevPos: PChar;
  CurLen, PrevLen, DiffLen: SizeInt;
  CharLen: Integer;
begin
  CurStart := PChar(ACurContent);
  CurPos := CurStart;
  PrevPos := PChar(FPrevContent);

  CurLen := Length(ACurContent);   // Use Length(string) DO NOT use Length(PChar)
  PrevLen := Length(FPrevContent);
  DiffLen := CurLen - PrevLen;

  if DiffLen < 0 then
    CurEnd := CurPos + CurLen - 1
  else if DiffLen > 0 then
    CurEnd := CurPos + PrevLen - 1
  else begin
    Result := False;
    Exit;
  end;

  while (CurPos <= CurEnd) do begin
    if CurPos^ <> PrevPos^ then Break;
    Inc(CurPos);
    Inc(PrevPos);
  end;

  Utf8TryFindCodepointStart(CurStart, CurPos, CharLen);
  ABytePos := CurPos - CurStart + 1;

  if DiffLen > 0 then begin
    AInsStr := Copy(ACurContent, ABytePos, DiffLen);
    ADelStr := '';
  end else begin
    AInsStr := '';
    ADelStr := Copy(FPrevContent, ABytePos, -DiffLen);
  end;
  ASelStart := FMemo.SelStart;

  Result := True;
end;

constructor THistory.Create(AMemo: TMemo);
begin
  inherited Create;
  FSteps := TList.Create;
  FMemo := AMemo;
  FOldOnChange := FMemo.OnChange;
  FMemo.OnChange := @MemoOnChange;
  FPrevContent := FMemo.Text;
  FIndex := -1;
  FInEdit := True;
end;

destructor THistory.Destroy;
begin
  FInEdit := False;
  FMemo.OnChange := FOldOnChange;
  FMemo := nil;
  FPrevContent := '';
  DelStep(0);
  FSteps.Free;
  inherited Destroy;
end;

function THistory.CanUndo: Boolean;
begin
  Result := FIndex >= 0;
end;

function THistory.CanRedo: Boolean;
begin
  Result := FIndex < FSteps.Count - 1;
end;

procedure THistory.Undo;
var
  NewSelStart: SizeInt;
begin
  if FIndex < 0 then Exit;
  FInEdit := False;

  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if InsStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(InsStr));
      NewSelStart := SelStart - UTF8LengthFast(InsStr);
    end;
    if DelStr <>'' then begin
      Insert(DelStr, FPrevContent, BytePos);
      NewSelStart := SelStart + UTF8LengthFast(DelStr);
    end;
  end;
  FMemo.Text := FPrevContent;
  FMemo.SelStart := NewSelStart;
  Dec(FIndex);

  FInEdit := True;
end;

procedure THistory.Redo;
var
  NewSelStart: SizeInt;
begin
  if FIndex >= FSteps.Count - 1 then Exit;
  FInEdit := False;

  Inc(FIndex);
  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if DelStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(DelStr));
      NewSelStart := SelStart;
    end;
    if InsStr <> '' then begin
      Insert(InsStr, FPrevContent, BytePos);
      NewSelStart := SelStart;
    end;
  end;
  FMemo.Text := FPrevContent;
  FMemo.SelStart := NewSelStart;

  FInEdit := True;
end;

function UTF8PosToBytePos(const AStr: string; const ASize: SizeInt; APos: SizeInt): SizeInt;
begin
  if APos < 1 then Result := -1 else Result := 0;

  while (APos > 1) and (Result < ASize) do begin
    case AStr[Result] of
      // #0..#127: Inc(Result);
      #192..#223: Inc(Result, 2);
      #224..#239: Inc(Result, 3);
      #240..#247: Inc(Result, 4);
      else Inc(Result);
    end;
    Dec(APos);
  end;

  Inc(Result)
end;

procedure THistory.Paste;
var
  BytePos: SizeInt;
  ClipBoardText: string;
begin
  FInEdit := False;

  ClipBoardText := ClipBoard.AsText;
  if ClipBoardText <> '' then begin
    // FPrevContent == FMemo.Text
    BytePos := UTF8PosToBytePos(FPrevContent, Length(FPrevContent), FMemo.SelStart + 1);
    AddStep(BytePos, FMemo.SelStart, ClipBoardText, FMemo.SelText);
    FMemo.SelText := ClipBoardText;
  end;

  FInEdit := True;
end;

end.

