#!/bin/bash
############################################################
# Copyright (c) 2021-2023 Alexey Kuryakin kouriakine@mail.ru
# Script dim-cpl.sh uses to install/start/stop/control DIM
############################################################
# Usage:
#   /opt/dim/dim-cpl.sh [options]
# options:
#   print,install,remove,enable,disable,start,stop,
#   restart,status,dim_ports_enable,dim_ports_view,
#   dns-exec,dns-stop,did-exec,did-stop
#   webdid-exec,webdid-stop
############################################################


readonly DNSEXE="dns";                 # DIM DNS executable basename
readonly DIMDIR="/opt/dim";            # DIM DNS directory location
readonly DIMDNS="$DIMDIR/$DNSEXE";     # DIM DNS executable file
readonly SYSDIR="/etc/systemd/system"; # systemd config directory
readonly UNITID="dimdns";              # Service identifier
readonly PIDFILE="/run/$UNITID.pid";   # PID file 

function is_root_user(){
 [ "$USER" == "root" ] || { echo "Error: root required. Use su/sudo to run."; return 1; };
 return 0;
}

function print_usage() {
 echo $*;
}

function dim_ports_view(){
 iptables -L INPUT;
 iptables -L OUTPUT;
 netstat -ntulp | grep -e '/dns *$';
}

function ip_port_enable() {
 local PROT="$1"; # "tcp","udp"
 local PORT="$2"; # port number
 iptables -A INPUT  -p $PROT --dport $PORT -j ACCEPT;
 iptables -A OUTPUT -p $PROT --sport $PORT -j ACCEPT;
}

function dim_ports_enable() {
 for port in 2500 2505 $(seq 5100 1 5111); do ip_port_enable tcp $port; done; return;
 ip_port_enable  tcp  2500; # WebDID
 ip_port_enable  tcp  2505; # dns
 ip_port_enable  tcp  5100; # dim data port 1
 ip_port_enable  tcp  5101; # dim data port 2
 ip_port_enable  tcp  5102; # dim data port 3
}

function print_unit() {
 echo "[Unit]";
 echo "Description=DIM DNS service";
 echo "After=network.target";
 echo "After=multi-user.target";
 #echo "After=graphical.target";
 echo "";
 echo "[Service]";
 echo "Type=simple";
 echo "ExecStart=";
 echo "ExecStart=$DIMDNS";
 echo "WorkingDirectory=$DIMDIR";
 echo "PIDFile=$PIDFILE";
 echo "Restart=always";
 #echo "WatchdogSec=30";
 #echo "ProtectHome=true";
 #echo "ProtectSystem=full";
 echo "";
 echo "[Install]";
 echo "Alias=$UNITID.service";
 #echo "WantedBy=multi-user.target";
 echo "";
}

function menu(){
 local title="DIM DNS Service Control…";
 local text="The script <b>/opt/dim/dim-cpl.sh</b>";
 text="$text\nuses to control the [<span fgcolor='blue'><b>dimdns.service</b></span>].";
 text="$text\n\n<big><b>Please <span fgcolor='green'>choose the command</span>:</b></big>"
 sleep 1;
 while true; do
  item="$(zenity --title "$title" --text "$text"                        \
          --width 500 --height 700                                      \
          --list --print-column 1                                       \
          --column Command --column "Comment"                           \
          _______________  "dimdns.service - Control: _______________"  \
          print            "dimdns.service - Print config as text   |"  \
          status           "dimdns.service - Status of service      |"  \
          install          "dimdns.service - Install service        |"  \
          remove           "dimdns.service - Remove  service        |"  \
          enable           "dimdns.service - Enable  service        |"  \
          disable          "dimdns.service - Disable service        |"  \
          start            "dimdns.service - Start   service        |"  \
          stop             "dimdns.service - Stop    service        |"  \
          restart          "dimdns.service - Restart service        |"  \
          ________________ "DIM Utilities: _________________________|"  \
          dns-exec         "DNS    Execute - DIM Name Server        |"  \
          dns-stop         "DNS    Stop    - DIM Name Server        |"  \
          did-exec         "DID    Execute - DIM Display program    |"  \
          did-stop         "DID    Stop    - DIM Display program    |"  \
          webdid-exec      "WebDID Execute - Web DIM Display server |"  \
          webdid-stop      "WebDID Stop    - Web DIM Display server |"  \
          dimtree          "DIMTree.exe    - DIM Tree Data Browser  |"  \
          dimstatgui       "DimStatGui     - View traffic/statistic |"  \
          dimmonitor       "DimMonitor     - Generic Data Monitor   |"  \
          ________________ "DIM Ports      - View/Enable TCP/IP ____|"  \
          dim_ports_enable "DIM ports      - Enable with iptables   |"  \
          dim_ports_view   "DIM ports      - View policy rules      |"  \
          ________________ "________________________________________|"  \
          exit             "Exit this menu (same as Cancel)         |"  \
          ¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯ "¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯¯"  \
          2>/dev/null
         )";
  [ -z "$item" ] && return 0;
  [ "$item" == "exit" ] && return 0;
  case $item in
   exit)
    return 0;
    ;;
   dns-exec|dns-stop|did-exec|did-stop|webdid-exec|webdid-stop|dimtree|dimstatgui|dimmonitor)
    /opt/dim/dim-cpl.sh $item;
    break;
    ;;
   print|install|remove|enable|disable|start|stop|restart|status|dim_ports_enable|dim_ports_view)
    sudo /opt/dim/dim-cpl.sh $item;
    ;;
   _*|¯*)
    continue;
    ;;
   *)
    break;
    ;;
  esac;
 done;
};

function dim_cpl() {
 [ -d "$DIMDIR" ] || { echo "Error: not found $DIMDIR"; return 1; };
 [ -e "$DIMDNS" ] || { echo "Error: not found $DIMDNS"; return 1; };
 [ -d "$SYSDIR" ] || { echo "Error: not found $SYSDIR"; return 1; };
 case "$1" in
  menu)
   menu;
   return 0;
   ;;
  print)
   print_unit;
   return 0;
   ;;
  install)
   is_root_user || return 1;
   print_unit > "$SYSDIR/$UNITID.service";
   systemctl status $UNITID;
   ;;
  remove)
   is_root_user || return 1;
   systemctl disable $UNITID;
   [ -e "$SYSDIR/$UNITID.service" ] && rm -f > "$SYSDIR/$UNITID.service";
   systemctl status $UNITID;
   ;;
  enable)
   is_root_user || return 1;
   systemctl enable $UNITID;
   systemctl status $UNITID;
   ;;
  disable)
   is_root_user || return 1;
   systemctl disable $UNITID;
   systemctl status $UNITID;
   ;;
  start)
   is_root_user || return 1;
   systemctl start $UNITID;
   systemctl status $UNITID;
   ;;
  stop)
   is_root_user || return 1;
   systemctl stop $UNITID;
   systemctl status $UNITID;
   ;;
  restart)
   is_root_user || return 1;
   systemctl restart $UNITID;
   systemctl status $UNITID;
   ;;
  status)
   systemctl status $UNITID;
   ;;
  dim_ports_enable)
   is_root_user || return 1;
   dim_ports_enable;
   ;;
  dim_ports_view)
   is_root_user || return 1;
   dim_ports_view;
   ;;
  dns-exec)
   fly-open -e /opt/dim/dns-exec.desktop;
   ;;
  dns-stop)
   fly-open -e /opt/dim/dns-stop.desktop;
   ;;
  did-exec)
   fly-open -e /opt/dim/did-exec.desktop;
   ;;
  did-stop)
   fly-open -e /opt/dim/did-stop.desktop;
   ;;
  webdid-exec)
   fly-open -e /opt/dim/webdid-exec.desktop;
   ;;
  webdid-stop)
   fly-open -e /opt/dim/webdid-stop.desktop;
   ;;
  dimtree)
  fly-open -e /opt/dim/dimtree.desktop;
   ;;
  dimstatgui)
  fly-open -e /opt/dim/dimstatgui.desktop;
   ;;
  dimmonitor)
  fly-open -e /opt/dim/dimmonitor.desktop;
   ;;
  *)
   print_usage "${0##*/} {print|install|remove|enable|disable|start|stop|restart|status|dim_ports_enable|dim_ports_view|dns-exec|dns-stop|did-exec|did-stop|webdid-exec|webdid-stop|dimtree|menu}"
 esac;
 return 0;
}

dim_cpl $* || exit 1;
exit 0;
