////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// OPC definitions.                                                           //
// All definitions derived from http://www.opcconnect.com.                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20051009 - Creation (A.K.)                                                 //
// 20230826 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_opc; // OPC definitions

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 {$IFDEF WINDOWS} activex, comobj, {$ENDIF}
 sysutils, classes,
 _crw_alloc;

{$IFDEF WINDOWS}

type // From D5 ActiveX
 PGUIDList = ^TGUIDList;
 TGUIDList = array[0..65535] of TGUID;


 {*******************************************************}
 {                                                       }
 {       OPCtypes.pas                                    }
 {                                                       }
 {       Standard type definitions shared across         }
 {       multiple OPC specifications                     }
 {                                                       }
 {*******************************************************}

type

  TOleEnum          = ActiveX.TOleEnum;

  OPCHANDLE         = DWORD;
  POPCHANDLE        = ^OPCHANDLE;
  OPCHANDLEARRAY    = array[0..65535] of OPCHANDLE;
  POPCHANDLEARRAY   = ^OPCHANDLEARRAY;

  PVarType          = ^TVarType;
  TVarTypeList      = array[0..65535] of TVarType;
  PVarTypeList      = ^TVarTypeList;

  POleVariant       = System.POleVariant;
  OleVariantArray   = array[0..65535] of OleVariant;
  POleVariantArray  = ^OleVariantArray;

  PLCID             = ^TLCID;

  BOOLARRAY         = array[0..65535] of BOOL;
  PBOOLARRAY        = ^BOOLARRAY;

  DWORDARRAY        = array[0..65535] of DWORD;
  PDWORDARRAY       = ^DWORDARRAY;

  SingleArray       = array[0..65535] of Single;
  PSingleArray      = ^SingleArray;

  TFileTimeArray    = array[0..65535] of TFileTime;
  PFileTimeArray    = ^TFileTimeArray;


 {*******************************************************}
 {                                                       }
 {       OPC status and error codes                      }
 {                                                       }
 {       Delphi conversion generated and supplied by     }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {*******************************************************}
 
 {
 Module Name:
     OpcError.h
 Author:
     OPC Task Force
 Revision History:
 Release 1.0A
     Removed Unused messages
     Added OPC_S_INUSE, OPC_E_INVALIDCONFIGFILE, OPC_E_NOTFOUND
 Release 2.0
     Added OPC_E_INVALID_PID
 Release 3.0
     Added new error codes for DA3.0
     Added error codes for complex data
 
 Module Name:
     opcae_er.h
 Author:
     Jim Luth - OPC Alarm & Events Committee
 Revision History:
 
 Module Name:
     OpcHDAError.h
 Author:
     Ayana Craven, OSI Software, Inc.
 Revision History:
 
 Module Name:
     OpcErrSec.h
 Author:
     OPC Security Task Force
 Revision History:
 Release 1.0 08/18/00
     OPC security HRESULTs
 }
 
 {
 Code Assignements:
   0000 to 0200 are reserved for Microsoft use
   (although some were inadverdantly used for OPC Data Access 1.0 errors).
   0200 to 7FFF are reserved for future OPC use.
   8000 to FFFF can be vendor specific.
 }

const

  //
  //  Values are 32 bit values laid out as follows:
  //
  //   3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
  //   1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
  //  +---+-+-+-----------------------+-------------------------------+
  //  |Sev|C|R|     Facility          |               Code            |
  //  +---+-+-+-----------------------+-------------------------------+
  //
  //  where
  //
  //      Sev - is the severity code
  //
  //          00 - Success
  //          01 - Informational
  //          10 - Warning
  //          11 - Error
  //
  //      C - is the Customer code flag
  //
  //      R - is a reserved bit
  //
  //      Facility - is the facility code
  //
  //      Code - is the facility's status code
  //

  // OPC Data Access

  //
  // MessageId: OPC_E_INVALIDHANDLE
  //
  // MessageText:
  //
  //  The value of the handle is invalid.
  //
  OPC_E_INVALIDHANDLE = HResult($C0040001);

  //
  // MessageId: OPC_E_BADTYPE
  //
  // MessageText:
  //
  //  The server cannot convert the data between the
  //  requested data type and the canonical data type.
  //
  OPC_E_BADTYPE = HResult($C0040004);

  //
  // MessageId: OPC_E_PUBLIC
  //
  // MessageText:
  //
  //  The requested operation cannot be done on a public group.
  //
  OPC_E_PUBLIC = HResult($C0040005);

  //
  // MessageId: OPC_E_BADRIGHTS
  //
  // MessageText:
  //
  //  The Items AccessRights do not allow the operation.
  //
  OPC_E_BADRIGHTS = HResult($C0040006);

  //
  // MessageId: OPC_E_UNKNOWNITEMID
  //
  // MessageText:
  //
  //  The item is no longer available in the server address space.
  //
  OPC_E_UNKNOWNITEMID = HResult($C0040007);

  //
  // MessageId: OPC_E_INVALIDITEMID
  //
  // MessageText:
  //
  //  The item definition doesn't conform to the server's syntax.
  //
  OPC_E_INVALIDITEMID = HResult($C0040008);

  //
  // MessageId: OPC_E_INVALIDFILTER
  //
  // MessageText:
  //
  //  The filter string was not valid.
  //
  OPC_E_INVALIDFILTER = HResult($C0040009);

  //
  // MessageId: OPC_E_UNKNOWNPATH
  //
  // MessageText:
  //
  //  The item's access path is not known to the server.
  //
  OPC_E_UNKNOWNPATH = HResult($C004000A);

  //
  // MessageId: OPC_E_RANGE
  //
  // MessageText:
  //
  //  The value was out of range.
  //
  OPC_E_RANGE = HResult($C004000B);

  //
  // MessageId: OPC_E_DUPLICATENAME
  //
  // MessageText:
  //
  //  Duplicate name not allowed.
  //
  OPC_E_DUPLICATENAME = HResult($C004000C);

  //
  // MessageId: OPC_S_UNSUPPORTEDRATE
  //
  // MessageText:
  //
  //  The server does not support the requested data rate
  //  but will use the closest available rate.
  //
  OPC_S_UNSUPPORTEDRATE = HResult($0004000D);

  //
  // MessageId: OPC_S_CLAMP
  //
  // MessageText:
  //
  //  A value passed to WRITE was accepted but the output was clamped.
  //
  OPC_S_CLAMP = HResult($0004000E);

  //
  // MessageId: OPC_S_INUSE
  //
  // MessageText:
  //
  //  The operation cannot be completed because the
  //  object still has references that exist.
  //
  OPC_S_INUSE = HResult($0004000F);

  //
  // MessageId: OPC_E_INVALIDCONFIGFILE
  //
  // MessageText:
  //
  //  The server's configuration file is an invalid format.
  //
  OPC_E_INVALIDCONFIGFILE = HResult($C0040010);

  //
  // MessageId: OPC_E_NOTFOUND
  //
  // MessageText:
  //
  //  The server could not locate the requested object.
  //
  OPC_E_NOTFOUND = HResult($C0040011);

  //
  // MessageId: OPC_E_INVALID_PID
  //
  // MessageText:
  //
  //  The server does not recognise the passed property ID.
  //
  OPC_E_INVALID_PID = HResult($C0040203);

  //
  // MessageId: OPC_E_DEADBANDNOTSET
  //
  // MessageText:
  //
  //  The item deadband has not been set for this item.
  //
  OPC_E_DEADBANDNOTSET = HResult($C0040400);

  //
  // MessageId: OPC_E_DEADBANDNOTSUPPORTED
  //
  // MessageText:
  //
  //  The item does not support deadband.
  //
  OPC_E_DEADBANDNOTSUPPORTED = HResult($C0040401);

  //
  // MessageId: OPC_E_NOBUFFERING
  //
  // MessageText:
  //
  //  The server does not support buffering of data items that are collected at
  //  a faster rate than the group update rate.
  //
  OPC_E_NOBUFFERING = HResult($C0040402);

  //
  // MessageId: OPC_E_INVALIDCONTINUATIONPOINT
  //
  // MessageText:
  //
  //  The continuation point is not valid.
  //
  OPC_E_INVALIDCONTINUATIONPOINT = HResult($C0040403);

  //
  // MessageId: OPC_S_DATAQUEUEOVERFLOW
  //
  // MessageText:
  //
  //  Data Queue Overflow - Some value transitions were lost.
  //
  OPC_S_DATAQUEUEOVERFLOW = HResult($00040404);

  //
  // MessageId: OPC_E_RATENOTSET
  //
  // MessageText:
  //
  //  Server does not support requested rate.
  //
  OPC_E_RATENOTSET = HResult($C0040405);

  //
  // MessageId: OPC_E_NOTSUPPORTED
  //
  // MessageText:
  //
  //  The server does not support writing of quality and/or timestamp.
  //
  OPC_E_NOTSUPPORTED = HResult($C0040406);

  //
  // MessageId: OPCCPX_E_TYPE_CHANGED
  //
  // MessageText:
  //
  //  The dictionary and/or type description for the item has changed.
  //
  OPCCPX_E_TYPE_CHANGED = HResult($C0040407);

  //
  // MessageId: OPCCPX_E_FILTER_DUPLICATE
  //
  // MessageText:
  //
  //  A data filter item with the specified name already exists.
  //
  OPCCPX_E_FILTER_DUPLICATE = HResult($C0040408);

  //
  // MessageId: OPCCPX_E_FILTER_INVALID
  //
  // MessageText:
  //
  //  The data filter value does not conform to the server's syntax.
  //
  OPCCPX_E_FILTER_INVALID = HResult($C0040409);

  //
  // MessageId: OPCCPX_E_FILTER_ERROR
  //
  // MessageText:
  //
  //  An error occurred when the filter value was applied to the source data.
  //
  OPCCPX_E_FILTER_ERROR = HResult($C004040A);

  //
  // MessageId: OPCCPX_S_FILTER_NO_DATA
  //
  // MessageText:
  //
  //  The item value is empty because the data filter has excluded all fields.
  //
  OPCCPX_S_FILTER_NO_DATA = HResult($0004040B);

  // OPC Alarms & Events

  //
  // MessageId: OPC_S_ALREADYACKED
  //
  // MessageText:
  //
  //  The condition has already been acknowleged
  //
  OPC_S_ALREADYACKED = HResult($00040200);

  //
  // MessageId: OPC_S_INVALIDBUFFERTIME
  //
  // MessageText:
  //
  //  The buffer time parameter was invalid
  //
  OPC_S_INVALIDBUFFERTIME = HResult($00040201);

  //
  // MessageId: OPC_S_INVALIDMAXSIZE
  //
  // MessageText:
  //
  //  The max size parameter was invalid
  //
  OPC_S_INVALIDMAXSIZE = HResult($00040202);

  //
  // MessageId: OPC_S_INVALIDKEEPALIVETIME
  //
  // MessageText:
  //
  //  The KeepAliveTime parameter was invalid
  //
  OPC_S_INVALIDKEEPALIVETIME = HResult($00040203);

  //
  // MessageId: OPC_E_INVALIDBRANCHNAME
  //
  // MessageText:
  //
  //  The string was not recognized as an area name
  //
  OPC_E_INVALIDBRANCHNAME = HResult($C0040203);

  //
  // MessageId: OPC_E_INVALIDTIME
  //
  // MessageText:
  //
  //  The time does not match the latest active time
  //
  OPC_E_INVALIDTIME = HResult($C0040204);

  //
  // MessageId: OPC_E_BUSY
  //
  // MessageText:
  //
  //  A refresh is currently in progress
  //
  OPC_E_BUSY = HResult($C0040205);

  //
  // MessageId: OPC_E_NOINFO
  //
  // MessageText:
  //
  //  Information is not available
  //
  OPC_E_NOINFO = HResult($C0040206);

  // OPC Historical Data Access

  //
  // MessageId: OPC_E_MAXEXCEEDED
  //
  // MessageText:
  //
  //  The maximum number of values requested exceeds the server's limit.
  //
  OPC_E_MAXEXCEEDED = HResult($C0041001);

  //
  // MessageId: OPC_S_NODATA
  //
  // MessageText:
  //
  //  There is no data within the specified parameters
  //
  OPC_S_NODATA = HResult($40041002);

  //
  // MessageId: OPC_S_MOREDATA
  //
  // MessageText:
  //
  // There is more data satisfying the query than was returned
  //
  OPC_S_MOREDATA = HResult($40041003);

  //
  // MessageId: OPC_E_INVALIDAGGREGATE
  //
  // MessageText:
  //
  //  The aggregate requested is not valid.
  //
  OPC_E_INVALIDAGGREGATE = HResult($C0041004);

  //
  // MessageId: OPC_S_CURRENTVALUE
  //
  // MessageText:
  //
  //  The server only returns current values for the requested item attributes.
  //
  OPC_S_CURRENTVALUE = HResult($40041005);

  //
  // MessageId: OPC_S_EXTRADATA
  //
  // MessageText:
  //
  //  Additional data satisfying the query was found.
  //
  OPC_S_EXTRADATA = HResult($40041006);

  //
  // MessageId: OPC_W_NOFILTER
  //
  // MessageText:
  //
  //  The server does not support this filter.
  //
  OPC_W_NOFILTER = HResult($80041007);

  //
  // MessageId: OPC_E_UNKNOWNATTRID
  //
  // MessageText:
  //
  //  The server does not support this attribute.
  //
  OPC_E_UNKNOWNATTRID = HResult($C0041008);

  //
  // MessageId: OPC_E_NOT_AVAIL
  //
  // MessageText:
  //
  //  The requested aggregate is not available for the specified item.
  //
  OPC_E_NOT_AVAIL = HResult($C0041009);

  //
  // MessageId: OPC_E_INVALIDDATATYPE
  //
  // MessageText:
  //
  //  The supplied value for the attribute is not a correct data type.
  //
  OPC_E_INVALIDDATATYPE = HResult($C004100A);

  //
  // MessageId: OPC_E_DATAEXISTS
  //
  // MessageText:
  //
  //  Unable to insert - data already present.
  //
  OPC_E_DATAEXISTS = HResult($C004100B);

  //
  // MessageId: OPC_E_INVALIDATTRID
  //
  // MessageText:
  //
  //  The supplied attribute ID is not valid.
  //
  OPC_E_INVALIDATTRID = HResult($C004100C);

  //
  // MessageId: OPC_E_NODATAEXISTS
  //
  // MessageText:
  //
  //  The server has no value for the specified time and item ID.
  //
  OPC_E_NODATAEXISTS = HResult($C004100D);

  //
  // MessageId: OPC_S_INSERTED
  //
  // MessageText:
  //
  //  The requested insert occurred.
  //
  OPC_S_INSERTED = HResult($4004100E);

  //
  // MessageId: OPC_S_REPLACED
  //
  // MessageText:
  //
  //  The requested replace occurred.
  //
  OPC_S_REPLACED = HResult($4004100F);

  // OPC Security

  //
  // MessageId: OPC_E_PRIVATE_ACTIVE
  //
  // MessageText:
  //
  //  OPC Security: A session using private OPC credentials is already active.
  //
  OPC_E_PRIVATE_ACTIVE = HResult($C0040301);

  //
  // MessageId: OPC_E_LOW_IMPERS_LEVEL
  //
  // MessageText:
  //
  //  OPC Security: Server requires higher impersonation level to access secured
  //  data.
  //
  OPC_E_LOW_IMPERS_LEVEL = HResult($C0040302);

  //
  // MessageId: OPC_S_LOW_AUTHN_LEVEL
  //
  // MessageText:
  //
  //  OPC Security: Server expected higher level of package privacy.
  //
  OPC_S_LOW_AUTHN_LEVEL = HResult($00040303);


 {*******************************************************}
 {                                                       }
 {       OPC Security 1.0                                }
 {                                                       }
 {       Delphi conversion maintained and supplied by    }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {       Originally contributed by                       }
 {       Illya Kysil                                     }
 {       mailto:ikysil@ua.fm                             }
 {                                                       }
 {*******************************************************}
 
 // ************************************************************************ //
 // Type Lib: opcSec_PS.dll
 // IID\LCID: {7AA83AFF-6C77-11D3-84F9-00008630A38B}\0
 // ************************************************************************ //
 
 // *********************************************************************//
 // GUIDS declared in the TypeLibrary                                    //
 // *********************************************************************//
const
 LIBID_OPCSEC: TIID = '{7AA83AFF-6C77-11D3-84F9-00008630A38B}';
 IID_IOPCSecurityNT: TIID = '{7AA83A01-6C77-11D3-84F9-00008630A38B}';
 IID_IOPCSecurityPrivate: TIID = '{7AA83A02-6C77-11D3-84F9-00008630A38B}';

type

 // *********************************************************************//
 // Forward declaration of types defined in TypeLibrary                  //
 // *********************************************************************//
 IOPCSecurityNT = interface;
 IOPCSecurityPrivate = interface;

 // *********************************************************************//
 // Interface: IOPCSecurityNT
 // GUID:      {7AA83A01-6C77-11D3-84F9-00008630A38B}
 // *********************************************************************//
 IOPCSecurityNT = interface(IUnknown)
  ['{7AA83A01-6C77-11D3-84F9-00008630A38B}']
  function IsAvailableNT(
     out   pbAvailable:                BOOL): HResult; stdcall;
   function QueryMinImpersonationLevel(
     out   pdwMinImpLevel:             DWORD): HResult; stdcall;
   function ChangeUser: HResult; stdcall;
 end;
 
 // *********************************************************************//
 // Interface: IOPCSecurityPrivate
 // GUID:      {7AA83A02-6C77-11D3-84F9-00008630A38B}
 // *********************************************************************//
 IOPCSecurityPrivate = interface(IUnknown)
   ['{7AA83A02-6C77-11D3-84F9-00008630A38B}']
   function IsAvailablePriv(
     out   pbAvailable:                BOOL): HResult; stdcall;
   function Logon(
           szUserID:                   POleStr;
           szPassword:                 POleStr): HResult; stdcall;
   function Logoff: HResult; stdcall;
 end;


 {*******************************************************}
 {                                                       }
 {       OPC Common 1.1                                  }
 {                                                       }
 {       Delphi conversion generated and supplied by     }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {*******************************************************}
 
 // ************************************************************************ //
 // Type Lib: opccomn_ps.dll
 // IID\LCID: {B28EEDB1-AC6F-11D1-84D5-00608CB8A7E9}\0
 // ************************************************************************ //
 
 // *********************************************************************//
 // GUIDS declared in the TypeLibrary                                    //
 // *********************************************************************//
const
 LIBID_OPCCOMN: TGUID = '{B28EEDB1-AC6F-11D1-84D5-00608CB8A7E9}';
 IID_IOPCCommon: TIID = '{F31DFDE2-07B6-11D2-B2D8-0060083BA1FB}';
 IID_IOPCShutdown: TIID = '{F31DFDE1-07B6-11D2-B2D8-0060083BA1FB}';
 IID_IOPCServerList: TIID = '{13486D50-4821-11D2-A494-3CB306C10000}';
 IID_IOPCEnumGUID: TIID = '{55C382C8-21C7-4e88-96C1-BECFB1E3F483}';
 IID_IOPCServerList2: TIID = '{9DD0B56C-AD9E-43ee-8305-487F3188BF7A}';
 CLSID_OPCServerList: TGUID = '{13486D51-4821-11D2-A494-3CB306C10000}';

type

 // *********************************************************************//
 // Forward declaration of interfaces defined in Type Library            //
 // *********************************************************************//
 IOPCCommon = interface;
 IOPCShutdown = interface;
 IOPCServerList = interface;
 IOPCEnumGUID = interface;
 IOPCServerList2 = interface;

 // *********************************************************************//
 // Declaration of structures, unions and aliases.                       //
 // *********************************************************************//
 LCIDARRAY = array[0..65535] of LCID;
 PLCIDARRAY = ^LCIDARRAY;

 // *********************************************************************//
 // Interface: IOPCCommon
 // GUID:      {F31DFDE2-07B6-11D2-B2D8-0060083BA1FB}
 // *********************************************************************//
 IOPCCommon = interface(IUnknown)
   ['{F31DFDE2-07B6-11D2-B2D8-0060083BA1FB}']
   function SetLocaleID(
           dwLcid:                     TLCID): HResult; stdcall;
   function GetLocaleID(
     out   pdwLcid:                    TLCID): HResult; stdcall;
   function QueryAvailableLocaleIDs(
     out   pdwCount:                   UINT;
     out   pdwLcid:                    PLCIDARRAY): HResult; stdcall;
   function GetErrorString(
           dwError:                    HResult;
     out   ppString:                   POleStr): HResult; stdcall;
   function SetClientName(
           szName:                     POleStr): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCShutdown
 // GUID:      {F31DFDE1-07B6-11D2-B2D8-0060083BA1FB}
 // *********************************************************************//
 IOPCShutdown = interface(IUnknown)
   ['{F31DFDE1-07B6-11D2-B2D8-0060083BA1FB}']
   function ShutdownRequest(
           szReason:                   POleStr): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCServerList
 // GUID:      {13486D50-4821-11D2-A494-3CB306C10000}
 // *********************************************************************//
 IOPCServerList = interface(IUnknown)
   ['{13486D50-4821-11D2-A494-3CB306C10000}']
   function EnumClassesOfCategories(
           cImplemented:               ULONG;
           rgcatidImpl:                PGUIDList;
           cRequired:                  ULONG;
           rgcatidReq:                 PGUIDList;
     out   ppenumClsid:                IEnumGUID): HResult; stdcall;
   function GetClassDetails(
     const clsid:                      TCLSID;
     out   ppszProgID:                 POleStr;
     out   ppszUserType:               POleStr): HResult; stdcall;
   function CLSIDFromProgID(
           szProgId:                   POleStr;
     out   clsid:                      TCLSID): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEnumGUID
 // GUID:      {55C382C8-21C7-4e88-96C1-BECFB1E3F483}
 // *********************************************************************//
 IOPCEnumGUID = interface(IUnknown)
   ['{55C382C8-21C7-4e88-96C1-BECFB1E3F483}']
   function Next(
           celt:                       UINT;
     out   rgelt:                      TGUID;
     out   pceltFetched:               UINT): HResult; stdcall;
   function Skip(
           celt:                       UINT): HResult; stdcall;
   function Reset: HResult; stdcall;
   function Clone(
     out   ppenum:                     IOPCEnumGUID): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCServerList2
 // GUID:      {9DD0B56C-AD9E-43ee-8305-487F3188BF7A}
 // *********************************************************************//
 IOPCServerList2 = interface(IUnknown)
   ['{9DD0B56C-AD9E-43ee-8305-487F3188BF7A}']
   function EnumClassesOfCategories(
           cImplemented:               ULONG;
           rgcatidImpl:                PGUIDList;
           cRequired:                  ULONG;
           rgcatidReq:                 PGUIDList;
     out   ppenumClsid:                IOPCEnumGUID): HResult; stdcall;
   function GetClassDetails(
     const clsid:                      TCLSID;
     out   ppszProgID:                 POleStr;
     out   ppszUserType:               POleStr;
     out   ppszVerIndProgID:           POleStr): HResult; stdcall;
   function CLSIDFromProgID(
           szProgId:                   POleStr;
     out   clsid:                      TCLSID): HResult; stdcall;
 end;


 {*******************************************************}
 {                                                       }
 {       OPC Alarms and Events 1.10                      }
 {                                                       }
 {       Delphi conversion generated and supplied by     }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {*******************************************************}
 
 // ************************************************************************ //
 // Type Lib: opc_aeps.dll
 // IID\LCID: {65168844-5783-11D1-84A0-00608CB8A7E9}\0
 // ************************************************************************ //
 
 // *********************************************************************//
 // GUIDS declared in the TypeLibrary                                    //
 // *********************************************************************//
const
 LIBID_OPC_AE: TGUID = '{65168844-5783-11D1-84A0-00608CB8A7E9}';
 IID_IOPCEventServer: TIID = '{65168851-5783-11D1-84A0-00608CB8A7E9}';
 IID_IOPCEventSubscriptionMgt: TIID = '{65168855-5783-11D1-84A0-00608CB8A7E9}';
 IID_IOPCEventAreaBrowser: TIID = '{65168857-5783-11D1-84A0-00608CB8A7E9}';
 IID_IOPCEventSink: TIID = '{6516885F-5783-11D1-84A0-00608CB8A7E9}';
 IID_IOPCEventServer2: TIID = '{71BBE88E-9564-4BCD-BCFC-71C558D94F2D}';
 IID_IOPCEventSubscriptionMgt2: TIID = '{94C955DC-3684-4CCB-AFAB-F898CE19AAC3}';
 OPCEventServerCATID: TGUID = '{58E13251-AC87-11D1-84D5-00608CB8A7E9}';

 // *********************************************************************//
 // Declaration of Enumerations defined in Type Library                  //
 // *********************************************************************//
type
 OPCAEBROWSEDIRECTION = TOleEnum;
const
 OPCAE_BROWSE_UP   = 1;
 OPCAE_BROWSE_DOWN = 2;
 OPCAE_BROWSE_TO   = 3;

type
 OPCAEBROWSETYPE = TOleEnum;
const
 OPC_AREA   = 1;
 OPC_SOURCE = 2;

type
 OPCEVENTSERVERSTATE = TOleEnum;
const
 OPCAE_STATUS_RUNNING    = 1;
 OPCAE_STATUS_FAILED     = 2;
 OPCAE_STATUS_NOCONFIG   = 3;
 OPCAE_STATUS_SUSPENDED  = 4;
 OPCAE_STATUS_TEST       = 5;
 OPCAE_STATUS_COMM_FAULT = 6;

 // *********************************************************************//
 // OPC Quality flags                                                    //
 // *********************************************************************//
 // Masks for extracting quality subfields
 // (note 'status' mask also includes 'Quality' bits)
(*** defined below in OPC DA ***
 OPC_QUALITY_MASK           = $C0;
 OPC_STATUS_MASK            = $FC;
 OPC_LIMIT_MASK             = $03;

 // Values for QUALITY_MASK bit field
 OPC_QUALITY_BAD            = $00;
 OPC_QUALITY_UNCERTAIN      = $40;
 OPC_QUALITY_GOOD           = $C0;

 // STATUS_MASK Values for Quality = BAD
 OPC_QUALITY_CONFIG_ERROR   = $04;
 OPC_QUALITY_NOT_CONNECTED  = $08;
 OPC_QUALITY_DEVICE_FAILURE = $0C;
 OPC_QUALITY_SENSOR_FAILURE = $10;
 OPC_QUALITY_LAST_KNOWN     = $14;
 OPC_QUALITY_COMM_FAILURE   = $18;
 OPC_QUALITY_OUT_OF_SERVICE = $1C;

 // STATUS_MASK Values for Quality = UNCERTAIN
 OPC_QUALITY_LAST_USABLE    = $44;
 OPC_QUALITY_SENSOR_CAL     = $50;
 OPC_QUALITY_EGU_EXCEEDED   = $54;
 OPC_QUALITY_SUB_NORMAL     = $58;

 // STATUS_MASK Values for Quality = GOOD
  OPC_QUALITY_LOCAL_OVERRIDE = $D8;
 ***)
 // State bit masks
 OPC_CONDITION_ENABLED = $0001;
 OPC_CONDITION_ACTIVE  = $0002;
 OPC_CONDITION_ACKED   = $0004;

 // bit masks for m_wChangeMask
 OPC_CHANGE_ACTIVE_STATE = $0001;
 OPC_CHANGE_ACK_STATE    = $0002;
 OPC_CHANGE_ENABLE_STATE = $0004;
 OPC_CHANGE_QUALITY      = $0008;
 OPC_CHANGE_SEVERITY     = $0010;
 OPC_CHANGE_SUBCONDITION = $0020;
 OPC_CHANGE_MESSAGE      = $0040;
 OPC_CHANGE_ATTRIBUTE    = $0080;

 // dwEventType
 OPC_SIMPLE_EVENT    = $0001;
 OPC_TRACKING_EVENT  = $0002;
 OPC_CONDITION_EVENT = $0004;
 OPC_ALL_EVENTS      = OPC_SIMPLE_EVENT or OPC_TRACKING_EVENT or OPC_CONDITION_EVENT;

 // QueryAvailableFilters() bit masks
 OPC_FILTER_BY_EVENT    = $0001;
 OPC_FILTER_BY_CATEGORY = $0002;
 OPC_FILTER_BY_SEVERITY = $0004;
 OPC_FILTER_BY_AREA     = $0008;
 OPC_FILTER_BY_SOURCE   = $0010;

type

 // *********************************************************************//
 // Forward declaration of interfaces defined in Type Library            //
 // *********************************************************************//
 IOPCEventServer = interface;
 IOPCEventSubscriptionMgt = interface;
 IOPCEventAreaBrowser = interface;
 IOPCEventSink = interface;
 IOPCEventServer2 = interface;
 IOPCEventSubscriptionMgt2 = interface;

 // *********************************************************************//
 // Declaration of structures, unions and aliases.                       //
 // *********************************************************************//
 ONEVENTSTRUCT = record
   wChangeMask:          Word;
   wNewState:            Word;
   szSource:             POleStr;
   ftTime:               TFileTime;
   szMessage:            POleStr;
   dwEventType:          DWORD;
   dwEventCategory:      DWORD;
   dwSeverity:           DWORD;
   szConditionName:      POleStr;
   szSubconditionName:   POleStr;
   wQuality:             Word;
   wReserved:            Word;     // added to ensure natural alignment
   bAckRequired:         BOOL;
   ftActiveTime:         TFileTime;
   dwCookie:             DWORD;
   dwNumEventAttrs:      DWORD;
   pEventAttributes:     POleVariantArray;
   szActorID:            POleStr;
 end;
 PONEVENTSTRUCT = ^ONEVENTSTRUCT;
 ONEVENTSTRUCTARRAY = array[0..65535] of ONEVENTSTRUCT;
 PONEVENTSTRUCTARRAY = ^ONEVENTSTRUCTARRAY;

 OPCEVENTSERVERSTATUS = record
   ftStartTime:          TFileTime;
   ftCurrentTime:        TFileTime;
   ftLastUpdateTime:     TFileTime;
   dwServerState:        OPCEVENTSERVERSTATE;
   wMajorVersion:        Word;
   wMinorVersion:        Word;
   wBuildNumber:         Word;
   wReserved:            Word;     // added to ensure natural alignment
   szVendorInfo:         POleStr;
 end;
 POPCEVENTSERVERSTATUS = ^OPCEVENTSERVERSTATUS;

 OPCCONDITIONSTATE = record
   wState:               Word;
   wReserved1:           Word;     // added to ensure natural alignment
   szActiveSubCondition: POleStr;
   szASCDefinition:      POleStr;
   dwASCSeverity:        DWORD;
   szASCDescription:     POleStr;
   wQuality:             Word;
   wReserved2:           Word;     // added to ensure natural alignment
   ftLastAckTime:        TFileTime;
   ftSubCondLastActive:  TFileTime;
   ftCondLastActive:     TFileTime;
   ftCondLastInactive:   TFileTime;
   szAcknowledgerID:     POleStr;
   szComment:            POleStr;
   dwNumSCs:             DWORD;
   pszSCNames:           POleStrList;
   pszSCDefinitions:     POleStrList;
   pdwSCSeverities:      PDWORDARRAY;
   pszSCDescriptions:    POleStrList;
   dwNumEventAttrs:      DWORD;
   pEventAttributes:     POleVariantArray;
   pErrors:              PResultList;
 end;
 POPCCONDITIONSTATE = ^OPCCONDITIONSTATE;

 // *********************************************************************//
 // Interface: IOPCEventServer
 // GUID:      {65168851-5783-11D1-84A0-00608CB8A7E9}
 // *********************************************************************//
 IOPCEventServer = interface(IUnknown)
   ['{65168851-5783-11D1-84A0-00608CB8A7E9}']
   function GetStatus(
     out   ppEventServerStatus:        POPCEVENTSERVERSTATUS):
           HResult; stdcall;
   function CreateEventSubscription(
           bActive:                    BOOL;
           dwBufferTime:               DWORD;
           dwMaxSize:                  DWORD;
           hClientSubscription:        OPCHANDLE;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown;
     out   pdwRevisedBufferTime:       DWORD;
     out   pdwRevisedMaxSize:          DWORD): HResult; stdcall;
   function QueryAvailableFilters(
     out   pdwFilterMask:              DWORD): HResult; stdcall;
   function QueryEventCategories(
           dwEventType:                DWORD;
     out   pdwCount:                   DWORD;
     out   ppdwEventCategories:        PDWORDARRAY;
     out   ppszEventCategoryDescs:     POleStrList): HResult; stdcall;
   function QueryConditionNames(
           dwEventCategory:            DWORD;
     out   pdwCount:                   DWORD;
     out   ppszConditionNames:         POleStrList): HResult; stdcall;
   function QuerySubConditionNames(
           szConditionName:            POleStr;
     out   pdwCount:                   DWORD;
     out   ppszSubConditionNames:      POleStrList): HResult; stdcall;
   function QuerySourceConditions(
           szSource:                   POleStr;
     out   pdwCount:                   DWORD;
     out   ppszConditionNames:         POleStrList): HResult; stdcall;
   function QueryEventAttributes(
           dwEventCategory:            DWORD;
     out   pdwCount:                   DWORD;
     out   ppdwAttrIDs:                PDWORDARRAY;
     out   ppszAttrDescs:              POleStrList;
     out   ppvtAttrTypes:              PVarTypeList): HResult; stdcall;
   function TranslateToItemIDs(
           szSource:                   POleStr;
           dwEventCategory:            DWORD;
           szConditionName:            POleStr;
           szSubconditionName:         POleStr;
           dwCount:                    DWORD;
           pdwAssocAttrIDs:            PDWORDARRAY;
     out   ppszAttrItemIDs:            POleStrList;
     out   ppszNodeNames:              POleStrList;
     out   ppCLSIDs:                   PGUIDList): HResult; stdcall;
   function GetConditionState(
           szSource:                   POleStr;
           szConditionName:            POleStr;
           dwNumEventAttrs:            DWORD;
           pdwAttributeIDs:            PDWORDARRAY;
     out   ppConditionState:           POPCCONDITIONSTATE): HResult; stdcall;
   function EnableConditionByArea(
           dwNumAreas:                 DWORD;
           pszAreas:                   POleStrList): HResult; stdcall;
   function EnableConditionBySource(
           dwNumSources:               DWORD;
           pszSources:                 POleStrList): HResult; stdcall;
   function DisableConditionByArea(
           dwNumAreas:                 DWORD;
           pszAreas:                   POleStrList): HResult; stdcall;
   function DisableConditionBySource(
           dwNumSources:               DWORD;
           pszSources:                 POleStrList): HResult; stdcall;
   function AckCondition(
           dwCount:                    DWORD;
           szAcknowledgerID:           POleStr;
           szComment:                  POleStr;
           pszSource:                  POleStrList;
           pszConditionName:           POleStrList;
           pftActiveTime:              PFileTimeArray;
           pdwCookie:                  PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function CreateAreaBrowser(
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEventSubscriptionMgt
 // GUID:      {65168855-5783-11D1-84A0-00608CB8A7E9}
 // *********************************************************************//
 IOPCEventSubscriptionMgt = interface(IUnknown)
   ['{65168855-5783-11D1-84A0-00608CB8A7E9}']
   function SetFilter(
           dwEventType:                DWORD;
           dwNumCategories:            DWORD;
           pdwEventCategories:         PDWORDARRAY;
           dwLowSeverity:              DWORD;
           dwHighSeverity:             DWORD;
           dwNumAreas:                 DWORD;
           pszAreaList:                POleStrList;
           dwNumSources:               DWORD;
           pszSourceList:              POleStrList): HResult; stdcall;
   function GetFilter(
     out   pdwEventType:               DWORD;
     out   pdwNumCategories:           DWORD;
     out   ppdwEventCategories:        PDWORDARRAY;
     out   pdwLowSeverity:             DWORD;
     out   pdwHighSeverity:            DWORD;
     out   pdwNumAreas:                DWORD;
     out   ppszAreaList:               POleStrList;
     out   pdwNumSources:              DWORD;
     out   ppszSourceList:             POleStrList): HResult; stdcall;
   function SelectReturnedAttributes(
           dwEventCategory:            DWORD;
           dwCount:                    DWORD;
           dwAttributeIDs:             PDWORDARRAY): HResult; stdcall;
   function GetReturnedAttributes(
           dwEventCategory:            DWORD;
     out   pdwCount:                   DWORD;
     out   ppdwAttributeIDs:           PDWORDARRAY): HResult; stdcall;
   function Refresh(
           dwConnection:               DWORD): HResult; stdcall;
   function CancelRefresh(
           dwConnection:               DWORD): HResult; stdcall;
   function GetState(
     out   pbActive:                   BOOL;
     out   pdwBufferTime:              DWORD;
     out   pdwMaxSize:                 DWORD;
     out   phClientSubscription:       OPCHANDLE): HResult; stdcall;
   function SetState(
           pbActive:                   PBOOL;
           pdwBufferTime:              PDWORD;
           pdwMaxSize:                 PDWORD;
           hClientSubscription:        OPCHANDLE;
     out   pdwRevisedBufferTime:       DWORD;
     out   pdwRevisedMaxSize:          DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEventAreaBrowser
 // GUID:      {65168857-5783-11D1-84A0-00608CB8A7E9}
 // *********************************************************************//
 IOPCEventAreaBrowser = interface(IUnknown)
   ['{65168857-5783-11D1-84A0-00608CB8A7E9}']
   function ChangeBrowsePosition(
           dwBrowseDirection:          OPCAEBROWSEDIRECTION;
           szString:                   POleStr): HResult; stdcall;
   function BrowseOPCAreas(
           dwBrowseFilterType:         OPCAEBROWSETYPE;
           szFilterCriteria:           POleStr;
     out   ppIEnumString:              IEnumString): HResult; stdcall;
   function GetQualifiedAreaName(
           szAreaName:                 POleStr;
     out   pszQualifiedAreaName:       POleStr): HResult; stdcall;
   function GetQualifiedSourceName(
           szSourceName:               POleStr;
     out   pszQualifiedSourceName:     POleStr): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEventSink
 // GUID:      {6516885F-5783-11D1-84A0-00608CB8A7E9}
 // *********************************************************************//
 IOPCEventSink = interface(IUnknown)
   ['{6516885F-5783-11D1-84A0-00608CB8A7E9}']
   function OnEvent(
           hClientSubscription:        OPCHANDLE;
           bRefresh:                   BOOL;
           bLastRefresh:               BOOL;
           dwCount:                    DWORD;
           pEvents:                    PONEVENTSTRUCTARRAY): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEventServer2
 // GUID:      {71BBE88E-9564-4BCD-BCFC-71C558D94F2D}
 // *********************************************************************//
 IOPCEventServer2 = interface(IOPCEventServer)
   ['{71BBE88E-9564-4BCD-BCFC-71C558D94F2D}']
   function EnableConditionByArea2(
           dwNumAreas:                 DWORD;
           pszAreas:                   POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function EnableConditionBySource2(
           dwNumSources:               DWORD;
           pszSources:                 POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DisableConditionByArea2(
           dwNumAreas:                 DWORD;
           pszAreas:                   POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DisableConditionBySource2(
           dwNumSources:               DWORD;
           pszSources:                 POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function GetEnableStateByArea(
           dwNumAreas:                 DWORD;
           pszAreas:                   POleStrList;
           pbEnabled:                  PBOOLARRAY;
           pbEffectivelyEnabled:       PBOOLARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function GetEnableStateBySource(
           dwNumSources:               DWORD;
           pszSources:                 POleStrList;
           pbEnabled:                  PBOOLARRAY;
           pbEffectivelyEnabled:       PBOOLARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCEventSubscriptionMgt2
 // GUID:      {94C955DC-3684-4CCB-AFAB-F898CE19AAC3}
 // *********************************************************************//
 IOPCEventSubscriptionMgt2 = interface(IOPCEventSubscriptionMgt)
   ['{94C955DC-3684-4CCB-AFAB-F898CE19AAC3}']
   function SetKeepAlive(
           dwKeepAliveTime:            DWORD;
     out   pdwRevisedKeepAliveTime:    DWORD): HResult; stdcall;
   function GetKeepAlive(
     out   pdwKeepAliveTime:           DWORD): HResult; stdcall;
 end;


 {*******************************************************}
 {                                                       }
 {       OPC Historical Data Access 1.2                  }
 {                                                       }
 {       Delphi conversion generated and supplied by     }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {       Based on OPC HDA 1.0 conversion by              }
 {       Illya Kysil                                     }
 {       mailto:ikysil@ua.fm                             }
 {                                                       }
 {*******************************************************}
 
 // ************************************************************************ //
 // Type Lib: opchda_ps.dll
 // IID\LCID: {1F1217BA-DEE0-11D2-A5E5-000086339399}\0
 // ************************************************************************ //
 
 // *********************************************************************//
 // GUIDS declared in the TypeLibrary                                    //
 // *********************************************************************//
const
 LIBID_OPCHDA: TGUID = '{1F1217BA-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_Server: TGUID = '{1F1217B0-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_Browser: TGUID = '{1F1217B1-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_SyncRead: TGUID = '{1F1217B2-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_SyncUpdate: TGUID = '{1F1217B3-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_SyncAnnotations: TGUID = '{1F1217B4-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_AsyncRead: TGUID = '{1F1217B5-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_AsyncUpdate: TGUID = '{1F1217B6-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_AsyncAnnotations: TGUID = '{1F1217B7-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_Playback: TGUID = '{1F1217B8-DEE0-11D2-A5E5-000086339399}';
 IID_IOPCHDA_DataCallback: TGUID = '{1F1217B9-DEE0-11D2-A5E5-000086339399}';
 CATID_OPCHDAServer10: TGUID = '{7DE5B060-E089-11d2-A5E6-000086339399}';

 // *********************************************************************//
 // Declaration of Enumerations defined in Type Library                  //
 // *********************************************************************//
type
 OPCHDA_SERVERSTATUS = TOleEnum;
const
 OPCHDA_UP            = 1;
 OPCHDA_DOWN          = 2;
 OPCHDA_INDETERMINATE = 3;

type
 OPCHDA_BROWSEDIRECTION = TOleEnum;
const
 OPCHDA_BROWSE_UP     = 1;
 OPCHDA_BROWSE_DOWN   = 2;
 OPCHDA_BROWSE_DIRECT = 3;

type
 OPCHDA_BROWSETYPE = TOleEnum;
const
 OPCHDA_BRANCH = 1;
 OPCHDA_LEAF   = 2;
 OPCHDA_FLAT   = 3;
 OPCHDA_ITEMS  = 4;

type
 OPCHDA_ANNOTATIONCAPABILITIES = TOleEnum;
const
 OPCHDA_READANNOTATIONCAP   = 1;
 OPCHDA_INSERTANNOTATIONCAP = 2;

type
 OPCHDA_UPDATECAPABILITIES = TOleEnum;
const
 OPCHDA_INSERTCAP        = $01;
 OPCHDA_REPLACECAP       = $02;
 OPCHDA_INSERTREPLACECAP = $04;
 OPCHDA_DELETERAWCAP     = $08;
 OPCHDA_DELETEATTIMECAP  = $10;

type
 OPCHDA_OPERATORCODES = TOleEnum;
const
 OPCHDA_EQUAL        = 1;
 OPCHDA_LESS         = 2;
 OPCHDA_LESSEQUAL    = 3;
 OPCHDA_GREATER      = 4;
 OPCHDA_GREATEREQUAL = 5;
 OPCHDA_NOTEQUAL     = 6;
type
 OPCHDA_OPERATORCODESARRAY = array[0..65535] of OPCHDA_OPERATORCODES;
 POPCHDA_OPERATORCODESARRAY = ^OPCHDA_OPERATORCODESARRAY;

type
 OPCHDA_EDITTYPE = TOleEnum;
const
 OPCHDA_INSERT        = 1;
 OPCHDA_REPLACE       = 2;
 OPCHDA_INSERTREPLACE = 3;
 OPCHDA_DELETE        = 4;
type
 OPCHDA_EDITTYPEARRAY = array[0..65535] of OPCHDA_EDITTYPE;
 POPCHDA_EDITTYPEARRAY = ^OPCHDA_EDITTYPEARRAY;

type
 OPCHDA_AGGREGATE = TOleEnum;
const
 OPCHDA_NOAGGREGATE       = 0;
 OPCHDA_INTERPOLATIVE     = 1;
 OPCHDA_TOTAL             = 2;
 OPCHDA_AVERAGE           = 3;
 OPCHDA_TIMEAVERAGE       = 4;
 OPCHDA_COUNT             = 5;
 OPCHDA_STDEV             = 6;
 OPCHDA_MINIMUMACTUALTIME = 7;
 OPCHDA_MINIMUM           = 8;
 OPCHDA_MAXIMUMACTUALTIME = 9;
 OPCHDA_MAXIMUM           = 10;
 OPCHDA_START             = 11;
 OPCHDA_END               = 12;
 OPCHDA_DELTA             = 13;
 OPCHDA_REGSLOPE          = 14;
 OPCHDA_REGCONST          = 15;
 OPCHDA_REGDEV            = 16;
 OPCHDA_VARIANCE          = 17;
 OPCHDA_RANGE             = 18;
 OPCHDA_DURATIONGOOD      = 19;
 OPCHDA_DURATIONBAD       = 20;
 OPCHDA_PERCENTGOOD       = 21;
 OPCHDA_PERCENTBAD        = 22;
 OPCHDA_WORSTQUALITY      = 23;
 OPCHDA_ANNOTATIONS       = 24;
type
 OPCHDA_AGGREGATEARRAY = array[0..65535] of OPCHDA_AGGREGATE;
 POPCHDA_AGGREGATEARRAY = ^OPCHDA_AGGREGATEARRAY;

const

 // AttributeID

 OPCHDA_DATA_TYPE          = $01;
 OPCHDA_DESCRIPTION        = $02;
 OPCHDA_ENG_UNITS          = $03;
 OPCHDA_STEPPED            = $04;
 OPCHDA_ARCHIVING          = $05;
 OPCHDA_DERIVE_EQUATION    = $06;
 OPCHDA_NODE_NAME          = $07;
 OPCHDA_PROCESS_NAME       = $08;
 OPCHDA_SOURCE_NAME        = $09;
 OPCHDA_SOURCE_TYPE        = $0A;
 OPCHDA_NORMAL_MAXIMUM     = $0B;
 OPCHDA_NORMAL_MINIMUM     = $0C;
 OPCHDA_ITEMID             = $0D;

 OPCHDA_MAX_TIME_INT       = $0E;
 OPCHDA_MIN_TIME_INT       = $0F;
 OPCHDA_EXCEPTION_DEV      = $10;
 OPCHDA_EXCEPTION_DEV_TYPE = $11;
 OPCHDA_HIGH_ENTRY_LIMIT   = $12;
 OPCHDA_LOW_ENTRY_LIMIT    = $13;

 // OPCHDA_QUALITY -- these are the high-order 16 bits, OPC DA Quality
 // occupies low-order 16 bits

 OPCHDA_EXTRADATA          = $00010000;
 OPCHDA_INTERPOLATED       = $00020000;
 OPCHDA_RAW                = $00040000;
 OPCHDA_CALCULATED         = $00080000;
 OPCHDA_NOBOUND            = $00100000;
 OPCHDA_NODATA             = $00200000;
 OPCHDA_DATALOST           = $00400000;
 OPCHDA_CONVERSION         = $00800000;
 OPCHDA_PARTIAL            = $01000000;

type

 // *********************************************************************//
 // Forward declaration of types defined in TypeLibrary                  //
 // *********************************************************************//
 IOPCHDA_Browser = interface;
 IOPCHDA_Server = interface;
 IOPCHDA_SyncRead = interface;
 IOPCHDA_SyncUpdate = interface;
 IOPCHDA_SyncAnnotations = interface;
 IOPCHDA_AsyncRead = interface;
 IOPCHDA_AsyncUpdate = interface;
 IOPCHDA_AsyncAnnotations = interface;
 IOPCHDA_Playback = interface;
 IOPCHDA_DataCallback = interface;

 // *********************************************************************//
 // Declaration of structures, unions and aliases.                       //
 // *********************************************************************//
 OPCHDA_ANNOTATION = record
   hClient:                            OPCHANDLE;
   dwNumValues:                        DWORD;
   ftTimeStamps:                       PFileTimeArray;
   szAnnotation:                       POleStrList;
   ftAnnotationTime:                   PFileTimeArray;
   szUser:                             POleStrList;
 end;
 POPCHDA_ANNOTATION = ^OPCHDA_ANNOTATION;
 OPCHDA_ANNOTATIONARRAY = array[0..65535] of OPCHDA_ANNOTATION;
 POPCHDA_ANNOTATIONARRAY = ^OPCHDA_ANNOTATIONARRAY;

 OPCHDA_MODIFIEDITEM = record
   hClient:                            OPCHANDLE;
   dwCount:                            DWORD;
   pftTimeStamps:                      PFileTimeArray;
   pdwQualities:                       PDWORDARRAY;
   pvDataValues:                       POleVariantArray;
   pftModificationTime:                PFileTimeArray;
   pEditType:                          POPCHDA_EDITTYPEARRAY;
   szUser:                             POleStrList;
 end;
 POPCHDA_MODIFIEDITEM = ^OPCHDA_MODIFIEDITEM;
 OPCHDA_MODIFIEDITEMARRAY = array[0..65535] of OPCHDA_MODIFIEDITEM;
 POPCHDA_MODIFIEDITEMARRAY = ^OPCHDA_MODIFIEDITEMARRAY;

 OPCHDA_ATTRIBUTE = record
   hClient:                            OPCHANDLE;
   dwNumValues:                        DWORD;
   dwAttributeID:                      DWORD;
   ftTimeStamps:                       PFileTimeArray;
   vAttributeValues:                   POleVariantArray;
 end;
 POPCHDA_ATTRIBUTE = ^OPCHDA_ATTRIBUTE;
 OPCHDA_ATTRIBUTEARRAY = array[0..65535] of OPCHDA_ATTRIBUTE;
 POPCHDA_ATTRIBUTEARRAY = ^OPCHDA_ATTRIBUTEARRAY;

 OPCHDA_TIME = record
   bString:                            BOOL;
   szTime:                             POleStr;
   ftTime:                             TFileTime;
 end;
 POPCHDA_TIME = ^OPCHDA_TIME;

 OPCHDA_ITEM = record
   hClient:                            OPCHANDLE;
   haAggregate:                        DWORD;
   dwCount:                            DWORD;
   pftTimeStamps:                      PFileTimeArray;
   pdwQualities:                       PDWORDARRAY;
   pvDataValues:                       POleVariantArray;
 end;
 POPCHDA_ITEM = ^OPCHDA_ITEM;
 OPCHDA_ITEMARRAY = array[0..65535] of OPCHDA_ITEM;
 POPCHDA_ITEMARRAY = ^OPCHDA_ITEMARRAY;
 OPCHDA_ITEMPTRARRAY = array[0..65535] of POPCHDA_ITEM;
 POPCHDA_ITEMPTRARRAY = ^OPCHDA_ITEMPTRARRAY;

 // *********************************************************************//
 // Interface: IOPCHDA_Browser
 // GUID:      {1F1217B1-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_Browser = interface(IUnknown)
   ['{1F1217B1-DEE0-11D2-A5E5-000086339399}']
   function GetEnum(
           dwBrowseType:               OPCHDA_BROWSETYPE;
     out   ppIEnumString:              IEnumString): HResult; stdcall;
   function ChangeBrowsePosition(
           dwBrowseDirection:          OPCHDA_BROWSEDIRECTION;
           szString:                   POleStr): HResult; stdcall;
   function GetItemID(
           szNode:                     POleStr;
     out   pszItemID:                  POleStr): HResult; stdcall;
   function GetBranchPosition(
     out   pszBranchPos:               POleStr): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_Server
 // GUID:      {1F1217B0-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_Server = interface(IUnknown)
   ['{1F1217B0-DEE0-11D2-A5E5-000086339399}']
   function GetItemAttributes(
     out   pdwCount:                   DWORD;
     out   ppdwAttrID:                 PDWORDARRAY;
     out   ppszAttrName:               POleStrList;
     out   ppszAttrDesc:               POleStrList;
     out   ppvtAttrDataType:           PVarTypeList): HResult; stdcall;
   function GetAggregates(
     out   pdwCount:                   DWORD;
     out   ppdwAggrID:                 PDWORDARRAY;
     out   ppszAggrName:               POleStrList;
     out   ppszAggrDesc:               POleStrList): HResult; stdcall;
   function GetHistorianStatus(
     out   pwStatus:                   OPCHDA_SERVERSTATUS;
     out   pftCurrentTime:             PFileTimeArray;
     out   pftStartTime:               PFileTimeArray;
     out   pwMajorVersion:             Word;
     out   pwMinorVersion:             Word;
     out   pwBuildNumber:              Word;
     out   pdwMaxReturnValues:         DWORD;
     out   ppszStatusString:           POleStr;
     out   ppszVendorInfo:             POleStr): HResult; stdcall;
   function GetItemHandles(
           dwCount:                    DWORD;
           pszItemID:                  POleStrList;
           phClient:                   POPCHANDLEARRAY;
     out   pphServer:                  POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReleaseItemHandles(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ValidateItemIDs(
           dwCount:                    DWORD;
           pszItemID:                  POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function CreateBrowse(
           dwCount:                    DWORD;
           pdwAttrID:                  PDWORDARRAY;
           pOperator:                  POPCHDA_OPERATORCODESARRAY;
           vFilter:                    POleVariantArray;
     out   pphBrowser:                 IOPCHDA_Browser;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_SyncRead
 // GUID:      {1F1217B2-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_SyncRead = interface(IUnknown)
   ['{1F1217B2-DEE0-11D2-A5E5-000086339399}']
   function ReadRaw(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumValues:                DWORD;
           bBounds:                    BOOL;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppItemValues:               POPCHDA_ITEMARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadProcessed(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           ftResampleInterval:         TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           haAggregate:                PDWORDARRAY;
     out   ppItemValues:               POPCHDA_ITEMARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadAtTime(
           dwNumTimeStamps:            DWORD;
           ftTimeStamps:               PFileTimeArray;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppItemValues:               POPCHDA_ITEMARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadModified(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumValues:                DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppItemValues:               POPCHDA_MODIFIEDITEMARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadAttribute(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           hServer:                    OPCHANDLE;
           dwNumAttributes:            DWORD;
           pdwAttributeIDs:            PDWORDARRAY;
     out   ppAttributeValues:          POPCHDA_ATTRIBUTEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_SyncUpdate
 // GUID:      {1F1217B3-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_SyncUpdate = interface(IUnknown)
   ['{1F1217B3-DEE0-11D2-A5E5-000086339399}']
   function QueryCapabilities(
     out   pCapabilities:              OPCHDA_UPDATECAPABILITIES):
           HResult; stdcall;
   function Insert(
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Replace(
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function InsertReplace(
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DeleteRaw(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DeleteAtTime(
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_SyncAnnotations
 // GUID:      {1F1217B4-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_SyncAnnotations = interface(IUnknown)
   ['{1F1217B4-DEE0-11D2-A5E5-000086339399}']
   function QueryCapabilities(
     out   pCapabilities:              OPCHDA_ANNOTATIONCAPABILITIES):
           HResult; stdcall;
   function Read(
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppAnnotationValues:         POPCHDA_ANNOTATIONARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Insert(
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           pAnnotationValues:          POPCHDA_ANNOTATIONARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_AsyncRead
 // GUID:      {1F1217B5-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_AsyncRead = interface(IUnknown)
   ['{1F1217B5-DEE0-11D2-A5E5-000086339399}']
   function ReadRaw(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumValues:                DWORD;
           bBounds:                    BOOL;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function AdviseRaw(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
           ftUpdateInterval:           TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadProcessed(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           ftResampleInterval:         TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           haAggregate:                PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function AdviseProcessed(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
           ftResampleInterval:         TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           haAggregate:                PDWORDARRAY;
           dwNumIntervals:             DWORD;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadAtTime(
           dwTransactionID:            DWORD;
           dwNumTimeStamps:            DWORD;
           ftTimeStamps:               PFileTimeArray;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadModified(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumValues:                DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadAttribute(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           hServer:                    OPCHANDLE;
           dwNumAttributes:            DWORD;
           dwAttributeIDs:             PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Cancel(
           dwCancelID:                 DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_AsyncUpdate
 // GUID:      {1F1217B6-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_AsyncUpdate = interface(IUnknown)
   ['{1F1217B6-DEE0-11D2-A5E5-000086339399}']
   function QueryCapabilities(
     out   pCapabilities:              OPCHDA_UPDATECAPABILITIES):
           HResult; stdcall;
   function Insert(
           dwTransactionID:            DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Replace(
           dwTransactionID:            DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function InsertReplace(
           dwTransactionID:            DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           vDataValues:                POleVariantArray;
           pdwQualities:               PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DeleteRaw(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function DeleteAtTime(
           dwTransactionID:            DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Cancel(
           dwCancelID:                 DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_AsyncAnnotations
 // GUID:      {1F1217B7-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_AsyncAnnotations = interface(IUnknown)
   ['{1F1217B7-DEE0-11D2-A5E5-000086339399}']
   function QueryCapabilities(
     out   pCapabilities:              OPCHDA_ANNOTATIONCAPABILITIES):
           HResult; stdcall;
   function Read(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Insert(
           dwTransactionID:            DWORD;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           ftTimeStamps:               PFileTimeArray;
           pAnnotationValues:          POPCHDA_ANNOTATIONARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Cancel(
           dwCancelID:                 DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_Playback
 // GUID:      {1F1217B8-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_Playback = interface(IUnknown)
   ['{1F1217B8-DEE0-11D2-A5E5-000086339399}']
   function ReadRawWithUpdate(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           dwNumValues:                DWORD;
           ftUpdateDuration:           TFileTime;
           ftUpdateInterval:           TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ReadProcessedWithUpdate(
           dwTransactionID:            DWORD;
     var   htStartTime:                OPCHDA_TIME;
     var   htEndTime:                  OPCHDA_TIME;
           ftResampleInterval:         TFileTime;
           dwNumIntervals:             DWORD;
           ftUpdateInterval:           TFileTime;
           dwNumItems:                 DWORD;
           phServer:                   POPCHANDLEARRAY;
           haAggregate:                PDWORDARRAY;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Cancel(
           dwCancelID:                 DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCHDA_DataCallback
 // GUID:      {1F1217B9-DEE0-11D2-A5E5-000086339399}
 // *********************************************************************//
 IOPCHDA_DataCallback = interface(IUnknown)
   ['{1F1217B9-DEE0-11D2-A5E5-000086339399}']
   function OnDataChange(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwNumItems:                 DWORD;
           pItemValues:                POPCHDA_ITEMARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnReadComplete(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwNumItems:                 DWORD;
           pItemValues:                POPCHDA_ITEMARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnReadModifiedComplete(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwNumItems:                 DWORD;
           pItemValues:                POPCHDA_MODIFIEDITEMARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnReadAttributeComplete(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           hClient:                    OPCHANDLE;
           dwNumItems:                 DWORD;
           pAttributeValues:           POPCHDA_ATTRIBUTEARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnReadAnnotations(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwNumItems:                 DWORD;
           pAnnotationValues:          POPCHDA_ANNOTATIONARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnInsertAnnotations(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwCount:                    DWORD;
           phClients:                  POPCHANDLEARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnPlayback(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwNumItems:                 DWORD;
           ppItemValues:               POPCHDA_ITEMPTRARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnUpdateComplete(
           dwTransactionID:            DWORD;
           hrStatus:                   HResult;
           dwCount:                    DWORD;
           phClients:                  POPCHANDLEARRAY;
           phrErrors:                  PResultList): HResult; stdcall;
   function OnCancelComplete(
           dwCancelID:                 DWORD): HResult; stdcall;
 end;


 {*******************************************************}
 {                                                       }
 {       OPC Data Access 3.0 (and earlier versions)      }
 {                                                       }
 {       Delphi conversion generated and supplied by     }
 {       Mike Dillamore                                  }
 {       OPC Programmers' Connection                     }
 {       http://www.opcconnect.com/                      }
 {       mailto:opc@opcconnect.com                       }
 {                                                       }
 {       Improvements contributed by                     }
 {       Martin Lafferty                                 }
 {       Production Robots Engineering Ltd               }
 {       http://www.prel.co.uk/                          }
 {       mailto:prOpcKit@prel.co.uk                      }
 {                                                       }
 {*******************************************************}
 
 // ************************************************************************ //
 // Type Lib: OPCProxy.dll
 // IID\LCID: {3B540B51-0378-4551-ADCC-EA9B104302BF}\0 - Data Access 3.0
 // IID\LCID: {B28EEDB2-AC6F-11D1-84D5-00608CB8A7E9}\0 - Data Access 2.0
 // ************************************************************************ //
 
 // *********************************************************************//
 // GUIDS declared in the TypeLibrary                                    //
 // *********************************************************************//
const
 LIBID_OPCDA: TGUID = '{3B540B51-0378-4551-ADCC-EA9B104302BF}';
 IID_IOPCServer: TIID = '{39C13A4D-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCServerPublicGroups: TIID = '{39C13A4E-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCBrowseServerAddressSpace: TIID = '{39C13A4F-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCGroupStateMgt: TIID = '{39C13A50-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCPublicGroupStateMgt: TIID = '{39C13A51-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCSyncIO: TIID = '{39C13A52-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCAsyncIO: TIID = '{39C13A53-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCItemMgt: TIID = '{39C13A54-011E-11D0-9675-0020AFD8ADB3}';
 IID_IEnumOPCItemAttributes: TIID = '{39C13A55-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCDataCallback: TIID = '{39C13A70-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCAsyncIO2: TIID = '{39C13A71-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCItemProperties: TIID = '{39C13A72-011E-11D0-9675-0020AFD8ADB3}';
 IID_IOPCItemDeadbandMgt: TIID = '{5946DA93-8B39-4ec8-AB3D-AA73DF5BC86F}';
 IID_IOPCItemSamplingMgt: TIID = '{3E22D313-F08B-41a5-86C8-95E95CB49FFC}';
 IID_IOPCBrowse: TIID = '{39227004-A18F-4b57-8B0A-5235670F4468}';
 IID_IOPCItemIO: TIID = '{85C0B427-2893-4cbc-BD78-E5FC5146F08F}';
 IID_IOPCSyncIO2: TIID = '{730F5F0F-55B1-4c81-9E18-FF8A0904E1FA}';
 IID_IOPCAsyncIO3: TIID = '{0967B97B-36EF-423e-B6F8-6BFF1E40D39D}';
 IID_IOPCGroupStateMgt2: TIID = '{8E368666-D72E-4f78-87ED-647611C61C9F}';
 CATID_OPCDAServer10: TGUID = '{63D5F430-CFE4-11d1-B2C8-0060083BA1FB}';
 CATID_OPCDAServer20: TGUID = '{63D5F432-CFE4-11d1-B2C8-0060083BA1FB}';
 CATID_OPCDAServer30: TGUID = '{CC603642-66D7-48f1-B69A-B625E73652D7}';
 CATID_XMLDAServer10: TGUID = '{3098EDA4-A006-48b2-A27F-247453959408}';

 // *********************************************************************//
 // Declaration of Enumerations defined in Type Library                  //
 // *********************************************************************//
type
 OPCDATASOURCE = TOleEnum;
const
 OPC_DS_CACHE  = 1;
 OPC_DS_DEVICE = 2;

type
 OPCBROWSETYPE = TOleEnum;
const
 OPC_BRANCH = 1;
 OPC_LEAF   = 2;
 OPC_FLAT   = 3;

type
 OPCNAMESPACETYPE = TOleEnum;
const
 OPC_NS_HIERARCHIAL = 1;
 OPC_NS_FLAT        = 2;

type
 OPCBROWSEDIRECTION = TOleEnum;
const
 OPC_BROWSE_UP   = 1;
 OPC_BROWSE_DOWN = 2;
 OPC_BROWSE_TO   = 3;

type
 OPCEUTYPE = TOleEnum;
const
 OPC_NOENUM     = 0;
 OPC_ANALOG     = 1;
 OPC_ENUMERATED = 2;

type
 OPCSERVERSTATE = TOleEnum;
const
 OPC_STATUS_RUNNING    = 1;
 OPC_STATUS_FAILED     = 2;
 OPC_STATUS_NOCONFIG   = 3;
 OPC_STATUS_SUSPENDED  = 4;
 OPC_STATUS_TEST       = 5;
 OPC_STATUS_COMM_FAULT = 6;

type
 OPCENUMSCOPE = TOleEnum;
const
 OPC_ENUM_PRIVATE_CONNECTIONS = 1;
 OPC_ENUM_PUBLIC_CONNECTIONS  = 2;
 OPC_ENUM_ALL_CONNECTIONS     = 3;
 OPC_ENUM_PRIVATE             = 4;
 OPC_ENUM_PUBLIC              = 5;
 OPC_ENUM_ALL                 = 6;

type
 OPCBROWSEFILTER = TOleEnum;
const
 OPC_BROWSE_FILTER_ALL      = 1;
 OPC_BROWSE_FILTER_BRANCHES = 2;
 OPC_BROWSE_FILTER_ITEMS    = 3;

 // *********************************************************************//
 // Constants                                                            //
 // *********************************************************************//
 // Category description strings
const
    OPC_CATEGORY_DESCRIPTION_DA10    = 'OPC Data Access Servers Version 1.0';
    OPC_CATEGORY_DESCRIPTION_DA20    = 'OPC Data Access Servers Version 2.0';
    OPC_CATEGORY_DESCRIPTION_DA30    = 'OPC Data Access Servers Version 3.0';
    OPC_CATEGORY_DESCRIPTION_XMLDA10 = 'OPC XML Data Access Servers Version 1.0';
 
 // Values for access rights mask
const
 OPC_READABLE  = $01;
 OPC_WRITEABLE = $02;

 // Values for browse element flags
const
 OPC_BROWSE_HASCHILDREN = $01;
 OPC_BROWSE_ISITEM      = $02;

 // *********************************************************************//
 // OPC Quality flags                                                    //
 // *********************************************************************//
 // Masks for extracting quality subfields
 // (note 'status' mask also includes 'Quality' bits)
 
const
 OPC_QUALITY_MASK           = $C0;
 OPC_STATUS_MASK            = $FC;
 OPC_LIMIT_MASK             = $03;

 // Values for QUALITY_MASK bit field
const
 OPC_QUALITY_BAD            = $00;
 OPC_QUALITY_UNCERTAIN      = $40;
 OPC_QUALITY_GOOD           = $C0;

 // STATUS_MASK Values for Quality = BAD
const
 OPC_QUALITY_CONFIG_ERROR              = $04;
 OPC_QUALITY_NOT_CONNECTED             = $08;
 OPC_QUALITY_DEVICE_FAILURE            = $0C;
 OPC_QUALITY_SENSOR_FAILURE            = $10;
 OPC_QUALITY_LAST_KNOWN                = $14;
 OPC_QUALITY_COMM_FAILURE              = $18;
 OPC_QUALITY_OUT_OF_SERVICE            = $1C;
 OPC_QUALITY_WAITING_FOR_INITIAL_DATA  = $20;

 // STATUS_MASK Values for Quality = UNCERTAIN
const
 OPC_QUALITY_LAST_USABLE    = $44;
 OPC_QUALITY_SENSOR_CAL     = $50;
 OPC_QUALITY_EGU_EXCEEDED   = $54;
 OPC_QUALITY_SUB_NORMAL     = $58;

 // STATUS_MASK Values for Quality = GOOD
const
 OPC_QUALITY_LOCAL_OVERRIDE = $D8;

 // Values for Limit Bitfield
const
 OPC_LIMIT_OK    = $00;
 OPC_LIMIT_LOW   = $01;
 OPC_LIMIT_HIGH  = $02;
 OPC_LIMIT_CONST = $03;

 // *********************************************************************//
 // Data Access 2.0 Property IDs:                                        //
 // *********************************************************************//
const
 OPC_PROP_CDT            = 1;
 OPC_PROP_VALUE          = 2;
 OPC_PROP_QUALITY        = 3;
 OPC_PROP_TIME           = 4;
 OPC_PROP_RIGHTS         = 5;
 OPC_PROP_SCANRATE       = 6;

 OPC_PROP_UNIT           = 100;
 OPC_PROP_DESC           = 101;
 OPC_PROP_HIEU           = 102;
 OPC_PROP_LOEU           = 103;
 OPC_PROP_HIRANGE        = 104;
 OPC_PROP_LORANGE        = 105;
 OPC_PROP_CLOSE          = 106;
 OPC_PROP_OPEN           = 107;
 OPC_PROP_TIMEZONE       = 108;

 OPC_PROP_DSP            = 200;
 OPC_PROP_FGC            = 201;
 OPC_PROP_BGC            = 202;
 OPC_PROP_BLINK          = 203;
 OPC_PROP_BMP            = 204;
 OPC_PROP_SND            = 205;
 OPC_PROP_HTML           = 206;
 OPC_PROP_AVI            = 207;

 OPC_PROP_ALMSTAT        = 300;
 OPC_PROP_ALMHELP        = 301;
 OPC_PROP_ALMAREAS       = 302;
 OPC_PROP_ALMPRIMARYAREA = 303;
 OPC_PROP_ALMCONDITION   = 304;
 OPC_PROP_ALMLIMIT       = 305;
 OPC_PROP_ALMDB          = 306;
 OPC_PROP_ALMHH          = 307;
 OPC_PROP_ALMH           = 308;
 OPC_PROP_ALML           = 309;
 OPC_PROP_ALMLL          = 310;
 OPC_PROP_ALMROC         = 311;
 OPC_PROP_ALMDEV         = 312;

// *********************************************************************//
// Data Access 3.0 Property IDs:                                        //
// *********************************************************************//
const
 OPC_PROPERTY_DATATYPE           = 1;
 OPC_PROPERTY_VALUE              = 2;
 OPC_PROPERTY_QUALITY            = 3;
 OPC_PROPERTY_TIMESTAMP          = 4;
 OPC_PROPERTY_ACCESS_RIGHTS      = 5;
 OPC_PROPERTY_SCAN_RATE          = 6;
 OPC_PROPERTY_EU_TYPE            = 7;
 OPC_PROPERTY_EU_INFO            = 8;
 OPC_PROPERTY_EU_UNITS           = 100;
 OPC_PROPERTY_DESCRIPTION        = 101;
 OPC_PROPERTY_HIGH_EU            = 102;
 OPC_PROPERTY_LOW_EU             = 103;
 OPC_PROPERTY_HIGH_IR            = 104;
 OPC_PROPERTY_LOW_IR             = 105;
 OPC_PROPERTY_CLOSE_LABEL        = 106;
 OPC_PROPERTY_OPEN_LABEL         = 107;
 OPC_PROPERTY_TIMEZONE           = 108;
 OPC_PROPERTY_CONDITION_STATUS   = 300;
 OPC_PROPERTY_ALARM_QUICK_HELP   = 301;
 OPC_PROPERTY_ALARM_AREA_LIST    = 302;
 OPC_PROPERTY_PRIMARY_ALARM_AREA = 303;
 OPC_PROPERTY_CONDITION_LOGIC    = 304;
 OPC_PROPERTY_LIMIT_EXCEEDED     = 305;
 OPC_PROPERTY_DEADBAND           = 306;
 OPC_PROPERTY_HIHI_LIMIT         = 307;
 OPC_PROPERTY_HI_LIMIT           = 308;
 OPC_PROPERTY_LO_LIMIT           = 309;
 OPC_PROPERTY_LOLO_LIMIT         = 310;
 OPC_PROPERTY_CHANGE_RATE_LIMIT  = 311;
 OPC_PROPERTY_DEVIATION_LIMIT    = 312;
 OPC_PROPERTY_SOUND_FILE         = 313;

 // *********************************************************************//
 // Data Access 3.0 Property Descriptions:                               //
 // *********************************************************************//
const
 OPC_PROPERTY_DESC_DATATYPE           = 'Item Canonical Data Type';
 OPC_PROPERTY_DESC_VALUE              = 'Item Value';
 OPC_PROPERTY_DESC_QUALITY            = 'Item Quality';
 OPC_PROPERTY_DESC_TIMESTAMP          = 'Item Timestamp';
 OPC_PROPERTY_DESC_ACCESS_RIGHTS      = 'Item Access Rights';
 OPC_PROPERTY_DESC_SCAN_RATE          = 'Server Scan Rate';
 OPC_PROPERTY_DESC_EU_TYPE            = 'Item EU Type';
 OPC_PROPERTY_DESC_EU_INFO            = 'Item EU Info';
 OPC_PROPERTY_DESC_EU_UNITS           = 'EU Units';
 OPC_PROPERTY_DESC_DESCRIPTION        = 'Item Description';
 OPC_PROPERTY_DESC_HIGH_EU            = 'High EU';
 OPC_PROPERTY_DESC_LOW_EU             = 'Low EU';
 OPC_PROPERTY_DESC_HIGH_IR            = 'High Instrument Range';
 OPC_PROPERTY_DESC_LOW_IR             = 'Low Instrument Range';
 OPC_PROPERTY_DESC_CLOSE_LABEL        = 'Contact Close Label';
 OPC_PROPERTY_DESC_OPEN_LABEL         = 'Contact Open Label';
 OPC_PROPERTY_DESC_TIMEZONE           = 'Item Timezone';
 OPC_PROPERTY_DESC_CONDITION_STATUS   = 'Condition Status';
 OPC_PROPERTY_DESC_ALARM_QUICK_HELP   = 'Alarm Quick Help';
 OPC_PROPERTY_DESC_ALARM_AREA_LIST    = 'Alarm Area List';
 OPC_PROPERTY_DESC_PRIMARY_ALARM_AREA = 'Primary Alarm Area';
 OPC_PROPERTY_DESC_CONDITION_LOGIC    = 'Condition Logic';
 OPC_PROPERTY_DESC_LIMIT_EXCEEDED     = 'Limit Exceeded';
 OPC_PROPERTY_DESC_DEADBAND           = 'Deadband';
 OPC_PROPERTY_DESC_HIHI_LIMIT         = 'HiHi Limit';
 OPC_PROPERTY_DESC_HI_LIMIT           = 'Hi Limit';
 OPC_PROPERTY_DESC_LO_LIMIT           = 'Lo Limit';
 OPC_PROPERTY_DESC_LOLO_LIMIT         = 'LoLo Limit';
 OPC_PROPERTY_DESC_CHANGE_RATE_LIMIT  = 'Rate of Change Limit';
 OPC_PROPERTY_DESC_DEVIATION_LIMIT    = 'Deviation Limit';
 OPC_PROPERTY_DESC_SOUND_FILE         = 'Sound File';

type

 // *********************************************************************//
 // Forward declaration of interfaces defined in Type Library            //
 // *********************************************************************//
 IOPCServer = interface;
 IOPCServerPublicGroups = interface;
 IOPCBrowseServerAddressSpace = interface;
 IOPCGroupStateMgt = interface;
 IOPCPublicGroupStateMgt = interface;
 IOPCSyncIO = interface;
 IOPCAsyncIO = interface;
 IOPCItemMgt = interface;
 IEnumOPCItemAttributes = interface;
 IOPCDataCallback = interface;
 IOPCAsyncIO2 = interface;
 IOPCItemProperties = interface;
 IOPCItemDeadbandMgt = interface;
 IOPCItemSamplingMgt = interface;
 IOPCBrowse = interface;
 IOPCItemIO = interface;
 IOPCSyncIO2 = interface;
 IOPCAsyncIO3 = interface;
 IOPCGroupStateMgt2 = interface;

 // *********************************************************************//
 // Declaration of structures, unions and aliases.                       //
 // *********************************************************************//
 OPCGROUPHEADER = record
   dwSize:               DWORD;
   dwItemCount:          DWORD;
   hClientGroup:         OPCHANDLE;
   dwTransactionID:      DWORD;
   hrStatus:             HResult;
 end;
 POPCGROUPHEADER = ^OPCGROUPHEADER;

 OPCITEMHEADER1 = record
   hClient:              OPCHANDLE;
   dwValueOffset:        DWORD;
   wQuality:             Word;
   wReserved:            Word;
   ftTimeStampItem:      TFileTime;
 end;
 POPCITEMHEADER1 = ^OPCITEMHEADER1;
 OPCITEMHEADER1ARRAY = array[0..65535] of OPCITEMHEADER1;
 POPCITEMHEADER1ARRAY = ^OPCITEMHEADER1ARRAY;

 OPCITEMHEADER2 = record
   hClient:              OPCHANDLE;
   dwValueOffset:        DWORD;
   wQuality:             Word;
   wReserved:            Word;
 end;
 POPCITEMHEADER2 = ^OPCITEMHEADER2;
 OPCITEMHEADER2ARRAY = array[0..65535] of OPCITEMHEADER2;
 POPCITEMHEADER2ARRAY = ^OPCITEMHEADER2ARRAY;

 OPCGROUPHEADERWRITE = record
   dwItemCount:          DWORD;
   hClientGroup:         OPCHANDLE;
   dwTransactionID:      DWORD;
   hrStatus:             HResult;
 end;
 POPCGROUPHEADERWRITE = ^OPCGROUPHEADERWRITE;

 OPCITEMHEADERWRITE = record
   hClient:              OPCHANDLE;
   dwError:              HResult;
 end;
 POPCITEMHEADERWRITE = ^OPCITEMHEADERWRITE;
 OPCITEMHEADERWRITEARRAY = array[0..65535] of OPCITEMHEADERWRITE;
 POPCITEMHEADERWRITEARRAY = ^OPCITEMHEADERWRITEARRAY;

 OPCITEMSTATE = record
   hClient:              OPCHANDLE;
   ftTimeStamp:          TFileTime;
   wQuality:             Word;
   wReserved:            Word;
   vDataValue:           OleVariant;
 end;
 POPCITEMSTATE = ^OPCITEMSTATE;
 OPCITEMSTATEARRAY = array[0..65535] of OPCITEMSTATE;
 POPCITEMSTATEARRAY = ^OPCITEMSTATEARRAY;

 OPCSERVERSTATUS = record
   ftStartTime:          TFileTime;
   ftCurrentTime:        TFileTime;
   ftLastUpdateTime:     TFileTime;
   dwServerState:        OPCSERVERSTATE;
   dwGroupCount:         DWORD;
   dwBandWidth:          DWORD;
   wMajorVersion:        Word;
   wMinorVersion:        Word;
   wBuildNumber:         Word;
   wReserved:            Word;
   szVendorInfo:         POleStr;
 end;
 POPCSERVERSTATUS = ^OPCSERVERSTATUS;

 OPCITEMDEF = record
   szAccessPath:         POleStr;
   szItemID:             POleStr;
   bActive:              BOOL;
   hClient:              OPCHANDLE;
   dwBlobSize:           DWORD;
   pBlob:                PByteArray;
   vtRequestedDataType:  TVarType;
   wReserved:            Word;
 end;
 POPCITEMDEF = ^OPCITEMDEF;
 OPCITEMDEFARRAY = array[0..65535] of OPCITEMDEF;
 POPCITEMDEFARRAY = ^OPCITEMDEFARRAY;

 OPCITEMATTRIBUTES = record
   szAccessPath:         POleStr;
   szItemID:             POleStr;
   bActive:              BOOL;
   hClient:              OPCHANDLE;
   hServer:              OPCHANDLE;
   dwAccessRights:       DWORD;
   dwBlobSize:           DWORD;
   pBlob:                PByteArray;
   vtRequestedDataType:  TVarType;
   vtCanonicalDataType:  TVarType;
   dwEUType:             OPCEUTYPE;
   vEUInfo:              OleVariant;
 end;
 POPCITEMATTRIBUTES = ^OPCITEMATTRIBUTES;
 OPCITEMATTRIBUTESARRAY = array[0..65535] of OPCITEMATTRIBUTES;
 POPCITEMATTRIBUTESARRAY = ^OPCITEMATTRIBUTESARRAY;

 OPCITEMRESULT = record
   hServer:              OPCHANDLE;
   vtCanonicalDataType:  TVarType;
   wReserved:            Word;
   dwAccessRights:       DWORD;
   dwBlobSize:           DWORD;
   pBlob:                PByteArray;
 end;
 POPCITEMRESULT = ^OPCITEMRESULT;
 OPCITEMRESULTARRAY = array[0..65535] of OPCITEMRESULT;
 POPCITEMRESULTARRAY = ^OPCITEMRESULTARRAY;

 OPCITEMPROPERTY = record
   vtDataType:           TVarType;
   wReserved:            Word;
   dwPropertyID:         DWORD;
   szItemID:             POleStr;
   szDescription:        POleStr;
   vValue:               OleVariant;
   hrErrorID:            HResult;
   dwReserved:           DWORD;
 end;
 POPCITEMPROPERTY = ^OPCITEMPROPERTY;
 OPCITEMPROPERTYARRAY = array[0..65535] of OPCITEMPROPERTY;
 POPCITEMPROPERTYARRAY = ^OPCITEMPROPERTYARRAY;

 OPCITEMPROPERTIES = record
   hrErrorID:            HResult;
   dwNumProperties:      DWORD;
   pItemProperties:      POPCITEMPROPERTYARRAY;
   dwReserved:           DWORD;
 end;
 POPCITEMPROPERTIES = ^OPCITEMPROPERTIES;
 OPCITEMPROPERTIESARRAY = array[0..65535] of OPCITEMPROPERTIES;
 POPCITEMPROPERTIESARRAY = ^OPCITEMPROPERTIESARRAY;

 OPCBROWSEELEMENT = record
   szName:               POleStr;
   szItemID:             POleStr;
   dwFlagValue:          DWORD;
   dwReserved:           DWORD;
   ItemProperties:       OPCITEMPROPERTIES;
 end;
 POPCBROWSEELEMENT = ^OPCBROWSEELEMENT;
 OPCBROWSEELEMENTARRAY = array[0..65535] of OPCBROWSEELEMENT;
 POPCBROWSEELEMENTARRAY = ^OPCBROWSEELEMENTARRAY;

 OPCITEMVQT = record
   vDataValue:           OleVariant;
   bQualitySpecified:    BOOL;
   wQuality:             Word;
   wReserved:            Word;
   bTimeStampSpecified:  BOOL;
   dwReserved:           DWORD;
   ftTimeStamp:          TFileTime;
 end;
 POPCITEMVQT = ^OPCITEMVQT;
 OPCITEMVQTARRAY = array[0..65535] of OPCITEMVQT;
 POPCITEMVQTARRAY = ^OPCITEMVQTARRAY;

 // *********************************************************************//
 // Interface: IOPCServer
 // GUID:      {39C13A4D-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCServer = interface(IUnknown)
   ['{39C13A4D-011E-11D0-9675-0020AFD8ADB3}']
   function AddGroup(
           szName:                     POleStr;
           bActive:                    BOOL;
           dwRequestedUpdateRate:      DWORD;
           hClientGroup:               OPCHANDLE;
           pTimeBias:                  PLongint;
           pPercentDeadband:           PSingle;
           dwLCID:                     DWORD;
     out   phServerGroup:              OPCHANDLE;
     out   pRevisedUpdateRate:         DWORD;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
   function GetErrorString(
           dwError:                    HResult;
           dwLocale:                   TLCID;
     out   ppString:                   POleStr): HResult; stdcall;
   function GetGroupByName(
           szName:                     POleStr;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
   function GetStatus(
     out   ppServerStatus:             POPCSERVERSTATUS): HResult; stdcall;
   function RemoveGroup(
           hServerGroup:               OPCHANDLE;
           bForce:                     BOOL): HResult; stdcall;
   function CreateGroupEnumerator(
           dwScope:                    OPCENUMSCOPE;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCServerPublicGroups
 // GUID:      {39C13A4E-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCServerPublicGroups = interface(IUnknown)
   ['{39C13A4E-011E-11D0-9675-0020AFD8ADB3}']
   function GetPublicGroupByName(
           szName:                     POleStr;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
   function RemovePublicGroup(
           hServerGroup:               OPCHANDLE;
           bForce:                     BOOL): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCBrowseServerAddressSpace
 // GUID:      {39C13A4F-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCBrowseServerAddressSpace = interface(IUnknown)
   ['{39C13A4F-011E-11D0-9675-0020AFD8ADB3}']
   function QueryOrganization(
     out   pNameSpaceType:             OPCNAMESPACETYPE): HResult; stdcall;
   function ChangeBrowsePosition(
           dwBrowseDirection:          OPCBROWSEDIRECTION;
           szString:                   POleStr): HResult; stdcall;
   function BrowseOPCItemIDs(
           dwBrowseFilterType:         OPCBROWSETYPE;
           szFilterCriteria:           POleStr;
           vtDataTypeFilter:           TVarType;
           dwAccessRightsFilter:       DWORD;
     out   ppIEnumString:              IEnumString): HResult; stdcall;
   function GetItemID(
           szItemDataID:               POleStr;
     out   szItemID:                   POleStr): HResult; stdcall;
   function BrowseAccessPaths(
           szItemID:                   POleStr;
     out   ppIEnumString:              IEnumString): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCGroupStateMgt
 // GUID:      {39C13A50-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCGroupStateMgt = interface(IUnknown)
   ['{39C13A50-011E-11D0-9675-0020AFD8ADB3}']
   function GetState(
     out   pUpdateRate:                DWORD;
     out   pActive:                    BOOL;
     out   ppName:                     POleStr;
     out   pTimeBias:                  Longint;
     out   pPercentDeadband:           Single;
     out   pLCID:                      TLCID;
     out   phClientGroup:              OPCHANDLE;
     out   phServerGroup:              OPCHANDLE): HResult; stdcall;
   function SetState(
           pRequestedUpdateRate:       PDWORD;
     out   pRevisedUpdateRate:         DWORD;
           pActive:                    PBOOL;
           pTimeBias:                  PLongint;
           pPercentDeadband:           PSingle;
           pLCID:                      PLCID;
           phClientGroup:              POPCHANDLE): HResult; stdcall;
   function SetName(
           szName:                     POleStr): HResult; stdcall;
   function CloneGroup(
           szName:                     POleStr;
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCPublicGroupStateMgt
 // GUID:      {39C13A51-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCPublicGroupStateMgt = interface(IUnknown)
   ['{39C13A51-011E-11D0-9675-0020AFD8ADB3}']
   function GetState(
     out   pPublic:                    BOOL): HResult; stdcall;
   function MoveToPublic: HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCSyncIO
 // GUID:      {39C13A52-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCSyncIO = interface(IUnknown)
   ['{39C13A52-011E-11D0-9675-0020AFD8ADB3}']
   function Read(
           dwSource:                   OPCDATASOURCE;
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppItemValues:               POPCITEMSTATEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Write(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pItemValues:                POleVariantArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCAsyncIO
 // GUID:      {39C13A53-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCAsyncIO = interface(IUnknown)
   ['{39C13A53-011E-11D0-9675-0020AFD8ADB3}']
   function Read(
           dwConnection:               DWORD;
           dwSource:                   OPCDATASOURCE;
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   pTransactionID:             DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Write(
           dwConnection:               DWORD;
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pItemValues:                POleVariantArray;
     out   pTransactionID:             DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Refresh(
           dwConnection:               DWORD;
           dwSource:                   OPCDATASOURCE;
     out   pTransactionID:             DWORD): HResult; stdcall;
   function Cancel(
           dwTransactionID:            DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCItemMgt
 // GUID:      {39C13A54-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCItemMgt = interface(IUnknown)
   ['{39C13A54-011E-11D0-9675-0020AFD8ADB3}']
   function AddItems(
           dwCount:                    DWORD;
           pItemArray:                 POPCITEMDEFARRAY;
     out   ppAddResults:               POPCITEMRESULTARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ValidateItems(
           dwCount:                    DWORD;
           pItemArray:                 POPCITEMDEFARRAY;
           bBlobUpdate:                BOOL;
     out   ppValidationResults:        POPCITEMRESULTARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function RemoveItems(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function SetActiveState(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           bActive:                    BOOL;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function SetClientHandles(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           phClient:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function SetDatatypes(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pRequestedDatatypes:        PVarTypeList;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function CreateEnumerator(
     const riid:                       TIID;
     out   ppUnk:                      IUnknown): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IEnumOPCItemAttributes
 // GUID:      {39C13A55-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IEnumOPCItemAttributes = interface(IUnknown)
   ['{39C13A55-011E-11D0-9675-0020AFD8ADB3}']
   function Next(
           celt:                       ULONG;
     out   ppItemArray:                POPCITEMATTRIBUTESARRAY;
     out   pceltFetched:               ULONG): HResult; stdcall;
   function Skip(
           celt:                       ULONG): HResult; stdcall;
   function Reset: HResult; stdcall;
   function Clone(
     out   ppEnumItemAttributes:       IEnumOPCItemAttributes):
           HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCDataCallback
 // GUID:      {39C13A70-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCDataCallback = interface(IUnknown)
   ['{39C13A70-011E-11D0-9675-0020AFD8ADB3}']
   function OnDataChange(
           dwTransid:                  DWORD;
           hGroup:                     OPCHANDLE;
           hrMasterquality:            HResult;
           hrMastererror:              HResult;
           dwCount:                    DWORD;
           phClientItems:              POPCHANDLEARRAY;
           pvValues:                   POleVariantArray;
           pwQualities:                PWordArray;
           pftTimeStamps:              PFileTimeArray;
           pErrors:                    PResultList): HResult; stdcall;
   function OnReadComplete(
           dwTransid:                  DWORD;
           hGroup:                     OPCHANDLE;
           hrMasterquality:            HResult;
           hrMastererror:              HResult;
           dwCount:                    DWORD;
           phClientItems:              POPCHANDLEARRAY;
           pvValues:                   POleVariantArray;
           pwQualities:                PWordArray;
           pftTimeStamps:              PFileTimeArray;
           pErrors:                    PResultList): HResult; stdcall;
   function OnWriteComplete(
           dwTransid:                  DWORD;
           hGroup:                     OPCHANDLE;
           hrMastererr:                HResult;
           dwCount:                    DWORD;
           pClienthandles:             POPCHANDLEARRAY;
           pErrors:                    PResultList): HResult; stdcall;
   function OnCancelComplete(
           dwTransid:                  DWORD;
           hGroup:                     OPCHANDLE): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCAsyncIO2
 // GUID:      {39C13A71-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCAsyncIO2 = interface(IUnknown)
   ['{39C13A71-011E-11D0-9675-0020AFD8ADB3}']
   function Read(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Write(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pItemValues:                POleVariantArray;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function Refresh2(
           dwSource:                   OPCDATASOURCE;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD): HResult; stdcall;
   function Cancel2(
           dwCancelID:                 DWORD): HResult; stdcall;
   function SetEnable(
           bEnable:                    BOOL): HResult; stdcall;
   function GetEnable(
     out   pbEnable:                   BOOL): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCItemProperties
 // GUID:      {39C13A72-011E-11D0-9675-0020AFD8ADB3}
 // *********************************************************************//
 IOPCItemProperties = interface(IUnknown)
   ['{39C13A72-011E-11D0-9675-0020AFD8ADB3}']
   function QueryAvailableProperties(
           szItemID:                   POleStr;
     out   pdwCount:                   DWORD;
     out   ppPropertyIDs:              PDWORDARRAY;
     out   ppDescriptions:             POleStrList;
     out   ppvtDataTypes:              PVarTypeList): HResult; stdcall;
   function GetItemProperties(
           szItemID:                   POleStr;
           dwCount:                    DWORD;
           pdwPropertyIDs:             PDWORDARRAY;
     out   ppvData:                    POleVariantArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function LookupItemIDs(
           szItemID:                   POleStr;
           dwCount:                    DWORD;
           pdwPropertyIDs:             PDWORDARRAY;
     out   ppszNewItemIDs:             POleStrList;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCItemDeadbandMgt
 // GUID:      {5946DA93-8B39-4ec8-AB3D-AA73DF5BC86F}
 // *********************************************************************//
 IOPCItemDeadbandMgt = interface(IUnknown)
   ['{5946DA93-8B39-4ec8-AB3D-AA73DF5BC86F}']
   function SetItemDeadband(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pPercentDeadband:           PSingleArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function GetItemDeadband(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppPercentDeadband:          PSingleArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ClearItemDeadband(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCItemSamplingMgt
 // GUID:      {3E22D313-F08B-41a5-86C8-95E95CB49FFC}
 // *********************************************************************//
 IOPCItemSamplingMgt = interface(IUnknown)
   ['{3E22D313-F08B-41a5-86C8-95E95CB49FFC}']
   function SetItemSamplingRate(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pdwRequestedSamplingRate:   PDWORDARRAY;
     out   ppdwRevisedSamplingRate:    PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function GetItemSamplingRate(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppdwSamplingRate:           PDWORDARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function ClearItemSamplingRate(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function SetItemBufferEnable(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pbEnable:                   PBOOLARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function GetItemBufferEnable(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
     out   ppbEnable:                  PBOOLARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCBrowse
 // GUID:      {39227004-A18F-4b57-8B0A-5235670F4468}
 // *********************************************************************//
 IOPCBrowse = interface(IUnknown)
   ['{39227004-A18F-4b57-8B0A-5235670F4468}']
   function GetProperties(
           dwItemCount:                DWORD;
           pszItemIDs:                 POleStrList;
           bReturnPropertyValues:      BOOL;
           dwPropertyCount:            DWORD;
           pdwPropertyIDs:             PDWORDARRAY;
     out   ppItemProperties:           POPCITEMPROPERTIESARRAY):
           HResult; stdcall;
   function Browse(
           szItemID:                   POleStr;
     var   pszContinuationPoint:       POleStr;
           dwMaxElementsReturned:      DWORD;
           dwBrowseFilter:             OPCBROWSEFILTER;
           szElementNameFilter:        POleStr;
           szVendorFilter:             POleStr;
           bReturnAllProperties:       BOOL;
           bReturnPropertyValues:      BOOL;
           dwPropertyCount:            DWORD;
           pdwPropertyIDs:             PDWORDARRAY;
     out   pbMoreElements:             BOOL;
     out   pdwCount:                   DWORD;
     out   ppBrowseElements:           POPCBROWSEELEMENTARRAY):
           HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCItemIO
 // GUID:      {85C0B427-2893-4cbc-BD78-E5FC5146F08F}
 // *********************************************************************//
 IOPCItemIO = interface(IUnknown)
   ['{85C0B427-2893-4cbc-BD78-E5FC5146F08F}']
   function Read(
           dwCount:                    DWORD;
           pszItemIDs:                 POleStrList;
           pdwMaxAge:                  PDWORDARRAY;
     out   ppvValues:                  POleVariantArray;
     out   ppwQualities:               PWordArray;
     out   ppftTimeStamps:             PFileTimeArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function WriteVQT(
           dwCount:                    DWORD;
           pszItemIDs:                 POleStrList;
           pItemVQT:                   POPCITEMVQTARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCSyncIO2
 // GUID:      {730F5F0F-55B1-4c81-9E18-FF8A0904E1FA}
 // *********************************************************************//
 IOPCSyncIO2 = interface(IOPCSyncIO)
   ['{730F5F0F-55B1-4c81-9E18-FF8A0904E1FA}']
   function ReadMaxAge(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pdwMaxAge:                  PDWORDARRAY;
     out   ppvValues:                  POleVariantArray;
     out   ppwQualities:               PWordArray;
     out   ppftTimeStamps:             PFileTimeArray;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function WriteVQT(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pItemVQT:                   POPCITEMVQTARRAY;
     out   ppErrors:                   PResultList): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCAsyncIO3
 // GUID:      {0967B97B-36EF-423e-B6F8-6BFF1E40D39D}
 // *********************************************************************//
 IOPCAsyncIO3 = interface(IOPCAsyncIO2)
   ['{0967B97B-36EF-423e-B6F8-6BFF1E40D39D}']
   function ReadMaxAge(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pdwMaxAge:                  PDWORDARRAY;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function WriteVQT(
           dwCount:                    DWORD;
           phServer:                   POPCHANDLEARRAY;
           pItemVQT:                   POPCITEMVQTARRAY;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD;
     out   ppErrors:                   PResultList): HResult; stdcall;
   function RefreshMaxAge(
           dwMaxAge:                   DWORD;
           dwTransactionID:            DWORD;
     out   pdwCancelID:                DWORD): HResult; stdcall;
 end;

 // *********************************************************************//
 // Interface: IOPCGroupStateMgt2
 // GUID:      {8E368666-D72E-4f78-87ED-647611C61C9F}
 // *********************************************************************//
 IOPCGroupStateMgt2 = interface(IOPCGroupStateMgt)
   ['{8E368666-D72E-4f78-87ED-647611C61C9F}']
   function SetKeepAlive(
           dwKeepAliveTime:            DWORD;
     out   pdwRevisedKeepAliveTime:    DWORD): HResult; stdcall;
   function GetKeepAlive(
     out   pdwKeepAliveTime:           DWORD): HResult; stdcall;
 end;

 // Register clipboard formats for use with IDataObject / IAdviseSink
procedure OPCDARegisterClipboardFormats;

var // OPC data stream formats
 OPCSTMFORMATDATA: UINT = 0;
 OPCSTMFORMATDATATIME: UINT = 0;
 OPCSTMFORMATWRITECOMPLETE: UINT = 0;

{$ENDIF ~WINDOWS}

implementation

procedure OPCDARegisterClipboardFormats;
begin
 {$IFDEF WINDOWS}
 OPCSTMFORMATDATA := RegisterClipboardFormat('OPCSTMFORMATDATA');
 OPCSTMFORMATDATATIME := RegisterClipboardFormat('OPCSTMFORMATDATATIME');
 OPCSTMFORMATWRITECOMPLETE := RegisterClipboardFormat('OPCSTMFORMATWRITECOMPLETE');
 {$ENDIF ~WINDOWS}
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_opc;
begin
 OPCDARegisterClipboardFormats;
end;

procedure Free_crw_opc;
begin
end;

initialization

 Init_crw_opc;

finalization

 Free_crw_opc;

end.

//////////////
// END OF FILE
//////////////

