#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

#####################################################
# The script to prepare environment for launch ROOT.
#####################################################

########################
# script identification.
########################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

declare -i has_web=0;
declare -i has_dat=0;

function fatal(){
 echo -ne "\n$2\n\n";
 exit $1;
};

function print_usage(){
 echo "$scriptname version 1.0";
 echo "Copyright (c) 2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Prepare root (CERN) environment & run command line.";
 echo "Usage: $scriptname [root [arg] | cmnd [arg] ]";
 echo " root [arg]  - run root with arguments [arg]";
 echo " cmnd [arg]  - run command (cmnd) with arguments [arg]";
 echo "Examples:";
 echo " $scriptname env | sort";
 echo " $scriptname root test.root";
 echo " $scriptname printenv ROOTSYS";
 echo " $scriptname '\$ROOTSYS/bin/root'";
};

function default_app(){
 local mime="$1";
 if [ -n "$mime" ] && which gio >/dev/null 2>&1; then
  env LANG=en gio mime $mime | grep 'Default application ' | sed 's/.*://' | xargs;
 fi;
};

function update_assoc(){
 local mime="$1"; local app="$2";
 if [ -n "$mime" ] && [ -n "$app" ] && which gio >/dev/null 2>&1; then
  local def="$(default_app "$mime")";
  if [ -z "$def" ] || [ "x$def" != "x$app" ]; then gio mime $mime $app; fi;
 fi;
};

function parse_root_options(){
 while test -n "$1"; do
  case $1 in
   --web*) let "has_web=1"; ;;
   *.root) let "has_dat=1"; ;;
   *) ;;
  esac;
  shift;
 done;
};

function exec_root(){
 exec $ROOTSYS/bin/root "$@";
};

function execute_root(){
 case $1 in
  -web|--web|-web=on|--web=on) shift; let "has_web=1"; ;;
  *) ;;
 esac;
 update_assoc application/x-root RootOpen.desktop;
 parse_root_options "$@";
 local opt_web="--web=off";
 if [ $has_web -eq 1 ]; then opt_web=""; fi;
 if [ $has_dat -eq 1 ]; then 
  exec_root $opt_web -e 'TBrowser bro;' "$@";
 else
  exec_root $opt_web "$@";
 fi;
};

function eval_exec(){
 eval exec "$@";
};

readonly rootdir="/opt/daqgroup/share/root";
readonly rootexe="$rootdir/bin/root";

if pushd $rootdir >/dev/null 2>&1; then
 source bin/thisroot.sh;
else
 fatal 1 "Error: could not cd $rootdir";
fi;
popd >/dev/null 2>&1;

###############
# main function
###############
function main(){
 if [ -z "$ROOTSYS" ]; then
  fatal 1 "Error: ROOTSYS is not set.\nPlease (re)install package 'daqgroup-root'.";
 fi;
 if [ -x "$ROOTSYS/bin/root" ]; then
  case $1 in
   '')   print_usage; return 0; ;;
   root) shift; execute_root "$@"; ;;
   *)    eval_exec "$@"; ;;
  esac;
 else
  fatal 1 "Error: ROOT installation wrong.\nPlease install package 'daqgroup-root'.";
 fi;
};

main "$@";

##############
## END OF FILE
##############
