////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// This unit contains API implementation to use in DLL plugins for CRWDAQ.    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20030215 - Creation                                                        //
// 20231207 - Modified for FPC (A.K.)                                         //
// 20241109 - Modified to use interfaces.                                     //
// 20250129 - Use TAtomicCounter                                              //
////////////////////////////////////////////////////////////////////////////////

unit _crw_crwapiserver; // CRW API Server

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, sockets,
 Form_CrwDaqSysChild, Form_CurveWindow,
 Form_ListBoxSelection, Form_TextEditDialog,
 Unit_SystemConsole, Form_TextEditor,
 _crw_alloc, _crw_fpu, _crw_str, _crw_ef, _crw_zm,
 _crw_fifo, _crw_rtc, _crw_fio, _crw_plut, _crw_proc,
 _crw_curves, _crw_sort, _crw_ee, _crw_snd, _crw_task,
 _crw_lsqpoly, _crw_couple, _crw_calib, _crw_sect,
 _crw_eldraw, _crw_dynar, _crw_guard, _crw_spcfld,
 _crw_daqdev, _crw_daqsys, _crw_daqevnt, _crw_daqtags,
 _crw_pio, _crw_funmin, _crw_rfadata, _crw_spline,
 _crw_simplex, _crw_svd, _crw_uart, _crw_softdev,
 _crw_adamdev, _crw_dcc32, _crw_fpcup, _crw_hl,
 _crw_regexp, _crw_pipeio, _crw_assoc, _crw_wmctrl,
 _crw_sharm, _crw_fsm, _crw_dbapi, _crw_crypt,
 _crw_base64, _crw_base32, _crw_bsencode,
 _crw_dbglog, _crw_colors, _crw_utf8,
 _crw_crwapi, _crw_plugin_wrapper, _crw_if_masters,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
 //
 // Эти классы реализуют абстрактные интерфейсы, от которых они наследованы.
 //
 TCrwApiServer = class;
 TSysApiServer = class;
 TGuiApiServer = class;
 TDaqApiServer = class;
 TDanApiServer = class;
 //
 // Вспомогательные классы.
 //
 TRfaDataBaseWrapServer = class;
 //
 // Реализация дерева интерфейсов CRWDAQ.
 //
 TCrwApiServer = class(TPluginApi,ICrwApi)
 public
  procedure ServerActionsBeforeExecution;                                override;
  procedure ServerActionsAfterExecution;                                 override;
  function  CrwApi:ICrwApi;                                              override;
 public // ICrwApi
  function  Version:Cardinal;
  function  Target:Cardinal;
  function  SysApi:ISysApi;
  function  GuiApi:IGuiApi;
  function  DaqApi:IDaqApi;
  function  DanApi:IDanApi;
  function  PluginName:LongString;
  function  PluginSourceFileName:LongString;
  function  PluginBinaryFileName:LongString;
  function  VerifyPluginType(aTarget,aExpected:Cardinal; Test:Boolean=false):Boolean;
  function  VerifyPluginDate(aEdition,aVersion:Cardinal; Test:Boolean=false):Boolean;
 private
  mySysApiServer : TSysApiServer;
  myGuiApiServer : TGuiApiServer;
  myDaqApiServer : TDaqApiServer;
  myDanApiServer : TDanApiServer;
  myTarget       : Cardinal;
  myName         : LongString;
  mySource       : LongString;
  myBinary       : LongString;
  myCrwApi       : ICrwApi;
  mySysApi       : ISysApi;
  myGuiApi       : IGuiApi;
  myDaqApi       : IDaqApi;
  myDanApi       : IDanApi;
  myFpuModes     : TFPUModes;
  myTmpModes     : TFPUModes;
 public
  constructor Create(aTarget:Cardinal; const aName,aSource,aBinary:LongString);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  function    ExecuteDataAnalysis(const aDLLPath:LongString;
                                        aSrcWin:TFormCurveWindow;
                                  const aArgums:LongString):Int64;
  function    ExecuteDataAcquisition(aPlugin:TPluginWrapper;
                                     aDaqDev:TDaqDevice;
                                     aDaqCmd:Integer):Int64;
 end;
 //
 // Реализация системного интерфейса.
 //
 TSysApiServer = class(TPluginApi,ISysApi)
 public
  procedure ServerActionsBeforeExecution;                                override;
  procedure ServerActionsAfterExecution;                                 override;
  function  CrwApi:ICrwApi;                                              override;
 public // ISysApi
  function  RedirectStdIn(var stdIn:Text):boolean;
  function  RedirectStdOut(var stdOut:Text):boolean;
  function  IncPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
  function  DecPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
  procedure SafeMove(const Source; var Dest; Count: LongInt);
  procedure SafeFillChar(var X; Count: LongInt; Value: Byte);
  function  AdjustBufferSize(Size:LongInt; Step:LongInt):LongInt;
  function  Allocate(N:LongInt):Pointer;
  procedure Deallocate(var P:Pointer);
  function  AllocSize(P:Pointer):LongInt;
  function  Reallocate(var P:Pointer; N:LongInt):boolean;
  function  isNAN(const X: Double): Boolean;
  function  isINF(const X: Double): Boolean;
  function  _NaN: Double;
  function  _PlusInf: Double;
  function  _MinusInf: Double;
  function  FindZero(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
  function  FindMin(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
  function  Quanc8(F:zmFunType;A,B,AbsErr,RelErr:Double; var ErrEst:Double;
            var NoFun:Integer; var Flag:Double; Custom:Pointer):Double;
  function  MachEps:double;
  function  NativeTimeNow:TSystemTime;
  function  MSecNow:double;
  function  AssignNativeTime(Year:word=1;Month:word=1;Day:word=1;
                             Hour:word=0;Minut:word=0;Sec:word=0;
                             mSec:word=0):TSystemTime;
  function  MSecToNativeTime(mSecond:Double):TSystemTime;
  function  NativeTimeToMSec(const T:TSystemTime):double;
  function  DateTimeToMSec(Year:Word=1;Month:Word=1;Day:Word=1;
                           Hour:Word=0;Minut:Word=0;Sec:Word=0;
                           mSec:Word=0):Double;
  procedure MSecToDateTime(T:Double;
                           var Year,Month,Day,Hour,Min,Sec,mSec:word);
  function  GetDateStr(ms:Double;Delim:Char='.';YMD_Order:Boolean=false;
                       ShortYear:Boolean=false):LongString;
  function  GetTimeStr(ms:double;Delim:Char=':';
                       ShowMSec:Boolean=false):LongString;
  function  StrLen(Str: PChar): LongInt;
  function  StrLLen(Str: PChar; MaxLen: Integer): LongInt;
  function  StrEnd(Str: PChar): PChar;
  function  StrCopy(Dest, Source: PChar): PChar;
  function  StrECopy(Dest, Source: PChar): PChar;
  function  StrLCopy(Dest, Source: PChar; MaxLen: LongInt): PChar;
  function  StrPCopy(Dest: PChar; const Source: LongString): PChar;
  function  StrCat(Dest, Source: PChar): PChar;
  function  StrPCat(Dest:PChar; const Source: LongString): PChar;
  function  StrLCat(Dest, Source: PChar; MaxLen: LongInt): PChar;
  function  StrComp(Str1, Str2: PChar): Integer;
  function  StrIComp(Str1, Str2: PChar): Integer;
  function  StrLComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
  function  StrLIComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
  function  StrScan(Str: PChar; Chr: Char): PChar;
  function  StrRScan(Str: PChar; Chr: Char): PChar;
  function  StrUpper(Str: PChar): PChar;
  function  StrLower(Str: PChar): PChar;
  function  StrPos(Str1, Str2: PChar): PChar;
  function  StrPas(Str: PChar): LongString;
  function  StrPass(Str:PChar; const PassChars:TCharSet):PChar;
  function  StrMove(Dest, Source: PChar; Count: LongInt): PChar;
  function  GetTextNumLines(Text:PChar; Count:LongInt=MaxLongInt;
                            UnixStyle:boolean=false):LongInt;
  function  UpCase(c:Char):Char;
  function  LoCase(c:Char):Char;
  function  LoCaseStr(const S:LongString):LongString;
  function  UpCaseStr(const S:LongString):LongString;
  function  WinToDos(c:Char):Char;
  function  DosToWin(c:Char):Char;
  function  WinToDosStr(const S:LongString):LongString;
  function  DosToWinStr(const S:LongString):LongString;
  function  WinToDosLongStr(const S:LongString):LongString;
  function  DosToWinLongStr(const S:LongString):LongString;
  function  WinToKoi(c:Char):Char;
  function  KoiToWin(c:Char):Char;
  function  WinToKoiStr(const S:LongString):LongString;
  function  KoiToWinStr(const S:LongString):LongString;
  function  WinToKoiLongStr(const S:LongString):LongString;
  function  KoiToWinLongStr(const S:LongString):LongString;
  function  Format(const Fmt:LongString;
                   const Args: array of const):LongString;
  function  Str2CharSet(const S:LongString):TCharSet;
  function  CharSet2Str(const S:TCharSet):LongString;
  function  RightStr(const S:LongString; pos:integer):LongString;
  function  LeftStr(const S:LongString; count:integer):LongString;
  function  CharStr(Len:Integer; Ch:Char=' '):LongString;
  function  Pad(const S:LongString; Len:Integer; Ch:Char=' '):LongString;
  function  LeftPad(const S:LongString; Len:Integer;
                    Ch:Char=' '):LongString;
  function  CenterStr(const S:LongString; Width:Integer;
                      Ch:Char=' '):LongString;
  function  TrimLeadChars(const S:LongString;
                          const TrimChars:TCharSet):LongString;
  function  TrimTrailChars(const S:LongString;
                           const TrimChars:TCharSet):LongString;
  function  TrimChars(const S:LongString;
                      const LeadTrim,TrailTrim:TCharSet):LongString;
  function  TrimLead(const S:LongString):LongString;
  function  TrimTrail(const S:LongString):LongString;
  function  Trim(const S:LongString):LongString;
  function  UnifyAlias(const Name:LongString; Mode:Integer=ua_NameDef):LongString;
  function  UnifyFileAlias(const FileName:LongString; Mode:Integer=ua_FileDef):LongString;
  function  UnifySection(const aSectionName:LongString; Mode:Integer=ua_SectDef):LongString;
  procedure WordWrap(const InSt:LongString; var OutSt, Overlap:LongString;
                     Margin:Integer; PadToMargin:Boolean);
  function  ReplaceString(const Str,Find,Replace:LongString):LongString;
  function  ReplaceAlignStr(const Str:LongString;
                           Invert:Boolean):LongString;
  function  WordCount(const S:LongString;
                      const WordDelims:TCharSet):Integer;
  function  ExtractWord(N:Integer; const S:LongString;
                                const WordDelims:TCharSet):LongString;
  function  WordIndex(const Name,Str:LongString;
                      const Delims:TCharSet):Integer;
  function  ExtractInt(N:Integer; const S:LongString; const WordDelims:TCharSet;
                               var Value:LongInt):boolean;
  function  ExtractReal(N:Integer; const S:LongString; const WordDelims:TCharSet;
                                var Value:Double):boolean;
  function  BinB(x:Byte):LongString;
  function  BinW(x:Word):LongString;
  function  BinL(x:LongWord):LongString;
  function  OctB(x:Byte):LongString;
  function  OctW(x:Word):LongString;
  function  OctL(x:LongInt):LongString;
  function  HexB(x:Byte):LongString;
  function  HexW(x:Word):LongString;
  function  HexL(x:LongWord):LongString;
  function  Long2Str(L:LongInt):LongString;
  function  Real2Str(R:Double; Width:Integer=0;
                     Places:Integer=0):LongString;
  function  d2s(d:LongInt; Width:Integer=0):LongString;
  function  f2s(f:Double):LongString;
  procedure f2sFormat(Width:Integer=0; Decimals:Integer=14);
  procedure f2sFormatOld;
  function  FormatG(X:Double; w:integer; d:integer):LongString;
  function  atoi(a:PChar; var i:LongInt):boolean;
  function  atof(a:PChar; var f:double):boolean;
  function  Str2Int(const S:LongString; var I:Integer):Boolean;
  function  Str2Word(const S:LongString; var I:Word):Boolean;
  function  Str2Long(const S:LongString; var I:LongInt):Boolean;
  function  Str2Real(const S:LongString; var R:Double):Boolean;
  function  SmartStr2Real(const S:LongString; var R:Double):Boolean;
  function  StrBin2Long(const S:LongString; var L:LongInt):Boolean;
  function  StrOct2Long(const S:LongString; var L:LongInt):Boolean;
  function  StrHex2Long(const S:LongString; var L:LongInt):Boolean;
  function  ScanVar(svMode:word; Str:PChar;
                    const Format:LongString; var Data):PChar;
  function  IsEmptyStr(const S:LongString):boolean;
  function  LastDelimiter(const Delimiters:TCharSet;
                          const S:LongString):Integer;
  function  AddBackSlash(const DirName:LongString):LongString;
  function  DropBackSlash(const DirName:LongString):LongString;
  function  IsWildCard(const FileName:LongString):boolean;
  function  IsRelativePath(const S:LongString):boolean;
  function  HasExtension(const Name:LongString;
                         var DotPos:Integer):Boolean;
  function  DefaultExtension(const Name,Ext:LongString):LongString;
  function  DefaultPath(const Name,Path:LongString):LongString;
  function  ForceExtension(const Name,Ext:LongString):LongString;
  function  ForcePath(const NewPath,Name:LongString):LongString;
  function  ExtractFileNameExt(const FileName: LongString):LongString;
  function  ExtractFileName(const FileName: LongString):LongString;
  function  ExtractFileExt(const FileName: LongString): LongString;
  function  ExtractFilePath(const FileName: LongString): LongString;
  function  ExtractFileDrive(FileName: LongString): LongString;
  function  ExtractFilterExt(const Filter:LongString;
                             Index:Integer):LongString;
  function  FExpand(const Path:LongString):LongString;
  function  MakeRelativePath(const Path,Base:LongString):LongString;
  function  RusEng(const Rus:LongString='';
                   const Eng:LongString=''):LongString;
  function  SysErrorMessage(ErrorCode: Integer): LongString;
  function  GetFileAttr(const FileName:LongString):Integer;
  function  SetFileAttr(const FileName: LongString;
                        Attr: Integer):boolean;
  function  GetFileSize(const FileName:LongString):LongInt;
  function  GetFileDate(const FileName:LongString;
                        ConvertToMsec:Boolean=false):Int64;
  function  FileExists(const FilePattern:LongString;
                       Attribut:Integer=faAnyFile):Boolean;
  function  DirExists(const DirName: LongString): Boolean;
  function  FileErase(const FileName:LongString;
                      CheckExistance:boolean=true):boolean;
  function  FileCopy(const CopyFrom,CopyTo:LongString;
                     FailIfExists:boolean=false):boolean;
  function  FileRename(const OldFileName,NewFileName:LongString;
                       FailIfExists:boolean=false):boolean;
  function  GetCurrDir: LongString;
  function  SetCurrDir(const DirName: LongString): Boolean;
  function  MkDir(const DirName: LongString): Boolean;
  function  RmDir(const DirName: LongString): Boolean;
  function  ReadFileToBuffer(const FileName : LongString;
                                   Buffer   : Pointer;
                                   Count    : LongInt;
                                   Offset   : LongInt = 0 ):LongInt;
  function  WriteBufferToFile(const FileName     : LongString;
                                    Buffer       : Pointer;
                                    Count        : LongInt;
                                    Offset       : LongInt = 0;
                                    FLAGS        : DWORD = OPEN_ALWAYS
                                               ) : LongInt;
  procedure ForEachFile(const RootDir         : LongString;
                        const FilePatterns    : LongString;
                              FileAction      : TForEachFileAction;
                              MaxSubDirLevels : Integer = 0;
                              CustomData      : Pointer = nil;
                        const Delimiters      : TCharSet = [' ',';',',',ASCII_Tab,ASCII_CR,ASCII_LF]
                                             );
  function  ForEachTextLine(const FileName   : LongString;
                            Action     : TForEachTextLineAction;
                            CustomData : Pointer = nil ):Integer;
  procedure DebugOut(n:byte; const S:LongString);
  function  DebugOutGetFile(n:byte):LongString;
  function  DebugOutGetErrors(n:byte):Int64;
  procedure DebugOutOpenFile(n           : byte;
                       const FileName    : LongString;
                             FifoSize    : Integer = 1024*64;
                             FlushPeriod : Integer = 50;
                             CreateNew   : boolean = true;
                             Hello       : boolean = false);
  function  ExtractTextSection(const FileName            : LongString;
                               const SectionName         : LongString;
                                     ExtractSectionFlags : Word
                                                       ) : LongString;
  function  ExtractWordList(const FileName            : LongString;
                            const SectionName         : LongString;
                                  ExtractSectionFlags : word;
                                  WordIndex           : word;
                            const Delimiters          : TCharSet
                                                    ) : LongString;
  function  ExtractEnumWordList(const IniFile             : LongString;
                                const SectionName         : LongString;
                                const Prefix              : LongString;
                                      ExtractSectionFlags : word
                                                        ) : LongString;
  function ReadIniFileVariable(const FileName    : LongString;
                               const SectionName : LongString;
                               const Format      : LongString;
                                 var Data      ) : Boolean;
  function ReadIniFilePath(const FileName    : LongString;
                           const SectionName : LongString;
                           const Name        : LongString;
                           const StartupPath : LongString;
                             var Path        : LongString ) : Boolean;
  function  ProgName:LongString;
  function  HomeDir:LongString;
  function  StartupPath:LongString;
  function  StartAtDir:LongString;
  function  SysIniFile:LongString;
  procedure GoHome;
  function  WindowsDir:LongString;
  function  WindowsSystemDir:LongString;
  function  WindowsTempDir:LongString;
  function  TempDir:LongString;
  function  CreateTempFile(const Template:LongString='###.TMP'):LongString;

  function  ReadRegistryString(RootKey:HKEY;
                               const Key,Name:LongString):LongString;
  function  GetWindowsShellFolder(const Name:LongString):LongString;
  function  GetWindowsShellDesktop:LongString;
  function  GetWindowsShellPrograms:LongString;
  function  GetWindowsShellStartup:LongString;
  function  GetWindowsShellStartMenu:LongString;
  function  GetWindowsShellFavorites:LongString;
  function  GetWindowsShellFonts:LongString;
  function  GetWindowsShellHistory:LongString;
  function  GetWindowsShellPersonal:LongString;
  function  GetWindowsShellSendTo:LongString;
  procedure CreateFileLink(const ObjectPath, LinkPath, Description,
                           Params: LongString);
  function  Point2I(x,y:Integer):TPoint2I;
  function  Point2D(x,y:Double):TPoint2D;
  function  Point3D(x,y,z:Double):TPoint3D;
  function  Rect2I(const a,b:TPoint2I):TRect2I;
  function  Rect2D(const a,b:TPoint2D):TRect2D;
  function  IsEqualPoint2I(const p1,p2:TPoint2I):Boolean;
  function  IsEqualPoint2D(const p1,p2:TPoint2D; AbsEps:Double=0;
                           RelEps:Double=1E-14):Boolean;
  function  IsEqualRect2I(const r1,r2:TRect2I):Boolean;
  function  IsEqualRect2D(const r1,r2:TRect2D; AbsEps:Double=0;
                          RelEps:Double=1E-14):Boolean;
  function  IsEmptyRect2I(const r:TRect2I):Boolean;
  function  IsEmptyRect2D(const r:TRect2D; AbsEps:Double=0;
                          RelEps:Double=1E-14):Boolean;
  function  Rect2IContainsPoint(const r:TRect2I;
                                const p:TPoint2I):Boolean;
  function  Rect2DContainsPoint(const r:TRect2D;
                                const p:TPoint2D):Boolean;
  function  Rect2IIntersection(const R1,R2:TRect2I):TRect2I;
  function  Rect2DIntersection(const R1,R2:TRect2D):TRect2D;
  function  Rect2IUnion(const R1,R2:TRect2I):TRect2I;
  function  Rect2DUnion(const R1,R2:TRect2D):TRect2D;
  function  Rect2IValidate(const r:TRect2I):TRect2I;
  function  Rect2DValidate(const r:TRect2D):TRect2D;
  procedure Point2IMove(var p:TPoint2I; dx,dy:Integer);
  procedure Point2DMove(var p:TPoint2D; dx,dy:Double);
  procedure Rect2IMove(var r:TRect2I; dx,dy:Integer);
  procedure Rect2DMove(var r:TRect2D; dx,dy:Double);
  procedure Point2IMult(var p:TPoint2I; mx,my:Integer);
  procedure Point2DMult(var p:TPoint2D; mx,my:Double);
  procedure Rect2IMult(var r:TRect2I; mx,my:Integer);
  procedure Rect2DMult(var r:TRect2D; mx,my:Double);
  procedure Rect2IGrow(var r:TRect2I; dx,dy:Integer);
  procedure Rect2DGrow(var r:TRect2D; dx,dy:double);
  function  Rect2ISize(const r:TRect2I):TPoint2I;
  function  Rect2DSize(const r:TRect2D):TPoint2D;
  function  Rect2ICenter(const r:TRect2I):TPoint2I;
  function  Rect2DCenter(const r:TRect2D):TPoint2D;
  function  CreateScriptInterpreter:IScriptInterpreter;
  function  GetBitMask(BitNumber:Integer):Integer;
  function  Precision(AbsEps,RelEps,Value:Double):Double;
  function  MaxIntSingle:Single;
  function  MaxIntDouble:Double;
  function  MaxIntExtended:Extended;
  function  fscale(X:Extended; N:integer): Extended;
  procedure fxtract(X: Extended; var Mantissa: Extended;
                                 var Exponent: Integer);
  function  fbitcompress(Argument:Extended;
                         NumberOfSignificantBits:integer):Extended;
  function  frelcompress(Arument,ScaleFactor:Extended):Extended;
  function  fabscompress(Argument,ScaleFactor:Extended):Extended;
  function  Gamma(x:Extended):Extended;
  function  GammaLn(x:Extended):Extended;
  function  GammaInc(x,a:Extended):Extended;
  function  GammaDistr(x,a,b:Extended):Extended;
  function  GammaCumDistr(x,a,b:Extended):Extended;
  function  GammaCumDistrInv(p,a,b:Extended):Extended;
  function  Chi2Distr(x,v:Extended):Extended;
  function  Chi2CumDistr(x,v:Extended):Extended;
  function  Chi2CumDistrInv(p,v:Extended):Extended;
  function  Trapezium(x1,y1,x2,y2:double):double;
  function  BesselN(N:integer; X:Double;Modify:boolean=false):Double;
  function  GaussSolverWithPartialPivoting(N,M:Integer; var A,X,Y:array of Double;
                                           Eps:Double=1E-14):Integer;
  function  CreatePolynom(aPower  : Integer = 1;
                          aCenter : Double  = 0;
                          aScale  : Double  = 1):IPolynomWrap;
  function GenMin(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer):Integer;
  function GenMinFix(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer;
                  var Fix:array of Boolean):Integer;
  function GenMinScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer; fscale:Double;
                  var xLo,xHi:array of Double):Integer;
  function GenMinFixScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer; var Fix:array of Boolean; fscale:Double;
                  var xLo,xHi:array of Double):Integer;
  function  LSQFit(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; Report:TFitReport;
                   n:Integer; var x:array of Double; var f:Double; var g:array of Double;
                   Custom:Pointer; var Count:Integer; MaxCount:Integer; Method:Integer;
                   tolx,tolf,tolg,Inf,Size,Step,EvalTol:Double; var a:array of Double;
                   var Fix:array of Boolean; fscale:Double; var xLo:array of Double;
                   var xHi:array of Double):Integer;
  function  LSQFitSumOfSquares(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; n:Integer;
                               var x:array of Double; var f:Double; var g:array of Double;
                               Custom:Pointer):Integer;
  function  Rfa:IRfaDataBaseWrap;
  function  Sort(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                 KeyOf:TKeyOfFunction; SwapData:TSwapProcedure;
                 Index:PIntegerArray; Method:Integer=0):Boolean;
  function  QuickFind(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                      KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                      Index:PIntegerArray=nil):Integer;
  function  QuickSearch(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                        KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                        var KeyIndex:Integer; Index:PIntegerArray=nil;
                        Duplicates:Boolean=false):Boolean;
  function  FindDoubleIndex(N:Integer; const x:array of Double; at:Double;
                            Index:PIntegerArray=nil):Integer;
  function  LinearInterpolation(const x,y:array of Double; N:Integer;
                                t:Double):Double;
  function  EvaluatePolynomAt(const P:array of Double; N:Integer;
                              t:Double):Double;
  function  FindLeastSquaresPolynom(var P:array of Double; N:Integer; X0,Scale:Double;
                        const x,y:array of Double; M:Integer):Boolean;
  function  EvaluateSplineAt(N:Integer; const x,z,k:array of Double;
                             t:Double; NDeriv:Integer=0):Double;
  function  FindSmoothingSpline(N:Integer; const x,y,p:array of Double;
                                var z,k:array of Double;
                                nL:Integer=2; pL:Double=0;
                                nR:Integer=2; pR:Double=0):Boolean;
  function  WindowedKernelSmoothing(const x,y:array of Double; N:Integer;
                                    t,Window:Double; Power:Integer; Kernel:TSmoothKernel;
                                    Custom:Pointer=nil):Double;
  function  WindowedIntegrationSmoothing(const x,y:array of Double; N:Integer;
                                         t,Window:Double; Kernel:TSmoothKernel;
                                         Custom:Pointer=nil):Double;
  procedure Solve3DiagonalSetByProgonka(N:Integer;
                                        var a,b,c,x,d:array of Double);
  procedure Solve3DiagonalSetByGauss(N:Integer;
                                     var a,b,c,x,d:array of Double);
  procedure Solve5DiagonalSetByGauss(N:Integer;
                                     var a,b,c,e,g,x,d:array of Double);
  function  CreateLongIntVector(aLength:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):ILongIntVectorWrap;
  function  CreateDoubleVector(aLength:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):IDoubleVectorWrap;
  function  CreateLongIntMatrix(aRows,aColumns:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):ILongIntMatrixWrap;
  function  CreateDoubleMatrix(aRows,aColumns:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):IDoubleMatrixWrap;
  function  LinearProgrammingBySimplex(Problem:IDoubleVectorWrap; MatrCoef:IDoubleMatrixWrap;
                            Signums:ILongIntVectorWrap; RightSide:IDoubleVectorWrap;
                            var ErrorCode:Integer; PrintMode:Word; var OutPutFile:Text;
                            Zero:Double=1E-12):IDoubleMatrixWrap;
  function  CreateSingularDecomposition(A:IDoubleMatrixWrap):ISVDWrap;
  function  CreateFifo(aSize:LongInt=1024;
                       const aName:LongString=''):IFifoWrap;
  function  MkSecNow:Extended;
  function  LockedAdd(var Target: Integer; Value: Integer): Integer;
  function  LockedCompareExchange(var Target: Integer;
                                  Exch, Comp: Integer): Integer;
  function  LockedDec(var Target: Integer): Integer;
  function  LockedExchange(var Target: Integer;
                               Value: Integer): Integer;
  function  LockedExchangeAdd(var Target: Integer;
                                   Value: Integer): Integer;
  function  LockedExchangeDec(var Target: Integer): Integer;
  function  LockedExchangeInc(var Target: Integer): Integer;
  function  LockedExchangeSub(var Target: Integer;
                                   Value: Integer): Integer;
  function  LockedInc(var Target: Integer): Integer;
  function  LockedSub(var Target: Integer; Value: Integer): Integer;
  function  RemoveBrackets(const s:LongString;
                           const Brackets:LongString='[]'):LongString;
  function  Mime_Encode(const S:LongString):LongString;
  function  Mime_Decode(const S:LongString):LongString;
  function  IsSameText(const S1,S2:LongString):Boolean;
  function  IsUnix:Boolean;
  function  IsLinux:Boolean;
  function  IsWindows:Boolean;
  function  IsWin32:Boolean;
  function  IsWin64:Boolean;
  function  IsCpu16:Boolean;
  function  IsCpu32:Boolean;
  function  IsCpu64:Boolean;
  function  IsCpuBE:Boolean;
  function  IsCpuLE:Boolean;
  function  IsSysUtf8:Boolean;
  function  IsFileNameCaseSensitive:Boolean;
  function  OsName(Mode:Integer=1):LongString;
  function  GetFpcVersion:LongString;
  function  GetFpcTargetOS:LongString;
  function  GetFpcTargetCPU:LongString;
  function  GetFpcTargetPlatform:LongString;
  function  IsMainThread:Boolean;
  function  GetCurrentProcessId:TPid;
  function  GetAllocMemCount:SizeInt;
  function  GetAllocMemSize:SizeInt;
  procedure ResetFpu;
  function  StrToWide(const s:LongString):WideString;
  function  WideToStr(const s:WideString):LongString;
  function  ConvertCP(const s:LongString; cpFrom,cpTo,cpSet:Word; silent:Boolean=false):LongString;
  function  AdaptFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function  AdaptExeFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function  AdaptDllFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function  AdaptLnkFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function  GetTickCount:Cardinal;
  function  GetTickCount64:QWord;
  function  ExtractBaseName(const FileName:LongString):LongString;
  function  ExtractDllBaseName(const FileName:LongString):LongString;
  function  ExtractFileDir(const FileName:LongString):LongString;
  function  IsLexeme(const arg:LongString; typ:Integer):Boolean;
  function  ExtractNameValuePair(const arg:LongString; out Name,Value:LongString;
                                 const Sign:Char='='; Mode:Integer=3):Integer;
  function  PosI(const Sub:LongString; const Str:LongString):SizeInt;
  function  PosEx(const Sub,Str:LongString; StartPos:SizeInt):SizeInt;
  function  LastPos(const Sub,Str:LongString):SizeInt;
  function  CountPos(const Sub,Str:LongString):SizeInt;
  function  NthPos(const Sub,Str:LongString; n:SizeInt):SizeInt;
  function  PosEol(const Buf:LongString; Start:SizeInt=1; SkipLines:SizeInt=0):SizeInt;
  function  TailStr(const S:LongString; Pos:SizeInt):LongString;
  function  StrAheadOf(const S:LongString; Delim:Char):LongString;
  function  StrAfterOf(const S:LongString; Delim:Char):LongString;
  function  CenterPad(const S:LongString; Width:SizeInt; Ch:Char=' '):LongString;
  function  TrimDef(const S,Def:LongString):LongString;
  function  SortTextLines(const aTextLines:LongString;
                          Comparator:TStringListSortCompare=nil):LongString;
  function  EnsureEndingEol(const S:LongString):LongString;
  function  EnsureHeapString(const S:LongString):LongString;
  function  CookieScan(const Buff,Name:LongString; Mode:Integer=csm_Default):LongString;
  function  SkipWords(N:Integer; const S:LongString;
                                 const WordDelims:TCharSet):LongString;
  function  PhraseCount(const S:LongString; const Delims:TCharSet;
                        Quotes:LongString=QuoteMark+Apostrophe):Integer;
  function  ExtractPhrase(N:Integer; const S:LongString; const Delims:TCharSet;
                          Quotes:LongString=QuoteMark+Apostrophe):LongString;
  function  SkipPhrases(N:Integer; const S:LongString; const Delims:TCharSet;
                        Quotes:LongString=QuoteMark+Apostrophe):LongString;
  function  PhraseListToTextLines(const S:LongString; const Delims:TCharSet;
                                  Quotes:LongString=QuoteMark+Apostrophe):LongString;
  function  ScanVarRecord(svMode:Cardinal; Str:PChar; const Format:LongString; var Data):PChar;
  function  ScanVarBoolean(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Boolean):PChar;
  function  ScanVarWord(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Word):PChar;
  function  ScanVarInteger(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Integer):PChar;
  function  ScanVarDouble(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Double):PChar;
  function  ScanVarAlpha(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
  function  ScanVarString(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
  function  AddPathDelim(const DirName:LongString; Delim:Char=PathDelim):LongString;
  function  DropPathDelim(const DirName:LongString):LongString;
  function  ValidatePathDelim(const FileName:LongString; Delim:Char=PathDelim):LongString;
  function  StrFetch(const s:LongString; i:SizeInt):AnsiChar;
  function  AnsiDeQuotedStr(const s:LongString; q:Char):LongString;
  function  AnsiSkipQuotedStr(const s:LongString; q:Char):LongString;
  function  ExtractFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  SkipFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  AnsiQuotedIfNeed(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  QArg(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  ExtractFirstParamUrl(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  Dequote_or_URL_Decode(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function  IsOption(const arg:LongString; const shortopt:LongString=''; const longopt:LongString='';
                    Mode:Integer=om_Default; Delim:Char='='):Boolean;
  function  GetOptionValue(const arg:LongString; Delim:Char='='):LongString;
  function  HasOptionValue(const arg:LongString; Delim:Char='='):Boolean;
  function  StringToSetOfChars(const S:LongString):TCharSet;
  function  SetOfCharsToString(const S:TCharSet):LongString;
  function  HasChars(const S:LongString; const C:TCharSet):Boolean;
  function  CountChars(const S:LongString; const C:TCharSet):SizeInt;
  function  GetRealFilePathName(const FileName:LongString; BuffSize:Integer=MAX_PATH; Mode:Integer=0):LongString;
  function  IsSameChar(C1,C2:Char):Boolean;
  function  IsSameStr(S1,S2:LongString):Boolean;
  function  IsSameFileName(const S1,S2:LongString; Mode:Integer=ua_FileDef):Boolean;
  function  IsSectionName(const aName:LongString):Boolean;
  function  HTTP_StatusMessage(StatusCode:Integer):LongString;
  function  URL_Packed(const S:LongString; Mode:Integer=um_Safe; const AllowChars:TCharSet=UrlAllowChars):LongString;
  function  URL_Encode(const S:LongString; Mode:Integer=um_Safe+um_Strict; const AllowChars:TCharSet=[]):LongString;
  function  URL_Decode(const S:LongString; Mode:Integer=um_Safe):LongString;
  function  StrIScan(Str:PChar; Chr:Char):PChar;
  function  StrRIScan(Str:PChar; Chr:Char):PChar;
  function  StrIPos(Str1,Str2:PChar):PChar;
  function  StrPLCopy(Dest:PChar; const Source:LongString; MaxLen:SizeInt):PChar;
  function  ValidateEOL(const aData:LongString; aTail:Integer=0; aEOL:LongString=EOL):LongString;
  function  RunCommand(cmdline:LongString; out outputstring:LongString):Boolean;
  function  RunCommandInDir(curdir,cmdline:LongString; out outputstring:LongString):Boolean;
  function  cpu_count:Integer;
  function  cpu_mhz(id:Integer=0):Double;
  function  ReadProcCpuInfo(CpuMask:Int64; What:LongString):LongString;
  function  IsATTY(Handle:LongInt):Boolean;
  function  GetProcessAffinityMask(pid:TPid=0):QWord;
  function  GetThreadAffinityMask(tid:SizeInt=0):QWord;
  function  SetThreadAffinityMask(tid:SizeInt; Mask:QWord):QWord;
  function  GetListOfProcesses(aPid,aPPid:SizeInt; const aName:LongString; Detail:Boolean=false):LongString;
  function  GetProcessName(pid:SizeInt=0):LongString;
  function  GetParentProcessID:SizeInt;
  function  GetParentProcessName:LongString;
  function  GetProcessTimesAsFileTime(out s,u:Int64):Boolean;
  function  GetThreadTimesAsFileTime(out s,u:Int64):Boolean;
  function  ProgBaseName:LongString;
  function  UserHomeDir(const SubDir:LongString=''):LongString;
  function  SystemRootDir:LongString;
  function  GlobalTempDir(const SubDir:LongString=''):LongString;
  function  LocalTempDir(const SubDir:LongString=''):LongString;
  function  GetCrwKitExe:LongString;
  function  GetShell:LongString;
  function  GetCommandLine:LongString;
  function  UserName:LongString;
  function  ComputerName:LongString;
  function  HostName(Method:Integer=0):LongString;
  function  UserDomain(const aUser:LongString=''; aDef:LongString=''):LongString;
  function  GetIPAddress(const aHostName:LongString=''):LongString;
  function  GetIfMacAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetIfIpAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetIPAddresses(const aHostName:LongString=''; Delim:LongString=','):LongString;
  function  GetMacAddresses(const Machine:LongString=''; Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetLocalUserList(Delim:LongString=EOL):LongString;
  function  GetEnv(const Name:LongString):LongString;
  function  SetEnv(const Name,Value:LongString):Boolean;
  function  ExpEnv(const Str:LongString):LongString;
  function  MaybeEnvStr(const arg:LongString):Boolean;
  function  MaybeTildeStr(const arg:LongString):Boolean;
  function  FileIsExecutable(const aFileName:LongString):Boolean;
  function  FileIsSymlink(const aFileName:LongString):Boolean;
  function  FileIsHardLink(const aFileName:LongString):Boolean;
  function  FileIsReadable(const aFileName:LongString):Boolean;
  function  FileIsWritable(const aFileName:LongString):Boolean;
  function  StringFromFile(const FileName:LongString; MaxSize:SizeInt):LongString;
  function  GetRootKeyByName(const Key:LongString):HKEY;
  function  ReadRegistryMultiStrings(RootKey:HKEY; const Key,Name:LongString; Delim:Char=ASCII_CR):LongString;
  function  ReadShellLinkAsText(const FileName:LongString):LongString;
  function  AnyFileMask:LongString;
  function  FindAllFilesAsText(const SearchPath:LongString;
                               const SearchMask:LongString='';
                                     SearchSubDirs:Boolean=True;
                                     DirAttr:Word=faDirectory;
                                     MaskSeparator:Char=';';
                                     PathSeparator:Char=';'
                                     ):LongString;
  function  ReadIniFileRecord(const FileName    : LongString;
                              const SectionName : LongString;
                              const Format      : LongString;
                                var Data;
                                    efMode      : Integer = efConfig;
                                    svMode      : Integer = svConfig
                                              ) : Boolean;
  function  ReadIniFileBoolean(const FileName    : LongString;
                               const SectionName : LongString;
                               const Key         : LongString;
                                 var Data        : Boolean;
                                     efMode      : Integer = efConfig;
                                     svMode      : Integer = svConfig
                                               ) : Boolean;
  function  ReadIniFileDouble(const FileName    : LongString;
                              const SectionName : LongString;
                              const Key         : LongString;
                                var Data        : Double;
                                    efMode      : Integer = efConfig;
                                    svMode      : Integer = svConfig
                                               ) : Boolean;
  function  ReadIniFileWord(const FileName    : LongString;
                            const SectionName : LongString;
                            const Key         : LongString;
                              var Data        : Word;
                                  efMode      : Integer = efConfig;
                                  svMode      : Integer = svConfig
                                            ) : Boolean;
  function  ReadIniFileInteger(const FileName    : LongString;
                               const SectionName : LongString;
                               const Key         : LongString;
                                 var Data        : Integer;
                                     efMode      : Integer = efConfig;
                                     svMode      : Integer = svConfig
                                               ) : Boolean;
  function  ReadIniFileAlpha(const FileName    : LongString;
                             const SectionName : LongString;
                             const Key         : LongString;
                               var Data        : LongString;
                                   efMode      : Integer = efConfig;
                                   svMode      : Integer = svConfig
                                             ) : Boolean;
  function  ReadIniFileString(const FileName    : LongString;
                              const SectionName : LongString;
                              const Key         : LongString;
                                var Data        : LongString;
                                    efMode      : Integer = efConfig;
                                    svMode      : Integer = svConfig
                                              ) : Boolean;
  function  GetFileType(hFile:THandle):LongWord;
  function  GetFileSize64(const FileName:LongString):Int64;
  function  FileHasNonBlockFlag(fd:THandle):Boolean;
  function  FileSetNonBlockFlag(fd:THandle; State:Boolean=true):Boolean;
  function  FileSetCloseOnExec(fd:THandle; State:Boolean=true):Boolean;
  function  ePendingFileOperation(ErrorCode:Integer):Boolean;
  function  eBrokenPipe(ErrorCode:Integer):Boolean;
  function  eOperationAborted(ErrorCode:Integer):Boolean;
  function  SmartFileRef(const FileRef   : LongString;
                         const DefExt    : LongString = '';
                         const BaseFile  : LongString = '';
                         const UserHome  : LongString = sfrDefUserHome;
                         const ProgHome  : LongString = sfrDefProgHome;
                         const Brackets  : LongString = sfrDefBrackets;
                               handleUrl : Boolean    = true
                                       ) : LongString;
  function  SmartFileRel(const FileRef   : LongString;
                         const BaseFile  : LongString;
                         const UserHome  : LongString = sfrDefUserHome;
                         const ProgHome  : LongString = sfrDefProgHome;
                         const Brackets  : LongString = sfrDefBrackets;
                               handleUrl : Boolean    = true
                                       ) : LongString;
  function  file_which(name:LongString):LongString;
  function  GetFileVersionInfoAsText(FileName:LongString):LongString;
  function  StdOutPrint(const Msg:LongString):LongInt;
  function  StdErrPrint(const Msg:LongString):LongInt;
  function  StdOutPrintLn(const Msg:LongString):LongInt;
  function  StdErrPrintLn(const Msg:LongString):LongInt;
  function  isFileClosed(var f:File):Boolean;
  function  isTextClosed(var f:Text):Boolean;
  procedure SmartFileClose(var f:File);
  procedure SmartTextClose(var f:Text);
  function  FileTypeToString(fType:LongWord; Prefix:LongString='FILE_TYPE_'):LongString;
  function  IoResultToString(Code:Integer):LongString;
  function  StdDateTimeStr(ms:Double=0):LongString;
  function  StdDateTimePrompt(ms:Double=0; Prompt:PChar=nil):LongString;
  function  WriteRegistryString(RootKey:HKEY; const Key,Name,Data:LongString):LongString;
  procedure DebugOutText(n:Byte; const S:LongString);
  function  ValidatePathList(const PathList:LongString; Delim:Char=PathSep):LongString;
  function  ValidateEnvPathList(const EnvPathList:LongString; Delim:Char=PathSep):Integer;
  function  SmartFileSearch(const FileName   : LongString;
                            const PathVarStr : LongString='PATH';
                            const PathExtVar : LongString='PATHEXT';
                            const PathExtStr : LongString=DefaultPathExt
                                           ) : LongString;
  function  ReadSymLink(const FileName:LongString; MaxLevel:Integer=10):LongString;
  function  GetSystemParamStr(const Category,ParamName:LongString):LongString;
  function  base64_encode(const S:LongString):LongString;
  function  base64_decode(const S:LongString):LongString;
  function  base32_encode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
  function  base32_decode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
  function  backslash_encode(const data:LongString; const esclist:TCharSet=[]; const hexlist:TCharSet=[]):LongString;
  function  backslash_decode(const data:LongString):LongString;
  function  StringOfCharRange(a,b:Char):LongString;
  function  task_init(const cmd_line:LongString):Integer;
  function  task_free(tid:Integer):Boolean;
  function  task_ref(tid:Integer):Integer;
  function  task_pid(tid:Integer):Integer;
  function  task_run(tid:Integer):Boolean;
  function  task_wait(tid,timeout:Integer):Boolean;
  function  task_send(tid:Integer; const data:LongString):Integer;
  function  task_recv(tid,maxlen:Integer):LongString;
  function  task_txcount(tid:Integer):Integer;
  function  task_rxcount(tid:Integer):Integer;
  function  task_txspace(tid:Integer):Integer;
  function  task_rxspace(tid:Integer):Integer;
  function  task_result(tid:Integer):Integer;
  function  task_kill(tid,how,exit_code,timeout:Integer):Boolean;
  function  task_ctrl(tid:Integer; const arg:LongString):LongString;
  function  SmartExecute(const CmdLine:LongString;
                               Display:Integer=SW_SHOWNORMAL;
                               ShellCm:LongString=''):Boolean;
  function  PidAffinity(pid:TPid; mask:Int64=0):Int64;
  function  KillProcess(aPid:TPid; aExitCode:Integer; sig:Integer=SIGTERM):Boolean;
  function  HashList_Init(aCaseSensitive:Boolean):Integer;
  function  HashList_Free(aRef:Integer):Boolean;
  function  HashList_Ref(aRef:Integer):Integer;
  function  HashList_Count(aRef:Integer):Integer;
  function  HashList_GetKey(aRef:Integer; aIndex:Integer):LongString;
  function  HashList_Delete(aRef:Integer; aKey:LongString):Boolean;
  function  HashList_IndexOf(aRef:Integer; aKey:LongString):Integer;
  function  HashList_GetData(aRef:Integer; aKey:LongString):Double;
  function  HashList_SetData(aRef:Integer; aKey:LongString; aData:Double):Boolean;
  function  HashList_GetLink(aRef:Integer; aKey:LongString):Integer;
  function  HashList_SetLink(aRef:Integer; aKey:LongString; aLink:Integer):Boolean;
  function  HashList_GetPara(aRef:Integer; aKey:LongString):LongString;
  function  HashList_SetPara(aRef:Integer; aKey:LongString; aParam:LongString):Boolean;
  function  regexp_init(engine:Integer; pattern:LongString; options:LongString=''):Integer;
  function  regexp_free(rex:Integer):Boolean;
  function  regexp_ref(rex:Integer):Integer;
  function  regexp_ctrl(rex:Integer; arg:LongString):LongString;
  function  regexp_test(rex:Integer; arg:LongString):Boolean;
  function  regexp_exec(rex:Integer; arg:LongString):Integer;
  function  regexp_replace(rex:Integer; arg,rep:LongString):LongString;
  function  regexp_matchnum(rex:Integer; i:Integer):Integer;
  function  regexp_matchpos(rex:Integer; i,j:Integer):Integer;
  function  regexp_matchlen(rex:Integer; i,j:Integer):Integer;
  function  regexp_matchstr(rex:Integer; i,j:Integer):LongString;
  function  regexp_escape(arg:LongString):LongString;
  function  pipe_init(const cmd_line:LongString):Integer;
  function  pipe_free(pid:Integer):Boolean;
  function  pipe_ref(pid:Integer):Integer;
  function  pipe_pid(pid:Integer):Integer;
  function  pipe_run(pid:Integer):Boolean;
  function  pipe_wait(pid,timeout:Integer):Boolean;
  function  pipe_send(pid:Integer; const data:LongString; awake:Boolean=false):Integer;
  function  pipe_recv(pid,maxlen:Integer):LongString;
  function  pipe_txcount(pid:Integer):Integer;
  function  pipe_rxcount(pid:Integer):Integer;
  function  pipe_txspace(pid:Integer):Integer;
  function  pipe_rxspace(pid:Integer):Integer;
  function  pipe_result(pid:Integer):Integer;
  function  pipe_kill(pid,how,exit_code,timeout:Integer):Boolean;
  function  pipe_ctrl(pid:Integer; const param:LongString):LongString;
  function  pipe_txclear(pid:Integer):Boolean;
  function  pipe_rxclear(pid:Integer):Boolean;
  function  pipe_count(pid:Integer):Integer;
  function  pipe_stream(pid,index:Integer):Integer;
  function  pipe_connected(pid:Integer):Integer;
  function  CSIDL_FolderByName(const Name:LongString; const sub:LongString;
                  Charset:Integer=-1; const Delim:LongString=''):LongString;
  function  CSIDL_ListAllAsText(const Prefix:LongString='CSIDL_';
                  Charset:Integer=-1; SkipEmpty:Boolean=true):LongString;
  function  GetSystemAssoc(const Ext:LongString):LongString;
  function  GetSystemFType(const FType:LongString):LongString;
  function  GetSystemAssocExe(const Ext:LongString):LongString;
  function  GetSystemFTypeExe(const FType:LongString):LongString;
  function  HasSystemAssocExe(const Ext:LongString):Boolean;
  function  HasSystemFTypeExe(const FType:LongString):Boolean;
  function  GetExeByExtension(const Extension:LongString):LongString;
  function  GetAppPath(args:LongString; AllowRun:Boolean=false; AllowWait:Boolean=false;
                      pExitCode:PInteger=nil; EnvPath:LongString='';
                      EnvPathExt:LongString=''):LongString;
  function  GetExeByFile(const FileName:LongString):LongString;
  function  WScriptShellRun(CmdLine:LongString; Show:Integer=SW_SHOWNORMAL; Wait:Boolean=false):Integer;
  function  ShellExecuteOpen(FileName:LongString; Cmd:LongString=''):Boolean;
  function  wmctrl_query(const args:LongString):LongString;
  function  shm_init(name:LongString; size,mode:Integer):Integer;
  function  shm_ref(ref:Integer):Integer;
  function  shm_free(ref:Integer):Boolean;
  function  shm_delink(name:LongString):Boolean;
  function  shm_iop(ref:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
  function  shm_rop(ref:Integer; offset:Integer; op:Char; data:Double):Double;
  function  shm_fop(ref:Integer; offset:Integer; op:Char; data:Single):Single;
  function  shm_sop(ref:Integer; offset:Integer; op:Char; data:LongString):LongString;
  function  shm_ctrl(ref:Integer; arg:LongString):LongString;
  function  shm_ioresult(code:Integer=0):Integer;
  function  fsm_new:Integer;
  function  fsm_free(ref:Integer):Boolean;
  function  fsm_ref(ref:Integer):Integer;
  function  fsm_root(ref:Integer):Integer;
  function  fsm_type(ref:Integer):Integer;
  function  fsm_parent(ref:Integer):Integer;
  function  fsm_name(ref:Integer):LongString;
  function  fsm_path(ref:Integer):LongString;
  function  fsm_ctrl(ref:Integer; arg:LongString):LongString;
  function  fsm_count(ref,typ:Integer):Integer;
  function  fsm_items(ref,typ,i:Integer):Integer;
  function  fsm_get_iparam(ref:Integer):Integer;
  function  fsm_set_iparam(ref:Integer; data:Integer):Boolean;
  function  fsm_get_fparam(ref:Integer):Double;
  function  fsm_set_fparam(ref:Integer; data:Double):Boolean;
  function  fsm_get_sparam(ref:Integer):LongString;
  function  fsm_set_sparam(ref:Integer; data:LongString):Boolean;
  function  fsm_add(ref:Integer; typ:integer; key:LongString):Integer;
  function  fsm_find(ref:Integer; typ:integer; key:LongString):Integer;
  function  fsm_get_state(ref:Integer):Integer;
  function  fsm_set_state(ref:Integer; state:Integer):Integer;
  function  fsm_put_state(ref:Integer; state:LongString):Integer;
  function  fsm_link(ref:Integer; arg:LongString):Integer;
  function  fsm_modified(ref:Integer; delta:Integer):Integer;
  function  fsm_name_rule(typ:Integer):Integer;
  function  fsm_valid_childtype(typ,childtyp:Integer):Boolean;
  function  Hex_Encode(const Str:LongString):LongString;
  function  Hex_Decode(const Str:LongString):LongString;
  function  ColorToString(ColorCode:TColor; Prefix:LongString=''):LongString;
  function  StringToColor(ColorName:LongString; DefaultColor:TColor=clNone):TColor;
  function  StringToSigCode(str:LongString; def:Integer=0):Integer;
  function  SigCodeToString(sig:Integer):LongString;
  function  text_new:Integer;
  function  text_ref(txt:Integer):Integer;
  function  text_free(txt:Integer):Boolean;
  function  text_getln(txt:Integer; i:Integer):LongString;
  function  text_putln(txt:Integer; i:Integer; s:LongString):Boolean;
  function  text_insln(txt:Integer; i:Integer; s:LongString):Boolean;
  function  text_addln(txt:Integer; s:LongString):Boolean;
  function  text_delln(txt:Integer; i:Integer):Boolean;
  function  text_numln(txt:Integer):Integer;
  function  text_tostring(txt:Integer):LongString;
  function  text_fromstring(txt:Integer; s:LongString):Integer;
  function  db_create(arg:LongString):Boolean;
  function  db_connection(eid:Integer; arg:LongString):Integer;
  function  db_recordset(dbo:Integer; arg:LongString):Integer;
  function  db_command(dbo:Integer; arg:LongString):Integer;
  function  db_free(dbo:Integer):Boolean;
  function  db_ref(dbo:Integer):Integer;
  function  db_root(dbo:Integer):Integer;
  function  db_type(dbo:Integer):Integer;
  function  db_parent(dbo:Integer):Integer;
  function  db_state(dbo:Integer):Integer;
  function  db_close(dbo:Integer):Boolean;
  function  db_open(dbo:Integer; opt:Integer):Boolean;
  function  db_ctrl(dbo:Integer; arg:LongString):LongString;
  function  db_bugscount(dbo:Integer):Integer;
  function  db_bugsclear(dbo:Integer):Integer;
  function  db_errors(dbo:Integer):LongString;
  function  db_errorscount(dbo:Integer):Integer;
  function  db_errorsclear(dbo:Integer):Integer;
  function  db_execute(dbo:Integer; arg:LongString; opt:Integer):Integer;
  function  db_cancel(dbo:Integer):Boolean;
  function  db_update(dbr:Integer):Boolean;
  function  db_cancelupdate(dbr:Integer):Boolean;
  function  db_begintrans(dbc:Integer):Integer;
  function  db_committrans(dbc:Integer):Boolean;
  function  db_rollbacktrans(dbc:Integer):Boolean;
  function  db_bof(dbr:Integer):Boolean;
  function  db_eof(dbr:Integer):Boolean;
  function  db_movefirst(dbr:Integer):Boolean;
  function  db_movelast(dbr:Integer):Boolean;
  function  db_movenext(dbr:Integer):Boolean;
  function  db_moveprevious(dbr:Integer):Boolean;
  function  db_fieldscount(dbr:Integer):Integer;
  function  db_fieldsnames(dbr:Integer; i:Integer):LongString;
  function  db_fieldstypes(dbr:Integer; id:LongString):Integer;
  function  db_fieldsasint(dbr:Integer; id:LongString; op:Char; arg:Integer):Integer;
  function  db_fieldsasfloat(dbr:Integer; id:LongString; op:Char; arg:Double):Double;
  function  db_fieldsasstring(dbr:Integer; id:LongString; op:Char; arg:LongString):LongString;
  function  db_addnew(dbr:Integer; arg:LongString):Boolean;
  function  db_delete(dbr:Integer; aff:Integer):Boolean;
  function  db_requery(dbr:Integer; opt:Integer):Boolean;
  function  db_resync(dbr:Integer; aff,res:Integer):Boolean;
  function  db_supports(dbr:Integer; opt:Integer):Boolean;
  function  db_save(dbr:Integer; dst:LongString; fmt:Integer):Boolean;
  function  HasFlags(Mode,Flags:Int64):Boolean;
  function  utf8_valid(const S:LongString):Boolean;
  function  utf8_errpos(const S:LongString):SizeInt;
  function  utf8_length(const S:LongString):SizeInt;
  function  utf8_encode_wide(const S:UnicodeString):LongString;
  function  utf8_decode_wide(const S:LongString):UnicodeString;
  function  utf8_encode_ansi(const S:LongString):LongString;
  function  utf8_decode_ansi(const S:LongString):LongString;
  function  utf8_copy(const S:LongString; i,n:SizeInt):LongString;
  function  utf8_chrpos(const S:LongString; i:SizeInt):SizeInt;
  function  utf8_ord(const S:LongString; i:SizeInt=1):SizeInt;
  function  utf8_chr(c:SizeInt):LongString;
  function  utf8_uppercase(const S:LongString):LongString;
  function  utf8_lowercase(const S:LongString):LongString;
  function  utf8_fixbroken(const S:LongString):LongString;
  function  utf8_sametext(const S1,S2:LongString):Boolean;
  function  utf8_fix_cp(const S:LongString; cp:Word=0):LongString;
  function  IntToStrBase(Value:LongInt; Base:Integer=10; Width:Integer=0):LongString;
  function  StrToIntBase(S:LongString; Base:Integer=10; Def:Integer=0):LongInt;
  function  htonl(l:Integer):Integer;
  function  ntohl(l:Integer):Integer;
  function  htons(s:Integer):Integer;
  function  ntohs(s:Integer):Integer;
  function  strtimefmt(const fmt:LongString; ms:Double):LongString;
  function  IsNonEmptyStr(const S:LongString):Boolean;
 private
  myOwner : TCrwApiServer;
  myRfa   : TRfaDataBaseWrapServer;
  myStore : TObjectStorage;
  procedure SaveRef(aRef:PtrInt);
  procedure DropRef(aRef:PtrInt);
  procedure FreeStoreRefs;
 private
  myGlops_Flags : Integer;
 public
  constructor Create(aOwner:TCrwApiServer);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация графического интерфейса.
 //
 TGuiApiServer = class(TPluginApi,IGuiApi)
 public
  procedure ServerActionsBeforeExecution; override;
  procedure ServerActionsAfterExecution;  override;
  function  CrwApi:ICrwApi;               override;
 public // IGuiApi
  procedure Echo(const Msg:LongString);
  function  Warning(const Msg:LongString):word;
  function  Information(const Msg:LongString):word;
  function  YesNo(const Msg:LongString):word;
  function  YesNoCancel(const Msg:LongString):word;
  function  Error(const Msg:LongString):word;
  function  Trouble(Check:boolean; const Msg:LongString):boolean;
  function  NoProblem(Check:boolean; const Msg:LongString):Boolean;
  function  ListBoxMenu(const Caption,Title:LongString;
                        const ItemList:LongString):Integer;
  function  ListBoxMultiSelection(const Caption,Title:LongString;
                                  const ItemList : LongString;
                                        Extended : Boolean = false;
                                        GetIndex : Boolean = false
                                               ) : LongString;
  function  TextEditDialog(const Caption,Title:LongString;
                           const TextToEdit:LongString):LongString;
  function  UsesBlaster:Boolean;
  procedure Voice(const PlayList:LongString);
  procedure StopBlaster;
  procedure AddSoundLibrary(const Path:LongString;
                                  MaxSubDirs:LongInt=0);
  procedure ApplicationProcessMessages;
  procedure UpdateSystemConsole;
  function  RGB(r,g,b:Integer):TColor;
  function  CgaColorNameEng(CgaColor:Integer):LongString;
  function  CgaColorNameRus(CgaColor:Integer):LongString;
  function  CgaColorByName(const S:LongString):Integer;
  function  CgaToRGBColor(CgaColor:Integer):TColor;
  function  ApplicationHandle:THandle;
  function  SendToMainConsole(const Msg:LongString):Integer;
 private
  myOwner : TCrwApiServer;
 public
  constructor Create(aOwner:TCrwApiServer);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация интерфейса системы сбора данных.
 //
 TDaqApiServer = class(TPluginApi,IDaqApi)
 public
  procedure ServerActionsBeforeExecution;                                override;
  procedure ServerActionsAfterExecution;                                 override;
  function  CrwApi:ICrwApi;                                              override;
 public // IDaqApi
  function  DaqCommand:Integer;
  function  DaqDataSheet(aSize:Integer):Pointer;
  function  DaqIsLoaded:Boolean;
  function  DaqIsStarted:Boolean;
  function  DaqConfigFile:LongString;
  function  DaqDataPath:LongString;
  function  DaqTempPath:LongString;
  function  FindTag(const Name:LongString):Integer;
  function  CreateTag(const Name:LongString; Typ:Integer):Integer;
  function  FreeTag(tag:Integer):Boolean;
  function  TypeTag(tag:Integer):Integer;
  function  NameTag(tag:Integer):LongString;
  function  iGetTag(tag:Integer):Integer;
  function  rGetTag(tag:Integer):Double;
  function  sGetTag(tag:Integer):LongString;
  function  iSetTag(tag:Integer; i:Integer):Boolean;
  function  rSetTag(tag:Integer; r:Double):Boolean;
  function  sSetTag(tag:Integer; const s:LongString):Boolean;
  function  ThermoCoupleCount:Integer;
  function  FindThermoCouple(const CoupleName:LongString):Integer;
  function  ThermoCoupleName(CoupleID:Integer):LongString;
  function  ThermoCoupleRange(CoupleID:Integer):TRect2D;
  function  ThermoCoupleValue(CoupleID:Integer; Argument:Double;
                              Inverted:Boolean):Double;
  function  RelationshipCount:Integer;
  function  FindRelationship(const RelationName:LongString):Integer;
  function  RelationshipName(RelationID:Integer):LongString;
  function  RelationshipRange(RelationID:Integer):TRect2D;
  function  RelationshipValue(RelationID:Integer; Argument:Double;
                              Inverted:Boolean):Double;
  function  Time:Double;
  function  TimeUnits:Double;
  function  NumAIs:Integer;
  function  NumDIs:Integer;
  function  NumAOs:Integer;
  function  NumDOs:Integer;
  function  NumCals:Integer;
  function  RefAI(n:Integer):Integer;
  function  RefDI(n:Integer):Integer;
  function  RefAO(n:Integer):Integer;
  function  RefDO(n:Integer):Integer;
  function  RefCalibr(n:Integer):Integer;
  function  GetCurveLength(c:Integer):Integer;
  procedure SetCurveLength(c,aLength:Integer);
  function  GetCurvePoint(c,i:Integer):TPoint2D;
  procedure SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
  procedure CurveAddPoint(c:Integer; const aPoint:TPoint2D);
  procedure CurveAddPoints(c:Integer; const aX,aY:array of Double;
                                            aCount:Integer);
  procedure CurveDeletePoint(c,i:Integer);
  procedure CurveDeletePoints(c,i,aCount:Integer);
  procedure CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
  procedure CurveInsertPoints(c,i:Integer; const aX,aY:array of Double;
                              aCount:LongInt);
  function  GetCurveName(c:Integer):LongString;
  function  GetCurveColor(c:Integer):Integer;
  function  GetCurveStyle(c:Integer):Integer;
  function  GetCurveComment(c:Integer):LongString;
  procedure SetCurveComment(c:Integer; const aComment:LongString);
  function  CurveGetIndexAt(c:Integer; t:Double):Integer;
  function  CurveInterpolateAt(c:Integer; t:Double):Double;
  function  CurveSmoothAt(c:Integer; t:Double;
                        Window:Double; Power,K1,K2:Integer):Double;
  function  CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double;
                         SortMethod:Integer=1):Boolean;
  function  CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10;
                       Small:Integer=2):LongInt;
  function  GetCurveLimits(c:Integer):TRect2D;
  function  GetCurveIntegral(c:Integer; a,b:Double):Double;
  function  GetCurveLastPoint(c:Integer):TPoint2D;
  function  PutEv(w,n:Integer; t,d0:Double; d1:Double=0):Boolean;
  function  PutAO(n:Integer; t,d:Double):Boolean;
  function  PutDO(n:Integer; t,d:Double):Boolean;
  function  getai(n:Integer;t:Double):Double;
  function  getai_n(n:Integer):Integer;
  function  getai_xn(n:Integer):Double;
  function  getai_yn(n:Integer):Double;
  function  getai_xi(n,i:Integer):Double;
  function  getai_yi(n,i:Integer):Double;
  function  getdi(n:Integer;t:Double):Double;
  function  getdi_n(n:Integer):Integer;
  function  getdi_xn(n:Integer):Double;
  function  getdi_yn(n:Integer):Double;
  function  getdi_xi(n,i:Integer):Double;
  function  getdi_yi(n,i:Integer):Double;
  function  calibr(n:Integer; t,d:Double):Double;
  function  fixerror(n:Integer):Boolean;
  function  aimap(i,n:Integer):Integer;
  function  dimap(i,n:Integer):Integer;
  function  aomap(i,n:Integer):Integer;
  function  domap(i,n:Integer):Integer;
  function  diword(i,n:Integer):Integer;
  function  ClickButton:Integer;
  function  ClickSensor:LongString;
  function  ClickTag:Integer;
  function  Voice(const msg:LongString):Boolean;
  function  WinDraw(const Caption:LongString):Boolean;
  function  WinShow(const Caption:LongString):Boolean;
  function  WinHide(const Caption:LongString):Boolean;
  function  WinSelect(const Caption:LongString):Boolean;
  function  ReadIni(const s:LongString):LongString;
  function  GetPortB(Address:Word):Byte;
  procedure SetPortB(Address:Word; Value:Byte);
  function  GetPortW(Address:Word):Word;
  procedure SetPortW(Address:Word; Value:Word);
  function  GetPortL(Address:Word):LongWord;
  procedure SetPortL(Address:Word; Value:LongWord);
  function  RegisterErr(const s:LongString):Integer;
  function  GetErrCount(n:Integer):Double;
  function  RunCount:Double;
  function  FindObjectRef(const What,Name:LongString):Integer;
  function  DeviceName(d:Integer):LongString;
  function  DeviceModel(d:Integer):LongString;
  function  DeviceFamily(d:Integer):LongString;
  function  DeviceAction(d:Integer):Boolean;
  function  DeviceClear(d:Integer):Boolean;
  function  DeviceClearDevice(d:Integer):Boolean;
  function  DeviceStart(d:Integer):Boolean;
  function  DeviceStop(d:Integer):Boolean;
  function  DeviceMessage(d:Integer; const msg:LongString):Double;
  function  SaveCrw(const ArgList:LongString):Boolean;
  function  SpecMarker(const Caption:LongString):Double;
  function  SpecMarkerL(const Caption:LongString):Double;
  function  SpecMarkerR(const Caption:LongString):Double;
  function  SpecRoiL(const Caption:LongString):Double;
  function  SpecRoiR(const Caption:LongString):Double;
  function  GlobalCalc(const Expression:LongString):Double;
  function  UartOpen(aPortNum:Integer; aBaudrate:Integer=9600; aParity:Integer=NOPARITY;
                     aDataBits:Integer=8; aStopBits:Integer=ONESTOPBIT; aXonXoff:Boolean=false;
                     aFifoSize:Integer=4096;aDcbFlags:LongInt=0):Boolean;
  function  UartOpenConfig(const Section:LongString='[SerialPort-COM1]'
                           ):Integer;
  procedure UartClose(aPortNum:Integer);
  function  UartActive(aPortNum:Integer):Boolean;
  function  UartWriteBuf(aPortNum:Integer; const Data:PChar;
                         Count:Cardinal):Cardinal;
  function  UartReadBuf(aPortNum:Integer; var Data:PChar;
                        MaxCount:Cardinal):Cardinal;
  function  UartWriteStr(aPortNum:Integer;
                         const Data:LongString):Boolean;
  function  UartReadStr(aPortNum:Integer;
                        MaxCount:Byte=255):LongString;
  procedure UartRxClear(aPortNum:Integer);
  procedure UartTxClear(aPortNum:Integer);
  function  UartRxCount(aPortNum:Integer):Integer;
  function  UartTxCount(aPortNum:Integer):Integer;
  function  UartRxSpace(aPortNum:Integer):Integer;
  function  UartTxSpace(aPortNum:Integer):Integer;
  function  UartFindCheckSum(Data:PChar; Length:Integer):Byte;
  function  UartCatCheckSum(const DataStr:LongString):LongString;
  function  UartCatCheckSumCR(const DataStr:LongString;
                              CR:Char=ASCII_CR):LongString;
  function  UartTruncCheckSum(const DataStr:LongString):LongString;
  function  UartTruncCheckSumCR(const DataStr:LongString;
                                CR:Char=ASCII_CR):LongString;
  function  AdamFindSlot(const aSlotDeviceName:LongString=''):Integer;
  function  AdamGetStatus(aSlot:Integer):Integer;
  function  AdamSendRequest(aSlot:Integer; aRequest:LongString;
                            aTimeOut:Integer):Boolean;
  function  AdamGetRequest(aSlot:Integer):LongString;
  function  AdamGetRequestTime(aSlot:Integer):Double;
  function  AdamGetAnswer(aSlot:Integer):LongString;
  function  AdamGetPort(aSlot:Integer):Integer;
  function  AdamGetAddress(aSlot:Integer):LongString;
  function  AdamGetUsesCheckSum(aSlot:Integer):Boolean;
  function  ClickParams(const Args:LongString):LongString;
  function  Add3D(x,y,z:Double):Boolean;
  function  Plot3D(Nx,Ny:Integer; x1,y1,x2,y2:Double;
                   const opt:LongString):Boolean;
  function  Edit(const s:LongString):LongString;
  function  EditState:Integer;
  function  DevSend(d:Integer; const msg:LongString):Double;
  function  FileRef(const FName,DefExt,BaseFile:LongString):LongString;
  function  DevPost(d:Integer; const msg:LongString):Double;
  procedure Echo(const Msg:LongString);
  function  RedirectStdIn(var stdIn:Text):boolean;
  function  RedirectStdOut(var stdOut:Text):boolean;
  function  str2shortcut(const s:LongString):Integer;
  function  shortcut2str(shortcut:Integer):LongString;
  function  awakeflag:Boolean;
  function  crypt_ctrl(const s:LongString):LongString;
  function  crypt_encode(const s,key:LongString):LongString;
  function  crypt_decode(const s,key:LongString):LongString;
  function  getmd5fromstr(const s:LongString):LongString;
  function  getmd5fromfile(const s:LongString):LongString;
  function  getmd5fromtext(ref:Integer):LongString;
  function  wdt_reset(flag:Boolean):Double;
  function  guard_check(const s:LongString):Integer;
  function  timebase:Double;
  function  daqfileref(f,e:LongString):LongString;
  function  GetClockRes(what:Integer):Double;
  function  SetClockRes(res:Double):Double;
  function  cpu_start:Boolean;
  function  cpu_clock:Double;
  function  pidaffinity(pid,mask:Integer):Integer;
  function  devaffinity(ref,mask:Integer):Integer;
  function  clickwrite(const lines:LongString):Integer;
  function  clickread:Integer;
  function  clickwhat:Integer;
  function  clickwrote:Integer;
  function  clickfilter(aFilter:Integer):Integer;
  function  clickawaker(aAwaker:Integer):Integer;
  function  taglock(lock:Boolean):Integer;
  function  getai_par(n:integer; id:integer):Double;
  function  getao_par(n:integer; id:integer):Double;
  function  getdi_par(n:integer; id:integer):Double;
  function  getdo_par(n:integer; id:integer):Double;
 private
  myOwner  : TCrwApiServer;
  myDaqDev : TDaqDevice;
  myDaqCmd : Integer;
  myDaqDS  : Pointer;
  function EncodeRef(Ref:TObject):Integer;
  function DecodeRef(Ref:Integer):TObject;
 public
  constructor Create(aOwner:TCrwApiServer);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    DaqDev         : TDaqDevice       read myDaqDev write myDaqDev;
  property    DaqCmd         : Integer          read myDaqCmd write myDaqCmd;
 end;
 //
 // Реализация интерфейса обработки данных.
 //
 TDanApiServer = class(TPluginApi,IDanApi)
 public
  procedure ServerActionsBeforeExecution;                                override;
  procedure ServerActionsAfterExecution;                                 override;
  function  CrwApi:ICrwApi;                                              override;
 public // IDanApi
  function  GetArguments:LongString;
  function  GetArgumentAsString(const aName:LongString):LongString;
  function  GetArgumentAsInteger(const aName:LongString;
                                 var aValue:Integer):boolean;
  function  GetArgumentAsDouble(const aName:LongString;
                                 var aValue:Double):boolean;
  function  WindowExists(w:Integer):Boolean;
  function  CurveExists(c:Integer):Boolean;
  function  GetWindowCaption(w:Integer):LongString;
  procedure SetWindowCaption(w:Integer;const aCaption:LongString);
  function  GetWindowTitle(w:Integer):LongString;
  procedure SetWindowTitle(w:Integer;const aTitle:LongString);
  function  GetWindowLegend(w:Integer):LongString;
  procedure SetWindowLegend(w:Integer;const aLegend:LongString);
  function  GetWindowRoi(w:Integer):TRect2D;
  procedure SetWindowRoi(w:Integer; const aRoi:TRect2D);
  function  GetWindowComment(w:Integer):LongString;
  procedure SetWindowComment(w:Integer;const aComment:LongString);
  function  GetSelectedCurve(w:Integer):Integer;
  procedure SetSelectedCurve(w,c:Integer);
  function  GetCurvesCount(w:Integer):Integer;
  procedure SetCurvesCount(w,aCount:Integer);
  function  GetCurveLength(c:Integer):Integer;
  procedure SetCurveLength(c,aLength:Integer);
  function  GetCurvePoint(c,i:Integer):TPoint2D;
  procedure SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
  procedure CurveAddPoint(c:Integer; const aPoint:TPoint2D);
  procedure CurveAddPoints(c:Integer; const aX,aY:array of Double;
                                            aCount:Integer);
  procedure CurveDeletePoint(c,i:Integer);
  procedure CurveDeletePoints(c,i,aCount:Integer);
  procedure CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
  procedure CurveInsertPoints(c,i:Integer; const aX,aY:array of Double;
                              aCount:LongInt);
  function  GetCurveName(c:Integer):LongString;
  procedure SetCurveName(c:Integer; const aName:LongString);
  function  GetCurveColor(c:Integer):Integer;
  procedure SetCurveColor(c:Integer; aColor:Integer);
  function  GetCurveStyle(c:Integer):Integer;
  procedure SetCurveStyle(c:Integer; aStyle:Integer);
  function  GetCurveComment(c:Integer):LongString;
  procedure SetCurveComment(c:Integer; const aComment:LongString);
  function  CurveGetIndexAt(c:Integer; t:Double):Integer;
  function  CurveInterpolateAt(c:Integer; t:Double):Double;
  function  CurveSmoothAt(c:Integer; t:Double;
                        Window:Double; Power,K1,K2:Integer):Double;
  function  CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double;
                         SortMethod:Integer=1):Boolean;
  function  CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10;
                       Small:Integer=2):LongInt;
  procedure CurveAssign(c1,c2:Integer);
  procedure CurveCatenate(c1,c2:Integer);
  function  CreateCurve(w:Integer; const aName:LongString='';
                        aColor:Integer=0;  aStyle:Integer=0;
                        aComment:LongString=''):Integer;
  procedure CurveSort(c:Integer; Flags:Integer=0; AbsEps:Double=0;
                      RelEps:Double=1E-10; Method:Integer=0);
  function  GetCurveLimits(c:Integer):TRect2D;
  procedure CurveIntegrate(c:Integer; AFreeConst:Double=0);
  procedure CurveDifferentiate(c:Integer);
  procedure CurveMedianFiltr(c:Integer; Width:Integer=1;
                             AbsEps:Double=0; RelEps:Double=1E-10);
  function  GetCurveIntegral(c:Integer; a,b:Double):Double;
  function  GetCurveLastPoint(c:Integer):TPoint2D;
 private
  myOwner  : TCrwApiServer;
  mySrcWin : TFormCurveWindow;
  myDstWin : TFormCurveWindow;
  mySrcFix : packed record dp:Boolean; ca:TCloseAction; ws:TWindowState; end;
  myDstFix : packed record dp:Boolean; ca:TCloseAction; ws:TWindowState; end;
  myArgums : LongString;
  function  iowin(w:Integer; e:Boolean=true):TFormCurveWindow;
  function  iocrv(c:Integer; e:Boolean=true):TCurve;
  function  iocheckindex(c,i:Integer; e:Boolean=true):Boolean;
 public
  constructor Create(aOwner:TCrwApiServer);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    SrcCurveWindow : TFormCurveWindow read mySrcWin write mySrcWin;
  property    DstCurveWindow : TFormCurveWindow read myDstWin write myDstWin;
  property    Arguments      : LongString       read myArgums write myArgums;
 end;
 //
 // Реализация интерфейса для интерпретатора Daq Script.
 //
 TScriptInterpreterServer = class(TInterfacedMasterObject,IScriptInterpreter)
 public
  function    GetValue(aName:PChar; var aValue:Double):Boolean;
  function    SetValue(aName:PChar; var aValue:Double):Boolean;
  procedure   Clear;
  function    Evaluate(aExpression:PChar):Integer;
  function    GetScript:LongString;
  procedure   SetScript(const aScript:LongString);
  function    RunScript:Integer;
  function    Status:Integer;
  function    StatusMessage:LongString;
 private
  myEe : TExpressionEvaluator;
 public
  constructor Create;
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация интерфейса для полинома МНК.
 //
 TPolynomWrapServer = class(TInterfacedMasterObject,IPolynomWrap)
  function  GetPower:Integer;
  procedure SetPower(aPower:Integer);
  function  GetCoeff(aIndex:Integer):Double;
  procedure SetCoeff(aIndex:Integer; aValue:Double);
  function  GetCenter:Double;
  procedure SetCenter(aCenter:Double);
  function  GetScale:Double;
  procedure SetScale(aScale:Double);
  function  Get(at:Double; aDerivNum:Integer=0):Double;
  function  Find(const X,Y:array of Double; NumPoints:Integer;
                 Eps:double=1E-14):boolean;
  function  FindWeight(const X,Y,W:array of Double; NumPoints:Integer;
                       Eps:double=1E-14):boolean;
 private
  myPoly : TPolynom;
 public
  constructor Create(aPower  : TPolynomPower = 1;
                     aCenter : Double        = 0;
                     aScale  : Double        = 1);
  destructor Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация интерфейса для базы данных РФА.
 //
 TRfaDataBaseWrapServer = class(TNoRefCountMasterObject,IRfaDataBaseWrap)
  function  GetAccessible(AtomId:Integer):Boolean;
  function  GetAsDouble(AtomId:Integer; FieldId:TRfaFieldId):Double;
  function  GetAsString(AtomId:Integer; FieldId:TRfaFieldId):LongString;
  function  GetXRayLine(AtomId:Integer; LineId:TRfaFieldId):TRfaXRayLine;
  function  GetAtomSymbol(AtomId:Integer):LongString;
  function  GetFieldName(FieldId:TRfaFieldId):LongString;
  function  GetLineName(LineId:TRfaFieldId):LongString;
  function  GetCrossSectionTableCount(AtomId:Integer):Integer;
  function  GetCrossSectionTable(AtomId:Integer; aChan:TRfaCrossSectionFieldId;
                                 aIndex:Integer):Double;
  function  FindAtom(const aSymbol:LongString):Integer;
  function  FindLine(const aLineName:LongString):TRfaFieldId;
  function  FindCrossSection(AtomID:Integer; CrossId:TRfaCrossSectionFieldId;
                               Energy:Double):Double;
 private
  myOwner : TSysApiServer;
  myIFace : IRfaDataBaseWrap;
 public
  constructor Create(aOwner:TSysApiServer);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация интерфейса для векторов и матриц.
 //
 TLongIntVectorWrapServer = class(TInterfacedMasterObject,ILongIntVectorWrap)
  procedure Lock;
  procedure UnLock;
  function  GetLength:LongInt;
  procedure SetLength(aLength:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aIndex:LongInt):LongInt;
  procedure SetItem(aIndex:LongInt; const aValue:LongInt);
  function  GetItemPtr(aIndex:LongInt):PLongInt;
 private
  myVector : TLongIntVector;
  myOwns   : Boolean;
 public
  constructor Create(aVector:TLongIntVector; aOwns:Boolean);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    TheVector : TLongIntVector read myVector;
 end;
 TDoubleVectorWrapServer = class(TInterfacedMasterObject,IDoubleVectorWrap)
  procedure Lock;
  procedure UnLock;
  function  GetLength:LongInt;
  procedure SetLength(aLength:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aIndex:LongInt):Double;
  procedure SetItem(aIndex:LongInt; const aValue:Double);
  function  GetItemPtr(aIndex:LongInt):PDouble;
 private
  myVector : TDoubleVector;
  myOwns   : Boolean;
 public
  constructor Create(aVector:TDoubleVector; aOwns:Boolean);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    TheVector : TDoubleVector read myVector;
 end;
 TLongIntMatrixWrapServer = class(TInterfacedMasterObject,ILongIntMatrixWrap)
  procedure Lock;
  procedure UnLock;
  function  GetRows:LongInt;
  procedure SetRows(aRows:LongInt);
  function  GetColumns:LongInt;
  procedure SetColumns(aColumns:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aRow,aColumn:LongInt):LongInt;
  procedure SetItem(aRow,aColumn:LongInt; const aValue:LongInt);
  function  GetItemPtr(aRow,aColumn:LongInt):PLongInt;
 private
  myMatrix : TLongIntMatrix;
  myOwns   : Boolean;
 public
  constructor Create(aMatrix:TLongIntMatrix; aOwns:Boolean);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    TheMatrix : TLongIntMatrix read myMatrix;
 end;
 TDoubleMatrixWrapServer = class(TInterfacedMasterObject,IDoubleMatrixWrap)
  procedure Lock;
  procedure UnLock;
  function  GetRows:LongInt;
  procedure SetRows(aRows:LongInt);
  function  GetColumns:LongInt;
  procedure SetColumns(aColumns:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aRow,aColumn:LongInt):Double;
  procedure SetItem(aRow,aColumn:LongInt; const aValue:Double);
  function  GetItemPtr(aRow,aColumn:LongInt):PDouble;
 private
  myMatrix : TDoubleMatrix;
  myOwns   : Boolean;
 public
  constructor Create(aMatrix:TDoubleMatrix; aOwns:Boolean);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property    TheMatrix : TDoubleMatrix read myMatrix;
 end;
 //
 // Реализация сервера сингулярных разложений
 //
 TSVDWrapServer=class(TInterfacedMasterObject,ISVDWrap)
  function  GetM:Integer;
  function  GetN:Integer;
  function  GetU(i,j:Integer):Double;
  function  GetW(i:Integer):Double;
  function  GetV(i,j:Integer):Double;
  function  Eps(AbsEps:Double=0; RelEps:Double=1E-10):Double;
  function  Rank(AbsEps:Double=0; RelEps:Double=1E-10):Integer;
  function  Solve(X,Y:IDoubleVectorWrap;
                  AbsEps:Double=0; RelEps:Double=1E-10):Boolean;
  function  PInvert(AbsEps:Double=0;
                    RelEps:Double=1E-10):IDoubleMatrixWrap;
 private
  mySVD : TSVD;
 public
  constructor Create(A:IDoubleMatrixWrap);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Реализация сервера FIFO
 //
 TFifoWrapServer = class(TInterfacedMasterObject,IFifoWrap)
  procedure   Lock;
  procedure   UnLock;
  procedure   Clear;
  function    GetCount:LongInt;
  function    GetSpace:LongInt;
  function    GetSize:LongInt;
  function    GetName:LongString;
  procedure   SetName(const NewName:LongString);
  function    GetTag(i:Integer):LongWord;
  procedure   SetTag(i:Integer; NewTag:LongWord);
  function    GetLost:Int64;
  procedure   SetLost(N:Int64);
  function    Put(Buffer:Pointer; BufferSize:LongInt;
                  TryMode:boolean=false):LongInt;
  function    Get(Buffer:Pointer; BufferSize:LongInt):LongInt;
  function    PutText(const S:LongString;
                      TryMode:boolean=false):Boolean;
  function    GetText(MaxLen:Integer=MaxInt):LongString;
 private
  myFifo : TFifo;
  myOwns : Boolean;
 public
  constructor Create(aFifo:TFifo; aOwns:Boolean);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 end;
 //
 // Оболочка для вызова DLL в CRW-DAQ.
 //
type
 EDaqDllWrapper = class(ESoftException);
 TDaqDllWrapper = class(TLatch)
 private
  myDevice : TDaqDevice;
  myItems  : packed array[1..15] of record
   Api     : TCrwApiServer;
   Plugin  : TPluginWrapper;
  end;
 public
  constructor Create(aDevice:TDaqDevice);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  function    DaqDllInit(aDllPath:LongString):Integer;
  function    DaqDllFree(aDllRef:Integer):Boolean;
  function    DaqDllCall(aDllRef,aCommand:Integer):Boolean;
 end;

function  NewCrwApiServer(aTarget:Cardinal; const aName,aSource,aBinary:LongString):TCrwApiServer;
procedure Kill(var TheObject:TCrwApiServer); overload;

function  NewDaqDllWrapper(aDevice:TDaqDevice):TDaqDllWrapper;
procedure Kill(var TheObject:TDaqDllWrapper); overload;

 {
 Directory of data analysis plugins.
 [System] DataAnalysisPlugins
 }
function  GetDataAnalysisPluginsFolder:LongString;

 {
 Directory of data analysis templates.
 [System] DataAnalysisTemplates
 }
function  GetDataAnalysisTemplatesFolder:LongString;

 {
 Pascal Project Type Code is char to identify Free/Object Pascal projects.
 }
const                   // Project Code  Type   IDE
 pptc_Lazarus    = 'l'; //         l     .lpr   Lazarus
 pptc_Delphi     = 'd'; //         d     .dpr   Delphi
 pptc_CodeTyphon = 't'; //         t     .ctpr  CodeTyphon

const // Supported project type codes.
 DefaultListOfPascalProjectTypeCodes='l';

 {
 Get EOL delimited list of project types (file extensions) by Codes.
 Use DefaultListOfPascalProjectTypeCodes if empty Codes.
 }
function ListOfPascalProjectTypes(Codes:LongString=''):LongString;

 {
 Get list of all projects in directory Dir by project type.
 }
function GetPluginsListFromDirectory(Dir:LongString; FileNamesOnly:Boolean;
                                     ProjectType:LongString):LongString;

 {
 Get list of all projects in DataAnalysisPlugins directory.
 }
function  GetDataAnalysisPluginsList(FileNamesOnly:Boolean;
                                     ProjectType:LongString):LongString;

function  CreateNewDataAnalysisPlugin(ProjectType,Params:LongString):LongString;

function CreateNewDataAcquisitionPlugin:LongString;

function dlc_PluginWrap:Integer;

implementation

uses
 _crw_daqpascaldevice,
 _crw_daqpascalcompiler,
 Form_CreateNewDataAnalysisPlugin;

 //////////////////////
 // Debug log routines.
 //////////////////////

const
 PluginWrapBalance:TAtomicCounter=nil;

procedure InitPliginCounters;
begin
 LockedInit(PluginWrapBalance);
end;

procedure FreePliginCounters;
begin
 LockedFree(PluginWrapBalance);
end;

function dlc_PluginWrap:Integer;
const dlc:Integer=0;
begin
 if (dlc=0) then dlc:=RegisterDebugLogChannel('_PluginWrap');
 Result:=dlc;
end;

procedure LogBorn(Obj:TObject);
begin
 if Assigned(Obj) then begin
  LockedInc(PluginWrapBalance);
  if DebugLogEnabled(dlc_PluginWrap)
  then DebugLog(dlc_PluginWrap,'Born '+Obj.ClassName);
 end;
end;

procedure LogKill(Obj:TObject);
begin
 if Assigned(Obj) then begin
  LockedDec(PluginWrapBalance);
  if DebugLogEnabled(dlc_PluginWrap)
  then DebugLog(dlc_PluginWrap,'Kill '+Obj.ClassName);
 end;
end;

 ///////////////////////////////////////////
 // As noted in very useful article:      //
 //  https://habr.com/ru/articles/534466  //
 // constant strings may cause exception, //
 // when DllExit try free strings. So all //
 // strings in PluginApi must use sb(…).  //
 // The sb(S) guarantee result in heap.   //
 ///////////////////////////////////////////
function sb(const S:LongString):LongString;
begin
 if (StringRefCount(S)<0)      // If S is constant string,
 then Result:=StringBuffer(S)  // then make a copy in heap
 else Result:=S;               // else use original string
end;

 //////////////////////////////////
 // TCrwApiServer implementation //
 //////////////////////////////////
constructor TCrwApiServer.Create(aTarget:Cardinal; const aName,aSource,aBinary:LongString);
begin
 inherited Create;
 mySysApiServer:=nil;
 myGuiApiServer:=nil;
 myDaqApiServer:=nil;
 myDanApiServer:=nil;
 myTarget:=aTarget;
 mySource:=Trim(aSource);
 myBinary:=Trim(aBinary);
 myName:=Trim(aName);
 myCrwApi:=nil;
 mySysApi:=nil;
 myGuiApi:=nil;
 myDaqApi:=nil;
 myDanApi:=nil;
 myFpuModes:=FpuGetCurrentModes;
end;

destructor  TCrwApiServer.Destroy;
begin
 myName:='';
 mySource:='';
 myBinary:='';
 mySysApiServer.Free;
 myGuiApiServer.Free;
 myDaqApiServer.Free;
 myDanApiServer.Free;
 inherited;
end;

procedure TCrwApiServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TCrwApiServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TCrwApiServer.ServerActionsBeforeExecution;
 procedure Check(IApi:IPluginApi);
 begin
  if not Assigned(IApi)
  then raise ECrwApi.Create('ServerActionsBeforeExecution failed.');
 end;
begin
 // Save FPU modes.
 myFpuModes:=FpuGetCurrentModes;
 // Check uses to initialize Api(s) if one not initialized.
 Check(SysApi); Check(GuiApi); Check(DaqApi); Check(DanApi);
 if Assigned(mySysApiServer) then mySysApiServer.ServerActionsBeforeExecution;
 if Assigned(myGuiApiServer) then myGuiApiServer.ServerActionsBeforeExecution;
 if Assigned(myDaqApiServer) then myDaqApiServer.ServerActionsBeforeExecution;
 if Assigned(myDanApiServer) then myDanApiServer.ServerActionsBeforeExecution;
end;

procedure TCrwApiServer.ServerActionsAfterExecution;
begin
 try
  if Assigned(myDanApiServer) then myDanApiServer.ServerActionsAfterExecution;
  if Assigned(myDaqApiServer) then myDaqApiServer.ServerActionsAfterExecution;
  if Assigned(myGuiApiServer) then myGuiApiServer.ServerActionsAfterExecution;
  if Assigned(mySysApiServer) then mySysApiServer.ServerActionsAfterExecution;
 finally
  // Restore FPU modes.
  myTmpModes:=FpuGetCurrentModes;
  if not IsSameFpuModes(myTmpModes,myFpuModes) then begin
   FpuSetCurrentModes(myFpuModes);
   myFpuModes:=FpuGetCurrentModes;
  end;
 end;
end;

function TCrwApiServer.CrwApi:ICrwApi;
begin
 if not Assigned(myCrwApi) then begin
  if not GetInterface(IID_ICrwApi,myCrwApi)
  then myCrwApi:=nil;
 end;
 Result:=myCrwApi;
end;

function TCrwApiServer.ExecuteDataAnalysis(const aDLLPath:LongString;
                                                 aSrcWin:TFormCurveWindow;
                                           const aArgums:LongString):Int64;
var Plugin:TPluginWrapper;
begin
 Result:=-1;
 if Assigned(Self) then
 try
  if FileExists(aDLLPath) then begin
   with DanApi as TDanApiServer do begin
    SrcCurveWindow:=aSrcWin;
    Arguments:=aArgums;
   end;
   Plugin:=NewPluginWrapper(aDLLPath,CRWDAQ_PLUGIN_ID);
   if Assigned(Plugin) then
   try
    Result:=Plugin.Execute(Self);
   finally
    Plugin.Free;
   end;
  end else Echo(Format(RusEng('Не найден файл "%s".',
                              'Not found file "%s".'),[aDLLPath]));
 except
  on E:Exception do BugReport(E,Self,'ExecuteDataAnalysis');
 end;
 Echo(Format(RusEng('%sПлагин %s вернул код %d.','%sPlugin %s return code %d.'),
            [StdDateTimePrompt(msecnow),ExtractDllBaseName(aDLLPath),Result]));
 if (Result<>0) then SystemConsole.Activate;
end;

function TCrwApiServer.ExecuteDataAcquisition(aPlugin:TPluginWrapper; aDaqDev:TDaqDevice; aDaqCmd:Integer):Int64;
begin
 Result:=-1;
 if Assigned(Self) then
 if Assigned(aPlugin) then
 if Assigned(aDaqDev) then
 try
  with DaqApi as TDaqApiServer do begin
   DaqDev:=aDaqDev;
   DaqCmd:=aDaqCmd;
  end;
  Result:=aPlugin.Execute(Self);
 except
  on E:Exception do BugReport(E,Self,'ExecuteDataAcquisition');
 end;
end;

function TCrwApiServer.Version:Cardinal;
begin
 Result:=CrwApiEdition;
end;

function TCrwApiServer.Target:Cardinal;
begin
 Result:=myTarget;
end;

function TCrwApiServer.SysApi:ISysApi;
begin
 if not Assigned(mySysApiServer) then begin
  mySysApiServer:=TSysApiServer.Create(Self);
  if Assigned(mySysApiServer) then begin
   mySysApiServer.Master:=@mySysApiServer;
   if not mySysApiServer.GetInterface(IID_ISysApi,mySysApi)
   then mySysApi:=nil;
  end else mySysApi:=nil;
 end;
 Result:=mySysApi;
end;

function TCrwApiServer.GuiApi:IGuiApi;
begin
 if not Assigned(myGuiApiServer) then begin
  myGuiApiServer:=TGuiApiServer.Create(Self);
  if Assigned(myGuiApiServer) then begin
   myGuiApiServer.Master:=@myGuiApiServer;
   if not myGuiApiServer.GetInterface(IID_IGuiApi,myGuiApi)
   then myGuiApi:=nil;
  end else myGuiApi:=nil;
 end;
 Result:=myGuiApi;
end;

function TCrwApiServer.DaqApi:IDaqApi;
begin
 if not Assigned(myDaqApiServer) then begin
  myDaqApiServer:=TDaqApiServer.Create(Self);
  if Assigned(myDaqApiServer) then begin
   myDaqApiServer.Master:=@myDaqApiServer;
   if not myDaqApiServer.GetInterface(IID_IDaqApi,myDaqApi)
   then myDaqApi:=nil;
  end else myDaqApi:=nil;
 end;
 Result:=myDaqApi;
end;

function TCrwApiServer.DanApi:IDanApi;
begin
 if not Assigned(myDanApiServer) then begin
  myDanApiServer:=TDanApiServer.Create(Self);
  if Assigned(myDanApiServer) then begin
   myDanApiServer.Master:=@myDanApiServer;
   if not myDanApiServer.GetInterface(IID_IDanApi,myDanApi)
   then myDanApi:=nil;
  end else myDanApi:=nil;
 end;
 Result:=myDanApi;
end;

function TCrwApiServer.PluginName:LongString;
begin
 Result:=sb(myName);
end;

function TCrwApiServer.PluginSourceFileName:LongString;
begin
 Result:=sb(mySource);
end;

function TCrwApiServer.PluginBinaryFileName:LongString;
begin
 Result:=sb(myBinary);
end;

function TargetStr(aTarget:Cardinal):LongString;
begin
 Result:='';
 case aTarget of
  ForCalculations    : Result:='ForCalculations';
  ForDataAnalysis    : Result:='ForDataAnalysis';
  ForDataAcquisition : Result:='ForDataAcquisition';
  else                 Result:='Unknown';
 end;
 Result:=Result+'('+IntToStr(aTarget)+')';
end;

function TCrwApiServer.VerifyPluginType(aTarget,aExpected:Cardinal; Test:Boolean=false):Boolean;
var msg:LongString;
begin
 Result:=(aTarget=aExpected);
 if Result or Test then Exit;
 msg:=RusEng('Неверный тип ','Invalid ')+'Target='+TargetStr(aTarget);
 msg:=msg+RusEng('. Ожидался ','. Expected ')+TargetStr(aExpected)+'.';
 raise ECrwApi.Create(msg);
end;

function TCrwApiServer.VerifyPluginDate(aEdition,aVersion:Cardinal; Test:Boolean=false):Boolean;
var msg:LongString;
begin
 Result:=(aEdition=aVersion);
 if Result or Test then Exit;
 msg:=RusEng('Негодная редакция плагина ','Invalid plugin edition ')+IntToStr(aEdition);
 msg:=msg+RusEng('. Ожидалась ','. Expected ')+IntToStr(aVersion)+'.';
 msg:=msg+RusEng(' Обновите плагин.',' Plugin update required.');
 raise ECrwApi.Create(msg);
end;

 //////////////////////////////////
 // TSysApiServer implementation //
 //////////////////////////////////
constructor TSysApiServer.Create(aOwner:TCrwApiServer);
begin
 inherited Create;
 myOwner:=aOwner;
 myStore:=nil;
 myRfa:=nil;
 myGlops_Flags:=glops_Default or glops_Threads;
end;

destructor  TSysApiServer.Destroy;
begin
 FreeStoreRefs;
 Kill(myStore);
 Kill(TObject(myRfa));
 if Assigned(myOwner) then myOwner.mySysApiServer:=nil;
 inherited;
end;

procedure TSysApiServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TSysApiServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TSysApiServer.SaveRef(aRef:PtrInt);
var P:Pointer;
begin
 P:=_crw_alloc.PtrIntToPointer(aRef);
 if Assigned(Self) and (aRef<>0) then begin
  if not Assigned(myStore) then begin
   myStore:=NewObjectStorage(false);
   myStore.OwnsObjects:=false;
   myStore.Master:=@myStore;
  end;
  if (myStore.IndexOf(P)<0) then myStore.Add(P);
 end;
end;

procedure TSysApiServer.DropRef(aRef:PtrInt);
var P:Pointer;
begin
 P:=_crw_alloc.PtrIntToPointer(aRef);
 if Assigned(Self) and (aRef<>0) then begin
  if Assigned(myStore) and (myStore.IndexOf(P)>=0)
  then myStore.Remove(P);
 end;
end;

procedure TSysApiServer.FreeStoreRefs;
var P:TObject; i:Integer; R:PtrInt;
begin
 if Assigned(Self) and Assigned(myStore) then
 try
  for i:= myStore.Count-1 downto 0 do begin
   R:=PointerToPtrInt(myStore[i]);
   P:=ObjectRegistry[R];
   myStore.Delete(i);
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,Self,'FreeStoreRefs');
 end;
end;

procedure TSysApiServer.ServerActionsBeforeExecution;
begin
end;

procedure TSysApiServer.ServerActionsAfterExecution;
begin
end;

function TSysApiServer.CrwApi:ICrwApi;
begin
 if Assigned(myOwner)
 then Result:=myOwner.CrwApi
 else Result:=nil;
end;

function TSysApiServer.RedirectStdIn(var stdIn:Text):boolean;
begin
 Result:=RedirectStdInToFifo(stdIn,StdInputFifo);
end;

function TSysApiServer.RedirectStdOut(var stdOut:Text):boolean;
begin
 Result:=RedirectStdOutToFifo(stdOut,StdOutputFifo);
end;

function  TSysApiServer.IncPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
begin
 Result:=_crw_alloc.IncPtr(Base,ByteOffset);
end;

function  TSysApiServer.DecPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
begin
 Result:=_crw_alloc.DecPtr(Base,ByteOffset);
end;

procedure TSysApiServer.SafeMove(const Source; var Dest; Count: LongInt);
begin
 _crw_alloc.SafeMove(Source,Dest,Count);
end;

procedure TSysApiServer.SafeFillChar(var X; Count: LongInt; Value: Byte);
begin
 _crw_alloc.SafeFillChar(X,Count,Value);
end;

function  TSysApiServer.AdjustBufferSize(Size:LongInt; Step:LongInt):LongInt;
begin
 Result:=_crw_alloc.AdjustBufferSize(Size,Step);
end;

function  TSysApiServer.Allocate(N:LongInt):Pointer;
begin
 Result:=_crw_alloc.Allocate(N);
end;

procedure TSysApiServer.Deallocate(var P:Pointer);
begin
 _crw_alloc.Deallocate(P);
end;

function  TSysApiServer.AllocSize(P:Pointer):LongInt;
begin
 Result:=_crw_alloc.AllocSize(P);
end;

function  TSysApiServer.Reallocate(var P:Pointer; N:LongInt):boolean;
begin
 Result:=_crw_alloc.Reallocate(P,N);
end;

function  TSysApiServer.isNAN(const X: Double): Boolean;
begin
 Result:=_crw_fpu.isNAN(X);
end;

function  TSysApiServer.isINF(const X: Double): Boolean;
begin
 Result:=_crw_fpu.isINF(X);
end;

function  TSysApiServer._NaN: Double;
begin
 Result:=_crw_fpu._NaN;
end;

function  TSysApiServer._PlusInf: Double;
begin
 Result:=_crw_fpu._PlusInf;
end;

function  TSysApiServer._MinusInf: Double;
begin
 Result:=_crw_fpu._MinusInf;
end;

function  TSysApiServer.FindZero(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
begin
 Result:=_crw_zm.FindZero(F,A,B,Tol,Custom);
end;

function  TSysApiServer.FindMin(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
begin
 Result:=_crw_zm.FindMin(F,A,B,Tol,Custom);
end;

function  TSysApiServer.Quanc8(F:zmFunType;A,B,AbsErr,RelErr:Double; var ErrEst:Double;
          var NoFun:Integer; var Flag:Double; Custom:Pointer):Double;
begin
 Result:=_crw_zm.Quanc8(F,A,B,AbsErr,RelErr,ErrEst,NoFun,Flag,Custom);
end;

function  TSysApiServer.MachEps:double;
begin
 Result:=_crw_zm.MachEps;
end;

function  TSysApiServer.NativeTimeNow:TSystemTime;
begin
 Result:=_crw_rtc.NativeTimeNow;
end;

function  TSysApiServer.MSecNow:double;
begin
 Result:=_crw_rtc.MSecNow;
end;

function  TSysApiServer.AssignNativeTime(Year:word=1;Month:word=1;Day:word=1;
                                         Hour:word=0;Minut:word=0;Sec:word=0;
                                         mSec:word=0):TSystemTime;
begin
 Result:=_crw_rtc.AssignNativeTime(Year,Month,Day,Hour,Minut,Sec,mSec);
end;

function  TSysApiServer.MSecToNativeTime(mSecond:Double):TSystemTime;
begin
 Result:=_crw_rtc.MSecToNativeTime(mSecond);
end;

function  TSysApiServer.NativeTimeToMSec(const T:TSystemTime):double;
begin
 Result:=_crw_rtc.NativeTimeToMSec(T);
end;

function  TSysApiServer.DateTimeToMSec(Year:Word=1;Month:Word=1;Day:Word=1;
                                       Hour:Word=0;Minut:Word=0;Sec:Word=0;
                                       mSec:Word=0):Double;
begin
 Result:=_crw_rtc.DateTimeToMSec(Year,Month,Day,Hour,Minut,Sec,mSec);
end;

procedure TSysApiServer.MSecToDateTime(T:Double; var Year,Month,Day,Hour,Min,Sec,mSec:word);
begin
 _crw_rtc.MSecToDateTime(T,Year,Month,Day,Hour,Min,Sec,mSec);
end;

function  TSysApiServer.GetDateStr(ms:Double;Delim:Char='.';YMD_Order:Boolean=false;
                                   ShortYear:Boolean=false):LongString;
begin
 Result:=sb(_crw_rtc.GetDateStr(ms,Delim,YMD_Order,ShortYear));
end;

function  TSysApiServer.GetTimeStr(ms:double;Delim:Char=':'; ShowMSec:Boolean=false):LongString;
begin
 Result:=sb(_crw_rtc.GetTimeStr(ms,Delim,ShowMSec));
end;

function  TSysApiServer.StrLen(Str: PChar): LongInt;
begin
 Result:=_crw_str.StrLen(Str);
end;

function  TSysApiServer.StrLLen(Str: PChar; MaxLen: Integer): LongInt;
begin
 Result:=_crw_str.StrLLen(Str,MaxLen);
end;

function  TSysApiServer.StrEnd(Str: PChar): PChar;
begin
 Result:=_crw_str.StrEnd(Str);
end;

function  TSysApiServer.StrCopy(Dest, Source: PChar): PChar;
begin
 Result:=_crw_str.StrCopy(Dest,Source);
end;

function  TSysApiServer.StrECopy(Dest, Source: PChar): PChar;
begin
 Result:=_crw_str.StrECopy(Dest,Source);
end;

function  TSysApiServer.StrLCopy(Dest, Source: PChar; MaxLen: LongInt): PChar;
begin
 Result:=_crw_str.StrLCopy(Dest,Source,MaxLen);
end;

function  TSysApiServer.StrPCopy(Dest: PChar; const Source: LongString): PChar;
begin
 Result:=_crw_str.StrPCopy(Dest,sb(Source));
end;

function  TSysApiServer.StrCat(Dest, Source: PChar): PChar;
begin
 Result:=_crw_str.StrCat(Dest,Source);
end;

function  TSysApiServer.StrPCat(Dest:PChar; const Source: LongString): PChar;
begin
 Result:=_crw_str.StrPCat(Dest,sb(Source));
end;

function  TSysApiServer.StrLCat(Dest, Source: PChar; MaxLen: LongInt): PChar;
begin
 Result:=_crw_str.StrLCat(Dest,Source,MaxLen);
end;

function  TSysApiServer.StrComp(Str1, Str2: PChar): Integer;
begin
 Result:=_crw_str.StrComp(Str1,Str2);
end;

function  TSysApiServer.StrIComp(Str1, Str2: PChar): Integer;
begin
 Result:=_crw_str.StrIComp(Str1,Str2);
end;

function  TSysApiServer.StrLComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
begin
 Result:=_crw_str.StrLComp(Str1,Str2,MaxLen);
end;

function  TSysApiServer.StrLIComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
begin
 Result:=_crw_str.StrLIComp(Str1,Str2,MaxLen);
end;

function  TSysApiServer.StrScan(Str: PChar; Chr: Char): PChar;
begin
 Result:=_crw_str.StrScan(Str,Chr);
end;

function  TSysApiServer.StrRScan(Str: PChar; Chr: Char): PChar;
begin
 Result:=_crw_str.StrRScan(Str,Chr);
end;

function  TSysApiServer.StrUpper(Str: PChar): PChar;
begin
 Result:=_crw_str.StrUpper(Str);
end;

function  TSysApiServer.StrLower(Str: PChar): PChar;
begin
 Result:=_crw_str.StrLower(Str);
end;

function  TSysApiServer.StrPos(Str1, Str2: PChar): PChar;
begin
 Result:=_crw_str.StrPos(Str1,Str2);
end;

function  TSysApiServer.StrPas(Str: PChar): LongString;
begin
 Result:=sb(_crw_str.StrPas(Str));
end;

function  TSysApiServer.StrPass(Str:PChar; const PassChars:TCharSet):PChar;
begin
 Result:=_crw_str.StrPass(Str,PassChars);
end;

function  TSysApiServer.StrMove(Dest, Source: PChar; Count: LongInt): PChar;
begin
 Result:=_crw_str.StrMove(Dest,Source,Count);
end;

function  TSysApiServer.GetTextNumLines(Text:PChar; Count:LongInt=MaxLongInt; UnixStyle:boolean=false):LongInt;
begin
 Result:=_crw_str.GetTextNumLines(Text,Count,UnixStyle);
end;

function  TSysApiServer.UpCase(c:Char):Char;
begin
 Result:=_crw_str.UpCase(c);
end;

function  TSysApiServer.LoCase(c:Char):Char;
begin
 Result:=_crw_str.LoCase(c);
end;

function  TSysApiServer.LoCaseStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.LoCaseStr(sb(S)));
end;

function  TSysApiServer.UpCaseStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.UpCaseStr(sb(S)));
end;

function  TSysApiServer.WinToDos(c:Char):Char;
begin
 Result:=_crw_str.WinToDos(c);
end;

function  TSysApiServer.DosToWin(c:Char):Char;
begin
 Result:=_crw_str.DosToWin(c);
end;

function  TSysApiServer.WinToDosStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.WinToDosStr(sb(S)));
end;

function  TSysApiServer.DosToWinStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.DosToWinStr(sb(S)));
end;

function  TSysApiServer.WinToDosLongStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.WinToDosStr(sb(S)));
end;

function  TSysApiServer.DosToWinLongStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.DosToWinStr(sb(S)));
end;

function  TSysApiServer.WinToKoi(c:Char):Char;
begin
 Result:=_crw_str.WinToKoi(c);
end;

function  TSysApiServer.KoiToWin(c:Char):Char;
begin
 Result:=_crw_str.KoiToWin(c);
end;

function  TSysApiServer.WinToKoiStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.WinToKoiStr(sb(S)));
end;

function  TSysApiServer.KoiToWinStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.KoiToWinStr(sb(S)));
end;

function  TSysApiServer.WinToKoiLongStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.WinToKoiStr(sb(S)));
end;

function  TSysApiServer.KoiToWinLongStr(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.KoiToWinStr(sb(S)));
end;

function  TSysApiServer.Format(const Fmt:LongString; const Args: array of const):LongString;
begin
 Result:=sb(_crw_str.Format(sb(Fmt),Args));
end;

function  TSysApiServer.Str2CharSet(const S:LongString):TCharSet;
begin
 Result:=_crw_str.Str2CharSet(sb(S));
end;

function  TSysApiServer.CharSet2Str(const S:TCharSet):LongString;
begin
 Result:=sb(_crw_str.CharSet2Str(S));
end;

function  TSysApiServer.RightStr(const S:LongString; pos:integer):LongString;
begin
 Result:=sb(_crw_str.RightStr(sb(S),pos));
end;

function  TSysApiServer.LeftStr(const S:LongString; count:integer):LongString;
begin
 Result:=sb(_crw_str.LeftStr(sb(S),count));
end;

function  TSysApiServer.CharStr(Len:Integer; Ch:Char=' '):LongString;
begin
 Result:=sb(_crw_str.CharStr(Len,Ch));
end;

function  TSysApiServer.Pad(const S:LongString; Len:Integer; Ch:Char=' '):LongString;
begin
 Result:=sb(_crw_str.Pad(sb(S),Len,Ch));
end;

function  TSysApiServer.LeftPad(const S:LongString; Len:Integer; Ch:Char=' '):LongString;
begin
 Result:=sb(_crw_str.LeftPad(sb(S),Len,Ch));
end;

function  TSysApiServer.CenterStr(const S:LongString; Width:Integer; Ch:Char=' '):LongString;
begin
 Result:=sb(_crw_str.CenterStr(sb(S),Width,Ch));
end;

function  TSysApiServer.TrimLeadChars(const S:LongString; const TrimChars:TCharSet):LongString;
begin
 Result:=sb(_crw_str.TrimLeadChars(sb(S),TrimChars));
end;

function  TSysApiServer.TrimTrailChars(const S:LongString; const TrimChars:TCharSet):LongString;
begin
 Result:=sb(_crw_str.TrimTrailChars(sb(S),TrimChars));
end;

function  TSysApiServer.TrimChars(const S:LongString; const LeadTrim,TrailTrim:TCharSet):LongString;
begin
 Result:=sb(_crw_str.TrimChars(sb(S),LeadTrim,TrailTrim));
end;

function  TSysApiServer.TrimLead(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.TrimLead(sb(S)));
end;

function  TSysApiServer.TrimTrail(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.TrimTrail(sb(S)));
end;

function  TSysApiServer.Trim(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.Trim(sb(S)));
end;

function  TSysApiServer.UnifyAlias(const Name:LongString; Mode:Integer=ua_NameDef):LongString;
begin
 Result:=sb(_crw_str.UnifyAlias(sb(Name),Mode));
end;

function  TSysApiServer.UnifyFileAlias(const FileName:LongString; Mode:Integer=ua_FileDef):LongString;
begin
 Result:=sb(_crw_str.UnifyFileAlias(sb(FileName),Mode));
end;

function  TSysApiServer.UnifySection(const aSectionName:LongString; Mode:Integer=ua_SectDef):LongString;
begin
 Result:=sb(_crw_str.UnifySection(sb(aSectionName),Mode));
end;

procedure TSysApiServer.WordWrap(const InSt:LongString; var OutSt, Overlap:LongString; Margin:Integer; PadToMargin:Boolean);
var aOutSt,aOverlap:LongString;
begin
 aOutSt:=sb(OutSt); aOverlap:=sb(Overlap);
 _crw_str.WordWrap(sb(InSt),aOutSt,aOverlap,Margin,PadToMargin);
 OutSt:=sb(aOutSt); Overlap:=sb(aOverlap);
end;

function TSysApiServer.ReplaceString(const Str,Find,Replace:LongString):LongString;
begin
 Result:=sb(_crw_str.ReplaceString(sb(Str),sb(Find),sb(Replace)));
end;

function TSysApiServer.ReplaceAlignStr(const Str:LongString; Invert:Boolean):LongString;
begin
 Result:=sb(_crw_str.ReplaceAlignStr(sb(Str),Invert));
end;

function  TSysApiServer.WordCount(const S:LongString; const WordDelims:TCharSet):Integer;
begin
 Result:=_crw_str.WordCount(sb(S),WordDelims);
end;

function  TSysApiServer.ExtractWord(N:Integer; const S:LongString; const WordDelims:TCharSet):LongString;
begin
 Result:=sb(_crw_str.ExtractWord(N,sb(S),WordDelims));
end;

function  TSysApiServer.WordIndex(const Name,Str:LongString; const Delims:TCharSet):Integer;
begin
 Result:=_crw_str.WordIndex(sb(Name),sb(Str),Delims);
end;

function  TSysApiServer.ExtractInt(N:Integer; const S:LongString; const WordDelims:TCharSet; var Value:LongInt):boolean;
begin
 Result:=_crw_str.ExtractInt(N,sb(S),WordDelims,Value);
end;

function  TSysApiServer.ExtractReal(N:Integer; const S:LongString; const WordDelims:TCharSet; var Value:Double):boolean;
begin
 Result:=_crw_str.ExtractReal(N,sb(S),WordDelims,Value);
end;

function  TSysApiServer.BinB(x:Byte):LongString;
begin
 Result:=sb(_crw_str.BinB(x));
end;

function  TSysApiServer.BinW(x:Word):LongString;
begin
 Result:=sb(_crw_str.BinW(x));
end;

function  TSysApiServer.BinL(x:LongWord):LongString;
begin
 Result:=sb(_crw_str.BinL(x));
end;

function  TSysApiServer.OctB(x:Byte):LongString;
begin
 Result:=sb(_crw_str.OctB(x));
end;

function  TSysApiServer.OctW(x:Word):LongString;
begin
 Result:=sb(_crw_str.OctW(x));
end;

function  TSysApiServer.OctL(x:LongInt):LongString;
begin
 Result:=sb(_crw_str.OctL(x));
end;

function  TSysApiServer.HexB(x:Byte):LongString;
begin
 Result:=sb(_crw_str.HexB(x));
end;

function  TSysApiServer.HexW(x:Word):LongString;
begin
 Result:=sb(_crw_str.HexW(x));
end;

function  TSysApiServer.HexL(x:LongWord):LongString;
begin
 Result:=sb(_crw_str.HexL(x));
end;

function  TSysApiServer.Long2Str(L:LongInt):LongString;
begin
 Result:=sb(_crw_str.Long2Str(L));
end;

function  TSysApiServer.Real2Str(R:Double; Width:Integer=0; Places:Integer=0):LongString;
begin
 Result:=sb(_crw_str.Real2Str(R,Width,Places));
end;

function  TSysApiServer.d2s(d:LongInt; Width:Integer=0):LongString;
begin
 Result:=sb(_crw_str.d2s(d,Width));
end;

function  TSysApiServer.f2s(f:Double):LongString;
begin
 Result:=sb(_crw_str.f2s(f));
end;

procedure TSysApiServer.f2sFormat(Width:Integer=0; Decimals:Integer=14);
begin
 _crw_str.f2sFormat(Width,Decimals);
end;

procedure TSysApiServer.f2sFormatOld;
begin
 _crw_str.f2sFormatOld;
end;

function  TSysApiServer.FormatG(X:Double; w:integer; d:integer):LongString;
begin
 Result:=sb(_crw_str.FormatG(X,w,d));
end;

function  TSysApiServer.atoi(a:PChar; var i:LongInt):boolean;
begin
 Result:=_crw_str.atoi(a,i);
end;

function  TSysApiServer.atof(a:PChar; var f:double):boolean;
begin
 Result:=_crw_str.atof(a,f);
end;

function  TSysApiServer.Str2Int(const S:LongString; var I:Integer):Boolean;
begin
 Result:=_crw_str.Str2Int(sb(S),I);
end;

function  TSysApiServer.Str2Word(const S:LongString; var I:Word):Boolean;
begin
 Result:=_crw_str.Str2Word(sb(S),I);
end;

function  TSysApiServer.Str2Long(const S:LongString; var I:LongInt):Boolean;
begin
 Result:=_crw_str.Str2Long(sb(S),I);
end;

function  TSysApiServer.Str2Real(const S:LongString; var R:Double):Boolean;
begin
 Result:=_crw_str.Str2Real(sb(S),R);
end;

function TSysApiServer.SmartStr2Real(const S:LongString; var R:Double):Boolean;
begin
 Result:=_crw_str.SmartStr2Real(sb(S),R);
end;

function  TSysApiServer.StrBin2Long(const S:LongString; var L:LongInt):Boolean;
begin
 Result:=_crw_str.StrBin2Long(sb(S),L);
end;

function  TSysApiServer.StrOct2Long(const S:LongString; var L:LongInt):Boolean;
begin
 Result:=_crw_str.StrOct2Long(sb(S),L);
end;

function  TSysApiServer.StrHex2Long(const S:LongString; var L:LongInt):Boolean;
begin
 Result:=_crw_str.StrHex2Long(sb(S),L);
end;

function  TSysApiServer.ScanVar(svMode:word; Str:PChar; const Format:LongString; var Data):PChar;
begin
 Result:=_crw_str.ScanVar(svMode,Str,sb(Format),Data);
end;

function  TSysApiServer.IsEmptyStr(const S:LongString):boolean;
begin
 Result:=_crw_str.IsEmptyStr(sb(S));
end;

function  TSysApiServer.LastDelimiter(const Delimiters:TCharSet; const S:LongString):Integer;
begin
 Result:=_crw_str.LastDelimiter(Delimiters,sb(S));
end;

function  TSysApiServer.AddBackSlash(const DirName:LongString):LongString;
begin
 Result:=sb(_crw_str.AddBackSlash(sb(DirName)));
end;

function  TSysApiServer.DropBackSlash(const DirName:LongString):LongString;
begin
 Result:=sb(_crw_str.DropBackSlash(sb(DirName)));
end;

function  TSysApiServer.IsWildCard(const FileName:LongString):boolean;
begin
 Result:=_crw_str.IsWildCard(sb(FileName));
end;

function  TSysApiServer.IsRelativePath(const S:LongString):boolean;
begin
 Result:=_crw_str.IsRelativePath(sb(S));
end;

function  TSysApiServer.HasExtension(const Name:LongString; var DotPos:Integer):Boolean;
var aDotPos:SizeInt;
begin
 aDotPos:=DotPos;
 Result:=_crw_str.HasExtension(sb(Name),aDotPos);
 DotPos:=aDotPos;
end;

function  TSysApiServer.DefaultExtension(const Name,Ext:LongString):LongString;
begin
 Result:=sb(_crw_str.DefaultExtension(sb(Name),sb(Ext)));
end;

function  TSysApiServer.DefaultPath(const Name,Path:LongString):LongString;
begin
 Result:=sb(_crw_str.DefaultPath(sb(Name),sb(Path)));
end;

function  TSysApiServer.ForceExtension(const Name,Ext:LongString):LongString;
begin
 Result:=sb(_crw_str.ForceExtension(sb(Name),sb(Ext)));
end;

function  TSysApiServer.ForcePath(const NewPath,Name:LongString):LongString;
begin
 Result:=sb(_crw_str.ForcePath(sb(NewPath),sb(Name)));
end;

function  TSysApiServer.ExtractFileNameExt(const FileName: LongString):LongString;
begin
 Result:=sb(_crw_str.ExtractFileNameExt(sb(FileName)));
end;

function  TSysApiServer.ExtractFileName(const FileName: LongString):LongString;
begin
 Result:=sb(_crw_str.ExtractFileName(sb(FileName)));
end;

function  TSysApiServer.ExtractFileExt(const FileName: LongString): LongString;
begin
 Result:=sb(_crw_str.ExtractFileExt(sb(FileName)));
end;

function  TSysApiServer.ExtractFilePath(const FileName: LongString): LongString;
begin
 Result:=sb(_crw_str.ExtractFilePath(sb(FileName)));
end;

function  TSysApiServer.ExtractFileDrive(FileName: LongString): LongString;
begin
 Result:=sb(_crw_str.ExtractFileDrive(sb(FileName)));
end;

function  TSysApiServer.ExtractFilterExt(const Filter:LongString; Index:Integer):LongString;
begin
 Result:=sb(_crw_str.ExtractFilterExt(sb(Filter),Index));
end;

function  TSysApiServer.FExpand(const Path:LongString):LongString;
begin
 Result:=sb(_crw_str.FExpand(sb(Path)));
end;

function  TSysApiServer.MakeRelativePath(const Path,Base:LongString):LongString;
begin
 Result:=sb(_crw_str.MakeRelativePath(sb(Path),sb(Base)));
end;

function  TSysApiServer.RusEng(const Rus:LongString=''; const Eng:LongString=''):LongString;
begin
 Result:=sb(_crw_str.RusEng(sb(Rus),sb(Eng)));
end;

function  TSysApiServer.SysErrorMessage(ErrorCode: Integer): LongString;
begin
 Result:=sb(_crw_str.SysErrorMessage(ErrorCode));
end;

function  TSysApiServer.GetFileAttr(const FileName:LongString):Integer;
begin
 Result:=_crw_fio.GetFileAttr(sb(FileName));
end;

function  TSysApiServer.SetFileAttr(const FileName: LongString; Attr: Integer):boolean;
begin
 Result:=_crw_fio.SetFileAttr(sb(FileName),Attr);
end;

function  TSysApiServer.GetFileSize(const FileName:LongString):LongInt;
begin
 Result:=_crw_fio.GetFileSize(sb(FileName));
end;

function  TSysApiServer.GetFileDate(const FileName:LongString; ConvertToMsec:Boolean=false):Int64;
begin
 Result:=_crw_fio.GetFileDate(sb(FileName),ConvertToMsec);
end;

function  TSysApiServer.FileExists(const FilePattern:LongString; Attribut:Integer=faAnyFile):Boolean;
begin
 Result:=_crw_fio.FileExists(sb(FilePattern),Attribut);
end;

function  TSysApiServer.DirExists(const DirName: LongString): Boolean;
begin
 Result:=_crw_fio.DirExists(sb(DirName));
end;

function  TSysApiServer.FileErase(const FileName:LongString; CheckExistance:boolean=true):boolean;
begin
 Result:=_crw_fio.FileErase(sb(FileName),CheckExistance);
end;

function  TSysApiServer.FileCopy(const CopyFrom,CopyTo:LongString; FailIfExists:boolean=false):boolean;
begin
 Result:=_crw_fio.FileCopy(sb(CopyFrom),sb(CopyTo),FailIfExists);
end;

function  TSysApiServer.FileRename(const OldFileName,NewFileName:LongString; FailIfExists:boolean=false):boolean;
begin
 Result:=_crw_fio.FileRename(sb(OldFileName),sb(NewFileName),FailIfExists);
end;

function  TSysApiServer.GetCurrDir: LongString;
begin
 Result:=sb(_crw_fio.GetCurrDir);
end;

function  TSysApiServer.SetCurrDir(const DirName: LongString): Boolean;
begin
 Result:=_crw_fio.SetCurrDir(sb(DirName));
end;

function  TSysApiServer.MkDir(const DirName: LongString): Boolean;
begin
 Result:=_crw_fio.MkDir(sb(DirName));
end;

function  TSysApiServer.RmDir(const DirName: LongString): Boolean;
begin
 Result:=_crw_fio.RmDir(sb(DirName));
end;

function  TSysApiServer.ReadFileToBuffer(const FileName:LongString; Buffer:Pointer; Count:LongInt; Offset:LongInt=0):LongInt;
begin
 Result:=_crw_fio.ReadFileToBuffer(sb(FileName),Buffer,Count,Offset);
end;

function  TSysApiServer.WriteBufferToFile(const FileName:LongString; Buffer:Pointer;
                            Count:LongInt; Offset:LongInt=0; FLAGS:DWORD=OPEN_ALWAYS) : LongInt;
begin
 Result:=_crw_fio.WriteBufferToFile(sb(FileName),Buffer,Count,Offset,FLAGS);
end;

procedure TSysApiServer.ForEachFile(const RootDir         : LongString;
                                    const FilePatterns    : LongString;
                                          FileAction      : TForEachFileAction;
                                          MaxSubDirLevels : Integer = 0;
                                          CustomData      : Pointer = nil;
                                    const Delimiters      : TCharSet = [' ',';',',',ASCII_Tab,ASCII_CR,ASCII_LF]);
begin
 _crw_fio.ForEachFile(sb(RootDir),sb(FilePatterns),FileAction,MaxSubDirLevels,CustomData,Delimiters);
end;

function  TSysApiServer.ForEachTextLine(const FileName:LongString; Action:TForEachTextLineAction; CustomData:Pointer=nil):Integer;
begin
 Result:=_crw_fio.ForEachTextLine(sb(FileName),Action,CustomData);
end;

procedure TSysApiServer.DebugOut(n:byte; const S:LongString);
begin
 _crw_fio.DebugOut(n,sb(S));
end;

function  TSysApiServer.DebugOutGetFile(n:byte):LongString;
begin
 Result:=sb(_crw_fio.DebugOutGetFile(n));
end;

function  TSysApiServer.DebugOutGetErrors(n:byte):Int64;
begin
 Result:=_crw_fio.DebugOutGetErrors(n);
end;

procedure TSysApiServer.DebugOutOpenFile(n           : byte;
                                   const FileName    : LongString;
                                         FifoSize    : Integer = 1024*64;
                                         FlushPeriod : Integer = 50;
                                         CreateNew   : boolean = true;
                                         Hello       : boolean = false);
begin
 _crw_fio.DebugOutOpenFile(n,sb(FileName),FifoSize,FlushPeriod,CreateNew,Hello);
end;

function  TSysApiServer.ExtractTextSection(const FileName            : LongString;
                                           const SectionName         : LongString;
                                                 ExtractSectionFlags : Word ) : LongString;
begin
 Result:=sb(_crw_fio.ExtractTextSection(sb(FileName),sb(SectionName),ExtractSectionFlags));
end;

function  TSysApiServer.ExtractWordList(const FileName            : LongString;
                                        const SectionName         : LongString;
                                              ExtractSectionFlags : word;
                                              WordIndex           : word;
                                        const Delimiters          : TCharSet
                                                                ) : LongString;
var p:TText;
begin
 p:=_crw_fio.ExtractWordList(sb(FileName),sb(SectionName),ExtractSectionFlags,WordIndex,Delimiters);
 if Assigned(p) then
 try
  Result:=sb(p.Text);
 finally
  Kill(p);
 end;
end;

function  TSysApiServer.ExtractEnumWordList(const IniFile             : LongString;
                                            const SectionName         : LongString;
                                            const Prefix              : LongString;
                                                  ExtractSectionFlags : word ) : LongString;
var p:TText;
begin
 p:=_crw_fio.ExtractEnumWordList(sb(IniFile),sb(SectionName),sb(Prefix),ExtractSectionFlags);
 if Assigned(p) then
 try
  Result:=sb(p.Text);
 finally
  Kill(p);
 end;
end;

function TSysApiServer.ReadIniFileVariable(const FileName    : LongString;
                                           const SectionName : LongString;
                                           const Format      : LongString;
                                             var Data      ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileVariable(sb(FileName),sb(SectionName),sb(Format),Data);
end;

function TSysApiServer.ReadIniFilePath(const FileName    : LongString;
                                       const SectionName : LongString;
                                       const Name        : LongString;
                                       const StartupPath : LongString;
                                         var Path        : LongString ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFilePath(sb(FileName),sb(SectionName),sb(Name),sb(StartupPath),Path);
end;

function  TSysApiServer.ProgName:LongString;
begin
 Result:=sb(_crw_fio.ProgName);
end;

function  TSysApiServer.HomeDir:LongString;
begin
 Result:=sb(_crw_fio.HomeDir);
end;

function  TSysApiServer.StartupPath:LongString;
begin
 Result:=sb(_crw_fio.StartupPath);
end;

function  TSysApiServer.StartAtDir:LongString;
begin
 Result:=sb(_crw_fio.StartAtDir);
end;

function  TSysApiServer.SysIniFile:LongString;
begin
 Result:=sb(_crw_fio.SysIniFile);
end;

procedure TSysApiServer.GoHome;
begin
 _crw_fio.GoHome;
end;

function  TSysApiServer.WindowsDir:LongString;
begin
 Result:=sb(_crw_fio.WindowsDir);
end;

function  TSysApiServer.WindowsSystemDir:LongString;
begin
 Result:=sb(_crw_fio.WindowsSystemDir);
end;

function  TSysApiServer.WindowsTempDir:LongString;
begin
 Result:=sb(_crw_fio.WindowsTempDir);
end;

function  TSysApiServer.TempDir:LongString;
begin
 Result:=sb(_crw_fio.TempDir);
end;

function  TSysApiServer.CreateTempFile(const Template:LongString='###.TMP'):LongString;
begin
 Result:=sb(_crw_fio.CreateTempFile(sb(Template)));
end;

function  TSysApiServer.ReadRegistryString(RootKey:HKEY; const Key,Name:LongString):LongString;
begin
 Result:=sb(_crw_fio.ReadRegistryString(RootKey,sb(Key),sb(Name)));
end;

function  TSysApiServer.GetWindowsShellFolder(const Name:LongString):LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellFolder(sb(Name)));
end;

function  TSysApiServer.GetWindowsShellDesktop:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellDesktop);
end;

function  TSysApiServer.GetWindowsShellPrograms:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellPrograms);
end;

function  TSysApiServer.GetWindowsShellStartup:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellStartup);
end;

function  TSysApiServer.GetWindowsShellStartMenu:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellStartMenu);
end;

function  TSysApiServer.GetWindowsShellFavorites:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellFavorites);
end;

function  TSysApiServer.GetWindowsShellFonts:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellFonts);
end;

function  TSysApiServer.GetWindowsShellHistory:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellHistory);
end;

function  TSysApiServer.GetWindowsShellPersonal:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellPersonal);
end;

function  TSysApiServer.GetWindowsShellSendTo:LongString;
begin
 Result:=sb(_crw_fio.GetWindowsShellSendTo);
end;

procedure TSysApiServer.CreateFileLink(const ObjectPath, LinkPath, Description, Params: LongString);
begin
 _crw_fio.CreateFileLink(sb(ObjectPath),sb(LinkPath),sb(Description),sb(Params));
end;

function  TSysApiServer.Point2I(x,y:Integer):TPoint2I;
begin
 Result.x:=x;
 Result.y:=y;
end;

function  TSysApiServer.Point2D(x,y:Double):TPoint2D;
begin
 Result.x:=x;
 Result.y:=y;
end;

function  TSysApiServer.Point3D(x,y,z:Double):TPoint3D;
begin
 Result.x:=x;
 Result.y:=y;
 Result.z:=z;
end;

function  TSysApiServer.Rect2I(const a,b:TPoint2I):TRect2I;
begin
 Result.a:=a;
 Result.b:=b;
end;

function  TSysApiServer.Rect2D(const a,b:TPoint2D):TRect2D;
begin
 Result.a:=a;
 Result.b:=b;
end;

function  TSysApiServer.IsEqualPoint2I(const p1,p2:TPoint2I):Boolean;
begin
 Result:=(p1.x=p2.x) and (p1.y=p2.y);
end;

function  TSysApiServer.IsEqualPoint2D(const p1,p2:TPoint2D; AbsEps:Double=0; RelEps:Double=1E-14):Boolean;
begin
 Result:=(abs(p1.x-p2.x)<=Precision(AbsEps,RelEps,abs(p1.x)+abs(p2.x))) and
         (abs(p1.y-p2.y)<=Precision(AbsEps,RelEps,abs(p1.y)+abs(p2.y)));
end;

function  TSysApiServer.IsEqualRect2I(const r1,r2:TRect2I):Boolean;
begin
 Result:=IsEqualPoint2I(r1.a,r2.a) and IsEqualPoint2I(r1.b,r2.b);
end;

function  TSysApiServer.IsEqualRect2D(const r1,r2:TRect2D; AbsEps:Double=0; RelEps:Double=1E-14):Boolean;
begin
 Result:=IsEqualPoint2D(r1.a,r2.a,AbsEps,RelEps) and IsEqualPoint2D(r1.b,r2.b,AbsEps,RelEps);
end;

function  TSysApiServer.IsEmptyRect2I(const r:TRect2I):Boolean;
begin
 with r do Result:=(a.x>=b.x) or (a.y>=b.y);
end;

function  TSysApiServer.IsEmptyRect2D(const r:TRect2D; AbsEps:Double=0; RelEps:Double=1E-14):Boolean;
begin
 with r do
 Result:=(a.x>=b.x) or (a.y>=b.y) or
         (abs(b.x-a.x)<=Precision(AbsEps,RelEps,abs(a.x)+abs(b.x))) or
         (abs(b.y-a.y)<=Precision(AbsEps,RelEps,abs(a.y)+abs(b.y)));
end;

function  TSysApiServer.Rect2IContainsPoint(const r:TRect2I; const p:TPoint2I):Boolean;
begin
 with r do Result:=(p.x>=a.x) and (p.x<=b.x) and (p.y>=a.y) and (p.y<=b.y);
end;

function  TSysApiServer.Rect2DContainsPoint(const r:TRect2D; const p:TPoint2D):Boolean;
begin
 with r do Result:=(p.x>=a.x) and (p.x<=b.x) and (p.y>=a.y) and (p.y<=b.y);
end;

function  TSysApiServer.Rect2IIntersection(const R1,R2:TRect2I):TRect2I;
begin
 if R1.a.x>R2.a.x then Result.a.x:=R1.a.x else Result.a.x:=R2.a.x;
 if R1.a.y>R2.a.y then Result.a.y:=R1.a.y else Result.a.y:=R2.a.y;
 if R1.b.x<R2.b.x then Result.b.x:=R1.b.x else Result.b.x:=R2.b.x;
 if R1.b.y<R2.b.y then Result.b.y:=R1.b.y else Result.b.y:=R2.b.y;
end;

function  TSysApiServer.Rect2DIntersection(const R1,R2:TRect2D):TRect2D;
begin
 if R1.a.x>R2.a.x then Result.a.x:=R1.a.x else Result.a.x:=R2.a.x;
 if R1.a.y>R2.a.y then Result.a.y:=R1.a.y else Result.a.y:=R2.a.y;
 if R1.b.x<R2.b.x then Result.b.x:=R1.b.x else Result.b.x:=R2.b.x;
 if R1.b.y<R2.b.y then Result.b.y:=R1.b.y else Result.b.y:=R2.b.y;
end;

function  TSysApiServer.Rect2IUnion(const R1,R2:TRect2I):TRect2I;
begin
 if R1.a.x<R2.a.x then Result.a.x:=R1.a.x else Result.a.x:=R2.a.x;
 if R1.a.y<R2.a.y then Result.a.y:=R1.a.y else Result.a.y:=R2.a.y;
 if R1.b.x>R2.b.x then Result.b.x:=R1.b.x else Result.b.x:=R2.b.x;
 if R1.b.y>R2.b.y then Result.b.y:=R1.b.y else Result.b.y:=R2.b.y;
end;

function  TSysApiServer.Rect2DUnion(const R1,R2:TRect2D):TRect2D;
begin
 if R1.a.x<R2.a.x then Result.a.x:=R1.a.x else Result.a.x:=R2.a.x;
 if R1.a.y<R2.a.y then Result.a.y:=R1.a.y else Result.a.y:=R2.a.y;
 if R1.b.x>R2.b.x then Result.b.x:=R1.b.x else Result.b.x:=R2.b.x;
 if R1.b.y>R2.b.y then Result.b.y:=R1.b.y else Result.b.y:=R2.b.y;
end;

function  TSysApiServer.Rect2IValidate(const r:TRect2I):TRect2I;
var d:Integer;
begin
 Result:=r;
 with Result do begin
  if b.x<a.x then begin d:=a.x; a.x:=b.x; b.x:=d; end;
  if b.y<a.y then begin d:=a.y; a.y:=b.y; b.y:=d; end;
 end;
end;

function  TSysApiServer.Rect2DValidate(const r:TRect2D):TRect2D;
var d:Double;
begin
 Result:=r;
 with Result do begin
  if b.x<a.x then begin d:=a.x; a.x:=b.x; b.x:=d; end;
  if b.y<a.y then begin d:=a.y; a.y:=b.y; b.y:=d; end;
 end;
end;

procedure TSysApiServer.Point2IMove(var p:TPoint2I; dx,dy:Integer);
begin
 p.x:=p.x+dx;
 p.y:=p.y+dy;
end;

procedure TSysApiServer.Point2DMove(var p:TPoint2D; dx,dy:Double);
begin
 p.x:=p.x+dx;
 p.y:=p.y+dy;
end;

procedure TSysApiServer.Rect2IMove(var r:TRect2I; dx,dy:Integer);
begin
 Point2IMove(r.a,dx,dy);
 Point2IMove(r.b,dx,dy);
end;

procedure TSysApiServer.Rect2DMove(var r:TRect2D; dx,dy:Double);
begin
 Point2DMove(r.a,dx,dy);
 Point2DMove(r.b,dx,dy);
end;

procedure TSysApiServer.Point2IMult(var p:TPoint2I; mx,my:Integer);
begin
 p.x:=p.x*mx;
 p.y:=p.y*my;
end;

procedure TSysApiServer.Point2DMult(var p:TPoint2D; mx,my:Double);
begin
 p.x:=p.x*mx;
 p.y:=p.y*my;
end;

procedure TSysApiServer.Rect2IMult(var r:TRect2I; mx,my:Integer);
begin
 Point2IMult(r.a,mx,my);
 Point2IMult(r.b,mx,my);
end;

procedure TSysApiServer.Rect2DMult(var r:TRect2D; mx,my:Double);
begin
 Point2DMult(r.a,mx,my);
 Point2DMult(r.b,mx,my);
end;

procedure TSysApiServer.Rect2IGrow(var r:TRect2I; dx,dy:Integer);
begin
 Point2IMove(r.a,-dx,-dy);
 Point2IMove(r.b,+dx,+dy);
end;

procedure TSysApiServer.Rect2DGrow(var r:TRect2D; dx,dy:double);
begin
 Point2DMove(r.a,-dx,-dy);
 Point2DMove(r.b,+dx,+dy);
end;

function  TSysApiServer.Rect2ISize(const r:TRect2I):TPoint2I;
begin
 Result.x:=r.b.x-r.a.x;
 Result.y:=r.b.y-r.a.y;
end;

function  TSysApiServer.Rect2DSize(const r:TRect2D):TPoint2D;
begin
 Result.x:=r.b.x-r.a.x;
 Result.y:=r.b.y-r.a.y;
end;

function  TSysApiServer.Rect2ICenter(const r:TRect2I):TPoint2I;
begin
 Result.x:=(r.b.x+r.a.x) shr 1;
 Result.y:=(r.b.y+r.a.y) shr 1;
end;

function  TSysApiServer.Rect2DCenter(const r:TRect2D):TPoint2D;
begin
 Result.x:=(r.b.x+r.a.x)*0.5;
 Result.y:=(r.b.y+r.a.y)*0.5;
end;

function  TSysApiServer.CreateScriptInterpreter:IScriptInterpreter;
var Obj:TScriptInterpreterServer; Intf:IScriptInterpreter;
begin
 Result:=nil;
 Obj:=TScriptInterpreterServer.Create;
 if Assigned(Obj) and Obj.GetInterface(IID_IScriptInterpreter,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function  TSysApiServer.GetBitMask(BitNumber:Integer):Integer;
begin
 Result:=_crw_ef.GetBitMask(BitNumber);
end;

function  TSysApiServer.Precision(AbsEps,RelEps,Value:Double):Double;
begin
 Result:=_crw_ef.Precision(AbsEps,RelEps,Value);
end;

function  TSysApiServer.MaxIntSingle:Single;
begin
 Result:=_crw_ef.MaxIntSingle;
end;

function  TSysApiServer.MaxIntDouble:Double;
begin
 Result:=_crw_ef.MaxIntDouble;
end;

function  TSysApiServer.MaxIntExtended:Extended;
begin
 Result:=_crw_ef.MaxIntExtended;
end;

function  TSysApiServer.fscale(X:Extended; N:integer): Extended;
begin
 Result:=_crw_ef.fscale(X,N);
end;

procedure TSysApiServer.fxtract(X: Extended; var Mantissa: Extended; var Exponent: Integer);
begin
 _crw_ef.fxtract(X,Mantissa,Exponent);
end;

function  TSysApiServer.fbitcompress(Argument:Extended; NumberOfSignificantBits:integer):Extended;
begin
 Result:=_crw_ef.fbitcompress(Argument,NumberOfSignificantBits);
end;

function  TSysApiServer.frelcompress(Arument,ScaleFactor:Extended):Extended;
begin
 Result:=_crw_ef.frelcompress(Arument,ScaleFactor);
end;

function  TSysApiServer.fabscompress(Argument,ScaleFactor:Extended):Extended;
begin
 Result:=_crw_ef.fabscompress(Argument,ScaleFactor);
end;

function  TSysApiServer.Gamma(x:Extended):Extended;
begin
 Result:=_crw_ef.Gamma(x);
end;

function  TSysApiServer.GammaLn(x:Extended):Extended;
begin
 Result:=_crw_ef.GammaLn(x);
end;

function  TSysApiServer.GammaInc(x,a:Extended):Extended;
begin
 Result:=_crw_ef.GammaInc(x,a);
end;

function  TSysApiServer.GammaDistr(x,a,b:Extended):Extended;
begin
 Result:=_crw_ef.GammaDistr(x,a,b);
end;

function  TSysApiServer.GammaCumDistr(x,a,b:Extended):Extended;
begin
 Result:=_crw_ef.GammaCumDistr(x,a,b);
end;

function  TSysApiServer.GammaCumDistrInv(p,a,b:Extended):Extended;
begin
 Result:=_crw_ef.GammaCumDistrInv(p,a,b);
end;

function  TSysApiServer.Chi2Distr(x,v:Extended):Extended;
begin
 Result:=_crw_ef.Chi2Distr(x,v);
end;

function  TSysApiServer.Chi2CumDistr(x,v:Extended):Extended;
begin
 Result:=_crw_ef.Chi2CumDistr(x,v);
end;

function  TSysApiServer.Chi2CumDistrInv(p,v:Extended):Extended;
begin
 Result:=_crw_ef.Chi2CumDistrInv(p,v);
end;

function  TSysApiServer.Trapezium(x1,y1,x2,y2:double):double;
begin
 Result:=_crw_ef.Trapezium(x1,y1,x2,y2);
end;

function  TSysApiServer.BesselN(N:integer; X:Double;Modify:boolean=false):Double;
begin
 Result:=_crw_ef.BesselN(N,X,Modify);
end;

function  TSysApiServer.GaussSolverWithPartialPivoting(N,M:Integer; var A,X,Y:array of Double; Eps:Double=1E-14):Integer;
begin
 Result:=_crw_lsqpoly.Partial_Pivoting(N,M,A,X,Y,Eps);
end;

function  TSysApiServer.CreatePolynom(aPower:Integer=1; aCenter:Double=0; aScale:Double=1):IPolynomWrap;
var Obj:TPolynomWrapServer; Intf:IPolynomWrap;
begin
 Result:=nil;
 Obj:=TPolynomWrapServer.Create(aPower,aCenter,aScale);
 if Assigned(Obj) and Obj.GetInterface(IID_IPolynomWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function TSysApiServer.GenMin(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer; MaxCount:Integer;
                  Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double; var v:array of Double; nv:Integer):Integer;
begin
 Result:=_crw_funmin.GenMin(Problem,Report,n,x,f,g,Custom,Count,MaxCount,Method,tolx,tolf,tolg,inf,Size,Step,EvalTol,v,nv);
end;

function TSysApiServer.GenMinFix(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer; MaxCount:Integer;
                  Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double; var v:array of Double; nv:Integer;
                  var Fix:array of Boolean):Integer;
begin
 Result:=_crw_funmin.GenMinFix(Problem,Report,n,x,f,g,Custom,Count,MaxCount,Method,tolx,tolf,tolg,inf,Size,Step,
                           EvalTol,v,nv,Fix);
end;

function TSysApiServer.GenMinScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer; MaxCount:Integer;
                  Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double; var v:array of Double; nv:Integer;
                  fscale:Double; var xLo,xHi:array of Double):Integer;
begin
 Result:=_crw_funmin.GenMinScaled(Problem,Report,n,x,f,g,Custom,Count,MaxCount,Method,tolx,tolf,tolg,inf,Size,Step,
                              EvalTol,v,nv,fscale,xLo,xHi);
end;

function TSysApiServer.GenMinFixScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer; MaxCount:Integer;
                  Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double; var v:array of Double; nv:Integer;
                  var Fix:array of Boolean; fscale:Double; var xLo,xHi:array of Double):Integer;
begin
 Result:=_crw_funmin.GenMinFixScaled(Problem,Report,n,x,f,g,Custom,Count,MaxCount,Method,tolx,tolf,tolg,inf,Size,Step,
                                 EvalTol,v,nv,Fix,fscale,xLo,xHi);
end;

function  TSysApiServer.LSQFit(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; Report:TFitReport;
                   n:Integer; var x:array of Double; var f:Double; var g:array of Double;
                   Custom:Pointer; var Count:Integer; MaxCount:Integer; Method:Integer;
                   tolx,tolf,tolg,Inf,Size,Step,EvalTol:Double; var a:array of Double;
                   var Fix:array of Boolean; fscale:Double; var xLo:array of Double;
                   var xHi:array of Double):Integer;
begin
 Result:=_crw_funmin.LSQFit(m,t,y,w,Fit,Report,n,x,f,g,Custom,Count,MaxCount,Method,
                        tolx,tolf,tolg,Inf,Size,Step,EvalTol,a,Fix,fscale,xLo,xHi);
end;

function  TSysApiServer.LSQFitSumOfSquares(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; n:Integer;
                               var x:array of Double; var f:Double; var g:array of Double;
                               Custom:Pointer):Integer;
begin
 Result:=_crw_funmin.LSQFitSumOfSquares(m,t,y,w,Fit,n,x,f,g,Custom);
end;

function TSysApiServer.Rfa:IRfaDataBaseWrap;
begin
 if not Assigned(myRfa) then begin
  myRfa:=TRfaDataBaseWrapServer.Create(Self);
  myRfa.Master:=@myRfa;
  if not myRfa.GetInterface(IID_IRfaDataBaseWrap,myRfa.myIFace)
  then myRfa.myIFace:=nil;
 end;
 if Assigned(myRfa)
 then Result:=myRfa.myIFace
 else Result:=nil;
end;

function  TSysApiServer.Sort(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                 KeyOf:TKeyOfFunction; SwapData:TSwapProcedure;
                 Index:PIntegerArray; Method:Integer=0):Boolean;
begin
 Result:=_crw_sort.Sort(Count,Data,Compare,KeyOf,SwapData,Pointer(Index),TSortMethod(Method));
end;

function  TSysApiServer.QuickFind(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                                  KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                                  Index:PIntegerArray=nil):Integer;
begin
 Result:=_crw_sort.QuickFind(Count,Data,Compare,KeyOf,KeyToSearch,Pointer(Index));
end;

function  TSysApiServer.QuickSearch(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                                    KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                                    var KeyIndex:Integer; Index:PIntegerArray=nil;
                                    Duplicates:Boolean=false):Boolean;
begin
 Result:=_crw_sort.QuickSearch(Count,Data,Compare,KeyOf,KeyToSearch,KeyIndex,Pointer(Index),Duplicates);
end;

function TSysApiServer.FindDoubleIndex(N:Integer; const x:array of Double; at:Double; Index:PIntegerArray=nil):Integer;
begin
 Result:=_crw_sort.FindIndex(N,x,at,Pointer(Index));
end;

function  TSysApiServer.LinearInterpolation(const x,y:array of Double; N:Integer; t:Double):Double;
begin
 Result:=_crw_spline.LinearInterpolation(x,y,N,t);
end;

function  TSysApiServer.EvaluatePolynomAt(const P:array of Double; N:Integer; t:Double):Double;
begin
 Result:=_crw_spline.EvalPoly(P,N,t);
end;

function  TSysApiServer.FindLeastSquaresPolynom(var P:array of Double; N:Integer; X0,Scale:Double;
                                                const x,y:array of Double; M:Integer):Boolean;
var i:Integer; Poly:_crw_alloc.PDoubleArray;
begin
 Result:=false;
 Poly:=_crw_spline.FindPoly(N,X0,Scale,x,y,M);
 if Assigned(Poly) then begin
  for i:=0 to N-1 do P[i]:=Poly[i];
  Deallocate(Pointer(Poly));
  Result:=true;
 end;
end;

function  TSysApiServer.EvaluateSplineAt(N:Integer; const x,z,k:array of Double; t:Double; NDeriv:Integer=0):Double;
begin
 Result:=_crw_spline.EvalSpline(N,x,z,k,t,NDeriv);
end;

function  TSysApiServer.FindSmoothingSpline(N:Integer; const x,y,p:array of Double; var z,k:array of Double;
                                            nL:Integer=2; pL:Double=0; nR:Integer=2; pR:Double=0):Boolean;
begin
 Result:=_crw_spline.LS_Spline(N,x,y,p,z,k,nL,pL,nR,pR);
end;

function  TSysApiServer.WindowedKernelSmoothing(const x,y:array of Double; N:Integer;
                                    t,Window:Double; Power:Integer; Kernel:TSmoothKernel;
                                    Custom:Pointer=nil):Double;
begin
 Result:=_crw_spline.WindowedKernelSmoothing(x,y,N,t,Window,Power,Kernel,Custom);
end;

function  TSysApiServer.WindowedIntegrationSmoothing(const x,y:array of Double; N:Integer;
                                         t,Window:Double; Kernel:TSmoothKernel;
                                         Custom:Pointer=nil):Double;
begin
 Result:=_crw_spline.WindowedIntegrationSmoothing(x,y,N,t,Window,Kernel,Custom);
end;

procedure TSysApiServer.Solve3DiagonalSetByProgonka(N:Integer; var a,b,c,x,d:array of Double);
begin
 _crw_spline.Progonka(N,a,b,c,x,d);
end;

procedure TSysApiServer.Solve3DiagonalSetByGauss(N:Integer; var a,b,c,x,d:array of Double);
begin
 _crw_spline.Solve3DiagonalSet(N,a,b,c,x,d);
end;

procedure TSysApiServer.Solve5DiagonalSetByGauss(N:Integer; var a,b,c,e,g,x,d:array of Double);
begin
 _crw_spline.Solve5DiagonalSet(N,a,b,c,e,g,x,d);
end;

function TSysApiServer.CreateLongIntVector(aLength:Integer; aOrigin:Integer=0; aExceptions:Boolean=true):ILongIntVectorWrap;
var Obj:TLongIntVectorWrapServer; Intf:ILongIntVectorWrap;
begin
 Result:=nil;
 Obj:=TLongIntVectorWrapServer.Create(NewLongIntVector(aLength,aOrigin),true);
 if Assigned(Obj) then Obj.Exceptions:=aExceptions;
 if Assigned(Obj) and Obj.GetInterface(IID_ILongIntVectorWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function TSysApiServer.CreateDoubleVector(aLength:Integer; aOrigin:Integer=0; aExceptions:Boolean=true):IDoubleVectorWrap;
var Obj:TDoubleVectorWrapServer; Intf:IDoubleVectorWrap;
begin
 Result:=nil;
 Obj:=TDoubleVectorWrapServer.Create(NewDoubleVector(aLength,aOrigin),true);
 if Assigned(Obj) then Obj.Exceptions:=aExceptions;
 if Assigned(Obj) and Obj.GetInterface(IID_IDoubleVectorWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function  TSysApiServer.CreateLongIntMatrix(aRows,aColumns:Integer; aOrigin:Integer=0; aExceptions:Boolean=true):ILongIntMatrixWrap;
var Obj:TLongIntMatrixWrapServer; Intf:ILongIntMatrixWrap;
begin
 Result:=nil;
 Obj:=TLongIntMatrixWrapServer.Create(NewLongIntMatrix(aRows,aColumns,aOrigin),true);
 if Assigned(Obj) then Obj.Exceptions:=aExceptions;
 if Assigned(Obj) and Obj.GetInterface(IID_ILongIntMatrixWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function  TSysApiServer.CreateDoubleMatrix(aRows,aColumns:Integer; aOrigin:Integer=0; aExceptions:Boolean=true):IDoubleMatrixWrap;
var Obj:TDoubleMatrixWrapServer; Intf:IDoubleMatrixWrap;
begin
 Result:=nil;
 Obj:=TDoubleMatrixWrapServer.Create(NewDoubleMatrix(aRows,aColumns,aOrigin),true);
 if Assigned(Obj) then Obj.Exceptions:=aExceptions;
 if Assigned(Obj) and Obj.GetInterface(IID_IDoubleMatrixWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function TSysApiServer.LinearProgrammingBySimplex(Problem:IDoubleVectorWrap; MatrCoef:IDoubleMatrixWrap;
                            Signums:ILongIntVectorWrap; RightSide:IDoubleVectorWrap;
                            var ErrorCode:Integer; PrintMode:Word; var OutPutFile:Text;
                            Zero:Double=1E-12):IDoubleMatrixWrap;
var m:TDoubleMatrix; sp:Double;
begin
 sp:=SimplexPrecision;
 SimplexPrecision:=Zero;
 m:=_crw_simplex.Simplex((Problem   as TDoubleVectorWrapServer).TheVector,
                     (MatrCoef  as TDoubleMatrixWrapServer).TheMatrix,
                     (Signums   as TLongIntVectorWrapServer).TheVector,
                     (RightSide as TDoubleVectorWrapServer).TheVector,
                     ErrorCode,PrintMode,OutPutFile);
 if Assigned(m)
 then Result:=TDoubleMatrixWrapServer.Create(m,true)
 else Result:=nil;
 SimplexPrecision:=sp;
end;

function TSysApiServer.CreateSingularDecomposition(A:IDoubleMatrixWrap):ISVDWrap;
var Obj:TSVDWrapServer; Intf:ISVDWrap;
begin
 Result:=nil;
 Obj:=TSVDWrapServer.Create(A);
 if Assigned(Obj) and Obj.GetInterface(IID_ISVDWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function TSysApiServer.CreateFifo(aSize:LongInt=1024; const aName:LongString=''):IFifoWrap;
var Obj:TFifoWrapServer; Intf:IFifoWrap;
begin
 Result:=nil;
 Obj:=TFifoWrapServer.Create(NewFifo(aSize,sb(aName)),true);
 if Assigned(Obj) and Obj.GetInterface(IID_IFifoWrap,Intf)
 then Result:=Intf
 else Obj.Free;
end;

function  TSysApiServer.MkSecNow:Extended;
begin
 Result:=_crw_rtc.MkSecNow;
end;

function  TSysApiServer.LockedAdd(var Target: Integer; Value: Integer): Integer;
begin
 Result:=_crw_alloc.LockedAdd(Target,Value);
end;

function  TSysApiServer.LockedCompareExchange(var Target: Integer; Exch, Comp: Integer): Integer;
begin
 Result:=_crw_alloc.LockedCompareExchange(Target,Exch,Comp);
end;

function  TSysApiServer.LockedDec(var Target: Integer): Integer;
begin
 Result:=_crw_alloc.LockedDec(Target);
end;

function  TSysApiServer.LockedExchange(var Target: Integer; Value: Integer): Integer;
begin
 Result:=_crw_alloc.LockedExchange(Target,Value);
end;

function  TSysApiServer.LockedExchangeAdd(var Target: Integer; Value: Integer): Integer;
begin
 Result:=_crw_alloc.LockedExchangeAdd(Target,Value);
end;

function  TSysApiServer.LockedExchangeDec(var Target: Integer): Integer;
begin
 Result:=_crw_alloc.LockedExchangeDec(Target);
end;

function  TSysApiServer.LockedExchangeInc(var Target: Integer): Integer;
begin
 Result:=_crw_alloc.LockedExchangeInc(Target);
end;

function  TSysApiServer.LockedExchangeSub(var Target: Integer; Value: Integer): Integer;
begin
 Result:=_crw_alloc.LockedExchangeSub(Target,Value);
end;

function  TSysApiServer.LockedInc(var Target: Integer): Integer;
begin
 Result:=_crw_alloc.LockedInc(Target);
end;

function  TSysApiServer.LockedSub(var Target: Integer; Value: Integer): Integer;
begin
 Result:=_crw_alloc.LockedSub(Target,Value);
end;

function TSysApiServer.RemoveBrackets(const s:LongString; const Brackets:LongString):LongString;
begin
 Result:=sb(_crw_str.RemoveBrackets(sb(s),sb(Brackets)));
end;

function TSysApiServer.Mime_Encode(const S:LongString):LongString;
begin
 Result:=sb(_crw_base64.Mime_Encode(sb(S)));
end;

function TSysApiServer.Mime_Decode(const S:LongString):LongString;
begin
 Result:=sb(_crw_base64.Mime_Decode(sb(S)));
end;

function TSysApiServer.IsSameText(const S1,S2:LongString):Boolean;
begin
 Result:=_crw_str.IsSameText(sb(S1),sb(S2));
end;

function TSysApiServer.IsUnix:Boolean;
begin
 Result:=_crw_alloc.IsUnix;
end;

function TSysApiServer.IsLinux:Boolean;
begin
 Result:=_crw_alloc.IsLinux;
end;

function TSysApiServer.IsWindows:Boolean;
begin
 Result:=_crw_alloc.IsWindows;
end;

function TSysApiServer.IsWin32:Boolean;
begin
 Result:=_crw_alloc.IsWin32;
end;

function TSysApiServer.IsWin64:Boolean;
begin
 Result:=_crw_alloc.IsWin64;
end;

function TSysApiServer.IsCpu16:Boolean;
begin
 Result:=_crw_alloc.IsCpu16;
end;

function TSysApiServer.IsCpu32:Boolean;
begin
 Result:=_crw_alloc.IsCpu32;
end;

function TSysApiServer.IsCpu64:Boolean;
begin
 Result:=_crw_alloc.IsCpu64;
end;

function TSysApiServer.IsCpuBE:Boolean;
begin
 Result:=_crw_alloc.IsCpuBE;
end;

function TSysApiServer.IsCpuLE:Boolean;
begin
 Result:=_crw_alloc.IsCpuLE;
end;

function TSysApiServer.IsSysUtf8:Boolean;
begin
 Result:=_crw_alloc.IsSysUtf8;
end;

function TSysApiServer.IsFileNameCaseSensitive:Boolean;
begin
 Result:=_crw_alloc.IsFileNameCaseSensitive;
end;

function TSysApiServer.OsName(Mode:Integer=1):LongString;
begin
 Result:=sb(_crw_alloc.OsName(Mode));
end;

function TSysApiServer.GetFpcVersion:LongString;
begin
 Result:=sb(_crw_alloc.GetFpcVersion);
end;

function TSysApiServer.GetFpcTargetOS:LongString;
begin
 Result:=sb(_crw_alloc.GetFpcTargetOS);
end;

function TSysApiServer.GetFpcTargetCPU:LongString;
begin
 Result:=sb(_crw_alloc.GetFpcTargetCPU);
end;

function TSysApiServer.GetFpcTargetPlatform:LongString;
begin
 Result:=sb(_crw_alloc.GetFpcTargetPlatform);
end;

function TSysApiServer.IsMainThread:Boolean;
begin
 Result:=_crw_alloc.IsMainThread;
end;

function TSysApiServer.GetCurrentProcessId:TPid;
begin
 Result:=_crw_alloc.GetCurrentProcessId;
end;

function TSysApiServer.GetAllocMemCount:SizeInt;
begin
 Result:=_crw_alloc.GetAllocMemCount;
end;

function TSysApiServer.GetAllocMemSize:SizeInt;
begin
 Result:=_crw_alloc.GetAllocMemSize;
end;

procedure TSysApiServer.ResetFpu;
begin
 FpuSetExceptions(false);
 FpuClearExceptions;
end;

function TSysApiServer.StrToWide(const s:LongString):WideString;
begin
 Result:=_crw_str.StrToWide(sb(s));
end;

function TSysApiServer.WideToStr(const s:WideString):LongString;
begin
 Result:=sb(_crw_str.WideToStr(s));
end;

function TSysApiServer.ConvertCP(const s:LongString; cpFrom,cpTo,cpSet:Word; silent:Boolean=false):LongString;
begin
 Result:=sb(_crw_str.ConvertCP(sb(s),cpFrom,cpTo,cpSet,silent));
end;

function TSysApiServer.AdaptFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
begin
 Result:=sb(_crw_str.AdaptFileName(sb(FileName),Mode));
end;

function TSysApiServer.AdaptExeFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
begin
 Result:=sb(_crw_str.AdaptExeFileName(sb(FileName),Mode));
end;

function TSysApiServer.AdaptDllFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
begin
 Result:=sb(_crw_str.AdaptDllFileName(sb(FileName),Mode));
end;

function TSysApiServer.AdaptLnkFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
begin
 Result:=sb(_crw_str.AdaptLnkFileName(sb(FileName),Mode));
end;

function TSysApiServer.GetTickCount:Cardinal;
begin
 Result:=_crw_rtc.GetTickCount;
end;

function TSysApiServer.GetTickCount64:QWord;
begin
 Result:=_crw_rtc.GetTickCount64;
end;

function TSysApiServer.ExtractBaseName(const FileName:LongString):LongString;
begin
 Result:=sb(_crw_str.ExtractBaseName(sb(FileName)));
end;

function TSysApiServer.ExtractDllBaseName(const FileName:LongString):LongString;
begin
 Result:=sb(_crw_str.ExtractDllBaseName(sb(FileName)));
end;

function TSysApiServer.ExtractFileDir(const FileName:LongString):LongString;
begin
 Result:=sb(_crw_str.ExtractFileDir(sb(FileName)));
end;

function TSysApiServer.IsLexeme(const arg:LongString; typ:Integer):Boolean;
begin
 Result:=_crw_str.IsLexeme(sb(arg),typ);
end;

function TSysApiServer.ExtractNameValuePair(const arg:LongString; out Name,Value:LongString;
                                const Sign:Char='='; Mode:Integer=3):Integer;
begin
 Result:=_crw_str.ExtractNameValuePair(sb(arg),Name,Value,Sign,Mode);
end;

function TSysApiServer.PosI(const Sub:LongString; const Str:LongString):SizeInt;
begin
 Result:=_crw_str.PosI(sb(Sub),sb(Str));
end;

function TSysApiServer.PosEx(const Sub,Str:LongString; StartPos:SizeInt):SizeInt;
begin
 Result:=_crw_str.PosEx(sb(Sub),sb(Str),StartPos);
end;

function TSysApiServer.LastPos(const Sub,Str:LongString):SizeInt;
begin
 Result:=_crw_str.LastPos(sb(Sub),sb(Str));
end;

function TSysApiServer.CountPos(const Sub,Str:LongString):SizeInt;
begin
 Result:=_crw_str.CountPos(sb(Sub),sb(Str));
end;

function TSysApiServer.NthPos(const Sub,Str:LongString; n:SizeInt):SizeInt;
begin
 Result:=_crw_str.NthPos(sb(Sub),sb(Str),n);
end;

function TSysApiServer.PosEol(const Buf:LongString; Start:SizeInt=1; SkipLines:SizeInt=0):SizeInt;
begin
 Result:=_crw_str.PosEol(sb(Buf),Start,SkipLines);
end;

function TSysApiServer.TailStr(const S:LongString; Pos:SizeInt):LongString;
begin
 Result:=sb(_crw_str.TailStr(sb(S),Pos));
end;

function TSysApiServer.StrAheadOf(const S:LongString; Delim:Char):LongString;
begin
 Result:=sb(_crw_str.StrAheadOf(sb(S),Delim));
end;

function TSysApiServer.StrAfterOf(const S:LongString; Delim:Char):LongString;
begin
 Result:=sb(_crw_str.StrAfterOf(sb(S),Delim));
end;

function TSysApiServer.CenterPad(const S:LongString; Width:SizeInt; Ch:Char=' '):LongString;
begin
 Result:=sb(_crw_str.CenterPad(sb(S),Width,Ch));
end;

function TSysApiServer.TrimDef(const S,Def:LongString):LongString;
begin
 Result:=sb(_crw_str.TrimDef(sb(S),sb(Def)));
end;

function TSysApiServer.SortTextLines(const aTextLines:LongString;
                       Comparator:TStringListSortCompare=nil):LongString;
begin
 Result:=sb(_crw_str.SortTextLines(sb(aTextLines),Comparator));
end;

function TSysApiServer.EnsureEndingEol(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.EnsureEndingEol(sb(S)));
end;

function TSysApiServer.EnsureHeapString(const S:LongString):LongString;
begin
 Result:=sb(_crw_str.EnsureHeapString(sb(S)));
end;

function TSysApiServer.CookieScan(const Buff,Name:LongString; Mode:Integer=csm_Default):LongString;
begin
 Result:=sb(_crw_str.CookieScan(sb(Buff),sb(Name),Mode));
end;

function TSysApiServer.SkipWords(N:Integer; const S:LongString;
                                 const WordDelims:TCharSet):LongString;
begin
 Result:=sb(_crw_str.SkipWords(N,sb(S),WordDelims));
end;

function TSysApiServer.PhraseCount(const S:LongString; const Delims:TCharSet;
                     Quotes:LongString=QuoteMark+Apostrophe):Integer;
begin
 Result:=_crw_str.PhraseCount(sb(S),Delims,sb(Quotes));
end;

function TSysApiServer.ExtractPhrase(N:Integer; const S:LongString; const Delims:TCharSet;
                       Quotes:LongString=QuoteMark+Apostrophe):LongString;
begin
 Result:=sb(_crw_str.ExtractPhrase(N,sb(S),Delims,sb(Quotes)));
end;

function TSysApiServer.SkipPhrases(N:Integer; const S:LongString; const Delims:TCharSet;
                     Quotes:LongString=QuoteMark+Apostrophe):LongString;
begin
 Result:=sb(_crw_str.SkipPhrases(N,sb(S),Delims,sb(Quotes)));
end;

function TSysApiServer.PhraseListToTextLines(const S:LongString; const Delims:TCharSet;
                               Quotes:LongString=QuoteMark+Apostrophe):LongString;
begin
 Result:=sb(_crw_str.PhraseListToTextLines(sb(S),Delims,sb(Quotes)));
end;

function TSysApiServer.ScanVarRecord(svMode:Cardinal; Str:PChar; const Format:LongString; var Data):PChar;
begin
 Result:=_crw_str.ScanVarRecord(svMode,Str,sb(Format),Data);
end;

function TSysApiServer.ScanVarBoolean(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Boolean):PChar;
begin
 Result:=_crw_str.ScanVarBoolean(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.ScanVarWord(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Word):PChar;
begin
 Result:=_crw_str.ScanVarWord(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.ScanVarInteger(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Integer):PChar;
begin
 Result:=_crw_str.ScanVarInteger(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.ScanVarDouble(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Double):PChar;
begin
 Result:=_crw_str.ScanVarDouble(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.ScanVarAlpha(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
begin
 Result:=_crw_str.ScanVarAlpha(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.ScanVarString(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
begin
 Result:=_crw_str.ScanVarString(svMode,Str,sb(Key),Data);
end;

function TSysApiServer.AddPathDelim(const DirName:LongString; Delim:Char=PathDelim):LongString;
begin
 Result:=sb(_crw_str.AddPathDelim(sb(DirName),Delim));
end;

function TSysApiServer.DropPathDelim(const DirName:LongString):LongString;
begin
 Result:=sb(_crw_str.DropPathDelim(sb(DirName)));
end;

function TSysApiServer.ValidatePathDelim(const FileName:LongString; Delim:Char=PathDelim):LongString;
begin
 Result:=sb(_crw_str.ValidatePathDelim(sb(FileName),Delim));
end;

function TSysApiServer.StrFetch(const s:LongString; i:SizeInt):AnsiChar;
begin
 Result:=_crw_str.StrFetch(sb(s),i);
end;

function TSysApiServer.AnsiDeQuotedStr(const s:LongString; q:Char):LongString;
begin
 Result:=sb(_crw_str.AnsiDeQuotedStr(sb(s),q));
end;

function TSysApiServer.AnsiSkipQuotedStr(const s:LongString; q:Char):LongString;
begin
 Result:=sb(_crw_str.AnsiSkipQuotedStr(sb(s),q));
end;

function TSysApiServer.ExtractFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.ExtractFirstParam(sb(s),quote,Spaces));
end;

function TSysApiServer.SkipFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.SkipFirstParam(sb(s),quote,Spaces));
end;

function TSysApiServer.AnsiQuotedIfNeed(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.AnsiQuotedIfNeed(sb(s),quote,Spaces));
end;

function TSysApiServer.QArg(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.QArg(sb(s),quote,Spaces));
end;

function TSysApiServer.ExtractFirstParamUrl(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.ExtractFirstParamUrl(sb(Line),quote,Spaces));
end;

function TSysApiServer.Dequote_or_URL_Decode(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
begin
 Result:=sb(_crw_str.Dequote_or_URL_Decode(sb(Line),quote,Spaces));
end;

function TSysApiServer.IsOption(const arg:LongString; const shortopt:LongString=''; const longopt:LongString='';
                  Mode:Integer=om_Default; Delim:Char='='):Boolean;
begin
 Result:=_crw_str.IsOption(sb(arg),sb(shortopt),sb(longopt),Mode,Delim);
end;

function TSysApiServer.GetOptionValue(const arg:LongString; Delim:Char='='):LongString;
begin
 Result:=sb(_crw_str.GetOptionValue(sb(arg),Delim));
end;

function TSysApiServer.HasOptionValue(const arg:LongString; Delim:Char='='):Boolean;
begin
 Result:=_crw_str.HasOptionValue(sb(arg),Delim);
end;

function TSysApiServer.StringToSetOfChars(const S:LongString):TCharSet;
begin
 Result:=_crw_str.StringToSetOfChars(sb(S));
end;

function TSysApiServer.SetOfCharsToString(const S:TCharSet):LongString;
begin
 Result:=sb(_crw_str.SetOfCharsToString(S));
end;

function TSysApiServer.HasChars(const S:LongString; const C:TCharSet):Boolean;
begin
 Result:=_crw_str.HasChars(sb(S),C);
end;

function TSysApiServer.CountChars(const S:LongString; const C:TCharSet):SizeInt;
begin
 Result:=_crw_str.CountChars(sb(S),C);
end;

function TSysApiServer.GetRealFilePathName(const FileName:LongString; BuffSize:Integer=MAX_PATH; Mode:Integer=0):LongString;
begin
 Result:=sb(_crw_fio.GetRealFilePathName(sb(FileName),BuffSize,Mode));
end;

function TSysApiServer.IsSameChar(C1,C2:Char):Boolean;
begin
 Result:=_crw_str.IsSameChar(C1,C2);
end;

function TSysApiServer.IsSameStr(S1,S2:LongString):Boolean;
begin
 Result:=_crw_str.IsSameStr(sb(S1),sb(S2));
end;

function TSysApiServer.IsSameFileName(const S1,S2:LongString; Mode:Integer=ua_FileDef):Boolean;
begin
 Result:=_crw_str.IsSameFileName(sb(S1),sb(S2),Mode);
end;

function TSysApiServer.IsSectionName(const aName:LongString):Boolean;
begin
 Result:=_crw_str.IsSectionName(sb(aName));
end;

function TSysApiServer.HTTP_StatusMessage(StatusCode:Integer):LongString;
begin
 Result:=sb(_crw_str.HTTP_StatusMessage(StatusCode));
end;

function TSysApiServer.URL_Packed(const S:LongString; Mode:Integer=um_Safe; const AllowChars:TCharSet=UrlAllowChars):LongString;
begin
 Result:=sb(_crw_str.URL_Packed(sb(S),Mode,AllowChars));
end;

function TSysApiServer.URL_Encode(const S:LongString; Mode:Integer=um_Safe+um_Strict; const AllowChars:TCharSet=[]):LongString;
begin
 Result:=sb(_crw_str.URL_Encode(sb(S),Mode,AllowChars));
end;

function TSysApiServer.URL_Decode(const S:LongString; Mode:Integer=um_Safe):LongString;
begin
 Result:=sb(_crw_str.URL_Decode(sb(S),Mode));
end;

function TSysApiServer.StrIScan(Str:PChar; Chr:Char):PChar;
begin
 Result:=_crw_str.StrIScan(Str,Chr);
end;

function TSysApiServer.StrRIScan(Str:PChar; Chr:Char):PChar;
begin
 Result:=_crw_str.StrRIScan(Str,Chr);
end;

function TSysApiServer.StrIPos(Str1,Str2:PChar):PChar;
begin
 Result:=_crw_str.StrIPos(Str1,Str2);
end;

function TSysApiServer.StrPLCopy(Dest:PChar; const Source:LongString; MaxLen:SizeInt):PChar;
begin
 Result:=_crw_str.StrPLCopy(Dest,sb(Source),MaxLen);
end;

function TSysApiServer.ValidateEOL(const aData:LongString; aTail:Integer=0; aEOL:LongString=EOL):LongString;
begin
 Result:=sb(_crw_str.ValidateEOL(sb(aData),aTail,aEol));
end;

function TSysApiServer.RunCommand(cmdline:LongString; out outputstring:LongString):Boolean;
begin
 Result:=_crw_proc.RunCommand(sb(cmdline),outputstring);
end;

function TSysApiServer.RunCommandInDir(curdir,cmdline:LongString; out outputstring:LongString):Boolean;
begin
 Result:=_crw_proc.RunCommandInDir(sb(curdir),sb(cmdline),outputstring);
end;

function TSysApiServer.cpu_count:Integer;
begin
 Result:=_crw_proc.cpu_count;
end;

function TSysApiServer.cpu_mhz(id:Integer=0):Double;
begin
 Result:=_crw_proc.cpu_mhz(id);
end;

function TSysApiServer.ReadProcCpuInfo(CpuMask:Int64; What:LongString):LongString;
begin
 Result:=sb(_crw_proc.ReadProcCpuInfo(CpuMask,sb(What)));
end;

function TSysApiServer.IsATTY(Handle:LongInt):Boolean;
begin
 Result:=_crw_proc.IsATTY(Handle);
end;

function TSysApiServer.GetProcessAffinityMask(pid:TPid=0):QWord;
begin
 Result:=_crw_proc.GetProcessAffinityMask(pid);
end;

function TSysApiServer.GetThreadAffinityMask(tid:SizeInt=0):QWord;
begin
 Result:=_crw_proc.GetThreadAffinityMask(tid);
end;

function TSysApiServer.SetThreadAffinityMask(tid:SizeInt; Mask:QWord):QWord;
begin
 Result:=_crw_proc.SetThreadAffinityMask(tid,Mask);
end;

function TSysApiServer.GetListOfProcesses(aPid,aPPid:SizeInt; const aName:LongString; Detail:Boolean=false):LongString;
begin
 Result:=sb(_crw_proc.GetListOfProcesses(aPid,aPPid,sb(aName),Detail,myGlops_Flags));
end;

function TSysApiServer.GetProcessName(pid:SizeInt=0):LongString;
begin
 Result:=sb(_crw_proc.GetProcessName(pid));
end;

function TSysApiServer.GetParentProcessID:SizeInt;
begin
 Result:=_crw_proc.GetParentProcessID;
end;

function TSysApiServer.GetParentProcessName:LongString;
begin
 Result:=sb(_crw_proc.GetParentProcessName);
end;

function TSysApiServer.GetProcessTimesAsFileTime(out s,u:Int64):Boolean;
begin
 Result:=_crw_proc.GetProcessTimesAsFileTime(s,u);
end;

function TSysApiServer.GetThreadTimesAsFileTime(out s,u:Int64):Boolean;
begin
 Result:=_crw_proc.GetThreadTimesAsFileTime(s,u);
end;

function TSysApiServer.ProgBaseName:LongString;
begin
 Result:=sb(_crw_fio.ProgBaseName);
end;

function TSysApiServer.UserHomeDir(const SubDir:LongString=''):LongString;
begin
 Result:=sb(_crw_fio.UserHomeDir(sb(SubDir)));
end;

function TSysApiServer.SystemRootDir:LongString;
begin
 Result:=sb(_crw_fio.SystemRootDir);
end;

function TSysApiServer.GlobalTempDir(const SubDir:LongString=''):LongString;
begin
 Result:=sb(_crw_fio.GlobalTempDir(sb(SubDir)));
end;

function TSysApiServer.LocalTempDir(const SubDir:LongString=''):LongString;
begin
 Result:=sb(_crw_fio.LocalTempDir(sb(SubDir)));
end;

function TSysApiServer.GetCrwKitExe:LongString;
begin
 Result:=sb(_crw_fio.GetCrwKitExe);
end;

function TSysApiServer.GetShell:LongString;
begin
 Result:=sb(_crw_fio.GetShell);
end;

function TSysApiServer.GetCommandLine:LongString;
begin
 Result:=sb(_crw_fio.GetCommandLine);
end;

function TSysApiServer.UserName:LongString;
begin
 Result:=sb(_crw_fio.UserName);
end;

function TSysApiServer.ComputerName:LongString;
begin
 Result:=sb(_crw_fio.ComputerName);
end;

function TSysApiServer.HostName(Method:Integer=0):LongString;
begin
 Result:=sb(_crw_fio.HostName(Method));
end;

function TSysApiServer.UserDomain(const aUser:LongString=''; aDef:LongString=''):LongString;
begin
 Result:=sb(_crw_fio.UserDomain(sb(aUser),sb(aDef)));
end;

function TSysApiServer.GetIPAddress(const aHostName:LongString=''):LongString;
begin
 Result:=sb(_crw_fio.GetIPAddress(sb(aHostName)));
end;

function TSysApiServer.GetIfMacAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
begin
 Result:=sb(_crw_fio.GetIfMacAddresses(sb(Delim),Mode));
end;

function TSysApiServer.GetIfIpAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
begin
 Result:=sb(_crw_fio.GetIfIpAddresses(sb(Delim),Mode));
end;

function TSysApiServer.GetIPAddresses(const aHostName:LongString=''; Delim:LongString=','):LongString;
begin
 Result:=sb(_crw_fio.GetIPAddresses(sb(aHostName),sb(Delim)));
end;

function TSysApiServer.GetMacAddresses(const Machine:LongString=''; Delim:LongString=','; Mode:Integer=0):LongString;
begin
 Result:=sb(_crw_fio.GetMacAddresses(sb(Machine),sb(Delim),Mode));
end;

function TSysApiServer.GetLocalUserList(Delim:LongString=EOL):LongString;
begin
 Result:=sb(_crw_fio.GetLocalUserList(sb(Delim)));
end;

function TSysApiServer.GetEnv(const Name:LongString):LongString;
begin
 Result:=sb(_crw_fio.GetEnv(sb(Name)));
end;

function TSysApiServer.SetEnv(const Name,Value:LongString):Boolean;
begin
 Result:=_crw_fio.SetEnv(sb(Name),sb(Value));
end;

function TSysApiServer.ExpEnv(const Str:LongString):LongString;
begin
 Result:=sb(_crw_fio.ExpEnv(sb(Str)));
end;

function TSysApiServer.MaybeEnvStr(const arg:LongString):Boolean;
begin
 Result:=_crw_fio.MaybeEnvStr(sb(arg));
end;

function TSysApiServer.MaybeTildeStr(const arg:LongString):Boolean;
begin
 Result:=_crw_fio.MaybeTildeStr(sb(arg));
end;

function TSysApiServer.FileIsExecutable(const aFileName:LongString):Boolean;
begin
 Result:=_crw_fio.FileIsExecutable(sb(aFileName));
end;

function TSysApiServer.FileIsSymlink(const aFileName:LongString):Boolean;
begin
 Result:=_crw_fio.FileIsSymlink(sb(aFileName));
end;

function TSysApiServer.FileIsHardLink(const aFileName:LongString):Boolean;
begin
 Result:=_crw_fio.FileIsHardLink(sb(aFileName));
end;

function TSysApiServer.FileIsReadable(const aFileName:LongString):Boolean;
begin
 Result:=_crw_fio.FileIsReadable(sb(aFileName));
end;

function TSysApiServer.FileIsWritable(const aFileName:LongString):Boolean;
begin
 Result:=_crw_fio.FileIsWritable(sb(aFileName));
end;

function TSysApiServer.StringFromFile(const FileName:LongString; MaxSize:SizeInt):LongString;
begin
 Result:=sb(_crw_fio.StringFromFile(sb(FileName),MaxSize));
end;

function  TSysApiServer.GetRootKeyByName(const Key:LongString):HKEY;
begin
 Result:=_crw_fio.GetRootKeyByName(sb(Key));
end;

function  TSysApiServer.ReadRegistryMultiStrings(RootKey:HKEY; const Key,Name:LongString; Delim:Char=ASCII_CR):LongString;
begin
 Result:=sb(_crw_fio.ReadRegistryMultiStrings(RootKey,sb(Key),sb(Name),Delim));
end;

function  TSysApiServer.ReadShellLinkAsText(const FileName:LongString):LongString;
begin
 Result:=sb(_crw_fio.ReadShellLinkAsText(sb(FileName)));
end;

function TSysApiServer.AnyFileMask:LongString;
begin
 Result:=sb(_crw_fio.AnyFileMask);
end;

function TSysApiServer.FindAllFilesAsText(const SearchPath:LongString;
                             const SearchMask:LongString='';
                                   SearchSubDirs:Boolean=True;
                                   DirAttr:Word=faDirectory;
                                   MaskSeparator:Char=';';
                                   PathSeparator:Char=';'
                                   ):LongString;
begin
 Result:=sb(_crw_fio.FindAllFilesAsText(sb(SearchPath),sb(SearchMask),
                            SearchSubDirs,DirAttr,MaskSeparator,PathSeparator));
end;

function TSysApiServer.ReadIniFileRecord(const FileName    : LongString;
                           const SectionName : LongString;
                           const Format      : LongString;
                             var Data;
                                 efMode      : Integer = efConfig;
                                 svMode      : Integer = svConfig
                                           ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileRecord(sb(FileName),sb(SectionName),sb(Format),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileBoolean(const FileName    : LongString;
                            const SectionName : LongString;
                            const Key         : LongString;
                              var Data        : Boolean;
                                  efMode      : Integer = efConfig;
                                  svMode      : Integer = svConfig
                                            ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileBoolean(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileDouble(const FileName    : LongString;
                           const SectionName : LongString;
                           const Key         : LongString;
                             var Data        : Double;
                                 efMode      : Integer = efConfig;
                                 svMode      : Integer = svConfig
                                            ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileDouble(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileWord(const FileName    : LongString;
                         const SectionName : LongString;
                         const Key         : LongString;
                           var Data        : Word;
                               efMode      : Integer = efConfig;
                               svMode      : Integer = svConfig
                                         ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileWord(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileInteger(const FileName    : LongString;
                            const SectionName : LongString;
                            const Key         : LongString;
                              var Data        : Integer;
                                  efMode      : Integer = efConfig;
                                  svMode      : Integer = svConfig
                                            ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileInteger(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileAlpha(const FileName    : LongString;
                          const SectionName : LongString;
                          const Key         : LongString;
                            var Data        : LongString;
                                efMode      : Integer = efConfig;
                                svMode      : Integer = svConfig
                                          ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileAlpha(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.ReadIniFileString(const FileName    : LongString;
                           const SectionName : LongString;
                           const Key         : LongString;
                             var Data        : LongString;
                                 efMode      : Integer = efConfig;
                                 svMode      : Integer = svConfig
                                           ) : Boolean;
begin
 Result:=_crw_fio.ReadIniFileString(sb(FileName),sb(SectionName),sb(Key),Data,efMode,svMode);
end;

function TSysApiServer.GetFileType(hFile:THandle):LongWord;
begin
 Result:=_crw_fio.GetFileType(hFile);
end;

function TSysApiServer.GetFileSize64(const FileName:LongString):Int64;
begin
 Result:=_crw_fio.GetFileSize64(sb(FileName));
end;

function TSysApiServer.FileHasNonBlockFlag(fd:THandle):Boolean;
begin
 Result:=_crw_fio.FileHasNonBlockFlag(fd);
end;

function TSysApiServer.FileSetNonBlockFlag(fd:THandle; State:Boolean=true):Boolean;
begin
 Result:=_crw_fio.FileSetNonBlockFlag(fd,State);
end;

function TSysApiServer.FileSetCloseOnExec(fd:THandle; State:Boolean=true):Boolean;
begin
 Result:=_crw_fio.FileSetCloseOnExec(fd,State);
end;

function TSysApiServer.ePendingFileOperation(ErrorCode:Integer):Boolean;
begin
 Result:=_crw_fio.ePendingFileOperation(ErrorCode);
end;

function TSysApiServer.eBrokenPipe(ErrorCode:Integer):Boolean;
begin
 Result:=_crw_fio.eBrokenPipe(ErrorCode);
end;

function TSysApiServer.eOperationAborted(ErrorCode:Integer):Boolean;
begin
 Result:=_crw_fio.eOperationAborted(ErrorCode);
end;

function TSysApiServer.SmartFileRef(const FileRef   : LongString;
                      const DefExt    : LongString = '';
                      const BaseFile  : LongString = '';
                      const UserHome  : LongString = sfrDefUserHome;
                      const ProgHome  : LongString = sfrDefProgHome;
                      const Brackets  : LongString = sfrDefBrackets;
                            handleUrl : Boolean    = true
                                      ) : LongString;
begin
 Result:=sb(_crw_fio.SmartFileRef(sb(FileRef),sb(DefExt),sb(BaseFile),
         sb(UserHome),sb(ProgHome),Brackets,HandleUrl));
end;

function TSysApiServer.SmartFileRel(const FileRef   : LongString;
                      const BaseFile  : LongString;
                      const UserHome  : LongString = sfrDefUserHome;
                      const ProgHome  : LongString = sfrDefProgHome;
                      const Brackets  : LongString = sfrDefBrackets;
                            handleUrl : Boolean    = true
                                    ) : LongString;
begin
 Result:=sb(_crw_fio.SmartFileRel(sb(FileRef),sb(BaseFile),
         sb(UserHome),sb(ProgHome),Brackets,HandleUrl));
end;

function TSysApiServer.file_which(name:LongString):LongString;
begin
 Result:=sb(_crw_fio.file_which(sb(name)));
end;

function TSysApiServer.GetFileVersionInfoAsText(FileName:LongString):LongString;
begin
 Result:=sb(_crw_fio.GetFileVersionInfoAsText(sb(FileName)));
end;

function TSysApiServer.StdOutPrint(const Msg:LongString):LongInt;
begin
 Result:=_crw_fio.StdOutPrint(sb(Msg));
end;

function TSysApiServer.StdErrPrint(const Msg:LongString):LongInt;
begin
 Result:=_crw_fio.StdErrPrint(sb(Msg));
end;

function TSysApiServer.StdOutPrintLn(const Msg:LongString):LongInt;
begin
 Result:=_crw_fio.StdOutPrintLn(sb(Msg));
end;

function TSysApiServer.StdErrPrintLn(const Msg:LongString):LongInt;
begin
 Result:=_crw_fio.StdErrPrintLn(sb(Msg));
end;

function  TSysApiServer.isFileClosed(var f:File):Boolean;
begin
 Result:=_crw_fio.isFileClosed(f);
end;

function  TSysApiServer.isTextClosed(var f:Text):Boolean;
begin
 Result:=_crw_fio.isFileClosed(f);
end;

procedure TSysApiServer.SmartFileClose(var f:File);
begin
 _crw_fio.SmartFileClose(f);
end;

procedure TSysApiServer.SmartTextClose(var f:Text);
begin
 _crw_fio.SmartFileClose(f);
end;

function TSysApiServer.FileTypeToString(fType:LongWord; Prefix:LongString='FILE_TYPE_'):LongString;
begin
 Result:=sb(_crw_fio.FileTypeToString(fType,sb(Prefix)));
end;

function TSysApiServer.IoResultToString(Code:Integer):LongString;
begin
 Result:=sb(_crw_fio.IoResultToString(Code));
end;

function TSysApiServer.StdDateTimeStr(ms:Double=0):LongString;
begin
 Result:=sb(_crw_apptools.StdDateTimeStr(ms));
end;

function TSysApiServer.StdDateTimePrompt(ms:Double=0; Prompt:PChar=nil):LongString;
begin
 Result:=sb(_crw_apptools.StdDateTimePrompt(ms,Prompt));
end;

function TSysApiServer.WriteRegistryString(RootKey:HKEY; const Key,Name,Data:LongString):LongString;
begin
 Result:=sb(_crw_fio.WriteRegistryString(RootKey,sb(Key),sb(Name),sb(Data)));
end;

procedure TSysApiServer.DebugOutText(n:Byte; const S:LongString);
begin
 _crw_fio.DebugOutText(n,sb(S));
end;

function TSysApiServer.ValidatePathList(const PathList:LongString; Delim:Char=PathSep):LongString;
begin
 Result:=sb(_crw_fio.ValidatePathList(sb(PathList),Delim));
end;

function TSysApiServer.ValidateEnvPathList(const EnvPathList:LongString; Delim:Char=PathSep):Integer;
begin
 Result:=_crw_fio.ValidateEnvPathList(sb(EnvPathList),Delim);
end;

function TSysApiServer.SmartFileSearch(const FileName   : LongString;
                           const PathVarStr : LongString='PATH';
                           const PathExtVar : LongString='PATHEXT';
                           const PathExtStr : LongString=DefaultPathExt
                                          ) : LongString;
begin
 Result:=sb(_crw_fio.SmartFileSearch(sb(FileName),sb(PathVarStr),sb(PathExtVar),sb(PathExtStr)));
end;

function TSysApiServer.ReadSymLink(const FileName:LongString; MaxLevel:Integer=10):LongString;
begin
 Result:=sb(_crw_fio.ReadSymLink(sb(FileName),MaxLevel));
end;

function TSysApiServer.GetSystemParamStr(const Category,ParamName:LongString):LongString;
begin
 Result:=sb(_crw_apptools.GetSystemParamStr(sb(Category),sb(ParamName)));
end;

function TSysApiServer.base64_encode(const S:LongString):LongString;
begin
 Result:=sb(_crw_base64.base64_encode(sb(S)));
end;

function TSysApiServer.base64_decode(const S:LongString):LongString;
begin
 Result:=sb(_crw_base64.base64_decode(sb(S)));
end;

function TSysApiServer.base32_encode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
begin
 Result:=sb(_crw_base32.base32_encode(sb(sData),nAlphabet));
end;

function TSysApiServer.base32_decode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
begin
 Result:=sb(_crw_base32.base32_decode(sb(sData),nAlphabet));
end;

function TSysApiServer.backslash_encode(const data:LongString; const esclist:TCharSet=[]; const hexlist:TCharSet=[]):LongString;
begin
 Result:=sb(_crw_bsencode.backslash_encode(sb(data),esclist,hexlist));
end;

function TSysApiServer.backslash_decode(const data:LongString):LongString;
begin
 Result:=sb(_crw_bsencode.backslash_decode(sb(data)));
end;

function TSysApiServer.StringOfCharRange(a,b:Char):LongString;
begin
 Result:=sb(_crw_bsencode.StringOfCharRange(a,b));
end;

function TSysApiServer.task_init(const cmd_line:LongString):Integer;
begin
 Result:=_crw_task.task_init(sb(cmd_line));
 SaveRef(task_ref(Result));
end;

function TSysApiServer.task_free(tid:Integer):Boolean;
begin
 DropRef(task_ref(tid));
 Result:=_crw_task.task_free(tid);
end;

function TSysApiServer.task_ref(tid:Integer):Integer;
begin
 Result:=_crw_task.task_ref(tid).Ref;
end;

function TSysApiServer.task_pid(tid:Integer):Integer;
begin
 Result:=_crw_task.task_pid(tid);
end;

function TSysApiServer.task_run(tid:Integer):Boolean;
begin
 Result:=_crw_task.task_run(tid);
end;

function TSysApiServer.task_wait(tid,timeout:Integer):Boolean;
begin
 Result:=_crw_task.task_wait(tid,timeout);
end;

function TSysApiServer.task_send(tid:Integer; const data:LongString):Integer;
begin
 Result:=_crw_task.task_send(tid,sb(data));
end;

function TSysApiServer.task_recv(tid,maxlen:Integer):LongString;
begin
 Result:=sb(_crw_task.task_recv(tid,maxlen));
end;

function TSysApiServer.task_txcount(tid:Integer):Integer;
begin
 Result:=_crw_task.task_txcount(tid);
end;

function TSysApiServer.task_rxcount(tid:Integer):Integer;
begin
 Result:=_crw_task.task_rxcount(tid);
end;

function TSysApiServer.task_txspace(tid:Integer):Integer;
begin
 Result:=_crw_task.task_txspace(tid);
end;

function TSysApiServer.task_rxspace(tid:Integer):Integer;
begin
 Result:=_crw_task.task_rxspace(tid);
end;

function TSysApiServer.task_result(tid:Integer):Integer;
begin
 Result:=_crw_task.task_result(tid);
end;

function TSysApiServer.task_kill(tid,how,exit_code,timeout:Integer):Boolean;
begin
 Result:=_crw_task.task_kill(tid,how,exit_code,timeout);
end;

function TSysApiServer.task_ctrl(tid:Integer; const arg:LongString):LongString;
begin
 Result:=sb(_crw_task.task_ctrl(tid,arg));
end;

function TSysApiServer.SmartExecute(const CmdLine:LongString;
                              Display:Integer=SW_SHOWNORMAL;
                              ShellCm:LongString=''):Boolean;
begin
 Result:=_crw_task.SmartExecute(sb(CmdLine),Display,sb(ShellCm));
end;

function TSysApiServer.PidAffinity(pid:TPid; mask:Int64=0):Int64;
begin
 Result:=_crw_task.PidAffinity(pid,mask);
end;

function TSysApiServer.KillProcess(aPid:TPid; aExitCode:Integer; sig:Integer=SIGTERM):Boolean;
begin
 Result:=_crw_task.KillProcess(aPid,aExitCode,sig);
end;

function TSysApiServer.HashList_Init(aCaseSensitive:Boolean):Integer;
begin
 Result:=_crw_hl.HashList_Init(aCaseSensitive,HashList_DefaultHasher);
 SaveRef(HashList_Ref(Result));
end;

function TSysApiServer.HashList_Free(aRef:Integer):Boolean;
begin
 DropRef(HashList_Ref(aRef));
 Result:=_crw_hl.HashList_Free(aRef);
end;

function TSysApiServer.HashList_Ref(aRef:Integer):Integer;
begin
 Result:=_crw_hl.HashList_Ref(aRef).Ref;
end;

function TSysApiServer.HashList_Count(aRef:Integer):Integer;
begin
 Result:=_crw_hl.HashList_Count(aRef);
end;

function TSysApiServer.HashList_GetKey(aRef:Integer; aIndex:Integer):LongString;
begin
 Result:=sb(_crw_hl.HashList_GetKey(aRef,aIndex));
end;

function TSysApiServer.HashList_Delete(aRef:Integer; aKey:LongString):Boolean;
begin
 Result:=_crw_hl.HashList_Delete(aRef,sb(aKey));
end;

function TSysApiServer.HashList_IndexOf(aRef:Integer; aKey:LongString):Integer;
begin
 Result:=_crw_hl.HashList_IndexOf(aRef,sb(aKey));
end;

function TSysApiServer.HashList_GetData(aRef:Integer; aKey:LongString):Double;
begin
 Result:=_crw_hl.HashList_GetData(aRef,sb(aKey));
end;

function TSysApiServer.HashList_SetData(aRef:Integer; aKey:LongString; aData:Double):Boolean;
begin
 Result:=_crw_hl.HashList_SetData(aRef,sb(aKey),aData);
end;

function TSysApiServer.HashList_GetLink(aRef:Integer; aKey:LongString):Integer;
begin
 Result:=_crw_hl.HashList_GetLink(aRef,sb(aKey));
end;

function TSysApiServer.HashList_SetLink(aRef:Integer; aKey:LongString; aLink:Integer):Boolean;
begin
 Result:=_crw_hl.HashList_SetLink(aRef,sb(aKey),aLink);
end;

function TSysApiServer.HashList_GetPara(aRef:Integer; aKey:LongString):LongString;
begin
 Result:=sb(_crw_hl.HashList_GetPara(aRef,sb(aKey)));
end;

function TSysApiServer.HashList_SetPara(aRef:Integer; aKey:LongString; aParam:LongString):Boolean;
begin
 Result:=_crw_hl.HashList_SetPara(aRef,sb(aKey),sb(aParam));
end;

function TSysApiServer.regexp_init(engine:Integer; pattern:LongString; options:LongString=''):Integer;
begin
 Result:=_crw_regexp.regexp_init(engine,sb(pattern),sb(options));
 SaveRef(regexp_ref(Result));
end;

function TSysApiServer.regexp_free(rex:Integer):Boolean;
begin
 DropRef(regexp_ref(rex));
 Result:=_crw_regexp.regexp_free(rex);
end;

function TSysApiServer.regexp_ref(rex:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_ref(rex).Ref;
end;

function TSysApiServer.regexp_ctrl(rex:Integer; arg:LongString):LongString;
begin
 Result:=sb(_crw_regexp.regexp_ctrl(rex,sb(arg)));
end;

function TSysApiServer.regexp_test(rex:Integer; arg:LongString):Boolean;
begin
 Result:=_crw_regexp.regexp_test(rex,sb(arg));
end;

function TSysApiServer.regexp_exec(rex:Integer; arg:LongString):Integer;
begin
 Result:=_crw_regexp.regexp_exec(rex,sb(arg));
end;

function TSysApiServer.regexp_replace(rex:Integer; arg,rep:LongString):LongString;
begin
 Result:=sb(_crw_regexp.regexp_replace(rex,sb(arg),sb(rep)));
end;

function TSysApiServer.regexp_matchnum(rex:Integer; i:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchnum(rex,i);
end;

function TSysApiServer.regexp_matchpos(rex:Integer; i,j:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchpos(rex,i,j);
end;

function TSysApiServer.regexp_matchlen(rex:Integer; i,j:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchlen(rex,i,j);
end;

function TSysApiServer.regexp_matchstr(rex:Integer; i,j:Integer):LongString;
begin
 Result:=sb(_crw_regexp.regexp_matchstr(rex,i,j));
end;

function TSysApiServer.regexp_escape(arg:LongString):LongString;
begin
 Result:=sb(_crw_regexp.regexp_escape(sb(arg)));
end;

function TSysApiServer.pipe_init(const cmd_line:LongString):Integer;
begin
 Result:=_crw_pipeio.pipe_init(sb(cmd_line));
 SaveRef(pipe_ref(Result));
end;

function TSysApiServer.pipe_free(pid:Integer):Boolean;
begin
 DropRef(pipe_ref(pid));
 Result:=_crw_pipeio.pipe_free(pid);
end;

function TSysApiServer.pipe_ref(pid:Integer):Integer;
var Obj:TObject;
begin
 Obj:=_crw_pipeio.pipe_ref(pid);
 if (Obj is TMasterObject)
 then Result:=TMasterObject(Obj).Ref
 else Result:=0;
end;

function TSysApiServer.pipe_pid(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_pid(pid);
end;

function TSysApiServer.pipe_run(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_run(pid);
end;

function TSysApiServer.pipe_wait(pid,timeout:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_wait(pid,timeout);
end;

function TSysApiServer.pipe_send(pid:Integer; const data:LongString; awake:Boolean=false):Integer;
begin
 Result:=_crw_pipeio.pipe_send(pid,sb(data),awake);
end;

function TSysApiServer.pipe_recv(pid,maxlen:Integer):LongString;
begin
 Result:=sb(_crw_pipeio.pipe_recv(pid,maxlen));
end;

function TSysApiServer.pipe_txcount(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_txcount(pid);
end;

function TSysApiServer.pipe_rxcount(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_rxcount(pid);
end;

function TSysApiServer.pipe_txspace(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_txspace(pid);
end;

function TSysApiServer.pipe_rxspace(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_rxspace(pid);
end;

function TSysApiServer.pipe_result(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_result(pid);
end;

function TSysApiServer.pipe_kill(pid,how,exit_code,timeout:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_kill(pid,how,exit_code,timeout);
end;

function TSysApiServer.pipe_ctrl(pid:Integer; const param:LongString):LongString;
begin
 Result:=sb(_crw_pipeio.pipe_ctrl(pid,sb(param)));
end;

function TSysApiServer.pipe_txclear(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_txclear(pid);
end;

function TSysApiServer.pipe_rxclear(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_rxclear(pid);
end;

function TSysApiServer.pipe_count(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_count(pid);
end;

function TSysApiServer.pipe_stream(pid,index:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_stream(pid,index);
end;

function TSysApiServer.pipe_connected(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_connected(pid);
end;

function TSysApiServer.CSIDL_FolderByName(const Name:LongString; const sub:LongString;
                 Charset:Integer=-1; const Delim:LongString=''):LongString;
begin
 Result:=sb(_crw_spcfld.CSIDL_FolderByName(sb(Name),sb(sub),Charset,sb(Delim)));
end;

function TSysApiServer.CSIDL_ListAllAsText(const Prefix:LongString='CSIDL_';
                 Charset:Integer=-1; SkipEmpty:Boolean=true):LongString;
begin
 Result:=sb(_crw_spcfld.CSIDL_ListAllAsText(sb(Prefix),Charset,SkipEmpty));
end;

function TSysApiServer.GetSystemAssoc(const Ext:LongString):LongString;
begin
 Result:=sb(_crw_assoc.GetSystemAssoc(sb(Ext)));
end;

function TSysApiServer.GetSystemFType(const FType:LongString):LongString;
begin
 Result:=sb(_crw_assoc.GetSystemFType(sb(FType)));
end;

function TSysApiServer.GetSystemAssocExe(const Ext:LongString):LongString;
begin
 Result:=sb(_crw_assoc.GetSystemAssocExe(sb(Ext)));
end;

function TSysApiServer.GetSystemFTypeExe(const FType:LongString):LongString;
begin
 Result:=sb(_crw_assoc.GetSystemFTypeExe(sb(FType)));
end;

function TSysApiServer.HasSystemAssocExe(const Ext:LongString):Boolean;
begin
 Result:=_crw_assoc.HasSystemAssocExe(sb(Ext));
end;

function TSysApiServer.HasSystemFTypeExe(const FType:LongString):Boolean;
begin
 Result:=_crw_assoc.HasSystemFTypeExe(sb(FType));
end;

function TSysApiServer.GetExeByExtension(const Extension:LongString):LongString;
begin
 Result:=_crw_assoc.GetExeByExtension(sb(Extension));
end;

function TSysApiServer.GetAppPath(args:LongString; AllowRun:Boolean=false; AllowWait:Boolean=false;
                   pExitCode:PInteger=nil; EnvPath:LongString='';
                   EnvPathExt:LongString=''):LongString;
begin
 Result:=sb(_crw_assoc.GetAppPath(sb(args),AllowRun,AllowWait,pExitCode,nil,nil,sb(EnvPath),sb(EnvPathExt)));
end;

function TSysApiServer.GetExeByFile(const FileName:LongString):LongString;
begin
 Result:=sb(_crw_assoc.GetExeByFile(sb(FileName)));
end;

function TSysApiServer.WScriptShellRun(CmdLine:LongString; Show:Integer=SW_SHOWNORMAL; Wait:Boolean=false):Integer;
begin
 Result:=_crw_assoc.WScriptShellRun(sb(CmdLine),Show,Wait);
end;

function TSysApiServer.ShellExecuteOpen(FileName:LongString; Cmd:LongString=''):Boolean;
begin
 Result:=_crw_assoc.ShellExecuteOpen(sb(FileName),sb(Cmd));
end;

function TSysApiServer.wmctrl_query(const args:LongString):LongString;
begin
 Result:=sb(_crw_wmctrl.wmctrl_query(sb(args)));
end;

function TSysApiServer.shm_init(name:LongString; size,mode:Integer):Integer;
begin
 Result:=_crw_sharm.shm_init(sb(name),size,mode);
 SaveRef(shm_ref(Result));
end;

function TSysApiServer.shm_ref(ref:Integer):Integer;
begin
 Result:=_crw_sharm.shm_ref(ref).Ref;
end;

function TSysApiServer.shm_free(ref:Integer):Boolean;
begin
 DropRef(shm_ref(ref));
 Result:=_crw_sharm.shm_free(ref);
end;

function TSysApiServer.shm_delink(name:LongString):Boolean;
begin
 Result:=_crw_sharm.shm_delink(sb(name));
end;

function TSysApiServer.shm_iop(ref:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
begin
 Result:=_crw_sharm.shm_iop(ref,offset,op,data);
end;

function TSysApiServer.shm_rop(ref:Integer; offset:Integer; op:Char; data:Double):Double;
begin
 Result:=_crw_sharm.shm_rop(ref,offset,op,data);
end;

function TSysApiServer.shm_fop(ref:Integer; offset:Integer; op:Char; data:Single):Single;
begin
 Result:=_crw_sharm.shm_fop(ref,offset,op,data);
end;

function TSysApiServer.shm_sop(ref:Integer; offset:Integer; op:Char; data:LongString):LongString;
begin
 Result:=sb(_crw_sharm.shm_sop(ref,offset,op,sb(data)));
end;

function TSysApiServer.shm_ctrl(ref:Integer; arg:LongString):LongString;
begin
 Result:=sb(_crw_sharm.shm_ctrl(ref,sb(arg)));
end;

function TSysApiServer.shm_ioresult(code:Integer=0):Integer;
begin
 Result:=_crw_sharm.shm_ioresult(code);
end;

function TSysApiServer.fsm_new:Integer;
begin
 Result:=_crw_fsm.fsm_new;
 SaveRef(fsm_ref(Result));
end;

function TSysApiServer.fsm_free(ref:Integer):Boolean;
begin
 DropRef(fsm_ref(ref));
 Result:=_crw_fsm.fsm_free(ref);
end;

function TSysApiServer.fsm_ref(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_ref(ref).Ref;
end;

function TSysApiServer.fsm_root(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_root(ref);
end;

function TSysApiServer.fsm_type(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_type(ref);
end;

function TSysApiServer.fsm_parent(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_parent(ref);
end;

function TSysApiServer.fsm_name(ref:Integer):LongString;
begin
 Result:=sb(_crw_fsm.fsm_name(ref));
end;

function TSysApiServer.fsm_path(ref:Integer):LongString;
begin
 Result:=sb(_crw_fsm.fsm_path(ref));
end;

function TSysApiServer.fsm_ctrl(ref:Integer; arg:LongString):LongString;
begin
 Result:=sb(_crw_fsm.fsm_ctrl(ref,sb(arg)));
end;

function TSysApiServer.fsm_count(ref,typ:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_count(ref,typ);
end;

function TSysApiServer.fsm_items(ref,typ,i:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_items(ref,typ,i);
end;

function TSysApiServer.fsm_get_iparam(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_get_iparam(ref);
end;

function TSysApiServer.fsm_set_iparam(ref:Integer; data:Integer):Boolean;
begin
 Result:=_crw_fsm.fsm_set_iparam(ref,data);
end;

function TSysApiServer.fsm_get_fparam(ref:Integer):Double;
begin
 Result:=_crw_fsm.fsm_get_fparam(ref);
end;

function TSysApiServer.fsm_set_fparam(ref:Integer; data:Double):Boolean;
begin
 Result:=_crw_fsm.fsm_set_fparam(ref,data);
end;

function TSysApiServer.fsm_get_sparam(ref:Integer):LongString;
begin
 Result:=sb(_crw_fsm.fsm_get_sparam(ref));
end;

function TSysApiServer.fsm_set_sparam(ref:Integer; data:LongString):Boolean;
begin
 Result:=_crw_fsm.fsm_set_sparam(ref,sb(data));
end;

function TSysApiServer.fsm_add(ref:Integer; typ:integer; key:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_add(ref,typ,sb(key));
end;

function TSysApiServer.fsm_find(ref:Integer; typ:integer; key:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_find(ref,typ,sb(key));
end;

function TSysApiServer.fsm_get_state(ref:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_get_state(ref);
end;

function TSysApiServer.fsm_set_state(ref:Integer; state:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_set_state(ref,state);
end;

function TSysApiServer.fsm_put_state(ref:Integer; state:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_set_state(ref,sb(state));
end;

function TSysApiServer.fsm_link(ref:Integer; arg:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_link(ref,sb(arg));
end;

function TSysApiServer.fsm_modified(ref:Integer; delta:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_modified(ref,delta);
end;

function TSysApiServer.fsm_name_rule(typ:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_name_rule(typ);
end;

function TSysApiServer.fsm_valid_childtype(typ,childtyp:Integer):Boolean;
begin
 Result:=_crw_fsm.fsm_valid_childtype(typ,childtyp);
end;

function TSysApiServer.Hex_Encode(const Str:LongString):LongString;
begin
 Result:=sb(_crw_crypt.Hex_Encode(sb(Str)));
end;

function TSysApiServer.Hex_Decode(const Str:LongString):LongString;
begin
 Result:=sb(_crw_crypt.Hex_Decode(sb(Str)));
end;

function TSysApiServer.ColorToString(ColorCode:TColor; Prefix:LongString=''):LongString;
begin
 Result:=sb(_crw_colors.ColorToString(ColorCode,sb(Prefix)));
end;

function TSysApiServer.StringToColor(ColorName:LongString; DefaultColor:TColor=clNone):TColor;
begin
 Result:=_crw_colors.StringToColor(sb(ColorName),DefaultColor);
end;

function TSysApiServer.StringToSigCode(str:LongString; def:Integer=0):Integer;
begin
 Result:=_crw_proc.StringToSigCode(sb(str),def);
end;

function TSysApiServer.SigCodeToString(sig:Integer):LongString;
begin
 Result:=sb(_crw_proc.SigCodeToString(sig));
end;

function TSysApiServer.text_new:Integer;
begin
 Result:=_crw_str.text_new.Ref;
 SaveRef(text_ref(Result));
end;

function TSysApiServer.text_ref(txt:Integer):Integer;
begin
 Result:=_crw_str.text_ref(txt).Ref;
end;

function TSysApiServer.text_free(txt:Integer):Boolean;
begin
 DropRef(text_ref(txt));
 Result:=_crw_str.text_free(txt);
end;

function TSysApiServer.text_getln(txt:Integer; i:Integer):LongString;
begin
 Result:=sb(_crw_str.text_getln(txt,i));
end;

function TSysApiServer.text_putln(txt:Integer; i:Integer; s:LongString):Boolean;
begin
 Result:=_crw_str.text_putln(txt,i,sb(s));
end;

function TSysApiServer.text_insln(txt:Integer; i:Integer; s:LongString):Boolean;
begin
 Result:=_crw_str.text_insln(txt,i,sb(s));
end;

function TSysApiServer.text_addln(txt:Integer; s:LongString):Boolean;
begin
 Result:=_crw_str.text_addln(txt,sb(s));
end;

function TSysApiServer.text_delln(txt:Integer; i:Integer):Boolean;
begin
 Result:=_crw_str.text_delln(txt,i);
end;

function TSysApiServer.text_numln(txt:Integer):Integer;
begin
 Result:=_crw_str.text_numln(txt);
end;

function TSysApiServer.text_tostring(txt:Integer):LongString;
begin
 Result:=sb(_crw_str.text_tostring(txt));
end;

function TSysApiServer.text_fromstring(txt:Integer; s:LongString):Integer;
begin
 Result:=_crw_str.text_fromstring(txt,sb(s));
end;

function TSysApiServer.db_create(arg:LongString):Boolean;
begin
 Result:=_crw_dbapi.db_create(sb(arg));
end;

function TSysApiServer.db_connection(eid:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_connection(eid,sb(arg));
 SaveRef(db_ref(Result));
end;

function TSysApiServer.db_recordset(dbo:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_recordset(dbo,sb(arg));
end;

function TSysApiServer.db_command(dbo:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_command(dbo,sb(arg));
end;

function TSysApiServer.db_free(dbo:Integer):Boolean;
begin
 DropRef(db_ref(dbo));
 Result:=_crw_dbapi.db_free(dbo);
end;

function TSysApiServer.db_ref(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_ref(dbo).Ref;
end;

function TSysApiServer.db_root(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_root(dbo).Ref;
end;

function TSysApiServer.db_type(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_type(dbo);
end;

function TSysApiServer.db_parent(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_parent(dbo);
end;

function TSysApiServer.db_state(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_state(dbo);
end;

function TSysApiServer.db_close(dbo:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_close(dbo);
end;

function TSysApiServer.db_open(dbo:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_open(dbo,opt);
end;

function TSysApiServer.db_ctrl(dbo:Integer; arg:LongString):LongString;
begin
 Result:=sb(_crw_dbapi.db_ctrl(dbo,sb(arg)));
end;

function TSysApiServer.db_bugscount(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_bugscount(dbo);
end;

function TSysApiServer.db_bugsclear(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_bugsclear(dbo);
end;

function TSysApiServer.db_errors(dbo:Integer):LongString;
begin
 Result:=sb(_crw_dbapi.db_errors(dbo));
end;

function TSysApiServer.db_errorscount(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_errorscount(dbo);
end;

function TSysApiServer.db_errorsclear(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_errorsclear(dbo);
end;

function TSysApiServer.db_execute(dbo:Integer; arg:LongString; opt:Integer):Integer;
begin
 Result:=_crw_dbapi.db_execute(dbo,sb(arg),opt).Ref;
end;

function TSysApiServer.db_cancel(dbo:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_cancel(dbo);
end;

function TSysApiServer.db_update(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_update(dbr);
end;

function TSysApiServer.db_cancelupdate(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_cancelupdate(dbr);
end;

function TSysApiServer.db_begintrans(dbc:Integer):Integer;
begin
 Result:=_crw_dbapi.db_begintrans(dbc);
end;

function TSysApiServer.db_committrans(dbc:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_committrans(dbc);
end;

function TSysApiServer.db_rollbacktrans(dbc:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_rollbacktrans(dbc);
end;

function TSysApiServer.db_bof(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_bof(dbr);
end;

function TSysApiServer.db_eof(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_eof(dbr);
end;

function TSysApiServer.db_movefirst(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movefirst(dbr);
end;

function TSysApiServer.db_movelast(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movelast(dbr);
end;

function TSysApiServer.db_movenext(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movelast(dbr);
end;

function TSysApiServer.db_moveprevious(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_moveprevious(dbr);
end;

function TSysApiServer.db_fieldscount(dbr:Integer):Integer;
begin
 Result:=_crw_dbapi.db_fieldscount(dbr);
end;

function TSysApiServer.db_fieldsnames(dbr:Integer; i:Integer):LongString;
begin
 Result:=sb(_crw_dbapi.db_fieldsnames(dbr,i));
end;

function TSysApiServer.db_fieldstypes(dbr:Integer; id:LongString):Integer;
begin
 Result:=_crw_dbapi.db_fieldstypes(dbr,sb(id));
end;

function TSysApiServer.db_fieldsasint(dbr:Integer; id:LongString; op:Char; arg:Integer):Integer;
begin
 Result:=_crw_dbapi.db_fieldsasint(dbr,sb(id),op,arg);
end;

function TSysApiServer.db_fieldsasfloat(dbr:Integer; id:LongString; op:Char; arg:Double):Double;
begin
 Result:=_crw_dbapi.db_fieldsasfloat(dbr,sb(id),op,arg);
end;

function TSysApiServer.db_fieldsasstring(dbr:Integer; id:LongString; op:Char; arg:LongString):LongString;
begin
 Result:=sb(_crw_dbapi.db_fieldsasstring(dbr,sb(id),op,sb(arg)));
end;

function TSysApiServer.db_addnew(dbr:Integer; arg:LongString):Boolean;
begin
 Result:=_crw_dbapi.db_addnew(dbr,sb(arg));
end;

function TSysApiServer.db_delete(dbr:Integer; aff:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_delete(dbr,aff);
end;

function TSysApiServer.db_requery(dbr:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_requery(dbr,opt);
end;

function TSysApiServer.db_resync(dbr:Integer; aff,res:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_resync(dbr,aff,res);
end;

function TSysApiServer.db_supports(dbr:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_supports(dbr,opt);
end;

function TSysApiServer.db_save(dbr:Integer; dst:LongString; fmt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_save(dbr,sb(dst),fmt);
end;

function TSysApiServer.HasFlags(Mode,Flags:Int64):Boolean;
begin
 Result:=_crw_ef.HasFlags(Mode,Flags);
end;

function TSysApiServer.utf8_valid(const S:LongString):Boolean;
begin
 Result:=_crw_utf8.utf8_valid(sb(S));
end;

function TSysApiServer.utf8_errpos(const S:LongString):SizeInt;
begin
 Result:=_crw_utf8.utf8_errpos(sb(S));
end;

function TSysApiServer.utf8_length(const S:LongString):SizeInt;
begin
 Result:=_crw_utf8.utf8_length(sb(S));
end;

function TSysApiServer.utf8_encode_wide(const S:UnicodeString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_encode_wide(S));
end;

function TSysApiServer.utf8_decode_wide(const S:LongString):UnicodeString;
begin
 Result:=_crw_utf8.utf8_decode_wide(sb(S));
end;

function TSysApiServer.utf8_encode_ansi(const S:LongString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_encode_ansi(sb(S)));
end;

function TSysApiServer.utf8_decode_ansi(const S:LongString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_decode_ansi(sb(S)));
end;

function TSysApiServer.utf8_copy(const S:LongString; i,n:SizeInt):LongString;
begin
 Result:=sb(_crw_utf8.utf8_copy(sb(S),i,n));
end;

function TSysApiServer.utf8_chrpos(const S:LongString; i:SizeInt):SizeInt;
begin
 Result:=_crw_utf8.utf8_chrpos(sb(S),i);
end;

function TSysApiServer.utf8_ord(const S:LongString; i:SizeInt=1):SizeInt;
begin
 Result:=_crw_utf8.utf8_ord(sb(S),i);
end;

function TSysApiServer.utf8_chr(c:SizeInt):LongString;
begin
 Result:=sb(_crw_utf8.utf8_chr(c));
end;

function TSysApiServer.utf8_uppercase(const S:LongString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_uppercase(sb(S)));
end;

function TSysApiServer.utf8_lowercase(const S:LongString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_lowercase(sb(S)));
end;

function TSysApiServer.utf8_fixbroken(const S:LongString):LongString;
begin
 Result:=sb(_crw_utf8.utf8_fixbroken(sb(S)));
end;

function TSysApiServer.utf8_sametext(const S1,S2:LongString):Boolean;
begin
 Result:=_crw_utf8.utf8_sametext(sb(S1),sb(S2));
end;

function TSysApiServer.utf8_fix_cp(const S:LongString; cp:Word=0):LongString;
begin
 Result:=sb(_crw_utf8.utf8_fix_cp(sb(S),cp));
end;

function TSysApiServer.IntToStrBase(Value:LongInt; Base:Integer=10; Width:Integer=0):LongString;
begin
 Result:=sb(_crw_str.IntToStrBase(Value,Base,Width));
end;

function TSysApiServer.StrToIntBase(S:LongString; Base:Integer=10; Def:Integer=0):LongInt;
begin
 Result:=_crw_str.StrToIntBase(sb(S),Base,Def);
end;

function TSysApiServer.htonl(l:Integer):Integer;
begin
 Result:=sockets.htonl(l);
end;

function TSysApiServer.ntohl(l:Integer):Integer;
begin
 Result:=sockets.ntohl(l);
end;

function TSysApiServer.htons(s:Integer):Integer;
begin
 Result:=sockets.htons(s);
end;

function TSysApiServer.ntohs(s:Integer):Integer;
begin
 Result:=sockets.ntohs(s);
end;

function TSysApiServer.strtimefmt(const fmt:LongString; ms:Double):LongString;
begin
 Result:=_crw_rtc.StrTimeFmt(fmt,ms,stfm_Ignore);
end;

function TSysApiServer.IsNonEmptyStr(const S:LongString):Boolean;
begin
 Result:=_crw_str.IsNonEmptyStr(sb(s));
end;

 //////////////////////////////////
 // TGuiApiServer implementation //
 //////////////////////////////////
constructor TGuiApiServer.Create(aOwner:TCrwApiServer);
begin
 inherited Create;
 myOwner:=aOwner;
end;

destructor  TGuiApiServer.Destroy;
begin
 if Assigned(myOwner) then myOwner.myGuiApiServer:=nil;
 inherited;
end;

procedure TGuiApiServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TGuiApiServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TGuiApiServer.ServerActionsBeforeExecution;
begin
end;

procedure TGuiApiServer.ServerActionsAfterExecution;
begin
end;

function TGuiApiServer.CrwApi:ICrwApi;
begin
 if Assigned(myOwner)
 then Result:=myOwner.CrwApi
 else Result:=nil;
end;

procedure TGuiApiServer.Echo(const Msg:LongString);
begin
 _crw_alloc.Echo(sb(Msg));
end;

function  TGuiApiServer.Warning(const Msg:LongString):word;
begin
 if IsMainThread
 then Result:=_crw_apptools.Warning(sb(Msg))
 else Result:=mrNone;
end;

function  TGuiApiServer.Information(const Msg:LongString):word;
begin
 if IsMainThread
 then Result:=_crw_apptools.Information(sb(Msg))
 else Result:=mrNone;
end;

function  TGuiApiServer.YesNo(const Msg:LongString):word;
begin
 if IsMainThread
 then Result:=_crw_apptools.YesNo(sb(Msg))
 else Result:=mrNone;
end;

function  TGuiApiServer.YesNoCancel(const Msg:LongString):word;
begin
 if IsMainThread
 then Result:=_crw_apptools.YesNoCancel(sb(Msg))
 else Result:=mrNone;
end;

function  TGuiApiServer.Error(const Msg:LongString):word;
begin
 if IsMainThread
 then Result:=_crw_apptools.Error(sb(Msg))
 else Result:=mrNone;
end;

function  TGuiApiServer.Trouble(Check:boolean; const Msg:LongString):boolean;
begin
 if IsMainThread
 then Result:=_crw_apptools.Trouble(Check,sb(Msg))
 else Result:=Check;
end;

function  TGuiApiServer.NoProblem(Check:boolean; const Msg:LongString):Boolean;
begin
 if IsMainThread
 then Result:=_crw_apptools.NoProblem(Check,sb(Msg))
 else Result:=Check;
end;

function  TGuiApiServer.ListBoxMenu(const Caption,Title:LongString; const ItemList:LongString):Integer;
begin
 if IsMainThread
 then Result:=Form_ListBoxSelection.ListBoxMenu(sb(Caption),sb(Title),sb(ItemList))
 else Result:=-1;
end;

function  TGuiApiServer.ListBoxMultiSelection(const Caption,Title:LongString; const ItemList:LongString;
                                              Extended:Boolean=false; GetIndex:Boolean=false):LongString;
begin
 if IsMainThread
 then Result:=sb(Form_ListBoxSelection.ListBoxMultiSelection(sb(Caption),sb(Title),sb(ItemList),Extended,GetIndex))
 else Result:='';
end;

function  TGuiApiServer.TextEditDialog(const Caption,Title:LongString; const TextToEdit:LongString):LongString;
begin
 if IsMainThread
 then Result:=sb(Form_TextEditDialog.TextEditDialog(sb(Caption),sb(Title),sb(TextToEdit)))
 else Result:=TextToEdit;
end;

function  TGuiApiServer.UsesBlaster:Boolean;
begin
 Result:=_crw_snd.UsesBlaster;
end;

procedure TGuiApiServer.Voice(const PlayList:LongString);
begin
 _crw_snd.Voice(sb(PlayList));
end;

procedure TGuiApiServer.StopBlaster;
begin
 _crw_snd.StopBlaster;
end;

procedure TGuiApiServer.AddSoundLibrary(const Path:LongString; MaxSubDirs:LongInt=0);
begin
 _crw_snd.AddSoundLibrary(sb(Path),MaxSubDirs);
end;

procedure TGuiApiServer.ApplicationProcessMessages;
begin
 if IsMainThread then SafeApplicationProcessMessages;
end;

procedure TGuiApiServer.UpdateSystemConsole;
begin
 if IsMainThread then ExecuteSystemConsoleMonitoring;
end;

function  TGuiApiServer.RGB(r,g,b:Integer):TColor;
begin
 Result:=_crw_plut.RGB(r,g,b);
end;

function  TGuiApiServer.CgaColorNameEng(CgaColor:Integer):LongString;
begin
 Result:=sb(_crw_plut.CgaColorNameEng(CgaColor));
end;

function  TGuiApiServer.CgaColorNameRus(CgaColor:Integer):LongString;
begin
 Result:=sb(_crw_plut.CgaColorNameRus(CgaColor));
end;

function  TGuiApiServer.CgaColorByName(const S:LongString):Integer;
begin
 Result:=_crw_plut.CgaColorByName(sb(S));
end;

function  TGuiApiServer.CgaToRGBColor(CgaColor:Integer):TColor;
begin
 Result:=_crw_plut.CgaToRGBColor(CgaColor);
end;

function  TGuiApiServer.ApplicationHandle:THandle;
begin
 Result:=_crw_apptools.ApplicationHandle;
end;

function TGuiApiServer.SendToMainConsole(const Msg:LongString):Integer;
begin
 Result:=_crw_alloc.SendToMainConsole(sb(Msg));
end;

 //////////////////////////////////
 // TDaqApiHelper implementation //
 //////////////////////////////////
type
 TDaqApiHelper = class helper for TDaqApiServer
  function DevDaqPascal:TDaqPascal;
  function HasDaqPascal:Boolean;
 end;

function TDaqApiHelper.DevDaqPascal:TDaqPascal;
begin
 Result:=nil;
 if (Self=nil) then raise EDaqApi.Create('Nil DaqApi.');
 if (myOwner=nil) then raise EDaqApi.Create('Nil CrwApi.');
 if (myOwner.myTarget<>ForDataAcquisition)
 then raise EDaqApi.Create('Expected Target=ForDataAcquisition.');
 if (myDaqDev=nil) then raise EDaqApi.Create('Nil DaqDevice.');
 Result:=(myDaqDev as TProgramDevice).DaqPascal;
 if (Result=nil) then raise EDaqApi.Create('Nil DaqPascal.');
end;

function TDaqApiHelper.HasDaqPascal:Boolean;
begin
 Result:=false;
 if (Self=nil) then Exit;
 if (myOwner=nil) then Exit;
 if (myDaqDev=nil) then Exit;
 if (myOwner.myTarget<>ForDataAcquisition) then Exit;
 Result:=Assigned((myDaqDev as TProgramDevice).DaqPascal);
end;

 //////////////////////////////////
 // TDaqApiServer implementation //
 //////////////////////////////////
constructor TDaqApiServer.Create(aOwner:TCrwApiServer);
begin
 inherited Create;
 myOwner:=aOwner;
 myDaqDev:=nil;
 myDaqCmd:=0;
 myDaqDS:=nil;
end;

destructor  TDaqApiServer.Destroy;
begin
 if Assigned(myOwner) then myOwner.myDaqApiServer:=nil;
 DaqDev:=nil;
 DaqCmd:=0;
 DaqDataSheet(0);
 inherited;
end;

procedure TDaqApiServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TDaqApiServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TDaqApiServer.ServerActionsBeforeExecution;
begin
 if myOwner.Target <> ForDataAcquisition then begin
  DaqDev:=nil;
  DaqCmd:=0;
 end;
end;

procedure TDaqApiServer.ServerActionsAfterExecution;
begin
 if myOwner.Target <> ForDataAcquisition then begin
  DaqDev:=nil;
  DaqCmd:=0;
 end;
end;

function TDaqApiServer.CrwApi:ICrwApi;
begin
 if Assigned(myOwner)
 then Result:=myOwner.CrwApi
 else Result:=nil;
end;

function TDaqApiServer.EncodeRef(Ref:TObject):Integer;
begin
 Result:=0;
 if Ref is TMasterObject then Result:=(Ref as TMasterObject).Ref else
 if Ref is TMasterForm   then Result:=(Ref as TMasterForm).Ref;
end;

function TDaqApiServer.DecodeRef(Ref:Integer):TObject;
begin
 Result:=ObjectRegistry[Ref];
end;

function  TDaqApiServer.DaqCommand:Integer;
begin
 Result:=DaqCmd;
end;

function  TDaqApiServer.DaqDataSheet(aSize:Integer):Pointer;
begin
 aSize:=Max(0,aSize);
 if AllocSize(myDaqDS)<>aSize then begin
  if aSize=0 then Deallocate(myDaqDS) else
  if not Reallocate(myDaqDS,aSize)
  then RAISE EDaqApi.Create(RusEng('Нет памяти!','Out of memory!'));
 end;
 Result:=myDaqDS;
end;

function  TDaqApiServer.DaqIsLoaded:Boolean;
begin
 Result:=Daq.Timer.isStart;
end;

function  TDaqApiServer.DaqIsStarted:Boolean;
begin
 Result:=Daq.AcqTimer.isStart;
end;

function  TDaqApiServer.DaqConfigFile:LongString;
begin
 Result:=sb(Daq.ConfigFile);
end;

function  TDaqApiServer.DaqDataPath:LongString;
begin
 Result:=sb(Daq.DataPath);
end;

function  TDaqApiServer.DaqTempPath:LongString;
begin
 Result:=sb(Daq.TempPath);
end;

function  TDaqApiServer.FindTag(const Name:LongString):Integer;
begin
 Result:=_crw_daqtags.FindTag(sb(Name));
end;

function  TDaqApiServer.CreateTag(const Name:LongString; Typ:Integer):Integer;
begin
 Result:=_crw_daqtags.InitTag(sb(Name),Typ);
end;

function  TDaqApiServer.FreeTag(tag:Integer):Boolean;
begin
 Result:=_crw_daqtags.FreeTag(tag);
end;

function  TDaqApiServer.TypeTag(tag:Integer):Integer;
begin
 Result:=_crw_daqtags.TypeTag(tag);
end;

function  TDaqApiServer.NameTag(tag:Integer):LongString;
begin
 Result:=sb(_crw_daqtags.NameTag(tag));
end;

function  TDaqApiServer.iGetTag(tag:Integer):Integer;
begin
 Result:=_crw_daqtags.iGetTag(tag);
end;

function  TDaqApiServer.rGetTag(tag:Integer):Double;
begin
 Result:=_crw_daqtags.rGetTag(tag);
end;

function  TDaqApiServer.sGetTag(tag:Integer):LongString;
begin
 Result:=sb(_crw_daqtags.sGetTag(tag));
end;

function  TDaqApiServer.iSetTag(tag:Integer; i:Integer):Boolean;
begin
 Result:=_crw_daqtags.iSetTag(tag,i);
end;

function  TDaqApiServer.rSetTag(tag:Integer; r:Double):Boolean;
begin
 Result:=_crw_daqtags.rSetTag(tag,r);
end;

function  TDaqApiServer.sSetTag(tag:Integer; const s:LongString):Boolean;
begin
 Result:=_crw_daqtags.sSetTag(tag,sb(s));
end;

function  TDaqApiServer.ThermoCoupleCount:Integer;
begin
 Result:=_crw_couple.ThermoCoupleCount;
end;

function  TDaqApiServer.FindThermoCouple(const CoupleName:LongString):Integer;
begin
 Result:=_crw_couple.FindThermoCouple(sb(CoupleName));
end;

function  TDaqApiServer.ThermoCoupleName(CoupleID:Integer):LongString;
begin
 Result:=sb(_crw_couple.ThermoCoupleName(CoupleID));
end;

function  TDaqApiServer.ThermoCoupleRange(CoupleID:Integer):TRect2D;
begin
 Result:=TRect2D(_crw_couple.ThermoCoupleRange(CoupleID));
end;

function  TDaqApiServer.ThermoCoupleValue(CoupleID:Integer; Argument:Double; Inverted:Boolean):Double;
begin
 Result:=_crw_couple.ThermoCouple(CoupleID,Argument,Inverted);
end;

function  TDaqApiServer.RelationshipCount:Integer;
begin
 Result:=_crw_couple.RelationshipCount;
end;

function  TDaqApiServer.FindRelationship(const RelationName:LongString):Integer;
begin
 Result:=_crw_couple.FindRelationship(sb(RelationName));
end;

function  TDaqApiServer.RelationshipName(RelationID:Integer):LongString;
begin
 Result:=sb(_crw_couple.RelationshipName(RelationID));
end;

function  TDaqApiServer.RelationshipRange(RelationID:Integer):TRect2D;
begin
 Result:=TRect2D(_crw_couple.RelationshipRange(RelationID));
end;

function  TDaqApiServer.RelationshipValue(RelationID:Integer; Argument:Double; Inverted:Boolean):Double;
begin
 Result:=_crw_couple.Relationship(RelationID,Argument,Inverted);
end;

function TDaqApiServer.Time;
begin
 Result:=Daq.Timer.LocalTime;
end;

function TDaqApiServer.TimeUnits:Double;
begin
 Result:=Daq.Timer.LocalTimeUnits;
end;

function  TDaqApiServer.NumAIs:Integer;
begin
 Result:=DaqDev.NumAnalogInputs;
end;

function  TDaqApiServer.NumDIs:Integer;
begin
 Result:=DaqDev.NumDigitalInputs;
end;

function  TDaqApiServer.NumAOs:Integer;
begin
 Result:=DaqDev.NumAnalogOutputs;
end;

function  TDaqApiServer.NumDOs:Integer;
begin
 Result:=DaqDev.NumDigitalOutputs;
end;

function  TDaqApiServer.NumCals:Integer;
begin
 Result:=DaqDev.NumCalibrations;
end;

function  TDaqApiServer.RefAI(n:Integer):Integer;
begin
 Result:=EncodeRef(DaqDev.AnalogInputCurve[n]);
end;

function  TDaqApiServer.RefDI(n:Integer):Integer;
begin
 Result:=EncodeRef(DaqDev.DigitalInputCurve[n]);
end;

function  TDaqApiServer.RefAO(n:Integer):Integer;
begin
 Result:=EncodeRef(DaqDev.AnalogOutputCurve[n]);
end;

function  TDaqApiServer.RefDO(n:Integer):Integer;
begin
 Result:=EncodeRef(DaqDev.DigitalOutputCurve[n]);
end;

function  TDaqApiServer.RefCalibr(n:Integer):Integer;
begin
 Result:=EncodeRef(DaqDev.Calibration[n]);
end;

function  TDaqApiServer.GetCurveLength(c:Integer):Integer;
begin
 Result:=(DecodeRef(c) as TCurve).Count;
end;

procedure TDaqApiServer.SetCurveLength(c,aLength:Integer);
begin
 (DecodeRef(c) as TCurve).Count:=aLength;
end;

function  TDaqApiServer.GetCurvePoint(c,i:Integer):TPoint2D;
begin
 Result:=TPoint2D((DecodeRef(c) as TCurve)[i]);
end;

procedure TDaqApiServer.SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
begin
 with aPoint do (DecodeRef(c) as TCurve)[i]:=Point2D(X,Y);
end;

procedure TDaqApiServer.CurveAddPoint(c:Integer; const aPoint:TPoint2D);
begin
 with aPoint do (DecodeRef(c) as TCurve).AddPoint(X,Y);
end;

procedure TDaqApiServer.CurveAddPoints(c:Integer; const aX,aY:array of Double; aCount:Integer);
begin
 (DecodeRef(c) as TCurve).AddPoints(aX,aY,aCount);
end;

procedure TDaqApiServer.CurveDeletePoint(c,i:Integer);
begin
 (DecodeRef(c) as TCurve).DeletePoint(i);
end;

procedure TDaqApiServer.CurveDeletePoints(c,i,aCount:Integer);
begin
 (DecodeRef(c) as TCurve).DeletePoints(i,aCount);
end;

procedure TDaqApiServer.CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
begin
 with aPoint do (DecodeRef(c) as TCurve).InsertPoint(i,X,Y);
end;

procedure TDaqApiServer.CurveInsertPoints(c,i:Integer; const aX,aY:array of Double; aCount:LongInt);
begin
 (DecodeRef(c) as TCurve).InsertPoints(i,aX,aY,aCount);
end;

function  TDaqApiServer.GetCurveName(c:Integer):LongString;
begin
 Result:=sb((DecodeRef(c) as TCurve).Name);
end;

function  TDaqApiServer.GetCurveColor(c:Integer):Integer;
begin
 Result:=(DecodeRef(c) as TCurve).Color;
end;

function  TDaqApiServer.GetCurveStyle(c:Integer):Integer;
begin
 Result:=(DecodeRef(c) as TCurve).Style;
end;

function  TDaqApiServer.GetCurveComment(c:Integer):LongString;
begin
 Result:=sb((DecodeRef(c) as TCurve).Comment.Text);
end;

procedure TDaqApiServer.SetCurveComment(c:Integer; const aComment:LongString);
begin
 (DecodeRef(c) as TCurve).Comment.Text:=sb(aComment);
end;

function  TDaqApiServer.CurveGetIndexAt(c:Integer; t:Double):Integer;
begin
 Result:=(DecodeRef(c) as TCurve).GetIndexAt(t);
end;

function  TDaqApiServer.CurveInterpolateAt(c:Integer; t:Double):Double;
begin
 Result:=(DecodeRef(c) as TCurve).Interpolate(t);
end;

function  TDaqApiServer.CurveSmoothAt(c:Integer; t:Double; Window:Double; Power,K1,K2:Integer):Double;
begin
 Result:=(DecodeRef(c) as TCurve).Smooth(t,Window,Power,K1,K2);
end;

function  TDaqApiServer.CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double; SortMethod:Integer=1):Boolean;
begin
 Result:=(DecodeRef(c) as TCurve).Mediana(i,j,aIndex,aValue,TSortMethod(SortMethod and 1));
end;

function  TDaqApiServer.CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10; Small:Integer=2):LongInt;
begin
 Result:=(DecodeRef(c) as TCurve).Flags(AbsEps,RelEps,Small);
end;

function  TDaqApiServer.GetCurveLimits(c:Integer):TRect2D;
begin
 Result:=TRect2D((DecodeRef(c) as TCurve).Limits);
end;

function  TDaqApiServer.GetCurveIntegral(c:Integer; a,b:Double):Double;
begin
 Result:=DaqIntegral(DecodeRef(c) as TCurve,a,b);
end;

function  TDaqApiServer.GetCurveLastPoint(c:Integer):TPoint2D;
begin
 Result:=TPoint2D((DecodeRef(c) as TCurve).LastPoint);
end;

function  TDaqApiServer.PutEv(w,n:Integer; t,d0:Double; d1:Double=0):Boolean;
begin
 Result:=DaqDev.PutDaqEvent(DaqEvent(w,n,t,d0,d1));
end;

function  TDaqApiServer.PutAO(n:Integer; t,d:Double):Boolean;
begin
 Result:=DaqDev.PutDaqEvent(DaqEvent(evAnalog+evCompress,n,t,d));
end;

function  TDaqApiServer.PutDO(n:Integer; t,d:Double):Boolean;
begin
 Result:=DaqDev.PutDaqEvent(DaqEvent(evDigital,n,t,d));
end;

function  TDaqApiServer.getai(n:Integer;t:Double):Double;
begin
 Result:=DaqDev.SmoothAnalogInputCurve(n,t);
end;

function  TDaqApiServer.getai_n(n:Integer):Integer;
begin
 Result:=DaqDev.AnalogInputCurve[n].Count;
end;

function  TDaqApiServer.getai_xn(n:Integer):Double;
begin
 Result:=DaqDev.AnalogInputCurve[n].LastPoint.X;
end;

function  TDaqApiServer.getai_yn(n:Integer):Double;
begin
 Result:=DaqDev.AnalogInputCurve[n].LastPoint.Y;
end;

function  TDaqApiServer.getai_xi(n,i:Integer):Double;
begin
 Result:=DaqDev.AnalogInputCurve[n][i-1].X;
end;

function  TDaqApiServer.getai_yi(n,i:Integer):Double;
begin
 Result:=DaqDev.AnalogInputCurve[n][i-1].Y;
end;

function  TDaqApiServer.getdi(n:Integer;t:Double):Double;
var
 c : TCurve;
begin
 Result:=0;
 c:=DaqDev.DigitalInputCurve[n];
 if Assigned(c) then begin
  c.Lock;
  try
   if (c.Count>0) and (t>=0) then begin
    if t>=c.LastPoint.X
    then Result:=c.LastPoint.Y
    else Result:=c[c.GetIndexAt(t)].Y;
   end;
  finally
   c.Unlock;
  end;
 end;
end;

function  TDaqApiServer.getdi_n(n:Integer):Integer;
begin
 Result:=DaqDev.DigitalInputCurve[n].Count;
end;

function  TDaqApiServer.getdi_xn(n:Integer):Double;
begin
 Result:=DaqDev.DigitalInputCurve[n].LastPoint.X;
end;

function  TDaqApiServer.getdi_yn(n:Integer):Double;
begin
 Result:=DaqDev.DigitalInputCurve[n].LastPoint.Y;
end;

function  TDaqApiServer.getdi_xi(n,i:Integer):Double;
begin
 Result:=DaqDev.DigitalInputCurve[n][i-1].X;
end;

function  TDaqApiServer.getdi_yi(n,i:Integer):Double;
begin
 Result:=DaqDev.DigitalInputCurve[n][i-1].Y;
end;

function  TDaqApiServer.calibr(n:Integer; t,d:Double):Double;
begin
 Result:=DaqDev.Transform(n,t,d);
end;

function  TDaqApiServer.fixerror(n:Integer):Boolean;
begin
 Result:=(n=byte(n));
 DaqDev.FixError(n);
end;

function  TDaqApiServer.aimap(i,n:Integer):Integer;
begin
 Result:=DaqDev.ConstructAnalogInputMap(i,n);
end;

function  TDaqApiServer.dimap(i,n:Integer):Integer;
begin
 Result:=DaqDev.ConstructDigitalInputMap(i,n);
end;

function  TDaqApiServer.aomap(i,n:Integer):Integer;
begin
 Result:=DaqDev.ConstructAnalogOutputMap(i,n);
end;

function  TDaqApiServer.domap(i,n:Integer):Integer;
begin
 Result:=DaqDev.ConstructDigitalOutputMap(i,n);
end;

function  TDaqApiServer.diword(i,n:Integer):Integer;
begin
 Result:=DaqDev.ConstructDigitalInputWord(i,n);
end;

function TDaqApiServer.ClickButton:Integer;
begin
 Result:=(DaqDev as TSoftwareDevice).ClickButton;
end;

function TDaqApiServer.ClickSensor:LongString;
begin
 Result:=sb((DaqDev as TSoftwareDevice).ClickSensor);
end;

function TDaqApiServer.ClickTag:Integer;
begin
 Result:=(DaqDev as TSoftwareDevice).ClickTag;
end;

function TDaqApiServer.Voice(const msg:LongString):Boolean;
begin
 Result:=Daq.SoundOn;
 Daq.Voice(sb(msg));
end;

function TDaqApiServer.WinDraw(const Caption:LongString):Boolean;
begin
 Result:=DevDaqPascal._windraw(sb(Caption));
end;

function TDaqApiServer.WinShow(const Caption:LongString):Boolean;
begin
 Result:=DevDaqPascal._winshow(sb(Caption));
end;

function TDaqApiServer.WinHide(const Caption:LongString):Boolean;
begin
 Result:=DevDaqPascal._winhide(sb(Caption));
end;

function TDaqApiServer.WinSelect(const Caption:LongString):Boolean;
begin
 Result:=DevDaqPascal._winselect(sb(Caption));
end;

function  TDaqApiServer.ReadIni(const s:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._readini(sb(s)));
end;

function  TDaqApiServer.GetPortB(Address:Word):Byte;
begin
 Result:=_crw_pio.Port[Address];
end;

procedure TDaqApiServer.SetPortB(Address:Word; Value:Byte);
begin
 _crw_pio.Port[Address]:=Value;
end;

function  TDaqApiServer.GetPortW(Address:Word):Word;
begin
 Result:=_crw_pio.PortW[Address];
end;

procedure TDaqApiServer.SetPortW(Address:Word; Value:Word);
begin
 _crw_pio.PortW[Address]:=Value;
end;

function  TDaqApiServer.GetPortL(Address:Word):LongWord;
begin
 Result:=_crw_pio.PortL[Address];
end;

procedure TDaqApiServer.SetPortL(Address:Word; Value:LongWord);
begin
 _crw_pio.PortL[Address]:=Value;
end;

function  TDaqApiServer.RegisterErr(const s:LongString):Integer;
begin
 Result:=Daq.RegisterErrMsg(sb(s));
end;

function  TDaqApiServer.GetErrCount(n:Integer):Double;
begin
 case n of
  -2:        Result:=FullDaqDeviceList.ErrorsTotal;
  -1:        Result:=DaqDev.ErrorsTotal;
  0..255:    Result:=DaqDev.ErrorsCount[n];
  256..511:  Result:=FullDaqDeviceList.ErrorsCount[n-256];
  else       Result:=0;
 end;
end;

function TDaqApiServer.RunCount:Double;
begin
 Result:=DevDaqPascal.runcount;
end;

function TDaqApiServer.FindObjectRef(const What,Name:LongString):Integer;
begin
  Result:=DevDaqPascal._reffind(sb(What)+' '+sb(Name));
end;

function  TDaqApiServer.DeviceName(d:Integer):LongString;
begin
 Result:=sb((DecodeRef(d) as TDaqDevice).Name);
end;

function  TDaqApiServer.DeviceModel(d:Integer):LongString;
begin
 Result:=sb((DecodeRef(d) as TDaqDevice).Model);
end;

function  TDaqApiServer.DeviceFamily(d:Integer):LongString;
begin
 Result:=sb((DecodeRef(d) as TDaqDevice).Family);
end;

function  TDaqApiServer.DeviceAction(d:Integer):Boolean;
var arg:LongString;
begin
 arg:=sb((DecodeRef(d) as TDaqDevice).Name);
 Result:=DevDaqPascal._action(arg);
end;

function  TDaqApiServer.DeviceClear(d:Integer):Boolean;
var arg:LongString;
begin
 arg:=sb((DecodeRef(d) as TDaqDevice).Name);
 Result:=DevDaqPascal._clear(arg);
end;

function  TDaqApiServer.DeviceClearDevice(d:Integer):Boolean;
var arg:LongString;
begin
 arg:=sb((DecodeRef(d) as TDaqDevice).Name);
 Result:=DevDaqPascal._cleardevice(arg);
end;

function  TDaqApiServer.DeviceStart(d:Integer):Boolean;
var arg:LongString;
begin
 arg:=sb((DecodeRef(d) as TDaqDevice).Name);
 Result:=DevDaqPascal._start(arg);
end;

function  TDaqApiServer.DeviceStop(d:Integer):Boolean;
var arg:LongString;
begin
 arg:=sb((DecodeRef(d) as TDaqDevice).Name);
 Result:=DevDaqPascal._stop(arg);
end;

function  TDaqApiServer.DeviceMessage(d:Integer; const msg:LongString):Double;
begin
 Result:=DevDaqPascal._devsend(d,sb(msg));
end;

function  TDaqApiServer.SaveCrw(const ArgList:LongString):Boolean;
var arg:LongString;
begin
 arg:=sb(ArgList);
 Result:=DevDaqPascal._SaveCrw(arg);
end;

function  TDaqApiServer.SpecMarker(const Caption:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Caption);
 Result:=DevDaqPascal._specmarker(arg);
end;

function  TDaqApiServer.SpecMarkerL(const Caption:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Caption);
 Result:=DevDaqPascal._specmarkerl(arg);
end;

function  TDaqApiServer.SpecMarkerR(const Caption:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Caption);
 Result:=DevDaqPascal._specmarkerr(arg);
end;

function  TDaqApiServer.SpecRoiL(const Caption:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Caption);
 Result:=DevDaqPascal._specroil(arg);
end;

function  TDaqApiServer.SpecRoiR(const Caption:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Caption);
 Result:=DevDaqPascal._specroir(arg);
end;

function  TDaqApiServer.GlobalCalc(const Expression:LongString):Double;
var arg:LongString;
begin
 arg:=sb(Expression);
 Result:=DevDaqPascal._global(arg);
end;

function TDaqApiServer.UartOpen(aPortNum:Integer; aBaudrate:Integer=9600;
                  aParity:Integer=NOPARITY; aDataBits:Integer=8; aStopBits:Integer=ONESTOPBIT;
                  aXonXoff:Boolean=false; aFifoSize:LongInt=4096;aDcbFlags:LongInt=0):Boolean;
begin
 Result:=uart[aPortNum].Open(aBaudrate,aParity,aDataBits,aStopBits,aXonXoff,aFifoSize,aDcbFlags);
end;

function  TDaqApiServer.UartOpenConfig(const Section:LongString='[SerialPort-COM1]'):Integer;
var s:LongString; buf:TParsingBuffer;
begin
 Result:=0; s:='';
 if ReadIniFileAlpha(Daq.ConfigFile,sb(Section),'Port%a',s) then
 if ScanVarInteger(svUpCase,StrCopyBuff(buf,s),'COM%i',Result)<>nil then
 if not uart[Result].OpenConfig(Daq.ConfigFile,sb(Section)) then Result:=0;
end;

procedure TDaqApiServer.UartClose(aPortNum:Integer);
begin
 uart[aPortNum].Close;
end;

function  TDaqApiServer.UartActive(aPortNum:Integer):Boolean;
begin
 Result:=uart[aPortNum].Active;
end;

function  TDaqApiServer.UartWriteBuf(aPortNum:Integer; const Data:PChar; Count:Cardinal):Cardinal;
begin
 Result:=uart[aPortNum].WriteBuf(Data,Count);
end;

function  TDaqApiServer.UartReadBuf(aPortNum:Integer; var Data:PChar; MaxCount:Cardinal):Cardinal;
begin
 Result:=uart[aPortNum].ReadBuf(Data,MaxCount);
end;

function  TDaqApiServer.UartWriteStr(aPortNum:Integer; const Data:LongString):Boolean;
begin
 Result:=uart[aPortNum].WriteStr(sb(Data));
end;

function  TDaqApiServer.UartReadStr(aPortNum:Integer; MaxCount:Byte=255):LongString;
begin
 Result:=sb(uart[aPortNum].ReadStr(MaxCount));
end;

procedure TDaqApiServer.UartRxClear(aPortNum:Integer);
begin
 uart[aPortNum].RxClear;
end;

procedure TDaqApiServer.UartTxClear(aPortNum:Integer);
begin
 uart[aPortNum].TxClear;
end;

function TDaqApiServer.UartRxCount(aPortNum:Integer):Integer;
begin
 Result:=uart[aPortNum].RxCount;
end;

function TDaqApiServer.UartTxCount(aPortNum:Integer):Integer;
begin
 Result:=uart[aPortNum].TxCount;
end;

function TDaqApiServer.UartRxSpace(aPortNum:Integer):Integer;
begin
 Result:=uart[aPortNum].RxSpace;
end;

function TDaqApiServer.UartTxSpace(aPortNum:Integer):Integer;
begin
 Result:=uart[aPortNum].TxSpace;
end;

function TDaqApiServer.UartFindCheckSum(Data:PChar; Length:Integer):Byte;
begin
 Result:=_crw_uart.FindCheckSum(Data,Length);
end;

function TDaqApiServer.UartCatCheckSum(const DataStr:LongString):LongString;
begin
 Result:=sb(_crw_uart.CatCheckSum(sb(DataStr)));
end;

function TDaqApiServer.UartCatCheckSumCR(const DataStr:LongString; CR:Char=ASCII_CR):LongString;
begin
 Result:=sb(_crw_uart.CatCheckSumCR(sb(DataStr),CR));
end;

function TDaqApiServer.UartTruncCheckSum(const DataStr:LongString):LongString;
begin
 Result:=sb(_crw_uart.TruncCheckSum(sb(DataStr)));
end;

function TDaqApiServer.UartTruncCheckSumCR(const DataStr:LongString; CR:Char=ASCII_CR):LongString;
begin
 Result:=sb(_crw_uart.TruncCheckSumCR(sb(DataStr),CR));
end;

function  TDaqApiServer.AdamFindSlot(const aSlotDeviceName:LongString=''):Integer;
var ref:Integer; dev:TDaqDevice;
begin
 Result:=0;
 if IsEmptyStr(sb(aSlotDeviceName)) then begin
  if DaqDev is TProgramDevice then begin
   dev:=DevDaqPascal.Adam_Slot;
   if dev is TAdamSlot then Result:=EncodeRef(dev);
  end;
 end else begin
  ref:=FindObjectRef('DEVICE',sb(aSlotDeviceName));
  if DecodeRef(ref) is TAdamSlot then Result:=ref;
 end;
end;

function  TDaqApiServer.AdamGetStatus(aSlot:Integer):Integer;
var AdamSlot:TAdamSlot;
begin
 Result:=0;
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then Result:=_Adam_Status(AdamSlot);
end;

function  TDaqApiServer.AdamSendRequest(aSlot:Integer; aRequest:LongString; aTimeOut:Integer):Boolean;
var AdamSlot:TAdamSlot;
begin
 Result:=false;
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then Result:=_Adam_Request(AdamSlot,sb(aRequest),aTimeOut);
end;

function  TDaqApiServer.AdamGetRequest(aSlot:Integer):LongString;
var AdamSlot:TAdamSlot; WhatToGet:LongString;
begin
 Result:='';
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then begin
  WhatToGet:='REQUEST';
  Result:=sb(_Adam_Get(AdamSlot,sb(WhatToGet)));
 end;
end;

function  TDaqApiServer.AdamGetRequestTime(aSlot:Integer):Double;
var AdamSlot:TAdamSlot;
begin
 Result:=0;
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then Result:=_Adam_ReqTime(AdamSlot); 
end;

function  TDaqApiServer.AdamGetAnswer(aSlot:Integer):LongString;
var AdamSlot:TAdamSlot; WhatToGet:LongString;
begin
 Result:='';
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then begin
  WhatToGet:='ANSWER';
  Result:=sb(_Adam_Get(AdamSlot,sb(WhatToGet)));
 end;
end;

function  TDaqApiServer.AdamGetPort(aSlot:Integer):Integer;
var PortNum:LongInt; AdamSlot:TAdamSlot; WhatToGet:LongString;
begin
 Result:=0;
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then begin
  WhatToGet:='PORT';
  if Str2Long(_Adam_Get(AdamSlot,sb(WhatToGet)),PortNum) then Result:=PortNum;
 end;
end;

function  TDaqApiServer.AdamGetAddress(aSlot:Integer):LongString;
var AdamSlot:TAdamSlot; WhatToGet:LongString;
begin
 Result:='';
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then begin
  WhatToGet:='ADDRESS';
  Result:=sb(_Adam_Get(AdamSlot,sb(WhatToGet)));
 end;
end;

function  TDaqApiServer.AdamGetUsesCheckSum(aSlot:Integer):Boolean;
var CheckSum:LongInt; AdamSlot:TAdamSlot; WhatToGet:LongString;
begin
 Result:=false;
 AdamSlot:=TAdamSlot(DecodeRef(aSlot));
 if TObject(AdamSlot) is TAdamSlot then begin
  WhatToGet:='USESCHECKSUM';
  if StrHex2Long(_Adam_Get(AdamSlot,sb(WhatToGet)),CheckSum) then Result:=CheckSum<>0;
 end;
end;

function  TDaqApiServer.ClickParams(const Args:LongString):LongString;
begin
 Result:=sb((DaqDev as TProgramDevice).ClickSensorParams(sb(Args)));
end;

function  TDaqApiServer.Add3D(x,y,z:Double):Boolean;
begin
 Result:=DevDaqPascal._add3d(x,y,z);
end;

function  TDaqApiServer.Plot3D(Nx,Ny:Integer; x1,y1,x2,y2:Double; const opt:LongString):Boolean;
var arg:LongString;
begin
 arg:=sb(opt);
 Result:=DevDaqPascal._plot3d(Nx,Ny,x1,y1,x2,y2,arg);
end;

function  TDaqApiServer.Edit(const s:LongString):LongString;
var tmp:LongString;
begin
 tmp:=sb(s);
 Result:=sb(DevDaqPascal._edit(tmp));
end;

function  TDaqApiServer.EditState:Integer;
begin
 Result:=DevDaqPascal._editstate;
end;

function  TDaqApiServer.DevSend(d:Integer; const msg:LongString):Double;
begin
 Result:=DevDaqPascal._devsend(d,sb(msg));
end;

function  TDaqApiServer.FileRef(const FName,DefExt,BaseFile:LongString):LongString;
begin
 Result:=sb(Daq.FileRef(sb(FName),sb(DefExt),sb(BaseFile)));
end;

function  TDaqApiServer.DevPost(d:Integer; const msg:LongString):Double;
begin
 Result:=DevDaqPascal._devpost(d,sb(msg));
end;

procedure TDaqApiServer.Echo(const Msg:LongString);
begin
 if HasDaqPascal
 then DevDaqPascal.OutFifo.PutText(sb(Msg)+EOL)
 else _crw_alloc.Echo(sb(Msg));
end;

function TDaqApiServer.RedirectStdIn(var stdIn:Text):boolean;
begin
 Result:=RedirectStdInToFifo(stdIn,DevDaqPascal.InpFifo);
end;

function TDaqApiServer.RedirectStdOut(var stdOut:Text):boolean;
begin
 Result:=RedirectStdOutToFifo(stdOut,DevDaqPascal.OutFifo);
end;

function TDaqApiServer.str2shortcut(const s:LongString):Integer;
begin
 Result:=DevDaqPascal._str2shortcut(sb(s));
end;

function TDaqApiServer.shortcut2str(shortcut:Integer):LongString;
begin
 Result:=sb(DevDaqPascal._shortcut2str(shortcut));
end;

function TDaqApiServer.awakeflag:Boolean;
begin
 Result:=DevDaqPascal._awakeflag;
end;

function TDaqApiServer.crypt_ctrl(const s:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._crypt_ctrl(sb(s)));
end;

function TDaqApiServer.crypt_encode(const s,key:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._crypt_encode(sb(s),sb(key)));
end;

function TDaqApiServer.crypt_decode(const s,key:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._crypt_decode(sb(s),sb(key)));
end;

function TDaqApiServer.getmd5fromstr(const s:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._getmd5fromstr(sb(s)));
end;

function TDaqApiServer.getmd5fromfile(const s:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._getmd5fromfile(sb(s)));
end;

function TDaqApiServer.getmd5fromtext(ref:Integer):LongString;
begin
 Result:=sb(DevDaqPascal._getmd5fromtext(ref));
end;

function TDaqApiServer.wdt_reset(flag:Boolean):Double;
begin
 Result:=DevDaqPascal._wdt_reset(flag);
end;

function TDaqApiServer.guard_check(const s:LongString):Integer;
begin
 Result:=DevDaqPascal._guard_check(sb(s));
end;

function TDaqApiServer.timebase:Double;
begin
 Result:=DevDaqPascal._timebase;
end;

function TDaqApiServer.daqfileref(f,e:LongString):LongString;
begin
 Result:=sb(DevDaqPascal._daqfileref(sb(f),sb(e)));
end;

function TDaqApiServer.GetClockRes(what:Integer):Double;
begin
 Result:=DevDaqPascal._GetClockRes(what);
end;

function TDaqApiServer.SetClockRes(res:Double):Double;
begin
 Result:=DevDaqPascal._SetClockRes(res);
end;

function TDaqApiServer.cpu_start:Boolean;
begin
 Result:=DevDaqPascal._cpu_start;
end;

function TDaqApiServer.cpu_clock:Double;
begin
 Result:=DevDaqPascal._cpu_clock;
end;

function TDaqApiServer.pidaffinity(pid,mask:Integer):Integer;
begin
 Result:=DevDaqPascal._pidaffinity(pid,mask);
end;

function TDaqApiServer.devaffinity(ref,mask:Integer):Integer;
begin
 Result:=DevDaqPascal._devaffinity(ref,mask);
end;

function TDaqApiServer.clickwrite(const lines:LongString):Integer;
begin
 Result:=DevDaqPascal._clickwrite(sb(lines));
end;

function TDaqApiServer.clickread:Integer;
begin
 Result:=DevDaqPascal._clickread;
end;

function TDaqApiServer.clickwhat:Integer;
begin
 Result:=DevDaqPascal._clickwhat;
end;

function TDaqApiServer.clickwrote:Integer;
begin
 Result:=DevDaqPascal._clickwrote;
end;

function TDaqApiServer.clickfilter(aFilter:Integer):Integer;
begin
 Result:=DevDaqPascal._clickfilter(aFilter);
end;

function TDaqApiServer.clickawaker(aAwaker:Integer):Integer;
begin
 Result:=DevDaqPascal._clickawaker(aAwaker);
end;

function TDaqApiServer.taglock(lock:Boolean):Integer;
begin
 Result:=DevDaqPascal._taglock(lock);
end;

function TDaqApiServer.getai_par(n:integer; id:integer):Double;
begin
 Result:=DevDaqPascal._getai_par(n,id);
end;

function TDaqApiServer.getao_par(n:integer; id:integer):Double;
begin
 Result:=DevDaqPascal._getao_par(n,id);
end;

function TDaqApiServer.getdi_par(n:integer; id:integer):Double;
begin
 Result:=DevDaqPascal._getdi_par(n,id);
end;

function TDaqApiServer.getdo_par(n:integer; id:integer):Double;
begin
 Result:=DevDaqPascal._getdo_par(n,id);
end;

 //////////////////////////////////
 // TDanApiServer implementation //
 //////////////////////////////////
constructor TDanApiServer.Create(aOwner:TCrwApiServer);
begin
 inherited Create;
 myOwner:=aOwner;
 mySrcWin:=nil;
 myDstWin:=nil;
 myArgums:='';
end;

destructor  TDanApiServer.Destroy;
begin
 if Assigned(myOwner) then myOwner.myDanApiServer:=nil;
 SrcCurveWindow:=nil;
 DstCurveWindow:=nil;
 Arguments:='';
 inherited;
end;

procedure TDanApiServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TDanApiServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TDanApiServer.ServerActionsBeforeExecution;
begin
 if myOwner.Target = ForDataAnalysis then begin
  DstCurveWindow:=NewCurveWindow;
  DstCurveWindow.Width:=400;
  DstCurveWindow.Height:=300;
  if SrcCurveWindow.Ok then begin
   mySrcFix.dp:=SrcCurveWindow.IsDataProtected;
   mySrcFix.ca:=SrcCurveWindow.CloseAction;
   mySrcFix.ws:=SrcCurveWindow.WindowState;
   SrcCurveWindow.IsDataProtected:=true;
   SrcCurveWindow.CloseAction:=caMinimize;
  end;
  if DstCurveWindow.Ok then begin
   myDstFix.dp:=DstCurveWindow.IsDataProtected;
   myDstFix.ca:=DstCurveWindow.CloseAction;
   myDstFix.ws:=DstCurveWindow.WindowState;
   DstCurveWindow.IsDataProtected:=true;
   DstCurveWindow.CloseAction:=caMinimize;
  end;
 end else begin
  SrcCurveWindow:=nil;
  DstCurveWindow:=nil;
  Arguments:='';
 end;
end;

procedure TDanApiServer.ServerActionsAfterExecution;
begin
 if myOwner.Target = ForDataAnalysis then begin
  if SrcCurveWindow.Ok then begin
   SrcCurveWindow.IsDataProtected:=mySrcFix.dp;
   SrcCurveWindow.CloseAction:=mySrcFix.ca;
   SrcCurveWindow.WindowState:=mySrcFix.ws;
  end;
  if DstCurveWindow.Ok then begin
   DstCurveWindow.IsDataProtected:=myDstFix.dp;
   DstCurveWindow.CloseAction:=myDstFix.ca;
   DstCurveWindow.WindowState:=myDstFix.ws;
   if DstCurveWindow.Curves.Count>0 then begin
    DstCurveWindow.AutoRange;
    SdiMan.ActivateChild(DstCurveWindow);
   end else begin
    DstCurveWindow.Free;
   end;
  end;
 end;
 SrcCurveWindow:=nil;
 DstCurveWindow:=nil;
 Arguments:='';
end;

function TDanApiServer.CrwApi:ICrwApi;
begin
 if Assigned(myOwner)
 then Result:=myOwner.CrwApi
 else Result:=nil;
end;

function TDanApiServer.iowin(w:Integer;e:Boolean=true):TFormCurveWindow;
begin
 if w>0 then Result:=mySrcWin else
 if w<0 then Result:=myDstWin else Result:=CurveClipboard;
 if e and not Assigned(Result) then
 case Sign(w) of
  -1 : RAISE EDanApi.Create(RusEng('Не задано окно - приемник.','Target window not found.'));
   0 : RAISE EDanApi.Create(RusEng('Нет окна - буфера обмена.', 'Clipboard window not found.'));
  +1 : RAISE EDanApi.Create(RusEng('Не задано окно - источник.','Source window not found.'));
 end;
end;

function TDanApiServer.iocrv(c:Integer;e:Boolean=true):TCurve;
begin
 if c<>0
 then Result:=iowin(c,e).Curves[abs(c)-1]
 else Result:=iowin(c,e).DefCurve;
 if e and not Assigned(Result)
 then RAISE EDanApi.Create(Format(RusEng('Недопустимая кривая %d.','Invalid curve %d.'),[c]));
end;

function TDanApiServer.iocheckindex(c,i:Integer; e:Boolean=true):Boolean;
begin
 Result:=Cardinal(i)<Cardinal(iocrv(c,e).Count);
 if e and not Result
 then RAISE EDanApi.Create(Format(RusEng('Недопустимый индекс %d кривой номер %d.',
                                         'Invalid index %d of curve number %d.'),[i,c]));
end;

function TDanApiServer.GetArguments:LongString;
begin
 Result:=sb(Arguments);
end;

function TDanApiServer.GetArgumentAsString(const aName:LongString):LongString;
var P:TText;
begin
 Result:='';
 P:=NewText;
 try
  P.Text:=GetArguments;
  Result:=sb(P.GetVar(sb(aName)));
 finally
  Kill(P);
 end;
end;

function  TDanApiServer.GetArgumentAsInteger(const aName:LongString; var aValue:Integer):boolean;
begin
 Result:=Str2Int(GetArgumentAsString(sb(aName)),aValue);
end;

function  TDanApiServer.GetArgumentAsDouble(const aName:LongString; var aValue:Double):boolean;
begin
 Result:=Str2Real(GetArgumentAsString(sb(aName)),aValue);
end;

function  TDanApiServer.WindowExists(w:Integer):Boolean;
begin
 Result:=iowin(w,false).Ok;
end;

function  TDanApiServer.CurveExists(c:Integer):Boolean;
begin
 Result:=iocrv(c,false).Ok;
end;

function  TDanApiServer.GetWindowCaption(w:Integer):LongString;
begin
 Result:=sb(iowin(w).Caption);
end;

procedure TDanApiServer.SetWindowCaption(w:Integer;const aCaption:LongString);
begin
 if w<>0 then iowin(w).Caption:=sb(aCaption);
end;

function  TDanApiServer.GetWindowTitle(w:Integer):LongString;
begin
 Result:=sb(iowin(w).Title);
end;

procedure TDanApiServer.SetWindowTitle(w:Integer;const aTitle:LongString);
begin
 if w<>0 then iowin(w).Title:=sb(ReplaceAlignStr(sb(aTitle),true));
end;

function  TDanApiServer.GetWindowLegend(w:Integer):LongString;
begin
 Result:=sb(iowin(w).Legend);
end;

procedure TDanApiServer.SetWindowLegend(w:Integer;const aLegend:LongString);
begin
 if w<>0 then iowin(w).Legend:=sb(ReplaceAlignStr(sb(aLegend),true));
end;

function  TDanApiServer.GetWindowRoi(w:Integer):TRect2D;
begin
 Result:=TRect2D(iowin(w).Roi);
end;

procedure TDanApiServer.SetWindowRoi(w:Integer; const aRoi:TRect2D);
begin
 if w<>0 then with aRoi do iowin(w).Roi:=Rect2D(A.X, A.Y, B.X, B.Y);
end;

function  TDanApiServer.GetWindowComment(w:Integer):LongString;
begin
 Result:=sb(iowin(w).Comment.Text);
end;

procedure TDanApiServer.SetWindowComment(w:Integer;const aComment:LongString);
begin
 if w<>0 then iowin(w).Comment.Text:=sb(aComment);
end;

function  TDanApiServer.GetSelectedCurve(w:Integer):Integer;
begin
 Result:=(iowin(w).DefCurveNum+1)*Sign(w);
end;

procedure TDanApiServer.SetSelectedCurve(w,c:Integer);
begin
 if w<>0 then iowin(w).DefCurveNum:=abs(c)-1;
end;

function  TDanApiServer.GetCurvesCount(w:Integer):Integer;
begin
 Result:=iowin(w,false).Curves.Count;
end;

procedure TDanApiServer.SetCurvesCount(w,aCount:Integer);
begin
 if w<>0 then
 with iowin(w) do begin
  aCount:=Max(0,aCount);
  Curves.Count:=Min(Curves.Count,aCount);
  while Curves.Count<aCount do AddCurve(NewCurve);
 end;
end;

function  TDanApiServer.GetCurveLength(c:Integer):Integer;
begin
 Result:=iocrv(c).Count;
end;

procedure TDanApiServer.SetCurveLength(c,aLength:Integer);
begin
 if c<>0 then iocrv(c).Count:=aLength;
end;

function  TDanApiServer.GetCurvePoint(c,i:Integer):TPoint2D;
begin
 iocheckindex(c,i);
 Result:=TPoint2D(iocrv(c)[i]);
end;

procedure TDanApiServer.SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
begin
 iocheckindex(c,i);
 if c<>0 then with aPoint do iocrv(c)[i]:=Point2D(X,Y);
end;

procedure TDanApiServer.CurveAddPoint(c:Integer; const aPoint:TPoint2D);
begin
 if c<>0 then with aPoint do iocrv(c).AddPoint(X,Y);
end;

procedure TDanApiServer.CurveAddPoints(c:Integer; const aX,aY:array of Double; aCount:Integer);
begin
 if c<>0 then iocrv(c).AddPoints(aX, aY, aCount);
end;

procedure TDanApiServer.CurveDeletePoint(c,i:Integer);
begin
 iocheckindex(c,i);
 if c<>0 then iocrv(c).DeletePoint(i);
end;

procedure TDanApiServer.CurveDeletePoints(c,i,aCount:Integer);
begin
 iocheckindex(c,i);
 if c<>0 then iocrv(c).DeletePoints(i,aCount);
end;

procedure TDanApiServer.CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
begin
 iocheckindex(c,i);
 if c<>0 then with aPoint do iocrv(c).InsertPoint(i,X,Y);
end;

procedure TDanApiServer.CurveInsertPoints(c,i:Integer; const aX,aY:array of Double; aCount:LongInt);
begin
 iocheckindex(c,i);
 if c<>0 then iocrv(c).InsertPoints(i,aX,aY,aCount);
end;

function  TDanApiServer.GetCurveName(c:Integer):LongString;
begin
 Result:=sb(iocrv(c).Name);
end;

procedure TDanApiServer.SetCurveName(c:Integer; const aName:LongString);
begin
 if c<>0 then iocrv(c).Name:=sb(aName);
end;

function  TDanApiServer.GetCurveColor(c:Integer):Integer;
begin
 Result:=iocrv(c).Color;
end;

procedure TDanApiServer.SetCurveColor(c:Integer; aColor:Integer);
begin
 if c<>0 then iocrv(c).Color:=aColor;
end;

function  TDanApiServer.GetCurveStyle(c:Integer):Integer;
begin
 Result:=iocrv(c).Style;
end;

procedure TDanApiServer.SetCurveStyle(c:Integer; aStyle:Integer);
begin
 if c<>0 then iocrv(c).Style:=aStyle;
end;

function  TDanApiServer.GetCurveComment(c:Integer):LongString;
begin
 Result:=sb(iocrv(c).Comment.Text);
end;

procedure TDanApiServer.SetCurveComment(c:Integer; const aComment:LongString);
begin
 if c<>0 then iocrv(c).Comment.Text:=sb(aComment);
end;

function  TDanApiServer.CurveGetIndexAt(c:Integer; t:Double):Integer;
begin
 Result:=iocrv(c).GetIndexAt(t);
end;

function  TDanApiServer.CurveInterpolateAt(c:Integer; t:Double):Double;
begin
 Result:=iocrv(c).Interpolate(t);
end;

function  TDanApiServer.CurveSmoothAt(c:Integer; t:Double; Window:Double; Power,K1,K2:Integer):Double;
begin
 Result:=iocrv(c).Smooth(t,Window,Power,K1,K2);
end;

function  TDanApiServer.CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double; SortMethod:Integer=1):Boolean;
begin
 iocheckindex(c,i);
 iocheckindex(c,j);
 Result:=iocrv(c).Mediana(i,j,aIndex,aValue,TSortMethod(SortMethod and 1));
end;

function  TDanApiServer.CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10; Small:Integer=2):LongInt;
begin
 Result:=iocrv(c,false).Flags(AbsEps,RelEps,Small);
end;

procedure TDanApiServer.CurveAssign(c1,c2:Integer);
begin
 iocrv(c1).Assign(iocrv(c2));
end;

procedure TDanApiServer.CurveCatenate(c1,c2:Integer);
begin
 iocrv(c1).Catenate(iocrv(c2));
end;

function  TDanApiServer.CreateCurve(w:Integer; const aName:LongString='';
                                    aColor:Integer=0;  aStyle:Integer=0;
                                    aComment:LongString=''):Integer;
begin
 Result:=0;
 if w<>0 then
 if iowin(w).Ok then begin
  iowin(w).AddCurve(NewCurve);
  Result:=iowin(w).Curves.Count*Sign(w);
  SetCurveName(Result,sb(aName));
  SetCurveColor(Result,aColor);
  SetCurveStyle(Result,aStyle);
  SetCurveComment(Result,sb(aComment));
 end;
end;

procedure TDanApiServer.CurveSort(c:Integer; Flags:Integer=0; AbsEps:Double=0;
                                  RelEps:Double=1E-10; Method:Integer=0);
var
 tmp:TCurve;
begin
 tmp:=nil;
 if c<>0 then
 try
  tmp:=NewCurveSorted(iocrv(c),Flags,AbsEps,RelEps,TSortMethod(Method and 1));
  if not tmp.Ok
  then RAISE EDanApi.Create(Format(RusEng('Не могу сортировать кривую %d.',
                                          'Could not sort curve %d.'),[c]));
  iocrv(c).Assign(tmp);
 finally
  Kill(tmp);
 end;
end;

function  TDanApiServer.GetCurveLimits(c:Integer):TRect2D;
begin
 Result:=TRect2D(iocrv(c).Limits);
end;

procedure TDanApiServer.CurveIntegrate(c:Integer; AFreeConst:Double=0);
var
 tmp:TCurve;
begin
 tmp:=nil;
 if c<>0 then
 try
  tmp:=NewCurveIntegral(iocrv(c),AFreeConst);
  if not tmp.Ok
  then RAISE EDanApi.Create(Format(RusEng('Не могу интегрировать кривую %d.',
                                          'Could not intergate curve %d.'),[c]));
  iocrv(c).Assign(tmp);
 finally
  Kill(tmp);
 end;
end;

procedure TDanApiServer.CurveDifferentiate(c:Integer);
var
 tmp:TCurve;
begin
 tmp:=nil;
 if c<>0 then
 try
  tmp:=NewCurveDerivative(iocrv(c));
  if not tmp.Ok
  then RAISE EDanApi.Create(Format(RusEng('Не могу дифференцировать кривую %d.',
                                          'Could not differentiate curve %d.'),[c]));
  iocrv(c).Assign(tmp);
 finally
  Kill(tmp);
 end;
end;

procedure TDanApiServer.CurveMedianFiltr(c:Integer; Width:Integer=1; AbsEps:Double=0; RelEps:Double=1E-10);
var
 tmp:TCurve;
begin
 tmp:=nil;
 if c<>0 then
 try
  tmp:=NewCurveMedianFiltr(iocrv(c),Width,AbsEps,RelEps);
  if not tmp.Ok
  then RAISE EDanApi.Create(Format(RusEng('Не могу фильтровать кривую %d.',
                                          'Could not filter curve %d.'),[c]));
  iocrv(c).Assign(tmp);
 finally
  Kill(tmp);
 end;
end;

function  TDanApiServer.GetCurveIntegral(c:Integer; a,b:Double):Double;
begin
 Result:=DaqIntegral(iocrv(c),a,b);
end;

function  TDanApiServer.GetCurveLastPoint(c:Integer):TPoint2D;
begin
 Result:=TPoint2D(iocrv(c).LastPoint);
end;

 /////////////////////////////////////////////
 // TScriptInterpreterServer implementation //
 /////////////////////////////////////////////
constructor TScriptInterpreterServer.Create;
begin
 inherited;
 myEe:=NewExpressionEvaluator;
end;

destructor  TScriptInterpreterServer.Destroy;
begin
 Kill(myEe);
 inherited;
end;

procedure TScriptInterpreterServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TScriptInterpreterServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

function TScriptInterpreterServer.GetValue(aName:PChar; var aValue:Double):Boolean;
begin
 Result:=myEe.GetValue(aName,aValue);
end;

function TScriptInterpreterServer.SetValue(aName:PChar; var aValue:Double):Boolean;
begin
 Result:=myEe.SetValue(aName,aValue);
end;

procedure TScriptInterpreterServer.Clear;
begin
 myEe.Clear;
end;

function TScriptInterpreterServer.Evaluate(aExpression:PChar):Integer;
begin
 Result:=myEe.EvaluateLine(aExpression);
end;

function TScriptInterpreterServer.GetScript:LongString;
begin
 Result:=sb(myEe.Script);
end;

procedure TScriptInterpreterServer.SetScript(const aScript:LongString);
begin
 myEe.Script:=sb(aScript);
end;

function TScriptInterpreterServer.RunScript:Integer;
begin
 Result:=myEe.RunScript;
end;

function TScriptInterpreterServer.Status:Integer;
begin
 Result:=myEe.Status;
end;

function TScriptInterpreterServer.StatusMessage:LongString;
begin
 Result:=sb(ee_ErrorMessage(Status));
end;

 ///////////////////////////////////////
 // TPolynomWrapServer implementation //
 ///////////////////////////////////////
function  TPolynomWrapServer.GetPower:Integer;
begin
 Result:=myPoly.Power;
end;

procedure TPolynomWrapServer.SetPower(aPower:Integer);
begin
 myPoly.Power:=aPower;
end;

function  TPolynomWrapServer.GetCoeff(aIndex:Integer):Double;
begin
 Result:=myPoly[aIndex];
end;

procedure TPolynomWrapServer.SetCoeff(aIndex:Integer; aValue:Double);
begin
 myPoly[aIndex]:=aValue;
end;

function  TPolynomWrapServer.GetCenter:Double;
begin
 Result:=myPoly.Center;
end;

procedure TPolynomWrapServer.SetCenter(aCenter:Double);
begin
 myPoly.Center:=aCenter;
end;

function  TPolynomWrapServer.GetScale:Double;
begin
 Result:=myPoly.Scale;
end;

procedure TPolynomWrapServer.SetScale(aScale:Double);
begin
 myPoly.Scale:=aScale;
end;

function  TPolynomWrapServer.Get(at:Double; aDerivNum:Integer=0):Double;
begin
 Result:=myPoly.Get(at,aDerivNum);
end;

function  TPolynomWrapServer.Find(const X,Y:array of Double; NumPoints:Integer; Eps:double=1E-14):boolean;
begin
 Result:=myPoly.Find(X,Y,NumPoints,Eps);
end;

function  TPolynomWrapServer.FindWeight(const X,Y,W:array of Double; NumPoints:Integer; Eps:double=1E-14):boolean;
begin
 Result:=myPoly.FindWeight(X,Y,W,NumPoints,Eps);
end;

constructor TPolynomWrapServer.Create(aPower:TPolynomPower=1; aCenter:Double=0; aScale:Double=1);
begin
 inherited Create;
 myPoly:=_crw_lsqpoly.NewPolynom(aPower,aCenter,aScale);
end;

destructor TPolynomWrapServer.Destroy;
begin
 Kill(myPoly);
 inherited Destroy;
end;

procedure TPolynomWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TPolynomWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

 ///////////////////////////////////////////
 // TRfaDataBaseWrapServer implementation //
 ///////////////////////////////////////////
constructor TRfaDataBaseWrapServer.Create(aOwner:TSysApiServer);
begin
 inherited Create;
 myOwner:=aOwner;
end;

destructor  TRfaDataBaseWrapServer.Destroy;
begin
 myOwner.myRfa:=nil;
 inherited;
end;

procedure TRfaDataBaseWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TRfaDataBaseWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

function  TRfaDataBaseWrapServer.GetAccessible(AtomId:Integer):Boolean;
begin
 Result:=Rfa.Accessible[AtomId] and (ord(rf_LAST_FIELD)=ord(rfa_LAST_FIELD));
end;

function  TRfaDataBaseWrapServer.GetAsDouble(AtomId:Integer; FieldId:TRfaFieldId):Double;
begin
 Result:=Rfa.AsDouble[AtomId,TRfaFields(FieldId)];
end;

function  TRfaDataBaseWrapServer.GetAsString(AtomId:Integer; FieldId:TRfaFieldId):LongString;
begin
 Result:=sb(Rfa.AsString[AtomId,TRfaFields(FieldId)]);
end;

function  TRfaDataBaseWrapServer.GetXRayLine(AtomId:Integer; LineId:TRfaFieldId):TRfaXRayLine;
begin
 Result:=TRfaXRayLine(Rfa.XRayLine[AtomId,TRfaFields(LineId)]);
end;

function  TRfaDataBaseWrapServer.GetAtomSymbol(AtomId:Integer):LongString;
begin
 Result:=sb(Rfa.AtomSymbol[AtomId]);
end;

function  TRfaDataBaseWrapServer.GetFieldName(FieldId:TRfaFieldId):LongString;
begin
 Result:=sb(Rfa.FieldName[TRfaFields(FieldId)]);
end;

function  TRfaDataBaseWrapServer.GetLineName(LineId:TRfaFieldId):LongString;
begin
 Result:=sb(Rfa.LineName[TRfaFields(LineId)]);
end;

function  TRfaDataBaseWrapServer.GetCrossSectionTableCount(AtomId:Integer):Integer;
begin
 Result:=Rfa.CrossSectionTableCount[AtomId];
end;

function  TRfaDataBaseWrapServer.GetCrossSectionTable(AtomId:Integer; aChan:TRfaCrossSectionFieldId; aIndex:Integer):Double;
begin
 Result:=Rfa.CrossSectionTable[AtomId,TRfaCrossSectionFields(aChan),aIndex];
end;

function  TRfaDataBaseWrapServer.FindAtom(const aSymbol:LongString):Integer;
begin
 Result:=Rfa.FindAtom(sb(aSymbol));
end;

function  TRfaDataBaseWrapServer.FindLine(const aLineName:LongString):TRfaFieldId;
begin
 Result:=TRfaFieldId(Rfa.FindLine(sb(aLineName)));
end;

function  TRfaDataBaseWrapServer.FindCrossSection(AtomID:Integer; CrossId:TRfaCrossSectionFieldId; Energy:Double):Double;
begin
 Result:=Rfa.FindCrossSection(AtomID,TRfaCrossSectionFields(CrossId),Energy);
end;

 /////////////////////////////////////////////
 // TLongIntVectorWrapServer implementation //
 /////////////////////////////////////////////
constructor TLongIntVectorWrapServer.Create(aVector:TLongIntVector; aOwns:Boolean);
begin
 inherited Create;
 myVector:=aVector;
 myOwns:=aOwns;
end;

destructor  TLongIntVectorWrapServer.Destroy;
begin
 if myOwns then Kill(myVector);
 inherited;
end;

procedure TLongIntVectorWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TLongIntVectorWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TLongIntVectorWrapServer.Lock;
begin
 TheVector.Lock;
end;

procedure TLongIntVectorWrapServer.UnLock;
begin
 TheVector.UnLock;
end;

function  TLongIntVectorWrapServer.GetLength:LongInt;
begin
 Result:=TheVector.Length;
end;

procedure TLongIntVectorWrapServer.SetLength(aLength:LongInt);
begin
 TheVector.Length:=aLength;
end;

function  TLongIntVectorWrapServer.GetOrigin:LongInt;
begin
 Result:=TheVector.Origin;
end;

procedure TLongIntVectorWrapServer.SetOrigin(aOrigin:LongInt);
begin
 TheVector.Origin:=aOrigin;
end;

function  TLongIntVectorWrapServer.GetExceptions:Boolean;
begin
 Result:=TheVector.Exceptions;
end;

procedure TLongIntVectorWrapServer.SetExceptions(aExcepts:Boolean);
begin
 TheVector.Exceptions:=aExcepts;
end;

function  TLongIntVectorWrapServer.GetItem(aIndex:LongInt):LongInt;
begin
 Result:=TheVector[aIndex];
end;

procedure TLongIntVectorWrapServer.SetItem(aIndex:LongInt; const aValue:LongInt);
begin
 TheVector[aIndex]:=aValue;
end;

function  TLongIntVectorWrapServer.GetItemPtr(aIndex:LongInt):PLongInt;
begin
 Result:=TheVector.ItemPtr[aIndex];
end;

 ////////////////////////////////////////////
 // TDoubleVectorWrapServer implementation //
 ////////////////////////////////////////////
constructor TDoubleVectorWrapServer.Create(aVector:TDoubleVector; aOwns:Boolean);
begin
 inherited Create;
 myVector:=aVector;
 myOwns:=aOwns;
end;

destructor  TDoubleVectorWrapServer.Destroy;
begin
 if myOwns then Kill(myVector);
 inherited;
end;

procedure TDoubleVectorWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TDoubleVectorWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TDoubleVectorWrapServer.Lock;
begin
 TheVector.Lock;
end;

procedure TDoubleVectorWrapServer.UnLock;
begin
 TheVector.UnLock;
end;

function  TDoubleVectorWrapServer.GetLength:LongInt;
begin
 Result:=TheVector.Length;
end;

procedure TDoubleVectorWrapServer.SetLength(aLength:LongInt);
begin
 TheVector.Length:=aLength;
end;

function  TDoubleVectorWrapServer.GetOrigin:LongInt;
begin
 Result:=TheVector.Origin;
end;

procedure TDoubleVectorWrapServer.SetOrigin(aOrigin:LongInt);
begin
 TheVector.Origin:=aOrigin;
end;

function  TDoubleVectorWrapServer.GetExceptions:Boolean;
begin
 Result:=TheVector.Exceptions;
end;

procedure TDoubleVectorWrapServer.SetExceptions(aExcepts:Boolean);
begin
 TheVector.Exceptions:=aExcepts;
end;

function  TDoubleVectorWrapServer.GetItem(aIndex:LongInt):Double;
begin
 Result:=TheVector[aIndex];
end;

procedure TDoubleVectorWrapServer.SetItem(aIndex:LongInt; const aValue:Double);
begin
 TheVector[aIndex]:=aValue;
end;

function  TDoubleVectorWrapServer.GetItemPtr(aIndex:LongInt):PDouble;
begin
 Result:=TheVector.ItemPtr[aIndex];
end;

 /////////////////////////////////////////////
 // TLongIntMatrixWrapServer implementation //
 /////////////////////////////////////////////
constructor TLongIntMatrixWrapServer.Create(aMatrix:TLongIntMatrix; aOwns:Boolean);
begin
 inherited Create;
 myMatrix:=aMatrix;
 myOwns:=aOwns;
end;

destructor  TLongIntMatrixWrapServer.Destroy;
begin
 if myOwns then Kill(myMatrix);
 inherited;
end;

procedure TLongIntMatrixWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TLongIntMatrixWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TLongIntMatrixWrapServer.Lock;
begin
 TheMatrix.Lock;
end;

procedure TLongIntMatrixWrapServer.UnLock;
begin
 TheMatrix.UnLock;
end;

function  TLongIntMatrixWrapServer.GetRows:LongInt;
begin
 Result:=TheMatrix.Rows;
end;

procedure TLongIntMatrixWrapServer.SetRows(aRows:LongInt);
begin
 TheMatrix.Rows:=aRows;
end;

function  TLongIntMatrixWrapServer.GetColumns:LongInt;
begin
 Result:=TheMatrix.Columns;
end;

procedure TLongIntMatrixWrapServer.SetColumns(aColumns:LongInt);
begin
 TheMatrix.Columns:=aColumns;
end;

function  TLongIntMatrixWrapServer.GetOrigin:LongInt;
begin
 Result:=TheMatrix.Origin;
end;

procedure TLongIntMatrixWrapServer.SetOrigin(aOrigin:LongInt);
begin
 TheMatrix.Origin:=aOrigin;
end;

function  TLongIntMatrixWrapServer.GetExceptions:Boolean;
begin
 Result:=TheMatrix.Exceptions;
end;

procedure TLongIntMatrixWrapServer.SetExceptions(aExcepts:Boolean);
begin
 TheMatrix.Exceptions:=aExcepts;
end;

function  TLongIntMatrixWrapServer.GetItem(aRow,aColumn:LongInt):LongInt;
begin
 Result:=TheMatrix[aRow,aColumn];
end;

procedure TLongIntMatrixWrapServer.SetItem(aRow,aColumn:LongInt; const aValue:LongInt);
begin
 TheMatrix[aRow,aColumn]:=aValue;
end;

function  TLongIntMatrixWrapServer.GetItemPtr(aRow,aColumn:LongInt):PLongInt;
begin
 Result:=TheMatrix.ItemPtr[aRow,aColumn];
end;

 /////////////////////////////////////////////
 // TDoubleMatrixWrapServer implementation //
 /////////////////////////////////////////////
constructor TDoubleMatrixWrapServer.Create(aMatrix:TDoubleMatrix; aOwns:Boolean);
begin
 inherited Create;
 myMatrix:=aMatrix;
 myOwns:=aOwns;
end;

destructor  TDoubleMatrixWrapServer.Destroy;
begin
 if myOwns then Kill(myMatrix);
 inherited;
end;

procedure TDoubleMatrixWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TDoubleMatrixWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure TDoubleMatrixWrapServer.Lock;
begin
 TheMatrix.Lock;
end;

procedure TDoubleMatrixWrapServer.UnLock;
begin
 TheMatrix.UnLock;
end;

function  TDoubleMatrixWrapServer.GetRows:LongInt;
begin
 Result:=TheMatrix.Rows;
end;

procedure TDoubleMatrixWrapServer.SetRows(aRows:LongInt);
begin
 TheMatrix.Rows:=aRows;
end;

function  TDoubleMatrixWrapServer.GetColumns:LongInt;
begin
 Result:=TheMatrix.Columns;
end;

procedure TDoubleMatrixWrapServer.SetColumns(aColumns:LongInt);
begin
 TheMatrix.Columns:=aColumns;
end;

function  TDoubleMatrixWrapServer.GetOrigin:LongInt;
begin
 Result:=TheMatrix.Origin;
end;

procedure TDoubleMatrixWrapServer.SetOrigin(aOrigin:LongInt);
begin
 TheMatrix.Origin:=aOrigin;
end;

function  TDoubleMatrixWrapServer.GetExceptions:Boolean;
begin
 Result:=TheMatrix.Exceptions;
end;

procedure TDoubleMatrixWrapServer.SetExceptions(aExcepts:Boolean);
begin
 TheMatrix.Exceptions:=aExcepts;
end;

function  TDoubleMatrixWrapServer.GetItem(aRow,aColumn:LongInt):Double;
begin
 Result:=TheMatrix[aRow,aColumn];
end;

procedure TDoubleMatrixWrapServer.SetItem(aRow,aColumn:LongInt; const aValue:Double);
begin
 TheMatrix[aRow,aColumn]:=aValue;
end;

function  TDoubleMatrixWrapServer.GetItemPtr(aRow,aColumn:LongInt):PDouble;
begin
 Result:=TheMatrix.ItemPtr[aRow,aColumn];
end;

 ///////////////////////////////////
 // TSVDWrapServer implementation //
 ///////////////////////////////////
constructor TSVDWrapServer.Create(A:IDoubleMatrixWrap);
begin
 inherited Create;
 if Assigned(A) and (A.Rows>0) and (A.Cols>0) and (A.Origin=0)
 then mySVD:=NewSVD(A.ItemPtr[0,0]^,A.Rows,A.Cols,A.Cols)
 else mySVD:=nil;
end;

destructor  TSVDWrapServer.Destroy;
begin
 Kill(mySVD);
 inherited;
end;

procedure TSVDWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TSVDWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

function  TSVDWrapServer.GetM:Integer;
begin
 Result:=mySVD.M;
end;

function  TSVDWrapServer.GetN:Integer;
begin
 Result:=mySVD.N;
end;

function  TSVDWrapServer.GetU(i,j:Integer):Double;
begin
 if (Cardinal(i)<Cardinal(mySVD.M)) and (Cardinal(j)<Cardinal(mySVD.N))
 then Result:=mySVD.U[i*mySVD.N+j]
 else Result:=0;
end;

function  TSVDWrapServer.GetW(i:Integer):Double;
begin
 if (Cardinal(i)<Cardinal(mySVD.N))
 then Result:=mySVD.W[i]
 else Result:=0;
end;

function  TSVDWrapServer.GetV(i,j:Integer):Double;
begin
 if (Cardinal(i)<Cardinal(mySVD.N)) and (Cardinal(j)<Cardinal(mySVD.N))
 then Result:=mySVD.V[i*mySVD.N+j]
 else Result:=0;
end;

function  TSVDWrapServer.Eps(AbsEps:Double=0; RelEps:Double=1E-10):Double;
begin
 Result:=mySVD.Eps(AbsEps,RelEps);
end;

function  TSVDWrapServer.Rank(AbsEps:Double=0; RelEps:Double=1E-10):Integer;
begin
 Result:=mySVD.Rank(Eps(AbsEps,RelEps));
end;

function  TSVDWrapServer.Solve(X,Y:IDoubleVectorWrap; AbsEps:Double=0; RelEps:Double=1E-10):Boolean;
begin
 Result:=false;
 if (GetM>0) and (GetN>0) and (X<>nil) and (Y<>nil) then
 if (X.Length=GetN) and (Y.Length=GetM) and (X.Origin=0) and (Y.Origin=0)
 then Result:=mySVD.Solve(X.ItemPtr[0]^,Y.ItemPtr[0]^,Eps(AbsEps,RelEps));
end;

function  TSVDWrapServer.PInvert(AbsEps:Double=0; RelEps:Double=1E-10):IDoubleMatrixWrap;
var Obj:TDoubleMatrixWrapServer; Intf:IDoubleMatrixWrap;
begin
 Result:=nil; Obj:=nil;
 if (GetM>0) and (GetN>0) then Obj:=TDoubleMatrixWrapServer.Create(NewDoubleMatrix(GetN,GetM),true);
 if Assigned(Obj)
    and Obj.GetInterface(IID_IDoubleMatrixWrap,Intf)
    and mySVD.PInvert(Obj.GetItemPtr(0,0)^,GetM,Eps(AbsEps,RelEps))
 then Result:=Intf
 else Obj.Free;
end;

 ////////////////////////////////////
 // TFifoWrapServer implementation //
 ////////////////////////////////////
constructor TFifoWrapServer.Create(aFifo:TFifo; aOwns:Boolean);
begin
 inherited Create;
 myFifo:=aFifo;
 myOwns:=aOwns;
end;

destructor  TFifoWrapServer.Destroy;
begin
 if myOwns then Kill(myFifo);
 inherited;
end;

procedure TFifoWrapServer.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TFifoWrapServer.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

procedure   TFifoWrapServer.Lock;
begin
 myFifo.Lock;
end;

procedure   TFifoWrapServer.UnLock;
begin
 myFifo.UnLock;
end;

procedure   TFifoWrapServer.Clear;
begin
 myFifo.Clear;
end;

function    TFifoWrapServer.GetCount:LongInt;
begin
 Result:=myFifo.Count;
end;

function    TFifoWrapServer.GetSpace:LongInt;
begin
 Result:=myFifo.Space;
end;

function    TFifoWrapServer.GetSize:LongInt;
begin
 Result:=myFifo.Size;
end;

function    TFifoWrapServer.GetName:LongString;
begin
 Result:=sb(myFifo.Name);
end;

procedure   TFifoWrapServer.SetName(const NewName:LongString);
begin
 myFifo.Name:=sb(NewName);
end;

function    TFifoWrapServer.GetTag(i:Integer):LongWord;
begin
 Result:=myFifo.Tag[i];
end;

procedure   TFifoWrapServer.SetTag(i:Integer; NewTag:LongWord);
begin
 myFifo.Tag[i]:=NewTag;
end;

function    TFifoWrapServer.GetLost:Int64;
begin
 Result:=myFifo.Lost;
end;

procedure   TFifoWrapServer.SetLost(N:Int64);
begin
 myFifo.Lost:=N;
end;

function    TFifoWrapServer.Put(Buffer:Pointer; BufferSize:LongInt; TryMode:boolean=false):LongInt;
begin
 Result:=myFifo.Put(Buffer,BufferSize,TryMode);
end;

function    TFifoWrapServer.Get(Buffer:Pointer; BufferSize:LongInt):LongInt;
begin
 Result:=myFifo.Get(Buffer,BufferSize);
end;

function    TFifoWrapServer.PutText(const S:LongString; TryMode:boolean=false):Boolean;
begin
 Result:=myFifo.PutText(sb(S),TryMode);
end;

function    TFifoWrapServer.GetText(MaxLen:Integer=MaxInt):LongString;
begin
 Result:=sb(myFifo.GetText(MaxLen));
end;

 ///////////////////////////////////
 // TDaqDllWrapper implementation //
 ///////////////////////////////////
constructor TDaqDllWrapper.Create(aDevice:TDaqDevice);
begin
 inherited Create;
 myDevice:=aDevice;
 FillChar(myItems,sizeof(myItems),0);
 Exceptions:=false;
end;

destructor  TDaqDllWrapper.Destroy;
var
 i : Integer;
begin
 myDevice:=nil;
 for i:=Low(myItems) to High(myItems) do DaqDllFree(i);
 inherited Destroy;
end;

procedure TDaqDllWrapper.AfterConstruction;
begin
 inherited;
 LogBorn(Self);
end;

procedure TDaqDllWrapper.BeforeDestruction;
begin
 LogKill(Self);
 inherited;
end;

function  TDaqDllWrapper.DaqDllInit(aDllPath:LongString):Integer;
var i:Integer; aName,aSource,aBinary:LongString;
begin
 Result:=0;
 if Ok then
 try
  if IsUnix then aDllPath:=DefaultExtension(aDllPath,'.so');
  if IsWindows then aDllPath:=DefaultExtension(aDllPath,'.dll');
  aDllPath:=UnifyFileAlias(AdaptDllFileName(aDllPath),ua_RealPath);
  if not FileExists(aDllPath)
  then RAISE EDaqDllWrapper.Create(Format('Could not find DLL "%s".',[aDllPath]));
  Lock;
  try
   aBinary:=aDllPath;
   aName:=LowerCase(ExtractDllBaseName(aBinary));
   aSource:=AddPathDelim(ExtractFileDir(aBinary))+aName+FavoritePascalProjectType;
   for i:=Low(myItems) to High(myItems) do with myItems[i] do
   if not Assigned(Api) and not Assigned(Plugin) then begin
    Api:=NewCrwApiServer(ForDataAcquisition,aName,aSource,aBinary);
    Plugin:=NewPluginWrapper(aDllPath,CRWDAQ_PLUGIN_ID);
    if Plugin.DllLoad <> 0 then begin
     Result:=i;
     Break;
    end else begin
     DaqDllFree(i);
     Break;
    end;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do ErrorFound(E,'DaqDllInit');
 end;
end;

function TDaqDllWrapper.DaqDllFree(aDllRef:Integer):Boolean;
begin
 Result:=false;
 if Ok then
 if aDllRef >= Low(myItems) then
 if aDllRef <= High(myItems) then
 try
  Lock;
  try
   with myItems[aDllRef] do begin
    Kill(Api);
    Kill(Plugin);
   end;
  finally
   Unlock;
  end;
  Result:=true;
 except
  on E:Exception do ErrorFound(E,'DaqDllFree');
 end;
end;

function TDaqDllWrapper.DaqDllCall(aDllRef,aCommand:Integer):Boolean;
var
 ErrCode : Int64;
begin
 Result:=false;
 if Ok then
 if aDllRef >= Low(myItems) then
 if aDllRef <= High(myItems) then
 try
  Lock;
  try
   with myItems[aDllRef] do
   if Assigned(Api) and Assigned(Plugin) then begin
    ErrCode:=Api.ExecuteDataAcquisition(Plugin,myDevice,aCommand);
    Result:=(ErrCode=0);
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do ErrorFound(E,'DaqDllCall');
 end;
end;

 ///////////////////////
 // Utility functions //
 ///////////////////////
function NewCrwApiServer(aTarget:Cardinal; const aName,aSource,aBinary:LongString):TCrwApiServer;
begin
 Result:=TCrwApiServer.Create(aTarget,aName,aSource,aBinary);
end;

procedure Kill(var TheObject:TCrwApiServer);
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function  NewDaqDllWrapper(aDevice:TDaqDevice):TDaqDllWrapper;
begin
 Result:=TDaqDllWrapper.Create(aDevice);
end;

procedure Kill(var TheObject:TDaqDllWrapper);
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function GetDataAnalysisPluginsFolder:LongString;
const Dir:LongString='';
begin
 if (Dir='') then begin
  if ReadIniFilePath(SysIniFile,SectSystem,'DataAnalysisPlugins',HomeDir,Dir)
  then Dir:=UnifyFileAlias(FExpand(Dir)) else Dir:='';
 end;
 Result:=Dir;
end;

function GetDataAnalysisTemplatesFolder:LongString;
const Dir:LongString='';
begin
 if (Dir='') then begin
  if ReadIniFilePath(SysIniFile,SectSystem,'DataAnalysisTemplates',HomeDir,Dir)
  then Dir:=UnifyFileAlias(FExpand(Dir)) else Dir:='';
 end;
 Result:=Dir;
end;

function ListOfPascalProjectTypes(Codes:LongString=''):LongString;
var i:Integer;
begin
 Result:='';
 Codes:=TrimDef(Codes,DefaultListOfPascalProjectTypeCodes);
 for i:=1 to Length(Codes) do
 case LoCase(Codes[i]) of
  'l': Result:=Result+'.lpr'+EOL;   // Lazarus
  'd': Result:=Result+'.dpr'+EOL;   // Delphi
  't': Result:=Result+'.ctpr'+EOL;  // CodeTyphon
 end;
end;

function GetPluginsListFromDirectory(Dir:LongString; FileNamesOnly:Boolean;
                                    ProjectType:LongString):LongString;
var P:TText; i:Integer; ProjectExt:LongString;
begin
 Result:='';
 try
  P:=NewText;
  try
   Dir:=UnifyFileAlias(Dir);
   if not DirExists(Dir) then Exit;
   ProjectExt:=Trim(ExtractFileExt(ProjectType));
   if (WordIndex(ProjectExt,ListOfPascalProjectTypes,ScanSpaces)>0)
   then P.Text:=FindAllFilesAsText(Dir,'*'+ProjectExt,true);
   if FileNamesOnly then for i:=0 to P.Count-1 do P[i]:=ExtractFileName(P[i]);
   Result:=P.Text;
  finally
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,nil,'GetPluginsListFromDirectory');
 end;
end;

function GetDataAnalysisPluginsList(FileNamesOnly:Boolean;
                                    ProjectType:LongString):LongString;
begin
 Result:=GetPluginsListFromDirectory(GetDataAnalysisPluginsFolder,FileNamesOnly,ProjectType);
end;

function CreateNewDataAnalysisPlugin(ProjectType,Params:LongString):LongString;
var samp,targ,sdir,tdir,list,exe,cmd:LongString;
begin
 Result:='';
 try
  samp:=''; targ:=''; exe:='';
  tdir:=GetDataAnalysisPluginsFolder;
  sdir:=GetDataAnalysisTemplatesFolder;
  if (sdir='') or not DirExists(sdir) then Exit;
  if (tdir='') or not DirExists(tdir) then Exit;
  ProjectType:=Trim(ExtractFileExt(ProjectType));
  list:=GetPluginsListFromDirectory(sdir,false,ProjectType);
  if (EnterNewDataAnalysisPluginName(list,samp,targ,Params)=mrOk) then begin
   exe:=Fpcup.IniFileLazClone;
   samp:=UnifyFileAlias(samp);
   targ:=LowerCase(Trim(targ));
   cmd:=QArg(MakeRelativePath(exe,ProgName));
   cmd:=cmd+' '+QArg(MakeRelativePath(samp,ProgName));
   cmd:=cmd+' '+QArg(MakeRelativePath(tdir,ProgName));
   cmd:=cmd+' '+targ;
   SendToMainConsole('@run -sw7 -cd ~~ '+cmd+EOL);
   Result:=UnifyFileAlias(AddPathDelim(AddPathDelim(tdir)+targ)+targ+ProjectType);
  end;
 except
  on E:Exception do BugReport(E,nil,'CreateNewDataAnalysisPlugin');
 end;
end;

procedure AddDaqFileToList(const FileName:LongString; const FileDetails:TSearchRec;
                        SubDirLevel:Integer; var Terminate:Boolean; CustomData:Pointer);
begin
 if (WordIndex(UnifyAlias(ExtractFileExt(FileName)),'.dpr .pas .cfg .crc .cal',ScanSpaces)>0)
 then TStringList(CustomData).Add(ExtractFileNameExt(FileName));
end;

function CreateNewDataAcquisitionPlugin:LongString;
var i:Integer; P:TStringList; s,d:LongString;
begin
 Result:='';
 try
  P:=TStringList.Create;
  try
   s:='';
   if ReadIniFilePath(SysIniFile,SectSystem,'DataAcquisitionTemplates',HomeDir,s)
   then if DirExists(s) then ForEachFile(s,'*.dpr *.pas *.cfg *.crc *.cal',AddDaqFileToList,0,P);
   if (P.Count>0) then begin
    P.Sort;
    i:=ListBoxMenu(RusEng('Создание утилиты DAQ','Create DAQ plugin'),
                   RusEng('Выбрать шаблон','Choose template'),P.Text);
    if (i>=0) then begin
     s:=AddPathDelim(s)+P[i];
     d:=ForcePath(TempDir,P[i]);
     if FileCopy(s,d) then Result:=d;
    end;
   end;
  finally
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,nil,'CreateNewDataAcquisitionPlugin');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_crwapiserver;
begin
 InitPliginCounters;
 dlc_PluginWrap; // Register DebugLog channel
end;

procedure Free_crw_crwapiserver;
begin
 ResourceLeakageLog(Format('%-60s = %d',['PluginWrapBalance',LockedGet(PluginWrapBalance)]));
 FreePliginCounters;
end;

initialization

 Init_crw_crwapiserver;

finalization

 Free_crw_crwapiserver;

end.

//////////////
// END OF FILE
//////////////


