////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Read *.dat.                                                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231202 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqreaddat; // Form DAQ Read *.dat

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Spin,
 lcltype, lclintf, CheckLst,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_daqsys,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormDaqReadDat }

  TFormDaqReadDat = class(TMasterForm)
    PanelControls: TPanel;
    PanelParams: TPanel;
    GroupBoxTimeUnits: TGroupBox;
    ComboBoxTimeUnits: TComboBox;
    EditTimeUnits: TEdit;
    LabelTimeUnitsMs: TLabel;
    PanelCurveList: TPanel;
    GroupBoxCurveList: TGroupBox;
    CheckListBoxCurves: TCheckListBox;
    GroupBoxTimeBase: TGroupBox;
    LabelYear: TLabel;
    LabelMonth: TLabel;
    LabelDay: TLabel;
    LabelHour: TLabel;
    LabelMin: TLabel;
    LabelSec: TLabel;
    LabelMSec: TLabel;
    SpinEditYear: TSpinEdit;
    SpinEditMonth: TSpinEdit;
    SpinEditDay: TSpinEdit;
    SpinEditHour: TSpinEdit;
    SpinEditMin: TSpinEdit;
    SpinEditSec: TSpinEdit;
    SpinEditMSec: TSpinEdit;
    ButtonTimeBaseNow: TButton;
    ButtonTimeBaseOrigin: TButton;
    ButtonTimeBaseDaq: TButton;
    ButtonTimeBaseXmas: TButton;
    ButtonTimeBaseFirst: TButton;
    GroupBoxPack: TGroupBox;
    CheckBoxSortX: TCheckBox;
    CheckBoxPackX: TCheckBox;
    CheckBoxPackY: TCheckBox;
    PanelTol: TPanel;
    LabelAbsRelTolX: TLabel;
    EditAbsTolX: TEdit;
    EditRelTolX: TEdit;
    LabelAbsRelTolY: TLabel;
    EditAbsTolY: TEdit;
    EditRelTolY: TEdit;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    BitBtnCheckAll: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure BitBtnCheckAllClick(Sender: TObject);
    procedure ComboBoxTimeUnitsChange(Sender: TObject);
    procedure ButtonTimeBaseNowClick(Sender: TObject);
    procedure ButtonTimeBaseOriginClick(Sender: TObject);
    procedure ButtonTimeBaseDaqClick(Sender: TObject);
    procedure ButtonTimeBaseXmasClick(Sender: TObject);
    procedure ButtonTimeBaseFirstClick(Sender: TObject);
  private
    { Private declarations }
    SaveTimeBase  : Double;
    SaveTimeUnits : Double;
    function  GetTimeBase:Double;
    procedure SetTimeBase(aTimeBase:Double);
    function  GetPackMode:Integer;
    procedure SetPackMode(aPackMode:Integer);
  public
    { Public declarations }
    property TimeBase : Double  read GetTimeBase write SetTimeBase;
    property PackMode : Integer read GetPackMode write SetPackMode;
  end;

function FormDaqReadDatExecute(var TimeUnits,TimeBase:Double;
                               var PackMode:Integer;
                               var AbsTolX,RelTolX,AbsTolY,RelTolY:Double;
                               CurveList:TText; Params:LongString=''):Integer;

implementation

{$R *.lfm}

function FormDaqReadDatExecute(var TimeUnits,TimeBase:Double;
                               var PackMode:Integer;
                               var AbsTolX,RelTolX,AbsTolY,RelTolY:Double;
                               CurveList:TText; Params:LongString=''):Integer;
const
 TheForm : TFormDaqReadDat = nil;
var
 i : Integer; r:Double;
begin
 Result:=mrCancel;
 if CanShowModal(TheForm) then
 try
  if not TheForm.Ok then begin
   Application.CreateForm(TFormDaqReadDat, TheForm);
   TheForm.Master:=@TheForm;
  end;
  if TheForm.Ok then begin
   TheForm.PackMode:=PackMode;
   TheForm.TimeBase:=TimeBase;
   TheForm.SaveTimeBase:=TimeBase;
   TheForm.SaveTimeUnits:=TimeUnits;
   TheForm.EditAbsTolX.Text:=Format('%.7g',[AbsTolX]);
   TheForm.EditRelTolX.Text:=Format('%.7g',[RelTolX]);
   TheForm.EditAbsTolY.Text:=Format('%.7g',[AbsTolY]);
   TheForm.EditRelTolY.Text:=Format('%.7g',[RelTolY]);
   TheForm.CheckListBoxCurves.Items.Text:=CurveList.Text;
   for i:=0 to TheForm.CheckListBoxCurves.Items.Count-1 do
   TheForm.CheckListBoxCurves.Checked[i]:=true;
   TheForm.CheckListBoxCurves.Update;
   TheForm.ComboBoxTimeUnits.ItemIndex:=4;
   TheForm.ComboBoxTimeUnitsChange(nil);
   TheForm.ApplyParams(Params);
   if (mrVoice(TheForm.ShowModal)=mrOk) then begin
    Result:=mrOk;
    CurveList.Count:=0;
    for i:=0 to TheForm.CheckListBoxCurves.Items.Count-1 do
    if TheForm.CheckListBoxCurves.Checked[i]
    then CurveList.AddLn(TheForm.CheckListBoxCurves.Items[i]);
    case TheForm.ComboBoxTimeUnits.ItemIndex of
     0 : TimeUnits:=1;
     1 : TimeUnits:=1000;
     2 : TimeUnits:=1000*60;
     3 : TimeUnits:=1000*60*60;
     4 : TimeUnits:=1000*60*60*24;
     else if not Str2Real(TheForm.EditTimeUnits.Text,TimeUnits) then begin
           Result:=mrCancel;
           Echo(RusEng('Недопустимые единицы времени ','Invalid time units ')+TheForm.EditTimeUnits.Text);
          end;
    end;
    TimeBase:=TheForm.TimeBase;
    PackMode:=TheForm.PackMode;
    if Str2Real(Trim(TheForm.EditAbsTolX.Text),r) then AbsTolX:=Max(0,r);
    if Str2Real(Trim(TheForm.EditRelTolX.Text),r) then RelTolX:=Max(0,r);
    if Str2Real(Trim(TheForm.EditAbsTolY.Text),r) then AbsTolY:=Max(0,r);
    if Str2Real(Trim(TheForm.EditRelTolY.Text),r) then RelTolY:=Max(0,r);
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormDaqReadDatExecute');
 end;
end;

function TFormDaqReadDat.GetTimeBase:Double;
begin
 Result:=_NaN;
 if Ok then
 try
  Result:=DateTimeToMSec(SpinEditYear.Value,
                         SpinEditMonth.Value,
                         SpinEditDay.Value,
                         SpinEditHour.Value,
                         SpinEditMin.Value,
                         SpinEditSec.Value,
                         SpinEditMSec.Value);
 except
  on E:Exception do BugReport(E,Self,'GetTimeBase');
 end;
end;

procedure TFormDaqReadDat.SetTimeBase(aTimeBase:Double);
begin
 if Ok then
 try
  with MSecToNativeTime(aTimeBase) do begin
   SpinEditYear.Value:=Year;
   SpinEditMonth.Value:=Month;
   SpinEditDay.Value:=Day;
   SpinEditHour.Value:=Hour;
   SpinEditMin.Value:=Minute;
   SpinEditSec.Value:=Second;
   SpinEditMSec.Value:=Millisecond;
  end;
 except
  on E:Exception do BugReport(E,Self,'SetTimeBase');
 end;
end;

function TFormDaqReadDat.GetPackMode:Integer;
begin
 Result:=0;
 if Ok then
 try
  if CheckBoxSortX.Checked then Result:=Result or 1;
  if CheckBoxPackX.Checked then Result:=Result or 2;
  if CheckBoxPackY.Checked then Result:=Result or 4;
 except
  on E:Exception do BugReport(E,Self,'GetPackMode');
 end;
end;

procedure TFormDaqReadDat.SetPackMode(aPackMode:Integer);
begin
 if Ok then
 try
  CheckBoxSortX.Checked:=HasFlags(aPackMode,1);
  CheckBoxPackX.Checked:=HasFlags(aPackMode,2);
  CheckBoxPackY.Checked:=HasFlags(aPackMode,4);
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormDaqReadDat.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Единицы времени и список кривых','Time units and curve list');
 BitBtnOk.Caption:=mrCaption(mrOk);
 BitBtnCancel.Caption:=mrCaption(mrCancel);
 GroupBoxTimeUnits.Caption:=RusEng('Eдиницы времени','Time units');
 GroupBoxTimeBase.Caption:=RusEng('Начало отсчета времени','Time zero base');
 GroupBoxCurveList.Caption:=RusEng('Список кривых','Curve list');
 BitBtnCheckAll.Caption:=RusEng('Все','&All');
 BitBtnCheckAll.ModalResult:=mrNone;
 ComboBoxTimeUnits.Items.Text:=RusEng('МиллиСекунды'+EOL+
                                     'Секунды'+EOL+
                                     'Минуты'+EOL+
                                     'Часы'+EOL+
                                     'Сутки'+EOL+
                                     'Другие единицы',
                                     'MilliSeconds'+EOL+
                                     'Seconds'+EOL+
                                     'Minutes'+EOL+
                                     'Hours'+EOL+
                                     'Days'+EOL+
                                     'Custom units');
 LabelYear.Caption:=RusEng('Год','Year');
 LabelMonth.Caption:=RusEng('Месяц','Month');
 LabelDay.Caption:=RusEng('День','Day');
 LabelHour.Caption:=RusEng('Час','Hour');
 LabelMin.Caption:=RusEng('Мин','Min');
 LabelSec.Caption:=RusEng('Сек','Sec');
 LabelMSec.Caption:=RusEng('мСек','mSec');
 ButtonTimeBaseOrigin.Caption:=RusEng('Полночь','Midnight');
 ButtonTimeBaseNow.Caption:=RusEng('Сейчас','Now');
 ButtonTimeBaseDaq.Caption:=RusEng('По часам DAQ','DAQ time');
 ButtonTimeBaseXmas.Caption:=RusEng('Рождество','Xmas');
 ButtonTimeBaseFirst.Caption:=RusEng('Первая точка кривой','First curve point');
 GroupBoxPack.Caption:=RusEng('Сортировка и сжатие','Sorting and packing');
 CheckBoxSortX.Caption:=RusEng('Сортировка','Sorting');
 CheckBoxPackX.Caption:=RusEng('Сжатие по X','X packing');
 CheckBoxPackY.Caption:=RusEng('Сжатие по Y','Y packing');
 LabelAbsRelTolX.Caption:=RusEng('Абс.\Отн. точность по X','Abs.\Rel. tolerance X');
 LabelAbsRelTolY.Caption:=RusEng('Абс.\Отн. точность по Y','Abs.\Rel. tolerance Y');
 LocateFormToCenterOfScreen(Self);
 TimeBase:=0;
end;

procedure TFormDaqReadDat.BitBtnCheckAllClick(Sender: TObject);
var
 i : Integer;
begin
 try
  for i:=0 to CheckListBoxCurves.Items.Count-1 do
  CheckListBoxCurves.Checked[i]:=not CheckListBoxCurves.Checked[i];
 except
  on E:Exception do BugReport(E,Self,'BitBtnCheckAllClick');
 end;
end;

procedure TFormDaqReadDat.ComboBoxTimeUnitsChange(Sender: TObject);
begin
 try
  case ComboBoxTimeUnits.ItemIndex of
   0 : EditTimeUnits.Text:=Format('%d',[1]);
   1 : EditTimeUnits.Text:=Format('%d',[1000]);
   2 : EditTimeUnits.Text:=Format('%d',[1000*60]);
   3 : EditTimeUnits.Text:=Format('%d',[1000*60*60]);
   4 : EditTimeUnits.Text:=Format('%d',[1000*60*60*24]);
   5 : EditTimeUnits.Text:=Format('%g',[SaveTimeUnits]);
  end;
  EditTimeUnits.Enabled:=(ComboBoxTimeUnits.ItemIndex=5);
 except
  on E:Exception do BugReport(E,Self,'ComboBoxTimeUnitsChange');
 end;
end;

procedure TFormDaqReadDat.ButtonTimeBaseNowClick(Sender: TObject);
begin
 if Ok then TimeBase:=msecnow;
end;

procedure TFormDaqReadDat.ButtonTimeBaseOriginClick(Sender: TObject);
begin
 if Ok then TimeBase:=SaveTimeBase;
end;

procedure TFormDaqReadDat.ButtonTimeBaseDaqClick(Sender: TObject);
begin
 if Ok and Daq.Timer.isStart then TimeBase:=Daq.Timer.StartTime;
end;

procedure TFormDaqReadDat.ButtonTimeBaseXmasClick(Sender: TObject);
begin
 if Ok then TimeBase:=0;
end;

procedure TFormDaqReadDat.ButtonTimeBaseFirstClick(Sender: TObject);
begin
 if Ok then
 try
  TimeBase:=0;
  SpinEditYear.Value:=0;
 except
  on E:Exception do BugReport(E,Self,'ButtonTimeBaseFirstClick');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqreaddat;
begin
end;

procedure Free_form_daqreaddat;
begin
end;

initialization

 Init_form_daqreaddat;

finalization

 Free_form_daqreaddat;

end.

//////////////
// END OF FILE
//////////////

