////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Demo shared memory client for CRW-DAQ.                                     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 2023xxxx - Created by A.K.                                                 //
// 20240312 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program demo_shm_client; // Demo shared memory client

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, interfaces, lclintf,
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_base64, _crw_fio, _crw_fifo,
 _crw_rtc, _crw_ascio, _crw_az, _crw_polling, _crw_task, _crw_proc,
 _crw_sharm;


 ////////////////////////////////////////
 // Include SHM constants to access data:
 // Shared Memory Block descriptor - WAVE
 ////////////////////////////////////////
const
 {$I shm_block_wave.inc}

 // Random [-0.5,0.5].
function urand:Double;
begin
 Result:=random-0.5;
end;

 // Count number of running instances.
function CountMyInstances:Integer;
begin
 Result:=WordCount(GetListOfProcesses(0,0,ApplicationName),EolnDelims);
end;

 ////////////////////////////////
 // Main DEMO_SHM client program.
 // Generate a demo SIN/COS wave.
 // IPC uses shared memory block.
 ////////////////////////////////
procedure ShmDemoClient;
var shm:Integer; name:LongString; IoCounter1,IoCounter2:Integer;
var dirty:Boolean; t0,t,wf,wa,wn,ws,wc:Double;
begin
 try
  //
  // Get & check Shared Memory block name from command line arguments.
  //
  name:=Trim(ParamStr(1));
  if (name='') then begin
   writeln('Error: Shared Memory name is not specified.');
   Flush(Output); Sleep(1000); ExitCode:=1;
   Exit;
  end;
  //
  // Initialize Shared Memory block.
  //
  shm:=shm_init(name,sz_WAVE,0);
  if (shm=0) then begin
   writeln('Error: Could not shm_init('+name+').');
   Flush(Output); Sleep(1000); ExitCode:=1;
   Exit;
  end;
  //
  // Main loop: read SIN/COS wave params from Shared Memory.
  // Calculate SIN/COS and write result back to Shared Memory.
  // Use IOCOUNTER atomic operation to synchronize I/O operations.
  //
  try
   if not shm_ref(shm).Ok then begin
    writeln('Error: SharedMemory data failed.');
    Flush(Output); Sleep(1000); ExitCode:=1;
    Exit;
   end;
   writeln('Shared Memory opened: '+name);
   t0:=mSecNow;
   // Set STARTED=-1 to terminate client
   while (shm_iop(shm,of_STARTED,'r',0)>=0) do begin
    if (shm_iop(shm,of_STARTED,'r',0)>0) then begin
     /////////////////////////////////////////////// Read wave parameters
     IoCounter1:=shm_iop(shm,of_IOCOUNTER,'+',0); // Read IO counter BEFORE
     wf:=shm_rop(shm,of_FREQUENCY,'r',0);         // Read Frecuency
     wa:=shm_rop(shm,of_AMPLITUDE,'r',0);         // Read Amplitude
     wn:=shm_rop(shm,of_NOISE,'r',0);             // Read Noise
     IoCounter2:=shm_iop(shm,of_IOCOUNTER,'+',0); // Read IO counter AFTER
     /////////////////////////////////////////////// Check reading is not "dirty"
     dirty:=(IoCounter1<>IoCounter2);             // Counter changed during read?
     dirty:=dirty or Odd(IoCounter1);             // Server write during reading?
     /////////////////////////////////////////////// Dirty reading = IO collision
     if dirty then begin                          // If data read dirty (failed),
      writeln('IO Collision, try again later.');  // Report and try again later
     end else begin
      t:=(mSecNow-t0)*1e-3;                       // Time since start, sec
      ws:=wa*Sin(2*pi*wf*t)+wn*urand;             // Evaluate demo SIN wave
      wc:=wa*Cos(2*pi*wf*t)+wn*urand;             // Evaluate demo COS wave
      ////////////////////////////////////////////// Write to Shared Memory
      shm_iop(shm,of_IOCOUNTER,'+',1);            // BEGIN WRITING SHM DATA
      shm_rop(shm,of_SIN,'w',ws);                 // Write shared data - SIN
      shm_rop(shm,of_COS,'w',wc);                 // Write shared data - COS
      shm_iop(shm,of_IOCOUNTER,'+',1);            // END WRITING SHM DATA
      //////////////////////////////////////////////
      writeln('SIN: ',ws:10:5,'  ',               // Report SIN
              'COS: ',wc:10:5);                   // Report COS
     end;
    end;
    Sleep(10);							       // Sleep to yield CPU time
   end;
  finally                                         // Anyway:
   shm_free(shm);                                 // Don't forget to free SHM
  end;
 except
  on E:Exception do BugReport(E,nil,'ShmDemoClient');
 end;
end;

begin
 if CountMyInstances>1 then begin
  writeln('Program is already running.');
  Sleep(1000);
  Exit;
 end;
 ShmDemoClient;
end.

//////////////
// END OF FILE
//////////////

