////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user program. //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Example of console application to connect to CRW-DAQ pipe.                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241024 - Sample created by A.K.                                          //
////////////////////////////////////////////////////////////////////////////////

program daq_user_task;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics, interfaces, forms,
 _crw_alloc, _crw_cmdargs, _crw_environ, _crw_proc, _crw_rtc, _crw_ef,
 _crw_str, _crw_utf8, _crw_fio, _crw_fifo, _crw_hl, _crw_ee, _crw_colors,
 _crw_base64, _crw_polling, _crw_task, _crw_ascio, _crw_az;

 //
 // General variables and constants
 //
const
 Terminated : Boolean = false; // Program should be terminated
 //
 // Request: @help
 // Reply:   help info
 // Comment: Show help.
 //
 procedure DoHelp(const cmnd,args:LongString);
 begin
  StdOut.Put:=Format('%s',[cmnd]);
  StdOut.Put:='>> Command list:';
  StdOut.Put:='>> @help                 This help.';
  StdOut.Put:='>> @exit=n               Exit program with code n.';
  StdOut.Put:='>> @errors               Return error counter.';
  StdOut.Put:='>> @memory               Return memory status.';
  StdOut.Put:='>> @processpriority=n    Set process priority:Idle/Lower/Normal/High/RealTime.';
  StdOut.Put:='>> @runcount=data        Demo data processing.';
 end;
 //
 // Request: @exit
 // Request: @exit=n
 // Reply:   @exit=n
 // Comment: Terminate program with exit code n.
 //
 procedure DoExit(const cmnd,args:LongString);
 begin
  Terminated:=true;
  System.ExitCode:=StrToIntDef(args,0);
  StdOut.Put:=Format('%s=%d',[cmnd,System.ExitCode]);
 end;
 //
 // Request: @errors
 // Request: @errors=n
 // Reply:   @errors=n
 // Comment: Return value n of error counter.
 //
 procedure DoErrors(const cmnd,args:LongString);
 var
  n : LongInt;
 begin
  if Str2Long(args,n)
  then n:=LockedExchange(StdIoErrorCount,n)
  else n:=LockedGet(StdIoErrorCount);
  StdOut.Put:=Format('%s=%d',[cmnd,n]);
 end;
 //
 // Request: @memory
 // Request: @memory=n
 // Comment: Return AllocMemSize.
 //
 procedure DoMemory(const cmnd,args:LongString);
 begin
  StdOut.Put:=Format('%s=%d',[cmnd,GetAllocMemSize]);
 end;
 //
 // Request: @ProcessPriority
 // Request: @ProcessPriority=n
 // Reply:   @ProcessPriority=n
 // Comment: Set process priority class.
 //
 procedure DoProcessPriority(const cmnd,args:LongString);
 var p:DWORD;
 begin
  if not IsEmptyStr(args) then begin
   p:=GetPriorityClassByName(args);
   if p>0 then SetPriorityClass(GetCurrentProcess,p);
  end;
  StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityName(GetProcessPriority(GetCurrentProcessId))]);
 end;
 //
 // Request: @RunCount=data
 // Reply:   @RunCount=data
 // Comment: Demo data processing.
 //
 procedure DoRunCount(const cmnd,args:LongString);
 const
  RunCount   : Double     = 1;
  ErrorCount : Integer    = 0;
 var
  s   : ShortString;
  r,t : Double;
 begin
  s:=mime_decode(args);
  r:=dump2r(Copy(s,1,8));
  t:=dump2r(Copy(s,9,8));
  if r<>RunCount+1 then Inc(ErrorCount);
  RunCount:=r;
  StdOut.Put:=Format('%s=%g'#9'Time=%11.6f'#9'ErrorCount=%d',[cmnd,RunCount,t,ErrorCount]);
 end;
 //
 // This callback handles unrecognized commands.
 //
 procedure DoSpecificCommands(const args:LongString);
 begin
  if Length(args)>0 then
  StdOut.Put:=Format('Could not recognize "%s"',[args]);
 end;
 //
 // Application specific initialization.
 //
 procedure SpecificInitialization;
  procedure ShowEnv(const aName:LongString);
  begin
   StdOut.Put:=Format('%s=%s',[aName,GetEnv(aName)]);
  end;
 begin
  // 
  // Register user commands coming from StdIn.
  //
  SystemEchoProcedure:=StdOutEcho;
  StdIn.SpecHandler:=DoSpecificCommands;
  StdIn.AddCommand('@Help',            DoHelp);
  StdIn.AddCommand('@Exit',            DoExit);
  StdIn.AddCommand('@Errors',          DoErrors);
  StdIn.AddCommand('@Memory',          DoMemory);
  StdIn.AddCommand('@ProcessPriority', DoProcessPriority);
  StdIn.AddCommand('@RunCount',        DoRunCount);
  //
  // Greetings.
  //
  StdOut.Put:='Welcome to '+SysUtils.ExtractFileName(ProgName)+'.';
  StdOut.Put:='Press @Help to get help.';
  //
  // Show environment variables.
  //
  ShowEnv('CRW_DAQ_SYS_HOME_DIR');      // Show crwdaq home directory
  ShowEnv('CRW_DAQ_SYS_EXE_FILE');      // Show full crwdaq file path
  ShowEnv('CRW_DAQ_SYS_INI_FILE');      // Show full crwdaq.ini file path
  ShowEnv('CRW_DAQ_SYS_EXE_PID');       // Show crwdaq process ID
  ShowEnv('CRW_DAQ_SYS_EXE_TID');       // Show crwdaq main thread ID
  ShowEnv('CRW_DAQ_SYS_MESSAGE');       // Show crwdaq unique application message
  ShowEnv('CRW_DAQ_CONFIG_FILE');       // Show DAQ configuration file
  ShowEnv('CRW_DAQ_CONFIG_DATA_PATH');  // Show DAQ data directory path
  ShowEnv('CRW_DAQ_CONFIG_LOAD_TIME');  // Show DAQ config load time
  ShowEnv('CRW_DAQ_CONFIG_START_TIME'); // Show DAQ config start time
 end;
 //
 // Application specific finalization.
 //
 procedure SpecificFinalization;
 begin
 end;
 //
 // Application specific polling.
 //
 procedure SpecificPolling;
 begin
  if BecameZombie(FILE_TYPE_PIPE,1000) then StdIn.Put:='@Exit';
 end;
 //
 // Main program
 //
begin
 try
  try
   SpecificInitialization;
   while not Terminated do begin
    while StdIn.Count>0 do StdIn.Process(StdIn.Get);
    SpecificPolling;
    Sleep(1);
   end;
  finally
   SpecificFinalization;
  end;
 except
  on E:Exception do StdOut.Put:=E.Message;
 end;
 Sleep(100);
 if BecameZombie(FILE_TYPE_PIPE) then ExitCode:=1;
end.

//////////////
// END OF FILE
//////////////
