////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020-2023 Alexey Kuryakin kouriakine@mail.ru - LGPL license. //
////////////////////////////////////////////////////////////////////////////////

unit dpSystem; // Diesel Pascal system routines.

interface

uses dpCmdArgs;

// Engine, Script and OS identification.
////////////////////////////////////////////////
function IsUnix:Boolean;                      // Is Unix    family OS?
function IsWindows:Boolean;                   // Is Windows family OS?
function GetOSName:String;                    // OS: Windows/Linux/MacOS/Unknown
function GetOSFamily:String;                  // OS Family: Windows/Unix/Unknown
function GetEngineName:String;                // Engine short filename with ext
function GetScriptName:String;                // Script short filename with ext
function GetEngineFullName:String;            // Engine full  filename with path,ext
function GetScriptFullName:String;            // Script full  filename with path,ext
function GetEngineBaseName:String;            // Engine base  filename without ext
function GetScriptBaseName:String;            // Script base  filename without ext
function GetEnvineVersionStr:String;          // Engine version like 1.14.32
function GetBriefEngineScriptInfo:String;     // Short Engine and Script info
function GetScriptParamCount:Integer;         // Get Script parameters Count
function GetScriptParamStr(n:Integer):String; // Get n-th parameter of Script

// Get parent process PID.
function GetParentProcessId(aPid:Cardinal=0):Cardinal;

// Environment variables.
////////////////////////////////////////////////
function GetEnv(Name:String):String;          // Get environment variable
function SetEnv(Name,Value:String):Boolean;   // Set environment variable
function ExpEnv(Name:String):String;          // Expand as %PATH% or $PATH
function EnvironmentVariableList:TStringList; // List all variables (NAME=VALUE)

const // List of DIESEL environment variables.
 DIESEL_ENV_VAR_NAMES = 'DIESEL_EXE_DIR,DIESEL_EXE_PID,DIESEL_LOCALSET_DIR,'
 +'DIESEL_SCRIPT_DIR,DIESEL_SCRIPT_NAME';

// Memory routines.
////////////////////////////////////////////////
function GetHeapStatusBrief:String;           // Brief HEAP memory status
function GetHeapStatusTotalAllocated:QWord;   // Total HEAP allocated bytes
function GetHeapStatusTotalFree:QWord;        // Total HEAP free bytes

// Log Events routines.
////////////////////////////////////////////////
var DateTimePromptUsesFiller:Char='-';        // Prompt char for AddTextAsEventLog
var DateTimePromptUsesMsec:Boolean=true;      // Prompt uses millisecond precision
var DateTimePromptUsesSuffix:String=' => ';   // Prompt uses trailing suffix string
function DateTimePrompt(When:TDateTime):String;   // "2020.07.24-10:15:20.321 => "
procedure MoveListToLogger(List,Logger:TStrings); // Move text from List to Logger
procedure MoveListToLoggerMemo(List:TStrings; Memo:TMemo; SelStart:Integer=MaxInt);
function AddTextAsEventLog(Logger:TStrings; When:TDateTime; TextLines:String):Integer;
function AddTextToLogger(Logger:TStrings; TextLines:String):Integer;

// Log Events list.
////////////////////////////////////////////////
var LogCountTotal:Int64=0;                    // Total count of log events
var UsesLogEvents:Boolean=true;               // Flag of log events usage
var LogEventsList:TStringList=nil;            // List of logs with timestamp
var LogEventsHistoryLimit:Integer=64;         // History limit for LogEventsList
function HasLogEvents:Boolean;
function GetLogEventsListCount:Integer;
procedure LogEvents(When:TDateTime; Message:String);
procedure MoveLogEventsListToLogger(Logger:TStrings);
procedure MoveLogEventsListToLoggerMemo(Memo:TMemo; SelStart:Integer=MaxInt);
function GetLogEventsListAsText(ClearMode:Integer):String;

// Exception handling.
// try Something except on E:Exception do BugReport(E,Sender,'Comment'); end;
////////////////////////////////////////////////
var BugCountTotal:Int64=0;                    // Total count of bugs
var UsesBugReport:Boolean=true;               // Flag of bug report usage
var ShowBugReport:Boolean=false;              // Flag of bug MessageBox show
var BugReportList:TStringList=nil;            // List of bugs with timestamp
var BugReportHistoryLimit:Integer=64;         // History limit for BugReportList
procedure SystemBugReport(E:Exception; Sender:TObject; Where:String);
procedure BugReport(E:Exception; Sender:TObject; Where:String);
function GetBugReportListAsText(ClearMode:Integer):String;
procedure MoveBugReportListToLogger(Logger:TStrings);
procedure MoveBugReportListToLoggerMemo(Memo:TMemo; SelStart:Integer=MaxInt);
function GetBugReportListCount:Integer;

implementation

var TheOSName     : String = '';
var TheOSFamily   : String = '';
var TheEnvineVers : String = '';

procedure InitEngineAndScriptConstants;
var Major,Minor,Built:Integer;
begin
 try
  DecimalSeparator:='.';
  if Linux   then TheOSName:='Linux'   else
  if MacOS   then TheOSName:='MacOS'   else
  if Windows then TheOSName:='Windows' else TheOSName:='Unknown';
  TheOSFamily:='Unknown';
  if IsUnix then TheOSFamily:='Unix';
  if IsWindows then TheOSFamily:='Windows';
  GetEngineVersion(Major,Minor,Built);
  TheEnvineVers:=IntToStr(Major)+'.'+IntToStr(Minor)+'.'+IntToStr(Built);
  SetEnv('DIESEL_ENGINE',GetEngineBaseName+' '+TheEnvineVers+' on '+TheOSName);
 except
  on E:Exception do BugReport(E,Application,'InitEngineAndScriptConstants');
 end;
end;

function IsUnix:Boolean;
begin
 Result:=Linux or MacOS;
end;

function IsWindows:Boolean;
begin
 Result:=Windows;
end;

function GetOSName:String;
begin
 Result:=TheOSName;
end;

function GetOSFamily:String;
begin
 Result:=TheOSFamily;
end;

function GetEngineFullName:String;
begin
 Result:=CmdArgs.EngineFile;
end;

function GetScriptFullName:String;
begin
 Result:=CmdArgs.ScriptFile;
end;

function GetEngineName:String;
begin
 Result:=CmdArgs.EngineName;
end;

function GetScriptName:String;
begin
 Result:=CmdArgs.ScriptName;
end;

function GetEngineBaseName:String;
begin
 Result:=CmdArgs.EngineBase;
end;

function GetScriptBaseName:String;
begin
 Result:=CmdArgs.ScriptBase;
end;

function GetEnvineVersionStr:String;
begin
 Result:=TheEnvineVers;
end;

function GetBriefEngineScriptInfo:String;
begin
 Result:='VisualTech DieselPascal '+GetEnvineVersionStr+LineEnding
        +'Engine: name '+GetEngineBaseName+' in file '+GetEngineName+' with path '+GetEngineFullName+LineEnding
        +'Script: name '+GetScriptBaseName+' in file '+GetScriptName+' with path '+GetScriptFullName+LineEnding
        +'Hosted: on '+GetOSFamily+' family OS';
end;

function GetScriptParamCount:Integer;
begin
 Result:=ParamCount-1;
end;

function GetScriptParamStr(n:Integer):String;
begin
 Result:=ParamStr(n+1);
end;

function GetParentProcessId(aPid:Cardinal=0):Cardinal;
var List:TStringList; i:Integer; s,Line,Delims:String;
begin
 Result:=0;
 try
  if (aPid=0) then aPid:=GetProcessId;
  if (aPid=0) then Exit;
  Delims:=',; '+CHR(9)+CHR(10)+CHR(13);
  if IsWindows {or IsUnix} then begin
   s:=GetListOfProcesses(aPid,0,'');
   Result:=StrToIntDef(ExtractWord(2,s,Delims),0);
   Exit;
  end;
  if IsUnix then begin
   List:=TStringList.Create;
   try
    List.LoadFromFile('/proc/'+IntToStr(aPid)+'/status');
    for i:=0 to List.Count-1 do begin
     Line:=List.Strings[i];
     s:=ExtractWord(1,Line,Delims);
     if SameText(s,'PPid:') then begin
      Result:=StrToIntDef(ExtractWord(2,Line,Delims),0);
      break;
     end;
    end;
   finally
    List.Free;
   end;
   Exit;
  end;
 except
  on E:Exception do BugReport(E,nil,'GetParentProcessId');
 end;
end;

var TheEnvList:TStringList=nil;

procedure TheEnvListInit;
var i:Integer; sn,sv,delims:String;
begin
 delims:=LineEnding+', ;'+Chr(9);
 Application.GetEnvironmentList(TheEnvList);
 for i:=1 to WordCount(DIESEL_ENV_VAR_NAMES,delims) do begin
  sn:=ExtractWord(i,DIESEL_ENV_VAR_NAMES,delims);
  sv:=GetEnv(sn); TheEnvList.Values[sn]:=sv;
 end;
 for i:=TheEnvList.Count-1 downto 0 do begin
  if (Pos('=',TheEnvList.Strings[i])<=1)
  then TheEnvList.Delete(i);
 end;
end;

function EnvironmentVariableList:TStringList;
begin
 if (TheEnvList=nil) then begin
  TheEnvList:=TStringList.Create;
  TheEnvListInit;
 end;
 Result:=TheEnvList;
end;

function GetEnv(Name:String):String;
begin
 Result:=GetEnvironmentVariable(Name);
end;

function SetEnv(Name,Value:String):Boolean;
begin
 if (Name='') then Exit;
 Result:=SetEnvironmentVariable(Name,Value);
 if Result then EnvironmentVariableList.Values[Name]:=Value;
end;

function ExpEnv(Name:String):String;
begin
 Result:=ExpandEnvironmentVariables(Name,emExpandDefaults);
end;

function GetHeapStatusBrief:String;
var S:String;
begin
 S:=GetHeapStatusAsText;
 Result:=FormatVar('Memory Space: %s',CookieScan(S,'TotalAddrSpace',0))
           +', '+FormatVar('Uses: %s',CookieScan(S,'TotalAllocated',0))
           +', '+FormatVar('Free: %s',CookieScan(S,'TotalFree',0));
end;

function GetHeapStatusTotalAllocated:QWord;
begin
 Result:=StrToQWordDef(CookieScan(GetHeapStatusAsText,'TotalAllocated',0),0);
end;

function GetHeapStatusTotalFree:QWord;
begin
 Result:=StrToQWordDef(CookieScan(GetHeapStatusAsText,'TotalFree',0),0);
end;

function DateTimePrompt(When:TDateTime):String;
begin
 if DateTimePromptUsesMsec
 then Result:=FormatDateTime('yyyy.mm.dd-hh:nn:ss.zzz',When)+DateTimePromptUsesSuffix
 else Result:=FormatDateTime('yyyy.mm.dd-hh:nn:ss',When)+DateTimePromptUsesSuffix;
end;

procedure MoveListToLogger(List,Logger:TStrings);
var i:Integer;
begin
 if (List=nil) then Exit;
 if (Logger<>nil) then
 for i:=0 to List.Count-1 do
 Logger.Add(List.Strings[i]);
 List.Clear;
end;

procedure MoveListToLoggerMemo(List:TStrings; Memo:TMemo; SelStart:Integer=MaxInt);
begin
 if (List<>nil) then
 if (List.Count>0) then begin
  if  (Memo<>nil) then begin
   try
    Memo.Lines.BeginUpdate;
    MoveListToLogger(List,Memo.Lines);
   finally
    Memo.Lines.EndUpdate;
   end;
   if (SelStart>=0) then Memo.SelStart:=SelStart;
  end else MoveListToLogger(List,nil);
 end;
end;

function AddTextAsEventLog(Logger:TStrings; When:TDateTime; TextLines:String):Integer;
var Stamp:String; List:TStringList; i:Integer;
begin
 Result:=0;
 if (Logger=nil) then Exit;
 if (TextLines='') then Exit;
 Stamp:=DateTimePrompt(When);
 if (Pos(LineEnding,TextLines)=0) then begin
  Logger.Add(Stamp+TextLines); Inc(Result);
 end else begin
  List:=TStringList.Create;
  try
   List.Text:=TextLines;
   for i:=0 to List.Count-1 do begin
    Logger.Add(Stamp+List.Strings[i]); Inc(Result);
    if (i=0) then Stamp:=StringOfChar(DateTimePromptUsesFiller,Length(Stamp)-Length(DateTimePromptUsesSuffix))+DateTimePromptUsesSuffix;
   end;
  finally
   List.Free;
  end;
 end;
end;

function AddTextToLogger(Logger:TStrings; TextLines:String):Integer;
var List:TStringList; i:Integer;
begin
 Result:=0;
 if (Logger=nil) then Exit;
 if (TextLines='') then Exit;
 if (Pos(LineEnding,TextLines)=0) then begin
  Logger.Add(TextLines); Inc(Result);
 end else begin
  List:=TStringList.Create;
  try
   List.Text:=TextLines;
   for i:=0 to List.Count-1 do begin
    Logger.Add(List.Strings[i]); Inc(Result);
   end;
  finally
   List.Free;
  end;
 end;
end;

function HasLogEvents:Boolean;
begin
 if (LogEventsList<>nil) then Result:=(LogEventsList.Count>0) else Result:=false;
end;

function GetLogEventsListCount:Integer;
begin
 if (LogEventsList<>nil) then Result:=LogEventsList.Count else Result:=0;
end;

procedure LogEvents(When:TDateTime; Message:String);
begin
 if (Message='') then Exit;
 LogCountTotal:=LogCountTotal+1;
 if not UsesLogEvents then Exit;
 if (LogEventsList=nil) then Exit;
 if (LogEventsHistoryLimit<0) then LogEventsHistoryLimit:=0;
 while (LogEventsList.Count>LogEventsHistoryLimit) do LogEventsList.Delete(0);
 if (LogEventsHistoryLimit>0) then begin
  if (When<=0)
  then AddTextToLogger(LogEventsList,Message)
  else AddTextAsEventLog(LogEventsList,When,Message);
 end;
end;

procedure MoveLogEventsListToLogger(Logger:TStrings);
begin
 MoveListToLogger(LogEventsList,Logger);
end;

procedure MoveLogEventsListToLoggerMemo(Memo:TMemo; SelStart:Integer=MaxInt);
begin
 MoveListToLoggerMemo(LogEventsList,Memo,SelStart);
end;

function GetLogEventsListAsText(ClearMode:Integer):String;
begin
 if (LogEventsList<>nil) then Result:=LogEventsList.Text else Result:='';
 if (LogEventsList<>nil) then if (ClearMode>0) then LogEventsList.Clear;
end;

function FormatBugMessage(E:Exception; Sender:TObject; Where,Delim:String):String;
begin
 Result:='Error: '+E.ClassName+Delim+'Message: '+E.Message+Delim;
 if (Where<>'') then Result:=Result+'Where: '+Where+Delim;
 if (Sender<>nil) then begin
  if (Sender is TControl)
  then Result:=Result+'Sender: '+TControl(Sender).Name
  else Result:=Result+'Sender: Unknown';
  Result:=Result+' as '+Sender.ClassName;
 end;
end;

procedure SystemBugReport(E:Exception; Sender:TObject; Where:String);
begin
 BugReport(E,Sender,Where);
end;

procedure BugReport(E:Exception; Sender:TObject; Where:String);
var When:TDateTime; Message:String;
begin
 When:=Now;
 if (E=nil) then Exit;
 BugCountTotal:=BugCountTotal+1;
 if not UsesBugReport then Exit;
 Message:=FormatBugMessage(E,Sender,Where,'; ')+'.';
 if UsesLogEvents and (LogEventsList<>nil) then LogEvents(When,Message);
 if (BugReportList<>nil) then begin
  if (BugReportHistoryLimit<0) then BugReportHistoryLimit:=0;
  while (BugReportList.Count>BugReportHistoryLimit) do BugReportList.Delete(0);
  if (BugReportHistoryLimit>0) then AddTextAsEventLog(BugReportList,When,Message);
 end;
 if ShowBugReport then begin
  Message:=Trim(FormatBugMessage(E,Sender,Where,LineEnding));
  MessageBox(DateTimePrompt(When)+Message+LineEnding,'Error: '+E.ClassName,0);
 end;
end;

function GetBugReportListAsText(ClearMode:Integer):String;
begin
 if (BugReportList<>nil) then Result:=BugReportList.Text else Result:='';
 if (BugReportList<>nil) then if (ClearMode>0) then BugReportList.Clear;
end;

procedure MoveBugReportListToLogger(Logger:TStrings);
begin
 MoveListToLogger(BugReportList,Logger);
end;

procedure MoveBugReportListToLoggerMemo(Memo:TMemo; SelStart:Integer=MaxInt);
begin
 MoveListToLoggerMemo(BugReportList,Memo,SelStart);
end;

function GetBugReportListCount:Integer;
begin
 if (BugReportList<>nil) then Result:=BugReportList.Count else Result:=0;
end;

initialization

 LogEventsList:=TStringList.Create;
 BugReportList:=TStringList.Create;
 InitEngineAndScriptConstants;

end.
