@echo off
SetLocal EnableExtensions EnableDelayedExpansion

:SetCodePageUtf8
chcp 65001 >nul

:Comment
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Copyright (c) 2025 Alexey Kuryakin daqgroup@mail.ru
:: Watcher for crwdaq: report result after crwdaq exit,
:: make analysis of leakage.log file, rotate logs, etc.
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

:Identification
set "scriptfile=%~f0"
set "scriptbase=%~n0"
set "scriptname=%~nx0"
for %%i in ( "%~dp0\." ) do set "scripthome=%%~fi"
call :save_arguments %*

:CrwDaqEnvironment
set "crwpid=%CRW_DAQ_SYS_EXE_PID%"
set "logdir=%CRW_DAQ_VAR_TMP_DIR%"
set "leaklog=%logdir%\leakage.log"
set "log=%logdir%\%scriptbase%.log"
set /a maxline=100

:InitCounters
set /a numleaks=0
set /a numsuccs=0
set /a numlines=0
set /a numleave=0
set /a numenter=0
set /a logsec=0

:CheckEnvironment
unix.exe >nul || ( call :fatal 1 "Error: unix.exe not found." & exit /b 1 )
if not defined crwpid ( call :fatal 1 "Error: crwdaq environ (PID)." & exit /b %ErrorLevel% )
if not defined logdir ( call :fatal 1 "Error: crwdaq environ (dir)." & exit /b %ErrorLevel% )
if not exist "%leaklog%" ( call :fatal 1 "Error: not found leakage.log." & exit /b %ErrorLevel% )
call :check_log
call :cmd_pid_running crwdaq.exe %crwpid% || ( call fatal 1 "Error: crwdaq PID %crwpid% is not running." & exit /b %ErrorLevel% )
set /a elapsed=0
:wait_process_termination_loop
call :cmd_pid_running crwdaq.exe %crwpid% || goto :crwdaq_stopped
call :println "crwdaq running %elapsed% second(s)."
unix sleep 5 || ping -n 6 localhost 2>&1 >nul
set /a elapsed+=5 & call :run_crwdaqlogroll !elapsed!
goto :wait_process_termination_loop
:crwdaq_stopped
call :println "crwdaq stopped."
call :report_result "%leaklog%"
:Done
echo Done %scriptname%.
exit /b %ErrorLevel%
goto :EOF

:run_crwdaqlogroll
 set /a "sec=%~1-%logsec%"
 if %sec% LSS 60 goto :EOF
 unix %scripthome%\crwdaqlogroll.cmd
 set /a "logsec=%~1"
goto :EOF

::::::::::::::::::::::::::::::::::::::::::
:: check command $1 with PID $2 is running
::::::::::::::::::::::::::::::::::::::::::
:cmd_pid_running
if "%~1" == "" exit /b 1
if "%~2" == "" exit /b 1
for /f "tokens=1,2,* delims= " %%i in ( 'tasklist /nh /fi "imagename eq %~1" 2^>nul ^| findstr /c:" %~2 "' ) do exit /b 0
exit /b 1
goto :EOF

:add_log
call :getdatetime
set "prog=%CRW_DAQ_SYS_TITLE%"
<nul set /p "_=%datetime% %prog%: %~1 %~2 %~3 %~4 %~5 %~6 %~7 %~8 %~9" | unix xargs | unix tee -a "%log%"
goto :EOF

:check_log
unix touch "%log%"
call :add_log "%scriptname% %arguments%"
goto :EOF

:leakage_analysis
set "line=%~1"
if "%line%" == "" goto :EOF
if %numenter% GTR 0 goto :EOF
if %numleave% GTR 0 goto :EOF
call :println_stderr "%line%"
set /a "numlines+=1" & set "value=" & set "hasvalue=0"
if /i "%line:~0,6%" == "Enter " ( set /a "numenter+=1" & goto :EOF )
if /i "%line:~0,6%" == "Leave " ( set /a "numleave+=1" & goto :EOF )
if /i "%line:~0,6%" == "value=" ( set /a "hasvalue+=1" & set /a "%line%" )
if /i "%hasvalue%" == "0" goto :EOF & if /i "%value%" == "" goto :EOF
if /i "%value%" == "0" ( set /a "numsuccs+=1" & goto :EOF )
set /a "numleaks+=1"
goto :EOF

:report_result
echo Report result:
set filter=findstr
set filter=%filter% /i /v /c:"ObjectRegistry.Space"
set filter=%filter% /i /v /c:"ObjectRegistry.Capacity"
set filter=%filter% /i /v /c:"Balance of GetMem/FreeMem"
set filter=%filter% /i /v /c:"TTask.DetachedPidList.Count"
set "hostname=%computername%" & call :LoCase hostname
set "sndr=%hostname%/%CRW_DAQ_SYS_SESSION_HEAD%/System"
set runsed=unix sed -re "s/\s*(\w[^=]*\w)\s*=\s*(\S*)\s*/value=\2/"
for /f "tokens=* delims=" %%i in ('unix tail -n %maxline% "%~1" ^| unix tac ^| %filter% ^| %runsed%') do call :leakage_analysis "%%~i"
call :println "numlines = %numlines%"
call :println "numenter = %numenter%"
call :println "numleave = %numleave%"
call :println "numleaks = %numleaks%"
call :println "numsuccs = %numsuccs%"
if %numleave% GTR 0 goto :found_leave
goto :not_found_leave
:found_leave
if %numleaks% GTR 0 goto :found_leaks
goto :not_found_leaks
:found_leaks
call :langstr ru "Процесс %crwpid% завершен успешно. Однако найдены утечки памяти" en "Process %crwpid% stop succeed. But memory leaks found"
call :warning 1 "%langstr% (%numleaks%)."
unix printc "%datetime% %prog%: PROBLEM - %langstr% (%numleaks%)." -n | unix tee -a "%log%"
call :syslog_report "WARN/PROBLEM: %sndr% - %langstr% (%numleaks%)."
exit /b 1
:not_found_leaks
if %numsuccs% GTR 0 goto :found_succs
goto :not_found_succs
:found_succs
call :langstr ru "Процесс %crwpid% завершен успешно. Проблем не обнаружено" en "Process %crwpid% stop succeed. No problems found"
call :succeed 0 "%langstr%."
unix printc "%datetime% %prog%: SUCCESS - %langstr%." -n | unix tee -a "%log%"
call :syslog_report "INFO/SUCCESS: %sndr% - %langstr%."
exit /b 0
:not_found_succs
:not_found_leave
if %numenter% gtr 0 goto :found_enter
goto :not_found_enter
:found_enter
call :langstr ru "Процесс %crwpid% завершен аварийно" en "Process %crwpid% abnormal termination"
call :syslog_report "FATAL/FAILURE: %sndr% - %langstr%."
call :fatal 1 "%langstr%."
unix printc "%datetime% %prog%: FAILURE - %langstr%." | unix tee -a "%log%"
exit /b 1
:not_found_enter
goto :EOF

:syslog_note
unix -c call "%scripthome%\crwdaq.syslog.cmd" %*
goto :EOF

:syslog_report
set "arg=%~1" & set "severity=" & set "sender=" & set "msg="
for /f "tokens=* delims=" %%i in ('unix printf "%%s" "%arg%" ^| unix cut -d: -f1 ^| unix xargs') do set "severity=%%~i"
for /f "tokens=* delims=" %%i in ('unix printf "%%s" "%arg%" ^| unix cut -d" " -f2 ^| unix xargs') do set "sender=%%~i"
for /f "tokens=* delims=" %%i in ('unix printf "%%s" "%arg%" ^| unix sed "s/.* - //" ^| unix xargs') do set "msg=%%~i"
if not defined severity goto :EOF & if not defined sender goto :EOF & if not defined msg goto :EOF
call :syslog_note                          ^
 "%severity%: %sender% - ________________" ^
 "%severity%: %sender% - Session Outcome." ^
 "%severity%: %sender% - ^^^^" ^
 "%severity%: %sender% - %msg%."
goto :EOF

:LoCase
:: Subroutine to convert a variable VALUE to all lower case.
:: The argument for this subroutine is the variable NAME.
set "%~1=!%~1:A=a!"
set "%~1=!%~1:B=b!"
set "%~1=!%~1:C=c!"
set "%~1=!%~1:D=d!"
set "%~1=!%~1:E=e!"
set "%~1=!%~1:F=f!"
set "%~1=!%~1:G=g!"
set "%~1=!%~1:H=h!"
set "%~1=!%~1:I=i!"
set "%~1=!%~1:J=j!"
set "%~1=!%~1:K=k!"
set "%~1=!%~1:L=l!"
set "%~1=!%~1:M=m!"
set "%~1=!%~1:N=n!"
set "%~1=!%~1:O=o!"
set "%~1=!%~1:P=p!"
set "%~1=!%~1:Q=q!"
set "%~1=!%~1:R=r!"
set "%~1=!%~1:S=s!"
set "%~1=!%~1:T=t!"
set "%~1=!%~1:U=u!"
set "%~1=!%~1:V=v!"
set "%~1=!%~1:W=w!"
set "%~1=!%~1:X=x!"
set "%~1=!%~1:Y=y!"
set "%~1=!%~1:Z=z!"
goto :EOF

:langstr
call :langstr_loop %*
set "langstr=%langstr_en%"
set "lng=%CRW_DAQ_SYS_LANG:~0,2%"
if /i "%lng%" == "ru" if defined langstr_ru set "langstr=%langstr_ru%"
if /i "%lng%" == "en" if defined langstr_en set "langstr=%langstr_en%"
goto :EOF
:langstr_loop
if /i "%~1" == "" goto :EOF
if /i "%~1" == "ru" set "langstr_ru=%~2"
if /i "%~1" == "en" set "langstr_en=%~2"
shift /1 & shift /1 & goto :langstr_loop
goto :EOF

:save_arguments
shift & set arguments=%*
goto :EOF

:print
<nul set /p "_=%~1"
goto :EOF

:println
<nul set /p "_=%~1"
echo.
goto :EOF

:print_stderr
<nul 1>&2 set /p "_=%~1"
goto :EOF

:println_stderr
1>&2 <nul set /p "_=%~1"
1>&2 echo.
goto :EOF

:getdatetime
set datetime=%date%-%time%
set datetime=%datetime: =0%
set datetime=%datetime:~6,4%.%datetime:~3,2%.%datetime:~0,2%-%datetime:~11,8%
goto :EOF

:succeed
call :getdatetime
set /a delay=1000*60
call :println_stderr "%~2"
set "prog=%CRW_DAQ_SYS_TITLE%"
unix tooltip-notifier text "%datetime% %prog%: %~2" preset stdSuccess delay %delay%
exit /b %1
goto :EOF

:warning
call :getdatetime
set /a delay=1000*60*60
call :println_stderr "%~2"
set "prog=%CRW_DAQ_SYS_TITLE%"
unix tooltip-notifier text "%datetime% %prog%: %~2" preset stdWarning delay %delay%
exit /b %1
goto :EOF

:fatal
call :getdatetime
set /a delay=1000*60*60*24
call :println_stderr "%~2"
set "prog=%CRW_DAQ_SYS_TITLE%"
unix tooltip-notifier text "%datetime% %prog%: %~2" preset stdError delay %delay%
exit /b %1
goto :EOF

::::::::::::::
:: END OF FILE
::::::::::::::
