/////////////////////////////////////////////////////////////////////
// SafeRun Copyright(c) 2016 Alexey Kuryakin kouriakine@mail.ru
//
// Purposes:
// SafeRun - free Win32 utility to run program in Safer mode.
// Safer mode means restricted process privileges for safety.
// Can be used for safe Web surfing or high security systems.
// 
// Requirements:
// Required client Windows XP or Server 2003 at least to run.
// Uses:  kernel32.dll, user32.dll, advapi32.dll, msvcrt.dll.
//
// Compilation:
// Compiled with Tiny C Compiler, it hyper fast & small code.
// Tiny C is free, open source, look http://bellard.org/tcc/.
// Example of how to compile saferun.c under Tiny C Compiler:
//
//  path c:\tcc;%path%                      path to find tcc
//  tiny_impdef advapi32.dll                make advapi32.def
//  tcc saferun.c -luser32 -ladvapi32       make saferun.exe
//
// License for use and distribution: GNU LGPL (see below).
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330,
// Boston, MA  02111-1307  USA
// 
// Credits & thanks & respect:
// SafeRun uses good ideas, tools, code fragments from other open source projects:
// Fabrice Bellard - TinyC, http://bellard.org/tcc/ (to compile hyper small code).
// Vadim Sterkin   - Hidec, http://forum.oszone.net/showthread.php?t=29630 (idea).
// Michael Howard  - DropMyRights, mikehow@microsoft.com (example of Safer usage).
//                   http://msdn.microsoft.com/en-us/library/ms972827.aspx.
// See also FYI:
//  http://cybercoyote.org/security/drop.shtml
//  http://blogs.technet.com/b/markrussinovich/archive/2006/03/02/
//         running-as-limited-user-the-easy-way.aspx
/////////////////////////////////////////////////////////////////////

char* About =
    "Help on SafeRun.exe:\n"
    "\n"
    "Copyright(c), 2016, Alexey Kuryakin, <kouriakine@mail.ru>.\n"
    "SafeRun - Win32 utility to run program in Safer mode,v1.3.\n"
    "Safer mode means restricted process privileges for safety.\n"
    "Required client Windows XP or Server 2003 at least to run.\n"
    "Use this program free under the terms of LGPL license.\n"
    "\n"
    "Usage: saferun [-opt] cmdline\n"
    "\n"
    "Options (started from - char):\n"
    " -v  - verbose mode: MessageBox on errors (default is silent)\n"
    "       it's better to place this option as 1st option in list\n"
    " -w  - wait while cmdline running, default is no waits\n"
    "       in wait mode SafeRun return cmdline's exit code\n"
    " -h  - run cmdline in hidden window (SW_HIDE), default is visible (SW_SHOW)\n"
    " -0..9 run with display mode assigned to 0=SW_HIDE(like -h), 1=SW_SHOWNORMAL,\n"
    "       2=SW_SHOWMINIMIZED, 3=SW_SHOWMAXIMIZED, 4=SW_SHOWNOACTIVATE,\n"
    "       5=SW_SHOW(default), 6=SW_MINIMIZE, 7=SW_SHOWMINNOACTIVE,\n"
    "       8=SW_SHOWNA, 9=SW_RESTORE\n"
    " -i  - run with IDLE_PRIORITY_CLASS         ( 4  lowest priority)\n"
    " -b  - run with BELOW_NORMAL_PRIORITY_CLASS ( 6  lower normal)\n"
    " -n  - run with NORMAL_PRIORITY_CLASS       ( 8  default)\n"
    " -a  - run with ABOVE_NORMAL_PRIORITY_CLASS ( 10 higher normal)\n"
    " -g  - run with HIGH_PRIORITY_CLASS         ( 13 greater normal)\n"
    " -r  - run with REALTIME_PRIORITY_CLASS     ( 24 highest)\n"
    " -t  - run cmdline in trusted mode, i.e. don't use Safer\n"
    " -o  - run cmdline in Safer mode as normal user, by default\n"
    " -c  - run cmdline in Safer mode as constrained user\n"
    " -u  - run cmdline in Safer mode as untrusted user\n"
    " -f  - run cmdline in Safer mode as fully trusted user\n"
    " -?  - run this help screen\n"
    "\n"
    "Exit code returned by SafeRun:\n"
    " 0   - OK, cmdline was successfully started\n"
    " 1   - invalid options specified (unsupported opt)\n"
    " 2   - no arguments specified (empty command line)\n"
    " 3   - could not create process (invalid cmdline or access)\n"
    " 4   - could not create Safer security level\n"
    " 5   - could not create Safer security token\n"
    " n   - in wait mode (-w) SafeRun return cmdline's exit code\n"
    "\n"
    "Examples:\n"
    " saferun firefox.exe - run Web browser in Safer mode with normal user level\n"
    " saferun -c firefox  - run Web browser in Safer mode with constrained user level\n"
    " saferun -u firefox  - run Web browser in Safer mode with untrusted user level\n"
    " saferun -f firefox  - run Web browser in Safer mode  with fully trusted user level\n"
    " saferun -t firefox  - run Web browser in normal mode, i.e. don't use Safer at all\n"
    " saferun -wv notepad - run notepad in Safer mode with normal user level, wait, verbose\n"
    " saferun -htw defrag - run defrag in fully trusted (i.e. normal) mode, hidden, wait result\n"
    " saferun -t7 defrag  - run deftag in trusted (i.e. normal) mode, minimized noactive, no wait\n"
    ;

#include <process.h>
#include <windows.h>
#include <winnt.h>
#include <stdio.h>

// Header derived from WinSafer.h to access advapi32.dll functions.
// Minimum supported client Windows XP, server Windows Server 2003.
///////////////////////////////////////////////////////////////////
DECLARE_HANDLE(SAFER_LEVEL_HANDLE);
#define SAFER_SCOPEID_MACHINE           1
#define SAFER_SCOPEID_USER              2
#define SAFER_LEVELID_FULLYTRUSTED      0x40000
#define SAFER_LEVELID_NORMALUSER        0x20000
#define SAFER_LEVELID_CONSTRAINED       0x10000
#define SAFER_LEVELID_UNTRUSTED         0x01000
#define SAFER_LEVELID_DISALLOWED        0x00000
#define SAFER_LEVEL_OPEN                1
WINBOOL WINAPI SaferCloseLevel(SAFER_LEVEL_HANDLE hLevelHandle);
WINBOOL WINAPI SaferCreateLevel(DWORD dwScopeId, DWORD dwLevelId, DWORD OpenFlags, SAFER_LEVEL_HANDLE *pLevelHandle, LPVOID lpReserved);
WINBOOL WINAPI SaferComputeTokenFromLevel(SAFER_LEVEL_HANDLE LevelHandle,HANDLE InAccessToken,PHANDLE OutAccessToken,DWORD dwFlags,LPVOID lpReserved);

// Special chars:
/////////////////
#define SpaceChar                       ' '
#define TabChar                         '\t'
#define DoubleQuota                     '\"'

// Utility functions for CmdLine parsing:
/////////////////////////////////////////
BOOL IsSameChar(char a, char b) { return (a == b) ? TRUE : FALSE; }
BOOL IsSameCharOrNul(char a, char b) { return IsSameChar(a,b) || IsSameChar(a,0); }
char TabToSpace(char c) { return (c == TabChar) ? SpaceChar : c; }
BOOL IsSpaceChar(char c) { return IsSameChar(c,SpaceChar); }
BOOL IsSpaceCharOrTab(char c) { return IsSpaceChar(TabToSpace(c)); }
BOOL IsSpaceCharOrNul(char c) { return IsSameCharOrNul(c,SpaceChar); }
BOOL IsSpaceCharOrTabOrNul(char c) { return IsSameCharOrNul(TabToSpace(c),SpaceChar); }
BOOL IsOptionChar(char c) { return IsSameChar(c,'/') || IsSameChar(c,'-'); }

// Exit codes returned by SafeRun:
//////////////////////////////////
#define ecOK                            0
#define ecBadOpt                        1
#define ecNoArgs                        2
#define ecFailStart                     3
#define ecFailAuthz                     4
#define ecFailToken                     5

// To be called on fatal errors:
////////////////////////////////
int Fatal(int exitcode, BOOL verb, char* Title, char* Content) {
    switch ( exitcode ) {
        case ecOK :
            {
            if ( verb ) MessageBox(0,Content,Title,MB_OK+MB_ICONINFORMATION);
            }
            break;
        case ecBadOpt :
        case ecNoArgs :
            {
            if ( verb ) MessageBox(0,Content,Title,MB_OK+MB_ICONWARNING);
            }
            break;
        case ecFailAuthz :
        case ecFailToken :
        case ecFailStart :
            {
            int LastError = GetLastError();
            char msg[1024*32]; ZeroMemory(&msg,sizeof(msg));
            char tmp[1024*16]; ZeroMemory(&tmp,sizeof(tmp));
            int len = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ARGUMENT_ARRAY,
                                    NULL, LastError, 0, (LPTSTR) &tmp, sizeof(tmp), NULL);
            while ( (len > 0) && (tmp[len-1] < SpaceChar)) { len--; tmp[len]=0; }
            snprintf(msg,sizeof(msg),"Win32 error %d occured: \"%s\"\n\nin commnd line:\n\n%s\n\n%s",
                                    LastError, &tmp, GetCommandLine(), Content);
            if ( verb ) MessageBox(0,msg,Title,MB_OK+MB_ICONERROR);
            }
            break;
        default :
            break;
    }
    return exitcode;
}

// Safer variables:
///////////////////
HANDLE hToken = NULL;
BOOL UsesSafer = TRUE;    
SAFER_LEVEL_HANDLE hAuthzLevel = NULL;
DWORD hSaferLevel = SAFER_LEVELID_NORMALUSER;

// Safer  cleanup on exit:
//////////////////////////
void CleanupSafer(void) {
    if ( hAuthzLevel != NULL ) SaferCloseLevel(hAuthzLevel);
    hAuthzLevel = NULL;
}

// Main program entry:
//////////////////////
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow) {
//int main(int argc, char **argv) {

    // Initialize.
    STARTUPINFO si; 
    DWORD exitcode = ecOK;
    DWORD WaitTimeOut = 0;
    PROCESS_INFORMATION pi;
    BOOL VerboseMode = FALSE;
    char stopchar = SpaceChar;
    WORD DisplayMode = SW_SHOW;
    BOOL CreateProcessResult = FALSE;
    char* lpszCmdLine = GetCommandLine();
    DWORD PriorityClass = 0;

    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) lpszCmdLine++;

    // Skip executable file name, maybe double quoted.
    if ( IsSameChar(lpszCmdLine[0],DoubleQuota) ) { stopchar = DoubleQuota; lpszCmdLine++; }
    while ( !IsSameCharOrNul(TabToSpace(lpszCmdLine[0]),stopchar) ) lpszCmdLine++;
    if ( IsSameChar(lpszCmdLine[0],stopchar) ) lpszCmdLine++;
    
    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) lpszCmdLine++;
    
    // Check options /opt or -opt.
    while ( IsOptionChar(lpszCmdLine[0]) ) {
        lpszCmdLine++;
        while ( !IsSpaceCharOrTabOrNul(lpszCmdLine[0]) ) {
            switch ( lpszCmdLine[0] ) {
                case 'W' :  // Set waiting timeout.
                case 'w' :  WaitTimeOut = INFINITE;
                            break;
                case 'H' :  // Set hidden mode.
                case 'h' :  DisplayMode = SW_HIDE;
                            break;
                case '0' :  DisplayMode = SW_HIDE;
                            break;
                case '1' :  DisplayMode = SW_SHOWNORMAL;
                            break;
                case '2' :  DisplayMode = SW_SHOWMINIMIZED;
                            break;
                case '3' :  DisplayMode = SW_SHOWMAXIMIZED;
                            break;
                case '4' :  DisplayMode = SW_SHOWNOACTIVATE;
                            break;
                case '5' :  DisplayMode = SW_SHOW;
                            break;
                case '6' :  DisplayMode = SW_MINIMIZE;
                            break;
                case '7' :  DisplayMode = SW_SHOWMINNOACTIVE;
                            break;
                case '8' :  DisplayMode = SW_SHOWNA;
                            break;
                case '9' :  DisplayMode = SW_RESTORE;
                            break;
                case 'T' :  // Set trusted mode,i.e. don't use safer.
                case 't' :  hSaferLevel = SAFER_LEVELID_FULLYTRUSTED;
                            UsesSafer = FALSE;
                            break;
                case 'F' :  // Set safer level for fully trusted user.
                case 'f' :  hSaferLevel = SAFER_LEVELID_FULLYTRUSTED;
                            break;
                case 'O' :  // Set safer level for normal user.
                case 'o' :  hSaferLevel = SAFER_LEVELID_NORMALUSER;
                            break;
                case 'C' :  // Set safer level for constrained user.
                case 'c' :  hSaferLevel = SAFER_LEVELID_CONSTRAINED; 
                            break;
                case 'U' :  // Set safer level for untrusted user.
                case 'u' :  hSaferLevel = SAFER_LEVELID_UNTRUSTED;
                            break;
                case 'V' :  // Set verbose mode.
                case 'v' :  VerboseMode = TRUE;
                            break;
                case 'I' :  // Set IDLE_PRIORITY_CLASS
                case 'i' :  PriorityClass = IDLE_PRIORITY_CLASS;
                            break;
                case 'B' :  // Set BELOW_NORMAL_PRIORITY_CLASS
                case 'b' :  PriorityClass = BELOW_NORMAL_PRIORITY_CLASS;
                            break;
                case 'N' :  // Set NORMAL_PRIORITY_CLASS
                case 'n' :  PriorityClass = NORMAL_PRIORITY_CLASS;
                            break;
                case 'A' :  // Set ABOVE_NORMAL_PRIORITY_CLASS
                case 'a' :  PriorityClass = ABOVE_NORMAL_PRIORITY_CLASS;
                            break;
                case 'G' :  // Set HIGH_PRIORITY_CLASS
                case 'g' :  PriorityClass = HIGH_PRIORITY_CLASS;
                            break;
                case 'R' :  // Set REALTIME_PRIORITY_CLASS
                case 'r' :  PriorityClass = REALTIME_PRIORITY_CLASS;
                case '?' :  return Fatal(ecOK,TRUE,"SafeRun -> Help.",About);
                            break;
                default  :  return Fatal(ecBadOpt,VerboseMode,"SafeRun -> Invalid option found.",About);
                            break;
            }
            lpszCmdLine++;
        }
        // Skip space & tab chars.
        while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) lpszCmdLine++;
    }

    // Skip space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) lpszCmdLine++;

    // No arguments?
    if (lpszCmdLine[0] == 0) return Fatal(ecNoArgs,TRUE,"SafeRun -> No arguments found.",About);
    
    //  Generate the restricted token that we will use.
    if ( UsesSafer ) {
        if ( SaferCreateLevel(SAFER_SCOPEID_USER, hSaferLevel, 0, &hAuthzLevel, NULL) )
            atexit( CleanupSafer );
        else
            return Fatal(ecFailAuthz,VerboseMode,"SafeRun -> SaferCreateLevel failed.","");
        if ( !SaferComputeTokenFromLevel(hAuthzLevel, NULL, &hToken, 0, NULL) )
            return Fatal(ecFailToken,VerboseMode,"SafeRun -> SaferComputeTokenFromLevel failed.","");
    }
    
    // Initialize startup info.
    ZeroMemory(&si, sizeof(si));
    si.cb = sizeof(si); 
    si.lpDesktop = NULL;
    si.dwFlags = STARTF_USESHOWWINDOW; 
    si.wShowWindow = DisplayMode;
    
    // Create process with new console, with\without Safer mode.
    // Uses tail of original command line to create new process.
    if ( UsesSafer )
        CreateProcessResult = CreateProcessAsUser( hToken, NULL, lpszCmdLine, NULL, NULL, FALSE, CREATE_NEW_CONSOLE | PriorityClass, NULL, NULL, &si, &pi );
    else
        CreateProcessResult = CreateProcess(               NULL, lpszCmdLine, NULL, NULL, FALSE, CREATE_NEW_CONSOLE | PriorityClass, NULL, NULL, &si, &pi );
    
    // Wait while child running (if /w option specified). Take child exit code.
    if( CreateProcessResult ) { 
        if ( WaitForSingleObject(pi.hProcess, WaitTimeOut) == WAIT_OBJECT_0 )
            GetExitCodeProcess(pi.hProcess, &exitcode);
        CloseHandle( pi.hProcess ); 
        CloseHandle( pi.hThread ); 
    } else {
        return Fatal(ecFailStart,VerboseMode,"SafeRun -> CreateProcess failed.","");
    }
    
    // Done!
    return exitcode;
}