 {
 ************************************************************************
 VTRD Modbus Driver.
 ************************************************************************
 Next text uses by @Help command. Do not remove it.
 ************************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Modbus.XXX    - response come from &ModbusProxy server:
| @Modbus.Reply    ref cid tim port uid fid $$ans
| @Modbus.Refuse   ref cid tim port uid fid   msg
| @Modbus.Timeout  ref cid tim port uid fid $$req
|  ref  - sender device reference, expected &ModbusProxy.
|  cid  - command id, any user defined custom data.
|  tim  - request-response time, ms.
|  port - logical port on &ModbusProxy, expected 1..16.
|  uid  - unit id, i.e.MODBUS address, expected 1..247.
|  fid  - function id, expected 1,2,3,4,5,6,15,16.
|  ans  - answer  (response), in HEX_ENCODE format.
|  req  - request (original), in HEX_ENCODE format.
|  msg  - plain text message to explain reason of Refuse.
| @PARAM.XX         - write parameter, i.e. send to MODBUS
| @PARAM.ZEROCALIB  - Калибровка нижнего предела измерений
| @PARAM.ATMCALIB   - Калибровка верхнего предела измерений
| @PARAM.ADDRESS n  - Установка адреса n устройства
| @PARAM.BAUDRATE n - Установка скорости передачи данных и контроля четности (см. таблицу в инструкции)
| @Edit PARAM.XX    - edit parameter and write it
| @AssignTag t v    - Assign tag t to value v
| @DimTagUpdate     - Update tag from DIM
|************************************************************************
[]

 ************************************************************************
  [DeviceList]
  &VTRD.DRIVER = device software program
  [&VTRD.DRIVER]
  Comment = Modbus RTU Driver for VTRD controller.
  InquiryPeriod  = 0
  DevicePolling  = 10, tpNormal
  ProgramSource  = ..\DaqPas\VTRD_drv.pas
  DigitalFifo    = 1024
  AnalogFifo     = 1024
  DebugFlags     = 3
  OpenConsole    = 2
  Simulator      = 0
  ModbusPort     = 1
  ModbusUnitId   = 10
  ModbusTimeout  = 250
  ModbusDeadline = 60000
  ModbusPolling  = 100
  DelayOnStart   = 1000
  tagPrefix      = VTRD.D1
  ...etc...
 ************************************************************************
 }
program VTRD;
const
 {-------------------------------}{ Declare uses program constants:     }
 {$I _con_StdLibrary}             { Include all Standard constants,     }
 {$I _con_NetLibrary}             { NetLibrary constants                }
 {-------------------------------}{ And add User defined constants:     }
 dfStatist         = 32;          { DebugFlags - polling statistics     }
 MaxCmdNum         = 7;           { Maximal number of commands:         }
 cm_ReadPressure   = 1;           {  Read Holding Registers UnInt 16    }
 cm_ReadSerialSett = 2;           {  Read Holding Registers UnInt 16    }
 cm_WriteZeroCalib = 3;           {  (reg.1003)                         }
 cm_WriteAtmCalib  = 4;           {  (reg. 1004)                        }
 cm_WriteAddress   = 5;           {  (reg. 999)                         }
 cm_WriteBaudrate  = 6;           {  (reg. 1000)                        }
 ao_PRESSURE       = 0;           { (reg. 1010)                         }
 do_STATE_INDIC    = 0;           {  Status indicator                   }
 do_POLLRATERX     = 1;           {  Poll rate Rx                       }
 do_POLLRATETX     = 2;           {  Poll rate Tx                       }
 do_POLLRATEEX     = 3;           {  Poll rate Ex                       }
 do_ERRORCOUNT     = 4;           {  Error counter                      }
 do_POLLSUMMRX     = 5;           {  Poll summ Rx                       }
 do_POLLSUMMTX     = 6;           {  Poll summ Tx                       }
 do_POLLSUMMEX     = 7;           {  Poll summ Ex                       }
 SwapModeFloat     = 1;           { Byte swap mode for float values     }
 SwapModeInt16     = 2;           { Byte swap mode for integers         }
 SwapModeInt32     = 3;           { Byte swap mode for long integers    }
 st_NORMAL         = 0;           { Status: Green                       }
 st_TIMEOUT        = 1;           {  Yellow timeout                     }
 st_REFUSE         = 2;           {  Red refuse                         }
 st_SIMULAT        = 3;           {  Simulator mode                     }
 st_DISABLE        = 4;           {  Polling disabled                   }

type
 {-------------------------------}{ Declare uses program types:         }
 {$I _typ_StdLibrary}             { Include all Standard types,         }
 {-------------------------------}{ And add User defined types:         }

var
 {-------------------------------}{ Declare uses program variables:     }
 {$I _var_StdLibrary}             { Include all Standard variables,     }
 {$I _var_NetLibrary}             { NetLibrary variables                }
 {-------------------------------}{ And add User defined variables:     }
 VTRD           : record          { VTRD Driver data record             }
  Simulator     : Boolean;        { Simulator or Driver mode            }
  Modbus        : record          { Modbus data record                  }
   Port         : Integer;        { Logical Port on &ModbusProxy        }
   UnitId       : Integer;        { Modbus unit id                      }
   Timeout      : Integer;        { Modbus timeout, ms                  }
   Polling      : Integer;        { Modbus polling period, ms           }
   Deadline     : Integer;        { Modbus deadline time, ms            }
   DelayOnStart : Integer;        { Command cycle delay on start        }
   Cmd          : record          { Command cycle data record           }
    Num         : Integer;        {  Current running command number     }
    Enabled     : array [1..MaxCmdNum] of Boolean; { Enable polling     }
    FuncId      : array [1..MaxCmdNum] of Integer; { Modbus Function Id }
    SAddr       : array [1..MaxCmdNum] of Integer; { Start Address      }
    Quant       : array [1..MaxCmdNum] of Integer; { Quantity of data   }
    OpData      : array [1..MaxCmdNum] of Real;    { Operation data     }
    OpBuff      : array [1..MaxCmdNum] of Real;    { Operation buffer   }
   end;                           {                                     }
   Poll         : record          { Polling transaction data record     }
    ref         : Integer;        { Last sent device reference          }
    cid         : Integer;        { Last sent command id                }
    tim         : Real;           { Last polling time, ms               }
    port        : Integer;        { Last polling port                   }
    uid         : Integer;        { Last polling unit id                }
    fid         : Integer;        { Last sent function id               }
    saddr       : Integer;        { Last sent start address             }
    quant       : Integer;        { Last sent quantity of registers     }
    dat         : String;         { Last sent PDU=(fid+dat) data        }
    Summ,                         { Poll summ counters                  }
    Rate        : record          { Poll rate on last second            }
     Rx         : Real;           {  Rx poll summ/rate - receiver       }
     Tx         : Real;           {  Tx poll summ/rate - transmitter    }
     Ex         : Real;           {  Ex poll summ/rate - errors         }
    end;                          {                                     }
   end;                           {                                     }
  end;                            {                                     }
  POLL          : record          { Poll parameters:                    }
   ENABLE       : TTagRef;        {  Poll enable flag                   }
  end;                            {                                     }
  PARAM         : record          { Device parameters:                  }
   PRESSURE     : TTagRef;        {  Pressure in Pa                     }
   STATE        : TTagRef;        {  Status indicator                   }
   ADDRESS      : TTagRef;        {  Device Address                     }
   BAUDRATE     : TTagRef;        {  Device Baudrate                    }
  end;                            {                                     }
  SIM           : record          { Simulator data:                     }
   PRESSURE     : Real;           {  Pressure in Pa                     }
   STATE        : Integer;        {  Status indicator                   }
   ADDRESS      : Integer;        {  Device Address                     }
   BAUDRATE     : Integer;        {  Device Baudrate                    }
  end;                            {                                     }
  POLLRATE      : record          {                                     }
   RX,TX,EX     : TTagRef;        {                                     }
  end;                            {                                     }
  POLLSUMM      : record          {                                     }
   RX,TX,EX     : TTagRef;        {                                     }
  end;                            {                                     }
  ERROR         : record          {                                     }
   COUNT        : TTagRef;        {                                     }
  end;                            {                                     }
 end;                             {                                     }
 cmd_mbReply      : Integer;      { Console commands:                   }
 cmd_mbPoll       : Integer;      {                                     }
 cmd_mbRefuse     : Integer;      {                                     }
 cmd_mbTimeout    : Integer;      {                                     }
 cmd_mbZeroPortCnt: Integer;      {                                     }
 cmd_mbClrSumm    : Integer;      {                                     }
 cmd_parZeroCalib : Integer;      {  Set Zero Calibration               }
 cmd_parAtmCalib  : Integer;      {  Set Atm Calibration                }
 cmd_parAddress   : Integer;      {  Set Device modbus adress           }
 cmd_parBaudrate  : Integer;      {  Set Serial interface settings      }
 cmd_Edit         : Integer;      {  Common commands:                   }
 cmd_menuBaudrate : Integer;      {  Start Baudrate menu                }
 cmd_DimTagUpdate : Integer;      {  @DimTagUpdate                      }
 cmd_AssignTag    : Integer;      {  @AssignTag                         }
 {-------------------------------}{ Declare procedures & functions:     }
 {$I _fun_StdLibrary}             { Include all Standard functions,     }
 {-------------------------------}{ And add User defined functions:     }
 {$I _fun_NetLibrary}             { NetLibrary functions                }
 {
 Prefix for DIM @remote commands.
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 //
 // Xor bit on click (local version)
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Xor bit on click (remote version).
 }
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(tag)+' '+Str(iXor(iGetTag(tag),XorMask)));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // Command cycle routines.
 //
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsUsableCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsUsableCmdNum:=(VTRD.Modbus.Cmd.FuncId[Num]<>0)
  else IsUsableCmdNum:=False;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsEnabledCmdNum:=(VTRD.Modbus.Cmd.FuncId[Num]<>0) and VTRD.Modbus.Cmd.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer;
 begin
  i:=0;
  while (i<MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   if IsEnabledCmdNum(Num)
   then i:=MaxCmdNum
   else i:=i+1;
  end;
  NextEnabledCmdNum:=Num;
 end;
 procedure EnableCmdNum(Num:Integer; Enabled:Boolean);
 begin
  if IsUsableCmdNum(Num) then VTRD.Modbus.Cmd.Enabled[Num]:=Enabled;
 end;
 procedure HoldCmdOpData(Num:Integer; OpData:Real);
 begin
  if IsUsableCmdNum(Num) then begin
   VTRD.Modbus.Cmd.Enabled[Num]:=not IsNaN(OpData);
   VTRD.Modbus.Cmd.OpBuff[Num]:=OpData;
  end;
 end;
 procedure ApplyCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then
  if not IsNaN(VTRD.Modbus.Cmd.OpBuff[Num]) then begin
   VTRD.Modbus.Cmd.OpData[Num]:=VTRD.Modbus.Cmd.OpBuff[Num];
   VTRD.Modbus.Cmd.OpBuff[Num]:=_NaN;
  end;
 end;
 procedure ReleaseCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then HoldCmdOpData(Num,VTRD.Modbus.Cmd.OpBuff[Num]);
 end;
 function GetCmdOpData(Num:Integer):Real;
 begin
  if IsUsableCmdNum(Num)
  then GetCmdOpData:=VTRD.Modbus.Cmd.OpData[Num]
  else GetCmdOpData:=_NaN;
 end;
 procedure InitCmdItem(Num:Integer; Enabled:Boolean; FuncId,SAddr,Quant:Integer; OpData,OpBuff:Real);
 begin
  if IsValidCmdNum(Num) then begin
   VTRD.Modbus.Cmd.Enabled[Num]:=Enabled;
   VTRD.Modbus.Cmd.FuncId[Num]:=FuncId;
   VTRD.Modbus.Cmd.SAddr[Num]:=SAddr;
   VTRD.Modbus.Cmd.Quant[Num]:=Quant;
   VTRD.Modbus.Cmd.OpData[Num]:=OpData;
   VTRD.Modbus.Cmd.OpBuff[Num]:=OpBuff;
  end;
 end;
 //
 // Command table cleanup and initialization.
 //
 procedure ClearCmdTable;
 var Num:Integer;
 begin
  VTRD.Modbus.Cmd.Num:=0;
  for Num:=1 to MaxCmdNum do InitCmdItem(Num,False,0,0,0,0,_NaN);
 end;
 procedure InitCmdTable;
 begin
  ClearCmdTable;
  // Table of CommandId          Enabled FunctionId        SAddr Quant OpData OpBuff
  InitCmdItem(cm_ReadPressure,   True,   modbus_fn_ReadIR, 1010, 2,    0,     _NaN);
  InitCmdItem(cm_ReadSerialSett, True,   modbus_fn_ReadHR, 999,  2,    0,     _NaN);
  InitCmdItem(cm_WriteZeroCalib, False,  modbus_fn_WritMC, 1003, 1,    0,     _NaN);
  InitCmdItem(cm_WriteAtmCalib,  False,  modbus_fn_WritMC, 1004, 1,    0,     _NaN);
  InitCmdItem(cm_WriteAddress,   False,  modbus_fn_WritMR, 999,  1,    0,     _NaN);
  InitCmdItem(cm_WriteBaudrate,  False,  modbus_fn_WritMR, 1000, 1,    0,     _NaN);
 end;
 //
 // Assign modbus last sent polling request record.
 //
 procedure AssignModbusPoll(ref,cid:Integer; tim:Real; port,uid,fid,saddr,quant:Integer; dat:String);
 begin
  VTRD.Modbus.Poll.ref:=ref;     VTRD.Modbus.Poll.cid:=cid;     VTRD.Modbus.Poll.tim:=tim;
  VTRD.Modbus.Poll.port:=port;   VTRD.Modbus.Poll.uid:=uid;     VTRD.Modbus.Poll.fid:=fid;
  VTRD.Modbus.Poll.saddr:=saddr; VTRD.Modbus.Poll.quant:=quant; VTRD.Modbus.Poll.dat:=dat;
 end;
 //
 // Clear modbus polling request to be ready for next polling.
 //
 procedure ClearModbusPoll;
 begin
  VTRD.Modbus.Poll.cid:=0;
  VTRD.Modbus.Poll.dat:='';
 end;
 //
 // Clear modbus poll summ counters.
 //
 procedure ClearModbusSumm;
 begin
  VTRD.Modbus.Poll.Summ.Rx:=0;
  VTRD.Modbus.Poll.Summ.Tx:=0;
  VTRD.Modbus.Poll.Summ.Ex:=0;
 end;
 //
 // Clear modbus poll rate counters.
 //
 procedure ClearModbusRate;
 begin
  VTRD.Modbus.Poll.Rate.Rx:=0;
  VTRD.Modbus.Poll.Rate.Tx:=0;
  VTRD.Modbus.Poll.Rate.Ex:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Real);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 //
 // Format error message with with error counter increment.
 // Usage is like: Trouble(GotBug(...)); Problem(GotBug(...));
 //
 function GotBug(msg:String):String;
 begin
  IncSummRate(VTRD.Modbus.Poll.Summ.Ex,VTRD.Modbus.Poll.Rate.Ex);
  GotBug:=msg;
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Меню "Инструменты"');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+DevName);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand(DimRemote+'@ClearModbusSumm');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+ModbusProxy);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand(DimRemote+'@ZeroPortCounters 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Перезапустить локальный драйвер '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   if DIM_IsClientMode then begin
    n:=n+EditAddInputLn('Перезапустить удаленный драйвер '+DevName);
    n:=n+EditAddConfirm(EditGetLastInputLn);
    n:=n+EditAddCommand(DimRemote+'@SysEval @Daq Compile '+DevName);
   end;
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VTRD: Калибровка нижнего предела измерений');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.ZEROCALIB');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VTRD: Калибровка верхнего предела измерений');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.ATMCALIB');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VTRD: Установка адреса устройства');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit ADDRESS');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('VTRD: Установка скорости передачи данных');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@MenuBaudrateStarter');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 {
 Menu Baudrate Starter to start editing.
 }
 procedure MenuBaudrateStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Установка скорости передачи');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('9600 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('19200 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('38400 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 2');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('57600 бод/с');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand(DimRemote+'@PARAM.BAUDRATE 3');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_BAUDRATE'),Str(iGetTag(VTRD.PARAM.BAUDRATE.tag)));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Baudrate Handler to handle editing.
 }
 procedure MenuBaudrateHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_BAUDRATE'));
 end;
 //
 // Find the current Proxy device to send @Modbus.Poll message.
 //
 function devTheProxy:Integer;
 begin
  if VTRD.Simulator                 // In simulation mode
  then devTheProxy:=devMySelf       // send messages to himself
  else devTheProxy:=devModbusProxy; // otherwise use &ModbusProxy
 end;
 //
 // Check if communication port is opened or not.
 // As long as we use Proxy device, check Port number & device reference.
 //
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(VTRD.Modbus.Port>0) and (devTheProxy<>0);
 end;
 //
 // Calculate Modbus request PDU by command id.
 // cid   - (in)  command id number.
 // fid   - (out) Modbus function id.
 // saddr - (out) zero-based start address of registers to read/write.
 // quant - (out) data quantity, i.e. number of r/w registers or single register r/w value.
 // dat   - (out) Modbus PDU data unit, PDU=(fid+dat).
 //
 function VTRD_CalcPDU(cid:Integer; var fid,saddr,quant:Integer; var dat:String):Boolean;
 begin
  fid:=0; saddr:=0; quant:=0; dat:='';
  if IsEnabledCmdNum(cid) then begin
   fid:=VTRD.Modbus.Cmd.FuncId[cid];
   saddr:=VTRD.Modbus.Cmd.SAddr[cid];
   quant:=VTRD.Modbus.Cmd.Quant[cid];
   if (cid=cm_ReadPressure)   then dat:=modbus_encode_pdu('R',fid,saddr,quant,'') else
   if (cid=cm_ReadSerialSett) then dat:=modbus_encode_pdu('R',fid,saddr,quant,'') else
   if (cid=cm_WriteZeroCalib) then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   if (cid=cm_WriteAtmCalib)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   if (cid=cm_WriteAddress)   then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   if (cid=cm_WriteBaudrate)  then dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeFloat)) else
   Trouble(GotBug('Invalid command id '+Str(cid)));
  end;
  VTRD_CalcPDU:=modbus_func_ok(fid) and (Length(dat)>0);
 end;
 //
 // Main Modbus command handler to process Modbus device reply.
 // cid   - (in)  command id number.
 // fid   - (in)  Modbus function id.
 // saddr - (in)  zero-based start address of registers to read/write.
 // quant - (in)  data quantity, i.e. number of r/w registers or single register r/w value.
 // raw   - (in)  raw coils/registers data array.
 //
 procedure VTRD_OnCommand(cid,fid,saddr,quant:Integer; raw:String);
 var addr,offs,indic:Integer; r:Real;
  procedure Cleanup;
  begin
   addr:=0; offs:=0; indic:=0; r:=0;
  end;
 begin
  Cleanup;
  if IsValidCmdNum(cid) then begin
   if cid=cm_ReadPressure then begin // Read Pressure
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=1010) and (quant-offs+1>=1) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(VTRD.PARAM.PRESSURE.tag,r));
      UpdateAo(ao_PRESSURE,time,r);
     end;
    end;
   end else if cid=cm_ReadSerialSett then begin // Read serial settings
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=999) and (quant-offs+1>=1) then begin
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(VTRD.PARAM.ADDRESS.tag,round(r)));
     end else
     if (addr=1000) and (quant-offs+1>=1) then begin
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(VTRD.PARAM.BAUDRATE.tag,round(r)));
      EnableCmdNum(cm_ReadSerialSett,False);
     end;
    end;
   end else
   if (cid=cm_WriteZeroCalib) then ReleaseCmdOpData(cid) else
   if (cid=cm_WriteAtmCalib)  then ReleaseCmdOpData(cid) else
   if (cid=cm_WriteAddress)   then ReleaseCmdOpData(cid) else
   if (cid=cm_WriteBaudrate)  then ReleaseCmdOpData(cid) else
   Trouble(GotBug('Unexpected command '+Str(cid)));
   if VTRD.Simulator then indic:=st_SIMULAT else indic:=st_NORMAL;
   UpdateDo(do_STATE_INDIC,time,indic);
   bNul(iSetTag(VTRD.PARAM.STATE.tag,indic));
  end;
  Cleanup;
 end;
 //
 // Data handler on @Modbus.Reply event. Process reply comes from Modbus device.
 //
 procedure VTRD_OnReply(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
  procedure Cleanup;
  begin
   raw:=''; saddr:=0; quant:=0;
  end;
 begin
  Cleanup;
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>VTRD.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>VTRD.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>VTRD.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>VTRD.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>VTRD.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=VTRD.Modbus.Poll.saddr; quant:=VTRD.Modbus.Poll.quant;
   if modbus_decode_pdu('A',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>VTRD.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>VTRD.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else
    if modbus_is_except(fid) then begin
     Problem(GotBug('MODBUS EXCEPTION '+Str(modbus_ext_byte(raw,0))+' ON COMMAND '+Str(cid)));
    end else begin
     IncSummRate(VTRD.Modbus.Poll.Summ.Rx,VTRD.Modbus.Poll.Rate.Rx);
     VTRD_OnCommand(cid,fid,saddr,quant,raw);
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  Cleanup;
 end;
 //
 // Data handler on @Modbus.Poll event.
 // This procedure calls in simulation mode only.
 //
 procedure VTRD_OnSimPoll(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
  procedure Cleanup;
  begin
   raw:=''; saddr:=0; quant:=0;
  end;
 begin
  Cleanup;
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>VTRD.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>VTRD.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>VTRD.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>VTRD.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>VTRD.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=VTRD.Modbus.Poll.saddr; quant:=VTRD.Modbus.Poll.quant;
   if modbus_decode_pdu('R',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>VTRD.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>VTRD.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else begin
     if DebugFlagEnabled(dfDetails) then Details('Simulate polling '+Str(cid));
     if (cid=cm_ReadPressure) then begin
      raw:=modbus_dump_float(VTRD.SIM.PRESSURE,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_ReadSerialSett) then begin
      raw:=modbus_dump_int16(VTRD.SIM.ADDRESS,SwapModeInt32)
          +modbus_dump_int16(VTRD.SIM.BAUDRATE,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_WriteAddress) then begin
      VTRD.SIM.ADDRESS:=modbus_ext_int16(raw,0,SwapModeFloat);
      EnableCmdNum(cm_ReadSerialSett,true);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else if (cid=cm_WriteBaudrate) then begin
      VTRD.SIM.BAUDRATE:=modbus_ext_int16(raw,0,SwapModeFloat);
      EnableCmdNum(cm_ReadSerialSett,true);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else DevPostCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,
              modbus_as_except(fid,true),Dump(Chr(modbus_er_ILLFUN))));
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  Cleanup;
 end;

 //
 // Pressure Simulation
 //
 procedure VTRD_Simulation;
 var
 begin
  VTRD.SIM.PRESSURE:=VTRD.SIM.PRESSURE-0.0002;
  if VTRD.SIM.PRESSURE<0 then VTRD.SIM.PRESSURE:=1.5;
 end;

 //
 // Send console command (cmd data) to set new tag value
 //
 procedure SendTagCmd(tag:Integer; cmd,data:String; min,max:Real);
 var v:Real;
 begin
  if IsRefTag(tag) then
  if not IsEmptyStr(cmd) then
  if not IsEmptyStr(data) then
  if TypeTag(tag)<=2 then begin
   v:=rVal(data);
   if (v<min) then begin v:=min; data:=Str(min); end;
   if (v>max) then begin v:=max; data:=Str(max); end;
   if IsNaN(v) then Trouble(GotBug('Invalid tag edit')) else
   if TypeTag(tag)=1 then DevPostCmdLocal(cmd+' '+Str(Round(v))) else
   if TypeTag(tag)=2 then DevPostCmdLocal(cmd+' '+Trim(data));
  end else if TypeTag(tag)=3 then begin
   DevPostCmdLocal(cmd+' '+Trim(data));
  end;
 end;

 {
 GUI Handler to process user input...
 }
 procedure GUIHandler;
 var s:String; ClickCurve:Integer;
  procedure Cleanup;
  begin
   DIM_GuiClickBuff:=''; s:=''; ClickCurve:=0;
  end;
 begin
  Cleanup;
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Copy GUI click to DIM buffer for remote execution.
   }
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Handle local clicks
     }
     if ClickIsLocal then begin
      //
      // Handle tag clicks...
      //
      ClickTagXorRemote(VTRD.POLL.ENABLE.tag,1);
      //
      // Handle sensor clicks...
      //
      if Pos('.TOOLS',ClickSensor)<>0 then begin
       bNul(Voice(snd_Click));
       MenuToolsStarter;
      end;
      if IsSameText(ClickSensor,'HELP') then begin
       DevPostCmdLocal('@BrowseHelp '+DaqFileRef(ReadIni('[DAQ] HelpFile'),'.md'));
       bNul(Voice(snd_Click));
      end;
      {
      Select Plot & Tab windows by curve...
      }
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      {
      Console commands: @url_encoded_sensor ...
      }
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     {
     Handle remote clicks comes from DIM via @DimGuiClick message.
     @DimGuiClick default handler decode and write events to FIFO,
     so we can find it as clicks and can handle it in usual way.
     }
     if ClickIsRemote then begin
      {
      Show time difference.
      }
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      {
      Handle remote console commands...
      }
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   //
   // If tag edit complete, send command to apply changes
   //
   if CheckEditTag(VTRD.PARAM.ADDRESS.tag, s) then SendTagCmd(VTRD.PARAM.ADDRESS.tag, DimRemote+'@PARAM.ADDRESS', s,1,247);
   if CheckEditTag(VTRD.PARAM.BAUDRATE.tag,s) then SendTagCmd(VTRD.PARAM.BAUDRATE.tag,DimRemote+'@PARAM.BAUDRATE',s,0,19);
   //
   // Menus handler
   //
   MenuToolsHandler;
   MenuBaudrateHandler;
   {
   Warning, Information dialog completion.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end;
  if EditStateError then begin
   Problem('Dialog error detected!');
   EditReset;
  end;
  Cleanup;
 end;
 //
 // VTRD Driver Command Cycle polling.
 //
 procedure VTRD_CMD_POLL;
  //
  // Send @Modbus.Poll ... command to &ModbusProxy and save sent request data in Modbus.Poll record.
  //
  procedure PollModbusProxy(cid:Integer);
  var dev,ref,tot,port,uid,fid,saddr,quant:Integer; dat:String;
   procedure Cleanup;
   begin
    dev:=0; ref:=0; tot:=0; port:=0; uid:=0; fid:=0; saddr:=0; quant:=0; dat:='';
   end;
  begin
   Cleanup;
   ApplyCmdOpData(cid);
   if VTRD_CalcPDU(cid,fid,saddr,quant,dat) then begin
    dev:=devTheProxy; ref:=devMySelf; tot:=VTRD.Modbus.Timeout; port:=VTRD.Modbus.Port; uid:=VTRD.Modbus.UnitId;
    if DevSend(dev,modbus_proxy_poll('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat)+CRLF)>0 then begin
     if DebugFlagEnabled(dfViewExp)
     then ViewExp('COM: '+modbus_proxy_nice('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat,32));
     AssignModbusPoll(dev,cid,mSecNow,port,uid,fid,saddr,quant,dat);
     IncSummRate(VTRD.Modbus.Poll.Summ.Tx,VTRD.Modbus.Poll.Rate.Tx);
    end else begin
     Trouble(GotBug('Fail to send command '+Str(cid)));
     ClearModbusPoll;
    end;
   end else begin
    Trouble(GotBug('Fail to calc command '+Str(cid)));
    ClearModbusPoll;
   end;
   Cleanup;
  end;
 begin
  if IsPortOpened and not DIM_IsClientMode then begin
   if iGetTag(VTRD.POLL.ENABLE.tag)<>0 then begin
    if IsValidCmdNum(VTRD.Modbus.Poll.cid) then begin
     //
     // Request in progress, waiting @Modbus.Reply/@Modbus.Refuse/@Modbus.Timeout.
     // Handle Deadline if was no responce for too long time: repeat polling again.
     //
     if (mSecNow>VTRD.Modbus.Poll.tim+VTRD.Modbus.Deadline) then begin
      Trouble(GotBug('Deadline detected, repeat polling again...'));
      PollModbusProxy(VTRD.Modbus.Poll.cid);
     end;
    end else begin
     //
     // If request is cleared, send new @Modbus.Poll request by timer.
     //
     if (mSecNow>=VTRD.Modbus.Poll.tim+VTRD.Modbus.Polling) then begin
      VTRD.Modbus.Cmd.Num:=NextEnabledCmdNum(VTRD.Modbus.Cmd.Num);
      if IsEnabledCmdNum(VTRD.Modbus.Cmd.Num)
      then PollModbusProxy(VTRD.Modbus.Cmd.Num);
     end;
    end;
   end else if iGetTag(VTRD.POLL.ENABLE.tag)=0 then begin
    UpdateDo(do_STATE_INDIC,time,st_DISABLE);
    bNul(iSetTag(VTRD.PARAM.STATE.tag,st_DISABLE));
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    bNul(rSetTag(VTRD.POLLRATE.RX.tag,VTRD.Modbus.Poll.Rate.Rx));
    bNul(rSetTag(VTRD.POLLRATE.TX.tag,VTRD.Modbus.Poll.Rate.Tx));
    bNul(rSetTag(VTRD.POLLRATE.EX.tag,VTRD.Modbus.Poll.Rate.Ex));
    bNul(rSetTag(VTRD.POLLSUMM.RX.tag,VTRD.Modbus.Poll.Summ.Rx));
    bNul(rSetTag(VTRD.POLLSUMM.TX.tag,VTRD.Modbus.Poll.Summ.Tx));
    bNul(rSetTag(VTRD.POLLSUMM.EX.tag,VTRD.Modbus.Poll.Summ.Ex));
    bNul(rSetTag(VTRD.ERROR.COUNT.tag,GetErrCount(-1)));
    UpdateDo(do_POLLRATERX,time,VTRD.Modbus.Poll.Rate.Rx);
    UpdateDo(do_POLLRATETX,time,VTRD.Modbus.Poll.Rate.Tx);
    UpdateDo(do_POLLRATEEX,time,VTRD.Modbus.Poll.Rate.Ex);
    UpdateDo(do_POLLSUMMRX,time,VTRD.Modbus.Poll.Summ.Rx);
    UpdateDo(do_POLLSUMMTX,time,VTRD.Modbus.Poll.Summ.Tx);
    UpdateDo(do_POLLSUMMEX,time,VTRD.Modbus.Poll.Summ.Ex);
    UpdateDo(do_ERRORCOUNT,time,GetErrCount(-1));
    if DebugFlagEnabled(dfStatist)
    then Success('PollRate: Rx='+Str(VTRD.Modbus.Poll.Rate.Rx)
                        +'  Tx='+Str(VTRD.Modbus.Poll.Rate.Tx)
                        +'  Ex='+Str(VTRD.Modbus.Poll.Rate.Ex));
    ClearModbusRate;
   end;
  end;
 end;
 {
 Handle message @AssignTag arg
 }
 procedure VTRD_OnAssignTag(arg:String);
 var tag:Integer; w1,w2:String;
  procedure Cleanup;
  begin
   w1:=''; w2:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   w1:=ExtractWord(1,arg);
   tag:=FindTag(w1);
   if (tag<>0) then begin
    w2:=ExtractWord(2,arg);
    if tag=VTRD.POLL.ENABLE.tag then UpdateTag(tag,w2,0,1);
   end;
  end;
  Cleanup;
 end;
 {
 Handle message @DimUpdateTag arg
 }
 procedure VTRD_OnDimUpdateTag(arg:String);
 var tag,typ:Integer; x,y:Real;
 begin
  if (arg<>'') then begin
   if DIM_IsClientMode and not DIM_IsServerMode then begin
    tag:=FindTag(ExtractWord(1,arg));
    if (tag<>0) then begin
     typ:=TypeTag(tag);
     if (typ=1) then y:=iGetTag(tag) else
     if (typ=2) then y:=rGetTag(tag) else y:=_Nan;
     x:=time;
     if not IsNan(y) then begin
      if tag=VTRD.PARAM.PRESSURE.tag then UpdateAo(ao_PRESSURE,x,y);
      if tag=VTRD.PARAM.STATE.tag    then UpdateDo(do_STATE_INDIC,x,y);
      if tag=VTRD.ERROR.COUNT.tag    then UpdateDo(do_ERRORCOUNT,x,y);
      if tag=VTRD.POLLRATE.RX.tag    then UpdateDo(do_POLLRATERX,x,y);
      if tag=VTRD.POLLRATE.TX.tag    then UpdateDo(do_POLLRATETX,x,y);
      if tag=VTRD.POLLRATE.EX.tag    then UpdateDo(do_POLLRATEEX,x,y);
      if tag=VTRD.POLLSUMM.RX.tag    then UpdateDo(do_POLLSUMMRX,x,y);
      if tag=VTRD.POLLSUMM.TX.tag    then UpdateDo(do_POLLSUMMTX,x,y);
      if tag=VTRD.POLLSUMM.EX.tag    then UpdateDo(do_POLLSUMMEX,x,y);
     end;
    end;
   end;
  end;
 end;
 //
 // Tags filling
 //
 procedure VTRD_FillTags(InitVal:Real);
 begin
  VTRD.POLL.ENABLE.val:=InitVal;
  VTRD.PARAM.PRESSURE.val:=InitVal;
  VTRD.PARAM.STATE.val:=InitVal;
  VTRD.PARAM.ADDRESS.val:=InitVal;
  VTRD.ERROR.COUNT.val:=InitVal;
  VTRD.POLLRATE.RX.val:=InitVal;
  VTRD.POLLRATE.TX.val:=InitVal;
  VTRD.POLLRATE.EX.val:=InitVal;
  VTRD.POLLSUMM.RX.val:=InitVal;
  VTRD.POLLSUMM.TX.val:=InitVal;
  VTRD.POLLSUMM.EX.val:=InitVal;
 end;
 {
 Update DIM services
 }
 procedure DimUpdateState;
 begin
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then VTRD_FillTags(-MaxReal);
   if ShouldRefresh(VTRD.POLL.ENABLE.val,    GetStampOfTag(VTRD.POLL.ENABLE.tag,0))>0    then DIM_UpdateTag(VTRD.POLL.ENABLE.tag,'');
   if SysTimer_Pulse(500)>0 then
   if ShouldRefresh(VTRD.PARAM.PRESSURE.val, GetStampOfTag(VTRD.PARAM.PRESSURE.tag,0))>0 then DIM_UpdateTag(VTRD.PARAM.PRESSURE.tag,'');
   if ShouldRefresh(VTRD.PARAM.STATE.val,    GetStampOfTag(VTRD.PARAM.STATE.tag,0))>0    then DIM_UpdateTag(VTRD.PARAM.STATE.tag,'');
   if ShouldRefresh(VTRD.PARAM.ADDRESS.val,  GetStampOfTag(VTRD.PARAM.ADDRESS.tag,0))>0  then DIM_UpdateTag(VTRD.PARAM.ADDRESS.tag,'');
   if ShouldRefresh(VTRD.PARAM.BAUDRATE.val, GetStampOfTag(VTRD.PARAM.BAUDRATE.tag,0))>0 then DIM_UpdateTag(VTRD.PARAM.BAUDRATE.tag,'');
   if ShouldRefresh(VTRD.ERROR.COUNT.val,    GetStampOfTag(VTRD.ERROR.COUNT.tag,0))>0    then DIM_UpdateTag(VTRD.ERROR.COUNT.tag,'');
   if ShouldRefresh(VTRD.POLLRATE.RX.val,    GetStampOfTag(VTRD.POLLRATE.RX.tag,0))+
      ShouldRefresh(VTRD.POLLRATE.TX.val,    GetStampOfTag(VTRD.POLLRATE.TX.tag,0))+
      ShouldRefresh(VTRD.POLLRATE.EX.val,    GetStampOfTag(VTRD.POLLRATE.EX.tag,0))>0 then DIM_UpdateTag(VTRD.POLLRATE.RX.tag,'');
   if ShouldRefresh(VTRD.POLLSUMM.RX.val,    GetStampOfTag(VTRD.POLLSUMM.RX.tag,0))+
      ShouldRefresh(VTRD.POLLSUMM.TX.val,    GetStampOfTag(VTRD.POLLSUMM.TX.tag,0))+
      ShouldRefresh(VTRD.POLLSUMM.EX.val,    GetStampOfTag(VTRD.POLLSUMM.EX.tag,0))>0 then DIM_UpdateTag(VTRD.POLLSUMM.RX.tag,'');
  end;
 end;
 //
 // VTRD tags initialization.
 //
 procedure VTRD_InitTags(Prefix:String);
 begin
  DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
  InitTag(VTRD.POLL.ENABLE.tag,    Prefix+'.ENABLE',      1);
  InitTag(VTRD.PARAM.PRESSURE.tag, Prefix+'.PRESSURE',    2);
  InitTag(VTRD.PARAM.STATE.tag,    Prefix+'.STATE',       1);
  InitTag(VTRD.PARAM.ADDRESS.tag,  Prefix+'.ADDRESS',     1);
  InitTag(VTRD.PARAM.BAUDRATE.tag, Prefix+'.BAUDRATE',    1);
  InitTag(VTRD.POLLRATE.RX.tag,    Prefix+'.POLLRATE.RX', 2);
  InitTag(VTRD.POLLRATE.TX.tag,    Prefix+'.POLLRATE.TX', 2);
  InitTag(VTRD.POLLRATE.EX.tag,    Prefix+'.POLLRATE.EX', 2);
  InitTag(VTRD.ERROR.COUNT.tag,    Prefix+'.ERROR.COUNT', 2);
  InitTag(VTRD.POLLSUMM.RX.tag,    Prefix+'.POLLSUMM.RX', 2);
  InitTag(VTRD.POLLSUMM.TX.tag,    Prefix+'.POLLSUMM.TX', 2);
  InitTag(VTRD.POLLSUMM.EX.tag,    Prefix+'.POLLSUMM.EX', 2);
 end;
 //
 // VTRD Driver cleanup.
 //
 procedure VTRD_Clear;
 begin
  VTRD.Simulator:=False;
  VTRD.Modbus.Port:=0;
  VTRD.Modbus.UnitId:=0;
  VTRD.Modbus.Timeout:=0;
  VTRD.Modbus.Polling:=0;
  VTRD.Modbus.Deadline:=0;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
  VTRD_FillTags(-MaxReal);
 end;
 //
 // VTRD Driver initialization...
 //
 procedure VTRD_Init;
 begin
  //
  // Read ini file variables
  //
  VTRD_InitTags(ReadIni('tagPrefix'));
  VTRD.Simulator:=iValDef(ReadIni('Simulator'),0)<>0; Success('Simulator='+Str(Ord(VTRD.Simulator)));
  VTRD.Modbus.Port:=iValDef(ReadIni('ModbusPort'),1); Success('ModbusPort='+Str(VTRD.Modbus.Port));
  VTRD.Modbus.UnitId:=iValDef(ReadIni('ModbusUnitId'),1); Success('ModbusUnitId='+Str(VTRD.Modbus.UnitId));
  VTRD.Modbus.Polling:=iValDef(ReadIni('ModbusPolling'),1000); Success('ModbusPolling='+Str(VTRD.Modbus.Polling));
  VTRD.Modbus.Timeout:=iValDef(ReadIni('ModbusTimeout'),250); Success('ModbusTimeout='+Str(VTRD.Modbus.Timeout));
  VTRD.Modbus.Deadline:=iValDef(ReadIni('ModbusDeadline'),60000); Success('ModbusDeadline='+Str(VTRD.Modbus.Deadline));
  VTRD.Modbus.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000); Success('DelayOnStart='+Str(VTRD.Modbus.DelayOnStart));
  if not DIM_IsClientMode then if not VTRD.Simulator then
  if IsRefDevice(RefFind('Device '+ModbusProxy)) then Success(ModbusProxy+' device found') else Trouble(ModbusProxy+' device not found!');
  if DIM_IsServerMode then Success('Run as Server') else if DIM_IsClientMode then Success('Run as Client');
  //
  // Initialize values
  //
  VTRD.SIM.ADDRESS:=VTRD.Modbus.UnitId;
  VTRD.SIM.BAUDRATE:=1;
  //
  // Initialize Cmd command table & clear Poll record.
  //
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  InitCmdTable;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  VTRD_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  //
  // Standard initializations.
  //
  InitNetLibrary;
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  VTRD_Init;
  if not DIM_IsClientMode then
  if Val(ReadIni('CustomIniAutoLoad'))=1 then DevPostCmdLocal('@LoadIni');
  //
  // StdIn Command registration.
  //
  cmd_mbReply      :=RegisterStdInCmd('@Modbus.Reply','');
  cmd_mbPoll       :=RegisterStdInCmd('@Modbus.Poll','');
  cmd_mbRefuse     :=RegisterStdInCmd('@Modbus.Refuse','');
  cmd_mbTimeout    :=RegisterStdInCmd('@Modbus.Timeout','');
  cmd_mbZeroPortCnt:=RegisterStdInCmd('@ZeroPortCounters','');
  cmd_mbClrSumm    :=RegisterStdInCmd('@ClearModbusSumm','');
  cmd_parZeroCalib :=RegisterStdInCmd('@PARAM.ZEROCALIB','');
  cmd_parAtmCalib  :=RegisterStdInCmd('@PARAM.ATMCALIB','');
  cmd_parAddress   :=RegisterStdInCmd('@PARAM.ADDRESS','');
  cmd_parBaudrate  :=RegisterStdInCmd('@PARAM.BAUDRATE','');
  cmd_Edit         :=RegisterStdInCmd('@Edit','');
  cmd_menuBaudrate :=RegisterStdInCmd('@MenuBaudrateStarter','');
  cmd_DimTagUpdate :=RegisterStdInCmd('@DimTagUpdate','');
  cmd_AssignTag    :=RegisterStdInCmd('@AssignTag',   '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if not DIM_IsClientMode then
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  FreeNetLibrary;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  GUIHandler;
  if mSecNow-FixmSecNow>VTRD.Modbus.DelayOnStart then
  if IsPortOpened then VTRD_CMD_POLL;
  if VTRD.Simulator then VTRD_Simulation;
  DimUpdateState;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,dat:String; cmdid,ref,cid,tim,port,uid,fid:Integer; v:Real;
  procedure Cleanup;
  begin
   cmd:=''; arg:=''; dat:='';
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Modbus.Reply ref cid tim port uid fid $$dat
   }
   if (cmdid = cmd_mbReply) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then VTRD_OnReply(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Poll ref cid tim port uid fid $$dat
   This message uses in simulation mode only
   }
   if (cmdid = cmd_mbPoll) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then VTRD_OnSimPoll(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    Data:='';
   end else
   {
   @Modbus.Refuse ...
   }
   if (cmdid = cmd_mbRefuse) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,0)))
    else Trouble(GotBug(cmd+' '+arg));
    UpdateDo(do_STATE_INDIC,time,st_REFUSE);
    bNul(iSetTag(VTRD.PARAM.STATE.tag,st_REFUSE));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Timeout ...
   }
   if (cmdid = cmd_mbTimeout) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Problem(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,64)))
    else Trouble(GotBug(cmd+' '+arg));
    UpdateDo(do_STATE_INDIC,time,st_TIMEOUT);
    bNul(iSetTag(VTRD.PARAM.STATE.tag,st_TIMEOUT));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @ZeroPortCounters 0
   }
   if (cmdid=cmd_mbZeroPortCnt) then begin
    if not DIM_IsClientMode then DevSendCmd(devModbusProxy,cmd+' '+arg);
    Data:='';
   end else
   {
   @ClearModbusSumm
   }
   if (cmdid = cmd_mbClrSumm) then begin
    if not DIM_IsClientMode then ClearModbusSumm;
    Data:='';
   end else
   {
   @PARAM.ZEROCALIB 1
   }
   if (cmdid = cmd_parZeroCalib) then begin
    if not DIM_IsClientMode then HoldCmdOpData(cm_WriteZeroCalib,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @PARAM.ATMCALIB 1
   }
   if (cmdid = cmd_parAtmCalib) then begin
    if not DIM_IsClientMode then HoldCmdOpData(cm_WriteAtmCalib,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @PARAM.ADDRESS 1
   }
   if (cmdid = cmd_parAddress) then begin
    if not DIM_IsClientMode then begin
     v:=rVal(ExtractWord(1,arg));
     HoldCmdOpData(cm_WriteAddress,v);
     if not IsNan(v) then begin
      bNul(iSetTag(VTRD.PARAM.ADDRESS.tag,Round(v)));
      DevPostCmdLocal('@SaveIni');
     end;
    end;
    Data:='';
   end else
   {
   @PARAM.BAUDRATE 1
   }
   if (cmdid = cmd_parBaudrate) then begin
    if not DIM_IsClientMode then begin
     v:=rVal(ExtractWord(1,arg));
     HoldCmdOpData(cm_WriteBaudrate,v);
     if not IsNan(v) then begin
      bNul(iSetTag(VTRD.PARAM.BAUDRATE.tag,Round(v)));
      DevPostCmdLocal('@SaveIni');
     end;
    end;
    Data:='';
   end else
   {
   @Edit ADDRESS 10
   @Edit BAUDRATE 2800
   }
   if (cmdid = cmd_Edit) then begin
    if IsSameText(ExtractWord(1,arg),'ADDRESS')  then StartEditTag(VTRD.PARAM.ADDRESS.tag, 'Введите адрес устройства (1-247)');
    if IsSameText(ExtractWord(1,arg),'BAUDRATE') then StartEditTag(VTRD.PARAM.BAUDRATE.tag,'Установка скорости передачи данных и контроля четности');
    Data:='';
   end else
   {
   @MenuBaudrateStarter
   }
   if (cmdid = cmd_menuBaudrate) then begin
    MenuBaudrateStarter;
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    VTRD_OnDimUpdateTag(arg);
    Data:='';
   end else
   {
   @AssignTag
   }
   if (cmdid=cmd_AssignTag) then begin
    VTRD_OnAssignTag(arg);
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
