 {
 ***********************************************************************
 BDGB14I Modbus Driver.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Modbus.XXX    - response come from &ModbusProxy server:
| @Modbus.Reply    ref cid tim port uid fid $$ans
| @Modbus.Refuse   ref cid tim port uid fid   msg
| @Modbus.Timeout  ref cid tim port uid fid $$req
|  ref  - sender device reference, expected &ModbusProxy.
|  cid  - command id, any user defined custom data.
|  tim  - request-response time, ms.
|  port - logical port on &ModbusProxy, expected 1..16.
|  uid  - unit id, i.e.MODBUS address, expected 1..247.
|  fid  - function id, expected 1,2,3,4,5,6,15,16.
|  ans  - answer  (response), in HEX_ENCODE format.
|  req  - request (original), in HEX_ENCODE format.
|  msg  - plain text message to explain reason of Refuse.
| @STATE.XX      - write state,     i.e. send to MODBUS
| @PARAM.XX      - write parameter, i.e. send to MODBUS
| @Edit PARAM.XX - edit parameter and write it
| @SetRegType n  - set regime - gas type (n) 0..6
| @SetRegCode n  - set regime - ADC range code (n) 0..2
|********************************************************
[]
 Configuration:
 ***********************************************************************
  [DeviceList]
  &BDGB14I.DRIVER = device software program
  [&BDGB14I.DRIVER]
  Comment        = BDGB14I MODBUS DRIVER.
  InquiryPeriod  = 0
  DevicePolling  = 10, tpNormal
  ProgramSource  = ..\DaqPas\mayak_bdgb14i.pas
  DigitalFifo    = 1024
  AnalogFifo     = 1024
  DebugFlags     = 3
  OpenConsole    = 2
  Simulator      = 0
  ModbusPort     = 1
  ModbusUnitId   = 1
  ModbusTimeout  = 250
  ModbusDeadline = 60000
  ModbusPolling  = 100
  DelayOnStart   = 1000
  tagPrefix      = BDGB14I
  ...etc...
 ***********************************************************************
 }
program BDGB14I_Modbus_Driver;   { BDGB14I Modbus Driver            }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_NetLibrary}            { NetLibrary constants             }
 dfStatist       = 32;           { DebugFlags - polling statistics  }
 MaxCmdNum       = 16;           { Maximal number of commands:      }
 cm_ReadPARAMS   = 1;            { - Read Parameters   $00..$09 10  }
 cm_ReadMEASUR   = 2;            { - Read Measure data $30..$4b 28  }
 cm_WritPREHV    = 3;            { - Write Preset of HV             }
 cm_WritRECTL    = 4;            { - Write Register of Control      }
 cm_WritREREG    = 5;            { - Write Register of Regime       }
 ao_PARAM_CURIA  = 0;            { Measured Current                 }
 ao_PARAM_GASVA  = 1;            { Measured Gas Volume Activity     }
 ao_PARAM_CURTC  = 2;            { Current temperature              }
 ao_PARAM_CURHV  = 3;            { Current HV                       }
 ao_PARAM_CURAD  = 4;            { Current ADC                      }
 ao_PARAM_CURU1  = 5;            { Current U1                       }
 ao_PARAM_CURU2  = 6;            { Current U2                       }
 ao_PARAM_CURU0  = 7;            { Current U0                       }
 ao_PARAM_CURR2  = 8;            { Current R2                       }
 ao_PARAM_CURI0  = 9;            { Current I0                       }
 ao_PARAM_PREHV  = 10;           { Preset  HV                       }
 ao_PARAM_DEFHV  = 11;           { Default HV                       }
 ao_PARAM_ALRM1  = 12;           { Alarm 1 level                    }
 ao_PARAM_ALRM2  = 13;           { Alarm 2 level                    }
 do_STATE_STNET  = 0;            { Status of network                }
 do_STATE_STADC  = 1;            { Status of ADC                    }
 do_STATE_STREG  = 2;            { Status of Regime                 }
 do_STATE_STCON  = 3;            { Status of Converter              }
 do_STATE_RECTL  = 4;            { Register of Control              }
 do_STATE_REREG  = 5;            { Register of Regime               }
 do_STATE_FACTN  = 6;            { Factory number                   }
 do_STATE_FACTD  = 7;            { Factory date                     }
 do_STATE_NETAD  = 8;            { Network address                  }
 do_STATE_NETIF  = 9;            { Network interface mode           }
 do_STATE_SOFTV  = 10;           { Software version                 }
 do_STATE_SOFTD  = 11;           { Software date                    }
 do_STATE_DEVID  = 12;           { Device identifier                }
 do_STATE_RECON  = 13;           { Register of Converter            }
 do_STATE_INDIC  = 14;           { Status indicator                 }
 do_POLLRATERX   = 15;           { DigitalOutput - Poll rate Rx     }
 do_POLLRATETX   = 16;           { DigitalOutput - Poll rate Tx     }
 do_POLLRATEEX   = 17;           { DigitalOutput - Poll rate Ex     }
 do_ERRORCOUNT   = 18;           { DigitalOutput - Error counter    }
 do_POLLSUMMRX   = 19;           { DigitalOutput - Poll summ Rx     }
 do_POLLSUMMTX   = 20;           { DigitalOutput - Poll summ Tx     }
 do_POLLSUMMEX   = 21;           { DigitalOutput - Poll summ Ex     }
 st_NORMAL       = 0;            { Green  volume activity level     }
 st_WARNING      = 1;            { Yellow volume activity level     }
 st_ALARM        = 2;            { Red    volume activity level     }
 st_CALIB        = 3;            { Yellow test/calibration mode     }
 st_ERROR        = 4;            { Red    error                     }
 st_TIMEOUT      = 5;            { Red    timeout                   }
 st_REFUSE       = 6;            { Red    refuse                    }
 st_SIMULAT      = 7;            { Yelow  simulator mode            }
 st_DISABLE      = 8;            { Yellow polling disabled          }
 mb_NORMAL       = 0;            { Modbus state normal              }
 mb_TIMEOUT      = 1;            { Modbus state timeout             }
 mb_REFUSE       = 2;            { Modbus state refuse              }
 mb_DEADLINE     = 3;            { Modbus state deadline            }
 SwapModeFloat   = 1;            { Byte swap mode for float values  }
 SwapModeInt32   = 1;            { Byte swap mode for integers      }

type
 TTagRef = record tag,nai,ndi,nao,ndo:Integer; val,tim:Real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_NetLibrary}            { NetLibrary variables             }
 BDGB14I    : record             { BDGB14I Driver data record       }
  Simulator : Boolean;           { Simulator or Driver mode         }
  Modbus    : record             { Modbus data record               }
   Port     : Integer;           { Logical Port on &ModbusProxy     }
   UnitId   : Integer;           { Modbus unit id                   }
   Timeout  : Integer;           { Modbus timeout, ms               }
   Polling  : Integer;           { Modbus polling period, ms        }
   Deadline : Integer;           { Modbus deadline time, ms         }
   DelayOnStart : Integer;       { Command cycle delay on start     }
   State    : Integer;           { State of Connection              }
   Cmd      : record             { Command cycle data record        }
    Num     : Integer;           { Current running command number   }
    Enabled : array [1..MaxCmdNum] of Boolean; { Enable polling     }
    FuncId  : array [1..MaxCmdNum] of Integer; { Modbus Function Id }
    SAddr   : array [1..MaxCmdNum] of Integer; { Start Address      }
    Quant   : array [1..MaxCmdNum] of Integer; { Quantity of data   }
    OpData  : array [1..MaxCmdNum] of Real;    { Operation data     }
    OpBuff  : array [1..MaxCmdNum] of Real;    { Operation buffer   }
   end;                          {                                  }
   Poll     : record             { Polling transaction data record  }
    ref     : Integer;           { Last sent device reference       }
    cid     : Integer;           { Last sent command id             }
    tim     : Real;              { Last polling time, ms            }
    port    : Integer;           { Last polling port                }
    uid     : Integer;           { Last polling unit id             }
    fid     : Integer;           { Last sent function id            }
    saddr   : Integer;           { Last sent start address          }
    quant   : Integer;           { Last sent quantity of registers  }
    dat     : String;            { Last sent PDU=(fid+dat) data     }
    Summ,                        { Poll summ counters               }
    Rate    : record             { Poll rate on last second         }
     Rx     : Real;              { Rx poll summ/rate - receiver     }
     Tx     : Real;              { Tx poll summ/rate - transmitter  }
     Ex     : Real;              { Ex poll summ/rate - errors       }
    end;                         {                                  }
   end;                          {                                  }
  end;                           {                                  }
  POLL      : record             { Poll parameters                  }
   ENABLE   : TTagRef;           { Poll enable flag                 }
  end;                           {                                  }
  EDIT      : record             { Edit parameters                  }
   ENABLE   : TTagRef;           { Edit enable flag                 }
  end;                           {                                  }
  AUTO      : record             { Auto                             }
   CALIBR   : TTagRef;           { Calibration flag                 }
  end;                           {                                  }
  NET       : record             { Network                          }
   RESTART  : TTagRef;           { Restart flag                     }
  end;                           {                                  }
  STATE     : record             { Device state parameters          }
   FACTN    : TTagRef;           { Factory number                   }
   FACTD    : TTagRef;           { Factory date                     }
   NETAD    : TTagRef;           { Network address preset           }
   NETIF    : TTagRef;           { Network mode preset              }
   SOFTV    : TTagRef;           { Software version                 }
   SOFTD    : TTagRef;           { Software date                    }
   DEVID    : TTagRef;           { Device identifier                }
   RECON    : TTagRef;           { Register of Regime               }
   STNET    : TTagRef;           { Status of Network                }
   STADC    : TTagRef;           { Status of ADC                    }
   STREG    : TTagRef;           { Status of Regime                 }
   STCON    : TTagRef;           { Status of Converter              }
   RECTL    : TTagRef;           { Register of Control              }
   REREG    : TTagRef;           { Register of Regime               }
   INDIC    : TTagRef;           { Register of Indicator            }
   FACTS    : TTagRef;           { Factory  date string             }
   SOFTS    : TTagRef;           { Software date string             }
  end;                           {                                  }
  PARAM     : record             { Device analog parameters         }
   CURIA    : TTagRef;           { Measured current                 }
   GASVA    : TTagRef;           { Current gas volume activity      }
   CURTC    : TTagRef;           { Current temperature              }
   CURHV    : TTagRef;           { Current HV                       }
   CURAD    : TTagRef;           { Current ADC                      }
   CURU1    : TTagRef;           { Current U1                       }
   CURU2    : TTagRef;           { Current U2                       }
   CURU0    : TTagRef;           { Current U0                       }
   CURR2    : TTagRef;           { Current R2                       }
   CURI0    : TTagRef;           { Current I0                       }
   PREHV    : TTagRef;           { Preset  HV                       }
   DEFHV    : TTagRef;           { Default HV                       }
   ALRM1    : TTagRef;           { Alarm 1 level                    }
   ALRM2    : TTagRef;           { Alarm 2 level                    }
   IDENT    : TTagRef;           { Device Identifier                }
  end;                           {                                  }
  SIM       : record             { Simulator data                   }
   FACTN    : Integer;           {                                  }
   FACTD    : Integer;           {                                  }
   NETAD    : Integer;           {                                  }
   NETIF    : Integer;           {                                  }
   SOFTV    : Integer;           {                                  }
   SOFTD    : Integer;           {                                  }
   DEVID    : Integer;           {                                  }
   RECON    : Integer;           {                                  }
   PREHV    : Real;              {                                  }
   STNET    : Integer;           {                                  }
   STADC    : Integer;           {                                  }
   STREG    : Integer;           {                                  }
   STCON    : Integer;           {                                  }
   CURIA    : Real;              {                                  }
   GASVA    : Real;              {                                  }
   CURTC    : Real;              {                                  }
   CURHV    : Real;              {                                  }
   CURAD    : Real;              {                                  }
   CURU1    : Real;              {                                  }
   CURU2    : Real;              {                                  }
   CURU0    : Real;              {                                  }
   CURR2    : Real;              {                                  }
   CURI0    : Real;              {                                  }
   DEFHV    : Real;              {                                  }
   RECTL    : Integer;           {                                  }
   REREG    : Integer;           {                                  }
  end;                           {                                  }
  winControl: String;            { Control window                   }
 end;                            {                                  }
 cmd_ClearModbusSumm       : Integer; { @ClearModbusSumm            }
 cmd_PARAM_PREHV           : Integer; { @PARAM.PREHV                }
 cmd_PARAM_ALRM1           : Integer; { @PARAM.ALRM1                }
 cmd_PARAM_ALRM2           : Integer; { @PARAM.ALRM2                }
 cmd_STATE_RECTL           : Integer; { @STATE.RECTL                }
 cmd_STATE_REREG           : Integer; { @STATE.REREG                }
 cmd_Edit                  : Integer; { @Edit                       }
 cmd_SetRegType            : Integer; { @SetRegType                 }
 cmd_SetRegCode            : Integer; { @SetRegCode                 }
 cmd_mayak_bdgb14i_htm     : Integer; { @mayak_bdgb14i.htm          }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_NetLibrary}            { NetLibrary functions             }

 //
 // Command cycle routines.
 //
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsUsableCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsUsableCmdNum:=(BDGB14I.Modbus.Cmd.FuncId[Num]<>0)
  else IsUsableCmdNum:=False;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsEnabledCmdNum:=(BDGB14I.Modbus.Cmd.FuncId[Num]<>0) and BDGB14I.Modbus.Cmd.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer;
 begin
  i:=0;
  while (i<MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   if IsEnabledCmdNum(Num)
   then i:=MaxCmdNum
   else i:=i+1;
  end;
  NextEnabledCmdNum:=Num;
 end;
 procedure EnableCmdNum(Num:Integer; Enabled:Boolean);
 begin
  if IsUsableCmdNum(Num) then BDGB14I.Modbus.Cmd.Enabled[Num]:=Enabled;
 end;
 procedure HoldCmdOpData(Num:Integer; OpData:Real);
 begin
  if IsUsableCmdNum(Num) then begin
   BDGB14I.Modbus.Cmd.Enabled[Num]:=not IsNaN(OpData);
   BDGB14I.Modbus.Cmd.OpBuff[Num]:=OpData;
  end;
 end;
 procedure ApplyCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then
  if not IsNaN(BDGB14I.Modbus.Cmd.OpBuff[Num]) then begin
   BDGB14I.Modbus.Cmd.OpData[Num]:=BDGB14I.Modbus.Cmd.OpBuff[Num];
   BDGB14I.Modbus.Cmd.OpBuff[Num]:=_NaN;
  end;
 end;
 procedure ReleaseCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then HoldCmdOpData(Num,BDGB14I.Modbus.Cmd.OpBuff[Num]);
 end;
 function GetCmdOpData(Num:Integer):Real;
 begin
  if IsUsableCmdNum(Num)
  then GetCmdOpData:=BDGB14I.Modbus.Cmd.OpData[Num]
  else GetCmdOpData:=_NaN;
 end;
 procedure InitCmdItem(Num:Integer; Enabled:Boolean; FuncId,SAddr,Quant:Integer; OpData,OpBuff:Real);
 begin
  if IsValidCmdNum(Num) then begin
   BDGB14I.Modbus.Cmd.Enabled[Num]:=Enabled;
   BDGB14I.Modbus.Cmd.FuncId[Num]:=FuncId;
   BDGB14I.Modbus.Cmd.SAddr[Num]:=SAddr;
   BDGB14I.Modbus.Cmd.Quant[Num]:=Quant;
   BDGB14I.Modbus.Cmd.OpData[Num]:=OpData;
   BDGB14I.Modbus.Cmd.OpBuff[Num]:=OpBuff;
  end;
 end;
 //
 // Command table cleanup and initialization.
 //
 procedure ClearCmdTable;
 var Num:Integer;
 begin
  BDGB14I.Modbus.Cmd.Num:=0;
  for Num:=1 to MaxCmdNum do InitCmdItem(Num,False,0,0,0,0,_NaN);
 end;
 procedure InitCmdTable;
 begin
  ClearCmdTable;
  // Table of CommandId      Enabled FunctionId        SAddr       Quant OpData OpBuff
  InitCmdItem(cm_ReadPARAMS, True,   modbus_fn_ReadHR, Val('$00'), 10,   0,     _NaN);
  InitCmdItem(cm_ReadMEASUR, True,   modbus_fn_ReadHR, Val('$30'), 28,   0,     _NaN);
  InitCmdItem(cm_WritPREHV,  False,  modbus_fn_WritMR, Val('$48'), 2,    0,     _NaN);
  InitCmdItem(cm_WritRECTL,  False,  modbus_fn_WritMR, Val('$4a'), 1,    0,     _NaN);
  InitCmdItem(cm_WritREREG,  False,  modbus_fn_WritMR, Val('$4b'), 1,    0,     _NaN);
 end;
 //
 // Assign modbus last sent polling request record.
 //
 procedure AssignModbusPoll(ref,cid:Integer; tim:Real; port,uid,fid,saddr,quant:Integer; dat:String);
 begin
  BDGB14I.Modbus.Poll.ref:=ref;     BDGB14I.Modbus.Poll.cid:=cid;      BDGB14I.Modbus.Poll.tim:=tim;
  BDGB14I.Modbus.Poll.port:=port;   BDGB14I.Modbus.Poll.uid:=uid;      BDGB14I.Modbus.Poll.fid:=fid;
  BDGB14I.Modbus.Poll.saddr:=saddr; BDGB14I.Modbus.Poll.quant:=quant;  BDGB14I.Modbus.Poll.dat:=dat;
 end;
 //
 // Clear modbus polling request to be ready for next polling.
 //
 procedure ClearModbusPoll;
 begin
  BDGB14I.Modbus.Poll.cid:=0;
  BDGB14I.Modbus.Poll.dat:='';
 end;
 //
 // Clear modbus poll summ counters.
 //
 procedure ClearModbusSumm;
 begin
  BDGB14I.Modbus.Poll.Summ.Rx:=0;
  BDGB14I.Modbus.Poll.Summ.Tx:=0;
  BDGB14I.Modbus.Poll.Summ.Ex:=0;
 end;
 //
 // Clear modbus poll rate counters.
 //
 procedure ClearModbusRate;
 begin
  BDGB14I.Modbus.Poll.Rate.Rx:=0;
  BDGB14I.Modbus.Poll.Rate.Tx:=0;
  BDGB14I.Modbus.Poll.Rate.Ex:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Real);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 //
 // Format error message with with error counter increment.
 // Usage is like: Trouble(GotBug(...)); Problem(GotBug(...));
 //
 function GotBug(msg:String):String;
 begin
  IncSummRate(BDGB14I.Modbus.Poll.Summ.Ex,BDGB14I.Modbus.Poll.Rate.Ex);
  GotBug:=msg;
 end;
 //
 // Convert date (days since 12/30/1899 12:00 am) to msec since Xmas.
 // 
 function DateToMsec(date:Real):Real;
 begin
  DateToMsec:=datetime2ms(1899,12,30,12,0,0,0)+date*86400000;
 end;
 //
 // Return string like 2016.08.26 by date (days since 12/30/1899 12:00 am).
 //
 function DateToStr(date:Real):String;
 begin
  DateToStr:=Copy(GetDateTime(DateToMsec(date)),1,10);
 end;
 //
 // Find the current Proxy device to send @Modbus.Poll message.
 //
 function devTheProxy:Integer;
 begin
  if BDGB14I.Simulator              // In simulation mode
  then devTheProxy:=devMySelf       // send messages to himself
  else devTheProxy:=devModbusProxy; // otherwise use &ModbusProxy
 end;
 //
 // Check if communication port is opened or not.
 // As long as we use Proxy device, check Port number & device reference.
 //
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(BDGB14I.Modbus.Port>0) and (devTheProxy<>0);
 end;
 //
 // Calculate Modbus request PDU by command id.
 // cid   - (in)  command id number.
 // fid   - (out) Modbus function id.
 // saddr - (out) zero-based start address of registers to read/write.
 // quant - (out) data quantity, i.e. number of r/w registers or single register r/w value.
 // dat   - (out) Modbus PDU data unit, PDU=(fid+dat).
 //
 function BDGB14I_CalcPDU(cid:Integer; var fid,saddr,quant:Integer; var dat:String):Boolean;
 begin
  fid:=0; saddr:=0; quant:=0; dat:='';
  if IsEnabledCmdNum(cid) then begin
   fid:=BDGB14I.Modbus.Cmd.FuncId[cid];
   saddr:=BDGB14I.Modbus.Cmd.SAddr[cid];
   quant:=BDGB14I.Modbus.Cmd.Quant[cid];
   if (cid=cm_ReadPARAMS) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_ReadMEASUR) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_WritPREHV) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_float(GetCmdOpData(cid),SwapModeFloat));
   end else
   if (cid=cm_WritRECTL) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(Round(GetCmdOpData(cid)),SwapModeInt32));
   end else
   if (cid=cm_WritREREG) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(Round(GetCmdOpData(cid)),SwapModeInt32));
   end else
   Trouble(GotBug('Invalid command id '+Str(cid)));
  end;
  BDGB14I_CalcPDU:=modbus_func_ok(fid) and (Length(dat)>0);
 end;
 //
 // Main Modbus command handler to process Modbus device reply.
 // cid   - (in)  command id number.
 // fid   - (in)  Modbus function id.
 // saddr - (in)  zero-based start address of registers to read/write.
 // quant - (in)  data quantity, i.e. number of r/w registers or single register r/w value.
 // raw   - (in)  raw coils/registers data array.
 //
 procedure BDGB14I_OnCommand(cid,fid,saddr,quant:Integer; raw:String);
 var addr,offs,stat:Integer; r:Real;
 begin
  if IsValidCmdNum(cid) then begin 
   if (cid=cm_ReadPARAMS) then begin // Read 10 holding registers starting from $00
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     // Addr[$00]:int16 = factory number
     if (addr=0) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.FACTN.tag,stat));
      UpdateDo(do_STATE_FACTN,time,stat);
     end else
     // Addr[$01]:int16 = factory date
     if (addr=1) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.FACTD.tag,stat));
      UpdateDo(do_STATE_FACTD,time,stat);
     end else
     // Addr[$02]:int16 = network address
     if (addr=2) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.NETAD.tag,stat));
      UpdateDo(do_STATE_NETAD,time,stat);
     end else
     // Addr[$03]:int16 = network interface mode
     if (addr=3) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.NETIF.tag,stat));
      UpdateDo(do_STATE_NETIF,time,stat);
     end else
     // Addr[$04]:int16 = software version
     if (addr=4) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.SOFTV.tag,stat));
      UpdateDo(do_STATE_SOFTV,time,stat);
     end else
     // Addr[$05]:int16 = software date
     if (addr=5) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.SOFTD.tag,stat));
      UpdateDo(do_STATE_SOFTD,time,stat);
     end else
     // Addr[$06]:int16 = device identifier
     if (addr=6) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.DEVID.tag,stat));
      UpdateDo(do_STATE_DEVID,time,stat);
     end else
     // Addr[$07]:int16 = register of Converter
     if (addr=7) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.RECON.tag,stat));
      UpdateDo(do_STATE_RECON,time,stat);
     end else
     // Addr[$08]:float = default HV
     if (addr=8) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.DEFHV.tag,r));
      UpdateAo(ao_PARAM_DEFHV,time,r);
     end;
    end;
   end else
   if (cid=cm_ReadMEASUR) then begin // Read 28 holding registers starting from $30
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     // Addr[$30]:int16 = status of network
     if (addr=48) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.STNET.tag,stat));
      UpdateDo(do_STATE_STNET,time,stat);
     end else
     // Addr[$31]:int16 = status of ADC
     if (addr=49) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.STADC.tag,stat));
      UpdateDo(do_STATE_STADC,time,stat);
     end else
     // Addr[$32]:int16 = status of Regime
     if (addr=50) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.STREG.tag,stat));
      UpdateDo(do_STATE_STREG,time,stat);
     end else
     // Addr[$33]:int16 = status of Converter
     if (addr=51) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.STCON.tag,stat));
      UpdateDo(do_STATE_STCON,time,stat);
     end else
     // Addr[$34]:float = measured current
     if (addr=52) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURIA.tag,r));
      UpdateAo(ao_PARAM_CURIA,time,r);
     end else
     // Addr[$36]:float = gas volume activity
     if (addr=54) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.GASVA.tag,r));
      UpdateAo(ao_PARAM_GASVA,time,r);
     end else
     // Addr[$38]:float = current temperature
     if (addr=56) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURTC.tag,r));
      UpdateAo(ao_PARAM_CURTC,time,r);
     end else
     // Addr[$3a]:float = current HV
     if (addr=58) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURHV.tag,r));
      UpdateAo(ao_PARAM_CURHV,time,r);
     end else
     // Addr[$3c]:float = current ADC
     if (addr=60) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURAD.tag,r));
      UpdateAo(ao_PARAM_CURAD,time,r);
     end else
     // Addr[$3e]:float = current U1
     if (addr=62) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURU1.tag,r));
      UpdateAo(ao_PARAM_CURU1,time,r);
     end else
     // Addr[$40]:float = current U2
     if (addr=64) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURU2.tag,r));
      UpdateAo(ao_PARAM_CURU2,time,r);
     end else
     // Addr[$42]:float = current U0
     if (addr=66) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURU0.tag,r));
      UpdateAo(ao_PARAM_CURU0,time,r);
     end else
     // Addr[$44]:float = current R2
     if (addr=68) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURR2.tag,r));
      UpdateAo(ao_PARAM_CURR2,time,r);
     end else
     // Addr[$46]:float = current I0
     if (addr=70) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.CURI0.tag,r));
      UpdateAo(ao_PARAM_CURI0,time,r);
     end else
     // Addr[$48]:float = preset HV
     if (addr=72) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(BDGB14I.PARAM.PREHV.tag,r));
      UpdateAo(ao_PARAM_PREHV,time,r);
     end else
     // Addr[$4a]:int16 = register of Control
     if (addr=74) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.RECTL.tag,stat));
      UpdateDo(do_STATE_RECTL,time,stat);
     end else
     // Addr[$4b]:int16 = register of Regime
     if (addr=75) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(BDGB14I.STATE.REREG.tag,stat));
      UpdateDo(do_STATE_REREG,time,stat);
     end;
    end;
   end else
   // Write register/coils: release confirmed command
   // to avoid rewrite the data which already written
   if (cid=cm_WritPREHV) then ReleaseCmdOpData(cid) else
   if (cid=cm_WritRECTL) then ReleaseCmdOpData(cid) else
   if (cid=cm_WritREREG) then ReleaseCmdOpData(cid) else
   Trouble(GotBug('Unexpected command '+Str(cid)));
  end;
 end;
 //
 // Data handler on @Modbus.Reply event. Process reply comes from Modbus device.
 //
 procedure BDGB14I_OnReply(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>BDGB14I.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>BDGB14I.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>BDGB14I.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>BDGB14I.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>BDGB14I.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=BDGB14I.Modbus.Poll.saddr; quant:=BDGB14I.Modbus.Poll.quant;
   if modbus_decode_pdu('A',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>BDGB14I.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>BDGB14I.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else
    if modbus_is_except(fid) then begin
     Problem(GotBug('MODBUS EXCEPTION '+Str(modbus_ext_byte(raw,0))+' ON COMMAND '+Str(cid)));
    end else begin
     IncSummRate(BDGB14I.Modbus.Poll.Summ.Rx,BDGB14I.Modbus.Poll.Rate.Rx);
     BDGB14I_OnCommand(cid,fid,saddr,quant,raw);
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // Data handler on @Modbus.Poll event.
 // This procedure calls in simulation mode only.
 //
 procedure BDGB14I_OnSimPoll(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant,offs,addr,state,resrc:Integer; r:Real;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>BDGB14I.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>BDGB14I.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>BDGB14I.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>BDGB14I.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>BDGB14I.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=BDGB14I.Modbus.Poll.saddr; quant:=BDGB14I.Modbus.Poll.quant;
   if modbus_decode_pdu('R',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>BDGB14I.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>BDGB14I.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else begin
     if DebugFlagEnabled(dfDetails) then Details('Simulate polling '+Str(cid));
     if (cid=cm_ReadPARAMS) then begin // 10 registers starting from $00
      raw:=modbus_dump_int16(BDGB14I.SIM.FACTN,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.FACTD,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.NETAD,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.NETIF,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.SOFTV,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.SOFTD,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.DEVID,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.RECON,SwapModeInt32)
          +modbus_dump_float(BDGB14I.SIM.DEFHV,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_ReadMEASUR) then begin // 28 registers starting from $30
      raw:=modbus_dump_int16(BDGB14I.SIM.STNET,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.STADC,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.STREG,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.STCON,SwapModeInt32)
          +modbus_dump_float(BDGB14I.SIM.CURIA,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.GASVA,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURTC,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURHV,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURAD,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURU1,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURU2,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURU0,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURR2,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.CURI0,SwapModeFloat)
          +modbus_dump_float(BDGB14I.SIM.PREHV,SwapModeFloat)
          +modbus_dump_int16(BDGB14I.SIM.RECTL,SwapModeInt32)
          +modbus_dump_int16(BDGB14I.SIM.REREG,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritPREHV) then begin
      BDGB14I.SIM.PREHV:=modbus_ext_float(raw,0,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritRECTL) then begin
      BDGB14I.SIM.RECTL:=modbus_ext_int16(raw,0,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritREREG) then begin
      BDGB14I.SIM.REREG:=modbus_ext_int16(raw,0,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,
                modbus_as_except(fid,true),Dump(Chr(modbus_er_ILLFUN))));
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // Update status indicator.
 //
 procedure BDGB14I_UpdateIndicator;
 var indic,stnet,stadc,streg,stcon,rectl,rereg,recon,stcod,rtype,rmode,factd,softd:Integer;
  gasva,alrm1,alrm2:Real;
  function InRange(x,a,b:Real):Boolean;
  begin
   InRange:=((a<=x) and (x<=b));
  end;
 begin
  stnet:=iGetTag(BDGB14I.STATE.STNET.tag);
  stadc:=iGetTag(BDGB14I.STATE.STADC.tag);
  streg:=iGetTag(BDGB14I.STATE.STREG.tag);
  stcon:=iGetTag(BDGB14I.STATE.STCON.tag);
  rectl:=iGetTag(BDGB14I.STATE.RECTL.tag);
  rereg:=iGetTag(BDGB14I.STATE.REREG.tag);
  recon:=iGetTag(BDGB14I.STATE.RECON.tag);
  gasva:=rGetTag(BDGB14I.PARAM.GASVA.tag);
  alrm1:=rGetTag(BDGB14I.PARAM.ALRM1.tag);
  alrm2:=rGetTag(BDGB14I.PARAM.ALRM2.tag);
  factd:=iGetTag(BDGB14I.STATE.FACTD.tag);
  softd:=iGetTag(BDGB14I.STATE.SOFTD.tag);
  stcod:=iAnd(stcon div 256,15);
  rtype:=iAnd(streg div 4096,15);
  rmode:=iAnd(streg div 256,3);
  if iGetTag(BDGB14I.POLL.ENABLE.tag)=0        then indic:=st_DISABLE else // Polling is OFF
  if BDGB14I.Simulator                         then indic:=st_SIMULAT else // Simulation mode ON
  if BDGB14I.Modbus.State=mb_DEADLINE          then indic:=st_TIMEOUT else // Deadline detected
  if BDGB14I.Modbus.State=mb_TIMEOUT           then indic:=st_TIMEOUT else // Timeout detected
  if BDGB14I.Modbus.State=st_REFUSE            then indic:=st_REFUSE  else // Connection refused
  if IsBit(stadc,8)                            then indic:=st_ERROR   else // ADC faulure
  if InRange(iAnd(stnet,255),255,255)          then indic:=st_ERROR   else // Network address controller failure
  if not InRange(rtype,0,6)                    then indic:=st_ERROR   else // Regime (gas type) invalid
  if not InRange(rmode,0,2)                    then indic:=st_ERROR   else // Regime (ADC range) invalid
  if not InRange(stcod,0,9)                    then indic:=st_ERROR   else // Converter state code invalid
  if IsNan(gasva+alrm1+alrm2)                  then indic:=st_ERROR   else // Volume activity invalid
  if IsInf(gasva+alrm1+alrm2)                  then indic:=st_ERROR   else // Volume activity invalid
  if not InRange(stcod,7,8)                    then indic:=st_CALIB   else // Converter runs calibration
  if (gasva>alrm2)                             then indic:=st_ALARM   else // Volume activity alarm 2 RED
  if (gasva>alrm1)                             then indic:=st_WARNING else // Volume activity alarm 1 YELLOW
  indic:=st_NORMAL;
  bNul(iSetTag(BDGB14I.STATE.INDIC.tag,indic));
  UpdateDo(do_STATE_INDIC,time,indic);
  UpdateAo(ao_PARAM_ALRM1,time,alrm1);
  UpdateAo(ao_PARAM_ALRM2,time,alrm2);
  bNul(sSetTag(BDGB14I.STATE.FACTS.tag,DateToStr(factd-1)));
  bNul(sSetTag(BDGB14I.STATE.SOFTS.tag,DateToStr(softd)));
  if ShouldRefresh(BDGB14I.PARAM.ALRM1.val,rGetTag(BDGB14I.PARAM.ALRM1.tag))
  +  ShouldRefresh(BDGB14I.PARAM.ALRM2.val,rGetTag(BDGB14I.PARAM.ALRM2.tag))>0 then begin
   if not IsEmptyStr(BDGB14I.winControl) then
   bNul(WinDraw(BDGB14I.winControl+'|Eval=(alarm1='+Str(BDGB14I.PARAM.ALRM1.val)
       +')+(alarm2='+Str(BDGB14I.PARAM.ALRM2.val)+')'));
  end;
 end;
 //
 // BDGB14I Driver Simulator mode polling.
 // This procedure calls in simulation mode only.
 //
 procedure BDGB14I_SIM_POLL;
 begin
  if BDGB14I.Simulator then begin
   BDGB14I.SIM.GASVA:=3.2949E+8*(1+0.1*sin(2*pi*time*60)+0.01*random(-1,1))*BDGB14I.SIM.PREHV/500;
   BDGB14I.SIM.CURIA:=BDGB14I.SIM.GASVA*1E-9;
   BDGB14I.SIM.CURAD:=BDGB14I.SIM.GASVA*0.5;
   BDGB14I.SIM.CURU1:=BDGB14I.SIM.GASVA*0.25;
   BDGB14I.SIM.CURU2:=BDGB14I.SIM.GASVA*0.125;
   BDGB14I.SIM.CURU0:=BDGB14I.SIM.GASVA*0.0625;
   BDGB14I.SIM.CURR2:=1.0446E+10*(1+0.001*random(-1,1));
   BDGB14I.SIM.CURI0:=BDGB14I.SIM.GASVA*1E-8;
   BDGB14I.SIM.CURHV:=BDGB14I.SIM.PREHV*(1+0.01*random(-1,1));
   BDGB14I.SIM.CURTC:=32.4*(1+0.01*random(-1,1));
  end;
 end;
 //
 // Get/set Regime Gas Type and Code of Diapason
 //
 function GetRegType:Integer;
 begin // Status Regime bit 12..14 - Gas Type
  GetRegType:=(iGetTag(BDGB14I.STATE.STREG.tag) div 4096) mod 8;
 end;
 function GetRegCode:Integer;
 begin // Status Regime bit 8..9 - Code of Diapason
  GetRegCode:=(iGetTag(BDGB14I.STATE.STREG.tag) div 256) mod 4;
 end;
 procedure SetRegType(gas:Integer);
 var rereg:Integer;
 begin
  if (gas>=0) and (gas<=6) then begin
   rereg:=gas*4096+GetRegCode*256;
   DevSendCmdLocal('@STATE.REREG '+Str(rereg));
  end;
 end;
 procedure SetRegCode(cod:Integer);
 var rereg:Integer;
 begin
  if (cod>=0) and (cod<=2) then begin
   rereg:=GetRegType*4096+cod*256;
   DevSendCmdLocal('@STATE.REREG '+Str(rereg));
  end;
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс счетчика ошибок драйвера '+ModbusProxy);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DevPost '+ModbusProxy+' @ZeroPortCounters 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс счетчика ошибок '+DevName);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@ClearModbusSumm');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Перезапустить драйвер '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:14\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 150 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  103 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 {
 Menu RegType Starter to start editing.
 }
 procedure MenuRegTypeStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('РЕЖИМ '+sGetTag(BDGB14I.PARAM.IDENT.tag)+'... ');
   n:=n+EditAddInputLn('Задайте используемый ТИП ГАЗА:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Тритий    T');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Углерод   C');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Ксенон   Xe');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 2');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Криптон  Kr');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Аргон    Ar');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 4');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Смесь газов');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 5');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Гамма излучение');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegType 6');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:14\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 450 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  107 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_REGTYPE'),Str(GetRegType));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu RegType Handler to handle editing.
 }
 procedure MenuRegTypeHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_REGTYPE'));
 end;
 {
 Menu RegCode Starter to start editing.
 }
 procedure MenuRegCodeStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('РЕЖИМ '+sGetTag(BDGB14I.PARAM.IDENT.tag)+'... ');
   n:=n+EditAddInputLn('Задайте ТИП ДИАПАЗОНА:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Автоматический Диапазон');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegCode 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Грубый         Диапазон');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegCode 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Чувствительный Диапазон');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@SetRegCode 2');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:14\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 520 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  107 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_REGCODE'),Str(GetRegCode));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu RegCode Handler to handle editing.
 }
 procedure MenuRegCodeHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_REGCODE'));
 end;
 //
 // BDGB14I Driver GUI polling.
 //
 procedure BDGB14I_GUI_POLL;
 var s:String; ClickCurve:Integer;
  //
  // Send to device (dev) command (cmd data) to set new tag value if one in range (min,max).
  //
  procedure DevSendCmdTag(dev,tag:Integer; cmd,data:String; min,max:Real);
  var v:Real;
  begin
   if IsRefTag(tag) then
   if IsRefDevice(dev) then
   if not IsEmptyStr(cmd) then
   if not IsEmptyStr(data) then
   if TypeTag(tag)<=2 then begin
    v:=rVal(data);
    if IsNaN(v) then Trouble(GotBug('Invalid tag edit')) else
    if (v<min) or (v>max) then Trouble(GotBug('Tag edit out of range')) else
    if TypeTag(tag)=1 then DevSendCmdLocal(cmd+' '+Str(Round(v))) else
    if TypeTag(tag)=2 then DevSendCmdLocal(cmd+' '+Trim(data));
   end else
   if TypeTag(tag)=3 then begin
    DevSendCmdLocal(cmd+' '+Trim(data));
   end;
  end;
  //
  // Send command (cmd) to device (dev) on tag click.
  //
  procedure ClickTagDevSendCmd(tag,dev:Integer; cmd:String);
  begin
   if Length(cmd)>0 then
   if IsRefTag(tag) then
   if IsRefDevice(dev) then
   if ClickTag=tag then begin
    bNul(Voice(snd_Click));
    DevSendCmd(dev,cmd);
   end;
  end;
  //
  // Xor bit on click (local version).
  //
  procedure ClickTagXorLocal(tag,XorMask:Integer);
  var nv:Integer;
  begin
   if (ClickTag=tag) then begin
    bNul(iSetTagXor(tag,XorMask));
    bNul(Voice(snd_Click));
   end;
  end;
  //
  // Check if clicked tag can be edit
  //
  function CanEdit(tip:Boolean):Boolean;
  var can:Boolean;
  begin
   can:=true;
   if (ClickTag=BDGB14I.PARAM.PREHV.tag)
   or (ClickTag=BDGB14I.STATE.STREG.tag)
   or (ClickTag=BDGB14I.PARAM.ALRM1.tag)
   or (ClickTag=BDGB14I.PARAM.ALRM2.tag)
   or (ClickTag=BDGB14I.AUTO.CALIBR.tag)
   or (ClickTag=BDGB14I.NET.RESTART.tag)
   then can:=(iGetTag(BDGB14I.EDIT.ENABLE.tag)<>0);
   if tip and not can then 
   ShowTooltip('guid MAYAK.NoEdit '
              +'text "Crw32.exe: Редактирование запрещено. Измените «Режим полей ввода»." '
              +'preset stdWarning delay 15000');
   CanEdit:=can;
  end;
 begin
  s:='';
  //
  // Update status indicator...
  //
  BDGB14I_UpdateIndicator;
  //
  // Command execution...
  //
  if (iGetTag(BDGB14I.AUTO.CALIBR.tag)<>0) then begin
   bNul(iSetTag(BDGB14I.AUTO.CALIBR.tag,0));
   DevSendCmdLocal('@STATE.RECTL 256');
  end;
  if (iGetTag(BDGB14I.NET.RESTART.tag)<>0) then begin
   bNul(iSetTag(BDGB14I.NET.RESTART.tag,0));
   DevSendCmdLocal('@STATE.RECTL 512');
  end;
  //
  // Handle sensor clicks...
  //
  if ClickButton=1 then begin
   //
   // Buttons.
   //
   ClickTagXorLocal(BDGB14I.POLL.ENABLE.tag,1);
   ClickTagXorLocal(BDGB14I.EDIT.ENABLE.tag,1);
   if CanEdit(true) then begin
    ClickTagXorLocal(BDGB14I.AUTO.CALIBR.tag,1);
    ClickTagXorLocal(BDGB14I.NET.RESTART.tag,1);
   end;
   if (ClickTag=BDGB14I.EDIT.ENABLE.tag) then begin
    if iGetTag(BDGB14I.EDIT.ENABLE.tag)<>0
    then ShowTooltip('delete MAYAK.NoEdit');
   end;
   //
   // Send command to edit on Click.
   //
   if CanEdit(true) then begin
    s:=mime_encode(SetFormUnderSensorLeftBottom(ClickParams('')));
    ClickTagDevSendCmd(BDGB14I.PARAM.PREHV.tag,devMySelf,'@Edit PARAM.PREHV '+s);
    ClickTagDevSendCmd(BDGB14I.PARAM.ALRM1.tag,devMySelf,'@Edit PARAM.ALRM1 '+s);
    ClickTagDevSendCmd(BDGB14I.PARAM.ALRM2.tag,devMySelf,'@Edit PARAM.ALRM2 '+s);
    if ClickTag=BDGB14I.STATE.STREG.tag then begin
     if IsSameText(ExtractFileExt(ClickSensor),'.TYPE') then begin
      bNul(Voice(snd_Click));
      MenuRegTypeStarter;
     end;
     if IsSameText(ExtractFileExt(ClickSensor),'.CODE') then begin
      bNul(Voice(snd_Click));
      MenuRegCodeStarter;
     end;
    end;
   end;
   //
   // Tools menu
   //
   if ClickTag=BDGB14I.PARAM.IDENT.tag then begin
    bNul(Voice(snd_Click));
    MenuToolsStarter;
   end;
   //
   // Plot & Tab windows
   //
   ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
   if IsRefCurve(ClickCurve) then begin
    iNul(WinSelectByCurve(ClickCurve,ClickCurve));
    bNul(Voice(snd_Wheel));
   end;
   //
   // User commands: @...
   //
   if LooksLikeCommand(ClickSensor) then begin
    DevSendCmdLocal(url_decode(ClickSensor));
    bNul(Voice(snd_Click));
   end;
  end;
  //
  // Edit tags...
  //
  if EditStateDone then begin
   //
   // If tag edit complete, send command to apply changes
   //
   if CheckEditTag(BDGB14I.PARAM.PREHV.tag,s)
   then DevSendCmdTag(devMySelf,BDGB14I.PARAM.PREHV.tag,'@PARAM.PREHV',s,0,_PlusInf);
   if CheckEditTag(BDGB14I.PARAM.ALRM1.tag,s)
   then DevSendCmdTag(devMySelf,BDGB14I.PARAM.ALRM1.tag,'@PARAM.ALRM1',s,0,_PlusInf);
   if CheckEditTag(BDGB14I.PARAM.ALRM2.tag,s)
   then DevSendCmdTag(devMySelf,BDGB14I.PARAM.ALRM2.tag,'@PARAM.ALRM2',s,0,_PlusInf);
   {
   Warning,Information.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
   {
   Menu TOOLS.
   }
   MenuToolsHandler;
   {
   Menu REGTYPE.
   }
   MenuRegTypeHandler;
   {
   Menu REGCODE.
   }
   MenuRegCodeHandler;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end;
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  s:='';
 end;
 //
 // BDGB14I Driver Command Cycle polling.
 //
 procedure BDGB14I_CMD_POLL;
  //
  // Send @Modbus.Poll ... command to &ModbusProxy and save sent request data in Modbus.Poll record.
  //
  procedure PollModbusProxy(cid:Integer);
  var dev,ref,tot,port,uid,fid,saddr,quant:Integer; dat:String;
  begin
   dat:='';
   ApplyCmdOpData(cid);
   if BDGB14I_CalcPDU(cid,fid,saddr,quant,dat) then begin
    dev:=devTheProxy; ref:=devMySelf;
    tot:=BDGB14I.Modbus.Timeout; port:=BDGB14I.Modbus.Port; uid:=BDGB14I.Modbus.UnitId;
    if DevSend(dev,modbus_proxy_poll('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat)+CRLF)>0 then begin
     if DebugFlagEnabled(dfViewExp) then 
     ViewExp('COM: '+modbus_proxy_nice('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat,32));
     AssignModbusPoll(dev,cid,mSecNow,port,uid,fid,saddr,quant,dat);
     IncSummRate(BDGB14I.Modbus.Poll.Summ.Tx,BDGB14I.Modbus.Poll.Rate.Tx);
    end else begin
     Trouble(GotBug('Fail to send command '+Str(cid)));
     ClearModbusPoll;
    end;
   end else begin
    Trouble(GotBug('Fail to calc command '+Str(cid)));
    ClearModbusPoll;
   end;
   dat:='';
  end;
 begin
  if IsPortOpened then begin
   if IsValidCmdNum(BDGB14I.Modbus.Poll.cid) then begin
    //
    // Request in progress, waiting @Modbus.Reply/@Modbus.Refuse/@Modbus.Timeout.
    // Handle Deadline if was no responce for too long time: repeat polling again.
    //
    if iGetTag(BDGB14I.POLL.ENABLE.tag)<>0 then
    if (mSecNow>BDGB14I.Modbus.Poll.tim+BDGB14I.Modbus.Deadline) then begin
     Trouble(GotBug('Deadline detected, repeat polling again...'));
     PollModbusProxy(BDGB14I.Modbus.Poll.cid);
     BDGB14I.Modbus.State:=mb_DEADLINE;
    end;
   end else begin
    //
    // If request is cleared, send new @Modbus.Poll request by timer.
    //
    if iGetTag(BDGB14I.POLL.ENABLE.tag)<>0 then
    if (mSecNow>=BDGB14I.Modbus.Poll.tim+BDGB14I.Modbus.Polling) then begin
     BDGB14I.Modbus.Cmd.Num:=NextEnabledCmdNum(BDGB14I.Modbus.Cmd.Num);
     if IsEnabledCmdNum(BDGB14I.Modbus.Cmd.Num)
     then PollModbusProxy(BDGB14I.Modbus.Cmd.Num);
    end;
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    UpdateDo(do_POLLRATERX,time,BDGB14I.Modbus.Poll.Rate.Rx);
    UpdateDo(do_POLLRATETX,time,BDGB14I.Modbus.Poll.Rate.Tx);
    UpdateDo(do_POLLRATEEX,time,BDGB14I.Modbus.Poll.Rate.Ex);
    UpdateDo(do_POLLSUMMRX,time,BDGB14I.Modbus.Poll.Summ.Rx);
    UpdateDo(do_POLLSUMMTX,time,BDGB14I.Modbus.Poll.Summ.Tx);
    UpdateDo(do_POLLSUMMEX,time,BDGB14I.Modbus.Poll.Summ.Ex);
    UpdateDo(do_ERRORCOUNT,time,GetErrCount(-1));
    if DebugFlagEnabled(dfStatist) then
    Success('PollRate: Rx='+Str(BDGB14I.Modbus.Poll.Rate.Rx)
                   +'  Tx='+Str(BDGB14I.Modbus.Poll.Rate.Tx)
                   +'  Ex='+Str(BDGB14I.Modbus.Poll.Rate.Ex));
    ClearModbusRate;
   end;
  end;
 end;
 //
 // BDGB14I tags initialization.
 //
 procedure BDGB14I_InitTags(Prefix:String);
 begin
  InitTag(BDGB14I.STATE.FACTN.tag,     Prefix+'.STATE.FACTN',     1);
  InitTag(BDGB14I.STATE.FACTD.tag,     Prefix+'.STATE.FACTD',     1);
  InitTag(BDGB14I.STATE.NETAD.tag,     Prefix+'.STATE.NETAD',     1);
  InitTag(BDGB14I.STATE.NETIF.tag,     Prefix+'.STATE.NETIF',     1);
  InitTag(BDGB14I.STATE.SOFTV.tag,     Prefix+'.STATE.SOFTV',     1);
  InitTag(BDGB14I.STATE.SOFTD.tag,     Prefix+'.STATE.SOFTD',     1);
  InitTag(BDGB14I.STATE.DEVID.tag,     Prefix+'.STATE.DEVID',     1);
  InitTag(BDGB14I.STATE.RECON.tag,     Prefix+'.STATE.RECON',     1);
  InitTag(BDGB14I.STATE.STNET.tag,     Prefix+'.STATE.STNET',     1);
  InitTag(BDGB14I.STATE.STADC.tag,     Prefix+'.STATE.STADC',     1);
  InitTag(BDGB14I.STATE.STREG.tag,     Prefix+'.STATE.STREG',     1);
  InitTag(BDGB14I.STATE.STCON.tag,     Prefix+'.STATE.STCON',     1);
  InitTag(BDGB14I.STATE.RECTL.tag,     Prefix+'.STATE.RECTL',     1);
  InitTag(BDGB14I.STATE.REREG.tag,     Prefix+'.STATE.REREG',     1);
  InitTag(BDGB14I.PARAM.CURIA.tag,     Prefix+'.PARAM.CURIA',     2);
  InitTag(BDGB14I.PARAM.GASVA.tag,     Prefix+'.PARAM.GASVA',     2);
  InitTag(BDGB14I.PARAM.CURTC.tag,     Prefix+'.PARAM.CURTC',     2);
  InitTag(BDGB14I.PARAM.CURAD.tag,     Prefix+'.PARAM.CURAD',     2);
  InitTag(BDGB14I.PARAM.CURHV.tag,     Prefix+'.PARAM.CURHV',     2);
  InitTag(BDGB14I.PARAM.CURU1.tag,     Prefix+'.PARAM.CURU1',     2);
  InitTag(BDGB14I.PARAM.CURU2.tag,     Prefix+'.PARAM.CURU2',     2);
  InitTag(BDGB14I.PARAM.CURU0.tag,     Prefix+'.PARAM.CURU0',     2);
  InitTag(BDGB14I.PARAM.CURR2.tag,     Prefix+'.PARAM.CURR2',     2);
  InitTag(BDGB14I.PARAM.CURI0.tag,     Prefix+'.PARAM.CURI0',     2);
  InitTag(BDGB14I.PARAM.PREHV.tag,     Prefix+'.PARAM.PREHV',     2);
  InitTag(BDGB14I.PARAM.DEFHV.tag,     Prefix+'.PARAM.DEFHV',     2);
  InitTag(BDGB14I.PARAM.ALRM1.tag,     Prefix+'.PARAM.ALRM1',     2);
  InitTag(BDGB14I.PARAM.ALRM2.tag,     Prefix+'.PARAM.ALRM2',     2);
  InitTag(BDGB14I.STATE.FACTS.tag,     Prefix+'.STATE.FACTS',     3);
  InitTag(BDGB14I.STATE.SOFTS.tag,     Prefix+'.STATE.SOFTS',     3);
  InitTag(BDGB14I.PARAM.IDENT.tag,     Prefix+'.PARAM.IDENT',     3);
  InitTag(BDGB14I.STATE.INDIC.tag,     Prefix+'.STATE.INDIC',     1);
  InitTag(BDGB14I.POLL.ENABLE.tag,     Prefix+'.POLL.ENABLE',     1);
  InitTag(BDGB14I.EDIT.ENABLE.tag,     Prefix+'.EDIT.ENABLE',     1);
  InitTag(BDGB14I.AUTO.CALIBR.tag,     Prefix+'.AUTO.CALIBR',     1);
  InitTag(BDGB14I.NET.RESTART.tag,     Prefix+'.NET.RESTART',     1);
  bNul(iSetTag(BDGB14I.EDIT.ENABLE.tag,0));
  bNul(iSetTag(BDGB14I.AUTO.CALIBR.tag,0));
  bNul(iSetTag(BDGB14I.NET.RESTART.tag,0));
 end;
 //
 // BDGB14I Driver cleanup.
 //
 procedure BDGB14I_Clear;
 begin
  BDGB14I.Simulator:=False;
  BDGB14I.Modbus.Port:=0;
  BDGB14I.Modbus.UnitId:=0;
  BDGB14I.Modbus.Timeout:=0;
  BDGB14I.Modbus.Polling:=0;
  BDGB14I.Modbus.Deadline:=0;
  BDGB14I.Modbus.State:=mb_NORMAL;
  BDGB14I.winControl:='';
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // BDGB14I Driver initialization...
 //
 procedure BDGB14I_Init;
 begin
  //
  // Read ini file variables
  //
  BDGB14I_InitTags(ReadIni('tagPrefix'));
  BDGB14I.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  Success('Simulator='+Str(Ord(BDGB14I.Simulator)));
  BDGB14I.Modbus.Port:=iValDef(ReadIni('ModbusPort'),1);
  Success('ModbusPort='+Str(BDGB14I.Modbus.Port));
  BDGB14I.Modbus.UnitId:=iValDef(ReadIni('ModbusUnitId'),1);
  Success('ModbusUnitId='+Str(BDGB14I.Modbus.UnitId));
  BDGB14I.Modbus.Polling:=iValDef(ReadIni('ModbusPolling'),1000);
  Success('ModbusPolling='+Str(BDGB14I.Modbus.Polling));
  BDGB14I.Modbus.Timeout:=iValDef(ReadIni('ModbusTimeout'),250);
  Success('ModbusTimeout='+Str(BDGB14I.Modbus.Timeout));
  BDGB14I.Modbus.Deadline:=iValDef(ReadIni('ModbusDeadline'),60000);
  Success('ModbusDeadline='+Str(BDGB14I.Modbus.Deadline));
  BDGB14I.Modbus.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000);
  Success('DelayOnStart='+Str(BDGB14I.Modbus.DelayOnStart));
  BDGB14I.winControl:=ReadIni('winControl');
  Success('winControl='+BDGB14I.winControl);
  BDGB14I.PARAM.ALRM1.val:=MaxReal;
  BDGB14I.PARAM.ALRM2.val:=MaxReal;
  BDGB14I.SIM.GASVA:=0;
  BDGB14I.SIM.DEFHV:=500;
  BDGB14I.SIM.PREHV:=BDGB14I.SIM.DEFHV;
  BDGB14I.SIM.NETAD:=Val(Copy(sGetTag(BDGB14I.PARAM.IDENT.tag),Length(sGetTag(BDGB14I.PARAM.IDENT.tag))-1));
  BDGB14I.SIM.NETIF:=2*3+256*3;
  BDGB14I.SIM.STNET:=BDGB14I.SIM.NETAD+256*1;
  BDGB14I.SIM.STADC:=0;
  BDGB14I.SIM.STCON:=255+(7+5*16)*256;
  BDGB14I.SIM.RECON:=0;
  BDGB14I.SIM.STREG:=4096*5+256*1+0;
  BDGB14I.SIM.FACTN:=16014;
  BDGB14I.SIM.FACTD:=42609; // 26.08.2016
  BDGB14I.SIM.SOFTV:=513;   // 2.1
  BDGB14I.SIM.SOFTD:=41781; // 22.05.2014
  BDGB14I.SIM.DEVID:=12;
  //
  // Initialize Cmd command table & clear Poll record.
  //
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  InitCmdTable;
 end;
 //
 // BDGB14I Driver Finalization.
 //
 procedure BDGB14I_Free;
 begin
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // BDGB14I Driver Polling.
 //
 procedure BDGB14I_Poll;
 begin
  BDGB14I_GUI_POLL;
  if mSecNow-FixmSecNow>BDGB14I.Modbus.DelayOnStart then
  if IsPortOpened then begin
   if BDGB14I.Simulator
   then BDGB14I_SIM_POLL;
   BDGB14I_CMD_POLL;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  BDGB14I_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  StdIn_SetScripts('@StartupScript','');
  StdIn_SetTimeouts(0,0,MaxInt,0);
  InitNetLibrary;
  BDGB14I_Init;
  cmd_ClearModbusSumm       := RegisterStdInCmd('@ClearModbusSumm',       '');
  cmd_PARAM_PREHV           := RegisterStdInCmd('@PARAM.PREHV',           '');
  cmd_PARAM_ALRM1           := RegisterStdInCmd('@PARAM.ALRM1',           '');
  cmd_PARAM_ALRM2           := RegisterStdInCmd('@PARAM.ALRM2',           '');
  cmd_STATE_RECTL           := RegisterStdInCmd('@STATE.RECTL',           '');
  cmd_STATE_REREG           := RegisterStdInCmd('@STATE.REREG',           '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',                  '');
  cmd_SetRegType            := RegisterStdInCmd('@SetRegType',            '');
  cmd_SetRegCode            := RegisterStdInCmd('@SetRegCode',            '');
  cmd_mayak_bdgb14i_htm     := RegisterStdInCmd('@mayak_bdgb14i.htm',     '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  FreeNetLibrary;
  BDGB14I_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  BDGB14I_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,dat:String; cmdid:Integer; ref,cid,tim,port,uid,fid:Integer; ms,v:Real;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  dat:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Modbus.Reply ref cid tim port uid fid $$dat
   }
   if (cmdid=cmd_NetModbusReply) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then BDGB14I_OnReply(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    BDGB14I.Modbus.State:=mb_NORMAL;
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Poll ref cid tim port uid fid $$dat
   This message uses in simulation mode only
   }
   if (cmdid=cmd_NetModbusPoll) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then BDGB14I_OnSimPoll(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    BDGB14I.Modbus.State:=mb_NORMAL;
    Data:='';
   end else
   {
   @Modbus.Refuse ...
   }
   if (cmdid=cmd_NetModbusRefuse) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,0)))
    else Trouble(GotBug(cmd+' '+arg));
    BDGB14I.Modbus.State:=mb_REFUSE;
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Timeout ...
   }
   if (cmdid=cmd_NetModbusTimeout) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Problem(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,64)))
    else Trouble(GotBug(cmd+' '+arg));
    BDGB14I.Modbus.State:=mb_TIMEOUT;
    ClearModbusPoll;
    Data:='';
   end else
   {
   @ClearModbusSumm
   }
   if (cmdid=cmd_ClearModbusSumm) then begin
    ClearModbusSumm;
    Data:='';
   end else
   {
   @PARAM.PREHV 300.5
   }
   if (cmdid=cmd_PARAM_PREHV) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then HoldCmdOpData(cm_WritPREHV,v);
    Data:='';
   end else
   {
   @PARAM.ALRM1 1E9
   }
   if (cmdid=cmd_PARAM_ALRM1) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then bNul(rSetTag(BDGB14I.PARAM.ALRM1.tag,v));
    Data:='';
   end else
   {
   @PARAM.ALRM2 1E10
   }
   if (cmdid=cmd_PARAM_ALRM2) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then bNul(rSetTag(BDGB14I.PARAM.ALRM2.tag,v));
    Data:='';
   end else
   {
   @STATE.RECTL 256
   }
   if (cmdid=cmd_STATE_RECTL) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then HoldCmdOpData(cm_WritRECTL,v);
    Data:='';
   end else
   {
   @STATE.REREG 2
   }
   if (cmdid=cmd_STATE_REREG) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then HoldCmdOpData(cm_WritREREG,v);
    if BDGB14I.Simulator then BDGB14I.SIM.STREG:=Round(v);
    Data:='';
   end else
   {
   @Edit PARAM.PREHV
   }
   if (cmdid=cmd_Edit) then begin
    if IsSameText(ExtractWord(1,arg),'PARAM.PREHV')
    then StartEditTagEx(BDGB14I.PARAM.PREHV.tag,'Уставка HV, Вольт',mime_decode(ExtractWord(2,arg)));
    if IsSameText(ExtractWord(1,arg),'PARAM.ALRM1')
    then StartEditTagEx(BDGB14I.PARAM.ALRM1.tag,'Порог Тревога 1 (желтый), Бк/м3',mime_decode(ExtractWord(2,arg)));
    if IsSameText(ExtractWord(1,arg),'PARAM.ALRM2')
    then StartEditTagEx(BDGB14I.PARAM.ALRM2.tag,'Порог Тревога 2 (красный), Бк/м3',mime_decode(ExtractWord(2,arg)));
    Data:='';
   end else
   {
   @SetRegType 1
   }
   if (cmdid=cmd_SetRegType) then begin
    if not IsEmptyStr(arg) then SetRegType(iValDef(Trim(arg),-1));
    Data:='';
   end else
   {
   @SetRegCode 0
   }
   if (cmdid=cmd_SetRegCode) then begin
    if not IsEmptyStr(arg) then SetRegCode(iValDef(Trim(arg),-1));
    Data:='';
   end else
   {
   @mayak_bdgb14i.htm
   }
   if (cmdid=cmd_mayak_bdgb14i_htm) then begin
    dat:=ParamStr('FileSearch mayak_bdgb14i.htm');
    if (dat<>'') then Cron('@BrowseHelp '+dat);
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  dat:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
