 {
 GUI for HVSS - Graphics User Interface for High Voltage Scan System.
 Last release:
  21.03.2012: kouriakine@mail.ru
 [@Help]
 |StdIn Command list: "@cmd=arg" or "@cmd arg"
 |******************************************************
 | @Shutdown w h   - Shutdown, w(hat)=crw/daq/win
 |                   h(ow)=exit,logout,restart
 | @SaveParams     - Save parameters to   INI file.
 | @LoadParams     - Load parameters from INI file.
 |******************************************************
 []
 }
program hvss_gui;
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 FixedWind         = True;       { Fixed plot window sizes          }
 UseAdjustment     = False;      { Adjust voltage to 2 or 5 V       }
 SL15MaxVolt       = 15000;      { SL15 voltage range               }
 SR60MaxVolt       = 60000;      { SR60 voltage range               }
 SL15MaxCurr       = 20;         { SL15 current range               }
 SR60MaxCurr       = 100;        { SR60 current range               }
 SL15MaxCode       = 4095;       { SL15 max DAC code                }
 SR60MaxCode       = 4095;       { SR60 max DAC code                }
 IBoxMaxCurr       = 2500;       { IBox max current                 }
 WaitMaxTime       = 60;         { Max time to wait                 }
 ExpoMaxTime       = 60;         { Max time to expose               }
 LvpsMinUDac       = 0;          { Minimal LVPS U DAC, mV           }        
 LvpsMaxUDac       = 10000;      { Maximal LVPS U DAC, mV           }        
 SmilePeriod       = 1000;       { Polling period for smile face    }
 ao_SL15VoltSet    = 0;          { SL15 wanted voltage              }
 ao_SR60VoltSet    = 1;          { SR60 wanted voltage              }
 ao_SL15CurrSet    = 2;          { SL15 wanted current              }
 ao_SR60CurrSet    = 3;          { SR60 wanted current              }
 ao_ScanIBox1      = 4;          { Scan current Box1                }
 ao_ScanIBox2      = 5;          { Scan current Box2                }
 ao_ScanIBox3      = 6;          { Scan current Box3                }
 ao_ScanIBox4      = 7;          { Scan current Box4                }
 ao_ScanIBox5      = 8;          { Scan current Box5                }
 ao_ScanISumm      = 9;          { Scan current Summ                }
 ao_ScanIDiap      = 10;         { Scan current Diaphragm           }
 ao_ScanIPock      = 11;         { Scan current Pocket              }
 ao_ScanKQual      = 12;         { Scan Kriteria of QUALity         }
 ao_ScanAddon      = 13;         { Scan ADDON user parameter        }
 ao_SpecIBox1      = 14;         { Spectr current Box1              }
 ao_SpecIBox2      = 15;         { Spectr current Box2              }
 ao_SpecIBox3      = 16;         { Spectr current Box3              }
 ao_SpecIBox4      = 17;         { Spectr current Box4              }
 ao_SpecIBox5      = 18;         { Spectr current Box5              }
 ao_SpecISumm      = 19;         { Spectr current Summ              }
 ao_SpecIDiap      = 20;         { Spectr current Diaphragm         }
 ao_SpecIPock      = 21;         { Spectr current Pocket            }
 ao_SpecKQual      = 22;         { Spectr Kriteria of QUALity       }
 ao_SpecAddon      = 23;         { Spectr ADDON user parameter      }
 ao_LvpsUSet       = 24;         { LVPS USET                        }
 ao_LvpsUDac       = 25;         { LVPS UDAC                        }
 ao_HvLvUGot       = 26;         { SR60.UGOT+LVPS.USET              }
 ao_HvLvUGet       = 27;         { SR60.UGET+LVPS.USET              }
 ao_HvLvUSet       = 28;         { SR60.USET+LVPS.USET              }
 do_SL15OnBtn      = 0;          { SL15 ONBT                        }
 do_SR60OnBtn      = 1;          { SR60 ONBT                        }
 do_SL15ResHr      = 2;          { SL15 Reset Hours                 }
 do_SR60ResHr      = 3;          { SR60 Reset Hours                 }
 do_BeamVolts      = 4;          { BEAM real Voltage                }
 do_BeamKQual      = 5;          { BEAM Kriteria of Quality         }
 do_BeamAddon      = 6;          { BEAM Addon user parameter        }
 ai_SL15VoltGot    = 0;          { SL15 got voltage                 }
 ai_SR60VoltGot    = 1;          { SR60 got voltage                 }
 ai_SL15CurrGot    = 2;          { SL15 got current                 }
 ai_SR60CurrGot    = 3;          { SR60 got current                 }
 ai_BeamIBox1      = 4;          { Beam current Box1                }
 ai_BeamIBox2      = 5;          { Beam current Box2                }
 ai_BeamIBox3      = 6;          { Beam current Box3                }
 ai_BeamIBox4      = 7;          { Beam current Box4                }
 ai_BeamIBox5      = 8;          { Beam current Box5                }
 ai_BeamISumm      = 9;          { Beam current Summ                }
 ai_BeamIDiap      = 10;         { Beam current Diaphragm           }
 ai_BeamIPock      = 11;         { Beam current Pocket              }
 ai_BeamUAdam      = 12;         { Beam voltage from ADAM           }
 ca_LvpsUSet       = 0;          { Calibration on LVPS USET         }
type
 TTagRef           = record tag,nai,ndi,nao,ndo:Integer; val:Real; end;
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }

 winHvssGui        : String;     { HVSS.GUI window                  }
 winHvssITab       : String;     { HVSS.ITAB window                 }
 winBeamIPlot      : String;     { HVSS.BEAM.I.PLOT window          }
 winBeamUPlot      : String;     { HVSS.BEAM.U.PLOT window          }
 winBeamKPlot      : String;     { HVSS.BEAM.K.PLOT window          }
 winScanIPlot      : String;     { HVSS.SCAN.I.PLOT window          }
 winSpecIPlot      : String;     { HVSS.SPEC.I.PLOT window          }
 winLvpsUPlot      : String;     { HVSS.LVPS.U.PLOT window          }
 winHvLvUPlot      : String;     { HVSS.HVLV.U.PLOT window          }
 CMD               : record      { CoMmandDs to control HVSS GUI    }
  HELP             : TTagRef;    { Open Help                        }
  OPEN             : TTagRef;    { Open DAT file(s)                 }
  ITAB             : TTagRef;    { Open ITAB window                 }
  SOUND            : TTagRef;    { Sound                            }
  TOOLS            : TTagRef;    { Tools menu                       }
  SMILE            : TTagRef;    { Smile menu                       }
  CLOSE            : TTagRef;    { Close DAQ/CRW/WIN                }
 end;
 SL15,SR60         : record      { SL15,SR60 related data           }
  devDrv           : Integer;    { Driver device                    }
  winUPlot         : String;     { HVSS.SXXX.U.PLOT window          }
  winIPlot         : String;     { HVSS.SXXX.I.PLOT window          }
  winTPlot         : String;     { HVSS.SXXX.T.PLOT window          }
  ONBT             : TTagRef;    { ON/OFF button                    }
  ON               : TTagRef;    { ON/OFF real state                }
  USET             : TTagRef;    { Wanted voltage                   }
  UGOT             : TTagRef;    { Accepted voltage                 }
  UDAC             : TTagRef;    { Accepted voltage DAC code        }
  UGET             : TTagRef;    { Real voltage from ADC            }
  UADC             : TTagRef;    { Real voltage ADC code            }
  ISET             : TTagRef;    { Wanted current                   }
  IGOT             : TTagRef;    { Accepted current                 }
  IDAC             : TTagRef;    { Accepted current DAC code        }
  IGET             : TTagRef;    { Real current from ADC            }
  IADC             : TTagRef;    { Real current ADC code            }
  HRS              : TTagRef;    { Hours counter                    }
  HRSRESET         : TTagRef;    { Reset hours counter button       }
  TSIC             : TTagRef;    { Temperature of electronics       }
  TMUL             : TTagRef;    { Temperature of multiplier        }
  INTERLOCK        : TTagRef;    { Interlock state                  }
  FAULT            : TTagRef;    { Fault state                      }
  CONNECT          : TTagRef;    { Connection state                 }
  ANSTIME          : TTagRef;    { Request-answer time, ms          }
 end;
 SCAN              : record      { Scanner data                     }
  UMIN             : TTagRef;    { Min scan range                   }
  UMAX             : TTagRef;    { Max scan range                   }
  WAIT             : TTagRef;    { Wait time                        }
  EXPO             : TTagRef;    { Exposition time                  }
  STEP             : TTagRef;    { Scan voltage step, V             }
  ONBT             : TTagRef;    { ON/OFF scanning                  }
  NUMP             : TTagRef;    { Number of points                 }
  REST             : TTagRef;    { Estimated time rest, sec         }
  PLOT             : TTagRef;    { Plot button                      }
  GOBT             : TTagRef;    { GoTo button                      }
  SAVE             : TTagRef;    { Save button                      }
 end;
 BEAM              : record      { Beam data                        }
  IBOX1            : TTagRef;    { Current of BOX1                  }
  IBOX2            : TTagRef;    { Current of BOX2                  }
  IBOX3            : TTagRef;    { Current of BOX3                  }
  IBOX4            : TTagRef;    { Current of BOX4                  }
  IBOX5            : TTagRef;    { Current of BOX5                  }
  ISUMM            : TTagRef;    { Current of SUMM                  }
  IDIAP            : TTagRef;    { Current of DIAPHRAGM             }
  IPOCK            : TTagRef;    { Current of POCKET                }
  UADAM            : TTagRef;    { Voltage readout from ADAM        }
  VOLTS            : TTagRef;    { Real Beam Voltage - value        }
  VFORM            : TTagRef;    { Real Beam Voltage - formula      }
  KQUAL            : TTagRef;    { Kriteria of Quality - value      }
  KFORM            : TTagRef;    { Kriteria of Quality - Formula    }
  ADDON            : TTagRef;    { Addon user parameter - value     }
  AFORM            : TTagRef;    { Addon user parameter - Formula   }
 end;
 LVPS              : record      { Beam data                        }
  USET             : TTagRef;    { Wanted voltage, V                }
  UDAC             : TTagRef;    { Wanted DAC, mV                   }
 end;
 HVLV              : record      { HV+LV data                       }
  PLOT             : TTagRef;    { Plot button                      }
  GOBT             : TTagRef;    { GoTo button                      }
  UGOT             : TTagRef;    { SR60.UGOT+LVPS.USET Voltage      }
  UGET             : TTagRef;    { SR60.UGET+LVPS.USET Voltage      }
  USET             : TTagRef;    { SR60.USET+LVPS.USET Voltage      }
 end;
 MenuClose         : Integer;    { Choice on menu Close             }
 MenuTools         : Integer;    { Choice on menu Tools             }
 SmileTimer        : Real;       { Timer for smile face             }
 SmileCount        : Real;       { Error counter                    }
 LastIncDecTime    : Real;       { Time of last Inc,Dec             }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Speak, i.e. send message to speech server.
 }
 procedure Speaker(msg:String);
 begin
  if iGetTag(CMD.SOUND.tag)>0 then Speak(msg);
 end;
 {
 Xor bit on click (local version)
 }
 procedure ClickBitXorLocal(tag,XorMask:Integer);
 var nv:Integer;
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  if Length(s)>0 then begin
   InfoBox(s);
   if Pos(':',s)>0 then s:=Trim(Copy(s,Pos(':',s)+1));
   Speaker(s);
  end;
 end;
 {
 Get text of Close menu items
 }
 function MenuCloseItem(MenuClose:Integer):String;
 begin
  if MenuClose=0 then MenuCloseItem:='Завершить CRW-DAQ' else
  if MenuClose=1 then MenuCloseItem:='Перезагрузить DAQ' else
  MenuCloseItem:='';
 end;
 {
 Execute Close menu commands
 }
 procedure MenuCloseCmnd(MenuClose:Integer);
 begin
  if MenuClose=0  then DevSendCmd(devMySelf,'@Shutdown Crw Exit');
  if MenuClose=1  then DevSendCmd(devMySelf,'@Shutdown Daq Restart');
 end;
 {
 Get text of Tools menu items
 }
 function MenuToolsItem(MenuTools:Integer):String;
 begin
  if MenuTools=0  then MenuToolsItem:='Загрузить настройки из INI файла : @LoadParams' else
  if MenuTools=1  then MenuToolsItem:='Сохранить настройки в  INI файле : @SaveParams' else
  MenuToolsItem:='';
 end;
 {
 Execute Tools menu commands
 }
 procedure MenuToolsCmnd(MenuTools:String);
 var p:Integer;
 begin
  p:=Pos(Dump('@'),MenuTools);
  if p>0 then DevSendCmd(devMySelf,Copy(MenuTools,p));
 end;
 {
 Truncate is Trunc replacement to avoid digital noise problem.
 CodeToVoltage(VoltageToCode(Code)) should be equal to Code.
 But it is not, if use simple Trunc instead of Truncate.
 }
 function Truncate(X:Real):Integer;
 begin
  Truncate:=Trunc(X+MachEps*2e3);
 end;
 {
 Adjust value x to be divider of n.
 }
 function AdjustValue(x:Real;n:Integer):Real;
 begin
  AdjustValue:=Truncate((x+n-1)/n)*n;
 end;
 {
 Convert code to Voltage,V.
 }
 function CodeToVoltage(code,MinVoltageCode,MaxVoltageCode,MinVoltageRange,MaxVoltageRange:Integer;
                        Adjust:Boolean):Real;
 var beta:Real; Voltage:Real;
 begin
  if code<MinVoltageCode then code:=MinVoltageCode;
  if code>MaxVoltageCode then code:=MaxVoltageCode;
  beta:=(code-MinVoltageCode)/(MaxVoltageCode-MinVoltageCode);
  Voltage:=MinVoltageRange+beta*(MaxVoltageRange-MinVoltageRange);
  if Adjust then begin
   beta:=(MaxVoltageRange-MinVoltageRange)/(MaxVoltageCode-MinVoltageCode);
   if beta>2 then Voltage:=Truncate(Voltage+0.999999999999);
   if UseAdjustment then
   if beta>10 then Voltage:=AdjustValue(Voltage,5) else
   if beta>3  then Voltage:=AdjustValue(Voltage,2);
  end;
  if Voltage<MinVoltageRange then Voltage:=MinVoltageRange;
  if Voltage>MaxVoltageRange then Voltage:=MaxVoltageRange;
  CodeToVoltage:=Voltage;
 end;
 {
 Convert Voltage,V to code.
 }
 function VoltageToCode(Voltage:Real;MinVoltageCode,MaxVoltageCode,MinVoltageRange,MaxVoltageRange:Integer):Integer;
 var beta:Real; code:Integer;
 begin
  if Voltage<MinVoltageRange then Voltage:=MinVoltageRange;
  if Voltage>MaxVoltageRange then Voltage:=MaxVoltageRange;
  beta:=(Voltage-MinVoltageRange)/(MaxVoltageRange-MinVoltageRange);
  code:=Truncate(MinVoltageCode+beta*(MaxVoltageCode-MinVoltageCode));
  if code<MinVoltageCode then code:=MinVoltageCode;
  if code>MaxVoltageCode then code:=MaxVoltageCode;
  VoltageToCode:=code;
 end;
 {
 Move voltage to Step quantums.
 }
 function QuantumStepMake(X:Real;Step,MaxCode,Range:Integer; Adjust:Boolean):Real;
 var code:Integer;
 begin
  code:=VoltageToCode(X,0,MaxCode,0,Range)+Step;
  QuantumStepMake:=CodeToVoltage(code,0,MaxCode,0,Range,Adjust);
 end;
 {
 SL15 related conversions.
 }
 function CodeToVoltageSL15(code:Integer; Adjust:Boolean):Real;
 begin
  CodeToVoltageSL15:=CodeToVoltage(code,0,SL15MaxCode,0,SL15MaxVolt,Adjust);
 end;
 function VoltageToCodeSL15(Voltage:Real):Integer;
 begin
  VoltageToCodeSL15:=VoltageToCode(Voltage,0,SL15MaxCode,0,SL15MaxVolt);
 end;
 function QuantumStepSL15(X:Real;Step:Integer; Adjust:Boolean):Real;
 begin
  QuantumStepSL15:=QuantumStepMake(X,Step,SL15MaxCode,SL15MaxVolt,Adjust);
 end;
 {
 SR60 retated conversions.
 }
 function CodeToVoltageSR60(code:Integer; Adjust:Boolean):Real;
 begin
  CodeToVoltageSR60:=CodeToVoltage(code,0,SR60MaxCode,0,SR60MaxVolt,Adjust);
 end;
 function VoltageToCodeSR60(Voltage:Real):Integer;
 begin
  VoltageToCodeSR60:=VoltageToCode(Voltage,0,SR60MaxCode,0,SR60MaxVolt);
 end;
 function QuantumStepSR60(X:Real;Step:Integer; Adjust:Boolean):Real;
 begin
  QuantumStepSR60:=QuantumStepMake(X,Step,SR60MaxCode,SR60MaxVolt,Adjust);
 end;
 {
 Test data convertion procedures.
 }
 procedure TestConvertion;
 var i,c,n:Integer; v,u,dmax,dave,ms:Real;
 begin
  dmax:=0; dave:=0;
  ms:=mkSecNow/1000; n:=0;
  for i:=0 to SL15MaxCode do begin
   v:=CodeToVoltageSL15(i,True);
   c:=VoltageToCodeSL15(v);
   if i<>c then n:=n+1;
   c:=VoltageToCodeSL15(QuantumStepSL15(v,+1,True));
   if i<SL15MaxCode then if i+1<>c then n:=n+1;
   c:=VoltageToCodeSL15(QuantumStepSL15(v,-1,True));
   if i>0 then if i-1<>c then n:=n+1;
  end;
  for i:=0 to SR60MaxCode do begin
   v:=CodeToVoltageSR60(i,True);
   c:=VoltageToCodeSR60(v);
   if i<>c then n:=n+1;
   c:=VoltageToCodeSR60(QuantumStepSR60(v,+1,True));
   if i<SR60MaxCode then if i+1<>c then n:=n+1;
   c:=VoltageToCodeSR60(QuantumStepSR60(v,-1,True));
   if i>0 then if i-1<>c then n:=n+1;
   u:=i*SR60MaxVolt/SR60MaxCode;
   dmax:=Max(dmax,Abs(v-u));
   dave:=dave+Abs(v-u)/(SR60MaxCode+1);
  end;
  ms:=Int(mkSecNow/1000-ms);
  Success('dmax='+StrFix(dmax,1,3)+', dave='+StrFix(dave,1,3));
  Success('TestConvertion: '+Str(n)+' errors,'+Str(ms)+' ms.');
 end;
 {
 Conver low voltage power supply DAC (mV) to output Voltage (volts).
 }
 function LvDacToVoltage(Dac_mV:Real):Real;
 begin
  if Dac_mV<LvpsMinUDac then Dac_mV:=LvpsMinUDac;
  if Dac_mV>LvpsMaxUDac then Dac_mV:=LvpsMaxUDac;
  LvDacToVoltage:=Max(0,Calibr(ca_LvpsUSet,Dac_mV,0));
 end;
 {
 Conver low voltage power supply output Voltage (volts) to DAC (mV).
 }
 function LvVoltageToDac(Volts:Real):Real;
 begin
  if Volts<=LvDacToVoltage(LvpsMinUDac) then LvVoltageToDac:=LvpsMinUDac else
  if Volts>=LvDacToVoltage(LvpsMaxUDac) then LvVoltageToDac:=LvpsMaxUDac else
  LvVoltageToDac:=InvCalibr(ca_LvpsUSet,Volts,0,LvpsMinUDac,LvpsMaxUDac);
 end;
 {
 Get units and title (comment label) for ref...
 }
 function GetRefUniTit(ref:Integer):String;
 var Name,UniTit:String;
  function Check(tag:Integer):Boolean;
  begin
   Check:=IsSameText(Name,RefInfo(tag,'Name'));
  end;
 begin
  UniTit:='';
  Name:=RefInfo(ref,'Name');
  if Length(Name)>0 then begin
   if Check(RefAi(ai_BeamIBox1)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIBox2)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIBox3)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIBox4)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIBox5)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamISumm)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIDiap)) then UniTit:='мкА Ток'         else
   if Check(RefAi(ai_BeamIPock)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIBox1)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIBox2)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIBox3)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIBox4)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIBox5)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanISumm)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIDiap)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanIPock)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_ScanKQual)) then UniTit:='УЕ  Критерий качества' else
   if Check(RefAo(ao_ScanAddon)) then UniTit:='УЕ  Дополнит.параметр' else
   if Check(RefAo(ao_SpecIBox1)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIBox2)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIBox3)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIBox4)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIBox5)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecISumm)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIDiap)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecIPock)) then UniTit:='мкА Ток'         else
   if Check(RefAo(ao_SpecKQual)) then UniTit:='УЕ  Критерий качества' else
   if Check(RefAo(ao_SpecAddon)) then UniTit:='УЕ  Дополнит.параметр' else
   if Check(BEAM.KQUAL.tag)      then UniTit:='УЕ  Критерий качества' else
   if Check(BEAM.ADDON.tag)      then UniTit:='УЕ  Дополнит.параметр' else
   if Check(BEAM.UADAM.tag)      then UniTit:='В   Напряжение'  else
   if Check(BEAM.VOLTS.tag)      then UniTit:='В   Напряжение'  else
   if Check(HVLV.UGOT.tag)       then UniTit:='В   Напряжение'  else
   if Check(HVLV.UGET.tag)       then UniTit:='В   Напряжение'  else
   if Check(HVLV.USET.tag)       then UniTit:='В   Напряжение'  else
   if Check(SL15.USET.tag)       then UniTit:='В   Напряжение'  else
   if Check(SL15.UGOT.tag)       then UniTit:='В   Напряжение'  else
   if Check(SL15.UDAC.tag)       then UniTit:='Код Напряжение'  else
   if Check(SL15.UGET.tag)       then UniTit:='В   Напряжение'  else
   if Check(SL15.UADC.tag)       then UniTit:='Код Напряжение'  else
   if Check(SL15.ISET.tag)       then UniTit:='мА  Ток'         else
   if Check(SL15.IGOT.tag)       then UniTit:='мА  Ток'         else
   if Check(SL15.IDAC.tag)       then UniTit:='Код Ток'         else
   if Check(SL15.IGET.tag)       then UniTit:='мА  Ток'         else
   if Check(SL15.IADC.tag)       then UniTit:='Код Ток'         else
   if Check(SL15.TSIC.tag)       then UniTit:='°C  Температура' else
   if Check(SL15.TMUL.tag)       then UniTit:='°C  Температура' else
   if Check(SR60.USET.tag)       then UniTit:='В   Напряжение'  else
   if Check(SR60.UGOT.tag)       then UniTit:='В   Напряжение'  else
   if Check(SR60.UDAC.tag)       then UniTit:='Код Напряжение'  else
   if Check(SR60.UGET.tag)       then UniTit:='В   Напряжение'  else
   if Check(SR60.UADC.tag)       then UniTit:='Код Напряжение'  else
   if Check(SR60.ISET.tag)       then UniTit:='мА  Ток'         else
   if Check(SR60.IGOT.tag)       then UniTit:='мА  Ток'         else
   if Check(SR60.IDAC.tag)       then UniTit:='Код Ток'         else
   if Check(SR60.IGET.tag)       then UniTit:='мА  Ток'         else
   if Check(SR60.IADC.tag)       then UniTit:='Код Ток'         else
   if Check(SR60.TSIC.tag)       then UniTit:='°C  Температура' else
   if Check(SR60.TMUL.tag)       then UniTit:='°C  Температура' else
   if Check(LVPS.USET.tag)       then UniTit:='В   Напряжение'  else
   if Check(LVPS.UDAC.tag)       then UniTit:='мВ  ЦАП'         else
   UniTit:='? ?';   
  end;
  GetRefUniTit:=UniTit;
  UniTit:='';
  Name:='';
 end;
 {
 Get measurement units for ref...
 }
 function GetRefUnits(ref:Integer):String;
 begin
  GetRefUnits:=ExtractWord(1,GetRefUniTit(ref));
 end;
 {
 Get title (comment label) for ref...
 }
 function GetRefTitle(ref:Integer):String;
 begin
  GetRefTitle:=Trim(SkipWords(1,GetRefUniTit(ref)));
 end;
 {
 Получить имя единиц времени.
 }
 function GetTimeUnitsName:String;
 begin
  if Round(TimeUnits/1000)=1        then GetTimeUnitsName:='Секунд' else
  if Round(TimeUnits/1000)=60       then GetTimeUnitsName:='Минут'  else
  if Round(TimeUnits/1000)=60*60    then GetTimeUnitsName:='Часов'  else
  if Round(TimeUnits/1000)=60*60*24 then GetTimeUnitsName:='Суток'  else
  GetTimeUnitsName:='???';
 end;
 {
 Return |Range=x1;y1;x2;y2
 }
 function GetRangeStr(x1,y1,x2,y2:Real):String;
 begin
  GetRangeStr:='|Range='+Str(x1)+';'+Str(y1)+';'+Str(x2)+';'+Str(y2);
 end;
 {
 Plot curve by curve or tag reference Ref.
 Mode 1 - Show
 Mode 2 - Select
 Mode 4 - Fix Size
 Mode 8 - Fix Pos
 }
 procedure winXPlot(win,opt:String;Ref,Mode:Integer);
 var crv,tag:Integer;
 begin
  opt:=Trim(opt);
  crv:=0; tag:=0;
  if Length(win)>0 then
  if RefFind('Window '+win)>0 then begin
   if Ref<>0 then
   if TypeTag(Ref)>0 then begin
    crv:=RefFind('Curve '+NameTag(Ref));
    tag:=Ref;
   end else begin
    crv:=RefFind('Curve '+CrvName(Ref));
    tag:=FindTag(CrvName(Ref));
   end;
   if iAnd(Mode,4)<>0
   then opt:='|Options=-Width,-Height'+opt
   else opt:='|Options=+Width,+Height'+opt;
   if iAnd(Mode,8)<>0
   then opt:='|Options=-Left,-Top'+opt
   else opt:='|Options=+Left,+Top'+opt;
   if IsSameText(ExtractFileExt(win),'.PLOT') then begin
    if IsSameText(win,winScanIPlot)
    or IsSameText(win,winSpecIPlot)
    then opt:='|LegendX=^R{В}   ^N^CНапряжение'+opt
    else opt:='|LegendX=^R{'+GetTimeUnitsName+'}   ^N^CВремя'+opt;
    if Length(GetRefTitle(crv))>0
    then opt:='|LegendY=^C'+GetRefTitle(crv)+'^N^L   {'+GetRefUnits(crv)+'}'+opt;
    opt:='|SelectCurve='+CrvName(crv)+opt;
   end;
   if IsSameText(win,winHvssGui) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=0|Width=1164|Height=317|ToolBarHeight=50'));
    bNul(WinDraw(win+'|Options=-Min,-Max,-Close,-HScroll,-VScroll'+opt));
   end;
   if IsSameText(win,winHvssITab) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=564|Top=0|Width=600|Height=830|ToolBarHeight=0'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,-VScroll'+opt));
   end;
   if IsSameText(win,winScanIPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=1164|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winSpecIPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=1164|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winLvpsUPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winBeamIPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=582|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winBeamUPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winHvLvUPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,winBeamKPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=0|Width=582|Height=830'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SL15.winUPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SL15.winIPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=582|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SL15.winTPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=582|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SR60.winUPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=0|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SR60.winIPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=582|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if IsSameText(win,SR60.winTPlot) then begin
    bNul(WinShow(win));
    bNul(WinDraw(win+'|Left=582|Top=317|Width=582|Height=513'));
    bNul(WinDraw(win+'|Options=-Min,-Max,+Close,-HScroll,+VScroll'+opt));
   end;
   if iAnd(Mode,2)<>0 then bNul(WinSelect(win)) else
   if iAnd(Mode,1)<>0 then bNul(WinShow(win))   else bNul(WinHide(win));
  end;
  opt:='';
 end;
 {
 Clear ROI markers.
 }
 procedure ClearRoiMarkers;
 begin
  rNul(Eval('@system ROI_X1=')+Eval('@system ROI_X2='));
  rNul(Eval('@system ROI_Y1=')+Eval('@system ROI_Y2='));
 end;
 {
 Set SR60 HV as specified in ROI marker.
 }
 procedure GoToRoiSR60;
 var roi,volt,ibox:Real; i,j:Integer;
 begin
  roi:=Eval('@system ROI_X1');
  if IsNan(roi) or IsInf(roi) or (roi<0) or (roi>SR60MaxVolt)
  or (roi<rGetTag(SCAN.UMIN.tag)) or (roi>rGetTag(SCAN.UMAX.tag)) then begin
   Warning('Маркер ROI ('+Str(roi)+') не выбран или выбран неверно! Не могу перейти к нему!');
  end else begin
   ibox:=0;
   volt:=QuantumStepSR60(roi,0,True);
   for j:=ao_ScanIBox1 to ao_ScanIBox5 do
   for i:=1 to Round(CrvLen(RefAo(j))) do ibox:=Max(ibox,CrvY(RefAo(j),i));
   bNul(WinDraw(winScanIPlot+'|Roi='+Str(volt)+';'+Str(ibox)+';roix2;roiy2'));
   StartEditTagStr(SR60.USET.tag,'Уставка HV SR60 по маркеру ROI -> ',Str(volt));
  end;
 end;
 {
 Set HV+LV as specified in ROI marker.
 }
 procedure GoToRoiHvLv;
 var roi,volt,ibox:Real; i,j:Integer;
 begin
  roi:=Eval('@system ROI_X1');
  if IsNan(roi) or IsInf(roi) or (roi<0) or (roi>SR60MaxVolt)
  or (roi<rGetTag(SCAN.UMIN.tag)) or (roi>rGetTag(SCAN.UMAX.tag)) then begin
   Warning('Маркер ROI ('+Str(roi)+') не выбран или выбран неверно! Не могу перейти к нему!');
  end else begin
   ibox:=0;
   volt:=Round(roi);
   for j:=ao_ScanIBox1 to ao_ScanIBox5 do
   for i:=1 to Round(CrvLen(RefAo(j))) do ibox:=Max(ibox,CrvY(RefAo(j),i));
   bNul(WinDraw(winSpecIPlot+'|Roi='+Str(volt)+';'+Str(ibox)+';roix2;roiy2'));
   StartEditTagStr(HVLV.USET.tag,'Уставка HV+LV по маркеру ROI -> ',Str(volt));
  end;
 end;
 {
 Save [CustomParameters]
 }
 procedure SaveParams;
 var f:String; i:Integer;
  procedure SaveTag(tag:Integer);
  begin
   if TypeTag(tag)=1 then Writeln(NameTag(tag),' = ',Str(iGetTag(tag)));
   if TypeTag(tag)=2 then Writeln(NameTag(tag),' = ',Str(rGetTag(tag)));
   if TypeTag(tag)=3 then Writeln(NameTag(tag),' = ',sGetTag(tag));
  end;
  procedure SaveTagHex(tag:Integer);
  begin
   if TypeTag(tag)=1 then Writeln(NameTag(tag),' = ',Hex_Encode(Dump(iGetTag(tag))));
   if TypeTag(tag)=2 then Writeln(NameTag(tag),' = ',Hex_Encode(Dump(rGetTag(tag))));
   if TypeTag(tag)=3 then Writeln(NameTag(tag),' = ',Hex_Encode(sGetTag(tag)));
  end;
 begin
  f:=AdaptFileName(ReadIni('CustomFile'));
  if Length(f)>0 then begin
   i:=Rewrite(DaqFileRef(f,''));
   writeln(ReadIni('CustomSection'));
   SaveTag(CMD.SOUND.tag);
   SaveTag(SL15.USET.tag); SaveTag(SL15.ISET.tag);
   SaveTag(SR60.USET.tag); SaveTag(SR60.ISET.tag);
   SaveTag(SCAN.UMIN.tag); SaveTag(SCAN.UMAX.tag);
   SaveTag(SCAN.WAIT.tag); SaveTag(SCAN.EXPO.tag);
   SaveTag(SCAN.STEP.tag); SaveTag(LVPS.UDAC.tag);
   SaveTagHex(BEAM.VFORM.tag);
   SaveTagHex(BEAM.KFORM.tag);
   SaveTagHex(BEAM.AFORM.tag);
   i:=Append('');
   {Save a copies}
   bNul(FileCopy(DaqFileRef(f,'')+' '
       +AddPathDelim(ExtractFilePath(DaqFileRef(f,'')))
       +ExtractFileName(DaqFileRef(f,''))+'-'+StrReplace(StrReplace(
        GetDateTime(msecnow),Dump('.'),'',3),Dump(':'),'',3)
       +ExtractFileExt(DaqFileRef(f,''))));
   bNul(FileCopy(DaqFileRef(f,'')+' '
       +AddPathDelim(DaqFileRef(AdaptFileName(ReadIni('[&DATSRV] DataPath')),''))
       +ExtractFileName(DaqFileRef(f,''))+'-'+StrReplace(StrReplace(
        GetDateTime(msecnow),Dump('.'),'',3),Dump(':'),'',3)
       +ExtractFileExt(DaqFileRef(f,''))));
  end;
  f:='';
 end;
 {
 Load [CustomParameters]
 }
 procedure LoadParams;
 var f:String;
  procedure LoadTag(tag:Integer);
  var s:String;
  begin
   s:=ReadIni(f+' '+NameTag(tag));
   if Length(s)>0 then begin
    if TypeTag(tag)=1 then bNul(iSetTag(tag,Val(s)));
    if TypeTag(tag)=2 then bNul(rSetTag(tag,rVal(s)));
    if TypeTag(tag)=3 then bNul(sSetTag(tag,s));
   end;
   s:='';
  end;
  procedure LoadTagHex(tag:Integer);
  var s:String;
  begin
   s:=ReadIni(f+' '+NameTag(tag));
   if Length(s)>0 then begin
    if TypeTag(tag)=1 then bNul(iSetTag(tag,Dump2i(Hex_Decode(s))));
    if TypeTag(tag)=2 then bNul(rSetTag(tag,Dump2r(Hex_Decode(s))));
    if TypeTag(tag)=3 then bNul(sSetTag(tag,Hex_Decode(s)));
   end;
   s:='';
  end;
 begin
  f:=AdaptFileName(ReadIni('CustomFile'));
  if Length(f)>0 then begin
   f:=DaqFileRef(f,'')+' '+ReadIni('CustomSection');
   LoadTag(CMD.SOUND.tag);
   LoadTag(SL15.USET.tag); LoadTag(SL15.ISET.tag);
   LoadTag(SR60.USET.tag); LoadTag(SR60.ISET.tag);
   LoadTag(SCAN.UMIN.tag); LoadTag(SCAN.UMAX.tag);
   LoadTag(SCAN.WAIT.tag); LoadTag(SCAN.EXPO.tag);
   LoadTag(SCAN.STEP.tag); LoadTag(LVPS.UDAC.tag);
   LoadTagHex(BEAM.VFORM.tag);
   LoadTagHex(BEAM.KFORM.tag);
   LoadTagHex(BEAM.AFORM.tag);
  end;
  f:='';
 end;
 {
 Finalize GUI system.
 }
 procedure GUI_Free;
 begin
  winHvssGui:='';    winHvssITab:='';   winHvLvUPlot:='';
  winScanIPlot:='';  winSpecIPlot:='';  winLvpsUPlot:='';
  winBeamIPlot:='';  winBeamUPlot:='';  winBeamKPlot:='';
  SL15.winUPlot:=''; SL15.winIPlot:=''; SL15.winTPlot:='';
  SR60.winUPlot:=''; SR60.winIPlot:=''; SR60.winTPlot:='';
  if Val(ReadIni('AutoSaveParams'))=1 then SaveParams;
 end;
 {
 Initialize GUI system.
 }
 procedure GUI_Init;
 var i:Integer;
  procedure InitWindow(var win:String;ref,opt:String;mode:Integer);
  begin
   win:=ReadIni(ref);
   if RefFind('Window '+win)<>0 then begin
    Success(ref+'='+win);
    WinXPlot(win,opt,0,mode);
   end else begin
    Problem(ref+'=?');
    win:='';
   end;
  end;
 begin
  {
  Initialize device references...
  }
  InitDevice(SL15.devDrv, ReadIni('devSL15Drv'), 1);
  InitDevice(SR60.devDrv, ReadIni('devSR60Drv'), 1);
  {
  Initialize tags...
  }
  InitTag(CMD.HELP.tag,      ReadIni('CMD.HELP'),      1);
  InitTag(CMD.OPEN.tag,      ReadIni('CMD.OPEN'),      1);
  InitTag(CMD.ITAB.tag,      ReadIni('CMD.ITAB'),      1);
  InitTag(CMD.SOUND.tag,     ReadIni('CMD.SOUND'),     1);
  InitTag(CMD.TOOLS.tag,     ReadIni('CMD.TOOLS'),     1);
  InitTag(CMD.SMILE.tag,     ReadIni('CMD.SMILE'),     1);
  InitTag(CMD.CLOSE.tag,     ReadIni('CMD.CLOSE'),     1);
  InitTag(SL15.ONBT.tag,     ReadIni('SL15.ONBT'),     1);
  InitTag(SL15.ON.tag,       ReadIni('SL15.ON'),       1);
  InitTag(SL15.USET.tag,     ReadIni('SL15.USET'),     2);
  InitTag(SL15.UGOT.tag,     ReadIni('SL15.UGOT'),     2);
  InitTag(SL15.UDAC.tag,     ReadIni('SL15.UDAC'),     1);
  InitTag(SL15.UGET.tag,     ReadIni('SL15.UGET'),     2);
  InitTag(SL15.UADC.tag,     ReadIni('SL15.UADC'),     1);
  InitTag(SL15.ISET.tag,     ReadIni('SL15.ISET'),     2);
  InitTag(SL15.IGOT.tag,     ReadIni('SL15.IGOT'),     2);
  InitTag(SL15.IDAC.tag,     ReadIni('SL15.IDAC'),     1);
  InitTag(SL15.IGET.tag,     ReadIni('SL15.IGET'),     2);
  InitTag(SL15.IADC.tag,     ReadIni('SL15.IADC'),     1);
  InitTag(SL15.TSIC.tag,     ReadIni('SL15.TSIC'),     2);
  InitTag(SL15.TMUL.tag,     ReadIni('SL15.TMUL'),     2);
  InitTag(SL15.HRS.tag,      ReadIni('SL15.HRS'),      2);
  InitTag(SL15.HRSRESET.tag, ReadIni('SL15.HRSRESET'), 1);
  InitTag(SL15.INTERLOCK.tag,ReadIni('SL15.INTERLOCK'),1);
  InitTag(SL15.FAULT.tag,    ReadIni('SL15.FAULT'),    1);
  InitTag(SL15.CONNECT.tag,  ReadIni('SL15.CONNECT'),  1);
  InitTag(SL15.ANSTIME.tag,  ReadIni('SL15.ANSTIME'),  1);
  InitTag(SR60.ONBT.tag,     ReadIni('SR60.ONBT'),     1);
  InitTag(SR60.ON.tag,       ReadIni('SR60.ON'),       1);
  InitTag(SR60.USET.tag,     ReadIni('SR60.USET'),     2);
  InitTag(SR60.UGOT.tag,     ReadIni('SR60.UGOT'),     2);
  InitTag(SR60.UDAC.tag,     ReadIni('SR60.UDAC'),     1);
  InitTag(SR60.UGET.tag,     ReadIni('SR60.UGET'),     2);
  InitTag(SR60.UADC.tag,     ReadIni('SR60.UADC'),     1);
  InitTag(SR60.ISET.tag,     ReadIni('SR60.ISET'),     2);
  InitTag(SR60.IGOT.tag,     ReadIni('SR60.IGOT'),     2);
  InitTag(SR60.IDAC.tag,     ReadIni('SR60.IDAC'),     1);
  InitTag(SR60.IGET.tag,     ReadIni('SR60.IGET'),     2);
  InitTag(SR60.IADC.tag,     ReadIni('SR60.IADC'),     1);
  InitTag(SR60.TSIC.tag,     ReadIni('SR60.TSIC'),     2);
  InitTag(SR60.TMUL.tag,     ReadIni('SR60.TMUL'),     2);
  InitTag(SR60.HRS.tag,      ReadIni('SR60.HRS'),      2);
  InitTag(SR60.HRSRESET.tag, ReadIni('SR60.HRSRESET'), 1);
  InitTag(SR60.INTERLOCK.tag,ReadIni('SR60.INTERLOCK'),1);
  InitTag(SR60.FAULT.tag,    ReadIni('SR60.FAULT'),    1);
  InitTag(SR60.CONNECT.tag,  ReadIni('SR60.CONNECT'),  1);
  InitTag(SR60.ANSTIME.tag,  ReadIni('SR60.ANSTIME'),  1);
  InitTag(SCAN.UMIN.tag,     ReadIni('SCAN.UMIN'),     2);
  InitTag(SCAN.UMAX.tag,     ReadIni('SCAN.UMAX'),     2);
  InitTag(SCAN.WAIT.tag,     ReadIni('SCAN.WAIT'),     2);
  InitTag(SCAN.EXPO.tag,     ReadIni('SCAN.EXPO'),     2);
  InitTag(SCAN.STEP.tag,     ReadIni('SCAN.STEP'),     2);
  InitTag(SCAN.ONBT.tag,     ReadIni('SCAN.ONBT'),     1);
  InitTag(SCAN.NUMP.tag,     ReadIni('SCAN.NUMP'),     1);
  InitTag(SCAN.REST.tag,     ReadIni('SCAN.REST'),     2);
  InitTag(SCAN.PLOT.tag,     ReadIni('SCAN.PLOT'),     1);
  InitTag(SCAN.GOBT.tag,     ReadIni('SCAN.GOTO'),     1);
  InitTag(SCAN.SAVE.tag,     ReadIni('SCAN.SAVE'),     1);
  InitTag(BEAM.IBOX1.tag,    ReadIni('BEAM.IBOX1'),    2);
  InitTag(BEAM.IBOX2.tag,    ReadIni('BEAM.IBOX2'),    2);
  InitTag(BEAM.IBOX3.tag,    ReadIni('BEAM.IBOX3'),    2);
  InitTag(BEAM.IBOX4.tag,    ReadIni('BEAM.IBOX4'),    2);
  InitTag(BEAM.IBOX5.tag,    ReadIni('BEAM.IBOX5'),    2);
  InitTag(BEAM.ISUMM.tag,    ReadIni('BEAM.ISUMM'),    2);
  InitTag(BEAM.IDIAP.tag,    ReadIni('BEAM.IDIAP'),    2);
  InitTag(BEAM.IPOCK.tag,    ReadIni('BEAM.IPOCK'),    2);
  InitTag(BEAM.UADAM.tag,    ReadIni('BEAM.UADAM'),    2);
  InitTag(BEAM.VOLTS.tag,    ReadIni('BEAM.VOLTS'),    2);
  InitTag(BEAM.KQUAL.tag,    ReadIni('BEAM.KQUAL'),    2);
  InitTag(BEAM.ADDON.tag,    ReadIni('BEAM.ADDON'),    2);
  InitTag(BEAM.VFORM.tag,    ReadIni('BEAM.VFORM'),    3);
  InitTag(BEAM.KFORM.tag,    ReadIni('BEAM.KFORM'),    3);
  InitTag(BEAM.AFORM.tag,    ReadIni('BEAM.AFORM'),    3);
  InitTag(LVPS.USET.tag,     ReadIni('LVPS.USET'),     2);
  InitTag(LVPS.UDAC.tag,     ReadIni('LVPS.UDAC'),     2);
  InitTag(HVLV.PLOT.tag,     ReadIni('HVLV.PLOT'),     1);
  InitTag(HVLV.GOBT.tag,     ReadIni('HVLV.GOTO'),     1);
  InitTag(HVLV.UGOT.tag,     ReadIni('HVLV.UGOT'),     2);
  InitTag(HVLV.UGET.tag,     ReadIni('HVLV.UGET'),     2);
  InitTag(HVLV.USET.tag,     ReadIni('HVLV.USET'),     2);
  {
  Initialize values
  }
  LastIncDecTime:=0;
  bNul(iSetTag(SCAN.NUMP.tag,0));
  SmileTimer:=0;     SmileCount:=0;
  MenuClose:=-1;     MenuTools:=-1;
  SL15.ONBT.val:=0;  SL15.ON.val:=0;
  SR60.ONBT.val:=0;  SR60.ON.val:=0;
  SCAN.ONBT.val:=0;  SCAN.PLOT.val:=0;
  SCAN.GOBT.val:=0;  SCAN.SAVE.val:=0;
  HVLV.PLOT.val:=0;  HVLV.GOBT.val:=0;
  {
  Initialize windows 
  }
  InitWindow(winHvssGui,'winHvssGui','',2+4+8);
  InitWindow(winHvssITab,'winHvssITab','',4+8);
  InitWindow(winScanIPlot,'winScanIPlot','',0);
  InitWindow(winSpecIPlot,'winSpecIPlot','',0);
  InitWindow(winHvLvUPlot,'winHvLvUPlot','',0);
  InitWindow(winLvpsUPlot,'winLvpsUPlot','',0);
  InitWindow(winBeamIPlot,'winBeamIPlot','',0);
  InitWindow(winBeamUPlot,'winBeamUPlot','',0);
  InitWindow(winBeamKPlot,'winBeamKPlot','',0);
  InitWindow(SL15.winUPlot,'SL15.winUPlot','',0);
  InitWindow(SL15.winIPlot,'SL15.winIPlot','',0);
  InitWindow(SL15.winTPlot,'SL15.winTPlot','',0);
  InitWindow(SR60.winUPlot,'SR60.winUPlot','',0);
  InitWindow(SR60.winIPlot,'SR60.winIPlot','',0);
  InitWindow(SR60.winTPlot,'SR60.winTPlot','',0);
 end;
 {
 GUI polling actions
 }
 procedure GUI_Polling;
 var kqual,addon,volts,udac,uset,errors:Real;
 begin
  udac:=rGetTag(LVPS.UDAC.tag);
  if udac<LvpsMinUDac then begin udac:=LvpsMinUDac; bNul(rSetTag(LVPS.UDAC.tag,udac)); end;
  if udac>LvpsMaxUDac then begin udac:=LvpsMaxUDac; bNul(rSetTag(LVPS.UDAC.tag,udac)); end;
  uset:=LvDacToVoltage(udac);
  bNul(rSetTag(LVPS.USET.tag,uset));
  UpdateAo(ao_LvpsUSet,time,uset);
  UpdateAo(ao_LvpsUDac,time,udac);
  UpdateAo(ao_SL15VoltSet,time,rGetTag(SL15.USET.tag));
  UpdateAo(ao_SR60VoltSet,time,rGetTag(SR60.USET.tag));
  UpdateAo(ao_SL15CurrSet,time,rGetTag(SL15.ISET.tag));
  UpdateAo(ao_SR60CurrSet,time,rGetTag(SR60.ISET.tag));
  UpdateDo(do_SL15OnBtn,time,iGetTag(SL15.ONBT.tag));
  UpdateDo(do_SR60OnBtn,time,iGetTag(SR60.ONBT.tag));
  UpdateDo(do_SL15ResHr,time,iGetTag(SL15.HRSRESET.tag));
  UpdateDo(do_SR60ResHr,time,iGetTag(SR60.HRSRESET.tag));
  bNul(rSetTag(BEAM.UADAM.tag,GetAi_Yn(ai_BeamUAdam)));
  bNul(rSetTag(BEAM.IBOX1.tag,GetAi_Yn(ai_BeamIBox1)));
  bNul(rSetTag(BEAM.IBOX2.tag,GetAi_Yn(ai_BeamIBox2)));
  bNul(rSetTag(BEAM.IBOX3.tag,GetAi_Yn(ai_BeamIBox3)));
  bNul(rSetTag(BEAM.IBOX4.tag,GetAi_Yn(ai_BeamIBox4)));
  bNul(rSetTag(BEAM.IBOX5.tag,GetAi_Yn(ai_BeamIBox5)));
  bNul(rSetTag(BEAM.ISUMM.tag,GetAi_Yn(ai_BeamISumm)));
  bNul(rSetTag(BEAM.IDIAP.tag,GetAi_Yn(ai_BeamIDiap)));
  bNul(rSetTag(BEAM.IPOCK.tag,GetAi_Yn(ai_BeamIPock)));
  bNul(rSetTag(HVLV.UGOT.tag,QuantumStepSR60(rGetTag(SR60.UGOT.tag),0,False)+rGetTag(LVPS.USET.tag)));
  bNul(rSetTag(HVLV.UGET.tag,QuantumStepSR60(rGetTag(SR60.UGET.tag),0,False)+rGetTag(LVPS.USET.tag)));
  bNul(rSetTag(HVLV.USET.tag,QuantumStepSR60(rGetTag(SR60.USET.tag),0,False)+rGetTag(LVPS.USET.tag)));
  UpdateAo(ao_HvLvUGot,time,rGetTag(HVLV.UGOT.tag));
  UpdateAo(ao_HvLvUGet,time,rGetTag(HVLV.UGET.tag));
  UpdateAo(ao_HvLvUSet,time,rGetTag(HVLV.USET.tag));
  if Evar('sl15_uset',rGetTag(SL15.USET.tag)) then
  if Evar('sl15_ugot',rGetTag(SL15.UGOT.tag)) then
  if Evar('sl15_udac',iGetTag(SL15.UDAC.tag)) then
  if Evar('sl15_uget',rGetTag(SL15.UGET.tag)) then
  if Evar('sl15_uadc',iGetTag(SL15.UADC.tag)) then
  if Evar('sl15_iset',rGetTag(SL15.ISET.tag)) then
  if Evar('sl15_igot',rGetTag(SL15.IGOT.tag)) then
  if Evar('sl15_idac',iGetTag(SL15.IDAC.tag)) then
  if Evar('sl15_iget',rGetTag(SL15.IGET.tag)) then
  if Evar('sl15_iadc',iGetTag(SL15.IADC.tag)) then
  if Evar('sr60_uset',rGetTag(SR60.USET.tag)) then
  if Evar('sr60_ugot',rGetTag(SR60.UGOT.tag)) then
  if Evar('sr60_udac',iGetTag(SR60.UDAC.tag)) then
  if Evar('sr60_uget',rGetTag(SR60.UGET.tag)) then
  if Evar('sr60_uadc',iGetTag(SR60.UADC.tag)) then
  if Evar('sr60_iset',rGetTag(SR60.ISET.tag)) then
  if Evar('sr60_igot',rGetTag(SR60.IGOT.tag)) then
  if Evar('sr60_idac',iGetTag(SR60.IDAC.tag)) then
  if Evar('sr60_iget',rGetTag(SR60.IGET.tag)) then
  if Evar('sr60_iadc',iGetTag(SR60.IADC.tag)) then
  if Evar('lvps_uset',rGetTag(LVPS.USET.tag)) then
  if Evar('lvps_udac',rGetTag(LVPS.UDAC.tag)) then
  if Evar('hvlv_ugot',rGetTag(HVLV.UGOT.tag)) then
  if Evar('hvlv_uget',rGetTag(HVLV.UGET.tag)) then
  if Evar('hvlv_uset',rGetTag(HVLV.USET.tag)) then
  if Evar('uadam',rGetTag(BEAM.UADAM.tag)) then
  if Evar('ibox1',rGetTag(BEAM.IBOX1.tag)) then
  if Evar('ibox2',rGetTag(BEAM.IBOX2.tag)) then
  if Evar('ibox3',rGetTag(BEAM.IBOX3.tag)) then
  if Evar('ibox4',rGetTag(BEAM.IBOX4.tag)) then
  if Evar('ibox5',rGetTag(BEAM.IBOX5.tag)) then
  if Evar('isumm',rGetTag(BEAM.ISUMM.tag)) then
  if Evar('idiap',rGetTag(BEAM.IDIAP.tag)) then
  if Evar('ipock',rGetTag(BEAM.IPOCK.tag)) then begin
   volts:=Eval(sGetTag(BEAM.VFORM.tag));
   kqual:=Eval(sGetTag(BEAM.KFORM.tag));
   addon:=Eval(sGetTag(BEAM.AFORM.tag));
   bNul(rSetTag(BEAM.VOLTS.tag,volts));
   bNul(rSetTag(BEAM.KQUAL.tag,kqual));
   bNul(rSetTag(BEAM.ADDON.tag,addon));
   if not IsNan(volts) and not IsInf(volts) then UpdateDo(do_BeamVolts,time,volts); 
   if not IsNan(kqual) and not IsInf(kqual) then UpdateDo(do_BeamKQual,time,kqual);
   if not IsNan(addon) and not IsInf(addon) then UpdateDo(do_BeamAddon,time,addon);
  end;
  if mSecNow>SmileTimer+SmilePeriod then begin
   errors:=GetErrCount(-2);
   if errors>SmileCount then bNul(iSetTag(CMD.SMILE.tag,2)) else
   if iGetTag(CMD.SMILE.tag)>1 then bNul(iSetTag(CMD.SMILE.tag,1));
   SmileTimer:=mSecNow;
   SmileCount:=errors;
  end;
 end;
 {
 Set HV+LV summary voltage.
 }
 procedure SetHvLvVoltage(Volt:Real);
 var uset,vgot,vdif,udac:Real;
 begin
  uset:=QuantumStepSR60(Volt,0,True);
  vgot:=QuantumStepSR60(uset,0,False);
  udac:=LvVoltageToDac(Volt-vgot);
  bNul(rSetTag(SR60.USET.tag,uset));
  bNul(rSetTag(LVPS.UDAC.tag,udac));
 end;
 {
 Return how many points scanned.
 }
 function GetScanNumPoints:Integer;
 begin
  GetScanNumPoints:=Round(CrvLen(RefAo(ao_ScanISumm)));
 end;
 {
 Estimate time range to make scan.
 }
 function GetScanDuration:Real;
 var cmin,cmax:Integer; step:Real;
 begin
  step:=(rGetTag(SCAN.WAIT.tag)+rGetTag(SCAN.EXPO.tag))*1000/TimeUnits;
  if rGetTag(SCAN.STEP.tag)>0 then begin
   cmin:=Truncate(rGetTag(SCAN.UMIN.tag)/rGetTag(SCAN.STEP.tag));
   cmax:=Truncate(rGetTag(SCAN.UMAX.tag)/rGetTag(SCAN.STEP.tag));
  end else begin
   cmin:=VoltageToCodeSR60(rGetTag(SCAN.UMIN.tag));
   cmax:=VoltageToCodeSR60(rGetTag(SCAN.UMAX.tag));
  end;
  GetScanDuration:=Max(0,(cmax-cmin+1)*step);
 end; 
 {
 Estimate time rest to make scan.
 }
 function GetScanRestSec:Real;
 var cmin,cmax,cgot:Integer; step:Real;
 begin
  step:=rGetTag(SCAN.WAIT.tag)+rGetTag(SCAN.EXPO.tag);
  if rGetTag(SCAN.STEP.tag)>0 then begin
   cgot:=Truncate(rGetTag(HVLV.UGOT.tag)/rGetTag(SCAN.STEP.tag));
   cmin:=Truncate(rGetTag(SCAN.UMIN.tag)/rGetTag(SCAN.STEP.tag));
   cmax:=Truncate(rGetTag(SCAN.UMAX.tag)/rGetTag(SCAN.STEP.tag));
  end else begin
   cgot:=VoltageToCodeSR60(rGetTag(SR60.UGOT.tag));
   cmin:=VoltageToCodeSR60(rGetTag(SCAN.UMIN.tag));
   cmax:=VoltageToCodeSR60(rGetTag(SCAN.UMAX.tag));
  end;
  if (cgot<cmin) or (iGetTag(SCAN.ONBT.tag)=0) then cgot:=cmin;
  GetScanRestSec:=Int(Max(0,(cmax-cgot+1)*step));
 end; 
 {
 Save scanned data to CRW file.
 }
 function SaveScanPlot(win,prefix:String;mode:Integer):Boolean;
 var fname:String; Ok:Boolean;
 begin
  fname:='';
  Ok:=False;
  if Length(win)>0 then
  if GetScanNumPoints>0 then begin
   fname:=AdaptFileName(ReadIni('['+DatSrv+'] DataPath'));
   if Length(fname)=0 then fname:=ParamStr('DaqDataPath');
   fname:=DaqFileRef(AddPathDelim(fname)+prefix+Str(GetDateAsNumber(mSecNow)),'.crw');
   Ok:=WinDraw(win+'|SaveCrw='+fname);
   if Ok
   then Success(win+' was saved to file '+fname)
   else Problem('Could not save '+win+' to '+fname);
  end else begin
   if mode>0 then
   Warning('Нет данных для сохранения! Сначала надо провести сканирование!');
  end;
  fname:='';
  SaveScanPlot:=Ok;
 end;
 {
 Handle scan point.
 }
 procedure ScanProcessing(tmin,tmax,volt:Real);
 var IBox1,IBox2,IBox3,IBox4,IBox5,ISumm,IDiap,IPock,Volts,KQual,Addon:Real;
  procedure UpdateAo(n:Integer;x,y:Real);
  begin
   bNul(CrvIns(RefAo(n),maxint,x,y)>0);
  end;
  function GetValue(crv:Integer):Real;
  var a,b,c,y:Real;
  begin
   bNul(CrvLock(crv));
   if CrvLen(crv)=0 then GetValue:=0 else begin
    a:=Max(tmin,CrvX(crv,1));
    b:=Min(tmax,CrvX(crv,CrvLen(crv)));
    c:=CrvY(crv,CrvLen(crv));
    if a>=b then GetValue:=c else begin
     y:=CrvInteg(crv,a,b)/(b-a);
     Details('     '+CrvName(crv)+' = '+StrFix(y,8,3));
     GetValue:=y;
    end;
   end;
   bNul(CrvUnLock(crv));
  end;
 begin
  Details('Scan '+Str(volt)+' Volt, '+StrFix((tmax-tmin)*TimeUnits,1,0)+' ms, since '+StrFix(tmin,1,3));
  IBox1:=GetValue(RefAi(ai_BeamIBox1));
  IBox2:=GetValue(RefAi(ai_BeamIBox2));
  IBox3:=GetValue(RefAi(ai_BeamIBox3));
  IBox4:=GetValue(RefAi(ai_BeamIBox4));
  IBox5:=GetValue(RefAi(ai_BeamIBox5));
  ISumm:=GetValue(RefAi(ai_BeamISumm));
  IDiap:=GetValue(RefAi(ai_BeamIDiap));
  IPock:=GetValue(RefAi(ai_BeamIPock));
  Volts:=GetValue(RefDo(do_BeamVolts));
  KQual:=GetValue(RefDo(do_BeamKQual));
  Addon:=GetValue(RefDo(do_BeamAddon));
  UpdateAo(ao_ScanIBox1,volt,IBox1);  UpdateAo(ao_SpecIBox1,Volts,IBox1);
  UpdateAo(ao_ScanIBox2,volt,IBox2);  UpdateAo(ao_SpecIBox2,Volts,IBox2);
  UpdateAo(ao_ScanIBox3,volt,IBox3);  UpdateAo(ao_SpecIBox3,Volts,IBox3);
  UpdateAo(ao_ScanIBox4,volt,IBox4);  UpdateAo(ao_SpecIBox4,Volts,IBox4);
  UpdateAo(ao_ScanIBox5,volt,IBox5);  UpdateAo(ao_SpecIBox5,Volts,IBox5);
  UpdateAo(ao_ScanISumm,volt,ISumm);  UpdateAo(ao_SpecISumm,Volts,ISumm);
  UpdateAo(ao_ScanIDiap,volt,IDiap);  UpdateAo(ao_SpecIDiap,Volts,IDiap);
  UpdateAo(ao_ScanIPock,volt,IPock);  UpdateAo(ao_SpecIPock,Volts,IPock);
  UpdateAo(ao_ScanKQual,volt,KQual);  UpdateAo(ao_SpecKQual,Volts,KQual);
  UpdateAo(ao_ScanAddon,volt,Addon);  UpdateAo(ao_SpecAddon,Volts,Addon);
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  winHvssGui:='';    winHvssITab:='';   winHvLvUPlot:='';
  winScanIPlot:='';  winSpecIPlot:='';  winLvpsUPlot:='';
  winBeamIPlot:='';  winBeamUPlot:='';  winBeamKPlot:=''; 
  SL15.winUPlot:=''; SL15.winIPlot:=''; SL15.winTPlot:='';
  SR60.winUPlot:=''; SR60.winIPlot:=''; SR60.winTPlot:='';
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  GUI_Init;
  TestConvertion;
  RunStartupScript;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  GUI_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 var s:String;
 begin
  s:='';
  {
  Load params on startup...
  }
  if RunCount=2 then if Val(ReadIni('AutoLoadParams'))=1 then LoadParams;
  {
  GUI polling actions
  }
  GUI_Polling;
  {
  Execute GUI commands.
  }
  if iGetTag(CMD.HELP.tag)<>0 then begin
   Cron('@Browse '+DaqFileRef(AdaptFileName(ReadIni('[DAQ] HelpFile')),'.htm'));
   bNul(iSetTag(CMD.HELP.tag,0));
  end;
  if iGetTag(CMD.OPEN.tag)<>0 then begin
   Cron('@FileOpenDialog '+
    URL_Packed(AddPathDelim(DaqFileRef(AdaptFileName(ReadIni('['+DatSrv+'] DataPath')),''))+'*.dat;*.crw'));
   bNul(iSetTag(CMD.OPEN.tag,0));
  end;
  if iGetTag(CMD.ITAB.tag)<>0 then begin
   WinXPlot(winHvssITab,'',0,2+4+8);
   bNul(iSetTag(CMD.ITAB.tag,0));
  end;
  if iGetTag(CMD.TOOLS.tag)<>0 then begin
   if editstate=0 then begin
    if pos('?',edit('(Вызвать утилиту...')
              +edit(' Что выбираете:')
              +edit(' '+MenuToolsItem(0))
              +edit(' '+MenuToolsItem(1))
              +edit(')MenuList MENU_'+NameTag(CMD.TOOLS.tag)))>0
    then Warning('Error initializing MenuList!');
   end else Warning('Cannot edit right now!');
   bNul(iSetTag(CMD.TOOLS.tag,0));
  end;
  if iGetTag(CMD.CLOSE.tag)<>0 then begin
   if editstate=0 then begin
    if pos('?',edit('(Команда "Закрыть"... ')
              +edit(' Что выбираете:')
              +edit(' '+MenuCloseItem(0))
              +edit(' '+MenuCloseItem(1))
              +edit(')MenuList MENU_'+NameTag(CMD.CLOSE.tag)))>0
    then Warning('Error initializing MenuList!');
   end else Warning('Cannot edit right now!');
   bNul(iSetTag(CMD.CLOSE.tag,0));
   MenuClose:=-1;
  end;
  {
  Handle HV ON/OFF
  }
  if ShouldRefresh(SL15.ON.val,iGetTag(SL15.ON.tag))>0 then begin
   if SL15.ON.val>0 then begin
    Speak('Подано высокое SL15.');
   end else begin
    Speaker('Снято высокое SL15.');
   end;
  end;
  if ShouldRefresh(SR60.ON.val,iGetTag(SR60.ON.tag))>0 then begin
   if SR60.ON.val>0 then begin
    Speaker('Подано высокое СР60.');
   end else begin
    Speaker('Снято высокое СР60.');
   end;
  end;
  {
  Handle SCAN ON/OFF
  }
  if ShouldRefresh(SCAN.ONBT.val,iGetTag(SCAN.ONBT.tag))>0 then begin
   if SCAN.ONBT.val>0 then begin // Start scan
    Speaker('Скан запущен.');
    bNul(iSetTag(SCAN.NUMP.tag,0));
    Success('Start scan at '+GetDateTime(mSecNow));
    SCAN.WAIT.val:=mSecNow+1000*rGetTag(SCAN.WAIT.tag);
    SCAN.EXPO.val:=SCAN.WAIT.val+1000*rGetTag(SCAN.EXPO.tag);
    if rGetTag(SCAN.STEP.tag)>0 then begin
     SetHvLvVoltage(rGetTag(SCAN.UMIN.tag));
    end else begin
     bNul(rSetTag(SR60.USET.tag,QuantumStepSR60(rGetTag(SCAN.UMIN.tag),0,True)));
     bNul(rSetTag(LVPS.UDAC.tag,LvVoltageToDac(Abs(rGetTag(SCAN.STEP.tag)))));
    end;
    ClearCurveData(RefAo(ao_ScanIBox1)); ClearCurveData(RefAo(ao_SpecIBox1));
    ClearCurveData(RefAo(ao_ScanIBox2)); ClearCurveData(RefAo(ao_SpecIBox2));
    ClearCurveData(RefAo(ao_ScanIBox3)); ClearCurveData(RefAo(ao_SpecIBox3));
    ClearCurveData(RefAo(ao_ScanIBox4)); ClearCurveData(RefAo(ao_SpecIBox4));
    ClearCurveData(RefAo(ao_ScanIBox5)); ClearCurveData(RefAo(ao_SpecIBox5));
    ClearCurveData(RefAo(ao_ScanISumm)); ClearCurveData(RefAo(ao_SpecISumm));
    ClearCurveData(RefAo(ao_ScanIDiap)); ClearCurveData(RefAo(ao_SpecIDiap));
    ClearCurveData(RefAo(ao_ScanIPock)); ClearCurveData(RefAo(ao_SpecIPock));
    ClearCurveData(RefAo(ao_ScanKQual)); ClearCurveData(RefAo(ao_SpecKQual));
    ClearCurveData(RefAo(ao_ScanAddon)); ClearCurveData(RefAo(ao_SpecAddon));
    WinXPlot(winSpecIPlot,GetRangeStr(rGetTag(SCAN.UMIN.tag),0,rGetTag(SCAN.UMAX.tag),IBoxMaxCurr),0,0);
    WinXPlot(winScanIPlot,GetRangeStr(rGetTag(SCAN.UMIN.tag),0,rGetTag(SCAN.UMAX.tag),IBoxMaxCurr),0,0);
    WinXPlot(winHvLvUPlot,GetRangeStr(Time,rGetTag(SCAN.UMIN.tag),
                                       Time+GetScanDuration,rGetTag(SCAN.UMAX.tag)),HVLV.UGOT.tag,2);
    WinXPlot(SR60.winUPlot,GetRangeStr(Time,rGetTag(SCAN.UMIN.tag),
                                       Time+GetScanDuration,rGetTag(SCAN.UMAX.tag)),SR60.UGOT.tag,2);
    if rGetTag(SCAN.STEP.tag)>0 then bNul(WinSelect(winHvLvUPlot));
    WinXPlot(winBeamIPlot,'|Roi=_Nan;_Nan;_Nan;_Nan',RefAi(ai_BeamISumm),2);
    WinXPlot(winBeamIPlot,GetRangeStr(Time,0,Time+GetScanDuration,IBoxMaxCurr),0,2);
    ClearRoiMarkers;
   end else begin // Stop scan
    ClearRoiMarkers; 
    Speaker('Скан завершен.');
    Success('Stop scan at '+GetDateTime(mSecNow));
    if GetScanNumPoints>0 then begin
     WinXPlot(winSpecIPlot,'|Range=xmin-(xmax-xmin)/20;0;xmax+(xmax-xmin)/20;ymax*1.2'+
                           '|Roi=_Nan;_Nan;_Nan;_Nan',RefAo(ao_SpecISumm),2);
     WinXPlot(winScanIPlot,'|Range=xmin-(xmax-xmin)/20;0;xmax+(xmax-xmin)/20;ymax*1.2'+
                           '|Roi=_Nan;_Nan;_Nan;_Nan',RefAo(ao_ScanISumm),2);
     if SaveScanPlot(winScanIPlot,AdaptFileName(ReadIni('ScanPrefix')),0)
     then Speaker('Скан сохранен в Файл.') else Speaker('Ошибка сохранения Скана.');
     if SaveScanPlot(winSpecIPlot,AdaptFileName(ReadIni('SpecPrefix')),0)
     then Speaker('Спектр сохранен в Файл.') else Speaker('Ошибка сохранения Спектра.');
    end;
   end;
  end;
  if SCAN.ONBT.val>0 then begin // Scan in process
   if iGetTag(SR60.ONBT.tag)=0 then begin // No HV, should stop
    Warning('Не могу делать скан, отключено высокое на SR60.');
    Speaker('Высокое напряжение СР60 не включено!');
    Success('Scan break, SR60 HV is OFF.');
    bNul(iSetTag(SCAN.ONBT.tag,0));
   end else
   if rGetTag(SR60.USET.tag)>=rGetTag(SCAN.UMAX.tag) then begin
    bNul(iSetTag(SCAN.ONBT.tag,0));
    Success('Scan done.');
   end else
   if mSecNow>SCAN.EXPO.val then begin
    bNul(iSetTag(SCAN.NUMP.tag,iGetTag(SCAN.NUMP.tag)+1));
    if rGetTag(SCAN.EXPO.tag)>0 then begin
     if rGetTag(SCAN.STEP.tag)>0
     then ScanProcessing((SCAN.WAIT.val-TimeBase)/TimeUnits,(SCAN.EXPO.val-TimeBase)/TimeUnits,rGetTag(HVLV.UGOT.tag))
     else ScanProcessing((SCAN.WAIT.val-TimeBase)/TimeUnits,(SCAN.EXPO.val-TimeBase)/TimeUnits,rGetTag(SR60.UGOT.tag));
    end;
    SCAN.WAIT.val:=mSecNow+1000*rGetTag(SCAN.WAIT.tag);
    SCAN.EXPO.val:=SCAN.WAIT.val+1000*rGetTag(SCAN.EXPO.tag);
    if rGetTag(SCAN.STEP.tag)>0 then begin
     SetHvLvVoltage(rGetTag(SCAN.UMIN.tag)+iGetTag(SCAN.NUMP.tag)*rGetTag(SCAN.STEP.tag));
    end else begin
     bNul(rSetTag(SR60.USET.tag,QuantumStepSR60(rGetTag(SR60.USET.tag),+1,True)));
     bNul(rSetTag(LVPS.UDAC.tag,LvVoltageToDac(Abs(rGetTag(SCAN.STEP.tag)))));
    end;
   end;
  end;
  bNul(rSetTag(SCAN.REST.tag,GetScanRestSec));
  if iGetTag(SCAN.PLOT.tag)<>0 then begin
   if GetScanNumPoints>0 then begin
    WinXPlot(winScanIPlot,'|Roi=_Nan;_Nan;_Nan;Nan',RefAo(ao_ScanISumm),1);
    WinXPlot(winScanIPlot,'|Range=xmin-0.05*(xmax-xmin);0;xmax+0.05*(xmax-xmin);ymax*1.05',0,2);
   end else begin
    Warning('Нет данных для графика! Сначала проведите сканирование!');
    Speaker('Нету данных. Проведите скан.');
   end;
   ClearRoiMarkers;
   bNul(iSetTag(SCAN.PLOT.tag,0));
  end;
  if iGetTag(HVLV.PLOT.tag)<>0 then begin
   if GetScanNumPoints>0 then begin
    WinXPlot(winSpecIPlot,'|Roi=_Nan;_Nan;_Nan;Nan',RefAo(ao_SpecISumm),1);
    WinXPlot(winSpecIPlot,'|Range=xmin-0.05*(xmax-xmin);0;xmax+0.05*(xmax-xmin);ymax*1.05',0,2);
   end else begin
    Warning('Нет данных для графика! Сначала проведите сканирование!');
    Speaker('Нету данных. Проведите скан.');
   end;
   ClearRoiMarkers;
   bNul(iSetTag(HVLV.PLOT.tag,0));
  end;
  if iGetTag(SCAN.GOBT.tag)<>0 then begin
   if GetScanNumPoints+1>0 then begin
    Speaker('Хотите установить высокое по маркеру РОИ?');
    GoToRoiSR60;
   end else begin
    Warning('Перед тем как делать GOTO, сначала надо провести сканирование!');
    Speaker('Нету данных. Проведите скан.');
   end; 
   ClearRoiMarkers;
   bNul(iSetTag(SCAN.GOBT.tag,0));
  end;
  if iGetTag(HVLV.GOBT.tag)<>0 then begin
   if GetScanNumPoints+1>0 then begin
    Speaker('Хотите установить высокое по маркеру РОИ?');
    GoToRoiHvLv;
   end else begin
    Warning('Перед тем как делать GOTO, сначала надо провести сканирование!');
    Speaker('Нету данных. Проведите скан.');
   end; 
   ClearRoiMarkers;
   bNul(iSetTag(HVLV.GOBT.tag,0));
  end;
  if iGetTag(SCAN.SAVE.tag)<>0 then begin
   if GetScanNumPoints>0 then begin
    Speaker('Хотите сохранить скан в файле?');
    if Pos('?',edit('(Задайте имя файла...')
              +edit(' '+DaqFileRef(AddPathDelim(AdaptFileName(ReadIni('['+DatSrv+'] DataPath')))+'*','.crw'))
              +edit(' CRW files (*.crw)|*.crw|')
              +edit(' All files (*.*)|*.*|')
              +edit(')FileOpenDialog SaveScanPlot'))>0
    then Problem('Error file open dialog!');
   end else begin
    Warning('Перед тем как сохранять, сначала надо провести сканирование!');
    Speaker('Нету данных. Проведите скан.');
   end; 
   bNul(iSetTag(SCAN.SAVE.tag,0));
  end;
  {
  Edit tags...
  }
  if editstate=ef_Done then begin
   if CheckEditTag(SL15.USET.tag,s) then UpdateTag(SL15.USET.tag, s, 0, SL15MaxVolt);
   if CheckEditTag(SL15.ISET.tag,s) then UpdateTag(SL15.ISET.tag, s, 0, SL15MaxCurr);
   if CheckEditTag(SR60.USET.tag,s) then begin
    UpdateTag(SR60.USET.tag, s, 0, SR60MaxVolt);
    bNul(rSetTag(LVPS.UDAC.tag,0));
   end;
   if CheckEditTag(SR60.ISET.tag,s) then UpdateTag(SR60.ISET.tag, s, 0, SR60MaxCurr);
   if CheckEditTag(SCAN.UMIN.tag,s) then UpdateTag(SCAN.UMIN.tag, s, 0, SR60MaxVolt);
   if CheckEditTag(SCAN.UMAX.tag,s) then UpdateTag(SCAN.UMAX.tag, s, 0, SR60MaxVolt);
   if CheckEditTag(SCAN.WAIT.tag,s) then UpdateTag(SCAN.WAIT.tag, s, 0, WaitMaxTime);
   if CheckEditTag(SCAN.EXPO.tag,s) then UpdateTag(SCAN.EXPO.tag, s, 0, ExpoMaxTime);
   if CheckEditTag(SCAN.STEP.tag,s) then UpdateTag(SCAN.STEP.tag, s, -LvDacToVoltage(LvpsMaxUDac), SR60MaxVolt);
   if CheckEditTag(BEAM.VFORM.tag,s) then UpdateTag(BEAM.VFORM.tag, UpCaseStr(Trim(s)), 0, 0);
   if CheckEditTag(BEAM.KFORM.tag,s) then UpdateTag(BEAM.KFORM.tag, UpCaseStr(Trim(s)), 0, 0);
   if CheckEditTag(BEAM.AFORM.tag,s) then UpdateTag(BEAM.AFORM.tag, UpCaseStr(Trim(s)), 0, 0);
   if CheckEditTag(LVPS.UDAC.tag,s) then UpdateTag(LVPS.UDAC.tag, s, LvpsMinUDac, LvpsMaxUDac);
   if CheckEditTag(LVPS.USET.tag,s) then UpdateTag(LVPS.UDAC.tag,Str(LvVoltageToDac(rVal(s))),LvpsMinUDac,LvpsMaxUDac);
   if CheckEditTag(HVLV.USET.tag,s) then SetHvLvVoltage(rVal(s));
   {
   Warning.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Warning') then begin
    s:=edit('');
   end;
   {
   Information.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Information') then begin
    s:=edit('');
   end;
   {
   SaveScanPlot.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'SaveScanPlot') then begin
    if Val(ExtractWord(2,edit('?ans 0')))=1 then begin
     if WinDraw(winScanIPlot+'|SaveCrw='+edit('?ans '+Str(1))) and
        WinDraw(winSpecIPlot+'|SaveCrw='+edit('?ans '+Str(1)))
     then Speaker('Файл сохранен.')
     else Speaker('Ошибка записи.');
    end;
    s:=edit('');
   end;
   {
   TOOLS menu.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'MENU_'+NameTag(CMD.TOOLS.tag)) then begin
    if Val(ExtractWord(2,edit('?ans 0')))=1 then begin
     MenuToolsCmnd(MenuToolsItem(Val(edit('?ans 1'))));
    end;
    s:=edit('');
   end;
   {
   CLOSE menu.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'MENU_'+NameTag(CMD.CLOSE.tag)) then begin
    if Val(ExtractWord(2,edit('?ans 0')))=1 then begin
     MenuClose:=Val(edit('?ans 1'));
     if MenuClose>=0 then
     if (MenuClose>2) then begin
      DevSendCmd(devMySelf,'@MenuCloseConfirmation '+UpCaseStr(StrReplace(StrReplace(
                  MenuCloseItem(MenuClose),'  ',Dump(' '),3),'  ',Dump(' '),3)));
     end else begin
      MenuCloseCmnd(MenuClose);
      MenuClose:=-1;
     end;
    end;
    s:=edit('');
   end;
   {
   CLOSE menu (after confirmation).
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'YesNo_'+NameTag(CMD.CLOSE.tag)) then begin
    if MenuClose>=0 then
    if Val(ExtractWord(2,edit('?ans 0')))=6 then MenuCloseCmnd(MenuClose);
    MenuClose:=-1;
    s:=edit('');
   end;
  end;
  if editstate=ef_Done then begin
   writeln('Unknown tag edition!');
   s:=edit('');
  end;
  if iAnd(editstate,ef_ErrorFound)<>0 then begin
   writeln('Dialog error detected!');
   s:=edit('');
  end;
  {
  Handle left button clicks...
  }
  if ClickButton=1 then begin
   {
   Handle clicks in HVSS.GUI window.
   }
   if IsSameText(ClickParams('Window'),winHvssGui)
   or IsSameText(ClickParams('Window'),winHvssITab) then begin
    {
    Toolbar buttons to run on local host...
    }
    ClickBitXorLocal(CMD.HELP.tag,1);
    ClickBitXorLocal(CMD.OPEN.tag,1);
    ClickBitXorLocal(CMD.ITAB.tag,1);
    ClickBitXorLocal(CMD.SOUND.tag,1);
    ClickBitXorLocal(CMD.TOOLS.tag,1);
    ClickBitXorLocal(CMD.CLOSE.tag,1);
    ClickBitXorLocal(SL15.ONBT.tag,1);
    ClickBitXorLocal(SL15.HRSRESET.tag,1);
    ClickBitXorLocal(SL15.CONNECT.tag,1);
    ClickBitXorLocal(SR60.ONBT.tag,1);
    ClickBitXorLocal(SR60.HRSRESET.tag,1);
    ClickBitXorLocal(SR60.CONNECT.tag,1);
    ClickBitXorLocal(SCAN.ONBT.tag,1);
    ClickBitXorLocal(SCAN.PLOT.tag,1);
    ClickBitXorLocal(SCAN.GOBT.tag,1);
    ClickBitXorLocal(SCAN.SAVE.tag,1);
    ClickBitXorLocal(HVLV.PLOT.tag,1);
    ClickBitXorLocal(HVLV.GOBT.tag,1);
    {
    Smile face button...
    }
    if ClickTag=CMD.SMILE.tag then begin
     bNul(Eval('@System @Async @Menu run FormDaqControlDialog.ActionDaqStatus')>0);
     bNul(iSetTag(CMD.SMILE.tag,0));
     bNul(Voice(snd_Click));
    end;
    {
    Inc,Dec buttons
    }
    if mSecNow>LastIncDecTime+rGetTag(SCAN.WAIT.tag)*1000 then begin
     if IsSameText(ClickSensor,NameTag(SL15.USET.tag)+'.INC') then begin
      bNul(rSetTag(SL15.USET.tag,QuantumStepSL15(rGetTag(SL15.USET.tag),+1,True)));
      LastIncDecTime:=mSecNow;
      bNul(Voice(snd_Inc));
     end;
     if IsSameText(ClickSensor,NameTag(SL15.USET.tag)+'.DEC') then begin
      bNul(rSetTag(SL15.USET.tag,QuantumStepSL15(rGetTag(SL15.USET.tag),-1,True)));
      LastIncDecTime:=mSecNow;
      bNul(Voice(snd_Dec));
     end;
     if SCAN.ONBT.val>0 then bNul(Voice(snd_Fails)) else begin
      if IsSameText(ClickSensor,NameTag(SR60.USET.tag)+'.INC') then begin
       bNul(rSetTag(SR60.USET.tag,QuantumStepSR60(rGetTag(SR60.USET.tag),+1,True)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Inc));
      end;
      if IsSameText(ClickSensor,NameTag(SR60.USET.tag)+'.DEC') then begin
       bNul(rSetTag(SR60.USET.tag,QuantumStepSR60(rGetTag(SR60.USET.tag),-1,True)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Dec));
      end;
      if IsSameText(ClickSensor,NameTag(HVLV.USET.tag)+'.INC') then begin
       SetHvLvVoltage(Truncate(Max(0,Round(rGetTag(HVLV.USET.tag))+1)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Inc));
      end;
      if IsSameText(ClickSensor,NameTag(HVLV.USET.tag)+'.DEC') then begin
       SetHvLvVoltage(Truncate(Max(0,Round(rGetTag(HVLV.USET.tag))-1)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Dec));
      end;
      if IsSameText(ClickSensor,NameTag(HVLV.USET.tag)+'.INC5') then begin
       SetHvLvVoltage(Truncate(Max(0,Round(rGetTag(HVLV.USET.tag))+5)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Inc));
      end;
      if IsSameText(ClickSensor,NameTag(HVLV.USET.tag)+'.DEC5') then begin
       SetHvLvVoltage(Truncate(Max(0,Round(rGetTag(HVLV.USET.tag))-5)));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Dec));
      end;
      if IsSameText(ClickSensor,NameTag(LVPS.USET.tag)+'.ZERO') then begin
       bNul(rSetTag(LVPS.UDAC.tag,LvpsMinUDac));
       LastIncDecTime:=mSecNow;
       bNul(Voice(snd_Inc));
      end;
     end;
    end;
    {
    Open plot window
    }
    if IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIBox1)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIBox2)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIBox3)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIBox4)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIBox5)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamISumm)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIDiap)))
    or IsSameText(ClickSensor,CrvName(RefAi(ai_BeamIPock)))
    then begin
     winXPlot(winBeamIPlot,'',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if IsSameText(ClickSensor,CrvName(RefAi(ai_BeamUAdam)))
    or IsSameText(ClickSensor,CrvName(RefDo(do_BeamVolts)))
    then begin
     winXPlot(winBeamUPlot,'',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if IsSameText(ClickSensor,CrvName(RefAo(ao_HvLvUGot)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_HvLvUGet)))
    then begin
     winXPlot(winHvLvUPlot,'',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if IsSameText(ClickSensor,CrvName(RefDo(do_BeamKQual)))
    or IsSameText(ClickSensor,CrvName(RefDo(do_BeamAddon)))
    then begin
     winXPlot(winBeamKPlot,'|Range=xmax-0.5;0;xmax+0.5;ymax*1.2',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIBox1)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIBox2)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIBox3)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIBox4)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIBox5)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanISumm)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIDiap)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanIPock)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanKQual)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_ScanAddon)))
    then begin
     WinXPlot(winScanIPlot,'|Range=xmin-(xmax-xmin)/20;0;xmax+(xmax-xmin)/20;ymax*1.2'+
                           '|Roi=_Nan;_Nan;_Nan;Nan',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIBox1)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIBox2)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIBox3)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIBox4)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIBox5)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecISumm)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIDiap)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecIPock)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecKQual)))
    or IsSameText(ClickSensor,CrvName(RefAo(ao_SpecAddon)))
    then begin
     WinXPlot(winSpecIPlot,'|Range=xmin-(xmax-xmin)/20;0;xmax+(xmax-xmin)/20;ymax*1.2'+
                           '|Roi=_Nan;_Nan;_Nan;Nan',RefFind('Curve '+ClickParams('Curve')),2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SL15.UGOT.tag) or (ClickTag=SL15.UGET.tag)
    or (ClickTag=SL15.UDAC.tag) or (ClickTag=SL15.UADC.tag)
    then begin
     winXPlot(SL15.winUPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SL15.IGOT.tag) or (ClickTag=SL15.IGET.tag)
    or (ClickTag=SL15.IDAC.tag) or (ClickTag=SL15.IADC.tag)
    then begin
     winXPlot(SL15.winIPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SL15.TSIC.tag) or (ClickTag=SL15.TMUL.tag)
    then begin
     winXPlot(SL15.winTPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SR60.UGOT.tag) or (ClickTag=SR60.UGET.tag)
    or (ClickTag=SR60.UDAC.tag) or (ClickTag=SR60.UADC.tag)
    then begin
     winXPlot(SR60.winUPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SR60.IGOT.tag) or (ClickTag=SR60.IGET.tag)
    or (ClickTag=SR60.IDAC.tag) or (ClickTag=SR60.IADC.tag)
    then begin
     winXPlot(SR60.winIPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    if (ClickTag=SR60.TSIC.tag) or (ClickTag=SR60.TMUL.tag)
    then begin
     winXPlot(SR60.winTPlot,'',ClickTag,2);
     bNul(Voice(snd_Wheel));
    end;
    {
    Edit parameters
    }
    if ClickTag=SL15.USET.tag then begin
     StartEditTag(ClickTag,'Уставка напряжения SL15, В ');
     Speaker('Задайте высокое напряжение SL15.');
     bNul(Voice(snd_Click));
    end;
    if ClickTag=SL15.ISET.tag then begin
     StartEditTag(ClickTag,'Уставка макс. тока SL15, мА ');
     Speaker('Задайте максимальный ток SL15.');
     bNul(Voice(snd_Click));
    end;
    if SCAN.ONBT.val>0 then bNul(Voice(snd_Fails)) else begin
     if ClickTag=SR60.USET.tag then begin
      StartEditTag(ClickTag,'Уставка напряжения SR60, В ');
      Speaker('Задайте высокое напряжение СР60.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SR60.ISET.tag then begin
      StartEditTag(ClickTag,'Уставка макс. тока SR60, мА ');
      Speaker('Задайте максимальный ток СР60.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SCAN.UMIN.tag then begin
      StartEditTag(ClickTag,'Начало сканирования SR60, В ');
      Speaker('Задайте начало скана СР60.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SCAN.UMAX.tag then begin
      StartEditTag(ClickTag,'Конец сканирования SR60, В ');
      Speaker('Задайте конец скана СР60.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SCAN.WAIT.tag then begin
      StartEditTag(ClickTag,'Задержка перед измерением точек, сек ');
      Speaker('Задайте задержку измерения точек.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SCAN.EXPO.tag then begin
      StartEditTag(ClickTag,'Время на измерение точки, сек ');
      Speaker('Задайте время измерения точек.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=SCAN.STEP.tag then begin
      StartEditTag(ClickTag,'Шаг сканирования, Вольт ');
      Speaker('Задайте шаг сканирования, Вольт.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=LVPS.USET.tag then begin
      StartEditTag(ClickTag,'Уставка напряжения низковольтного источника, В ');
      Speaker('Задайте низкое напряжение.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=LVPS.UDAC.tag then begin
      StartEditTag(ClickTag,'Уставка ЦАП низковольтного источника, мВ ');
      Speaker('Задайте ЦАП низковольтного источника.');
      bNul(Voice(snd_Click));
     end;
     if ClickTag=HVLV.USET.tag then begin
      StartEditTag(ClickTag,'Уставка суммарного напряжения HV+LV, В ');
      Speaker('Задайте сумму напряжений источников высокого и низкого питания.');
      bNul(Voice(snd_Click));
     end;
    end;
    if IsSameText(ClickSensor,NameTag(BEAM.VOLTS.tag)+'.LAB') then begin
     StartEditTag(BEAM.VFORM.tag,'Формула для напряжения пучка (Voltage) ');
     Speaker('Задайте формулу для напряжения пучка.');
     bNul(Voice(snd_Click));
    end;
    if IsSameText(ClickSensor,NameTag(BEAM.KQUAL.tag)+'.LAB') then begin
     StartEditTag(BEAM.KFORM.tag,'Формула для критерия качества (Kriteria of Quality) ');
     Speaker('Задайте формулу для критерия, качества пучка.');
     bNul(Voice(snd_Click));
    end;
    if IsSameText(ClickSensor,NameTag(BEAM.ADDON.tag)+'.LAB') then begin
     StartEditTag(BEAM.AFORM.tag,'Формула для дополнительного параметра (Addon) ');
     Speaker('Задайте формулу для дополнительного параметра пучка.');
     bNul(Voice(snd_Click));
    end;
   end;
  end;
  {
  Handle right button clicks...
  }
  if ClickButton=2 then begin
   bNul(Voice(snd_Click));
   SensorHelp(Url_Decode(ClickParams('Hint')));
  end;
  s:='';
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; i:Integer; r:Real;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   Example: @Browse http://localhost/
   }
   if IsSameText(cmd,'@Browse') then begin
    Cron('@Browse '+arg);
    Data:='';
   end else
   {
   Example: @open \\localhost\hvss_data\*.dat
   }
   if IsSameText(cmd,'@Open') then begin
    Cron('@FileOpenDialog '+URL_Packed(Trim(arg)));
    Data:='';
   end else
   {
   Example: @Shutdown Daq Exit
   }
   if IsSameText('@Shutdown',cmd) then begin
    if IsSameText('Daq',ExtractWord(1,arg)) then begin
     if IsSameText('Exit',    ExtractWord(2,arg)) then Cron('@Shutdown Daq Exit Beep');
     if IsSameText('Restart', ExtractWord(2,arg)) then Cron('@Shutdown Daq Restart Beep 1 1');
    end;
    if IsSameText('Crw',ExtractWord(1,arg)) then begin
     if IsSameText('Exit',    ExtractWord(2,arg)) then Cron('@Shutdown Crw Exit Beep');
    end;
    if IsSameText('Win',ExtractWord(1,arg)) then begin
     if IsSameText('Exit',    ExtractWord(2,arg)) then Cron('@Shutdown Win Exit Beep');
     if IsSameText('Logout',  ExtractWord(2,arg)) then Cron('@Shutdown Win Logout Beep');
     if IsSameText('Restart', ExtractWord(2,arg)) then Cron('@Shutdown Win Restart Beep');
    end;
    Data:='';
   end else
   {
   Example: @SaveParams
   }
   if IsSameText(cmd,'@SaveParams') then begin
    SaveParams;
    Success(cmd);
    Data:='';
   end else
   {
   Example: @LoadParams
   }
   if IsSameText(cmd,'@LoadParams') then begin
    LoadParams;
    Success(cmd);
    Data:='';
   end else
   {
   Example: @WinHide HVSS.GUI
   }
   if IsSameText(cmd,'@WinHide') then begin
    if not IsEmptyStr(arg) then bNul(WinHide(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinShow HVSS.GUI
   }
   if IsSameText(cmd,'@WinShow') then begin
    if not IsEmptyStr(arg) then bNul(WinShow(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinDraw HVSS.GUI
   }
   if IsSameText(cmd,'@WinDraw') then begin
    if not IsEmptyStr(arg) then bNul(WinDraw(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinSelect HVSS.GUI
   }
   if IsSameText(cmd,'@WinSelect') then begin
    if not IsEmptyStr(arg) then bNul(WinSelect(Trim(arg)));
    Data:='';
   end else
   {
   Example: @Async @WinSelect HVSS.GUI
   }
   if IsSameText(cmd,'@Async') then begin
    if not IsEmptyStr(arg) then DevSendCmd(devMySelf,Trim(arg));
    Data:='';
   end else
   {
   Example: @Eval @System @Async @View Hide Crw32.ToolBar
   }
   if IsSameText(cmd,'@Eval') then begin
    if not IsEmptyStr(arg) then rNul(Eval(Trim(arg)));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
