 {
 ***********************************************************************
 UDGB01 Modbus Driver.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Modbus.XXX    - response come from &ModbusProxy server:
| @Modbus.Reply    ref cid tim port uid fid $$ans
| @Modbus.Refuse   ref cid tim port uid fid   msg
| @Modbus.Timeout  ref cid tim port uid fid $$req
|  ref  - sender device reference, expected &ModbusProxy.
|  cid  - command id, any user defined custom data.
|  tim  - request-response time, ms.
|  port - logical port on &ModbusProxy, expected 1..16.
|  uid  - unit id, i.e.MODBUS address, expected 1..247.
|  fid  - function id, expected 1,2,3,4,5,6,15,16.
|  ans  - answer  (response), in HEX_ENCODE format.
|  req  - request (original), in HEX_ENCODE format.
|  msg  - plain text message to explain reason of Refuse.
| @PARAM.XX      - write parameter, i.e. send to MODBUS
| @Edit PARAM.XX - edit parameter and write it
| @Remote @cmd   - run @cmd remotely: send to DIM server
| @LoadIni       - load params from INI file.
| @SaveIni       - save params to   INI file.
| @MenuToolsOpen - open Menu Tools dialog.
| @DimGuiClick   - handle remote click from DIM
|********************************************************
[]
 Configuration:
 ***********************************************************************
  [DeviceList]
  &UDGB01.DRIVER = device software program
  [&UDGB01.DRIVER]
  Comment        = UDGB01 MODBUS DRIVER.
  InquiryPeriod  = 0
  DevicePolling  = 10, tpNormal
  ProgramSource  = ..\DaqPas\doza_udgb01.pas
  DigitalFifo    = 1024
  AnalogFifo     = 1024
  DebugFlags     = 3
  OpenConsole    = 2
  Simulator      = 0
  ModbusPort     = 1
  ModbusUnitId   = 1
  ModbusTimeout  = 250
  ModbusDeadline = 60000
  ModbusPolling  = 100
  DelayOnStart   = 1000
  tagPrefix      = UDGB01
  ...etc...
 ***********************************************************************
 }
program UDGB01_Modbus_Driver;    { UDGB01 Modbus Driver             }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_NetLibrary}            { NetLibrary constants             }
 dfStatist       = 32;           { DebugFlags - polling statistics  }
 MaxCmdNum       = 16;           { Maximal number of commands:      }
 cm_ReadALLIR    = 1;            { - Read All Input Registers       }
 cm_WritYELVA    = 2;            { - Write Yellow Vol. Act. level   }
 cm_WritREDVA    = 3;            { - Write Red Volume Activity lev. }
 cm_WritDRYVA    = 4;            { - Write Dry contact Vol.Ac.level }
 cm_WritRTIME    = 5;            { - Write registers of Date Time   }
 ao_PARAM_GASVA  = 0;            { Gas    (measure) Volume Activity }
 ao_PARAM_YELVA  = 1;            { Yellow (warning) Volume Activity }
 ao_PARAM_REDVA  = 2;            { Red    (alarm)   Volume Activity }
 ao_PARAM_DRYVA  = 3;            { Dry contact      Volume Activity }
 ao_PARAM_RESRC  = 4;            { Resource                         }
 ao_PARAM_RTIME  = 5;            { Real Time by device clock        }
 do_STATE_FLAGS  = 0;            { Status flags                     }
 do_STATE_INDIC  = 1;            { Status indicator                 }
 do_POLLRATERX   = 2;            { Poll rate Rx                     }
 do_POLLRATETX   = 3;            { Poll rate Tx                     }
 do_POLLRATEEX   = 4;            { Poll rate Ex                     }
 do_ERRORCOUNT   = 5;            { Error counter                    }
 do_POLLSUMMRX   = 6;            { Poll summ Rx                     }
 do_POLLSUMMTX   = 7;            { Poll summ Tx                     }
 do_POLLSUMMEX   = 8;            { Poll summ Ex                     }
 SwapModeFloat   = 1;            { Byte swap mode for float values  }
 SwapModeInt32   = 3;            { Byte swap mode for integers      }
 st_NORMAL       = 0;            { Green                            }
 st_WARNING      = 1;            { Yellow                           }
 st_ALARM        = 2;            { Red level                        }
 st_TEST         = 3;            { Yellow test mode                 }
 st_ERROR        = 4;            { Red error                        }
 st_TIMEOUT      = 5;            { Red timeout                      }
 st_REFUSE       = 6;            { Red refuse                       }
 st_SIMULAT      = 7;            { Simulator mode                   }
 st_DISABLE      = 8;            { Polling disabled                 }
 TestingMode     = false;        { Just for tests                   }

type
 TTagRef = record tag,nai,ndi,nao,ndo:Integer; val,tim:Real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_NetLibrary}            { NetLibrary variables             }
 UDGB01     : record             { UDGB01 Driver data record        }
  Simulator : Boolean;           { Simulator or Driver mode         }
  Modbus    : record             { Modbus data record               }
   Port     : Integer;           { Logical Port on &ModbusProxy     }
   UnitId   : Integer;           { Modbus unit id                   }
   Timeout  : Integer;           { Modbus timeout, ms               }
   Polling  : Integer;           { Modbus polling period, ms        }
   Deadline : Integer;           { Modbus deadline time, ms         }
   DelayOnStart : Integer;       { Command cycle delay on start     }
   Cmd      : record             { Command cycle data record        }
    Num     : Integer;           { Current running command number   }
    Enabled : array [1..MaxCmdNum] of Boolean; { Enable polling     }
    FuncId  : array [1..MaxCmdNum] of Integer; { Modbus Function Id }
    SAddr   : array [1..MaxCmdNum] of Integer; { Start Address      }
    Quant   : array [1..MaxCmdNum] of Integer; { Quantity of data   }
    OpData  : array [1..MaxCmdNum] of Real;    { Operation data     }
    OpBuff  : array [1..MaxCmdNum] of Real;    { Operation buffer   }
   end;                          {                                  }
   Poll     : record             { Polling transaction data record  }
    ref     : Integer;           { Last sent device reference       }
    cid     : Integer;           { Last sent command id             }
    tim     : Real;              { Last polling time, ms            }
    port    : Integer;           { Last polling port                }
    uid     : Integer;           { Last polling unit id             }
    fid     : Integer;           { Last sent function id            }
    saddr   : Integer;           { Last sent start address          }
    quant   : Integer;           { Last sent quantity of registers  }
    dat     : String;            { Last sent PDU=(fid+dat) data     }
    Summ,                        { Poll summ counters               }
    Rate    : record             { Poll rate on last second         }
     Rx     : Real;              { Rx poll summ/rate - receiver     }
     Tx     : Real;              { Tx poll summ/rate - transmitter  }
     Ex     : Real;              { Ex poll summ/rate - errors       }
    end;                         {                                  }
   end;                          {                                  }
  end;                           {                                  }
  POLL      : record             { Poll parameters                  }
   ENABLE   : TTagRef;           { Poll enable flag                 }
  end;                           {                                  }
  PARAM     : record             { Device analog parameters         }
   GASVA    : TTagRef;           { Gas volume activity level        }
   YELVA    : TTagRef;           { Yellow (warning) vol. act. level }
   REDVA    : TTagRef;           { Red    (alarm)   vol. act. level }
   DRYVA    : TTagRef;           { Dry contact      vol. act. level }
   DTIME    : TTagRef;           { Date/time of device clock        }
   RESRC    : TTagRef;           { Resource time                    }
   IDENT    : TTagRef;           { Device Identifier                }
  end;                           {                                  }
  STATE     : record             { Device state parameters          }
   FLAGS    : TTagRef;           { Flags                            }
   INDIC    : TTagRef;           { Indicator                        }
  end;                           {                                  }
  POLLRATE  : record             {                                  }
   RX,TX,EX : TTagRef;           {                                  }
  end;                           {                                  }
  POLLSUMM  : record             {                                  }
   RX,TX,EX : TTagRef;           {                                  }
  end;                           {                                  }
  ERROR     : record             {                                  }
   COUNT    : TTagRef;           {                                  }
  end;                           {                                  }
  SIM       : record             { Simulator data                   }
   GASVA    : Real;              {                                  }
   YELVA    : Real;              {                                  }
   REDVA    : Real;              {                                  }
   DRYVA    : Real;              {                                  }
   RTIME    : Real;              {                                  }
   RESRC    : Integer;           {                                  }
   STATE    : Integer;           {                                  }
  end;                           {                                  }
 end;                            {                                  }
 cmd_ZeroPortCounters : Integer; { @ZeroPortCounters                }
 cmd_DimTagUpdate     : Integer; { @DimTagUpdate                    }
 cmd_ClearModbusSumm  : Integer; { @ClearModbusSumm                 }
 cmd_PARAM_YELVA      : Integer; { @PARAM.YELVA                     }
 cmd_PARAM_REDVA      : Integer; { @PARAM.REDVA                     }
 cmd_PARAM_DRYVA      : Integer; { @PARAM.DRYVA                     }
 cmd_PARAM_RTIME      : Integer; { @PARAM.RTIME                     }
 cmd_Edit             : Integer; { @Edit                            }
 cmd_Remote           : Integer; { @Remote                          }
 cmd_LoadIni          : Integer; { @LoadIni                         }
 cmd_SaveIni          : Integer; { @SaveIni                         }
 cmd_MenuToolsOpen    : Integer; { @MenuToolsOpen                   }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_NetLibrary}            { NetLibrary functions             }

 //
 // Command cycle routines.
 //
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsUsableCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsUsableCmdNum:=(UDGB01.Modbus.Cmd.FuncId[Num]<>0)
  else IsUsableCmdNum:=False;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsEnabledCmdNum:=(UDGB01.Modbus.Cmd.FuncId[Num]<>0) and UDGB01.Modbus.Cmd.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer;
 begin
  i:=0;
  while (i<MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   if IsEnabledCmdNum(Num)
   then i:=MaxCmdNum
   else i:=i+1;
  end;
  NextEnabledCmdNum:=Num;
 end;
 procedure EnableCmdNum(Num:Integer; Enabled:Boolean);
 begin
  if IsUsableCmdNum(Num) then UDGB01.Modbus.Cmd.Enabled[Num]:=Enabled;
 end;
 procedure HoldCmdOpData(Num:Integer; OpData:Real);
 begin
  if IsUsableCmdNum(Num) then begin
   UDGB01.Modbus.Cmd.Enabled[Num]:=not IsNaN(OpData);
   UDGB01.Modbus.Cmd.OpBuff[Num]:=OpData;
  end;
 end;
 procedure ApplyCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then
  if not IsNaN(UDGB01.Modbus.Cmd.OpBuff[Num]) then begin
   UDGB01.Modbus.Cmd.OpData[Num]:=UDGB01.Modbus.Cmd.OpBuff[Num];
   UDGB01.Modbus.Cmd.OpBuff[Num]:=_NaN;
  end;
 end;
 procedure ReleaseCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then HoldCmdOpData(Num,UDGB01.Modbus.Cmd.OpBuff[Num]);
 end;
 function GetCmdOpData(Num:Integer):Real;
 begin
  if IsUsableCmdNum(Num)
  then GetCmdOpData:=UDGB01.Modbus.Cmd.OpData[Num]
  else GetCmdOpData:=_NaN;
 end;
 procedure InitCmdItem(Num:Integer; Enabled:Boolean; FuncId,SAddr,Quant:Integer; OpData,OpBuff:Real);
 begin
  if IsValidCmdNum(Num) then begin
   UDGB01.Modbus.Cmd.Enabled[Num]:=Enabled;
   UDGB01.Modbus.Cmd.FuncId[Num]:=FuncId;
   UDGB01.Modbus.Cmd.SAddr[Num]:=SAddr;
   UDGB01.Modbus.Cmd.Quant[Num]:=Quant;
   UDGB01.Modbus.Cmd.OpData[Num]:=OpData;
   UDGB01.Modbus.Cmd.OpBuff[Num]:=OpBuff;
  end;
 end;
 //
 // Command table cleanup and initialization.
 //
 procedure ClearCmdTable;
 var Num:Integer;
 begin
  UDGB01.Modbus.Cmd.Num:=0;
  for Num:=1 to MaxCmdNum do InitCmdItem(Num,False,0,0,0,0,_NaN);
 end;
 procedure InitCmdTable;
 begin
  ClearCmdTable;
  // Table of CommandId     Enabled FunctionId        SAddr Quant OpData OpBuff
  InitCmdItem(cm_ReadALLIR, True,   modbus_fn_ReadIR, 0,    21,   0,     _NaN);
  if TestingMode then
  InitCmdItem(cm_ReadALLIR, True,   modbus_fn_ReadHR, 0,    21,   0,     _NaN);
  InitCmdItem(cm_WritYELVA, False,  modbus_fn_WritMR, 10,   2,    0,     _NaN);
  InitCmdItem(cm_WritREDVA, False,  modbus_fn_WritMR, 12,   2,    0,     _NaN);
  InitCmdItem(cm_WritDRYVA, False,  modbus_fn_WritMR, 14,   2,    0,     _NaN);
  InitCmdItem(cm_WritRTIME, False,  modbus_fn_WritMR, 18,   3,    0,     _NaN);
 end;
 //
 // Assign modbus last sent polling request record.
 //
 procedure AssignModbusPoll(ref,cid:Integer; tim:Real; port,uid,fid,saddr,quant:Integer; dat:String);
 begin
  UDGB01.Modbus.Poll.ref:=ref;     UDGB01.Modbus.Poll.cid:=cid;      UDGB01.Modbus.Poll.tim:=tim;
  UDGB01.Modbus.Poll.port:=port;   UDGB01.Modbus.Poll.uid:=uid;      UDGB01.Modbus.Poll.fid:=fid;
  UDGB01.Modbus.Poll.saddr:=saddr; UDGB01.Modbus.Poll.quant:=quant;  UDGB01.Modbus.Poll.dat:=dat;
 end;
 //
 // Clear modbus polling request to be ready for next polling.
 //
 procedure ClearModbusPoll;
 begin
  UDGB01.Modbus.Poll.cid:=0;
  UDGB01.Modbus.Poll.dat:='';
 end;
 //
 // Clear modbus poll summ counters.
 //
 procedure ClearModbusSumm;
 begin
  UDGB01.Modbus.Poll.Summ.Rx:=0;
  UDGB01.Modbus.Poll.Summ.Tx:=0;
  UDGB01.Modbus.Poll.Summ.Ex:=0;
 end;
 //
 // Clear modbus poll rate counters.
 //
 procedure ClearModbusRate;
 begin
  UDGB01.Modbus.Poll.Rate.Rx:=0;
  UDGB01.Modbus.Poll.Rate.Tx:=0;
  UDGB01.Modbus.Poll.Rate.Ex:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Real);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 //
 // Format error message with with error counter increment.
 // Usage is like: Trouble(GotBug(...)); Problem(GotBug(...));
 //
 function GotBug(msg:String):String;
 begin
  IncSummRate(UDGB01.Modbus.Poll.Summ.Ex,UDGB01.Modbus.Poll.Rate.Ex);
  GotBug:=msg;
 end;
 //
 // Nice tooltip notifier.
 //
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
 end;
 //
 // Post command to local/remote server.
 //
 procedure PostCmdLocal(cmd:String);
 begin
  DevPostCmdLocal(cmd);
 end;
 procedure PostCmdRemote(cmd:String);
 begin
  Dim_GuiConsoleSend(DevName,cmd);
 end;
 //
 // Menu Tools Starter to start editing.
 //
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Вызвать справочное окно (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+DevName);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @ClearModbusSumm');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Очистка счетчиков ошибок '+ModbusProxy);
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @ZeroPortCounters 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Restart Local  '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Restart Remote '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 150 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  105 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 //
 // Find the current Proxy device to send @Modbus.Poll message.
 //
 function devTheProxy:Integer;
 begin
  if UDGB01.Simulator               // In simulation mode
  then devTheProxy:=devMySelf       // send messages to himself
  else devTheProxy:=devModbusProxy; // otherwise use &ModbusProxy
 end;
 //
 // Check if communication port is opened or not.
 // As long as we use Proxy device, check Port number & device reference.
 //
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(UDGB01.Modbus.Port>0) and (devTheProxy<>0);
 end;
 //
 // Encode/decode date/time in 3 registers.
 //
 function encode_datetime(ms:Real):String;
 var year,month,day,hour,min,sec,sec100,reg18,reg19,reg20:Integer;
 begin
  year:=ms2year(ms); month:=ms2month(ms); day:=ms2day(ms);
  hour:=ms2hour(ms); min:=ms2min(ms);     sec:=ms2sec(ms);
  sec100:=Round((ms-datetime2ms(year,month,day,hour,min,sec,0))/10);
  reg18:=iAnd(day,31)+iShift(iAnd(month,15),5)+iShift(iAnd(year-1980,127),9);
  reg19:=min+256*hour;
  reg20:=sec100+256*sec;
  encode_datetime:=modbus_dump_int16(reg18,SwapModeInt32)
                  +modbus_dump_int16(reg19,SwapModeInt32)
                  +modbus_dump_int16(reg20,SwapModeInt32);
 end;
 function decode_datetime(reg18,reg19,reg20:Integer):Real;
 var year,month,day,hour,min,sec,sec100:Integer; ms:Real;
 begin
  day    := iAnd(reg18,31);                  // Bit 0..4  - 5 bits
  month  := iAnd(iShift(reg18,-5),15);       // Bit 5..8  - 4 bits
  year   := iAnd(iShift(reg18,-9),127)+1980; // Bit 9..15 - 7 bits
  hour   := reg19 div 256;                   // Hi byte
  min    := reg19 mod 256;                   // Lo byte
  sec    := reg20 div 256;                   // Hi byte
  sec100 := reg20 mod 256;                   // Lo byte
  ms:=datetime2ms(year,month,day,hour,min,sec,sec100*10);
  decode_datetime:=ms;
 end;
 function IsValidDateTime(ms:Real):Boolean;
 begin
  if IsNaN(ms) then IsValidDateTime:=False else
  if IsInf(ms) then IsValidDateTime:=False else
  if ms<datetime2ms(1900,1,1,0,0,0,0) then IsValidDateTime:=False else
  if ms>datetime2ms(2100,1,1,0,0,0,0) then IsValidDateTime:=False else
  IsValidDateTime:=True;
 end;
 function extract_datetime(arg:String):Real;
 var r,v:Real; year,month,day,hour,min,sec,msec,p:Integer;
 begin
  r:=rVal(ExtractWord(1,arg));
  if not IsValidDateTime(r) then r:=_NaN;
  if IsNan(r) then begin
   arg:=Trim(arg);
   p:=Pos('.',arg);
   if p>1 then begin
    year:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos('.',arg);
    if (2000<=year) and (year<=3000) and (p>1) then begin
     month:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos('-',arg);
     if (1<=month) and (month<=12) and (p>1) then begin
      day:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos(':',arg);
       if (1<=day) and (day<=31) and (p>1) then begin
        hour:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos(':',arg);
        if (0<=hour) and (hour<=60) and (p>1) then begin
         min:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1);
         if (0<=min) and (min<=60) and not IsEmptyStr(arg) then begin
          v:=rVal(arg);
          if not IsNan(v) then begin
           sec:=Trunc(v);
           msec:=Round(Frac(v)*1000);
           r:=datetime2ms(year,month,day,hour,min,sec,msec);
          end;
         end;
        end;
       end;
     end;
    end;
   end;
  end;
  extract_datetime:=r;
 end;
 //
 // UDGB01 tags initialization.
 //
 procedure UDGB01_FillTags(InitVal:Real);
 begin
  UDGB01.POLL.ENABLE.val:=InitVal;
  UDGB01.STATE.FLAGS.val:=InitVal;
  UDGB01.STATE.INDIC.val:=InitVal;
  UDGB01.PARAM.IDENT.val:=InitVal;
  UDGB01.PARAM.DTIME.val:=InitVal;
  UDGB01.PARAM.GASVA.val:=InitVal;
  UDGB01.PARAM.YELVA.val:=InitVal;
  UDGB01.PARAM.REDVA.val:=InitVal;
  UDGB01.PARAM.DRYVA.val:=InitVal;
  UDGB01.PARAM.RESRC.val:=InitVal;
  UDGB01.POLLRATE.RX.val:=InitVal;
  UDGB01.POLLRATE.TX.val:=InitVal;
  UDGB01.POLLRATE.EX.val:=InitVal;
  UDGB01.POLLSUMM.RX.val:=InitVal;
  UDGB01.POLLSUMM.TX.val:=InitVal;
  UDGB01.POLLSUMM.EX.val:=InitVal;
  UDGB01.ERROR.COUNT.val:=InitVal;
 end;
 procedure UDGB01_InitTags(Prefix:String; InitVal:Real);
 begin
  DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
  InitTag(UDGB01.POLL.ENABLE.tag,     Prefix+'.POLL.ENABLE',     1);
  InitTag(UDGB01.STATE.FLAGS.tag,     Prefix+'.STATE.FLAGS',     1);
  InitTag(UDGB01.STATE.INDIC.tag,     Prefix+'.STATE.INDIC',     1);
  InitTag(UDGB01.PARAM.IDENT.tag,     Prefix+'.PARAM.IDENT',     3);
  InitTag(UDGB01.PARAM.DTIME.tag,     Prefix+'.PARAM.DTIME',     3);
  InitTag(UDGB01.PARAM.GASVA.tag,     Prefix+'.PARAM.GASVA',     2);
  InitTag(UDGB01.PARAM.YELVA.tag,     Prefix+'.PARAM.YELVA',     2);
  InitTag(UDGB01.PARAM.REDVA.tag,     Prefix+'.PARAM.REDVA',     2);
  InitTag(UDGB01.PARAM.DRYVA.tag,     Prefix+'.PARAM.DRYVA',     2);
  InitTag(UDGB01.PARAM.RESRC.tag,     Prefix+'.PARAM.RESRC',     2);
  InitTag(UDGB01.POLLRATE.RX.tag,     Prefix+'.POLLRATE.RX',     2);
  InitTag(UDGB01.POLLRATE.TX.tag,     Prefix+'.POLLRATE.TX',     2);
  InitTag(UDGB01.POLLRATE.EX.tag,     Prefix+'.POLLRATE.EX',     2);
  InitTag(UDGB01.ERROR.COUNT.tag,     Prefix+'.ERROR.COUNT',     2);
  InitTag(UDGB01.POLLSUMM.RX.tag,     Prefix+'.POLLSUMM.RX',     2);
  InitTag(UDGB01.POLLSUMM.TX.tag,     Prefix+'.POLLSUMM.TX',     2);
  InitTag(UDGB01.POLLSUMM.EX.tag,     Prefix+'.POLLSUMM.EX',     2);
  UDGB01_FillTags(InitVal);
 end;
 //
 // Set State Flags and Indicator.
 //
 procedure UDGB01_SetStateFlags(flags:Integer);
 begin
  UpdateDo(do_STATE_FLAGS,time,flags);
  bNul(iSetTag(UDGB01.STATE.FLAGS.tag,flags));
 end;
 procedure UDGB01_SetStateIndic(indic:Integer; dtime:String);
 begin
  UpdateDo(do_STATE_INDIC,time,indic);
  bNul(iSetTag(UDGB01.STATE.INDIC.tag,indic));
  if (dtime<>'') then bNul(sSetTag(UDGB01.PARAM.DTIME.tag,dtime));
 end;
 //
 // Calculate Modbus request PDU by command id.
 // cid   - (in)  command id number.
 // fid   - (out) Modbus function id.
 // saddr - (out) zero-based start address of registers to read/write.
 // quant - (out) data quantity, i.e. number of r/w registers or single register r/w value.
 // dat   - (out) Modbus PDU data unit, PDU=(fid+dat).
 //
 function UDGB01_CalcPDU(cid:Integer; var fid,saddr,quant:Integer; var dat:String):Boolean;
 begin
  fid:=0; saddr:=0; quant:=0; dat:='';
  if IsEnabledCmdNum(cid) then begin
   fid:=UDGB01.Modbus.Cmd.FuncId[cid];
   saddr:=UDGB01.Modbus.Cmd.SAddr[cid];
   quant:=UDGB01.Modbus.Cmd.Quant[cid];
   if (cid=cm_ReadALLIR) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_WritYELVA) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_float(GetCmdOpData(cid),SwapModeFloat));
   end else
   if (cid=cm_WritREDVA) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_float(GetCmdOpData(cid),SwapModeFloat));
   end else
   if (cid=cm_WritDRYVA) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_float(GetCmdOpData(cid),SwapModeFloat));
   end else
   if (cid=cm_WritRTIME) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,encode_datetime(GetCmdOpData(cid)));
   end else
   Trouble(GotBug('Invalid command id '+Str(cid)));
  end;
  UDGB01_CalcPDU:=modbus_func_ok(fid) and (Length(dat)>0);
 end;
 //
 // Main Modbus command handler to process Modbus device reply.
 // cid   - (in)  command id number.
 // fid   - (in)  Modbus function id.
 // saddr - (in)  zero-based start address of registers to read/write.
 // quant - (in)  data quantity, i.e. number of r/w registers or single register r/w value.
 // raw   - (in)  raw coils/registers data array.
 //
 procedure UDGB01_OnCommand(cid,fid,saddr,quant:Integer; raw:String);
 var addr,offs,stat,indic:Integer; r:Real;
 begin
  if IsValidCmdNum(cid) then begin 
   if (cid=cm_ReadALLIR) then begin // Read 21 input registers
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs; // PLC reg.address
     if (addr=0) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(UDGB01.PARAM.GASVA.tag,r));
      UpdateAo(ao_PARAM_GASVA,time,r);
     end else
     if (addr=2) and (quant-offs+1>=1) then begin
      stat:=modbus_ext_int16(raw,offs,SwapModeInt32);
      if iGetTag(UDGB01.POLL.ENABLE.tag)=0                         then indic:=st_DISABLE else
      if UDGB01.Simulator                                          then indic:=st_SIMULAT else
      if IsBit(stat,3) or IsBit(stat,8)                            then indic:=st_TEST    else
      if IsBit(stat,0) and not IsBit(stat,3) and not IsBit(stat,8) then indic:=st_ERROR   else
      if IsBit(stat,4) and not IsBit(stat,3)                       then indic:=st_ERROR   else
      if IsBit(stat,6)                       and not IsBit(stat,8) then indic:=st_ERROR   else
      if IsBit(stat,7) and not IsBit(stat,3) and not IsBit(stat,8) then indic:=st_ERROR   else
      if IsBit(stat,2)                                             then indic:=st_ALARM   else
      if IsBit(stat,1)                                             then indic:=st_WARNING else
      indic:=st_NORMAL;
      UDGB01_SetStateFlags(stat);
      UDGB01_SetStateIndic(indic,'');
     end else
     if (addr=10) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(UDGB01.PARAM.YELVA.tag,r));
      UpdateAo(ao_PARAM_YELVA,time,r);
     end else
     if (addr=12) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(UDGB01.PARAM.REDVA.tag,r));
      UpdateAo(ao_PARAM_REDVA,time,r);
     end else
     if (addr=14) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(UDGB01.PARAM.DRYVA.tag,r));
      UpdateAo(ao_PARAM_DRYVA,time,r);
     end else
     if (addr=16) and (quant-offs+1>=2) then begin
      r:=modbus_ext_int32(raw,offs,SwapModeInt32);
      bNul(rSetTag(UDGB01.PARAM.RESRC.tag,r));
      UpdateAo(ao_PARAM_RESRC,time,r);
     end else
     if (addr=18) and (quant-offs+1>=3) then begin
      r:=decode_datetime(modbus_ext_int16(raw,offs+0,SwapModeInt32),
                         modbus_ext_int16(raw,offs+1,SwapModeInt32),
                         modbus_ext_int16(raw,offs+2,SwapModeInt32));
      if IsValidDateTime(r) then begin
       bNul(sSetTag(UDGB01.PARAM.DTIME.tag,GetDateTime(r)));
       UpdateAo(ao_PARAM_RTIME,time,r);
      end;
     end;
    end;
   end else
   // Write register/coils: release confirmed command
   // to avoid rewrite the data which already written
   if (cid=cm_WritYELVA) then ReleaseCmdOpData(cid) else
   if (cid=cm_WritREDVA) then ReleaseCmdOpData(cid) else
   if (cid=cm_WritDRYVA) then ReleaseCmdOpData(cid) else
   if (cid=cm_WritRTIME) then ReleaseCmdOpData(cid) else
   Trouble(GotBug('Unexpected command '+Str(cid)));
  end;
 end;
 //
 // Data handler on @Modbus.Reply event. Process reply comes from Modbus device.
 //
 procedure UDGB01_OnReply(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>UDGB01.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>UDGB01.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>UDGB01.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>UDGB01.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>UDGB01.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=UDGB01.Modbus.Poll.saddr; quant:=UDGB01.Modbus.Poll.quant;
   if modbus_decode_pdu('A',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>UDGB01.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>UDGB01.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else
    if modbus_is_except(fid) then begin
     Problem(GotBug('MODBUS EXCEPTION '+Str(modbus_ext_byte(raw,0))+' ON COMMAND '+Str(cid)));
    end else begin
     IncSummRate(UDGB01.Modbus.Poll.Summ.Rx,UDGB01.Modbus.Poll.Rate.Rx);
     UDGB01_OnCommand(cid,fid,saddr,quant,raw);
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // Data handler on @Modbus.Poll event.
 // This procedure calls in simulation mode only.
 //
 procedure UDGB01_OnSimPoll(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant,offs,addr,state,resrc:Integer; r:Real;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>UDGB01.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>UDGB01.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>UDGB01.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>UDGB01.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>UDGB01.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=UDGB01.Modbus.Poll.saddr; quant:=UDGB01.Modbus.Poll.quant;
   if modbus_decode_pdu('R',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>UDGB01.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>UDGB01.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else begin
     if DebugFlagEnabled(dfDetails) then Details('Simulate polling '+Str(cid));
     if (cid=cm_ReadALLIR) then begin // 21 register
      raw:=modbus_dump_float(UDGB01.SIM.GASVA,SwapModeFloat)
          +modbus_dump_int16(UDGB01.SIM.STATE,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_int16(0,SwapModeInt32)
          +modbus_dump_float(UDGB01.SIM.YELVA,SwapModeFloat)
          +modbus_dump_float(UDGB01.SIM.REDVA,SwapModeFloat)
          +modbus_dump_float(UDGB01.SIM.DRYVA,SwapModeFloat)
          +modbus_dump_int32(UDGB01.SIM.RESRC,SwapModeInt32)
          +encode_datetime(mSecNow-UDGB01.SIM.RTIME);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritYELVA) then begin
      UDGB01.SIM.YELVA:=modbus_ext_float(raw,0,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritREDVA) then begin
      UDGB01.SIM.REDVA:=modbus_ext_float(raw,0,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritDRYVA) then begin
      UDGB01.SIM.DRYVA:=modbus_ext_float(raw,0,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritRTIME) then begin
      r:=decode_datetime(modbus_ext_int16(raw,offs+0,SwapModeInt32),
                         modbus_ext_int16(raw,offs+1,SwapModeInt32),
                         modbus_ext_int16(raw,offs+2,SwapModeInt32));
      if IsValidDateTime(r) then UDGB01.SIM.RTIME:=mSecNow-r;
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     DevSendCmdLocal(modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,
                modbus_as_except(fid,true),Dump(Chr(modbus_er_ILLFUN))));
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // UDGB01 Driver Simulator mode polling.
 // This procedure calls in simulation mode only.
 //
 procedure UDGB01_SIM_POLL;
 begin
  if UDGB01.Simulator then begin
   UDGB01.SIM.GASVA:=10+3*sin(2*pi*time*60)+0.1*random(-1,1);
   UDGB01.SIM.STATE:=iSetBitState(UDGB01.SIM.STATE,1,UDGB01.SIM.GASVA>UDGB01.SIM.YELVA);
   UDGB01.SIM.STATE:=iSetBitState(UDGB01.SIM.STATE,2,UDGB01.SIM.GASVA>UDGB01.SIM.REDVA);
  end;
 end;
 //
 // Xor bit on click (local version).
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 var nv:Integer;
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Xor bit on click (remote version).
 }
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 var b:Boolean; nv:Integer;
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DIM_GuiClickSend(DIM_GuiClickBuff+'NewValue='+Str(iXor(iGetTag(tag),XorMask)));
   b:=Voice(snd_Click);
  end;
 end;
 //
 // UDGB01 Driver GUI polling.
 //
 procedure UDGB01_GUI_POLL;
 var s:String; ClickCurve:Integer;
  //
  // Send command (cmd) to device (dev) on tag click.
  //
  procedure ClickTagDevSendCmd(tag,dev:Integer; cmd:String);
  begin
   if Length(cmd)>0 then
   if IsRefTag(tag) then
   if IsRefDevice(dev) then
   if ClickTag=tag then begin
    bNul(Voice(snd_Click));
    DevSendCmd(dev,cmd);
   end;
  end;
 begin
  s:='';
  DIM_GuiClickBuff:='';
  //
  // Handle user mouse/keyboard clicks...
  // ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  // ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  //
  if ClickWhat<>0 then
  repeat
   //
   // Copy GUI click to DIM buffer for remote execution.
   //
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   //
   // Handle MouseDown/KeyDown
   //
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    //
    // Handle Left mouse button click
    //
    if (ClickButton=VK_LBUTTON) then begin
     //
     // Handle local clicks
     //
     if ClickIsLocal then begin
      //
      // Handle sensor clicks...
      //
      if (ClickTag<>0) then begin
       //
       // Buttons...
       //
       ClickTagXorRemote(UDGB01.POLL.ENABLE.tag,1);
       //
       // Send command to edit on Click.
       //
       s:=mime_encode(SetFormUnderSensorLeftBottom(ClickParams('')));
       ClickTagDevSendCmd(UDGB01.PARAM.YELVA.tag,devMySelf,'@Edit PARAM.YELVA '+s);
       ClickTagDevSendCmd(UDGB01.PARAM.REDVA.tag,devMySelf,'@Edit PARAM.REDVA '+s);
       ClickTagDevSendCmd(UDGB01.PARAM.DRYVA.tag,devMySelf,'@Edit PARAM.DRYVA '+s);
       ClickTagDevSendCmd(UDGB01.PARAM.DTIME.tag,devMySelf,'@Edit PARAM.DTIME '+s);
       //
       // Tools menu
       //
       if ClickTag=UDGB01.PARAM.IDENT.tag then begin
        DevSendCmdLocal('@MenuToolsOpen');
        bNul(Voice(snd_Click));
       end;
      end;
      //
      // Plot & Tab windows
      //
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      //
      // User commands: @...
      //
      if LooksLikeCommand(ClickSensor) then begin
       DevSendCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     //
     // Handle remote clicks comes from DIM via @DimGuiClick message.
     // @DimGuiClick default handler decode and write events to FIFO,
     // so we can find it as clicks and can handle it in usual way.
     //
     if ClickIsRemote then begin
      //
      // Show time difference.
      //
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      //
      // Handle remote console commands...
      //
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
      //
      // Handle remote sensor clicks...
      //
      if TypeTag(ClickTag)>0 then begin
       s:=ClickParams('NewValue');
       if Length(s)>0 then begin
        if ClickTag=UDGB01.POLL.ENABLE.tag then UpdateTag(ClickTag,s,0,1);
       end;
      end;
     end;
    end;
   end;
  until (ClickRead=0);
  //
  // Edit handling...
  //
  if EditStateDone then begin
   //
   // If tag edit complete, send command to apply changes
   //
   if CheckEditTag(UDGB01.PARAM.YELVA.tag,s) then PostCmdRemote('@PARAM.YELVA '+s);
   if CheckEditTag(UDGB01.PARAM.REDVA.tag,s) then PostCmdRemote('@PARAM.REDVA '+s);
   if CheckEditTag(UDGB01.PARAM.DRYVA.tag,s) then PostCmdRemote('@PARAM.DRYVA '+s);
   if CheckEditTag(UDGB01.PARAM.DTIME.tag,s) then PostCmdRemote('@PARAM.RTIME '+s);
   //
   // Menu TOOLS.
   //
   MenuToolsHandler;
   //
   // Warning, Information dialog completion.
   //
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end;
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  DIM_GuiClickBuff:='';
  s:='';
 end;
 //
 // UDGB01 Driver Command Cycle polling.
 //
 procedure UDGB01_CMD_POLL;
  //
  // Send @Modbus.Poll ... command to &ModbusProxy and save sent request data in Modbus.Poll record.
  //
  procedure PollModbusProxy(cid:Integer);
  var dev,ref,tot,port,uid,fid,saddr,quant:Integer; dat:String;
  begin
   dat:='';
   ApplyCmdOpData(cid);
   if UDGB01_CalcPDU(cid,fid,saddr,quant,dat) then begin
    dev:=devTheProxy; ref:=devMySelf; tot:=UDGB01.Modbus.Timeout; port:=UDGB01.Modbus.Port; uid:=UDGB01.Modbus.UnitId;
    if DevSend(dev,modbus_proxy_poll('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat)+EOL)>0 then begin
     if DebugFlagEnabled(dfViewExp) then 
     ViewExp('COM: '+modbus_proxy_nice('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat,32));
     AssignModbusPoll(dev,cid,mSecNow,port,uid,fid,saddr,quant,dat);
     IncSummRate(UDGB01.Modbus.Poll.Summ.Tx,UDGB01.Modbus.Poll.Rate.Tx);
    end else begin
     Trouble(GotBug('Fail to send command '+Str(cid)));
     ClearModbusPoll;
    end;
   end else begin
    Trouble(GotBug('Fail to calc command '+Str(cid)));
    ClearModbusPoll;
   end;
   dat:='';
  end;
 begin
  if IsPortOpened and not DIM_IsClientMode then begin
   if IsValidCmdNum(UDGB01.Modbus.Poll.cid) then begin
    //
    // Request in progress, waiting @Modbus.Reply/@Modbus.Refuse/@Modbus.Timeout.
    // Handle Deadline if was no responce for too long time: repeat polling again.
    //
    if iGetTag(UDGB01.POLL.ENABLE.tag)<>0 then
    if (mSecNow>UDGB01.Modbus.Poll.tim+UDGB01.Modbus.Deadline) then begin
     UDGB01_SetStateIndic(st_REFUSE,'DEADLINE DETECTED');
     Trouble(GotBug('Deadline detected, repeat polling again...'));
     PollModbusProxy(UDGB01.Modbus.Poll.cid);
    end;
   end else begin
    //
    // If request is cleared, send new @Modbus.Poll request by timer.
    //
    if iGetTag(UDGB01.POLL.ENABLE.tag)<>0 then
    if (mSecNow>=UDGB01.Modbus.Poll.tim+UDGB01.Modbus.Polling) then begin
     UDGB01.Modbus.Cmd.Num:=NextEnabledCmdNum(UDGB01.Modbus.Cmd.Num);
     if IsEnabledCmdNum(UDGB01.Modbus.Cmd.Num)
     then PollModbusProxy(UDGB01.Modbus.Cmd.Num);
    end;
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    bNul(rSetTag(UDGB01.POLLRATE.RX.tag,UDGB01.Modbus.Poll.Rate.Rx));
    bNul(rSetTag(UDGB01.POLLRATE.TX.tag,UDGB01.Modbus.Poll.Rate.Tx));
    bNul(rSetTag(UDGB01.POLLRATE.EX.tag,UDGB01.Modbus.Poll.Rate.Ex));
    bNul(rSetTag(UDGB01.POLLSUMM.RX.tag,UDGB01.Modbus.Poll.Summ.Rx));
    bNul(rSetTag(UDGB01.POLLSUMM.TX.tag,UDGB01.Modbus.Poll.Summ.Tx));
    bNul(rSetTag(UDGB01.POLLSUMM.EX.tag,UDGB01.Modbus.Poll.Summ.Ex));
    bNul(rSetTag(UDGB01.ERROR.COUNT.tag,GetErrCount(-1)));
    UpdateDo(do_POLLRATERX,time,UDGB01.Modbus.Poll.Rate.Rx);
    UpdateDo(do_POLLRATETX,time,UDGB01.Modbus.Poll.Rate.Tx);
    UpdateDo(do_POLLRATEEX,time,UDGB01.Modbus.Poll.Rate.Ex);
    UpdateDo(do_POLLSUMMRX,time,UDGB01.Modbus.Poll.Summ.Rx);
    UpdateDo(do_POLLSUMMTX,time,UDGB01.Modbus.Poll.Summ.Tx);
    UpdateDo(do_POLLSUMMEX,time,UDGB01.Modbus.Poll.Summ.Ex);
    UpdateDo(do_ERRORCOUNT,time,GetErrCount(-1));
    if DebugFlagEnabled(dfStatist) then
    Success('PollRate: Rx='+Str(UDGB01.Modbus.Poll.Rate.Rx)
                   +'  Tx='+Str(UDGB01.Modbus.Poll.Rate.Tx)
                   +'  Ex='+Str(UDGB01.Modbus.Poll.Rate.Ex));
    ClearModbusRate;
   end;
  end;
  //
  // Update Indicator when polling OFF...
  //
  if SysTimer_Pulse(1000)>0 then begin
   if iGetTag(UDGB01.POLL.ENABLE.tag)=0 then begin
    UDGB01_SetStateIndic(st_DISABLE,'DISCONNECTED');
   end;
  end;
  //
  // Update DIM services
  //
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then UDGB01_FillTags(-MaxReal);
   if ShouldRefresh(UDGB01.POLL.ENABLE.val,iGetTag(UDGB01.POLL.ENABLE.tag))>0
   then DIM_UpdateTag(UDGB01.POLL.ENABLE.tag,'');
   if ShouldRefresh(UDGB01.STATE.FLAGS.val,iGetTag(UDGB01.STATE.FLAGS.tag))
     +ShouldRefresh(UDGB01.STATE.INDIC.val,iGetTag(UDGB01.STATE.INDIC.tag))>0
   then DIM_UpdateTag(UDGB01.STATE.FLAGS.tag,'');
   if ShouldRefresh(UDGB01.PARAM.GASVA.val,rGetTag(UDGB01.PARAM.GASVA.tag))>0
   then DIM_UpdateTag(UDGB01.PARAM.GASVA.tag,'');
   if ShouldRefresh(UDGB01.PARAM.YELVA.val,rGetTag(UDGB01.PARAM.YELVA.tag))
     +ShouldRefresh(UDGB01.PARAM.REDVA.val,rGetTag(UDGB01.PARAM.REDVA.tag))
     +ShouldRefresh(UDGB01.PARAM.DRYVA.val,rGetTag(UDGB01.PARAM.DRYVA.tag))
     +ShouldRefresh(UDGB01.PARAM.RESRC.val,rGetTag(UDGB01.PARAM.RESRC.tag))>0
   then DIM_UpdateTag(UDGB01.PARAM.YELVA.tag,'');
   if ShouldRefresh(UDGB01.PARAM.DTIME.val,HashIndexOf(sGetTag(UDGB01.PARAM.DTIME.tag),0,0))>0
   then DIM_UpdateTag(UDGB01.PARAM.DTIME.tag,'');
   if ShouldRefresh(UDGB01.POLLRATE.RX.val,rGetTag(UDGB01.POLLRATE.RX.tag))
     +ShouldRefresh(UDGB01.POLLRATE.TX.val,rGetTag(UDGB01.POLLRATE.TX.tag))
     +ShouldRefresh(UDGB01.POLLRATE.EX.val,rGetTag(UDGB01.POLLRATE.EX.tag))>0
   then DIM_UpdateTag(UDGB01.POLLRATE.RX.tag,'');
   if ShouldRefresh(UDGB01.POLLSUMM.RX.val,rGetTag(UDGB01.POLLSUMM.RX.tag))
     +ShouldRefresh(UDGB01.POLLSUMM.TX.val,rGetTag(UDGB01.POLLSUMM.TX.tag))
     +ShouldRefresh(UDGB01.POLLSUMM.EX.val,rGetTag(UDGB01.POLLSUMM.EX.tag))>0
   then DIM_UpdateTag(UDGB01.POLLSUMM.RX.tag,'');
   if ShouldRefresh(UDGB01.ERROR.COUNT.val,rGetTag(UDGB01.ERROR.COUNT.tag))>0
   then DIM_UpdateTag(UDGB01.ERROR.COUNT.tag,'');
  end;
 end;
 //
 // UDGB01 Driver cleanup.
 //
 procedure UDGB01_Clear;
 begin
  UDGB01.Simulator:=False;
  UDGB01.Modbus.Port:=0;
  UDGB01.Modbus.UnitId:=0;
  UDGB01.Modbus.Timeout:=0;
  UDGB01.Modbus.Polling:=0;
  UDGB01.Modbus.Deadline:=0;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // UDGB01 Driver initialization...
 //
 procedure UDGB01_Init;
 begin
  //
  // Initialize variables
  //
  UDGB01.SIM.GASVA:=0;
  UDGB01.SIM.YELVA:=10;
  UDGB01.SIM.REDVA:=12;
  UDGB01.SIM.DRYVA:=13;
  UDGB01.SIM.RESRC:=12345;
  UDGB01.SIM.RTIME:=0;
  //
  // Read ini file variables
  //
  UDGB01_InitTags(ReadIni('tagPrefix'),-MaxReal);
  UDGB01.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  Success('Simulator='+Str(Ord(UDGB01.Simulator)));
  UDGB01.Modbus.Port:=iValDef(ReadIni('ModbusPort'),1);
  Success('ModbusPort='+Str(UDGB01.Modbus.Port));
  UDGB01.Modbus.UnitId:=iValDef(ReadIni('ModbusUnitId'),1);
  Success('ModbusUnitId='+Str(UDGB01.Modbus.UnitId));
  UDGB01.Modbus.Polling:=iValDef(ReadIni('ModbusPolling'),1000);
  Success('ModbusPolling='+Str(UDGB01.Modbus.Polling));
  UDGB01.Modbus.Timeout:=iValDef(ReadIni('ModbusTimeout'),250);
  Success('ModbusTimeout='+Str(UDGB01.Modbus.Timeout));
  UDGB01.Modbus.Deadline:=iValDef(ReadIni('ModbusDeadline'),60000);
  Success('ModbusDeadline='+Str(UDGB01.Modbus.Deadline));
  UDGB01.Modbus.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000);
  Success('DelayOnStart='+Str(UDGB01.Modbus.DelayOnStart));
  //
  // Initialize Cmd command table & clear Poll record.
  //
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  InitCmdTable;
 end;
 //
 // UDGB01 Driver Finalization.
 //
 procedure UDGB01_Free;
 begin
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // UDGB01 Driver Polling.
 //
 procedure UDGB01_Poll;
 begin
  UDGB01_GUI_POLL;
  if mSecNow-FixmSecNow>UDGB01.Modbus.DelayOnStart then
  if IsPortOpened then begin
   if UDGB01.Simulator
   then UDGB01_SIM_POLL;
   UDGB01_CMD_POLL;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  UDGB01_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('@StartupScript','');
  StdIn_SetTimeouts(0,0,MaxInt,0);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  InitNetLibrary;
  UDGB01_Init;
  cmd_ZeroPortCounters      := RegisterStdInCmd('@ZeroPortCounters',      '');
  cmd_DimTagUpdate          := RegisterStdInCmd('@DimTagUpdate',          '');
  cmd_ClearModbusSumm       := RegisterStdInCmd('@ClearModbusSumm',       '');
  cmd_PARAM_YELVA           := RegisterStdInCmd('@PARAM.YELVA',           '');
  cmd_PARAM_REDVA           := RegisterStdInCmd('@PARAM.REDVA',           '');
  cmd_PARAM_DRYVA           := RegisterStdInCmd('@PARAM.DRYVA',           '');
  cmd_PARAM_RTIME           := RegisterStdInCmd('@PARAM.RTIME',           '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',                  '');
  cmd_Remote                := RegisterStdInCmd('@Remote',                '');
  cmd_LoadIni               := RegisterStdInCmd('@LoadIni',               '');
  cmd_SaveIni               := RegisterStdInCmd('@SaveIni',               '');
  cmd_MenuToolsOpen         := RegisterStdInCmd('@MenuToolsOpen',         '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  FreeNetLibrary;
  UDGB01_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  UDGB01_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,dat,par:String; cmdid:Integer; ref,cid,tim,port,uid,fid,tag:Integer; ms:Real;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  dat:='';
  par:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Modbus.Reply ref cid tim port uid fid $$dat
   }
   if (cmdid=cmd_NetModbusReply) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then UDGB01_OnReply(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Poll ref cid tim port uid fid $$dat
   This message uses in simulation mode only
   }
   if (cmdid=cmd_NetModbusPoll) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then UDGB01_OnSimPoll(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    Data:='';
   end else
   {
   @Modbus.Refuse ...
   }
   if (cmdid=cmd_NetModbusRefuse) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,0)))
    else Trouble(GotBug(cmd+' '+arg));
    ClearModbusPoll;
    UDGB01_SetStateIndic(st_REFUSE,'CONNECTION REFUSED');
    Data:='';
   end else
   {
   @Modbus.Timeout ...
   }
   if (cmdid=cmd_NetModbusTimeout) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,64)))
    else Trouble(GotBug(cmd+' '+arg));
    UDGB01_SetStateIndic(st_TIMEOUT,'TIMEOUT DETECTED');
    ClearModbusPoll;
    Data:='';
   end else
   {
   @DimGuiClick=<mime_encoded_encrypted_remote_click>
   if IsSameText('@DimGuiClick',cmd) then begin
    DIM_GuiClickHandler(arg);
    Data:='';
   end else
   }
   {
   @ZeroPortCounters 0
   }
   if (cmdid=cmd_ZeroPortCounters) then begin
    if not DIM_IsClientMode then begin
     DevSendCmd(devModbusProxy,cmd+' '+arg);
    end;
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if tag=UDGB01.PARAM.GASVA.tag then UpdateAo(ao_PARAM_GASVA,time,rGetTag(tag));
     if tag=UDGB01.PARAM.YELVA.tag then UpdateAo(ao_PARAM_YELVA,time,rGetTag(tag));
     if tag=UDGB01.PARAM.REDVA.tag then UpdateAo(ao_PARAM_REDVA,time,rGetTag(tag));
     if tag=UDGB01.PARAM.DRYVA.tag then UpdateAo(ao_PARAM_DRYVA,time,rGetTag(tag));
     if tag=UDGB01.PARAM.RESRC.tag then UpdateAo(ao_PARAM_RESRC,time,rGetTag(tag));
     if tag=UDGB01.PARAM.DTIME.tag then ;
     if tag=UDGB01.STATE.FLAGS.tag then UpdateDo(do_STATE_FLAGS,time,iGetTag(tag));
     if tag=UDGB01.STATE.INDIC.tag then UpdateDo(do_STATE_INDIC,time,iGetTag(tag));
     if tag=UDGB01.POLLRATE.RX.tag then UpdateDo(do_POLLRATERX,time,rGetTag(tag));
     if tag=UDGB01.POLLRATE.TX.tag then UpdateDo(do_POLLRATETX,time,rGetTag(tag));
     if tag=UDGB01.POLLRATE.EX.tag then UpdateDo(do_POLLRATEEX,time,rGetTag(tag));
     if tag=UDGB01.ERROR.COUNT.tag then UpdateDo(do_ERRORCOUNT,time,rGetTag(tag));
     if tag=UDGB01.POLLSUMM.RX.tag then UpdateDo(do_POLLSUMMRX,time,rGetTag(tag));
     if tag=UDGB01.POLLSUMM.TX.tag then UpdateDo(do_POLLSUMMTX,time,rGetTag(tag));
     if tag=UDGB01.POLLSUMM.EX.tag then UpdateDo(do_POLLSUMMEX,time,rGetTag(tag));
    end;
   end else
   {
   @ClearModbusSumm
   }
   if (cmdid=cmd_ClearModbusSumm) then begin
    if not DIM_IsClientMode then begin
     ClearModbusSumm;
    end;
    Data:='';
   end else
   {
   @PARAM.YELVA 1.234
   }
   if (cmdid=cmd_PARAM_YELVA) then begin
    if not DIM_IsClientMode then begin
     HoldCmdOpData(cm_WritYELVA,rVal(ExtractWord(1,arg)));
    end;
    Data:='';
   end else
   {
   @PARAM.REDVA 1.234
   }
   if (cmdid=cmd_PARAM_REDVA) then begin
    if not DIM_IsClientMode then begin
     HoldCmdOpData(cm_WritREDVA,rVal(ExtractWord(1,arg)));
    end;
    Data:='';
   end else
   {
   @PARAM.DRYVA 1.234
   }
   if (cmdid=cmd_PARAM_DRYVA) then begin
    if not DIM_IsClientMode then begin
     HoldCmdOpData(cm_WritDRYVA,rVal(ExtractWord(1,arg)));
    end;
    Data:='';
   end else
   {
   @PARAM.DTIME Now
   @PARAM.RTIME 2016.12.22-08:16:22.123
   }
   if (cmdid=cmd_PARAM_RTIME) or IsSameText(cmd,'@PARAM.DTIME') then begin
    if not DIM_IsClientMode then begin
     if IsSameText(ExtractWord(1,arg),'Now') then ms:=mSecNow else 
     ms:=extract_datetime(ExtractWord(1,arg));
     HoldCmdOpData(cm_WritRTIME,ms);
    end;
    Data:='';
   end else
   {
   @Edit PARAM.YELVA
   @Edit PARAM.DTIME Now
   }
   if (cmdid=cmd_Edit) then begin
    par:=mime_decode(ExtractWord(2,arg));
    if IsSameText(ExtractWord(1,arg),'PARAM.YELVA')
    then StartEditTagEx(UDGB01.PARAM.YELVA.tag,'Уставка YELLOW - предупредительный порог, Бк/м3',par);
    if IsSameText(ExtractWord(1,arg),'PARAM.REDVA')
    then StartEditTagEx(UDGB01.PARAM.REDVA.tag,'Уставка RED - аварийный порог, Бк/м3',par);
    if IsSameText(ExtractWord(1,arg),'PARAM.DRYVA')
    then StartEditTagEx(UDGB01.PARAM.DRYVA.tag,'Уставка DRY - порог сухого контакта, Бк/м3',par);
    if IsSameText(ExtractWord(1,arg),'PARAM.DTIME') then begin
     if IsSameText(ExtractWord(2,arg),'Now')
     or not IsValidDateTime(extract_datetime(sGetTag(UDGB01.PARAM.DTIME.tag)))
     then bNul(sSetTag(UDGB01.PARAM.DTIME.tag,GetDateTime(mSecNow)));
     StartEditTagEx(UDGB01.PARAM.DTIME.tag,'Уставка DATE TIME - время по часам устройства',par);
    end;
    Data:='';
   end else
   {
   @Remote
   }
   if (cmdid=cmd_Remote) then begin
    if not IsEmptyStr(arg) then PostCmdRemote(Trim(arg));
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    if not DIM_IsClientMode then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    if not DIM_IsClientMode then begin
     iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   @MenuToolsOpen
   }
   if (cmdid=cmd_MenuToolsOpen) then begin
    MenuToolsStarter;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  dat:='';
  par:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
