////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020-2023 Alexey Kuryakin kouriakine@mail.ru - LGPL license. //
////////////////////////////////////////////////////////////////////////////////

unit dpGuiUtils; // Diesel Pascal routines for useful for GUI.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpWindows,dpVbs,dpColors,dpLinux;

// Locate Form to center of Screen, set new size in percents.
procedure FormCenterToScreen(Form:TForm; PercentW,PercentH:Integer);

// Set crHandPoint cursor for all button types.
procedure SetButtonCursorToHandPoint(Comp:TComponent);

// Set Font.Name to default value.
procedure SetDefaultFontName(Font:TFont);
procedure SetDefaultMonoFontName(Font:TFont);
procedure SetDefaultSansFontName(Font:TFont);
procedure SetDefaultSerifFontName(Font:TFont);
procedure SetDefaultRomanFontName(Font:TFont);
procedure SetDefaultNarrowFontName(Font:TFont);

const // Constants for DefaultFontType.
 ft_Mono=0; ft_Sans=1; ft_Serif=2; ft_Roman=3; ft_Narrow=4;

var // Type of SetDefaultFontName.
 DefaultFontType : Integer = ft_Mono;

// Find available font name from font name list or return fallback value.
// Font.Name:=GetAvailableFontName(DefaultMonoFonts,FallbackMonoFont);
function GetAvailableFontName(FontNameList,Fallback:String):String;

const // List of popular Mono Fonts to use by default.
 DefaultMonoFonts = 'PT Mono, Consola Mono, Consolas, '
                  + 'Code New Roman, Source Code Pro, '
                  + 'Liberation Mono, DejaVu Sans Mono, '
                  + 'Droid Sans Mono, Roboto Mono, Fira Mono, '
                  + 'Ubuntu Mono, Cousine, Anonymous Pro, FreeMono'
                  + 'Courier New, Courier, Lucida Console, Fixedsys';
 DefaultSansFonts = 'PT Sans, PT Astra Sans, Sans, Liberation Sans, '
                  + 'DejaVu Sans, Droid Sans, FreeSans, Fira Sans, Source Sans Pro, '
                  + 'Arial, Verdana, Tahoma, Lucida Sans, Microsoft Sans Serif, MS Sans Serif';
 DefaultSerifFonts = 'PT Serif, PT Astra Serif, Serif, Liberation Serif, '
                  + 'DejaVu Serif, Droid Serif, FreeSerif, Source Serif Pro'
                  + 'Microsoft Sans Serif, MS Serif';
 DefaultRomanFonts = 'PT Astra Serif, Times New Roman, Roman, '
                  + 'Code New Roman, Royal Times New Roman, MS Serif';
 DefaultNarrowFonts = 'PT Sans Narrow, Liberation Sans Narrow, Arial Narrow, '
                    + 'Microsoft Sans Serif, MS Sans Serif';

const // Fallback font if none found.
 FallbackMonoFont   = 'Courier New';
 FallbackSansFont   = 'MS Sans Serif';
 FallbackSerifFont  = 'MS Serif';
 FallbackRomanFont  = 'MS Serif';
 FallbackNarrowFont = 'MS Sans Serif';

 // Name:PT_Mono\Size:10\Color:Black\Style:[Bold]
function GetFontStyleAsText(Font:TFont):String;
function GetFontStylesAsText(Styles:TFontStyles):String;
function GetSetFontParams(Font:TFont; Params:String):String;

 // Get Application main Form
function GetApplicationMainForm:TForm;
function GetApplicationMainFormWindowId:String;

 // Hide Application icon from TaskBar. To use with TrayIcon.
procedure HideApplicationIconFromTaskBar(Hide:Boolean);
procedure HandlePendingHideApplicationIconFromTaskBar;
var LastPendingHideApplicationIconFromTaskBar:Integer=0;

 // Show window by handle and bring to front.
procedure ShowAndActivateFormByHandle(Handle:THandle; nShow:Integer);

 // Show window by reference and bring to front.
procedure ShowAndActivateFormByReference(Form:TForm; nShow:Integer);

 // Get registered Application Path from Registry.
 // Example: firefox:=GetRegAppPath('firefox.exe')
function GetRegAppPath(app:String):String;

 // Get HTML Browser from applist list.
 // if applist is empty, use applist=WellKnownHtmlBrowsers.
 // mode bits: 1=quoted; 2=check file existance
function GetHtmlBrowser(applist:String=''; mode:Integer=3):String;

var // List of well known HTML browsers.
 WellKnownHtmlBrowsers:String='firefox.exe opera.exe chrome.exe tor.exe iexplore.exe';

 // ListSysInfoAsText(what) - list system information as text by (what) query
 // list environ            - list environment of process
 // list session            - list session information
 // list process 0 0 *      - list process pid ppid exe
 // list windows 0 * *      - list process pid class title
 // list modules 0          - list modules pid
 // exec wmctrl -l          - list windows (linux only)
 // exec wmctrl -l -p       - list windows (with pid`s)
 // exec c                  - execute command c
function ListSysInfoAsText(what:String; timeout:Integer=5000):String;
function GetItemsOfListSysInfoAsText:String; // For ComboBox.Items

implementation

procedure FormCenterToScreen(Form:TForm; PercentW,PercentH:Integer);
begin
 if (Form<>nil) then with Form do begin
  if (PercentW>0) then Width:=(ScreenWidth * PercentW) div 100;
  if (PercentH>0) then Height:=(ScreenHeight * PercentH) div 100;
  Left:=(ScreenWidth-Width) div 2;
  Top:=(ScreenHeight-Height) div 2;
 end;
end;

procedure SetButtonCursorToHandPoint(Comp:TComponent);
var i:Integer;
begin
 if (Comp <> nil) then
 if (Comp is TButton) then TButton(Comp).Cursor:=crHandPoint else
 if (Comp is TBitBtn) then TBitBtn(Comp).Cursor:=crHandPoint else
 if (Comp is TSpeedButton) then TSpeedButton(Comp).Cursor:=crHandPoint else
 if (Comp is TForm) then begin for i:=0 to TForm(Comp).ComponentCount-1 do SetButtonCursorToHandPoint(TForm(Comp).Components[i]); end;
end;

function GetAvailableFontName(FontNameList,Fallback:String):String;
var i:Integer; S,Delims:String;
begin
 Result:=Fallback; Delims:=';,'+LineEnding;
 for i:=1 to WordCount(FontNameList,Delims) do begin
  S:=Trim(ExtractWord(i,FontNameList,Delims));
  if (S<>'') and (Screen.Fonts.IndexOf(S)>=0) then begin
   Result:=S;
   Break;
  end;
 end;
end;

procedure SetDefaultFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 case DefaultFontType of
  0: SetDefaultMonoFontName(Font);
  1: SetDefaultSansFontName(Font);
  2: SetDefaultSerifFontName(Font);
  3: SetDefaultRomanFontName(Font);
  4: SetDefaultNarrowFontName(Font);
 end;
end;

procedure SetDefaultMonoFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 Font.Name:=GetAvailableFontName(DefaultMonoFonts,FallbackMonoFont);
end;

procedure SetDefaultSansFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 Font.Name:=GetAvailableFontName(DefaultSansFonts,FallbackSansFont);
end;

procedure SetDefaultSerifFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 Font.Name:=GetAvailableFontName(DefaultSerifFonts,FallbackSerifFont);
end;

procedure SetDefaultRomanFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 Font.Name:=GetAvailableFontName(DefaultRomanFonts,FallbackRomanFont);
end;

procedure SetDefaultNarrowFontName(Font:TFont);
begin
 if (Font=nil) then Exit;
 Font.Name:=GetAvailableFontName(DefaultNarrowFonts,FallbackNarrowFont);
end;

function GetFontStylesAsText(Style:TFontStyles):String;
begin
 Result:='';
 if (Styles=[]) then begin Result:='[]'; Exit; end;
 if (fsBold in Styles) then Result:=Result+' Bold';
 if (fsItalic in Styles) then Result:=Result+' Italic';
 if (Result='') then Result:='Normal';
 if (fsStrikeOut in Styles) then Result:=Result+' StrikeOut';
 if (fsUnderline in Styles) then Result:=Result+' Underline';
 Result:='['+StringReplace(Trim(Result),' ',',',rfReplaceAll)+']';
end;

function GetFontStyleAsText(Font:TFont):String;
begin
 Result:='';
 if (Font=nil) then Exit;
 Result:=GetFontStylesAsText(Font.Style);
end;

function GetSetFontParams(Font:TFont; Params:String):String;
var Lines:TStrings; s:String;
begin
 Result:='';
 try
  Lines:=TStringList.Create;
  try
   Params:=Trim(Params);
   if (Params<>'') then begin
    Params:=StringReplace(Params,'_',' ',rfReplaceAll);
    Params:=StringReplace(Params,':','=',rfReplaceAll);
    Params:=StringReplace(Params,'\',LineEnding,rfReplaceAll);
    Lines.Text:=AdjustLineBreaks(Params);
   end;
   if (Font<>nil) then begin
    s:=Trim(Lines.Values['Name']);  if (s<>'') then Font.Name:=s;
    s:=Trim(Lines.Values['Size']);  if (s<>'') then Font.Size:=StrToIntDef(s,Font.Size);
    s:=Trim(Lines.Values['Color']); if (s<>'') then Font.Color:=StrToIntDef(s,Font.Color);
    s:=Trim(Lines.Values['Style']);
    if (s<>'') then begin
     if (WordIndex('Bold',s,'[,]')>0) then Font.Style:=Font.Style+[fsBold] else Font.Style:=Font.Style-[fsBold];
     if (WordIndex('Italic',s,'[,]')>0) then Font.Style:=Font.Style+[fsItalic] else Font.Style:=Font.Style-[fsItalic];
     if (WordIndex('StrikeOut',s,'[,]')>0) then Font.Style:=Font.Style+[fsStrikeOut] else Font.Style:=Font.Style-[fsStrikeOut];
     if (WordIndex('Underline',s,'[,]')>0) then Font.Style:=Font.Style+[fsUnderline] else Font.Style:=Font.Style-[fsUnderline];

    end;
    Lines.Clear;
    Lines.Values['Name']:=Font.Name;
    Lines.Values['Size']:=IntToStr(Font.Size);
    Lines.Values['Color']:=ColorToString(Font.Color);
    Lines.Values['Style']:=GetFontStyleAsText(Font);
   end;
   Result:=Trim(Lines.Text);
   Result:=StringReplace(Result,' ','_',rfReplaceAll);
   Result:=StringReplace(Result,'=',':',rfReplaceAll);
   Result:=StringReplace(Result,LineEnding,'\',rfReplaceAll);
  finally
   Lines.Free;
  end;
 except
  on E:Exception do BugReport(E,Application,'GetSetFontParams');
 end;
end;

function GetApplicationMainForm:TForm;
begin
 Result:=nil;
 if Application=nil then Exit;
 Result:=Application.MainForm;
end;

function GetApplicationMainFormWindowId:String;
var Form:TForm; List:TStringList; i:Integer; pid,line,delims:String;
begin
 Result:='';
 try
  Form:=GetApplicationMainForm;
  if not (Form is TForm) then Exit;
  if IsWindows then begin
   Result:=FormatVar('0x%x',Form.Handle);
  end else
  if IsUnix then begin
   List:=TStringList.Create;
   try
    pid:=IntToStr(GetProcessId);
    delims:=LineEnding+' '+vbTab;
    List.Text:=execute_wmctrl('-l -p');
    for i:=0 to List.Count-1 do begin
     line:=Trim(List.Strings[i]);
     if SameText(ExtractWord(3,line,delims),pid) then
     if SameText(SkipWords(4,line,delims),Trim(Form.Caption)) then begin
      Result:=ExtractWord(1,line,delims);
      Break;
     end;
    end;
   finally
    List.Free;
   end;
  end;
 except
  on E:Exception do BugReport(E,Application,'GetApplicationMainFormWindowId');
 end;
end;

procedure HideApplicationIconFromTaskBar(Hide:Boolean);
var GWL:Integer; arg,wid:String; Form:TForm;
const UsesWindowName=false;
begin
 if IsWindows then begin
  GWL:=WinApi.GetWindowLong(Application.Handle, GWL_EXSTYLE);
  if (Hide)
  then GWL:=GWL OR WS_EX_TOOLWINDOW AND NOT WS_EX_APPWINDOW
  else GWL:=GWL OR WS_EX_APPWINDOW AND NOT WS_EX_TOOLWINDOW;
  WinApi.SetWindowLong(Application.Handle, GWL_EXSTYLE, GWL);
  LastPendingHideApplicationIconFromTaskBar:=0;
 end else
 if IsUnix then begin
  Form:=GetApplicationMainForm;
  if not (Form is TForm) then Exit;
  if FileExists(file_which('wmctrl')) then begin
   wid:=GetApplicationMainFormWindowId;
   if (wid<>'') then begin
    LogEvents(Now,wid+' = '+FormatVar('"%s"',Form.Caption));
    if UsesWindowName then begin
     if Hide
     then arg:=FormatVar('-r "%s" -b add,skip_taskbar',Form.Caption)
     else arg:=FormatVar('-r "%s" -b remove,skip_taskbar',Form.Caption);
    end else begin
     if Hide
     then arg:=FormatVar('-i -r "%s" -b add,skip_taskbar',wid)
     else arg:=FormatVar('-i -r "%s" -b remove,skip_taskbar',wid);
    end;
    if (arg<>'') then LogEvents(Now,'wmctrl '+arg+LineEnding+execute_wmctrl(arg));
   end;
   if Hide
   then LastPendingHideApplicationIconFromTaskBar:=+1
   else LastPendingHideApplicationIconFromTaskBar:=-1;
  end;
 end;
end;

procedure HandlePendingHideApplicationIconFromTaskBar;
begin
 if (LastPendingHideApplicationIconFromTaskBar<>0) then begin
  HideApplicationIconFromTaskBar(LastPendingHideApplicationIconFromTaskBar>0);
  LastPendingHideApplicationIconFromTaskBar:=0;
 end;
end;

procedure ShowAndActivateFormByHandle(Handle:THandle; nShow:Integer);
begin
 if (Handle<>0) and WinApi.IsWindow(Handle) then begin
  WinApi.ShowWindow(Handle,nShow);
  WinApi.BringWindowToTop(Handle);
  WinApi.SetForegroundWindow(Handle);
 end;
end;

procedure ShowAndActivateFormByReference(Form:TForm; nShow:Integer);
begin
 if (Form is TForm) then
 if IsWindows then begin
  ShowAndActivateFormByHandle(Form.Handle,nShow);
 end else begin
  Application.Restore; Application.BringToFront;
  Form.Show; Form.WindowState:=wsNormal; Form.SetFocus;
 end;
end;

function GetRegAppPath(app:String):String;
const regPath='HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\';
begin
 Result:=WshRegRead(regPath+app+'\');
end;

function GetHtmlBrowser(applist:String=''; mode:Integer=3):String;
var i:Integer; app:String; const delims=', ;';
begin
 Result:='';
 if (applist='') then applist:=WellKnownHtmlBrowsers;
 for i:=1 to WordCount(applist,delims) do begin
  app:=ExtractWord(i,applist,delims);
  if IsWindows
  then Result:=Trim(GetRegAppPath(app))
  else Result:=file_which(ReplaceFileExt(app,''));
  if HasFlags(mode,2) then // Check file existance if needed
  if (Result<>'') and not FileExists(Result) then Result:='';
  if (Result<>'') then break;
 end;
 if (Pos(' ',Result)>0) and HasFlags(mode,1) then Result:=AnsiQuotedStr(Result,'"');
end;

function ListSysInfoAsText(what:String; timeout:Integer=5000):String;
var Lines:TStringList; cmd,arg,exe,opt,cls,tit,delims:String; pid,ppid,ExitCode,aff:Integer;
begin
 Result:='';
 if (what<>'') then
 try
  delims:=LineEnding+', ;'+vbTab;
  Lines:=TStringList.Create;
  try
   cmd:=ExtractWord(1,what,delims);
   arg:=Trim(SkipWords(1,what,delims));
   if SameText(cmd,'list') then begin
    cmd:=ExtractWord(2,what,delims);
    arg:=Trim(SkipWords(2,what,delims));
    if SameText(cmd,'environ') then begin
     pid:=StrToIntDef(ExtractWord(1,arg,delims),0);
     if (pid=0)
     then Lines.Text:=EnvironmentVariableList.Text
     else Lines.Text:=StringReplace(read_proc_pid_file(pid,'environ'),Chr(0),LineEnding,rfReplaceAll);
    end else
    if SameText(cmd,'session') then begin
     aff:=GetPidAffinityMask(0);
     Lines.Add('HOSTNAME='+GetHostName);
     Lines.Add('USERNAME='+GetUserName);
     Lines.Add('AFFINITY=$'+FormatVar('%x',aff));
     Lines.Add('BROWSER='+GetHtmlBrowser);
     Lines.Add('SHELL='+GetShell);
     Lines.Add('HOME='+GetHomeDir);
     Lines.Add('TEMP='+GetTempDir);
     Lines.Add('OS='+GetOsName);
    end else
    if SameText(cmd,'process') then begin
     pid:=StrToIntDef(ExtractWord(1,arg,delims),0);
     ppid:=StrToIntDef(ExtractWord(2,arg,delims),0);
     exe:=StringReplace(Trim(SkipWords(2,arg,delims)),'*','',rfReplaceAll);
     Lines.Text:=GetListOfProcesses(pid,ppid,exe);
    end else
    if SameText(cmd,'windows') then begin
     pid:=StrToIntDef(ExtractWord(1,arg,delims),0);
     cls:=StringReplace(ExtractWord(2,arg,delims),'*','',rfReplaceAll);
     tit:=StringReplace(Trim(SkipWords(2,arg,delims)),'*','',rfReplaceAll);
     if IsWindows
     then Lines.Text:=GetListOfWindows(pid,cls,tit)
     else Lines.Text:=execute_wmctrl('-l -p');
    end else
    if SameText(cmd,'modules') then begin
     pid:=StrToIntDef(ExtractWord(1,arg,delims),GetProcessId);
     if IsWindows
     then Lines.Text:=GetListOfModules(pid)
     else Lines.Text:=read_proc_pid_modules(pid);
    end;
   end else
   if SameText(cmd,'exec') then begin
    cmd:=ExtractWord(2,what,delims);
    arg:=Trim(SkipWords(2,what,delims));
    if SameText(cmd,'wmctrl') then begin
     if IsWindows
     then Lines.Add('wmctrl is not avail in Windows')
     else Lines.Text:=execute_wmctrl(arg);
    end else
    if (cmd<>'') then begin
     opt:=execute_command_as_text_defaults('$TMPDIR','SW_HIDE',timeout);
     Lines.Text:=execute_command_as_text(Trim(cmd+' '+arg),pid,ExitCode,opt);
    end;
   end;
   Result:=Lines.Text;
   if (Result='') then Result:='Error';
  finally
   Lines.Free;
  end;
 except
  on E:Exception do BugReport(E,Application,'ListSysInfoAsText');
 end;
end;

function GetItemsOfListSysInfoAsText:String;
var List:TStringList;
begin
 Result:='';
 try
  List:=TStringList.Create;
  try
   List.Add('list environ');
   List.Add('list session');
   List.Add('list modules 0');
   List.Add('list process 0 0 *');
   List.Add('list windows 0 * *');
   if IsWindows then begin
    // Nothing to add
   end else
   if IsUnix then begin
    List.Add('list environ 0');
    List.Add('exec wmctrl -l');
    List.Add('exec wmctrl -l -p');
    List.Add('exec wmctrl -l -p -x');
    List.Add('exec lscpu');
    List.Add('exec lsblk');
    List.Add('exec lsipc');
    List.Add('exec lsusb');
    List.Add('exec who');
    List.Add('exec df');
   end;
   Result:=List.Text;
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,Application,'GetItemsOfListSysInfoAsText');
 end;
end;

end.
