////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Утилита сортирует точки кривой.
Это часто бывает нужно, например, если точки кривой были заданы в произвольном
порядке, в то время как большинство математических утилит предполагают,
что абсциссы точек кривой X[i] расположены в порядке возрастания: X[i+1]>=X[i].
Для сортировки надо указать ключ сортировки (x или y), 
направление ( > возрастание, < убывание ), упаковка ( + или -), 
абсолютный и относительный триггер упаковки. Упаковка означает, что точки,
ключ которых с заданной триггерами точностью совпадают, объединяются в одну
среднюю точку.
[]
[Manual.Eng]
This utility uses to sort curve points.
Most of plugins works only with curves, sorted along X axis: X[i+1]>=X[i].
If curve points has random order, it should be sorted.
To sort curve, use Key (x or y), Direction ( < decrease or > increase),
Pack ( + or - ), absolute and relative trigger for Pack.
Packing is the next: if some keys equals with tolerrance given by triggers,
this point should be replaced to single point with average value.
[]
[Arguments.Rus]
Имя окна = Результат:"Сортировка кривой"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
Ключ = x
Направление = >
Упаковка = -
Абс.Триггер = 0
Отн.Триггер = 0.01
[]
[Arguments.Eng]
Caption = Result:"Curve sort"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
Key = x
Direction = >
Pack = -
Abs.Trigger = 0
Rel.Trigger = 0.01
[]
}

library _curve_sort;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfNanInf;
var
 SortFlags, c1, c2, cFirst, cLast : Integer;
 AbsEps, RelEps : Double;
 skey,sdir,spak:ShortString;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('Неверное значение Target!',
                                   'Invalid Target!'));
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  skey:=LoCaseStr(GetArgumentAsString(RusEng('Ключ','Key')));
  sdir:=LoCaseStr(GetArgumentAsString(RusEng('Направление','Direction')));
  spak:=LoCaseStr(GetArgumentAsString(RusEng('Упаковка','Pack')));
  if ((skey<>'x') and (skey<>'y')) or
     ((sdir<>'<') and (sdir<>'>')) or
     ((spak<>'+') and (spak<>'-')) or
     not GetArgumentAsDouble(RusEng('Абс.Триггер','Abs.Trigger'),AbsEps) or
     not GetArgumentAsDouble(RusEng('Отн.Триггер','Rel.Trigger'),RelEps)
  then Raise EDanApi.Create(RusEng('Ошибка задания аргументов!',
                                   'Invalid input arguments!'));
  SortFlags:=ord(skey='y')*sfPrimaryY+
             ord(sdir='<')*sfReversPrimary+
             ord(spak='+')*sfPack;
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для сортировки!',
                                           'Curve %d is not convenient for sort!'),[c1]));
   c2:=CreateCurve(twin);
   CurveAssign(c2,c1);
   CurveSort(c2,SortFlags,AbsEps,RelEps);
   if CurveFlags(c2,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Ошибка сортировки кривой %d !',
                                           'Error on curve %d sort !'),[c1]));
   Refreshment(100);
  end;
  if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;			
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
 
