////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2022 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpCrcUtils; // Helper routines for CRC file construction.

interface

uses  dpCmdArgs,dpSystem,dpSysUtils,dpVbs,dpWindows,dpGuiUtils,dpCrwDaqSys,
      dpDaqConfig,dpColors;

type
 TCrcMaker = class(TObject)
 private
  myLines      : TStringList;
  mySensorName : String;
 public
  constructor Create;
  destructor  Destroy;
 public
  procedure ClearVars;
  procedure Add(Line:String);
  procedure AddSensor(id:String);
  procedure AddPos(x,y:Integer);
  procedure AddLed(w,d,v:Integer; Fmt,Font:String);
  procedure AddTagEval(expression:String);
  procedure AddPainter(expression:String);
  procedure AddHint(hint:String);
  procedure AddTag(n,tag:Integer; bmp,title:String);
  function  UndotLowerName(s:String):String;
  function  StdBarBmp(w,h,bits:Integer; color:String):String;
 public
  property  Lines      : TStringList read myLines;
  property  SensorName : String      read mySensorName;
 end;

implementation

constructor TCrcMaker.Create(aLines:TStrings);
begin
 inherited Create;
 myLines:=TStringList.Create;
 ClearVars;
end;

destructor TCrcMaker.Destroy;
begin
 ClearVars;
 myLines.Free;
 inherited Destroy;
end;

procedure TCrcMaker.ClearVars;
begin
 mySensorName:='';
end;

procedure TCrcMaker.Add(Line:String);
begin
 if (Lines<>nil) then Lines.Add(Line);
end;

procedure TCrcMaker.AddSensor(id:String);
begin
 id:=Trim(id);
 mySensorName:=id;
 Add('[SensorList]');
 Add('Sensor = '+id);
 Add('['+id+']');
end;

procedure TCrcMaker.AddPos(x,y:Integer);
begin
 Add('Pos = '+IntToStr(x)+', '+IntToStr(y));
end;

procedure TCrcMaker.AddLed(w,d,v:Integer; Fmt,Font:String);
var line:String;
begin
 w:=imax(0,w); d:=imax(0,d);
 Font:=Trim(Font); Fmt:=Trim(Fmt); if (Fmt='') then Fmt:='*';
 line:='LED = '+IntToStr(w)+', '+IntToStr(d)+', '+IntToStr(v)+', '+Fmt;
 if (Font<>'') then line:=Line+', '+Font;
 Add(line);
end;

procedure TCrcMaker.AddTagEval(expression:String);
begin
 if IsEmptyStr(expression) then Exit;
 Add('TagEval(v) = '+expression);
end;

procedure TCrcMaker.AddPainter(expression:String);
begin
 if IsEmptyStr(expression) then Exit;
 Add('Painter(v) = '+expression);
end;

procedure TCrcMaker.AddHint(hint:String);
begin
 if IsEmptyStr(hint) then Exit;
 Add('Hint = '+hint);
end;

procedure TCrcMaker.AddTag(n,tag:Integer; bmp,title:String);
begin
 Add(Trim('Tag#'+IntToStr(n)+' = '+IntToStr(tag)+', '+Trim(bmp)+' '+Trim(title)));
end;

function TCrcMaker.UndotLowerName(s:String):String;
begin
 Result:=LowerCase(StringReplace(s,'.','_',rfReplaceAll));
end;

function TCrcMaker.StdBarBmp(w,h,bits:Integer; color:String):String;
begin
 Result:='~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_'
        +IntToStr(w)+'_'+IntToStr(h)+'_'+IntToStr(bits)+'_'+Trim(color)+'.bmp';
end;



end.
