////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Application modal services.                                                //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20250902 - Created by A.K.                                                 //
////////////////////////////////////////////////////////////////////////////////

unit _crw_appmodal; // 

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
forms, controls, lmessages, lcltype,
 _crw_alloc, _crw_ef, _crw_fio, _crw_str, _crw_wmctrl,
 _crw_rtc, _crw_proc, _crw_polling, _crw_dbglog,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
 TAppModal = class(TMasterObject)
 private
  myLastPoll:Int64;
  myInterval:Integer;
 private
  function  GetInterval:Integer;
  procedure SetInterval(i:Integer);
  function  GetStarted:Boolean;
  procedure SetStarted(s:Boolean);
 public
  constructor Create(aInterval:Integer=5000);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property  Interval:Integer read GetInterval write SetInterval;
  property  Started:Boolean read GetStarted write SetStarted;
 public
  procedure Start;
  procedure Stop;
 public
  function  PollPulser:Boolean;
  function  AwakeModalWindows:Integer;
  function  ApplicationModalLevel:Integer;
  function  ProcessModalWindowsCount:Integer;
 public
  procedure ActivateTopmostModalForm;
  function  FindTopmostModalForm:TCustomForm;
  function  GetModalFormZOrderIndex(Form:TCustomForm):Integer;
 end;

function AppModal:TAppModal;

procedure Timer_CheckTopmostModalForm;

implementation

procedure Timer_CheckTopmostModalForm;
begin
 if IsMainThread then
 if AppModal.PollPulser
 then AppModal.AwakeModalWindows;
end;

///////////////////////////
// TAppModal implementation
///////////////////////////

constructor TAppModal.Create(aInterval:Integer=5000);
begin
 inherited Create;
 Interval:=aInterval;
end;

destructor TAppModal.Destroy;
begin
 inherited Destroy;
end;

procedure TAppModal.AfterConstruction;
begin
 inherited AfterConstruction;
end;

procedure TAppModal.BeforeDestruction;
begin
 Stop;
 inherited BeforeDestruction;
end;

function TAppModal.GetInterval:Integer;
begin
 if Assigned(Self)
 then Result:=myInterval
 else Result:=0;
end;

procedure TAppModal.SetInterval(i:Integer);
begin
 if Assigned(Self)
 then myInterval:=Max(0,i);
end;

function TAppModal.PollPulser:Boolean;
var ms,dt:Int64;
begin
 Result:=False;
 if Assigned(Self) then
 if (myInterval>0) then begin
  ms:=IntMsecNow; dt:=ms-myLastPoll;
  if (dt>=myInterval) then begin
   myLastPoll:=ms;
   Result:=True;
  end;
 end;
end;

function TAppModal.GetStarted:Boolean;
begin
 Result:=False;
 if Assigned(Self) then begin
  Result:=SecondActions.HasAction(Timer_CheckTopmostModalForm);
 end;
end;

procedure TAppModal.SetStarted(s:Boolean);
begin
 if Assigned(Self) then begin
  if s then Start else Stop;
 end;
end;

procedure TAppModal.Start;
begin
 if Assigned(Self) then begin
  SecondActions.Add(Timer_CheckTopmostModalForm);
 end;
end;

procedure TAppModal.Stop;
begin
 if Assigned(Self) then begin
  SecondActions.Remove(Timer_CheckTopmostModalForm);
 end;
end;

function TAppModal.ApplicationModalLevel:Integer;
begin
 if Assigned(Self) and Assigned(Application)
 then Result:=Application.ModalLevel
 else Result:=0;
end;

function pmwcIter(n:SizeInt; Line:LongString; Custom:Pointer):Boolean;
var Wnd:HWND; wsf:QWord; MatchFound:Boolean;
begin
 Result:=True;
 if IsNonEmptyStr(Line) then begin
  Wnd:=wmctrl.StrToWnd(ExtractWord(1,Line,ScanSpaces));
  if wmctrl.IsWindow(Wnd) then begin
   wsf:=wmctrl.WindowStateFlags(Wnd);
   MatchFound:=False;
   if HasFlags(wsf,WSF_MODAL) then begin
    MatchFound:=True;
   end;
   if MatchFound then begin
    if Assigned(Custom) then Inc(Integer(Custom^));
   end;
  end;
 end;
end;

function TAppModal.ProcessModalWindowsCount:Integer;
var wList:LongString; Count:Integer;
begin
 Result:=0; Count:=0;
 if Assigned(Self) then
 try
  wList:=wmctrl.ListWindows(GetCurrentProcessId,'','');
  if (ForEachStringLine(wList,pmwcIter,@Count)>0)
  then Result:=Count;
 except
  on E:Exception do BugReport(E,nil,'ProcessModalWindowsCount');
 end;
end;

function amwIter(n:SizeInt; Line:LongString; Custom:Pointer):Boolean;
var Wnd:HWND; wsf:QWord; MatchFound:Boolean;
begin
 Result:=True;
 if IsNonEmptyStr(Line) then begin
  Wnd:=wmctrl.StrToWnd(ExtractWord(1,Line,ScanSpaces));
  if wmctrl.IsWindow(Wnd) then begin
   wsf:=wmctrl.WindowStateFlags(Wnd);
   MatchFound:=False;
   if HasFlags(wsf,WSF_MODAL) then begin
    if HasFlags(wsf,WSF_HIDDEN) then MatchFound:=True;
   end;
   if MatchFound then begin
    wmctrl.SetWindowStateFlags(Wnd,WSC_REMOVE,WSF_HIDDEN);
    if Assigned(Custom) then Inc(Integer(Custom^));
   end;
  end;
 end;
end;

function TAppModal.AwakeModalWindows:Integer;
var wList:LongString; Count:Integer;
begin
 Result:=0; Count:=0;
 if Assigned(Self) then
 try
  wList:=wmctrl.ListWindows(GetCurrentProcessId,'','');
  if (ForEachStringLine(wList,amwIter,@Count)>0)
  then Result:=Count;
 except
  on E:Exception do BugReport(E,nil,'AwakeModalWindows');
 end;
end;

function TAppModal.FindTopmostModalForm:TCustomForm;
begin
 if Assigned(Self) and Assigned(Screen)
 then Result:=Screen.GetCurrentModalForm
 else Result:=nil;
end;

function TAppModal.GetModalFormZOrderIndex(Form:TCustomForm):Integer;
begin
 if Assigned(Self) and Assigned(Screen)
 then Result:=Screen.GetCurrentModalFormZIndex
 else Result:=-1;
end;

procedure TAppModal.ActivateTopmostModalForm;
var CForm:TCustomForm;
begin
 if IsMainThread then
 if Assigned(Self) then
 try
  CForm:=FindTopmostModalForm;
  if Assigned(CForm) then begin
   if (CForm.WindowState=wsMinimized)
   or (GetModalFormZOrderIndex(CForm)>0)
   or not CForm.Visible
   then CForm.ShowOnTop;
  end;
 except
  on E:Exception do BugReport(E,nil,'ActivateTopmostModalForm');
 end;
end;

//////////////////////////
// AppModal implementation
//////////////////////////
const
 TheAppModal:TAppModal=nil;

function AppModal:TAppModal;
begin
 if not Assigned(TheAppModal) then begin
  TheAppModal:=TAppModal.Create;
  TheAppModal.Master:=@TheAppModal;
 end;
 Result:=TheAppModal;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_appmodal;
begin
 AppModal.Ok;
end;

procedure Free_crw_appmodal;
begin
 Kill(TObject(TheAppModal));
end;

initialization

 Init_crw_appmodal;

finalization

 Free_crw_appmodal;

end.

//////////////
// END OF FILE
//////////////

