////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form CrwDaq WatchDog to detect hanging threads.                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231116 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_crwdaqwatchdog; // Form CrwDaq WatchDog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, strutils,
 Graphics, Interfaces, LMessages, lclintf,
 Forms, Controls, StdCtrls, Dialogs,
 Spin, ExtCtrls, ComCtrls,
 ActnList, Menus, ToolWin, ImgList, Printers, Clipbrd, Buttons, FileUtil,
 _crw_alloc, _crw_str, _crw_eldraw, _crw_guard, _crw_sect, _crw_ef,
 _crw_polling, _crw_fio, _crw_rtc, _crw_sesman, _crw_proc, _crw_gloss,
 _crw_appforms, _crw_apptools, _crw_apputils, _crw_task;

type

  { TFormCrwDaqWatchDog }

  TFormCrwDaqWatchDog = class(TMasterForm)
    TimerWdt: TTimer;
    PageControlWdt: TPageControl;
    TabSheetWdtStatus: TTabSheet;
    TabSheetWdtControl: TTabSheet;
    PanelWdtImage: TPanel;
    ImageWdt: TImage;
    PanelWdtList: TPanel;
    GroupBoxWdtList: TGroupBox;
    MemoWdtList: TMemo;
    GroupBoxWdtPeriod: TGroupBox;
    LabelWdtPeriod: TLabel;
    LabelWdtZero: TLabel;
    SpinEditWdtPeriod: TSpinEdit;
    LabelWdtAlert: TLabel;
    TabSheetWdtLog: TTabSheet;
    GroupBoxWdtLog: TGroupBox;
    LabelWdtLogPath: TLabel;
    LabelWdtLogName: TLabel;
    BitBtnWdtLogView: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormActivate(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure TimerWdtTimer(Sender: TObject);
    procedure SpinEditWdtPeriodChange(Sender: TObject);
    procedure PageControlWdtChange(Sender: TObject);
    procedure LabelWdtLogNameDblClick(Sender: TObject);
    procedure LabelWdtLogPathDblClick(Sender: TObject);
    procedure BitBtnWdtLogViewClick(Sender: TObject);
  private
    { Private declarations }
    myFileItem : Integer;
    myFileList : TStringList;
    procedure UpdateTimerState;
    procedure Alert(Alarm:Boolean);
  public
    { Public declarations }
    procedure UpdateLogStatus;
    procedure WatchDogLogView(exe:LongString='');
  end;

procedure InitWatchDogTimer;
procedure DoneWatchDogTimer;
procedure CheckWatchDogTimer;
procedure OpenFormCrwDaqWatchDog(Show:Boolean=true; const aParams:LongString='');
procedure HideFormCrwDaqWatchDog;
procedure KillFormCrwDaqWatchDog;
procedure SaveWatchdogLog(ms:Double; AppendMode:Boolean; msg:LongString);
function  GetWatchDogTimerLog:LongString;

implementation

{$R *.lfm}

const
 WatchDogTime       : Integer            = -1;
 WatchDogDelay      : Integer            = 1000*60*60*24;
 WatchDogUseTooltip : Boolean            = true;
 WatchDogCmdOpen    : LongString         = 'resource\shell\crwdaqopenwatchdog.cmd';
 WatchdogLog        : LongString         = '';
 WatchDogLogRec                          = 64;
 WatchDogLogTime    : Double             = 0;
 WatchDogLogLast    : Integer            = 0;
 FormCrwDaqWatchDog : TFormCrwDaqWatchDog = nil;
 fmtState           = '%2d hanging threads found';
 fmtLogin           = '%s %s@%s';

const
 WatchDogHangList:TStringList=nil; // List of hanging threads

procedure InitWatchDogHangList;
begin
 WatchDogHangList:=TStringList.Create;
end;

procedure FreeWatchDogHangList;
begin
 Kill(WatchDogHangList);
end;

function  GetWatchDogTimerLog:LongString;
begin
 Result:=WatchdogLog;
end;

procedure InitWatchDogTimer;
var b:Boolean;
begin
 SecondActions.Add(CheckWatchDogTimer);
 DoneSubSystems.Add(DoneWatchDogTimer);
 SaveWatchdogLog(msecnow,true,Format(fmtLogin,['++',UserName,ComputerName]));
 SaveWatchdogLog(msecnow,true,Format(fmtState,[0]));
 b:=WatchDogUseTooltip;
 if ReadIniFileBoolean(SysIniFile,SectSystem,'WatchDogUseTooltip%b',b)
 then WatchDogUseTooltip:=b;
 WatchDogCmdOpen:=SysGlossary.ReadIniParamDef(SysIniFile,SectSystem,'WatchDogCmdOpen',WatchDogCmdOpen);
end;

procedure DoneWatchDogTimer;
begin
 WatchDogTime:=0;
 KillFormCrwDaqWatchDog;
 SaveWatchdogLog(msecnow,true,Format(fmtLogin,['--',UserName,ComputerName]));
 WatchdogLog:='';
end;

function TooltipWatchdogMessage:LongString;
var pid:DWORD; ProgName:LongString;
begin
 Result:='';
 ProgName:=ParamStr(0); pid:=GetCurrentProcessId;
 if WatchDogUseTooltip then
 // The crwdaqopenwatchdog.cmd should send @menu run FormCrwDaq.ActionWindowsWatchDogControl
 Result:=Format('guid %s:WDT text "%s: %s" preset stdFails delay %d btn1 "%s" cmd1 "%s -p %d"'
               ,[SessionManager.SessionHead,SessionManager.SessionSign
               ,RusEng('СТОРОЖЕВОЙ ТАЙМЕР','WATCHDOG TIMER'),WatchDogDelay,RusEng('Открыть','Open')
               ,SmartFileRef(AdaptExeFileName(WatchDogCmdOpen),'',ProgName),pid]);
end;

procedure CheckPollingWdt(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
var Code,Counter,aTimeout,dTimeout:Integer; pName:LongString; List:TStringList; aPolling:TPolling;
begin
 if Assigned(CustomData) then
 if (aObject is TPolling) then begin
  aPolling:=TPolling(aObject);
  dTimeout:=WatchDogTime*MSecsPerSec;
  aTimeout:=aPolling.WdtTimeoutDef(dTimeout);
  if aPolling.WdtAlert(aTimeout) then begin
   pName:=aPolling.Name;
   List:=TStringList(CustomData);
   Echo(Format('%s WatchDog alert: %s',[StdDateTimeStr(msecnow),pName]));
   Counter:=List.Count;
   if (Counter=0) then begin
    if PreferTooltip and WatchDogUseTooltip
    then Code:=ShowTooltip(TooltipWatchdogMessage)
    else Code:=0;
    OpenFormCrwDaqWatchDog(Code=0);
   end;
   if FormCrwDaqWatchDog.Ok then begin
    if (FormCrwDaqWatchDog.MemoWdtList.Lines.IndexOf(pName)<0)
    then FormCrwDaqWatchDog.MemoWdtList.Lines.Add(pName);
   end;
   List.Add(pName);
  end;
 end;
end;

procedure UnixKillHangingChild;
var pid:TPid; cmd,ps,msg:LongString;
begin
 if IsUnix then begin
  cmd:=ExtractFileName(ProgName);
  ps:=GetListOfProcesses(0,GetCurrentProcessId,cmd,true,glops_FixName);
  pid:=StrToIntDef(ExtractWord(1,ps,ScanSpaces),0);
  if (pid<=0) then Exit;
  msg:=RusEng('Обнаружен подвисший','Detected hanging')+' PID '+IntToStr(pid);
  SendToMainConsole('@silent echo '+msg+EOL);
  if KillProcess(pid,1)
  then SendToMainConsole('@silent echo kill '+IntToStr(pid)+EOL);
 end;
end;

procedure CheckWatchDogTimer;
var n:Integer;
begin
 try
  n:=0;
  if (WatchDogTime<0) then begin
   if ReadIniFileInteger(SysIniFile,SectSystem,'WatchDogTime%i',WatchDogTime)
   then WatchDogTime:=max(0,WatchDogTime)
   else WatchDogTime:=0;
  end else
  if (WatchDogTime>0) then begin
   if Assigned(WatchDogHangList) then begin
    WatchDogHangList.Clear;
    FullPollingList.ForEach(CheckPollingWdt,WatchDogHangList);
    n:=WatchDogHangList.Count;
    if (n>0) then begin
     if SysLogNotable(SeverityOfWatchdog)
     then SysLogNote(0,SeverityOfWatchdog,sdr_System,'Thread(s) WatchDog alert: '+WatchDogHangList.CommaText);
     WatchDogHangList.Clear;
    end;
   end;
   if (msecnow-WatchDogLogTime>WatchDogTime*1000) then begin
    SaveWatchdogLog(msecnow,(n>0) or (WatchdogLogLast>0),Format(fmtState,[n]));
    WatchDogLogTime:=msecnow;
    WatchdogLogLast:=n;
    if (n>0) and IsUnix then UnixKillHangingChild;
   end;
   if FormCrwDaqWatchDog.Ok then FormCrwDaqWatchDog.Alert(n>0);
  end;
 except
  on E:Exception do BugReport(E,nil,'CheckWatchDogTimer');
 end;
end;

procedure OpenFormCrwDaqWatchDog(Show:Boolean=true; const aParams:LongString='');
var apFlags:Integer;
begin
 try
  if not FormCrwDaqWatchDog.Ok then begin
   Application.CreateForm(TFormCrwDaqWatchDog, FormCrwDaqWatchDog);
   FormCrwDaqWatchDog.Master:=@FormCrwDaqWatchDog;
   FormCrwDaqWatchDog.Visible:=Show;
  end;
  if FormCrwDaqWatchDog.Ok then begin
   if Show then begin
    Application.Restore;
    Application.BringToFront;
    FormCrwDaqWatchDog.Show;
    FormCrwDaqWatchDog.WindowState:=wsNormal;
    FormCrwDaqWatchDog.BringToFront;
    apFlags:=FormCrwDaqWatchDog.ApplyParams(aParams);
    if not HasFlags(apFlags,apf_FormPos)
    then LocateFormToCenterOfScreen(FormCrwDaqWatchDog);
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormCrwDaqWatchDog');
 end;
end;

procedure HideFormCrwDaqWatchDog;
begin
 try
  if FormCrwDaqWatchDog.Ok then FormCrwDaqWatchDog.Hide;
 except
  on E:Exception do BugReport(E,nil,'HideFormCrwDaqWatchDog');
 end;
end;
 
procedure KillFormCrwDaqWatchDog;
begin
 Kill(TForm(FormCrwDaqWatchDog));
end;

function GetWatchdogLog:LongString;
begin
 Result:=SessionManager.VarTmpFile('watchdog.log');
end;

procedure SaveWatchdogLog(ms:Double; AppendMode:Boolean; msg:LongString);
var f:THandle; width:Integer;
begin
 if Length(WatchdogLog)=0
 then WatchdogLog:=GetWatchdogLog;
 if Length(msg)>0 then
 if Length(WatchdogLog)>0 then
 try
  if (DebugOutFileLimit>0) then
  if GetFileSize(WatchDogLog)>DebugOutFileLimit
  then FileRename(WatchDogLog,WatchDogLog+'.old');
  if FileExists(WatchdogLog)
  then f:=FileOpen(WatchdogLog,fmOpenReadWrite+fmShareDenyWrite)
  else f:=FileCreate(WatchDogLog);
  if (f<>INVALID_HANDLE_VALUE) then
  try
   width:=WatchdogLogRec-Length(EOL);
   msg:=Format('%s : %s',[StdDateTimeStr(ms),msg]);
   msg:=Format('%-*.*s%s',[width,width,msg,EOL]);
   if AppendMode
   then FileSeek(f,0,fsFromEnd)
   else FileSeek(f,-Length(msg),fsFromEnd);
   FileWrite(f,msg[1],Length(msg));
  finally
   FileClose(f);
  end;
 except
  on E:Exception do BugReport(E,nil,'SaveWatchdogLog');
 end;
end;

procedure AddFileList(const FileName:LongString; const FileDetails:TSearchRec;
                      SubDirLevel:Integer;  var Terminate:Boolean; CustomData:Pointer);
begin
 if TObject(CustomData) is TStringList then TStringList(CustomData).Add(FileName);
end;

procedure TFormCrwDaqWatchDog.FormCreate(Sender: TObject);
var s:LongString;
begin
 inherited;
 myFileItem:=0;
 myFileList:=TStringList.Create;
 myFileList.Sorted:=true; s:='';
 if ReadIniFilePath(SysIniFile,SectSystem,'WatchDogPath',HomeDir,s) and DirExists(s)
 then ForEachFile(s,'*.jpg',AddFileList,0,myFileList);
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=Format('%s - %s',[SessionManager.SessionHead,RusEng('СТОРОЖЕВОЙ ТАЙМЕР','WATCHDOG TIMER')]);
 GroupBoxWdtList.Caption:=RusEng('Список висящих потоков','Hanging thread list');
 GroupBoxWdtPeriod.Caption:=RusEng('Период сторожа','Watchdog period');
 LabelWdtPeriod.Caption:=RusEng('Период, сек','Period, sec');
 LabelWdtZero.Caption:=RusEng('Нуль = запрет сторожа','Zero = disable watchdog');
 TabSheetWdtControl.Caption:=RusEng('Контроль','Control');
 TabSheetWdtStatus.Caption:=RusEng('Статус','Status');
 TabSheetWdtLog.Caption:=RusEng('Журнал','Log');
 GroupBoxWdtLog.Caption:=RusEng('Журнал Сторожевого Таймера','WatchDog Log');
 UpdateTimerState;
 UpdateLogStatus;
 Alert(false);
end;

procedure TFormCrwDaqWatchDog.BitBtnWdtLogViewClick(Sender: TObject);
begin
 WatchDogLogView;
end;

procedure TFormCrwDaqWatchDog.FormDestroy(Sender: TObject);
begin
 Kill(myFileList);
 inherited;
end;

procedure TFormCrwDaqWatchDog.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 Action:=caFree;
end;

procedure TFormCrwDaqWatchDog.FormActivate(Sender: TObject);
begin
 if Ok then
 try
  PageControlWdt.ActivePage:=TabSheetWdtStatus;
  SpinEditWdtPeriod.Value:=WatchDogTime;
  UpdateTimerState;
  UpdateLogStatus;
 except
  on E:Exception do BugReport(E,Self,'FormActivate');
 end;
end;

procedure TFormCrwDaqWatchDog.FormDeactivate(Sender: TObject);
begin
 if Ok then
 try
  TimerWdt.Enabled:=false;
 except
  on E:Exception do BugReport(E,Self,'FormDeactivate');
 end;
end;

procedure TFormCrwDaqWatchDog.TimerWdtTimer(Sender: TObject);
begin
 if Ok then
 try
  if ImageWdt.Visible then
  if Assigned(myFileList) and (myFileList.Count>0) then begin
   myFileItem:=myFileItem+1;
   if myFileItem<=0 then myFileItem:=0;
   if myFileItem>=myFileList.Count then myFileItem:=0;
   if FileExists(myFileList[myFileItem])
   then ImageWdt.Picture.LoadFromFile(myFileList[myFileItem]);
  end;
 except
  on E:Exception do BugReport(E,Self,'TimerWdtTimer');
 end;
end;

procedure TFormCrwDaqWatchDog.SpinEditWdtPeriodChange(Sender: TObject);
begin
 if Ok then
 try
  WatchDogTime:=max(0,SpinEditWdtPeriod.Value);
  if WatchDogTime=0 then Alert(false);
  UpdateTimerState;
 except
  on E:Exception do BugReport(E,Self,'SpinEditWdtPeriodChange');
 end;
end;

procedure TFormCrwDaqWatchDog.PageControlWdtChange(Sender: TObject);
begin
 UpdateTimerState;
 UpdateLogStatus;
end;

procedure TFormCrwDaqWatchDog.UpdateTimerState;
begin
 if Ok then
 try
  TimerWdt.Enabled:=Assigned(myFileList) and (myFileList.Count>0) and
                    (WatchDogTime>0) and (PageControlWdt.ActivePage=TabSheetWdtStatus);
 except
  on E:Exception do BugReport(E,Self,'UpdateTimerState');
 end;
end;

procedure TFormCrwDaqWatchDog.Alert(Alarm:Boolean);
begin
 if Ok then
 try
  if Alarm then begin
   LabelWdtAlert.Caption:=RusEng('Повисли?','Hanging?');
   LabelWdtAlert.Color:=clRed;
   LabelWdtAlert.Font.Color:=clYellow;
   PanelWdtImage.Caption:=RusEng('Есть проблемы!','Problems found!');
   PanelWdtImage.Color:=clRed;
   PanelWdtImage.Font.Color:=clYellow;
   ImageWdt.Visible:=True;
  end else begin
   MemoWdtList.Lines.Clear;
   LabelWdtAlert.Caption:='Ok.';
   LabelWdtAlert.Color:=clInfoBk;
   LabelWdtAlert.Font.Color:=clBtnText;
   PanelWdtImage.Caption:=RusEng('Нет проблем!','No problems!');
   PanelWdtImage.Color:=clBtnFace;
   PanelWdtImage.Font.Color:=clBtnText;
   ImageWdt.Visible:=False;
  end;
 except
  on E:Exception do BugReport(E,Self,'Alert');
 end;
end;

procedure TFormCrwDaqWatchDog.LabelWdtLogNameDblClick(Sender: TObject);
var cmd,opt:LongString;
begin
 opt:=IfThen(IsWindows,'-hide ','');
 cmd:='@silent @run '+opt+'unix lister '+AnsiQuotedIfNeed(GetWatchDogTimerLog);
 SendToMainConsole(cmd+EOL);
end;

procedure TFormCrwDaqWatchDog.LabelWdtLogPathDblClick(Sender: TObject);
var cmd:LongString;
begin
 cmd:='@silent @run FileManager '+AnsiQuotedIfNeed(ExtractFileDir(GetWatchDogTimerLog));
 SendToMainConsole(cmd+EOL);
end;

procedure TFormCrwDaqWatchDog.UpdateLogStatus;
var Log,Dir,Base:LongString;
begin
 Log:=GetWatchDogTimerLog; Dir:=ExtractFileDir(Log); Base:=ExtractFileNameExt(Log);
 LabelWdtLogPath.Caption:=RusEng('Папка: ','Folder: ')+MinimizeFileName(Dir,Canvas,Width-70,255);
 LabelWdtLogPath.Hint:=RusEng('Папка: ','Folder: ')+Dir;
 LabelWdtLogPath.ShowHint:=true;
 LabelWdtLogName.Caption:=RusEng('Файл: ','File: ')+Base;
 LabelWdtLogName.Hint:=LabelWdtLogName.Caption;
 LabelWdtLogName.ShowHint:=true;
 BitBtnWdtLogView.Caption:=RusEng('Открыть журнал','View Log file');
end;

procedure TFormCrwDaqWatchDog.WatchDogLogView(exe:LongString='');
var cmd,opt:LongString;
begin
 cmd:='';
 if SysGlossary.ReadIniString(SysIniFile,SectActionList(1),'ActionWatchDogLogView',cmd)
 then cmd:=StringReplace(Trim(cmd),'watchdog.log',AnsiQuotedIfNeed(GetWatchDogTimerLog),[rfIgnoreCase]);
 if IsEmptyStr(cmd) then begin
  exe:=TrimDef(exe,'unix wintail');
  opt:=IfThen(IsWindows,'-hide ','');
  cmd:='@silent @run '+opt+exe+' '+AnsiQuotedIfNeed(GetWatchDogTimerLog);
 end;
 SendToMainConsole(cmd+EOL);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_crwdaqwatchdog;
begin
 InitWatchDogHangList;
end;

procedure Free_form_crwdaqwatchdog;
begin
 WatchdogLog:='';
 FreeWatchDogHangList;
end;

initialization

 Init_form_crwdaqwatchdog;

finalization

 Free_form_crwdaqwatchdog;

end.

//////////////
// END OF FILE
//////////////

