////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Curve Window.                                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231130 - Modified for FPC (A.K.)                                         //
// 20240626 - PaintBoxPosParams                                               //
////////////////////////////////////////////////////////////////////////////////

unit form_curvewindow; // Form Curve Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, Printers,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_fifo, _crw_rtc, _crw_ef,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_sort,
 _crw_dynar, _crw_snd, _crw_guard, _crw_zm, _crw_syscal,
 _crw_colors, _crw_curves, _crw_riff, _crw_lttb,
 _crw_appforms, _crw_apptools, _crw_apputils;


type
 TMouseMode=(mmSelector,mmZoomIn,mmZoomOut,mmDragCopy,mmDragMove,
             mmSelectRoi,mmSelectRoiL,mmSelectRoiR);

type
  TFormCurveWindow = class(TFormCrwDaqSysChild)
    PaintBox: TPaintBox;
    ScrollBar: TScrollBar;
    ActionCurveMouseSelector: TAction;
    ActionCurveMouseZoomIn: TAction;
    ActionCurveMouseZoomOut: TAction;
    ActionCurveMouseDragCopy: TAction;
    ActionCurveMouseDragMove: TAction;
    ActionCurveMouseSelectRoi: TAction;
    ActionCurveMouseSelectRoiL: TAction;
    ActionCurveMouseSelectRoiR: TAction;
    ActionCurveMouseClearRoiL: TAction;
    ActionCurveMouseClearRoiR: TAction;
    ActionCurveRangeAuto: TAction;
    ActionCurveRangeSelector: TAction;
    ActionCurveRangeZoomIn: TAction;
    ActionCurveRangeZoomOut: TAction;
    ActionCurveRangeLeft: TAction;
    ActionCurveRangeRight: TAction;
    ActionCurveRangeUp: TAction;
    ActionCurveRangeDown: TAction;
    ActionCurveRangeTimeAxisX: TAction;
    ActionCurveEditDelete: TAction;
    ActionCurveEditCut: TAction;
    ActionCurveEditCopy: TAction;
    ActionCurveEditPaste: TAction;
    ActionCurveEditSelect0: TAction;
    ActionCurveEditSelect1: TAction;
    ActionCurveEditSelect2: TAction;
    ActionCurveEditSelect3: TAction;
    ActionCurveEditSelect4: TAction;
    ActionCurveEditSelect5: TAction;
    ActionCurveEditSelect6: TAction;
    ActionCurveEditSelect7: TAction;
    ActionCurveEditSelect8: TAction;
    ActionCurveEditSelect9: TAction;
    ActionCurveEditSelectMore: TAction;
    ActionCurveEditCloneWindow: TAction;
    ActionCurveEditComment: TAction;
    ActionCurveEditStyle: TAction;
    ActionCurveEditWindowStyle: TAction;
    ActionCurveEditDownSampling: TAction;
    ActionCurveTools: TAction;
    ActionCurveToolsRunMacroDialog: TAction;
    ActionCurveToolsRunPluginDialog: TAction;
    ActionCurveToolsWriteTable: TAction;
    ActionCurveToolsComposeSurface: TAction;
    MenuCurveMouse: TMenuItem;
    MenuCurveMouseSelector: TMenuItem;
    MenuCurveMouseZoomIn: TMenuItem;
    MenuCurveMouseZoomOut: TMenuItem;
    MenuCurveMouseDragCopy: TMenuItem;
    MenuCurveMouseDragMove: TMenuItem;
    MenuCurveMouseSelectRoi: TMenuItem;
    MenuCurveMouseSelectRoiL: TMenuItem;
    MenuCurveMouseSelectRoiR: TMenuItem;
    MenuCurveMouseClearRoiL: TMenuItem;
    MenuCurveMouseClearRoiR: TMenuItem;
    MenuCurveRange: TMenuItem;
    MenuCurveRangeAuto: TMenuItem;
    MenuCurveRangeSelector: TMenuItem;
    MenuCurveRangeZoomIn: TMenuItem;
    MenuCurveRangeZoomOut: TMenuItem;
    NameCurveRangeLeft: TMenuItem;
    MenuCurveRangeRight: TMenuItem;
    MenuCurveRangeUp: TMenuItem;
    MenuCurveRangeDown: TMenuItem;
    MenuCurveRangeTimeAxisX: TMenuItem;
    MenuCurveEdit: TMenuItem;
    MenuCurveEditSelect: TMenuItem;
    MenuCurveEditSelect0: TMenuItem;
    MenuCurveEditSelect1: TMenuItem;
    MenuCurveEditSelect2: TMenuItem;
    MenuCurveEditSelect3: TMenuItem;
    MenuCurveEditSelect4: TMenuItem;
    MenuCurveEditSelect5: TMenuItem;
    MenuCurveEditSelect6: TMenuItem;
    MenuCurveEditSelect7: TMenuItem;
    MenuCurveEditSelect8: TMenuItem;
    MenuCurveEditSelect9: TMenuItem;
    MenuCurveEditSelectMore: TMenuItem;
    MenuCurveEditDelete: TMenuItem;
    MenuCurveEditCut: TMenuItem;
    MenuCurveEditCopy: TMenuItem;
    MenuCurveEditPaste: TMenuItem;
    MenuCurveEditCloneWindow: TMenuItem;
    MenuCurveEditComment: TMenuItem;
    MenuCurveEditStyle: TMenuItem;
    MenuCurveEditWindowStyle: TMenuItem;
    MenuCurveEditDownSampling: TMenuItem;
    MenuCurveEditMore: TMenuItem;
    MenuCurve: TMenuItem;
    MenuCurveTools: TMenuItem;
    MenuCurveToolsRunMacroDialog: TMenuItem;
    MenuCurveToolsRunPluginDialog: TMenuItem;
    MenuCurveToolsWriteTable: TMenuItem;
    MenuCurveToolsComposeSurface: TMenuItem;
    ToolButtonCurveTools: TToolButton;
    ToolButtonCurveToolsRunMacroDialog: TToolButton;
    ToolButtonCurveMouseSeparator: TToolButton;
    ToolButtonCurveMouseSelector: TToolButton;
    ToolButtonCurveMouseZoomIn: TToolButton;
    ToolButtonCurveMouseZoomOut: TToolButton;
    ToolButtonCurveMouseDragCopy: TToolButton;
    ToolButtonCurveMouseDragMove: TToolButton;
    ToolButtonCurveMouseSelectRoi: TToolButton;
    ToolButtonCurveMouseSelectRoiL: TToolButton;
    ToolButtonCurveMouseSelectRoiR: TToolButton;
    ToolButtonCurveMouseClearRoiL: TToolButton;
    ToolButtonCurveMouseClearRoiR: TToolButton;
    ToolButtonCurveRangeAutoSeparator: TToolButton;
    ToolButtonCurveRangeAuto: TToolButton;
    ToolButtonCurveEditSelectMore: TToolButton;
    ToolButtonCurveEditCloneWindow: TToolButton;
    ToolButtonCurveRangeSelectorSeparator: TToolButton;
    ToolButtonCurveRangeSelector: TToolButton;
    ToolButtonCurveRangeZoomIn: TToolButton;
    ToolButtonCurveRangeZoomOut: TToolButton;
    ToolButtonCurveRangeLeftSeparator: TToolButton;
    ToolButtonCurveRangeLeft: TToolButton;
    ToolButtonCurveRangeRight: TToolButton;
    ToolButtonCurveRangeUp: TToolButton;
    ToolButtonCurveRangeDown: TToolButton;
    ToolButtonCurveRangeTimeAxisX: TToolButton;
    ToolButtonEditDeleteSeparator: TToolButton;
    ToolButtonCurveEditDelete: TToolButton;
    ToolButtonCurveEditCut: TToolButton;
    ToolButtonCurveEditCopy: TToolButton;
    ToolButtonCurveEditPaste: TToolButton;
    ToolButtonCurveEditCommentSeparator: TToolButton;
    ToolButtonCurveEditComment: TToolButton;
    ToolButtonCurveEditStyle: TToolButton;
    ToolButtonCurveEditWindowStyle: TToolButton;
    ToolButtonCurveEditDownSampling: TToolButton;
    ToolButtonCurveToolsRunPluginDialog: TToolButton;
    PopupMenuFile: TMenuItem;
    PopupMenuFileSave: TMenuItem;
    PopupMenuFileSaveAs: TMenuItem;
    PopupMenuFilePrint: TMenuItem;
    PopupMenuFilePrinterSetup: TMenuItem;
    PopupMenuCurveEdit: TMenuItem;
    PopupMenuCurveEditDelete: TMenuItem;
    PopupMenuCurveEditCut: TMenuItem;
    PopupMenuCurveEditCopy: TMenuItem;
    PopupMenuCurveEditPaste: TMenuItem;
    PopupMenuCurveEditWindowStyle: TMenuItem;
    PopupMenuCurveEditDownSampling: TMenuItem;
    PopupMenuCurveEditStyle: TMenuItem;
    PopupMenuCurveEditComment: TMenuItem;
    PopupMenuCurveMouse: TMenuItem;
    PopupMenuCurveMouseSelector: TMenuItem;
    PopupMenuCurveMouseZoomIn: TMenuItem;
    PopupMenuCurveMouseZoomOut: TMenuItem;
    PopupMenuCurveMouseDragCopy: TMenuItem;
    PopupMenuCurveMouseDragMove: TMenuItem;
    PopupMenuCurveMouseSelectRoi: TMenuItem;
    PopupMenuCurveMouseSelectRoiL: TMenuItem;
    PopupMenuCurveMouseSelectRoiR: TMenuItem;
    PopupMenuCurveMouseClearRoiL: TMenuItem;
    PopupMenuCurveMouseClearRoiR: TMenuItem;
    PopupMenuCurveRange: TMenuItem;
    PopupMenuCurveRangeAuto: TMenuItem;
    PopupMenuCurveRangeSelector: TMenuItem;
    PopupMenuCurveRangeZoomIn: TMenuItem;
    PopupMenuCurveRangeZoomOut: TMenuItem;
    PopupMenuCurveRangeLeft: TMenuItem;
    PopupMenuCurveRangeRight: TMenuItem;
    PopupMenuCurveRangeUp: TMenuItem;
    PopupMenuCurveRangeDown: TMenuItem;
    PopupMenuCurveRangeTimeAxisX: TMenuItem;
    PopupMenuCurveEditSelect: TMenuItem;
    PopupMenuCurveEditSelectMore: TMenuItem;
    PopupMenuCurveEditSelect0: TMenuItem;
    PopupMenuCurveEditSelect1: TMenuItem;
    PopupMenuCurveEditSelect2: TMenuItem;
    PopupMenuCurveEditSelect3: TMenuItem;
    PopupMenuCurveEditSelect4: TMenuItem;
    PopupMenuCurveEditSelect5: TMenuItem;
    PopupMenuCurveEditSelect6: TMenuItem;
    PopupMenuCurveEditSelect7: TMenuItem;
    PopupMenuCurveEditSelect8: TMenuItem;
    PopupMenuCurveEditSelect9: TMenuItem;
    PopupMenuCurveTool: TMenuItem;
    PopupMenuCurveTools: TMenuItem;
    PopupMenuCurveToolsRunMacroDialog: TMenuItem;
    PopupMenuCurveToolsRunPluginDialog: TMenuItem;
    PopupMenuCurveEditCloneWindow: TMenuItem;
    PopupMenuCurveToolsWriteTable: TMenuItem;
    PopupMenuCurveToolsComposeSurface: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ScrollBarChange(Sender: TObject);
    procedure PaintBoxPaint(Sender: TObject);
    procedure PaintBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
    procedure PaintBoxMouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure PaintBoxMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
    procedure PaintBoxStartDrag(Sender: TObject; var DragObject: TDragObject);
    procedure PaintBoxEndDrag(Sender, Target: TObject; X, Y: Integer);
    procedure PaintBoxDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure PaintBoxDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ActionCurveMouseSelectorExecute(Sender: TObject);
    procedure ActionCurveMouseZoomInExecute(Sender: TObject);
    procedure ActionCurveMouseZoomOutExecute(Sender: TObject);
    procedure ActionCurveMouseDragCopyExecute(Sender: TObject);
    procedure ActionCurveMouseDragMoveExecute(Sender: TObject);
    procedure ActionCurveMouseSelectRoiExecute(Sender: TObject);
    procedure ActionCurveMouseSelectRoiLExecute(Sender: TObject);
    procedure ActionCurveMouseSelectRoiRExecute(Sender: TObject);
    procedure ActionCurveMouseClearRoiLExecute(Sender: TObject);
    procedure ActionCurveMouseClearRoiRExecute(Sender: TObject);
    procedure ActionCurveRangeAutoExecute(Sender: TObject);
    procedure ActionCurveRangeSelectorExecute(Sender: TObject);
    procedure ActionCurveRangeZoomInExecute(Sender: TObject);
    procedure ActionCurveRangeZoomOutExecute(Sender: TObject);
    procedure ActionCurveRangeLeftExecute(Sender: TObject);
    procedure ActionCurveRangeRightExecute(Sender: TObject);
    procedure ActionCurveRangeUpExecute(Sender: TObject);
    procedure ActionCurveRangeDownExecute(Sender: TObject);
    procedure ActionCurveRangeTimeAxisXExecute(Sender: TObject);
    procedure ActionCurveEditDeleteExecute(Sender: TObject);
    procedure ActionCurveEditCutExecute(Sender: TObject);
    procedure ActionCurveEditCopyExecute(Sender: TObject);
    procedure ActionCurveEditPasteExecute(Sender: TObject);
    procedure ActionCurveEditSelectMoreExecute(Sender: TObject);
    procedure ActionCurveEditCloneWindowExecute(Sender: TObject);
    procedure ActionCurveEditSelect0Execute(Sender: TObject);
    procedure ActionCurveEditSelect1Execute(Sender: TObject);
    procedure ActionCurveEditSelect2Execute(Sender: TObject);
    procedure ActionCurveEditSelect3Execute(Sender: TObject);
    procedure ActionCurveEditSelect4Execute(Sender: TObject);
    procedure ActionCurveEditSelect5Execute(Sender: TObject);
    procedure ActionCurveEditSelect6Execute(Sender: TObject);
    procedure ActionCurveEditSelect7Execute(Sender: TObject);
    procedure ActionCurveEditSelect8Execute(Sender: TObject);
    procedure ActionCurveEditSelect9Execute(Sender: TObject);
    procedure ActionCurveEditCommentExecute(Sender: TObject);
    procedure ActionCurveEditStyleExecute(Sender: TObject);
    procedure ActionCurveEditWindowStyleExecute(Sender: TObject);
    procedure ActionCurveEditDownSamplingExecute(Sender: TObject);
    procedure ActionCurveToolsExecute(Sender: TObject);
    procedure ActionCurveToolsRunMacroDialogExecute(Sender: TObject);
    procedure ActionCurveToolsRunPluginDialogExecute(Sender: TObject);
    procedure ActionCurveToolsWriteTableExecute(Sender: TObject);
    procedure ActionCurveToolsComposeSurfaceExecute(Sender: TObject);
  private
    {
    Private declarations
    }
    myWorld         : TRect2D;
    myOrigin        : TPoint2D;
    myScale         : TPoint2D;
    myGridX         : TAxisGrid;
    myGridY         : TAxisGrid;
    myAxisFlags     : Cardinal;
    myGroundColor   : TColor;
    myLegendColor   : TColor;
    myNumberColor   : TColor;
    myBoxColor      : TColor;
    myRectColor     : TColor;
    myGridColor     : TColor;
    myRoiLColor     : TColor;
    myRoiRColor     : TColor;
    myTitle         : LongString;
    myLegend        : LongString;
    myCurves        : TCurveList;
    myComment       : TText;
    myDefCurveNum   : Integer;
    myMouseMode     : TMouseMode;
    myRoi           : TRect2D;
    myXorSelector   : TXorSelector;
    myXorRoi        : TXorSelector;
    myXorRoiL       : TXorSelector;
    myXorRoiR       : TXorSelector;
    mySmartDrawer   : TSmartDrawer;
    myMousePos      : TPoint2D;
    myTimeBase      : Double;
    myTimeUnits     : Double;
    myDrawEntry     : Cardinal;
    myWantTimeAxisX : Boolean;
    myDownSampling  : TDownSamplingParams;
    myDownSampCount : Integer;
    myUpdateLoops   : Integer;
    function   GetIsClipBoard:Boolean;
    function   GetIsDataProtected:Boolean;
    procedure  SetIsDataProtected(Protection:Boolean);
    function   GetWorld:TRect2D;
    procedure  SetWorld(const aWorld:TRect2D);
    function   GetPlotArea:TRect2I;
    function   GetScreenWorld:TRect2I;
    function   GetAxisFlags:Cardinal;
    procedure  SetAxisFlags(aFlags:Cardinal);
    function   GetGroundColor:TColor;
    procedure  SetGroundColor(aColor:TColor);
    function   GetLegendColor:TColor;
    procedure  SetLegendColor(aColor:TColor);
    function   GetNumberColor:TColor;
    procedure  SetNumberColor(aColor:TColor);
    function   GetBoxColor:TColor;
    procedure  SetBoxColor(aColor:TColor);
    function   GetRectColor:TColor;
    procedure  SetRectColor(aColor:TColor);
    function   GetGridColor:TColor;
    procedure  SetGridColor(aColor:TColor);
    function   GetRoiLColor:TColor;
    procedure  SetRoiLColor(aColor:TColor);
    function   GetRoiRColor:TColor;
    procedure  SetRoiRColor(aColor:TColor);
    function   GetTitle:LongString;
    procedure  SetTitle(const aTitle:LongString);
    function   GetLegend:LongString;
    procedure  SetLegend(const aLegend:LongString);
    function   GetCurves:TCurveList;
    function   GetComment:TText;
    function   GetDefCurveNum:Integer;
    procedure  SetDefCurveNum(aNumber:Integer);
    function   GetDefCurve:TCurve;
    procedure  SetDefCurve(aCurve:TCurve);
    function   GetDefComment:TText;
    function   GetHasSelection:Boolean;
    function   GetMouseMode:TMouseMode;
    procedure  SetMouseMode(aMode:TMouseMode);
    function   GetRoi:TRect2D;
    procedure  SetRoi(const aRoi:TRect2D);
    function   GetTimeBase:Double;
    procedure  SetTimeBase(aTimeBase:Double);
    function   GetTimeUnits:Double;
    procedure  SetTimeUnits(aTimeUnits:Double);
    function   GetCanUseTimeAxisX:Boolean;
    function   GetUsesTimeAxisX:Boolean;
    function   GetWantTimeAxisX:Boolean;
    procedure  SetWantTimeAxisX(aWant:Boolean);
    function   GetDownSampling:TDownSamplingParams;
    procedure  SetDownSampling(const aParams:TDownSamplingParams);
    function   GetUsesDownSampling:Boolean;
    procedure  SetUsesDownSampling(aUses:Boolean);
    procedure  UpdateTimeAxisX;
  public
    {
    Public declarations
    }
    property  IsClipboard      : Boolean     read GetIsClipboard;
    property  IsDataProtected  : Boolean     read GetIsDataProtected write SetIsDataProtected;
    property  World            : TRect2D     read GetWorld           write SetWorld;
    property  PlotArea         : TRect2I     read GetPlotArea;
    property  ScreenWorld      : TRect2I     read GetScreenWorld;
    property  AxisFlags        : Cardinal    read GetAxisFlags       write SetAxisFlags;
    property  GroundColor      : TColor      read GetGroundColor     write SetGroundColor;
    property  LegendColor      : TColor      read GetLegendColor     write SetLegendColor;
    property  NumberColor      : TColor      read GetNumberColor     write SetNumberColor;
    property  BoxColor         : TColor      read GetBoxColor        write SetBoxColor;
    property  RectColor        : TColor      read GetRectColor       write SetRectColor;
    property  GridColor        : TColor      read GetGridColor       write SetGridColor;
    property  RoiLColor        : TColor      read GetRoiLColor       write SetRoiLColor;
    property  RoiRColor        : TColor      read GetRoiRColor       write SetRoiRColor;
    property  Title            : LongString  read GetTitle           write SetTitle;
    property  Legend           : LongString  read GetLegend          write SetLegend;
    property  Curves           : TCurveList  read GetCurves;
    property  Comment          : TText       read GetComment;
    property  DefCurveNum      : Integer     read GetDefCurveNum     write SetDefCurveNum;
    property  DefCurve         : TCurve      read GetDefCurve        write SetDefCurve;
    property  DefComment       : TText       read GetDefComment;
    property  HasSelection     : Boolean     read GetHasSelection;
    property  MouseMode        : TMouseMode  read GetMouseMode       write SetMouseMode;
    property  Roi              : TRect2D     read GetRoi             write SetRoi;
    property  TimeBase         : Double      read GetTimeBase        write SetTimeBase;
    property  TimeUnits        : Double      read GetTimeUnits       write SetTimeUnits;
    property  WantTimeAxisX    : Boolean     read GetWantTimeAxisX   write SetWantTimeAxisX;
    property  DownSampling     : TDownSamplingParams read GetDownSampling write SetDownSampling;
    property  UsesDownSampling : Boolean     read GetUsesDownSampling write SetUsesDownSampling;
    property  UsesTimeAxisX    : Boolean     read GetUsesTimeAxisX;
    property  CanUseTimeAxisX  : Boolean     read GetCanUseTimeAxisX;
  public
    function  DownSamplingMethod:Integer;
    function  DownSamplingLength:Integer;
    procedure UpdateDownSamplingStatus;
    function  DownSamplingTail:Integer;
  public
    procedure  AfterConstruction; override;
    procedure  BeforeDestruction; override;
    procedure  UpdateCommands; override;
    procedure  FileSave; override;
    procedure  FileSaveAs; override;
    procedure  ResetWorld;
    procedure  SomeGrowWorld(Lim:TRect2D; Flags:Cardinal);
    function   WorldToLocal(const P:TPoint2D):TPoint2I;
    function   WorldToLocalX(X:Double):Integer;
    function   WorldToLocalY(Y:Double):Integer;
    function   WorldToLocalRect(const R:TRect2D):TRect2I;
    function   LocalToWorld(const P:TPoint2I):TPoint2D;
    function   LocalToWorldX(X:Integer):Double;
    function   LocalToWorldY(Y:Integer):Double;
    function   LocalToWorldRect(const R:TRect2I):TRect2D;
    procedure  DrawView; override;
    procedure  PrintView; override;
    function   GetImage(BlackAndWhite:Boolean):TBitmap;
    procedure  DrawAxis(Flags:Cardinal);
    procedure  DrawCurveXY(Curve:TCurve; Start,N:Integer; Color,Style:Cardinal);
    procedure  DrawCurveXYPrime(Curve:TCurve; Start,N:Integer; Color,Style:Cardinal);
    procedure  DrawCurve(aCurve:TCurve);
    procedure  DrawCurvePart(Curve:TCurve; StartIndex,StopIndex:LongInt);
    procedure  DrawCurves;
    procedure  DrawAlignedStr(y:integer; s:LongString; Color,Back:TColor);
    procedure  DrawAlignedStrings(y:integer;s:LongString; Color,Back:TColor);
    procedure  InsertCurve(Index:integer; aCurve:TCurve);
    procedure  AddCurve(aCurve:TCurve);
    procedure  DeleteCurveNum(Num:Integer);
    procedure  DeleteCurve(aCurve:TCurve);
    procedure  UpdateScroller;
    procedure  UpdatePaintBox;
    function   GetAutoRange:TRect2D;
    procedure  AutoRange;
    function   CheckIsDataProtected(Msg:Boolean=true):Boolean;
    function   Clone:TFormCurveWindow;
    procedure  StartMonitoring;
    procedure  StopMonitoring;
    procedure  Monitoring;
    procedure  MonitoringDone;
    procedure  RoiWasSelected(Print:Boolean=True);
    function   PaintBoxPosParams:LongString;
  end;
  TCurveWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormCurveWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormCurveWindow);
  public
    property   Window[i:Integer]:TFormCurveWindow read GetWindow write SetWindow; default;
  end;

const
 CurveClipboard      : TFormCurveWindow = nil;
 FullCurveWindowList : TCurveWindowList = nil;
 CurveWindowsMonitor : TCurveWindowList = nil;
 MaxClipboardCurves  : Integer          = 3;
 DefGroundColor      : TColor           = clLtGray;
 DefLegendColor      : TColor           = clBlack;
 DefNumberColor      : TColor           = clBlack;
 DefBoxColor         : TColor           = clLtGray;
 DefRectColor        : TColor           = clRed;
 DefGridColor        : TColor           = clRed;
 DefRoiLColor        : TColor           = clYellow;
 DefRoiRColor        : TColor           = clBlue;
 WorldAbsEps         : Double           = 0;
 WorldRelEps         : Double           = 1E-8;
 ZoomFactor          : Double           = 1.5;
 AlphaStayFree       : Double           = 0.7;
 AxisSettings:record
  Width         : Integer;
  Decim         : Integer;
  Inter         : TPoint;
  Stick         : TPoint;
  Space         : TPoint;
  Margin        : TRect2I;
  GrowStep      : TPoint2D;
  IncStep       : TPoint2D;
  MoveStep      : TPoint2D;
  MinWinSize    : TPoint;
  GridPattern   : Integer
 end=(
  Width         : 16;
  Decim         : 10;
  Inter         : (X:4;Y:4);
  Stick         : (X:3;Y:3);
  Space         : (X:2;Y:2);
  Margin        : (A:(X:68;Y:52); B:(X:10;Y:70)); // was (A:(X:68;Y:36); B:(X:10;Y:70))
  GrowStep      : ( X:0.05; Y:0.05 );
  IncStep       : ( X:2;    Y:2    );
  MoveStep      : ( X:0.25;  Y:0.25);
  MinWinSize    : (X:250;Y:250);
  GridPattern   : $FFFFFF
 );
 {
 axis flags
 }
 afReversX         = $0001;                   {screen x have reverse direction}
 afReversY         = $0002;                   {screen y have reverse direction}
 afCheckX          = $0004;                   {check x-limits when draw}
 afCheckY          = $0008;                   {check y-limits when draw}
 afCheck           = afCheckX+afCheckY;       {check limits when draw}
 afCheckInsX       = $0010;                   {check x-limits when insert}
 afCheckInsY       = $0020;                   {check y-limits when insert}
 afCheckIns        = afCheckInsX+afCheckInsY; {check limits when insert}
 afUnion           = $0040;                   {union limits with older limits}
 afSelectIns       = $0080;                   {select curve when insert}
 afLineBreak1      = $0100;                   {break line if one point not visible}
 afLineBreak2      = $0200;                   {break line if two points not visible}
 afUpdate          = $0400;
 DefaultAxisFlags  = afReversY+afUnion+afCheckIns+afSelectIns;
 {
 draw flags
 }
 dfGridLineX     = $0001;
 dfGridLineY     = $0002;
 dfGridTextX     = $0004;
 dfGridTextY     = $0008;
 dfPlotBox       = $0010;
 dfGround        = $0020;
 dfTitle         = $0040;
 dfLegend        = $0080;
 dfRoiL          = $0100;
 dfRoiR          = $0200;
 dfGridLines     = dfGridLineX+dfGridLineY;
 dfGridTexts     = dfGridTextX+dfGridTextY;
 dfGridX         = dfGridLineX+dfGridTextX;
 dfGridY         = dfGridLineY+dfGridTextY;
 dfRoiLR         = dfRoiL+dfRoiR;
 dfDefault       = dfGround+dfPlotBox+dfGridX+dfGridY+dfTitle+dfLegend+dfRoiLR;
 DefaultTimeAxisFont : TFontParams = (
  CharSet : RUSSIAN_CHARSET;
  Color   : clBlack;
  Height  : -9; // or -11;
  Name    : 'PT Mono';
  Pitch   : fpFixed;
  Style   : [];
 );

function NewCurveWindow(const aCaption : LongString  = '???';
                        const aTitle   : LongString  = '???';
                        const aLegend  : LongString  = '???';
                              aState   : TWindowState = wsNormal):TFormCurveWindow;
procedure Kill(var TheObject:TFormCurveWindow); overload;

function  NewCurveWindowList(aOwnsObjects : Boolean = true;
                             aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                             aStep        : LongInt = DefaultTObjectStorageStep
                                        ) : TCurveWindowList;
procedure Kill(var TheObject:TCurveWindowList); overload;

function NewCurveWindowByFormula:TFormCurveWindow;
function ActiveCurveWindow:TFormCurveWindow;
function FindCurveWindow(const Caption:LongString; CreateNew:Boolean=false):TFormCurveWindow;
function NewCurveReadTextTable(Form:TForm):TFormCurveWindow;
function SaveCurveWindowToCRW(FileName:LongString; ITEM:TFormCurveWindow):Boolean;
function LoadCurveWindowFromCRW(FileName:LongString; const Chunk:TRiffChunk):TFormCurveWindow;

procedure ReadCurveWindowsPalette(TheIniFile:LongString);

function IsValidTimeScale(const TimeBase,TimeUnit:Double):Boolean;

 {
 ********************
 CurveWindowsProfiler
 ********************
 }
procedure CurveWindowsProfiler_Clear;
procedure CurveWindowsProfiler_Start;
procedure CurveWindowsProfiler_Stop;
procedure CurveWindowsProfiler_Poll;

var
 CurveWindowsProfiler : record
  Curr, Prev, Rate    : record
   TickCount          : Cardinal;
   DrawView           : Int64;
   DrawCurve          : Int64;
   DrawPoint          : Int64;
   MonitorCall        : Int64;
   MonitorDraw        : Int64;
   MonitorCurve       : Int64;
   MonitorPoint       : Int64;
  end;
  RateFactor          : Double;
 end;

implementation

uses
 Form_TextEditor,
 Form_TextEditDialog,
 Form_ListBoxSelection,
 Form_CurveEditDownSamplingDialog,
 Form_CurveEditWindowStyleDialog,
 Form_CurveToolsWriteTableDialog,
 Form_CurveToolsReadTableDialog,
 Form_CurveToolsRunMacroDialog,
 Form_CurveRangeSelector,
 Form_CurveStyleDialog,
 Form_CurveTools,
 Form_SurfWindow,
 Form_CrwDaq,
 Form_CurveToolsRunPluginDialog,
 Form_CurveByFormula;

{$R *.lfm}

 {
 ********************
 CurveWindowsProfiler
 ********************
 }
procedure CurveWindowsProfiler_Clear;
begin
 FillChar(CurveWindowsProfiler, SizeOf(CurveWindowsProfiler), 0);
end;

procedure CurveWindowsProfiler_Start;
begin
 CurveWindowsProfiler_Clear;
 SecondActions.Add(CurveWindowsProfiler_Poll);
end;

procedure CurveWindowsProfiler_Stop;
begin
 SecondActions.Remove(CurveWindowsProfiler_Poll);
 CurveWindowsProfiler_Clear;
end;

procedure CurveWindowsProfiler_Poll;
begin
 with CurveWindowsProfiler do begin
  Curr.TickCount     := GetTickCount;
  Rate.TickCount     := Curr.TickCount    - Prev.TickCount;
  if (Rate.TickCount=0) then RateFactor:=0.0 else RateFactor:=1000.0/Rate.TickCount;
  Rate.DrawView      := Curr.DrawView     - Prev.DrawView;
  Rate.DrawCurve     := Curr.DrawCurve    - Prev.DrawCurve;
  Rate.DrawPoint     := Curr.DrawPoint    - Prev.DrawPoint;
  Rate.MonitorCall   := Curr.MonitorCall  - Prev.MonitorCall;
  Rate.MonitorDraw   := Curr.MonitorDraw  - Prev.MonitorDraw;
  Rate.MonitorCurve  := Curr.MonitorCurve - Prev.MonitorCurve;
  Rate.MonitorPoint  := Curr.MonitorPoint - Prev.MonitorPoint;
  Prev:=Curr;
 end;
end;

 {
 ****************
 Utility routines
 ****************
 }

function IsValidTimeScale(const TimeBase,TimeUnit:Double):Boolean;
const
 TimeBaseMin = UnixTimeBase; // 1970.01.01-00:00:00:000 - min date
 TimeBaseMax = 3155063616e5; // 9999.01.01-00:00:00:000 - max date
begin
 Result:=false;
 if IsNanOrInf(TimeBase) then Exit;
 if IsNanOrInf(TimeUnit) then Exit;
 if TimeBase<TimeBaseMin then Exit;
 if TimeBase>TimeBaseMax then Exit;
 if TimeUnit<=0 then Exit;
 Result:=true;
end;

 {
 *******************************
 TCurveWindowList implementation
 *******************************
 }
function   TCurveWindowList.GetWindow(i:Integer):TFormCurveWindow;
begin
 Result:=TFormCurveWindow(Items[i]);
end;

procedure  TCurveWindowList.SetWindow(i:Integer; aWindow:TFormCurveWindow);
begin
 Items[i]:=aWindow;
end;

function  NewCurveWindowList(aOwnsObjects : Boolean = true;
                             aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                             aStep        : LongInt = DefaultTObjectStorageStep
                                        ) : TCurveWindowList;
begin
 Result:=nil;
 try
  Result:=TCurveWindowList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewCurveWindowList');
 end;
end;

procedure Kill(var TheObject:TCurveWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 ************************
 General purpose utilites
 ************************
 }
function NewCurveWindow(const aCaption : LongString  = '???';
                        const aTitle   : LongString  = '???';
                        const aLegend  : LongString  = '???';
                              aState   : TWindowState = wsNormal):TFormCurveWindow;
const WinNum:Integer=0;
begin
 Result:=nil;
 try
  Application.CreateForm(TFormCurveWindow,Result);
  with Result do begin
   WindowState:=aState;
   try
    LockDraw;
    Inc(WinNum);
    if aCaption='???'
    then Caption:=RusEng('НОВОЕ ОКНО КРИВЫХ ','NEW CURVE WINDOW ')+IntToStr(WinNum)
    else Caption:=aCaption;
    if aTitle='???'
    then Title:=RusEng(^C'ЗАГОЛОВОК'+ASCII_CR+^L' Y',^C'TITLE'+ASCII_CR+^L' Y')
    else Title:=aTitle;
    if aLegend='???'
    then Legend:=RusEng(^R'X '+ASCII_CR+^C'ЛЕГЕНДА',^R'X '+ASCII_CR+^C'LEGEND')
    else Legend:=aLegend;
    World:=Rect2D(0,0,1,1);
   finally
    UnlockDraw;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewCurveWindow');
 end;
end;

procedure Kill(var TheObject:TFormCurveWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

function NewCurveWindowByFormula:TFormCurveWindow;
var Curve:TCurve; Par:LongString;
begin
 Result:=nil;
 try
  if FormCrwDaq.Ok
  then Par:=ControlPosParams(FormCrwDaq.ToolButtonToolsPlot2D,'BR',-300,4)
  else Par:='';
  Curve:=CurveByFormulaDialog(Par);
  if Assigned(Curve) then begin
   Result:=NewCurveWindow;
   Result.AddCurve(Curve);
  end;
 except
  on E:Exception do BugReport(E,nil,'NewCurveWindowByFormula');
 end;
end;

function ActiveCurveWindow:TFormCurveWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormCurveWindow) then Result:=TFormCurveWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveCurveWindow');
 end;
end;

type
 TFindRec = packed record
  Match   : TFormCurveWindow;
  Caption : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TFindRec(Custom^) do
 if (Form is TFormCurveWindow) and SameText(Form.Caption,Caption) then begin
  Match:=Form as TFormCurveWindow;
  Terminate:=true;
 end;
end;

function FindCurveWindow(const Caption:LongString; CreateNew:Boolean=false):TFormCurveWindow;
var FindRec:TFindRec;
begin
 FindRec:=Default(TFindRec);
 try
  FindRec.Match:=nil;
  FindRec.Caption:=StringBuffer(Caption);
  if (Caption<>'') then SdiMan.ForEachChild(FindForm,@FindRec);
  Result:=FindRec.Match;
  if CreateNew and not Assigned(Result) then Result:=NewCurveWindow(Caption);
 finally
  FindRec.Caption:='';
 end;
end;

function NewCurveReadTextTable(Form:TForm):TFormCurveWindow;
var p:TText; i,colx,coly,ss,sl:Integer; buf:TParsingBuffer; c:TCurve;
var TableText,Par,st:LongString;
const WinNum:Integer=0;
begin
 Result:=nil;
 try
  ss:=0; sl:=0; st:='';
  if not Assigned(Form) then Exit;
  if (Form is TFormTextEditor) then with (Form as TFormTextEditor) do begin
   st:=PerformSelText; if (st='') then ReadLastSelection(ss,sl,st);
   sl:=Length(st);
  end;
  if (Form is TFormTextEditor) then with (Form as TFormTextEditor) do
  if NoProblem(sl>0,RusEng('В окне редактора не выделен текст таблицы!',
                           'No table text selected in text editor!'))
  then TableText:=st else TableText:='';
  if IsEmptyStr(TableText) then Exit;
  Par:=ControlPosParams((Form as TFormTextEditor).Editor);
  if CurveToolsReadTableDialogExecute(Form,Par)=mrOk then begin
   p:=NewText;
   try
    Inc(WinNum);
    colx:=0; coly:=0;
    p.Text:=TableText;
    Result:=NewCurveWindow(RusEng('Таблица ','Table ')+IntToStr(WinNum));
    for i:=1 to 256 do begin
     StrCopyBuff(buf,UpcaseStr(FormCurveToolsReadTableDialogTableFormat));
     if (ScanVarInteger(svUpCase,buf,'X'+d2s(i)+':%i',colx)<>nil) and
        (ScanVarInteger(svUpCase,buf,'Y'+d2s(i)+':%i',coly)<>nil)
     then begin
      c:=NewCurve;
      c.Style:=$1F;
      c.ReadTable(p,colx,coly);
      if Trouble(c.Count=0,RusEng('Не могу прочитать таблицу!','Could not read table!')) then begin
       Kill(c);
       Kill(Result);
       break;
      end;
      Result.AddCurve(c);
     end else break;
    end;
    if Assigned(Result) and Trouble(Result.Curves.Count=0,RusEng('Не могу прочитать таблицу!','Could not read table!'))
    then Kill(Result);
   finally
    Kill(p);
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewCurveReadTextTable');
 end;
end;

function SaveCurveWindowToCRW(FileName:LongString; ITEM:TFormCurveWindow):Boolean;
var Filer:TRiffFiler; Form,SubForm,Chunk:TRiffChunk; i,j:Integer; Curve:TCurve;
begin
 Result:=false;
 FileName:=UnifyFileAlias(FileName,ua_FileDefLow);
 Filer:=NewRiffFiler(FileName,fmOpenReadWrite,sig_CRWF);
 if Filer.Ok then
 try
  {
  seek to end of file and create form ITEM CRVW
  }
  Filer.Validate;
  Filer.SeekChunkNext(Filer.RootForm);
  Form:=Filer.CreateForm(sig_ITEM,sig_CRVW);
   {
   Save ITEM name - window name
   }
   Chunk:=Filer.CreateChunk(sig_name);
   Filer.WriteString(ITEM.Caption);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM time - save time
   }
   Chunk:=Filer.CreateChunk(sig_time);
   Filer.WriteDouble(msecnow);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM vers - version
   }
   Chunk:=Filer.CreateChunk(sig_vers);
   Filer.WriteLongInt(sig_0001);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM ltwh - left,top,width,height
   }
   Chunk:=Filer.CreateChunk(sig_ltwh);
   Filer.WriteLongInt(ITEM.Left);
   Filer.WriteLongInt(ITEM.Top);
   Filer.WriteLongInt(ITEM.Width);
   Filer.WriteLongInt(ITEM.Height);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM titl - Title
   }
   Chunk:=Filer.CreateChunk(sig_titl);
   Filer.WriteString(ITEM.Title);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM lgnd - Legend
   }
   Chunk:=Filer.CreateChunk(sig_lgnd);
   Filer.WriteString(ITEM.Legend);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM text - Comment
   }
   Chunk:=Filer.CreateChunk(sig_text);
   Filer.WriteString(ITEM.Comment.Text);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM clrp - Color palette
   }
   Chunk:=Filer.CreateChunk(sig_clrp);
   Filer.WriteLongInt(ITEM.GroundColor);
   Filer.WriteLongInt(ITEM.LegendColor);
   Filer.WriteLongInt(ITEM.NumberColor);
   Filer.WriteLongInt(ITEM.BoxColor);
   Filer.WriteLongInt(ITEM.RectColor);
   Filer.WriteLongInt(ITEM.GridColor);
   Filer.WriteLongInt(ITEM.RoiLColor);
   Filer.WriteLongInt(ITEM.RoiRColor);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM wrld - World
   }
   Chunk:=Filer.CreateChunk(sig_wrld);
   Filer.WriteRect2D(ITEM.World);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM roi  - Roi
   }
   Chunk:=Filer.CreateChunk(sig_roi_);
   Filer.WriteRect2D(ITEM.Roi);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM tbtu - TimeBase, TimeUnits
   }
   Chunk:=Filer.CreateChunk(sig_tbtu);
   Filer.WriteDouble(ITEM.TimeBase);
   Filer.WriteDouble(ITEM.TimeUnits);
   Filer.FlushChunk(Chunk);
   {
   Сохранить список кривых в форме ITEM CURV.
   Список кривых и сами кривые запираются Lock/Unlock для потоковой безопасности.
   }
   try
    ITEM.Curves.Lock;
    for i:=0 to ITEM.Curves.Count-1 do begin
     Curve:=ITEM.Curves[i];
     if Curve.Ok then
     try
      Curve.Lock;
      SubForm:=Filer.CreateForm(sig_ITEM,sig_CURV);
       {
       Save ITEM name - curve name
       }
       Chunk:=Filer.CreateChunk(sig_name);
       Filer.WriteString(Curve.Name);
       Filer.FlushChunk(Chunk);
       {
       Save ITEM clrp - curve color palette
       }
       Chunk:=Filer.CreateChunk(sig_clrp);
       Filer.WriteLongInt(Curve.Color);
       Filer.FlushChunk(Chunk);
       {
       Save ITEM styl - curve style
       }
       Chunk:=Filer.CreateChunk(sig_styl);
       Filer.WriteLongInt(Curve.Style);
       Filer.FlushChunk(Chunk);
       {
       Save ITEM step - curve step
       }
       Chunk:=Filer.CreateChunk(sig_step);
       Filer.WriteLongInt(Curve.Step);
       Filer.FlushChunk(Chunk);
       {
       Save ITEM data - curve points as array of TPoint2D
       }
       Chunk:=Filer.CreateChunk(sig_data);
       for j:=0 to Curve.Count-1 do Filer.WritePoint2D(Curve[j]);
       Filer.FlushChunk(Chunk);
       {
       Save ITEM text - curve comment text
       }
       Chunk:=Filer.CreateChunk(sig_text);
       Filer.WriteString(Curve.Comment.Text);
       Filer.FlushChunk(Chunk);
      {
      В конце записи формы надо обновить ее заголовок вызовом FlushChunk.
      }
      Filer.FlushChunk(SubForm);
     finally
      Curve.Unlock;
     end;
    end;
   finally
    ITEM.Curves.Unlock;
   end;
   {
   Save ITEM defc - DefCurveNum
   }
   Chunk:=Filer.CreateChunk(sig_defc);
   Filer.WriteLongInt(ITEM.DefCurveNum);
   Filer.FlushChunk(Chunk);
  {
  В конце записи формы надо обновить ее заголовок вызовом FlushChunk.
  }
  Filer.FlushChunk(Form);
  Filer.Flush;
  Result:=true;
  SendToMainConsole('@silent @integrity save.crw '+FileName+EOL);
 except
  on EReadError  do begin
   Filer.Modified:=false;
   Echo(RusEng('Ошибка чтения файла ','Error reading file ')+FileName);
  end;
  on EWriteError do begin
   Filer.Modified:=false;
   Echo(RusEng('Ошибка записи файла ','Error writing file ')+FileName);
  end;
  on E:Exception do begin
   Filer.Modified:=false;
   BugReport(E,nil,'SaveCurveWindowToCRW');
  end;
 end else Echo(RusEng('Ошибка открытия файла ','Error open file ')+FileName);
 Kill(Filer);
end;

type
 TScanRec  = packed record
  Template : TRiffChunk;
  Found    : Boolean;
 end;

procedure ScanChunk(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
begin
 with TScanRec(Custom^) do
 if (Template.dwSign   = Chunk.dwSign) and
    (Template.dwSize   = Chunk.dwSize) and
    (Template.dwFormID = Chunk.dwFormID) and
    (Template.dwOffset = Chunk.dwOffset)
 then begin
  Found:=true;
  Terminate:=true;
 end;
end;

procedure LoadCurve(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
var
 i     : LongInt;
 p     : TPoint2D;
 Curve : TCurve absolute Custom;
begin
 if Curve.Ok then
 case Chunk.dwSign of
  sig_name : Curve.Name:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_clrp : Curve.Color:=Filer.ReadLongInt;
  sig_styl : Curve.Style:=Filer.ReadLongInt;
  sig_step : Curve.Step:=Filer.ReadLongInt;
  sig_data : for i:=0 to (Chunk.dwSize div sizeof(p))-1 do begin
              p:=Filer.ReadPoint2D;
              Curve.AddPoint(p.x,p.y);
             end;
  sig_text : Curve.Comment.Text:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
 end;
end;

procedure LoadCurveWindow(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
var
 Window : TFormCurveWindow absolute Custom;
 Curve  : TCurve;
 r,r1   : TRect2I;
 left,top,width,height,af : LongInt;
begin
 if Window.Ok then
 case Chunk.dwSign of
  sig_name : Window.Caption:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_titl : Window.Title:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_lgnd : Window.Legend:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_text : Window.Comment.Text:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_ltwh : begin
              left:=Filer.ReadLongInt;
              top:=Filer.ReadLongInt;
              width:=Filer.ReadLongInt;
              height:=Filer.ReadLongInt;
              r:=Rect2I(left,top,left+width,top+height);
              if Assigned(FormCrwDaq)
              then r1:=TRect2I(FormCrwDaq.ClientRect)
              else r1:=r;
              if RectIsEqual(r1,RectUnion(r1,r)) then begin
               Window.Left:=Left;
               Window.Top:=top;
               Window.Width:=width;
               Window.Height:=height;
              end;
             end;
  sig_clrp : begin
              Window.GroundColor:=Filer.ReadLongInt;
              Window.LegendColor:=Filer.ReadLongInt;
              Window.NumberColor:=Filer.ReadLongInt;
              Window.BoxColor:=Filer.ReadLongInt;
              Window.RectColor:=Filer.ReadLongInt;
              Window.GridColor:=Filer.ReadLongInt;
              Window.RoiLColor:=Filer.ReadLongInt;
              Window.RoiRColor:=Filer.ReadLongInt;
             end;
  sig_roi_ : Window.Roi:=Filer.ReadRect2D;
  sig_tbtu : begin
              Window.TimeBase:=Filer.ReadDouble;
              Window.TimeUnits:=Filer.ReadDouble;
             end;
  sig_wrld : Window.World:=Filer.ReadRect2D;
  sig_defc : Window.DefCurveNum:=Filer.ReadLongInt;
  sig_ITEM : if Chunk.dwFormID = sig_CURV then begin
              af:=Window.myAxisFlags;
              Window.myAxisFlags:=Window.myAxisFlags and not afCheckIns;
              Curve:=NewCurve;
              Window.AddCurve(Curve);
              Filer.ForEach(LoadCurve,Chunk,Curve);
              Window.myAxisFlags:=af;
             end;
 end;
end;


function LoadCurveWindowFromCRW(FileName:LongString; const Chunk:TRiffChunk):TFormCurveWindow;
var
 Filer   : TRiffFiler;
 ScanRec : TScanRec;
begin
 Result:=nil;
 FileName:=UnifyFileAlias(FileName);
 if FileExists(FileName) and (Chunk.dwSign=sig_ITEM) and (Chunk.dwFormID=sig_CRVW) then begin
  Filer:=NewRiffFiler(FileName,fmOpenReadWrite,sig_CRWF);
  if Filer.Ok then
  try
   Filer.Validate;
   ScanRec.Template:=Chunk;
   ScanRec.Found:=false;
   Filer.ForEach(ScanChunk,Filer.RootForm,@ScanRec);
   if ScanRec.Found then begin
    Result:=NewCurveWindow;
    try
     Result.LockDraw;
     Filer.ForEach(LoadCurveWindow,Chunk,Result);
    finally
     Result.UnlockDraw;
    end;
   end;
   SendToMainConsole('@silent @integrity load.crw '+FileName+EOL);
  except
   on EReadError  do begin
    Kill(Result);
    Filer.Modified:=false;
    Echo(RusEng('Ошибка чтения файла ','Error reading file ')+FileName);
   end;
   on EWriteError do begin
    Kill(Result);
    Filer.Modified:=false;
    Echo(RusEng('Ошибка записи файла ','Error writing file ')+FileName);
   end;
   on E:Exception do begin
    Kill(Result);
    Filer.Modified:=false;
    BugReport(E,nil,'LoadCurveWindowFromCRW');
   end;
  end else Echo(RusEng('Ошибка открытия файла ','Error open file ')+FileName);
  Kill(Filer);
 end;
end;

procedure ReadCurveWindowsPalette(TheIniFile:LongString);
 function ReadColor(const ColName:LongString; DefColor:TColor):TColor;
 var s:LongString;
 begin
  Result:=DefColor;
  try
   s:='';
   TheIniFile:=UnifyFileAlias(TheIniFile);
   if ReadIniFileAlpha(TheIniFile,'[Curve Windows Palette]',ColName+'%a',s)
   or ReadIniFileAlpha(SysIniFile,'[Curve Windows Palette]',ColName+'%a',s)
   then Result:=StringToColor(s);
  except
   on E:Exception do Result:=DefColor;
  end;
 end;
begin
 DefGroundColor := ReadColor('GroundColor', DefGroundColor);
 DefLegendColor := ReadColor('LegendColor', DefLegendColor);
 DefNumberColor := ReadColor('NumberColor', DefNumberColor);
 DefBoxColor    := ReadColor('BoxColor',    DefBoxColor);
 DefRectColor   := ReadColor('RectColor',   DefRectColor);
 DefGridColor   := ReadColor('GridColor',   DefGridColor);
 DefRoiLColor   := ReadColor('RoiLColor',   DefRoiLColor);
 DefRoiRColor   := ReadColor('RoiRColor',   DefRoiRColor);
end;

 {
 ***********************************************
 TFormCurveWindow private methods implementation
 ***********************************************
 }
function TFormCurveWindow.GetIsClipboard: Boolean;
begin
 Result:=Assigned(Self) and (Self=CurveClipboard);
end;

function TFormCurveWindow.GetIsDataProtected: Boolean;
begin
 Result:=Assigned(Self) and not Curves.OwnsObjects;
end;

procedure TFormCurveWindow.SetIsDataProtected(Protection:Boolean);
begin
 if Assigned(Self) then Curves.OwnsObjects:=not Protection;
end;

function TFormCurveWindow.GetWorld:TRect2D;
begin
 if Assigned(Self) then Result:=myWorld else Result:=Rect2D(0,0,0,0);
end;

procedure TFormCurveWindow.SetWorld(const aWorld:TRect2D);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myWorld:=aWorld;
   ResetWorld;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCurveWindow.GetPlotArea:TRect2I;
begin
 if Assigned(Self)
 then Result:=Rect2I(0,0,PaintBox.Width,PaintBox.Height)
 else Result:=Rect2I(0,0,0,0);
end;

function TFormCurveWindow.GetScreenWorld:TRect2I;
begin
 if Assigned(Self) then Result:=GetPlotArea else Result:=Rect2I(0,0,0,0);
 with AxisSettings do begin
  Inc(Result.a.x,Margin.a.x);
  Inc(Result.a.y,Margin.a.y);
  Dec(Result.b.x,Margin.b.x);
  Dec(Result.b.y,Margin.b.y);
 end;
end;

function  TFormCurveWindow.GetAxisFlags:Cardinal;
begin
 if Assigned(Self) then Result:=myAxisFlags else Result:=0;
end;

procedure TFormCurveWindow.SetAxisFlags(aFlags:Cardinal);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myAxisFlags:=aFlags;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetGroundColor:TColor;
begin
 if Assigned(Self) then Result:=myGroundColor else Result:=0;
end;

procedure TFormCurveWindow.SetGroundColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myGroundColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetLegendColor:TColor;
begin
 if Assigned(Self) then Result:=myLegendColor else Result:=0;
end;

procedure TFormCurveWindow.SetLegendColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myLegendColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetNumberColor:TColor;
begin
 if Assigned(Self) then Result:=myNumberColor else Result:=0;
end;

procedure TFormCurveWindow.SetNumberColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myNumberColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetBoxColor:TColor;
begin
 if Assigned(Self) then Result:=myBoxColor else Result:=0;
end;

procedure TFormCurveWindow.SetBoxColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myBoxColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetRectColor:TColor;
begin
 if Assigned(Self) then Result:=myRectColor else Result:=0;
end;

procedure TFormCurveWindow.SetRectColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myRectColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetGridColor:TColor;
begin
 if Assigned(Self) then Result:=myGridColor else Result:=0;
end;

procedure TFormCurveWindow.SetGridColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myGridColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetRoiLColor:TColor;
begin
 if Assigned(Self) then Result:=myRoiLColor else Result:=0;
end;

procedure TFormCurveWindow.SetRoiLColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myRoiLColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormCurveWindow.GetRoiRColor:TColor;
begin
 if Assigned(Self) then Result:=myRoiRColor else Result:=0;
end;

procedure TFormCurveWindow.SetRoiRColor(aColor:TColor);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myRoiRColor:=aColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCurveWindow.GetTitle:LongString;
begin
 if Assigned(Self) then Result:=myTitle else Result:='';
end;

procedure TFormCurveWindow.SetTitle(const aTitle:LongString);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myTitle:=aTitle;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCurveWindow.GetLegend:LongString;
begin
 if Assigned(Self) then Result:=myLegend else Result:='';
end;

procedure TFormCurveWindow.SetLegend(const aLegend:LongString);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myLegend:=aLegend;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCurveWindow.GetCurves:TCurveList;
begin
 if Assigned(Self) then Result:=myCurves else Result:=nil;
end;

function TFormCurveWindow.GetComment:TText;
begin
 if Assigned(Self) then Result:=myComment else Result:=nil;
end;

function TFormCurveWindow.GetDefCurveNum:Integer;
begin
 if Assigned(Self) and (Cardinal(myDefCurveNum)<Cardinal(myCurves.Count))
 then Result:=myDefCurveNum
 else Result:=-1;
end;

procedure TFormCurveWindow.SetDefCurveNum(aNumber:Integer);
begin
 if Assigned(Self) and (DefCurveNum<>aNumber) then begin
  try
   LockDraw;
   if Cardinal(aNumber)<Cardinal(myCurves.Count)
   then myDefCurveNum:=aNumber
   else myDefCurveNum:=-1;
   UpdateScroller;
   UpdatePaintBox;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCurveWindow.GetDefCurve:TCurve;
begin
 if Assigned(Self) then Result:=Curves[DefCurveNum] else Result:=nil;
end;

procedure TFormCurveWindow.SetDefCurve(aCurve:TCurve);
begin
 if Assigned(Self) then DefCurveNum:=Curves.IndexOf(aCurve);
end;

function TFormCurveWindow.GetDefComment:TText;
begin
 if HasSelection then Result:=DefCurve.Comment else Result:=Comment;
end;

function TFormCurveWindow.GetHasSelection:Boolean;
begin
 if Assigned(Self) then Result:=(DefCurve<>nil) else Result:=false;
end;

function TFormCurveWindow.GetMouseMode:TMouseMode;
begin
 if Assigned(Self) then Result:=myMouseMode else Result:=mmSelector;
end;

procedure TFormCurveWindow.SetMouseMode(aMode:TMouseMode);
 procedure Setup(aDragMode:TDragMode; aCursor:TCursor; aToolButton:TToolButton);
 begin
  PaintBox.DragMode:=aDragMode;
  PaintBox.Cursor:=aCursor;
  aToolButton.Down:=true;
 end;
begin
 if Assigned(Self) then
 try
  myXorSelector.Stop(PaintBox.Canvas);
  myXorRoi.Stop(PaintBox.Canvas);
  myXorRoiL.Stop(PaintBox.Canvas);
  myXorRoiR.Stop(PaintBox.Canvas);
  myMouseMode:=aMode;
  case myMouseMode of
   mmSelector   : Setup(dmManual,    crCross,     ToolButtonCurveMouseSelector);
   mmZoomIn     : Setup(dmManual,    crHandPoint, ToolButtonCurveMouseZoomIn);
   mmZoomOut    : Setup(dmManual,    crHandPoint, ToolButtonCurveMouseZoomOut);
   mmDragCopy   : Setup(dmAutomatic, crMultiDrag, ToolButtonCurveMouseDragCopy);
   mmDragMove   : Setup(dmAutomatic, crMultiDrag, ToolButtonCurveMouseDragMove);
   mmSelectRoi  : Setup(dmManual,    crCross,     ToolButtonCurveMouseSelectRoi);
   mmSelectRoiL : begin
                   Setup(dmManual, crCross, ToolButtonCurveMouseSelectRoiL);
                   myXorRoiL.Start(PaintBox.Canvas, TPoint2I(PaintBox.ScreenToClient(Mouse.CursorPos)));
                   Roi:=Rect2D(_Nan,_Nan,Roi.B.X,Roi.B.Y);
                  end;
   mmSelectRoiR : begin
                   Setup(dmManual, crCross, ToolButtonCurveMouseSelectRoiR);
                   myXorRoiR.Start(PaintBox.Canvas, TPoint2I(PaintBox.ScreenToClient(Mouse.CursorPos)));
                   Roi:=Rect2D(Roi.A.X,Roi.A.Y,_Nan,_Nan);
                  end;
  end;
 except
  on E:Exception do BugReport(E,Self,'SetMouseMode');
 end;
end;

function TFormCurveWindow.GetRoi:TRect2D;
begin
 if Assigned(Self) then Result:=myRoi else Result:=Rect2D(_Nan,_Nan,_Nan,_Nan);
end;

procedure TFormCurveWindow.SetRoi(const aRoi:TRect2D);
begin
 if Assigned(Self) then myRoi:=aRoi;
end;

function TFormCurveWindow.GetTimeBase:Double;
begin
 if Assigned(Self) then Result:=myTimeBase else Result:=_Nan;
end;

procedure TFormCurveWindow.SetTimeBase(aTimeBase:Double);
begin
 if Assigned(Self) then begin
  myTimeBase:=aTimeBase;
  UpdateTimeAxisX;
 end;
end;

function TFormCurveWindow.GetTimeUnits:Double;
begin
 if Assigned(Self) then Result:=myTimeUnits else Result:=_Nan;
end;

procedure TFormCurveWindow.SetTimeUnits(aTimeUnits:Double);
begin
 if Assigned(Self) then begin
  myTimeUnits:=aTimeUnits;
  UpdateTimeAxisX;
 end;
end;

function TFormCurveWindow.GetCanUseTimeAxisX:Boolean;
begin
 if Assigned(Self)
 then Result:=IsValidTimeScale(myTimeBase,myTimeUnits)
 else Result:=false;
end;

function TFormCurveWindow.GetWantTimeAxisX:Boolean;
begin
 if Assigned(Self)
 then Result:=myWantTimeAxisX
 else Result:=false;
end;

procedure TFormCurveWindow.SetWantTimeAxisX(aWant:Boolean);
begin
 if Assigned(Self) then begin
  myWantTimeAxisX:=aWant;
  UpdateTimeAxisX;
 end;
end;

function TFormCurveWindow.GetDownSampling:TDownSamplingParams;
const ZeroParams:TDownSamplingParams=(Method:0;Mode:0;AbsLen:0;PerPix:0;Tail:0);
begin
 if Assigned(Self)
 then Result:=myDownSampling
 else Result:=ZeroParams;
end;

procedure TFormCurveWindow.SetDownSampling(const aParams:TDownSamplingParams);
begin
 if Assigned(Self) then begin
  if CompareMem(@myDownSampling,@aParams,SizeOf(myDownSampling)) then Exit; // Nothing to do
  LockDraw;
  myDownSampling:=aParams;
  UnlockDraw;
 end;
end;

function TFormCurveWindow.GetUsesDownSampling:Boolean;
begin
 if Assigned(Self)
 then Result:=HasFlags(myDownSampling.Mode,dsm_Enable)
 else Result:=false;
end;

procedure TFormCurveWindow.SetUsesDownSampling(aUses:Boolean);
begin
 if Assigned(Self) then begin
  if (UsesDownSampling=aUses) then Exit; // Nothing to do
  LockDraw;
  LiftFlags(myDownSampling.Mode,dsm_Enable,aUses);
  UnlockDraw;
 end;
end;

function TFormCurveWindow.DownSamplingMethod:Integer;
begin
 if Assigned(Self)
 then Result:=myDownSampling.Method
 else Result:=0;
end;

function TFormCurveWindow.DownSamplingLength:Integer;
var AbsLen,PixLen,PixWidth:Integer;
begin
 Result:=0;
 if Assigned(Self) then
 if (myDownSampling.Method in dsm_GOOD) then
 if HasFlags(myDownSampling.Mode,dsm_Enable) then
 if HasFlags(myDownSampling.Mode,dsm_AbsLen+dsm_PerPix) then begin
  AbsLen:=MaxInt; PixLen:=MaxInt; PixWidth:=RectSizeX(PlotArea);
  if HasFlags(myDownSampling.Mode,dsm_AbsLen) then AbsLen:=Max(0,myDownSampling.AbsLen);
  if HasFlags(myDownSampling.Mode,dsm_PerPix) then PixLen:=Max(0,Round(myDownSampling.PerPix*PixWidth));
  Result:=Min(AbsLen,PixLen); if (Result=MaxInt) then Result:=0;
 end;
end;

procedure TFormCurveWindow.UpdateDownSamplingStatus;
var Img:Integer;
begin
 if Assigned(Self) then begin
  Img:=250; // Transparent lightning
  if HasFlags(myDownSampling.Mode,dsm_Enable) then
  if (myDownSampling.Method in dsm_GOOD) then begin
   if (myDownSampCount=0)
   then Img:=258  // White
   else Img:=253; // Lime
  end;
  if (ActionCurveEditDownSampling.ImageIndex<>Img)
  then ActionCurveEditDownSampling.ImageIndex:=Img;
 end;
end;

function TFormCurveWindow.DownSamplingTail:Integer;
begin
 if Assigned(Self)
 then Result:=myDownSampling.Tail
 else Result:=0;
end;

function TFormCurveWindow.GetUsesTimeAxisX:Boolean;
begin
 Result:=WantTimeAxisX and CanUseTimeAxisX;
end;

procedure TFormCurveWindow.UpdateTimeAxisX;
begin
 if Assigned(Self) then
 try
  ActionCurveRangeTimeAxisX.Enabled:=CanUseTimeAxisX;
  ActionCurveRangeTimeAxisX.Checked:=myWantTimeAxisX;
  if ActionCurveRangeTimeAxisX.Checked
  then ActionCurveRangeTimeAxisX.ImageIndex:=225
  else ActionCurveRangeTimeAxisX.ImageIndex:=224;
 except
  on E:Exception do BugReport(E,Self,'UpdateTimeAxisX');
 end;
end;

procedure TFormCurveWindow.UpdateCommands;
var s:LongString; Exposed,NotProtected,CanCopy,CanCut,CanPaste,CanTimeAxisX:Boolean;
    VisibleCurves:Integer; ms:Double;
begin
 inherited UpdateCommands;
 try
  Exposed:=FormIsExposed(Self);
  NotProtected:=Exposed and not IsDataProtected;
  if Exposed then VisibleCurves:=Curves.Count else VisibleCurves:=0;
  CanCopy:=Exposed and HasSelection;
  CanCut:=CanCopy and NotProtected;
  CanPaste:=NotProtected and CurveClipboard.HasSelection;
  CanTimeAxisX:=CanUseTimeAxisX;
  {}
  ActionFileSave.Enabled:=Exposed;
  ActionFileSaveAs.Enabled:=Exposed;
  ActionFilePrint.Enabled:=Exposed;
  ActionCurveMouseSelector.Enabled:=Exposed;
  ActionCurveMouseZoomIn.Enabled:=Exposed;
  ActionCurveMouseZoomOut.Enabled:=Exposed;
  ActionCurveMouseDragCopy.Enabled:=Exposed;
  ActionCurveMouseSelectRoi.Enabled:=Exposed;
  ActionCurveMouseSelectRoiL.Enabled:=Exposed;
  ActionCurveMouseSelectRoiR.Enabled:=Exposed;
  ActionCurveRangeAuto.Enabled:=Exposed;
  ActionCurveRangeSelector.Enabled:=Exposed;
  ActionCurveRangeZoomIn.Enabled:=Exposed;
  ActionCurveRangeZoomOut.Enabled:=Exposed;
  ActionCurveRangeLeft.Enabled:=Exposed;
  ActionCurveRangeRight.Enabled:=Exposed;
  ActionCurveRangeUp.Enabled:=Exposed;
  ActionCurveRangeDown.Enabled:=Exposed;
  ActionCurveRangeTimeAxisX.Enabled:=CanTimeAxisX;
  ActionCurveEditCloneWindow.Enabled:=Exposed;
  ActionCurveEditComment.Enabled:=Exposed;
  ActionCurveTools.Enabled:=Exposed;
  ActionCurveToolsWriteTable.Enabled:=Exposed;
  ActionCurveToolsComposeSurface.Enabled:=Exposed;
  ActionCurveEditWindowStyle.Enabled:=NotProtected;
  ActionCurveEditDownSampling.Enabled:=Exposed;
  ActionCurveMouseDragMove.Enabled:=NotProtected;
  ActionCurveToolsRunMacroDialog.Enabled:=NotProtected;
  ActionCurveToolsRunPluginDialog.Enabled:=NotProtected;
  ActionCurveEditSelect0.Enabled:=(VisibleCurves>0);
  ActionCurveEditSelect1.Enabled:=(VisibleCurves>0);
  ActionCurveEditSelectMore.Enabled:=(VisibleCurves>0);
  ActionCurveEditSelect2.Enabled:=(VisibleCurves>1);
  ActionCurveEditSelect3.Enabled:=(VisibleCurves>2);
  ActionCurveEditSelect4.Enabled:=(VisibleCurves>3);
  ActionCurveEditSelect5.Enabled:=(VisibleCurves>4);
  ActionCurveEditSelect6.Enabled:=(VisibleCurves>5);
  ActionCurveEditSelect7.Enabled:=(VisibleCurves>6);
  ActionCurveEditSelect8.Enabled:=(VisibleCurves>7);
  ActionCurveEditSelect9.Enabled:=(VisibleCurves>8);
  ActionCurveEditCopy.Enabled:=CanCopy;
  ActionCurveEditDelete.Enabled:=CanCut;
  ActionCurveEditCut.Enabled:=CanCut;
  ActionCurveEditStyle.Enabled:=CanCut;
  ActionCurveEditPaste.Enabled:=CanPaste;
  with Roi do begin
   ActionCurveMouseClearRoiL.Enabled:=Exposed and not isNanOrInf(A);
   ActionCurveMouseClearRoiR.Enabled:=Exposed and not isNanOrInf(B);
  end;
  {}
  s:='';
  if isNanOrInf(myMousePos) then begin
   s:=s+RusEng(' Кривых:','Curves: ')+d2s(Curves.Count);
   if HasSelection then begin
    s:=s+RusEng(' Выбрана:',' Selected:')+d2s(DefCurveNum+1)+
         RusEng(' Имя:',' Name:')+DefCurve.Name+
         RusEng(' Точек:',' Points:')+d2s(DefCurve.Count);
   end else begin
    s:=s+RusEng(' Выбрана:НЕТ',' Selected:NONE');
   end;
  end else begin
   s:=s+RusEng(' Мышь:',' Mouse:');
   case MouseMode of
    mmSelector   : s:=s+RusEng('Селектор','Selector');
    mmZoomIn     : s:=s+RusEng('Увеличение','Zoom In');
    mmZoomOut    : s:=s+RusEng('Уменьшение','Zoom Out');
    mmDragCopy   : s:=s+RusEng('Копирование','Drag & Copy');
    mmDragMove   : s:=s+RusEng('Перемещение','Drag & Move');
    mmSelectRoi  : s:=s+RusEng('Маркеры РОИ','ROI markers');
    mmSelectRoiL : s:=s+RusEng('Левый маркер РОИ','Left ROI mark');
    mmSelectRoiR : s:=s+RusEng('Правый маркер РОИ','Right ROI mark');
    else s:=s+'?';
   end;
   s:=s+Format('  X=%12.8f  Y=%12.8f',[myMousePos.X/myGridX.Scale,myMousePos.Y/myGridY.Scale]);
   {Show date/time}
   if IsValidTimeScale(myTimeBase,myTimeUnits) then begin
    ms:=myTimeBase+myMousePos.x*myTimeUnits;
    if ms>=0 then                        // 0001.01.01-00:00:00:000 - min date
    if ms<=315506361600000.0 then begin  // 9999.01.01-00:00:00:000 - max date
     s:=s+'  T='+GetDateStr(ms,'.',true)+'-'+GetTimeStr(ms,':',true);
    end;
   end;
  end;
  if StatusBar.SimpleText<>s then begin
   StatusBar.SimpleText:=s;
   StatusBar.Update;
  end;
  if (myUpdateLoops mod 8 = 0) then UpdateDownSamplingStatus;
  Inc(myUpdateLoops);
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormCurveWindow.FileSave;
const FName:LongString='';
var Ext,Params:LongString;
begin
 try
  SaveDialog.Title:=RusEng('Файл\Сохранить','File\Save');
  SaveDialog.Filter:=RusEng('Архивные файлы CRW (*.crw)|*.crw|'+
                            'Архивные файлы DAQ (*.daq)|*.daq|'+
                            'Все остальные файлы  (*.*)|*.*|',
                            'CRW archive files (*.crw)|*.crw|'+
                            'DAQ archive files (*.daq)|*.daq|'+
                            'All other files     (*.*)|*.*|');
  if IsEmptyStr(SaveDialog.FileName) then SaveDialog.FileName:=FName;
  if IsEmptyStr(SaveDialog.FileName)
  then OpenDialogSelectType(SaveDialog,'*.crw')
  else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
  Params:=ControlPosParams(ToolBar,'LB');
  if ExecuteFileDialog(GuardOpenDialog(SaveDialog),Params) then begin
   FName:=UnifyFileAlias(SaveDialog.FileName);
   Ext:=UnifyAlias(SaveDialog.DefaultExt);
   if IsEmptyStr(Ext) or IsWildCard(Ext)
   then FName:=UnifyFileAlias(FName,ua_FileDefLow)
   else FName:=UnifyFileAlias(ForceExtension(FName,Ext),ua_FileDefLow);
   SaveCurveWindowToCrw(FName,Self);
  end;
 except
  on E:Exception do BugReport(E,Self,'FileSave');
 end;
end;

procedure TFormCurveWindow.FileSaveAs;
var Params:LongString;
begin
 if Ok and not CheckIsDataProtected then
 try
  Params:=IfThen(Visible,ControlPosParams(PaintBox,'LT'),'');
  RunCurveWindowStyleDialog(Self,Params);
  FileSave;
 except
  on E:Exception do BugReport(E,Self,'FileSaveAs');
 end;
end;

procedure TFormCurveWindow.ResetWorld;
 procedure ValidateLimits(var R:TRect2D);
 var
  Scale:TPoint2D;
  Sizes:TPoint2D;
 begin
  if isNanOrInf(R) then R:=Rect2D(0,0,1,1);
  Scale.X:=(abs(R.A.X)+abs(R.B.X));
  Scale.Y:=(abs(R.A.Y)+abs(R.B.Y));
  Sizes.X:=R.B.X-R.A.X;
  Sizes.Y:=R.B.Y-R.A.Y;
  if abs(Sizes.X)<=Precision(WorldAbsEps,WorldRelEps,Scale.X) then begin
   R.A.X:=R.A.X-Scale.X;
   R.B.X:=R.B.X+Scale.X;
  end;
  if abs(Sizes.Y)<=Precision(WorldAbsEps,WorldRelEps,Scale.Y) then begin
   R.A.Y:=R.A.Y-Scale.Y;
   R.B.Y:=R.B.X+Scale.Y;
  end;
 end;
 procedure CheckWorldEmpty;
 begin
  myWorld:=RectValidate(myWorld);
  if RectIsEmpty(myWorld) then ValidateLimits(myWorld);
  if RectIsEmpty(myWorld) then myWorld:=Rect2D(0,0,1,1);
 end;
var
 R:TRect2I;
begin
 if Assigned(Self) then
 try
  CheckWorldEmpty;
  R:=GetScreenWorld;
  myScale.X:=(R.B.X-R.A.X)/(myWorld.B.X-myWorld.A.X);
  myScale.Y:=(R.B.Y-R.A.Y)/(myWorld.B.Y-myWorld.A.Y);
  if myScale.X=0 then myScale.X:=1;
  if myScale.Y=0 then myScale.Y:=1;
  myOrigin.X:=R.A.X-myWorld.A.X*myScale.X;
  myOrigin.Y:=R.A.Y-myWorld.A.Y*myScale.Y;
  if myAxisFlags and afReversX <> 0 then begin
   myScale.X:=-myScale.X;
   myOrigin.X:=R.B.X-myWorld.A.X*myScale.X;
  end;
  if myAxisFlags and afReversY <> 0 then begin
   myScale.Y:=-myScale.Y;
   myOrigin.Y:=R.B.Y-myWorld.A.Y*myScale.Y;
  end;
  myGridX:=AxisGrid(myWorld.A.X,myWorld.B.X);
  myGridY:=AxisGrid(myWorld.A.Y,myWorld.B.Y);
 except
  on E:Exception do BugReport(E,Self,'ResetWorld');
 end;
end;

procedure TFormCurveWindow.SomeGrowWorld(Lim:TRect2D; Flags:Cardinal);
var W:TRect2D;
const
   NearlyZero=1.0E-5;
begin
 if Assigned(Self) then
 try
  if Flags and afCheckX <> 0
  then RectGrow(Lim, RectSizeX(Lim)*AxisSettings.GrowStep.X, 0)
  else begin
   Lim.a.x:=myWorld.a.x;
   Lim.b.x:=myWorld.b.x;
  end;
  if Flags and afCheckY <> 0
  then RectGrow(Lim, 0, RectSizeY(Lim)*AxisSettings.GrowStep.Y)
  else begin
   Lim.a.Y:=myWorld.a.Y;
   Lim.b.Y:=myWorld.b.Y;
  end;
  W:=myWorld;
  RectGrow(W, RectSizeX(W)*NearlyZero, RectSizeY(W)*NearlyZero);
  if (Flags and afUpdate <> 0)
  or not RectContainsPoint(W,Lim.A)
  or not RectContainsPoint(W,Lim.B)
  then begin
   if Flags and afUnion <> 0 then Lim:=RectUnion(Lim,myWorld);
   World:=Lim;
  end;
 except
  on E:Exception do BugReport(E,Self,'SomeGrowWorld');
 end;
end;

function TFormCurveWindow.WorldToLocal(const P:TPoint2D):TPoint2I;
var Local:TPoint2D;
begin
 if Assigned(Self) then begin
  Local.X:=myOrigin.X + myScale.X * P.X;
  Local.Y:=myOrigin.Y + myScale.Y * P.Y;
  if abs(Local.X)<MaxInt then Result.X:=Trunc(Local.X) else Result.X:=MaxInt;
  if abs(Local.Y)<MaxInt then Result.Y:=Trunc(Local.Y) else Result.Y:=MaxInt;
 end else begin
  Result.X:=0;
  Result.Y:=0;
 end;
end;

function TFormCurveWindow.WorldToLocalX(X:Double):Integer;
var Local:Double;
begin
 if Assigned(Self) then begin
  Local:=myOrigin.X + myScale.X * X;
  if abs(Local)<MaxInt then Result:=Trunc(Local) else Result:=MaxInt;
 end else Result:=0;
end;

function TFormCurveWindow.WorldToLocalY(Y:Double):Integer;
var Local:Double;
begin
 if Assigned(Self) then begin
  Local:=myOrigin.Y + myScale.Y * Y;
  if abs(Local)<MaxInt then Result:=Trunc(Local) else Result:=MaxInt;
 end else Result:=0;
end;

function TFormCurveWindow.WorldToLocalRect(const R:TRect2D):TRect2I;
begin
 if Assigned(Self) then begin
  if myAxisFlags and afReversX <> 0 then begin
   Result.A.X:=WorldToLocalX(R.B.X);
   Result.B.X:=WorldToLocalX(R.A.X);
  end else begin
   Result.A.X:=WorldToLocalX(R.A.X);
   Result.B.X:=WorldToLocalX(R.B.X);
  end;
  if myAxisFlags and afReversY <> 0 then begin
   Result.A.Y:=WorldToLocalY(R.B.Y);
   Result.B.Y:=WorldToLocalY(R.A.Y);
  end else begin
   Result.A.Y:=WorldToLocalY(R.A.Y);
   Result.B.Y:=WorldToLocalY(R.B.Y);
  end;
 end else Result:=Rect2I(0,0,0,0);
end;

function TFormCurveWindow.LocalToWorld(const P:TPoint2I):TPoint2D;
begin
 if Assigned(Self) then begin
  if myScale.X<>0 then Result.X:=(P.X-myOrigin.X)/myScale.X else Result.X:=0;
  if myScale.Y<>0 then Result.Y:=(P.Y-myOrigin.Y)/myScale.Y else Result.Y:=0;
 end else begin
  Result.X:=0;
  Result.Y:=0;
 end;
end;

function TFormCurveWindow.LocalToWorldX(X:Integer):Double;
begin
 if Assigned(Self) and (myScale.X<>0)
 then Result:=(X-myOrigin.X)/myScale.X
 else Result:=0;
end;

function TFormCurveWindow.LocalToWorldY(Y: Integer): Double;
begin
 if Assigned(Self) and (myScale.Y<>0)
 then Result:=(Y-myOrigin.Y)/myScale.Y
 else Result:=0;
end;

function TFormCurveWindow.LocalToWorldRect(const R:TRect2I):TRect2D;
begin
 if Assigned(Self) then begin
  if myAxisFlags and afReversX <> 0 then begin
   Result.A.X:=LocalToWorldX(R.B.X);
   Result.B.X:=LocalToWorldX(R.A.X);
  end else begin
   Result.A.X:=LocalToWorldX(R.A.X);
   Result.B.X:=LocalToWorldX(R.B.X);
  end;
  if myAxisFlags and afReversY <> 0 then begin
   Result.A.Y:=LocalToWorldY(R.B.Y);
   Result.B.Y:=LocalToWorldY(R.A.Y);
  end else begin
   Result.A.Y:=LocalToWorldY(R.A.Y);
   Result.B.Y:=LocalToWorldY(R.B.Y);
  end;
 end;
end;

procedure TFormCurveWindow.InsertCurve(Index:integer; aCurve:TCurve);
var Lim:TRect2D;
begin
 if Assigned(Self) and Assigned(aCurve) and (Curves.IndexOf(aCurve)<0) then
 try
  try
   LockDraw;
   if IsClipBoard then while Curves.Count>MaxClipboardCurves do Curves.Delete(0);
   if AxisFlags and afCheckIns <> 0 then begin
    Lim:=aCurve.Limits;
    if IsClipboard then World:=Lim;
    SomeGrowWorld(Lim,(AxisFlags and afUnion) or afCheck);
   end;
   if Cardinal(Index)<Cardinal(Curves.Count)
   then Curves.Insert(Index,aCurve)
   else Curves.Add(aCurve);
   if AxisFlags and afSelectIns<>0 then DefCurve:=aCurve;
   UpdateScroller;
   UpdatePaintBox;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'InsertCurve');
 end;
end;

procedure TFormCurveWindow.AddCurve(aCurve:TCurve);
begin
 InsertCurve(-1,aCurve);
end;

procedure TFormCurveWindow.DeleteCurveNum(Num:Integer);
var aLimits:TRect2D;
begin
 if Assigned(Self) and (Cardinal(Num)<Cardinal(Curves.Count)) then
 try
  try
   LockDraw;
   Curves.Delete(Num);
   if IsClipBoard then begin
    DefCurveNum:=Curves.Count-1;
    if HasSelection then begin
     aLimits:=DefCurve.Limits;
     World:=aLimits;
     SomeGrowWorld(aLimits,afCheck);
    end;
   end else DefCurveNum:=-1;
   UpdateScroller;
   UpdatePaintBox;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'DeleteCurveNum');
 end;
end;

procedure TFormCurveWindow.DeleteCurve(aCurve:TCurve);
begin
 DeleteCurveNum(Curves.IndexOf(aCurve));
end;

procedure TFormCurveWindow.UpdateScroller;
begin
 if Assigned(Self) then
 if (ScrollBar.Position<>DefCurveNum+1) or (ScrollBar.Max<>Curves.Count) then
 try
  ScrollBar.Max:=Curves.Count;
  ScrollBar.Position:=DefCurveNum+1;
 except
  on E:Exception do BugReport(E,Self,'UpdateScroller');
 end;
end;

procedure TFormCurveWindow.UpdatePaintBox;
begin
 if Assigned(Self) then PaintBox.Update;
end;

procedure TFormCurveWindow.DrawView;
begin
 if Assigned(Self) then
 if IsFormViewable then
 try
  DebugLogReport_DrawView;
  Inc(CurveWindowsProfiler.Curr.DrawView);
  myDownSampCount:=0;
  UpdateTimeAxisX;
  DrawAxis(dfDefault);
  DrawCurves;
  UpdateDownSamplingStatus;
 except
  on E:Exception do BugReport(E,Self,'DrawView');
 end;
end;

procedure TFormCurveWindow.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu; tb,sb,sy:Boolean;
var Params:LongString;
 function GetTable:LongString;
 var List:TStringList; s:LongString; i,j,i1,i2,Len:Integer;
 begin
  Result:='';
  if Curves.Count>0 then begin
   if DefCurveNum>=0 then begin
    i1:=DefCurveNum;
    i2:=DefCurveNum;
   end else begin
    i1:=0;
    i2:=Curves.Count-1;
   end;
   Len:=0;
   List:=TStringList.Create;
   try
    for i:=i1 to i2 do Len:=Max(Len,Curves[i].Count);
    for j:=0 to Len-1 do begin
     s:='';
     for i:=i1 to i2 do begin
      if Length(s)>0 then s:=s+#9;
      if j<Curves[i].Count
      then with Curves[i][j] do s:=s+Format('%g'#9'%g',[x,y])
      else s:=s+'-'#9'-';
     end;
     List.Add(s);
    end;
    Result:=List.Text;
    Echo(RusEng('Таблица скопирована в буфер обмена!','Table copied to clipboard!'));
   finally
    List.Free;
    Finalize(s);
   end;
  end;
 end;
begin
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +PaintBoxPosParams;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать как текстовую таблицу в буфер обмена'+EOL+
                         'Скопировать изображение в буфер обмена (цвет.)'+EOL+
                         'Скопировать изображение в буфер обмена (ч./б.)'+EOL+
                         'Скопировать изображение в буфер обмена (серый)'+EOL+
                         'Напечатать на Принтере '+Printer.PrinterName,
                         'Copy as text table to clipboard'+EOL+
                         'Copy bitmap to clipboard (color)'+EOL+
                         'Copy bitmap to clipboard (b./w.)'+EOL+
                         'Copy bitmap to clipboard (gray)'+EOL+
                         'Print hardcopy on Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  tb:=ToolBar.Visible;
  sb:=StatusBar.Visible;
  sy:=ScrollBar.Visible;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   ResetWorld;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0:    Clipboard.AsText:=GetTable;
    1..3: begin
           Bmp:=GetPrintableImage;
           if Assigned(Bmp) then
           try
            if (Key=2) then ConvertBmpToBlackAndWhite(Bmp,GroundColor,BoxColor,1);
            if (Key=3) then ConvertBmpToBlackAndWhite(Bmp,GroundColor,BoxColor,3);
            CopyFormBmpToClipboard(Bmp,true);
           finally
            Kill(Bmp);
           end;
          end;
    4:    if HasPrintersDialog then Print;
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=tb;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=sb;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBar.Visible:=sy;
  end;
  ResetWorld;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

function TFormCurveWindow.GetImage(BlackAndWhite:Boolean):TBitmap;
begin
 Result:=nil;
 if Ok then
 try
  Toolbar.Hide;
  StatusBar.Hide;
  ScrollBar.Hide;
  ResetWorld;
  Result:=GetFormImage;
  if BlackAndWhite then ConvertBmpToBlackAndWhite(Result,GroundColor,BoxColor);
  Toolbar.Show;
  StatusBar.Show;
  ScrollBar.Show;
  ResetWorld;
 except
  on E:Exception do BugReport(E,Self,'GetImage');
 end;
end;

procedure TFormCurveWindow.DrawAxis(Flags:Cardinal);
var R,ViewBox,PlotBox:TRect2I; p:TPoint2I; i,k:Integer; s:LongString; x,y:Double;
 {
 draw message '*10^Pw' with big and small font, righttop-justify
 }
 procedure DrawPowerMessage(aPower,px,py:integer);
 var p:TPoint2I; Bmp:TBitmap;
 begin
  if aPower<>0 then begin
   str(aPower,s);
   p:=Point2I(px-PaintBox.Canvas.TextWidth(s), py);
   mySmartDrawer.DrawText(PaintBox.Canvas,p,s,NumberColor,GroundColor,0,0);
   s:='*10';
   with PaintBox.Font do
   Bmp:=CreateTextBitmap(s,CharSet,NumberColor,2*Height,Name,Pitch,Style,GroundColor);
   try
    p:=Point2I(p.x-Bmp.Width, py);
    mySmartDrawer.DrawBitmap(PaintBox.Canvas,p,Bmp);
   finally
    Kill(Bmp);
   end;
  end;
 end;
 {
 draw date-time message with small font, centered bottom justify
 }
 procedure DrawDateTime(ms:Double; px,py:integer; const F:TFontParams);
 var p:TPoint2I; Bmp:TBitmap; ds,ts:String[15];
 begin
  if IsValidTimeScale(ms,myTimeUnits) then begin
   ds:=CenterStr(GetDateStr(ms,'.',false,true), 12,' '); // Must uses monospace font
   ts:=CenterStr(GetTimeStr(ms,':',false),      12,' '); // like PT Mono or Courier New
   Bmp:=CreateTextBitmap(ds,F.CharSet,NumberColor,F.Height,F.Name,F.Pitch,F.Style,GroundColor);
   try
    p:=Point2I(px-(Bmp.Width div 2), py-Bmp.Height*2-1);
    mySmartDrawer.DrawBitmap(PaintBox.Canvas,p,Bmp);
   finally
    Kill(Bmp);
   end;
   Bmp:=CreateTextBitmap(ts,F.CharSet,NumberColor,F.Height,F.Name,F.Pitch,F.Style,GroundColor);
   try
    p:=Point2I(px-(Bmp.Width div 2), py-Bmp.Height-1);
    mySmartDrawer.DrawBitmap(PaintBox.Canvas,p,Bmp);
   finally
    Kill(Bmp);
   end;
  end;
 end;
 {
 Format string for grid text
 }
 function NumToStr(X:Double):LongString;
 var S:LongString;
 begin
  Str(X:AxisSettings.Width:AxisSettings.Decim,S);
  S:=TrimChars(S,[' '], [' ','0'] );
  if pos('.',S)>0 then begin
   S:=TrimTrailChars(S,['0']);
   S:=TrimTrailChars(S,['.']);
  end;
  Result:=S;
 end;
begin
 if Assigned(Self) then
 if IsFormViewable then
 try
  mySmartDrawer.StartDraw;
  ViewBox:=PlotArea;
  PlotBox:=ScreenWorld;
  {
  Draw bars
  }
  if Flags and dfGround <> 0 then begin
   R:=Rect2I(ViewBox.A.X, ViewBox.A.Y, ViewBox.B.X, PlotBox.A.Y);
   DrawBar(PaintBox.Canvas,R,GroundColor);
   R:=Rect2I(ViewBox.A.X, PlotBox.A.Y, PlotBox.A.X, PlotBox.B.Y);
   DrawBar(PaintBox.Canvas,R,GroundColor);
   R:=Rect2I(PlotBox.B.X, PlotBox.A.Y, ViewBox.B.X, PlotBox.B.Y);
   DrawBar(PaintBox.Canvas,R,GroundColor);
   R:=Rect2I(ViewBox.A.X, PlotBox.B.Y, ViewBox.B.X, ViewBox.B.Y);
   DrawBar(PaintBox.Canvas,R,GroundColor);
  end;
  if Flags and dfPlotBox <> 0 then begin
   DrawCustomRect(PaintBox.Canvas,PlotBox,RectColor,psSolid,pmCopy,1,BoxColor,bsSolid);
  end;
  {
  Draw x and y power such as *10^5
  }
  if Flags and dfGridTextX <> 0
  then DrawPowerMessage(myGridX.Pow,PlotBox.B.X,PlotBox.B.Y+AxisSettings.Inter.Y+AxisSettings.Space.Y);
  if Flags and dfGridTextY <> 0
  then DrawPowerMessage(myGridY.Pow,PlotBox.A.X-AxisSettings.Inter.X-AxisSettings.Space.X, PlotBox.A.Y);
  {
  Draw Х axis gridlines and numbers
  }
  R:=Rect2I(PlotBox.B.X+1, ViewBox.A.Y, ViewBox.B.X, ViewBox.B.Y);
  mySmartDrawer.LockRect(R);
  for i:=myGridX.Num downto 0 do begin
   x:=myGridX.Start+i*myGridX.Step;
   if (x>=myWorld.a.x) and (x<=myWorld.b.x) then begin
    k:=WorldToLocalX(x);
    if Flags and dfGridLineX <> 0 then begin
     DrawPatternLine(PaintBox.Canvas,Point2I(k,PlotBox.a.y),Point2I(k,PlotBox.b.y-1),GridColor,AxisSettings.GridPattern);
    end;
    if Flags and dfGridTextX <> 0 then begin
     s:=NumToStr(x/myGridX.Scale);
     p.x:=k-PaintBox.Canvas.TextWidth(s) shr 1;
     p.y:=PlotBox.B.Y+AxisSettings.Inter.Y+AxisSettings.Space.Y;
     if p.x<0 then continue;
     if p.x+PaintBox.Canvas.TextWidth(s)>PlotBox.B.X then continue;
     if mySmartDrawer.DrawText(PaintBox.Canvas,p,s,NumberColor,GroundColor,2,2)
     then DrawLine(PaintBox.Canvas,Point2I(k,PlotBox.B.Y),Point2I(k,PlotBox.B.Y+AxisSettings.Stick.y),GridColor);
     if UsesTimeAxisX then DrawDateTime(myTimeBase+myTimeUnits*x,k,PlotBox.a.y,DefaultTimeAxisFont);
    end;
   end;
  end;
  {
  Draw Y axis gridlines and numbers
  }
  for i:=myGridY.Num downto 0 do begin
   y:=myGridY.Start+i*myGridY.Step;
   if (y>=myWorld.a.y) and (y<=myWorld.b.y) then begin
    k:=WorldToLocalY(y);
    if Flags and dfGridLineY <> 0 then begin
     DrawPatternLine(PaintBox.Canvas,Point2I(PlotBox.a.x,k),Point2I(PlotBox.b.x-1,k),GridColor,AxisSettings.GridPattern);
    end;
    if Flags and dfGridTextY <> 0 then begin
     s:=NumToStr(y/myGridY.Scale);
     p.x:=PlotBox.a.x-PaintBox.Canvas.TextWidth(s)-AxisSettings.Inter.X-AxisSettings.Space.X;
     p.y:=k-abs(PaintBox.Font.Height) shr 1;
     if p.x<0 then p.x:=0;
     if mySmartDrawer.DrawText(PaintBox.Canvas,p,s,NumberColor,GroundColor,2,2)
     then DrawLine(PaintBox.Canvas,Point2I(PlotBox.A.X-AxisSettings.Stick.X,k),Point2I(PlotBox.A.X,k),GridColor);
    end;
   end;
  end;
  {
  Draw ROI
  }
  if (Flags and dfRoiL<>0) and not isNanOrInf(Roi.A) then begin
   if RectContainsPoint(World,Point2D(Roi.A.X,RectCenterY(World))) then begin
    k:=WorldToLocalX(Roi.A.X);
    DrawLine(PaintBox.Canvas,Point2I(k,PlotBox.A.Y),Point2I(k,PlotBox.B.Y),RoiLColor);
   end;
   if RectContainsPoint(World,Point2D(RectCenterX(World),Roi.A.Y)) then begin
    k:=WorldToLocalY(Roi.A.Y);
    DrawLine(PaintBox.Canvas,Point2I(PlotBox.A.X,k),Point2I(PlotBox.B.X,k),RoiLColor);
   end;
  end;
  if (Flags and dfRoiR<>0) and not isNanOrInf(Roi.B) then begin
   if RectContainsPoint(World,Point2D(Roi.B.X,RectCenterY(World))) then begin
    k:=WorldToLocalX(Roi.B.X);
    DrawLine(PaintBox.Canvas,Point2I(k,PlotBox.A.Y),Point2I(k,PlotBox.B.Y),RoiRColor);
   end;
   if RectContainsPoint(World,Point2D(RectCenterX(World),Roi.B.Y)) then begin
    k:=WorldToLocalY(Roi.B.Y);
    DrawLine(PaintBox.Canvas,Point2I(PlotBox.A.X,k),Point2I(PlotBox.B.X,k),RoiRColor);
   end;
  end;
  {
  Draw title and legend
  }
  if Flags and dfTitle <> 0 then begin
   k:=1;
   DrawAlignedStrings(k,Title,LegendColor,GroundColor);
  end;
  if Flags and dfLegend <> 0 then begin
   k:=PlotBox.B.Y+AxisSettings.Inter.X+AxisSettings.Space.x+abs(PaintBox.Font.Height)*2+1;
   DrawAlignedStrings(k,Legend,LegendColor,GroundColor);
  end;
  mySmartDrawer.StopDraw;
 except
  on E:Exception do BugReport(E,Self,'DrawAxis');
 end;
end;

function CheckCurveIsValidForDownSampling(crv:TCurve):Boolean;
var i,n,ne:Integer; PX,PY:PDoubleArray;
begin
 Result:=false;
 if Assigned(crv) then
 try
  crv.Lock;
  n:=crv.Count; ne:=0;
  if (n>=3) then begin
   px:=crv.PX; py:=crv.PY;
   for i:=0 to n-1 do begin
    if IsNan(PX[i]) or IsInf(PX[i])  then begin inc(ne); Break; end;  // X[i] is NaN or INF ?
    if IsNan(PY[i]) or IsInf(PY[i])  then begin inc(ne); Break; end;  // Y[i] is NaN or INF ?
    if (i>0) then if (PX[i]<PX[i-1]) then begin inc(ne); Break; end;  // X[i] is not sorted ?
   end;
   Result:=(ne=0);
  end;
 finally
  crv.UnLock;
 end;
end;

procedure TFormCurveWindow.DrawCurveXY(Curve:TCurve; Start,N:Integer; Color,Style:Cardinal);
var DownSamplingOn:Boolean; DownLeng,DownMeth,DownTail:Integer; Sampled,Working:TCurve;
begin
 if Assigned(Self) and Assigned(Curve) then
 try
  // Detect if DownSampling required.
  if (Start<0) then Start:=0; N:=Min(N,Curve.Count-Start);
  DownSamplingOn:=HasFlags(myDownSampling.Mode,dsm_Enable);
  DownMeth:=0; DownLeng:=0;
  if DownSamplingOn then begin
   DownMeth:=myDownSampling.Method;
   if (DownMeth in dsm_GOOD) then DownLeng:=DownSamplingLength;
   if (DownLeng<3) or (N<=DownLeng) then DownSamplingOn:=false;
  end;
  if HasFlags(myDownSampling.Mode,dsm_Tail)
  then DownTail:=Max(0,myDownSampling.Tail) else DownTail:=0;
  // Check Curve is good for DownSampling: sorted by X and not NaN or INF.
  if DownSamplingOn then DownSamplingOn:=CheckCurveIsValidForDownSampling(Curve);
  // Without DownSampling we just call primary drawing procedure - DrawCurveXYPrime.
  if not DownSamplingOn then begin DrawCurveXYPrime(Curve,Start,N,Color,Style); Exit; end;
  // DownSampling required.
  Sampled:=nil; Working:=nil;
  try
   Working:=NewCurveCopy(Curve,Start,Start+N-1);
   Sampled:=CurveDownSampling(Working,DownLeng,DownMeth,DownTail);
   DrawCurveXYPrime(Sampled,0,Sampled.Count,Color,Style);
   if Assigned(Sampled) and (Sampled<>Curve) and (Sampled<>Working) then Inc(myDownSampCount);
  finally
   if Assigned(Sampled) and (Sampled<>Curve) and (Sampled<>Working) then Kill(Sampled);
   if Assigned(Working) and (Working<>Curve) then Kill(Working);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawCurveXY');
 end;
end;

procedure TFormCurveWindow.DrawCurveXYPrime(Curve:TCurve; Start,N:Integer; Color,Style:Cardinal);
const
 MaxDrawEntry = 3;
 LongDrawing  = 200;
 TickCheckMask = 255;
var
 i,j:Integer; Ticks,FixTicks:QWORD;
 Previous,Current:TPoint2I;
 Limit:TRect2D;
 Vertex:array[1..4] of TPoint2D;
 p,p1,p2,p3:TPoint2D;
 PreviousVisible,CurrentNotVisible:Boolean;
 MarkerStyle,MarkerColor,LineStyle:Cardinal;
 function UserBreak:Boolean;
 begin
  UserBreak:=false;
  if GetTickCount64>FixTicks+LongDrawing then begin
  end;
 end;
begin
 if Assigned(Self) and Assigned(Curve) and (Curve.Count>0) and (N>0) and
    (myDrawEntry<=MaxDrawEntry)
 then
 try
  inc(myDrawEntry);
  try
   Inc(CurveWindowsProfiler.Curr.DrawCurve);
   if Start<0 then Start:=0;
   Ticks:=GetTickCount64;
   FixTicks:=Ticks;
   MarkerColor:=Color;
   MarkerStyle:=Style and $F;
   LineStyle:=(Style shr 4) and $F;
   Limit:=World;
   p1:=LocalToWorld(ScreenWorld.A);
   p2:=LocalToWorld(Point2I(ScreenWorld.A.X+AxisSettings.Inter.x,
                            ScreenWorld.A.Y+AxisSettings.Inter.y));
   p3:=p1;
   p.x:=abs(p1.x-p2.x);
   p.y:=abs(p1.y-p2.y);
   Limit.a.x:=Limit.a.x-P.X;
   Limit.a.y:=Limit.a.y-P.y;
   Limit.b.x:=Limit.b.x+P.X;
   Limit.b.y:=Limit.b.y+P.y;
   Vertex[1]:=Point2D(Limit.a.x,Limit.a.y);
   Vertex[2]:=Point2D(Limit.b.x,Limit.a.y);
   Vertex[3]:=Point2D(Limit.b.x,Limit.b.y);
   Vertex[4]:=Point2D(Limit.a.x,Limit.b.y);
   PreviousVisible:=false;
   for i:=0 to N-1 do begin
    if (i and TickCheckMask) = TickCheckMask then
    if Ticks<>GetTickCount64 then if UserBreak then break;
    j:=Start+i;
    if j>=Curve.Count then break; {out of range}
    p:=Curve[j];
    if isNanOrInf(p) then continue; {NAN or INF}
    Inc(CurveWindowsProfiler.Curr.DrawPoint);
    CurrentNotVisible:=(p.x<Limit.a.x) or (p.x>Limit.b.x) or (p.y<Limit.a.y) or (p.y>Limit.b.y);
    if CurrentNotVisible then begin
     if PreviousVisible then begin
      {case current point not visible but previous point visible}
      if (LineStyle<>0) and (myAxisFlags and afLineBreak1=0) then
      if SegmentsHasIntersection(Vertex[1],Vertex[2],p,p2,p1) or
         SegmentsHasIntersection(Vertex[3],Vertex[4],p,p2,p1) or
         SegmentsHasIntersection(Vertex[2],Vertex[3],p,p2,p1) or
         SegmentsHasIntersection(Vertex[1],Vertex[4],p,p2,p1)
      then begin
       Current:=WorldToLocal(p1);
       Previous:=WorldToLocal(p2);
       DrawLineMarker(PaintBox.Canvas,Previous,Current,MarkerColor,LineStyle);
      end;
     end else begin
      {case current and previous point not visible; commented to draw more fast}
      if (i>0) and (myAxisFlags and afLineBreak2=0) then
      if LineStyle<>0 then
      if SegmentsHasIntersection(Vertex[1],Vertex[2],p,p2,p1) or
         SegmentsHasIntersection(Vertex[3],Vertex[4],p,p2,p1) or
         SegmentsHasIntersection(Vertex[2],Vertex[3],p,p2,p1) or
         SegmentsHasIntersection(Vertex[1],Vertex[4],p,p2,p1)
      then
      if SegmentsHasIntersection(Vertex[1],Vertex[4],p,p2,p3) or
         SegmentsHasIntersection(Vertex[2],Vertex[3],p,p2,p3) or
         SegmentsHasIntersection(Vertex[3],Vertex[4],p,p2,p3) or
         SegmentsHasIntersection(Vertex[1],Vertex[2],p,p2,p3)
      then begin
       Current:=WorldToLocal(p1);
       Previous:=WorldToLocal(p3);
       DrawLineMarker(PaintBox.Canvas,Previous,Current,MarkerColor,LineStyle);
      end;
     end;
     PreviousVisible:=false;
    end else begin
     Current:=WorldToLocal(p);
     if PreviousVisible then begin
      {case current point visible and previous point visible}
      if (Current.x<>Previous.x) or (Current.y<>Previous.y) then begin
       DrawPointMarker(PaintBox.Canvas,Current,MarkerColor,MarkerStyle);
       if LineStyle<>0 then DrawLineMarker(PaintBox.Canvas,Previous,Current,MarkerColor,LineStyle);
      end;
     end else begin
      {case current point visible but previous point not visible}
      DrawPointMarker(PaintBox.Canvas,Current,MarkerColor,MarkerStyle);
      if (i>0) and (LineStyle<>0) and (myAxisFlags and afLineBreak1=0) then begin
       if SegmentsHasIntersection(Vertex[1],Vertex[2],p,p2,p1) or
          SegmentsHasIntersection(Vertex[3],Vertex[4],p,p2,p1) or
          SegmentsHasIntersection(Vertex[2],Vertex[3],p,p2,p1) or
          SegmentsHasIntersection(Vertex[1],Vertex[4],p,p2,p1)
       then begin
        Previous:=WorldToLocal(p1);
        DrawLineMarker(PaintBox.Canvas,Previous,Current,MarkerColor,LineStyle);
       end;
      end;
     end;
     PreviousVisible:=true;
     Previous:=Current;
    end;
    p2:=p;
   end;
  finally
   dec(myDrawEntry);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawCurveXYPrime');
 end;
end;

procedure FindVisibleCurveIndexRange(aCurve:TCurve; const aWorld:TRect2D; out iStart,iStop:LongInt);
 function IsMonotonous(var x:array of double; N:LongInt):Boolean;
 var i:LongInt;
 begin
  Result:=true;
  for i:=0 to N-2 do if x[i]>x[i+1] then begin
   Result:=false;
   break;
  end;
 end;
begin
 if aCurve.Ok then with aCurve do
 try
  Lock;
  if (Count>1) and IsMonotonous(PX[0],Count) then begin
   iStart:=max(0,       FindIndex(Count, PX[0], aWorld.A.X) - 1);
   iStop :=min(Count-1, FindIndex(Count, PX[0], aWorld.B.X) + 1);
  end else begin
   iStart:=0;
   iStop:=Count-1;
  end;
 finally
  Unlock;
 end else begin
  iStart:=0;
  iStop:=-1;
 end;
end;

procedure ExtractVisibleCurvePart(SourceCurve,DestCurve:TCurve; const Rect:TRect2D);
var iStart,iStop:LongInt;
begin
 if SourceCurve.Ok and DestCurve.Ok then
 try
  SourceCurve.Lock;
  FindVisibleCurveIndexRange(SourceCurve,Rect,iStart,iStop);
  if iStart<=iStop
  then DestCurve.AssignData(SourceCurve.PX[iStart], SourceCurve.PY[iStart], iStop-iStart+1)
  else DestCurve.Count:=0;
 finally
  SourceCurve.Unlock;
 end;
end;

procedure TFormCurveWindow.DrawCurve(aCurve:TCurve);
var Lim:TRect2D; iStart,iStop:Integer;
begin
 if Assigned(Self) and Assigned(aCurve) then
 try
  if myAxisFlags and afCheck <> 0 then begin
   Lim:=aCurve.Limits;
   SomeGrowWorld(Lim,myAxisFlags);
  end;
  if aCurve.Color<>BoxColor then begin
   FindVisibleCurveIndexRange(aCurve,World,iStart,iStop);
   if iStart<=iStop then
   DrawCurveXY(aCurve,iStart,iStop-iStart+1,aCurve.Color,aCurve.Style);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawCurve');
 end;
end;

procedure TFormCurveWindow.DrawCurvePart(Curve:TCurve; StartIndex,StopIndex:LongInt);
begin
 if Assigned(Self) and Assigned(Curve) then
 try
  if StartIndex<0 then StartIndex:=0;
  if StopIndex>Curve.Count-1 then StopIndex:=Curve.Count-1;
  if (StartIndex<=StopIndex) and (Curve.Color<>BoxColor)
  then DrawCurveXY(Curve,StartIndex,StopIndex-StartIndex+1,Curve.Color,Curve.Style);
 except
  on E:Exception do BugReport(E,Self,'DrawCurvePart');
 end;
end;

procedure TFormCurveWindow.DrawCurves;
var i:Integer; Temp:TCurve; SaveWorld:TRect2D;
 procedure PlotCurve(Curve:TCurve);
 begin
  ExtractVisibleCurvePart(Curve,Temp,World);
  if Temp.Count>0 then begin
   Temp.Color:=Curve.Color;
   Temp.Style:=Curve.Style;
   DrawCurve(Temp);
  end;
  Curve.SmartAddMarker:=Point2D(_Nan,_Nan);
 end;
begin
 if Curves.Count>0 then
 try
  Temp:=NewCurve;
  try
   if DefCurve.Ok then PlotCurve(DefCurve) else begin
    for i:=0 to Curves.Count-1 do begin
     SaveWorld:=myWorld;
     PlotCurve(Curves[i]);
     if not RectIsEqual(SaveWorld,myWorld) then break;
    end;
   end;
  finally
   Kill(Temp);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawCurves');
 end;
end;

procedure TFormCurveWindow.DrawAlignedStr(y:integer; s:LongString; Color,Back:TColor);
var P:TPoint2I;
begin
 if pos('^L',s)=1 then begin
  delete(s,1,2);
  p:=Point2I(0,y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 if pos(^L,s)=1 then begin
  delete(s,1,1);
  p:=Point2I(0,y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 if pos('^R',s)=1 then begin
  delete(s,1,2);
  p:=Point2I(PaintBox.Width-PaintBox.Canvas.TextWidth(s),y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 if pos(^R,s)=1 then begin
  delete(s,1,1);
  p:=Point2I(PaintBox.Width-PaintBox.Canvas.TextWidth(s),y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 if pos('^C',s)=1 then begin
  delete(s,1,2);
  p:=Point2I((PaintBox.Width-PaintBox.Canvas.TextWidth(s)) shr 1,y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 if pos(^C,s)=1 then begin
  delete(s,1,1);
  p:=Point2I((PaintBox.Width-PaintBox.Canvas.TextWidth(s)) shr 1,y);
  DrawText(PaintBox.Canvas,p,s,Color,Back)
 end else
 begin
  p:=Point2I(0,y);
  DrawText(PaintBox.Canvas,p,s,Color,Back);
 end;
end;

procedure TFormCurveWindow.DrawAlignedStrings(y:integer;s:LongString; Color,Back:TColor);
var ls:LongString; p:Integer;
begin
 while (s<>'') do begin
  p:=PosEol(s);
  if (p>0) then begin
   ls:=system.copy(s,1,p-1);
   s:=system.copy(s,PosEol(s,p,1),MaxInt);
  end else begin
   ls:=s;
   s:='';
  end;
  if (ls<>'') then DrawAlignedStr(y,ls,Color,Back);
  inc(y,abs(PaintBox.Canvas.Font.Height)+1);
 end;
end;

function   TFormCurveWindow.GetAutoRange:TRect2D;
var i:Integer;
begin
 Result:=World;
 if HasSelection
 then Result:=DefCurve.Limits
 else for i:=0 to Curves.Count-1 do
      if i=0 then Result:=Curves[0].Limits
             else Result:=RectUnion(Result,Curves[i].Limits);
end;

procedure TFormCurveWindow.AutoRange;
begin
 SomeGrowWorld(GetAutoRange,afCheck+afUpdate);
end;

function TFormCurveWindow.CheckIsDataProtected(Msg:Boolean=true):Boolean;
begin
 Result:=IsDataProtected;
 if Msg then Trouble(Result, RusEng(
  'Операция запрещена для защищенного окна!'+EOL+'Скопируйте окно и делайте с ним что хотите.',
  'Could not complete this operation for protected window!'+EOL+'Copy window and make operation in new window.'));
end;

function TFormCurveWindow.Clone:TFormCurveWindow;
var
 i:Integer;
begin
 Result:=nil;
 if Assigned(Self) then
 try
  Application.CreateForm(TFormCurveWindow,Result);
  Result.LockDraw;
  try
   Result.Caption:=RusEng('КОПИЯ:','COPY:')+Caption;
   Result.Width:=Width;
   Result.Height:=Height;
   for i:=0 to Curves.Count-1 do Result.AddCurve(Curves[i].Clone);
   Result.Title:=Title;
   Result.Legend:=Legend;
   Result.Comment.Text:=Comment.Text;
   Result.GroundColor:=GroundColor;
   Result.LegendColor:=LegendColor;
   Result.NumberColor:=NumberColor;
   Result.BoxColor:=BoxColor;
   Result.RectColor:=RectColor;
   Result.GridColor:=GridColor;
   Result.RoiLColor:=RoiLColor;
   Result.RoiRColor:=RoiRColor;
   Result.MouseMode:=MouseMode;
   Result.DefCurveNum:=DefCurveNum;
   Result.World:=World;
   Result.Roi:=Roi;
   Result.TimeBase:=TimeBase;
   Result.TimeUnits:=TimeUnits;
   Result.WantTimeAxisX:=WantTimeAxisX;
  finally
   Result.UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'Clone');
 end;
end;

type
 TMonitoringRec = packed record
  TheCommand : (idMonitorEvent,idMonitoring,idMonitoringDone);
  EventCount : Cardinal;
 end;

procedure DoMonitoring(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if aObject is TFormCurveWindow then
 with aObject as TFormCurveWindow do
 with TMonitoringRec(CustomData^) do begin
  if IsFormExposed then
  case TheCommand of
   idMonitorEvent   : inc(EventCount,ord(MonitorEvent));
   idMonitoring     : Monitoring;
   idMonitoringDone : MonitoringDone;
  end;
  if (TheCommand=idMonitorEvent) then
  Inc(CurveWindowsProfiler.Curr.MonitorCall);
 end;
end;

procedure CurveWindowsMonitoring;
var R:TMonitoringRec;
begin
 R.EventCount:=0;
 R.TheCommand:=idMonitorEvent;
 CurveWindowsMonitor.ForEach(DoMonitoring,@R);
 if R.EventCount>0 then begin
  R.TheCommand:=idMonitoring;
  CurveWindowsMonitor.ForEach(DoMonitoring,@R);
  R.TheCommand:=idMonitoringDone;
  CurveWindowsMonitor.ForEach(DoMonitoring,@R);
 end;
end;

procedure TFormCurveWindow.StartMonitoring;
begin
 if Ok and CurveWindowsMonitor.Ok then begin
  if CurveWindowsMonitor.IndexOf(Self) < 0 then begin
   CurveWindowsMonitor.Add(Self);
   Tick55Actions.Add(CurveWindowsMonitoring);
  end;
 end;
end;

procedure TFormCurveWindow.StopMonitoring;
begin
 if Ok and CurveWindowsMonitor.Ok then begin
  if CurveWindowsMonitor.IndexOf(Self) >= 0 then begin
   CurveWindowsMonitor.Remove(Self);
   if CurveWindowsMonitor.Count=0 then
   Tick55Actions.Remove(CurveWindowsMonitoring);
  end;
 end;
end;

procedure TFormCurveWindow.Monitoring;
var Temp:TCurve; i,First,Last:LongInt; WorldChanged:Boolean;
 procedure PlotCurve(Curve:TCurve);
 var Rect,NewWorld:TRect2D;
 begin
  Rect.A:=Curve.SmartAddMarker;
  if not isNan(Rect.A) then begin
   Rect.B:=World.B;
   if (Curve.Count>0) and (Curve.LastPoint.X>Rect.B.X) then begin
    WorldChanged:=true;
    NewWorld:=World;
    RectMove(NewWorld, Curve.LastPoint.X - World.B.X + AlphaStayFree * RectSizeX(World), 0);
    World:=NewWorld;
   end else begin
    ExtractVisibleCurvePart(Curve,Temp,Rect);
    if Temp.Count>0 then begin
     Temp.Color:=Curve.Color;
     Temp.Style:=Curve.Style;
     DrawCurve(Temp);
    end;
    Inc(CurveWindowsProfiler.Curr.MonitorCurve);
    Inc(CurveWindowsProfiler.Curr.MonitorPoint,Temp.Count);
   end;
  end;
 end;
begin
 if Ok and (Curves.Count>0) then
 try
  Inc(CurveWindowsProfiler.Curr.MonitorDraw);
  if HasSelection then begin
   First:=DefCurveNum;
   Last:=First;
  end else begin
   First:=0;
   Last:=Curves.Count-1;
  end;
  WorldChanged:=false;
  Temp:=NewCurve;
  try
   for i:=First to Last do begin
    PlotCurve(Curves[i]);
    if WorldChanged then break;
   end;
  finally
   Kill(Temp);
  end;
 except
  on E:Exception do BugReport(E,Self,'Monitoring');
 end;
end;

procedure TFormCurveWindow.MonitoringDone;
var i:LongInt;
begin
 if Ok then
 for i:=0 to Curves.Count-1 do Curves[i].SmartAddMarker:=Point2D(_Nan,_Nan);
end;

procedure TFormCurveWindow.RoiWasSelected(Print:Boolean);
begin
 if Ok then begin
  if Print then begin
   Echo(StdDateTimePrompt+'ROI selection:');
   Echo(Format(' ROI_X1=%-18.12g  ROI_Y1=%-18.12g',[Roi.A.X,Roi.A.Y]));
   Echo(Format(' ROI_X2=%-18.12g  ROI_Y2=%-18.12g',[Roi.B.X,Roi.B.Y]));
   Echo(Format(' SIZE_X=%-18.12g  SIZE_Y=%-18.12g',[RectSizeX(Roi),RectSizeY(Roi)]));
  end;
  SystemCalculator.SetConst('roi_x1',Roi.A.X);
  SystemCalculator.SetConst('roi_y1',Roi.A.Y);
  SystemCalculator.SetConst('roi_x2',Roi.B.X);
  SystemCalculator.SetConst('roi_y2',Roi.B.Y);
 end;
end;

function TFormCurveWindow.PaintBoxPosParams:LongString;
begin
 if Ok then Result:=ControlPosParams(PaintBox) else Result:='';
end;

procedure TFormCurveWindow.FormCreate(Sender: TObject);
begin
 inherited;
 SetStandardFont(Self);
 PaintBox.Font.Assign(Self.Font);
 SetAllButtonsCursor(Self,crHandPoint);
 AutoScroll:=false;
 myDrawEntry:=0;
 try
  LockDraw;
  myWorld:=Rect2D(0,0,1,1);
  myAxisFlags:=DefaultAxisFlags;
  GroundColor:=DefGroundColor;
  LegendColor:=DefLegendColor;
  NumberColor:=DefNumberColor;
  BoxColor:=DefBoxColor;
  RectColor:=DefRectColor;
  GridColor:=DefGridColor;
  RoiLColor:=DefRoiLColor;
  RoiRColor:=DefRoiRColor;
  Title:='';
  Legend:='';
  MouseMode:=mmSelector;
  myMousePos:=Point2D(_Nan,_Nan);
  Roi:=Rect2D(_Nan,_Nan,_Nan,_Nan);
  TimeBase:=_Nan;
  TimeUnits:=_Nan;
  myCurves:=NewCurveList;           myCurves.Master:=@myCurves;
  myComment:=NewText;               myComment.Master:=@myComment;
  myDefCurveNum:=-1;
  myXorSelector:=NewXorSelector;    myXorSelector.Master:=@myXorSelector;
  myXorRoi:=NewXorSelector;         myXorRoi.Master:=@myXorRoi;
  myXorRoiL:=NewXorSelector;        myXorRoiL.Master:=@myXorRoiL;  myXorRoiL.Mode:=sm_BigCross; myXorRoiL.Width:=3;
  myXorRoiR:=NewXorSelector;        myXorRoiR.Master:=@myXorRoiR;  myXorRoiR.Mode:=sm_BigCross; myXorRoiR.Width:=3;
  mySmartDrawer:=NewSmartDrawer;    mySmartDrawer.Master:=@mySmartDrawer;
  myDownSampling:=DefDownSamplingParams;
  myDownSampCount:=0;
  ResetWorld;
  UpdateScroller;
 finally
  UnlockDraw;
 end;
 UpdateMenu(MenuCurveMouse,
            RusEng('Мышь','Mouse'),
            RusEng('Меню режима мыши','Mouse mode menu'),
            0);
 UpdateMenu(MenuCurveMouseSelector,
            RusEng('Выбор','Selector'),
            RusEng('Мышь:выбор пределов графика','Mouse: range selector'),
            0);
 UpdateMenu(MenuCurveMouseZoomIn,
            RusEng('Увеличение','Increase'),
            RusEng('Мышь:увеличение масштаба','Mouse: range increase'),
            0);
 UpdateMenu(MenuCurveMouseZoomOut,
            RusEng('Уменьшение','Decrease'),
            RusEng('Мышь:уменьшение масштаба','Mouse: range decrease'),
            0);
 UpdateMenu(MenuCurveMouseDragCopy,
            RusEng('Копия','Drag copy'),
            RusEng('Мышь:копирование данных','Mouse: Drag & copy data'),
            0);
 UpdateMenu(MenuCurveMouseDragMove,
            RusEng('Перенос','Drag move'),
            RusEng('Мышь:перемещение данных','Mouse: Drag & move data'),
            0);
 UpdateMenu(MenuCurveMouseSelectRoi,
            RusEng('Выбрать РОИ','Select ROI '),
            RusEng('Мышь:выбрать маркеры РОИ','Mouse: select ROI markers'),
            0);
 UpdateMenu(MenuCurveMouseSelectRoiL,
            RusEng('Выбрать левый  РОИ','Select ROI Left '),
            RusEng('Мышь:выбрать левый  маркер РОИ','Mouse: select left  ROI marker'),
            0);
 UpdateMenu(MenuCurveMouseSelectRoiR,
            RusEng('Выбрать правый РОИ','Select ROI Right'),
            RusEng('Мышь:выбрать правый маркер РОИ','Mouse: select right ROI marker'),
            0);
 UpdateMenu(MenuCurveMouseClearRoiL,
            RusEng('Убрать левый  РОИ','Clear ROI Left '),
            RusEng('Очистить левый  маркер РОИ','Clear left  ROI marker'),
            0);
 UpdateMenu(MenuCurveMouseClearRoiR,
            RusEng('Убрать правый РОИ','Clear ROI Right'),
            RusEng('Очистить правый маркер РОИ','Clear right ROI marker'),
            0);
 UpdateMenu(MenuCurveRange,
            RusEng('Пределы','Range'),
            RusEng('Задание пределов графика','Grapchics range selection'),
            0);
 UpdateMenu(MenuCurveRangeAuto,
            RusEng('Автомасштаб','Auto range'),
            RusEng('Автоматически масштабировать','Auto range to see all'),
            VK_F8);
 UpdateMenu(MenuCurveRangeSelector,
            RusEng('Выбор','Range select'),
            RusEng('Выбрать пределы X,Y графика','Select range dialog'),
            ShortCut(VK_F8, [ssCtrl]));
 UpdateMenu(MenuCurveRangeZoomIn,
            RusEng('Увеличить','Increase'),
            RusEng('Увеличить масштаб','Range increase'),
            ShortCut(VK_F8, [ssAlt]));
 UpdateMenu(MenuCurveRangeZoomOut,
            RusEng('Уменьшить','Decrease'),
            RusEng('Уменьшить масштаб','Range decrease'),
            ShortCut(VK_F8, [ssShift]));
 UpdateMenu(NameCurveRangeLeft,
            RusEng('Влево','Left'),
            RusEng('Сдвинуть график влево','Move range left'),
            ShortCut(VK_LEFT, [ssAlt]));
 UpdateMenu(MenuCurveRangeRight,
            RusEng('Вправо','Right'),
            RusEng('Сдвинуть график вправо','Move range right'),
            ShortCut(VK_RIGHT, [ssAlt]));
 UpdateMenu(MenuCurveRangeUp,
            RusEng('Вверх','Up'),
            RusEng('Сдвинуть график вверх','Move range up'),
            ShortCut(VK_UP, [ssAlt]));
 UpdateMenu(MenuCurveRangeDown,
            RusEng('Вниз','Down'),
            RusEng('Сдвинуть график вниз','Move range down'),
            ShortCut(VK_DOWN, [ssAlt]));
 UpdateMenu(MenuCurveRangeTimeAxisX,
            RusEng('Шкала времени по X','Time Scale along X'),
            RusEng('Разрешить шкалу времени по оси X','Enable Time Scale along X axis'),
            0);
 UpdateMenu(MenuCurveEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню выбора/редактирования','Edit and selection menu'),
            0);
 UpdateMenu(MenuCurveEditSelect,
            RusEng('Выбор кривой','Select curve'),
            RusEng('Меню выбора кривой в окне','Select curve menu'),
            0);
 UpdateMenu(MenuCurveEditSelect0,
            RusEng('Отмена выбора','No selection'),
            RusEng('Нет выбранной кривой в окне','No cirve selected in window'),
            ShortCut(Word('0'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect1,
            RusEng('Кривая 1','Curve 1'),
            RusEng('Выбрать кривую 1 в окне','Select curve 1 in window'),
            ShortCut(Word('1'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect2,
            RusEng('Кривая 2','Curve 2'),
            RusEng('Выбрать кривую 2 в окне','Select curve 2 in window'),
            ShortCut(Word('2'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect3,
            RusEng('Кривая 3','Curve 3'),
            RusEng('Выбрать кривую 3 в окне','Select curve 3 in window'),
            ShortCut(Word('3'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect4,
            RusEng('Кривая 4','Curve 4'),
            RusEng('Выбрать кривую 4 в окне','Select curve 4 in window'),
            ShortCut(Word('4'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect5,
            RusEng('Кривая 5','Curve 5'),
            RusEng('Выбрать кривую 5 в окне','Select curve 5 in window'),
            ShortCut(Word('5'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect6,
            RusEng('Кривая 6','Curve 6'),
            RusEng('Выбрать кривую 6 в окне','Select curve 6 in window'),
            ShortCut(Word('6'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect7,
            RusEng('Кривая 7','Curve 7'),
            RusEng('Выбрать кривую 7 в окне','Select curve 7 in window'),
            ShortCut(Word('7'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect8,
            RusEng('Кривая 8','Curve 8'),
            RusEng('Выбрать кривую 8 в окне','Select curve 8 in window'),
            ShortCut(Word('8'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelect9,
            RusEng('Кривая 9','Curve 9'),
            RusEng('Выбрать кривую 9 в окне','Select curve 9 in window'),
            ShortCut(Word('9'),[ssCtrl]));
 UpdateMenu(MenuCurveEditSelectMore,
            RusEng('Кривая № …','Curve № …'),
            RusEng('Диалог выбора кривой в окне','Curve selection dialog...'),
            ShortCut(Word('F'),[ssCtrl]));
 UpdateMenu(MenuCurveEditDelete,
            RusEng('Удалить','Delete'),
            RusEng('Удалить кривую из окна','Delete curve from window'),
            ShortCut(VK_DELETE,[ssCtrl]));
 UpdateMenu(MenuCurveEditCut,
            RusEng('Вырезать','Cut'),
            RusEng('Вырезать кривую в буфер','Cut curve to clipboard'),
            ShortCut(Word('X'),[ssCtrl]));
 UpdateMenu(MenuCurveEditCopy,
            RusEng('Копировать','Copy'),
            RusEng('Копировать кривую в буфер','Copy curve to clipboard'),
            ShortCut(Word('C'),[ssCtrl]));
 UpdateMenu(MenuCurveEditPaste,
            RusEng('Вставить','Paste'),
            RusEng('Вставить кривую из буфеера','Paste curve from clipboard'),
            ShortCut(Word('V'),[ssCtrl]));
 UpdateMenu(MenuCurveEditCloneWindow,
            RusEng('Клонировать','Clone'),
            RusEng('Создать точную копию окна','Create exact copy of window'),
            ShortCut(Word('D'),[ssCtrl]));
 UpdateMenu(MenuCurveEditMore,
            RusEng('Править еще...','Edit more...'),
            RusEng('Дополнительное меню правки','Edit comment,color,style...'),
            0);
 UpdateMenu(MenuCurveEditComment,
            RusEng('Легенда (паспорт)','Legend (comment'),
            RusEng('Редактировать легенду (паспорт) кривой','Edit curve legend (comment)'),
            ShortCut(Word('L'),[ssCtrl]));
 UpdateMenu(MenuCurveEditStyle,
            RusEng('Стиль кривой','Curve style'),
            RusEng('Имя,цвет,стиль кривой','Edit curve name,color&style'),
            ShortCut(Word('Y'),[ssCtrl]));
 UpdateMenu(MenuCurveEditWindowStyle,
            RusEng('Стиль окна','Window style'),
            RusEng('Имя,цвет,стиль окна','Edit window name,color&style'),
            ShortCut(Word('W'),[ssCtrl]));
 UpdateMenu(MenuCurveEditDownSampling,
            RusEng('Дискретность','DownSampling'),
            RusEng('DownSampling - понижение дискретности для быстрого рисования','DownSampling parameters for fast draw'),
            ShortCut(VK_DOWN,[ssCtrl,ssAlt]));
 UpdateMenu(MenuCurve,
            RusEng('Кривая','Curve')+MenuRightSpace,
            RusEng('Меню операций с кривыми','Curve operation menu'),
            0);
 UpdateMenu(MenuCurveTools,
            RusEng('Все утилиты …','Curve tools …'),
            RusEng('Вызов панели управления для окна кривых','Show curve tools window'),
            0);
 UpdateMenu(MenuCurveToolsRunMacroDialog,
            RusEng('Макрос анализа данных','Macros for data analysis'),
            RusEng('Запустить макрос анализа данных','Run macros for data analysis'),
            ShortCut(Word('M'),[ssCtrl]));
 UpdateMenu(MenuCurveToolsRunPluginDialog,
            RusEng('Утилита анализа данных','Plugin  for data analysis'),
            RusEng('Запустить утилиту анализа данных','Run plugin for data analysis'),
            ShortCut(Word('U'),[ssCtrl]));
 UpdateMenu(MenuCurveToolsWriteTable,
            RusEng('Таблица кривых','Curve table'),
            RusEng('Сформировать таблицу кривых','Create curve table as text'),
            ShortCut(Word('T'),[ssCtrl]));
 UpdateMenu(MenuCurveToolsComposeSurface,
            RusEng('Построить поверхность','Compose surface'),
            RusEng('Сформировать из кривых поверхность','Create surface from curves'),
            ShortCut(Word('S'),[ssCtrl]));
 PopupMenuFile.Caption:=RusEng('Файл …','File …');
 PopupMenuCurveEdit.Caption:=RusEng('Правка …','Edit …');
 PopupMenuCurveMouse.Caption:=RusEng('Мышка …','Mouse …');
 PopupMenuCurveRange.Caption:=RusEng('Масштаб …','Range …');
 PopupMenuCurveEditSelect.Caption:=RusEng('Выбор …','Select …');
 PopupMenuCurveTool.Caption:=RusEng('Утилиты …','Utility …');
end;

procedure TFormCurveWindow.FormDestroy(Sender: TObject);
begin
 myTitle:='';
 myLegend:='';
 Kill(myCurves);
 Kill(myComment);
 Kill(myXorSelector);
 Kill(myXorRoi);
 Kill(myXorRoiL);
 Kill(myXorRoiR);
 Kill(mySmartDrawer);
 inherited;
end;

procedure TFormCurveWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullCurveWindowList.Add(Self);
 AddonSdiFlags(sf_SdiCurveWin);
end;

procedure TFormCurveWindow.BeforeDestruction;
begin
 StopMonitoring;
 FullCurveWindowList.Remove(Self);
 if IsClipBoard then CurveClipboard:=nil;
 inherited BeforeDestruction;
end;

procedure TFormCurveWindow.FormResize(Sender: TObject);
begin
 inherited;
 ResetWorld;
end;

procedure TFormCurveWindow.ScrollBarChange(Sender: TObject);
begin
 DefCurveNum:=ScrollBar.Position-1;
end;

procedure TFormCurveWindow.PaintBoxPaint(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormCurveWindow.PaintBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
 procedure AdjustLocation(var A:TPoint2D; const B:TPoint2I; Curve:TCurve);
 var i:LongInt; P,Q:TPoint2D; Dist,DMin:Double;
 begin
  if Curve.Ok then begin
   DMin:=MaxInt;
   for i:=0 to Curve.Count-1 do begin
    Q:=Curve[i];
    if RectContainsPoint(World,Q) then begin
     Dist:=Hypot(WorldToLocal(Q).X-B.X,WorldToLocal(Q).Y-B.Y);
     if Dist<DMin then begin
      DMin:=Dist;
      P:=Q;
     end;
    end;
   end;
   if DMin <= 12 then A:=P;
  end;
 end;
var P,S:TPoint2D;
begin
 try
  if ssLeft in Shift then
  case myMouseMode of
   mmSelector   : myXorSelector.Start(PaintBox.Canvas,Point2I(x,y));
   mmZoomIn     : begin
                   P:=LocalToWorld(Point2I(X,Y));
                   S:=Point2D(0.5*RectSizeX(World)/ZoomFactor,0.5*RectSizeY(World)/ZoomFactor);
                   World:=Rect2D(P.X-S.X,P.Y-S.Y,P.X+S.X,P.Y+S.Y);
                  end;
   mmZoomOut    : begin
                   P:=LocalToWorld(Point2I(x,y));
                   S:=Point2D(0.5*RectSizeX(World)*ZoomFactor,0.5*RectSizeY(World)*ZoomFactor);
                   World:=Rect2D(P.X-S.X,P.Y-S.Y,P.X+S.X,P.Y+S.Y);
                  end;
   mmDragCopy   : ;
   mmDragMove   : ;
   mmSelectRoi  : myXorRoi.Start(PaintBox.Canvas,Point2I(x,y));
   mmSelectRoiL : if myXorRoiL.Stop(PaintBox.Canvas) then begin
                   myRoi.A:=LocalToWorld(myXorRoiL.Selection.B);
                   if HasSelection and (ssShift in Shift)
                   then AdjustLocation(myRoi.A, myXorRoiL.Selection.B, DefCurve);
                  end;
   mmSelectRoiR : if myXorRoiR.Stop(PaintBox.Canvas) then begin
                   myRoi.B:=LocalToWorld(myXorRoiR.Selection.B);
                   if HasSelection and (ssShift in Shift)
                   then AdjustLocation(myRoi.B, myXorRoiR.Selection.B, DefCurve);
                  end;
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseDown');
 end;
end;

procedure TFormCurveWindow.PaintBoxMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
 try
  case myMouseMode of
   mmSelector   : myXorSelector.Replace(PaintBox.Canvas,Point2I(X,Y));
   mmZoomIn     : ;
   mmZoomOut    : ;
   mmDragCopy   : ;
   mmDragMove   : ;
   mmSelectRoi  : myXorRoi.Replace(PaintBox.Canvas,Point2I(X,Y));
   mmSelectRoiL : myXorRoiL.Replace(PaintBox.Canvas,Point2I(X,Y));
   mmSelectRoiR : myXorRoiR.Replace(PaintBox.Canvas,Point2I(X,Y));
  end;
  if RectContainsPoint(ScreenWorld,Point2I(X,Y))
  then myMousePos:=LocalToWorld(Point2I(X,Y))
  else myMousePos:=Point2D(_Nan,_Nan);
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseMove');
 end;
end;

procedure TFormCurveWindow.PaintBoxMouseUp(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
begin
 try
  case myMouseMode of
   mmSelector   : if myXorSelector.Stop(PaintBox.Canvas) then begin
                   if min(abs(RectSizeX(myXorSelector.Selection)),abs(RectSizeY(myXorSelector.Selection)))>2
                   then World:=RectValidate(LocalToWorldRect(myXorSelector.Selection));
                  end;
   mmZoomIn     : ;
   mmZoomOut    : ;
   mmDragCopy   : ;
   mmDragMove   : ;
   mmSelectRoi  : if myXorRoi.Stop(PaintBox.Canvas) then begin
                   if min(abs(RectSizeX(myXorRoi.Selection)),abs(RectSizeY(myXorRoi.Selection)))>2 then begin
                    Roi:=LocalToWorldRect(myXorRoi.Selection);
                    PaintBox.Repaint;
                    RoiWasSelected;
                   end;
                   MouseMode:=mmSelector;
                  end;
   mmSelectRoiL : begin
                   MouseMode:=mmSelector;
                   PaintBox.Repaint;
                   RoiWasSelected;
                  end;
   mmSelectRoiR : begin
                   MouseMode:=mmSelector;
                   PaintBox.Repaint;
                   RoiWasSelected;
                  end;
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseUp');
 end;
end;

procedure TFormCurveWindow.PaintBoxStartDrag(Sender: TObject;  var DragObject: TDragObject);
begin
 { Start drag }
end;

procedure TFormCurveWindow.PaintBoxEndDrag(Sender, Target: TObject; X,Y: Integer);
begin
 MouseMode:=mmSelector;
end;

function OwnerIsCurveWindow(AObject:TObject):Boolean;
begin
 Result:=(AObject is TPaintBox) and ((AObject as TPaintBox).Owner is TFormCurveWindow);
end;

procedure TFormCurveWindow.PaintBoxDragOver(Sender, Source: TObject; X,Y: Integer; State: TDragState; var Accept: Boolean);
begin
 Accept:=(Sender<>Source) and
         OwnerIsCurveWindow(Source) and
         OwnerIsCurveWindow(Sender) and
         (((Source as TPaintBox).Owner as TFormCurveWindow).Curves.Count>0);
end;

procedure TFormCurveWindow.PaintBoxDragDrop(Sender, Source: TObject; X,Y: Integer);
var i:Integer; Src,Dst:TFormCurveWindow; aCurve:TCurve; SaveOwns:Boolean;
begin
 if (Sender<>Source) and OwnerIsCurveWindow(Sender) and OwnerIsCurveWindow(Source) then
 try
  Src:=(Source as TPaintBox).Owner as TFormCurveWindow;
  Dst:=(Sender as TPaintBox).Owner as TFormCurveWindow;
  if not Dst.CheckIsDataProtected then
  case Src.MouseMode of
   mmDragCopy: if Src.HasSelection then begin
                Dst.AddCurve(NewCurveCopy(Src.DefCurve));
               end else begin
                for i:=0 to Src.Curves.Count-1 do
                Dst.AddCurve(NewCurveCopy(Src.Curves[i]));
               end;
   mmDragMove: if not Src.CheckIsDataProtected then 
               if Src.HasSelection then begin
                SaveOwns:=Src.Curves.OwnsObjects;
                Src.Curves.OwnsObjects:=false;
                aCurve:=Src.DefCurve;
                Src.DeleteCurve(aCurve);
                Dst.AddCurve(aCurve);
                Src.Curves.OwnsObjects:=SaveOwns;
               end else begin
                SaveOwns:=Src.Curves.OwnsObjects;
                Src.Curves.OwnsObjects:=false;
                while Src.Curves.Count>0 do begin
                 aCurve:=Src.Curves[0];
                 Src.DeleteCurve(aCurve);
                 Dst.AddCurve(aCurve);
                end;
                Src.Curves.OwnsObjects:=SaveOwns;
               end;
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxDragDrop');
 end;
end;

 {
 **********************
 Actions implementation
 **********************
 }
procedure TFormCurveWindow.ActionCurveMouseSelectorExecute(Sender: TObject);
begin
 MouseMode:=mmSelector;
end;

procedure TFormCurveWindow.ActionCurveMouseZoomInExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseZoomIn)<0 then Exit;
 MouseMode:=mmZoomIn;
end;

procedure TFormCurveWindow.ActionCurveMouseZoomOutExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseZoomOut)<0 then Exit;
 MouseMode:=mmZoomOut;
end;

procedure TFormCurveWindow.ActionCurveMouseDragCopyExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseDragCopy)<0 then Exit;
 MouseMode:=mmDragCopy;
end;

procedure TFormCurveWindow.ActionCurveMouseDragMoveExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseDragMove)<0 then Exit;
 MouseMode:=mmDragMove;
end;

procedure TFormCurveWindow.ActionCurveMouseSelectRoiLExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseSelectRoiL)<0 then Exit;
 MouseMode:=mmSelectRoiL;
end;

procedure TFormCurveWindow.ActionCurveMouseSelectRoiRExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseSelectRoiR)<0 then Exit;
 MouseMode:=mmSelectRoiR;
end;

procedure TFormCurveWindow.ActionCurveMouseClearRoiLExecute(Sender: TObject);
var ShouldDraw:Boolean;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseClearRoiL)<0 then Exit;
 if Ok then begin
  ShouldDraw:=not isNanOrInf(Roi.A);
  Roi:=Rect2D(_Nan,_Nan,Roi.B.X,Roi.B.Y);
  if ShouldDraw then PaintBox.Repaint;
  RoiWasSelected(false);
 end;
end;

procedure TFormCurveWindow.ActionCurveMouseClearRoiRExecute(Sender: TObject);
var ShouldDraw:Boolean;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseClearRoiR)<0 then Exit;
 if Ok then begin
  ShouldDraw:=not isNanOrInf(Roi.B);
  Roi:=Rect2D(Roi.A.X,Roi.A.Y,_Nan,_Nan);
  if ShouldDraw then PaintBox.Repaint;
  RoiWasSelected(false);
 end;
end;

procedure TFormCurveWindow.ActionCurveMouseSelectRoiExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveMouseSelectRoi)<0 then Exit;
 MouseMode:=mmSelectRoi;
end;

procedure TFormCurveWindow.ActionCurveRangeAutoExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeAuto)<0 then Exit;
 AutoRange;
end;

procedure TFormCurveWindow.ActionCurveRangeSelectorExecute(Sender: TObject);
var Rw:TRect2D; Tb,Tu:Double;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeSelector)<0 then Exit;
 Rw:=World; Tb:=TimeBase; Tu:=TimeUnits;
 if (CurveRangeSelectorDialog(Rw,Tb,Tu,PaintBoxPosParams)=mrOk) then
 try
  LockDraw;
  try
   if not IsNanOrInf(Rw) and not RectIsEmpty(Rw) then begin
    World:=Rw; TimeBase:=Tb; TimeUnits:=Tu;
   end;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionCurveRangeSelectorExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveRangeZoomInExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeZoomIn)<0 then Exit;
 R.A:=RectCenter(World);
 R.B:=R.A;
 RectGrow(R,0.5*RectSizeX(World)/ZoomFactor,0.5*RectSizeY(World)/ZoomFactor);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeZoomOutExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeZoomOut)<0 then Exit;
 R.A:=RectCenter(World);
 R.B:=R.A;
 RectGrow(R,0.5*RectSizeX(World)*ZoomFactor,0.5*RectSizeY(World)*ZoomFactor);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeLeftExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeLeft)<0 then Exit;
 R:=World;
 RectMove(R,-0.25*RectSizeX(R)/ZoomFactor,0);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeRightExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeRight)<0 then Exit;
 R:=World;
 RectMove(R,+0.25*RectSizeX(R)/ZoomFactor,0);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeUpExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeUp)<0 then Exit;
 R:=World;
 RectMove(R,0,+0.25*RectSizeY(R)/ZoomFactor);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeDownExecute(Sender: TObject);
var R:TRect2D;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveRangeDown)<0 then Exit;
 R:=World;
 RectMove(R,0,-0.25*RectSizeY(R)/ZoomFactor);
 World:=R;
end;

procedure TFormCurveWindow.ActionCurveRangeTimeAxisXExecute(Sender: TObject);
begin
 inherited;
 if ActionCurveRangeTimeAxisX.Enabled then
 try
  LockDraw;
  try
   WantTimeAxisX:=not WantTimeAxisX;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionCurveRangeTimeAxisXExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveEditDeleteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditDelete)<0 then Exit;
 if not CheckIsDataProtected and HasSelection then DeleteCurve(DefCurve);
end;

procedure TFormCurveWindow.ActionCurveEditCutExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditCut)<0 then Exit;
 if not CheckIsDataProtected and HasSelection then begin
  ActionCurveEditCopyExecute(Sender);
  ActionCurveEditDeleteExecute(Sender);
 end;
end;

procedure TFormCurveWindow.ActionCurveEditCopyExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditCopy)<0 then Exit;
 if not Assigned(CurveClipboard) then begin
  CurveClipboard:=NewCurveWindow(RusEng('БУФЕР ОБМЕНА КРИВЫХ',
                                        'CURVE CLIPBOARD'),
                                 RusEng(^C'КРИВЫЕ ИЗ БУФЕРА ОБМЕНА'+ASCII_CR+'  Y',
                                        ^C'CLIPBOARD CURVES'+ASCII_CR+'  Y'),
                                 RusEng(^R'X  '+ASCII_CR+^C'Y(X)',
                                        ^R'X  '+ASCII_CR+^C'Y(X)'), wsMinimized);
  CurveClipboard.Master:=@CurveClipboard;
  CurveClipboard.CloseAction:=caMinimize;
 end;
 if HasSelection then CurveClipboard.AddCurve(NewCurveCopy(DefCurve));
end;

procedure TFormCurveWindow.ActionCurveEditPasteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditPaste)<0 then Exit;
 if not CheckIsDataProtected then
 if CurveClipboard.HasSelection then AddCurve(NewCurveCopy(CurveClipboard.DefCurve));
end;

procedure TFormCurveWindow.ActionCurveEditSelectMoreExecute(Sender: TObject);
const
 aWidth=3;
var
 aList:TText; aFocus:Integer; aCurve:TCurve; i:Integer; aParams:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelectMore)<0 then Exit;
 if Assigned(Self) and (Curves.Count>0) then
 try
  aList:=NewText;
  try
   aList.Addln(d2s(0,aWidth)+'  '+RusEng('ОТМЕНА ВЫБОРА КРИВОЙ (показать все кривые)',
                                         'SKIP CURVE SELECTION (show all curves)'));
   for i:=0 to Curves.Count-1 do begin
    aCurve:=Curves[i];
    if aCurve.Name<>''
    then aList.Addln(d2s(i+1,aWidth)+'  '+aCurve.Name)
    else aList.Addln(d2s(i+1,aWidth)+'  '+RusEng('БЕЗЫМЯННАЯ','NONAME'));
   end;
   aFocus:=DefCurveNum+1;
   aParams:=ControlPosParams(PaintBox);
   if ListBoxSelection(RusEng('Диалог выбора кривой в окне','Curve selection dialog'),
                       LeftPad('N#',aWidth)+'  '+RusEng('ИМЯ КРИВОЙ','CURVE NAME'),
                       aList.Text,aFocus,aParams)=mrOk
   then DefCurveNum:=aFocus-1;
  finally
   Kill(aList);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionCurveEditSelectMoreExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveEditSelect0Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect0)<0 then Exit;
 DefCurveNum:=-1;
end;

procedure TFormCurveWindow.ActionCurveEditSelect1Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect1)<0 then Exit;
 DefCurveNum:=0;
end;

procedure TFormCurveWindow.ActionCurveEditSelect2Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect2)<0 then Exit;
 DefCurveNum:=1;
end;

procedure TFormCurveWindow.ActionCurveEditSelect3Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect3)<0 then Exit;
 DefCurveNum:=2;
end;

procedure TFormCurveWindow.ActionCurveEditSelect4Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect4)<0 then Exit;
 DefCurveNum:=3;
end;

procedure TFormCurveWindow.ActionCurveEditSelect5Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect5)<0 then Exit;
 DefCurveNum:=4;
end;

procedure TFormCurveWindow.ActionCurveEditSelect6Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect6)<0 then Exit;
 DefCurveNum:=5;
end;

procedure TFormCurveWindow.ActionCurveEditSelect7Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect7)<0 then Exit;
 DefCurveNum:=6;
end;

procedure TFormCurveWindow.ActionCurveEditSelect8Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect8)<0 then Exit;
 DefCurveNum:=7;
end;

procedure TFormCurveWindow.ActionCurveEditSelect9Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditSelect9)<0 then Exit;
 DefCurveNum:=8;
end;

procedure TFormCurveWindow.ActionCurveEditCloneWindowExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditCloneWindow)<0 then Exit;
 Clone;
end;

procedure TFormCurveWindow.ActionCurveEditDownSamplingExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditDownSampling)<0 then Exit;
 try
  if Ok then RunCurveDownSamplingDialog(Self);
 except
  on E:Exception do BugReport(E,Self,'ActionCurveEditDownSamplingExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveEditCommentExecute(Sender: TObject);
var Cap,Tit:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditComment)<0 then Exit;
 try
  if HasSelection then begin
   Cap:=RusEng('Редактирование паспорта кривой','Edit curve comment');
   Tit:=RusEng('Отредактируйте паспорт кривой ','Please, edit comment of curve ')+
               d2s(DefCurveNum+1)+' "'+DefCurve.Name+'"';
  end else begin
   Cap:=RusEng('Редактирование паспорта окна','Edit window comment');
   Tit:=RusEng('Отредактируйте паспорт окна "','Please, edit comment of window "')+Caption+'"';
  end;
  TextEditDialog(Cap,Tit,DefComment,PaintBoxPosParams);
 except
  on E:Exception do BugReport(E,Self,'ActionCurveEditCommentExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveEditStyleExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditStyle)<0 then Exit;
 if Ok and not CheckIsDataProtected then RunCurveStyleDialog(Self,PaintBoxPosParams);
end;

procedure TFormCurveWindow.ActionCurveEditWindowStyleExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveEditWindowStyle)<0 then Exit;
 if Ok and not CheckIsDataProtected then RunCurveWindowStyleDialog(Self,PaintBoxPosParams)
end;

procedure TFormCurveWindow.ActionCurveToolsExecute(Sender: TObject);
var Where:TPoint2I;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveTools)<0 then Exit;
 Where:=Point2I(PaintBox.ClientToScreen(Point(0,0)));
 ShowCurveTools(Self,Where);
end;

procedure TFormCurveWindow.ActionCurveToolsRunMacroDialogExecute(Sender: TObject);
var SrcWin,DstWin:TFormCurveWindow;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveToolsRunMacroDialog)<0 then Exit;
 if Ok and not CheckIsDataProtected then
 try
  SrcWin:=Self;
  DstWin:=NewCurveWindow;
  if (CurveToolsRunMacroDialogExecute(SrcWin,DstWin,DefaultMacroName,PaintBoxPosParams)=mrOk) then begin
   if DstWin.Ok then begin
    SdiMan.ActivateChild(DstWin);
    if (DstWin.Curves.Count=0) then Kill(DstWin);
   end;
  end else begin
   Kill(DstWin);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionCurveToolsRunMacroDialogExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveToolsRunPluginDialogExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveToolsRunPluginDialog)<0 then Exit;
 inherited;
 if Ok and not CheckIsDataProtected
 then ExecuteFormCurveToolsRunPluginDialog(Self,PaintBoxPosParams);
end;

type
 TWriTab_Rec=record
  Dest  : Text;
  Delim : Char;
 end;

procedure WriTab(const aFormat:LongString; const aValue:Double; Custom:Pointer);
begin
 with TWriTab_Rec(Custom^) do
 write(Dest,Delim,ReplaceString(Format(aFormat,[aValue]),',','.'));
end;

procedure TFormCurveWindow.ActionCurveToolsWriteTableExecute(Sender: TObject);
var Editor:TFormTextEditor; Temp:LongString; WriTab_Rec:TWriTab_Rec;
var SaveIORes:Integer;
begin
 if Guard.CheckAction(ga_Guest,ActionCurveToolsWriteTable)<0 then Exit;
 if Ok and NoProblem(Curves.Count>0,RusEng('В окне нет данных для вывода!','No curves in window!'))
       and  not CheckIsDataProtected then
 if CurveToolsWriteTableDialogExecute(Self,PaintBoxPosParams)=mrOk then
 try
  Temp:=CreateTempFile('tmp.tab');
  if NoProblem(Temp<>'',RusEng('Ошибка создания временного файла!','Could not create temporary file!'))
  then begin
   System.Assign(WriTab_Rec.Dest,Temp);
   WriTab_Rec.Delim:=FormCurveToolsWriteTableDialogDelimiter;
   SaveIORes:=IOResult;
   try
    System.Rewrite(WriTab_Rec.Dest);
    if Curves.FormatTable(FormCurveToolsWriteTableDialogTableFormat,WriTab,@WriTab_Rec) then begin
     SmartFileClose(WriTab_Rec.Dest);
     if NoProblem(IOResult=0,RusEng('Ошибка записи в файл!','File write error!'))
     then begin
      Editor:=NewTextEditor(Temp);
      Editor.PathName:='';
     end;
    end else if IOResult=0 then Error(RusEng('Ошибка форматирования!','Formatting error!'))
                           else Error(RusEng('Ошибка записи в файл!','File write error!'));
   finally
    SmartFileClose(WriTab_Rec.Dest);
    FileErase(Temp);
    SetInOutRes(SaveIORes);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionCurveToolsWriteTableExecute');
 end;
end;

procedure TFormCurveWindow.ActionCurveToolsComposeSurfaceExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionCurveToolsComposeSurface)<0 then Exit;
 inherited;
 NewSurfWindowComposedFromCurves(Self);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_curvewindow;
begin
 CurveWindowsProfiler_Clear;
 FullCurveWindowList:=NewCurveWindowList(false);
 FullCurveWindowList.Master:=@FullCurveWindowList;
 CurveWindowsMonitor:=NewCurveWindowList(false);
 CurveWindowsMonitor.Master:=@CurveWindowsMonitor;
 ZoomFactor:=sqrt(2);
end;

procedure Free_form_curvewindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullCurveWindowList.Count', FullCurveWindowList.Count]));
 ResourceLeakageLog(Format('%-60s = %d',['CurveWindowsMonitor.Count', CurveWindowsMonitor.Count]));
 Kill(FullCurveWindowList);
 Kill(CurveWindowsMonitor);
end;

initialization

 Init_form_curvewindow;

finalization

 Free_form_curvewindow;

end.

//////////////
// END OF FILE
//////////////

