////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// FP-QUI Broker to be connected to CRW-DAQ pipe to call tooltip-notifier.    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 2000xxxx - Created by A.K.                                                 //
// 20240129 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program fpquibrk; // FP-QUI Broker

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 interfaces, lclintf,
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_base64, _crw_fio, _crw_fifo,
 _crw_rtc, _crw_ascio, _crw_az, _crw_polling, _crw_task, _crw_fpqui;

 //
 // General variables and constants
 //
 const
 Terminated : Boolean = false; // Program should be terminated
 //
 // Get EXE file version info.
 //
 function GetVersionInfo(const Name:LongString):LongString;
 begin
  Result:=CookieScan(GetFileVersionInfoAsText(ProgName),Name);
 end;
 {$IFDEF WINDOWS}
 //
 // Exception handler for FpQuiManager.
 //
 procedure FpQuiTipExceptionHandler(E:Exception);
 begin
  if Assigned(E) then StdOut.Put:='@OnException='+E.ClassName+' '+E.Message;
 end;
 //
 // Echo procedure for FpQuiManager.
 //
 procedure FpQuiTipEcho(const Msg:LongString);
 begin
  StdOut.Put:=Msg;
 end;
 {$ENDIF ~WINDOWS}
 //
 // Print greetings message.
 //
 procedure Greetings;
 begin
  StdOut.Put:=GetVersionInfo('ProductName')+' version '+GetVersionInfo('ProductVersion')+'.';
  StdOut.Put:=GetVersionInfo('LegalCopyright')+'.';
  StdOut.Put:=GetVersionInfo('ProductName')+' - '+GetVersionInfo('FileDescription');
  StdOut.Put:=GetVersionInfo('Comments');
  StdOut.Put:='Use @help command to get short help.';
  StdOut.Put:='Use @tooltip=--help command to get detail help.';
  StdOut.Put:='Use @tooltip=text msg ... to show popup tooltips.';
 end;

 //
 // Initialize FpQuiManager.
 //
 procedure InitFpQuiManager;
 begin
  if IsUnix then Greetings;
  {$IFDEF WINDOWS}
  TheFpQuiEchoProcedure:=FpQuiTipEcho;
  TheFpQuiExceptionHandlerProcedure:=FpQuiTipExceptionHandler;
  FpQuiManager.DemoDelay:=3000;
  if IsEmptyStr(FpQuiManager.GetCmdLineArguments) then Greetings else begin
   ExitCode:=FpQuiManager.RunFpQuiTipExe;
   Terminated:=true;
  end;
  {$ENDIF ~WINDOWS}
 end;
 //
 // Handred parts of second like ".45".
 //
 function Hund(ms:Double):LongString;
 begin
  Result:=Format('%4.2f',[frac(ms/1000)]);
  Result:=Copy(Result,Pos('.',Result),3);
 end;
 //
 // Expand
 //
 function ExpandEnvVars(const S:LongString):LongString;
 var ms:Double;
 begin
  Result:=S;
  if MaybeEnvStr(S) then begin
   Result:=ExpEnv(Result); ms:=mSecNow;
   Result:=StringReplace(Result,'%CD%',GetCurrentDir,[rfReplaceAll,rfIgnoreCase]);
   Result:=StringReplace(Result,'%DATE%',GetDateStr(ms),[rfReplaceAll,rfIgnoreCase]);
   Result:=StringReplace(Result,'%TIME%',GetTimeStr(ms)+Hund(ms),[rfReplaceAll,rfIgnoreCase]);
   Result:=StringReplace(Result,'%RANDOM%',IntToStr(Random(1024*32)),[rfReplaceAll,rfIgnoreCase]);
  end;
 end;
 //
 // Request: @Tooltip=msg
 // Comment: Return FpQuiManager.RunFpQuiTipExe(msg) result.
 //
 procedure DoTooltip(const cmnd,args:LongString);
 var n,pid:Integer; msg,cmd:LongString; task:TTask;
 begin
  n:=0;
  if not IsEmptyStr(args) then begin
   StdOut.Put:=Format('%s=%s',[cmnd,args]);
   msg:=Trim(ExpandEnvVars(args));
   if IsUnix then begin
    cmd:='unix tooltip-notifier '+msg;
    task:=NewTask('',cmd);
    if task.Run then begin
     n:=EcfpQuiSuccess; pid:=task.Pid;
     StdOut.Put:=Format('PID %d started: %s',[pid,cmd]);
    end else begin
     StdOut.Put:='Could not run: '+cmd;
     n:=EcFpQuiNotSent;
    end;
    Kill(task);
   end;
   {$IFDEF WINDOWS}
   if IsWindows then msg:=DefToAnsiCP(msg);
   n:=FpQuiManager.RunFpQuiTipExe(msg);
   {$ENDIF ~WINDOWS}
  end;
  StdOut.Put:=Format('%s=%d',[cmnd,n]);
 end;
 //
 // Request: @help
 // Reply:   help info
 // Comment: Show help.
 //
 procedure DoHelp(const cmnd,args:LongString);
 begin
  StdOut.Put:=Format('%s',[cmnd]);
  StdOut.Put:='>> Command list:';
  StdOut.Put:='>> @help                 This help.';
  StdOut.Put:='>> @exit=n               Exit program with code n.';
  StdOut.Put:='>> @errors               Return error counter.';
  StdOut.Put:='>> @memory               Return memory status.';
  StdOut.Put:='>> @processpriority=n    Set process priority:Idle/Low/Normal/High/RealTime.';
  StdOut.Put:='>> @threadpriority=n     Set thread priority:tpIdle/tpLow/tpNormal/tpHigh/tpTimeCritical.';
  StdOut.Put:='>> @tooltip=msg          Equivalent to fpquitip.exe msg call. Use --help to get help.';
  StdOut.Put:='>> Example:';
  StdOut.Put:='>> @tooltip=text "Hello world." preset stdTooltip delay 15000';
 end;
 //
 // Request: @exit
 // Request: @exit=n
 // Reply:   @exit=n
 // Comment: Terminate program with exit code n.
 //
 procedure DoExit(const cmnd,args:LongString);
 begin
  Terminated:=true;
  System.ExitCode:=StrToIntDef(args,0);
  StdOut.Put:=Format('%s=%d',[cmnd,System.ExitCode]);
 end;
 //
 // Request: @errors
 // Request: @errors=n
 // Reply:   @errors=n
 // Comment: Return value n of error counter.
 //
 procedure DoErrors(const cmnd,args:LongString);
 var
  n : LongInt;
 begin
  if Str2Long(args,n)
  then n:=LockedExchange(StdIoErrorCount,n)
  else n:=LockedGet(StdIoErrorCount);
  StdOut.Put:=Format('%s=%d',[cmnd,n]);
 end;
 //
 // Request: @memory
 // Request: @memory=n
 // Comment: Return AllocMemSize.
 //
 procedure DoMemory(const cmnd,args:LongString);
 begin
  StdOut.Put:=Format('%s=%d',[cmnd,GetAllocMemSize]);
 end;
 //
 // Request: @ProcessPriority
 // Request: @ProcessPriority=n
 // Reply:   @ProcessPriority=n
 // Comment: Set process priority class.
 //
 procedure DoProcessPriority(const cmnd,args:LongString);
 var p:DWORD;
 begin
  if not IsEmptyStr(args) then begin
   p:=GetPriorityClassByName(args);
   if (p>0) then SetPriorityClass(GetCurrentProcess,p);
  end;
  p:=ProcessPriorityToClass(GetProcessPriority);
  StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityClassName(p)]);
 end;
 //
 // Request: @ThreadPriority
 // Request: @ThreadPriority=n
 // Reply:   @ThreadPriority=n
 // Comment: Set thread priority class.
 //
 procedure DoThreadPriority(const cmnd,args:LongString);
 var p:TThreadPriority;
 begin
  if not IsEmptyStr(args) then begin
   p:=GetPriorityByName(args);
   StdIn.Priority:=p;
   StdOut.Priority:=p;
   {$IFDEF WINDOWS}
   case p of
    tpIdle         : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_IDLE);
    tpLowest       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_LOWEST);
    tpLower        : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_BELOW_NORMAL);
    tpNormal       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
    tpHigher       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_ABOVE_NORMAL);
    tpHighest      : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_HIGHEST);
    tpTimeCritical : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
   end;
   {$ENDIF ~WINDOWS}
  end;
  StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityName(StdIn.Priority)]);
 end;
 //
 // This callback handles unrecognized commands.
 //
 procedure DoSpecificCommands(const args:LongString);
 begin
  if Length(args)>0 then
  StdOut.Put:=Format('Could not recognize "%s"',[args]);
 end;
 //
 // Application specific initialization.
 //
 procedure SpecificInitialization;
 begin
  //
  // Register user commands coming from StdIn.
  //
  SystemEchoProcedure:=StdOutEcho;
  StdIn.SpecHandler:=DoSpecificCommands;
  StdIn.AddCommand('@Help',            DoHelp);
  StdIn.AddCommand('@Exit',            DoExit);
  StdIn.AddCommand('@Errors',          DoErrors);
  StdIn.AddCommand('@Memory',          DoMemory);
  StdIn.AddCommand('@ProcessPriority', DoProcessPriority);
  StdIn.AddCommand('@ThreadPriority',  DoThreadPriority);
  StdIn.AddCommand('@Tooltip',         DoTooltip);
  //
  // Initialize FpQuiManager.
  //
  InitFpQuiManager;
 end;
 //
 // Application specific finalization.
 //
 procedure SpecificFinalization;
 begin
  {$IFDEF WINDOWS}
  FpQuiManager.Free;
  {$ENDIF ~WINDOWS}
 end;
 //
 // Application specific polling.
 //
 procedure SpecificPolling;
 const LastTicks:QWord=0;
 var CurrTicks:QWord;
 begin
  CurrTicks:=GetTickCount64;
  if (CurrTicks>LastTicks+1000) then begin
   TTask.PollDetachedPids;
   LastTicks:=CurrTicks;
  end;
  if BecameZombie(FILE_TYPE_PIPE,1000) then StdIn.Put:='@Exit';
 end;
 //
 // Main program
 //
begin
 try
  try
   SpecificInitialization;
   while not Terminated do begin
    while (StdIn.Count>0) do StdIn.Process(StdIn.Get);
    SpecificPolling;
    Sleep(TPolling.DefPollPeriod);
   end;
  finally
   SpecificFinalization;
  end;
 except
  on E:Exception do StdOut.Put:=E.Message;
 end;
 Sleep(100);
 if BecameZombie(FILE_TYPE_PIPE,0) then ExitCode:=1;
end.

//////////////
// END OF FILE
//////////////

