////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// L-CARD E14-140 driver for CRW-DAQ: shared memory data.                     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20101112 - Created by A.K.                                                 //
// 20241208 - Modified for FPC.                                               //
////////////////////////////////////////////////////////////////////////////////

unit _crw_e140_shm; //  E140 shared memory.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 _crw_alloc, _crw_lcard;

const                            // Shared memory buffer:
 E140BuffFactor   = 2;           // To calculate buf.size
 E140NumBuffers   = 64;          // Number of I/O buffers
 E140BuffAdjust   = 32;          // Buf.length adjustment
 E140MinBuffLen   = 32;          // Minimal buffer length
 E140MaxBuffLen   = 1024*1024;   // Maximal buffer length
 E140HeaderSize   = 1024*16;     // Size of buffer header
 E140DefTimeout   = 1000;        // Default timeout in ms
 E140DefBufTime   = 50;          // Default buff.time, ms
 E140DefBuffLen   = E140MaxBuffLen div E140NumBuffers;
 E140ListBuffSize = 1024*4;      // Buffer for LCard device list

type
 pULONG   = ^ULONG;
 pSHORT   = ^SmallInt;
 pADC_PAR = ^ADC_PAR;

 //
 // Next record uses for shared memory buffer to transfer data.
 //
type
 PE140_SHARED_BUFFER = ^TE140_SHARED_BUFFER;
 TE140_SHARED_BUFFER = packed record
  case Integer of
   0: ( 
        Header : packed array[0..E140HeaderSize-1] of Char;
        Buffer : packed array[0..E140BuffFactor*E140MaxBuffLen*SizeOf(SmallInt)-1] of SmallInt;
       );
   1: (
        SlotNum     : Integer;
        SlotPar     : SLOT_PAR;
        PlataDescr  : PLATA_DESCR_U2;
        AdcInited   : ADC_PAR;
        AdcWanted   : ADC_PAR;
        AdcActual   : ADC_PAR;
        LastError   : ULONG;
        StartTime   : Double;
        ListLDev    : packed array[0..E140ListBuffSize-1] of Char;
       );
 end;

function LPlataSerNum(const Buf:TE140_SHARED_BUFFER):LongString; overload;

function GetE140SerialNumber(const Buf:TE140_SHARED_BUFFER):LongString; overload;
function GetE140Description(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2; const Prefix:AnsiString=''):LongString;
function GetE140AdcPars(const aAdc:ADC_PAR; const Prefix:AnsiString):AnsiString;

implementation

function LPlataSerNum(const Buf:TE140_SHARED_BUFFER):LongString;
begin
 Result:=LPlataSerNum(Buf.SlotPar,Buf.PlataDescr);
end;

function GetE140SerialNumber(const Buf:TE140_SHARED_BUFFER):LongString; overload;
begin
 Result:=LPlataSerNum(Buf);
end;

function GetE140Description(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2; const Prefix:AnsiString=''):LongString;
var i:Integer; Buff:array[0..255] of Char;
 procedure Add(s:AnsiString);
 begin
  if Length(Result)=0
  then Result:=Prefix+s
  else Result:=Result+#13#10+Prefix+s; 
 end;
begin
 Result:='';
 Add('BoardType = '+LBoardTypeName(sp));
 Add('BoardName = '+ LPlataBrdName(sp,pd));
end;

function GetE140AdcPars(const aAdc:ADC_PAR; const Prefix:AnsiString):AnsiString;
var i:Integer; 
 procedure Add(s:AnsiString);
 begin
  if Length(Result)=0
  then Result:=Prefix+s
  else Result:=Result+#13#10+Prefix+s; 
 end;
begin
 Result:='';
 with aAdc.t1 do begin
  Add(Format('s_Type             = %d',[s_Type]));
  Add(Format('FIFO               = %d',[FIFO]));
  Add(Format('IrqStep            = %d',[IrqStep]));
  Add(Format('Pages              = %d',[Pages]));
  Add(Format('AutoInit           = %d',[AutoInit]));
  Add(Format('dRate              = %g',[dRate]));
  Add(Format('dKadr              = %g',[dKadr]));
  Add(Format('dScale             = %g',[dScale]));
  Add(Format('Rate               = %d',[Rate]));
  Add(Format('Kadr               = %d',[Kadr]));
  Add(Format('Scale              = %d',[Scale]));
  Add(Format('FPDelay            = %d',[FPDelay]));
  Add(Format('SynchroType        = %d',[SynchroType]));
  Add(Format('SynchroSensitivity = %d',[SynchroSensitivity]));
  Add(Format('SynchroMode        = %d',[SynchroMode]));
  Add(Format('AdChannel          = %d',[AdChannel]));
  Add(Format('AdPorog            = %d',[AdPorog]));
  Add(Format('NCh                = %d',[NCh]));
  for i:=0 to Min(Integer(Nch)-1,High(Chn)) do 
  Add(Format('Chn[%2d]           = %d',[i,Chn[i]]));
  Add(Format('IrqEna             = %d',[IrqEna]));
  Add(Format('AdcEna             = %d',[AdcEna]));
 end;
end;


///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_e140_shm;
begin
end;

procedure Free_crw_e140_shm;
begin
end;

initialization

 Init_crw_e140_shm;

finalization

 Free_crw_e140_shm;

end.

//////////////
// END OF FILE
//////////////

