{*********************************************************************
 Дополнительный модуль, облегчающий работу с низкоуровневыми типами 
 данных GMP:
 - определениы арифметические и сравнивающие операторы;
 - некоторые частоупотребляемые процедуры сделаны функциями.
 *********************************************************************}
{$mode objfpc}{$h+}
{$packrecords c}
Unit gmp2;

interface

Uses gmp;

Type
  // Для совместимости с gmp.h
  mp_bitcnt_t	= LongWord;
  mp_ptr	= ^mp_limb_t;
  
  // Для преобразования числа в строку
  TTypeNumber = (tnFixed, tnScientific);

  // Для математики
  TmpfVector = array of mpf_t;
  TmpfMatrix = array of array of mpf_t;

{--------------------------------------------
 Десятичные дроби
---------------------------------------------}
// Инициализация заранее неизвестного количества переменных
procedure mpf_inits(op: mpf_ptr {; ...}); cdecl; varargs; external LIB name '__gmpf_inits';
// Очистка заранее неизвестного количества переменных
procedure mpf_clears(op: mpf_ptr {; ...}); cdecl; varargs; external LIB name '__gmpf_clears';

{********************************************
 Операторы
*********************************************}

// Операторы присваивания значений
operator := (op: AnsiString): mpf_t;
operator := (op: pchar): mpf_t;
operator := (op: longword): mpf_t;
operator := (op: integer): mpf_t;
operator := (op: double): mpf_t;
operator := (op: mpq_t): mpf_t;
operator := (op: mpz_t): mpf_t;

// Функция извлечения кв. корня
function sqrt(op: mpf_t): mpf_t;
function sqrt(op: longword): mpf_t;

// Операторы сравнения
operator =  (op1: mpf_t; op2: mpf_t): boolean;
operator <> (op1: mpf_t; op2: mpf_t): boolean;
operator >  (op1: mpf_t; op2: mpf_t): boolean;
operator <  (op1: mpf_t; op2: mpf_t): boolean;
operator >= (op1: mpf_t; op2: mpf_t): boolean;
operator <= (op1: mpf_t; op2: mpf_t): boolean;

{*********************************************************
 Функции
**********************************************************}
// Проверка знака числа
function mpf_sgn(op: mpf_t): integer;

// Абсолютное значение
function abs(op: mpf_t): mpf_t;

// Округление
function ceil(op: mpf_t): mpf_t;
function floor(op: mpf_t): mpf_t;
function trunc(op: mpf_t): mpf_t;

// Преобразование в строку
function mpf_ToString(op: mpf_t; tn: TTypeNumber = tnFixed): AnsiString;

// Служебная информация
//
// Количество разрядов в мантиссе по кол-ву бит
function GetDigits(prec: LongWord): longWord;

// Максимально возможное количество разрядов числа
function mpg_GetMaxDigits(op: mpf_t): LongWord;

// Количество битов по кол-ву разрядов.
// Результат округляется до ближайшего большего
// целого кол-ва лимбов
function GetPrec(digits: LongWord): valuint;

{--------------------------------------------
 Целые числа
---------------------------------------------}
// Инициализация заранее неизвестного количества переменных
procedure mpz_inits(op: mpz_ptr {; ...}); cdecl; varargs; external LIB name '__gmpz_inits';
// Очистка заранее неизвестного количества переменных
procedure mpz_clears(op: mpz_ptr {; ...}); cdecl; varargs; external LIB name '__gmpz_clears';

{********************************************
 Операторы
*********************************************}

// Операторы присваивания значений
operator := (op: string): mpz_t;
operator := (op: pchar): mpz_t;
operator := (op: longword): mpz_t;
operator := (op: integer): mpz_t;
operator := (op: double): mpz_t;
operator := (op: mpf_t): mpz_t;
operator := (op: mpq_t): mpz_t;

// Операторы сравнения
operator =  (op1: mpz_t; op2: mpz_t): boolean;
operator <> (op1: mpz_t; op2: mpz_t): boolean;
operator >  (op1: mpz_t; op2: mpz_t): boolean;
operator <  (op1: mpz_t; op2: mpz_t): boolean;
operator >= (op1: mpz_t; op2: mpz_t): boolean;
operator <= (op1: mpz_t; op2: mpz_t): boolean;

{*********************************************************
 Функции
**********************************************************}
// Проверка знака числа
function mpz_sgn(op: mpz_t): integer;

// Абсолютное значение
function abs(op: mpz_t): mpz_t;

{--------------------------------------------
 Обыкновенные дроби
---------------------------------------------}
// Инициализация заранее неизвестного количества переменных
procedure mpq_inits(op: mpq_ptr {; ...}); cdecl; varargs; external LIB name '__gmpq_inits';
// Очистка заранее неизвестного количества переменных
procedure mpq_clears(op: mpq_ptr {; ...}); cdecl; varargs; external LIB name '__gmpq_clears';

{********************************************
 Операторы
*********************************************}

// Операторы присваивания значений
operator := (op: string): mpq_t;
operator := (op: pchar): mpq_t;
operator := (op: mpz_t): mpq_t;

// Операторы сравнения
operator =  (op1: mpq_t; op2: mpq_t): boolean;
operator <> (op1: mpq_t; op2: mpq_t): boolean;
operator >  (op1: mpq_t; op2: mpq_t): boolean;
operator <  (op1: mpq_t; op2: mpq_t): boolean;
operator >= (op1: mpq_t; op2: mpq_t): boolean;
operator <= (op1: mpq_t; op2: mpq_t): boolean;

{*********************************************************
 Функции
**********************************************************}
// Проверка знака числа
function mpq_sgn(op: mpq_t): integer;

// Абсолютное значение
function abs(op: mpq_t): mpq_t;



implementation

Uses SysUtils, Math;

Var
  TmpResult_f: mpf_t;
  TmpResult_z: mpz_t;
  TmpResult_q: mpq_t;

{--------------------------------------------
 Десятичные дроби
---------------------------------------------}

{------ Присваивание --------------------------------}
operator := (op: string): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_str(result, pchar(op), 0);
end;

operator := (op: pchar): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_str(result, op, 0);
end;

operator := (op: longword): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_ui(result, op);
end;

operator := (op: integer): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_si(result, op);
end;

operator := (op: double): mpf_t;
begin
  result:=TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_d(result, op);
end;

operator := (op: mpz_t): mpf_t;
begin
  result:=TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_z(result, op);
end;

operator := (op: mpq_t): mpf_t;
begin
  result:=TmpResult_f;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_set_q(result, op);
end;

{------ Корень квадратный ---------------------------}
function sqrt(op: mpf_t): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_prec(op));
  mpf_sqrt(result, op);
end;

function sqrt(op: longword): mpf_t;
begin
  result := TmpResult_z;
  mpf_set_prec(result, mpf_get_default_prec());
  mpf_sqrt_ui(result, op);
end;

{------ Сравнение ------------------------------------}
operator = (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) = 0 Then
    result:=True;
end;

operator <> (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) <> 0 Then
    result:=True;
end;

operator > (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) > 0 Then
    result:=True;
end;

operator < (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) < 0 Then
    result:=True;
end;

operator >= (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) >= 0 Then
    result:=True;
end;

operator <= (op1: mpf_t; op2: mpf_t): boolean;
begin
  result:=False;
  If mpf_cmp(op1, op2) <= 0 Then
    result:=True;
end;

{*************************************************************
 Функции
**************************************************************}
{------ Проверка знака числа ---------------------------------}
function mpf_sgn(op: mpf_t): integer;
begin
  If op.size<0 Then
    Result:=-1
  Else If op.size>0 Then
    Result:=1
  Else
    Result:=0;
end;

{------ Абсолютное значение ----------------------------------}
function abs(op: mpf_t): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_prec(op));
  mpf_abs(result, op);
end;

{------ Округление ------------------------------------------}
function ceil(op: mpf_t): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_prec(op));
  mpf_ceil(result, op);
end;

function floor(op: mpf_t): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_prec(op));
  mpf_floor(result, op);
end;

function trunc(op: mpf_t): mpf_t;
begin
  result := TmpResult_f;
  mpf_set_prec(result, mpf_get_prec(op));
  mpf_trunc(result, op);
end;

{------ Преобразование в строку ----------------------}
function mpf_ToString(op: mpf_t; tn: TTypeNumber = tnFixed): AnsiString;
Var
  pc: PChar;
  cnt: longword;
  view: string;
begin
  cnt := GetDigits(mpf_get_prec(op));
  GetMem(pc, cnt);
  If tn = tnFixed Then
    view := '.Ff'
  Else
    view := '.Fe';
  mp_sprintf(pc, PChar('%'+IntToStr(cnt)+view), op);
  result := pc;
  FreeMem(pc);
  result := Trim(result);
end;


{------ Служебная информация -------------------------}
function GetDigits(prec: LongWord): longWord;
begin
  result := Math.floor(prec * LOG_10_2);
end;

function mpg_GetMaxDigits(op: mpf_t): LongWord;
begin
  result:=GetDigits(mpf_get_prec(op));
end;

function GetPrec(digits: LongWord): valuint;
Var
  ost: LongWord;
  bpl: LongWord;
begin
  result:=Math.ceil(digits/LOG_10_2);
  bpl:=bits_per_limb();
  ost:=result mod bpl;
  result:=result + bpl - ost;
end;

{--------------------------------------------
 Целые числа
---------------------------------------------}
{------ Присваивание --------------------------------}
operator := (op: string): mpz_t;
begin
  result := TmpResult_z;
  mpz_set_str(result, pchar(op), 0);
end;

operator := (op: pchar): mpz_t;
begin
  result := TmpResult_z;
  mpz_set_str(result, op, 0);
end;

operator := (op: longword): mpz_t;
begin
  result := TmpResult_z;
  mpz_set_ui(result, op);
end;

operator := (op: integer): mpz_t;
begin
  result := TmpResult_z;
  mpz_set_si(result, op);
end;

operator := (op: double): mpz_t;
begin
  result:=TmpResult_z;
  mpz_set_d(result, op);
end;

operator := (op: mpf_t): mpz_t;
begin
  result:=TmpResult_z;
  mpz_set_f(result, op);
end;

operator := (op: mpq_t): mpz_t;
begin
  result:=TmpResult_z;
  mpz_set_q(result, op);
end;

{------ Корень квадратный ---------------------------}
function sqrt(op: mpz_t): mpz_t;
begin
  result := TmpResult_z;
  mpz_sqrt(result, op);
end;

{------ Сравнение ------------------------------------}
operator = (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) = 0 Then
    result:=True;
end;

operator <> (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) <> 0 Then
    result:=True;
end;

operator > (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) > 0 Then
    result:=True;
end;

operator < (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) < 0 Then
    result:=True;
end;

operator >= (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) >= 0 Then
    result:=True;
end;

operator <= (op1: mpz_t; op2: mpz_t): boolean;
begin
  result:=False;
  If mpz_cmp(op1, op2) <= 0 Then
    result:=True;
end;

{*************************************************************
 Функции
**************************************************************}
{------ Проверка знака числа ---------------------------------}
function mpz_sgn(op: mpz_t): integer;
begin
  If op.size<0 Then
    Result:=-1
  Else If op.size>0 Then
    Result:=1
  Else
    Result:=0;
end;

{------ Абсолютное значение ----------------------------------}
function abs(op: mpz_t): mpz_t;
begin
  result := TmpResult_z;
  mpz_abs(result, op);
end;

{--------------------------------------------
 Обычные дроби
---------------------------------------------}
{------ Присваивание --------------------------------}
operator := (op: string): mpq_t;
begin
  result := TmpResult_q;
  mpq_set_str(result, pchar(op), 0);
end;

operator := (op: pchar): mpq_t;
begin
  result := TmpResult_q;
  mpq_set_str(result, op, 0);
end;

operator := (op: mpz_t): mpq_t;
begin
  result:=TmpResult_q;
  mpq_set_z(result, op);
end;

{------ Сравнение ------------------------------------}
operator = (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) = 0 Then
    result:=True;
end;

operator <> (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) <> 0 Then
    result:=True;
end;

operator > (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) > 0 Then
    result:=True;
end;

operator < (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) < 0 Then
    result:=True;
end;

operator >= (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) >= 0 Then
    result:=True;
end;

operator <= (op1: mpq_t; op2: mpq_t): boolean;
begin
  result:=False;
  If mpq_cmp(op1, op2) <= 0 Then
    result:=True;
end;

{*************************************************************
 Функции
**************************************************************}
{------ Проверка знака числа ---------------------------------}
function mpq_sgn(op: mpq_t): integer;
begin
  If op.num.size<0 Then
    Result:=-1
  Else If op.num.size>0 Then
    Result:=1
  Else
    Result:=0;
end;

{------ Абсолютное значение ----------------------------------}
function abs(op: mpq_t): mpq_t;
begin
  result := TmpResult_q;
  mpq_abs(result, op);
end;

Initialization
  mpf_init(TmpResult_f);
  mpz_init(TmpResult_z);
  mpq_init(TmpResult_q);

finalization
  mpf_clear(TmpResult_f);
  mpz_clear(TmpResult_z);
  mpq_clear(TmpResult_q);

End.  