'****************************************************************
'** Copyright (c) 2018-2021 Alexey Kuryakin kouriakine@mail.ru **
'** Under MIT License, see https://opensource.org/licenses/MIT **
'****************************************************************
'** Command line tool to invoke InputBox with options.         **
'****************************************************************

Option Explicit

'******* System objects
'**********************

dim StdIn  : set StdIn  = WScript.StdIn
dim StdOut : set StdOut = WScript.StdOut
dim StdErr : set StdErr = WScript.StdErr
dim Shell  : set Shell  = WScript.CreateObject("WScript.Shell")
dim FSO    : set FSO    = WScript.CreateObject("Scripting.FileSystemObject")

'******* Utility routines
'************************

sub WriteLn(line)
 StdOut.WriteLine(line)
end sub

sub PrintStdErr(line)
 StdErr.WriteLine(line)
end sub

sub Abort(ExitCode,ErrorMessage)
 PrintStdErr(ErrorMessage)
 WScript.Quit(ExitCode)
end sub

sub Assert(Condition,ExitCode,ErrorMessage)
 if not cbool(Condition) then
  call Abort(ExitCode,ErrorMessage)
 end if
end sub

function IntBool(cond)
 if cbool(cond) then : IntBool = 1 : else : IntBool = 0 : end if
end function

function IsEmptyStr(s)
 IsEmptyStr=not(trim(s)<>"")
end function

function evaldef(s,def)
 On Error Resume Next
 dim l : l=eval(s)
 if (Err.Number<>0) then : l=def : end if : Err.Clear
 On Error Goto 0
 evaldef=l
end function

function GetBaseName(path)
 GetBaseName = FSO.GetBaseName(path)
end function

'******* Print Version
'*********************

sub PrintVersion(name)
 WriteLn(ucase(name)+" version 1.0")
end sub

'******* Print help screen
'*************************

sub PrintHelp(name)
 PrintVersion(name)
 WriteLn("Command line tool to invoke InputBox with options.")
 WriteLn("Copyright (c) 2018-2021 Alexey Kuryakin kouriakine@mail.ru")
 WriteLn("Under MIT License, see https://opensource.org/licenses/MIT")
 WriteLn("Help on "+ucase(name)+":")
 WriteLn(" ===================> Syntax:")
 WriteLn("  1) "+ucase(name)+" Caption Message [Default] [Xpos] [Ypos] [NameVar]")
 WriteLn("     Open "+name+" with fixed, position-dependent arguments.")
 WriteLn("     Only single-line messages is available in this form.")
 WriteLn("  2) "+ucase(name)+" -c Caption -m Message1 -m Message2.. -d Default -x Xpos -y Ypos -n NameVar")
 WriteLn("     Open "+name+" with option-dependent arguments (multi-line messages is available)")
 WriteLn("     Short options -c,-m,-d,-x,-y,-n may be used as synonym of corresponded")
 WriteLn("      long options --caption,--message,--default,--xpos,--ypos,--namevar")
 WriteLn(" ===================> Options:")
 WriteLn("   --              => options ended, next is params")
 WriteLn("   --version       => print program version and exit")
 WriteLn("   -h,--help       => print this help screen and exit")
 WriteLn("   -c,--caption c  => set Caption string 'c'")
 WriteLn("   -m,--message m  => add line 'm' to Message text (multi-line messages)")
 WriteLn("   -d,--default d  => set 'd' as Default input text")
 WriteLn("   -x,--xpos x     => set Xpos by expression 'x', in twips")
 WriteLn("   -y,--ypos y     => set Ypos by expression 'y', in twips")
 WriteLn("                      twips=pixels*1440/ppi (usually ppi=PixelsPerInch=96)")
 WriteLn("   -n,--namevar n  => set NameVar 'n' = name of result variable to print")
 WriteLn(" ===================> Print result of input:")
 WriteLn("  <Empty>             if Escape/Cancel/Error")
 WriteLn("  <Default>           if User change nothing")
 WriteLn("  <user data>         if User input some data")
 WriteLn(" ===================> Exit Code:")
 WriteLn("   0               => success input done")
 WriteLn("   1               => input dialog cancelled or empty input")
 WriteLn("   2               => some error found (bad arguments/options)")
 WriteLn("   else            => some error found (internal script error)")
 WriteLn(" ===================> Examples:")
 WriteLn("  call "+name+" --help")
 WriteLn("  call "+name+" --version")
 WriteLn("  call "+name+" ""Test"" ""Please enter something..."" ")
 WriteLn("  call "+name+" ""Authorization"" ""Please enter your name"" %UserName% ")
 WriteLn("  call "+name+" -c ""Demo Caption"" -m ""Demo line 1"" -m ""Demo line 2"" -d ""User data"" -x 300*1440/96 -y 200*1440/96 -n Result")
end sub

'******* Program data and options
'********************************

dim argnum     : argnum     = 0
dim theXpos    : theXpos    = Empty
dim theYpos    : theYpos    = Empty
dim theResult  : theResult  = Empty
dim theDefault : theDefault = Empty
dim theCaption : theCaption = ""
dim theMessage : theMessage = ""
dim theNameVar : theNameVar = ""
dim ExitCode   : ExitCode   = 0

'******* Program routines
'************************

sub AddMessage(arg)
 if theMessage<>"" then
  theMessage=theMessage+vbCrLf
 end if
 theMessage=theMessage+cstr(arg)
end sub

sub SetCaption(arg)
 theCaption=cstr(arg)
end sub

sub SetXpos(arg)
 if not IsEmptyStr(arg) then
  dim x : x=evaldef(cstr(arg),Empty)
  if not IsEmpty(x) then theXpos=cint(x) end if
 end if
end sub

sub SetYpos(arg)
 if not IsEmptyStr(arg) then
  dim y : y=evaldef(cstr(arg),Empty)
  if not IsEmpty(y) then theYpos=cint(y) end if
 end if
end sub

sub SetDefault(arg)
 theDefault=cstr(arg)
end sub

sub SetNameVar(arg)
 theNameVar=cstr(arg)
end sub

'******* Handle arguments
'************************

sub HandleArgs(arg)
 argnum=argnum+1
 select case argnum
  case 1     : call SetCaption(cstr(arg))
  case 2     : call AddMessage(cstr(arg))
  case 3     : call SetDefault(cstr(arg))
  case 4     : call SetXpos(cstr(arg))
  case 5     : call SetYpos(cstr(arg))
  case 6     : call SetNameVar(cstr(arg))
  case else : call Abort(2,"Error: excess argument found. See --help.")
 end select
end sub

'******* Command line parsing
'****************************

dim i,arg,opt,isopt : arg="" : opt="" : isopt=true
for i=0 to WScript.Arguments.Count-1
 arg=WScript.Arguments(i)
 if (left(arg,1)="-") and isopt and (opt="") then
  select case arg
   case "--"                    : isopt=false ' end of options, interpret next arguments as params
   case "--version"             : PrintVersion(GetBaseName(WScript.ScriptName)) : WScript.Quit(0)
   case "-h","--help"           : PrintHelp(GetBaseName(WScript.ScriptName))    : WScript.Quit(0)
   case "-c","--caption"        : opt=arg
   case "-m","--message"        : opt=arg
   case "-d","--default"        : opt=arg
   case "-x","--xpos"           : opt=arg
   case "-y","--ypos"           : opt=arg
   case "-n","--namevar"        : opt=arg
   case else                    : call Abort(2,"Error: unknown option "+arg+". See --help.")
  end select
 else
  select case opt
   case "-c","--caption"        : call SetCaption(arg)
   case "-m","--message"        : call AddMessage(arg)
   case "-d","--default"        : call SetDefault(arg)
   case "-x","--xpos"           : call SetXpos(arg)
   case "-y","--ypos"           : call SetYpos(arg)
   case "-n","--namevar"        : call SetNameVar(arg)
   case ""                      : call HandleArgs(arg)
   case else                    : call Abort(2,"Error: unknown option "+opt+". See --help.")
  end select
  opt=""
 end if
next

'******* Task execution
'**********************

if (WScript.Arguments.Count=0) then : PrintHelp(GetBaseName(WScript.ScriptName)) : WScript.Quit(0) : end if

call Assert(WScript.Interactive,2,"Error: "+WScript.Name+" is not Interactive. Use //I option.")

call Assert(theCaption<>"",2,"Error: empty --caption. See --help.")
call Assert(theMessage<>"",2,"Error: empty --message. See --help.")

select case (1*IntBool(not IsEmpty(theDefault))+2*IntBool(not IsEmpty(theXpos))+4*IntBool(not IsEmpty(theYpos)))
 case 0    : theResult=InputBox(theMessage,theCaption                           )
 case 1    : theResult=InputBox(theMessage,theCaption,theDefault                )
 case 2    : theResult=InputBox(theMessage,theCaption,          ,theXpos        )
 case 3    : theResult=InputBox(theMessage,theCaption,theDefault,theXpos        )
 case 4    : theResult=InputBox(theMessage,theCaption,          ,       ,theYpos)
 case 5    : theResult=InputBox(theMessage,theCaption,theDefault,       ,theYpos)
 case 6    : theResult=InputBox(theMessage,theCaption,          ,theXpos,theYpos)
 case 7    : theResult=InputBox(theMessage,theCaption,theDefault,theXpos,theYpos)
 case else : theResult=InputBox(theMessage,theCaption,theDefault,theXpos,theYpos)
end select

if (theResult="") then ExitCode=1 end if

if theNameVar<>"" then theResult=cstr(theNameVar)+"="+cstr(theResult) end if

WriteLn(cstr(theResult))

WScript.Quit(ExitCode)

'******* Done
'************
