////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// This unit contains API interface to use in DLL plugins for CRW-DAQ.        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 2003xxxx - Created by A.K.                                                 //
// 20241108 - Modified for FPC (A.K.)                                         //
// 20241109 - Modified to use interfaces.                                     //
// 20241111 - all classes replaced to interfaces                              //
////////////////////////////////////////////////////////////////////////////////

unit _crw_crwapi; // CRW API - program application interface for plugins.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics, types, lcltype;

const //////////////////////////////////////////////////////////////////////////
 TheCrwApiEdition = 20241211; // YearMonthDay of the current CrwApi edition.  //
 ///////////////////////////////////////////////////////////////////////////////

 ///////////////////////////////////////////////////////////////////////////////
 // Usage is like: CrwApi.VerifyPluginDate(CrwApiEdition,CrwApi.Version);     //
 ///////////////////////////////////////////////////////////////////////////////
function CrwApiEdition:Cardinal;

 ///////////////////////////////////////////////////////////////////////////////
 // The CrwAPI is program interface for crwdaq plugins (with dynamic load).   //
 // All CrwAPI implemented by using Interfaces for binary compatibility.      //
 // Please remember simple rule: you can add new methods to interfaces,       //
 // but don't change existing methods to avoid backward incompatibility.      //
 ///////////////////////////////////////////////////////////////////////////////

type // Base interface for plugin API's.
 IPluginApi = interface(IUnknown)
 end;

const // Identifiers for API's interfaces.
 SIID_ICrwApi = '{E3D9744C-721C-4F8D-B25F-FA25613E0A83}';
 SIID_ISysApi = '{E255F12A-34E6-427D-BA29-9E34E012078B}';
 SIID_IGuiApi = '{FD530AA4-1355-470C-88FD-32D807E0C74A}';
 SIID_IDaqApi = '{98995C87-C191-4E53-A6E9-EF2AB55BD639}';
 SIID_IDanApi = '{E751D9C0-3CDE-49B4-AEFD-A2CF05EB2E99}';
const // GUIDs for API's interfaces.
 IID_ICrwApi  : TGUID = SIID_ICrwApi;
 IID_ISysApi  : TGUID = SIID_ISysApi;
 IID_IGuiApi  : TGUID = SIID_IGuiApi;
 IID_IDaqApi  : TGUID = SIID_IDaqApi;
 IID_IDanApi  : TGUID = SIID_IDanApi;

type // Base interface for plugin wrappers.
 IPluginWrap = interface(IUnknown)
 end;

const // Identifiers for wrapper's interfaces.
 SIID_IRfaDataBaseWrap   = '{F45FAC1F-630B-4C38-818A-53B197596EB8}';
 SIID_IScriptInterpreter = '{41AF38CC-A2B1-4B94-879F-1DF74470D875}';
 SIID_IPolynomWrap       = '{EAE12521-41EE-4711-8928-6D9316540C2D}';
 SIID_ILongIntVectorWrap = '{E527477A-5B7A-4569-9B05-1AA6344A0BBA}';
 SIID_IDoubleVectorWrap  = '{C417CFA7-ED3E-43DA-B6BD-9F5FCC9D5747}';
 SIID_ILongIntMatrixWrap = '{6FE70CD8-24B9-48E9-8A8B-F75E0637DF67}';
 SIID_IDoubleMatrixWrap  = '{4351BA1A-4B89-423B-BF4B-313F4723BBE4}';
 SIID_ISVDWrap           = '{89EF7BA5-B9A3-4E7D-ACA0-B207D3D9913C}';
 SIID_IFifoWrap          = '{B94BCB60-8B40-4824-BE55-CC61E73D6287}';
const // GUIDs for wrapper's interfaces.
 IID_IRfaDataBaseWrap    : TGuid = SIID_IRfaDataBaseWrap;
 IID_IScriptInterpreter  : TGuid = SIID_IScriptInterpreter;
 IID_IPolynomWrap        : TGuid = SIID_IPolynomWrap;
 IID_ILongIntVectorWrap  : TGuid = SIID_ILongIntVectorWrap;
 IID_IDoubleVectorWrap   : TGuid = SIID_IDoubleVectorWrap;
 IID_ILongIntMatrixWrap  : TGuid = SIID_ILongIntMatrixWrap;
 IID_IDoubleMatrixWrap   : TGuid = SIID_IDoubleMatrixWrap;
 IID_ISVDWrap            : TGuid = SIID_ISVDWrap;
 IID_IFifoWrap           : TGuid = SIID_IFifoWrap;

 ///////////////////////////////////////////////////////////////////////////////
 // Base types                                                                //
 ///////////////////////////////////////////////////////////////////////////////
type
 TPid       = SizeInt;
 TCharSet   = TSysCharSet;
 TByteSet   = set of Byte;
 PureString = ShortString;
 LongString = RawByteString;

 ///////////////////////////////////////////////////////////////////////////////
 // Определения констант и типов общего применения.                           //
 ///////////////////////////////////////////////////////////////////////////////
const                                             // CPU/RAM bitness
 NumBitsPerByte = 8;                              // Number of Bits per Byte
 CpuBitness     = SizeOf(Pointer)*NumBitsPerByte; // System CPU/RAM Bitness
const
 // Константы памяти.
 KiloByte          = 1024;
 MegaByte          = KiloByte*KiloByte;
 GigaByte          = MegaByte*KiloByte;
const                // Разделители строк
 CRLF = #13#10;      // Text line delimiter for DOS, Windows.
 EOL  = LineEnding;  // End Of Line delimeter for current OS.
const                // Posix PIPE_BUF = atomic pipe IO buffer.
 OS_PIPE_BUF = 4096; // Number of bytes in atomic write to a pipe (Linux).
const                // Succeed error codes.
 NO_ERROR      = 0;  // No error found.
 ERROR_SUCCESS = 0;  // Success status.
const                // Task reference range
 task_ref_min = 1;   // Minimal task reference
 task_ref_max = 255; // Maximal task reference
const                             // From Windows API: useful constants.
 STATUS_PENDING = $103; {259}     // Operation requested pending completion.
 STILL_ACTIVE   = STATUS_PENDING; // The thread or process is still active.
const // Standard formats for FormatDateTime function
 StdTimeOnlyFormat   = 'hh:nn:ss';                // Standard     Time Format
 StdDateOnlyFormat   = 'yyyy.mm.dd';              // Standard Date     Format
 StdTimeOnlyFormatMs = 'hh:nn:ss.zzz';            // Standard Time Format +ms
 StdDateTimeFormat   = 'yyyy.mm.dd-hh:nn:ss';     // Standard DateTime Format
 StdDateTimeFormatMs = 'yyyy.mm.dd-hh:nn:ss.zzz'; // Date Time Format with ms
const // Severity levels for error handlers and messages
 SEVERITY_DEBUG      = 0;            // No error(s), just for debugging
 SEVERITY_INFO       = 1;            // No error(s), just for information
 SEVERITY_WARN       = 2;            // Warning, maybe it`s not error yet
 SEVERITY_ERROR      = 3;            // Error found (recoverable)
 SEVERITY_FATAL      = 4;            // Fatal error (unrecoverable)
 SEVERITY_OFF        = 5;            // Any kind of messages is OFF
 SEVERITY_NAMES      = 'DEBUG'+EOL+'INFO'+EOL+'WARN'+EOL+'ERROR'+EOL+'FATAL'+EOL+'OFF'+EOL;
 SEVERITY_WARNING    = SEVERITY_WARN; // Synonym for backward compatibility
const // Unix/Posix file access constants   // Possible values  for mode
 S_IRUSR = &400; // %0100000000;            // Read permission  for owner
 S_IWUSR = &200; // %0010000000;            // Write permission for owner
 S_IXUSR = &100; // %0001000000;            // Exec  permission for owner
 S_IRGRP = &040; // %0000100000;            // Read permission  for group
 S_IWGRP = &020; // %0000010000;            // Write permission for group
 S_IXGRP = &010; // %0000001000;            // Exec permission  for group
 S_IROTH = &004; // %0000000100;            // Read permission  for other
 S_IWOTH = &002; // %0000000010;            // Write permission for other
 S_IXOTH = &001; // %0000000001;            // Exec permission  for other
 S_IRWXU = S_IRUSR or S_IWUSR or S_IXUSR;   // Read Write Exec  for owner
 S_IRWXG = S_IRGRP or S_IWGRP or S_IXGRP;   // Read Write Exec  for group
 S_IRWXO = S_IROTH or S_IWOTH or S_IXOTH;   // Read Write Exec  for other
const                                       // Over Posix
 S_IRWXUG  = S_IRWXU or S_IRWXG;            // Read Write Exec  for USR/GRP
 S_IRWXUGO = S_IRWXU or S_IRWXG or S_IRWXO; // Read Write Exec  for all
const // Marker for file errors.
 INVALID_HANDLE_VALUE = THandle(-1);
const // GetFileType results
 FILE_TYPE_UNKNOWN = 0;     // Unknown or error  (all platforms )
 FILE_TYPE_DISK    = 1;     // Regular file      (all platforms )
 FILE_TYPE_CHAR    = 2;     // Character special (all platforms )
 FILE_TYPE_PIPE    = 3;     // Named pipe (fifo) (all platforms )
 ////////////////////////////////////////////////////////////////
 FILE_TYPE_NONE    = 0;     // Unknown or error  (synonym       )
 FILE_TYPE_REG     = 1;     // Regular file      (synonym       )
 FILE_TYPE_CHR     = 2;     // Character special (synonym       )
 FILE_TYPE_FIFO    = 3;     // Named pipe (fifo) (synonym       )
 ////////////////////////////////////////////////////////////////
 FILE_TYPE_DIR     = 4;     // Directory         (unix extension)
 FILE_TYPE_BLK     = 5;     // Block special     (unix extension)
 FILE_TYPE_LNK     = 6;     // Symbolic link     (unix extension)
 FILE_TYPE_SOCK    = 7;     // Socket            (unix extension)
 FILE_TYPE_REMOTE  = $8000; // Remote file       (winapi, unused)
const                  // Windows Registry Root keys:
 HKEY_CLASSES_ROOT     = HKEY($80000000);
 HKEY_CURRENT_USER     = HKEY($80000001);
 HKEY_LOCAL_MACHINE    = HKEY($80000002);
 HKEY_USERS            = HKEY($80000003);
 HKEY_PERFORMANCE_DATA = HKEY($80000004);
 HKEY_CURRENT_CONFIG   = HKEY($80000005);
 HKEY_DYN_DATA         = HKEY($80000006);
const // WinApi ShowWindow() Commands
 SW_HIDE            = 0;
 SW_SHOWNORMAL      = 1;
 SW_NORMAL          = 1;
 SW_SHOWMINIMIZED   = 2;
 SW_SHOWMAXIMIZED   = 3;
 SW_MAXIMIZE        = 3;
 SW_SHOWNOACTIVATE  = 4;
 SW_SHOW            = 5;
 SW_MINIMIZE        = 6;
 SW_SHOWMINNOACTIVE = 7;
 SW_SHOWNA          = 8;
 SW_RESTORE         = 9;
 SW_SHOWDEFAULT     = 10;
 SW_MAX             = 10;
const // Unix signals
 SIGHUP  = 1;  SIGINT    = 2;  SIGQUIT = 3;  SIGILL   = 4;  SIGTRAP = 5;
 SIGABRT = 6;  SIGIOT    = 6;  SIGBUS  = 7;  SIGFPE   = 8;  SIGKILL = 9;
 SIGUSR1 = 10; SIGSEGV   = 11; SIGUSR2 = 12; SIGPIPE  = 13; SIGALRM = 14;
 SIGTERM = 15; SIGSTKFLT = 16; SIGCHLD = 17; SIGCONT  = 18; SIGSTOP = 19;
 SIGTSTP = 20; SIGTTIN   = 21; SIGTTOU = 22; SIGURG   = 23; SIGXCPU = 24;
 SIGXFSZ = 25; SIGVTALRM = 26; SIGPROF = 27; SIGWINCH = 28; SIGIO   = 29;
 SIGPWR  = 30; SIGSYS    = 31;
 SIGCLD  = SIGCHLD;
 SIGPOLL = SIGIO;
const                       // SmartFileRef constants
 sfrDefUserHome    = '~';   // Marker of User Home,   i.e. %UserProfile%    like ~\Dir\demo.txt
 sfrDefProgHome    = '~~';  // Marker of ProgramHome, i.e. main EXE folder  like ~~\Resource\Manual\crw-daq.htm
 sfrDefBrackets    = '*';   // Marker of default brackets usage
const                       // SmartFileSearch constants
 {$IFDEF WINDOWS}
 DefaultPathExt    = '.com;.exe;.bat;.cmd;.vbs;.vbe;.js;.jse;.wsf;.wsh';
 {$ELSE}
 DefaultPathExt    = '.sh;.bash';
 {$ENDIF ~WINDOWS}
 DefaultPathVarStr = 'CRW_DAQ_CONFIG_PATH;CRW_DAQ_SYS_PATH;PATH';
const                         // GetRealFilePathName Mode flags:
 fm_ReplaceSlash = $00000001; // Replace / to \
 fm_ApplyFExpand = $00000002; // Apply GetFullPathName to resolve .\, ..\, \\
 fm_ApplyGLongFN = $00000004; // Apply GetLongFileName to convert file names to long file names (but not path)
 fm_ApplyFind1st = $00000008; // Apply FindFirstFile to correct char case of file name and full path
 fm_ReadSymLinks = $00000010; // Use ReadSymLink to resolve symbolic links
const                                                             // Popular alphabets for Base32 encoders/decodes
 Base32_Alphabet_base32    = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567='; // [1] RFC4648 table 3 - default for GNU/base32 = 0
 Base32_Alphabet_base32hex = '0123456789ABCDEFGHIJKLMNOPQRSTUV='; // [1] RFC4648 table 4 - first 16 like base16   = 1
 Base32_Alphabet_crockford = '0123456789ABCDEFGHJKMNPQRSTVWXYZ';  // [3] Popular alternative, more human-readable = 2
 Base32_Alphabet_zooko     = 'ybndrfg8ejkmcpqxot1uwisza345h769';  // [4] Popular alternative, very human-readable = 3
 Base32_Alphabet_Alias_0   = '0 base32 rfc4648.3';                // Alias names of alphabet 0 - base32
 Base32_Alphabet_Alias_1   = '1 base32hex rfc4648.4';             // Alias names of alphabet 1 - base32hex
 Base32_Alphabet_Alias_2   = '2 crockford';                       // Alias names of alphabet 2 - crockford
 Base32_Alphabet_Alias_3   = '3 zooko human nice zbase32';        // Alias names of alphabet 3 - zooko
 Base32_Alphabet_Default   = Base32_Alphabet_zooko;               // Default alphabet
const                              // nAlphabet identifiers
 Base32_Alphabet_Id_base32    = 0; // Identify base32
 Base32_Alphabet_Id_base32hex = 1; // Identify base32hex
 Base32_Alphabet_Id_crockford = 2; // Identify crockford
 Base32_Alphabet_Id_zbase32   = 3; // Identify zbase32
 Base32_Alphabet_Id_Default   = 3; // Default alphabet ID
const ////////////////////// Special ASCII chars
 ASCII_NUL  = char($00);  // -- Null
 ASCII_SOH  = char($01);  // ^A Start Of Heading
 ASCII_STX  = char($02);  // ^B Start of TeXt
 ASCII_ETX  = char($03);  // ^C End of TeXt
 ASCII_EOT  = char($04);  // ^D End Of Transmission
 ASCII_ENQ  = char($05);  // ^E ENQuiry
 ASCII_ACK  = char($06);  // ^F ACKnowledge
 ASCII_BEL  = char($07);  // ^G Bell
 ASCII_BS   = char($08);  // ^H BackSpace
 ASCII_HT   = char($09);  // ^I Horizontal Tabulation
 ASCII_LF   = char($0A);  // ^J Line Feed
 ASCII_VT   = char($0B);  // ^K Vertical Tabulation
 ASCII_FF   = char($0C);  // ^L Form Feed
 ASCII_CR   = char($0D);  // ^M Carriage Return
 ASCII_SO   = char($0E);  // ^N Shift out
 ASCII_SI   = char($0F);  // ^O Shift in
 ASCII_DLE  = char($10);  // ^P Data Link Escape
 ASCII_DC1  = char($11);  // ^Q Data Control 1
 ASCII_DC2  = char($12);  // ^R Data Control 2
 ASCII_DC3  = char($13);  // ^S Data Control 3
 ASCII_DC4  = char($14);  // ^T Data Control 4
 ASCII_NAK  = char($15);  // ^U Negative AcKnowledge
 ASCII_SYN  = char($16);  // ^V SYNcronize idle
 ASCII_ETB  = char($17);  // ^W End of Transmission Block
 ASCII_CAN  = char($18);  // ^X CANcel
 ASCII_EM   = char($19);  // ^Y End of Medium
 ASCII_SUB  = char($1A);  // ^Z SUBstitute
 ASCII_ESC  = char($1B);  // -- ESCape
 ASCII_FS   = char($1C);  // -- File Separator
 ASCII_GS   = char($1D);  // -- Group Separator
 ASCII_RS   = char($1E);  // -- Record Separator
 ASCII_US   = char($1F);  // -- Unit Separator
 ASCII_DEL  = char($7F);  // -- Delete
 ASCII_SP   = char($20);  // -- Space
 ASCII_NBSP = char($A0);  // -- NO-BREAK SPACE
 /////////////////////////// another magic chars
 ASCII_TAB  = ASCII_HT;   // Horizontal Tabulation, synonym
 ASCII_XON  = ASCII_DC1;  // Uses for RS-232 software flow control
 ASCII_XOFF = ASCII_DC3;  // Uses for RS-232 software flow control
 NBSP       = ASCII_NBSP; // NO-BREAK SPACE
// CR         = ASCII_CR;   // Carriage Return
// LF         = ASCII_LF;   // Line Feed
// FF         = ASCII_FF;   // Form Feed
// TAB        = ASCII_HT;   // Tabulator
 QuoteMark  = '"';        // "Quotation Mark" as defined in Unicode also known as "quote marks"
 Apostrophe = '''';       // "Apostrophe"     as defined in Unicode also known as "apostrophe-quote" , "apl quote"
 // Наборы символов
 JustSpaces = [#0..' '];           // Uses by Trim, IsEmptyStr etc.
 JustBlanks = [ASCII_HT,' '];      // Space chars uses as blank
 EolnDelims = [ASCII_CR,ASCII_LF]; // EndOfLine delimiters
const               // Lexeme POSIX-class     RegExp/Pascal equivalent              Comment
 lex_Ansi    = 0;   // Lexeme is ANSI string  length(s)>0                           any non-zero string (s<>'')
 lex_Utf8    = 1;   // Lexeme is UTF8 string  --                                    any valid UTF8 encoded string
 lex_Name    = 2;   // Lexeme is name,        --                                    [:word:] and not start with digit
 lex_Word    = 3;   // Lexeme [:word:]        [[:alnum:]_]                          word of letters,digits,underscore
 lex_Blank   = 4;   // Lexeme [:blank:]       [ \t]                                 spaces and tabulations
 lex_Space   = 5;   // Lexeme [:space:]       [[:blank:]\v\r\n\f]                   space chars
 lex_Cntrl   = 6;   // Lexeme [:cntrl:]       [\x00-\x1F\x7F]                       control chars
 lex_Alpha   = 7;   // Lexeme [:alpha:]       [[:upper:][:lower:]]                  latin letters
 lex_Lower   = 8;   // Lexeme [:lower:]       [a-z]                                 lower case latin letters
 lex_Upper   = 9;   // Lexeme [:upper:]       [A-Z]                                 upper case latin letters
 lex_Digit   = 10;  // Lexeme [:digit:]       [0-9]                                 digits
 lex_Alnum   = 11;  // Lexeme [:alnum:]       [[:alpha:][:digit:]]                  letters,digits
 lex_xDigit  = 12;  // Lexeme [:xdigit:]      [[:digit:]A-Fa-f]                     hexadecimal digits
 lex_Punct   = 13;  // Lexeme [:punct:]       [!"#$%&'()*+,-./:;<=>?@[\\\]_`{|}~]   punctuation
 lex_Print   = 14;  // Lexeme [:print:]       [\x20-\x7E], i.e. [[:graph:] ]        printable chars with space
 lex_Graph   = 15;  // Lexeme [:graph:]       [\x21-\x7E]                           printable chars
 lex_Ascii   = 16;  // Lexeme [:ascii:]       [\x00-\x7F]                           ASCII characters
 lex_iParam  = 17;  // Lexeme is Integer      $hex or decimal integer               valid integer parameter
 lex_fParam  = 18;  // Lexeme is Float        float value                           valid float   parameter
 lex_sParam  = 19;  // Lexeme is String       "string param" or StringParam         valid string  parameter
 lex_Base64  = 20;  // Lexeme is Base64       [0-9A-Za-z\+/=]*                      valid Base64 data string
 lex_FsmName = 21;  // Lexeme is FSM name                                           name with [&:-] allowed
 lex_SmiName = 22;  // Lexeme is SMI name                                           name compatible with SMI
 lex_DimName = 23;  // Lexeme is DIM name                                           name compatible with DIM
 lex_SqlName = 24;  // Lexeme is SQL name     [a-zA-Z][0-9_a-zA-Z]*                 name compatible with SQL GOST R ISO/MEK 9075-93
 lex_Fbdname = 25;  // ---- Fireberd name     [a-zA-Z][0-9_\$a-zA-Z]*               name compatible with SQL Firebird
 lex_Section = 26;  // ---- Section name      [\[][^[:cntrl:]]*[\]]                 [Section Name]
 lex_AtCall  = 27;  // ---- @Command          [@][^[:cntrl:] ]                      Call of @Command ... (fast short  test)
 lex_AtCmnd  = 28;  // ---- @Command          [@][\t^[:cntrl:]]+                    Text of @Command ... (more strict test)
const                       // CookieScan Mode flags:
 csm_CaseSense = $00000100; // Name comparison is case sensitive
 csm_SkipTrimN = $00000200; // Don't Trim name before comparison
 csm_SkipTrimV = $00000400; // Don't Trim result Value
 csm_Default   = 0;         // Default is case insensitive, trim name and value
const                        // IsOption flags
 CmdOptionChars = ['-','/']; // Chars which uses for command options like -a or /a
 om_NoSlash     = 1;         // Option mode flag disable slash char in options (as Linix does)
 om_UseCase     = 2;         // Option mode flag to use case sensitive compare (as Linux does)
 {$IFDEF WINDOWS}
 om_Default     = 0;         // Option mode uses by default (Windows).
 {$ELSE}
 om_Default     = 3;         // Option mode uses by default (Unix).
 {$ENDIF}
const                             // Constants uses by UnifyAlias,UnifyFileAlias
 ua_None     = $0000;             // No conversions
 ua_Upper    = $0001;             // Convert to upper
 ua_Lower    = $0002;             // Convert to lower
 ua_TrimL    = $0004;             // Trim Left
 ua_TrimR    = $0008;             // Trim Right
 ua_DirSep   = $0010;             // Convert directory separators - / or \
 ua_FExpand  = $0020;             // Apply FExpand
 ua_GetReal  = $0040;             // Apply GetRealFilePathName for Windows
 ua_NameDef  = $0100;             // Use default for names
 ua_FileDef  = $0200;             // Use default for files
 ua_SectDef  = $0400;             // Use default for sections
 ua_Trim     = ua_TrimL+ua_TrimR; // Trim Left and Right
 ua_Case     = ua_Upper+ua_Lower; // Mask for Up/Lo case
 ua_ModeDef  = ua_NameDef+ua_FileDef+ua_SectDef;
 ua_FileLow  = ua_FileDef+ua_Lower;
 ua_RealPath = ua_FileDef+ua_GetReal;
function ua_FileDefLow:Integer; // ua_FileLow if not FileNameCaseSensitive

const // Константы диалогов ModalResult.
 mrNone            = 0;
 mrOk              = idOk;
 mrCancel          = idCancel;
 mrAbort           = idAbort;
 mrRetry           = idRetry;
 mrIgnore          = idIgnore;
 mrYes             = idYes;
 mrNo              = idNo;
 mrAll             = mrNo + 1;
 mrNoToAll         = mrAll + 1;
 mrYesToAll        = mrNoToAll + 1;
const // Константы для файлов
 CREATE_NEW        = 1;
 CREATE_ALWAYS     = 2;
 OPEN_EXISTING     = 3;
 OPEN_ALWAYS       = 4;
 TRUNCATE_EXISTING = 5;
const // Флаги функции ScanVar.
 svNoCase          = 0;                                 // Регистр Format не преобразуется.
 svUpCase          = $0001;                             // Format преобразуется в верхний регистр.
 svLoCase          = $0002;                             // Format преобразуется в нижний регистр.
 svCaseMask        = svUpCase+svLoCase;                 // Маска для преобразования регистра.
 svLSpace          = $0004;                             // Перед именем должен стоять разделитель.
 svRSpace          = $0008;                             // После имени должен стоять разделитель.
 svSpaces          = svLSpace+svRSpace;                 // До и после имени - разделитель (поиск по полному слову).
 svOrigin          = $0010;                             // Каждое имя ищется с начала строки.
 svAsIs            = 0;                                 // Чтение без преобразований, по порядку.
 svConfig          = svUpCase+svSpaces+svOrigin;        // Стандартный способ разбора ini-файлов.
 svConfigNC        = svConfig and not svCaseMask;       // Разбор ini-файлов с сохранением регистра.
 // Флаги функции ExtractTextSection.
 efNoCase          = 0;                                 // Регистр не преобразуется.
 efUpCase          = $0001;                             // Символы преобразуются в нижний  регистр.
 efLoCase          = $0002;                             // Символы преобразуются в верхний регистр.
 efCaseMask        = efUpCase+efLoCase;                 // Маска для преобразования регистра.
 efLTrim           = $0004;                             // Удалять незначащие пробелы слева.
 efRTrim           = $0008;                             // Удалять незначащие пробелы справа.
 efTrimMask        = efLTrim+efRTrim;                   // Удалять незначащие пробелы слева и справа.
 efDelCom          = $0010;                             // Удалять комментарии.
 efAsIs            = 0;                                 // Никаких преобразований.
 efConfig          = efUpCase+efLTrim+efRTrim+efDelCom; // Стандартный способ чтения ini-файлов.
 esComments        = [';'];                             // Разделители комментариев в ini-файлах.
 // Стандартные отладочные файлы DebugOut.
 stdfDebug         = 0;                                 // Стандартный журнал отладки программы.
 stdfError         = 1;                                 // Стандартный журнал ошибок программы.
 stdfReadIniLog    = 2;                                 // Стандартный журнал чтения INI файла.
 stdfSound         = 3;                                 // Стандартный журнал звуковой системы.
 // Наборы символов для синтаксического анализа.
 DirDelimiters     = ['\','/'];                         // Разделители каталогов.
 DosDelimiters     = ['\','/',':'];                     // Разделители каталогов и дисков.
 ScanSpaces        = [' ',ASCII_Tab,ASCII_CR,ASCII_LF,',',';','=']; // Разделители при разборе слов.
 ScanWordDelims    = [ASCII_CR,';'];                    // Разделители строк форматирования.
 WordSet           = ['_','a'..'z','A'..'Z','0'..'9'];  // Символы слов
 // Флаги функции CurveFlags.
 cfInvalid         = $00000001;                         // Кривая не существует.
 cfNoData          = $00000002;                         // В кривой нет точек.
 cfNoComment       = $00000004;                         // В кривой нет комментария.
 cfTooSmall        = $00000008;                         // Кривая имеет мало точек.
 cfNotSortedX      = $00000010;                         // Кривая не отсортирована по x.
 cfNotSortedY      = $00000020;                         // Кривая не отсортирована по y.
 cfDuplicatesX     = $00000040;                         // Кривая содержит точки с одинаковым значением x.
 cfDuplicatesY     = $00000080;                         // Кривая содержит точки с одинаковым значением y.
 cfNotEquiDistantX = $00000100;                         // Кривая имеет непостоянный шаг по x.
 cfNotEquiDistantY = $00000200;                         // Кривая имеет непостоянный шаг по y.
 cfHaveZeroX       = $00000400;                         // Кривая имеет точки с x=0.
 cfHaveZeroY       = $00000800;                         // Кривая имеет точки с y=0.
 cfHavePozitivX    = $00001000;                         // Кривая имеет точки с x>0.
 cfHavePozitivY    = $00002000;                         // Кривая имеет точки с y>0.
 cfHaveNegativX    = $00004000;                         // Кривая имеет точки с x<0.
 cfHaveNegativY    = $00008000;                         // Кривая имеет точки с y<0.
 cfNaNX            = $00010000;                         // Кривая имеет точки с x=NAN.
 cfNaNY            = $00020000;                         // Кривая имеет точки с y=NAN.
 cfInfX            = $00040000;                         // Кривая имеет точки с x=INF.
 cfInfY            = $00080000;                         // Кривая имеет точки с y=INF.
 cfNanXY           = cfNanX+cfNanY;
 cfInfXY           = cfInfX+cfInfY;
 cfNanInf          = cfNanXY+cfInfXY;
 // Флаги функции CurveSort
 sfPrimaryY        = $0001;                             // Основной ключ сортировки - Y, а не Х, как обычно
 sfReversPrimary   = $0002;                             // Сортировка в обратном порядке по основному ключу
 sfReversSecondary = $0004;                             // Сортировка в обратном порядке по дополнительному ключу
 sfPack            = $0008;                             // Упаковка точек с одинаковым значением ключей
 // Флаги событий DaqApi.putev
 evAnalog          = $00000000;                         // Событие направляется в AnalogOutput
 evDigital         = $00000001;                         // Событие направляется в DigitalOutput
 evNormal          = $00000000;                         // Событие обычное, может быть упаковано
 evImportant       = $00000002;                         // Событие важное, упаковка подавляется
 evNoCompress      = $00000000;                         // Событие не упаковывается
 evCompress        = $00000004;                         // Событие упаковывается
 evNoSpectral      = $00000000;                         // Событие не спектральное, добавляется в конец кривой
 evSpectral        = $00000008;                         // Событие спектральное, накапливается в спектре
 // Команды DaqApi.DaqCommand
 Daq_Cmd_Init      = 1;                                 // Команда "Начало работы"
 Daq_Cmd_Free      = 2;                                 // Команда "Завершение работы"
 Daq_Cmd_Poll      = 3;                                 // Команда "Опрос устройства"
 Daq_Cmd_User      = $1000;                             // Начало пользовательских команд
 // Коды методов минимизации
 mNelderMead             = 0;                           // NM  - Нелдер-Мид
 mDavidonFletcherPowell  = 1;                           // DFP - Давидон-Флетчер-Пауэлл
 mBroyden                = 2;                           // B   - Бройден
 mPearson2               = 3;                           // P2  - Пирсон 2
 mPearson3               = 4;                           // P3  - Пирсон 3
 mZoutendijk             = 5;                           // Z   - Заудентайк
 mSteward                = 6;                           // S   - Стюарт
 mFletcherReeves         = 7;                           // FR  - Флетчер-Ривз
 mDavidon2               = 8;                           // D2  - Давидон 2
 // Коды ошибок минимизации, minimization error code
 mecOk             = 0;                                 // Ok,minimum found
 mecBadDim         = 1;                                 // Bad dimension
 mecBadMethod      = 2;                                 // Unknown method
 mecBadArg         = 3;                                 // Invalid argument
 mecBadVal         = 4;                                 // Invalid value
 mecUserBreak      = 5;                                 // User breaks minimization process
 mecFail           = 6;                                 // Convergence not achieved
 mecInfPassed      = 7;                                 // Function value < user-defined infimum
 mecBounds         = 8;                                 // Out of available bounds
 mecInitErr        = 9;                                 // Init error
 mecDoneErr        = 10;                                // Finit error
 mecBadScale       = 11;                                // Invalid scale in GenMinScaled
 mecBadWeight      = 12;                                // Invalid weights in least squares
 mecDLLError       = 13;                                // Error loading DLL
 mecOutOfMem       = 14;                                // Detect out of memory
 // Коды LinearProgrammingBySimplex
 si_Ok               = 0;                               // решение получено, успех
 si_SolutionInfinite = 1;                               // решение неограничено
 si_ConditionsError  = 2;                               // ограничения несовместны, решения нет
 si_OutOfMemory      = -1;                              // Нет памяти для рабочих массивов
 si_InvalidInput     = -2;                              // среди входных данных есть NIL-указатели
 si_InvalidSignum    = -3;                              // ошибка в векторе знаков Signums
 si_InvalidRange     = -4;                              // ошибка размерности входных матриц
 si_Exception        = -5;                              // возникло исключение
 si_Input            = $0001;                           // печать входных данных
 si_Result           = $0002;                           // печать результата
 si_IterBasis        = $0004;                           // печатать базис на каждой итерации
 si_IterFun          = $0008;                           // печатать значение функции на каждой итерации
 si_IterSigm         = $0010;                           // печать симплекс-множителей
 si_Echo             = $8000;                           // вывод на экран сведений о итерации
 // Цветовые константы старого стиля CGA, см. CgaToRgbColor
 cgaBlack            = 0;
 cgaBlue             = 1;
 cgaGreen            = 2;
 cgaCyan             = 3;
 cgaRed              = 4;
 cgaMagenta          = 5;
 cgaBrown            = 6;
 cgaLightGray        = 7;
 cgaDarkGray         = 8;
 cgaLightBlue        = 9;
 cgaLightGreen       = 10;
 cgaLightCyan        = 11;
 cgaLightRed         = 12;
 cgaLightMagenta     = 13;
 cgaYellow           = 14;
 cgaWhite            = 15;
 // Коды AdamGetStatus. Сокращение от Request Status.
 rs_NotAvail       = 0;                                 // Сеть RS-485 недоступна
 rs_NoRequest      = 1;                                 // Запрос не возбуждался, канал свободен
 rs_WaitQueue      = 2;                                 // Запрос возбужден, но в сеть еще не передан
 rs_WaitAnswer     = 3;                                 // Запрос возбужден, передан в сеть, ожидание ответа
 rs_Answer         = 4;                                 // Пришел ответ на сообщение из сети RS-485
 rs_TimeOut        = 5;                                 // TimeOut : ответ не пришел в назначеннное время
 // Идентификаторы химических элементов таблицы Менделеева.
 ChemEl_H  = 1;    ChemEl_He = 2;     ChemEl_Li = 3;     ChemEl_Be = 4;     ChemEl_B  = 5;
 ChemEl_C  = 6;    ChemEl_N  = 7;     ChemEl_O  = 8;     ChemEl_F  = 9;     ChemEl_Ne = 10;
 ChemEl_Na = 11;   ChemEl_Mg = 12;    ChemEl_Al = 13;    ChemEl_Si = 14;    ChemEl_P  = 15;
 ChemEl_S  = 16;   ChemEl_Cl = 17;    ChemEl_Ar = 18;    ChemEl_K  = 19;    ChemEl_Ca = 20;
 ChemEl_Sc = 21;   ChemEl_Ti = 22;    ChemEl_V  = 23;    ChemEl_Cr = 24;    ChemEl_Mn = 25;
 ChemEl_Fe = 26;   ChemEl_Co = 27;    ChemEl_Ni = 28;    ChemEl_Cu = 29;    ChemEl_Zn = 30;
 ChemEl_Ga = 31;   ChemEl_Ge = 32;    ChemEl_As = 33;    ChemEl_Se = 34;    ChemEl_Br = 35;
 ChemEl_Kr = 36;   ChemEl_Rb = 37;    ChemEl_Sr = 38;    ChemEl_Y  = 39;    ChemEl_Zr = 40;
 ChemEl_Nb = 41;   ChemEl_Mo = 42;    ChemEl_Tc = 43;    ChemEl_Ru = 44;    ChemEl_Rh = 45;
 ChemEl_Pd = 46;   ChemEl_Ag = 47;    ChemEl_Cd = 48;    ChemEl_In = 49;    ChemEl_Sn = 50;
 ChemEl_Sb = 51;   ChemEl_Te = 52;    ChemEl_I  = 53;    ChemEl_Xe = 54;    ChemEl_Cs = 55;
 ChemEl_Ba = 56;   ChemEl_La = 57;    ChemEl_Ce = 58;    ChemEl_Pr = 59;    ChemEl_Nd = 60;
 ChemEl_Pm = 61;   ChemEl_Sm = 62;    ChemEl_Eu = 63;    ChemEl_Gd = 64;    ChemEl_Tb = 65;
 ChemEl_Dy = 66;   ChemEl_Ho = 67;    ChemEl_Er = 68;    ChemEl_Tm = 69;    ChemEl_Yb = 70;
 ChemEl_Lu = 71;   ChemEl_Hf = 72;    ChemEl_Ta = 73;    ChemEl_W  = 74;    ChemEl_Re = 75;
 ChemEl_Os = 76;   ChemEl_Ir = 77;    ChemEl_Pt = 78;    ChemEl_Au = 79;    ChemEl_Hg = 80;
 ChemEl_Tl = 81;   ChemEl_Pb = 82;    ChemEl_Bi = 83;    ChemEl_Po = 84;    ChemEl_At = 85;
 ChemEl_Rn = 86;   ChemEl_Fr = 87;    ChemEl_Ra = 88;    ChemEl_Ac = 89;    ChemEl_Th = 90;
 ChemEl_Pa = 91;   ChemEl_U  = 92;    ChemEl_Np = 93;    ChemEl_Pu = 94;    ChemEl_Am = 95;
 ChemEl_Cm = 96;   ChemEl_Bk = 97;    ChemEl_Cf = 98;    ChemEl_Es = 99;    ChemEl_Fm = 100;
 ChemEl_Md = 101;  ChemEl_No = 102;   ChemEl_Lr = 103;   ChemEl_Ku = 104;   ChemEl_Ns = 105;
 rfa_MinAtomId = ChemEl_H;
 rfa_MaxAtomId = ChemEl_Ns;
 // Parity control type
 NOPARITY           = 0;
 ODDPARITY          = 1;
 EVENPARITY         = 2;
 MARKPARITY         = 3;
 SPACEPARITY        = 4;
 // ByteSize = DataBits
 ONESTOPBIT         = 0;
 ONE5STOPBITS       = 1;
 TWOSTOPBITS        = 2;
const                           // AdaptFileName Mode flags:
 afnm_Trim  = 1;                // Use Trim
 afnm_Delim = 2;                // Fix directory delimiters (\,/).
 afnm_Drive = 4;                // Fix drive (c:) on Unix
 afnm_Lower = 8;                // Lower case on Unix
 afnm_Utf8  = 16;               // Use UTF8  on Unix
 afnm_NoDup = 32;               // Drop duplicates like //
 afnm_LibSo = 64;               // Add lib prefix on Unix
 afnm_DefNC = 1+2+4  +16+32+64; // Default but not use LoCase
 afnm_Def   = 1+2+4+8+16+32+64; // Default
const                      // URL encode/decode modes
 UrlAllowChars = [#33..#255]-['%','+',' ',ASCII_TAB,ASCII_CR,ASCII_LF,',',';','='];
 um_Safe          = 1;     // Do not raise exception
 um_StrictSpace   = 2;     // Use %20 instead of + to encode space char
 um_StrictPercent = 4;     // Use %25 instead of %% to encode % char
 um_StrictDecode  = 8;     // Use strict URL decoding
 um_Strict        = um_StrictSpace+um_StrictPercent+um_StrictDecode;
const ////////////////////// FSM entity type identifiers
 fsm_type_nil       = 0;  // Nil pointer, entity is not exist
 fsm_type_int       = 1;  // Parameter type Integer
 fsm_type_float     = 2;  // Parameter type Float
 fsm_type_string    = 3;  // Parameter type String
 fsm_type_parameter = 4;  // TFsmParameter, abstract
 fsm_type_manager   = 5;  // TFsmManager
 fsm_type_domain    = 6;  // TFsmDomain
 fsm_type_class     = 7;  // TFsmClass
 fsm_type_object    = 8;  // TFsmObject
 fsm_type_state     = 9;  // TFsmState
 fsm_type_action    = 10; // TFsmAction
 fsm_type_objectset = 11; // TFsmObjectSet
 fsm_type_function  = 12; // TFsmFunction
 fsm_type_parent    = 13; // TFsmParent, abstract
 fsm_type_entity    = 14; // TFsmEntity, abstract
 fsm_type_any       = 14; // Synonym of fsm_type_entity
type /////////////////////// Types of  TFsmEntity
 TFsmEntityTypeId   = fsm_type_nil..fsm_type_any;
const
 FsmParamTypes    = [fsm_type_int,fsm_type_float,fsm_type_string];
 FsmParentTypes   = [fsm_type_manager,fsm_type_domain,fsm_type_class,
                     fsm_type_object,fsm_type_state,fsm_type_action,
                     fsm_type_objectset,fsm_type_function];
 FsmAbstractTypes = [fsm_type_parameter,fsm_type_parent,fsm_type_entity];
 FsmValidTypes    = FsmParamTypes+FsmParentTypes;
 FsmAnyValidTypes = FsmValidTypes+FsmAbstractTypes;
 FsmAnyParamTypes = FsmParamTypes+[fsm_type_parameter];

const //////////////////// DB entity type identifiers
 db_type_nil             = 0;  // Nil pointer, entity is not exist
 db_type_connection      = 1;  // TDbConnection
 db_type_recordset       = 2;  // TDbRecordset
 db_type_command         = 3;  // TDbCommand
 db_type_parent          = 4;  // TDbParent, abstract
 db_type_entity          = 5;  // TDbEntity, abstract
 db_type_any             = db_type_entity;
 {
 Some special chars and codes.
 }
 _NUL          = 0;    // ASCII code Null
 _SOH          = 1;    // ASCII code Start of heading
 _STX          = 2;    // ASCII code Start of Text
 _ETX          = 3;    // ASCII code End of Text
 _EOT          = 4;    // ASCII code End of Transmission
 _ENQ          = 5;    // ASCII code Enquire
 _ACK          = 6;    // ASCII code Acknowledgement
 _BEL          = 7;    // ASCII code Bell
 _BS           = 8;    // ASCII code Backspace
 _HT           = 9;    // ASCII code Horizontal Tabulation
 _LF           = 10;   // ASCII code Line Feed
 _VT           = 11;   // ASCII code Vertical Tab
 _FF           = 12;   // ASCII code Form Feed
 _CR           = 13;   // ASCII code Carriage Return
 _SO           = 14;   // ASCII code Shift Out
 _SI           = 15;   // ASCII code Shift In
 _DLE          = 16;   // ASCII code Data Link Escape
 _DC1          = 17;   // ASCII code Device Control 1
 _DC2          = 18;   // ASCII code Device Control 2
 _DC3          = 19;   // ASCII code Device Control 3
 _DC4          = 20;   // ASCII code Device Control 4
 _NAK          = 21;   // ASCII code Negative Acknowledgment
 _SYN          = 22;   // ASCII code Synchronization
 _ETB          = 23;   // ASCII code End of Text Block
 _CAN          = 24;   // ASCII code Cancel
 _EM           = 25;   // ASCII code End of Medium
 _SUB          = 26;   // ASCII code Substitute
 _ESC          = 27;   // ASCII code Escape
 _FS           = 28;   // ASCII code File Separator
 _GS           = 29;   // ASCII code Group Separator
 _RS           = 30;   // ASCII code Record Separator
 _US           = 31;   // ASCII code Unit Separator
 _SP           = 32;   // ASCII code Space
 _DEL          = 127;  // ASCII code Delete
 _Space        = ' ';  // Char Space
 _NBSP         = ' ';  // U+00A0 NO-BREAK SPACE
 _Exclamation  = '!';  // Char Exclamation Mark
 _DoubleQuote  = '"';  // Char Double Quote or Quotation Mark
 _NumberSign   = '#';  // Char Number Sign
 _DollarSign   = '$';  // Char Dollar Sign
 _PercentSign  = '%';  // Char Percent Sign
 _Ampersand    = '&';  // Char Ampersand
 _SingleQuote  = ''''; // Char Single Quote or Apostrophe
 _LeftParent   = '(';  // Char Left Parenthesis
 _RightParent  = ')';  // Char Right Parenthesis
 _Asterisk     = '*';  // Char Asterisk
 _PlusSign     = '+';  // Char Plus Sign
 _Comma        = ',';  // Char Comma
 _MinusSign    = '-';  // Char Minus Sign or Hypen-Minus
 _Dot          = '.';  // Char Dot or Period or Full Stop
 _Slash        = '/';  // Char Slash or Solidus
 _Colon        = ':';  // Char Colon
 _SemiColon    = ';';  // Char SemiColon
 _LessSign     = '<';  // Char Less-Then Sign
 _EqualsSign   = '=';  // Char Equals Sign
 _GreaterSign  = '>';  // Char Greater-Then Sign
 _Question     = '?';  // Char Question Mark
 _AtSign       = '@';  // Char At Sign
 _LeftBracket  = '[';  // Char Left Square Bracket
 _BackSlash    = '\';  // Char BackSlash or Reverse Solidus
 _RightBracket = ']';  // Char Right Square Bracket
 _Caret        = '^';  // Char Caret or Circumflex Accent
 _Underscore   = '_';  // Char Underscore or Low Line
 _GraveAccent  = '`';  // Char Grave Accent
 _LeftCurly    = '{';  // Char Left Curly Bracket
 _PipeSign     = '|';  // Char Pipe Sign or Vertical Line
 _RightCurly   = '}';  // Char Right Curly Bracket
 _Tilde        = '~';  // Char Tilde
 _CurrencySign = '¤';  // Char Currency Sign
 _TripleDot    = '…';  // Char Horizontal Ellipsis or Triple Dot
 _Dagger       = '†';  // Char Dagger
 _DoubleDagger = '‡';  // Char Double Dagger
 _EuroSign     = '€';  // Char Euro Sign
 _PerMilleSign = '‰';  // Char Per Mille Sign
 _Bullet       = '•';  // Char Bullet
 _Trademark    = '®';  // Char Trademark Sign
 _SectionSign  = '§';  // Char Section Sign
 _Copyright    = '©';  // Char Copyright Sign
 _NotSign      = '¬';  // Char Not Sign
 _DegreeSign   = '°';  // Char Degree Sign
 _PlusMinus    = '±';  // Char Plus-Minus Sign
 _MicroSign    = 'µ';  // Char Micro Sign
 _PilcrowSign  = '¶';  // Char Pilcrow Sign
 _NumeroSign   = '№';  // Char Numero Sign
 {
 ClickWhat codes:
 }
 cw_Nothing    = 0;    // Nothing happened
 cw_MouseDown  = 1;    // Mouse button down
 cw_MouseUp    = 2;    // Mouse button up
 cw_MouseMove  = 3;    // Mouse cursor moved
 cw_KeyDown    = 4;    // Keyboard key down
 cw_KeyUp      = 5;    // Keyboard key up
 cw_NameList   = 'NOTHING,MOUSEDOWN,MOUSEUP,MOUSEMOVE,KEYDOWN,KEYUP,MOUSEWHEEL,MOUSEWHEELDOWN,MOUSEWHEELUP';
 cw_MouseWheel     = 6;    // Mouse wheel move
 cw_MouseWheelDown = 7;    // Mouse wheel down
 cw_MouseWheelUp   = 8;    // Mouse wheel up
 {
 Virtual Key codes:
 }
 VK_LBUTTON    = 1;    // 0x01 Left mouse button
 VK_RBUTTON    = 2;    // 0x02 Right mouse button
 VK_CANCEL     = 3;    // 0x03 Control-break processing
 VK_MBUTTON    = 4;    // 0x04 Middle mouse button (three-button mouse)
 VK_BACK       = 8;    // 0x08 BACKSPACE key
 VK_TAB        = 9;    // 0x09 TAB key
 VK_CLEAR      = 12;   // 0x0C CLEAR key
 VK_RETURN     = 13;   // 0x0D ENTER key
 VK_SHIFT      = 16;   // 0x10 SHIFT key
 VK_CONTROL    = 17;   // 0x11 CTRL key
 VK_MENU       = 18;   // 0x12 ALT key
 VK_PAUSE      = 19;   // 0x13 PAUSE key
 VK_CAPITAL    = 20;   // 0x14 CAPS LOCK key
 VK_ESCAPE     = 27;   // 0x1B ESC key
 VK_SPACE      = 32;   // 0x20 SPACEBAR
 VK_PRIOR      = 33;   // 0x21 PAGE UP key
 VK_NEXT       = 34;   // 0x22 PAGE DOWN key
 VK_END        = 35;   // 0x23 END key
 VK_HOME       = 36;   // 0x24 HOME key
 VK_LEFT       = 37;   // 0x25 LEFT ARROW key
 VK_UP         = 38;   // 0x26 UP ARROW key
 VK_RIGHT      = 39;   // 0x27 RIGHT ARROW key
 VK_DOWN       = 40;   // 0x28 DOWN ARROW key
 VK_SELECT     = 41;   // 0x29 SELECT key
 VK_PRINT      = 42;   // 0x2A PRINT key
 VK_EXECUTE    = 43;   // 0x2B EXECUTE key
 VK_SNAPSHOT   = 44;   // 0x2C PRINT SCREEN key
 VK_INSERT     = 45;   // 0x2D INS key
 VK_DELETE     = 46;   // 0x2E DEL key
 VK_HELP       = 47;   // 0x2F HELP key
 VK_F1         = 112;  // 0x70 F1 key
 VK_F2         = 113;  // 0x71 F2 key
 VK_F3         = 114;  // 0x72 F3 key
 VK_F4         = 115;  // 0x73 F4 key
 VK_F5         = 116;  // 0x74 F5 key
 VK_F6         = 117;  // 0x75 F6 key
 VK_F7         = 118;  // 0x76 F7 key
 VK_F8         = 119;  // 0x77 F8 key
 VK_F9         = 120;  // 0x78 F9 key
 VK_F10        = 121;  // 0x79 F10 key
 VK_F11        = 122;  // 0x7A F11 key
 VK_F12        = 123;  // 0x7B F12 key
 VK_NUMLOCK    = 144;  // 0x90 NUM LOCK key
 VK_SCROLL     = 145;  // 0x91 SCROLL LOCK key
 VK_LSHIFT     = 160;  // 0xA0 Left SHIFT key
 VK_RSHIFT     = 161;  // 0xA1 Right SHIFT key
 VK_LCONTROL   = 162;  // 0xA2 Left CONTROL key
 VK_RCONTROL   = 163;  // 0xA3 Right CONTROL key
 VK_LMENU      = 164;  // 0xA4 Left MENU key
 VK_RMENU      = 165;  // 0xA5 Right MENU key

type
 // Идентификаторы полей базы данных РФА
 TRfaFieldId=(                                          // Поля базы данных РФА:
  rfa_ATOM_ID,        rfa_ATOM_SYMBOL,                  // Номер и символ элемента
  rfa_ATOM_NAME_RUS,  rfa_ATOM_NAME_ENG,                // Название на русском/английском
  rfa_ATOM_GROUP,     rfa_ATOM_PERIOD,                  // Группа и период
  rfa_ATOM_COLOR,     rfa_CROSS_SECTION,                // Цвет и таблица сечений
  rfa_ATOM_WEIGHT,    rfa_ATOM_DENSITY,                 // Атомный вес и плотность
  rfa_EKAB,  rfa_EL1AB,  rfa_EL2AB,  rfa_EL3AB,         // Энергии абсорбции для K,L1,L2,L3-уровней линий ХРИ
  rfa_EKA,   rfa_IKA,                                   // Линия ХРИ K a1 + a2
  rfa_EKA1,  rfa_IKA1,                                  // Линия ХРИ K a1
  rfa_EKA2,  rfa_IKA2,                                  // Линия ХРИ K a2
  rfa_EKB,   rfa_IKB,                                   // Линия ХРИ K b1 + b2
  rfa_EKB1,  rfa_IKB1,                                  // Линия ХРИ K b1
  rfa_EKB2,  rfa_IKB2,                                  // Линия ХРИ K b2
  rfa_EKB3,  rfa_IKB3,                                  // Линия ХРИ K b3
  rfa_ELA12, rfa_ILA12,                                 // Линия ХРИ L3 a1 + a2
  rfa_ELA1,  rfa_ILA1,                                  // Линия ХРИ L3 a1
  rfa_ELA2,  rfa_ILA2,                                  // Линия ХРИ L3 a2
  rfa_ELB1,  rfa_ILB1,                                  // Линия ХРИ L2 b1
  rfa_ELB2,  rfa_ILB2,                                  // Линия ХРИ L3 b2
  rfa_ELB3,  rfa_ILB3,                                  // Линия ХРИ L1 b3
  rfa_ELB4,  rfa_ILB4,                                  // Линия ХРИ L1 b4
  rfa_ELB5,  rfa_ILB5,                                  // Линия ХРИ L3 b5
  rfa_ELB6,  rfa_ILB6,                                  // Линия ХРИ L3 b6
  rfa_ELB10, rfa_ILB10,                                 // Линия ХРИ L1 b10
  rfa_ELB15, rfa_ILB15,                                 // Линия ХРИ L3 b15
  rfa_ELB17, rfa_ILB17,                                 // Линия ХРИ L2 b17
  rfa_ELG1,  rfa_ILG1,                                  // Линия ХРИ L2 g1
  rfa_ELG2,  rfa_ILG2,                                  // Линия ХРИ L1 g2
  rfa_ELG3,  rfa_ILG3,                                  // Линия ХРИ L1 g3
  rfa_ELG4,  rfa_ILG4,                                  // Линия ХРИ L1 g4
  rfa_ELG4_, rfa_ILG4_,                                 // Линия ХРИ L1 g4_
  rfa_ELG5,  rfa_ILG5,                                  // Линия ХРИ L2 g5
  rfa_ELG6,  rfa_ILG6,                                  // Линия ХРИ L2 g6
  rfa_ELG8,  rfa_ILG8,                                  // Линия ХРИ L2 g6
  rfa_ELG11, rfa_ILG11,                                 // Линия ХРИ L1 g11
  rfa_ELmu,  rfa_ILmu,                                  // Линия ХРИ L2 Mu
  rfa_ELnu,  rfa_ILnu,                                  // Линия ХРИ L2 Nu
  rfa_ELL,   rfa_ILL,                                   // Линия ХРИ L3 L
  rfa_RSRV01, rfa_RSRV02, rfa_RSRV03, rfa_RSRV04,       // Зарезервировано на будущее
  rfa_RSRV05, rfa_RSRV06, rfa_RSRV07, rfa_RSRV08,       // Зарезервировано на будущее
  rfa_RSRV09, rfa_RSRV10, rfa_RSRV11, rfa_RSRV12,       // Зарезервировано на будущее
  rfa_RSRV13, rfa_RSRV14, rfa_RSRV15, rfa_RSRV16,       // Зарезервировано на будущее
  rfa_RSRV17, rfa_RSRV18, rfa_RSRV19, rfa_RSRV20,       // Зарезервировано на будущее
  rfa_RSRV21, rfa_RSRV22, rfa_RSRV23, rfa_RSRV24,       // Зарезервировано на будущее
  rfa_RSRV25, rfa_RSRV26, rfa_RSRV27, rfa_RSRV28,       // Зарезервировано на будущее
  rfa_RSRV29, rfa_RSRV30, rfa_RSRV31, rfa_RSRV32,       // Зарезервировано на будущее
  rfa_RSRV33, rfa_RSRV34, rfa_RSRV35, rfa_RSRV36,       // Зарезервировано на будущее
  rfa_RSRV37, rfa_RSRV38, rfa_RSRV39, rfa_RSRV40,       // Зарезервировано на будущее
  rfa_RSRV41, rfa_RSRV42, rfa_RSRV43, rfa_RSRV44,       // Зарезервировано на будущее
  rfa_RSRV45, rfa_RSRV46, rfa_RSRV47, rfa_RSRV48,       // Зарезервировано на будущее
  rfa_RSRV49, rfa_RSRV50, rfa_RSRV51, rfa_RSRV52,       // Зарезервировано на будущее
  rfa_RSRV53, rfa_RSRV54, rfa_RSRV55, rfa_RSRV56,       // Зарезервировано на будущее
  rfa_RSRV57, rfa_RSRV58, rfa_RSRV59, rfa_RSRV60,       // Зарезервировано на будущее
  rfa_RSRV61, rfa_RSRV62, rfa_RSRV63, rfa_RSRV64,       // Зарезервировано на будущее
  rfa_RSRV65, rfa_RSRV66, rfa_RSRV67, rfa_RSRV68,       // Зарезервировано на будущее
  rfa_RSRV69, rfa_RSRV70, rfa_RSRV71, rfa_RSRV72,       // Зарезервировано на будущее
  rfa_RSRV73, rfa_RSRV74, rfa_RSRV75, rfa_RSRV76,       // Зарезервировано на будущее
  rfa_RSRV77, rfa_RSRV78, rfa_RSRV79, rfa_RSRV80,       // Зарезервировано на будущее
  rfa_RSRV81, rfa_RSRV82, rfa_RSRV83, rfa_RSRV84,       // Зарезервировано на будущее
  rfa_RSRV85, rfa_RSRV86, rfa_RSRV87, rfa_RSRV88,       // Зарезервировано на будущее
  rfa_RSRV89, rfa_RSRV90, rfa_RSRV91, rfa_RSRV92,       // Зарезервировано на будущее
  rfa_RSRV93, rfa_RSRV94, rfa_RSRV95, rfa_RSRV96,       // Зарезервировано на будущее
  rfa_RSRV97, rfa_RSRV98, rfa_RSRV99, rfa_RSRV100,      // Зарезервировано на будущее
  rfa_KEMISSION,   rfa_L31EMISSION, rfa_L32EMISSION,    // Факторы эмиссии линий ХРИ
  rfa_L33EMISSION, rfa_L21EMISSION, rfa_L22EMISSION,    // Факторы эмиссии линий ХРИ
  rfa_ATOMS_MM1,                                        // атомная масса наиболее распространенного изотопа
  rfa_ATOMS_M1,                                         // атомный вес наиболее распространенного изотопа
  rfa_ATOMS_M2,                                         // атомный вес естественной смеси изотопов
  rfa_ATOMS_RHO,                                        // плотность г/куб.см
  rfa_ATOMS_ATRHO,                                      // атомная плотность, 10**22 атом/куб.см
  rfa_ATOMS_VFERMI,                                     // скорость Ферми в единицах скорости Бора
  rfa_ATOMS_LFCTR,                                      // множитель длины экранирования
  rfa_STRAG_S1, rfa_STRAG_S2, rfa_STRAG_S3, rfa_STRAG_S4,   // Коэффициенты для вычисления страгглинга
  rfa_PSTOP_P1, rfa_PSTOP_P2, rfa_PSTOP_P3, rfa_PSTOP_P4,   // Коэффициенты протонного торможения
  rfa_PSTOP_P5, rfa_PSTOP_P6, rfa_PSTOP_P7, rfa_PSTOP_P8,   // Коэффициенты протонного торможения
  rfa_ELEMPOS_X, rfa_ELEMPOS_Y,                         // Положения кнопок для отображения таблицы Менделеева
  rfa_INVALID_FIELD                                     // Такого поля быть не должно
 );
 TRfaFieldIdSet = set of TRfaFieldId;                   // Для наборов полей базы данных
 TRfaCrossSectionFieldId = (                            // Поля таблиц сечений
  rfa_ENERGY,                                           // Энергия в кэв
  rfa_PHOTO,                                            // Сечение фотоэффекта
  rfa_COHER,                                            // Сечение когерентного рассеяния
  rfa_INCOHER,                                          // Сечение некогерентного рассеяния
  rfa_TOTAL                                             // Полное сечение
 );
 TRfaXRayLine = packed record Energy,Height:Double; end;// Описывает линию ХРИ
 // Интерфейс инкапсулирует функции для доступа к базе данных РФА.
 IRfaDataBaseWrap = interface(IPluginWrap) [SIID_IRfaDataBaseWrap]
  //
  // 0    - Возвращает True, если атомный номер доступен, то есть если он допустим и база данных загружена.
  //
  function  GetAccessible(AtomId:Integer):Boolean;
  //
  // 1    - Дает доступ к полям базы данных как к числу; нечисловые поля возвращают длину строки/таблицы.
  //
  function  GetAsDouble(AtomId:Integer; FieldId:TRfaFieldId):Double;
  //
  // 2    - Дает доступ к полям базы данных как к строкам; числовые поля преобразуются в строку.
  //
  function  GetAsString(AtomId:Integer; FieldId:TRfaFieldId):LongString;
  //
  // 3    - Возвращает энергию и интенсивность линии ХРИ. Возвращает ноль, если LineId неверное.
  //
  function  GetXRayLine(AtomId:Integer; LineId:TRfaFieldId):TRfaXRayLine;
  //
  // 4    - Возвращает химический символ атома.
  //
  function  GetAtomSymbol(AtomId:Integer):LongString;
  //
  // 5    - Возвращает символьное имя поля таблицы РФА.
  //
  function  GetFieldName(FieldId:TRfaFieldId):LongString;
  //
  // 6    - Возвращает название линии ХРИ.
  //
  function  GetLineName(LineId:TRfaFieldId):LongString;
  //
  // 7    - Возвращает длину таблицы сечений для данного элемента.
  //
  function  GetCrossSectionTableCount(AtomId:Integer):Integer;
  //
  // 8    - Дает доступ к таблице сечений.
  //
  function  GetCrossSectionTable(AtomId:Integer; aChan:TRfaCrossSectionFieldId;
                                 aIndex:Integer):Double;
  //
  // 9    - Находит номер атома по его химическому символу или возвращает 0.
  //
  function  FindAtom(const aSymbol:LongString):Integer;
  //
  // 10   - Находит линию ХРИ по ее названию или возвращает rfa_INVALID_FIELD.
  //
  function  FindLine(const aLineName:LongString):TRfaFieldId;
  //
  // 11   - Вычисляет сечение, используя логарифмическую интерполяцию таблицы сечений.
  //
  function  FindCrossSection(AtomID:Integer; CrossId:TRfaCrossSectionFieldId;
                               Energy:Double):Double;
  //
  // Properties
  //
  property    Accessible[AtomId:Integer]                   : Boolean      read GetAccessible;
  property    AsDouble[AtomId:Integer;FieldId:TRfaFieldId] : Double       read GetAsDouble; default;
  property    AsString[AtomId:Integer;FieldId:TRfaFieldId] : LongString   read GetAsString;
  property    XRayLine[AtomId:Integer;LineId:TRfaFieldId]  : TRfaXRayLine read GetXRayLine;
  property    AtomSymbol[AtomId:Integer]                   : LongString   read GetAtomSymbol;
  property    FieldName[FieldId:TRfaFieldId]               : LongString   read GetFieldName;
  property    LineName[LineId:TRfaFieldId]                 : LongString   read GetLineName;
  property    CrossSectionTableCount[AtomId:Integer]       : Integer      read GetCrossSectionTableCount;
  property    CrossSectionTable[AtomId:Integer; Chan:TRfaCrossSectionFieldId; aIndex:Integer] : Double  read GetCrossSectionTable;
 end;

const
 rfa_FIRST_FIELD        = Low(TRfaFieldId);             // Первое допустимое поле
 rfa_LAST_FIELD         = Pred(rfa_INVALID_FIELD);      // Последнее допустимое поле
 // Различные наборы, идентифицирующие линии ХРИ.
 rfa_K_Series_Shorten   = [rfa_EKA,             rfa_EKB];
 rfa_K_Series_Perfect   = [rfa_EKA1, rfa_EKA2,  rfa_EKB1, rfa_EKB2, rfa_EKB3];
 rfa_L1_Series_Shorten  = [rfa_ELB3, rfa_ELB4,  rfa_ELB10, rfa_ELG2, rfa_ELG3, rfa_ELG4, rfa_ELG4_, rfa_ELG11];
 rfa_L1_Series_Perfect  = [rfa_ELB3, rfa_ELB4,  rfa_ELB10, rfa_ELG2, rfa_ELG3, rfa_ELG4, rfa_ELG4_, rfa_ELG11];
 rfa_L2_Series_Shorten  = [rfa_ELB1, rfa_ELB17, rfa_ELG1, rfa_ELG5, rfa_ELG6, rfa_ELG8, rfa_ELmu, rfa_ELnu];
 rfa_L2_Series_Perfect  = [rfa_ELB1, rfa_ELB17, rfa_ELG1, rfa_ELG5, rfa_ELG6, rfa_ELG8, rfa_ELmu, rfa_ELnu];
 rfa_L3_Series_Shorten  = [rfa_ELA12,           rfa_ELB2, rfa_ELB5, rfa_ELB6, rfa_ELB15, rfa_ELL];
 rfa_L3_Series_Perfect  = [rfa_ELA1,  rfa_ELA2, rfa_ELB2, rfa_ELB5, rfa_ELB6, rfa_ELB15, rfa_ELL];
 rfa_K_Series           = rfa_K_Series_Shorten  + rfa_K_Series_Perfect;
 rfa_L1_Series          = rfa_L1_Series_Shorten + rfa_L1_Series_Perfect;
 rfa_L2_Series          = rfa_L2_Series_Shorten + rfa_L2_Series_Perfect;
 rfa_L3_Series          = rfa_L3_Series_Shorten + rfa_L3_Series_Perfect;
 rfa_L_Series_Shorten   = rfa_L1_Series_Shorten + rfa_L2_Series_Shorten + rfa_L3_Series_Shorten;
 rfa_L_Series_Perfect   = rfa_L1_Series_Perfect + rfa_L2_Series_Perfect + rfa_L3_Series_Perfect;
 rfa_L_Series           = rfa_L_Series_Shorten + rfa_L_Series_Perfect;
 rfa_All_Series_Shorten = rfa_K_Series_Shorten + rfa_L_Series_Shorten;
 rfa_All_Series_Perfect = rfa_K_Series_Perfect + rfa_L_Series_Perfect;
 rfa_All_Series         = rfa_K_Series + rfa_L_Series;

type
 PByte           = ^Byte;
 PByteArray      = ^TByteArray;
 TByteArray      = packed array[0..(MaxInt div sizeof(Byte))-1]     of Byte;
 TCharArray      = packed array[0..(MaxInt div sizeof(Char))-1]     of Char;
 PShortInt       = ^ShortInt;
 PShortIntArray  = ^TShortIntArray;
 TShortIntArray  = packed array[0..(MaxInt div sizeof(ShortInt))-1] of ShortInt;
 PBoolean        = ^Boolean;
 PBooleanArray   = ^TBooleanArray;
 TBooleanArray   = packed array[0..(MaxInt div sizeof(Boolean))-1]  of Boolean;
 PWord           = ^Word;
 PWordArray      = ^TWordArray;
 TWordArray      = packed array[0..(MaxInt div sizeof(Word))-1]     of Word;
 PSmallInt       = ^SmallInt;
 PSmallIntArray  = ^TSmallIntArray;
 TSmallIntArray  = packed array[0..(MaxInt div sizeof(SmallInt))-1] of SmallInt;
 PLongWord       = ^LongWord;
 PLongWordArray  = ^TLongWordArray;
 TLongWordArray  = packed array[0..(MaxInt div sizeof(LongWord))-1] of LongWord;
 PDWord          = ^DWord;
 PDWordArray     = ^TDWordArray;
 TDWordArray     = packed array[0..(MaxInt div sizeof(DWord))-1]    of DWord;
 PLongInt        = ^LongInt;
 PLongIntArray   = ^TLongIntArray;
 TLongIntArray   = packed array[0..(MaxInt div sizeof(LongInt))-1]  of LongInt;
 PCardinal       = ^Cardinal;
 PCardinalArray  = ^TCardinalArray;
 TCardinalArray  = packed array[0..(MaxInt div sizeof(Cardinal))-1] of Cardinal;
 PInteger        = ^Integer;
 PIntegerArray   = ^TIntegerArray;
 TIntegerArray   = packed array[0..(MaxInt div sizeof(Integer))-1]  of Integer;
 PInt64          = ^Int64;
 PInt64Array     = ^TInt64Array;
 TInt64Array     = packed array[0..(MaxInt div sizeof(Int64))-1]    of Int64;
 PSingle         = ^Single;
 PSingleArray    = ^TSingleArray;
 TSingleArray    = packed array[0..(MaxInt div sizeof(Single))-1]   of Single;
 PDouble         = ^Double;
 PDoubleArray    = ^TDoubleArray;
 TDoubleArray    = packed array[0..(MaxInt div sizeof(Double))-1]   of Double;
 PComp           = ^Comp;
 PCompArray      = ^TCompArray;
 TCompArray      = packed array[0..(MaxInt div sizeof(Comp))-1]     of Comp;
 PExtended       = ^Extended;
 PExtendedArray  = ^TExtendedArray;
 TExtendedArray  = packed array[0..(MaxInt div sizeof(Extended))-1] of Extended;
 PPointerArray   = ^TPointerArray;
 TPointerArray   = packed array[0..(MaxInt div sizeof(Pointer))-1]  of Pointer;
 // Точки и прямоугольники
 TPoint2I        = packed record x,y : Integer; end;
 TPoint2D        = packed record x,y : Double;  end;
 TPoint3D        = packed record x,y,z : Double; end;
 TRect2I         = packed record a,b : TPoint2I; end;
 TRect2D         = packed record a,b : TPoint2D; end;
 PPoint2IArray   = ^TPoint2IArray;
 TPoint2IArray   = packed array[0..MaxInt div sizeof(TPoint2I)-1] of TPoint2I;
 PPoint2DArray   = ^TPoint2DArray;
 TPoint2DArray   = packed array[0..MaxInt div sizeof(TPoint2D)-1] of TPoint2D;
 PPoint3DArray   = ^TPoint3DArray;
 TPoint3DArray   = packed array[0..MaxInt div sizeof(TPoint3D)-1] of TPoint3D;
 PRect2IArray    = ^TRect2IArray;
 TRect2IArray    = packed array[0..MaxInt div sizeof(TRect2I)-1] of TRect2I;
 PRect2DArray    = ^TRect2DArray;
 TRect2DArray    = packed array[0..MaxInt div sizeof(TRect2D)-1] of TRect2D;
 // Для функций FindZero, FindMin и т.д.
 zmFunType       = function ( x : Double; Custom : Pointer ) : Double;
 // Для функции ForEachFile.
 TForEachFileAction     = procedure ( const FileName    : LongString;
                                      const FileDetails : TSearchRec;
                                            SubDirLevel : Integer;
                                        var Terminate   : Boolean;
                                            CustomData  : Pointer);
 // Для функции ForEachTexLine.
 TForEachTextLineAction = procedure ( const FileName    : LongString;
                                      const Line        : LongString;
                                            Count       : SizeInt;
                                        var Terminate   : Boolean;
                                            CustomData  : Pointer);
 // Целевая функция для процедур минимизации
 // n=размерность, x=аргумент, f=значение, g=градиент, Custom=пользовательские данные.
 TMinProblem  = function(n:Integer; var x:array of Double; var f:Double;
                                    var g:array of Double; Custom:Pointer):Integer;
 // Процедура оповещения для процедур минимизации
 TMinReport  = procedure(n:Integer; var x:array of Double; var f:Double;
                                    var g:array of Double; Custom:Pointer;
                                    Count:Integer; const Method,Comment:LongString);
 // Целевая функция для фитирования
 // t=аргумент, n=размерность параметров,  x=параметры, f=значение, g=градиент по параметрам,
 // Custom=пользовательские данные.
 TFitProblem = function(t:Double; n:Integer; var x:array of Double; var f:Double;
                                             var g:array of Double; Custom:Pointer):Integer;
 // Процедура оповещения для процедур фитирования
 TFitReport=procedure(m:Integer; const t,y,w:array of Double; n:Integer;
                      const x:array of Double; const f:Double; const g:array of Double;
                      Custom:Pointer; Count:Integer; const Method,Comment:LongString);
 // Функция сравнения для процедур сортировки и поиска.
 // Compare(Data,Key1,Key2) сравнивает ключи Key1,Key2 и возвращает
 // 1 если Key1^ > Key2^; 0 если Key1^ = Key2^; -1 если Key1^ < Key2^.
 // Через Data передаются пользовательские данные.
 TCompareFunction = function(Data,Key1,Key2:Pointer):Integer;
 // Функция ключа для процедур сортировки и поиска.
 // KeyOf(Data,j) возвращает ключ Data[j] с индексом j=0..Count-1 в массиве Data.
 TKeyOfFunction = function(Data:Pointer; i:Integer):Pointer;
 // Процедура перестановки для сортировки массива "на месте".
 // Swap(Data,i,j) меняет местами элементы i,j массива Data.
 TSwapProcedure = procedure(Data:Pointer; i,j:Integer);
 // Процедура для определения ядра оконного сглаживания
 TSmoothKernel = function(x:Double; CustomData:Pointer):Double;
 //
 // Класс TScriptInterpreter применяется для вычисления аналитически заданных
 // выражений и выполнения сценариев на языке Daq Script.
 //
 IScriptInterpreter = interface(IPluginWrap) [SIID_IScriptInterpreter]
  //
  // 0    - Чтение значения переменной.
  //
  function  GetValue(aName:PChar; var aValue:Double):Boolean;
  //
  // 1    - Запись значения переменной. Создание переменной.
  //
  function  SetValue(aName:PChar; var aValue:Double):Boolean;
  //
  // 2    - Очистка всех переменных.
  //
  procedure Clear;
  //
  // 3    - Вычисляет выражение.
  //
  function  Evaluate(aExpression:PChar):Integer;
  //
  // 4    - Возвращает текст сценария.
  //
  function  GetScript:LongString;
  //
  // 5    - Задает текст сценария.
  //
  procedure SetScript(const aScript:LongString);
  //
  // 6    - Выполняет сценарий Script.
  //
  function  RunScript:Integer;
  //
  // 7    - Возвращает статус последней операции.
  //
  function  Status:Integer;
  //
  // 8    - Возвращает расшифровку статуса.
  //
  function  StatusMessage:LongString;
  //
  // Properties
  //
  property  Script : LongString read GetScript write SetScript;
 end;
 //
 // Класс TPolynomWrap используется для построения полиномов методом наименьших
 // квадратов и может использоваться для параметрической аппроксимации гладких
 // данных. Полином задается в следующей форме:
 //  P(x) = P(z) = C[0]+C[1]*z+C[2]*z^2..., где z=(x-Center)/Scale
 // Сдвиг Center и масштаб Scale нужны, чтобы избежать вырожденности базисной
 // системы функций полинома в случае, когда массив данных смещен и растянут
 // по оси абсцисс. Оптимальный выбор состоит в том, чтобы после масштабирования
 // переменная z менялась в интервале [0..1].
 //
 IPolynomWrap = interface(IPluginWrap) [SIID_IPolynomWrap]
  //
  // 0    - Возвращает степень полинома.
  //
  function  GetPower:Integer;
  //
  // 1    - Задает степень полинома.
  //
  procedure SetPower(aPower:Integer);
  //
  // 2    - Возвращает коэффициент полинома.
  //
  function  GetCoeff(aIndex:Integer):Double;
  //
  // 3    - Задает коэффициент полинома.
  //
  procedure SetCoeff(aIndex:Integer; aValue:Double);
  //
  // 4    - Возвращает центр (смещение) полинома.
  //
  function  GetCenter:Double;
  //
  // 5    - Задает центр (смещение) полинома.
  //
  procedure SetCenter(aCenter:Double);
  //
  // 6    - Возвращает масштабный фактор полинома.
  //
  function  GetScale:Double;
  //
  // 7    - Задает масштабный фактор полинома.
  //
  procedure SetScale(aScale:Double);
  //
  // 8    - Возвращает значение полинома или значение производной с номером
  //        aDerivNum в точке at.
  //
  function  Get(at:Double; aDerivNum:Integer=0):Double;
  //
  // 9    - Находит методом наименьших квадратов коэффициетны полинома, которые
  //        аппроксимируют выборку (X[i],Y[i]), i=0..NumPoints-1 с единичными весами
  //
  function  Find(const X,Y:array of Double; NumPoints:Integer;
                 Eps:double=1E-14):boolean;
  //
  // 10   - Находит методом наименьших квадратов коэффициетны полинома, которые
  //        аппроксимируют выборку (X[i],Y[i]), i=0..NumPoints-1 с заданными
  //        весовыми факторами W[i].
  //
  function  FindWeight(const X,Y,W:array of Double; NumPoints:Integer;
                       Eps:double=1E-14):boolean;
  //
  // Properties
  //
  property  Power:Integer                read GetPower  write SetPower;
  property  Coeff[aIndex:Integer]:Double read GetCoeff  write SetCoeff; default;
  property  Center:Double                read GetCenter write SetCenter;
  property  Scale:Double                 read GetScale  write SetScale;
 end;
 //
 // Классы TLongIntVectorWrap,TDoubleVectorWrap,TLongIntMatrixWrap,TDoubleMatrixWrap
 // реализуют защищенные в смысле ошибок индексации и потоков векторы и матрицы.
 // Lock/UnLock - служит для входа-выхода в потокозащищенную область кода.
 // Origin      - номер первого индекса массива, по умолчанию ноль.
 // Length      - текущая длина вектора.
 // Rows        - текущее число строк матрицы.
 // Cols        - текущее число столбцов матрицы.
 // Exceptions  - разрешение генерации исключений при выходе индекса за диапазон.
 // Пример:
 //  var x:TDoubleVectorWrap; y:TDoubleMatrixWrap; i,j:Integer;
 //  x:=CreateDoubleVector(100);
 //  y:=CreateDoubleMatrix(100,100);
 //  x.Lock;
 //  y.Lock;
 //  try
 //   for i:=0 to x.Length-1 do x[i]:=i;
 //   for i:=0 to y.Rows-1 do for j:=0 to y.Cols-1 do y[i,j]:=x[i]*j;
 //  finally
 //   y.UnLock;
 //   x.UnLock;
 //  end;
 //  x.Free;
 //  y.Free;
 //
 ILongIntVectorWrap = interface(IPluginWrap) [SIID_ILongIntVectorWrap]
  procedure Lock;
  procedure UnLock;
  function  GetLength:LongInt;
  procedure SetLength(aLength:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aIndex:LongInt):LongInt;
  procedure SetItem(aIndex:LongInt; const aValue:LongInt);
  function  GetItemPtr(aIndex:LongInt):PLongInt;
  //
  // Properties
  //
  property    Length             : LongInt  read GetLength     write SetLength;
  property    Origin             : LongInt  read GetOrigin     write SetOrigin;
  property    Exceptions         : Boolean  read GetExceptions write SetExceptions;
  property    Item[i:LongInt]    : LongInt  read GetItem       write SetItem; default;
  property    ItemPtr[i:LongInt] : PLongInt read GetItemPtr;
 end;
 IDoubleVectorWrap = interface(IPluginWrap) [SIID_IDoubleVectorWrap]
  procedure Lock;
  procedure UnLock;
  function  GetLength:LongInt;
  procedure SetLength(aLength:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aIndex:LongInt):Double;
  procedure SetItem(aIndex:LongInt; const aValue:Double);
  function  GetItemPtr(aIndex:LongInt):PDouble;
  //
  // Properties
  //
  property    Length             : LongInt read GetLength     write SetLength;
  property    Origin             : LongInt read GetOrigin     write SetOrigin;
  property    Exceptions         : Boolean read GetExceptions write SetExceptions;
  property    Item[i:LongInt]    : Double  read GetItem       write SetItem; default;
  property    ItemPtr[i:LongInt] : PDouble read GetItemPtr;
 end;
 ILongIntMatrixWrap = interface(IPluginWrap) [SIID_ILongIntMatrixWrap]
  procedure Lock;
  procedure UnLock;
  function  GetRows:LongInt;
  procedure SetRows(aRows:LongInt);
  function  GetColumns:LongInt;
  procedure SetColumns(aColumns:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aRow,aColumn:LongInt):LongInt;
  procedure SetItem(aRow,aColumn:LongInt; const aValue:LongInt);
  function  GetItemPtr(aRow,aColumn:LongInt):PLongInt;
  //
  // Properties
  //
  property  Rows                 : LongInt  read GetRows       write SetRows;
  property  Cols                 : LongInt  read GetColumns    write SetColumns;
  property  Origin               : LongInt  read GetOrigin     write SetOrigin;
  property  Exceptions           : Boolean  read GetExceptions write SetExceptions;
  property  Item[r,c:LongInt]    : LongInt  read GetItem       write SetItem; default;
  property  ItemPtr[r,c:LongInt] : PLongInt read GetItemPtr;
 end;
 IDoubleMatrixWrap = interface(IPluginWrap) [SIID_IDoubleMatrixWrap]
  procedure Lock;
  procedure UnLock;
  function  GetRows:LongInt;
  procedure SetRows(aRows:LongInt);
  function  GetColumns:LongInt;
  procedure SetColumns(aColumns:LongInt);
  function  GetOrigin:LongInt;
  procedure SetOrigin(aOrigin:LongInt);
  function  GetExceptions:Boolean;
  procedure SetExceptions(aExcepts:Boolean);
  function  GetItem(aRow,aColumn:LongInt):Double;
  procedure SetItem(aRow,aColumn:LongInt; const aValue:Double);
  function  GetItemPtr(aRow,aColumn:LongInt):PDouble;
  //
  // Properties
  //
  property  Rows                 : LongInt  read GetRows       write SetRows;
  property  Cols                 : LongInt  read GetColumns    write SetColumns;
  property  Origin               : LongInt  read GetOrigin     write SetOrigin;
  property  Exceptions           : Boolean  read GetExceptions write SetExceptions;
  property  Item[r,c:LongInt]    : Double   read GetItem       write SetItem; default;
  property  ItemPtr[r,c:LongInt] : PDouble  read GetItemPtr;
 end;
 //
 // Класс TSVDWrap - оболочка для выполнения сингулярного разложения матриц,
 // решения линейных уравнений в смысле наименьших квадратов и т.д.
 // Теория.
 // Сингулярное разложение (singular values decomposition,SVD) матрицы A размера
 // M строк на N столбцов A[M*N] - это разложение на произведение трех матриц
 // U[M*M],W[M*N],V[N*N] вида A=U*W*V', где V' означает "V транспонированное".
 // Столбцы U[M*M], "left singular vectors",  являются собственными векторами A*A'.
 // Столбцы V[N*N], "right singular vectors", являются собственными векторами A'*A.
 // Матрицы U и V унитарные, это значит, U'*U=I(M), V'*V=I(N), где I(N),I(M)
 // обозначает единичную матрицу размера N или M. Матрица W[M*N] - диагональная,
 // так что W[i,j]=0 при i<>j. Диагональные элементы W[i]=W[i,i],i=1..N - это
 // сингулярные числа (singular values) матрицы A. Их квадраты Sqr(W[i]) являются
 // собственными числам матрицы A'*A. Сингулярные числа всегда неотрицательны,
 // то есть W[i]>=0. Сингулярное разложение существует для любых матриц, в том
 // числе вырожденных. Если матрица невырождена, все W[i]>0. Если матрица вырождена,
 // ее ранг равен числу ненулевых W[i]. На практике, где играют роль ошибки вычислений,
 // используется малое число Eps для определения "нулевых" сингулярных чисел.
 // Число Cond(A)=Max(W[i])/Min(W[i]) называется "числом обусловленности".
 // Чем больше значение этого числа, тем матрица ближе к вырожденной, такие
 // матрицы называют плохо обусловленными.
 // Применение в МНК.
 // В методе наименьших квадратов возникают линейные системы A*X=Y, где
 // A[M*N],X[N],Y[M] и M>=N. В точном смысле мы ищем вектор X, минимизирующий
 // сумму квадратов отклонений, или длину вектора невязки R=A*X-Y, то есть
 // ищется минимум квадратичной функции {R'*R}. Что дает сингулярное разложение?
 // Во-первых, |R|=|U'*R|, так как умножение на унитарную матрицу не меняет
 // длину вектора. Возьмем сингулярное разложение A=U*W*V'. Введем новый вектор
 // переменных Z=V'*X и вектор констант D=U'*Y.
 // Имеем: |R|=|U'*(A*X-Y)|=|U'*U*W*V'*X-U'*Y|=|W*Z-D|, так как U'*U=I.
 // Минимизация стала тривиальной, так как W - диагональная матрица.
 // Решение Z=pinv(W,Eps)*D, где pinv(W,Eps) - диагональная "псевдообратная"
 // матрица: pinv(W,Eps)[i]=1/W[i],если W[i]>Eps; pinv(W,Eps)[i]=0,если W[i]<=Eps,
 // где Eps - порог "нулевых" сингулярных чисел. Eps используется для повышения
 // устойчивости решения. Теперь V'*X=pinv(W,Eps)*U'*Y, так что окончательное
 // решение МНК запишется в виде X=V*pinv(W,Eps)*U'*Y.
 // Матрица pinv(A,Eps)=V*pinv(W,Eps)*U' - "псевдообратная" с порогом Eps.
 // Итак, МНК система A*X=Y имеет решение X=pinv(A,Eps)*Y, где "псевдообратная"
 // матрица pinv(A,Eps)=V*pinv(W,Eps)*U' и Eps - малое число для устойчивости.
 // Примечание 1:
 // На практике используют матрицу U[M*N] вместо U[M*M] и вектор W[N] вместо W[M*N].
 // Это обусловлено тем, что W[i,j]=0, при i<>j. Поэтому нет практического смысла
 // хранить недиагональные элементы W[i,j], а также столбцы N..M-1 матрицы U.
 // Класс TSVDWrap поэтому также использует матрицу U[M*N] и вектор W[N].
 // Методы:
 // M,N   - размерность исходной матрицы A[M*N]
 // Eps   - порог "малости" для сингулярных чисел с заданной абсолютной и относительной точностью.
 // Rank  - ранг матрицы, то есть число сингулярных чисел, превышающих пороговое значение Eps.
 // Solve - решение системы линейных уравнений A*X=Y размера M*N в смысле наименьших квадратов, A=U*W*V'.
 // PInv  - вычисление (псевдо) обратной матрицы размера N*M.
 //
 ISVDWrap=interface(IPluginWrap) [SIID_ISVDWrap]
  function  GetM:Integer;
  function  GetN:Integer;
  function  GetU(i,j:Integer):Double;
  function  GetW(i:Integer):Double;
  function  GetV(i,j:Integer):Double;
  function  Eps(AbsEps:Double=0; RelEps:Double=1E-10):Double;
  function  Rank(AbsEps:Double=0; RelEps:Double=1E-10):Integer;
  function  Solve(X,Y:IDoubleVectorWrap;
                  AbsEps:Double=0; RelEps:Double=1E-10):Boolean;
  function  PInvert(AbsEps:Double=0;
                    RelEps:Double=1E-10):IDoubleMatrixWrap;
  //
  // Properties
  //
  property M               : Integer read GetM;  // Число строк матрицы  A
  property N               : Integer read GetN;  // Число столбцов матрицы A
  property U[i,j:Integer]  : Double  read GetU;  // U[0..M-1,0..N-1] -> A=U*W*V'
  property W[i:Integer]    : Double  read GetW;  // W[0..N-1]        -> A=U*W*V'
  property V[i,j:Integer]  : Double  read GetV;  // V[0..N-1,0..N-1] -> A=U*W*V'
 end;
 //
 // Класс TFifoWrap служит для потокобезопасной передачи данных между потоками
 // через буфер FIFO. Операция передачи данных является атомарной, она либо
 // проходит целиком, либо аннулируется. Это гарантирует корректную передачу
 // потока данных между потоками.
 //
 IFifoWrap = interface(IPluginWrap) [SIID_IFifoWrap]
  procedure   Lock;
  procedure   UnLock;
  procedure   Clear;
  function    GetCount:LongInt;
  function    GetSpace:LongInt;
  function    GetSize:LongInt;
  function    GetName:LongString;
  procedure   SetName(const NewName:LongString);
  function    GetTag(i:Integer):LongWord;
  procedure   SetTag(i:Integer; NewTag:LongWord);
  function    GetLost:Int64;
  procedure   SetLost(N:Int64);
  function    Put(Buffer:Pointer; BufferSize:LongInt;
                  TryMode:Boolean=false):LongInt;
  function    Get(Buffer:Pointer; BufferSize:LongInt):LongInt;
  function    PutText(const S:LongString;
                      TryMode:Boolean=false):Boolean;
  function    GetText(MaxLen:Integer=MaxInt):LongString;
  //
  // Properties
  //
  property    Name           : LongString read GetName  write SetName;  // Имя буфера
  property    Tag[i:Integer] : LongWord   read GetTag   write SetTag;   // Теги для хранения пользовательских данных
  property    Count          : LongInt    read GetCount;                // Сколько байт хранится в буфере
  property    Space          : LongInt    read GetSpace;                // Сколько байт в буфере свободно
  property    Size           : LongInt    read GetSize;                 // Размер буфера
  property    Lost           : Int64      read GetLost  write SetLost;  // Число потерянных байт
 end;
 //
 //
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Определения интерфейсов динамических расширений CRWDAQ.                   //
 // 1) Для чего это надо.                                                     //
 //    Известно, что написать программу на все случаи жизни невозможно, хотя  //
 //    и хочется. Однако можно сделать средства, позволяющие разрабатывать,   //
 //    редактировать, компилировать и тут же, на ходу, вызывать на исполнение //
 //    динамические библиотеки DLL. Если DLL будет уметь общаться программой, //
 //    можно серьезно расширить функциональность программы без модификации, а //
 //    за счет создания внешних динамических расширений или утилит. Если при  //
 //    этом встроить компилятор в программу, программа приобретает новое      //
 //    качество - становится саморасширяемой проблеммно-ориентированной       //
 //    средой разработки. Пакет CRWDAQ, например, содержит интегрированный    //
 //    компилятор Delphi, точнее, его подмножество.                           //
 // 2) Как это реализуется.                                                   //
 //    Создается виртуальный абстрактный интерфейсный класс. Это значит, что  //
 //    он содержит ТОЛЬКО виртуальные абстрактные методы, дающие динамически  //
 //    вызываемой DLL (клиенту) доступ к данным и процедурам сервера, то есть //
 //    программный интерфейс (API). Допустимы также свойства (property), если //
 //    они определяются через виртуальные методы. Фактически абстрактный      //
 //    класс - это просто таблица указателей на функции, общая для сервера и  //
 //    клиента. Практически то же, что COM - интерфейс, только проще.         //
 //    Основная программа (сервер) создает порожденный класс, перекрывающий   //
 //    все абстрактные методы, но клиент не знает, как сервер их реализует.   //
 //    Клиент, то есть DLL, экспортирует функцию вида:                        //
 //      function Plugin(CrwApi:ICrwApi):Integer; StdCall;                    //
 //    Через экземпляр объекта API клиент получает доступ к тем данным и      //
 //    методам, которые сервер считает нужным предоставить. После создания и  //
 //    компиляции утилиты сервер может вызвать клиента, так как способ вызова //
 //    четко зафиксирован. Клиент же может сделать, все, что позволяет данный //
 //    ему сервером интерфейс.                                                //
 // 3) Некоторые договоренности.                                              //
 //    Чтобы обеспечить непротиворечивую расширяемость интерфейса, он должен  //
 //    строиться как дерево интерфейсов, то есть интерфейс может содержать    //
 //    в качестве методов функции для доступа к другим интерфейсам. Дерево    //
 //    интерфейсов может только расти, удалять или менять сигнатуры или       //
 //    порядок ранее объявленных методов запрещено. Поэтому клиент может быть //
 //    уверен, что при обновлении версии сервера его интерфейс будет верным,  //
 //    хотя про новые методы ему ничего не известно.                          //
 //    Клиент возвращает 0, если все нормально, или код ошибки. Все прочие    //
 //    данные клиент передает и получает через функции API.                   //
 ///////////////////////////////////////////////////////////////////////////////
const
 //
 // Идентификатор экспортируемой функции DLL расширения CRWDAQ.
 //
 CRWDAQ_PLUGIN_ID    = 'CRWDAQ_PLUGIN_FUNCTION';
 //
 // Идентификаторы назначения утилиты.
 //
 ForCalculations    = 0; // Для вычислений общего назначения.
 ForDataAnalysis    = 1; // Для анализа экспериментальных данных.
 ForDataAcquisition = 2; // Для системы сбора данных.

type
 //
 // Интерфейсы CRWDAQ.
 //
 ICrwApi    = interface;        // Корень дерева интерфейсов CRWDAQ API.
 ISysApi    = interface;        // API для функций общего назначения.
 IGuiApi    = interface;        // API для создания графического интерфейса.
 IDaqApi    = interface;        // API для систем сбора данных.
 IDanApi    = interface;        // API для анализа данных.
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Динамическая библиотека для расширений CRWDAQ должна экспортировать       //
 // функцию с такой сигнатурой и именем CRWDAQ_PLUGIN_ID.                     //
 ///////////////////////////////////////////////////////////////////////////////
 TPluginFun = function(CrwApi:ICrwApi):Integer; {$I _crw_plugin_abicall.inc};
 //
 // Исключения интерфейсов CRWDAQ.
 //
 ECrwApi    = class(Exception);
 ESysApi    = class(Exception);
 EDaqApi    = class(Exception);
 EDanApi    = class(Exception);
 ///////////////////////////////////////////////////////////////////////////////
 // Класс CrwApi служит корнем дерева интерфейсов CRW API и служит для        //
 // доступа к другим проблеммно - ориентированным интерфейсам.                //
 ///////////////////////////////////////////////////////////////////////////////
 ICrwApi    = interface(IPluginApi) [SIID_ICrwApi]
  //
  // 0    - Версия API, содержит дату создания, например, 20030215.
  //
  function  Version:Cardinal;
  //
  // 1    - Проблемная ориентация интерфейса.
  //        Это может быть ForCalculations, ForDataAnalysis, ForDataAcquisition.
  //
  function  Target:Cardinal;
  //
  // 2    - API для функций общего назначения.
  //
  function  SysApi:ISysApi;
  //
  // 3    - API для функций графического интерфейса.
  //
  function  GuiApi:IGuiApi;
  //
  // 4    - API для систем сбора данных.
  //
  function  DaqApi:IDaqApi;
  //
  // 5    - API для анализа экспериментальных данных.
  //
  function  DanApi:IDanApi;
  //
  // 6    - Имя плагина. Номинально - базовое имя файла исходного кода.
  //
  function  PluginName:LongString;
  //
  // 7    - Имя файла исходного кода плагина (предполагаемое).
  //
  function  PluginSourceFileName:LongString;
  //
  // 8    - Имя исполняемого файла плагина (*.dll или *.so).
  //
  function  PluginBinaryFileName:LongString;
  //
  // 9    - Проверка типа плагина (aTarget=aExpected).
  //
  function VerifyPluginType(aTarget,aExpected:Cardinal; Test:Boolean=false):Boolean;
  //
  // 10   - Проверка версии плагина (aEdition=aVersion).
  //
  function VerifyPluginDate(aEdition,aVersion:Cardinal; Test:Boolean=false):Boolean;
 end;
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Класс TSysApi содержит функции общего назначения. Он доступен при всех    //
 // значениях Target.                                                         //
 ///////////////////////////////////////////////////////////////////////////////
 ISysApi = interface(IPluginApi) [SIID_ISysApi]
  //
  // 0    - Переназначает стандартный ввод на ввод с окна консоли приложения.
  //
  function  RedirectStdIn(var stdIn:Text):boolean;
  //
  // 1    - Переназначает стандартный вывод на вывод в окно консоли приложения.
  //
  function  RedirectStdOut(var stdOut:Text):boolean;
  //
  // 2    - Инкремент указателя.
  //
  function  IncPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
  //
  // 3    - Декремент указателя.
  //
  function  DecPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
  //
  // 4    - Перемещение данных с проверкой указателей.
  //
  procedure SafeMove(const Source; var Dest; Count: LongInt);
  //
  // 5    - Заполнение области памяти заданным значением.
  //
  procedure SafeFillChar(var X; Count: LongInt; Value: Byte);
  //
  // 6    - Применяется для выровнивания размер буфера Size по модулю Step.
  //
  function  AdjustBufferSize(Size:LongInt; Step:LongInt):LongInt;
  //
  // 7    - Выделяет блок памяти заданного размера или nil при неудаче.
  //        Блок памяти автоматически заполняется нулями.
  //
  function  Allocate(N:LongInt):Pointer;
  //
  // 8    - Освободить блок памяти, выделенный вызовом Allocate.
  //
  procedure Deallocate(var P:Pointer);
  //
  // 9    - Узнать размер блока памяти, выделенного вызовом Allocate.
  //
  function  AllocSize(P:Pointer):LongInt;
  //
  // 10   - Перераспределить память, выделенную вызовом Allocate.
  //        Содержимое блока не меняется, дополнительная память зануляется.
  //
  function  Reallocate(var P:Pointer; N:LongInt):boolean;
  //
  // 11   - Проверка вещественного числа на значение Not A Number.
  //
  function  isNAN(const X: Double): Boolean;
  //
  // 12   - Проверка вещественного числа на значение Infinity.
  //
  function  isINF(const X: Double): Boolean;
  //
  // 13   - Возвращает значение Not A Number, "не число".
  //
  function  _NaN: Double;
  //
  // 14   - Возвращает значение +Infinity, "плюс бесконечность".
  //
  function  _PlusInf: Double;
  //
  // 15   - Возвращает значение -Infinity, "минус бесконечность".
  //
  function  _MinusInf: Double;
  //
  // 16   - Поиск нуля функции F(x) на интервале [A,B] с точностью Tol.
  //        Указатель Custom служит для передачи в F(x) пользовательских данных.
  //
  function  FindZero(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
  //
  // 17   - Поиск минимума функции F(x)на интервале [A,B] с точностью Tol.
  //        Указатель Custom служит для передача в F(x) пользовательских данных.
  //
  function  FindMin(F:zmFunType;A,B,Tol:Double;Custom:Pointer):Double;
  //
  // 18   - Поиск интеграла F(x) на интервале [A,B] с заданной абсолютной AbsErr
  //        и относительной RelErr точностью. Возвращает значение интеграла,
  //        а также оценку точности ErrEst, флаг ошибок Flag, который равен 0
  //        при успешном интегрировании, а также счетчик вызовов функции NoFun.
  //        Указатель Custom служит для передача в F(x) пользовательских данных.
  //        
  function  Quanc8(F:zmFunType;A,B,AbsErr,RelErr:Double; var ErrEst:Double;
            var NoFun:Integer; var Flag:Double; Custom:Pointer):Double;
  //
  // 19   - Относительная точность представления вещественных чисел.
  //
  function  MachEps:double;
  //
  // 20   - Текущее время в календарном формате.
  //
  function  NativeTimeNow:TSystemTime;
  //
  // 21   - Текущее время в миллисекундах от Рождества Христа (01.01.0001-00:00:00).
  //
  function  MSecNow:double;
  //
  // 22   - Назначить значение записи календарного времени.
  //
  function  AssignNativeTime(Year:word=1;Month:word=1;Day:word=1;
                             Hour:word=0;Minut:word=0;Sec:word=0;
                             mSec:word=0):TSystemTime;
  //
  // 23   - Преобразовать время из миллисекунд в календарное.
  //
  function  MSecToNativeTime(mSecond:Double):TSystemTime;
  //
  // 24   - Преобразовать время из календарного в миллисекунды.
  //
  function  NativeTimeToMSec(const T:TSystemTime):double;
  //
  // 25   - Преобразовать время из календарного в миллисекунды.
  //
  function  DateTimeToMSec(Year:Word=1;Month:Word=1;Day:Word=1;
                           Hour:Word=0;Minut:Word=0;Sec:Word=0;
                           mSec:Word=0):Double;
  //
  // 26   - Преобразовать время из миллисекунд в календарное.
  //
  procedure MSecToDateTime(T:Double;
                           var Year,Month,Day,Hour,Min,Sec,mSec:word);
  //
  // 27   - Получить строку даты по времени в миллисекундах от Рождества Христа.
  //
  function  GetDateStr(ms:Double;Delim:Char='.';YMD_Order:Boolean=false;
                       ShortYear:Boolean=false):LongString;
  //
  // 28   - Получить строку времени суток по времени в миллисекундах от Рождества Христа.
  //
  function  GetTimeStr(ms:double;Delim:Char=':';
                       ShowMSec:Boolean=false):LongString;
  //
  // 29   - Длина строки, заверщающейся нулем.
  //
  function  StrLen(Str: PChar): LongInt;
  //
  // 30   - Длина строки, но не более MaxLen.
  //
  function  StrLLen(Str: PChar; MaxLen: Integer): LongInt;
  //
  // 31   - Возвращает указатель на ноль, завершающий строку Str.
  //
  function  StrEnd(Str: PChar): PChar;
  //
  // 32   - Копирует строку Source в строку Dest и возвращает Dest.
  //
  function  StrCopy(Dest, Source: PChar): PChar;
  //
  // 33   - Копирует Source в Dest и возвращает StrEnd(Dest).
  //
  function  StrECopy(Dest, Source: PChar): PChar;
  //
  // 34   - копирует не более MaxLen символов из Source в Dest и возвращает Dest.
  //
  function  StrLCopy(Dest, Source: PChar; MaxLen: LongInt): PChar;
  //
  // 35   - Копирует Pascal-строку Source в строку Dest и возвращает Dest.
  //
  function  StrPCopy(Dest: PChar; const Source: LongString): PChar;
  //
  // 36   - Добавляет копию Source в конец строки Dest и возвращает Dest.
  //
  function  StrCat(Dest, Source: PChar): PChar;
  //
  // 37   - Добавляет копию Pascal-строки Source в конец Dest и возвращает Dest.
  //
  function  StrPCat(Dest:PChar; const Source: LongString): PChar;
  //
  // 38   - Добавляет не более MaxLen - StrLen(Dest) символов из Source в хвост
  //        Dest, возвращает Dest.
  //
  function  StrLCat(Dest, Source: PChar; MaxLen: LongInt): PChar;
  //
  // 39   - Сравнивает Str1 и Str2. Возвращает значение < 0 если Str1 < Str2,
  //        0 если Str1 = Str2, или значение > 0 если Str1 > Str2.
  //
  function  StrComp(Str1, Str2: PChar): Integer;
  //
  // 40   - Сравнивает Str1 и Str2, в режиме, нечувствительном к регистру символов.
  //
  function  StrIComp(Str1, Str2: PChar): Integer;
  //
  // 41   - Сравнивает Str1 и Str2, но не более MaxLen символов.
  //
  function  StrLComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
  //
  // 42   - Сравнивает Str1 и Str2, но не более MaxLen символов, без учета регистра.
  //
  function  StrLIComp(Str1, Str2: PChar; MaxLen: LongInt): Integer;
  //
  // 43   - Возвращает указатель на первое вхождение символа Chr в строку Str.
  //        Возвращает Nil, если в строке нет такого символа.
  //
  function  StrScan(Str: PChar; Chr: Char): PChar;
  //
  // 44   - Возвращает указатель на последнее вхождение символа Chr в строку Str.
  //        Возвращает Nil, если в строке нет такого символа.
  //
  function  StrRScan(Str: PChar; Chr: Char): PChar;
  //
  // 45   - Преобразует строку Str к верхнему регистру и возвращает Str.
  //
  function  StrUpper(Str: PChar): PChar;
  //
  // 46   - Преобразует строку Str к нижнему регистру и возвращает Str.
  //
  function  StrLower(Str: PChar): PChar;
  //
  // 47   - Возвращает указатель на первое вхождение строки Str2 в строку Str1.
  //        Возвращает NIL, если в строке нет вхождения искомой подстроки.
  //
  function  StrPos(Str1, Str2: PChar): PChar;
  //
  // 48   - Преобразует строку Str с завершающим нулем в Pascal-строку.
  //
  function  StrPas(Str: PChar): LongString;
  //
  // 49   - Пропускает ведущие символы, входящие в множество PassChars.
  //        Возвращает указатель на первый символ, не входящий в PassChars.
  //
  function  StrPass(Str:PChar; const PassChars:TCharSet):PChar;
  //
  // 50   - Копирует не более Count символов из Source в Dest и возвращает Dest.
  //        Source и Dest могут перекрываться.
  //
  function  StrMove(Dest, Source: PChar; Count: LongInt): PChar;
  //
  // 51   - Вычисляет число строк текста, то есть число разделителей CRLF плюс 1.
  //        Count задает число символов в тексте. UnixStyle задает разделитель
  //        CR вместо CRLF.
  //
  function  GetTextNumLines(Text:PChar; Count:LongInt=MaxLongInt;
                            UnixStyle:boolean=false):LongInt;
  //
  // 52   - Преобразование к верхнему регистру.
  //
  function  UpCase(c:Char):Char;
  //
  // 53   - Преобразование к нижнему регистру.
  //
  function  LoCase(c:Char):Char;
  //
  // 54   - Преобразование к нижнему регистру.
  //
  function  LoCaseStr(const S:LongString):LongString;
  //
  // 55   - Преобразование к верхнему регистру.
  //
  function  UpCaseStr(const S:LongString):LongString;
  //
  // 56   - Преобразование кодировки Windows(1251) в кодировку Dos(866).
  //
  function  WinToDos(c:Char):Char;
  //
  // 57   - Преобразование кодировки Dos в кодировку Windows.
  //
  function  DosToWin(c:Char):Char;
  //
  // 58   - Преобразование кодировки Windows в кодировку Dos.
  //
  function  WinToDosStr(const S:LongString):LongString;
  //
  // 59   - Преобразование кодировку Dos в кодировку Windows.
  //
  function  DosToWinStr(const S:LongString):LongString;
  //
  // 60   - Преобразование кодировки Windows в кодировку Dos.
  //
  function  WinToDosLongStr(const S:LongString):LongString;
  //
  // 61   - Преобразование кодировки Dos в кодировку Windows.
  //
  function  DosToWinLongStr(const S:LongString):LongString;
  //
  // 62   - Преобразование кодировки Windows в кодировку KOI-8.
  //
  function  WinToKoi(c:Char):Char;
  //
  // 63   - Преобразование кодировки KOI-8 в кодировку Windows.
  //
  function  KoiToWin(c:Char):Char;
  //
  // 64   - Преобразование кодировки Windows в кодировку KOI-8.
  //
  function  WinToKoiStr(const S:LongString):LongString;
  //
  // 65   - Преобразование кодировки KOI-8 в кодировку Windows.
  //
  function  KoiToWinStr(const S:LongString):LongString;
  //
  // 66   - Преобразование кодировки Windows в кодировку KOI-8.
  //
  function  WinToKoiLongStr(const S:LongString):LongString;
  //
  // 67   - Преобразование кодировки KOI-8 в кодировку Windows.
  //
  function  KoiToWinLongStr(const S:LongString):LongString;
  //
  // 68   - Защищенная версия функции Format.
  //
  function  Format(const Fmt:LongString;
                   const Args: array of const):LongString;
  //
  // 69   - Создает множество символов, присутствующих в строке S.
  //
  function  Str2CharSet(const S:LongString):TCharSet;
  //
  // 70   - Создать строку из символов, присутствующих в множестве S.
  //
  function  CharSet2Str(const S:TCharSet):LongString;
  //
  // 71   - Копировать из строки S символы pos..length(S).
  //
  function  RightStr(const S:LongString; pos:integer):LongString;
  //
  // 72   - Копировать из строки символы 1..count.
  //
  function  LeftStr(const S:LongString; count:integer):LongString;
  //
  // 73   - Вернет строку длиной len, заполненную символами ch.
  //
  function  CharStr(Len:Integer; Ch:Char=' '):LongString;
  //
  // 74   - Возвращает строку S, дополненную справа до длины len символами ch.
  //
  function  Pad(const S:LongString; Len:Integer; Ch:Char=' '):LongString;
  //
  // 75   - Возвращает строку S, дополненную слева до длины len символами ch.
  //
  function  LeftPad(const S:LongString; Len:Integer; Ch:Char=' '):LongString;
  //
  // 76   - Возвращает строку S, дополненную симметрично слева и справа до длины
  //        Width символами Ch, так что исходная строка оказывается в центре.
  //
  function  CenterStr(const S:LongString; Width:Integer; Ch:Char=' '):LongString;
  //
  // 77   - Возвращает строку без пробелов в начале строки. TrimChars определяет,
  //        что считается пробелом.
  //
  function  TrimLeadChars(const S:LongString;
                          const TrimChars:TCharSet):LongString;
  //
  // 78   - Возвращает строку без пробелов в конце строки. TrimChars определяет,
  //        что считается пробелом.
  //
  function  TrimTrailChars(const S:LongString;
                           const TrimChars:TCharSet):LongString;
  //
  // 79   - Возвращает строку без пробелов в начале и в конце строки.
  //        LeadTrim,TrailTrim определяют, что считается пробелом.
  //
  function  TrimChars(const S:LongString;
                      const LeadTrim,TrailTrim:TCharSet):LongString;
  //
  // 80   - Возвращает строку без пробелов в начале строки.
  //        Пробелами считаются символы [#0..' '].
  //
  function  TrimLead(const S:LongString):LongString;
  //
  // 81   - Возвращает строку без пробелов в конце строки.
  //        Пробелами считаются символы [#0..' '].
  //
  function  TrimTrail(const S:LongString):LongString;
  //
  // 82   - Возвращает строку без пробелов в начале и конце строки.
  //        Пробелами считаются символы [#0..' '].
  //
  function  Trim(const S:LongString):LongString;
  //
  // 83   - Функция преобразует строки имен к виду, позволяющему непосредственно
  //        сравнивать их. Например, имена " one " и "One" эквивалентны, хотя их
  //        строковое представление различно. Функция преобразует их в "ONE".
  //        Считается что в именах регистр и пробелы - незначащие.
  //        Текущая реализация - UpcaseStr(Trim(Name)).
  //
  function  UnifyAlias(const Name:LongString; Mode:Integer=ua_NameDef):LongString;
  //
  // 84   - Преобразует имена файлов к каноническому виду, допускающему
  //        непосредственное сравнение имен файлов как строк. Для этого строка
  //        имени файла преобразуется к полному имени, незначащие пробелы удаляются,
  //        имя преобразуется в верхний регистр. Например, имена файлов
  //        "c:\daq\1110\config\1110.cfg" и "C:\DAQ\1110\DATA\..\CONFIG\1110.CFG"
  //        эквивалентны, хотя имеют разное строковое предстваление. Функция
  //        преобразует их к "C:\DAQ\1110\CONFIG\1110.CFG".
  //        Надо отметить, что для имен файлов Unix это не работает - там имена
  //        файлов чувствительны к регистру.
  //        Текущая реализация - UpcaseStr(Trim(FExpand(Name))).
  //
  function  UnifyFileAlias(const FileName:LongString; Mode:Integer=ua_FileDef):LongString;
  //
  // 85   - Преобразует любую строку в корректный заголовок секции.
  //        Например, строки 'Section', '[Section', 'Section]', '[Section]'
  //        будут преобразованы в '[SECTION]'.
  //
  function UnifySection(const aSectionName:LongString; Mode:Integer=ua_SectDef):LongString;
  //
  // 86   - Делает перенос строки InSt, чтобы строка умещалась в поле Margin,
  //        сохраняет результат OutSt, а остаток строки в Overlap.
  //
  procedure WordWrap(const InSt:LongString; var OutSt, Overlap:LongString;
                     Margin:Integer; PadToMargin:Boolean);
  //
  // 87   - Заменяет в строке Str все вхождения подстроки Find на строку Replace.
  //
  function ReplaceString(const Str,Find,Replace:LongString):LongString;
  //
  // 88   - Делает замену символов форматирования ^L,^R,^C,^N,^B на строки
  //        '^L','^R','^C','^N','^B' или наоборот. Символы форматирования:
  //        ^L - выравнивание по левому краю
  //        ^R - выравнивание по правому краю
  //        ^C - выравнивание по центру
  //        ^N - переход на новую строку
  //
  function ReplaceAlignStr(const Str:LongString;
                           Invert:Boolean):LongString;
  //
  // 89   - Найти в строке S число слов, разделенных пробелами WordDelims.
  //
  function  WordCount(const S:LongString;
                      const WordDelims:TCharSet):Integer;
  //
  // 90   - Выделить из строки S слово номер N. Слова разделены пробелами WordDelims.
  //
  function  ExtractWord(N:Integer; const S:LongString;
                                   const WordDelims:TCharSet):LongString;
  //
  // 91   - Возвращает порядковый номер слова Name в наборе слов, составляющих
  //        строку Str. Слова разделены пробелами WordDelims.
  //        Возвращает ноль, если слово не найдено.
  //
  function  WordIndex(const Name,Str:LongString;
                      const Delims:TCharSet):Integer;
  //
  // 92   - Выделяет слово, а затем пытается преобразовать его в целое число.
  //
  function  ExtractInt(N:Integer; const S:LongString; const WordDelims:TCharSet;
                                  var Value:LongInt):boolean;
  //
  // 93   - Выделяет слово, а затем пытается преобразовать его в вещественное число.
  //
  function  ExtractReal(N:Integer; const S:LongString; const WordDelims:TCharSet;
                                   var Value:Double):boolean;
  //
  // 94   - Преобразует байт в строку в двоичной системе счисления.
  //
  function  BinB(x:Byte):LongString;
  //
  // 95   - Преобразует слово в строку в двоичной системе счисления.
  //
  function  BinW(x:Word):LongString;
  //
  // 96   - Преобразует целое число в строку в двоичной системе счисления.
  //
  function  BinL(x:LongWord):LongString;
  //
  // 97   - Преобразует байт в строку в восьмиричной системе счисления.
  //
  function  OctB(x:Byte):LongString;
  //
  // 98   - Преобразует слово в строку в восьмиричной системе счисления.
  //
  function  OctW(x:Word):LongString;
  //
  // 99   - Преобразует целое число в строку в восьмиричной системе счисления.
  //
  function  OctL(x:LongInt):LongString;
  //
  // 100  - Преобразует байт в строку в шестнадцатиричной системе счисления.
  //
  function  HexB(x:Byte):LongString;
  //
  // 101  - Преобразует слово в строку в шестнадцатиричной системе счисления.
  //
  function  HexW(x:Word):LongString;
  //
  // 102  - Преобразует целое число в строку в шестнадцатиричной системе счисления.
  //
  function  HexL(x:LongWord):LongString;
  //
  // 103  - Преобразует целое число в строку.
  //
  function  Long2Str(L:LongInt):LongString;
  //
  // 104  - Преобразует вещественное число в строку в фиксированном формате.
  //
  function  Real2Str(R:Double; Width:Integer=0;
                     Places:Integer=0):LongString;
  //
  // 105  - Преобразует целое число в строку.
  //
  function  d2s(d:LongInt; Width:Integer=0):LongString;
  //
  // 106  - Преобразует вещественное число в строку в свободном формате.
  //        Параметры задаются в f2sFormat.
  //
  function  f2s(f:Double):LongString;
  //
  // 107  - Задает параметры для функции f2s - ширину и число значащих цифр.
  //
  procedure f2sFormat(Width:Integer=0; Decimals:Integer=14);
  //
  // 108  - Восстанавливает предыдущее значение параметров f2s.
  //
  procedure f2sFormatOld;
  //
  // 109  - Преобразует вещественное число в строку в свободном формате.
  //        Аналогично Format('%w.dg',[X]).
  //
  function  FormatG(X:Double; w:integer; d:integer):LongString;
  //
  // 110  - Преобразует строку "a" в целое число "i".
  //        Если первый символ строки "$", преобразует как шестнадцатиричное.
  //
  function  atoi(a:PChar; var i:LongInt):boolean;
  //
  // 111  - Преобразует строку "a" в вещественное число "f".
  //        Если первый символ строки "$", преобразует как шестнадцатеричное целое.
  //
  function  atof(a:PChar; var f:double):boolean;
  //
  // 112  - Преобразует строку в целое число.
  //
  function  Str2Int(const S:LongString; var I:Integer):Boolean;
  //
  // 113  - Преобразует строку в целое беззнаковое.
  //
  function  Str2Word(const S:LongString; var I:Word):Boolean;
  //
  // 114  - Преобразует строку в целое число.
  //
  function  Str2Long(const S:LongString; var I:LongInt):Boolean;
  //
  // 115  - Преобразует строку в вещественное число.
  //
  function  Str2Real(const S:LongString; var R:Double):Boolean;
  //
  // 116  - Преобразует строку в вещественное число.
  //        Присвоение делает, только если строковые представления старого
  //        и нового значения различны. Это полезно при обработке полей ввода.
  //
  function SmartStr2Real(const S:LongString; var R:Double):Boolean;
  //
  // 117  - Преобразование строки в целое число в двоичной системе.
  //
  function  StrBin2Long(const S:LongString; var L:LongInt):Boolean;
  //
  // 118  - Преобразование строки в целое число в восьмеричной системе.
  //
  function  StrOct2Long(const S:LongString; var L:LongInt):Boolean;
  //
  // 119  - Преобразование строки в целое число в шестнадцатеричной системе.
  //
  function  StrHex2Long(const S:LongString; var L:LongInt):Boolean;
  //
  // 120  - Форматное чтение переменных с префиксом для идентификации.
  //        Формат состоит из слов,разделенных символом CR или ; типа
  //        prefix1%format1+CR+
  //        prefix2%format2+CR+
  //        ....
  //        prefixN%formatN
  //        Префикс-строка для поиска в тексте (может отсутствовать)
  //        Форматы должны быть одной из указанных строк:
  //        %s - чтение PureString(Data) после префикса до символа CR или конца строки,
  //             но не более 255 символов
  //        %a - чтение PureString(Data) после префикса и пробелов слова,содержащего
  //             символы, отличные от пробелов (ScanSpaces), но не более 255 символов
  //        %d - чтение LongInt(Data) после префикса и пробелов
  //        %w - чтение Word
  //        %i - чтение Integer
  //        %b - чтение Boolean (0,1,False,True,Yes,No-по первой букве)
  //        %f - чтение Double(Data) после префикса и пробелов
  //        %c - чтение Char(Data) сразу после префикса
  //        ScanVar возвращает указатель на символ после последней прочтенной
  //        переменной в тексте или nil в случае:
  //         если не найден префикс поиска в тексте ( если он был в формате)
  //         если не найден ни один из форматов %s,%f,%d,%c,%a,%w,%i,%b
  //         в строке формата
  //        пример:
  //         var
  //          Data:record S:PureString; D:LongInt; F:Double; C:Char end;
  //         const
  //          InputStr='Найти Var1=Cтрока'+CR+
  //                   '      Var2= 100'+CR+
  //                   '      Var3= -3.14'+CR+
  //                   '      Var4=&';
  //          begin
  //           if ScanVar(svAsIs,InputStr,'Var1=%s'+CR+
  //                                      'Var2=%d'+CR+
  //                                      'Var3=%f'+CR+
  //                                      'Var4=%c',Data)=nil then error;
  //         результат:
  //          Data.S='Строка';
  //          Data.D=100;
  //          Data.F:=-3.14;
  //          Data.C='&';
  //        Режим чтения задает способ выделения переменной
  //        svUpCase/svLoCase
  //         Переводят строку формата в верхний/нижний регистр и нужны для того чтобы
  //         переменные читались независимо от регистра, надо только сам текст тоже
  //         привести к этому регистру. Например:
  //          S=StrUpper('SomeVar=1000')
  //          ScanVar(svUpCase,S,'somevar%d'...    - прочитается
  //          ScanVar(svNoCase,S,'somevar%d'...    - не прочитается, разные регистры
  //        svLSpace/svRSpace
  //         Анализируют наличие пробелов слева/справа от выделенного префикса, чтобы
  //         одинаковые фрагменты слов не приводили к ошибочному чтению.
  //         Например, в строке
  //          S='Var10 = 1, Var1 = 10'
  //         фрагмент Var1 - общий для обеих переменных, тогда
  //          ScanVar(0,S,'Var1%s'... прочитает 0, так как воспримет в качестве
  //          префикса вхождение Var1 в строку Var10
  //          ScanVar(svRSpace+svLSpace прочитает правильно, так как выделит только
  //          цельное слово Var1 (так чтобы слева и справа от слова были пробелы)
  //         В то же время в конструкции S='Sin(10)' надо читать
  //         ScanVar(0,'Sin(%f'... , а не ScanVar(svRSpace,'Sin(%f'...
  //         так как значение идет сразу после префикса без пробела
  //        svOrigin
  //         начинает поиск каждого префикса с начала текста, чтобы чтение переменных
  //         не зависело от их порядка, например
  //         если S='Var2 = 1, Var1 = 2', то
  //              ScanVar(0,'Var1%d;Var2%d' - не прочитает так как Var1 - после Var2
  //              ScanVar(svOrigin,'Var1%d;Var2%d'- прочитает
  //         в то же время для жестких конструкций svOrigin не пригоден, например
  //         если S='Var = 1 , 2, 3, 4', то надо читать ScanVar(0,'Var%w;,%w;,%w;,%w
  //         так как при чтении ScanVar(svOrigin,.... вместо 3, 4 будет прочитано
  //         2, 2 - число после ПЕРВОЙ запятой в тексте
  //         Отметим, что пустой префикс не ведет переходу на начало текста то есть
  //         в предыдущем примере можно читать ScanVar(0,'Var%w;%w;%w;%w'....
  //
  function  ScanVar(svMode:word; Str:PChar;
                    const Format:LongString; var Data):PChar;
  //
  // 121  - Проверяет, содержит ли строка S отличные от пробелов символы.
  //
  function  IsEmptyStr(const S:LongString):boolean;
  //
  // 122  - Найти индекс последнего вхождения символа - разделителя из набора
  //        символов Delimiters в строке S. Возвращает ноль, если в строке нет
  //        символов - разделителей.
  //
  function  LastDelimiter(const Delimiters:TCharSet;
                          const S:LongString):Integer;
  //
  // 123  - Удалить из имени каталога DirName незначащие пробелы и добавить
  //        обратный слеш "\" в конец строки, если его нет. Используется при
  //        конструировании имен файлов Windows.
  //
  function  AddBackSlash(const DirName:LongString):LongString;
  //
  // 124  - Удалить из имени каталога незначащие пробелы, а также обратный слеш
  //        в конце строки. Используется при конструировании имен файлов.
  //
  function  DropBackSlash(const DirName:LongString):LongString;
  //
  // 125  - Проверяет, является ли имя файла групповой маской, типа "*.*" или "?.?".
  //
  function  IsWildCard(const FileName:LongString):boolean;
  //
  // 126  - Проверяет, является ли имя файла относительным или абсолютным.
  //        Абсолютное имя содержит полный путь к файлу, относительное же
  //        предполагает отсчет от какого-то начального каталога, например,
  //        ".\doc\readme.txt".
  //
  function  IsRelativePath(const S:LongString):boolean;
  //
  // 127  - Проверяет наличие в имени файла расширения и находит его положение,
  //        то есть положение точки, с которой начинается расширение.
  //
  function  HasExtension(const Name:LongString;
                         var DotPos:Integer):Boolean;
  //
  // 128  - Возвращает имя файла с присоединением заданного расширения, если
  //        в имени нет расширения. Если расширение уже есть, то имя файла
  //        не изменяется. Используется для задания расширния по умолчанию.
  //
  function  DefaultExtension(const Name,Ext:LongString):LongString;
  //
  // 129  - Возвращает имя файла с присоединенным путем, если исходное имя файла
  //        содержит относительный путь. Абсолютное имя файла не изменяется.
  //        Используется для задания пути по умолчанию.
  //
  function  DefaultPath(const Name,Path:LongString):LongString;
  //
  // 130  - Задает расширение имени файла. Если у файла уже есть расширение, оно
  //        будет заменено новым.
  //
  function  ForceExtension(const Name,Ext:LongString):LongString;
  //
  // 131  - Задает путь к файлу. Если у файла уже был путь, он будет заменен новым.
  //
  function  ForcePath(const NewPath,Name:LongString):LongString;
  //
  // 132  - Удаляет из имени файла путь, оставляя имя и расширение.
  //
  function  ExtractFileNameExt(const FileName: LongString):LongString;
  //
  // 133  - Выделяет из полного имени файла только имя, без пути и расширения.
  //
  function  ExtractFileName(const FileName: LongString):LongString;
  //
  // 134  - Выделяет из полного имени файла расширение, включая точку.
  //
  function  ExtractFileExt(const FileName: LongString): LongString;
  //
  // 135  - Выделяет из полного имени файла путь, без имени и расширения.
  //        Обратный слеш в конце пути тоже удаляется.
  //
  function  ExtractFilePath(const FileName: LongString): LongString;
  //
  // 136  - Выделяет из полного пути файла имя диска или имя сетевого сервера.
  //
  function  ExtractFileDrive(FileName: LongString): LongString;
  //
  // 137  - Выделяет расширение файла из описания фильтра, как это принято
  //        Delphi: 'Text files | *.txt | Data files | *.dat |'.
  //
  function  ExtractFilterExt(const Filter:LongString;
                             Index:Integer):LongString;
  //
  // 138  - Преобразует имя файла к канонической форме, удаляя все относительные
  //        ссылки. В результате получается абсолютное имя файла.
  //        Если имя файла пустое, вернет текущий каталог.
  //
  function  FExpand(const Path:LongString):LongString;
  //
  // 139  - Преобразует имя файла Path из абсолютного в относительное, считая
  //        базовым путем путь файла Base.
  //
  function  MakeRelativePath(const Path,Base:LongString):LongString;
  //
  // 140  - Возвращает русскую строку Rus или английскую Eng в зависимости от
  //        того, в каком режиме запущена программа.
  //
  function  RusEng(const Rus:LongString='';
                   const Eng:LongString=''):LongString;
  //
  // 141  - Возвращает строку системной ошибки Windows по коду GetLastError.
  //
  function  SysErrorMessage(ErrorCode: Integer): LongString;
  //
  // 142  - Возвращает атрибуты файла с именем FileName.
  //        Возвращает -1, если  имя файла недопустимо, или если файл является
  //        групповой маской (*,?), или если файл не существует.
  //
  function  GetFileAttr(const FileName:LongString):Integer;
  //
  // 143  - Задает атрибуты файла с именем FileName.
  //        Возвращает false, если  имя файла недопустимо, или если файл является
  //        групповой маской (*,?), или если файл не существует.
  //
  function  SetFileAttr(const FileName: LongString;
                        Attr: Integer):boolean;
  //
  // 144  - Возвращает размер файла или -1, если файл не существует или ошибка.
  //
  function  GetFileSize(const FileName:LongString):LongInt;
  //
  // 145  - Возвращает время и дату файла или -1, если файла не существует.
  //        Возвращаемое значение задано в миллисекундах от Рождества Христа,
  //        если ConvertToMSec=true. Иначе время задано в упакованной форме,
  //        которую можно распаковать функцией FileDateToDateTime.
  //
  function  GetFileDate(const FileName:LongString;
                        ConvertToMsec:Boolean=false):Int64;
  //
  // 146  - Возвращает true, если заданный файл или файлы существуют.
  //        FilePattern задает имя файла или групповую маску, содержащую
  //        шаблон поиска, например "*.*".
  //
  function  FileExists(const FilePattern:LongString;
                       Attribut:Integer=faAnyFile):Boolean;
  //
  // 147  - Проверяет, существует ли заданный каталог.
  //        Имя каталога не должно быть групповой маской.
  //
  function  DirExists(const DirName: LongString): Boolean;
  //
  // 148  - Удаляет файл. Возвращает true, если файл не существовал или был
  //        успешно удален с диска.
  //
  function  FileErase(const FileName:LongString;
                      CheckExistance:boolean=true):boolean;
  //
  // 149  - Копирует файл CopyFrom в файл CopyTo.
  //        Возвращает false при ошибке ввода-вывода, отсутствии файла CopyFrom,
  //        а также при наличии файла CopyTo и флаге FailIfExists=true.
  //
  function  FileCopy(const CopyFrom,CopyTo:LongString;
                     FailIfExists:boolean=false):boolean;
  //
  // 150  - Перемещает OldFileName в файл NewFileName.
  //        Возвращает false при ошибке ввода-вывода, отсутствии файла OldFileName,
  //        а также при наличии файла NewFileName и флаге FailIfExists=true.
  //
  function  FileRename(const OldFileName,NewFileName:LongString;
                       FailIfExists:boolean=false):boolean;
  //
  // 151  - Возвращает текущий каталог без обратного слеша в конце.
  //
  function  GetCurrDir: LongString;
  //
  // 152  - Задает текущий каталог. Имя каталога не должно быть групповой маской.
  //
  function  SetCurrDir(const DirName: LongString): Boolean;
  //
  // 153  - Проверяет наличие и при необходимости создает каталог с заданным
  //        именем DirName. Если родительские каталоги не существуют, они также
  //        создаются. Возвращает true, если каталог существовал или был успешно
  //        создан.
  //
  function  MkDir(const DirName: LongString): Boolean;
  //
  // 154  - Удаляет каталог DirName. Каталог должен быть пустым.
  //        Возвращает false, если каталог не существует или не пуст.
  function  RmDir(const DirName: LongString): Boolean;
  //
  // 155  - Открывает файл FileName, смещается на Offset байт от начала файла,
  //        читает Count байтов в Buffer.
  //        Возвращает число прочитанных байт или -1 при ошибке.
  //
  function  ReadFileToBuffer(const FileName : LongString;
                                   Buffer   : Pointer;
                                   Count    : LongInt;
                                   Offset   : LongInt = 0 ):LongInt;
  //
  // 156  - Открывает или создает FileName, смещается на Offset байт от начала
  //        файла, записывает Count байт из Buffer.
  //        Возвращает число записанных байт или -1 при ошибке.
  //        FLAGS может быть OPEN_EXISTING, OPEN_ALWAYS, CREATE_NEW, CREATE_ALWAYS
  //
  function  WriteBufferToFile(const FileName     : LongString;
                                    Buffer       : Pointer;
                                    Count        : LongInt;
                                    Offset       : LongInt = 0;
                                    FLAGS        : DWORD = OPEN_ALWAYS
                                               ) : LongInt;
  //
  // 157  - Мощный файловый итератор. Для каждого файла, заданного шаблоном
  //        поиска FilePatterns, начиная с каталога RootDir, вызывает процедуру
  //        FileAction, затем вызывает себя рекурсивно для обхода подкаталогов,
  //        максимальный уровень вложенности подкаталогов задается в переменной
  //        MaxSubDirLevels. Указатель CustomData служит для передачи в процедуру
  //        FileAction пользовательских данных. FilePatterns может содержать
  //        список имен файлов или список групповых шаблонов поиска, например,
  //        "*.bmp, config.*, *.wav". Delimiters определяет набор символов,
  //        разделяющих шаблоны поиска в списке FilePatterns.
  //        
  procedure ForEachFile(const RootDir         : LongString;
                        const FilePatterns    : LongString;
                              FileAction      : TForEachFileAction;
                              MaxSubDirLevels : Integer = 0;
                              CustomData      : Pointer = nil;
                        const Delimiters      : TCharSet = [' ',';',',',ASCII_Tab,ASCII_CR,ASCII_LF]
                                             );
  //
  // 158  - Мощный итератор для текстовых файлов. Для каждой строки текстового
  //        файла FileName вызывает процедуру Action. Возвращает 0, если все в
  //        порядке, -1 если файл не существует или IOResult при ошибке.
  //
  function  ForEachTextLine(const FileName   : LongString;
                                  Action     : TForEachTextLineAction;
                                  CustomData : Pointer = nil
                                           ) : Integer;
  //
  // 159  - Вывод строки S в отладочный файл номер n. Отладочный файл n надо
  //        открыть процедурой DebugOutOpenFile. Если файл не открыт, ничего
  //        не происходит. Вывод в отладочный файл потокобезопасен и
  //        буферизован, так что вызывать его можно отовсюду.
  //
  procedure DebugOut(n:byte; const S:LongString);
  //
  // 160  - Возвращает имя отладочного файла номер n.
  //        Возвращает пустую строку, если отладочный файл не был открыт.
  //
  function  DebugOutGetFile(n:byte):LongString;
  //
  // 161  - Возвращает 0 если все OK, или число потерянных байт в отладочном
  //        файле n или -1 если файл не был открыт.
  //
  function  DebugOutGetErrors(n:byte):Int64;
  //
  // 162  - Открывает отладочный файл номер n. Для каждого отладочного файла
  //        создается буфер FIFO, а также поток для его выталкивания на диск.
  //        FlushPeriod - период выталкивания FIFO на диск в миллисекундах.
  //        Для закрытия отладочного файла надо вызывать DebugOutOpenFile
  //        с пустым именем файла FileName.
  //
  procedure DebugOutOpenFile(n           : byte;
                       const FileName    : LongString;
                             FifoSize    : Integer = 1024*64;
                             FlushPeriod : Integer = 50;
                             CreateNew   : boolean = true;
                             Hello       : boolean = false);
  //
  // 163  - Выделяет текст секции [SectionName] из ini-файла FileName.
  //        Биты ExtractSectionFlags задают разные флаги алгоритма выделения.
  //        Обычно используются флаги efAsIs или efConfig.
  //
  function  ExtractTextSection(const FileName            : LongString;
                               const SectionName         : LongString;
                                     ExtractSectionFlags : Word
                                                       ) : LongString;
  //
  // 164  - Выделяет список слов с заданным индексом WordIndex из секции
  //        [SectionName] из ini-файла FileName.
  //
  //
  function  ExtractWordList(const FileName            : LongString;
                            const SectionName         : LongString;
                                  ExtractSectionFlags : word;
                                  WordIndex           : word;
                            const Delimiters          : TCharSet
                                                    ) : LongString;
  //
  // 165  - Выделяет из секции [SectionName] из ini-файла FileName список всех
  //        переменных типа "Prefix = var1, var2, ..".
  //
  function  ExtractEnumWordList(const IniFile             : LongString;
                                const SectionName         : LongString;
                                const Prefix              : LongString;
                                      ExtractSectionFlags : word
                                                        ) : LongString;
  //
  // 166  - Чтение переменных из ini-файла. FileName,SectionName задают имя
  //        файла и секции. Format задает формат чтения - список выражений
  //        вида 'Name%x', разделенных символами CR или ";". Здесь Name - имя
  //        переменной, %x - формат, это одно из выражений
  //         %b - Boolean,
  //         %f - Double,
  //         %d - LongInt,
  //         %w - Word,
  //         %i - Integer,
  //         %s - PureString, строка символов до конца строки текста,
  //         %a - PureString, символьная строка, только первое слово.
  //        Запись данных Data должна содержать упакованные (packed) данные
  //        в соответствии с форматом чтения. Например, для чтения данных
  //         [Person]
  //         Age = 34
  //         Height = 182.5
  //         User = Alex, Physics department
  //        можно написать:
  //         var Person : packed record
  //              Age:Integer; Height:Double; Name,Comment:PureString;
  //             end;
  //         if ReadIniFileVariable(File,'[Person]','Age%i;Height%f;User%a;%s',Data)
  //         then ... Data.Age=34, Data.Height=182.5, Data.Name='ALEX', Data.Comment='PHYSICS DEPARTMENT'
  //         Примечания:
  //         1) Числа могут вводиться в шестнадцатеричном виде типа $F123.
  //         2) Строки всегда преобразуются в верхний регистр. Если это не
  //            подходит, используйте комбинацию ExtractTextSection и ScanVar.
  //
  function ReadIniFileVariable(const FileName    : LongString;
                               const SectionName : LongString;
                               const Format      : LongString;
                                 var Data      ) : Boolean;
  //
  // 167  - Читает имя файла из ini-файла. Возвращается всегда абсолютное (или
  //        полное) имя файла. Если в файле задано относительное имя файла,
  //        то абсолютное имя вычисляется относительно пути StartupPath.
  //
  function ReadIniFilePath(const FileName    : LongString;
                           const SectionName : LongString;
                           const Name        : LongString;
                           const StartupPath : LongString;
                             var Path        : LongString ) : Boolean;
  //
  // 168  - Возвращает абсолютное имя EXE-файла текущей выполняемой программы.
  //
  function  ProgName:LongString;
  //
  // 169  - Домашний каталог программы, то есть каталог, где расположен EXE-файл
  //        выполняемой программы. Обычно все пути файлов программы указаны
  //        относительно ее домашнего каталога.
  //
  function  HomeDir:LongString;
  //
  // 170  - Домашний каталог программы, синоним HomeDir.
  //
  function  StartupPath:LongString;
  //
  // 171  - Текущий каталог в момент старта программы. Он не обязательно
  //        совпадает с домашним каталогом программы.
  //
  function  StartAtDir:LongString;
  //
  // 172  - Основной ini-файл программы, в котором содержатся параметры для ее
  //        запуска и настройки. Этот файл получается заменой расширения .EXE
  //        исполняемого файла программы на расширение .INI.
  //
  function  SysIniFile:LongString;
  //
  // 173  - Перейти в домашний каталог программы.
  //
  procedure GoHome;
  //
  // 174  - Каталог, где расположен Windows, например,  C:\Windows.
  //
  function  WindowsDir:LongString;
  //
  // 175  - Каталог системных файлов Windows, например, C:\Windows\System32.
  //
  function  WindowsSystemDir:LongString;
  //
  // 176  - Каталог временных файлов Windows.
  //
  function  WindowsTempDir:LongString;
  //
  // 177  - Каталог временных файлов программы. Описывается в SysIniFile,
  //        в секции [System], переменная TempDir.
  //
  function  TempDir:LongString;
  //
  // 178  - Создает имя временного файла, которое гарантированно не совпадает
  //        ни с каким существующим файлом. Имя имеет вид ###NNNN.TMP, где
  //        ### - заданный префикс, NNNN - уникальный номер, .TMP - заданное
  //        расширение.
  //
  function  CreateTempFile(const Template:LongString='###.TMP'):LongString;

  //
  // 179  - Чтение переменной - строки Name из каталога Key системного реестра Windows.
  //        RootKey задает ключ, например, HKEY_CURRENT_USER.
  //
  function  ReadRegistryString(RootKey:HKEY;
                               const Key,Name:LongString):LongString;
  //
  // 180  - Чтение переменной - строки Name из системного реестра Windows,
  //        ключ HKEY_CURRENT_USER,
  //        каталог 'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders'
  //
  function  GetWindowsShellFolder(const Name:LongString):LongString;
  //
  // 181  - Возвращает каталог "Рабочий стол" текущего пользователя  Windows.
  //
  function  GetWindowsShellDesktop:LongString;
  //
  // 182  - Возвращает каталог "Программы" текущего пользователя  Windows.
  //
  function  GetWindowsShellPrograms:LongString;
  //
  // 183  - Возвращает каталог "Программы\Автозагрузка" текущего пользователя  Windows.
  //
  function  GetWindowsShellStartup:LongString;
  //
  // 184  - Возвращает каталог "Старт" текущего пользователя  Windows.
  //
  function  GetWindowsShellStartMenu:LongString;
  //
  // 185  - Возвращает каталог "Избранное" текущего пользователя  Windows.
  //
  function  GetWindowsShellFavorites:LongString;
  //
  // 186  - Возвращает каталог фонтов Windows.
  //
  function  GetWindowsShellFonts:LongString;
  //
  // 187  - Возвращает каталог "История" текущего пользователя Windows.
  //
  function  GetWindowsShellHistory:LongString;
  //
  // 188  - Возвращает каталог "Мои документы" текущего пользователя  Windows.
  //
  function  GetWindowsShellPersonal:LongString;
  //
  // 189  - Возвращает каталог "Send To"  текущего пользователя Windows.
  //
  function  GetWindowsShellSendTo:LongString;
  //
  // 190  - Создает ярлык с именем LinkPath, который ссылается на файл ObjectPath.
  //        Description содержит информационную строку описания файла.
  //        Params содержит параметры командной строки, передаваемой программе.
  //
  procedure CreateFileLink(const ObjectPath, LinkPath, Description,
                           Params: LongString);
  //
  // 191  - Задает целочисленную точку.
  //
  function  Point2I(x,y:Integer):TPoint2I;
  //
  // 192  - Задает вещественную точку.
  //
  function  Point2D(x,y:Double):TPoint2D;
  //
  // 193  - Задает вещественную трехмерную точку.
  //
  function  Point3D(x,y,z:Double):TPoint3D;
  //
  // 194  - Задает целочисленный прямоугольник.
  //
  function  Rect2I(const a,b:TPoint2I):TRect2I;
  //
  // 195  -  Задает вещественный прямоугольник.
  //
  function  Rect2D(const a,b:TPoint2D):TRect2D;
  //
  // 196  - Сравнивает, равны ли две целочисленные точки.
  //
  function  IsEqualPoint2I(const p1,p2:TPoint2I):Boolean;
  //
  // 197  - Сравнивает, равны ли две вещественные точки.
  //
  function  IsEqualPoint2D(const p1,p2:TPoint2D; AbsEps:Double=0;
                           RelEps:Double=1E-14):Boolean;
  //
  // 198  - Сравнивает, равны ли два целочисленных прямоугольника.
  //
  function  IsEqualRect2I(const r1,r2:TRect2I):Boolean;
  //
  // 199  - Сравнивает, равны ли два вещественных прямоугольника.
  //
  function  IsEqualRect2D(const r1,r2:TRect2D; AbsEps:Double=0;
                          RelEps:Double=1E-14):Boolean;
  //
  // 200  - Определяет, является ли целочисленный прямоугольник пустым.
  //
  function  IsEmptyRect2I(const r:TRect2I):Boolean;
  //
  // 201  - Определяет, является ли вещественный прямоугольник пустым.
  //
  function  IsEmptyRect2D(const r:TRect2D; AbsEps:Double=0;
                          RelEps:Double=1E-14):Boolean;
  //
  // 202  - Проверяет, содержит ли целочисленный прямоугольник заданную точку.
  //
  function  Rect2IContainsPoint(const r:TRect2I;
                                const p:TPoint2I):Boolean;
  //
  // 203  - Проверяет, содержит ли вещественный прямоугольник заданную точку.
  //
  function  Rect2DContainsPoint(const r:TRect2D;
                                const p:TPoint2D):Boolean;
  //
  // 204  - Находит пересечение целочисленных прямоугольников.
  //
  function  Rect2IIntersection(const R1,R2:TRect2I):TRect2I;
  //
  // 205  - Находит пересечение вещественных прямоугольников.
  //
  function  Rect2DIntersection(const R1,R2:TRect2D):TRect2D;
  //
  // 206  - Находит объединение целочисленных прямоугольников.
  //
  function  Rect2IUnion(const R1,R2:TRect2I):TRect2I;
  //
  // 207  - Находит объединение вещественных прямоугольников.
  //
  function  Rect2DUnion(const R1,R2:TRect2D):TRect2D;
  //
  // 208  - Нормализует целочисленный прямоугольник, если он "вывернут наизнанку".
  //        То есть меняет a.x и b.x если a.x > b.x; меняет a.y и b.y если a.y > b.y.
  //
  function  Rect2IValidate(const r:TRect2I):TRect2I;
  //
  // 209  - Нормализует вещественный прямоугольник, если он "вывернут наизнанку".
  //        То есть меняет a.x и b.x если a.x > b.x; меняет a.y и b.y если a.y > b.y.
  //
  function  Rect2DValidate(const r:TRect2D):TRect2D;
  //
  // 210  - Сдвиг целочисленной точки.
  //
  procedure Point2IMove(var p:TPoint2I; dx,dy:Integer);
  //
  // 211  - Сдвиг вещественной точки.
  //
  procedure Point2DMove(var p:TPoint2D; dx,dy:Double);
  //
  // 212  - Сдвиг целочисленного прямоугольника.
  //
  procedure Rect2IMove(var r:TRect2I; dx,dy:Integer);
  //
  // 213  - Сдвиг вещественного прямоугольника.
  //
  procedure Rect2DMove(var r:TRect2D; dx,dy:Double);
  //
  // 214  - Умножение целочисленной точки.
  //
  procedure Point2IMult(var p:TPoint2I; mx,my:Integer);
  //
  // 215  - Умножение вещественной точки.
  //
  procedure Point2DMult(var p:TPoint2D; mx,my:Double);
  //
  // 216  - Умножение целочисленного прямоугольника.
  //
  procedure Rect2IMult(var r:TRect2I; mx,my:Integer);
  //
  // 217  - Умножение вещественного прямоугольника.
  //
  procedure Rect2DMult(var r:TRect2D; mx,my:Double);
  //
  // 218  - Симметричное расширение целочисленного прямоугольника.
  //
  procedure Rect2IGrow(var r:TRect2I; dx,dy:Integer);
  //
  // 219  - Симметричное расширение вещественного прямоугольника.
  //
  procedure Rect2DGrow(var r:TRect2D; dx,dy:double);
  //
  // 220  - Возвращает размер целочисленного прямоугольника.
  //
  function  Rect2ISize(const r:TRect2I):TPoint2I;
  //
  // 221  - Возвращает размер вещественного прямоугольника.
  //
  function  Rect2DSize(const r:TRect2D):TPoint2D;
  //
  // 222  - Возвращает центр целочисленного прямоугольника.
  //
  function  Rect2ICenter(const r:TRect2I):TPoint2I;
  //
  // 223  - Возвращает центр вещественного прямоугольника.
  //
  function  Rect2DCenter(const r:TRect2D):TPoint2D;
  //
  // 224  - Создание экземпляра интерпретатора выражений.
  //        Не забудьте удалить его после использования вызовом Kill(…).
  //
  function  CreateScriptInterpreter:IScriptInterpreter;
  //
  // 225  - Вычисляет 1 SHL BitNumber, то есть число, у которого все биты
  //        кроме заданного равны 0.
  //
  function  GetBitMask(BitNumber:Integer):Integer;
  //
  // 226  - Вычисляет для числа точность, по заданным абсолютным и относительным
  //        порогам, которую можно использовать при сравнении с другим числом.
  //
  function  Precision(AbsEps,RelEps,Value:Double):Double;
  //
  // 227  - Возвращает максимальное целое число, которое точно представляется
  //        в виде Single. 16.777.216 <-> 24 bit
  //
  function  MaxIntSingle:Single;
  //
  // 228  - Возвращает максимальное целое число, которое точно представляется
  //        в виде Double. 9.007.199.254.740.992 <-> 53 bit
  //
  function  MaxIntDouble:Double;
  //
  // 229  - Возвращает максимальное целое число, которое точно представляется
  //        в виде Extended. 18.446.744.073.709.551.600 <-> 64 bit
  //
  function  MaxIntExtended:Extended;
  //
  // 230  - Сдвиг вещественного числа. Эквивалентно умножению на степень 2:
  //        fscale(X,N) = X * 2 ^ N
  //
  function  fscale(X:Extended; N:integer): Extended;
  //
  // 231  - Выделение экспоненты и мантиссы числа X.
  //        X = Mantissa * 2 ^ Exponent, 1<=abs(Mantissa)<2
  //
  procedure fxtract(X: Extended; var Mantissa: Extended;
                                 var Exponent: Integer);
  //
  // 232  - Сжимает число, то есть округляет его так, что в нем остается
  //        только NumberOfSignificantBits значимых двоичных знаков.
  //
  function  fbitcompress(Argument:Extended;
                         NumberOfSignificantBits:integer):Extended;
  //
  // 233  - Сжимает число, то есть округляет его так, что оно представляет
  //        исходное значение с заданной относительной точностью, равной
  //        1/ScaleFactor.
  //
  function  frelcompress(Arument,ScaleFactor:Extended):Extended;
  //
  // 234  - Сжимает число, то есть округляет его так, что оно представляет
  //        исходное значение с заданной абсолютной точностью, равной
  //        1/ScaleFactor.
  //
  function  fabscompress(Argument,ScaleFactor:Extended):Extended;
  //
  // 235  - Вычисляет гамма - функцию. Для целых чисел gamma(n+1) = n!
  //
  function  Gamma(x:Extended):Extended;
  //
  // 236  - Вычисляет натуральный логарифм гамма-функции.
  //
  function  GammaLn(x:Extended):Extended;
  //
  // 237  - Вычисляет неполную гамма - функцию.
  //
  function  GammaInc(x,a:Extended):Extended;
  //
  // 238  - Вычисляет плотность распределения гамма функции.
  //
  function  GammaDistr(x,a,b:Extended):Extended;
  //
  // 239  - Вычисляет кумулятивное распределение гамма - функции.
  //
  function  GammaCumDistr(x,a,b:Extended):Extended;
  //
  // 240  - Вычисляет обратное кумулятивное распределение гамма-функции.
  //
  function  GammaCumDistrInv(p,a,b:Extended):Extended;
  //
  // 241  - Вычисляет плотность распределения Хи-квадрат с v степенями свободы.
  //
  function  Chi2Distr(x,v:Extended):Extended;
  //
  // 242  - Вычисляет кумулятивное распределение Хи-квадрат с v степенями свободы.
  //
  function  Chi2CumDistr(x,v:Extended):Extended;
  //
  // 243  - Вычисляет обратное кумулятивное распределение Хи-квадрат с v степенями свободы.
  //
  function  Chi2CumDistrInv(p,v:Extended):Extended;
  //
  // 244  - Вычисляет площадь трапеции по отрезком (x1,y1)-(x2,y2).
  //        Полезно для интегрирования методом трапеций, например.
  //
  function  Trapezium(x1,y1,x2,y2:double):double;
  //
  // 245  - Вычисляет простую и модифицированную функцию Бесселя порядка N.
  //
  function  BesselN(N:integer; X:Double;Modify:boolean=false):Double;
  //
  // 246  - Решает квадратную систему линейных уравнений A*X=Y размера N*N
  //        методом Гаусса с выбором главного диагонального элемента.
  //        Матрица A хранится в одномерном массиве, причем для доступа к
  //        элементам используется правило:
  //          A[i,j] <-> A[i*M+j], i,j=0..N-1
  //        При таком способе хранения матрицу A можно хранить также в двумерном
  //        массиве размерности M, например:
  //         const N=3; M=5;
  //         var A:array[0..M-1,0..M-1] of double; X,Y:array[0..M-1] of double;
  //         if GaussSolverWithPartialPivoting(N,M,A[0,0],X[0],Y[0])=0 then ...
  //        Параметр Eps задает порог для ведущего диагонального элемента,
  //        ниже которого матрица считается вырожденной.
  //        Возвращает значение 0 - Ok.
  //                            1 - Неверная размерность N или M.
  //                            2 - Матрица сингулярна, ведущая диагональ <= Eps.
  //
  function  GaussSolverWithPartialPivoting(N,M:Integer; var A,X,Y:array of Double;
                                           Eps:Double=1E-14):Integer;
  //
  // 247  - Создает экземпляр полинома для аппроксимации данных методом
  //        наименьших квадратов.
  //        Не забудьте удалить объект после использования вызовом Kill(…).
  //
  function  CreatePolynom(aPower  : Integer = 1;
                          aCenter : Double  = 0;
                          aScale  : Double  = 1):IPolynomWrap;
  //
  // 248  - Общая процедура минимизации при отсутствии ограничений.
  //        Problem=целевая функция, Report=процедура оповещения, n=размерность,
  //        x[n]=аргумент, f=значение, g[n]=градиент, Custom=пользовательские данные, Count=счетчик вызовов,
  //        MaxCount=ограничение на счетчик вызовов, Method=метод минимизации 0..8, tolx,tolf,tolg=точность по x,f,g,
  //        inf=оценка минимума, Size=начальный симплекс, Step=начальный шаг, EvalTol=точность вычисления функции,
  //        v[]=рабочий массив не менее (n+9)*nv, nv>n, nv=размерность рабочего массива,nv>=n+1.
  //
  function  GenMin(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer):Integer;
  //
  // 249  - Общая процедура минимизации при отсутствии ограничений.
  //        Добавляется массив фиксации переменных Fix[n].
  //
  function  GenMinFix(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer;
                  var Fix:array of Boolean):Integer;
  //
  // 250  - Общая процедура минимизации при отсутствии ограничений.
  //        Добавляется масштабирование функции fscale и диапазон аргумента xLo[n]..xHi[n].
  //
  function  GenMinScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer; fscale:Double;
                  var xLo,xHi:array of Double):Integer;
  //
  // 251  - Общая процедура минимизации при отсутствии ограничений.
  //        Используется фиксация и масштабирование.
  //
  function GenMinFixScaled(Problem:TMinProblem; Report:TMinReport; n:Integer; var x:array of Double;
                  var f:Double; var g:array of Double; Custom:Pointer; var Count:Integer;
                  MaxCount:Integer; Method:Integer; tolx,tolf,tolg,inf,Size,Step,EvalTol:Double;
                  var v:array of Double; nv:Integer; var Fix:array of Boolean; fscale:Double;
                  var xLo,xHi:array of Double):Integer;
  //
  // 252  - Общая процедура фитирования по МНК.
  //        m=число точек, t[m],y[m],w[m]=аргументы,значения,веса фитируемых точек,
  //        Fit=аппроксимирующая функция, Report=процедура отображения, n=размерность,
  //        x[n]=параметры, f=сумма квадратов, g[n]=градиент по параметрам, Custom=данные юзера,
  //        Count=счетчик вызовов, MaxCount=ограничение на счетчик вызовов, Method=метод минимизации 0..8,
  //        tolx,tolf,tolg=точность по x,f,g, inf=оценка минимума, Size=начальный симплекс, Step=начальный шаг,
  //        EvalTol=точность вычисления функции, a[]=рабочий массив не менее (n+10)*(n+1),
  //        Fix[n]=массив фиксации переменных, fscale=масштаб функции, xLo[n],xHi[n]=диапазон параметров
  //
  function  LSQFit(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; Report:TFitReport;
                   n:Integer; var x:array of Double; var f:Double; var g:array of Double;
                   Custom:Pointer; var Count:Integer; MaxCount:Integer; Method:Integer;
                   tolx,tolf,tolg,Inf,Size,Step,EvalTol:Double; var a:array of Double;
                   var Fix:array of Boolean; fscale:Double; var xLo:array of Double;
                   var xHi:array of Double):Integer;
  //
  // 253  - Находит сумму квадратов отклонений, аргументы как в LSQFit.
  //
  function  LSQFitSumOfSquares(m:Integer; var t,y,w:array of Double; Fit:TFitProblem; n:Integer;
                               var x:array of Double; var f:Double; var g:array of Double;
                               Custom:Pointer):Integer;
  //
  // 254  - Дает доступ к базе данных РФА (свойства химических элементов и линии ХРИ).
  //
  function  Rfa:IRfaDataBaseWrap;
  //
  // 255  - Функция быстрой сортировки массива произвольного типа и метода хранения.
  //        Надо задать функцию "Compare" для сравнения двух элементов массива,
  //        функцию "KeyOf" для получения ключа сортировки для данного индекса.
  //        Кроме того, задается один из следующих методов сортировки:
  //         "SwapData"  - процедура для сортировки массива "на месте".
  //         "Index"     - массив индексов при "индексированной" сортировке.
  //        При задании сортировки "на месте", в массиве Data будет произведена
  //        перестановка элементов, так что будет выполнено условие:
  //           Compare(Data,KeyOf(Data,i),KeyOf(Data,j))<=0, i<=j
  //        Это значит, Data[i] будет отсортирован как Data[i]<=Data[j], i<=j.
  //        Процедура "SwapData" задает способ перестановки двух элементов массива.
  //        Массив индексов "Index" в этом случае должен быть равен nil.
  //          Если массив данных очень велик или доступен только для чтения, лучше
  //        использовать "индексированную" сортировку (при этом "SwapData"=nil).
  //        При "индексированной" сортировке исходный массив данных не меняется.
  //        Вместо этого заполняется массив индексов "Index", сортирующий массив:
  //           Compare(Data,KeyOf(Data,Index[i]),KeyOf(Data,Index[j]))<=0, i<=j
  //        Это значит, что Data[Index[i]] будет отсортирован:
  //           Data[Index[i]]<=Data[Index[j]], i<=j
  //        Параметр "Method" задает алгоритм 0=ShellSort, 1=QuickSort.
  //        ShellSort - "рабочая лошадка" сортировки, простой нерекурсивный алгоритм,
  //        достаточно быстрый для большинства задач, хотя несколько уступающий по
  //        скорости алгоритму Хаара.
  //        smQuickSort - рекурсивная версия быстрой сортировки Хаара. Не следует
  //        использовать ее при очень большой размерности массива из-за опасности
  //        переполнения стека. Используйте в этом случае ShellSort.
  //        Аргументы:
  //           Count    - Размерность массива Data.
  //           Data     - дает доступ к массиву[0..Count-1] данных для сортировки.
  //           Compare  - сравнивает ключи сортировки:Compare(Data,Key1,Key2)=-1/0/+1=LT/EQ/GT.
  //           KeyOf    - KeyOf(Data,j), j=0..N-1 возвращает ключ сортировки j в массиве Data.
  //           SwapData - процедура перестановки элементов при сортировке "на месте".
  //           Index    - указатель на массив индексов: Data[Index[i]] будет отсортирован.
  //           Method   - 0=ShellSort, 1=QuickSort.
  //
  function  Sort(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                 KeyOf:TKeyOfFunction; SwapData:TSwapProcedure;
                 Index:PIntegerArray; Method:Integer=0):Boolean;
  //
  // 256  - Быстрый поиск для процедур интерполяции.
  // 257 -  Быстрый поиск для работы со списками.
  //        Обе функции выполняют быстрый двоичный поиск в массиве, который
  //        был отсортирован "на месте"  (в этом случае Index=nil) или через
  //        "индексированную" сортировку (в этом случае Index<>nil).
  //        QuickFind возвращает индекс i в отсортированном массиве данных
  //        Data[0..Count-1] который удовлетворяет условиям:
  //            i=0        если Key <  Data[0],
  //            i=Count-2  если Key >= Data[Count-1]
  //            Data[i] <= Key < Data[i+1] в ином случае
  //        QuickFind предназначен специально для процедур интерполяции, где
  //        неважно, принадлежит Key массиву данных или нет, а важно только его
  //        положение относительно элементов массива. QuickFind возвращает
  //        фактически индекс интервала x[i]..x[i+1], в который попадает точка.
  //        QuickSearch ищет ключ Key в отсортированном массиве Data[0..Count-1]
  //        и возвращает true/false если ключ найден/не найден, а также индекс
  //        KeyIndex, удовлетворяющий условиям:
  //             KeyIndex=0        если Key<=Data[0],
  //             KeyIndex=Count    если Key>Data[Count-1]
  //             x[KeyIndex-1] < Key <= x[KeyIndex] в иных случаях
  //        Индекс KeyIndex показывает, куда должен быть вставлен элемент Key,
  //        чтобы после вставки массив был по-прежнему отсортирован.
  //        QuickSearch предназначен специально для процедур работы с сортированными
  //        списками, когда перед вставкой элемента надо проверить факт его
  //        существования в списке, а также найти индекс для вставки.
  //
  function  QuickFind(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                      KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                      Index:PIntegerArray=nil):Integer;
  function  QuickSearch(Count:Integer; Data:Pointer; Compare:TCompareFunction;
                        KeyOf:TKeyOfFunction; KeyToSearch:Pointer;
                        var KeyIndex:Integer; Index:PIntegerArray=nil;
                        Duplicates:Boolean=false):Boolean;
  //
  // 258  - Найти индекс принадлежности точки в отсортированном массиве.
  //        i:=FindIndex(N,x,at,index) возвращает индекс i в массиве x[0..N-1],
  //        отсортированном как x[i]<=x[i+1], для которого выполнено, условие
  //        x[i]<=at<x[i+1]; при этом i=0, если at<x[0]; i=N-2, если at>x[N-2].
  //        Если задан index<>nil, x[i] заменяется везде на x[index[i]].
  //        Предназначена для процедур интерполяции; это быстрый эквивалент
  //        i:=QuickFind(N,@x,CompareDoubles,KeyOfDoubles,@at,index).
  //
  function  FindDoubleIndex(N:Integer; const x:array of Double; at:Double;
                            Index:PIntegerArray=nil):Integer;
  //
  // 259  - Линейная интерполяция таблично заданной функции y(x) в точке t.
  //        Таблица функции (x[0..N-1],y[0..N-1]) должна быть отсортирована
  //        как x[i]<=x[i+1]. Если это условие не выполнено, результат будет
  //        неверным.
  //
  function  LinearInterpolation(const x,y:array of Double; N:Integer;
                                t:Double):Double;
  //
  // 260  - Вычислить полином степени N-1, с коэффициентами P[0..N-1], в точке t.
  //        Вычисления идут по схеме Горнера, то есть
  //         P(t)=(..(P[N-1]*t+P[N-2])*t+..+P[1])*t+P[0]
  //
  function  EvaluatePolynomAt(const P:array of Double; N:Integer;
                              t:Double):Double;
  //
  // 261  - Находит МНК коэффициенты полинома P[0..N-1] степени N-1, который
  //        аппроксимирует таблично заданную функцию y(x)=(x[0..M-1],y[0..M-1]).
  //        Полином вычисляется для масштабированной переменной t=(x-X0)/Scale,
  //        чтобы избежать численной неустойчивости. Потом можно вызывать
  //        EvaluatePolynomAt(P,N,(x-X0)/Scale);
  //
  function  FindLeastSquaresPolynom(var P:array of Double; N:Integer; X0,Scale:Double;
                        const x,y:array of Double; M:Integer):Boolean;
  //
  // 262 - Вычисляет кубический сплайн и его производные в заданной точке t.
  //       Задается массив узлов сплайна x[0..N-1], массивы z[0..N-1],k[0..N-1]
  //       вычисляются процедурой FindSmoothingSpline. Параметр NDeriv=0/1/2/3
  //       задает вычисление значения/1/2/3 производной.
  //
  function  EvaluateSplineAt(N:Integer; const x,z,k:array of Double;
                             t:Double; NDeriv:Integer=0):Double;
  //
  // 263 - Вычисляет коэффициенты k[0..N-1],z[0..N-1] сглаживающего кубического
  //       сплайна, аппроксимирующего функцию y(x), таблично заданную через
  //       массивы x[0..N-1],y[0..N-1], причем x[i] должен быть отсортирован.
  //       Массив p[0..N-1] задает дисперсии точек. Если задать p[i]=0, сплайн
  //       будет проходить точно через точку (x[i],y[i]). Если задать p[i]>0,
  //       получится сглаженное значение с весом 1/p[i]. Если задать все p[i]=0,
  //       получится классический интерполяционный полином. pL,pR,nL,nR задают
  //       левое и правое граничное условие на сплайн, то есть номер и значение
  //       производной на концах.
  //
  function  FindSmoothingSpline(N:Integer; const x,y,p:array of Double;
                                var z,k:array of Double;
                                nL:Integer=2; pL:Double=0;
                                nR:Integer=2; pR:Double=0):Boolean;
  //
  // 264 - Оконное полиномиальное сглаживание.
  // 265 - Оконное интегрирующее сглаживание.
  //       Оконное сглаживание в данной точке t функции y(x), заданной таблицей
  //       (x[0..N-1],y[0..N-1]), отсортированной по x как x[i+1]>=x[i].
  //       Сглаживание на интервале [t-Window..t+Window] окна с шириной Window
  //       вокруг данной точки происходит с ядром w(t)=Kernel((t-Window)/Window).
  //       При полиномиальном сглаживании по точкам окна строится МНК полином
  //       степени Power. При интегральном сглаживании берется интеграл с данной
  //       весовой функцией.
  //
  function  WindowedKernelSmoothing(const x,y:array of Double; N:Integer;
                                    t,Window:Double; Power:Integer; Kernel:TSmoothKernel;
                                    Custom:Pointer=nil):Double;
  function  WindowedIntegrationSmoothing(const x,y:array of Double; N:Integer;
                                         t,Window:Double; Kernel:TSmoothKernel;
                                         Custom:Pointer=nil):Double;
  //
  // 266 - Решение трехдиагональной системы линейных уравнений методом прогонки.
  // 267 - Решение трехдиагональной системы линейных уравнений методом Гаусса.
  // 268 - Решение пятидиагональной системы линейных уравнений методом Гаусса.
  //       Трехдиагональная система относительно x[0..N-1] имеет вид:
  //       a[i]*x[i-1]+b[i]*x[i]+c[i]*x[i+1]=d[i]
  //       Пятидиагональная система относительно x[0..N-1] имеет вид:
  //       a[i]*x[i-2]+b[i]*x[i-1]+c[i]*x[i]+e[i]*x[i+1]+g[i]*x[i+2]=d[i]
  //       Все массивы имеют размерность N и меняются в процессе вычислений.
  //
  procedure Solve3DiagonalSetByProgonka(N:Integer;
                                        var a,b,c,x,d:array of Double);
  procedure Solve3DiagonalSetByGauss(N:Integer;
                                     var a,b,c,x,d:array of Double);
  procedure Solve5DiagonalSetByGauss(N:Integer;
                                     var a,b,c,e,g,x,d:array of Double);
  //
  // 269 - Создает экземпляр защищенного целочисленного вектора.
  // 270 - Создает экземпляр защищенного вещественного  вектора.
  // 271 - Создает экземпляр защищенной целочисленной матицы.
  // 272 - Создает экземпляр защищенной вещественной  матрицы.
  //       После использования не забудьте удалить вектор/матрицу вызовом Kill().
  //
  function  CreateLongIntVector(aLength:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):ILongIntVectorWrap;
  function  CreateDoubleVector(aLength:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):IDoubleVectorWrap;
  function  CreateLongIntMatrix(aRows,aColumns:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):ILongIntMatrixWrap;
  function  CreateDoubleMatrix(aRows,aColumns:Integer; aOrigin:Integer=0;
                         aExceptions:Boolean=true):IDoubleMatrixWrap;
  //
  // 273  - Решение задач линейного программирования симплексным алгоритмом.
  //        LinearProgrammingBySimplex решает задачу линейного программирования вида:
  //          F = minimum( F(X)=-G(X)= C(1)*X(1)+C(2)*X(2)+...+C(N)*X(N) ) либо
  //          G = maximum( G(X)=-F(X)=-C(1)*X(1)-C(2)*X(2)-...-C(N)*X(N) )
  //        при ограничениях:
  //               A(1,1)*X(1)+A(1,2)*X(2)+....+A(1,N)*X(N) (<=,>=,=) B(1)
  //               ......
  //               A(M,1)*X(1)+A(M,2)*X(2)+....+A(M,N)*X(N) (<=,>=,=) B(M)
  //        а также
  //               X(i) >= 0, i=1..N
  //        B(i) могут иметь любой знак.
  //        Входные данные:
  //        Матрица A(i,j) ограничений передается в MatrCoef[1..M,1..N];
  //        Коэффициенты С(i)            - в векторе Problem[1..N];
  //        Правые части неравенств B(i) - в векторе RightSide[1..M]
  //        Знаки неравенств (<=,>=,=)   - в целом векторе Signums[1..M]:
  //         Signum =+1 соответствует знаку '>='
  //         Signum = 0 соответствует знаку '='
  //         Signum =-1 соответствует знаку '<='
  //        Предполагается, что все матрица MatrCoef и векторы Problem, RightSide имеют
  //        индексацию от 1. Это означает, что свойство Origin этих объектов должно иметь
  //        значение 1.
  //        PrintMode-режим вывода: 0-нет вывода; см.флаги si_XXXX
  //        OutPutFile-файл вывода результата или комментариев.
  //        Можно использовать файл OutPut(экран) или Null(нет вывода).
  //        Результат:
  //        Функция возвращает матрицу,строки которой заполнены векторами решений задачи
  //        или nil, если решения не существует или его не удалось получить.
  //        Если матрица решений содержит более одной сторки, это значит, что
  //        симплекс-задача имеет бесконечное число решений,а множество решений
  //        является выпуклой оболочкой строк матрицы решений. Матрица решений также
  //        индексируется с 1.
  //        Решение ведется с точностью, хранящейся в константе Zero; если решений
  //        много, то это значит, что с точностью Zero все они дают оптимальное решение.
  //        Zero не может быть меньше чем машинная точность MachEps.
  //        В переменной  ErrorCode возвращается код ошибки,смотри константы si_XXXX
  //        Требования к памяти:
  //        Размещение рабочих массивов внутри Simplex требует порядка M*M + 5(2*M+N)
  //        дополнительных ячеек кроме входных данных и решения
  //
  function  LinearProgrammingBySimplex(Problem:IDoubleVectorWrap; MatrCoef:IDoubleMatrixWrap;
                            Signums:ILongIntVectorWrap; RightSide:IDoubleVectorWrap;
                            var ErrorCode:Integer; PrintMode:Word; var OutPutFile:Text;
                            Zero:Double=1E-12):IDoubleMatrixWrap;
  //
  // 274  - Вычисление сингулярного разложения матрицы. Применяется для решения
  //        линейных уравнений.
  //        После использования не забудьте удалить вызовом Kill(…).
  //
  function  CreateSingularDecomposition(A:IDoubleMatrixWrap):ISVDWrap;
  //
  // 275  - Создает буфер FIFO для безопасного обмена данными между потоками.
  //        После использования не забудьте удалить вызовом Kill(…).
  //
  function CreateFifo(aSize:LongInt=1024;
                      const aName:LongString=''):IFifoWrap;
  //
  // 276  - Возвращает локальное время в микросекундах, считая от старта
  //        программы. Вызов занимает около 2 микросекунд, против 0.2 микросекунд
  //        для вызова msecnow, поэтому, если микросекундное разрешение
  //        не требуется, лучше использовать msecnow. С другой стороны,
  //        время msecnow имеет квант порядка 10 миллисекунд, а время mksecnow
  //        порядка микросекунды. Кроме того, следует учесть, что время mksecnow
  //        зависит от времени старта программы, а у msecnow время глобальное.
  //        Можно повысить точность глобального времени комбинацией вызова
  //        msecnow для фиксации старта и mksecnow для фиксации интервала,
  //        однако следует иметь в виду, что разрядность мантиссы (53) типа
  //        Double, принятого в CRW-DAQ, позволяет определять глобальное время
  //        с точностью порядка 10 микросекунд, не лучше.
  //
  function  MkSecNow:Extended;
  //
  // 277  - Атомарная операция Target:=Target+Value, возвращает Target+Value.
  //        Может применяться для синхронизации потоков или процессов.
  //        Например:
  //         const Latch:Integer=0;
  //         if LockedAdd(Latch,1)=1 then DoSomething;
  //         LockedSub(Latch,1);
  //        Процедура  DoSomething защищена от одновременного выполнения более
  //        чем одним потоком, если все потоки используют этот код.
  //        Адрес переменной Target должен быть выровнен по границе 32 бит. 
  //
  function  LockedAdd(var Target: Integer; Value: Integer): Integer;
  //
  // 278  - Атомарная операция if Target=Comp then Target:=Exch, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedCompareExchange(var Target: Integer;
                                  Exch, Comp: Integer): Integer;
  //
  // 279  - Атомарная операция Target:=Target-1, возвращает Target-1.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedDec(var Target: Integer): Integer;
  //
  // 280  - Атомарная операция Target:=Value, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedExchange(var Target: Integer;
                               Value: Integer): Integer;
  //
  // 281  - Атомарная операция Target:=Target+Value, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedExchangeAdd(var Target: Integer;
                                   Value: Integer): Integer;
  //
  // 282  - Атомарная операция Target:=Target-1, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedExchangeDec(var Target: Integer): Integer;
  //
  // 283  - Атомарная операция Target:=Target+1, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedExchangeInc(var Target: Integer): Integer;
  //
  // 284  - Атомарная операция Target:=Target-Value, возвращает Target.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedExchangeSub(var Target: Integer;
                                   Value: Integer): Integer;
  //
  // 285  - Атомарная операция Target:=Target+1, возвращает Target+1.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedInc(var Target: Integer): Integer;
  //
  // 286  - Атомарная операция Target:=Target-Value, возвращает Target-Value.
  //        Может применяться для синхронизации потоков или процессов.
  //
  function  LockedSub(var Target: Integer; Value: Integer): Integer;
  //
  // 287 - Удалить начальную и конечную скобку. Полезна при анализе конфигураций.
  //       Например, RemoveBrackets('[Section]')='Section'.
  //
  function  RemoveBrackets(const s:LongString;
                           const Brackets:LongString='[]'):LongString;
  //
  // 288 - Кодировка двоичных данных в MIME, 64-base.
  // В входной строке присутствуют любые символы.
  // В выходной строке присутствуют только символы a..z, A..Z, 0..9, +, /.
  // Применяется для передачи двоичных данных по текстовому каналу связи.
  // Например:
  //  var x,y:double; i,j:integer; s:string;
  //  s:=mime_encode(r2dump(x)+r2dump(y)+i2dump(i)+i2dump(j));
  //  ... send data ...
  //  s:=mime_decode(s);
  //  x:=dump2r(copy(s,1,8));
  //  y:=dump2r(copy(s,9,8));
  //  i:=dump2i(copy(s,17,4));
  //  j:=dump2i(copy(s,21,4));
  //
  function  Mime_Encode(const S:LongString):LongString;
  //
  // 289 - Декодирование данных, закодированных при помощи Mime_Encode.
  //
  function  Mime_Decode(const S:LongString):LongString;
  //
  // 290 - Сравнение строк, не чувствительное к регистру.
  //
  function  IsSameText(const S1,S2:LongString):Boolean;
  //
  // 291 - Detect Unix
  // 292 - Detect Linux
  // 293 - Detect Windows
  // 294 - Detect win32
  // 295 - Detect win64
  // 296 - Detect 16-bit CPU
  // 297 - Detect 32-bit CPU
  // 298 - Detect 64-bit CPU
  // 299 - Detect BigEndian
  // 300 - Detect Little Endian
  // 301 - System CodePage is UTF8
  // 302 - File system is case sensitive
  // 303 - OS name Mode=1:(Unix,Windows) Mode=2:(Unix,Linux,Windows)
  // 304 - Get FPC version         like 3.2.0
  // 305 - Get FPC target OS       like linux
  // 306 - Get FPC target CPU      like x86_64
  // 307 - Get FPC target platform like x86_64-linux
  //
  function IsUnix:Boolean;
  function IsLinux:Boolean;
  function IsWindows:Boolean;
  function IsWin32:Boolean;
  function IsWin64:Boolean;
  function IsCpu16:Boolean;
  function IsCpu32:Boolean;
  function IsCpu64:Boolean;
  function IsCpuBE:Boolean;
  function IsCpuLE:Boolean;
  function IsSysUtf8:Boolean;
  function IsFileNameCaseSensitive:Boolean;
  function OsName(Mode:Integer=1):LongString;
  function GetFpcVersion:LongString;
  function GetFpcTargetOS:LongString;
  function GetFpcTargetCPU:LongString;
  function GetFpcTargetPlatform:LongString;
  //
  // 308 - Detect main thread
  // 309 - Get PID of current process
  // 310 - Allocated memory block count
  // 311 - Allocated memory size, bytes
  //
  function IsMainThread:Boolean;
  function GetCurrentProcessId:TPid;
  function GetAllocMemCount:SizeInt;
  function GetAllocMemSize:SizeInt;
  //
  // 312 - Сброс FPU в состояние по умолчанию.
  //
  procedure ResetFpu;
  //
  // 313 - Convert String to WideString.
  // 314 - Convert WideString to String.
  // 315 - Convert string codepage.
  //
  function StrToWide(const s:LongString):WideString;
  function WideToStr(const s:WideString):LongString;
  function ConvertCP(const s:LongString; cpFrom,cpTo,cpSet:Word; silent:Boolean=false):LongString;
  //
  // 316 - Adapt File Name for current OS rules.
  // 317 - Adapt Executable File Name for current OS rules.
  // 318 - Adapt Dynamic Link Library File Name for current OS rules.
  // 319 - Adapt desktop shortcut/ shell link file (.lnk or .desktop).
  //
  function AdaptFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function AdaptExeFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function AdaptDllFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  function AdaptLnkFileName(const FileName:LongString; Mode:Integer=afnm_Def):LongString;
  //
  // 320 - Get milliseconds since system start (32 bit).
  // 321 - Get milliseconds since system start (64 bit).
  //
  function GetTickCount:Cardinal;
  function GetTickCount64:QWord;
  //
  // 322 - Extract base file name without path and extension.
  // 323 - Extract base name of *.dll or lib*.so library.
  // 324 - Extract file directory path without trailing backslash.
  //
  function  ExtractBaseName(const FileName: LongString):LongString;
  function  ExtractDllBaseName(const FileName:LongString):LongString;
  function  ExtractFileDir(const FileName: LongString): LongString;
  //
  // 325 - Lexical analyser. Check arg is lexeme of specified type, see lex_xxx.
  //
  function IsLexeme(const arg:LongString; typ:Integer):Boolean;
  //
  // 326 - Extract (Name,Value) pair from argument arg like "Name <Sign> Value".
  // Default Sign is "=", so default statement is "Name=Value".
  // Mode flags (1,2) uses to Trim result (Name,Value).
  // Return position of Sign in arg string or 0.
  //
  function ExtractNameValuePair(const arg:LongString; out Name,Value:LongString;
                                const Sign:Char='='; Mode:Integer=3):Integer;
  //
  // 327 - Case insensitive version of Pos
  // 328 - Extended version of Pos with StartPos option.
  // 329 - Return last position of substring (Sub) in string (Str) or return 0.
  // 330 - Return counter of substrings (sub) in string (str) or 0.
  // 331 - Return N-th position of substrings (sub) in string (str) or 0.
  // 332 - Return position (1-based) of CR or LF or zero if one not found.
  //
  function PosI(const Sub:LongString; const Str:LongString):SizeInt;
  function PosEx(const Sub,Str:LongString; StartPos:SizeInt):SizeInt;
  function LastPos(const Sub,Str:LongString):SizeInt;
  function CountPos(const Sub,Str:LongString):SizeInt;
  function NthPos(const Sub,Str:LongString; n:SizeInt):SizeInt;
  function PosEol(const Buf:LongString; Start:SizeInt=1; SkipLines:SizeInt=0):SizeInt;
  //
  // 333 - Copy string tail starting from pos.
  // 334 - String left  part ahead of delimeter or empty string.
  // 335 - String right part after of delimeter or original str.
  // 336 - Return a string centered in a string of Ch with specified width.
  // 337 - Return Trim(S) or Def if result is empty.
  //
  function  TailStr(const S:LongString; Pos:SizeInt):LongString;
  function  StrAheadOf(const S:LongString; Delim:Char):LongString;
  function  StrAfterOf(const S:LongString; Delim:Char):LongString;
  function  CenterPad(const S:LongString; Width:SizeInt; Ch:Char=' '):LongString;
  function  TrimDef(const S,Def:LongString):LongString;
  //
  // 338 - Sort text lines with given comparator or use default sort.
  // 339 - Ensure ending EOL exists.
  // 340 - Ensure result string is located in Heap.
  //
  function SortTextLines(const aTextLines:LongString;
                         Comparator:TStringListSortCompare=nil):LongString;
  function EnsureEndingEol(const S:LongString):LongString;
  function EnsureHeapString(const S:LongString):LongString;
  //
  // 341 - Scan cookies by Name in buffer Buf, i.e. extract Name=Value clause.
  //
  function CookieScan(const Buff,Name:LongString; Mode:Integer=csm_Default):LongString;
  //
  // 342  - Skip N words of S delimited by WordDelims, return string tail.
  //
  function  SkipWords(N:Integer; const S:LongString;
                                 const WordDelims:TCharSet):LongString;
  //
  // 343 - Calculate number of phrases.
  // 344 - Extract phrase with given index number, starting from 1.
  // 345 - Return tail after N phrases, and Delims skipped.
  // 346 - Convert list of phrases to text lines separated by EOL.
  //
  function PhraseCount(const S:LongString; const Delims:TCharSet;
                       Quotes:LongString=QuoteMark+Apostrophe):Integer;
  function ExtractPhrase(N:Integer; const S:LongString; const Delims:TCharSet;
                         Quotes:LongString=QuoteMark+Apostrophe):LongString;
  function SkipPhrases(N:Integer; const S:LongString; const Delims:TCharSet;
                       Quotes:LongString=QuoteMark+Apostrophe):LongString;
  function PhraseListToTextLines(const S:LongString; const Delims:TCharSet;
                                 Quotes:LongString=QuoteMark+Apostrophe):LongString;
  //
  // 347 - Scan for any data record.
  // 348 - Scan for boolean variable %b.
  // 349 - Scan for word    variable %w.
  // 350 - Scan for integer variable %i.
  // 351 - Scan for double  variable %f.
  // 352 - Scan for alpha, i.e. word %a.
  // 353 - Scan for string  variable %s.
  //
  function ScanVarRecord(svMode:Cardinal; Str:PChar; const Format:LongString; var Data):PChar;
  function ScanVarBoolean(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Boolean):PChar;
  function ScanVarWord(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Word):PChar;
  function ScanVarInteger(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Integer):PChar;
  function ScanVarDouble(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:Double):PChar;
  function ScanVarAlpha(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
  function ScanVarString(svMode:Cardinal; Str:PChar; const Key:LongString; var Data:LongString):PChar;
  //
  // 354 - Add \ or / delimiter to tail of path if one not present.
  // 355 - Drop \ or / delimiter from tail of path if one not present.
  // 356 - Validate path delimeters ('\' or '/') in given FileName.
  //
  function  AddPathDelim(const DirName:LongString; Delim:Char=PathDelim):LongString;
  function  DropPathDelim(const DirName:LongString):LongString;
  function ValidatePathDelim(const FileName:LongString; Delim:Char=PathDelim):LongString;
  //
  // 357 - Fetch chaar from string (safe).
  // 358 - Extract first quoted phrase from string (s).
  // 359 - Skip first quoted phrase from string (s).
  // 360 - Extract first parameter - word or quoted string.
  // 361 - Skip    first parameter - word or quoted string.
  // 362 - Quote string if needed (if has spaces inside).
  // 363 - Quote trimmed argument string if needed.
  // 364 - Extract quoted string or URL-encoded (depending on first char).
  // 365 - Dequote or decode URL-encoded string (depending on first char).
  // 366 - Check if string contains option -opt or --opt=value.
  // 367 - Get option value from -opt=value.
  // 368 - Check if argument is option.
  //
  function StrFetch(const s:LongString; i:SizeInt):AnsiChar;
  function AnsiDeQuotedStr(const s:LongString; q:Char):LongString;
  function AnsiSkipQuotedStr(const s:LongString; q:Char):LongString;
  function ExtractFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function SkipFirstParam(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function AnsiQuotedIfNeed(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function QArg(const s:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function ExtractFirstParamUrl(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function Dequote_or_URL_Decode(Line:LongString; quote:Char=QuoteMark; const Spaces:TCharSet=JustSpaces):LongString;
  function IsOption(const arg:LongString; const shortopt:LongString=''; const longopt:LongString='';
                    Mode:Integer=om_Default; Delim:Char='='):Boolean;
  function GetOptionValue(const arg:LongString; Delim:Char='='):LongString;
  function HasOptionValue(const arg:LongString; Delim:Char='='):Boolean;
  //
  // 369 - Convert string to set of chars.
  // 370 - Convert set of chars to string.
  // 371 - Return true if string S has chars in C charset.
  // 372 - Return counter  of chars in C charset if string S.
  //
  function StringToSetOfChars(const S:LongString):TCharSet;
  function SetOfCharsToString(const S:TCharSet):LongString;
  function HasChars(const S:LongString; const C:TCharSet):Boolean;
  function CountChars(const S:LongString; const C:TCharSet):SizeInt;
  //
  // 373 - Get real path name (validate file name and dereference).
  //
  function GetRealFilePathName(const FileName:LongString; BuffSize:Integer=MAX_PATH; Mode:Integer=0):LongString;
  //
  // 374 - Compare chars without   case sensivity.
  // 375 - Compare strings with    case sensivity.
  // 376 - Compare file names.
  // 377 - Detect section name.
  //
  function IsSameChar(C1,C2:Char):Boolean;
  function IsSameStr(S1,S2:LongString):Boolean;
  function IsSameFileName(const S1,S2:LongString; Mode:Integer=ua_FileDef):Boolean;
  function IsSectionName(const aName:LongString):Boolean;
  //
  // 378 - HTTP status message by status code.
  //
  function HTTP_StatusMessage(StatusCode:Integer):LongString;
  //
  // 379 - URL packing (light encoding).
  // 380 - URL encode.
  // 381 - URL decode.
  //
  function URL_Packed(const S:LongString; Mode:Integer=um_Safe; const AllowChars:TCharSet=UrlAllowChars):LongString;
  function URL_Encode(const S:LongString; Mode:Integer=um_Safe+um_Strict; const AllowChars:TCharSet=[]):LongString;
  function URL_Decode(const S:LongString; Mode:Integer=um_Safe):LongString;
  //
  // 382 - Case insensitive scan char.
  // 383 - Reverse case insensitive scan char.
  // 384 - Case insensitive Pos.
  // 385 - Safe string copy.
  //
  function  StrIScan(Str:PChar; Chr:Char):PChar;
  function  StrRIScan(Str:PChar; Chr:Char):PChar;
  function  StrIPos(Str1,Str2:PChar):PChar;
  function  StrPLCopy(Dest:PChar; const Source:LongString; MaxLen:SizeInt):PChar;
  //
  // 386 - Validate EOL
  //
  function ValidateEOL(const aData:LongString; aTail:Integer=0; aEOL:LongString=EOL):LongString;
  //
  // 387 - Run command (cmdline) and read output (outputstring).
  // 388 - Run command (cmdline) in directory (curdir) and read output (outputstring).
  // 389 - Get number of CPU's.
  // 390 - CPU frequency, MHz, by source id. id=(0:default,1:nominal,2:eastimated).
  // 391 - Read /proc/cpuinfo (What field) for CPU`s in CpuMask (1=CPU0,2=CPU1,etc).
  //       Use What='model name' to detect processor model name
  //       like 'Intel(R) Core(TM) i7-4700MQ CPU @ 2.40GHz'.
  // 392 - Return true if handle is terminal.
  // 393 - Get process (pid) affinity mask, PID 0 means the current process.
  // 394 - Get thread (tid) affinity mask.
  // 395 - Get/Set thread (tid) affinity mask.
  //       Tid 0 means the current thread, Mask=0 uses to Get only.
  // 396 - Return list of running processes as lines of: PID, PPID, PRIO, NAME.
  // 397 - Get Process Name with PID specified.
  // 398 - Get Parent Process ID.
  // 399 - Get Parent Process Name.
  // 400 - Current Process times (system s, user u) as FileTime (100 ns units).
  // 401 - Current Thread times (system s, user u) as FileTime (100 ns units).
  //
  function RunCommand(cmdline:LongString; out outputstring:LongString):Boolean;
  function RunCommandInDir(curdir,cmdline:LongString; out outputstring:LongString):Boolean;
  function cpu_count:Integer;
  function cpu_mhz(id:Integer=0):Double;
  function ReadProcCpuInfo(CpuMask:Int64; What:LongString):LongString;
  function IsATTY(Handle:LongInt):Boolean;
  function GetProcessAffinityMask(pid:TPid=0):QWord;
  function GetThreadAffinityMask(tid:SizeInt=0):QWord;
  function SetThreadAffinityMask(tid:SizeInt; Mask:QWord):QWord;
  function GetListOfProcesses(aPid,aPPid:SizeInt; const aName:LongString; Detail:Boolean=false):LongString;
  function GetProcessName(pid:SizeInt=0):LongString;
  function GetParentProcessID:SizeInt;
  function GetParentProcessName:LongString;
  function GetProcessTimesAsFileTime(out s,u:Int64):Boolean;
  function GetThreadTimesAsFileTime(out s,u:Int64):Boolean;
  //
  // 402 - Program base file name.
  // 403 - User $HOME or %USERPROFILE% directory.
  // 404 - System root directory: / for UNIX, %WINDIR% for Windows.
  // 405 - Global (system) temporary directory.
  // 406 - Local (current user) temporary directory.
  // 407 - Find executable of CrwKit - (crwkit,crwrun,unix).
  // 408 - Get command processor %COMSPEC% or $SHELL.
  // 409 - Get command line as it known from OS.
  // 410 - Current user name.
  // 411 - Computer name.
  // 412 - Hostname 0:simple, 1:domain.
  // 413 - Get user domain name.
  // 414 - Get IP address.
  // 415 - Get list of MAC addresses of all local interfaces.
  // 416 - Get list of IP addresses of all local interfaces.
  // 417 - Get list of IP addresses with delemiter (Delim).
  // 418 - List of MAC ADDRESS for Eternet cards. Mode(1)=WithNames.
  // 419 - Get list of local host users.
  //
  function  ProgBaseName:LongString;
  function  UserHomeDir(const SubDir:LongString=''):LongString;
  function  SystemRootDir:LongString;
  function  GlobalTempDir(const SubDir:LongString=''):LongString;
  function  LocalTempDir(const SubDir:LongString=''):LongString;
  function  GetCrwKitExe:LongString;
  function  GetShell:LongString;
  function  GetCommandLine:LongString;
  function  UserName:LongString;
  function  ComputerName:LongString;
  function  HostName(Method:Integer=0):LongString;
  function  UserDomain(const aUser:LongString=''; aDef:LongString=''):LongString;
  function  GetIPAddress(const aHostName:LongString=''):LongString;
  function  GetIfMacAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetIfIpAddresses(Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetIPAddresses(const aHostName:LongString=''; Delim:LongString=','):LongString;
  function  GetMacAddresses(const Machine:LongString=''; Delim:LongString=','; Mode:Integer=0):LongString;
  function  GetLocalUserList(Delim:LongString=EOL):LongString;
  //
  // 420 - Get environment variable.
  // 421 - Set environment variable.
  // 423 - Expand environment variables.
  // 424 - Maybe arg contains environment string.
  // 425 - Maybe arg contains ~ or ~~ reference.
  //
  function GetEnv(const Name:LongString):LongString;
  function SetEnv(const Name,Value:LongString):Boolean;
  function ExpEnv(const Str:LongString):LongString;
  function MaybeEnvStr(const arg:LongString):Boolean;
  function MaybeTildeStr(const arg:LongString):Boolean;
  //
  // 426 - Check file is executable.
  // 427 - Check file is symbolic link.
  // 428 - Check file is hard link.
  // 429 - Check file is readable.
  // 430 - Check file is writable.
  //
  function FileIsExecutable(const aFileName:LongString):Boolean;
  function FileIsSymlink(const aFileName:LongString):Boolean;
  function FileIsHardLink(const aFileName:LongString):Boolean;
  function FileIsReadable(const aFileName:LongString):Boolean;
  function FileIsWritable(const aFileName:LongString):Boolean;
  //
  // 431 - Read (binary) string from file.
  //
  function StringFromFile(const FileName:LongString; MaxSize:SizeInt):LongString;
  //
  // 432 - Get ROOT key my name like HKLM,HKCU etc.
  // 433 - Read Windows registry multistrings.
  // 434 - Read content of *.lnk or *.desktop file and return it as text.
  //
  function  GetRootKeyByName(const Key:LongString):HKEY;
  function  ReadRegistryMultiStrings(RootKey:HKEY; const Key,Name:LongString; Delim:Char=ASCII_CR):LongString;
  function  ReadShellLinkAsText(const FileName:LongString):LongString;
  //
  // 435 - Mask for any file: *.* for Windows, * for Unix.
  // 436 - Find all files in SearchPath directories by names in SearchMask.
  //
  function AnyFileMask:LongString;
  function FindAllFilesAsText(const SearchPath:LongString;
                              const SearchMask:LongString='';
                                    SearchSubDirs:Boolean=True;
                                    DirAttr:Word=faDirectory;
                                    MaskSeparator:Char=';';
                                    PathSeparator:Char=';'
                              ):LongString;
  //
  // 437 - Read data (any record) from Ini file with format specified.
  // 438 - Read data (boolean)    from Ini file with format %b.
  // 439 - Read data (double)     from Ini file with format %f.
  // 440 - Read data (word)       from Ini file with format %w.
  // 441 - Read data (integer)    from Ini file with format %i or %d.
  // 442 - Read data (string)     from Ini file with format %a - extract word.
  // 443 - Read data (string)     from Ini file with format %s - whole string.
  //
  function ReadIniFileRecord(const FileName    : LongString;
                             const SectionName : LongString;
                             const Format      : LongString;
                               var Data;
                                   efMode      : Integer = efConfig;
                                   svMode      : Integer = svConfig
                                             ) : Boolean;
  function ReadIniFileBoolean(const FileName    : LongString;
                              const SectionName : LongString;
                              const Key         : LongString;
                                var Data        : Boolean;
                                    efMode      : Integer = efConfig;
                                    svMode      : Integer = svConfig
                                              ) : Boolean;
  function ReadIniFileDouble(const FileName    : LongString;
                             const SectionName : LongString;
                             const Key         : LongString;
                               var Data        : Double;
                                   efMode      : Integer = efConfig;
                                   svMode      : Integer = svConfig
                                              ) : Boolean;
  function ReadIniFileWord(const FileName    : LongString;
                           const SectionName : LongString;
                           const Key         : LongString;
                             var Data        : Word;
                                 efMode      : Integer = efConfig;
                                 svMode      : Integer = svConfig
                                           ) : Boolean;
  function ReadIniFileInteger(const FileName    : LongString;
                              const SectionName : LongString;
                              const Key         : LongString;
                                var Data        : Integer;
                                    efMode      : Integer = efConfig;
                                    svMode      : Integer = svConfig
                                              ) : Boolean;
  function ReadIniFileAlpha(const FileName    : LongString;
                            const SectionName : LongString;
                            const Key         : LongString;
                              var Data        : LongString;
                                  efMode      : Integer = efConfig;
                                  svMode      : Integer = svConfig
                                            ) : Boolean;
  function ReadIniFileString(const FileName    : LongString;
                             const SectionName : LongString;
                             const Key         : LongString;
                               var Data        : LongString;
                                   efMode      : Integer = efConfig;
                                   svMode      : Integer = svConfig
                                             ) : Boolean;
  //
  // 444 - Get type of file handle.
  // 445 - Return size of big file or -1 if file not exists or error occured.
  // 446 - File has non-block flag (O_NONBLOCK)? Unix Only.
  // 447 - Set or clear non-block flag (O_NONBLOCK) of file. Unix Only.
  // 448 - Set or clear close-on-exec flag (FD_CLOEXEC) of file descritor. Unix Only.
  // 449 - Check ErrorCode for pending file operations on non-block file.
  //       Windows: ERROR_IO_INCOMPLETE. Unix: EAGAIN or EWOULDBLOCK.
  // 450 - ErrorCode is BROKEN_PIPE?
  // 452 - ErrorCode is OPERATION_ABORTED?
  //
  //
  function GetFileType(hFile:THandle):LongWord;
  function GetFileSize64(const FileName:LongString):Int64;
  function FileHasNonBlockFlag(fd:THandle):Boolean;
  function FileSetNonBlockFlag(fd:THandle; State:Boolean=true):Boolean;
  function FileSetCloseOnExec(fd:THandle; State:Boolean=true):Boolean;
  function ePendingFileOperation(ErrorCode:Integer):Boolean;
  function eBrokenPipe(ErrorCode:Integer):Boolean;
  function eOperationAborted(ErrorCode:Integer):Boolean;
  //
  // 453 - Get smart file reference:
  //  1) If FileRef has FULL pathname with extension, return FileRef
  //  2) If FileRef has no extension, set extension DefExt by default
  //  3) If FileRef starts from ~~, it's relative path from program HomeDir
  //  4) If FileRef starts from ~,  it's relative path from user home directory
  //  5) If FileRef has relative path, it's path form BaseFile
  // Example:
  //  SmartFileRef('c:\daq\demo\config\test.cfg') - c:\daq\demo\config\test.cfg
  //  SmartFileRef('test','.cfg',Daq.ConfigFile)  - c:\daq\demo\config\test.cfg
  //  SmartFileRef('~\test','.cfg')               - c:\Documents Add settings\Alex\test.cfg
  //  SmartFileRef('~~\test','.cfg')              - c:\Crw32exe\test.cfg
  //
  function SmartFileRef(const FileRef   : LongString;
                        const DefExt    : LongString = '';
                        const BaseFile  : LongString = '';
                        const UserHome  : LongString = sfrDefUserHome;
                        const ProgHome  : LongString = sfrDefProgHome;
                        const Brackets  : LongString = sfrDefBrackets;
                              handleUrl : Boolean    = true
                                      ) : LongString;

  //
  // 454 - Get short file reference, back from SmartFileRef:
  //  1) Replace program home dir to ~~
  //  2) Replace user home dir    to ~
  //  3) Make relative path, form BaseFile
  //
  function SmartFileRel(const FileRef   : LongString;
                        const BaseFile  : LongString;
                        const UserHome  : LongString = sfrDefUserHome;
                        const ProgHome  : LongString = sfrDefProgHome;
                        const Brackets  : LongString = sfrDefBrackets;
                              handleUrl : Boolean    = true
                                      ) : LongString;
  //
  // 455 - Search file in PATH like unix's which does.
  //       Return empty string if file not found.
  //
  function file_which(name:LongString):LongString;
  //
  // 456 - Get EXE file information:
  // CompanyName,FileDescription,FileVersion,InternalName,LegalCopyright,
  // LegalTradeMarks,OriginalFilename,ProductName,ProductVersion,Comments
  //
  function GetFileVersionInfoAsText(FileName:LongString):LongString;
  //
  // 457 - Print string to stdout.
  // 458 - Print string to stderr.
  // 459 - Print line   to stdout.
  // 460 - Print line   to stderr.
  //
  function StdOutPrint(const Msg:LongString):LongInt;
  function StdErrPrint(const Msg:LongString):LongInt;
  function StdOutPrintLn(const Msg:LongString):LongInt;
  function StdErrPrintLn(const Msg:LongString):LongInt;
  //
  // 461 - checks is file closed or not. File must be assigned before.
  // 462 - checks is text closed or not. File must be assigned before.
  // 463 - close file, if it was opened. File must be assigned before.
  // 464 - close text, if it was opened. File must be assigned before.
  //
  function  isFileClosed(var f:File):Boolean;
  function  isTextClosed(var f:Text):Boolean;
  procedure SmartFileClose(var f:File);
  procedure SmartTextClose(var f:Text);
  //
  // 465 - Convert result of GetFileType to string.
  // 466 - Convert IOResult code to string message.
  //
  function FileTypeToString(fType:LongWord; Prefix:LongString='FILE_TYPE_'):LongString;
  function IoResultToString(Code:Integer):LongString;
  //
  // 467 - Get date/time string as '2024.11.12.31-08:15:30'.
  // 468 - Get date/time prompt as '2024.11.12.31-08:15:30 => '.
  //
  function  StdDateTimeStr(ms:Double=0):LongString;
  function  StdDateTimePrompt(ms:Double=0; Prompt:PChar=nil):LongString;
  //
  // 469 - Write string to registry. Windows only.
  //
  function  WriteRegistryString(RootKey:HKEY; const Key,Name,Data:LongString):LongString;
  //
  // 470 - Debug output text lines.
  //
  procedure DebugOutText(n:Byte; const S:LongString);
  //
  // 471 - Validate PATH list like: SetEnv('PATH',ValidatePathList(GetEnv('PATH')));
  //       Check directory exists, convert short file names to long etc, see GetRealFilePathName.
  // 472 - Validate file search paths from list of environment variables, like:
  //       ValidateEnvPathList('PATH;CRW_DAQ_CONFIG_PATH');
  //
  function ValidatePathList(const PathList:LongString; Delim:Char=PathSep):LongString;
  function ValidateEnvPathList(const EnvPathList:LongString; Delim:Char=PathSep):Integer;
  //
  // 473 - Smart file seach.
  // FileName   is short (without path) file name to seach.
  // PathVarStr is semicolon-delimited list of environment variables, each
  //            variable contains semicolon-delimited list of search
  //            directories. For example, Path environment variable
  //            contains system Windows seach directories list.
  // PathExtVar is environment variable with semicolon-delimeted list of
  //            extensions to seach for, if one not specified in FileName.
  // PathExtStr is direct semicolon-delimeted list of extensions to seach
  //            for, if one not specified in FileName and PathExtVar.
  // Example: arj:=SmartFileSearch('arj.exe','CRW_DAQ_SYS_PATH;Path');
  //
  function SmartFileSearch(const FileName   : LongString;
                           const PathVarStr : LongString='PATH';
                           const PathExtVar : LongString='PATHEXT';
                           const PathExtStr : LongString=DefaultPathExt
                                          ) : LongString;
  //
  // 474 - Read symlink (rerursive) or return regular file name.
  //
  function ReadSymLink(const FileName:LongString; MaxLevel:Integer=10):LongString;
  //
  // 475 - Get System Parameter String.
  //       Category=(Id,OS,Ram,Process,Metrics,Screen).
  //       ParamName depends on category (long list). See crwlib code.
  //
  function GetSystemParamStr(const Category,ParamName:LongString):LongString;
  //
  // 476 - Base64 encode data string.
  // 477 - Base64 decode data string.
  // 478 - Base32 encode data string.
  // 479 - Base64 decode data string.
  // 480 - Backslash encode data string.
  // 481 - Backslash decode data string.
  //
  function  base64_encode(const S:LongString):LongString;
  function  base64_decode(const S:LongString):LongString;
  function  base32_encode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
  function  base32_decode(const sData:LongString; nAlphabet:Integer=Base32_Alphabet_Id_Default):LongString;
  function  backslash_encode(const data:LongString; const esclist:TCharSet=[]; const hexlist:TCharSet=[]):LongString;
  function  backslash_decode(const data:LongString):LongString;
  function  StringOfCharRange(a,b:Char):LongString;
  //
  // 482 - 496 - task API, see DaqPascal help
  //
  function  task_init(const cmd_line:LongString):Integer;
  function  task_free(tid:Integer):Boolean;
  function  task_ref(tid:Integer):Integer;
  function  task_pid(tid:Integer):Integer;
  function  task_run(tid:Integer):Boolean;
  function  task_wait(tid,timeout:Integer):Boolean;
  function  task_send(tid:Integer; const data:LongString):Integer;
  function  task_recv(tid,maxlen:Integer):LongString;
  function  task_txcount(tid:Integer):Integer;
  function  task_rxcount(tid:Integer):Integer;
  function  task_txspace(tid:Integer):Integer;
  function  task_rxspace(tid:Integer):Integer;
  function  task_result(tid:Integer):Integer;
  function  task_kill(tid,how,exit_code,timeout:Integer):Boolean;
  function  task_ctrl(tid:Integer; const arg:LongString):LongString;
  //
  // 497 - Выполнить командную строку CmdLine.
  //       Если ShellCm='Open','Print','Explore', вызвать через ShellExecute
  // 498 - Set/Get process affinity mask. Set mask=0 to get only.
  // 499 - KillProcess     - Terminate process by given PID.
  //
  function SmartExecute(const CmdLine:LongString;
                              Display:Integer=SW_SHOWNORMAL;
                              ShellCm:LongString=''):Boolean;
  function PidAffinity(pid:TPid; mask:Int64=0):Int64;
  function KillProcess(aPid:TPid; aExitCode:Integer; sig:Integer=SIGTERM):Boolean;
  //
  // 500 - 512 - hash list API, see DaqPascal help.
  //
  function HashList_Init(aCaseSensitive:Boolean):Integer;
  function HashList_Free(aRef:Integer):Boolean;
  function HashList_Ref(aRef:Integer):Integer;
  function HashList_Count(aRef:Integer):Integer;
  function HashList_GetKey(aRef:Integer; aIndex:Integer):LongString;
  function HashList_Delete(aRef:Integer; aKey:LongString):Boolean;
  function HashList_IndexOf(aRef:Integer; aKey:LongString):Integer;
  function HashList_GetData(aRef:Integer; aKey:LongString):Double;
  function HashList_SetData(aRef:Integer; aKey:LongString; aData:Double):Boolean;
  function HashList_GetLink(aRef:Integer; aKey:LongString):Integer;
  function HashList_SetLink(aRef:Integer; aKey:LongString; aLink:Integer):Boolean;
  function HashList_GetPara(aRef:Integer; aKey:LongString):LongString;
  function HashList_SetPara(aRef:Integer; aKey:LongString; aParam:LongString):Boolean;
  //
  // 513 - 524 regexp API, see DaqPascal help.
  //
  function regexp_init(engine:Integer; pattern:LongString; options:LongString=''):Integer;
  function regexp_free(rex:Integer):Boolean;
  function regexp_ref(rex:Integer):Integer;
  function regexp_ctrl(rex:Integer; arg:LongString):LongString;
  function regexp_test(rex:Integer; arg:LongString):Boolean;
  function regexp_exec(rex:Integer; arg:LongString):Integer;
  function regexp_replace(rex:Integer; arg,rep:LongString):LongString;
  function regexp_matchnum(rex:Integer; i:Integer):Integer;
  function regexp_matchpos(rex:Integer; i,j:Integer):Integer;
  function regexp_matchlen(rex:Integer; i,j:Integer):Integer;
  function regexp_matchstr(rex:Integer; i,j:Integer):LongString;
  function regexp_escape(arg:LongString):LongString;
  //
  // 525 - 544
  //
  function pipe_init(const cmd_line:LongString):Integer;
  function pipe_free(pid:Integer):Boolean;
  function pipe_ref(pid:Integer):Integer;
  function pipe_pid(pid:Integer):Integer;
  function pipe_run(pid:Integer):Boolean;
  function pipe_wait(pid,timeout:Integer):Boolean;
  function pipe_send(pid:Integer; const data:LongString; awake:Boolean=false):Integer;
  function pipe_recv(pid,maxlen:Integer):LongString;
  function pipe_txcount(pid:Integer):Integer;
  function pipe_rxcount(pid:Integer):Integer;
  function pipe_txspace(pid:Integer):Integer;
  function pipe_rxspace(pid:Integer):Integer;
  function pipe_result(pid:Integer):Integer;
  function pipe_kill(pid,how,exit_code,timeout:Integer):Boolean;
  function pipe_ctrl(pid:Integer; const param:LongString):LongString;
  function pipe_txclear(pid:Integer):Boolean;
  function pipe_rxclear(pid:Integer):Boolean;
  function pipe_count(pid:Integer):Integer;
  function pipe_stream(pid,index:Integer):Integer;
  function pipe_connected(pid:Integer):Integer;
  //
  // 545 - Get special folder (CSIDL) by name.
  // 546 - List all special folders.
  //       CSIDL = constant special identifier list.
  //       CSIDL = (DESKTOP,INTERNET,PROGRAMS,CONTROLS,PRINTERS,PERSONAL,
  //                FAVORITES,STARTUP,RECENT,SENDTO,BITBUCKET,STARTMENU,
  //                MYDOCUMENTS,MYMUSIC,MYVIDEO,DESKTOPDIRECTORY,DRIVES,
  //                NETWORK,NETHOOD,FONTS,TEMPLATES,COMMON_STARTMENU,
  //                COMMON_PROGRAMS,COMMON_STARTUP,COMMON_DESKTOPDIRECTORY,
  //                APPDATA,PRINTHOOD,LOCAL_APPDATA,ALTSTARTUP,COMMON_ALTSTARTUP,
  //                COMMON_FAVORITES,INTERNET_CACHE,COOKIES,HISTORY,COMMON_APPDATA,
  //                WINDOWS,SYSTEM,PROGRAM_FILES,MYPICTURES,PROFILE,SYSTEMX86,
  //                PROGRAM_FILESX86,PROGRAM_FILES_COMMON,PROGRAM_FILES_COMMONX86,
  //                COMMON_TEMPLATES,COMMON_DOCUMENTS,COMMON_ADMINTOOLS,ADMINTOOLS,
  //                CONNECTIONS,COMMON_MUSIC,COMMON_PICTURES,COMMON_VIDEO,RESOURCES,
  //                RESOURCES_LOCALIZED,COMMON_OEM_LINKS,CDBURN_AREA,COMPUTERSNEARME,
  //                XDG_HOME,XDG_DATA_HOME,XDG_CONFIG_HOME,XDG_STATE_HOME,
  //                XDG_CACHE_HOME,XDG_DATA_DIRS,XDG_CONFIG_DIRS,XDG_RUNTIME_DIR,
  //                XDG_DATA_DIR,XDG_CONFIG_DIR,XDG_DESKTOP_DIR,XDG_DOWNLOAD_DIR,
  //                XDG_TEMPLATES_DIR,XDG_PUBLICSHARE_DIR,XDG_DOCUMENTS_DIR,
  //                XDG_MUSIC_DIR,XDG_PICTURES_DIR,XDG_VIDEOS_DIR)
  //
  function CSIDL_FolderByName(const Name:LongString; const sub:LongString;
                 Charset:Integer=-1; const Delim:LongString=''):LongString;
  function CSIDL_ListAllAsText(const Prefix:LongString='CSIDL_';
                 Charset:Integer=-1; SkipEmpty:Boolean=true):LongString;
  //
  // 547 - 553 - Handle file associations
  //
  function  GetSystemAssoc(const Ext:LongString):LongString;
  function  GetSystemFType(const FType:LongString):LongString;
  function  GetSystemAssocExe(const Ext:LongString):LongString;
  function  GetSystemFTypeExe(const FType:LongString):LongString;
  function  HasSystemAssocExe(const Ext:LongString):Boolean;
  function  HasSystemFTypeExe(const FType:LongString):Boolean;
  function  GetExeByExtension(const Extension:LongString):LongString;
  //
  // 554 - Get application path by short name (like firefox) or EXE name
  //       (like firefox.exe) or full file name (like %windir%\notepad.exe)
  //       or file extension type association (like .html) or file type
  //       identifier (like htmlfile); many specifications may be user in list
  //       of arguments (args); also options supported: (-f,-e,-b,-r,-a,-t,-c).
  // 555 - Get executable file name by document file name.
  // 556 - Run application via WScript.Shell.Run COM object. Windows only.
  // 557 - Execute file via ShellExecute('open',..)
  //
  function GetAppPath(args:LongString; AllowRun:Boolean=false; AllowWait:Boolean=false;
                     pExitCode:PInteger=nil; EnvPath:LongString='';
                     EnvPathExt:LongString=''):LongString;
  function GetExeByFile(const FileName:LongString):LongString;
  function WScriptShellRun(CmdLine:LongString; Show:Integer=SW_SHOWNORMAL; Wait:Boolean=false):Integer;
  function ShellExecuteOpen(FileName:LongString; Cmd:LongString=''):Boolean;
  //
  // 558 - Window Manager control query.
  // wmctrl_query(args) - query wmctrl to get/set desktop window parameters.
  // wmctrl_query DisplayInfo -mode m
  // wmctrl_query XlibInfo -mode m
  // wmctrl_query ListWindows -pid p -class "c" -title "t" -mode m
  // wmctrl_query FindWindow -pid p -class "c" -title "t" -index i
  // wmctrl_query WindowManagerName
  // wmctrl_query DesktopCount
  // wmctrl_query ActiveDesktop
  // wmctrl_query SwitchToDesktop -desktop d
  // wmctrl_query ActiveWindow
  // wmctrl_query ActivateWindow -wnd w -switch-desktop s
  // wmctrl_query WindowPid -wnd w
  // wmctrl_query WindowDesktop -wnd w
  // wmctrl_query WindowHost -wnd w
  // wmctrl_query WindowTitle -wnd w
  // wmctrl_query WindowClass -wnd w
  // wmctrl_query WindowBounds -wnd w
  // wmctrl_query WindowStateList -wnd w
  // wmctrl_query WindowTypeList -wnd w
  // wmctrl_query WindowStateFlags -wnd w
  // wmctrl_query WindowTypeFlags -wnd w
  // wmctrl_query SetWindowStateList -wnd w -wsc c -state-list l
  // wmctrl_query SetWindowStateFlags -wnd w -wsc c -state-flags f
  // wmctrl_query SetWindowDesktop -wnd w -desktop d
  // wmctrl_query SetWindowTitle -wnd w -title "t"
  // wmctrl_query SupportedList
  // wmctrl_query MoveResizeWindow -wnd w -geom g
  // wmctrl_query CloseWindow -wnd w -timeout t
  // wmctrl_query KillWindow -wnd w -sig s -timeout t
  // wmctrl_query IcccmClass
  // wmctrl_query ListDesktopManagers
  // wmctrl_query DesktopManager -index i
  // wmctrl_query ListTerminals
  // wmctrl_query Terminal -index i
  //
  function wmctrl_query(const args:LongString):LongString;
  //
  // 559 - 568 - shared memory API, see DaqPascal help.
  //
  function shm_init(name:LongString; size,mode:Integer):Integer;
  function shm_ref(ref:Integer):Integer;
  function shm_free(ref:Integer):Boolean;
  function shm_delink(name:LongString):Boolean;
  function shm_iop(ref:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
  function shm_rop(ref:Integer; offset:Integer; op:Char; data:Double):Double;
  function shm_fop(ref:Integer; offset:Integer; op:Char; data:Single):Single;
  function shm_sop(ref:Integer; offset:Integer; op:Char; data:LongString):LongString;
  function shm_ctrl(ref:Integer; arg:LongString):LongString;
  function shm_ioresult(code:Integer=0):Integer;
  //
  // 569 - 595 - Finite State Machine (FSM) API, see DaqPascal help.
  //
  function fsm_new:Integer;
  function fsm_free(ref:Integer):Boolean;
  function fsm_ref(ref:Integer):Integer;
  function fsm_root(ref:Integer):Integer;
  function fsm_type(ref:Integer):Integer;
  function fsm_parent(ref:Integer):Integer;
  function fsm_name(ref:Integer):LongString;
  function fsm_path(ref:Integer):LongString;
  function fsm_ctrl(ref:Integer; arg:LongString):LongString;
  function fsm_count(ref,typ:Integer):Integer;
  function fsm_items(ref,typ,i:Integer):Integer;
  function fsm_get_iparam(ref:Integer):Integer;
  function fsm_set_iparam(ref:Integer; data:Integer):Boolean;
  function fsm_get_fparam(ref:Integer):Double;
  function fsm_set_fparam(ref:Integer; data:Double):Boolean;
  function fsm_get_sparam(ref:Integer):LongString;
  function fsm_set_sparam(ref:Integer; data:LongString):Boolean;
  function fsm_add(ref:Integer; typ:integer; key:LongString):Integer;
  function fsm_find(ref:Integer; typ:integer; key:LongString):Integer;
  function fsm_get_state(ref:Integer):Integer;
  function fsm_set_state(ref:Integer; state:Integer):Integer;
  function fsm_put_state(ref:Integer; state:LongString):Integer;
  function fsm_link(ref:Integer; arg:LongString):Integer;
  function fsm_modified(ref:Integer; delta:Integer):Integer;
  function fsm_name_rule(typ:Integer):Integer;
  function fsm_valid_childtype(typ,childtyp:Integer):Boolean;
  //
  // 596 - 597 - Encode/decode data in hexadecimal format.
  //
  function  Hex_Encode(const Str:LongString):LongString;
  function  Hex_Decode(const Str:LongString):LongString;
  //
  // 598 - Convert ColorCode to known color name or $GGBBRR code of RGB.
  // 599 - Convert ColorName (known color name or $GGBBRR code) to color code.
  //
  function ColorToString(ColorCode:TColor; Prefix:LongString=''):LongString;
  function StringToColor(ColorName:LongString; DefaultColor:TColor=clNone):TColor;
  //
  // 600 - Convert string to signal code.
  // 601 - Convert signal code to string.
  //
  function StringToSigCode(str:LongString; def:Integer=0):Integer;
  function SigCodeToString(sig:Integer):LongString;
  //
  // 602 - 612 - text API, see DaqPascal help.
  //
  function text_new:Integer;
  function text_ref(txt:Integer):Integer;
  function text_free(txt:Integer):Boolean;
  function text_getln(txt:Integer; i:Integer):LongString;
  function text_putln(txt:Integer; i:Integer; s:LongString):Boolean;
  function text_insln(txt:Integer; i:Integer; s:LongString):Boolean;
  function text_addln(txt:Integer; s:LongString):Boolean;
  function text_delln(txt:Integer; i:Integer):Boolean;
  function text_numln(txt:Integer):Integer;
  function text_tostring(txt:Integer):LongString;
  function text_fromstring(txt:Integer; s:LongString):Integer;
  //
  // 613 - 655 database API, see DaqPascal help.
  //
  function db_create(arg:LongString):Boolean;
  function db_connection(eid:Integer; arg:LongString):Integer;
  function db_recordset(dbo:Integer; arg:LongString):Integer;
  function db_command(dbo:Integer; arg:LongString):Integer;
  function db_free(dbo:Integer):Boolean;
  function db_ref(dbo:Integer):Integer;
  function db_root(dbo:Integer):Integer;
  function db_type(dbo:Integer):Integer;
  function db_parent(dbo:Integer):Integer;
  function db_state(dbo:Integer):Integer;
  function db_close(dbo:Integer):Boolean;
  function db_open(dbo:Integer; opt:Integer):Boolean;
  function db_ctrl(dbo:Integer; arg:LongString):LongString;
  function db_bugscount(dbo:Integer):Integer;
  function db_bugsclear(dbo:Integer):Integer;
  function db_errors(dbo:Integer):LongString;
  function db_errorscount(dbo:Integer):Integer;
  function db_errorsclear(dbo:Integer):Integer;
  function db_execute(dbo:Integer; arg:LongString; opt:Integer):Integer;
  function db_cancel(dbo:Integer):Boolean;
  function db_update(dbr:Integer):Boolean;
  function db_cancelupdate(dbr:Integer):Boolean;
  function db_begintrans(dbc:Integer):Integer;
  function db_committrans(dbc:Integer):Boolean;
  function db_rollbacktrans(dbc:Integer):Boolean;
  function db_bof(dbr:Integer):Boolean;
  function db_eof(dbr:Integer):Boolean;
  function db_movefirst(dbr:Integer):Boolean;
  function db_movelast(dbr:Integer):Boolean;
  function db_movenext(dbr:Integer):Boolean;
  function db_moveprevious(dbr:Integer):Boolean;
  function db_fieldscount(dbr:Integer):Integer;
  function db_fieldsnames(dbr:Integer; i:Integer):LongString;
  function db_fieldstypes(dbr:Integer; id:LongString):Integer;
  function db_fieldsasint(dbr:Integer; id:LongString; op:Char; arg:Integer):Integer;
  function db_fieldsasfloat(dbr:Integer; id:LongString; op:Char; arg:Double):Double;
  function db_fieldsasstring(dbr:Integer; id:LongString; op:Char; arg:LongString):LongString;
  function db_addnew(dbr:Integer; arg:LongString):Boolean;
  function db_delete(dbr:Integer; aff:Integer):Boolean;
  function db_requery(dbr:Integer; opt:Integer):Boolean;
  function db_resync(dbr:Integer; aff,res:Integer):Boolean;
  function db_supports(dbr:Integer; opt:Integer):Boolean;
  function db_save(dbr:Integer; dst:LongString; fmt:Integer):Boolean;
  //
  // 656 - Check if flag present in mode.
  //
  function HasFlags(Mode,Flags:Int64):Boolean;
  //
  // 657 - 672 - UTF8 support.
  //
  function utf8_valid(const S:LongString):Boolean;
  function utf8_errpos(const S:LongString):SizeInt;
  function utf8_length(const S:LongString):SizeInt;
  function utf8_encode_wide(const S:UnicodeString):LongString;
  function utf8_decode_wide(const S:LongString):UnicodeString;
  function utf8_encode_ansi(const S:LongString):LongString;
  function utf8_decode_ansi(const S:LongString):LongString;
  function utf8_copy(const S:LongString; i,n:SizeInt):LongString;
  function utf8_chrpos(const S:LongString; i:SizeInt):SizeInt;
  function utf8_ord(const S:LongString; i:SizeInt=1):SizeInt;
  function utf8_chr(c:SizeInt):LongString;
  function utf8_uppercase(const S:LongString):LongString;
  function utf8_lowercase(const S:LongString):LongString;
  function utf8_fixbroken(const S:LongString):LongString;
  function utf8_sametext(const S1,S2:LongString):Boolean;
  function utf8_fix_cp(const S:LongString; cp:Word=0):LongString;
  //
  // 673 - Convert Value to string with Base in [2,8,10,16]. Then left pad with 0 to Width.
  // 674 - Convert string S to integer with Base in [2,8,10,16] or return Def value on error.
  // 675 - 678 - sockets: network to native convertion.
  // 679 - format time.
  //
  function  IntToStrBase(Value:LongInt; Base:Integer=10; Width:Integer=0):LongString;
  function  StrToIntBase(S:LongString; Base:Integer=10; Def:Integer=0):LongInt;
  function  htonl(l:Integer):Integer;
  function  ntohl(l:Integer):Integer;
  function  htons(s:Integer):Integer;
  function  ntohs(s:Integer):Integer;
  function  strtimefmt(const fmt:LongString; ms:Double):LongString;
  //
  // 680  -  Return TRUE if S is non-empty string which contains non-space chars.
  //
  function  IsNonEmptyStr(const S:LongString):Boolean;
 end;
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Класс TGuiApi содержит функции для организации графического интерфейса    //
 // пользователя. Он доступен при всех значениях Target. При создании утилит  //
 // обработки данных можно использовать все эти функции без ограничений. При  //
 // создании драйверов CRW-DAQ надо помнить, что драйвер выполняется, как     //
 // правило, в отдельном потоке, поэтому вызов модальных диалогов, например,  //
 // недопустим. Большинство таких функций защищено, в том смысле, что они     //
 // ничего не делают, если вызваны из другого потока, а возвращают mrNone.    //
 ///////////////////////////////////////////////////////////////////////////////
 IGuiApi = interface(IPluginApi) [SIID_IGuiApi]
  //
  // 0    - Вывод текста в консольное окно. Работает всегда, даже если не было
  //        вызова RedirectStdOut(..). Вызов потокобезопасен, не дает задержки,
  //        так как фактически выполняется через FIFO. Поэтому может применяться
  //        в Real Time задачах.
  //
  procedure Echo(const Msg:LongString);
  //
  // 1    - Модальный диалог для предупреждений.
  //
  function  Warning(const Msg:LongString):word;
  //
  // 2    - Модальный диалог для информационных сообщений.
  //
  function  Information(const Msg:LongString):word;
  //
  // 3    - Модальный диалог для запросов Yes/No.
  //
  function  YesNo(const Msg:LongString):word;
  //
  // 4    - Модальный диалог для запросов Yes/No/Cancel.
  //
  function  YesNoCancel(const Msg:LongString):word;
  //
  // 5    - Модальный диалог для сообщений об ошибках.
  //
  function  Error(const Msg:LongString):word;
  //
  // 6    - Если выполнено условие, вызывает модальный диалог для сообщений об ошибках.
  //
  function  Trouble(Check:boolean; const Msg:LongString):Boolean;
  //
  // 7    - Если не выполнено условие, вызывает модальной диалог для сообщений об ошибках.
  //
  function  NoProblem(Check:boolean; const Msg:LongString):Boolean;
  //
  // 8    - Модальный диалог для выбора элемента списка, заданного текстом ItemList.
  //        Возвращает индекс выбранного пункта (начиная с нуля).
  //        Возвращает -1, если выбор был отменен.
  //
  function  ListBoxMenu(const Caption,Title:LongString;
                        const ItemList:LongString):Integer;
  //
  // 9    - Модальный диалог для выбора группы элементов списка, заданного текстом ItemList.
  //        Возвращает список выбранных пунктов или список их индексов,
  //        в зависимости от значения GetIndex. Возвращает пустую строку при
  //        отмене выбора.
  //
  function  ListBoxMultiSelection(const Caption,Title:LongString;
                                  const ItemList : LongString;
                                        Extended : Boolean = false;
                                        GetIndex : Boolean = false
                                               ) : LongString;
  //
  // 10   - Модальный диалог для редактирования текста.
  //
  function  TextEditDialog(const Caption,Title:LongString;
                           const TextToEdit:LongString):LongString;
  //
  // 11  - Проверяет, доступна ли звуковая подсистема программы.
  //
  function UsesBlaster:Boolean;
  //
  // 12   - Инициирует проигрывание списка звуков, заданных списком PlayList.
  //        На самом деле процедура просто помещает список звуков в FIFO и
  //        сразу возвращает управление. Проигрывание звуков выполняется
  //        в отдельном потоке. Список PlayList состоит из имен wav-файлов,
  //        расположенных в каталогах, заданных AddSoundLibrary. Целые числа
  //        автоматически преобразуются в произносимую последовательность
  //        звуков.
  //
  procedure Voice(const PlayList:LongString);
  //
  // 13   - Очищает FIFO звуков, отменяя все задания, инициированные командой Voice.
  //
  procedure StopBlaster;
  //
  // 14   - Добавляет в список доступных звуков каталог и его подкаталоги.
  //        Все *.WAV файлы из этих каталогов можно будет произносить.
  //
  procedure AddSoundLibrary(const Path:LongString;
                                  MaxSubDirs:LongInt=0);
  //
  // 15   - Вызывает Application.ProcessMessages, для длительных вычислений.
  //
  procedure ApplicationProcessMessages;
  //
  // 16   - Вызывает обновление консоли, для длительных вычислений.
  //
  procedure UpdateSystemConsole;
  //
  // 17   - Составляет цвет из отдельных компонент R,G,B, лежащих в диапазоне 0..255.
  //
  function  RGB(r,g,b:Integer):TColor;
  //
  // 18   - Возвращает английское имя цвета старого стиля CGA.
  //
  function  CgaColorNameEng(CgaColor:Integer):LongString;
  //
  // 19   - Возвращает русское имя цвета старого стиля CGA.
  //
  function  CgaColorNameRus(CgaColor:Integer):LongString;
  //
  // 20   - Возвращает цвет старого стиля CGA по имени.
  //
  function  CgaColorByName(const S:LongString):Integer;
  //
  // 21   - Преобразует цветовую кодировку старого стиля CGA, принятую в DOS,
  //        в цветовую кодировку RGB, принятую в Windows. CgaColor меняется
  //        в пределах 0=Black...15=White.
  //
  function  CgaToRGBColor(CgaColor:Integer):TColor;
  //
  // 22   - Вызывает Application.Handle.
  //
  function  ApplicationHandle:THandle;
  //
  // 23  - Посылает сообщение в Главную Консоль.
  //
  function  SendToMainConsole(const Msg:LongString):Integer;
 end;
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Класс TDaqApi содержит функции для создания драйверов для подсистемы      //
 // сбора данных CRW-DAQ. При этом Target должен быть ForDataAcquisition.     //
 // Общие замечания.                                                          //
 // 1) Вызов DLL. В системе CRW-DAQ с устройством Program (на Daq Pascal)     //
 //    можно связать несколько DLL, вызываемых через функции Daq Pascal:      //
 //     DaqDllInit(DllPath)      - загружает DLL с заданным именем файла.     //
 //     DaqDllFree(hDll)         - удаляет загруженную ранее DLL.             //
 //     DaqDllCall(hDll,Command) - вызывает DLL для выполнения данной команды.//
 //    Через Command передается команда, общее указание, что надо делать.     //
 //    Все остальные параметры передаются/принимаются через теги/кривые.      //
 //    Команды, за исключением нескольких, не заданы жестко, они являются     //
 //    предметом договоренности вызывающей программы на Daq Pascal и DLL.     //
 //    Однако несколько команд зарезервированы:                               //
 //     a)При старте CRW-DAQ надо вызывать DLL с командой Daq_Cmd_Init        //
 //       для выполнения инициализирующих действий при старте устройства.     //
 //       Например, инициализация ссылок на кривые и теги, выделение памяти.  //
 //       Если при старте возник код ошибки, работа с DLL должна быть         //
 //       завершена, а опрос должен быть отменен.                             //
 //     b)В процессе опроса DAQ надо вызывать DLL с командой Daq_Cmd_Poll     //
 //       для выполнения полезной работы. Команда опроса вызывается           //
 //       многократно, в цикле опроса устройства. Устройство должно быстро    //
 //       выполнять свои функции, по возможности быстро возвращая управление. //
 //       Если нужны длительные операции, лучше создать для них свой поток,   //
 //       чем задерживать цикл опроса.                                        //
 //     c)При остановке CRW DAQ, надо вызывать DLL с командой Daq_Cmd_Free    //
 //       для выполнения завершающих операций,таких как освобождение памяти   //
 //       и т.д.                                                              //
 //  2) Хранение временных данных. При вызове DaqDllInit создается новый      //
 //     экземпляр API, с которым связан свой собственный сегмент данных.      //
 //     Это надо потому, что одна и та же DLL может обслуживать несколько     //
 //     устройств. Чтобы не путать одноименные переменные, которые относятся  //
 //     к разным устройствам, каждое устройство имеет свой экземпляр API со   //
 //     своим сегментом данных. Сегмент данных сохраняется от вызова к вызову,//
 //     что позволяет хранить в нем долговременные данные.                    //
 //     При вызове DaqDllFree сегмент данных автоматически уничтожается.      //
 //     Но все же лучше уничтожать его явно по команде Daq_Cmd_Free вызовом   //
 //     DaqDataSheet(0), так как сегмент данных может содержать указатели на  //
 //     другие данные,которые надо освободить до уничтожения сегмента данных. //
 //     Выделение, доступ и удаление сегмента данных осуществляется через     //
 //     одну функцию DaqDataSheet(Size). Сегмент данных обычно связан с       //
 //     записью некоторого типа, Size получается через sizeof(..)             //
 //     Пример:                                                               //
 //          type TDataRec:record ... end;                                    //
 //          begin                                                            //
 //           with Daq, TDataRec(DaqDataSheet(sizeof(TDataRec))^) do ...      //
 //          end;                                                             //
 //  3) Теги. Система CRW DAQ содержит базу данных тегов, к которой DLL имеет //
 //     через функции FindTag...sSetTag. Теги являются поименованными         //
 //     переменными системы CRW-DAQ общего пользования с потокобезопасным     //
 //     доступом. Каждый тег имеет имя, ссылку, тип, и значение. Имя служит   //
 //     для идентификаци тега с точки зрения программиста и конфигурационного //
 //     файла (запоминать легче). Однако из соображений эффективности доступ  //
 //     к тегу возможен только по ссылке. Обычно при старте программа DAQ     //
 //     определяет ссылки тегов по имени, а затем использует ссылки для       //
 //     доступа к данным. Тип тега определяет содержимое тега и может быть    //
 //     равен 1(Integer), 2(Double), 3(LongString). Нулевой тип значит, что   //
 //     тег с данным индексом не существует. То же самое означает нулевая     //
 //     ссылка на тег.                                                        //
 //  4) Термопарные таблицы. Система Crw Daq содержит базу данных стандартных //
 //     калибровок - термопар, резистивных датчиков, термодиодов и т.д.       //
 //     Для доступа к базе данных служат функции Thermocouple,Relationsheep.. //
 //  5) Кривые. Система CRW DAQ содержит базу данных кривых, то есть          //
 //     поименованных динамических массивов. Для работы с кривыми есть очень  //
 //     много функций. Часть из них использует для доступа к кривой номер     //
 //     аналогового/цифрового входа/выхода. Эти функции по большей части      //
 //     аналогичны функциям Daq Pascal. Например, основной способ записи      //
 //     в кривую - это помещение в очередь события вызовом putev,putao,putdo. //
 //     При этом автоматически будут обновляться данные в окне.               //
 //     Другая группа функций использует для доступа к кривой ссылку, которую //
 //     можно получить через refAI..refDO, FindObjectRef('Curve',..).         //
 //     При прямой модификации кривой окна надо обновлять вручную, вызовом    //
 //     WinDraw. При использовании функций этой группы надо быть внимательным,//
 //     так как контроль сведен к минимуму (например, можно модифицировать    //
 //     кривую, подключенную к входу).                                        //
 //  6) Устройства. В отличие от DaqPascal, для работы с устройствами CRW DAQ //
 //     в основном используются ссылки, получаемые через вызов функции        //
 //     FindObjectRef('Device',..).                                           //
 ///////////////////////////////////////////////////////////////////////////////
 IDaqApi = interface(IPluginApi) [SIID_IDaqApi]
  //
  // 0    - Возвращает команду, переданную в DLL из программы Daq Pascal
  //        при вызове функции DaqDllCall(..,Command).
  //
  function  DaqCommand:Integer;
  //
  // 1    - Возвращает указатель области данных драйвера DAQ - системы.
  //        Параметр aSize задает размер области данных. Если он равен нулю,
  //        область данных уничтожается. Если размер области данных изменился,
  //        память перераспределяется. Область данных используется для хранения
  //        данных между вызовами DLL, обычно она связана с записью какого-то
  //        фиксированного типа и используется в таком стиле:
  //          type TDataRec:record ... end;
  //          begin
  //           with Daq, TDataRec(DaqDataSheet(sizeof(TDataRec))^) do ...
  //          end;
  //
  function  DaqDataSheet(aSize:Integer):Pointer;
  //
  // 2    - Проверяет, загружена ли DAQ - система.
  //
  function  DaqIsLoaded:Boolean;
  //
  // 3    - Проверяет, запущена ли на выполнение DAQ - система.
  //
  function  DaqIsStarted:Boolean;
  //
  // 4    - Возвращает имя основного конфигурационного файла DAQ.
  //
  function  DaqConfigFile:LongString;
  //
  // 5    - Возвращает каталог для файлов данных загруженной системы DAQ.
  //
  function  DaqDataPath:LongString;
  //
  // 6    - Возвращает каталог для временных файлов загруженной системы DAQ.
  //
  function  DaqTempPath:LongString;
  //
  // 7    - Возвращает ссылку тега по его имени. Сссылка должна быть ненулевой.
  //        Если возвращается ноль, значит тега с таким именем не существует.
  //
  function  FindTag(const Name:LongString):Integer;
  //
  // 8    - Возвращает ссылку тега с данным именем, если он существует.
  //        Создает новый тег, если его еще не существует.
  //        Возвращает ноль, если таблица тегов исчерпана или если существует
  //        тег с заданным именем, но другого типа.
  //
  function  CreateTag(const Name:LongString; Typ:Integer):Integer;
  //
  // 9    - Уничтожает тег, заданный ссылкой tag.
  //
  function  FreeTag(tag:Integer):Boolean;
  //
  // 10    - Возвращает тип тега, заданного ссылкой tag. Может возвратить:
  //        0 - Тег не существует.
  //        1 - Тег имеет тип Integer    (Integer в Daq Pascal).
  //        2 - Тег имеет тип Double     (Real    в Daq Pascal).
  //        3 - Тег имеет тип LongString (String  в Daq Pascal).
  //
  function  TypeTag(tag:Integer):Integer;
  //
  // 11   - Возвращает имя тега с заданной ссылкой tag.
  //        Возвращает пустую строку, если тег не существует.
  //
  function  NameTag(tag:Integer):LongString;
  //
  // 12   - Возвращает значение тега типа Integer.
  //        Возвращает 0, если тег не существует или тип тега неверен.
  //
  function  iGetTag(tag:Integer):Integer;
  //
  // 13   - Возвращает значение тега типа Real.
  //        Возвращает 0, если тег не существует или тип тега неверен.
  //
  function  rGetTag(tag:Integer):Double;
  //
  // 14   - Возвращает значение тега типа String.
  //        Возвращает пустую строку, если тег не существует или тип тега неверен.
  //
  function  sGetTag(tag:Integer):LongString;
  //
  // 15   - Задает новое значение тега типа Integer.
  //        Возвращает false, если тег не существует или тип тега неверен.
  //
  function  iSetTag(tag:Integer; i:Integer):Boolean;
  //
  // 16   - Задает новое значение тега типа Real.
  //        Возвращает false, если тег не существует или тип тега неверен.
  //
  function  rSetTag(tag:Integer; r:Double):Boolean;
  //
  // 17   - Задает новое значение тега типа String.
  //        Возвращает false, если тег не существует или тип тега неверен.
  //
  function  sSetTag(tag:Integer; const s:LongString):Boolean;
  //
  // 18   - Число загруженных стандартных термопарных таблиц.
  //
  function  ThermoCoupleCount:Integer;
  //
  // 19   - Находит индекс от 0 до ThermoCoupleCount-1 стандартной термопарной
  //        таблицы по имени. Возвращает -1, если термопары с таким именем нет.
  //
  function  FindThermoCouple(const CoupleName:LongString):Integer;
  //
  // 20   - Возвращает имя стандартной термопарной таблицы по индексу.
  //
  function  ThermoCoupleName(CoupleID:Integer):LongString;
  //
  // 21   - Возвращает диапазон стандартной термопарной таблицы.
  //
  function  ThermoCoupleRange(CoupleID:Integer):TRect2D;
  //
  // 22   - Возвращает значение прямой или обратной стандартной термопарной
  //        таблицы, в зависимости от Inverted. Для интерполяции используется
  //        сглаживающий сплайн.
  //
  function  ThermoCoupleValue(CoupleID:Integer; Argument:Double;
                              Inverted:Boolean):Double;
  //
  // 23   - Число загруженных таблиц стандартных калибровок.
  //
  function  RelationshipCount:Integer;
  //
  // 24   - Возвращает индекс от 0 до RelationshipCount-1 таблицы стандартных
  //        калибровок по имени. Возвращает -1, если нет такой таблицы.
  //
  function  FindRelationship(const RelationName:LongString):Integer;
  //
  // 25   - Возвращает имя таблицы стандартных калибровок по индексу.
  //
  function  RelationshipName(RelationID:Integer):LongString;
  //
  // 26   - Возвращает диапазон стандартной таблицы калибровок.
  //
  function  RelationshipRange(RelationID:Integer):TRect2D;
  //
  // 27   - Возвращает значение прямой или обратной стандартной таблицы
  //        калибровок, в зависимости от Inverted. Для интерполяции используется
  //        сглаживающий сплайн.
  //
  function  RelationshipValue(RelationID:Integer; Argument:Double;
                              Inverted:Boolean):Double;
  //
  // 28   - Возвращает текущее время по часам DAQ, выраженное в единицах TimeUnits.
  //
  function  Time:Double;
  //
  // 29   - Возвращает единицы измерения времени DAQ в миллисекундах.
  //
  function  TimeUnits:Double;
  //
  // 30   - Возвращает число аналоговых входов вызвавшего устройства DAQ.
  //
  function  NumAIs:Integer;
  //
  // 31   - Возвращает число цифровых входов вызвавшего устройства DAQ.
  //
  function  NumDIs:Integer;
  //
  // 32   - Возвращает число аналоговых выходов вызвавшего устройства DAQ.
  //
  function  NumAOs:Integer;
  //
  // 33   - Возвращает число цифровых выходов вызвавшего устройства DAQ.
  //
  function  NumDOs:Integer;
  //
  // 34   - Возвращает число калибровок вызвавшего устройства DAQ.
  //
  function  NumCals:Integer;
  //
  // 35   - Возвращает ссылку кривой, подключенной к аналоговому входу n.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  RefAI(n:Integer):Integer;
  //
  // 36   - Возвращает ссылку кривой, подключенной к цифровому входу n.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  RefDI(n:Integer):Integer;
  //
  // 37   - Возвращает ссылку кривой, подключенной к аналоговому выходу n.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  RefAO(n:Integer):Integer;
  //
  // 38   - Возвращает ссылку кривой, подключенной к цифровому выходу n.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  RefDO(n:Integer):Integer;
  //
  // 39   - Возвращает ссылку калибровки n.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  RefCalibr(n:Integer):Integer;
  //
  // 40   - Возвращает длину кривой, заданной ссылкой с.
  //
  function  GetCurveLength(c:Integer):Integer;
  //
  // 41   - Задает новую длину кривой c.
  //
  procedure SetCurveLength(c,aLength:Integer);
  //
  // 42   - Возвращает точку номер i из кривой, заданной ссылкой c.
  //        Напоминаем, что нумерация точек идет с 0.
  //
  function  GetCurvePoint(c,i:Integer):TPoint2D;
  //
  // 43   - Задает значение точки номер i на кривой c.
  //
  procedure SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
  //
  // 44   - Добавляет точку в конец кривой c.
  //
  procedure CurveAddPoint(c:Integer; const aPoint:TPoint2D);
  //
  // 45   - Добавляет aCount точек в конец кривой c.
  //
  procedure CurveAddPoints(c:Integer; const aX,aY:array of Double;
                                            aCount:Integer);
  //
  // 46   - Удаляет точку i из кривой c.
  //
  procedure CurveDeletePoint(c,i:Integer);
  //
  // 47   - Удаляет aCount точек начиная с i из кривой c.
  //
  procedure CurveDeletePoints(c,i,aCount:Integer);
  //
  // 48   - Вставляет точку в позицию i кривой c.
  //
  procedure CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
  //
  // 49   - Вставляет aCount точек в позицию i кривой c.
  //
  procedure CurveInsertPoints(c,i:Integer; const aX,aY:array of Double;
                              aCount:LongInt);
  //
  // 50   - Возвращает имя кривой, заданной ссылкой c.
  //
  function  GetCurveName(c:Integer):LongString;
  //
  // 51   - Возвращает цвет кривой, заданной ссылкой c.
  //
  function  GetCurveColor(c:Integer):Integer;
  //
  // 52   - Возвращает стиль кривой, заданной ссылкой c.
  //
  function  GetCurveStyle(c:Integer):Integer;
  //
  // 53   - Возвращает комментарий кривой, заданной ссылкой c.
  //
  function  GetCurveComment(c:Integer):LongString;
  //
  // 54   - Задает комментарий кривой, заданной ссылкой c.
  //
  procedure SetCurveComment(c:Integer; const aComment:LongString);
  //
  // 55   - Быстрый поиск индекса ближайшей к t точки кривой c.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveGetIndexAt(c:Integer; t:Double):Integer;
  //
  // 56   - Линейная интерполяция кривой c в точке t.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveInterpolateAt(c:Integer; t:Double):Double;
  //
  // 57   - Сглаживание кривой c в точке t окном шириной Window, а также ядром
  //        типа K(x)=(1-x^k1)^k2. Power - степень полинома или интеграл при Power=-1.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveSmoothAt(c:Integer; t:Double;
                        Window:Double; Power,K1,K2:Integer):Double;
  //
  // 58   - Вычисляет медиану кривой c в интервале точек [i..j].
  //        Возвращает индекс aIndex и значение aValue медианы.
  //        Можно выбирать метод сортировки 0=Shell,1=Quick.
  //
  function  CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double;
                         SortMethod:Integer=1):Boolean;
  //
  // 59   - Вычисляет массив флагов типа "кривая отсортирована" и т.д.
  //        Смотрите описание констант cfXXXX.
  //
  function  CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10;
                       Small:Integer=2):LongInt;
  //
  // 60   - Возвращает минимальный прямоугольник, содержащий все точки кривой c.
  //
  function  GetCurveLimits(c:Integer):TRect2D;
  //
  // 61   - Возвращает интеграл участка (a,b) кривой c.
  //
  function  GetCurveIntegral(c:Integer; a,b:Double):Double;
  //
  // 62   - Возвращает последнюю точку кривой c.
  //
  function  GetCurveLastPoint(c:Integer):TPoint2D;
  //
  // 63   - Помещает событие в очередь событий.
  //        w  - тип события, например, evAnalog+evCompress или evDigital.
  //        n  - номер канала, то есть аналогового или цифрового выхода.
  //        t  - время по часам DAQ, см. функцию Time.
  //        d0 - первый элемент данных
  //        d1 - второй элемент данных
  //        Запись данных в кривые через очередь событий является наиболее
  //        предпочтительным способом обновления кривых, так как при этом выполнется
  //        автоматическое обновление графиков, сжатие данных и другие
  //        предусмотренные диспетчером событий действия.
  //
  function  PutEv(w,n:Integer; t,d0:Double; d1:Double=0):Boolean;
  //
  // 64   - Помещает аналоговое событие в очередь событий.
  //        Эквивалент PutEv(evAnalog+evCompress,n,t,d,0).
  //        Запись данных в кривые через очередь событий является наиболее
  //        предпочтительным способом обновления кривых, так как при этом выполнется
  //        автоматическое обновление графиков, сжатие данных и другие
  //        предусмотренные диспетчером событий действия.
  //
  function  PutAO(n:Integer; t,d:Double):Boolean;
  //
  // 65   - Помещает цифровое событие в очередь событий.
  //        Эквивалент PutEv(evDigital,n,t,d,0).
  //        Запись данных в кривые через очередь событий является наиболее
  //        предпочтительным способом обновления кривых, так как при этом выполнется
  //        автоматическое обновление графиков, сжатие данных и другие
  //        предусмотренные диспетчером событий действия.
  //
  function  PutDO(n:Integer; t,d:Double):Boolean;
  //
  // 66   - Возвращает данные аналогового входа n в момент времени t с применением сглаживания.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai(n:Integer;t:Double):Double;
  //
  // 67   - Возвращает число точек кривой с аналогового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai_n(n:Integer):Integer;
  //
  // 68   - Возвращает координату X последней точки кривой с аналогового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai_xn(n:Integer):Double;
  //
  // 69   - Возвращает координату Y последней точки кривой с аналогового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai_yn(n:Integer):Double;
  //
  // 70   - Возвращает координату X[i] кривой с аналогового входа n.
  //        Нумерация точек начинается с 1, как в Daq Pascal.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai_xi(n,i:Integer):Double;
  //
  // 71   - Возвращает координату Y[i] кривой с аналогового входа n.
  //        Нумерация точек начинается с 1, как в Daq Pascal.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getai_yi(n,i:Integer):Double;
  //
  // 72   - Возвращает данные цифрового входа n в момент времени t.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi(n:Integer;t:Double):Double;
  //
  // 73   - Возвращает число точек кривой с цифрового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi_n(n:Integer):Integer;
  //
  // 74   - Возвращает координату X последней точки кривой с цифрового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi_xn(n:Integer):Double;
  //
  // 75   - Возвращает координату Y последней точки кривой с цифрового входа n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi_yn(n:Integer):Double;
  //
  // 76   - Возвращает координату X[i] кривой с цифрового входа n.
  //        Нумерация точек начинается с 1, как в Daq Pascal.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi_xi(n,i:Integer):Double;
  //
  // 77   - Возвращает координату Y[i] кривой с цифрового входа n.
  //        Нумерация точек начинается с 1, как в Daq Pascal.
  //        Аналог одноименной функции DaqPascal.
  //
  function  getdi_yi(n,i:Integer):Double;
  //
  // 78   - Преобразует величину t по калибровке номер n, используя параметр d.
  //        Аналог одноименной функции DaqPascal.
  //
  function  calibr(n:Integer; t,d:Double):Double;
  //
  // 79   - Фиксирует ошибку с кодом n.
  //        Аналог одноименной функции DaqPascal.
  //
  function  fixerror(n:Integer):Boolean;
  //
  // 80   - Возвращает число, в битах которого закодирована карта подключения
  //        аналоговых входов i..i+n-1. Наличие бита в числе означает, что
  //        на соответствующем входе подключена кривая.
  //        Аналог одноименной функции DaqPascal.
  //
  function  aimap(i,n:Integer):Integer;
  //
  // 81   - Возвращает число, в битах которого закодирована карта подключения
  //        цифровых входов i..i+n-1. Наличие бита в числе означает, что
  //        на соответствующем входе подключена кривая.
  //        Аналог одноименной функции DaqPascal.
  //
  function  dimap(i,n:Integer):Integer;
  //
  // 82   - Возвращает число, в битах которого закодирована карта подключения
  //        аналоговых выходов i..i+n-1. Наличие бита в числе означает, что
  //        на соответствующем выходе подключена кривая.
  //        Аналог одноименной функции DaqPascal.
  //
  function  aomap(i,n:Integer):Integer;
  //
  // 83   - Возвращает число, в битах которого закодирована карта подключения
  //        цифровых выходов i..i+n-1. Наличие бита в числе означает, что
  //        на соответствующем выходе подключена кривая.
  //        Аналог одноименной функции DaqPascal.
  //
  function  domap(i,n:Integer):Integer;
  //
  // 84   - Конструирует число, в битах которого закодирован входной цифровой
  //        сигнал на входах i..i+n-1. Наличие бита в числе означает, что
  //        на соответствующем входе соответствующий бит числа установлен.
  //        Аналог одноименной функции DaqPascal.
  //
  function  diword(i,n:Integer):Integer;
  //
  // 85   - Возвращает код при нажатии кнопки мыши на сенсоре.
  //        0 - ничего не нажато.
  //        1 - нажата левая кнопка.
  //        2 - нажата правая кнопка.
  //        3 - нажата средняя кнопка.
  //        Аналог одноименной функции DaqPascal.
  //
  function  ClickButton:Integer;
  //
  // 86   - Возвращает имя нажатого сенсора.
  //        Аналог одноименной функции DaqPascal.
  //
  function  ClickSensor:LongString;
  //
  // 87   - Возвращает ссылку нажатого тега.
  //        Аналог одноименной функции DaqPascal.
  //
  function  ClickTag:Integer;
  //
  // 88   - Посылает звуковое сообщение. Отличается рядом деталей от GuiApi.Voice.
  //        Например, звук DaqApi.Voice можно запретить кнопкой на панели DAQ.
  //        Аналог одноименной функции DaqPascal.
  //
  function  Voice(const msg:LongString):Boolean;
  //
  // 89   - Находит и перерисовывает окно, имеющее заголовок Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  WinDraw(const Caption:LongString):Boolean;
  //
  // 90   - Находит и разворачивает окно, имеющее заголовок Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  WinShow(const Caption:LongString):Boolean;
  //
  // 91   - Находит и сворачивает окно, имеющее заголовок Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  WinHide(const Caption:LongString):Boolean;
  //
  // 92   - Находит и активизирует окно, имеющее заголовок Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  WinSelect(const Caption:LongString):Boolean;
  //
  // 93   - Читает из конфигурации переменную. Возможны варианты строки s:
  //         s = Var         - чтение переменной Var из секции устройства.
  //         s = Sec Var     - чтение переменной Var из секции Sec конфигурации DAQ.
  //         s = Cfg Sec Var - чтение переменной Var из секции Sec файла Cfg.
  //        Аналог одноименной функции DaqPascal.
  //
  function  ReadIni(const s:LongString):LongString;
  //
  // 94   - Чтение байта из порта ввода-вывода процессора. Будьте осторожны!
  //
  function  GetPortB(Address:Word):Byte;
  //
  // 95   - Запись байта в порт ввода-вывода процессора. Будьте осторожны!
  //
  procedure SetPortB(Address:Word; Value:Byte);
  //
  // 96   - Чтение слова из порта ввода-вывода процессора. Будьте осторожны!
  //
  function  GetPortW(Address:Word):Word;
  //
  // 97   - Запись слова в порт ввода-вывода процессора. Будьте осторожны!
  //
  procedure SetPortW(Address:Word; Value:Word);
  //
  // 98   - Чтение двойного слова из порта ввода-вывода процессора. Будьте осторожны!
  //
  function  GetPortL(Address:Word):LongWord;
  //
  // 99   - Запись двойного слова в порт ввода-вывода процессора. Будьте осторожны!
  //
  procedure SetPortL(Address:Word; Value:LongWord);
  //
  // 100  -  Регистрирует в DAQ системе новый код ошибки с текстом сообщения s.
  //        Аналог одноименной функции DaqPascal.
  //
  function  RegisterErr(const s:LongString):Integer;
  //
  // 101  - Счетчик ошибок с кодом n.
  //         n = -2       : все ошибки всех устройств
  //         n = -1       : все ошибки данного устройства
  //         n = 0..255   : ошибки устройства с кодом n
  //         n = 256..511 : ошибки всех устройств с кодом n-256
  //        Аналог одноименной функции DaqPascal.
  //
  function  GetErrCount(n:Integer):Double;
  //
  // 102  - Возвращает счетчик вызовов программы DAQ PASCAL, с которой связана DLL.
  //        Аналог одноименной функции DaqPascal.
  //
  function  RunCount:Double;
  //
  // 103  - Поиск объекта типа What с именем Name в базе данных CRW DAQ.
  //        Тип What может иметь значение:
  //        TAG    - возвращает ссылку на тег, аналогично вызову FindTag.
  //        CURVE  - возвращает ссылку на кривую с данным именем.
  //        DEVICE - возвращает ссылку на устройство DAQ с данным именем.
  //                 Если задано пустое имя, возвращает текущее устройство.
  //        Если возвращается ноль, объект не найден.
  //        Полученную ссылку надо использовать по назначению, например,
  //        ссылку на кривую - только в функциях работы с кривыми и т.д.
  //        Так как функция связана с поиском в базе данных, рекомендуется
  //        вызывать ее один раз в начале программы, а полученную ссылку
  //        запоминать для дальнейшего использования.
  //
  function  FindObjectRef(const What,Name:LongString):Integer;
  //
  // 104  - Возвращает имя устройства по его ссылке.
  //
  function  DeviceName(d:Integer):LongString;
  //
  // 105  - Возвращает название модели устройства по его ссылке.
  //
  function  DeviceModel(d:Integer):LongString;
  //
  // 106  - Возвращает название семейства устройства по его ссылке.
  //
  function  DeviceFamily(d:Integer):LongString;
  //
  // 107  - Вызов метода Action устройства, заданного ссылкой d, см.FindObjectRef.
  //        Вызов отсрочен, всвязи с тем, что выполняется в основном потоке VCL.
  //
  function  DeviceAction(d:Integer):Boolean;
  //
  // 108  - Вызов метода Clear устройства, заданного ссылкой d, см.FindObjectRef.
  //        Вызов отсрочен, всвязи с тем, что выполняется в основном потоке VCL.
  //
  function  DeviceClear(d:Integer):Boolean;
  //
  // 109  - Вызов метода ClearDevice устройства, заданного ссылкой d, см.FindObjectRef.
  //        Вызов отсрочен, всвязи с тем, что выполняется в основном потоке VCL.
  //
  function  DeviceClearDevice(d:Integer):Boolean;
  //
  // 110  - Вызов метода Start устройства, заданного ссылкой d, см.FindObjectRef.
  //        Вызов отсрочен, всвязи с тем, что выполняется в основном потоке VCL.
  //
  function  DeviceStart(d:Integer):Boolean;
  //
  // 111  - Вызов метода Stop устройства, заданного ссылкой d, см.FindObjectRef.
  //        Вызов отсрочен, всвязи с тем, что выполняется в основном потоке VCL.
  //
  function  DeviceStop(d:Integer):Boolean;
  //
  // 112  - Вызов метода HandleMessage устройства, заданного ссылкой d, см.FindObjectRef.
  //
  function  DeviceMessage(d:Integer; const msg:LongString):Double;
  //
  // 113  - Сохранение списка кривых в файл *.CRW.
  //        arglist =  fname winname wintitle winlable curvelist
  //         fname     - имя файла, можно относительно основного конфигфайла и без расширения
  //         winname   - имя окна под которым оно попадет в crw-архив
  //         wintitle  - заголовок в верхней части окна
  //         winlable  - метка в нижней части окна
  //         curvelist - список сохраняемых кривых
  //        Например:
  //        SaveCrw('..\data\pressure.crw, Pressures, ^CP(t)^N^L{P}, ^R{t}^N^Ct, p1,p2,p3,p4,p5');
  //        Аналог одноименной функции DaqPascal.
  //
  function  SaveCrw(const ArgList:LongString):Boolean;
  //
  // 114  - Возвращает текущий маркер спектрометрического окна с заголовком Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  SpecMarker(const Caption:LongString):Double;
  //
  // 115  - Возвращает левый маркер спектрометрического окна с заголовком Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  SpecMarkerL(const Caption:LongString):Double;
  //
  // 116  - Возвращает правый маркер спектрометрического окна с заголовком Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  SpecMarkerR(const Caption:LongString):Double;
  //
  // 117  - Возвращает левую границу ROI спектрометрического окна с заголовком Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  SpecRoiL(const Caption:LongString):Double;
  //
  // 118  - Возвращает правую границу ROI спектрометрического окна с заголовком Caption.
  //        Аналог одноименной функции DaqPascal.
  //
  function  SpecRoiR(const Caption:LongString):Double;
  //
  // 119  - Вычисляет выражение Expression, используя общий калькулятор.
  //        Аналог функции global() DaqPascal.
  //
  function  GlobalCalc(const Expression:LongString):Double;
  //
  // 120    - Открывает порт aPortNum последовательной связи COM.
  //          aParity   = (NOPARITY,ODDPARITY,EVENPARITY,MARKPARITY,SPACEPARITY).
  //          aDataBits = 5..8.
  //          aStopBits = (ONESTOPBIT,ONE5STOPBITS,TWOSTOPBITS).
  //          Будьте внимательны. Функции Uartxxx работают с портом напрямую.
  //          Это полезно при написании драйверов RS-232, однако недопустимо
  //          работать с портом напрямую, если на порт подключены устройства
  //          семейства ADAM, работающие по логике RS-485, типа серии IC-7000.
  //          Прямая работа с портом в этом случае нарушает логику опроса.
  //
  function  UartOpen(aPortNum:Integer; aBaudrate:Integer=9600; aParity:Integer=NOPARITY;
                     aDataBits:Integer=8; aStopBits:Integer=ONESTOPBIT; aXonXoff:Boolean=false;
                     aFifoSize:Integer=4096;aDcbFlags:LongInt=0):Boolean;
  //
  // 121  - Открывает порт, используя конфигурационный файл CRW-DAQ.
  //
  function  UartOpenConfig(const Section:LongString='[SerialPort-COM1]'
                           ):Integer;
  //
  // 122  - Закрывает порт с данным номером.
  //
  procedure UartClose(aPortNum:Integer);
  //
  // 123  - Проверяет, не был ли уже открыт порт с данным номером.
  //
  function  UartActive(aPortNum:Integer):Boolean;
  //
  // 124  - Записывает данные в буфер передатчика для последующей передачи.
  //        Возвращает число реально записанных в передачик символов.
  //
  function  UartWriteBuf(aPortNum:Integer; const Data:PChar;
                         Count:Cardinal):Cardinal;
  //
  // 125  - Читает принятые данные из буфера приемника. Возвращает число прочитанных символов.
  //
  function  UartReadBuf(aPortNum:Integer; var Data:PChar;
                        MaxCount:Cardinal):Cardinal;
  //
  // 126  - Записывает строку данных в буфер передатчика для последующей передачи.
  //
  function  UartWriteStr(aPortNum:Integer;
                         const Data:LongString):Boolean;
  //
  // 127  - Читает строку данных из буфера приемника.
  //
  function  UartReadStr(aPortNum:Integer;
                        MaxCount:Byte=255):LongString;
  //
  // 128  - Очищает приемник и его буфер.
  //
  procedure UartRxClear(aPortNum:Integer);
  //
  // 129  - Очищает передатчик и его буфер.
  //
  procedure UartTxClear(aPortNum:Integer);
  //
  // 130  - Возвращает число символов в буфере приемника.
  //
  function  UartRxCount(aPortNum:Integer):Integer;
  //
  // 131  - Возвращает число символов в буфере передатчика.
  //
  function  UartTxCount(aPortNum:Integer):Integer;
  //
  // 132  - Возвращает свободное пространство в буфере приемника.
  //
  function  UartRxSpace(aPortNum:Integer):Integer;
  //
  // 133  - Возвращает свободное пространство в буфере передатчика.
  //
  function  UartTxSpace(aPortNum:Integer):Integer;
  //
  // 133 - Находит контрольную сумму строки, то есть сумму всех байтов без учета переполнения.
  //       Эта и 4 следующие функции ориентированы на работу с модулями серии ADAM,IC-700 и т.д.
  //
  function  UartFindCheckSum(Data:PChar; Length:Integer):Byte;
  //
  // 134  - Добавляет в конец строки контрольную сумму в виде HEX - строки.
  //
  function  UartCatCheckSum(const DataStr:LongString):LongString;
  //
  // 135  - Добавляет в конец строки перед символом CR контрольную сумму в виде HEX - строки.
  //
  function  UartCatCheckSumCR(const DataStr:LongString;
                              CR:Char=ASCII_CR):LongString;
  //
  // 136  - Делает проверку контрольной суммы и выделение исходной строки без контрольной суммы.
  //        Если сумма не совпала, возвращается пустая строка.
  //
  function  UartTruncCheckSum(const DataStr:LongString):LongString;
  //
  // 137  - Делает проверку контрольной суммы и выделение исходной строки без контрольной суммы.
  //        Если сумма не совпала, возвращается пустая строка. Если в конце строки CR, он подавляется.
  //
  function  UartTruncCheckSumCR(const DataStr:LongString;
                                CR:Char=ASCII_CR):LongString;
  //
  // 138  - Получить ссылку на устройство Adam_Slot. Указывается имя устройства.
  //        Если имя пустое, используется ссылка Adam_Slot в секции описания
  //        устройства program, из которого вызвана DLL.
  //        Группа функций типа AdamXXX специально создана для обслуживания
  //        устройств в сети RS-485 типа ADAM-4000,IC-7000,IC-87000 и т.д.
  //        Для работы в сети RS-485 используется специальная дисциплина опроса
  //        устройств - они должны опрашиваться последовательно, а не асинхронно,
  //        так как все устройства используют один канал связи. Устройство типа
  //        Adam_Slot служит для того, чтобы через него программа получила
  //        корректный доступ к RS-485. Эти устройства обеспечивают корректный
  //        опрос, добавление/проверку/удаление контрольных сумм и разделителей,
  //        так что можно посылать/принимать сообщения, не заботясь о суммах.
  //
  function  AdamFindSlot(const aSlotDeviceName:LongString=''):Integer;
  //
  // 139  - Возвращает статус устройства Adam_Slot в сети RS-485.
  //        rs_NotAvail   = Сеть RS-485 недоступна. Скорее всего, неверная ссылка Adam_Slot.
  //        rs_NoRequest  = Запрос не был возбужден. Можно вызывать AdamSendRequest.
  //        rs_WaitQueue  = Запрос возбужден и ожидает очереди. Значит, запрос передан в устройство
  //                        Adam_Slot, но еще не передан в сеть из-за того, что она была занята. Ждите.
  //        rs_WaitAnswer = Запрос возбужден и передан по линии связи, ожидается ответ.
  //        rs_Answer     = Принят ответ на запрос. Вызывайте AdamGetAnswer.
  //        rs_TimeOut    = Ошибка TimeOut. Ответ на запрос не был принят в течение установленного времени.
  //        Если статус был Answer или TimeOut, то после вызова функции статус
  //        автоматически сбрасывается в rs_NoRequest, чтобы исключить возможность
  //        ошибочной повторной обработки уже обработанных запросов.
  //
  function  AdamGetStatus(aSlot:Integer):Integer;
  //
  // 140  - Посылает запрос в сеть RS-485. Запрос передается устройству AdamSlot,
  //        которое обеспечивает синхронизацию с другими устройствами. Указывается
  //        также время ожидания в миллисекундах, после которого регистрируется
  //        "TimeOut". Функцию надо вызывать, когда AdamGetStatus=rs_NoRequest.
  //        Запросы посылаются простые, без контрольных сумм и разделителя CR.
  //        Устройство AdamDevice само их прибавит, как надо.
  //
  function  AdamSendRequest(aSlot:Integer; aRequest:LongString;
                            aTimeOut:Integer):Boolean;
  //
  // 141  - Возвращает данные последнего посланного запроса.
  //
  function  AdamGetRequest(aSlot:Integer):LongString;
  //
  // 142  - Возвращает время посылки последнего запроса в миллисекундах.
  //
  function  AdamGetRequestTime(aSlot:Integer):Double;
  //
  // 143  - Возвращает ответ от устройства в сети RS-485 на посланный запрос.
  //        В ответе уже проверены и удалены контрольная сумма и разделитель CR.
  //        Вызывается, когда AdamGetStatus=rs_Answer.
  //
  function  AdamGetAnswer(aSlot:Integer):LongString;
  //
  // 144  - Возвращает номер порта, к которому подключен AdamSlot.
  //
  function  AdamGetPort(aSlot:Integer):Integer;
  //
  // 145  - Возвращает в виде 2-х символьной HEX-строки адрес устройства
  //        AdamSlot в сети RS-485. Используется при формировании сообщений.
  //
  function  AdamGetAddress(aSlot:Integer):LongString;
  //
  // 146  - Возвращает признак испльзования контрольных сумм в AdamSlot.
  //        Добавление/анализ/удаление контрольных сумм и разделителя CR
  //        происходит автоматически, так что при посылке сообщения формируются
  //        так, будто контрольные суммы не используются.
  //
  function  AdamGetUsesCheckSum(aSlot:Integer):Boolean;
  //
  // 147  - Точный аналог Daq Pascal функции clickparams. Смотри документацию по CRW-DAQ.
  //
  function  ClickParams(const Args:LongString):LongString;
  //
  // 148  - Точный аналог Daq Pascal функции add3d. Смотри документацию по CRW-DAQ.
  //
  function  Add3D(x,y,z:Double):Boolean;
  //
  // 149  - Точный аналог Daq Pascal функции plot3d. Смотри документацию по CRW-DAQ.
  //
  function  Plot3D(Nx,Ny:Integer; x1,y1,x2,y2:Double;
                   const opt:LongString):Boolean;
  //
  // 150  - Точный аналог Daq Pascal функции edit. Смотри документацию по CRW-DAQ.
  //
  function  Edit(const s:LongString):LongString;
  //
  // 151  - Точный аналог Daq Pascal функции editstate. Смотри документацию по CRW-DAQ.
  //
  function  EditState:Integer;
  //
  // 152  - Точный аналог Daq Pascal функции devsend. Смотри документацию по CRW-DAQ.
  //
  function  DevSend(d:Integer; const msg:LongString):Double;
  //
  // 153 - Точный аналог Daq Pascal функции daqfileref. Смотри документацию по CRW-DAQ.
  //
  function  FileRef(const FileName : LongString;
                    const DefExt   : LongString='';
                    const BaseFile : LongString='*'):LongString;
  //
  // 154  - Точный аналог Daq Pascal функции devpost. Смотри документацию по CRW-DAQ.
  //
  function  DevPost(d:Integer; const msg:LongString):Double;
  //
  // 155  - Вывод текста в консольное окно DAQ устройства.
  //        Работает всегда, даже если не было вызова RedirectStdOut(..).
  //        Вызов потокобезопасен, не дает задержки, выполняется через FIFO.
  //        Поэтому может применяться в Real Time задачах.
  //
  procedure Echo(const Msg:LongString);
  //
  // 156  - Переназначает стандартный ввод на ввод из консоли DAQ-устройства.
  //
  function  RedirectStdIn(var stdIn:Text):boolean;
  //
  // 157  - Переназначает стандартный вывод на вывод в консоль DAQ-устройства.
  //
  function  RedirectStdOut(var stdOut:Text):boolean;
  //
  // 158 - 187 - functions from DaqPascal API.
  //
  function  str2shortcut(const s:LongString):Integer;
  function  shortcut2str(shortcut:Integer):LongString;
  function  awakeflag:Boolean;
  function  crypt_ctrl(const s:LongString):LongString;
  function  crypt_encode(const s,key:LongString):LongString;
  function  crypt_decode(const s,key:LongString):LongString;
  function  getmd5fromstr(const s:LongString):LongString;
  function  getmd5fromfile(const s:LongString):LongString;
  function  getmd5fromtext(ref:Integer):LongString;
  function  wdt_reset(flag:Boolean):Double;
  function  guard_check(const s:LongString):Integer;
  function  timebase:Double;
  function  daqfileref(f,e:LongString):LongString;
  function  GetClockRes(what:Integer):Double;
  function  SetClockRes(res:Double):Double;
  function  cpu_start:Boolean;
  function  cpu_clock:Double;
  function  pidaffinity(pid,mask:Integer):Integer;
  function  devaffinity(ref,mask:Integer):Integer;
  function  clickwrite(const lines:LongString):Integer;
  function  clickread:Integer;
  function  clickwhat:Integer;
  function  clickwrote:Integer;
  function  clickfilter(aFilter:Integer):Integer;
  function  clickawaker(aAwaker:Integer):Integer;
  function  taglock(lock:Boolean):Integer;
  function  getai_par(n:integer; id:integer):Double;
  function  getao_par(n:integer; id:integer):Double;
  function  getdi_par(n:integer; id:integer):Double;
  function  getdo_par(n:integer; id:integer):Double;
  //
  // Properties
  //
  property  CurveLength[c:Integer]   : Integer     read GetCurveLength   write SetCurveLength;
  property  CurvePoint[c,i:Integer]  : TPoint2D    read GetCurvePoint    write SetCurvePoint;
  property  CurveName[c:Integer]     : LongString  read GetCurveName;
  property  CurveColor[c:Integer]    : Integer     read GetCurveColor;
  property  CurveStyle[c:Integer]    : Integer     read GetCurveStyle;
  property  CurveComment[c:Integer]  : LongString  read GetCurveComment  write SetCurveComment;
  property  CurveLimits[c:Integer]   : TRect2D     read GetCurveLimits;
  property  CurveLastPoint[c:Integer]: TPoint2D    read GetCurveLastPoint;
  //
  // Properties of Port
  //
  property  PortB[Address:Word]      : Byte        read GetPortB         write SetPortB;
  property  PortW[Address:Word]      : Word        read GetPortW         write SetPortW;
  property  PortL[Address:Word]      : LongWord    read GetPortL         write SetPortL;
 end;
 //
 ///////////////////////////////////////////////////////////////////////////////
 // Класс TDanApi содержит интерфейс для создания утилит анализа данных.      //
 // Перед вызовом утилиты анализа данных пользователь вводит в диалоговом     //
 // окне аргументы, для доступа к ним есть GetArguments,GetArgumentAsXX.      //
 // Интерфейс дает доступ к трем окнам:                                       //
 // 1) Окно - источник, активное окно с кривыми в момент вызова утилиты.      //
 //    Оно служит источником данных, которые надо обрабатывать.               //
 // 2) Окно - приемник, вновь созданное окно для помещения в него результата. //
 // 3) Окно - буфер обмена, служит для того, чтобы задавать дополнительную    //
 //    кривую в качестве аргумента. Например, для сложения кривых одна        //
 //    берется из окна - источника, другая копируется в буфер обмена.         //
 // Для доступа к кривым и окнам интерфейс использует целочисленные ссылки.   //
 // Будем обозначать ссылки для доступа к кривым буквой c, к окнам - w, а     //
 // к точкам кривой - i. При этом:                                            //
 //   w > 0 - идентифицирует окно - источник                                  //
 //   w < 0 - идентифицирует окно - приемник                                  //
 //   w = 0 - идентифицирует окно - буфер обмена                              //
 //   c > 0 - идентифицирует кривую с номером "+c"  в окне источника          //
 //   c < 0 - идентифицирует кривую с номером "-c" в окне приемника           //
 //   c = 0 - идентифицирует кривую, скопированную в буфер обмена             //
 // Нумерация кривых в окне начинается с 1, при этом абсолютное значение      //
 // ссылки дает номер кривой в окне, а знак-ссылку на окно (+1 = источник,    //
 // -1 = приемник, 0 = буфер обмена). В буфере обмена доступна одна кривая,   //
 // та, которая была вставлена туда последней, что вроде бы логично.          //
 // При такой нумерации ссылка кривой может служить одновременно и  ссылкой   //
 // для доступа к окну, которому она принадлежит.                             //
 // Нумерация точек кривых начинается с 0 .                                   //
 // Значения, связанные с буфером обмена, можно только читать, но не писать.  //
 // Например, вызов:                                                          //
 //  WindowCaption[+1]:='Source' - задаст имя окна источника                  //
 //  WindowCaption[-1]:='Target' - задаст имя окна приемника                  //
 //  WindowCaption[0]:='Clipbrd' - ничего не сделает                          //
 // Дополнительную информацию можно получить, изучая примеры меющихся утилит. //
 ///////////////////////////////////////////////////////////////////////////////
 //
 IDanApi = interface(IPluginApi) [SIID_IDanApi]
  //
  // 0    - Возвращает список аргументов вызова, которые пользователь редактирует
  //        перед вызовом утилиты в диалоговом окне. Сам список аргументов и его
  //        начальное состояние описывается в секциях [Arguments.Rus] и
  //        [Arguments.Eng] в комментариях текста программы.
  //
  function  GetArguments:LongString;
  //
  // 1    - Возвращает аргумент вызова по имени в виде строки или пустую строку
  //        если аргумента с таким именем нет.
  //
  function  GetArgumentAsString(const aName:LongString):LongString;
  //
  // 2    - Возвращает аргумент вызова по имени в виде целого числа.
  //
  function  GetArgumentAsInteger(const aName:LongString;
                                 var aValue:Integer):boolean;
  //
  // 3    - Возвращает аргумент вызова по имени в виде вещественного числа.
  //
  function  GetArgumentAsDouble(const aName:LongString;
                                 var aValue:Double):boolean;
  //
  // 4    - Проверяет, существует ли окно, заданное ссылкой w.
  //
  function  WindowExists(w:Integer):Boolean;
  //
  // 5    - Проверяет, существует ли кривая, заданная ссылкой c.
  //
  function  CurveExists(c:Integer):Boolean;
  //
  // 6    - Возвращает заголовок окна, заданного ссылкой w.
  //
  function  GetWindowCaption(w:Integer):LongString;
  //
  // 7    - Задает заголовок окна, заданного ссылкой w.
  //
  procedure SetWindowCaption(w:Integer;const aCaption:LongString);
  //
  // 8    - Возвращает верхний комментарий окна, заданного ссылкой w.
  //
  function  GetWindowTitle(w:Integer):LongString;
  //
  // 9    - Задает верхний комментарий окна, заданного ссылкой w.
  //
  procedure SetWindowTitle(w:Integer;const aTitle:LongString);
  //
  // 10   - Возвращает нижний комментарий окна, заданного ссылкой w.
  //
  function  GetWindowLegend(w:Integer):LongString;
  //
  // 11   - Задает нижний комментарий окна, заданного ссылкой w.
  //
  procedure SetWindowLegend(w:Integer;const aLegend:LongString);
  //
  // 12   - Возвращает прямоугольник ROI - "область интересов" окна w.
  //        Прямоугольник содержит значения NAN, если ROI не задано.
  //
  function  GetWindowRoi(w:Integer):TRect2D;
  //
  // 13   - Задает прямоугольник ROI - "область интересов" окна w.
  //        Для отмены области интересов надо задать NAN.
  //
  procedure SetWindowRoi(w:Integer; const aRoi:TRect2D);
  //
  // 14   - Возвращает текст комментария окна, заданного ссылкой w.
  //
  function  GetWindowComment(w:Integer):LongString;
  //
  // 15   - Задает текст комментария окна, заданного ссылкой w.
  //
  procedure SetWindowComment(w:Integer;const aComment:LongString);
  //
  // 16   - Возвращает ссылку выделенной кривой в окне w или 0, если не выбрана.
  //        Абсолютное значение ссылки дает номер кривой в окне.
  //        Для окна буфера обмена всегда возвращается 0.
  //
  function  GetSelectedCurve(w:Integer):Integer;
  //
  // 17   - Выделяет кривую с в окне w. Можно использовать как ссылку,
  //        так и номер кривой, так как c берется по модулю.
  //
  procedure SetSelectedCurve(w,c:Integer);
  //
  // 18   - Возвращает текущее число кривых в окне, заданном ссылкой w.
  //
  function  GetCurvesCount(w:Integer):Integer;
  //
  // 19   - Задает число кривых в окне, заданном ссылкой w.
  //        При необходимости уничтожает лишние кривые или создает новые пустые
  //        кривые, в зависимости от того, сколько кривых уже было в окне.
  //        Поэтому функцию можно использовать для создания кривых вместо
  //        CurveCreate, а также для удаления кривых. Например, вызов типа
  //        CurveCount[-1]:=0; удалит все кривые в окне приемника.
  //
  procedure SetCurvesCount(w,aCount:Integer);
  //
  // 20   - Возвращает длину кривой c.
  //
  function  GetCurveLength(c:Integer):Integer;
  //
  // 21   - Задает новую длину кривой c.
  //
  procedure SetCurveLength(c,aLength:Integer);
  //
  // 22   - Возвращает точку номер i из кривой, заданной ссылкой c.
  //        Напоминаем, что нумерация точек идет с 0, а кривых с 1.
  //
  function  GetCurvePoint(c,i:Integer):TPoint2D;
  //
  // 23   - Задает значение точки номер i на кривой c.
  //
  procedure SetCurvePoint(c,i:Integer; const aPoint:TPoint2D);
  //
  // 24   - Добавляет точку в конец кривой c.
  //
  procedure CurveAddPoint(c:Integer; const aPoint:TPoint2D);
  //
  // 25   - Добавляет aCount точек в конец кривой c.
  //
  procedure CurveAddPoints(c:Integer; const aX,aY:array of Double;
                                            aCount:Integer);
  //
  // 26   - Удаляет точку i из кривой c.
  //
  procedure CurveDeletePoint(c,i:Integer);
  //
  // 27   - Удаляет aCount точек начиная с i из кривой c.
  //
  procedure CurveDeletePoints(c,i,aCount:Integer);
  //
  // 28   - Вставляет точку в позицию i кривой c.
  //
  procedure CurveInsertPoint(c,i:Integer; const aPoint:TPoint2D);
  //
  // 29   - Вставляет aCount точек в позицию i кривой c.
  //
  procedure CurveInsertPoints(c,i:Integer; const aX,aY:array of Double;
                              aCount:LongInt);
  //
  // 30   - Возвращает имя кривой, заданной ссылкой c.
  //
  function  GetCurveName(c:Integer):LongString;
  //
  // 31   - Задает имя кривой, заданной ссылкой c.
  //
  procedure SetCurveName(c:Integer; const aName:LongString);
  //
  // 32   - Возвращает цвет кривой, заданной ссылкой c.
  //
  function  GetCurveColor(c:Integer):Integer;
  //
  // 33   - Задает цвет кривой, заданной ссылкой c.
  //
  procedure SetCurveColor(c:Integer; aColor:Integer);
  //
  // 34   - Возвращает стиль кривой, заданной ссылкой c.
  //
  function  GetCurveStyle(c:Integer):Integer;
  //
  // 35   - Задает стиль кривой, заданной ссылкой c.
  //
  procedure SetCurveStyle(c:Integer; aStyle:Integer);
  //
  // 36   - Возвращает комментарий кривой, заданной ссылкой c.
  //
  function  GetCurveComment(c:Integer):LongString;
  //
  // 37   - Задает комментарий кривой, заданной ссылкой c.
  //
  procedure SetCurveComment(c:Integer; const aComment:LongString);
  //
  // 38   - Быстрый поиск индекса ближайшей к t точки кривой c.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveGetIndexAt(c:Integer; t:Double):Integer;
  //
  // 39   - Линейная интерполяция кривой c в точке t.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveInterpolateAt(c:Integer; t:Double):Double;
  //
  // 40   - Сглаживание кривой c в точке t окном шириной Window, а также ядром
  //        типа K(x)=(1-x^k1)^k2. Power - степень полинома или интеграл при Power=-1.
  //        Точки кривой должны быть отсортированы в порядке возрастания по оси X.
  //
  function  CurveSmoothAt(c:Integer; t:Double;
                        Window:Double; Power,K1,K2:Integer):Double;
  //
  // 41   - Вычисляет медиану кривой c в интервале точек [i..j].
  //        Возвращает индекс aIndex и значение aValue медианы.
  //        Можно выбирать метод сортировки 0=Shell,1=Quick.
  //
  function  CurveMedianaAt(c,i,j:Integer; var aIndex:Integer; var aValue:Double;
                         SortMethod:Integer=1):Boolean;
  //
  // 42   - Вычисляет массив флагов типа "кривая отсортирована" и т.д.
  //        Смотрите описание констант cfXXXX.
  //
  function  CurveFlags(c:Integer; AbsEps:Double=0;RelEps:Double=1E-10;
                       Small:Integer=2):LongInt;
  //
  // 43   - Копирование всех данных из кривой c2 в кривую c1.
  //
  procedure CurveAssign(c1,c2:Integer);
  //
  // 44   - Добавление к кривой c1 точек из кривой c2.
  //
  procedure CurveCatenate(c1,c2:Integer);
  //
  // 45  - Создание в окне w кривой с заданным именем, цветом, стилем, комментарием.
  //       Возвращает ссылку на кривую.
  //
  function  CreateCurve(w:Integer; const aName:LongString='';
                        aColor:Integer=0;  aStyle:Integer=0;
                        aComment:LongString=''):Integer;
  //
  // 46  - Сортирует кривую c, используя флаги Flags, смотри константы sfXXXX.
  //       Если указана упаковка, абсолютная и относительная точность определяют
  //       сжатие кривой. При этом точки, у которых ключ сортировки одинаков
  //       в пределах точности, объединяются в одну среднюю точку.
  //       Можно выбирать метод сортировки 0=Shell,1=Quick.
  //
  procedure CurveSort(c:Integer; Flags:Integer=0; AbsEps:Double=0;
                      RelEps:Double=1E-10; Method:Integer=0);
  //
  // 47   - Возвращает минимальный прямоугольник, содержащий все точки кривой c.
  //
  function  GetCurveLimits(c:Integer):TRect2D;
  //
  // 48   - Интегрирует кривую c методом трапеций, возвращает первообразную.
  //
  procedure CurveIntegrate(c:Integer; AFreeConst:Double=0);
  //
  // 49   - Дифференциирует кривую c, возвращает производную.
  //
  procedure CurveDifferentiate(c:Integer);
  //
  // 50   - Применяет к кривой c пороговый медианный фильтр.
  //
  procedure CurveMedianFiltr(c:Integer; Width:Integer=1;
                             AbsEps:Double=0; RelEps:Double=1E-10);
  //
  // 51   - Возвращает интеграл участка (a,b) кривой c.
  //
  function  GetCurveIntegral(c:Integer; a,b:Double):Double;
  //
  // 52   - Возвращает последнюю точку кривой c.
  //
  function  GetCurveLastPoint(c:Integer):TPoint2D;
  //
  // Properties of CurveWindow +n/0/-n=Source/Clipboars/Target
  //
  property  WindowCaption[w:Integer] : LongString  read GetWindowCaption write SetWindowCaption;
  property  WindowTitle[w:Integer]   : LongString  read GetWindowTitle   write SetWindowTitle;
  property  WindowLegend[w:Integer]  : LongString  read GetWindowLegend  write SetWindowLegend;
  property  WindowRoi[w:Integer]     : TRect2D     read GetWindowRoi     write SetWindowRoi;
  property  WindowComment[w:Integer] : LongString  read GetWindowComment write SetWindowComment;
  property  SelectedCurve[w:Integer] : Integer     read GetSelectedCurve write SetSelectedCurve;
  property  CurvesCount[w:Integer]   : Integer     read GetCurvesCount   write SetCurvesCount;
  property  CurveLength[c:Integer]   : Integer     read GetCurveLength   write SetCurveLength;
  property  CurvePoint[c,i:Integer]  : TPoint2D    read GetCurvePoint    write SetCurvePoint;
  property  CurveName[c:Integer]     : LongString  read GetCurveName     write SetCurveName;
  property  CurveColor[c:Integer]    : Integer     read GetCurveColor    write SetCurveColor;
  property  CurveStyle[c:Integer]    : Integer     read GetCurveStyle    write SetCurveStyle;
  property  CurveComment[c:Integer]  : LongString  read GetCurveComment  write SetCurveComment;
  property  CurveLimits[c:Integer]   : TRect2D     read GetCurveLimits;
  property  CurveLastPoint[c:Integer]: TPoint2D    read GetCurveLastPoint;
 end;
 //
 // Пример функций сравнения, ключа и перестановки для поиска/сортировки.
 //
function  CompareDoubles(Data,Key1,Key2:Pointer):Integer;
function  KeyOfDoubles(Data:Pointer; KeyIndex:Integer):Pointer;
procedure SwapDoubles(Data:Pointer; i,j:Integer);
 //
 // Dump/undump routines to be used with MIME
 // For example:
 //  var x,y:double; i,j:integer; s:string;
 //  s:=mime_encode(r2dump(x)+r2dump(y)+i2dump(i)+i2dump(j));
 //  s:=mime_decode(s);
 //  x:=dump2r(copy(s,1,8));
 //  y:=dump2r(copy(s,9,8));
 //  i:=dump2i(copy(s,17,4));
 //  j:=dump2i(copy(s,21,4));
 //
function c2dump(const x:Char):LongString;
function b2dump(const x:Boolean):LongString;
function i2dump(const x:Integer):LongString;
function r2dump(const x:Double):LongString;
function x2dump(var x; Size:Integer):LongString;
function dump2c(const x:LongString):Char;
function dump2b(const x:LongString):Boolean;
function dump2i(const x:LongString):Integer;
function dump2r(const x:LongString):Double;

procedure Kill(var What:IScriptInterpreter); overload;
procedure Kill(var What:IPolynomWrap); overload;
procedure Kill(var What:ILongIntVectorWrap); overload;
procedure Kill(var What:IDoubleVectorWrap); overload;
procedure Kill(var What:ILongIntMatrixWrap); overload;
procedure Kill(var What:IDoubleMatrixWrap); overload;
procedure Kill(var What:ISVDWrap); overload;
procedure Kill(var What:IFifoWrap); overload;

procedure ResetFpu;

implementation

function CrwApiEdition:Cardinal;
begin
 Result:=TheCrwApiEdition;
end;

function ua_FileDefLow:Integer;
begin
 Result:=IfThen(FileNameCaseSensitive,ua_FileDef,ua_FileLow);
end;

function CompareDoubles(Data,Key1,Key2:Pointer):Integer;
begin
 if Double(Key1^)<Double(Key2^) then Result:=-1 else
 if Double(Key1^)>Double(Key2^) then Result:=+1 else Result:= 0;
end;

function KeyOfDoubles(Data:Pointer; KeyIndex:Integer):Pointer;
begin
 Result:=@TDoubleArray(Data^)[KeyIndex];
end;

procedure SwapDoubles(Data:Pointer; i,j:Integer);
var dummy:Double;
begin
 dummy:=TDoubleArray(Data^)[i];
 TDoubleArray(Data^)[i]:=TDoubleArray(Data^)[j];
 TDoubleArray(Data^)[j]:=dummy;
end;

function c2dump(const x:Char):LongString;
begin
 SetString(Result,PChar(@x),sizeof(x));
end;

function b2dump(const x:Boolean):LongString;
begin
 SetString(Result,PChar(@x),sizeof(x));
end;

function i2dump(const x:Integer):LongString;
begin
 SetString(Result,PChar(@x),sizeof(x));
end;

function r2dump(const x:Double):LongString;
begin
 SetString(Result,PChar(@x),sizeof(x));
end;

function x2dump(var x; Size:Integer):LongString;
begin
 SetString(Result,PChar(@x),Size);
end;

procedure SafeMove(const Source; var Dest; Count: LongInt);
begin
 if (Count>0) and (@Source<>nil) and (@Dest<>nil) then Move(Source,Dest,Count);
end;

function dump2b(const x:LongString):Boolean;
begin
 Result:=false;
 SafeMove(x[1],Result,min(sizeof(Result),Length(x)));
end;

function dump2c(const x:LongString):Char;
begin
 Result:=#0;
 SafeMove(x[1],Result,min(sizeof(Result),Length(x)));
end;

function dump2i(const x:LongString):Integer;
begin
 Result:=0;
 SafeMove(x[1],Result,min(sizeof(Result),Length(x)));
end;

function dump2r(const x:LongString):Double;
begin
 Result:=0;
 SafeMove(x[1],Result,min(sizeof(Result),Length(x)));
end;

function IncPtr(Base:Pointer; ByteOffset:LongInt):Pointer;
begin
 Result:=PChar(Base)+ByteOffset;
end;

procedure Kill(var What:IScriptInterpreter);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:IPolynomWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:ILongIntVectorWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:IDoubleVectorWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:ILongIntMatrixWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:IDoubleMatrixWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:ISVDWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure Kill(var What:IFifoWrap);
begin
 if Assigned(What) then What:=nil;
end;

procedure ResetFpu; // Reset FPU to safe state, clear and mask all exceptions.
begin
 Math.SetExceptionMask([Low(TFPUException)..High(TFPUException)]);
 Math.ClearExceptions;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_crwapi;
begin
 if isLibrary then begin
  ResetFpu;
 end;
end;

procedure Free_crw_crwapi;
begin
end;

initialization

 Init_crw_crwapi;

finalization

 Free_crw_crwapi;

end.

//////////////
// END OF FILE
//////////////

