////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// DAQ System Device base.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqdev; // DAQ System Device base

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}
{$WARN 6058 off : Call to subroutine "$1" marked as inline is not inlined}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf, lclproc,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator, Form_ListBoxSelection,
 Form_UartTerminal, Form_CalibDialog, Form_DaqEditTagDialog,
  Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo, _crw_ef,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_calib,
 _crw_curves, _crw_daqtags, _crw_daqsys, _crw_daqevnt,
 _crw_pio, _crw_couple, _crw_riff, _crw_syslog,
 _crw_dynar, _crw_snd, _crw_guard, _crw_assoc,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 TSensorEvent описывает событие нажатия сенсора
 *******************************************************************************
 }
type
 TSensorEvent = packed record
  What       : TEventWhat;
  Key        : Word;
  Shift      : TShiftState;
  Where      : TPoint2I;
  Wheel      : Integer;
  When       : Int64;
  Button     : Integer;
  Sensor     : PureString;
  Tag        : Integer;
  Wrote      : Cardinal;
  Cookies    : array[0..(1024*4-1)-292] of Char;
 end;

const
 TDaqDeviceSensorEventsFifoLength = 16;

function SensorEvent:TSensorEvent; overload;
procedure SensorEvent(out Event:TSensorEvent); overload;
function SensorEvent(const Lines:LongString):TSensorEvent; overload;
function SensorEvent(Circuit:TFormCircuitWindow; var Event:TCircuitEvent;
         Sensor:TCircuitSensor):TSensorEvent; overload;
function SensorEventParams(Event:TSensorEvent;const Name:LongString):LongString;

 {
 *******************************************************************************
 Устройство DAQ.
 *******************************************************************************
 }
const                           // HandleMessage flags
 hf_SkipAwake      = $00000001; // Skip Awake (for devPost)
 hf_Default        = $00000000; // Default HandleMessage flags
const                           // ClickFilter bits
 cf_MouseDown      = 1 shl Ord(evMouseDown);      // 2
 cf_MouseUp        = 1 shl Ord(evMouseUp);        // 4
 cf_MouseMove      = 1 shl Ord(evMouseMove);      // 8
 cf_KeyDown        = 1 shl Ord(evKeyDown);        // 16
 cf_KeyUp          = 1 shl Ord(evKeyUp);          // 32
 cf_MouseWheel     = 1 shl Ord(evMouseWheel);     // 64
 cf_MouseWheelDown = 1 shl Ord(evMouseWheelDown); // 128
 cf_MouseWheelUp   = 1 shl Ord(evMouseWheelUp);   // 256
 cf_Default        = cf_MouseDown+cf_KeyDown;
type
 TDaqDevicePropertyDialog = class;
 TDaqDeviceCommonPropertyDialog = class;
 TDaqDevice = class(TLatch)
 private
  myName                     : LongString;
  myFamily                   : LongString;
  myModel                    : LongString;
  mySysLogSign               : LongString;
 protected
  procedure SetDeviceFamily(aFamily:LongString);
  procedure SetDeviceModel(aModel:LongString);
 private
  myComment                  : LongString;
  myInquiryTimer             : TIntervalTimer;
  myAnalogFifo               : TFifo;
  myDigitalFifo              : TFifo;
  myDigitalFirst             : Boolean;
  myErrorsTotal              : Integer;
  myErrorsCount              : packed array[byte] of Integer;
  myGroupTag                 : Integer;
  myStartingOrder            : Integer;
  myStoppingOrder            : Integer;
  myDispatcherFilter         : Integer;
  myAnalogInputCurves        : TObjectStorage;
  myAnalogInputSmooth        : TObjectStorage;
  myDigitalInputCurves       : TObjectStorage;
  myDigitalInputBits         : TLongIntVector;
  myAnalogOutputCurves       : TObjectStorage;
  myPrimaryAOs               : Integer;
  mySecondaryAOs             : Integer;
  myAnalogOutputCompressors  : TObjectStorage;
  myAnalogOutputHistoryLens  : TLongIntVector;
  myDigitalOutputCurves      : TObjectStorage;
  myDigitalOutputHistoryLens : TLongIntVector;
  myCalibrations             : TObjectStorage;
  myPropertyDialog           : TDaqDevicePropertyDialog;
  mySensorClicked            : TFifo;
  myClickFilter              : Cardinal;
  myClickAwaker              : Cardinal;
  myCommonPropertyDialog     : TDaqDeviceCommonPropertyDialog;
  function    GetName:LongString;
  function    GetFamily:LongString;
  function    GetModel:LongString;
  function    GetDevSection:LongString;
  function    GetDeviceDescription:LongString;
  function    GetComment:LongString;
  procedure   SetComment(const aComment:LongString);
  function    GetInquiryTimer:TIntervalTimer;
  function    GetAnalogFifoSize:Integer;
  procedure   SetAnalogFifoSize(aSize:Integer);
  function    GetDigitalFifoSize:Integer;
  procedure   SetDigitalFifoSize(aSize:Integer);
  function    GetErrorsTotal:Integer;
  function    GetErrorsCount(ErrorCode:Byte):Integer;
  function    GetGroupTag:Integer;
  procedure   SetGroupTag(aGroupTag:Integer);
  function    GetStartingOrder:Integer;
  procedure   SetStartingOrder(aStartingOrder:Integer);
  function    GetStoppingOrder:Integer;
  procedure   SetStoppingOrder(aStoppingOrder:Integer);
  function    GetDispatcherFilter:Integer;
  procedure   SetDispatcherFilter(aDispatcherFilter:Integer);
  function    GetNumAnalogInputs:Integer;
  procedure   SetNumAnalogInputs(aCount:Integer);
  function    GetAnalogInputCurve(Index:Integer):TCurve;
  procedure   SetAnalogInputCurve(Index:Integer; aCurve:TCurve);
  function    GetAnalogInputSmoother(Index:Integer):TDaqSmoother;
  procedure   SetAnalogInputSmoother(Index:Integer; aSmoother:TDaqSmoother);
  function    GetNumDigitalInputs:Integer;
  procedure   SetNumDigitalInputs(aCount:Integer);
  function    GetDigitalInputCurve(Index:Integer):TCurve;
  procedure   SetDigitalInputCurve(Index:Integer; aCurve:TCurve);
  function    GetNumAnalogOutputs:Integer;
  procedure   SetNumAnalogOutputs(aCount:Integer);
  function    GetNumPrimaryAnalogOutputs:Integer;
  function    GetNumSecondaryAnalogOutputs:Integer;
  function    GetAnalogOutputCurve(Index:Integer):TCurve;
  procedure   SetAnalogOutputCurve(Index:Integer; aCurve:TCurve);
  function    GetAnalogOutputCompressor(Index:Integer):TDaqCompressor;
  procedure   SetAnalogOutputCompressor(Index:Integer; aCompressor:TDaqCompressor);
  function    GetAnalogOutputHistoryLen(Index:Integer):Integer;
  procedure   SetAnalogOutputHistoryLen(Index:Integer; aHistoryLen:Integer);
  function    GetNumDigitalOutputs:Integer;
  procedure   SetNumDigitalOutputs(aCount:Integer);
  function    GetDigitalOutputCurve(Index:Integer):TCurve;
  procedure   SetDigitalOutputCurve(Index:Integer; aCurve:TCurve);
  function    GetDigitalOutputHistoryLen(Index:Integer):Integer;
  procedure   SetDigitalOutputHistoryLen(Index:Integer; aHistoryLen:Integer);
  function    GetNumCalibrations:Integer;
  procedure   SetNumCalibrations(aCount:Integer);
  function    GetCalibration(Index:Integer):TPolynomCalibration;
  procedure   SetCalibration(Index:Integer; aCalibration:TPolynomCalibration);
  function    GetPropertyDialog:TDaqDevicePropertyDialog;
  function    GetClickFilter:Cardinal;
  procedure   SetClickFilter(aClickFilter:Cardinal);
  function    GetClickAwaker:Cardinal;
  procedure   SetClickAwaker(aClickAwaker:Cardinal);
 public
  property    Name                              : LongString     read GetName;
  property    Family                            : LongString     read GetFamily;
  property    Model                             : LongString     read GetModel;
  property    DevSection                        : LongString     read GetDevSection;
  property    DeviceDescription                 : LongString     read GetDeviceDescription;
  property    Comment                           : LongString     read GetComment                 write SetComment;
  property    InquiryTimer                      : TIntervalTimer read GetInquiryTimer;
  property    AnalogFifoSize                    : Integer        read GetAnalogFifoSize          write SetAnalogFifoSize;
  property    DigitalFifoSize                   : Integer        read GetDigitalFifoSize         write SetDigitalFifoSize;
  property    ErrorsTotal                       : Integer        read GetErrorsTotal;
  property    ErrorsCount[i:Byte]               : Integer        read GetErrorsCount;
  property    GroupTag                          : Integer        read GetGroupTag                write SetGroupTag;
  property    StartingOrder                     : Integer        read GetStartingOrder           write SetStartingOrder;
  property    StoppingOrder                     : Integer        read GetStoppingOrder           write SetStoppingOrder;
  property    DispatcherFilter                  : Integer        read GetDispatcherFilter        write SetDispatcherFilter;
  property    NumAnalogInputs                   : Integer        read GetNumAnalogInputs         write SetNumAnalogInputs;
  property    AnalogInputCurve[i:Integer]       : TCurve         read GetAnalogInputCurve        write SetAnalogInputCurve;
  property    AnalogInputSmoother[i:Integer]    : TDaqSmoother   read GetAnalogInputSmoother     write SetAnalogInputSmoother;
  property    NumDigitalInputs                  : Integer        read GetNumDigitalInputs        write SetNumDigitalInputs;
  property    DigitalInputCurve[i:Integer]      : TCurve         read GetDigitalInputCurve       write SetDigitalInputCurve;
  property    NumAnalogOutputs                  : Integer        read GetNumAnalogOutputs        write SetNumAnalogOutputs;
  property    NumPrimaryAnalogOutputs           : Integer        read GetNumPrimaryAnalogOutputs;
  property    NumSecondaryAnalogOutputs         : Integer        read GetNumSecondaryAnalogOutputs;
  property    AnalogOutputCurve[i:Integer]      : TCurve         read GetAnalogOutputCurve       write SetAnalogOutputCurve;
  property    AnalogOutputCompressor[i:Integer] : TDaqCompressor read GetAnalogOutputCompressor  write SetAnalogOutputCompressor;
  property    AnalogOutputHistoryLen[i:Integer] : Integer        read GetAnalogOutputHistoryLen  write SetAnalogOutputHistoryLen;
  property    NumDigitalOutputs                 : Integer        read GetNumDigitalOutputs       write SetNumDigitalOutputs;
  property    DigitalOutputCurve[i:Integer]     : TCurve         read GetDigitalOutputCurve      write SetDigitalOutputCurve;
  property    DigitalOutputHistoryLen[i:Integer]: Integer        read GetDigitalOutputHistoryLen write SetDigitalOutputHistoryLen;
  property    NumCalibrations                   : Integer        read GetNumCalibrations         write SetNumCalibrations;
  property    Calibration[i:Integer]            : TPolynomCalibration read GetCalibration        write SetCalibration;
  property    PropertyDialog                    : TDaqDevicePropertyDialog read GetPropertyDialog;
  property    ClickFilter                       : Cardinal       read GetClickFilter             write SetClickFilter;
  property    ClickAwaker                       : Cardinal       read GetClickAwaker             write SetClickAwaker;
  procedure   SetSensorClicked(const aSensorEvent:TSensorEvent);
  procedure   GetSensorClicked(var aSensorEvent:TSensorEvent);
  function    SuitableClick(What:TEventWhat):Boolean;
  function    AwakeOnClick(What:TEventWhat):Boolean;
  function    SysLogSign(const Sub:LongString=''):LongString;
 public
  constructor Create(const aName:LongString);
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
  destructor  Destroy; override;
 public
  procedure   ClearAnalogFifo;
  procedure   ClearDigitalFifo;
  procedure   FixError(ErrorCode:Byte);
  procedure   ClearErrors;
  function    GetDaqEvent(out Event:TDaqEvent):Boolean;
  function    PutDaqEvent(const Event:TDaqEvent):Boolean;
  procedure   DispatchDaqEvent(var Event:TDaqEvent);
  procedure   DispatchEvents;
  procedure   ClearAnalogInputCurve(Index:Integer);
  function    SmoothAnalogInputCurve(Index:Integer; Where:Double):Double;
  function    ConstructAnalogInputMap(ofs,n:Integer):Integer;
  procedure   ClearDigitalInputCurve(Index:Integer);
  function    GetDigitalInputBit(Index:Integer; out Inversion:Boolean):Integer;
  procedure   SetDigitalInputBit(Index:Integer; aBit:Integer; aInversion:Boolean);
  function    ConstructDigitalInputWord(ofs,n:Integer):Integer;
  function    ConstructDigitalInputMap(ofs,n:Integer):Integer;
  procedure   SetNumAnalogOutputsMux(nPrimary,nSecondary:Integer);
  procedure   ClearAnalogOutputCurve(Index:word);
  function    PackAnalogOutputIndex(iPrimary,iSecondary:Integer):Integer;
  procedure   UnpackAnalogOutputIndex(LinearIndex:Integer; out iPrimary,iSecondary:Integer);
  function    ConstructAnalogOutputMap(ofs,n:Integer):Integer;
  procedure   ClearDigitalOutputCurve(Index:Integer);
  function    ConstructDigitalOutputMap(ofs,n:Integer):Integer;
  procedure   InitCalibration(Index:Integer);
  function    Transform(Index:Integer; Argument,Parametr:Double):Double;
  function    ReadCalibration(Index:Integer; FileName,SectionName,CalibName:LongString):Boolean;
  procedure   EditCalibration(Index:Integer);
  procedure   OpenPropertyDialog;
  function    NewPropertyDialog:TDaqDevicePropertyDialog; virtual;
  function    UpdatePropertyDialog:Boolean; virtual;
  procedure   AdvancedPropertyDialog; virtual;
  procedure   RightButtonPropertyDialog; virtual;
  function    CommonPropertyDialog:TDaqDeviceCommonPropertyDialog;
  procedure   OpenCommonPropertyDialog;
  function    NewCommonPropertyDialog:TDaqDeviceCommonPropertyDialog;
  function    UpdateCommonPropertyDialog:Boolean;
 public
  {виртуальный интерфейс высокого уровня}
  procedure   Idle; virtual;
  procedure   Poll; virtual;
  procedure   Clear; virtual;
  procedure   ClearDevice; virtual;
  function    CheckDevice:Boolean; virtual;
  procedure   Config(FileName:LongString); virtual;
  procedure   Animate; virtual;
  function    GetProperty(TheText:TText):TText; virtual;
  function    Start:Boolean; virtual;
  procedure   Stop; virtual;
  procedure   Awake;virtual;
  procedure   Action; virtual;
  procedure   Watchdog; virtual;
  function    GotEvents:Boolean; virtual;
  function    HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double; virtual;
  procedure   HandleSensorClick(const aEvent : TSensorEvent); virtual;
 end;
 TDaqDevicePropertyDialog = class(TMasterForm)
 private
  myLinkedDevice : TDaqDevice;
  function  GetLinkedDevice:TDaqDevice;
  procedure SetLinkedDevice(aDevice:TDaqDevice);
 public
  property    LinkedDevice : TDaqDevice read GetLinkedDevice write SetLinkedDevice;
 end;
 TDaqDeviceCommonPropertyDialog = class(TDaqDevicePropertyDialog)
 end;

procedure Kill(var TheObject:TDaqDevice); overload;

 {
 *******************************************************************************
 Тип для функций, создающих устройства по имени модели.
 Должен проанализировать Family,Model и создать устройство с именем Name, взяв
 дополнительные данные из файла Config, секции [Name].
 Если устройство не создано, вернуть nil.
 *******************************************************************************
 }
type
 TDaqDeviceConstructor = function(const Name,Family,Model,Config:LongString):TDaqDevice;

 {
 *******************************************************************************
 Список устройств DAQ
 *******************************************************************************
 }
type
 TDaqDeviceList = class(TObjectStorage)
 private
  myNamePad       : Integer;
  myFamilyPad     : Integer;
  myModelPad      : Integer;
  myStartingIndex : TLongIntVector;
  myStoppingIndex : TLongIntVector;
  function  GetDevices(Index:Integer):TDaqDevice;
  procedure SetDevices(Index:Integer; aDevice:TDaqDevice);
  function  GetErrorsTotal:Integer;
  function  GetErrorsCount(ErrorCode:Byte):Integer;
  function  GetStartingIndex:TLongIntVector;
  function  GetStoppingIndex:TLongIntVector;
  function  GetStartingList:LongString;
  function  GetStoppingList:LongString;
 public
  constructor Create(aOwnsObjects : Boolean = true;
                     aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                     aStep        : LongInt = DefaultTObjectStorageStep);
  destructor  Destroy; override;
 public
  property  Devices[i:Integer]  : TDaqDevice     read GetDevices write SetDevices; default;
  property  ErrorsTotal         : Integer        read GetErrorsTotal;
  property  ErrorsCount[i:Byte] : Integer        read GetErrorsCount;
  property  StartingIndex       : TLongIntVector read GetStartingIndex;
  property  StoppingIndex       : TLongIntVector read GetStoppingIndex;
  property  StartingList        : LongString     read GetStartingList;
  property  StoppingList        : LongString     read GetStoppingList;
 public
  procedure StartingSort;
  procedure StoppingSort;
  function  Find(const DeviceName:LongString):TDaqDevice;
  procedure ClearErrors;
  procedure DispatchEvents;
  procedure Idle; virtual;
  procedure Poll; virtual;
  procedure Clear; virtual;
  procedure ClearDevice; virtual;
  function  CheckDevice:Boolean; virtual;
  procedure Action; virtual;
  procedure Watchdog; virtual;
  function  Start:Boolean; virtual;
  procedure Stop; virtual;
  procedure Animate; virtual;
  procedure StartSession; virtual;
  procedure StopSession; virtual;
  function  GetNumCurveLinks(Curve:TCurve):Integer;
  procedure CheckCurveLinks;
  function  GetNameList(aText:TText):TText;
  function  GetDescriptionList(aText:TText):TText;
  function  GetFullList(aText:TText):TText;
  function  SelectDevice(const Caption:LongString):TDaqDevice;
  procedure Config(FileName,Msg:LongString; DeviceConstructor:TDaqDeviceConstructor); virtual;
 end;

function FullDaqDeviceList:TDaqDeviceList;

function  NewDaqDeviceList(aOwnsObjects : Boolean = true;
                           aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                           aStep        : LongInt = DefaultTObjectStorageStep
                                      ) : TDaqDeviceList;
procedure Kill(var TheObject:TDaqDeviceList); overload;

 {
 *******************************************************************************
 Разные утилиты
 *******************************************************************************
 }
procedure ViewTagList;
procedure DefaultDaqCircuitMonitor(Circuit:TFormCircuitWindow);
procedure DefaultDaqCircuitStrategy(Circuit:TFormCircuitWindow; var Event:TCircuitEvent; Sensor:TCircuitSensor);
procedure AnimateCircuits(aStrategy:TCircuitStrategy; aMonitor:TCircuitMonitor);
function  FormatCalibration(const aName:LongString; aCalibration:TPolynomCalibration):LongString;
function  DeviceFamilyPresent(Config,Family,Model:LongString):Boolean;

implementation

uses
 Form_DaqDeviceControl,
 Form_DaqDeviceCommonPropertyDialog;
const
 MaxDim = MaxInt div sizeof(Pointer);

 {
 *******************************************************************************
 SensorEvent implementation
 *******************************************************************************
 }
procedure SensorEvent(out Event:TSensorEvent);
begin
 Event.What:=evNothing;
 Event.Key:=0;
 Event.Shift:=[];
 Event.Where.x:=0;
 Event.Where.y:=0;
 Event.Wheel:=0;
 Event.When:=0;
 Event.Button:=0;
 Event.Sensor:='';
 Event.Tag:=0;
 Event.Wrote:=0;
 Event.Cookies:='';
end;

function SensorEvent:TSensorEvent;
begin
 SensorEvent(Result);
end;

function SensorEvent(Circuit:TFormCircuitWindow; var Event:TCircuitEvent; Sensor:TCircuitSensor):TSensorEvent;
var
 BitMap  : TBitmap;
 x,y     : Integer;
 CookPtr : PChar;
 CookLim : Integer;
 R       : TRect;
 procedure AppendCookies(const Name,Value:LongString);
 var Line:LongString; Leng:Integer;
 begin
  if (Length(Name)>0) and (Length(Value)>0) then begin
   Line:=Name+'='+Value+EOL;
   Leng:=Length(Line);
   if Leng<CookLim then begin
    StrLCopy(CookPtr,PChar(Line),Leng);
    CookPtr:=StrEnd(CookPtr);
    Dec(CookLim,Leng);
   end;
  end;
 end;
begin
 SensorEvent(Result);
 if Event.What<>evNothing then
 try
  CookPtr:=Result.Cookies;
  CookLim:=SizeOf(Result.Cookies);
  Result.What:=Event.What;
  Result.Key:=Event.Key;
  Result.Shift:=Event.Shift;
  Result.Where:=Event.Where;
  Result.Wheel:=Event.Wheel;
  Result.When:=IntMSecNow;
  Result.Button:=ShortCut(Event.Key,Event.Shift);
  Result.Sensor:=Sensor.Name;
  Result.Tag:=Sensor.LinkedTag;
  AppendCookies('TagName',NameTag(Result.Tag));
  AppendCookies('Curve',Sensor.LinkedCurve.Name);
  if Circuit.Ok then AppendCookies('Window',Circuit.Caption);
  if Sensor.LinkedObject is TDaqDevice then AppendCookies('Device',(Sensor.LinkedObject as TDaqDevice).Name);
  with Sensor.Bounds do AppendCookies('Bounds',Format('%d,%d,%d,%d',[A.X,A.Y,B.X,B.Y]));
  AppendCookies('ShortCut',IntToStr(Sensor.ShortCut));
  Result.Wrote:=0;
  if Sensor.Glyph.Ok then
  if Result.What in [evMouseDown..evKeyUp] then
  if RectContainsPoint(Sensor.Bounds,Result.Where) then begin
   Bitmap:=Sensor.Glyph.CreateBitmap;
   if Assigned(Bitmap) then
   try
    x:=Result.Where.x-Sensor.Bounds.a.x;
    y:=Result.Where.y-Sensor.Bounds.a.y;
    if (x>=0) and (x<Bitmap.Width) then
    if (y>=0) and (y<Bitmap.Height) then
    AppendCookies('Pixel',Format('$%6.6x',[Bitmap.Canvas.Pixels[x,y]]));
   finally
    Kill(Bitmap);
   end;
  end;
  case typetag(Sensor.LinkedTag) of
   1:AppendCookies('Value',Format('%d',[igettag(Sensor.LinkedTag)]));
   2:AppendCookies('Value',Format('%g',[rgettag(Sensor.LinkedTag)]));
   3:AppendCookies('Value',Format('%s',[sgettag(Sensor.LinkedTag)]));
   else
   if Sensor.LinkedCurve.Ok
   then AppendCookies('Value',Format('%g',[Sensor.LinkedCurve.LastPoint.Y]));
  end;
  AppendCookies('Bitmap',Sensor.Glyph.Source);
  AppendCookies('Hint',URL_Packed(Sensor.HintText));
  if Sensor.HasTagEval then AppendCookies('TagEval(v)',Trim(Sensor.TagEval));
  if Sensor.HasLedEval then AppendCookies('LedEval(v)',Trim(Sensor.LedEval));
  if Sensor.HasPainter then AppendCookies('Painter(v)',Trim(Sensor.SourceSection));
  AppendCookies('CrcFile',Trim(Sensor.SourceCrcFile));
  AppendCookies('Section',Trim(Sensor.SourceSection));
  AppendCookies('LedValue',Sensor.LedDrawn);
  AppendCookies('BookMark',IntToStr(Sensor.BookMark));
  if Circuit.Ok then begin
   with Circuit do R:=Rect(0,0,Width,Height);
   R.TopLeft:=Circuit.ClientToScreen(R.TopLeft);
   R.BottomRight:=Circuit.ClientToScreen(R.BottomRight);
   AppendCookies('WindowRect',Format('%d,%d,%d,%d',[R.Left,R.Top,R.Right,R.Bottom]));
   with Circuit.PaintBox do R:=Rect(0,0,Width,Height);
   R.TopLeft:=Circuit.PaintBox.ClientToScreen(R.TopLeft);
   R.BottomRight:=Circuit.PaintBox.ClientToScreen(R.BottomRight);
   AppendCookies('PaintBoxRect',Format('%d,%d,%d,%d',[R.Left,R.Top,R.Right,R.Bottom]));
  end;
  AppendCookies('AppFormBounds',GetAppFormBoundsStr);
  AppendCookies('AppClientBounds',GetAppClientBoundsStr);
 except
  on E:Exception do BugReport(E,nil,'SensorEvent');
 end;
end;

type ESensorEventFormat=class(Exception);

function SensorEvent(const Lines:LongString):TSensorEvent;
var p:TText; key,s,w:LongString; i,n:Integer; Cookies:LongString;
 function ExtractVar(p:TText; const aName:LongString):LongString;
 var i,n:Integer;
 begin
  Result:=p.GetVar(aName);
  for n:=0 to p.Count-1 do begin i:=p.FindVar(aName); if i>=0 then p.DelLn(i) else break; end;
 end;
 function DropGarbage(p:TText):LongString;
 var i,k:Integer; n,v:LongString;
 begin
  for i:=p.Count-1 downto 0 do begin
   k:=ExtractNameValuePair(p[i],n,v);
   if (k=0) or (n='') or (v='') then p.DelLn(i);
  end;
  Result:=p.Text;
 end;
 procedure Failure(key,s:LongString);
 begin
  Raise ESensorEventFormat.Create('SensorEvent: Bad '+key+' = '+s);
 end;
begin
 SensorEvent(Result);
 try
  p:=NewText;
  Cookies:='';
  try
   p.Text:=Lines;
   //
   key:='What';
   s:=Trim(ExtractVar(p,key));
   if not Str2Int(s,n) then n:=WordIndex(UpCaseStr(s),EventWhatNameList,ScanSpaces)-1;
   if (n<Ord(Low(TEventWhat))) or (n>Ord(High(TEventWhat))) then Failure(key,s);
   Result.What:=TEventWhat(n);
   //
   key:='Button';
   s:=Trim(ExtractVar(p,key));
   if Str2Int(s,n) then Result.Button:=n else Failure(key,s);
   ShortCutToKey(Result.Button,Result.Key,Result.Shift);
   //
   key:='Key';
   s:=Trim(ExtractVar(p,key));
   if Str2Int(s,n) then Result.Key:=n;
   //
   key:='Shift';
   s:=Trim(ExtractVar(p,key));
   for i:=1 to WordCount(s,ScanSpaces) do begin
    w:=ExtractWord(i,s,ScanSpaces);
    if SameText(w,'SHIFT')  then Include(Result.Shift,ssShift)  else
    if SameText(w,'ALT')    then Include(Result.Shift,ssAlt)    else
    if SameText(w,'CTRL')   then Include(Result.Shift,ssCtrl)   else
    if SameText(w,'LEFT')   then Include(Result.Shift,ssLeft)   else
    if SameText(w,'RIGHT')  then Include(Result.Shift,ssRight)  else
    if SameText(w,'MIDDLE') then Include(Result.Shift,ssMiddle) else
    if SameText(w,'DOUBLE') then Include(Result.Shift,ssDouble);
   end;
   //
   key:='Where';
   s:=Trim(ExtractVar(p,key));
   if Str2Int(ExtractWord(1,s,ScanSpaces),n) then Result.Where.x:=n else Failure(key,s);
   if Str2Int(ExtractWord(2,s,ScanSpaces),n) then Result.Where.y:=n else Failure(key,s);
   //
   key:='Wheel';
   s:=Trim(ExtractVar(p,key));
   Result.Wheel:=StrToIntDef(s,0);
   //
   key:='When';
   s:=Trim(ExtractVar(p,key));
   Result.When:=StrToInt64Def(s,0);
   //
   key:='Sensor';
   s:=Trim(ExtractVar(p,key));
   Result.Sensor:=s;
   //
   key:='Tag';
   s:=Trim(ExtractVar(p,key));
   Result.Tag:=FindTag(s);
   //
   key:='Wrote';
   s:=Trim(ExtractVar(p,key));
   if Str2Int(s,n) then Result.Wrote:=n;
   Inc(Result.Wrote);
   //
   key:='Cookies';
   Cookies:=DropGarbage(p);
   StrLCopy(Result.Cookies,PChar(Cookies),SizeOf(Result.Cookies)-1);
  finally
   Kill(p);
   Cookies:='';
  end;
 except
  on ESensorEventFormat do SensorEvent(Result);
  on E:Exception do BugReport(E,nil,'SensorEvent');
 end;
end;

function SensorEventParams(Event:TSensorEvent;const Name:LongString):LongString;
var Ident:LongString;
begin
 Result:='';
 try
  Ident:=SysUtils.Trim(Name);
  if Ident='' then begin
   Result:=SysUtils.Trim('What='+ExtractWord(1+Ord(Event.What),EventWhatNameList,ScanSpaces)+EOL+
                         'Key='+IntToStr(Event.Key)+EOL+
                         'Shift='+SensorEventParams(Event,'Shift')+EOL+
                         'Where='+IntToStr(Event.Where.X)+','+IntToStr(Event.Where.Y)+EOL+
                         'Wheel='+IntToStr(Event.Wheel)+EOL+
                         'When='+IntToStr(Event.When)+EOL+
                         'Button='+IntToStr(Event.Button)+EOL+
                         'Sensor='+Event.Sensor+EOL+
                         'Tag='+NameTag(Event.Tag)+EOL+
                         'Wrote='+IntToStr(Event.Wrote)+EOL+
                         Event.Cookies);
   Exit;
  end;
  if SameText(Ident,'What')  then Result:=ExtractWord(1+Ord(Event.What),EventWhatNameList,ScanSpaces) else
  if SameText(Ident,'Key')  then Result:=IntToStr(Event.Key) else
  if SameText(Ident,'Shift')  then begin
   if ssShift  in Event.Shift then Result:=Result+',SHIFT';
   if ssAlt    in Event.Shift then Result:=Result+',ALT';
   if ssCtrl   in Event.Shift then Result:=Result+',CTRL';
   if ssLeft   in Event.Shift then Result:=Result+',LEFT';
   if ssRight  in Event.Shift then Result:=Result+',RIGHT';
   if ssMiddle in Event.Shift then Result:=Result+',MIDDLE';
   if ssDouble in Event.Shift then Result:=Result+',DOUBLE';
   if Pos(',',Result)=1 then Result:=Copy(Result,2,255);
  end else
  if SameText(Ident,'Where')     then Result:=IntToStr(Event.Where.X)+','+IntToStr(Event.Where.Y) else
  if SameText(Ident,'Wheel')     then Result:=IntToStr(Event.Wheel) else
  if SameText(Ident,'When')      then Result:=IntToStr(Event.When) else
  if SameText(Ident,'Button')    then Result:=IntToStr(Event.Button) else
  if SameText(Ident,'Sensor')    then Result:=Event.Sensor else
  if SameText(Ident,'Tag')       then Result:=NameTag(Event.Tag) else
  if SameText(Ident,'Wrote')     then Result:=IntToStr(Event.Wrote) else
  if Event.Cookies[0]<>ASCII_NUL then Result:=CookieScan(Event.Cookies,SizeOf(Event.Cookies),Ident);
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 *******************************************************************************
 TDaqDevice implementation
 *******************************************************************************
 }
constructor TDaqDevice.Create(const aName:LongString);
begin
 inherited Create;
 myName:=Trim(aName);
 myFamily:='ABSTRACT';
 myModel:='ABSTRACT';
 myComment:='';
 myInquiryTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(55,1,nil));
 myInquiryTimer.Master:=@myInquiryTimer;
 myAnalogFifo:=nil;
 myDigitalFifo:=nil;
 myDigitalFirst:=false;
 myErrorsTotal:=0;
 SafeFillChar(myErrorsCount,sizeof(myErrorsCount),0);
 myGroupTag:=0;
 myStartingOrder:=0;
 myStoppingOrder:=0;
 myDispatcherFilter:=afDefault;
 myAnalogInputCurves:=NewObjectStorage(false);
 myAnalogInputSmooth:=NewObjectStorage(true);
 myDigitalInputCurves:=NewObjectStorage(false);
 myDigitalInputBits:=NewLongIntVector(0);
 myDigitalInputBits.Exceptions:=false;
 myAnalogOutputCurves:=NewObjectStorage(false);
 myPrimaryAOs:=0;
 mySecondaryAOs:=0;
 myAnalogOutputCompressors:=NewObjectStorage(true);
 myAnalogOutputHistoryLens:=NewLongIntVector(0);
 myAnalogOutputHistoryLens.Exceptions:=false;
 myDigitalOutputCurves:=NewObjectStorage(false);
 myDigitalOutputHistoryLens:=NewLongIntVector(0);
 myDigitalOutputHistoryLens.Exceptions:=false;
 myCalibrations:=NewObjectStorage(true);
 myPropertyDialog:=nil;
 myCommonPropertyDialog:=nil;
 mySensorClicked:=NewFifo(SizeOf(TSensorEvent)*TDaqDeviceSensorEventsFifoLength);
 mySensorClicked.Master:=@mySensorClicked;
 mySensorClicked.GrowLimit:=1024*256;
 mySensorClicked.GrowFactor:=2;
 myClickFilter:=cf_Default;
 myClickAwaker:=cf_Default;
end;

procedure TDaqDevice.AfterConstruction;
begin
 inherited AfterConstruction;
 FullDaqDeviceList.Add(Self);
end;

procedure TDaqDevice.BeforeDestruction;
begin
 FullDaqDeviceList.Remove(Self);
 inherited BeforeDestruction;
end;

destructor  TDaqDevice.Destroy;
begin
 Kill(mySensorClicked);
 Kill(TObject(myCommonPropertyDialog));
 Kill(TObject(myPropertyDialog));
 KillFormCalibDialog;
 Kill(myCalibrations);
 Kill(myDigitalOutputHistoryLens);
 Kill(myDigitalOutputCurves);
 Kill(myAnalogOutputHistoryLens);
 Kill(myAnalogOutputCompressors);
 mySecondaryAOs:=0;
 myPrimaryAOs:=0;
 Kill(myAnalogOutputCurves);
 Kill(myDigitalInputBits);
 Kill(myDigitalInputCurves);
 Kill(myAnalogInputSmooth);
 Kill(myAnalogInputCurves);
 myGroupTag:=0;
 myStartingOrder:=0;
 myStoppingOrder:=0;
 myDispatcherFilter:=afDefault;
 SafeFillChar(myErrorsCount,sizeof(myErrorsCount),0);
 myErrorsTotal:=0;
 Kill(myDigitalFifo);
 Kill(myAnalogFifo);
 Kill(myInquiryTimer);
 myComment:='';
 myModel:='';
 myFamily:='';
 myName:='';
 mySysLogSign:='';
 inherited Destroy;
end;

function TDaqDevice.GetName:LongString;
begin
 if Assigned(Self) then Result:=myName else Result:='';
end;

function TDaqDevice.GetFamily:LongString;
begin
 if Assigned(Self) then Result:=myFamily else Result:='';
end;

procedure TDaqDevice.SetDeviceFamily(aFamily:LongString);
begin
 if Assigned(Self) then myFamily:=Trim(aFamily);
end;

function TDaqDevice.GetModel:LongString;
begin
 if Assigned(Self) then Result:=myModel else Result:='';
end;

procedure TDaqDevice.SetDeviceModel(aModel:LongString);
begin
 if Assigned(Self) then myModel:=Trim(aModel);
end;

function TDaqDevice.GetDeviceDescription:LongString;
begin
 if Assigned(Self)
 then Result:=Name+' = DEVICE '+Family+' '+Model
 else Result:='';
end;

function TDaqDevice.GetDevSection:LongString;
begin
 if Assigned(Self)
 then Result:='['+Name+']'
 else Result:='';
end;

function  TDaqDevice.GetComment:LongString;
begin
 if Assigned(Self) then Result:=myComment else Result:='';
end;

procedure TDaqDevice.SetComment(const aComment:LongString);
begin
 if Assigned(Self) then myComment:=aComment;
end;

function TDaqDevice.GetInquiryTimer:TIntervalTimer;
begin
 if Assigned(Self)
 then Result:=myInquiryTimer
 else Result:=nil;
end;

function TDaqDevice.GetAnalogFifoSize:Integer;
begin
 if Assigned(Self)
 then Result:=myAnalogFifo.Size div sizeof(TDaqEvent)
 else Result:=0;
end;

procedure TDaqDevice.SetAnalogFifoSize(aSize:Integer);
begin
 if Assigned(Self) then begin
  Kill(myAnalogFifo);
  if aSize>0 then myAnalogFifo:=NewFifo(aSize*SizeOf(TDaqEvent));
 end;
end;

function TDaqDevice.GetDigitalFifoSize:Integer;
begin
 if Assigned(Self)
 then Result:=myDigitalFifo.Size div sizeof(TDaqEvent)
 else Result:=0;
end;

procedure TDaqDevice.SetDigitalFifoSize(aSize:Integer);
begin
 if Assigned(Self) then begin
  Kill(myDigitalFifo);
  if aSize>0 then myDigitalFifo:=NewFifo(aSize*SizeOf(TDaqEvent));
 end;
end;

procedure TDaqDevice.ClearAnalogFifo;
begin
 if Assigned(Self) then  myAnalogFifo.Clear;
end;

procedure TDaqDevice.ClearDigitalFifo;
begin
 if Assigned(Self) then  myDigitalFifo.Clear;
end;

function TDaqDevice.GetErrorsTotal:Integer;
begin
 if Assigned(Self) then Result:=myErrorsTotal else Result:=0;
end;

function TDaqDevice.GetErrorsCount(ErrorCode:Byte):Integer;
begin
 if Assigned(Self) then Result:=myErrorsCount[ErrorCode] else Result:=0;
end;

procedure TDaqDevice.FixError(ErrorCode:Byte);
begin
 if Assigned(Self) then begin
  inc(myErrorsTotal);
  inc(myErrorsCount[ErrorCode]);
 end;
end;

procedure TDaqDevice.ClearErrors;
begin
 if Assigned(Self) then begin
  myErrorsTotal:=0;
  SafeFillChar(myErrorsCount,sizeof(myErrorsCount),0);
 end;
end;

function TDaqDevice.GetGroupTag:Integer;
begin
 if Assigned(Self) then Result:=myGroupTag else Result:=0;
end;

procedure TDaqDevice.SetGroupTag(aGroupTag:Integer);
begin
 if Assigned(Self) then myGroupTag:=aGroupTag;
end;

function TDaqDevice.GetStartingOrder:Integer;
begin
 if Assigned(Self) then Result:=myStartingOrder else Result:=0;
end;

procedure TDaqDevice.SetStartingOrder(aStartingOrder:Integer);
begin
 if Assigned(Self) then myStartingOrder:=aStartingOrder;
end;

function TDaqDevice.GetStoppingOrder:Integer;
begin
 if Assigned(Self) then Result:=myStoppingOrder else Result:=0;
end;

procedure TDaqDevice.SetStoppingOrder(aStoppingOrder:Integer);
begin
 if Assigned(Self) then myStoppingOrder:=aStoppingOrder;
end;

function TDaqDevice.GetDispatcherFilter:Integer;
begin
 if Assigned(Self) then Result:=myDispatcherFilter else Result:=0;
end;

procedure TDaqDevice.SetDispatcherFilter(aDispatcherFilter:Integer);
begin
 if Assigned(Self) then myDispatcherFilter:=aDispatcherFilter;
end;

function TDaqDevice.GetDaqEvent(out Event:TDaqEvent):Boolean;
begin
 if Assigned(Self) then begin
  if myDigitalFirst then begin
   Result:=myDigitalFifo.Get(@Event,sizeof(Event))=sizeof(Event);
   if not Result then
   Result:=myAnalogFifo.Get(@Event,sizeof(Event))=sizeof(Event);
  end else begin
   Result:=myAnalogFifo.Get(@Event,sizeof(Event))=sizeof(Event);
   if not Result then
   Result:=myDigitalFifo.Get(@Event,sizeof(Event))=sizeof(Event);
  end;
  myDigitalFirst:=not myDigitalFirst;
 end else begin
  Result:=false;
  SafeFillChar(Event, sizeof(Event), 0);
 end;
end;

function TDaqDevice.PutDaqEvent(const Event:TDaqEvent):Boolean;
begin
 Result:=false;
 if Assigned(Self) then begin
  if Event.What and evDigital <> 0 then begin
   if myDigitalFifo.Put(@Event,sizeof(Event))=sizeof(Event)
   then Result:=true
   else FixError(ecDigitalFifoLost);
  end else begin
   if myAnalogFifo.Put(@Event,sizeof(Event))=sizeof(Event)
   then Result:=true
   else FixError(ecAnalogFifoLost);
  end;
 end; 
end;

 {
 Диспетчер событий
 }
procedure TDaqDevice.DispatchDaqEvent(var Event:TDaqEvent);
var
 Curve      : TCurve;
 HistLen    : Integer;
 Compressor : TDaqCompressor;
 Chan       : Integer;
 P          : TPoint2D;
begin
 if Assigned(Self) then begin
  {
  Неспектрометрические события обрабатываются здесь по такому алгоритму:
  Берем кривую номер Event.Chan, проверяем что она динамическая (не спектр!).
  При необходимости компрессируем аналоговые данные.
  Добавляем точку (Event.Time,Event.Data[0]), если изменилось время
  или событие отмечено как "важное".
  Проверяем длину истории и ограничиваем ее если надо.
  }
  if Event.What and evSpectral = evNoSpectral then begin
   if Event.What and evDigital <> 0 then begin
    Curve:=DigitalOutputCurve[Event.Chan];
    if Assigned(Curve) and Curve.IsDynamic then
    try
     Curve.Lock;
     if (Curve.Count=0) or (Event.Time>Curve.LastPoint.X) or
        (Event.What and evImportant<>0) or (myDispatcherFilter and afIgnoreLEX<>afIgnoreLEX)
     then begin
      Curve.SmartAddPoint(Event.Time, Event.Data[0], myDispatcherFilter);
      HistLen:=DigitalOutputHistoryLen[Event.Chan];
      if (HistLen>0) and (Curve.Count>HistLen shl 1)
      then Curve.ClearHistory(HistLen);
     end;
    finally
     Curve.Unlock;
    end else FixError(ecDigitalDispatch);
   end else begin
    Curve:=AnalogOutputCurve[Event.Chan];
    if Assigned(Curve) and Curve.IsDynamic then
    try
     Curve.Lock;
     if (Curve.Count=0) or (Event.Time>Curve.LastPoint.X) or
        (Event.What and evImportant<>0) or (myDispatcherFilter and afIgnoreLEX<>afIgnoreLEX)
     then begin
      if Event.What and evCompress<>0 then begin
       Compressor:=AnalogOutputCompressor[Event.Chan];
       if Assigned(Compressor)
       then Event.Data[0]:=Compressor.Compress(Event.Data[0]);
      end;
      Curve.SmartAddPoint(Event.Time, Event.Data[0], myDispatcherFilter);
      HistLen:=AnalogOutputHistoryLen[Event.Chan];
      if (HistLen>0) and (Curve.Count>HistLen shl 1)
      then Curve.ClearHistory(HistLen);
     end;
    finally
     Curve.Unlock;
    end else FixError(ecAnalogDispatch);
   end;
  end;
  {
  Спектрометрические события обрабатываются здесь по такому алгоритму:
  Берем кривую номер Event.Chan, проверяем, что она статическая (спектр!).
  Определяем номер канала Event.Data[0], если он допустим, то приращение
  этого канала на величину Event.Data[1]
  }
  if Event.What and evSpectral = evSpectral then begin
   if Event.What and evDigital <> 0 then begin
    Curve:=DigitalOutputCurve[Event.Chan];
    if Assigned(Curve) and Curve.IsStatic then
    try
     Curve.Lock;
     Chan:=round(Event.Data[0]);
     if Cardinal(Chan)<Cardinal(Curve.Count) then begin
      P:=Curve[Chan];
      P.Y:=P.Y+Event.Data[1];
      Curve[Chan]:=P;
     end else FixError(ecSpectrChan);
    finally
     Curve.Unlock;
    end else FixError(ecDigitalDispatch);
   end else begin
    Curve:=AnalogOutputCurve[Event.Chan];
    if Assigned(Curve) and Curve.IsStatic then
    try
     Curve.Lock;
     Chan:=round(Event.Data[0]);
     if Cardinal(Chan)<Cardinal(Curve.Count) then begin
      P:=Curve[Chan];
      P.Y:=P.Y+Event.Data[1];
      Curve[Chan]:=P;
     end else FixError(ecSpectrChan);
    finally
     Curve.Unlock;
    end else FixError(ecAnalogDispatch);
   end;
  end;
 end;
end;

procedure TDaqDevice.DispatchEvents;
var Event:TDaqEvent;
begin
 if Assigned(Self) then while GetDaqEvent(Event) do DispatchDaqEvent(Event);
end;

function TDaqDevice.GetNumAnalogInputs:Integer;
begin
 if Assigned(Self)
 then Result:=myAnalogInputCurves.Count
 else Result:=0;
end;

procedure TDaqDevice.SetNumAnalogInputs(aCount:Integer);
var
 i : Integer;
begin
 if Assigned(Self) then begin
  if (aCount<=0) or (aCount>MaxDim) then aCount:=0;
  myAnalogInputCurves.Count:=aCount;
  myAnalogInputSmooth.Count:=aCount;
  for i:=0 to aCount-1 do begin
   AnalogInputCurve[i]:=nil;
   AnalogInputSmoother[i]:=nil;
  end;
 end;
end;

function TDaqDevice.GetAnalogInputCurve(Index:Integer):TCurve;
begin
 if Assigned(Self)
 then Result:=TCurve(myAnalogInputCurves[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetAnalogInputCurve(Index:Integer; aCurve:TCurve);
begin
 if Assigned(Self) then myAnalogInputCurves[Index]:=aCurve;
end;

procedure TDaqDevice.ClearAnalogInputCurve(Index:Integer);
begin
 DaqClearCurve(AnalogInputCurve[Index],0);
end;

function TDaqDevice.GetAnalogInputSmoother(Index:Integer):TDaqSmoother;
begin
 if Assigned(Self)
 then Result:=TDaqSmoother(myAnalogInputSmooth[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetAnalogInputSmoother(Index:Integer; aSmoother:TDaqSmoother);
begin
 if Assigned(Self) then myAnalogInputSmooth[Index]:=aSmoother;
end;

function TDaqDevice.SmoothAnalogInputCurve(Index:Integer; Where:Double):Double;
var
 P        : TPoint2D;
 Curve    : TCurve;
 Smoother : TDaqSmoother;
begin
 Curve:=AnalogInputCurve[Index];
 if Curve.Ok then begin
  Smoother:=AnalogInputSmoother[Index];
  if Smoother.Ok
  then Result:=Smoother.Smooth(Curve,Where)
  else if Curve.Count>0 then begin
   P:=Curve.LastPoint;
   if Where>=P.X then Result:=P.Y else Result:=Curve.Interpolate(Where);
  end else Result:=0;
 end else Result:=0;
end;

function TDaqDevice.ConstructAnalogInputMap(ofs,n:Integer):Integer;
var
 i : Integer;
begin
 Result:=0;
 if Assigned(Self) then
 for i:=0 to n-1 do
 if AnalogInputCurve[i+ofs].Ok then Result:=Result or GetBitMask(i);
end;

function TDaqDevice.GetNumDigitalInputs:Integer;
begin
 if Assigned(Self)
 then Result:=myDigitalInputCurves.Count
 else Result:=0;
end;

procedure TDaqDevice.SetNumDigitalInputs(aCount:Integer);
var
 i : Integer;
begin
 if Assigned(Self) then begin
  if (aCount<=0) or (aCount>MaxDim) then aCount:=0;
  myDigitalInputCurves.Count:=aCount;
  myDigitalInputBits.Length:=aCount;
  for i:=0 to aCount-1 do begin
   DigitalInputCurve[i]:=nil;
   SetDigitalInputBit(i, 0, false);
  end;
 end;
end;

function TDaqDevice.GetDigitalInputCurve(Index:Integer):TCurve;
begin
 if Assigned(Self)
 then Result:=TCurve(myDigitalInputCurves[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetDigitalInputCurve(Index:Integer; aCurve:TCurve);
begin
 if Assigned(Self) then myDigitalInputCurves[Index]:=aCurve;
end;

procedure TDaqDevice.ClearDigitalInputCurve(Index:Integer);
begin
 DaqClearCurve(DigitalInputCurve[Index],0);
end;

const
 InvFlag = Integer($80000000);

function TDaqDevice.GetDigitalInputBit(Index:Integer; out Inversion:Boolean):Integer;
begin
 if Assigned(Self) then begin
  Result:=myDigitalInputBits[Index] and not InvFlag;
  Inversion:=myDigitalInputBits[Index] and InvFlag <> 0;
 end else begin
  Result:=0;
  Inversion:=false;
 end;
end;

procedure TDaqDevice.SetDigitalInputBit(Index:Integer; aBit:Integer; aInversion:Boolean);
begin
 if Assigned(Self) then begin
  aBit:=aBit and not InvFlag;
  if aInversion then aBit:=aBit or InvFlag;
  myDigitalInputBits[Index]:=aBit;
 end;
end;

function TDaqDevice.ConstructDigitalInputWord(ofs,n:Integer):Integer;
var
 i     : Integer;
 aBit  : Integer;
 aInv  : Boolean;
 Curve : TCurve;
begin
 Result:=0;
 if Assigned(Self) then
 for i:=0 to n-1 do begin
  Curve:=DigitalInputCurve[i+ofs];
  if Curve.Ok then
  if Curve.Count>0 then begin
   aBit:=GetDigitalInputBit(i+ofs, aInv);
   if aInv then begin
    if Round(Curve.LastPoint.Y) and GetBitMask(aBit) = 0
    then Result:=Result or GetBitMask(i);
   end else begin
    if Round(Curve.LastPoint.Y) and GetBitMask(aBit)<> 0
    then Result:=Result or GetBitMask(i);
   end;
  end;
 end;
end;

function TDaqDevice.ConstructDigitalInputMap(ofs,n:Integer):Integer;
var
 i : Integer;
begin
 Result:=0;
 if Assigned(Self) then
 for i:=0 to n-1 do
 if DigitalInputCurve[i+ofs].Ok then Result:=Result or GetBitMask(i);
end;

function TDaqDevice.GetNumAnalogOutputs:Integer;
begin
 if Assigned(Self)
 then Result:=myAnalogOutputCurves.Count
 else Result:=0;
end;

function TDaqDevice.GetNumPrimaryAnalogOutputs:Integer;
begin
 if Assigned(Self)
 then Result:=myPrimaryAOs
 else Result:=0;
end;

function TDaqDevice.GetNumSecondaryAnalogOutputs:Integer;
begin
 if Assigned(Self)
 then Result:=mySecondaryAOs
 else Result:=0;
end;

procedure TDaqDevice.SetNumAnalogOutputs(aCount:Integer);
begin
 SetNumAnalogOutputsMux(aCount,1);
end;

procedure TDaqDevice.SetNumAnalogOutputsMux(nPrimary,nSecondary:Integer);
var
 i : Integer;
begin
 if Assigned(Self) then begin
  if (nPrimary<=0) or (nPrimary>MaxDim) then nPrimary:=0;
  if (nSecondary<=0) or (nSecondary>MaxDim) then nSecondary:=0;
  if (nPrimary*nSecondary<=0) or (nPrimary*nSecondary>MaxDim) then begin
   nPrimary:=0;
   nSecondary:=0;
  end;
  myAnalogOutputCurves.Count:=nPrimary*nSecondary;
  myPrimaryAOs:=nPrimary;
  mySecondaryAOs:=nSecondary;
  myAnalogOutputCompressors.Count:=nPrimary*nSecondary;
  myAnalogOutputHistoryLens.Length:=nPrimary*nSecondary;
  for i:=0 to nPrimary*nSecondary-1 do begin
   AnalogOutputCurve[i]:=nil;
   AnalogOutputCompressor[i]:=nil;
   AnalogOutputHistoryLen[i]:=0;
  end;
 end;
end;

function TDaqDevice.GetAnalogOutputCurve(Index:Integer):TCurve;
begin
 if Assigned(Self)
 then Result:=TCurve(myAnalogOutputCurves[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetAnalogOutputCurve(Index:Integer; aCurve:TCurve);
begin
 if Assigned(Self) then myAnalogOutputCurves[Index]:=aCurve;
end;

procedure TDaqDevice.ClearAnalogOutputCurve(Index:word);
begin
 DaqClearCurve(AnalogOutputCurve[Index],0);
end;

function TDaqDevice.GetAnalogOutputCompressor(Index:Integer):TDaqCompressor;
begin
 if Assigned(Self)
 then Result:=TDaqCompressor(myAnalogOutputCompressors[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetAnalogOutputCompressor(Index:Integer; aCompressor:TDaqCompressor);
begin
 if Assigned(Self) then myAnalogOutputCompressors[Index]:=aCompressor;
end;

function TDaqDevice.GetAnalogOutputHistoryLen(Index:Integer):Integer;
begin
 if Assigned(Self)
 then Result:=myAnalogOutputHistoryLens[Index]
 else Result:=0;
end;

procedure TDaqDevice.SetAnalogOutputHistoryLen(Index:Integer; aHistoryLen:Integer);
begin
 if Assigned(Self) then myAnalogOutputHistoryLens[Index]:=aHistoryLen;
end;

function TDaqDevice.PackAnalogOutputIndex(iPrimary,iSecondary:Integer):Integer;
begin
 if Assigned(Self)
 then Result:=iPrimary+iSecondary*myPrimaryAOs
 else Result:=0;
end;

procedure TDaqDevice.UnpackAnalogOutputIndex(LinearIndex:Integer; out iPrimary,iSecondary:Integer);
begin
 if Assigned(Self) and (myPrimaryAOs>0) then begin
  iPrimary:=LinearIndex mod myPrimaryAOs;
  iSecondary:=LinearIndex div myPrimaryAOs;
 end else begin
  iPrimary:=0;
  iSecondary:=0;
 end;
end;

function TDaqDevice.ConstructAnalogOutputMap(ofs,n:Integer):Integer;
var
 i : Integer;
begin
 Result:=0;
 if Assigned(Self) then
 for i:=0 to n-1 do
 if AnalogOutputCurve[i+ofs].Ok then Result:=Result or GetBitMask(i);
end;

function TDaqDevice.GetNumDigitalOutputs:Integer;
begin
 if Assigned(Self)
 then Result:=myDigitalOutputCurves.Count
 else Result:=0;
end;

procedure TDaqDevice.SetNumDigitalOutputs(aCount:Integer);
var
 i : Integer;
begin
 if Assigned(Self) then begin
  if (aCount<=0) or (aCount>MaxDim) then aCount:=0;
  myDigitalOutputCurves.Count:=aCount;
  myDigitalOutputHistoryLens.Length:=aCount;
  for i:=0 to aCount-1 do begin
   DigitalOutputCurve[i]:=nil;
  end;
 end;
end;

function TDaqDevice.GetDigitalOutputCurve(Index:Integer):TCurve;
begin
 if Assigned(Self)
 then Result:=TCurve(myDigitalOutputCurves[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetDigitalOutputCurve(Index:Integer; aCurve:TCurve);
begin
 if Assigned(Self) then myDigitalOutputCurves[Index]:=aCurve;
end;

procedure TDaqDevice.ClearDigitalOutputCurve(Index:Integer);
begin
 DaqClearCurve(DigitalOutputCurve[Index],0);
end;

function TDaqDevice.GetDigitalOutputHistoryLen(Index:Integer):Integer;
begin
 if Assigned(Self)
 then Result:=myDigitalOutputHistoryLens[Index]
 else Result:=0;
end;

procedure TDaqDevice.SetDigitalOutputHistoryLen(Index:Integer; aHistoryLen:Integer);
begin
 if Assigned(Self) then myDigitalOutputHistoryLens[Index]:=aHistoryLen;
end;

function TDaqDevice.ConstructDigitalOutputMap(ofs,n:Integer):Integer;
var
 i : Integer;
begin
 Result:=0;
 if Assigned(Self) then
 for i:=0 to n-1 do
 if DigitalOutputCurve[i+ofs].Ok then Result:=Result or GetBitMask(i);
end;

function TDaqDevice.GetNumCalibrations:Integer;
begin
 if Assigned(Self)
 then Result:=myCalibrations.Count
 else Result:=0;
end;

procedure TDaqDevice.SetNumCalibrations(aCount:Integer);
var
 i : Integer;
begin
 if Assigned(Self) then begin
  if (aCount<0) or (aCount>MaxDim) then aCount:=0;
  myCalibrations.Count:=aCount;
  for i:=0 to aCount-1 do begin
   Calibration[i]:=nil;
  end;
 end;
end;

function TDaqDevice.GetCalibration(Index:Integer):TPolynomCalibration;
begin
 if Assigned(Self)
 then Result:=TPolynomCalibration(myCalibrations[Index])
 else Result:=nil;
end;

procedure TDaqDevice.SetCalibration(Index:Integer; aCalibration:TPolynomCalibration);
begin
 if Assigned(Self) then myCalibrations[Index]:=aCalibration;
end;

procedure TDaqDevice.InitCalibration(Index:Integer);
begin
 if Assigned(Self) and (Cardinal(Index)<Cardinal(NumCalibrations))
 then Calibration[Index]:=NewCalibration('X', 'Y', '', 'Line', 'Line', 1, 0, 1, 0, 1000);
end;

function TDaqDevice.Transform(Index:Integer; Argument,Parametr:Double):Double;
var
 aCalibration : TPolynomCalibration;
begin
 aCalibration:=Calibration[Index];
 if aCalibration.Ok 
 then Result:=aCalibration.GetY(Argument,Parametr)
 else Result:=Argument;
end;

function TDaqDevice.ReadCalibration(Index:Integer; FileName,SectionName,CalibName:LongString):Boolean;
var
 s       : LongString;
 Section : LongString;
 Temp    : TParsingBuffer;
 Data    : packed record
  fName  : PureString;
  xName  : PureString;
  yName  : PureString;
  zName  : PureString;
  xScale : PureString;
  yScale : PureString;
  BoundA : Double;
  BoundB : Double;
 end;
 {проверка имени-если '*', то имя не задано}
 procedure CheckName(var s:PureString);
 begin
  if (s='-') or (s='*') then s:='';
 end;
begin
 Result:=false;
 if Assigned(Self) then
 if InRange(Index,0,NumCalibrations-1) then begin
  {занулим все данные калибровки}
  SafeFillChar(Data,sizeof(Data),0);
  {
  прочитаем секцию и в ней строку в формате
  name = file xname yname zname xscale yscale a b
  }
  FileName:=UnifyFileAlias(FileName);
  Section:=ExtractTextSection(FileName,SectionName,efConfig and not efCaseMask);
  s:='';
  ScanVarString(svConfig,PChar(Section),CalibName+'%s',s);
  ScanVarRecord(svAsIs,StrCopyBuff(Temp,s),'%a;%a;%a;%a;%a;%a;%f;%f',Data);
  Section:='';
  {проверка имен - если '*', то не задано}
  CheckName(Data.xName);
  CheckName(Data.yName);
  CheckName(Data.zName);
  CheckName(Data.xScale);
  CheckName(Data.yScale);
  {строка калибровки прочитана?}
  if IsNonEmptyStr(Data.fName) then begin
   {коррекция имени файла}
   Data.fName:=Daq.FileRef(Data.fName,'.cal',FileName);
   {проверим наличие калибровки}
   if Calibration[Index]=nil then InitCalibration(Index);
   {если есть дополнительные данные, скорректируем поля калибровки}
   if Calibration[Index]<>nil then begin
    if IsNonEmptyStr(Data.xName)  then Calibration[Index].NameX:=Data.xName;
    if IsNonEmptyStr(Data.yName)  then Calibration[Index].NameY:=Data.yName;
    if IsNonEmptyStr(Data.zName)  then Calibration[Index].NameZ:=Data.zName;
    if IsNonEmptyStr(Data.xScale) then Calibration[Index].TransformX:=Data.xScale;
    if IsNonEmptyStr(Data.yScale) then Calibration[Index].TransformY:=Data.yScale;
    if Data.BoundB>Data.BoundA then begin
     Calibration[Index].BoundA:=Data.BoundA;
     Calibration[Index].BoundB:=Data.BoundB;
    end;
    {если задано корректное имя файла, читаем калибровку}
    if not IsWildCard(Data.fName) then begin
     if FileExists(Data.fName) then begin
      if Calibration[Index].LoadFromFile(Data.fName)
      then Result:=true
      else Daq.AddWarning('['+Name+'] Invalid -> '+Data.fName);
     end else Daq.AddWarning('['+Name+'] Not found -> '+Data.fName);
    end;
   end;
  end;
 end else Daq.AddWarning('['+Name+'] Invalid calibration index -> '+d2s(Index));
end;

procedure TDaqDevice.EditCalibration(Index:Integer);
begin
 if Assigned(Self) then
 if NoProblem(NumCalibrations>0,RusEng('Устройство '+Name+' не имеет калибровок!',
                                       'Device '+Name+' has no calibratins!'))
 then
 if NoProblem(Cardinal(Index)<Cardinal(NumCalibrations), RusEng('Неверный индекс калибровки!',
                                                                'Invalid calibration index'))
 then begin
  KillFormCalibDialog;
  if Calibration[Index]=nil then
  if YesNo(RusEng('Калибровка не инициирована!'+EOL+'Создать новую?',
                  'Calibration does not exists!'+EOL+'Create new one?'))=mrYes
  then begin
   InitCalibration(Index);
   NoProblem(Calibration[Index].Ok,RusEng('Не могу создать калибровку!','Could not create calibration!'));
  end;
  if Calibration[Index].Ok then OpenFormCalibDialog(Calibration[Index],false);
 end;
end;

function TDaqDevice.GetClickFilter:Cardinal;
begin
 if Assigned(Self) then Result:=myClickFilter else Result:=0;
end;

procedure TDaqDevice.SetClickFilter(aClickFilter:Cardinal);
begin
 if Assigned(Self) then myClickFilter:=aClickFilter;
end;

function TDaqDevice.SuitableClick(What:TEventWhat):Boolean;
begin
 if Assigned(Self) and (What<>evNothing)
 then Result:= (myClickFilter and (1 shl Ord(What)) <> 0)
 else Result:=false;
end;

function TDaqDevice.GetClickAwaker:Cardinal;
begin
 if Assigned(Self) then Result:=myClickAwaker else Result:=0;
end;

procedure TDaqDevice.SetClickAwaker(aClickAwaker:Cardinal);
begin
 if Assigned(Self) then myClickAwaker:=aClickAwaker;
end;

function TDaqDevice.AwakeOnClick(What:TEventWhat):Boolean;
begin
 if Assigned(Self) and (What<>evNothing)
 then Result:= (myClickAwaker and (1 shl Ord(What)) <> 0)
 else Result:=false;
end;

function TDaqDevice.SysLogSign(const Sub:LongString=''):LongString;
begin
 Result:='';
 if Assigned(Self) then begin
  if (mySysLogSign='')
  then mySysLogSign:=Daq.SysLogSign(1,myName);
  if (Sub='')
  then Result:=mySysLogSign
  else Result:=mySysLogSign+SysLog.SenderSep+Sub;
 end;
end;

procedure TDaqDevice.GetSensorClicked(var aSensorEvent:TSensorEvent);
begin
 SensorEvent(aSensorEvent);
 if Assigned(Self) then mySensorClicked.Get(@aSensorEvent,SizeOf(aSensorEvent));
end;

procedure TDaqDevice.SetSensorClicked(const aSensorEvent:TSensorEvent);
begin
 if Assigned(Self) and (aSensorEvent.What<>evNothing) then begin
  if mySensorClicked.Put(@aSensorEvent,SizeOf(aSensorEvent))=0
  then FixError(ecSensorClickFifoLost);
  if AwakeOnClick(aSensorEvent.What) then Awake;
 end;
end;

function TDaqDevice.GetPropertyDialog:TDaqDevicePropertyDialog;
begin
 if Assigned(Self) then Result:=myPropertyDialog else Result:=nil;
end;

procedure TDaqDevice.OpenPropertyDialog;
begin
 if Assigned(Self) then begin
  if myPropertyDialog.Ok then begin
   myPropertyDialog.Show;
   myPropertyDialog.WindowState:=wsNormal;
   myPropertyDialog.BringToFront;
  end else begin
   myPropertyDialog:=NewPropertyDialog;
   if myPropertyDialog.Ok then begin
    myPropertyDialog.Master:=@myPropertyDialog;
    myPropertyDialog.LinkedDevice:=Self;
   end;
  end;
 end;
end;

function TDaqDevice.NewPropertyDialog:TDaqDevicePropertyDialog;
begin
 Result:=nil;
end;

function TDaqDevice.UpdatePropertyDialog:Boolean;
begin
 Result:=false;
end;

procedure TDaqDevice.AdvancedPropertyDialog;
begin
end;

procedure TDaqDevice.RightButtonPropertyDialog;
begin
 OpenPropertyDialog;
end;

function  TDaqDevice.CommonPropertyDialog:TDaqDeviceCommonPropertyDialog;
begin
 if Assigned(Self) then Result:=myCommonPropertyDialog else Result:=nil;
end;

procedure TDaqDevice.OpenCommonPropertyDialog;
begin
 if Assigned(Self) then begin
  if myCommonPropertyDialog.Ok then begin
   myCommonPropertyDialog.Show;
   myCommonPropertyDialog.WindowState:=wsNormal;
   myCommonPropertyDialog.BringToFront;
  end else begin
   myCommonPropertyDialog:=NewCommonPropertyDialog;
   if myCommonPropertyDialog.Ok then begin
    myCommonPropertyDialog.Master:=@myCommonPropertyDialog;
    myCommonPropertyDialog.LinkedDevice:=Self;
    myCommonPropertyDialog.Show;
    myCommonPropertyDialog.WindowState:=wsNormal;
    myCommonPropertyDialog.BringToFront;
   end;
  end;
 end;
end;

function TDaqDevice.NewCommonPropertyDialog:TDaqDeviceCommonPropertyDialog;
begin
 Result:=NewFormDaqDeviceCommonPropertyDialog(Self);
end;

function TDaqDevice.UpdateCommonPropertyDialog:Boolean;
begin
 if (CommonPropertyDialog is TFormDaqDeviceCommonPropertyDialog)
 then Result:=TFormDaqDeviceCommonPropertyDialog(myCommonPropertyDialog).HaveToUpdate
 else Result:=false;
end;

procedure TDaqDevice.Idle;
begin
end;

procedure TDaqDevice.Poll;
begin
end;

procedure TDaqDevice.Clear;
var
 Index : Integer;
begin
 ClearAnalogFifo;
 ClearDigitalFifo;
 for Index:=0 to NumAnalogOutputs-1 do ClearAnalogOutputCurve(Index);
 for Index:=0 to NumDigitalOutputs-1 do ClearDigitalOutputCurve(Index);
end;

procedure TDaqDevice.ClearDevice;
begin
end;

function TDaqDevice.CheckDevice:Boolean;
begin
 Result:=true;
end;

procedure TDaqDevice.Config(FileName:LongString);
var f:Double; d:LongInt; s:LongString;
begin
 f:=0; d:=0; s:='';
 FileName:=UnifyFileAlias(FileName);
 {строка комментария}
 if ReadIniFileString(FileName,'['+Name+']','Comment%s',s)
 then Comment:=TrimChars(s,ScanSpaces,ScanSpaces)
 else Comment:='';
 {период опроса}
 if ReadIniFileDouble(FileName,'['+Name+']','InquiryPeriod%f',f)
 then InquiryTimer.IntervalMs[0]:=f
 else InquiryTimer.IntervalMs[0]:=Daq.AcqTimer.IntervalMs[0];
 {групповой тег}
 if ReadIniFileLongInt(FileName,'['+Name+']','GroupTag%d',d)
 then GroupTag:=d
 else GroupTag:=0;
 {starting order}
 if ReadIniFileLongInt(FileName,'['+Name+']','StartingOrder%d',d)
 then StartingOrder:=d
 else StartingOrder:=0;
 {stopping order}
 if ReadIniFileLongInt(FileName,'['+Name+']','StoppingOrder%d',d)
 then StoppingOrder:=d
 else StoppingOrder:=0;
 {чтение fifo}
 if ReadIniFileLongInt(FileName,'['+Name+']','AnalogFifo%d',d)
 then AnalogFifoSize:=d;
 if ReadIniFileLongInt(FileName,'['+Name+']','DigitalFifo%d',d)
 then DigitalFifoSize:=d;
 {чтение фильтра диспетчера}
 if ReadIniFileString(FileName,'['+Name+']','DispatcherFilter%s',s)
 or ReadIniFileString(FileName,'[DAQ]',     'DispatcherFilter%s',s)
 or ReadIniFileString(SysIniFile,'[DaqSys]','DispatcherFilter%s',s)
 then begin
  s:=Trim(s);
  if s<>'' then begin
   if Pos('-NANX',   s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreNanX;
   if Pos('+NANX',   s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreNanX;
   if Pos('-NANY',   s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreNanY;
   if Pos('+NANY',   s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreNanY;
   if Pos('-INFX',   s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreInfX;
   if Pos('+INFX',   s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreInfX;
   if Pos('-INFY',   s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreInfY;
   if Pos('+INFY',   s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreInfY;
   if Pos('-LESSX',  s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreLessX;
   if Pos('+LESSX',  s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreLessX;
   if Pos('-EQUALX', s)>0 then DispatcherFilter:=DispatcherFilter or      afIgnoreEqualX;
   if Pos('+EQUALX', s)>0 then DispatcherFilter:=DispatcherFilter and not afIgnoreEqualX;
   if Pos('-PACKX',  s)>0 then DispatcherFilter:=DispatcherFilter and not afPackDublicatesX;
   if Pos('+PACKX',  s)>0 then DispatcherFilter:=DispatcherFilter or      afPackDublicatesX;
   if Pos('-PACKY',  s)>0 then DispatcherFilter:=DispatcherFilter and not afPackDublicatesY;
   if Pos('+PACKY',  s)>0 then DispatcherFilter:=DispatcherFilter or      afPackDublicatesY;
   if Pos('-UPDATE', s)>0 then DispatcherFilter:=DispatcherFilter and not afUpdateMarker;
   if Pos('+UPDATE', s)>0 then DispatcherFilter:=DispatcherFilter or      afUpdateMarker;
  end;
 end;
end;

procedure TDaqDevice.Animate;
var
 i       : Integer;
 Section : TText;
 {
 Процедура коррекции шага приращения памяти кривой для оптимизации
 }
 procedure StepCorrection(C:TCurve; Hist:LongInt);
 var Step:LongInt;
 begin
  if Assigned(C) and (Hist>0) then begin
   Step:=round(Hist*2.5);
   if C.IsDynamic and (C.Step<Step) then C.Step:=Step;
  end;
 end;
 {
 Разбор строк типа
 Link AnalogInput   nnn with curve yyy smoothing www ppp kk1 kk2
 Link AnalogOutput  nnn with curve yyy tolerance aaa rrr history hhh
 Link DigitalInput  nnn with curve yyy bit zzz
 Link DigitalInput  nnn with curve yyy inverted bit zzz
 Link DigitalOutput nnn with curve yyy history hhh
 }
 procedure AnalizeLine(const S:LongString);
 var
  Curve      : TCurve;
  Wrd        : LongString;
  f,f1       : Double;
  hl         : LongInt;
  icr,iai,idi,iao,ido,ibt,itl,ism,ihl,nai,ndi,nao,mao,ndo,nbt,pw,k1,k2 : Integer;
 begin
  if Daq.Ok then
  if (Length(S)>0) and (ExtractWord(1,S,ScanSpaces)='LINK') then begin
   Curve:=nil;
   {
   curve nnn
   }
   icr:=WordIndex('CURVE',S,ScanSpaces);
   if icr>0 then begin
    Wrd:=ExtractWord(icr+1,S,ScanSpaces);
    Curve:=Daq.Curves.Find(Wrd);
    if Curve=nil then begin
     Daq.AddWarning('['+Name+'] -> bad CURVE in "'+S+'"');
     exit;
    end;
   end;
   if Curve=nil then exit;
   {
   AnalogInput nnn smoothing w p k1 k2
   }
   iai:=WordIndex('ANALOGINPUT',S,ScanSpaces);
   if iai>0 then begin
    Wrd:=ExtractWord(iai+1,S,ScanSpaces);
    if not Str2Int(Wrd,nai) or (nai<0) or (nai>=NumAnalogInputs) then begin
     Daq.AddWarning('['+Name+'] -> bad ANALOGINPUT in "'+S+'"');
     exit;
    end;
    AnalogInputCurve[nai]:=Curve;
    ism:=WordIndex('SMOOTHING',S,ScanSpaces);
    if (ism>0) then begin
     Wrd:=ExtractWord(ism+1,S,ScanSpaces);
     if not Str2Real(Wrd,f) then f:=0 else begin
      Wrd:=ExtractWord(ism+2,S,ScanSpaces);
      if not Str2Int(Wrd,pw) then begin
       pw:=1;
       k1:=2;
       k2:=1;
      end else begin
       Wrd:=ExtractWord(ism+3,S,ScanSpaces);
       if not Str2Int(Wrd,k1) then begin
        k1:=2;
        k2:=1;
       end else begin
        Wrd:=ExtractWord(ism+4,S,ScanSpaces);
        if not Str2Int(Wrd,k2) then k2:=1;
       end;
      end;
     end;
     if f>0 then begin
      if (nai>=0) and (nai<NumAnalogInputs)
      then AnalogInputSmoother[nai]:=NewDaqSmoother(f,pw,k1,k2);
     end;
    end;
    exit;
   end;
   {
   AnalogOutput nnn mmm tolerance aaa rrr history hhh
   }
   iao:=WordIndex('ANALOGOUTPUT',S,ScanSpaces);
   if iao>0 then begin
    Wrd:=ExtractWord(iao+1,S,ScanSpaces);
    if not Str2Int(Wrd,nao) or (nao<0) or (nao>=NumAnalogOutputs) then begin
     Daq.AddWarning('['+Name+'] -> bad ANALOGOUTPUT in "'+S+'"');
     exit;
    end;
    Wrd:=ExtractWord(iao+2,S,ScanSpaces);
    if (Wrd<>'WITH') and Str2Int(Wrd,mao) then begin
     if (nao<0) or (nao>=NumPrimaryAnalogOutputs) or
        (mao<0) or (mao>=NumSecondaryAnalogOutputs)
     then begin
      Daq.AddWarning('['+Name+'] -> bad ANALOGOUTPUT in "'+S+'"');
      exit;
     end;
     nao:=PackAnalogOutputIndex(nao,mao);
    end;
    AnalogOutputCurve[nao]:=Curve;
    itl:=WordIndex('TOLERANCE',S,ScanSpaces);
    if (itl>0) then begin
     Wrd:=ExtractWord(itl+1,S,ScanSpaces);
     if not Str2Real(Wrd,f) then begin
      f:=0;
      f1:=0;
     end else begin
      Wrd:=ExtractWord(itl+2,S,ScanSpaces);
      if not Str2Real(Wrd,f1) then f1:=0;
     end;
     if (f>0) or (f1>0) then begin
      if (nao>=0) and (nao<NumAnalogOutputs)
      then AnalogOutputCompressor[nao]:=NewDaqCompressor(f,f1);
     end;
    end;
    ihl:=WordIndex('HISTORY',S,ScanSpaces);
    if ihl>0 then begin
     Wrd:=ExtractWord(ihl+1,S,ScanSpaces);
     if not Str2Long(Wrd,hl) then hl:=0;
     AnalogOutputHistoryLen[nao]:=hl;
     StepCorrection(AnalogOutputCurve[nao],hl);
    end;
    exit;
   end;
   {
   DigitalInput nnn bit bbb
   DigitalInput nnn inverted bit bbb
   }
   idi:=WordIndex('DIGITALINPUT',S,ScanSpaces);
   if idi>0 then begin
    Wrd:=ExtractWord(idi+1,S,ScanSpaces);
    if not Str2Int(Wrd,ndi) or (ndi<0) or (ndi>=NumDigitalInputs) then begin
     Daq.AddWarning('['+Name+'] -> bad DIGITALINPUT in "'+S+'"');
     exit;
    end;
    ibt:=WordIndex('BIT',S,ScanSpaces);
    Wrd:=ExtractWord(ibt+1,S,ScanSpaces);
    if (ibt=0) or not Str2Int(Wrd,nbt) then begin
     Daq.AddWarning('['+Name+'] -> bad DIGITALINPUT Bit in "'+S+'"');
     exit;
    end;
    DigitalInputCurve[ndi]:=Curve;
    SetDigitalInputBit(ndi,nbt,WordIndex('INVERTED',S,ScanSpaces)=ibt-1);
    exit;
   end;
   {
   DigitalOutput nnn history hhh
   }
   ido:=WordIndex('DIGITALOUTPUT',S,ScanSpaces);
   if ido>0 then begin
    Wrd:=ExtractWord(ido+1,S,ScanSpaces);
    if not Str2Int(Wrd,ndo) or (ndo<0) or (ndo>=NumDigitalOutputs) then begin
     Daq.AddWarning('['+Name+'] -> bad DIGITALOUTPUT in "'+S+'"');
     exit;
    end;
    DigitalOutputCurve[ndo]:=Curve;
    ihl:=WordIndex('HISTORY',S,ScanSpaces);
    if ihl>0 then begin
     Wrd:=ExtractWord(ihl+1,S,ScanSpaces);
     if not Str2Long(Wrd,hl) then hl:=0;
     DigitalOutputHistoryLen[ndo]:=hl;
     StepCorrection(DigitalOutputCurve[ndo],hl);
    end;
    exit;
   end;
  end;
 end;
begin
 if Daq.Ok then begin
  {очистить все входы-выходы}
  for i:=0 to NumAnalogInputs-1 do AnalogInputCurve[i]:=nil;
  for i:=0 to NumDigitalInputs-1 do DigitalInputCurve[i]:=nil;
  for i:=0 to NumDigitalInputs-1 do SetDigitalInputBit(i,0,false);
  for i:=0 to NumAnalogOutputs-1 do AnalogOutputCurve[i]:=nil;
  for i:=0 to NumAnalogOutputs-1 do AnalogOutputCompressor[i]:=nil;
  for i:=0 to NumAnalogOutputs-1 do AnalogOutputHistoryLen[i]:=0;
  for i:=0 to NumDigitalOutputs-1 do DigitalOutputCurve[i]:=nil;
  for i:=0 to NumDigitalOutputs-1 do DigitalOutputHistoryLen[i]:=0;
  {выделить секцию блока и связать}
  Section:=ExtractListSection(Daq.ConfigFile,'['+Name+']',efConfig);
  if Section.Ok then
  for i:=0 to Section.Count-1 do AnalizeLine(Section[i]);
  Kill(Section);
 end;
end;

function TDaqDevice.GetProperty(TheText:TText):TText;
const
 sig : packed array[boolean] of char = ('-','+');
var
 i     : Integer;
 Curve : TCurve;
 j     : Integer;
 k     : Integer;
 s     : LongString;
 inv   : Boolean;
begin
 Result:=TheText;
 if Assigned(TheText) then with TheText do begin
  Addln('[DeviceList]');
  Addln(DeviceDescription);
  Addln('['+Name+']');
  AddLn('Comment = '+Comment);
  AddLn('GroupTag = '+d2s(GroupTag));
  AddLn('StartingOrder = '+d2s(StartingOrder));
  AddLn('StoppingOrder = '+d2s(StoppingOrder));
  Addln('InquiryPeriod = '+f2s(InquiryTimer.IntervalMs[0])+' msec');
  AddLn('AnalogFifo = '+d2s(AnalogFifoSize));
  AddLn('DigitalFifo = '+d2s(DigitalFifoSize));
  Addln('AnalogInputs = '+d2s(NumAnalogInputs));
  Addln('AnalogOutputs = '+d2s(NumAnalogOutputs)+
        ' ( '+d2s(NumPrimaryAnalogOutputs)+
        ' * '+d2s(NumSecondaryAnalogOutputs)+' )');
  Addln('DigitalInputs = '+d2s(NumDigitalInputs));
  Addln('DigitalOutputs = '+d2s(NumDigitalOutputs));
  for i:=0 to NumAnalogInputs-1 do begin
   Curve:=AnalogInputCurve[i];
   if Assigned(Curve) then begin
    s:='Link AnalogInput '+d2s(i)+' with curve '+Curve.Name;
    if AnalogInputSmoother[i]<>nil then
    s:=s+' smoothing '+f2s(AnalogInputSmoother[i].Window)+' '+
                       d2s(AnalogInputSmoother[i].Power)+' '+
                       d2s(AnalogInputSmoother[i].k1)+' '+
                       d2s(AnalogInputSmoother[i].k2);
    AddLn(s);
   end;
  end;
  for i:=0 to NumAnalogOutputs-1 do begin
   Curve:=AnalogOutputCurve[i];
   if Assigned(Curve) then begin
    if NumSecondaryAnalogOutputs=1
    then s:='Link AnalogOutput '+d2s(i)+' with curve '+Curve.Name
    else begin
     UnpackAnalogOutputIndex(i,j,k);
     s:='Link AnalogOutput '+d2s(j)+' '+d2s(k)+' with curve '+Curve.Name;
    end;
    if AnalogOutputCompressor[i]<>nil then
    s:=s+' tolerance '+f2s(AnalogOutputCompressor[i].AbsTol)+' '+
                       f2s(AnalogOutputCompressor[i].RelTol);
    if AnalogOutputHistoryLen[i]>0 then
    s:=s+' history '+d2s(AnalogOutputHistoryLen[i]);
    Addln(s);
   end;
  end;
  for i:=0 to NumDigitalInputs-1 do begin
   Curve:=DigitalInputCurve[i];
   if Assigned(Curve) then begin
    s:='Link DigitalInput '+d2s(i)+' with curve '+Curve.Name;
    j:=GetDigitalInputBit(i,inv);
    if inv then s:=s+' inverted';
    s:=s+' bit '+d2s(j);
    Addln(s);
   end;
  end;
  for i:=0 to NumDigitalOutputs-1 do begin
   Curve:=DigitalOutputCurve[i];
   if Assigned(Curve) then begin
    s:='Link DigitalOutput '+d2s(i)+' with curve '+Curve.Name;
    if DigitalOutputHistoryLen[i]>0 then
    s:=s+' history '+d2s(DigitalOutputHistoryLen[i]);
    Addln(s);
   end;
  end;
  Addln('DispatcherFilter = '
        +sig[DispatcherFilter and afIgnoreNanX       = 0]+'nanx,'
        +sig[DispatcherFilter and afIgnoreNanY       = 0]+'nany,'
        +sig[DispatcherFilter and afIgnoreInfX       = 0]+'infx,'
        +sig[DispatcherFilter and afIgnoreInfY       = 0]+'infy,'
        +sig[DispatcherFilter and afIgnoreLessX      = 0]+'lessx,'
        +sig[DispatcherFilter and afIgnoreEqualX     = 0]+'equalx,'
        +sig[DispatcherFilter and afPackDublicatesX <> 0]+'packx,'
        +sig[DispatcherFilter and afPackDublicatesY <> 0]+'packy,'
        +sig[DispatcherFilter and afUpdateMarker    <> 0]+'update'
        +Format(' ; $%8.8x',[DispatcherFilter]));
 end;
end;

function TDaqDevice.Start:Boolean;
begin
 Result:=false;
 if Ok and Daq.Ok and not InquiryTimer.IsStart then begin
  Result:=true;
  InquiryTimer.Start;
  mySensorClicked.Clear;
 end;
end;

procedure TDaqDevice.Stop;
begin
 InquiryTimer.Stop;
end;

procedure TDaqDevice.Awake;
begin
end;

procedure TDaqDevice.Action;
begin
end;

procedure TDaqDevice.Watchdog;
begin
end;

function TDaqDevice.GotEvents:Boolean;
begin
 Result:=InquiryTimer.Event;
end;

function TDaqDevice.HandleMessage(const aMsg: LongString; aFlags:Cardinal=hf_Default):Double;
begin
 Result:=0;
end;

procedure TDaqDevice.HandleSensorClick(const aEvent: TSensorEvent);
begin
 SetSensorClicked(aEvent);
 if aEvent.What in [evMouseDown,evKeyDown] then
 case aEvent.Button of
  VK_LBUTTON : Action;
  VK_RBUTTON : RightButtonPropertyDialog;
 end;
end;

procedure Kill(var TheObject:TDaqDevice); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;
 {
 *******************************************************************************
 TDaqDevicePropertyDialog implementation
 *******************************************************************************
 }
function TDaqDevicePropertyDialog.GetLinkedDevice:TDaqDevice;
begin
 if Ok then Result:=myLinkedDevice else Result:=nil;
end;

procedure TDaqDevicePropertyDialog.SetLinkedDevice(aDevice:TDaqDevice);
begin
 if Ok then myLinkedDevice:=aDevice;
end;

 {
 *******************************************************************************
 TDaqDeviceList implementation
 *******************************************************************************
 }
constructor TDaqDeviceList.Create(aOwnsObjects : Boolean = true;
                                  aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                                  aStep        : LongInt = DefaultTObjectStorageStep);
begin
 inherited Create(aOwnsObjects,aCapacity,aStep);
 myStartingIndex:=NewLongIntVector(0);
 myStartingIndex.Master:=@myStartingIndex;
 myStoppingIndex:=NewLongIntVector(0);
 myStoppingIndex.Master:=@myStoppingIndex;
end;

destructor TDaqDeviceList.Destroy;
begin
 Kill(myStartingIndex);
 Kill(myStoppingIndex);
 inherited Destroy;
end;

function TDaqDeviceList.GetStartingIndex:TLongIntVector;
begin
 if Assigned(Self) then Result:=myStartingIndex else Result:=nil;
end;

function TDaqDeviceList.GetStoppingIndex:TLongIntVector;
begin
 if Assigned(Self) then Result:=myStoppingIndex else Result:=nil;
end;

function TDaqDeviceList.GetStartingList:LongString;
var i:Integer; t:TText;
begin
 Result:='';
 if Assigned(Self) then
 try
  t:=NewText;
  try
   if StartingIndex.Length=Count then
   for i:=0 to Count-1 do t.Addln(Self[StartingIndex[i]].Name) else
   for i:=0 to Count-1 do t.Addln(Self[i].Name);
   Result:=t.Text;
  finally
   Kill(t);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TDaqDeviceList.GetStoppingList:LongString;
var i:Integer; t:TText;
begin
 Result:='';
 if Assigned(Self) then
 try
  t:=NewText;
  try
   if StoppingIndex.Length=Count then
   for i:=0 to Count-1 do t.Addln(Self[StoppingIndex[i]].Name) else
   for i:=0 to Count-1 do t.Addln(Self[i].Name);
   Result:=t.Text;
  finally
   Kill(t);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TDaqDeviceList.StartingSort;
var i:Integer;
 procedure ShellSortStarting(Index:TLongIntVector);
 var gap,i,j,k:Integer;
 begin
  gap:=Count shr 1;
  while gap>0 do begin
   i:=gap;
   while i<Count do begin
    j:=i-gap;
    while (j>=0) and (Self[Index[j+gap]].StartingOrder<Self[Index[j]].StartingOrder)
    do begin
     k:=Index[j+gap]; Index[j+gap]:=Index[j]; Index[j]:=k;
     dec(j,gap);
    end;
    inc(i);
   end;
   gap:=gap shr 1;
  end;
 end;
begin
 if Assigned(Self) then
 try
  StartingIndex.Length:=Count;
  for i:=0 to Count-1 do StartingIndex[i]:=i;
  ShellSortStarting(StartingIndex);
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TDaqDeviceList.StoppingSort;
var i:Integer;
 procedure ShellSortStopping(Index:TLongIntVector);
 var gap,i,j,k:Integer;
 begin
  gap:=Count shr 1;
  while gap>0 do begin
   i:=gap;
   while i<Count do begin
    j:=i-gap;
    while (j>=0) and (Self[Index[j+gap]].StoppingOrder<Self[Index[j]].StoppingOrder)
    do begin
     k:=Index[j+gap]; Index[j+gap]:=Index[j]; Index[j]:=k;
     dec(j,gap);
    end;
    inc(i);
   end;
   gap:=gap shr 1;
  end;
 end;
begin
 if Assigned(Self) then
 try
  StoppingIndex.Length:=Count;
  for i:=0 to Count-1 do StoppingIndex[i]:=i;
  ShellSortStopping(StoppingIndex);
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TDaqDeviceList.GetDevices(Index:Integer):TDaqDevice;
begin
 Result:=TDaqDevice(Items[Index]);
end;

procedure TDaqDeviceList.SetDevices(Index:Integer; aDevice:TDaqDevice);
begin
 Items[Index]:=aDevice;
end;

function TDaqDeviceList.GetErrorsTotal:Integer;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then inc(Result, Item.ErrorsTotal);
 end;
begin
 Result:=0;
 for i:=0 to Count-1 do Handle(Self[i]);
 inc(Result,TagBugs);
 inc(Result,Daq.WarningList.Count);
 inc(Result,TCircuitSensorTagEvalErrorCount);
 inc(Result,TCircuitSensorLedEvalErrorCount);
 inc(Result,TCircuitSensorPainterErrorCount);
end;

function TDaqDeviceList.GetErrorsCount(ErrorCode:Byte):Integer;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then inc(Result, Item.ErrorsCount[ErrorCode]);
 end;
begin
 Result:=0;
 for i:=0 to Count-1 do Handle(Self[i]);
 case ErrorCode of
  ecTagBug        : inc(Result,TagBugs);
  ecWarningList   : inc(Result,Daq.WarningList.Count);
  ecSensorTagEval : inc(Result,TCircuitSensorTagEvalErrorCount);
  ecSensorLedEval : inc(Result,TCircuitSensorLedEvalErrorCount);
  ecSensorPainter : inc(Result,TCircuitSensorPainterErrorCount);
 end;
end;

function TDaqDeviceList.Find(const DeviceName:LongString):TDaqDevice;
var
 i : Integer;
begin
 Result:=nil;
 if IsNonEmptyStr(DeviceName) then
 for i:=0 to Count-1 do
 if SameText(UnifyAlias(Self[i].Name),UnifyAlias(DeviceName)) then begin
  Result:=Self[i];
  break;
 end;
end;

procedure TDaqDeviceList.ClearErrors;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.ClearErrors;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
 TCircuitSensorTagEvalErrorCount:=0;
 TCircuitSensorLedEvalErrorCount:=0;
 TCircuitSensorPainterErrorCount:=0;
 SetTagBugs(0);
end;

procedure TDaqDeviceList.DispatchEvents;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.DispatchEvents;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Idle;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Idle;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Poll;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Poll;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Clear;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Clear;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.ClearDevice;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.ClearDevice;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

function  TDaqDeviceList.CheckDevice:Boolean;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Result:=Item.CheckDevice and Result;
 end;
begin
 Result:=Daq.Ok;
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Action;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Action;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Watchdog;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Watchdog;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

function TDaqDeviceList.Start:Boolean;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Result:=Item.Start and Result;
 end;
begin
 Result:=Daq.Ok;
 if StartingIndex.Length=Count then
 for i:=0 to Count-1 do Handle(Self[StartingIndex[i]]) else
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Stop;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Stop;
 end;
begin
 if StoppingIndex.Length=Count then
 for i:=0 to Count-1 do Handle(Self[StoppingIndex[i]]) else
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.Animate;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 begin
  if Item.Ok then Item.Animate;
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.StartSession;
begin
 StartingSort;
 StoppingSort;
end;

procedure TDaqDeviceList.StopSession;
begin
end;

function TDaqDeviceList.GetNumCurveLinks(Curve:TCurve):Integer;
var
 i : Integer;
 procedure Handle(Item:TDaqDevice);
 var
  i : Integer;
 begin
  if Item.Ok then begin
   for i:=0 to Item.NumAnalogOutputs-1 do
   if Curve = Item.AnalogOutputCurve[i] then inc(Result);
   for i:=0 to Item.NumDigitalOutputs-1 do
   if Curve = Item.DigitalOutputCurve[i] then inc(Result);
  end;
 end;
begin
 Result:=0;
 if Curve.Ok then for i:=0 to Count-1 do Handle(Self[i]);
end;

procedure TDaqDeviceList.CheckCurveLinks;
var
 i : Integer;
 n : Integer;
 procedure Handle(Curve:TCurve);
 var
  Links : Integer;
 begin
  if Curve.Ok then begin
   Links:=GetNumCurveLinks(Curve);
   if Links<>1 then begin
    if n=0 then Daq.AddWarning('Link curve<->device error(s) detected:');
    inc(n);
    Daq.AddWarning(Format(' Curve %12s have %d links',[Curve.Name,Links]));
   end;
  end;
 end;
begin
 n:=0;
 if Daq.Ok then for i:=0 to Daq.Curves.Count-1 do Handle(Daq.Curves[i]);
end;

function TDaqDeviceList.GetNameList(aText:TText):TText;
var
 i : Integer;
begin
 Result:=aText;
 for i:=0 to Count-1 do aText.Addln(Self[i].Name);
end;

function TDaqDeviceList.GetDescriptionList(aText:TText):TText;
var
 i : Integer;
 procedure CheckItem(Device:TDaqDevice);
 begin
  if Device.Ok then begin
   myNamePad:=max(myNamePad,Length(Device.Name));
   myFamilyPad:=max(myFamilyPad,Length(Device.Family));
   myModelPad:=max(myModelPad,Length(Device.Model));
  end;
 end;
 procedure AddItem(Device:TDaqDevice);
 begin
  if Device.Ok then
  aText.Addln(Pad(Device.Name,myNamePad)+' = DEVICE '+
              Pad(Device.Family,myFamilyPad)+' '+Pad(Device.Model,myModelPad));
 end;
begin
 Result:=aText;
 if Count>0 then begin
  myNamePad:=0;
  myFamilyPad:=0;
  myModelPad:=0;
  for i:=0 to Count-1 do CheckItem(Self[i]);
  for i:=0 to Count-1 do AddItem(Self[i]);
 end;
end;

function TDaqDeviceList.GetFullList(aText:TText):TText;
var
 i : Integer;
 procedure AddSection(Device:TDaqDevice);
 begin
  if Device.Ok then Device.GetProperty(aText);
 end;
begin
 Result:=aText;
 if Ok and aText.Ok then begin
  aText.Addln('[DeviceList]');
  GetDescriptionList(aText);
  for i:=0 to Count-1 do AddSection(Self[i]);
 end;
end;

function TDaqDeviceList.SelectDevice(const Caption:LongString):TDaqDevice;
var
 List : TText;
 Key  : Integer;
begin
 Result:=nil;
 if (Count>0) then begin
  List:=GetDescriptionList(NewText);
  Key:=ListBoxMenu(Caption, Pad('Name',myNamePad)+' = DEVICE '+
                            Pad('Family',myFamilyPad)+' '+
                            Pad('Model',myModelPad), List.Text);
  if (Key>=0) then Result:=Find(ExtractWord(1,List[Key],ScanSpaces));
  Kill(List);
 end;
end;

procedure TDaqDeviceList.Config(FileName,Msg:LongString;
                                DeviceConstructor:TDaqDeviceConstructor);
var
 i       : Integer;
 Section : TText;
 procedure InsertDevice(const s:LongString);
 label Quit;
 var Ident,DeviceID,Family,Model:LongString; Device:TDaqDevice;
     DeviceAlreadyExists:Boolean;
 begin
  if Daq.Ok then
  if IsNonEmptyStr(s) then begin
   {разбить на слова ident = device family model}
   Ident   :=ExtractWord(1,s,ScanSpaces);
   DeviceID:=ExtractWord(2,s,ScanSpaces);
   Family  :=ExtractWord(3,s,ScanSpaces);
   Model   :=ExtractWord(4,s,ScanSpaces);
   {нет слова device?}
   if DeviceID<>'DEVICE' then begin
    Daq.AddWarning('[DeviceList] -> Syntax error in "'+s+'"');
    goto Quit;
   end;
   {нет конструктора?}
   if not Assigned(DeviceConstructor) then begin
    Daq.AddWarning('[DeviceList] -> NIL device constructor in "'+s+'"');
    goto Quit;
   end;
   {устройство с таким именем уже есть?}
   DeviceAlreadyExists:=FullDaqDeviceList.Find(Ident).Ok;
   {создаем Device}
   Device:=DeviceConstructor(Ident,Family,Model,FileName);
   if Device.Ok then begin
    {попытка создания устройства с таким же именем?}
    if DeviceAlreadyExists then begin
     Daq.AddWarning('[DeviceList] -> Duplicate device name in "'+s+'"');
     Kill(Device);
     goto Quit;
    end;
    {неверная модель?}
    if Device.Model<>Model then begin
     Daq.AddWarning('[DeviceList] -> Invalid device model "'+Device.Model+'" in "'+s+'"');
     Kill(Device);
     goto Quit;
    end;
    {все в порядке, читаем конфигурацию}
    Device.Config(FileName);
    {и вставляем в список}
    Add(Device);
   end;
  end;
  Quit:
 end;
begin
 if Ok and Daq.Ok then
 try
  FileName:=UnifyFileAlias(FileName);
  Section:=ExtractListSection(FileName,'[DeviceList]',efConfig);
  try
   for i:=0 to Section.Count-1 do InsertDevice(Section[i]);
   StartingSort;
   StoppingSort;
  finally
   Kill(Section);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function FullDaqDeviceList:TDaqDeviceList;
const
 myFullDaqDeviceList : TDaqDeviceList = nil;
begin
 if not Assigned(myFullDaqDeviceList) then begin
  myFullDaqDeviceList:=TDaqDeviceList.Create(false);
  myFullDaqDeviceList.Master:=@myFullDaqDeviceList;
 end;
 Result:=myFullDaqDeviceList;
end;

function  NewDaqDeviceList(aOwnsObjects : Boolean = true;
                           aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                           aStep        : LongInt = DefaultTObjectStorageStep
                                      ) : TDaqDeviceList;
begin
 Result:=TDaqDeviceList.Create(aOwnsObjects,aCapacity,aStep);
end;

procedure Kill(var TheObject:TDaqDeviceList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 {
 *******************************************************************************
 Utilites
 *******************************************************************************
 }
 {
 Просмотр списка тегов.
 }
procedure ViewTagList;
var P:TText; Key,tag:Integer; d:Longint; r:Double; s:LongString;
begin
 P:=GetTagList(NewText);
 Key:=ListBoxMenu(RusEng('СПИСОК ТЕГОВ','TAG LIST'),'',P.Text);
 if Key>=0 then begin
  tag:=FindTag(ExtractWord(1,P[Key],ScanSpaces));
  case TypeTag(tag) of
   1:begin
      d:=igettag(tag);
      s:=Format('%d',[d]);
      if FormDaqEditTagDialogExecute(NameTag(tag),s)=mrOk then
      if NoProblem(Str2Long(s,d),RusEng('Неверный формат числа!',
                                        'Invalid numeral format!'))
      then iSetTag(tag,d);
     end;
   2:begin
      r:=rgettag(tag);
      s:=Format('%g',[r]);
      if FormDaqEditTagDialogExecute(NameTag(tag),s)=mrOk then
      if NoProblem(Str2Real(s,r),RusEng('Неверный формат числа!',
                                        'Invalid numeral format!'))
      then rSetTag(tag,r);
     end;
   3:begin
      s:=sGetTag(tag);
      if FormDaqEditTagDialogExecute(NameTag(tag),s)=mrOk
      then sSetTag(tag,s);
     end;
  end;
 end;
 Kill(P);
end;

 {
 Стандартная процедура сканирования мнемосхем обновляет состояние сенсоров
 по последнему значению ассоциированной с ней кривой или тега
 }
procedure DefaultDaqCircuitMonitor(Circuit:TFormCircuitWindow);
var
 i : Integer;
 procedure ScanItem(Sensor:TCircuitSensor);
 var
  LinkedTag   : Integer;
  LinkedCurve : TCurve;
  TempBuffer  : TParsingBuffer;
 begin
  if Sensor.Ok then begin
   LinkedTag:=Sensor.LinkedTag;
   LinkedCurve:=Sensor.LinkedCurve;
   if Assigned(LinkedCurve) then begin   { если подключена кривая }
    if LinkedCurve.Count>0 then begin
     //Sensor.TextColor:=LinkedCurve.Color;
     Sensor.TextColor:=Sensor.LedFontColor;
     Circuit.UpdateSensor(Sensor,LinkedCurve.LastPoint.Y);
    end;
   end else
   case TypeTag(LinkedTag) of            { если подключен тег }
    1:begin
       Sensor.TextColor:=Sensor.LedFontColor;
       Circuit.UpdateSensor(Sensor,iGetTag(LinkedTag));
      end;
    2:begin
       Sensor.TextColor:=Sensor.LedFontColor;
       Circuit.UpdateSensor(Sensor,rGetTag(LinkedTag));
      end;
    3:begin
       Sensor.TextColor:=Sensor.LedFontColor;
       Circuit.UpdateSensor(Sensor,0,StrCopyBuff(TempBuffer,sGetTag(LinkedTag)));
      end;
   end;
  end;
 end;
begin
 if Circuit.Ok then for i:=0 to Circuit.Count-1 do ScanItem(Circuit[i]);
end;

 {
 Стандартная процедура стратегии мнемосхем использует ассоциированный
 с сенсором обьект типа TDaqDevice для операций с сенсорами (по левой кнопке
 вызывается Action, по правой Property). В поле SensorInfo записывается
 при этом нажатая кнопка, имя сенсора и тег (для использования в программах
 на языке DAQ PASCAL)
 При нажатии правой кнопки над основным полем показывает список связей.
 }
procedure DefaultDaqCircuitStrategy(Circuit:TFormCircuitWindow; var Event:TCircuitEvent; Sensor:TCircuitSensor);
var Device:TDaqDevice;
 { Get sensor's linked device }
 function LinkedDevice(Sensor:TCircuitSensor):TDaqDevice;
 begin
  if Sensor.Ok and (Sensor.LinkedObject is TDaqDevice)
  then Result:=TDaqDevice(Sensor.LinkedObject)
  else Result:=nil;
 end;
 { Показать описание подключений мнемосхемы }
 procedure ViewLinks;
 var
  i    : Integer;
  List : TText;
  ws,wd,wc,wt,wu : Integer;
  procedure AddItem(Sensor:TCircuitSensor;Fit:Bool);
  var s:LongString;
  begin
   if Sensor.Ok then begin
    if Fit then begin
     ws:=Max(ws,Length(Sensor.Name));
     if LinkedDevice(Sensor).Ok
     then wd:=Max(wd,Length(LinkedDevice(Sensor).Name));
     if Sensor.LinkedCurve is TCurve
     then wc:=Max(wc,Length(Sensor.LinkedCurve.Name));
     if TypeTag(Sensor.LinkedTag) in [1,2,3]
     then wt:=Max(wt,Length(NameTag(Sensor.LinkedTag)));
     if Sensor.ShortCut<>0
     then wu:=Max(wu,Length(ShortCutToText(Sensor.ShortCut)));
    end else begin
     s:='Link sensor '+Pad(Sensor.Name,ws)+' with';
     if Sensor.LinkedCurve is TCurve
     then s:=s+' curve '+Pad(Sensor.LinkedCurve.Name,max(wc,wt))
     else
     if TypeTag(Sensor.LinkedTag) in [1,2,3]
     then s:=s+' tag   '+Pad(NameTag(Sensor.LinkedTag),max(wc,wt))
     else s:=s+StringOfChar(' ',Length(' curve ')+max(wc,wt));
     if LinkedDevice(Sensor).Ok
     then s:=s+' device '+Pad(LinkedDevice(Sensor).Name,wd);
     if Sensor.ShortCut<>0
     then s:=s+' shortcut '+Pad(ShortCutToText(Sensor.ShortCut),wu);
     List.Addln(s);
    end;
   end;
  end;
 begin
  if Circuit.Ok then begin
   List:=NewText;
   try
    ws:=1; wd:=1; wc:=1; wt:=1; wu:=1;
    for i:=0 to Circuit.Count-1 do AddItem(Circuit[i],True);
    for i:=0 to Circuit.Count-1 do AddItem(Circuit[i],False);
    ListBoxMenu(RusEng('Характеристики связей мнемосхемы','Mnemonic scheme links'),
                RusEng('Мнемоcхема ','Mnemonic Scheme ')+Circuit.Caption, List.Text);
   finally
    Kill(List);
   end;
  end;
 end;
 procedure DeviceClickHandler;
 var msg:LongString; opt:Integer;
 begin
  if Event.What in [evMouseDown,evKeyDown,evMouseWheelDown]
  then begin msg:='* ClickSensor('+Sensor.Name+')'; opt:=gf_Default; end
  else begin msg:=''; opt:=0; end;
  if Sensor.Ok and Circuit.Ok then
  if Guard.Check(Sensor.GuardLevel,msg,opt)>=0 then
  if Device.Ok and Device.SuitableClick(Event.What) then
  Device.HandleSensorClick(SensorEvent(Circuit,Event,Sensor));
 end;
begin
 try
  if Circuit.Ok then begin
   Device:=LinkedDevice(Sensor);
   case Event.What of
    evMouseMove:
    if Sensor.Ok then begin
     if Length(Sensor.HintLine)=0 then begin
      if (Sensor.ShortCut<>0) and (ShortCutToText(Sensor.ShortCut)<>'') then
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Shortcut=%s',[ShortCutToText(Sensor.ShortCut)]);
      if typetag(Sensor.LinkedTag)>0 then
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Tag=%s',[nametag(Sensor.LinkedTag)]);
      if Sensor.LinkedCurve.Ok then
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Curve=%s',[Sensor.LinkedCurve.Name]);
      if Device.Ok then
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Device=%s',[Device.Name]);
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Mouse=%d,%d',[Event.Where.X,Event.Where.Y]);
     end;
     DeviceClickHandler;
    end else begin
     if Length(Circuit.HintLine)=0 then begin
      Circuit.StatusBar.SimpleText:=Circuit.StatusBar.SimpleText+
       Format(' Mouse=%d,%d',[Event.Where.X,Event.Where.Y]);
     end;
    end;
    evMouseDown:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end else begin
     case Event.Key of
      VK_RBUTTON : ViewLinks;
     end;
    end;
    evKeyDown:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
    evMouseUp:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
    evKeyUp:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
    evMouseWheel:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
    evMouseWheelDown:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
    evMouseWheelUp:
    if Sensor.Ok then begin
     DeviceClickHandler;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 'Оживить' сенсоры мнемосхем.
 При загрузке DAQ все мнемосхемы загружаются, но остаются 'мертвыми',
 то есть поля связей, стратегии и сканеры пусты.
 Эта процедура берет готовый список схем и 'оживляет' их, то есть
 читает из конфигурации связи.
 Необходимость процедуры связана с тем, что на этапе чтения окон
 списка функциональных блоков еще нет. Окна читаются, но 'оживляются'
 только после загрузки списка блоков.
 Формат описания связей:
 Link sensor xxx with device yyy, curve zzz
 }
procedure AnimateCircuits(aStrategy:TCircuitStrategy; aMonitor:TCircuitMonitor);
var
 i : Integer;
 procedure ConfigCircWin(Win:TFormCircuitWindow);
 var
  i       : Integer;
  Section : TText;
  {
  Разбор строки типа
  Link sensor xxx with device yyy, curve zzz
  }
  procedure AnalizeLine(const S:LongString);
  var
   Sensor : TCircuitSensor;
   Curve  : TCurve;
   Device : TDaqDevice;
   HotKey : TShortCut;
   Wrd    : LongString;
   isen   : Integer;
   idev   : Integer;
   icur   : Integer;
   itag   : Integer;
   ikey   : Integer;
   Tag    : Integer;
  begin
   if Daq.Ok then
   if (Length(S)>0) and SameText(ExtractWord(1,S,ScanSpaces),'LINK') then begin
    Sensor:=nil;
    Curve:=nil;
    Device:=nil;
    HotKey:=0;
    Tag:=0;
    isen:=WordIndex('SENSOR',S,ScanSpaces);
    idev:=WordIndex('DEVICE',S,ScanSpaces);
    icur:=WordIndex('CURVE',S,ScanSpaces);
    itag:=WordIndex('TAG',S,ScanSpaces);
    ikey:=WordIndex('SHORTCUT',S,ScanSpaces);
    if isen>0 then begin
     Wrd:=ExtractWord(isen+1,S,ScanSpaces);
     Sensor:=Win.SensorByName(Wrd);
     if not Assigned(Sensor) then begin
      Daq.AddWarning(Win.SourceSection+' -> bad SENSOR in "'+S+'"');
      exit;
     end;
    end;
    if idev>0 then begin
     Wrd:=ExtractWord(idev+1,S,ScanSpaces);
     Device:=FullDaqDeviceList.Find(Wrd);
     if not Assigned(Device) then begin
      Daq.AddWarning(Win.SourceSection+' -> bad DEVICE in "'+S+'"');
      exit;
     end;
    end;
    if icur>0 then begin
     Wrd:=ExtractWord(icur+1,S,ScanSpaces);
     Curve:=Daq.Curves.Find(Wrd);
     if not Assigned(Curve) then begin
      Daq.AddWarning(Win.SourceSection+' -> bad CURVE in "'+S+'"');
      exit;
     end;
    end;
    if itag>0 then begin
     Wrd:=ExtractWord(itag+1,S,ScanSpaces);
     Tag:=FindTag(Wrd);
     if not (TypeTag(Tag) in [1,2,3]) then begin
      Daq.AddWarning(Win.SourceSection+' -> bad TAG in "'+S+'"');
      exit;
     end;
    end;
    if (icur>0) and (itag>0) then begin
     Daq.AddWarning(Win.SourceSection+' -> CURVE & TAG in "'+S+'"');
     exit;
    end;
    if ikey>0 then begin
     Wrd:=ExtractWord(ikey+1,S,ScanSpaces);
     HotKey:=TextToShortCut(Wrd);
    end;
    if Assigned(Sensor) then begin
     Sensor.LinkedTag:=Tag;
     Sensor.LinkedCurve:=Curve;
     Sensor.LinkedObject:=Device;
     Sensor.ShortCut:=HotKey;
    end;
   end;
  end;
 begin
  if Daq.Ok and Win.Ok then begin
   Win.Strategy:=aStrategy;
   Win.Monitor:=aMonitor;
   Section:=ExtractListSection(Win.SourceCfgFile, Win.SourceSection, efConfig);
   if Section.Ok then for i:=0 to Section.Count-1 do AnalizeLine(Section[i]);
   Kill(Section);
  end;
 end;
begin
 if Daq.Ok then
 for i:=0 to Daq.CirWinList.Count-1 do ConfigCircWin(Daq.CirWinList[i]);
end;

 {
 Форматировать строку описания калибровки
 }
function FormatCalibration(const aName:LongString; aCalibration:TPolynomCalibration):LongString;
begin
 Result:=aName+' = ';
 if Daq.Ok and aCalibration.Ok then begin
  if IsNonEmptyStr(aCalibration.FileName)
  then Result:=Result+Daq.FileRel(aCalibration.FileName)
  else Result:=Result+'*';
  if IsNonEmptyStr(aCalibration.NameX)
  then Result:=Result+' '+aCalibration.NameX
  else Result:=Result+' *';
  if IsNonEmptyStr(aCalibration.NameY)
  then Result:=Result+' '+aCalibration.NameY
  else Result:=Result+' *';
  if IsNonEmptyStr(aCalibration.NameZ)
  then Result:=Result+' '+aCalibration.NameZ
  else Result:=Result+' *';
  Result:=Result+' '+ExtractCalibAlias(aCalibration.TransformX)
                +' '+ExtractCalibAlias(aCalibration.TransformY);
  Result:=Result+Format(' %g %g', [aCalibration.BoundA, aCalibration.BoundB]);
 end else Result:=Result+'(not init)';
end;

 {
 Проверяет наличие в секции [DaqDevice] каких-либо устройств заданного
 семейства Family и/или типа Model
 }
function DeviceFamilyPresent(Config,Family,Model:LongString):Boolean;
var
 i       : Integer;
 Section : TText;
 function CheckLine(const s:LongString):Boolean;
 begin
  Result:=(WordCount(s,ScanSpaces)=4) and
          SameText(ExtractWord(2,s,ScanSpaces),'DEVICE') and
          SameText(ExtractWord(3,s,ScanSpaces),Family) and
          SameText(ExtractWord(4,s,ScanSpaces),Model);
 end;
begin
 Result:=false;
 Config:=UnifyFileAlias(Config);
 Section:=ExtractListSection(Config,'[DeviceList]',efConfig);
 try
  if Section.Ok then
  for i:=0 to Section.Count-1 do
  if CheckLine(Section[i]) then begin
   Result:=true;
   break;
  end;
 finally
  Kill(Section);
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqdev;
begin
 FullDaqDeviceList.Ok;
end;

procedure Free_crw_daqdev;
begin
 FullDaqDeviceList.Free;
end;

initialization

 Init_crw_daqdev;

finalization

 Free_crw_daqdev;

end.

//////////////
// END OF FILE
//////////////

