//////////////////////////////////////////////////////
// Threadsafe queue (FIFO) for DIM client/server code.
// Copyright (c) 2024 Alexey Kuryakin daqgroup@mail.ru
//////////////////////////////////////////////////////

#include <stdlib.h>
#include <dim_common.h>

/////////////////////////////////
// Structures required for queue.
/////////////////////////////////

// Note: Union uses to make DIM data access easy.
// dimData are sharing pointer of different type.

typedef union dimData {
  void *P;
  char *C;
  int  *I;
  long *L;
  float *F;
  double *D;
  long long *X;
} dimData;

typedef struct queueNode{
  void *next;
  dim_long tag;
  int   size;
  dimData data;
} queueNode;

typedef struct queueHead {
  queueNode *head;
  queueNode *tail;
  queueNode *node;
  CRITICAL_SECTION critSect;
} queueHead;

////////////////////////////////////////////////////////////////////////////////
// Usage:
// Initialization:
//  queueHead* q=queue_init();
// Server:
//  queue_push(q,tag,buffer,size);
// Client:
//  while(queue_pop(q)){
//     queueNode *node = queue_node(q);
//     // Note: node valid only inside this block,
//     // until new queue_pop replace node to new one.
//     printf('Got tag[%d] data: %s',node->tag,node->data);
//  };
// Finalization:
//  queue_free(q);
////////////////////////////////////////////////////////////////////////////////
queueHead* queue_init();
void queue_free(queueHead *queue);
void queue_lock(queueHead *queue);
void queue_unlock(queueHead *queue);
void queue_push(queueHead *queue, dim_long tag, void *data, int size);
queueNode* queue_pop(queueHead *queue);
queueNode* queue_node(queueHead *queue);

queueHead* queue_init() {
  queueHead *queue = malloc(sizeof(*queue));
  if(queue){
   queue->head = NULL;
   queue->tail = NULL;
   queue->node = NULL;
   InitializeCriticalSection(&(queue->critSect));
  };
  return queue;
};

void queue_free(queueHead *queue) {
  if(queue){
   while(queue_pop(queue)); queue_pop(queue);
   DeleteCriticalSection(&(queue->critSect));
   free(queue);
   queue = NULL;
  };
};

void queue_lock(queueHead *queue) {
  if(queue){ EnterCriticalSection(&(queue->critSect)); };
};

void queue_unlock(queueHead *queue) {
  if(queue){ LeaveCriticalSection(&(queue->critSect)); };
};

void queue_push(queueHead *queue, dim_long tag, void *data, int size) {
  if(queue == NULL) return;
  queueNode *node = malloc(sizeof(*node));
  if(size<0) size=0;
  node->tag  = tag; 
  node->size = size;
  node->data.P = NULL;
  if((size>0)&&(data != NULL)) node->data.P = memcpy(malloc(size),data,size);
  //printf("New data tag[%d], data[%d]: %s\n",tag,size,data);
  //printf("Set data tag[%d], data[%d]: %s\n",node->tag,node->size,node->data.C);
  node->next = NULL;
  queue_lock(queue);
  if (queue->head == NULL) {
    queue->head = node;
    queue->tail = node;
  } else {
    queueNode* old_tail = queue->tail;
    old_tail->next = node;
    queue->tail = node;
  }
  queue_unlock(queue);
};

queueNode* queue_pop(queueHead *queue) {
  if(queue == NULL) return NULL;
  queue_lock(queue);
  if(queue->node){
   free(queue->node->data.P);
   free(queue->node);
   queue->node = NULL;
  };
  if (queue->head != NULL) {
    queue->node = queue->head;
    queue->head = queue->head->next;
  };
  if (queue->head == NULL) {
    queue->tail = NULL;
  }
  queueNode *node=queue->node;
  queue_unlock(queue);
  return node;
};

queueNode* queue_node(queueHead *queue) {
  if(queue == NULL) return NULL;
  queue_lock(queue);
  queueNode *node=queue->node;
  queue_unlock(queue);
  return node;
};

void queue_tester(){
  // Initialize
  queueHead *q = queue_init();
  // Put data to queue
  for(int i=1; i<10; i++){
   char buff[80]; sprintf(buff,"Loop %d",i);
   queue_push(q,i,buff,strlen(buff)+1);
  };
  // Get data from queue
  while(queue_pop(q)){
   queueNode *node=queue_node(q);
   printf("Got data tag[%d], data[%d]: %s\n",node->tag,node->size,node->data.C);
  };
  // Finalize
  queue_free(q);
};
