////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpDaqConfig; // Diesel Pascal DAQ system config files reader.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpVbs,dpCrwDaqSys;

const efDrop     = 1;    // flag for efMode - Drop comments
const efTrim     = 2;    // flag for efMode - Trim
const efLower    = 4;    // flag for efMode - Lowercase
const efUpper    = 8;    // flag for efMode - Uppercase
const efAsAnsi   = 16;   // flag for efMode - I/O as Ansi format
const efConfig   = efDrop+efTrim+efAsAnsi+efUpper;  // Default for configs
const efConfigNC = efDrop+efTrim+efAsAnsi;          // Config with case sense

type
 TDaqConfig = class(TObject) // Config files reader
 protected
  FRecursion  : Integer;
  FTotalList  : TStringList;
 public
  constructor Create;
  destructor  Destroy;
 public
  // Drop comment after ; char
  function DropComment(S:String):String;
  // Detect 0:not a section, 1:any section, 2:given section
  function DetectSectionHeader(S,Section:String):Integer;
  // Get variable value from Line contains 'Name = Value' expression
  function GetVarValue(Line,Name:String; var Value:String):Boolean;
  // Extract section from List/Text/File with given mode
  // Extraction from file uses [ConfigFileList] features
  function ExtractSectionFromList(List:TStringList; Section:String; efMode:Integer):String;
  function ExtractSectionFromText(TextLines:String; Section:String; efMode:Integer):String;
  function ExtractSectionFromFile(FileName:String; Section:String; efMode:Integer):String;
  // Read string value (Key = Value) from Section of INI file FileName.
  function ReadIniStr(FileName,Section,Key:String; Mode:Integer=efConfigNC):String;
 end;

function DaqConfig:TDaqConfig; // General TDaqConfig class instance

////////////////////////////////////////////////////////////////////////////////
// Next functions provides Section names which can be system dependent.
// For example:
// Section [System]    with Mode=0.
// Section [System:OS] with Mode=1, where OS=(Unix,Windows).
////////////////////////////////////////////////////////////////////////////////

 // Section [Name] by given Name.
function SectByName(const Name:String; Mode:Integer=0):String;

// Section [System]
function SectSystem(Mode:Integer=0):String;

// Section [System.StartupScript]
function SectSystemStartupScript(Mode:Integer=0):String;

// Section [DaqSys]
function SectDaqSys(Mode:Integer=0):String;

// Section [DAQ]
function SectDaq(Mode:Integer=0):String;

// Section [SysVoice]
function SectSysVoice(Mode:Integer=0):String;

// Section [@run]
function SectAtRun(Mode:Integer=0):String;

// Section [ActionList]
function SectActionList(Mode:Integer=0):String;

// Section [HandlerList]
function SectHandlerList(Mode:Integer=0):String;

// Section [i18n/codes/iso3166.csv]
function SectI18nCodesIso3166Csv(Mode:Integer=0):String;

// Section [i18n/codes/ru/world.csv]
function SectI18nCodesRuWorldCsv(Mode:Integer=0):String;

// Section [i18n/codes/cca2tld.csv]
function SectI18nCodesCca2TldCsv(Mode:Integer=0):String;

// Section [i18n/codes/iso639-1.csv]
function SectI18nCodesIso639v1Csv(Mode:Integer=0):String;

// Section [i18n/codes/iso639-2.csv]
function SectI18nCodesIso639v2Csv(Mode:Integer=0):String;

// Section [i18n/codes/gost7.75-97.csv]
function SectI18nCodesGost7p75v97Csv(Mode:Integer=0):String;

// Section [i18n/codes/ms-lcid.csv]
function SectI18nCodesMsLcidCsv(Mode:Integer=0):String;

// Section [Text Editors Palette]
function SectTextEditorsPalette(Mode:Integer=0):String;

// Section [ConsoleUtilities]
function SectConsoleUtilities(Mode:Integer=0):String;

implementation

var TheDaqConfig : TDaqConfig = nil;

function DaqConfig:TDaqConfig;
begin
 if (TheDaqConfig=nil) then TheDaqConfig:=TDaqConfig.Create;
 Result:=TheDaqConfig;
end;

constructor TDaqConfig.Create;
begin
 inherited Create;
 FRecursion:=0;
 FTotalList:=TStringList.Create;
 if (TheDaqConfig=nil) then TheDaqConfig:=Self;
end;

destructor TDaqConfig.Destroy;
begin
 if (TheDaqConfig=Self) then TheDaqConfig:=nil;
 FTotalList.Free;
 FRecursion:=0;
 inherited Destroy;
end;

function TDaqConfig.DropComment(S:String):String;
var p:Integer;
begin
 p:=Pos(';',S);
 if (p>0) then S:=Copy(S,1,p-1);
 Result:=S;
end;

function TDaqConfig.DetectSectionHeader(S,Section:String):Integer;
begin
 Result:=0; // It' not section header
 S:=Trim(DropComment(S)); if (S='') then Exit;
 if (S[1]<>'[') then Exit; if (S[Length(S)]<>']') then Exit;
 Result:=1; // It's some section header found
 if (Section<>'') and not IsSameText(S,Section) then Exit;
 Result:=2; // It's section specified found
end;

function TDaqConfig.GetVarValue(Line,Name:String; var Value:String):Boolean;
var p:Integer;
begin
 Result:=false; Value:='';
 Line:=Trim(Line); Name:=Trim(Name);
 if (Line='') or (Name='') then Exit;
 p:=Pos('=',Line); if (p<2) then Exit;
 if not IsSameText(Trim(Copy(Line,1,p-1)),Name) then Exit;
 Value:=Trim(Copy(Line,p+1,Length(Line)-p));
 Result:=true;
end;

function TDaqConfig.ExtractSectionFromList(List:TStringList; Section:String; efMode:Integer):String;
var i:Integer; Temp:TStringList; InSection:Boolean; S:String;
begin
 Result:='';
 Section:=Trim(Section);
 if (List<>nil) and (List.Count>0) then
 if (DetectSectionHeader(Section,'')>0) then
 try
  Temp:=TStringList.Create;
  try
   InSection:=false;
   for i:=0 to List.Count-1 do begin
    S:=List.Strings[i];
    case DetectSectionHeader(S,Section) of
     0: if InSection then begin
         if ((efMode and efDrop)  <> 0) then S:=DropComment(S);
         if ((efMode and efTrim)  <> 0) then S:=Trim(S);
         if ((efMode and efLower) <> 0) then S:=Lowercase(S);
         if ((efMode and efUpper) <> 0) then S:=Uppercase(S);
         if (S<>'') then Temp.Add(S);
        end;
     1: InSection:=false;
     2: InSection:=true;
    end;
   end;
   Result:=Temp.Text;
  finally
   Temp.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromList');
   Result:='';
  end;
 end;
end;

function TDaqConfig.ExtractSectionFromText(TextLines:String; Section:String; efMode:Integer):String;
var List:TStringList;
begin
 Result:='';
 if (TextLines<>'') and (Section<>'') then
 try
  List:=TStringList.Create;
  try
   List.Text:=TextLines;
   Result:=ExtractSectionFromList(List,Section,efMode);
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromText');
   Result:='';
  end;
 end;
end;

function TDaqConfig.ExtractSectionFromFile(FileName:String; Section:String; efMode:Integer):String;
var Lines,Sect:String; List:TStringList; i,ioMode:Integer; S,FN,UserProfile,HomeDir:String;
begin
 Result:='';
 Section:=Trim(Section);
 FileName:=Trim(FileName);
 if (DetectSectionHeader(Section,'')>0) then
 if (FileName<>'') and FileExists(FileName) then
 try
  ioMode:=ioModeAsIs;
  if (FRecursion=0) then FTotalList.Clear;
  if ((efMode and efAsAnsi)<>0) then ioMode:=ioMode or ioModeAnsi;
  Lines:=FileReadAsText(FileName,ioMode);
  Result:=ExtractSectionFromText(Lines,Section,efMode);
  Sect:=ExtractSectionFromText(Lines,'[ConfigFileList]',efConfigNC);
  List:=TStringList.Create;
  try
   List.Text:=Sect;
   for i:=0 to List.Count-1 do begin
    if GetVarValue(List.Strings[i],'ConfigFile',S) then begin
     FN:=AdaptFileName(S);
     if (FN<>'') then
     if (Pos(':\',FN)=0) then
     if (Pos('\\',FN)<>1) then begin
      if (Pos('~\',FN)=1) or (Pos('~/',FN)=1) then begin
       UserProfile:=GetEnv('USERPROFILE');
       if (UserProfile<>'') and DirectoryExists(UserProfile)
       then FN:=AddPathDelim(UserProfile)+Copy(FN,2,Length(FN)-1)
       else FN:='';
      end else
      if (Pos('~~\',FN)=1) or (Pos('~~/',FN)=1) then begin
       HomeDir:=GetEnv('CRW_DAQ_SYS_HOME_DIR');
       if (HomeDir<>'') and DirectoryExists(HomeDir)
       then FN:=AddPathDelim(HomeDir)+Copy(FN,3,Length(FN)-2)
       else FN:='';
      end else
      FN:=AddPathDelim(ExtractFileDir(FileName))+FN;
      FN:=ExpandFileName(FN);
     end;
     if (FN<>'') and FileExists(FN) then
     if (FTotalList.IndexOf(FN)<0) then begin
      FTotalList.Add(FN);
      try
       Inc(FRecursion);
       Result:=Result+ExtractSectionFromFile(FN,Section,efMode);
      finally
       Dec(FRecursion);
       if FTotalList.IndexOf(FN)>=0 then
       FTotalList.Delete(FTotalList.IndexOf(FN));
      end;
     end;
    end;
   end;
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromFile');
   Result:='';
  end;
 end;
end;

function TDaqConfig.ReadIniStr(FileName,Section,Key:String; Mode:Integer=efConfigNC):String;
var Lines:TStringList; i:Integer; Name,Value:String;
begin
 Result:='';
 FileName:=Trim(FileName); if IsEmptyStr(FileName) then Exit;
 Section:=Trim(Section); if IsEmptyStr(Section) then Exit;
 Key:=Trim(Key); if IsEmptyStr(Key) then Exit;
 if FileExists(FileName) then
 if (DaqConfig.DetectSectionHeader(Section,Section)>0) then
 try
  Lines:=TStringList.Create;
  try
   Lines.Text:=DaqConfig.ExtractSectionFromFile(FileName,Section,Mode);
   for i:=0 to Lines.Count-1 do
   if (ExtractNameValuePair(Lines.Strings[i],Name,Value)>0) then
   if (Name<>'') and IsSameText(Name,Key) then begin
    Result:=Value;
    Exit;
   end;
  finally
   Lines.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'ReadIniStr');
 end;
end;


///////////////////////
// Well known sections.
///////////////////////

function OsSuffix(Mode:Integer):String;
begin
 Result:='';
 if (Mode>0) then Result:=GetOSFamily;
 if (Result<>'') then Result:=':'+Result;
end;

function SectName(const Name:String; Mode:Integer):String;
begin
 Result:='['+Trim(Name)+OsSuffix(Mode)+']';
end;

function SectByName(const Name:String; Mode:Integer=0):String;
begin
 Result:=SectName(Name,Mode);
end;

function SectSystem(Mode:Integer=0):String;
begin
 Result:=SectName('System',Mode);
end;

function SectSystemStartupScript(Mode:Integer=0):String;
begin
 Result:=SectName('System.StartupScript',Mode);
end;

function SectDaqSys(Mode:Integer=0):String;
begin
 Result:=SectName('DaqSys',Mode);
end;

function SectDaq(Mode:Integer=0):String;
begin
 Result:=SectName('DAQ',Mode);
end;

function SectSysVoice(Mode:Integer=0):String;
begin
 Result:=SectName('SysVoice',Mode);
end;

function SectAtRun(Mode:Integer=0):String;
begin
 Result:=SectName('@run',Mode);
end;

function SectActionList(Mode:Integer=0):String;
begin
 Result:=SectName('ActionList',Mode);
end;

function SectHandlerList(Mode:Integer=0):String;
begin
 Result:=SectName('HandlerList',Mode);
end;

function SectI18nCodesIso3166Csv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/iso3166.csv',Mode);
end;

function SectI18nCodesRuWorldCsv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/ru/world.csv',Mode);
end;

function SectI18nCodesCca2TldCsv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/cca2tld.csv',Mode);
end;

function SectI18nCodesIso639v1Csv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/iso639-1.csv',Mode);
end;

function SectI18nCodesIso639v2Csv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/iso639-2.csv',Mode);
end;

function SectI18nCodesGost7p75v97Csv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/gost7.75-97.csv',Mode);
end;

function SectI18nCodesMsLcidCsv(Mode:Integer=0):String;
begin
 Result:=SectName('i18n/codes/ms-lcid.csv',Mode);
end;

function SectTextEditorsPalette(Mode:Integer=0):String;
begin
 Result:=SectName('Text Editors Palette',Mode);
end;

function SectConsoleUtilities(Mode:Integer=0):String;
begin
 Result:=SectName('ConsoleUtilities',Mode);
end;

initialization

 DaqConfig;

end.
