 {
 ***********************************************************************
 Daq Pascal application program uniheater_ctrl.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
|********************************************************
[]
 ***********************************************************************
 Программа управления нагревателями (печами) UniHeater (Unified Heaters).
 Подсистема UniHeater обслуживает от 1 до 32 каналов нагревателей, может
 использоваться для термостабилизации и управляемого нагрева по графику.
 Конфигурирование и описание: (здесь i-номер нагревателя, i=1..32):
 1)  Имя устройства имеет вид &XXXX.UNIHEATER.CTRL, где XXXX - общее имя
     подсистемы UniHeater, например, DEMO или TEST.UH (точки допустимы).
 2)  Все имена тегов, кривых, окон имеют вид XXXX.UNIHEATER.PPP, который
     имеет тот же префикс XXXX.UNIHEATER и берется из имени устройства.
     Здесь PPP - имя конкретного параметра, например T_i, GATE_i и т.д.
 3)  Имена стандартных окон (winCtrl,winSettings):
     XXXX.UNIHEATER.CTRL     - общая контрольная панель UniHeater,
     XXXX.UNIHEATER.SETTINGS - диалог настройки свойств одного канала.
 4)  К входам AnalogInput(1..32)  подключаются каналы температур T_i.
 5)  К входам DigitalInput(1..32) подключаются каналы давлений   P_i.
 6)  На выходы AnalogOutput(1..32) выдается вычисленная мощность QX_i
     (точнее, коэффициент заполнения меандра в %) для подачи на вход
     генератора  управляющего меандра импульсов скважности GATE_i
     (программа uniheater_pulser.pas).
 7)  На выходы DigitalOutput(1..32) выдаются сигналы блокировки BLK_i.
 8)  Число нагревателей определяется по числу тегов его имени HNAME_i.
     Следите за соответствием числа нагревателей и указанных тегов.
 9)  Блокировка делается по:
     a) уровню давления    - порог в теге PBLK_i,
     b) уровню температуры - порог в теге TBLK_i,
     c) сторожу WDT_i, чтобы сигнализировать обрыв связи.
     d) внешней блокировки EBLK_i, за неё отвечает внешняя программа.
     Обрыв связи регистрируется по метке времени последнего обновления
     давления и температуры (Watchdog - сторожевой таймер WDT_i секунд).
     Значение WDT_i=0 означает отмену WatchDog - контроля.
     Блокировка заключается в записи флага (1) блокировки в тег BLK_i,
     а также в кривую DigitalOutput(i) для подачи на реле блокировки.
     Если к входу не подключено давление или температура, блокировка
     по ним не делается (выдается значение 0).
     При блокировке периодически выдается сообщение "Блокировка i".
 10) Алгоритм вычисления мощности qx:
     if T<T0-dt1 then qx:=Q1 else
     if T<T0     then qx:=Q2 else
     if T<T0+dt2 then qx:=Q3 else qx:=0;
 11) Программа использует номера датчиков давления и температуры
     в тегах TNUM_i, PNUM_i, (см. программу uniheater_setect.pas).
     Также в диалогах используются имена каналов TNAME_i, PNAME_i,
     которые вычисляются по списку имен датчиков TNAMES, PNAMES вида
     T1/T2/T3/... (разделитель - слеш). При этом длина названия для
     каждого датчика должно быть не более 4 символов, например TK12.
     Следите за корректностью списка имен датчиков  TNAMES, PNAMES.
 12) При обнаружении сигнала ICTRL_i>0 при включенной кнопке печи
     регистрируется обрыв цепи и выдается периодическое сообщение
     "Разрыв i".
 13) На сигналы блокировки влияют флаги для разрешения этих блокировок:
     TENABL_i, PENABL_i, WENABL_i. Если блокировки не разрешены (0), то
     сигналы блокировки не возникают (равны 0).
 14) Канал нагревателя можно запретить флагом HENABL_i=0.
     В этом случае всегда выдается BT_i=0, BLK_i=1 и QX_i=0.
 15) 20030202-Alexey Kuryakin
     Теперь блокировка (кривая BLK_i) включается при выключении нагрева.
     Это надо чтобы охранное блокирующее реле отключалось при выключении
     нагрева, что важно, например, при пробое твердотельного реле.
     Однако теперь кривая BLK_i и тег BLK_i уже не совпадают.
     Кривая используется для управления, а тег для индикации.
 16) Программа для управления нагревателями UniHeater также отрабатывает
     нажатие элементов пользовательского интерфейса: кнопок, полей ввода
     в окнах UNIHEATER.CTRL и UNIHEATER.SETTINGS.
 ***********************************************************************
 20000903 Alexey Kuryakin
 20131012 Alexey Kuryakin
 20211215 Alexey Kuryakin
 ***********************************************************************
 }
{
[Compiler.Options]
Compiler.dtabmax = 1024*32  ; Data segment
Compiler.stabmax = 1024*8   ; String table
Compiler.dtabmin = 1024*1   ; Min stack  space
Compiler.stabmin = 512*1    ; Min string space
[]
}
program uniheater_ctrl;          { UniHeater control                }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 snd_BLK         = 'Блокиров';   { Sound on blocking                }
 snd_ICTRL       = 'Обрыв';      { Sound on wire break              }
 spk_BLK         = 'Блокировка печи';   { Speaker message           }
 spk_ICTRL       = 'Обрыв печи';        { Speaker message           }
 MaxHeater       = 32;           { Max. number of heaters           }
 MsgPeriod       = 30;           { Period for sound messages        }
 NamesPeriod     = 100;          { Period to update sensor names    }
 SensorDelims    = '/|\';        { Delimeters for sensor list       }
 UseSensorMenu   = True;         { Use menu to select sensors       }
 mode_Heat       = 0;            { Use heating mode                 }
 mode_Cool       = 1;            { Use cooling mode                 }
 sb_BT           = 1;            { StatusByte - BT     state        }
 sb_GATE         = 2;            { StatusByte - GATE   signal       }
 sb_SBLK         = 4;            { StatusByte - BLK    state        }
 sb_BLK          = 8;            { StatusByte - BLK    signal       }
 sb_EBLK         = 16;           { StatusByte - EBLK   state        }
 sb_ICTRL        = 32;           { StatusByte - ICTRL  state        }
 sb_TWDT         = 64;           { StatusByte - T WDT  state        }
 sb_PWDT         = 128;          { StatusByte - P WDT  state        }
 sb_TBLK         = 256;          { StatusByte - T>TBLK state        }
 sb_PBLK         = 512;          { StatusByte - P>PBLK state        }
 sb_QXON         = 1024;         { StatusByte - QX>0   state        }
 sb_NETW         = 2048;         { StatusByte - NETW   state        }
 DimDeadline     = 5000;         { Detect DIM server is dead        }
 UsesStatSums    = true;         { Uses stat. sums for mean/rmsd    }

type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }
 TUniTagRefs     = array [0..MaxHeater] of TTagRef; { Array of Tags }
 {------ N.B. Fake Heater[0] uses just for settings editing only!   }
 TUniStatSums    = array [0..MaxHeater] of TStatSum2D;

var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 UH              : record        { All UniHeater data               }
  NumHeaters     : Integer;      { Number of heaters                }
  HNAME          : TUniTagRefs;  { HNAME_i  Heater name like HEAT_1 }
  TNAME          : TUniTagRefs;  { TNAME_i  T sensor name           }
  PNAME          : TUniTagRefs;  { PNAME_i  P sensor name           }
  TNUM           : TUniTagRefs;  { T_i      T Sensor number, 1..32  }
  PNUM           : TUniTagRefs;  { P_i      P Sensor number, 1..32  }
  MODE           : TUniTagRefs;  { MODE_i   0/1=Heat/Cool mode      }
  CONF           : TUniTagRefs;  { CONF_i   Confirmation flags      }
  HENABL         : TUniTagRefs;  { HENABL_i Enable Heater i         }
  TENABL         : TUniTagRefs;  { TENABL_i Enable blocking by T_i  }
  PENABL         : TUniTagRefs;  { PENABL_i Enable blocking by P_i  }
  WENABL         : TUniTagRefs;  { WENABL_i Enable Watchdog: WDT_i  }
  PERIOD         : TUniTagRefs;  { PERIOD_i Period of meander, ms   }
  TGOAL          : TUniTagRefs;  { TGOAL_i  Goal T value            }
  TREAL          : TUniTagRefs;  { TREAL_i  Real T value            }
  PREAL          : TUniTagRefs;  { PREAL_i  Real P value            }
  GATE           : TUniTagRefs;  { GATE_i   Meander GATE signal     }
  BLK            : TUniTagRefs;  { BLK_i    Signal of Blocking      }
  BT             : TUniTagRefs;  { BT_i     Button on/off Heater i  }
  ST             : TUniTagRefs;  { ST_i     State 0..6 of Heater i  }
  SB             : TUniTagRefs;  { SB_i     StatusByte of Heater i  }
  QX             : TUniTagRefs;  { QX_i     QX, duty factor, %      }
  Q1             : TUniTagRefs;  { Q1_i     Q1, starting QX value   }
  Q2             : TUniTagRefs;  { Q2_i     Q2, underheat           }
  Q3             : TUniTagRefs;  { Q3_i     Q3, overheat            }
  DT1            : TUniTagRefs;  { DT1_i    dT1 underheat           }
  DT2            : TUniTagRefs;  { DT2_i    dT2 overheat            }
  WDT            : TUniTagRefs;  { WDT_i    Watchdog, sec           }
  PBLK           : TUniTagRefs;  { PBLK_i   Hi limit for P blocking }
  TBLK           : TUniTagRefs;  { TBLK_i   Hi limit for T blocking }
  EBLK           : TUniTagRefs;  { EBLK_i   Extern blocking signal  }
  ICTRL          : TUniTagRefs;  { ICTRL_i  Flag of Wire break      }
  QMEAN          : TUniTagRefs;  { QMEAN_i  QMEAN, average power, % }
  TMEAN          : TUniTagRefs;  { TMEAN_i  TMEAN, average temperat }
  TRMSD          : TUniTagRefs;  { TRMSD_i  TRMSD, T RMS deviation  }
  TSTAT          : TUniTagRefs;  { TSTAT_i  TSTAT time of statistic }
  TNAMES         : TTagRef;      { List of T sensor names T1/T2/..  }
  PNAMES         : TTagRef;      { List of P sensor names P1/P2/..  }
  SOUNDS         : TTagRef;      { Sounds on/off                    }
  SERVID         : TTagRef;      { Server Identifier pid@hostname   }
  CLOCK          : TTagRef;      { Server Date-Time                 }
  tStart         : Real;         { Start time                       }
  SelfId         : String;       { Self pid@hostname                }
  STAT_TREAL     : TUniStatSums; { Statistics for TREAL             }
 end;
 EditingHNUM     : Integer;      { Current editing heater number    }
 UsesSpeakSrv    : Boolean;      { Uses SpeakSrv server or not      }
 ColorNorm       : Integer;      { Color in normal state: lime,aqua }
 ColorWarn       : Integer;      { Color in warning state: yellow   }
 cmd_DimTagUpdate     : Integer; { @DimTagUpdate                    }
 cmd_AssignTag        : Integer; { @AssignTag                       }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_StdStatSum}            { Add Standard Stat.Sum routines.  }

 {
 Get tag prefix by device name.
 }
 function tagPrefix:String;
 begin
  tagPrefix:=ExtractFileName(Copy(DevName,2));
 end;
 {
 Get Control window name by device name.
 }
 function winCtrl:String;
 begin
  winCtrl:=Copy(DevName,2);
 end;
 {
 Get Settings window name by device name.
 }
 function winSettings:String;
 begin
  winSettings:=ForceExtension(Copy(DevName,2),'.SETTINGS');
 end;
 {
 Get UniHeater full sensor name.
 }
 function UniSenName(Name:String; i:Integer):String;
 begin
  if (i>=0) then Name:=Name+Str(i);
  Name:=tagPrefix+'.'+Name;
  UniSenName:=Name;
 end;
 {
 Get UniHeater full tag name with prefix.
 }
 function UniTagName(Name:String; i:Integer):String;
 begin
  if (i>=0) then Name:=Name+Str(i);
  Name:=tagPrefix+'.'+Name;
  UniTagName:=Name;
 end;
 {
 Identifier for YesNo dialog with tagMode.
 }
 function uidYesNoMode:String;
 begin
  uidYesNoMode:=EditGetUID('YesNo:'+NameTag(UH.MODE[0].tag));
 end;
 {
 Identifier for YesNo dialog with given tag.
 }
 function uidYesNoTag(tag:Integer):String;
 begin
  uidYesNoTag:=EditGetUID('YesNo:'+NameTag(tag));
 end;
 {
 Identifier for YesNo dialog to SaveIni/LoadIni.
 }
 function uidYesNoSaveIni:String;
 begin
  uidYesNoSaveIni:=EditGetUID('YesNo:@SaveIni');
 end;
 function uidYesNoLoadIni:String;
 begin
  uidYesNoLoadIni:=EditGetUID('YesNo:@LoadIni');
 end;
 {
 Check if operation (n) need confirmation for Heater[i].
 The n=0..31 is operation code:
 0 : Heat ON
 1 : Heat OFF
 }
 function NeedConfirm(i,n:Integer):Boolean;
 begin
  if (i>=0) and (i<=MaxHeater) and (n>=0) and (n<32)
  then NeedConfirm:=IsBit(iGetTag(UH.CONF[i].tag),n)
  else NeedConfirm:=false;
 end;
 {
 Warning tooltip notifier.
 }
 procedure WarnNotify(aText:String; aDelay:Integer);
 begin
  if (Length(aText)>0) then begin
   ShowTooltip('guid '+Str(getpid)+'@'+ProgName+' text "UniHeater: '+aText+'" preset stdWarning delay '+Str(aDelay));
  end;
 end;
 {
 Procedure to show sensor help.
 }
 procedure SensorHelp(s:String);
 begin
  if (Length(s)>0) then begin
   ShowTooltip('guid '+Str(getpid)+'@'+ProgName+' text "'+s+'" preset stdHelp delay 15000'
   +' btn1 Справка cmd1 "cmd /c start '+fexpand(ProgramSourceDir+'\uniheater.htm')+'"');
   if (Pos(':',s)>0) then s:=Trim(Copy(s,Pos(':',s)+1));
   if (Pos('(',s)>0) then s:=Trim(Copy(s,1,Pos('(',s)-1));
   if (iGetTag(UH.SOUNDS.tag)<>0) and (s<>'') then Speak(s);
  end;
 end;
 {
 Check if heater number n is valid to use.
 }
 function ValidHNUM(n:Integer):Boolean;
 begin
  ValidHNUM:=((n>=1) and (n<=UH.NumHeaters));
 end;
 {
 State of Settings editing.
 }
 function IsEditingSettings:Boolean;
 begin
  IsEditingSettings:=ValidHNUM(EditingHNUM);
 end;
 {
 Get number of sensors in list like T1/T2/T3/...
 }
 function GetNumSensors(SensorNames:String):Integer;
 begin
  GetNumSensors:=WordCountDelims(SensorNames,WordDelims('')+SensorDelims);
 end;
 {
 Get sensors name from list like T1/T2/T3/.. by index n.
 }
 function GetSensorName(n:Integer; SensorNames:String):String;
 begin
  GetSensorName:=ExtractWordDelims(n,SensorNames,WordDelims('')+SensorDelims);
 end;
 {
 Update T,P sensor names.
 }
 procedure UpdateSensorNames;
 var i:Integer; st,sp,tnames,pnames:String; ies:Boolean;
 begin
  st:=''; sp:=''; tnames:=''; pnames:='';
  tnames:=sGetTag(UH.TNAMES.tag);
  pnames:=sGetTag(UH.PNAMES.tag); ies:=IsEditingSettings;
  for i:=0 to UH.NumHeaters do if (i>0) or ies then begin
   st:=GetSensorName(iGetTag(UH.TNUM[i].tag),tnames);
   sp:=GetSensorName(iGetTag(UH.PNUM[i].tag),pnames);
   if (sGetTag(UH.TNAME[i].tag)<>st) then bNul(sSetTag(UH.TNAME[i].tag,st));
   if (sGetTag(UH.PNAME[i].tag)<>sp) then bNul(sSetTag(UH.PNAME[i].tag,sp));
  end;
  st:=''; sp:=''; tnames:=''; pnames:='';
 end;
 {
 Prefix for DIM @remote commands.
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 {
 Compose command for local/remote assign tag.
 }
 function CmndLocalAssignTag(tag:Integer; NewValue:String):String;
 begin
  if (TypeTag(tag)>0) and (NewValue<>'')
  then CmndLocalAssignTag:='@AssignTag '+NameTag(tag)+' '+NewValue
  else CmndLocalAssignTag:='';
 end;
 function CmndRemoteAssignTag(tag:Integer; NewValue:String):String;
 begin
  if (TypeTag(tag)>0) and (NewValue<>'')
  then CmndRemoteAssignTag:=DimRemote+'@AssignTag '+NameTag(tag)+' '+NewValue
  else CmndRemoteAssignTag:='';
 end;
 {
 Post @Remote @AssignTag Name(tag) NewValue.
 Post @AssignTag Name(tag) NewValue.
 }
 procedure PostRemoteAssignTag(tag:Integer; NewValue:String);
 begin
  if (TypeTag(tag)>0) and (NewValue<>'') then
  DevPostCmdLocal(CmndRemoteAssignTag(tag,NewValue));
 end;
 procedure PostLocalAssignTag(tag:Integer; NewValue:String);
 begin
  if (TypeTag(tag)>0) and (NewValue<>'') then
  DevPostCmdLocal(CmndLocalAssignTag(tag,NewValue));
 end;
 {
 Check tag editing. If done, post @Remote @AssignTag NewValue.
 }
 procedure CheckEditTagRemote(tag:Integer; n:Integer);
 var s:String;
 begin
  s:='';
  if CheckEditTag(tag,s) then begin
   if (n>0)
   then PostRemoteAssignTag(tag,s)
   else PostLocalAssignTag(tag,s);
  end;
  s:='';
 end;
 {
 Copy Settings: copy all UniHeater x[n] tags to x[m] tags.
 }
 procedure CopySettings(n,m:Integer);
 begin
  if (m<>n) then
  if (m>=0) and (m<=MaxHeater) then
  if (n>=0) and (n<=MaxHeater) then begin
   bNul(sSetTag(UH.HNAME[m].tag,  sGetTag(UH.HNAME[n].tag)));
   bNul(sSetTag(UH.TNAME[m].tag,  sGetTag(UH.TNAME[n].tag)));
   bNul(sSetTag(UH.PNAME[m].tag,  sGetTag(UH.PNAME[n].tag)));
   bNul(iSetTag(UH.TNUM[m].tag,   iGetTag(UH.TNUM[n].tag)));
   bNul(iSetTag(UH.PNUM[m].tag,   iGetTag(UH.PNUM[n].tag)));
   bNul(iSetTag(UH.MODE[m].tag,   iGetTag(UH.MODE[n].tag)));
   bNul(iSetTag(UH.CONF[m].tag,   iGetTag(UH.CONF[n].tag)));
   bNul(iSetTag(UH.HENABL[m].tag, iGetTag(UH.HENABL[n].tag)));
   bNul(iSetTag(UH.TENABL[m].tag, iGetTag(UH.TENABL[n].tag)));
   bNul(iSetTag(UH.PENABL[m].tag, iGetTag(UH.PENABL[n].tag)));
   bNul(iSetTag(UH.WENABL[m].tag, iGetTag(UH.WENABL[n].tag)));
   bNul(rSetTag(UH.PERIOD[m].tag, rGetTag(UH.PERIOD[n].tag)));
   bNul(rSetTag(UH.TGOAL[m].tag,  rGetTag(UH.TGOAL[n].tag)));
   bNul(rSetTag(UH.TREAL[m].tag,  rGetTag(UH.TREAL[n].tag)));
   bNul(rSetTag(UH.PREAL[m].tag,  rGetTag(UH.PREAL[n].tag)));
   bNul(iSetTag(UH.GATE[m].tag,   iGetTag(UH.GATE[n].tag)));
   bNul(iSetTag(UH.BLK[m].tag,    iGetTag(UH.BLK[n].tag)));
   bNul(iSetTag(UH.BT[m].tag,     iGetTag(UH.BT[n].tag)));
   bNul(iSetTag(UH.ST[m].tag,     iGetTag(UH.ST[n].tag)));
   bNul(iSetTag(UH.SB[m].tag,     iGetTag(UH.SB[n].tag)));
   bNul(rSetTag(UH.QX[m].tag,     rGetTag(UH.QX[n].tag)));
   bNul(rSetTag(UH.Q1[m].tag,     rGetTag(UH.Q1[n].tag)));
   bNul(rSetTag(UH.Q2[m].tag,     rGetTag(UH.Q2[n].tag)));
   bNul(rSetTag(UH.Q3[m].tag,     rGetTag(UH.Q3[n].tag)));
   bNul(rSetTag(UH.DT1[m].tag,    rGetTag(UH.DT1[n].tag)));
   bNul(rSetTag(UH.DT2[m].tag,    rGetTag(UH.DT2[n].tag)));
   bNul(rSetTag(UH.WDT[m].tag,    rGetTag(UH.WDT[n].tag)));
   bNul(rSetTag(UH.PBLK[m].tag,   rGetTag(UH.PBLK[n].tag)));
   bNul(rSetTag(UH.TBLK[m].tag,   rGetTag(UH.TBLK[n].tag)));
   bNul(iSetTag(UH.EBLK[m].tag,   iGetTag(UH.EBLK[n].tag)));
   bNul(iSetTag(UH.ICTRL[m].tag,  iGetTag(UH.ICTRL[n].tag)));
   bNul(rSetTag(UH.QMEAN[m].tag,  rGetTag(UH.QMEAN[n].tag)));
   bNul(rSetTag(UH.TMEAN[m].tag,  rGetTag(UH.TMEAN[n].tag)));
   bNul(rSetTag(UH.TRMSD[m].tag,  rGetTag(UH.TRMSD[n].tag)));
   bNul(rSetTag(UH.TSTAT[m].tag,  rGetTag(UH.TSTAT[n].tag)));
  end;
 end;
 {
 Zero Settings: zero all UniHeater x[n] tags.
 }
 procedure ZeroSettings(n:Integer);
 begin
  if (n>=0) and (n<=MaxHeater) then begin
   bNul(sSetTag(UH.HNAME[n].tag, ''));
   bNul(sSetTag(UH.TNAME[n].tag, ''));
   bNul(sSetTag(UH.PNAME[n].tag, ''));
   bNul(iSetTag(UH.TNUM[n].tag,   0));
   bNul(iSetTag(UH.PNUM[n].tag,   0));
   bNul(iSetTag(UH.MODE[n].tag,   0));
   bNul(iSetTag(UH.CONF[n].tag,   0));
   bNul(iSetTag(UH.HENABL[n].tag, 0));
   bNul(iSetTag(UH.TENABL[n].tag, 0));
   bNul(iSetTag(UH.PENABL[n].tag, 0));
   bNul(iSetTag(UH.WENABL[n].tag, 0));
   bNul(rSetTag(UH.PERIOD[n].tag, 0));
   bNul(rSetTag(UH.TGOAL[n].tag,  0));
   bNul(rSetTag(UH.TREAL[n].tag,  0));
   bNul(rSetTag(UH.PREAL[n].tag,  0));
   bNul(iSetTag(UH.GATE[n].tag,   0));
   bNul(iSetTag(UH.BLK[n].tag,    0));
   bNul(iSetTag(UH.BT[n].tag,     0));
   bNul(iSetTag(UH.ST[n].tag,     0));
   bNul(iSetTag(UH.SB[n].tag,     0));
   bNul(rSetTag(UH.QX[n].tag,     0));
   bNul(rSetTag(UH.Q1[n].tag,     0));
   bNul(rSetTag(UH.Q2[n].tag,     0));
   bNul(rSetTag(UH.Q3[n].tag,     0));
   bNul(rSetTag(UH.DT1[n].tag,    0));
   bNul(rSetTag(UH.DT2[n].tag,    0));
   bNul(rSetTag(UH.WDT[n].tag,    0));
   bNul(rSetTag(UH.TBLK[n].tag,   0));
   bNul(rSetTag(UH.PBLK[n].tag,   0));
   bNul(iSetTag(UH.EBLK[n].tag,   0));
   bNul(iSetTag(UH.ICTRL[n].tag,  0));
   bNul(rSetTag(UH.QMEAN[n].tag,  0));
   bNul(rSetTag(UH.TMEAN[n].tag,  0));
   bNul(rSetTag(UH.TRMSD[n].tag,  0));
   bNul(rSetTag(UH.TSTAT[n].tag,  0));
  end;
 end;
 {
 Send Settings to remote host (after editing).
 }
 procedure SendSettings(n:Integer);
  procedure SendTag(tag:Integer);
  begin
   if (TypeTag(tag)=1) then PostRemoteAssignTag(tag,Str(iGetTag(tag)));
   if (TypeTag(tag)=2) then PostRemoteAssignTag(tag,Str(rGetTag(tag)));
  end;
 begin
  if DIM_IsClientMode then begin
   if ValidHNUM(n) then begin
    SendTag(UH.TNUM[n].tag);
    SendTag(UH.PNUM[n].tag);
    SendTag(UH.MODE[n].tag);
    SendTag(UH.CONF[n].tag);
    SendTag(UH.HENABL[n].tag);
    SendTag(UH.TENABL[n].tag);
    SendTag(UH.PENABL[n].tag);
    SendTag(UH.WENABL[n].tag);
    SendTag(UH.PERIOD[n].tag);
    SendTag(UH.TGOAL[n].tag);
    SendTag(UH.BT[n].tag);
    SendTag(UH.Q1[n].tag);
    SendTag(UH.Q2[n].tag);
    SendTag(UH.Q3[n].tag);
    SendTag(UH.DT1[n].tag);
    SendTag(UH.DT2[n].tag);
    SendTag(UH.WDT[n].tag);
    SendTag(UH.TBLK[n].tag);
    SendTag(UH.PBLK[n].tag);
    SendTag(UH.TSTAT[n].tag);
   end;
   SendTag(UH.SOUNDS.tag);
  end;
 end;
 {
 Close SETTINGS window and zero all Settings in x[0] tags.
 }
 procedure CloseSettings;
 begin
  EditingHNUM:=0;
  ZeroSettings(0);
  bNul(WinHide(winSettings));
 end;
 {
 Start edit SETTINGS: copy x[n] tags to x[0] tags. Then open SETTINGS editor window.
 }
 procedure StartEditSettings(n:Integer);
 begin
  if ValidHNUM(n) then begin
   if (ClickButton<>0) then WinSelectUnderClickSensor(winSettings,ClickParams(''),'-Left,-Top');
   bNul(WinSelect(winSettings));
   CopySettings(n,0);
   EditingHNUM:=n;
  end;
 end;
 {
 Stop edit SETTINGS: accept changes and clear flag.
 }
 procedure StopEditSettings;
 begin
  if IsEditingSettings then begin
   CopySettings(0,EditingHNUM);
   SendSettings(EditingHNUM);
   CloseSettings;
  end;
 end;
 {
 Start menu list to select sensor.
 }
 procedure CheckStartSensorMenu(n,tag:Integer; SensorNames,Id:String);
 var i,ns,m:Integer;
 begin
  if (ClickTag=tag) then if ValidHNUM(n) then if (TypeTag(tag)=1) then
  if (iGetTag(UH.HENABL[n].tag)=0) then begin
   WarnNotify('Нагреватель '+sGetTag(UH.HNAME[n].tag)+' выключен',10000);
   bNul(Voice(snd_Error));
  end else begin
   ns:=GetNumSensors(SensorNames);
   if (ns>0) then
   if ValidHNUM(n) then begin
    if EditStateReady then begin
     m:=0+EditAddOpening(StrFmt('Выбор датчика %s...',Id));
     m:=m+EditAddInputLn(StrFmt('Датчик %s',Id)+StrFmt(' нагревателя %d:',n));
     for i:=1 to ns do
     m:=m+EditAddInputLn(StrFmt('Датчик %-3d  ',i)+GetSensorName(i,SensorNames));
     m:=m+EditAddSetting('@set ListBox.Font Size:14\Style:[Bold]');
     m:=m+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
     m:=m+EditAddClosing('MenuList',EditGetUID('MENU_'+NameTag(tag)),Str(iGetTag(tag)-1));
     if (m>0) then Problem('Error initializing MenuList!');
    end else Problem('Cannot edit right now!');
   end;
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Check and accept sensor menu.
 }
 procedure CheckEditSensor(tag:Integer; n:Integer);
 var ns:Integer;
 begin
  if (TypeTag(tag)=1) then
  if EditTestResultName(EditGetUID('MENU_'+NameTag(tag))) then begin
   if EditTestResultCode(mr_Ok) then begin
    ns:=EditGetMenuListSelectedIndex+1;
    if (n>0)
    then PostRemoteAssignTag(tag,Str(ns))
    else PostLocalAssignTag(tag,Str(ns));
   end;
   EditReset;
  end;
 end;
 {
 Check tag clicked, heater number n is Ok.
 If heater is enabled, start edit tag.
 }
 procedure CheckStartEditTag(n,tag:Integer; Comment:String);
 begin
  if (ClickTag=tag) then if ValidHNUM(n) then if (TypeTag(tag)>0) then
  if (iGetTag(UH.HENABL[n].tag)=0) then begin
   WarnNotify('Нагреватель '+sGetTag(UH.HNAME[n].tag)+' выключен',10000);
   bNul(Voice(snd_Error));
  end else begin
   StartEditTagEx(tag,Comment,SetFormUnderSensorLeftBottom(ClickParams('')));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Check start edit all tags of heater (n>0) or editing settings (n=0).
 }
 procedure CheckStartEditHeaterSettings(n:Integer);
 var m:Integer;
 begin
  m:=n;
  if (n=0) then n:=EditingHNUM;
  if ValidHNUM(n) then begin
   CheckStartEditTag(n,UH.HNAME[m].tag,StrFmt('Наименование нагревателя канала %d: ',n));
   CheckStartEditTag(n,UH.TGOAL[m].tag,StrFmt('Целевая температура стабилизации канала %d (°C): ',n));
   CheckStartEditTag(n,UH.Q1[m].tag,StrFmt('Мощность Q1 начального нагрева канала %d (%%): ',n));
   CheckStartEditTag(n,UH.Q2[m].tag,StrFmt('Мощность Q2 (нижняя) подогрева канала %d (%%): ',n));
   CheckStartEditTag(n,UH.Q3[m].tag,StrFmt('Мощность Q3 (верхняя) подогрева канала %d (%%): ',n));
   CheckStartEditTag(n,UH.DT1[m].tag,StrFmt('Коридор dT1 (нижний) подогрева канала %d (°C): ',n));
   CheckStartEditTag(n,UH.DT2[m].tag,StrFmt('Коридор dT2 (верхний) подогрева канала %d (°C): ',n));
   CheckStartEditTag(n,UH.TBLK[m].tag,StrFmt('Температура блокировки канала %d (°C): ',n));
   CheckStartEditTag(n,UH.PBLK[m].tag,StrFmt('Давление блокировки канала %d: ',n));
   CheckStartEditTag(n,UH.WDT[m].tag,StrFmt('Сторожевой таймер канала %d (сек): ',n));
   CheckStartEditTag(n,UH.PERIOD[m].tag,StrFmt('Период меандра канала %d (мсек): ',n));
   if UseSensorMenu then begin
    CheckStartSensorMenu(n,UH.PNUM[m].tag,sGetTag(UH.PNAMES.tag),'давления');
    CheckStartSensorMenu(n,UH.TNUM[m].tag,sGetTag(UH.TNAMES.tag),'температуры');
   end else begin
    CheckStartEditTag(n,UH.PNUM[m].tag,StrFmt('Номер датчика давления канала %d: ',n));
    CheckStartEditTag(n,UH.TNUM[m].tag,StrFmt('Номер датчика температуры канала %d: ',n));
   end;
   CheckStartEditTag(n,UH.TSTAT[m].tag,StrFmt('Время накопления статистики канала %d (сек): ',n));
  end;
 end;
 {
 UniHeater: Clear tags.
 }
 procedure UniHeater_ClearTags;
 var i:Integer;
 begin
  for i:=0 to MaxHeater do begin
   UH.HNAME[i].tag:=0;
   UH.TNAME[i].tag:=0;
   UH.PNAME[i].tag:=0;
   UH.TNUM[i].tag:=0;
   UH.PNUM[i].tag:=0;
   UH.MODE[i].tag:=0;
   UH.CONF[i].tag:=0;
   UH.HENABL[i].tag:=0;
   UH.TENABL[i].tag:=0;
   UH.PENABL[i].tag:=0;
   UH.WENABL[i].tag:=0;
   UH.PERIOD[i].tag:=0;
   UH.TGOAL[i].tag:=0;
   UH.TREAL[i].tag:=0;
   UH.PREAL[i].tag:=0;
   UH.GATE[i].tag:=0;
   UH.BLK[i].tag:=0;
   UH.BT[i].tag:=0;
   UH.ST[i].tag:=0;
   UH.SB[i].tag:=0;
   UH.QX[i].tag:=0;
   UH.Q1[i].tag:=0;
   UH.Q2[i].tag:=0;
   UH.Q3[i].tag:=0;
   UH.DT1[i].tag:=0;
   UH.DT2[i].tag:=0;
   UH.WDT[i].tag:=0;
   UH.TBLK[i].tag:=0;
   UH.PBLK[i].tag:=0;
   UH.EBLK[i].tag:=0;
   UH.ICTRL[i].tag:=0;
   UH.QMEAN[i].tag:=0;
   UH.TMEAN[i].tag:=0;
   UH.TRMSD[i].tag:=0;
   UH.TSTAT[i].tag:=0;
   UH.BLK[i].tim:=0;
   UH.ICTRL[i].tim:=0;
   UH.TSTAT[i].tim:=0;
  end;
  UH.TNAMES.tag:=0;
  UH.PNAMES.tag:=0;
  UH.SOUNDS.tag:=0;
  UH.SERVID.tag:=0;
  UH.CLOCK.tag:=0;
  UH.SelfId:='';
 end;
 {
 UniHeater: Link tags with AI/DI/AO/DO.
 }
 procedure UniHeater_LinkTags;
 var i:Integer;
  procedure LinkTagRef(var R:TTagRef);
  begin
   R.nai:=-1; R.nao:=-1; R.ndi:=-1; R.ndo:=-1;
   if (R.tag<>0) then
   if (TypeTag(R.tag)>0) then
   if (TypeTag(R.tag)<3) then begin
    R.nai:=FindNaiByName(NameTag(R.tag));
    R.nao:=FindNaoByName(NameTag(R.tag));
    R.ndi:=FindNdiByName(NameTag(R.tag));
    R.ndo:=FindNdoByName(NameTag(R.tag));
    R.crv:=RefFind('Curve '+NameTag(R.tag));
   end;
  end;
 begin
  for i:=0 to MaxHeater do begin
   LinkTagRef(UH.HNAME[i]);
   LinkTagRef(UH.TNAME[i]);
   LinkTagRef(UH.PNAME[i]);
   LinkTagRef(UH.TNUM[i]);
   LinkTagRef(UH.PNUM[i]);
   LinkTagRef(UH.MODE[i]);
   LinkTagRef(UH.CONF[i]);
   LinkTagRef(UH.HENABL[i]);
   LinkTagRef(UH.TENABL[i]);
   LinkTagRef(UH.PENABL[i]);
   LinkTagRef(UH.WENABL[i]);
   LinkTagRef(UH.PERIOD[i]);
   LinkTagRef(UH.TGOAL[i]);
   LinkTagRef(UH.TREAL[i]);
   LinkTagRef(UH.PREAL[i]);
   LinkTagRef(UH.GATE[i]);
   LinkTagRef(UH.BLK[i]);
   LinkTagRef(UH.BT[i]);
   LinkTagRef(UH.ST[i]);
   LinkTagRef(UH.SB[i]);
   LinkTagRef(UH.QX[i]);
   LinkTagRef(UH.Q1[i]);
   LinkTagRef(UH.Q2[i]);
   LinkTagRef(UH.Q3[i]);
   LinkTagRef(UH.DT1[i]);
   LinkTagRef(UH.DT2[i]);
   LinkTagRef(UH.WDT[i]);
   LinkTagRef(UH.TBLK[i]);
   LinkTagRef(UH.PBLK[i]);
   LinkTagRef(UH.EBLK[i]);
   LinkTagRef(UH.ICTRL[i]);
   LinkTagRef(UH.QMEAN[i]);
   LinkTagRef(UH.TMEAN[i]);
   LinkTagRef(UH.TRMSD[i]);
   LinkTagRef(UH.TSTAT[i]);
  end;
  LinkTagRef(UH.TNAMES);
  LinkTagRef(UH.PNAMES);
  LinkTagRef(UH.SOUNDS);
  LinkTagRef(UH.SERVID);
  LinkTagRef(UH.CLOCK);
 end;
 {
 UniHeater: Assign Values.
 }
 procedure UniHeater_SetVals(v:Real);
 var i:Integer;
 begin
  for i:=0 to MaxHeater do begin
   UH.HNAME[i].val:=v;
   UH.TNAME[i].val:=v;
   UH.PNAME[i].val:=v;
   UH.TNUM[i].val:=v;
   UH.PNUM[i].val:=v;
   UH.MODE[i].val:=v;
   UH.CONF[i].val:=v;
   UH.HENABL[i].val:=v;
   UH.TENABL[i].val:=v;
   UH.PENABL[i].val:=v;
   UH.WENABL[i].val:=v;
   UH.PERIOD[i].val:=v;
   UH.TGOAL[i].val:=v;
   UH.TREAL[i].val:=v;
   UH.PREAL[i].val:=v;
   UH.GATE[i].val:=v;
   UH.BLK[i].val:=v;
   UH.BT[i].val:=v;
   UH.ST[i].val:=v;
   UH.SB[i].val:=v;
   UH.QX[i].val:=v;
   UH.Q1[i].val:=v;
   UH.Q2[i].val:=v;
   UH.Q3[i].val:=v;
   UH.DT1[i].val:=v;
   UH.DT2[i].val:=v;
   UH.WDT[i].val:=v;
   UH.TBLK[i].val:=v;
   UH.PBLK[i].val:=v;
   UH.EBLK[i].val:=v;
   UH.ICTRL[i].val:=v;
   UH.QMEAN[i].val:=v;
   UH.TMEAN[i].val:=v;
   UH.TRMSD[i].val:=v;
   UH.TSTAT[i].val:=v;
  end;
  UH.TNAMES.val:=v;
  UH.PNAMES.val:=v;
  UH.SOUNDS.val:=v;
  UH.SERVID.val:=v;
 end;
 {
 Find curve mean value on x range [ax,bx].
 }
 function crvEvalMean(crv:Integer; ax,bx:Real):Real;
 var mean:Real;
 begin
  mean:=0;
  if (crv<>0) and (CrvLen(crv)>0) and (bx>ax) then begin
   mean:=CrvInteg(crv,ax,bx)/(bx-ax);
  end;
  crvEvalMean:=mean;
 end;
 {
 Find curve RMS deviation value on x range [ax,bx].
 }
 function crvEvalRmsd(crv:Integer; ax,bx:Real; m:Integer):Real;
 var rmsd,x,y,sy,syy,sn:Real; i,n,ai,bi:Integer;
 begin
  rmsd:=0; sy:=0; syy:=0;
  if (crv<>0) and (CrvLen(crv)>0) and (bx>ax) then begin
   ai:=Round(crvWhere(crv,ax));
   bi:=Round(crvWhere(crv,bx));
   n:=imax(m,(bi-ai)*m);
   for i:=0 to n-1 do begin
    x:=ax+i*(bx-ax)/(n-1);
    y:=crvget(crv,x);
    syy:=syy+y*y;
    sy:=sy+y;
   end;
   rmsd:=sqrt(syy/n-sqr(sy/n));
   if IsNan(rmsd) then rmsd:=0;
   if IsInf(rmsd) then rmsd:=0;
  end;
  crvEvalRmsd:=rmsd;
 end;
 {
 UniHeter: Polling routine to calculate statistics.
 }
 procedure UniHeater_Poll_Stat;
 var i,crv,nai,ns:Integer; ms,dt,dx,perTSTAT,qmean,tmean,trmsd,mks,xn,yn:Real;
 begin
  // Add TREAL statistics sums
  if UsesStatSums then
  for i:=1 to UH.NumHeaters do begin
   nai:=UH.TREAL[i].nai;
   if (nai>=0) then begin
    xn:=GetAi_Xn(nai);
    if (ShouldRefresh(UH.STAT_TREAL[i].TimeStamp,xn)>0) then begin
     yn:=GetAi_Yn(nai);
     TStatSum2D_Add(UH.STAT_TREAL[i],xn,yn);
    end;
   end;
  end;
  // Calculate mean/rmsd
  ms:=mSecNow; mks:=0; ns:=0;
  for i:=1 to UH.NumHeaters do begin
   dt:=ms-UH.TSTAT[i].tim;
   perTSTAT:=rGetTag(UH.TSTAT[i].tag)*1000;
   if (perTSTAT>0) and (dt>perTSTAT) then begin
    if (UH.TSTAT[i].tim>0) then begin
     dx:=dt/TimeUnits;
     if DebugFlagEnabled(dfDetails) then ns:=ns+1;
     if (ns>0) and (mks=0) then mks:=mksecnow;
     qmean:=crvEvalMean(UH.GATE[i].crv,time-dx,time)*100;
     bNul(rSetTag(UH.QMEAN[i].tag,qmean)); UpdateAo(UH.QMEAN[i].nao,time,qmean);
     if UsesStatSums then TStatSum2D_Calculate(UH.STAT_TREAL[i],1);
     if UsesStatSums
     then tmean:=UH.STAT_TREAL[i].Mean.Y
     else tmean:=crvEvalMean(UH.TREAL[i].crv,time-dx,time);
     bNul(rSetTag(UH.TMEAN[i].tag,tmean)); UpdateAo(UH.TMEAN[i].nao,time,tmean);
     if UsesStatSums
     then trmsd:=UH.STAT_TREAL[i].Sigma.Y
     else trmsd:=crvEvalRmsd(UH.TREAL[i].crv,time-dx,time,5);
     bNul(rSetTag(UH.TRMSD[i].tag,trmsd)); UpdateAo(UH.TRMSD[i].nao,time,trmsd);
     if UsesStatSums then TStatSum2D_Reset(UH.STAT_TREAL[i]);
    end;
    UH.TSTAT[i].tim:=ms;
   end;
  end;
  if (ns>0) and (mks>0) then begin
   mks:=mksecnow-mks;
   Details(StrFmt('Stat %d items ',ns)+StrFmt('took %.5g mksec',mks));
  end;
 end;
 {
 UniHeater: Handle message @AssignTag arg.
 }
 procedure UniHeater_OnAssignTag(arg:String);
 var i,tag:Integer; w1,w2:String;
 begin
  w1:=''; w2:='';
  if (arg<>'') then begin
   w1:=ExtractWord(1,arg);
   tag:=FindTag(w1);
   if (tag<>0) then begin
    w2:=ExtractWord(2,arg);
    for i:=0 to UH.NumHeaters do begin
     if (tag=UH.HNAME[i].tag)  then UpdateTag(tag,w2,0,1);
     if (tag=UH.TNAME[i].tag)  then UpdateTag(tag,w2,0,1);
     if (tag=UH.PNAME[i].tag)  then UpdateTag(tag,w2,0,1);
     if (tag=UH.TNUM[i].tag)   then UpdateTag(tag,w2,1,GetNumSensors(sGetTag(UH.TNAMES.tag)));
     if (tag=UH.PNUM[i].tag)   then UpdateTag(tag,w2,1,GetNumSensors(sGetTag(UH.PNAMES.tag)));
     if (tag=UH.MODE[i].tag)   then UpdateTag(tag,w2,0,1);
     if (tag=UH.CONF[i].tag)   then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.HENABL[i].tag) then UpdateTag(tag,w2,0,1);
     if (tag=UH.TENABL[i].tag) then UpdateTag(tag,w2,0,1);
     if (tag=UH.PENABL[i].tag) then UpdateTag(tag,w2,0,1);
     if (tag=UH.WENABL[i].tag) then UpdateTag(tag,w2,0,1);
     if (tag=UH.PERIOD[i].tag) then UpdateTag(tag,w2,1000,60000);
     if (tag=UH.TGOAL[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.TREAL[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.PREAL[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.GATE[i].tag)   then UpdateTag(tag,w2,0,1);
     if (tag=UH.BLK[i].tag)    then UpdateTag(tag,w2,0,1);
     if (tag=UH.BT[i].tag)     then UpdateTag(tag,w2,0,1);
     if (tag=UH.ST[i].tag)     then UpdateTag(tag,w2,0,5);
     if (tag=UH.SB[i].tag)     then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.QX[i].tag)     then UpdateTag(tag,w2,0,100);
     if (tag=UH.Q1[i].tag)     then UpdateTag(tag,w2,0,100);
     if (tag=UH.Q2[i].tag)     then UpdateTag(tag,w2,0,100);
     if (tag=UH.Q3[i].tag)     then UpdateTag(tag,w2,0,100);
     if (tag=UH.DT1[i].tag)    then UpdateTag(tag,w2,0,1e4);
     if (tag=UH.DT2[i].tag)    then UpdateTag(tag,w2,0,1e4);
     if (tag=UH.WDT[i].tag)    then UpdateTag(tag,w2,0,1e4);
     if (tag=UH.TBLK[i].tag)   then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.PBLK[i].tag)   then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.EBLK[i].tag)   then UpdateTag(tag,w2,0,1);
     if (tag=UH.ICTRL[i].tag)  then UpdateTag(tag,w2,0,1);
     if (tag=UH.QMEAN[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.TMEAN[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.TRMSD[i].tag)  then UpdateTag(tag,w2,_MinusInf,_PlusInf);
     if (tag=UH.TSTAT[i].tag)  then UpdateTag(tag,w2,1,3600);
    end;
    if (tag=UH.TNAMES.tag)     then UpdateTag(tag,w2,0,1);
    if (tag=UH.PNAMES.tag)     then UpdateTag(tag,w2,0,1);
    if (tag=UH.SOUNDS.tag)     then UpdateTag(tag,w2,0,1);
   end;
  end;
  w1:=''; w2:='';
 end;
 {
 UniHeater: Handle message @DimUpdateTag arg.
 }
 procedure UniHeater_OnDimUpdateTag(arg:String);
 var i,tag,typ:Integer; x,y:Real;

 begin
  if (arg<>'') then begin
   if DIM_IsClientMode and not DIM_IsServerMode then begin
    tag:=FindTag(ExtractWord(1,arg));
    if (tag<>0) then begin
     typ:=TypeTag(tag);
     if (typ=1) then y:=iGetTag(tag) else
     if (typ=2) then y:=rGetTag(tag) else y:=_Nan;
     i:=0; x:=time;
     while (i<=UH.NumHeaters) and not IsNan(y) do begin
      if (tag=UH.QX[i].tag)    then begin UpdateAo(UH.QX[i].nao,x,y);    y:=_nan; end;
      if (tag=UH.QMEAN[i].tag) then begin UpdateAo(UH.QMEAN[i].nao,x,y); y:=_nan; end;
      if (tag=UH.TMEAN[i].tag) then begin UpdateAo(UH.TMEAN[i].nao,x,y); y:=_nan; end;
      if (tag=UH.TRMSD[i].tag) then begin UpdateAo(UH.TRMSD[i].nao,x,y); y:=_nan; end;
      i:=i+1;
     end;
    end;
   end;
  end;
 end;
 {
 UniHeater: Update DIM services.
 }
 procedure UniHeater_DimUpdate;
 var i,bt,timer_tp:Integer;
 begin
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then UniHeater_SetVals(-MaxReal);
   for i:=0 to UH.NumHeaters do begin
    if ShouldRefresh(UH.HENABL[i].val,GetStampOfTag(UH.HENABL[i].tag,0))
      +ShouldRefresh(UH.HNAME[i].val,GetStampOfTag(UH.HNAME[i].tag,0))>0
    then DIM_UpdateTag(UH.HENABL[i].tag,'');
    if ShouldRefresh(UH.TENABL[i].val,GetStampOfTag(UH.TENABL[i].tag,0))
      +ShouldRefresh(UH.TBLK[i].val,GetStampOfTag(UH.TBLK[i].tag,0))
      +ShouldRefresh(UH.TNUM[i].val,GetStampOfTag(UH.TNUM[i].tag,0))
      +ShouldRefresh(UH.TNAME[i].val,GetStampOfTag(UH.TNAME[i].tag,0))>0
    then DIM_UpdateTag(UH.TENABL[i].tag,'');
    if ShouldRefresh(UH.PENABL[i].val,GetStampOfTag(UH.PENABL[i].tag,0))
      +ShouldRefresh(UH.PBLK[i].val,GetStampOfTag(UH.PBLK[i].tag,0))
      +ShouldRefresh(UH.PNUM[i].val,GetStampOfTag(UH.PNUM[i].tag,0))
      +ShouldRefresh(UH.PNAME[i].val,GetStampOfTag(UH.PNAME[i].tag,0))>0
    then DIM_UpdateTag(UH.PENABL[i].tag,'');
    if ShouldRefresh(UH.WENABL[i].val,GetStampOfTag(UH.WENABL[i].tag,0))
      +ShouldRefresh(UH.WDT[i].val,GetStampOfTag(UH.WDT[i].tag,0))>0
    then DIM_UpdateTag(UH.WENABL[i].tag,'');
    if ShouldRefresh(UH.TGOAL[i].val,GetStampOfTag(UH.TGOAL[i].tag,0))
      +ShouldRefresh(UH.MODE[i].val,GetStampOfTag(UH.MODE[i].tag,0))
      +ShouldRefresh(UH.PERIOD[i].val,GetStampOfTag(UH.PERIOD[i].tag,0))
      +ShouldRefresh(UH.Q1[i].val,GetStampOfTag(UH.Q1[i].tag,0))
      +ShouldRefresh(UH.Q2[i].val,GetStampOfTag(UH.Q2[i].tag,0))
      +ShouldRefresh(UH.Q3[i].val,GetStampOfTag(UH.Q3[i].tag,0))
      +ShouldRefresh(UH.DT1[i].val,GetStampOfTag(UH.DT1[i].tag,0))
      +ShouldRefresh(UH.DT2[i].val,GetStampOfTag(UH.DT2[i].tag,0))>0
    then DIM_UpdateTag(UH.TGOAL[i].tag,'');
    if ShouldRefresh(UH.CONF[i].val,GetStampOfTag(UH.CONF[i].tag,0))>0
    then DIM_UpdateTag(UH.CONF[i].tag,'');
    if ShouldRefresh(UH.BT[i].val,GetStampOfTag(UH.BT[i].tag,0))
      +ShouldRefresh(UH.ST[i].val,GetStampOfTag(UH.ST[i].tag,0))
      +ShouldRefresh(UH.SB[i].val,GetStampOfTag(UH.SB[i].tag,0))
      +ShouldRefresh(UH.BLK[i].val,GetStampOfTag(UH.BLK[i].tag,0))
      +ShouldRefresh(UH.QX[i].val,GetStampOfTag(UH.QX[i].tag,0))>0
    then DIM_UpdateTag(UH.BT[i].tag,'');
    if ShouldRefresh(UH.GATE[i].val,GetStampOfTag(UH.GATE[i].tag,0))>0
    then DIM_UpdateTag(UH.GATE[i].tag,'');
    if ShouldRefresh(UH.EBLK[i].val,GetStampOfTag(UH.EBLK[i].tag,0))
      +ShouldRefresh(UH.ICTRL[i].val,GetStampOfTag(UH.ICTRL[i].tag,0))>0
    then DIM_UpdateTag(UH.EBLK[i].tag,'');
    if ShouldRefresh(UH.TSTAT[i].val,GetStampOfTag(UH.TSTAT[i].tag,0))
      +ShouldRefresh(UH.QMEAN[i].val,GetStampOfTag(UH.QMEAN[i].tag,0))
      +ShouldRefresh(UH.TMEAN[i].val,GetStampOfTag(UH.TMEAN[i].tag,0))
      +ShouldRefresh(UH.TRMSD[i].val,GetStampOfTag(UH.TRMSD[i].tag,0))>0
    then DIM_UpdateTag(UH.TSTAT[i].tag,'');
    {
    Limit frequency of TReal,PReal updates
    }
    bt:=iGetTag(UH.BT[i].tag);
    timer_tp:=200*Ord(bt<>0)+1000*Ord(bt=0);
    if (SysTimer_Pulse(timer_tp)>0) then begin
     if ShouldRefresh(UH.TREAL[i].val,UH.TREAL[i].tim)>0
     then DIM_UpdateTag(UH.TREAL[i].tag,'');
     if ShouldRefresh(UH.PREAL[i].val,UH.PREAL[i].tim)>0
     then DIM_UpdateTag(UH.PREAL[i].tag,'');
    end;
   end;
   if ShouldRefresh(UH.SOUNDS.val,iGetTag(UH.SOUNDS.tag))>0
   then DIM_UpdateTag(UH.SOUNDS.tag,'');
   if ShouldRefresh(UH.TNAMES.val,GetStampOfTag(UH.TNAMES.tag,0))>0
   then DIM_UpdateTag(UH.TNAMES.tag,'');
   if ShouldRefresh(UH.PNAMES.val,GetStampOfTag(UH.PNAMES.tag,0))>0
   then DIM_UpdateTag(UH.PNAMES.tag,'');
   if ShouldRefresh(UH.CLOCK.val,GetStampOfTag(UH.CLOCK.tag,0))>0
   then DIM_UpdateTag(UH.CLOCK.tag,'');
   if ShouldRefresh(UH.SERVID.val,GetStampOfTag(UH.SERVID.tag,0))>0
   then DIM_UpdateTag(UH.SERVID.tag,'');
  end;
 end;
 {
 UniHeater: Initialization.
 }
 procedure UniHeater_Init;
 var i:Integer;
 begin
  {
  Find number of heaters by max. number of tags HNAME_i
  }
  UH.NumHeaters:=0;
  for i:=1 to MaxHeater do if (FindTag(UniTagName('HNAME_',i))<>0) then UH.NumHeaters:=i;
  Success(StrFmt('NumHeaters = %d',UH.NumHeaters));
  {
  Initialize tags...
  }
  if (FindTag(tagPrefix+'.DIMGUICLICK')<>0)
  then DIM_GuiClickInit(tagPrefix+'.DIMGUICLICK');
  i:=Ord(DIM_IsServerMode); Success('DIM_IsServerMode = '+Str(i));
  i:=Ord(DIM_IsClientMode); Success('DIM_IsClientMode = '+Str(i));
  for i:=0 to UH.NumHeaters do begin
   InitTag( UH.HNAME[i].tag,  UniTagName('HNAME_',i),  3);
   InitTag( UH.TNAME[i].tag,  UniTagName('TNAME_',i),  3);
   InitTag( UH.PNAME[i].tag,  UniTagName('PNAME_',i),  3);
   InitTag( UH.TNUM[i].tag,   UniTagName('TNUM_',i),   1);
   InitTag( UH.PNUM[i].tag,   UniTagName('PNUM_',i),   1);
   InitTag( UH.MODE[i].tag,   UniTagName('MODE_',i),   1);
   InitTag( UH.CONF[i].tag,   UniTagName('CONF_',i),  -1);
   InitTag( UH.HENABL[i].tag, UniTagName('HENABL_',i), 1);
   InitTag( UH.TENABL[i].tag, UniTagName('TENABL_',i), 1);
   InitTag( UH.PENABL[i].tag, UniTagName('PENABL_',i), 1);
   InitTag( UH.WENABL[i].tag, UniTagName('WENABL_',i), 1);
   InitTag( UH.PERIOD[i].tag, UniTagName('PERIOD_',i), 2);
   InitTag( UH.TGOAL[i].tag,  UniTagName('TGOAL_',i),  2);
   InitTag( UH.TREAL[i].tag,  UniTagName('TREAL_',i),  2);
   InitTag( UH.PREAL[i].tag,  UniTagName('PREAL_',i),  2);
   InitTag( UH.GATE[i].tag,   UniTagName('GATE_',i),   1);
   InitTag( UH.BLK[i].tag,    UniTagName('BLK_',i),    1);
   InitTag( UH.BT[i].tag,     UniTagName('BT_',i),     1);
   InitTag( UH.ST[i].tag,     UniTagName('ST_',i),     1);
   InitTag( UH.SB[i].tag,     UniTagName('SB_',i),     1);
   InitTag( UH.QX[i].tag,     UniTagName('QX_',i),     2);
   InitTag( UH.Q1[i].tag,     UniTagName('Q1_',i),     2);
   InitTag( UH.Q2[i].tag,     UniTagName('Q2_',i),     2);
   InitTag( UH.Q3[i].tag,     UniTagName('Q3_',i),     2);
   InitTag( UH.DT1[i].tag,    UniTagName('DT1_',i),    2);
   InitTag( UH.DT2[i].tag,    UniTagName('DT2_',i),    2);
   InitTag( UH.WDT[i].tag,    UniTagName('WDT_',i),    2);
   InitTag( UH.PBLK[i].tag,   UniTagName('PBLK_',i),   2);
   InitTag( UH.TBLK[i].tag,   UniTagName('TBLK_',i),   2);
   InitTag( UH.EBLK[i].tag,   UniTagName('EBLK_',i),   1);
   InitTag( UH.ICTRL[i].tag,  UniTagName('ICTRL_',i),  1);
   InitTag( UH.QMEAN[i].tag,  UniTagName('QMEAN_',i),  2);
   InitTag( UH.TMEAN[i].tag,  UniTagName('TMEAN_',i),  2);
   InitTag( UH.TRMSD[i].tag,  UniTagName('TRMSD_',i),  2);
   InitTag( UH.TSTAT[i].tag,  UniTagName('TSTAT_',i),  2);
  end;
  InitTag( UH.TNAMES.tag,    UniTagName('TNAMES',-1),  3);
  InitTag( UH.PNAMES.tag,    UniTagName('PNAMES',-1),  3);
  InitTag( UH.SOUNDS.tag,    UniTagName('SOUNDS',-1),  1);
  InitTag( UH.SERVID.tag,    UniTagName('SERVID',-1),  3);
  InitTag( UH.CLOCK.tag,     UniTagName('CLOCK',-1),   3);
  ZeroSettings(0);
  {
  Bind tags with AI/DI/AO/AO...
  }
  UniHeater_LinkTags;
  for i:=0 to UH.NumHeaters do begin
   UH.QX[i].nao:=FindNaoByName(NameTag(UH.QX[i].tag));
   UH.QMEAN[i].nao:=FindNaoByName(NameTag(UH.QMEAN[i].tag));
   UH.TMEAN[i].nao:=FindNaoByName(NameTag(UH.TMEAN[i].tag));
   UH.TRMSD[i].nao:=FindNaoByName(NameTag(UH.TRMSD[i].tag));
   UH.TREAL[i].nai:=FindNaiByName(NameTag(UH.TREAL[i].tag));
   UH.TREAL[i].crv:=RefFind('Curve '+NameTag(UH.TREAL[i].tag));
   UH.PREAL[i].ndi:=FindNdiByName(NameTag(UH.PREAL[i].tag));
   UH.BLK[i].ndo:=FindNdoByName(NameTag(UH.BLK[i].tag));
   UH.GATE[i].crv:=RefFind('Curve '+NameTag(UH.GATE[i].tag));
  end;
  {
  Server Identifier.
  }
  if DIM_IsServerMode
  then UH.SelfId:=Str(getpid)+'@'+ParamStr('HostName')
  else UH.SelfId:=Str(getpid)+'@'+ParamStr('ComputerName');
  {
  Colors
  }
  ColorNorm:=StringToColor('Aqua');
  if DIM_IsServerMode then ColorNorm:=StringToColor('Lime');
  if DIM_IsClientMode then ColorNorm:=StringToColor('Lime');
  ColorWarn:=StringToColor('Yellow');
  bNul(SetTagColor(UH.SERVID.tag,ColorNorm));
  bNul(SetTagColor(UH.CLOCK.tag,ColorNorm));
  {
  Initialize timers
  }
  UH.tStart:=Time;
  for i:=1 to UH.NumHeaters do begin
   UH.ICTRL[i].tim:=SecNow;
   UH.BLK[i].tim:=SecNow;
  end;
  UH.CLOCK.tim:=mSecNow;
  {
  Initialize Statistics sums
  }
  if UsesStatSums then
  for i:=0 to UH.NumHeaters do begin
   TStatSum2D_Init(UH.STAT_TREAL[i]);
  end;
  {
  Initialize GUI
  }
  UsesSpeakSrv:=(Val(ReadIni('UsesSpeakSrv'))>0) and (devSpeakSrv<>0);
  bNul(WinHide(winSettings));
  EditingHNUM:=0;
  {
  Startup script & INI
  }
  RunStartupScript;
  if not DIM_IsClientMode then
  if (Val(ReadIni('CustomIniAutoLoad'))=1)
  then iNul(CustomIniRw('R','',2));
 end;
 {
 UniHeater: Finalization.
 }
 procedure UniHeater_Free;
 begin
  {
  Finally script & INI
  }
  if not DIM_IsClientMode then
  if (Val(ReadIni('CustomIniAutoSave'))=1)
  then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  {
  Finalize GUI
  }
  bNul(WinHide(winSettings));
  bNul(WinHide(winCtrl));
 end;
 {
 UniHeater: Poll DIM Client.
 }
 procedure UniHeater_Poll_Client;
 var i:Integer;
 begin
  if DIM_IsClientMode then begin
   if (ShouldRefresh(UH.CLOCK.dat,GetStampOfTag(UH.CLOCK.tag,0))>0) then begin
    bNul(SetTagColor(UH.SERVID.tag,ColorNorm));
    bNul(SetTagColor(UH.CLOCK.tag,ColorNorm));
    UH.CLOCK.tim:=mSecNow;
   end;
   if (SysTimer_Pulse(1000)>0) then
   if (mSecNow-UH.CLOCK.tim>DimDeadline) then begin
    bNul(sSetTag(UH.SERVID.tag,'Server Disconnected'));
    bNul(SetTagColor(UH.SERVID.tag,ColorWarn));
    bNul(SetTagColor(UH.CLOCK.tag,ColorWarn));
    for i:=1 to UH.NumHeaters do begin
     bNul(iSetTag(UH.ST[i].tag,6));
     bNul(iAtomicTagOp(UH.SB[i].tag,'|',sb_NETW));
    end;
   end;
  end;
 end;
 {
 UniHeater: Poll Logic for temperature stabilization.
 }
 procedure UniHeater_Poll_Logic;
 var i,MODE,ST,SB:Integer;
     BT,GATE,HENABL,BLK,BLK_WT,BLK_WP,BLK_VT,BLK_VP,EBLK,WENABL,ICTRL:Boolean;
     TimeOut,qx,TReal,PReal,TWhen,PWhen,TGoal,dt1,dt2:Real;
 begin
  {
  Update T,P names
  }
  if SysTimer_Pulse(NamesPeriod)>0 then UpdateSensorNames;
  {
  For all heaters: calculate off-duty factor, blocking, button states
  }
  for i:=1 to UH.NumHeaters do begin
   {
   Get TREAL, PREAL values and it`s update time.
   }
   TReal:=GetAi_Yn(i); TWhen:=GetAi_Xn(i); bNul(rSetTag(UH.TREAL[i].tag,TReal)); UH.TREAL[i].tim:=TWhen;
   PReal:=GetDi_Yn(i); PWhen:=GetDi_Xn(i); bNul(rSetTag(UH.PREAL[i].tag,PReal)); UH.PREAL[i].tim:=PWhen;
   {
   Get BT,HENABL,ICTRL state.
   }
   BT:=(iGetTag(UH.BT[i].tag)<>0);
   GATE:=(iGetTag(UH.GATE[i].tag)<>0);
   HENABL:=(iGetTag(UH.HENABL[i].tag)<>0);
   ICTRL:=(iGetTag(UH.ICTRL[i].tag)<>0);
   {
   Disabled heaters is always OFF
   }
   if not HENABL then begin
    if BT then bNul(iSetTag(UH.BT[i].tag,0));
    BT:=false;
   end;
   {
   Blocking by P, T, WDT & external EBLK
   }
   BLK:=false;    EBLK:=false;
   BLK_WT:=false; BLK_WP:=false;
   BLK_VT:=false; BLK_VP:=false;
   if HENABL then begin
    {
    Blocking by WDT
    }
    WENABL:=(iGetTag(UH.WENABL[i].tag)<>0) and (rGetTag(UH.WDT[i].tag)>0);
    TimeOut:=rGetTag(UH.WDT[i].tag)*1000/TimeUnits;
    {
    Blocking by T
    }
    if (iGetTag(UH.TENABL[i].tag)<>0) then
    if (RefAi(i)<>_Nil) then begin
     MODE:=iGetTag(UH.MODE[i].tag);
     if (MODE=mode_Heat) then begin
      BLK_WT:=(Time-Max(TWhen,UH.tStart)>TimeOut) and WENABL;
      BLK_VT:=(TReal>rGetTag(UH.TBLK[i].tag));
      BLK:=BLK or BLK_WT or BLK_VT;
     end else
     if (MODE=mode_Cool) then begin
      BLK_WT:=(Time-Max(TWhen,UH.tStart)>TimeOut) and WENABL;
      BLK_VT:=(TReal<rGetTag(UH.TBLK[i].tag));
      BLK:=BLK or BLK_WT or BLK_VT;
     end;
    end;
    {
    Blocking by P
    }
    if (iGetTag(UH.PENABL[i].tag)<>0) then
    if (RefDi(i)<>_Nil) then begin
     BLK_WP:=(Time-Max(PWhen,UH.tStart)>TimeOut) and WENABL;
     BLK_VP:=(PReal>rGetTag(UH.PBLK[i].tag));
     BLK:=BLK or BLK_WP or BLK_VP;
    end;
    {
    External blocking
    }
    EBLK:=(iGetTag(UH.EBLK[i].tag)<>0);
    BLK:=BLK or EBLK;
   end;
   {
   Write blockings to tag & curve
   Blocking tag   - by P, T overrange only
   Blocking curve - by P, T overrange & button OFF
   This helps to disconnect power in case of SSR shortcut.
   In general case use curve for hardware control and tag for GUI.
   }
   bNul(iSetTag(UH.BLK[i].tag,Ord(BLK)));
   UpdateDo(i,Time,Ord(BLK or not BT));
   {
   Off-duty factor calculation (0-100%):
   If button OFF or blocking present, off-duty = 0.
   If button ON and no blockings, return Q1,Q2,Q3,0 - depend on TReal vs TGoal.
   Then write off-duty to curve.
   }
   qx:=0;
   if BT and not BLK then begin
    TGoal:=rGetTag(UH.TGOAL[i].tag);
    dt1:=rGetTag(UH.DT1[i].tag);
    dt2:=rGetTag(UH.DT2[i].tag);
    MODE:=iGetTag(UH.MODE[i].tag);
    if (MODE=mode_Heat) then begin
     if (TReal<TGoal-dt1) then qx:=rGetTag(UH.Q1[i].tag) else
     if (TReal<TGoal)     then qx:=rGetTag(UH.Q2[i].tag) else
     if (TReal<TGoal+dt2) then qx:=rGetTag(UH.Q3[i].tag) else qx:=0;
    end else
    if (MODE=mode_Cool) then begin
     if (TReal>TGoal+dt1) then qx:=rGetTag(UH.Q1[i].tag) else
     if (TReal>TGoal)     then qx:=rGetTag(UH.Q2[i].tag) else
     if (TReal>TGoal-dt2) then qx:=rGetTag(UH.Q3[i].tag) else qx:=0;
    end;
   end;
   bNul(rSetTag(UH.QX[i].tag,qx));
   UpdateAo(UH.QX[i].nao,time,qx);
   {
   Heater button state BT_i:
   0 - Heater button OFF
   1 - Heater button ON, qx=0
   2 - Heater button ON, qx>0
   3 - Heater button ON, ICTRL show wire cut
   4 - Heater button OFF, BLOCKING
   5 - Heater button OFF, BLOCKING
   Priority order: blocking, button, ICTRL, state of qx>0
   }
   if BLK    then ST:=4+Ord(BT) else
   if not BT then ST:=0         else
   if ICTRL  then ST:=3         else
   if (qx>0) then ST:=2         else ST:=1;
   bNul(iSetTag(UH.ST[i].tag,ST));
   {
   Status byte.
   }
   SB:=Ord(BT)               * sb_BT
      +Ord(GATE)             * sb_GATE
      +Ord(BLK)              * sb_SBLK
      +Ord(BLK or not BT)    * sb_BLK
      +Ord(EBLK)             * sb_EBLK
      +Ord(ICTRL)            * sb_ICTRL
      +Ord(BLK_WT)           * sb_TWDT
      +Ord(BLK_WP)           * sb_PWDT
      +Ord(BLK_VT)           * sb_TBLK
      +Ord(BLK_VP)           * sb_PBLK
      +Ord(qx>0)             * sb_QXON;
   bNul(iSetTag(UH.SB[i].tag,SB));
   {
   Voice on blocking
   }
   if BLK then begin
    if (SecNow-UH.BLK[i].tim>MsgPeriod) then begin
     if iGetTag(UH.SOUNDS.tag)<>0 then begin
      if UsesSpeakSrv
      then Speak(spk_BLK+' '+Str(i))
      else bNul(Voice(snd_BLK+' '+Str(i)));
     end;
     UH.BLK[i].tim:=SecNow;
    end;
   end else UH.BLK[i].tim:=0;
   {
   Voice on ICTRL
   }
   if (ST=3) then begin
    if (SecNow-UH.ICTRL[i].tim>MsgPeriod) then begin
     if (iGetTag(UH.SOUNDS.tag)<>0) then begin
      if UsesSpeakSrv
      then Speak(spk_ICTRL+' '+Str(i))
      else bNul(Voice(snd_ICTRL+' '+Str(i)));
     end;
     UH.ICTRL[i].tim:=SecNow;
    end;
   end else UH.ICTRL[i].tim:=0;
  end;
  {
  Update Host Date-Time.
  }
  if (SysTimer_Pulse(1000)>0) then begin
   bNul(sSetTag(UH.SERVID.tag,UH.SelfId));
   bNul(sSetTag(UH.CLOCK.tag,GetDateTime(mSecNow)));
   bNul(SetTagColor(UH.SERVID.tag,ColorNorm));
   bNul(SetTagColor(UH.CLOCK.tag,ColorNorm));
  end;
  {
  Calculate statistics.
  }
  UniHeater_Poll_Stat;
  {
  Update DIM tags.
  }
  UniHeater_DimUpdate;
 end;
 {
 UniHeater: Poll GUI.
 }
 procedure UniHeater_Poll_Gui;
 var i,ib,mx,bti,nv,ClickCurve:Integer; s:String;
  //
  // Search curves for open plot window
  //
  function SearchCurve(n:Integer;sec:String):Integer;
  var i,txt,crv:Integer;
  begin
   crv:=0;
   txt:=readinisection(text_new,16,'',sec);
   for i:=0 to text_numln(txt)-1 do begin
    if ExtractWord(1,text_getln(txt,i))='Link' then
    if ExtractWord(2,text_getln(txt,i))='AnalogInput' then
    if ExtractWord(4,text_getln(txt,i))='with' then
    if ExtractWord(5,text_getln(txt,i))='curve' then
    if n=val(ExtractWord(3,text_getln(txt,i))) then begin
     crv:=RefFind('Curve '+ExtractWord(6,text_getln(txt,i)));
     bNul(text_free(txt));
    end;
   end;
   SearchCurve:=crv;
  end;
 begin
  s:='';
  DIM_GuiClickBuff:='';
  {
  Update PREAL_0,TREAL_0,.. tags in editing state.
  }
  if IsEditingSettings then begin
   bNul(iSetTag(UH.SB[0].tag,iGetTag(UH.SB[EditingHNUM].tag)));
   bNul(rSetTag(UH.QX[0].tag,rGetTag(UH.QX[EditingHNUM].tag)));
   bNul(rSetTag(UH.TREAL[0].tag,rGetTag(UH.TREAL[EditingHNUM].tag)));
   bNul(rSetTag(UH.PREAL[0].tag,rGetTag(UH.PREAL[EditingHNUM].tag)));
   bNul(rSetTag(UH.QMEAN[0].tag,rGetTag(UH.QMEAN[EditingHNUM].tag)));
   bNul(rSetTag(UH.TMEAN[0].tag,rGetTag(UH.TMEAN[EditingHNUM].tag)));
   bNul(rSetTag(UH.TRMSD[0].tag,rGetTag(UH.TRMSD[EditingHNUM].tag)));
  end;
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Copy GUI click to DIM buffer for remote execution.
   }
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    if (ClickButton=VK_LBUTTON)  then begin
     {
     Handle local clicks
     }
     if ClickIsLocal then begin
      for i:=1 to UH.NumHeaters do begin
       {
       Open temperature plot window out of UniHeater control dialog
       }
       if IsSameText(ClickSensor,UniSenName('TREAL_',i)) then begin
        ClickCurve:=SearchCurve(iGetTag(UH.TNUM[i].tag),'['+ForceExtension(DevName,'SELECT_T')+']');
        if ClickCurve<>0 then begin
         iNul(WinSelectByCurve(ClickCurve,ClickCurve));
         bNul(Voice(snd_Wheel));
        end;
       end;
       {
       Press heater button if enabled
       }
       if IsSameText(ClickSensor,UniSenName('BT_',i)) then
       if (iGetTag(UH.HENABL[i].tag)=0) then begin
        WarnNotify('Нагреватель '+sGetTag(UH.HNAME[i].tag)+' выключен',10000);
        bNul(Voice(snd_Error));
       end else begin
        bti:=iGetTag(UH.BT[i].tag);
        if NeedConfirm(i,iAnd(bti,1)) then begin
         if EditStateReady then begin
          if (EditAddOpening('Требуется Ваше подтверждение!')
             +EditAddInputLn('Вы действительно хотите '+ExtractWord(1+iAnd(bti,1),'ВКЛЮЧИТЬ,ВЫКЛЮЧИТЬ')+' нагреватель '+sGetTag(UH.HNAME[i].tag)+'?')
             +EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')))
             +EditAddClosing('YesNo',uidYesNoTag(UH.BT[i].tag),'')>0)
          then Problem('Error initializing YesNo dialog!');
         end else Problem('Cannot edit right now!');
         bNul(Voice(snd_Click));
        end else begin
         nv:=Ord(bti=0); s:=Str(nv);
         PostRemoteAssignTag(UH.BT[i].tag,s);
         bNul(Voice(snd_Click));
        end;
       end else
       {
       Open Settings window
       }
       if IsSameText(ClickSensor,UniSenName('PROP_',i)) or IsSameText(ClickSensor,UniSenName('HENABL_',i)) then begin
        bNul(Voice(snd_Click));
        StartEditSettings(i);
       end else
       {
       Edit tags from main GUI
       }
       CheckStartEditHeaterSettings(i);
      end;
      {
      Open temperature plot window out of Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('TREAL_',0)) then begin
       ClickCurve:=SearchCurve(iGetTag(UH.TNUM[EditingHNUM].tag),'['+ForceExtension(DevName,'SELECT_T')+']');
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
       ClickCurve:=RefFind('Curve '+NameTag(UH.TREAL[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Open pressure plot window out of Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('PREAL_',0)) then begin
       ClickCurve:=SearchCurve(iGetTag(UH.PNUM[EditingHNUM].tag),'['+ForceExtension(DevName,'SELECT_P')+']');
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
       ClickCurve:=RefFind('Curve '+NameTag(UH.PREAL[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Open QX plot
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('QX_',0)) then begin
       ClickCurve:=RefFind('Curve '+NameTag(UH.QX[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Open QMEAN plot
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('QMEAN_',0)) then begin
       ClickCurve:=RefFind('Curve '+NameTag(UH.QMEAN[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Open TMEAN plot
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('TMEAN_',0)) then begin
       ClickCurve:=RefFind('Curve '+NameTag(UH.TMEAN[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Open TRMSD plot
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('TRMSD_',0)) then begin
       ClickCurve:=RefFind('Curve '+NameTag(UH.TRMSD[EditingHNUM].tag));
       if ClickCurve<>0 then begin
        iNul(WinSelectByCurve(ClickCurve,ClickCurve));
        bNul(Voice(snd_Wheel));
       end;
      end;
      {
      Edit tags from Settings window
      }
      if IsEditingSettings then CheckStartEditHeaterSettings(0);
      {
      Accept button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('ACCEPT',-1)) then begin
       bNul(Voice(snd_Click));
       StopEditSettings;
      end;
      {
      Cancel button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('CANCEL',-1)) then begin
       bNul(Voice(snd_Cancel));
       CloseSettings;
      end;
      {
      Close button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('CLOSE',-1)) then begin
       bNul(Voice(snd_Cancel));
       CloseSettings;
      end;
      {
      SaveIni button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('SAVEINI',-1)) then begin
       if NeedConfirm(0,2) then begin
        if EditStateReady then begin
         if (EditAddOpening('Требуется Ваше подтверждение!')
            +EditAddInputLn('Вы действительно хотите СОХРАНИТЬ параметры в INI файле?')
            +EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')))
            +EditAddClosing('YesNo',uidYesNoSaveIni,'')>0)
         then Problem('Error initializing YesNo dialog!');
        end else Problem('Cannot edit right now!');
        bNul(Voice(snd_Click));
       end else begin
        DevPostCmdLocal(DimRemote+'@SaveIni');
        bNul(Voice(snd_Click));
        StopEditSettings;
       end;
      end;
      {
      LoadIni button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('LOADINI',-1)) then begin
       if NeedConfirm(0,2) then begin
        if EditStateReady then begin
         if (EditAddOpening('Требуется Ваше подтверждение!')
            +EditAddInputLn('Вы действительно хотите ЗАГРУЗИТЬ параметры в INI файле?')
            +EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')))
            +EditAddClosing('YesNo',uidYesNoLoadIni,'')>0)
         then Problem('Error initializing YesNo dialog!');
        end else Problem('Cannot edit right now!');
        bNul(Voice(snd_Click));
       end else begin
        DevPostCmdLocal(DimRemote+'@LoadIni');
        bNul(Voice(snd_Click));
        CloseSettings;
       end;
      end;
      {
      Manual button pressed in Settings window
      }
      if IsEditingSettings then
      if IsSameText(ClickSensor,UniSenName('MANUAL',-1)) then begin
       DevPostCmdLocal('@BrowseHelp '+ReadIni('UniHeaterManual'));
       bNul(Voice(snd_Click));
      end;
      {
      Demo.UniHeater.Make.lm9 button pressed
      }
      if IsSameText(ClickSensor,UniSenName('MAKE.LM9',-1)) then begin
       bNul(Voice(snd_Click));
       Cron('@run '+AddPathDelim(ProgramSourceDir)+'uniheater_make.lm9'+' --config='+ParamStr('DaqConfigFile'));
      end;
      {
      Sound on/off button
      }
      if IsEditingSettings then
      if (ClickTag=UH.SOUNDS.tag) then begin
       bNul(iSetTag(UH.SOUNDS.tag,Ord(iGetTag(UH.SOUNDS.tag)=0)));
       bNul(Voice(snd_Click));
      end;
      {
      Enable/disable buttons for heaters & blocking
      }
      if IsEditingSettings then
      if (ClickTag=UH.HENABL[0].tag) or (ClickTag=UH.TENABL[0].tag)
      or (ClickTag=UH.PENABL[0].tag) or (ClickTag=UH.WENABL[0].tag) then begin
       nv:=Ord(iGetTag(ClickTag)=0); s:=Str(nv);
       PostLocalAssignTag(ClickTag,s);
       bNul(Voice(snd_Click));
      end;
      {
      Mode button for heaters
      }
      if IsEditingSettings then
      if (ClickTag=UH.MODE[0].tag) then begin
       if EditStateReady then begin
        mx:=iGetTag(UH.MODE[0].tag);
        if (EditAddOpening('Операция смены режима требует Вашего подтверждения!')
           +EditAddInputLn('Вы действительно хотите изменить режим работы нагревателя '+sGetTag(UH.HNAME[0].tag)+'?')
           +EditAddInputLn('Режим '+ExtractWord(1+mx,'НАГРЕВ,ОХЛАЖДЕНИЕ')+' заменится на режим '+ExtractWord(1+mx,'ОХЛАЖДЕНИЕ,НАГРЕВ')+'.')
           +EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')))
           +EditAddClosing('YesNo',uidYesNoMode,'')>0)
        then Problem('Error initializing YesNo dialog!');
       end else Problem('Cannot edit right now!');
       bNul(Voice(snd_Click));
      end;
      {
      Confirmation checkboxes
      }
      if IsEditingSettings then
      if (ClickTag=UH.CONF[0].tag) then begin
       nv:=iGetTag(ClickTag);
       for ib:=0 to 31 do begin
        s:='_'+Str(ib); // Sensor has name XX.CONF_i_n where n is bit number
        if IsSameText(s,Copy(ClickSensor,Length(ClickSensor)-Length(s)+1,Length(s)))
        then nv:=iXor(nv,iShift(1,ib));
       end;
       s:=Str(nv);
       PostLocalAssignTag(ClickTag,s);
       bNul(Voice(snd_Click));
      end;
      {
      Select Plot & Tab windows by curve...
      }
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      {
      Console commands: @url_encoded_sensor ...
      }
      if LooksLikeCommand(ClickSensor) then begin
       DevSendCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     {
     Handle remote clicks comes from DIM via @DimGuiClick message.
     @DimGuiClick default handler decode and write events to FIFO,
     so we can find it as clicks and can handle it in usual way.
     }
     if ClickIsRemote then begin
      {
      Show time difference.
      }
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      {
      Handle remote console commands...
      }
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit tags...
  }
  if EditStateDone then begin
   {
   Warning,Information.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
   {
   Edit heater tags
   }
   for i:=0 to UH.NumHeaters do begin
    CheckEditTagRemote(UH.HNAME[i].tag,i);
    CheckEditTagRemote(UH.TGOAL[i].tag,i);
    CheckEditTagRemote(UH.PERIOD[i].tag,i);
    CheckEditTagRemote(UH.TSTAT[i].tag,i);
    CheckEditTagRemote(UH.Q1[i].tag,i);
    CheckEditTagRemote(UH.Q2[i].tag,i);
    CheckEditTagRemote(UH.Q3[i].tag,i);
    CheckEditTagRemote(UH.DT1[i].tag,i);
    CheckEditTagRemote(UH.DT2[i].tag,i);
    CheckEditTagRemote(UH.WDT[i].tag,i);
    CheckEditTagRemote(UH.TBLK[i].tag,i);
    CheckEditTagRemote(UH.PBLK[i].tag,i);
    CheckEditTagRemote(UH.TNUM[i].tag,i);
    CheckEditTagRemote(UH.PNUM[i].tag,i);
    CheckEditSensor(UH.TNUM[i].tag,i);
    CheckEditSensor(UH.PNUM[i].tag,i);
   end;
   // Mode after confirm
   if EditTestResultName(uidYesNoMode) then begin
    if EditTestResultCode(mr_Yes) then begin
     nv:=Ord(iGetTag(UH.MODE[0].tag)=0); s:=Str(nv);
     PostLocalAssignTag(UH.MODE[0].tag,s);
    end;
    EditReset;
   end;
   // BT_i after confirm
   if EditStateDone then
   for i:=1 to UH.NumHeaters do begin
    if EditTestResultName(uidYesNoTag(UH.BT[i].tag)) then begin
     if EditTestResultCode(mr_Yes) then begin
      bti:=iGetTag(UH.BT[i].tag);
      nv:=Ord(bti=0); s:=Str(nv);
      PostRemoteAssignTag(UH.BT[i].tag,s);
     end;
     EditReset;
    end;
   end;
   // SaveIni after confirm
   if EditTestResultName(uidYesNoSaveIni) then begin
    if EditTestResultCode(mr_Yes) then begin
     DevPostCmdLocal(DimRemote+'@SaveIni');
     StopEditSettings;
    end;
    EditReset;
   end;
   // LoadIni after confirm
   if EditTestResultName(uidYesNoLoadIni) then begin
    if EditTestResultCode(mr_Yes) then begin
     DevPostCmdLocal(DimRemote+'@LoadIni');
     CloseSettings;
    end;
    EditReset;
   end;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end;
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  DIM_GuiClickBuff:='';
  s:='';
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  UniHeater_ClearTags;
  UniHeater_SetVals(-MaxReal);
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  cmd_DimTagUpdate := RegisterStdInCmd('@DimTagUpdate', '');
  cmd_AssignTag    := RegisterStdInCmd('@AssignTag',    '');
  UniHeater_Init;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  UniHeater_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  if DIM_IsClientMode
  then UniHeater_Poll_Client
  else UniHeater_Poll_Logic;
  UniHeater_Poll_Gui;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid:Integer;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    UniHeater_OnDimUpdateTag(arg);
    Data:='';
   end else
   {
   @AssignTag DEMO.UNIHEATER.TNUM_1 1
   }
   if (cmdid=cmd_AssignTag) then begin
    UniHeater_OnAssignTag(arg);
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
